"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AlarmRule = exports.AlarmState = void 0;
/**
 * Enumeration indicates state of Alarm used in building Alarm Rule.
 */
var AlarmState;
(function (AlarmState) {
    AlarmState["ALARM"] = "ALARM";
    AlarmState["OK"] = "OK";
    AlarmState["INSUFFICIENT_DATA"] = "INSUFFICIENT_DATA";
})(AlarmState = exports.AlarmState || (exports.AlarmState = {}));
/**
 * Enumeration of supported Composite Alarms operators.
 */
var Operator;
(function (Operator) {
    Operator["AND"] = "AND";
    Operator["OR"] = "OR";
    Operator["NOT"] = "NOT";
})(Operator || (Operator = {}));
/**
 * Class with static functions to build AlarmRule for Composite Alarms.
 */
class AlarmRule {
    /**
     * function to join all provided AlarmRules with AND operator.
     *
     * @param operands IAlarmRules to be joined with AND operator.
     */
    static allOf(...operands) {
        return this.concat(Operator.AND, ...operands);
    }
    /**
     * function to join all provided AlarmRules with OR operator.
     *
     * @param operands IAlarmRules to be joined with OR operator.
     */
    static anyOf(...operands) {
        return this.concat(Operator.OR, ...operands);
    }
    /**
     * function to wrap provided AlarmRule in NOT operator.
     *
     * @param operand IAlarmRule to be wrapped in NOT operator.
     */
    static not(operand) {
        return new class {
            renderAlarmRule() {
                return `(NOT (${operand.renderAlarmRule()}))`;
            }
        };
    }
    /**
     * function to build TRUE/FALSE intent for Rule Expression.
     *
     * @param value boolean value to be used in rule expression.
     */
    static fromBoolean(value) {
        return new class {
            renderAlarmRule() {
                return `${String(value).toUpperCase()}`;
            }
        };
    }
    /**
     * function to build Rule Expression for given IAlarm and AlarmState.
     *
     * @param alarm IAlarm to be used in Rule Expression.
     * @param alarmState AlarmState to be used in Rule Expression.
     */
    static fromAlarm(alarm, alarmState) {
        return new class {
            renderAlarmRule() {
                return `${alarmState}(${alarm.alarmArn})`;
            }
        };
    }
    /**
     * function to build Rule Expression for given Alarm Rule string.
     *
     * @param alarmRule string to be used in Rule Expression.
     */
    static fromString(alarmRule) {
        return new class {
            renderAlarmRule() {
                return alarmRule;
            }
        };
    }
    static concat(operator, ...operands) {
        return new class {
            renderAlarmRule() {
                const expression = operands
                    .map(operand => `${operand.renderAlarmRule()}`)
                    .join(` ${operator} `);
                return `(${expression})`;
            }
        };
    }
}
exports.AlarmRule = AlarmRule;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYWxhcm0tcnVsZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImFsYXJtLXJ1bGUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBS0EsSUFBWSxVQWlCWDtBQWpCRCxXQUFZLFVBQVU7SUFLcEIsNkJBQWUsQ0FBQTtJQUtmLHVCQUFTLENBQUE7SUFLVCxxREFBdUMsQ0FBQTtBQUV6QyxDQUFDLEVBakJXLFVBQVUsR0FBVixrQkFBVSxLQUFWLGtCQUFVLFFBaUJyQjtBQUVEOztHQUVHO0FBQ0gsSUFBSyxRQU1KO0FBTkQsV0FBSyxRQUFRO0lBRVgsdUJBQVcsQ0FBQTtJQUNYLHFCQUFTLENBQUE7SUFDVCx1QkFBVyxDQUFBO0FBRWIsQ0FBQyxFQU5JLFFBQVEsS0FBUixRQUFRLFFBTVo7Ozs7QUFLRCxNQUFhLFNBQVM7Ozs7OztJQU9iLE1BQU0sQ0FBQyxLQUFLLENBQUMsR0FBRyxRQUFzQjtRQUMzQyxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLEdBQUcsRUFBRSxHQUFHLFFBQVEsQ0FBQyxDQUFDO0lBQ2hELENBQUM7Ozs7OztJQU9NLE1BQU0sQ0FBQyxLQUFLLENBQUMsR0FBRyxRQUFzQjtRQUMzQyxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLEVBQUUsRUFBRSxHQUFHLFFBQVEsQ0FBQyxDQUFDO0lBQy9DLENBQUM7Ozs7OztJQU9NLE1BQU0sQ0FBQyxHQUFHLENBQUMsT0FBbUI7UUFDbkMsT0FBTyxJQUFJO1lBQ0YsZUFBZTtnQkFDcEIsT0FBTyxTQUFTLE9BQU8sQ0FBQyxlQUFlLEVBQUUsSUFBSSxDQUFDO1lBQ2hELENBQUM7U0FDRixDQUFDO0lBQ0osQ0FBQzs7Ozs7O0lBT00sTUFBTSxDQUFDLFdBQVcsQ0FBQyxLQUFjO1FBQ3RDLE9BQU8sSUFBSTtZQUNGLGVBQWU7Z0JBQ3BCLE9BQU8sR0FBRyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsV0FBVyxFQUFFLEVBQUUsQ0FBQztZQUMxQyxDQUFDO1NBQ0YsQ0FBQztJQUNKLENBQUM7Ozs7Ozs7SUFRTSxNQUFNLENBQUMsU0FBUyxDQUFDLEtBQWEsRUFBRSxVQUFzQjtRQUMzRCxPQUFPLElBQUk7WUFDRixlQUFlO2dCQUNwQixPQUFPLEdBQUcsVUFBVSxJQUFJLEtBQUssQ0FBQyxRQUFRLEdBQUcsQ0FBQztZQUM1QyxDQUFDO1NBQ0YsQ0FBQztJQUNKLENBQUM7Ozs7OztJQU9NLE1BQU0sQ0FBQyxVQUFVLENBQUMsU0FBaUI7UUFDeEMsT0FBTyxJQUFJO1lBQ0YsZUFBZTtnQkFDcEIsT0FBTyxTQUFTLENBQUM7WUFDbkIsQ0FBQztTQUNGLENBQUM7SUFDSixDQUFDO0lBRU8sTUFBTSxDQUFDLE1BQU0sQ0FBQyxRQUFrQixFQUFFLEdBQUcsUUFBc0I7UUFDakUsT0FBTyxJQUFJO1lBQ0YsZUFBZTtnQkFDcEIsTUFBTSxVQUFVLEdBQUcsUUFBUTtxQkFDeEIsR0FBRyxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsR0FBRyxPQUFPLENBQUMsZUFBZSxFQUFFLEVBQUUsQ0FBQztxQkFDOUMsSUFBSSxDQUFDLElBQUksUUFBUSxHQUFHLENBQUMsQ0FBQztnQkFDekIsT0FBTyxJQUFJLFVBQVUsR0FBRyxDQUFDO1lBQzNCLENBQUM7U0FDRixDQUFDO0lBQ0osQ0FBQztDQUNGO0FBbkZELDhCQW1GQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IElBbGFybSwgSUFsYXJtUnVsZSB9IGZyb20gJy4vYWxhcm0tYmFzZSc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBlbnVtIEFsYXJtU3RhdGUge1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBBTEFSTSA9ICdBTEFSTScsXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBPSyA9ICdPSycsXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgSU5TVUZGSUNJRU5UX0RBVEEgPSAnSU5TVUZGSUNJRU5UX0RBVEEnLFxuXG59XG5cbi8qKlxuICogRW51bWVyYXRpb24gb2Ygc3VwcG9ydGVkIENvbXBvc2l0ZSBBbGFybXMgb3BlcmF0b3JzLlxuICovXG5lbnVtIE9wZXJhdG9yIHtcblxuICBBTkQgPSAnQU5EJyxcbiAgT1IgPSAnT1InLFxuICBOT1QgPSAnTk9UJyxcblxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgQWxhcm1SdWxlIHtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBhbGxPZiguLi5vcGVyYW5kczogSUFsYXJtUnVsZVtdKTogSUFsYXJtUnVsZSB7XG4gICAgcmV0dXJuIHRoaXMuY29uY2F0KE9wZXJhdG9yLkFORCwgLi4ub3BlcmFuZHMpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGFueU9mKC4uLm9wZXJhbmRzOiBJQWxhcm1SdWxlW10pOiBJQWxhcm1SdWxlIHtcbiAgICByZXR1cm4gdGhpcy5jb25jYXQoT3BlcmF0b3IuT1IsIC4uLm9wZXJhbmRzKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgbm90KG9wZXJhbmQ6IElBbGFybVJ1bGUpOiBJQWxhcm1SdWxlIHtcbiAgICByZXR1cm4gbmV3IGNsYXNzIGltcGxlbWVudHMgSUFsYXJtUnVsZSB7XG4gICAgICBwdWJsaWMgcmVuZGVyQWxhcm1SdWxlKCk6IHN0cmluZyB7XG4gICAgICAgIHJldHVybiBgKE5PVCAoJHtvcGVyYW5kLnJlbmRlckFsYXJtUnVsZSgpfSkpYDtcbiAgICAgIH1cbiAgICB9O1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGZyb21Cb29sZWFuKHZhbHVlOiBib29sZWFuKTogSUFsYXJtUnVsZSB7XG4gICAgcmV0dXJuIG5ldyBjbGFzcyBpbXBsZW1lbnRzIElBbGFybVJ1bGUge1xuICAgICAgcHVibGljIHJlbmRlckFsYXJtUnVsZSgpOiBzdHJpbmcge1xuICAgICAgICByZXR1cm4gYCR7U3RyaW5nKHZhbHVlKS50b1VwcGVyQ2FzZSgpfWA7XG4gICAgICB9XG4gICAgfTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGZyb21BbGFybShhbGFybTogSUFsYXJtLCBhbGFybVN0YXRlOiBBbGFybVN0YXRlKTogSUFsYXJtUnVsZSB7XG4gICAgcmV0dXJuIG5ldyBjbGFzcyBpbXBsZW1lbnRzIElBbGFybVJ1bGUge1xuICAgICAgcHVibGljIHJlbmRlckFsYXJtUnVsZSgpOiBzdHJpbmcge1xuICAgICAgICByZXR1cm4gYCR7YWxhcm1TdGF0ZX0oJHthbGFybS5hbGFybUFybn0pYDtcbiAgICAgIH1cbiAgICB9O1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGZyb21TdHJpbmcoYWxhcm1SdWxlOiBzdHJpbmcpOiBJQWxhcm1SdWxlIHtcbiAgICByZXR1cm4gbmV3IGNsYXNzIGltcGxlbWVudHMgSUFsYXJtUnVsZSB7XG4gICAgICBwdWJsaWMgcmVuZGVyQWxhcm1SdWxlKCk6IHN0cmluZyB7XG4gICAgICAgIHJldHVybiBhbGFybVJ1bGU7XG4gICAgICB9XG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgc3RhdGljIGNvbmNhdChvcGVyYXRvcjogT3BlcmF0b3IsIC4uLm9wZXJhbmRzOiBJQWxhcm1SdWxlW10pOiBJQWxhcm1SdWxlIHtcbiAgICByZXR1cm4gbmV3IGNsYXNzIGltcGxlbWVudHMgSUFsYXJtUnVsZSB7XG4gICAgICBwdWJsaWMgcmVuZGVyQWxhcm1SdWxlKCk6IHN0cmluZyB7XG4gICAgICAgIGNvbnN0IGV4cHJlc3Npb24gPSBvcGVyYW5kc1xuICAgICAgICAgIC5tYXAob3BlcmFuZCA9PiBgJHtvcGVyYW5kLnJlbmRlckFsYXJtUnVsZSgpfWApXG4gICAgICAgICAgLmpvaW4oYCAke29wZXJhdG9yfSBgKTtcbiAgICAgICAgcmV0dXJuIGAoJHtleHByZXNzaW9ufSlgO1xuICAgICAgfVxuICAgIH07XG4gIH1cbn1cbiJdfQ==