"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EdgeFunction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const ssm = require("@aws-cdk/aws-ssm");
const core_1 = require("@aws-cdk/core");
/**
 * A Lambda@Edge function.
 *
 * Convenience resource for requesting a Lambda function in the 'us-east-1' region for use with Lambda@Edge.
 * Implements several restrictions enforced by Lambda@Edge.
 *
 * Note that this construct requires that the 'us-east-1' region has been bootstrapped.
 * See https://docs.aws.amazon.com/cdk/latest/guide/bootstrapping.html or 'cdk bootstrap --help' for options.
 *
 * @stability stable
 * @resource AWS::Lambda::Function
 */
class EdgeFunction extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Whether or not this Lambda function was bound to a VPC.
         *
         * If this is is `false`, trying to access the `connections` object will fail.
         *
         * @stability stable
         */
        this.isBoundToVpc = false;
        // Create a simple Function if we're already in us-east-1; otherwise create a cross-region stack.
        const regionIsUsEast1 = !core_1.Token.isUnresolved(this.stack.region) && this.stack.region === 'us-east-1';
        const { edgeFunction, edgeArn } = regionIsUsEast1
            ? this.createInRegionFunction(props)
            : this.createCrossRegionFunction(id, props);
        this.edgeArn = edgeArn;
        this.functionArn = edgeArn;
        this._edgeFunction = edgeFunction;
        this.functionName = this._edgeFunction.functionName;
        this.grantPrincipal = this._edgeFunction.role;
        this.permissionsNode = this._edgeFunction.permissionsNode;
        this.version = lambda.extractQualifierFromArn(this.functionArn);
        this.node.defaultChild = this._edgeFunction;
    }
    /**
     * The underlying AWS Lambda function.
     *
     * @stability stable
     */
    get lambda() {
        return this._edgeFunction;
    }
    /**
     * Convenience method to make `EdgeFunction` conform to the same interface as `Function`.
     *
     * @stability stable
     */
    get currentVersion() {
        return this;
    }
    /**
     * Defines an alias for this version.
     *
     * @stability stable
     */
    addAlias(aliasName, options = {}) {
        return new lambda.Alias(this._edgeFunction, `Alias${aliasName}`, {
            aliasName,
            version: this._edgeFunction.currentVersion,
            ...options,
        });
    }
    /**
     * Not supported.
     *
     * Connections are only applicable to VPC-enabled functions.
     *
     * @stability stable
     */
    get connections() {
        throw new Error('Lambda@Edge does not support connections');
    }
    /**
     * The `$LATEST` version of this function.
     *
     * Note that this is reference to a non-specific AWS Lambda version, which
     * means the function this version refers to can return different results in
     * different invocations.
     *
     * To obtain a reference to an explicit version which references the current
     * function configuration, use `lambdaFunction.currentVersion` instead.
     *
     * @stability stable
     */
    get latestVersion() {
        throw new Error('$LATEST function version cannot be used for Lambda@Edge');
    }
    /**
     * Adds an event source that maps to this AWS Lambda function.
     *
     * @stability stable
     */
    addEventSourceMapping(id, options) {
        return this.lambda.addEventSourceMapping(id, options);
    }
    /**
     * Adds a permission to the Lambda resource policy.
     *
     * @stability stable
     */
    addPermission(id, permission) {
        return this.lambda.addPermission(id, permission);
    }
    /**
     * Adds a statement to the IAM role assumed by the instance.
     *
     * @stability stable
     */
    addToRolePolicy(statement) {
        return this.lambda.addToRolePolicy(statement);
    }
    /**
     * Grant the given identity permissions to invoke this Lambda.
     *
     * @stability stable
     */
    grantInvoke(identity) {
        return this.lambda.grantInvoke(identity);
    }
    /**
     * Return the given named metric for this Lambda Return the given named metric for this Function.
     *
     * Return the given named metric for this Lambda Return the given named metric for this Function.
     *
     * @stability stable
     */
    metric(metricName, props) {
        return this.lambda.metric(metricName, { ...props, region: EdgeFunction.EDGE_REGION });
    }
    /**
     * Metric for the Duration of this Lambda How long execution of this Lambda takes.
     *
     * Average over 5 minutes
     * Metric for the Duration of this Lambda How long execution of this Lambda takes.
     *
     * Average over 5 minutes
     *
     * @stability stable
     */
    metricDuration(props) {
        return this.lambda.metricDuration({ ...props, region: EdgeFunction.EDGE_REGION });
    }
    /**
     * How many invocations of this Lambda fail.
     *
     * Sum over 5 minutes
     *
     * @stability stable
     */
    metricErrors(props) {
        return this.lambda.metricErrors({ ...props, region: EdgeFunction.EDGE_REGION });
    }
    /**
     * Metric for the number of invocations of this Lambda How often this Lambda is invoked.
     *
     * Sum over 5 minutes
     * Metric for the number of invocations of this Lambda How often this Lambda is invoked.
     *
     * Sum over 5 minutes
     *
     * @stability stable
     */
    metricInvocations(props) {
        return this.lambda.metricInvocations({ ...props, region: EdgeFunction.EDGE_REGION });
    }
    /**
     * Metric for the number of throttled invocations of this Lambda How often this Lambda is throttled.
     *
     * Sum over 5 minutes
     * Metric for the number of throttled invocations of this Lambda How often this Lambda is throttled.
     *
     * Sum over 5 minutes
     *
     * @stability stable
     */
    metricThrottles(props) {
        return this.lambda.metricThrottles({ ...props, region: EdgeFunction.EDGE_REGION });
    }
    /**
     * Adds an event source to this function.
     *
     * @stability stable
     */
    addEventSource(source) {
        return this.lambda.addEventSource(source);
    }
    /**
     * Configures options for asynchronous invocation.
     *
     * @stability stable
     */
    configureAsyncInvoke(options) {
        return this.lambda.configureAsyncInvoke(options);
    }
    /** Create a function in-region */
    createInRegionFunction(props) {
        const edgeFunction = new lambda.Function(this, 'Fn', props);
        addEdgeLambdaToRoleTrustStatement(edgeFunction.role);
        return { edgeFunction, edgeArn: edgeFunction.currentVersion.edgeArn };
    }
    /** Create a support stack and function in us-east-1, and a SSM reader in-region */
    createCrossRegionFunction(id, props) {
        const parameterNamePrefix = '/cdk/EdgeFunctionArn';
        if (core_1.Token.isUnresolved(this.env.region)) {
            throw new Error('stacks which use EdgeFunctions must have an explicitly set region');
        }
        const parameterName = `${parameterNamePrefix}/${this.env.region}/${this.node.path}`;
        const functionStack = this.edgeStack(props.stackId);
        const edgeFunction = new lambda.Function(functionStack, id, props);
        addEdgeLambdaToRoleTrustStatement(edgeFunction.role);
        // Store the current version's ARN to be retrieved by the cross region reader below.
        const version = edgeFunction.currentVersion;
        new ssm.StringParameter(edgeFunction, 'Parameter', {
            parameterName,
            stringValue: version.edgeArn,
        });
        const edgeArn = this.createCrossRegionArnReader(parameterNamePrefix, parameterName, version);
        return { edgeFunction, edgeArn };
    }
    createCrossRegionArnReader(parameterNamePrefix, parameterName, version) {
        // Prefix of the parameter ARN that applies to all EdgeFunctions.
        // This is necessary because the `CustomResourceProvider` is a singleton, and the `policyStatement`
        // must work for multiple EdgeFunctions.
        const parameterArnPrefix = this.stack.formatArn({
            service: 'ssm',
            region: EdgeFunction.EDGE_REGION,
            resource: 'parameter',
            resourceName: parameterNamePrefix + '/*',
            sep: '',
        });
        const resourceType = 'Custom::CrossRegionStringParameterReader';
        const serviceToken = core_1.CustomResourceProvider.getOrCreate(this, resourceType, {
            codeDirectory: path.join(__dirname, 'edge-function'),
            runtime: core_1.CustomResourceProviderRuntime.NODEJS_12_X,
            policyStatements: [{
                    Effect: 'Allow',
                    Resource: parameterArnPrefix,
                    Action: ['ssm:GetParameter'],
                }],
        });
        const resource = new core_1.CustomResource(this, 'ArnReader', {
            resourceType: resourceType,
            serviceToken,
            properties: {
                Region: EdgeFunction.EDGE_REGION,
                ParameterName: parameterName,
                // This is used to determine when the function has changed, to refresh the ARN from the custom resource.
                //
                // Use the logical id of the function version. Whenever a function version changes, the logical id must be
                // changed for it to take effect - a good candidate for RefreshToken.
                RefreshToken: core_1.Lazy.uncachedString({
                    produce: () => {
                        const cfn = version.node.defaultChild;
                        return this.stack.resolve(cfn.logicalId);
                    },
                }),
            },
        });
        return resource.getAttString('FunctionArn');
    }
    edgeStack(stackId) {
        const stage = core_1.Stage.of(this);
        if (!stage) {
            throw new Error('stacks which use EdgeFunctions must be part of a CDK app or stage');
        }
        const edgeStackId = stackId !== null && stackId !== void 0 ? stackId : `edge-lambda-stack-${this.stack.node.addr}`;
        let edgeStack = stage.node.tryFindChild(edgeStackId);
        if (!edgeStack) {
            edgeStack = new core_1.Stack(stage, edgeStackId, {
                env: {
                    region: EdgeFunction.EDGE_REGION,
                    account: core_1.Stack.of(this).account,
                },
            });
        }
        this.stack.addDependency(edgeStack);
        return edgeStack;
    }
}
exports.EdgeFunction = EdgeFunction;
_a = JSII_RTTI_SYMBOL_1;
EdgeFunction[_a] = { fqn: "@aws-cdk/aws-cloudfront.experimental.EdgeFunction", version: "1.104.0" };
EdgeFunction.EDGE_REGION = 'us-east-1';
function addEdgeLambdaToRoleTrustStatement(role) {
    if (role instanceof iam.Role && role.assumeRolePolicy) {
        const statement = new iam.PolicyStatement();
        const edgeLambdaServicePrincipal = new iam.ServicePrincipal('edgelambda.amazonaws.com');
        statement.addPrincipals(edgeLambdaServicePrincipal);
        statement.addActions(edgeLambdaServicePrincipal.assumeRoleAction);
        role.assumeRolePolicy.addStatements(statement);
    }
}
//# sourceMappingURL=data:application/json;base64,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