"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LinuxUbuntuComponents = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const common_1 = require("../common");
const common_2 = require("./common");
/**
 * Components for Ubuntu Linux that can be used with AWS Image Builder based builders. These cannot be used by {@link CodeBuildImageBuilder}.
 */
class LinuxUbuntuComponents {
    static requiredPackages(scope, id, architecture) {
        let archUrl;
        if (architecture.is(common_1.Architecture.X86_64)) {
            archUrl = 'amd64';
        }
        else if (architecture.is(common_1.Architecture.ARM64)) {
            archUrl = 'arm64';
        }
        else {
            throw new Error(`Unsupported architecture for required packages: ${architecture.name}`);
        }
        return new common_2.ImageBuilderComponent(scope, id, {
            platform: 'Linux',
            displayName: 'Required packages',
            description: 'Install packages required for GitHub Runner and upgrade all packages',
            commands: [
                'set -ex',
                'apt-get update',
                'DEBIAN_FRONTEND=noninteractive apt-get upgrade -y',
                'DEBIAN_FRONTEND=noninteractive apt-get install -y curl sudo jq bash zip unzip iptables software-properties-common ca-certificates',
                `curl -sfLo /tmp/amazon-cloudwatch-agent.deb https://s3.amazonaws.com/amazoncloudwatch-agent/ubuntu/${archUrl}/latest/amazon-cloudwatch-agent.deb`,
                'dpkg -i -E /tmp/amazon-cloudwatch-agent.deb',
                'rm /tmp/amazon-cloudwatch-agent.deb',
            ],
        });
    }
    static runnerUser(scope, id, _architecture) {
        return new common_2.ImageBuilderComponent(scope, id, {
            platform: 'Linux',
            displayName: 'GitHub Runner user',
            description: 'Install latest version of AWS CLI',
            commands: [
                'set -ex',
                'addgroup runner',
                'adduser --system --disabled-password --home /home/runner --ingroup runner runner',
            ],
        });
    }
    static awsCli(scope, id, architecture) {
        let archUrl;
        if (architecture.is(common_1.Architecture.X86_64)) {
            archUrl = 'x86_64';
        }
        else if (architecture.is(common_1.Architecture.ARM64)) {
            archUrl = 'aarch64';
        }
        else {
            throw new Error(`Unsupported architecture for awscli: ${architecture.name}`);
        }
        return new common_2.ImageBuilderComponent(scope, id, {
            platform: 'Linux',
            displayName: 'AWS CLI',
            description: 'Install latest version of AWS CLI',
            commands: [
                'set -ex',
                `curl -fsSL "https://awscli.amazonaws.com/awscli-exe-linux-${archUrl}.zip" -o awscliv2.zip`,
                'unzip -q awscliv2.zip',
                './aws/install',
                'rm -rf awscliv2.zip aws',
            ],
        });
    }
    static githubCli(scope, id, _architecture) {
        return new common_2.ImageBuilderComponent(scope, id, {
            platform: 'Linux',
            displayName: 'GitHub CLI',
            description: 'Install latest version of gh',
            commands: [
                'set -ex',
                'curl -fsSL https://cli.github.com/packages/githubcli-archive-keyring.gpg | sudo dd of=/usr/share/keyrings/githubcli-archive-keyring.gpg',
                'echo "deb [arch=$(dpkg --print-architecture) signed-by=/usr/share/keyrings/githubcli-archive-keyring.gpg] ' +
                    '  https://cli.github.com/packages stable main" | sudo tee /etc/apt/sources.list.d/github-cli.list > /dev/null',
                'apt-get update',
                'DEBIAN_FRONTEND=noninteractive apt-get install -y gh',
            ],
        });
    }
    static git(scope, id, _architecture) {
        return new common_2.ImageBuilderComponent(scope, id, {
            platform: 'Linux',
            displayName: 'Git',
            description: 'Install latest version of git',
            commands: [
                'set -ex',
                'add-apt-repository ppa:git-core/ppa',
                'apt-get update',
                'DEBIAN_FRONTEND=noninteractive apt-get install -y git',
            ],
        });
    }
    static githubRunner(scope, id, runnerVersion, architecture) {
        let versionCommand;
        if (runnerVersion.is(common_1.RunnerVersion.latest())) {
            versionCommand = 'RUNNER_VERSION=`curl -w "%{redirect_url}" -fsS https://github.com/actions/runner/releases/latest | grep -oE "[^/v]+$"`';
        }
        else {
            versionCommand = `RUNNER_VERSION='${runnerVersion.version}'`;
        }
        let archUrl;
        if (architecture.is(common_1.Architecture.X86_64)) {
            archUrl = 'x64';
        }
        else if (architecture.is(common_1.Architecture.ARM64)) {
            archUrl = 'arm64';
        }
        else {
            throw new Error(`Unsupported architecture for GitHub Runner: ${architecture.name}`);
        }
        return new common_2.ImageBuilderComponent(scope, id, {
            platform: 'Linux',
            displayName: 'GitHub Actions Runner',
            description: 'Install latest version of GitHub Actions Runner',
            commands: [
                'set -ex',
                versionCommand,
                `curl -fsSLO "https://github.com/actions/runner/releases/download/v\${RUNNER_VERSION}/actions-runner-linux-${archUrl}-\${RUNNER_VERSION}.tar.gz"`,
                `tar xzf "actions-runner-linux-${archUrl}-\${RUNNER_VERSION}.tar.gz"`,
                `rm actions-runner-linux-${archUrl}-\${RUNNER_VERSION}.tar.gz`,
                './bin/installdependencies.sh',
            ],
        });
    }
    static docker(scope, id, _architecture) {
        return new common_2.ImageBuilderComponent(scope, id, {
            platform: 'Linux',
            displayName: 'Docker',
            description: 'Install latest version of Docker',
            commands: [
                'set -ex',
                'curl -fsSL https://download.docker.com/linux/ubuntu/gpg | sudo gpg --dearmor -o /usr/share/keyrings/docker.gpg',
                'echo ' +
                    '  "deb [arch=$(dpkg --print-architecture) signed-by=/usr/share/keyrings/docker.gpg] https://download.docker.com/linux/ubuntu ' +
                    '  $(lsb_release -cs) stable" | sudo tee /etc/apt/sources.list.d/docker.list > /dev/null',
                'apt-get update',
                'DEBIAN_FRONTEND=noninteractive apt-get install -y docker-ce docker-ce-cli containerd.io docker-compose-plugin',
                'usermod -aG docker runner',
                'ln -s /usr/libexec/docker/cli-plugins/docker-compose /usr/bin/docker-compose',
            ],
        });
    }
    static extraCertificates(scope, id, path) {
        return new common_2.ImageBuilderComponent(scope, id, {
            platform: 'Linux',
            displayName: 'Extra certificates',
            description: 'Install self-signed certificates to provide access to GitHub Enterprise Server',
            commands: [
                'set -ex',
                'cp certs/certs.pem /usr/local/share/ca-certificates/github-enterprise-server.crt',
                'update-ca-certificates',
            ],
            assets: [
                {
                    path: 'certs',
                    asset: new aws_cdk_lib_1.aws_s3_assets.Asset(scope, `${id} Asset`, { path }),
                },
            ],
        });
    }
}
exports.LinuxUbuntuComponents = LinuxUbuntuComponents;
_a = JSII_RTTI_SYMBOL_1;
LinuxUbuntuComponents[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.LinuxUbuntuComponents", version: "0.7.1" };
//# sourceMappingURL=data:application/json;base64,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