"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContainerImageBuilder = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ecr_1 = require("aws-cdk-lib/aws-ecr");
const utils_1 = require("../../utils");
const common_1 = require("../common");
const common_2 = require("./common");
const linux_components_1 = require("./linux-components");
const windows_components_1 = require("./windows-components");
const dockerfileTemplate = `FROM {{{ imagebuilder:parentImage }}}
ENV RUNNER_VERSION=___RUNNER_VERSION___
{{{ imagebuilder:environments }}}
{{{ imagebuilder:components }}}`;
/**
 * Image builder recipe for a Docker container image.
 */
class ContainerRecipe extends common_2.ImageBuilderObjectBase {
    constructor(scope, id, props) {
        super(scope, id);
        const name = common_2.uniqueImageBuilderName(this);
        let components = props.components.map(component => {
            return {
                componentArn: component.arn,
            };
        });
        const recipe = new aws_cdk_lib_1.aws_imagebuilder.CfnContainerRecipe(this, 'Recipe', {
            name: name,
            version: this.version('ContainerRecipe', name, {
                platform: props.platform,
                components,
                dockerfileTemplate,
            }),
            parentImage: props.parentImage ?? 'mcr.microsoft.com/windows/servercore:ltsc2019-amd64',
            components,
            containerType: 'DOCKER',
            targetRepository: {
                service: 'ECR',
                repositoryName: props.targetRepository.repositoryName,
            },
            dockerfileTemplateData: props.dockerfileTemplate,
        });
        this.arn = recipe.attrArn;
        this.name = name;
    }
}
/**
 * An image builder that uses AWS Image Builder to build Docker images pre-baked with all the GitHub Actions runner requirements. Builders can be used with runner providers.
 *
 * The CodeBuild builder is better and faster. Only use this one if you have no choice. For example, if you need Windows containers.
 *
 * Each builder re-runs automatically at a set interval to make sure the images contain the latest versions of everything.
 *
 * You can create an instance of this construct to customize the image used to spin-up runners. Some runner providers may require custom components. Check the runner provider documentation. The default components work with CodeBuild and Fargate.
 *
 * For example, to set a specific runner version, rebuild the image every 2 weeks, and add a few packages for the Fargate provider, use:
 *
 * ```
 * const builder = new ContainerImageBuilder(this, 'Builder', {
 *     runnerVersion: RunnerVersion.specific('2.293.0'),
 *     rebuildInterval: Duration.days(14),
 * });
 * new CodeBuildRunner(this, 'CodeBuild provider', {
 *     label: 'custom-codebuild',
 *     imageBuilder: builder,
 * });
 * ```
 */
class ContainerImageBuilder extends common_2.ImageBuilderBase {
    constructor(scope, id, props) {
        super(scope, id, {
            os: props?.os,
            supportedOs: [common_1.Os.WINDOWS],
            architecture: props?.architecture,
            supportedArchitectures: [common_1.Architecture.X86_64],
            instanceType: props?.instanceType,
            vpc: props?.vpc,
            securityGroups: props?.securityGroup ? [props.securityGroup] : props?.securityGroups,
            subnetSelection: props?.subnetSelection,
            logRemovalPolicy: props?.logRemovalPolicy,
            logRetention: props?.logRetention,
            runnerVersion: props?.runnerVersion,
            rebuildInterval: props?.rebuildInterval,
            imageTypeName: 'image',
        });
        this.parentImage = props?.parentImage;
        // create repository that only keeps one tag
        this.repository = new aws_cdk_lib_1.aws_ecr.Repository(this, 'Repository', {
            imageScanOnPush: true,
            imageTagMutability: aws_ecr_1.TagMutability.MUTABLE,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            lifecycleRules: [
                {
                    description: 'Remove all but the latest image',
                    tagStatus: aws_ecr_1.TagStatus.ANY,
                    maxImageCount: 1,
                },
            ],
        });
        // add all basic components
        this.addBaseWindowsComponents();
    }
    addBaseWindowsComponents() {
        this.addComponent(windows_components_1.WindowsComponents.awsCli(this, 'AWS CLI'));
        this.addComponent(windows_components_1.WindowsComponents.githubCli(this, 'GitHub CLI'));
        this.addComponent(windows_components_1.WindowsComponents.git(this, 'git'));
        this.addComponent(windows_components_1.WindowsComponents.githubRunner(this, 'GitHub Actions Runner', this.runnerVersion));
    }
    /**
     * Add a component to be installed before any other components. Useful for required system settings like certificates or proxy settings.
     * @param component
     */
    prependComponent(component) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        if (component.platform != this.platform) {
            throw new Error('Component platform doesn\'t match builder platform');
        }
        this.components = [component].concat(this.components);
    }
    /**
     * Add a component to be installed.
     * @param component
     */
    addComponent(component) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        if (component.platform != this.platform) {
            throw new Error('Component platform doesn\'t match builder platform');
        }
        this.components.push(component);
    }
    /**
     * Add extra trusted certificates. This helps deal with self-signed certificates for GitHub Enterprise Server.
     *
     * All first party Dockerfiles support this. Others may not.
     *
     * @param path path to directory containing a file called certs.pem containing all the required certificates
     */
    addExtraCertificates(path) {
        if (this.platform == 'Linux') {
            this.prependComponent(linux_components_1.LinuxUbuntuComponents.extraCertificates(this, 'Extra Certs', path));
        }
        else if (this.platform == 'Windows') {
            this.prependComponent(windows_components_1.WindowsComponents.extraCertificates(this, 'Extra Certs', path));
        }
        else {
            throw new Error(`Unknown platform: ${this.platform}`);
        }
    }
    /**
     * Called by IRunnerProvider to finalize settings and create the image builder.
     */
    bind() {
        if (this.boundImage) {
            return this.boundImage;
        }
        const dist = new aws_cdk_lib_1.aws_imagebuilder.CfnDistributionConfiguration(this, 'Distribution', {
            name: common_2.uniqueImageBuilderName(this),
            description: this.description,
            distributions: [
                {
                    region: aws_cdk_lib_1.Stack.of(this).region,
                    containerDistributionConfiguration: {
                        ContainerTags: ['latest'],
                        TargetRepository: {
                            Service: 'ECR',
                            RepositoryName: this.repository.repositoryName,
                        },
                    },
                },
            ],
        });
        const recipe = new ContainerRecipe(this, 'Container Recipe', {
            platform: this.platform,
            components: this.components,
            targetRepository: this.repository,
            dockerfileTemplate: dockerfileTemplate.replace('___RUNNER_VERSION___', this.runnerVersion.version),
            parentImage: this.parentImage,
        });
        const log = this.createLog(recipe.name);
        const infra = this.createInfrastructure([
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilderECRContainerBuilds'),
        ]);
        const image = this.createImage(infra, dist, log, undefined, recipe.arn);
        this.createPipeline(infra, dist, log, undefined, recipe.arn);
        this.imageCleaner(image, recipe.name);
        this.boundImage = {
            // There are simpler ways to get the ARN, but we want an image object that depends on the newly built image.
            // We want whoever is using this image to automatically wait for Image Builder to finish building before using the image.
            imageRepository: aws_cdk_lib_1.aws_ecr.Repository.fromRepositoryName(this, 'Dependable Image', 
            // we can't use image.attrName because it comes up with upper case
            cdk.Fn.split(':', cdk.Fn.split('/', image.attrImageUri, 2)[1], 2)[0]),
            imageTag: 'latest',
            os: this.os,
            architecture: this.architecture,
            logGroup: log,
            runnerVersion: this.runnerVersion,
        };
        return this.boundImage;
    }
    imageCleaner(image, recipeName) {
        const crHandler = utils_1.BundledNodejsFunction.singleton(this, 'build-image', {
            description: 'Custom resource handler that triggers CodeBuild to build runner images, and cleans-up images on deletion',
            timeout: cdk.Duration.minutes(3),
        });
        const policy = new aws_cdk_lib_1.aws_iam.Policy(this, 'CR Policy', {
            statements: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ecr:BatchDeleteImage', 'ecr:ListImages'],
                    resources: [this.repository.repositoryArn],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['imagebuilder:ListImages', 'imagebuilder:ListImageBuildVersions', 'imagebuilder:DeleteImage'],
                    resources: ['*'],
                }),
            ],
        });
        crHandler.role?.attachInlinePolicy(policy);
        const cr = new aws_cdk_lib_1.CustomResource(this, 'Deleter', {
            serviceToken: crHandler.functionArn,
            resourceType: 'Custom::ImageDeleter',
            properties: {
                RepoName: this.repository.repositoryName,
                ImageBuilderName: recipeName,
                DeleteOnly: true,
            },
        });
        // add dependencies to make sure resources are there when we need them
        cr.node.addDependency(image);
        cr.node.addDependency(policy);
        cr.node.addDependency(crHandler);
        return cr;
    }
}
exports.ContainerImageBuilder = ContainerImageBuilder;
_a = JSII_RTTI_SYMBOL_1;
ContainerImageBuilder[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.ContainerImageBuilder", version: "0.7.1" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29udGFpbmVyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL3Byb3ZpZGVycy9pbWFnZS1idWlsZGVycy9jb250YWluZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxtQ0FBbUM7QUFDbkMsNkNBVXFCO0FBQ3JCLGlEQUErRDtBQUUvRCx1Q0FBb0Q7QUFDcEQsc0NBQXdGO0FBQ3hGLHFDQUFtSDtBQUNuSCx5REFBMkQ7QUFDM0QsNkRBQXlEO0FBRXpELE1BQU0sa0JBQWtCLEdBQUc7OztnQ0FHSyxDQUFDO0FBMklqQzs7R0FFRztBQUNILE1BQU0sZUFBZ0IsU0FBUSwrQkFBc0I7SUFJbEQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFnQztRQUN4RSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLE1BQU0sSUFBSSxHQUFHLCtCQUFzQixDQUFDLElBQUksQ0FBQyxDQUFDO1FBRTFDLElBQUksVUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLFNBQVMsQ0FBQyxFQUFFO1lBQ2hELE9BQU87Z0JBQ0wsWUFBWSxFQUFFLFNBQVMsQ0FBQyxHQUFHO2FBQzVCLENBQUM7UUFDSixDQUFDLENBQUMsQ0FBQztRQUVILE1BQU0sTUFBTSxHQUFHLElBQUksOEJBQVksQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsUUFBUSxFQUFFO1lBQ2pFLElBQUksRUFBRSxJQUFJO1lBQ1YsT0FBTyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsaUJBQWlCLEVBQUUsSUFBSSxFQUFFO2dCQUM3QyxRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVE7Z0JBQ3hCLFVBQVU7Z0JBQ1Ysa0JBQWtCO2FBQ25CLENBQUM7WUFDRixXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVcsSUFBSSxxREFBcUQ7WUFDdkYsVUFBVTtZQUNWLGFBQWEsRUFBRSxRQUFRO1lBQ3ZCLGdCQUFnQixFQUFFO2dCQUNoQixPQUFPLEVBQUUsS0FBSztnQkFDZCxjQUFjLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGNBQWM7YUFDdEQ7WUFDRCxzQkFBc0IsRUFBRSxLQUFLLENBQUMsa0JBQWtCO1NBQ2pELENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxHQUFHLEdBQUcsTUFBTSxDQUFDLE9BQU8sQ0FBQztRQUMxQixJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksQ0FBQztJQUNuQixDQUFDO0NBQ0Y7QUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0dBcUJHO0FBQ0gsTUFBYSxxQkFBc0IsU0FBUSx5QkFBZ0I7SUFLekQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFrQztRQUMxRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNmLEVBQUUsRUFBRSxLQUFLLEVBQUUsRUFBRTtZQUNiLFdBQVcsRUFBRSxDQUFDLFdBQUUsQ0FBQyxPQUFPLENBQUM7WUFDekIsWUFBWSxFQUFFLEtBQUssRUFBRSxZQUFZO1lBQ2pDLHNCQUFzQixFQUFFLENBQUMscUJBQVksQ0FBQyxNQUFNLENBQUM7WUFDN0MsWUFBWSxFQUFFLEtBQUssRUFBRSxZQUFZO1lBQ2pDLEdBQUcsRUFBRSxLQUFLLEVBQUUsR0FBRztZQUNmLGNBQWMsRUFBRSxLQUFLLEVBQUUsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxFQUFFLGNBQWM7WUFDcEYsZUFBZSxFQUFFLEtBQUssRUFBRSxlQUFlO1lBQ3ZDLGdCQUFnQixFQUFFLEtBQUssRUFBRSxnQkFBZ0I7WUFDekMsWUFBWSxFQUFFLEtBQUssRUFBRSxZQUFZO1lBQ2pDLGFBQWEsRUFBRSxLQUFLLEVBQUUsYUFBYTtZQUNuQyxlQUFlLEVBQUUsS0FBSyxFQUFFLGVBQWU7WUFDdkMsYUFBYSxFQUFFLE9BQU87U0FDdkIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFdBQVcsR0FBRyxLQUFLLEVBQUUsV0FBVyxDQUFDO1FBRXRDLDRDQUE0QztRQUM1QyxJQUFJLENBQUMsVUFBVSxHQUFHLElBQUkscUJBQUcsQ0FBQyxVQUFVLENBQUMsSUFBSSxFQUFFLFlBQVksRUFBRTtZQUN2RCxlQUFlLEVBQUUsSUFBSTtZQUNyQixrQkFBa0IsRUFBRSx1QkFBYSxDQUFDLE9BQU87WUFDekMsYUFBYSxFQUFFLDJCQUFhLENBQUMsT0FBTztZQUNwQyxjQUFjLEVBQUU7Z0JBQ2Q7b0JBQ0UsV0FBVyxFQUFFLGlDQUFpQztvQkFDOUMsU0FBUyxFQUFFLG1CQUFTLENBQUMsR0FBRztvQkFDeEIsYUFBYSxFQUFFLENBQUM7aUJBQ2pCO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCwyQkFBMkI7UUFDM0IsSUFBSSxDQUFDLHdCQUF3QixFQUFFLENBQUM7SUFDbEMsQ0FBQztJQUVPLHdCQUF3QjtRQUM5QixJQUFJLENBQUMsWUFBWSxDQUFDLHNDQUFpQixDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUsU0FBUyxDQUFDLENBQUMsQ0FBQztRQUM3RCxJQUFJLENBQUMsWUFBWSxDQUFDLHNDQUFpQixDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsWUFBWSxDQUFDLENBQUMsQ0FBQztRQUNuRSxJQUFJLENBQUMsWUFBWSxDQUFDLHNDQUFpQixDQUFDLEdBQUcsQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLENBQUMsQ0FBQztRQUN0RCxJQUFJLENBQUMsWUFBWSxDQUFDLHNDQUFpQixDQUFDLFlBQVksQ0FBQyxJQUFJLEVBQUUsdUJBQXVCLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUM7SUFDdkcsQ0FBQztJQUVEOzs7T0FHRztJQUNILGdCQUFnQixDQUFDLFNBQWdDO1FBQy9DLElBQUksSUFBSSxDQUFDLFVBQVUsRUFBRTtZQUNuQixNQUFNLElBQUksS0FBSyxDQUFDLDBGQUEwRixDQUFDLENBQUM7U0FDN0c7UUFDRCxJQUFJLFNBQVMsQ0FBQyxRQUFRLElBQUksSUFBSSxDQUFDLFFBQVEsRUFBRTtZQUN2QyxNQUFNLElBQUksS0FBSyxDQUFDLG9EQUFvRCxDQUFDLENBQUM7U0FDdkU7UUFDRCxJQUFJLENBQUMsVUFBVSxHQUFHLENBQUMsU0FBUyxDQUFDLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUN4RCxDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsWUFBWSxDQUFDLFNBQWdDO1FBQzNDLElBQUksSUFBSSxDQUFDLFVBQVUsRUFBRTtZQUNuQixNQUFNLElBQUksS0FBSyxDQUFDLDBGQUEwRixDQUFDLENBQUM7U0FDN0c7UUFDRCxJQUFJLFNBQVMsQ0FBQyxRQUFRLElBQUksSUFBSSxDQUFDLFFBQVEsRUFBRTtZQUN2QyxNQUFNLElBQUksS0FBSyxDQUFDLG9EQUFvRCxDQUFDLENBQUM7U0FDdkU7UUFDRCxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CLENBQUMsSUFBWTtRQUN0QyxJQUFJLElBQUksQ0FBQyxRQUFRLElBQUksT0FBTyxFQUFFO1lBQzVCLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyx3Q0FBcUIsQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUM7U0FDM0Y7YUFBTSxJQUFJLElBQUksQ0FBQyxRQUFRLElBQUksU0FBUyxFQUFFO1lBQ3JDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxzQ0FBaUIsQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUM7U0FDdkY7YUFBTTtZQUNMLE1BQU0sSUFBSSxLQUFLLENBQUMscUJBQXFCLElBQUksQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO1NBQ3ZEO0lBQ0gsQ0FBQztJQUVEOztPQUVHO0lBQ0gsSUFBSTtRQUNGLElBQUksSUFBSSxDQUFDLFVBQVUsRUFBRTtZQUNuQixPQUFPLElBQUksQ0FBQyxVQUFVLENBQUM7U0FDeEI7UUFFRCxNQUFNLElBQUksR0FBRyxJQUFJLDhCQUFZLENBQUMsNEJBQTRCLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtZQUMvRSxJQUFJLEVBQUUsK0JBQXNCLENBQUMsSUFBSSxDQUFDO1lBQ2xDLFdBQVcsRUFBRSxJQUFJLENBQUMsV0FBVztZQUM3QixhQUFhLEVBQUU7Z0JBQ2I7b0JBQ0UsTUFBTSxFQUFFLG1CQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU07b0JBQzdCLGtDQUFrQyxFQUFFO3dCQUNsQyxhQUFhLEVBQUUsQ0FBQyxRQUFRLENBQUM7d0JBQ3pCLGdCQUFnQixFQUFFOzRCQUNoQixPQUFPLEVBQUUsS0FBSzs0QkFDZCxjQUFjLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxjQUFjO3lCQUMvQztxQkFDRjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxNQUFNLEdBQUcsSUFBSSxlQUFlLENBQUMsSUFBSSxFQUFFLGtCQUFrQixFQUFFO1lBQzNELFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUTtZQUN2QixVQUFVLEVBQUUsSUFBSSxDQUFDLFVBQVU7WUFDM0IsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLFVBQVU7WUFDakMsa0JBQWtCLEVBQUUsa0JBQWtCLENBQUMsT0FBTyxDQUFDLHNCQUFzQixFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsT0FBTyxDQUFDO1lBQ2xHLFdBQVcsRUFBRSxJQUFJLENBQUMsV0FBVztTQUM5QixDQUFDLENBQUM7UUFFSCxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN4QyxNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsb0JBQW9CLENBQUM7WUFDdEMscUJBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMsOEJBQThCLENBQUM7WUFDMUUscUJBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMscURBQXFELENBQUM7U0FDbEcsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFLEdBQUcsRUFBRSxTQUFTLEVBQUUsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ3hFLElBQUksQ0FBQyxjQUFjLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRSxHQUFHLEVBQUUsU0FBUyxFQUFFLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUU3RCxJQUFJLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFdEMsSUFBSSxDQUFDLFVBQVUsR0FBRztZQUNoQiw0R0FBNEc7WUFDNUcseUhBQXlIO1lBQ3pILGVBQWUsRUFBRSxxQkFBRyxDQUFDLFVBQVUsQ0FBQyxrQkFBa0IsQ0FDaEQsSUFBSSxFQUFFLGtCQUFrQjtZQUN4QixrRUFBa0U7WUFDbEUsR0FBRyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsR0FBRyxFQUFFLEdBQUcsQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsWUFBWSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUNyRTtZQUNELFFBQVEsRUFBRSxRQUFRO1lBQ2xCLEVBQUUsRUFBRSxJQUFJLENBQUMsRUFBRTtZQUNYLFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWTtZQUMvQixRQUFRLEVBQUUsR0FBRztZQUNiLGFBQWEsRUFBRSxJQUFJLENBQUMsYUFBYTtTQUNsQyxDQUFDO1FBRUYsT0FBTyxJQUFJLENBQUMsVUFBVSxDQUFDO0lBQ3pCLENBQUM7SUFFTyxZQUFZLENBQUMsS0FBNEIsRUFBRSxVQUFrQjtRQUNuRSxNQUFNLFNBQVMsR0FBRyw2QkFBcUIsQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRTtZQUNyRSxXQUFXLEVBQUUsMEdBQTBHO1lBQ3ZILE9BQU8sRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7U0FDakMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxNQUFNLEdBQUcsSUFBSSxxQkFBRyxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUsV0FBVyxFQUFFO1lBQy9DLFVBQVUsRUFBRTtnQkFDVixJQUFJLHFCQUFHLENBQUMsZUFBZSxDQUFDO29CQUN0QixPQUFPLEVBQUUsQ0FBQyxzQkFBc0IsRUFBRSxnQkFBZ0IsQ0FBQztvQkFDbkQsU0FBUyxFQUFFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUM7aUJBQzNDLENBQUM7Z0JBQ0YsSUFBSSxxQkFBRyxDQUFDLGVBQWUsQ0FBQztvQkFDdEIsT0FBTyxFQUFFLENBQUMseUJBQXlCLEVBQUUscUNBQXFDLEVBQUUsMEJBQTBCLENBQUM7b0JBQ3ZHLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztpQkFDakIsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsU0FBUyxDQUFDLElBQUksRUFBRSxrQkFBa0IsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUUzQyxNQUFNLEVBQUUsR0FBRyxJQUFJLDRCQUFjLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUM3QyxZQUFZLEVBQUUsU0FBUyxDQUFDLFdBQVc7WUFDbkMsWUFBWSxFQUFFLHNCQUFzQjtZQUNwQyxVQUFVLEVBQUU7Z0JBQ1YsUUFBUSxFQUFFLElBQUksQ0FBQyxVQUFVLENBQUMsY0FBYztnQkFDeEMsZ0JBQWdCLEVBQUUsVUFBVTtnQkFDNUIsVUFBVSxFQUFFLElBQUk7YUFDakI7U0FDRixDQUFDLENBQUM7UUFFSCxzRUFBc0U7UUFDdEUsRUFBRSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDN0IsRUFBRSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDOUIsRUFBRSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUyxDQUFDLENBQUM7UUFFakMsT0FBTyxFQUFFLENBQUM7SUFDWixDQUFDOztBQS9MSCxzREFnTUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjZGsgZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHtcbiAgYXdzX2VjMiBhcyBlYzIsXG4gIGF3c19lY3IgYXMgZWNyLFxuICBhd3NfaWFtIGFzIGlhbSxcbiAgYXdzX2ltYWdlYnVpbGRlciBhcyBpbWFnZWJ1aWxkZXIsXG4gIGF3c19sb2dzIGFzIGxvZ3MsXG4gIEN1c3RvbVJlc291cmNlLFxuICBEdXJhdGlvbixcbiAgUmVtb3ZhbFBvbGljeSxcbiAgU3RhY2ssXG59IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7IFRhZ011dGFiaWxpdHksIFRhZ1N0YXR1cyB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1lY3InO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBCdW5kbGVkTm9kZWpzRnVuY3Rpb24gfSBmcm9tICcuLi8uLi91dGlscyc7XG5pbXBvcnQgeyBBcmNoaXRlY3R1cmUsIElJbWFnZUJ1aWxkZXIsIE9zLCBSdW5uZXJJbWFnZSwgUnVubmVyVmVyc2lvbiB9IGZyb20gJy4uL2NvbW1vbic7XG5pbXBvcnQgeyBJbWFnZUJ1aWxkZXJCYXNlLCBJbWFnZUJ1aWxkZXJDb21wb25lbnQsIEltYWdlQnVpbGRlck9iamVjdEJhc2UsIHVuaXF1ZUltYWdlQnVpbGRlck5hbWUgfSBmcm9tICcuL2NvbW1vbic7XG5pbXBvcnQgeyBMaW51eFVidW50dUNvbXBvbmVudHMgfSBmcm9tICcuL2xpbnV4LWNvbXBvbmVudHMnO1xuaW1wb3J0IHsgV2luZG93c0NvbXBvbmVudHMgfSBmcm9tICcuL3dpbmRvd3MtY29tcG9uZW50cyc7XG5cbmNvbnN0IGRvY2tlcmZpbGVUZW1wbGF0ZSA9IGBGUk9NIHt7eyBpbWFnZWJ1aWxkZXI6cGFyZW50SW1hZ2UgfX19XG5FTlYgUlVOTkVSX1ZFUlNJT049X19fUlVOTkVSX1ZFUlNJT05fX19cbnt7eyBpbWFnZWJ1aWxkZXI6ZW52aXJvbm1lbnRzIH19fVxue3t7IGltYWdlYnVpbGRlcjpjb21wb25lbnRzIH19fWA7XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgQ29udGFpbmVySW1hZ2VCdWlsZGVyIGNvbnN0cnVjdC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDb250YWluZXJJbWFnZUJ1aWxkZXJQcm9wcyB7XG4gIC8qKlxuICAgKiBJbWFnZSBhcmNoaXRlY3R1cmUuXG4gICAqXG4gICAqIEBkZWZhdWx0IEFyY2hpdGVjdHVyZS5YODZfNjRcbiAgICovXG4gIHJlYWRvbmx5IGFyY2hpdGVjdHVyZT86IEFyY2hpdGVjdHVyZTtcblxuICAvKipcbiAgICogSW1hZ2UgT1MuXG4gICAqXG4gICAqIEBkZWZhdWx0IE9TLkxJTlVYXG4gICAqL1xuICByZWFkb25seSBvcz86IE9zO1xuXG4gIC8qKlxuICAgKiBQYXJlbnQgaW1hZ2UgZm9yIHRoZSBuZXcgRG9ja2VyIEltYWdlLiBZb3UgY2FuIHVzZSBlaXRoZXIgSW1hZ2UgQnVpbGRlciBpbWFnZSBBUk4gb3IgcHVibGljIHJlZ2lzdHJ5IGltYWdlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAnbWNyLm1pY3Jvc29mdC5jb20vd2luZG93cy9zZXJ2ZXJjb3JlOmx0c2MyMDE5LWFtZDY0J1xuICAgKi9cbiAgcmVhZG9ubHkgcGFyZW50SW1hZ2U/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFZlcnNpb24gb2YgR2l0SHViIFJ1bm5lcnMgdG8gaW5zdGFsbC5cbiAgICpcbiAgICogQGRlZmF1bHQgbGF0ZXN0IHZlcnNpb24gYXZhaWxhYmxlXG4gICAqL1xuICByZWFkb25seSBydW5uZXJWZXJzaW9uPzogUnVubmVyVmVyc2lvbjtcblxuICAvKipcbiAgICogU2NoZWR1bGUgdGhlIGltYWdlIHRvIGJlIHJlYnVpbHQgZXZlcnkgZ2l2ZW4gaW50ZXJ2YWwuIFVzZWZ1bCBmb3Iga2VlcGluZyB0aGUgaW1hZ2UgdXAtZG8tZGF0ZSB3aXRoIHRoZSBsYXRlc3QgR2l0SHViIHJ1bm5lciB2ZXJzaW9uIGFuZCBsYXRlc3QgT1MgdXBkYXRlcy5cbiAgICpcbiAgICogU2V0IHRvIHplcm8gdG8gZGlzYWJsZS5cbiAgICpcbiAgICogQGRlZmF1bHQgRHVyYXRpb24uZGF5cyg3KVxuICAgKi9cbiAgcmVhZG9ubHkgcmVidWlsZEludGVydmFsPzogRHVyYXRpb247XG5cbiAgLyoqXG4gICAqIFZQQyB0byBsYXVuY2ggdGhlIHJ1bm5lcnMgaW4uXG4gICAqXG4gICAqIEBkZWZhdWx0IGRlZmF1bHQgYWNjb3VudCBWUENcbiAgICovXG4gIHJlYWRvbmx5IHZwYz86IGVjMi5JVnBjO1xuXG4gIC8qKlxuICAgKiBTZWN1cml0eSBncm91cCB0byBhc3NpZ24gdG8gbGF1bmNoZWQgYnVpbGRlciBpbnN0YW5jZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IG5ldyBzZWN1cml0eSBncm91cFxuICAgKlxuICAgKiBAZGVwcmVjYXRlZCB1c2Uge0BsaW5rIHNlY3VyaXR5R3JvdXBzfVxuICAgKi9cbiAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cD86IGVjMi5JU2VjdXJpdHlHcm91cDtcblxuICAvKipcbiAgICogU2VjdXJpdHkgZ3JvdXBzIHRvIGFzc2lnbiB0byBsYXVuY2hlZCBidWlsZGVyIGluc3RhbmNlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgbmV3IHNlY3VyaXR5IGdyb3VwXG4gICAqL1xuICByZWFkb25seSBzZWN1cml0eUdyb3Vwcz86IGVjMi5JU2VjdXJpdHlHcm91cFtdO1xuXG4gIC8qKlxuICAgKiBXaGVyZSB0byBwbGFjZSB0aGUgbmV0d29yayBpbnRlcmZhY2VzIHdpdGhpbiB0aGUgVlBDLlxuICAgKlxuICAgKiBAZGVmYXVsdCBkZWZhdWx0IFZQQyBzdWJuZXRcbiAgICovXG4gIHJlYWRvbmx5IHN1Ym5ldFNlbGVjdGlvbj86IGVjMi5TdWJuZXRTZWxlY3Rpb247XG5cbiAgLyoqXG4gICAqIFRoZSBpbnN0YW5jZSB0eXBlIHVzZWQgdG8gYnVpbGQgdGhlIGltYWdlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBtNS5sYXJnZVxuICAgKi9cbiAgcmVhZG9ubHkgaW5zdGFuY2VUeXBlPzogZWMyLkluc3RhbmNlVHlwZTtcblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBkYXlzIGxvZyBldmVudHMgYXJlIGtlcHQgaW4gQ2xvdWRXYXRjaCBMb2dzLiBXaGVuIHVwZGF0aW5nXG4gICAqIHRoaXMgcHJvcGVydHksIHVuc2V0dGluZyBpdCBkb2Vzbid0IHJlbW92ZSB0aGUgbG9nIHJldGVudGlvbiBwb2xpY3kuIFRvXG4gICAqIHJlbW92ZSB0aGUgcmV0ZW50aW9uIHBvbGljeSwgc2V0IHRoZSB2YWx1ZSB0byBgSU5GSU5JVEVgLlxuICAgKlxuICAgKiBAZGVmYXVsdCBsb2dzLlJldGVudGlvbkRheXMuT05FX01PTlRIXG4gICAqL1xuICByZWFkb25seSBsb2dSZXRlbnRpb24/OiBsb2dzLlJldGVudGlvbkRheXM7XG5cbiAgLyoqXG4gICAqIFJlbW92YWwgcG9saWN5IGZvciBsb2dzIG9mIGltYWdlIGJ1aWxkcy4gSWYgZGVwbG95bWVudCBmYWlscyBvbiB0aGUgY3VzdG9tIHJlc291cmNlLCB0cnkgc2V0dGluZyB0aGlzIHRvIGBSZW1vdmFsUG9saWN5LlJFVEFJTmAuIFRoaXMgd2F5IHRoZSBDb2RlQnVpbGQgbG9ncyBjYW4gc3RpbGwgYmUgdmlld2VkLCBhbmQgeW91IGNhbiBzZWUgd2h5IHRoZSBidWlsZCBmYWlsZWQuXG4gICAqXG4gICAqIFdlIHRyeSB0byBub3QgbGVhdmUgYW55dGhpbmcgYmVoaW5kIHdoZW4gcmVtb3ZlZC4gQnV0IHNvbWV0aW1lcyBhIGxvZyBzdGF5aW5nIGJlaGluZCBpcyB1c2VmdWwuXG4gICAqXG4gICAqIEBkZWZhdWx0IFJlbW92YWxQb2xpY3kuREVTVFJPWVxuICAgKi9cbiAgcmVhZG9ubHkgbG9nUmVtb3ZhbFBvbGljeT86IFJlbW92YWxQb2xpY3k7XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgQ29udGFpbmVyUmVjaXBlIGNvbnN0cnVjdC5cbiAqL1xuaW50ZXJmYWNlIENvbnRhaW5lclJlY2lwZVByb3BlcnRpZXMge1xuICAvKipcbiAgICogVGFyZ2V0IHBsYXRmb3JtLiBNdXN0IG1hdGNoIGJ1aWxkZXIgcGxhdGZvcm0uXG4gICAqL1xuICByZWFkb25seSBwbGF0Zm9ybTogJ0xpbnV4JyB8ICdXaW5kb3dzJztcblxuICAvKipcbiAgICogQ29tcG9uZW50cyB0byBhZGQgdG8gdGFyZ2V0IGNvbnRhaW5lciBpbWFnZS5cbiAgICovXG4gIHJlYWRvbmx5IGNvbXBvbmVudHM6IEltYWdlQnVpbGRlckNvbXBvbmVudFtdO1xuXG4gIC8qKlxuICAgKiBFQ1IgcmVwb3NpdG9yeSB3aGVyZSByZXN1bHRpbmcgY29udGFpbmVyIGltYWdlIHdpbGwgYmUgdXBsb2FkZWQuXG4gICAqL1xuICByZWFkb25seSB0YXJnZXRSZXBvc2l0b3J5OiBlY3IuSVJlcG9zaXRvcnk7XG5cbiAgLyoqXG4gICAqIERvY2tlcmZpbGUgdGVtcGxhdGUgd2hlcmUgYWxsIHRoZSBjb21wb25lbnRzIHdpbGwgYmUgYWRkZWQuXG4gICAqXG4gICAqIE11c3QgY29udGFpbiBhdCBsZWFzdCB0aGUgZm9sbG93aW5nIHBsYWNlaG9sZGVyczpcbiAgICpcbiAgICogYGBgXG4gICAqIEZST00ge3t7IGltYWdlYnVpbGRlcjpwYXJlbnRJbWFnZSB9fX1cbiAgICoge3t7IGltYWdlYnVpbGRlcjplbnZpcm9ubWVudHMgfX19XG4gICAqIHt7eyBpbWFnZWJ1aWxkZXI6Y29tcG9uZW50cyB9fX1cbiAgICogYGBgXG4gICAqL1xuICByZWFkb25seSBkb2NrZXJmaWxlVGVtcGxhdGU6IHN0cmluZztcblxuICAvKipcbiAgICogUGFyZW50IGltYWdlIGZvciB0aGUgbmV3IERvY2tlciBJbWFnZS5cbiAgICpcbiAgICogQGRlZmF1bHQgJ21jci5taWNyb3NvZnQuY29tL3dpbmRvd3Mvc2VydmVyY29yZTpsdHNjMjAxOS1hbWQ2NCdcbiAgICovXG4gIHJlYWRvbmx5IHBhcmVudEltYWdlPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIEltYWdlIGJ1aWxkZXIgcmVjaXBlIGZvciBhIERvY2tlciBjb250YWluZXIgaW1hZ2UuXG4gKi9cbmNsYXNzIENvbnRhaW5lclJlY2lwZSBleHRlbmRzIEltYWdlQnVpbGRlck9iamVjdEJhc2Uge1xuICBwdWJsaWMgcmVhZG9ubHkgYXJuOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBuYW1lOiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IENvbnRhaW5lclJlY2lwZVByb3BlcnRpZXMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgY29uc3QgbmFtZSA9IHVuaXF1ZUltYWdlQnVpbGRlck5hbWUodGhpcyk7XG5cbiAgICBsZXQgY29tcG9uZW50cyA9IHByb3BzLmNvbXBvbmVudHMubWFwKGNvbXBvbmVudCA9PiB7XG4gICAgICByZXR1cm4ge1xuICAgICAgICBjb21wb25lbnRBcm46IGNvbXBvbmVudC5hcm4sXG4gICAgICB9O1xuICAgIH0pO1xuXG4gICAgY29uc3QgcmVjaXBlID0gbmV3IGltYWdlYnVpbGRlci5DZm5Db250YWluZXJSZWNpcGUodGhpcywgJ1JlY2lwZScsIHtcbiAgICAgIG5hbWU6IG5hbWUsXG4gICAgICB2ZXJzaW9uOiB0aGlzLnZlcnNpb24oJ0NvbnRhaW5lclJlY2lwZScsIG5hbWUsIHtcbiAgICAgICAgcGxhdGZvcm06IHByb3BzLnBsYXRmb3JtLFxuICAgICAgICBjb21wb25lbnRzLFxuICAgICAgICBkb2NrZXJmaWxlVGVtcGxhdGUsXG4gICAgICB9KSxcbiAgICAgIHBhcmVudEltYWdlOiBwcm9wcy5wYXJlbnRJbWFnZSA/PyAnbWNyLm1pY3Jvc29mdC5jb20vd2luZG93cy9zZXJ2ZXJjb3JlOmx0c2MyMDE5LWFtZDY0JyxcbiAgICAgIGNvbXBvbmVudHMsXG4gICAgICBjb250YWluZXJUeXBlOiAnRE9DS0VSJyxcbiAgICAgIHRhcmdldFJlcG9zaXRvcnk6IHtcbiAgICAgICAgc2VydmljZTogJ0VDUicsXG4gICAgICAgIHJlcG9zaXRvcnlOYW1lOiBwcm9wcy50YXJnZXRSZXBvc2l0b3J5LnJlcG9zaXRvcnlOYW1lLFxuICAgICAgfSxcbiAgICAgIGRvY2tlcmZpbGVUZW1wbGF0ZURhdGE6IHByb3BzLmRvY2tlcmZpbGVUZW1wbGF0ZSxcbiAgICB9KTtcblxuICAgIHRoaXMuYXJuID0gcmVjaXBlLmF0dHJBcm47XG4gICAgdGhpcy5uYW1lID0gbmFtZTtcbiAgfVxufVxuXG4vKipcbiAqIEFuIGltYWdlIGJ1aWxkZXIgdGhhdCB1c2VzIEFXUyBJbWFnZSBCdWlsZGVyIHRvIGJ1aWxkIERvY2tlciBpbWFnZXMgcHJlLWJha2VkIHdpdGggYWxsIHRoZSBHaXRIdWIgQWN0aW9ucyBydW5uZXIgcmVxdWlyZW1lbnRzLiBCdWlsZGVycyBjYW4gYmUgdXNlZCB3aXRoIHJ1bm5lciBwcm92aWRlcnMuXG4gKlxuICogVGhlIENvZGVCdWlsZCBidWlsZGVyIGlzIGJldHRlciBhbmQgZmFzdGVyLiBPbmx5IHVzZSB0aGlzIG9uZSBpZiB5b3UgaGF2ZSBubyBjaG9pY2UuIEZvciBleGFtcGxlLCBpZiB5b3UgbmVlZCBXaW5kb3dzIGNvbnRhaW5lcnMuXG4gKlxuICogRWFjaCBidWlsZGVyIHJlLXJ1bnMgYXV0b21hdGljYWxseSBhdCBhIHNldCBpbnRlcnZhbCB0byBtYWtlIHN1cmUgdGhlIGltYWdlcyBjb250YWluIHRoZSBsYXRlc3QgdmVyc2lvbnMgb2YgZXZlcnl0aGluZy5cbiAqXG4gKiBZb3UgY2FuIGNyZWF0ZSBhbiBpbnN0YW5jZSBvZiB0aGlzIGNvbnN0cnVjdCB0byBjdXN0b21pemUgdGhlIGltYWdlIHVzZWQgdG8gc3Bpbi11cCBydW5uZXJzLiBTb21lIHJ1bm5lciBwcm92aWRlcnMgbWF5IHJlcXVpcmUgY3VzdG9tIGNvbXBvbmVudHMuIENoZWNrIHRoZSBydW5uZXIgcHJvdmlkZXIgZG9jdW1lbnRhdGlvbi4gVGhlIGRlZmF1bHQgY29tcG9uZW50cyB3b3JrIHdpdGggQ29kZUJ1aWxkIGFuZCBGYXJnYXRlLlxuICpcbiAqIEZvciBleGFtcGxlLCB0byBzZXQgYSBzcGVjaWZpYyBydW5uZXIgdmVyc2lvbiwgcmVidWlsZCB0aGUgaW1hZ2UgZXZlcnkgMiB3ZWVrcywgYW5kIGFkZCBhIGZldyBwYWNrYWdlcyBmb3IgdGhlIEZhcmdhdGUgcHJvdmlkZXIsIHVzZTpcbiAqXG4gKiBgYGBcbiAqIGNvbnN0IGJ1aWxkZXIgPSBuZXcgQ29udGFpbmVySW1hZ2VCdWlsZGVyKHRoaXMsICdCdWlsZGVyJywge1xuICogICAgIHJ1bm5lclZlcnNpb246IFJ1bm5lclZlcnNpb24uc3BlY2lmaWMoJzIuMjkzLjAnKSxcbiAqICAgICByZWJ1aWxkSW50ZXJ2YWw6IER1cmF0aW9uLmRheXMoMTQpLFxuICogfSk7XG4gKiBuZXcgQ29kZUJ1aWxkUnVubmVyKHRoaXMsICdDb2RlQnVpbGQgcHJvdmlkZXInLCB7XG4gKiAgICAgbGFiZWw6ICdjdXN0b20tY29kZWJ1aWxkJyxcbiAqICAgICBpbWFnZUJ1aWxkZXI6IGJ1aWxkZXIsXG4gKiB9KTtcbiAqIGBgYFxuICovXG5leHBvcnQgY2xhc3MgQ29udGFpbmVySW1hZ2VCdWlsZGVyIGV4dGVuZHMgSW1hZ2VCdWlsZGVyQmFzZSBpbXBsZW1lbnRzIElJbWFnZUJ1aWxkZXIge1xuICByZWFkb25seSByZXBvc2l0b3J5OiBlY3IuSVJlcG9zaXRvcnk7XG4gIHByaXZhdGUgcmVhZG9ubHkgcGFyZW50SW1hZ2U6IHN0cmluZyB8IHVuZGVmaW5lZDtcbiAgcHJpdmF0ZSBib3VuZEltYWdlPzogUnVubmVySW1hZ2U7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM/OiBDb250YWluZXJJbWFnZUJ1aWxkZXJQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgb3M6IHByb3BzPy5vcyxcbiAgICAgIHN1cHBvcnRlZE9zOiBbT3MuV0lORE9XU10sXG4gICAgICBhcmNoaXRlY3R1cmU6IHByb3BzPy5hcmNoaXRlY3R1cmUsXG4gICAgICBzdXBwb3J0ZWRBcmNoaXRlY3R1cmVzOiBbQXJjaGl0ZWN0dXJlLlg4Nl82NF0sXG4gICAgICBpbnN0YW5jZVR5cGU6IHByb3BzPy5pbnN0YW5jZVR5cGUsXG4gICAgICB2cGM6IHByb3BzPy52cGMsXG4gICAgICBzZWN1cml0eUdyb3VwczogcHJvcHM/LnNlY3VyaXR5R3JvdXAgPyBbcHJvcHMuc2VjdXJpdHlHcm91cF0gOiBwcm9wcz8uc2VjdXJpdHlHcm91cHMsXG4gICAgICBzdWJuZXRTZWxlY3Rpb246IHByb3BzPy5zdWJuZXRTZWxlY3Rpb24sXG4gICAgICBsb2dSZW1vdmFsUG9saWN5OiBwcm9wcz8ubG9nUmVtb3ZhbFBvbGljeSxcbiAgICAgIGxvZ1JldGVudGlvbjogcHJvcHM/LmxvZ1JldGVudGlvbixcbiAgICAgIHJ1bm5lclZlcnNpb246IHByb3BzPy5ydW5uZXJWZXJzaW9uLFxuICAgICAgcmVidWlsZEludGVydmFsOiBwcm9wcz8ucmVidWlsZEludGVydmFsLFxuICAgICAgaW1hZ2VUeXBlTmFtZTogJ2ltYWdlJyxcbiAgICB9KTtcblxuICAgIHRoaXMucGFyZW50SW1hZ2UgPSBwcm9wcz8ucGFyZW50SW1hZ2U7XG5cbiAgICAvLyBjcmVhdGUgcmVwb3NpdG9yeSB0aGF0IG9ubHkga2VlcHMgb25lIHRhZ1xuICAgIHRoaXMucmVwb3NpdG9yeSA9IG5ldyBlY3IuUmVwb3NpdG9yeSh0aGlzLCAnUmVwb3NpdG9yeScsIHtcbiAgICAgIGltYWdlU2Nhbk9uUHVzaDogdHJ1ZSxcbiAgICAgIGltYWdlVGFnTXV0YWJpbGl0eTogVGFnTXV0YWJpbGl0eS5NVVRBQkxFLFxuICAgICAgcmVtb3ZhbFBvbGljeTogUmVtb3ZhbFBvbGljeS5ERVNUUk9ZLFxuICAgICAgbGlmZWN5Y2xlUnVsZXM6IFtcbiAgICAgICAge1xuICAgICAgICAgIGRlc2NyaXB0aW9uOiAnUmVtb3ZlIGFsbCBidXQgdGhlIGxhdGVzdCBpbWFnZScsXG4gICAgICAgICAgdGFnU3RhdHVzOiBUYWdTdGF0dXMuQU5ZLFxuICAgICAgICAgIG1heEltYWdlQ291bnQ6IDEsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0pO1xuXG4gICAgLy8gYWRkIGFsbCBiYXNpYyBjb21wb25lbnRzXG4gICAgdGhpcy5hZGRCYXNlV2luZG93c0NvbXBvbmVudHMoKTtcbiAgfVxuXG4gIHByaXZhdGUgYWRkQmFzZVdpbmRvd3NDb21wb25lbnRzKCkge1xuICAgIHRoaXMuYWRkQ29tcG9uZW50KFdpbmRvd3NDb21wb25lbnRzLmF3c0NsaSh0aGlzLCAnQVdTIENMSScpKTtcbiAgICB0aGlzLmFkZENvbXBvbmVudChXaW5kb3dzQ29tcG9uZW50cy5naXRodWJDbGkodGhpcywgJ0dpdEh1YiBDTEknKSk7XG4gICAgdGhpcy5hZGRDb21wb25lbnQoV2luZG93c0NvbXBvbmVudHMuZ2l0KHRoaXMsICdnaXQnKSk7XG4gICAgdGhpcy5hZGRDb21wb25lbnQoV2luZG93c0NvbXBvbmVudHMuZ2l0aHViUnVubmVyKHRoaXMsICdHaXRIdWIgQWN0aW9ucyBSdW5uZXInLCB0aGlzLnJ1bm5lclZlcnNpb24pKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBjb21wb25lbnQgdG8gYmUgaW5zdGFsbGVkIGJlZm9yZSBhbnkgb3RoZXIgY29tcG9uZW50cy4gVXNlZnVsIGZvciByZXF1aXJlZCBzeXN0ZW0gc2V0dGluZ3MgbGlrZSBjZXJ0aWZpY2F0ZXMgb3IgcHJveHkgc2V0dGluZ3MuXG4gICAqIEBwYXJhbSBjb21wb25lbnRcbiAgICovXG4gIHByZXBlbmRDb21wb25lbnQoY29tcG9uZW50OiBJbWFnZUJ1aWxkZXJDb21wb25lbnQpIHtcbiAgICBpZiAodGhpcy5ib3VuZEltYWdlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0ltYWdlIGlzIGFscmVhZHkgYm91bmQuIFVzZSB0aGlzIG1ldGhvZCBiZWZvcmUgcGFzc2luZyB0aGUgYnVpbGRlciB0byBhIHJ1bm5lciBwcm92aWRlci4nKTtcbiAgICB9XG4gICAgaWYgKGNvbXBvbmVudC5wbGF0Zm9ybSAhPSB0aGlzLnBsYXRmb3JtKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0NvbXBvbmVudCBwbGF0Zm9ybSBkb2VzblxcJ3QgbWF0Y2ggYnVpbGRlciBwbGF0Zm9ybScpO1xuICAgIH1cbiAgICB0aGlzLmNvbXBvbmVudHMgPSBbY29tcG9uZW50XS5jb25jYXQodGhpcy5jb21wb25lbnRzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBjb21wb25lbnQgdG8gYmUgaW5zdGFsbGVkLlxuICAgKiBAcGFyYW0gY29tcG9uZW50XG4gICAqL1xuICBhZGRDb21wb25lbnQoY29tcG9uZW50OiBJbWFnZUJ1aWxkZXJDb21wb25lbnQpIHtcbiAgICBpZiAodGhpcy5ib3VuZEltYWdlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0ltYWdlIGlzIGFscmVhZHkgYm91bmQuIFVzZSB0aGlzIG1ldGhvZCBiZWZvcmUgcGFzc2luZyB0aGUgYnVpbGRlciB0byBhIHJ1bm5lciBwcm92aWRlci4nKTtcbiAgICB9XG4gICAgaWYgKGNvbXBvbmVudC5wbGF0Zm9ybSAhPSB0aGlzLnBsYXRmb3JtKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0NvbXBvbmVudCBwbGF0Zm9ybSBkb2VzblxcJ3QgbWF0Y2ggYnVpbGRlciBwbGF0Zm9ybScpO1xuICAgIH1cbiAgICB0aGlzLmNvbXBvbmVudHMucHVzaChjb21wb25lbnQpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBleHRyYSB0cnVzdGVkIGNlcnRpZmljYXRlcy4gVGhpcyBoZWxwcyBkZWFsIHdpdGggc2VsZi1zaWduZWQgY2VydGlmaWNhdGVzIGZvciBHaXRIdWIgRW50ZXJwcmlzZSBTZXJ2ZXIuXG4gICAqXG4gICAqIEFsbCBmaXJzdCBwYXJ0eSBEb2NrZXJmaWxlcyBzdXBwb3J0IHRoaXMuIE90aGVycyBtYXkgbm90LlxuICAgKlxuICAgKiBAcGFyYW0gcGF0aCBwYXRoIHRvIGRpcmVjdG9yeSBjb250YWluaW5nIGEgZmlsZSBjYWxsZWQgY2VydHMucGVtIGNvbnRhaW5pbmcgYWxsIHRoZSByZXF1aXJlZCBjZXJ0aWZpY2F0ZXNcbiAgICovXG4gIHB1YmxpYyBhZGRFeHRyYUNlcnRpZmljYXRlcyhwYXRoOiBzdHJpbmcpIHtcbiAgICBpZiAodGhpcy5wbGF0Zm9ybSA9PSAnTGludXgnKSB7XG4gICAgICB0aGlzLnByZXBlbmRDb21wb25lbnQoTGludXhVYnVudHVDb21wb25lbnRzLmV4dHJhQ2VydGlmaWNhdGVzKHRoaXMsICdFeHRyYSBDZXJ0cycsIHBhdGgpKTtcbiAgICB9IGVsc2UgaWYgKHRoaXMucGxhdGZvcm0gPT0gJ1dpbmRvd3MnKSB7XG4gICAgICB0aGlzLnByZXBlbmRDb21wb25lbnQoV2luZG93c0NvbXBvbmVudHMuZXh0cmFDZXJ0aWZpY2F0ZXModGhpcywgJ0V4dHJhIENlcnRzJywgcGF0aCkpO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFVua25vd24gcGxhdGZvcm06ICR7dGhpcy5wbGF0Zm9ybX1gKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQ2FsbGVkIGJ5IElSdW5uZXJQcm92aWRlciB0byBmaW5hbGl6ZSBzZXR0aW5ncyBhbmQgY3JlYXRlIHRoZSBpbWFnZSBidWlsZGVyLlxuICAgKi9cbiAgYmluZCgpOiBSdW5uZXJJbWFnZSB7XG4gICAgaWYgKHRoaXMuYm91bmRJbWFnZSkge1xuICAgICAgcmV0dXJuIHRoaXMuYm91bmRJbWFnZTtcbiAgICB9XG5cbiAgICBjb25zdCBkaXN0ID0gbmV3IGltYWdlYnVpbGRlci5DZm5EaXN0cmlidXRpb25Db25maWd1cmF0aW9uKHRoaXMsICdEaXN0cmlidXRpb24nLCB7XG4gICAgICBuYW1lOiB1bmlxdWVJbWFnZUJ1aWxkZXJOYW1lKHRoaXMpLFxuICAgICAgZGVzY3JpcHRpb246IHRoaXMuZGVzY3JpcHRpb24sXG4gICAgICBkaXN0cmlidXRpb25zOiBbXG4gICAgICAgIHtcbiAgICAgICAgICByZWdpb246IFN0YWNrLm9mKHRoaXMpLnJlZ2lvbixcbiAgICAgICAgICBjb250YWluZXJEaXN0cmlidXRpb25Db25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgICBDb250YWluZXJUYWdzOiBbJ2xhdGVzdCddLFxuICAgICAgICAgICAgVGFyZ2V0UmVwb3NpdG9yeToge1xuICAgICAgICAgICAgICBTZXJ2aWNlOiAnRUNSJyxcbiAgICAgICAgICAgICAgUmVwb3NpdG9yeU5hbWU6IHRoaXMucmVwb3NpdG9yeS5yZXBvc2l0b3J5TmFtZSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgICBjb25zdCByZWNpcGUgPSBuZXcgQ29udGFpbmVyUmVjaXBlKHRoaXMsICdDb250YWluZXIgUmVjaXBlJywge1xuICAgICAgcGxhdGZvcm06IHRoaXMucGxhdGZvcm0sXG4gICAgICBjb21wb25lbnRzOiB0aGlzLmNvbXBvbmVudHMsXG4gICAgICB0YXJnZXRSZXBvc2l0b3J5OiB0aGlzLnJlcG9zaXRvcnksXG4gICAgICBkb2NrZXJmaWxlVGVtcGxhdGU6IGRvY2tlcmZpbGVUZW1wbGF0ZS5yZXBsYWNlKCdfX19SVU5ORVJfVkVSU0lPTl9fXycsIHRoaXMucnVubmVyVmVyc2lvbi52ZXJzaW9uKSxcbiAgICAgIHBhcmVudEltYWdlOiB0aGlzLnBhcmVudEltYWdlLFxuICAgIH0pO1xuXG4gICAgY29uc3QgbG9nID0gdGhpcy5jcmVhdGVMb2cocmVjaXBlLm5hbWUpO1xuICAgIGNvbnN0IGluZnJhID0gdGhpcy5jcmVhdGVJbmZyYXN0cnVjdHVyZShbXG4gICAgICBpYW0uTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ0FtYXpvblNTTU1hbmFnZWRJbnN0YW5jZUNvcmUnKSxcbiAgICAgIGlhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnRUMySW5zdGFuY2VQcm9maWxlRm9ySW1hZ2VCdWlsZGVyRUNSQ29udGFpbmVyQnVpbGRzJyksXG4gICAgXSk7XG4gICAgY29uc3QgaW1hZ2UgPSB0aGlzLmNyZWF0ZUltYWdlKGluZnJhLCBkaXN0LCBsb2csIHVuZGVmaW5lZCwgcmVjaXBlLmFybik7XG4gICAgdGhpcy5jcmVhdGVQaXBlbGluZShpbmZyYSwgZGlzdCwgbG9nLCB1bmRlZmluZWQsIHJlY2lwZS5hcm4pO1xuXG4gICAgdGhpcy5pbWFnZUNsZWFuZXIoaW1hZ2UsIHJlY2lwZS5uYW1lKTtcblxuICAgIHRoaXMuYm91bmRJbWFnZSA9IHtcbiAgICAgIC8vIFRoZXJlIGFyZSBzaW1wbGVyIHdheXMgdG8gZ2V0IHRoZSBBUk4sIGJ1dCB3ZSB3YW50IGFuIGltYWdlIG9iamVjdCB0aGF0IGRlcGVuZHMgb24gdGhlIG5ld2x5IGJ1aWx0IGltYWdlLlxuICAgICAgLy8gV2Ugd2FudCB3aG9ldmVyIGlzIHVzaW5nIHRoaXMgaW1hZ2UgdG8gYXV0b21hdGljYWxseSB3YWl0IGZvciBJbWFnZSBCdWlsZGVyIHRvIGZpbmlzaCBidWlsZGluZyBiZWZvcmUgdXNpbmcgdGhlIGltYWdlLlxuICAgICAgaW1hZ2VSZXBvc2l0b3J5OiBlY3IuUmVwb3NpdG9yeS5mcm9tUmVwb3NpdG9yeU5hbWUoXG4gICAgICAgIHRoaXMsICdEZXBlbmRhYmxlIEltYWdlJyxcbiAgICAgICAgLy8gd2UgY2FuJ3QgdXNlIGltYWdlLmF0dHJOYW1lIGJlY2F1c2UgaXQgY29tZXMgdXAgd2l0aCB1cHBlciBjYXNlXG4gICAgICAgIGNkay5Gbi5zcGxpdCgnOicsIGNkay5Gbi5zcGxpdCgnLycsIGltYWdlLmF0dHJJbWFnZVVyaSwgMilbMV0sIDIpWzBdLFxuICAgICAgKSxcbiAgICAgIGltYWdlVGFnOiAnbGF0ZXN0JyxcbiAgICAgIG9zOiB0aGlzLm9zLFxuICAgICAgYXJjaGl0ZWN0dXJlOiB0aGlzLmFyY2hpdGVjdHVyZSxcbiAgICAgIGxvZ0dyb3VwOiBsb2csXG4gICAgICBydW5uZXJWZXJzaW9uOiB0aGlzLnJ1bm5lclZlcnNpb24sXG4gICAgfTtcblxuICAgIHJldHVybiB0aGlzLmJvdW5kSW1hZ2U7XG4gIH1cblxuICBwcml2YXRlIGltYWdlQ2xlYW5lcihpbWFnZTogaW1hZ2VidWlsZGVyLkNmbkltYWdlLCByZWNpcGVOYW1lOiBzdHJpbmcpIHtcbiAgICBjb25zdCBjckhhbmRsZXIgPSBCdW5kbGVkTm9kZWpzRnVuY3Rpb24uc2luZ2xldG9uKHRoaXMsICdidWlsZC1pbWFnZScsIHtcbiAgICAgIGRlc2NyaXB0aW9uOiAnQ3VzdG9tIHJlc291cmNlIGhhbmRsZXIgdGhhdCB0cmlnZ2VycyBDb2RlQnVpbGQgdG8gYnVpbGQgcnVubmVyIGltYWdlcywgYW5kIGNsZWFucy11cCBpbWFnZXMgb24gZGVsZXRpb24nLFxuICAgICAgdGltZW91dDogY2RrLkR1cmF0aW9uLm1pbnV0ZXMoMyksXG4gICAgfSk7XG5cbiAgICBjb25zdCBwb2xpY3kgPSBuZXcgaWFtLlBvbGljeSh0aGlzLCAnQ1IgUG9saWN5Jywge1xuICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgYWN0aW9uczogWydlY3I6QmF0Y2hEZWxldGVJbWFnZScsICdlY3I6TGlzdEltYWdlcyddLFxuICAgICAgICAgIHJlc291cmNlczogW3RoaXMucmVwb3NpdG9yeS5yZXBvc2l0b3J5QXJuXSxcbiAgICAgICAgfSksXG4gICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBhY3Rpb25zOiBbJ2ltYWdlYnVpbGRlcjpMaXN0SW1hZ2VzJywgJ2ltYWdlYnVpbGRlcjpMaXN0SW1hZ2VCdWlsZFZlcnNpb25zJywgJ2ltYWdlYnVpbGRlcjpEZWxldGVJbWFnZSddLFxuICAgICAgICAgIHJlc291cmNlczogWycqJ10sIC8vIEltYWdlIEJ1aWxkZXIgZG9lc24ndCBzdXBwb3J0IHNjb3BpbmcgdGhpcyA6KFxuICAgICAgICB9KSxcbiAgICAgIF0sXG4gICAgfSk7XG4gICAgY3JIYW5kbGVyLnJvbGU/LmF0dGFjaElubGluZVBvbGljeShwb2xpY3kpO1xuXG4gICAgY29uc3QgY3IgPSBuZXcgQ3VzdG9tUmVzb3VyY2UodGhpcywgJ0RlbGV0ZXInLCB7XG4gICAgICBzZXJ2aWNlVG9rZW46IGNySGFuZGxlci5mdW5jdGlvbkFybixcbiAgICAgIHJlc291cmNlVHlwZTogJ0N1c3RvbTo6SW1hZ2VEZWxldGVyJyxcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgUmVwb05hbWU6IHRoaXMucmVwb3NpdG9yeS5yZXBvc2l0b3J5TmFtZSxcbiAgICAgICAgSW1hZ2VCdWlsZGVyTmFtZTogcmVjaXBlTmFtZSwgLy8gd2UgZG9uJ3QgdXNlIGltYWdlLm5hbWUgYmVjYXVzZSBDbG91ZEZvcm1hdGlvbiBjb21wbGFpbnMgaWYgaXQgd2FzIGRlbGV0ZWQgYWxyZWFkeVxuICAgICAgICBEZWxldGVPbmx5OiB0cnVlLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIGFkZCBkZXBlbmRlbmNpZXMgdG8gbWFrZSBzdXJlIHJlc291cmNlcyBhcmUgdGhlcmUgd2hlbiB3ZSBuZWVkIHRoZW1cbiAgICBjci5ub2RlLmFkZERlcGVuZGVuY3koaW1hZ2UpO1xuICAgIGNyLm5vZGUuYWRkRGVwZW5kZW5jeShwb2xpY3kpO1xuICAgIGNyLm5vZGUuYWRkRGVwZW5kZW5jeShjckhhbmRsZXIpO1xuXG4gICAgcmV0dXJuIGNyO1xuICB9XG59XG4iXX0=