"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FargateRunner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const common_1 = require("./common");
const codebuild_1 = require("./image-builders/codebuild");
/**
 * Our special launch target that can use spot instances and set EnableExecuteCommand.
 */
class EcsFargateLaunchTarget {
    constructor(props) {
        this.props = props;
    }
    /**
     * Called when the Fargate launch type configured on RunTask
     */
    bind(_task, launchTargetOptions) {
        if (!launchTargetOptions.taskDefinition.isFargateCompatible) {
            throw new Error('Supplied TaskDefinition is not compatible with Fargate');
        }
        return {
            parameters: {
                EnableExecuteCommand: this.props.enableExecute,
                CapacityProviderStrategy: [
                    {
                        CapacityProvider: this.props.spot ? 'FARGATE_SPOT' : 'FARGATE',
                    },
                ],
            },
        };
    }
}
/**
 * GitHub Actions runner provider using Fargate to execute jobs.
 *
 * Creates a task definition with a single container that gets started for each job.
 *
 * This construct is not meant to be used by itself. It should be passed in the providers property for GitHubRunners.
 */
class FargateRunner extends common_1.BaseProvider {
    constructor(scope, id, props) {
        super(scope, id);
        this.labels = this.labelsFromProperties('fargate', props?.label, props?.labels);
        this.vpc = props?.vpc ?? aws_cdk_lib_1.aws_ec2.Vpc.fromLookup(this, 'default vpc', { isDefault: true });
        this.subnetSelection = props?.subnetSelection;
        this.securityGroups = props?.securityGroup ? [props.securityGroup] : (props?.securityGroups ?? [new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'security group', { vpc: this.vpc })]);
        this.connections = new aws_cdk_lib_1.aws_ec2.Connections({ securityGroups: this.securityGroups });
        this.assignPublicIp = props?.assignPublicIp ?? true;
        this.cluster = props?.cluster ? props.cluster : new aws_cdk_lib_1.aws_ecs.Cluster(this, 'cluster', {
            vpc: this.vpc,
            enableFargateCapacityProviders: true,
        });
        this.spot = props?.spot ?? false;
        const imageBuilder = props?.imageBuilder ?? new codebuild_1.CodeBuildImageBuilder(this, 'Image Builder', {
            dockerfilePath: FargateRunner.LINUX_X64_DOCKERFILE_PATH,
        });
        const image = this.image = imageBuilder.bind();
        let arch;
        if (image.architecture.is(common_1.Architecture.ARM64)) {
            arch = aws_cdk_lib_1.aws_ecs.CpuArchitecture.ARM64;
        }
        else if (image.architecture.is(common_1.Architecture.X86_64)) {
            arch = aws_cdk_lib_1.aws_ecs.CpuArchitecture.X86_64;
        }
        else {
            throw new Error(`${image.architecture.name} is not supported on Fargate`);
        }
        let os;
        if (image.os.is(common_1.Os.LINUX)) {
            os = aws_cdk_lib_1.aws_ecs.OperatingSystemFamily.LINUX;
        }
        else if (image.os.is(common_1.Os.WINDOWS)) {
            os = aws_cdk_lib_1.aws_ecs.OperatingSystemFamily.WINDOWS_SERVER_2019_CORE;
            if (props?.ephemeralStorageGiB) {
                throw new Error('Ephemeral storage is not supported on Fargate Windows');
            }
        }
        else {
            throw new Error(`${image.os.name} is not supported on Fargate`);
        }
        this.task = new aws_cdk_lib_1.aws_ecs.FargateTaskDefinition(this, 'task', {
            cpu: props?.cpu ?? 1024,
            memoryLimitMiB: props?.memoryLimitMiB ?? 2048,
            ephemeralStorageGiB: props?.ephemeralStorageGiB ?? !image.os.is(common_1.Os.WINDOWS) ? 25 : undefined,
            runtimePlatform: {
                operatingSystemFamily: os,
                cpuArchitecture: arch,
            },
        });
        this.container = this.task.addContainer('runner', {
            image: aws_cdk_lib_1.aws_ecs.AssetImage.fromEcrRepository(image.imageRepository, image.imageTag),
            logging: aws_cdk_lib_1.aws_ecs.AwsLogDriver.awsLogs({
                logGroup: new aws_cdk_lib_1.aws_logs.LogGroup(this, 'logs', {
                    retention: props?.logRetention ?? aws_logs_1.RetentionDays.ONE_MONTH,
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                }),
                streamPrefix: 'runner',
            }),
            command: this.runCommand(),
        });
        this.grantPrincipal = new aws_cdk_lib_1.aws_iam.UnknownPrincipal({ resource: this.task.taskRole });
    }
    /**
     * Generate step function task(s) to start a new runner.
     *
     * Called by GithubRunners and shouldn't be called manually.
     *
     * @param parameters workflow job details
     */
    getStepFunctionTask(parameters) {
        return new aws_cdk_lib_1.aws_stepfunctions_tasks.EcsRunTask(this, this.labels.join(', '), {
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
            taskDefinition: this.task,
            cluster: this.cluster,
            launchTarget: new EcsFargateLaunchTarget({
                spot: this.spot,
                enableExecute: this.image.os.is(common_1.Os.LINUX),
            }),
            subnets: this.subnetSelection,
            assignPublicIp: this.assignPublicIp,
            securityGroups: this.securityGroups,
            containerOverrides: [
                {
                    containerDefinition: this.container,
                    environment: [
                        {
                            name: 'RUNNER_TOKEN',
                            value: parameters.runnerTokenPath,
                        },
                        {
                            name: 'RUNNER_NAME',
                            value: parameters.runnerNamePath,
                        },
                        {
                            name: 'RUNNER_LABEL',
                            value: this.labels.join(','),
                        },
                        {
                            name: 'GITHUB_DOMAIN',
                            value: parameters.githubDomainPath,
                        },
                        {
                            name: 'OWNER',
                            value: parameters.ownerPath,
                        },
                        {
                            name: 'REPO',
                            value: parameters.repoPath,
                        },
                    ],
                },
            ],
        });
    }
    grantStateMachine(_) {
    }
    status(statusFunctionRole) {
        this.image.imageRepository.grant(statusFunctionRole, 'ecr:DescribeImages');
        return {
            type: this.constructor.name,
            labels: this.labels,
            vpcArn: this.vpc?.vpcArn,
            securityGroups: this.securityGroups.map(sg => sg.securityGroupId),
            roleArn: this.task.taskRole.roleArn,
            image: {
                imageRepository: this.image.imageRepository.repositoryUri,
                imageTag: this.image.imageTag,
                imageBuilderLogGroup: this.image.logGroup?.logGroupName,
            },
        };
    }
    runCommand() {
        let runnerFlags = '';
        if (this.image.runnerVersion.is(common_1.RunnerVersion.latest())) {
            runnerFlags = '--disableupdate';
        }
        if (this.image.os.is(common_1.Os.LINUX)) {
            return [
                'sh', '-c',
                `./config.sh --unattended --url "https://$GITHUB_DOMAIN/$OWNER/$REPO" --token "$RUNNER_TOKEN" --ephemeral --work _work --labels "$RUNNER_LABEL" ${runnerFlags} --name "$RUNNER_NAME" && ./run.sh`,
            ];
        }
        else if (this.image.os.is(common_1.Os.WINDOWS)) {
            return [
                'powershell', '-Command',
                `cd \\actions ; ./config.cmd --unattended --url "https://\${Env:GITHUB_DOMAIN}/\${Env:OWNER}/\${Env:REPO}" --token "\${Env:RUNNER_TOKEN}" --ephemeral --work _work --labels "\${Env:RUNNER_LABEL}" ${runnerFlags} --name "\${Env:RUNNER_NAME}" ; ./run.cmd`,
            ];
        }
        else {
            throw new Error(`Fargate runner doesn't support ${this.image.os.name}`);
        }
    }
}
exports.FargateRunner = FargateRunner;
_a = JSII_RTTI_SYMBOL_1;
FargateRunner[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.FargateRunner", version: "0.7.1" };
/**
 * Path to Dockerfile for Linux x64 with all the requirement for Fargate runner. Use this Dockerfile unless you need to customize it further than allowed by hooks.
 *
 * Available build arguments that can be set in the image builder:
 * * `BASE_IMAGE` sets the `FROM` line. This should be an Ubuntu compatible image.
 * * `EXTRA_PACKAGES` can be used to install additional packages.
 */
FargateRunner.LINUX_X64_DOCKERFILE_PATH = path.join(__dirname, 'docker-images', 'fargate', 'linux-x64');
/**
 * Path to Dockerfile for Linux ARM64 with all the requirement for Fargate runner. Use this Dockerfile unless you need to customize it further than allowed by hooks.
 *
 * Available build arguments that can be set in the image builder:
 * * `BASE_IMAGE` sets the `FROM` line. This should be an Ubuntu compatible image.
 * * `EXTRA_PACKAGES` can be used to install additional packages.
 */
FargateRunner.LINUX_ARM64_DOCKERFILE_PATH = path.join(__dirname, 'docker-images', 'fargate', 'linux-arm64');
//# sourceMappingURL=data:application/json;base64,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