Python Bindings for the DUNE-FEM module
=======================================

*Authors: Andreas Dedner, Martin Nolte, and Robert Kl&ouml;fkorn*

dune-fempy aims to provide Python bindings for the [dune-fem][femlink] discretization
module. Here an example for solving Poisson's equation:

```python
import math
from ufl import *
import dune.ufl
import dune.create as create

grid = create.grid("ALUConform", dune.grid.cartesianDomain([0, 0], [1, 1], [8, 8]), dimgrid=2)

# set up a diffusion reaction model using UFL
uflSpace = dune.ufl.Space((grid.dimGrid, grid.dimWorld), 1, field="double")
u = TrialFunction(uflSpace)
v = TestFunction(uflSpace)
x = SpatialCoordinate(uflSpace.cell())
a = (inner(grad(u), grad(v)) + inner(u,v)) * dx
# provide an exact solution that will be used to add suitable forcing and dirichlet b.c.
exact = as_vector( [cos(2.*pi*x[0])*cos(2.*pi*x[1])] )
model = create.model("elliptic", grid, a==0, exact=exact, dirichlet={ 1:exact } )

# set up a space and a conforming finite element scheme and solve the PDE
space  = dune.create.space("Lagrange", grid, dimrange=1, order=1)
scheme = create.scheme("h1", space, model, "cg")
uh,info = scheme.solve()

# make 'exact' into a grid function for output and uh into an UFL coefficient for error computation
exact_gf = create.function("ufl", grid, "exact", 5, exact)
uh_coeff = dune.ufl.GridCoefficient(uh)
# now define a grid function representing the pointwise error
l2error_gf = create.function("ufl", grid, "error", 5, as_vector([(exact[0]-uh_coeff[0])**2]) )

error = math.sqrt( l2error_gf.integrate() )
print("size:",grid.size(0),"L2-error:",error)
grid.writeVTK("laplace", pointdata=[ uh, l2error_gf, exact_gf ])
```

See the file COPYING for full copying permissions.

If you find this tutorial helpful for getting your own projects up and
running please cite

Title: Python Bindings for the DUNE-FEM module
*Authors: Andreas Dedner, Martin Nolte, and Robert Klöfkorn*
Publisher: Zenodoo, 2020
DOI 10.5281/zenodo.3706994

[![DOI](https://zenodo.org/badge/246695081.svg)](https://zenodo.org/badge/latestdoi/246695081)

Dependencies
------------

dune-fempy depends on the following DUNE modules:
- dune-common 2.8+
- dune-geometry 2.8+
- dune-grid 2.8+
- dune-python 2.8+
- dune-fem 2.8+

In addition to the dependencies of these DUNE modules, the following software
packages are required:

- a C++17 compatible C++ compiler
- Python 2.7+ (although mostly tested with 3.4+)

We strongly recommend installing the following Python packages to make full
use of dune-fem's Python bindings:

- numpy
- mpi4py
- ufl (2016.1.0+)

[femlink]: https://gitlab.dune-project.org/dune-fem/dune-fem


git-2d404a0a28ab3122f8ae41ce03c19f1b9699eb79
