(function () {
    'use strict';

    /*! *****************************************************************************
    Copyright (c) Microsoft Corporation.

    Permission to use, copy, modify, and/or distribute this software for any
    purpose with or without fee is hereby granted.

    THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH
    REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
    AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT,
    INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
    LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
    OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
    PERFORMANCE OF THIS SOFTWARE.
    ***************************************************************************** */

    function __decorate(decorators, target, key, desc) {
        var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
        if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
        else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
        return c > 3 && r && Object.defineProperty(target, key, r), r;
    }

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    /**
     * True if the custom elements polyfill is in use.
     */
    const isCEPolyfill = typeof window !== 'undefined' &&
        window.customElements != null &&
        window.customElements.polyfillWrapFlushCallback !==
            undefined;
    /**
     * Removes nodes, starting from `start` (inclusive) to `end` (exclusive), from
     * `container`.
     */
    const removeNodes = (container, start, end = null) => {
        while (start !== end) {
            const n = start.nextSibling;
            container.removeChild(start);
            start = n;
        }
    };

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    /**
     * An expression marker with embedded unique key to avoid collision with
     * possible text in templates.
     */
    const marker = `{{lit-${String(Math.random()).slice(2)}}}`;
    /**
     * An expression marker used text-positions, multi-binding attributes, and
     * attributes with markup-like text values.
     */
    const nodeMarker = `<!--${marker}-->`;
    const markerRegex = new RegExp(`${marker}|${nodeMarker}`);
    /**
     * Suffix appended to all bound attribute names.
     */
    const boundAttributeSuffix = '$lit$';
    /**
     * An updatable Template that tracks the location of dynamic parts.
     */
    class Template {
        constructor(result, element) {
            this.parts = [];
            this.element = element;
            const nodesToRemove = [];
            const stack = [];
            // Edge needs all 4 parameters present; IE11 needs 3rd parameter to be null
            const walker = document.createTreeWalker(element.content, 133 /* NodeFilter.SHOW_{ELEMENT|COMMENT|TEXT} */, null, false);
            // Keeps track of the last index associated with a part. We try to delete
            // unnecessary nodes, but we never want to associate two different parts
            // to the same index. They must have a constant node between.
            let lastPartIndex = 0;
            let index = -1;
            let partIndex = 0;
            const { strings, values: { length } } = result;
            while (partIndex < length) {
                const node = walker.nextNode();
                if (node === null) {
                    // We've exhausted the content inside a nested template element.
                    // Because we still have parts (the outer for-loop), we know:
                    // - There is a template in the stack
                    // - The walker will find a nextNode outside the template
                    walker.currentNode = stack.pop();
                    continue;
                }
                index++;
                if (node.nodeType === 1 /* Node.ELEMENT_NODE */) {
                    if (node.hasAttributes()) {
                        const attributes = node.attributes;
                        const { length } = attributes;
                        // Per
                        // https://developer.mozilla.org/en-US/docs/Web/API/NamedNodeMap,
                        // attributes are not guaranteed to be returned in document order.
                        // In particular, Edge/IE can return them out of order, so we cannot
                        // assume a correspondence between part index and attribute index.
                        let count = 0;
                        for (let i = 0; i < length; i++) {
                            if (endsWith(attributes[i].name, boundAttributeSuffix)) {
                                count++;
                            }
                        }
                        while (count-- > 0) {
                            // Get the template literal section leading up to the first
                            // expression in this attribute
                            const stringForPart = strings[partIndex];
                            // Find the attribute name
                            const name = lastAttributeNameRegex.exec(stringForPart)[2];
                            // Find the corresponding attribute
                            // All bound attributes have had a suffix added in
                            // TemplateResult#getHTML to opt out of special attribute
                            // handling. To look up the attribute value we also need to add
                            // the suffix.
                            const attributeLookupName = name.toLowerCase() + boundAttributeSuffix;
                            const attributeValue = node.getAttribute(attributeLookupName);
                            node.removeAttribute(attributeLookupName);
                            const statics = attributeValue.split(markerRegex);
                            this.parts.push({ type: 'attribute', index, name, strings: statics });
                            partIndex += statics.length - 1;
                        }
                    }
                    if (node.tagName === 'TEMPLATE') {
                        stack.push(node);
                        walker.currentNode = node.content;
                    }
                }
                else if (node.nodeType === 3 /* Node.TEXT_NODE */) {
                    const data = node.data;
                    if (data.indexOf(marker) >= 0) {
                        const parent = node.parentNode;
                        const strings = data.split(markerRegex);
                        const lastIndex = strings.length - 1;
                        // Generate a new text node for each literal section
                        // These nodes are also used as the markers for node parts
                        for (let i = 0; i < lastIndex; i++) {
                            let insert;
                            let s = strings[i];
                            if (s === '') {
                                insert = createMarker();
                            }
                            else {
                                const match = lastAttributeNameRegex.exec(s);
                                if (match !== null && endsWith(match[2], boundAttributeSuffix)) {
                                    s = s.slice(0, match.index) + match[1] +
                                        match[2].slice(0, -boundAttributeSuffix.length) + match[3];
                                }
                                insert = document.createTextNode(s);
                            }
                            parent.insertBefore(insert, node);
                            this.parts.push({ type: 'node', index: ++index });
                        }
                        // If there's no text, we must insert a comment to mark our place.
                        // Else, we can trust it will stick around after cloning.
                        if (strings[lastIndex] === '') {
                            parent.insertBefore(createMarker(), node);
                            nodesToRemove.push(node);
                        }
                        else {
                            node.data = strings[lastIndex];
                        }
                        // We have a part for each match found
                        partIndex += lastIndex;
                    }
                }
                else if (node.nodeType === 8 /* Node.COMMENT_NODE */) {
                    if (node.data === marker) {
                        const parent = node.parentNode;
                        // Add a new marker node to be the startNode of the Part if any of
                        // the following are true:
                        //  * We don't have a previousSibling
                        //  * The previousSibling is already the start of a previous part
                        if (node.previousSibling === null || index === lastPartIndex) {
                            index++;
                            parent.insertBefore(createMarker(), node);
                        }
                        lastPartIndex = index;
                        this.parts.push({ type: 'node', index });
                        // If we don't have a nextSibling, keep this node so we have an end.
                        // Else, we can remove it to save future costs.
                        if (node.nextSibling === null) {
                            node.data = '';
                        }
                        else {
                            nodesToRemove.push(node);
                            index--;
                        }
                        partIndex++;
                    }
                    else {
                        let i = -1;
                        while ((i = node.data.indexOf(marker, i + 1)) !== -1) {
                            // Comment node has a binding marker inside, make an inactive part
                            // The binding won't work, but subsequent bindings will
                            // TODO (justinfagnani): consider whether it's even worth it to
                            // make bindings in comments work
                            this.parts.push({ type: 'node', index: -1 });
                            partIndex++;
                        }
                    }
                }
            }
            // Remove text binding nodes after the walk to not disturb the TreeWalker
            for (const n of nodesToRemove) {
                n.parentNode.removeChild(n);
            }
        }
    }
    const endsWith = (str, suffix) => {
        const index = str.length - suffix.length;
        return index >= 0 && str.slice(index) === suffix;
    };
    const isTemplatePartActive = (part) => part.index !== -1;
    // Allows `document.createComment('')` to be renamed for a
    // small manual size-savings.
    const createMarker = () => document.createComment('');
    /**
     * This regex extracts the attribute name preceding an attribute-position
     * expression. It does this by matching the syntax allowed for attributes
     * against the string literal directly preceding the expression, assuming that
     * the expression is in an attribute-value position.
     *
     * See attributes in the HTML spec:
     * https://www.w3.org/TR/html5/syntax.html#elements-attributes
     *
     * " \x09\x0a\x0c\x0d" are HTML space characters:
     * https://www.w3.org/TR/html5/infrastructure.html#space-characters
     *
     * "\0-\x1F\x7F-\x9F" are Unicode control characters, which includes every
     * space character except " ".
     *
     * So an attribute is:
     *  * The name: any character except a control character, space character, ('),
     *    ("), ">", "=", or "/"
     *  * Followed by zero or more space characters
     *  * Followed by "="
     *  * Followed by zero or more space characters
     *  * Followed by:
     *    * Any character except space, ('), ("), "<", ">", "=", (`), or
     *    * (") then any non-("), or
     *    * (') then any non-(')
     */
    const lastAttributeNameRegex = 
    // eslint-disable-next-line no-control-regex
    /([ \x09\x0a\x0c\x0d])([^\0-\x1F\x7F-\x9F "'>=/]+)([ \x09\x0a\x0c\x0d]*=[ \x09\x0a\x0c\x0d]*(?:[^ \x09\x0a\x0c\x0d"'`<>=]*|"[^"]*|'[^']*))$/;

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    const walkerNodeFilter = 133 /* NodeFilter.SHOW_{ELEMENT|COMMENT|TEXT} */;
    /**
     * Removes the list of nodes from a Template safely. In addition to removing
     * nodes from the Template, the Template part indices are updated to match
     * the mutated Template DOM.
     *
     * As the template is walked the removal state is tracked and
     * part indices are adjusted as needed.
     *
     * div
     *   div#1 (remove) <-- start removing (removing node is div#1)
     *     div
     *       div#2 (remove)  <-- continue removing (removing node is still div#1)
     *         div
     * div <-- stop removing since previous sibling is the removing node (div#1,
     * removed 4 nodes)
     */
    function removeNodesFromTemplate(template, nodesToRemove) {
        const { element: { content }, parts } = template;
        const walker = document.createTreeWalker(content, walkerNodeFilter, null, false);
        let partIndex = nextActiveIndexInTemplateParts(parts);
        let part = parts[partIndex];
        let nodeIndex = -1;
        let removeCount = 0;
        const nodesToRemoveInTemplate = [];
        let currentRemovingNode = null;
        while (walker.nextNode()) {
            nodeIndex++;
            const node = walker.currentNode;
            // End removal if stepped past the removing node
            if (node.previousSibling === currentRemovingNode) {
                currentRemovingNode = null;
            }
            // A node to remove was found in the template
            if (nodesToRemove.has(node)) {
                nodesToRemoveInTemplate.push(node);
                // Track node we're removing
                if (currentRemovingNode === null) {
                    currentRemovingNode = node;
                }
            }
            // When removing, increment count by which to adjust subsequent part indices
            if (currentRemovingNode !== null) {
                removeCount++;
            }
            while (part !== undefined && part.index === nodeIndex) {
                // If part is in a removed node deactivate it by setting index to -1 or
                // adjust the index as needed.
                part.index = currentRemovingNode !== null ? -1 : part.index - removeCount;
                // go to the next active part.
                partIndex = nextActiveIndexInTemplateParts(parts, partIndex);
                part = parts[partIndex];
            }
        }
        nodesToRemoveInTemplate.forEach((n) => n.parentNode.removeChild(n));
    }
    const countNodes = (node) => {
        let count = (node.nodeType === 11 /* Node.DOCUMENT_FRAGMENT_NODE */) ? 0 : 1;
        const walker = document.createTreeWalker(node, walkerNodeFilter, null, false);
        while (walker.nextNode()) {
            count++;
        }
        return count;
    };
    const nextActiveIndexInTemplateParts = (parts, startIndex = -1) => {
        for (let i = startIndex + 1; i < parts.length; i++) {
            const part = parts[i];
            if (isTemplatePartActive(part)) {
                return i;
            }
        }
        return -1;
    };
    /**
     * Inserts the given node into the Template, optionally before the given
     * refNode. In addition to inserting the node into the Template, the Template
     * part indices are updated to match the mutated Template DOM.
     */
    function insertNodeIntoTemplate(template, node, refNode = null) {
        const { element: { content }, parts } = template;
        // If there's no refNode, then put node at end of template.
        // No part indices need to be shifted in this case.
        if (refNode === null || refNode === undefined) {
            content.appendChild(node);
            return;
        }
        const walker = document.createTreeWalker(content, walkerNodeFilter, null, false);
        let partIndex = nextActiveIndexInTemplateParts(parts);
        let insertCount = 0;
        let walkerIndex = -1;
        while (walker.nextNode()) {
            walkerIndex++;
            const walkerNode = walker.currentNode;
            if (walkerNode === refNode) {
                insertCount = countNodes(node);
                refNode.parentNode.insertBefore(node, refNode);
            }
            while (partIndex !== -1 && parts[partIndex].index === walkerIndex) {
                // If we've inserted the node, simply adjust all subsequent parts
                if (insertCount > 0) {
                    while (partIndex !== -1) {
                        parts[partIndex].index += insertCount;
                        partIndex = nextActiveIndexInTemplateParts(parts, partIndex);
                    }
                    return;
                }
                partIndex = nextActiveIndexInTemplateParts(parts, partIndex);
            }
        }
    }

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    const directives = new WeakMap();
    /**
     * Brands a function as a directive factory function so that lit-html will call
     * the function during template rendering, rather than passing as a value.
     *
     * A _directive_ is a function that takes a Part as an argument. It has the
     * signature: `(part: Part) => void`.
     *
     * A directive _factory_ is a function that takes arguments for data and
     * configuration and returns a directive. Users of directive usually refer to
     * the directive factory as the directive. For example, "The repeat directive".
     *
     * Usually a template author will invoke a directive factory in their template
     * with relevant arguments, which will then return a directive function.
     *
     * Here's an example of using the `repeat()` directive factory that takes an
     * array and a function to render an item:
     *
     * ```js
     * html`<ul><${repeat(items, (item) => html`<li>${item}</li>`)}</ul>`
     * ```
     *
     * When `repeat` is invoked, it returns a directive function that closes over
     * `items` and the template function. When the outer template is rendered, the
     * return directive function is called with the Part for the expression.
     * `repeat` then performs it's custom logic to render multiple items.
     *
     * @param f The directive factory function. Must be a function that returns a
     * function of the signature `(part: Part) => void`. The returned function will
     * be called with the part object.
     *
     * @example
     *
     * import {directive, html} from 'lit-html';
     *
     * const immutable = directive((v) => (part) => {
     *   if (part.value !== v) {
     *     part.setValue(v)
     *   }
     * });
     */
    const directive = (f) => ((...args) => {
        const d = f(...args);
        directives.set(d, true);
        return d;
    });
    const isDirective = (o) => {
        return typeof o === 'function' && directives.has(o);
    };

    /**
     * @license
     * Copyright (c) 2018 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    /**
     * A sentinel value that signals that a value was handled by a directive and
     * should not be written to the DOM.
     */
    const noChange = {};
    /**
     * A sentinel value that signals a NodePart to fully clear its content.
     */
    const nothing = {};

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    /**
     * An instance of a `Template` that can be attached to the DOM and updated
     * with new values.
     */
    class TemplateInstance {
        constructor(template, processor, options) {
            this.__parts = [];
            this.template = template;
            this.processor = processor;
            this.options = options;
        }
        update(values) {
            let i = 0;
            for (const part of this.__parts) {
                if (part !== undefined) {
                    part.setValue(values[i]);
                }
                i++;
            }
            for (const part of this.__parts) {
                if (part !== undefined) {
                    part.commit();
                }
            }
        }
        _clone() {
            // There are a number of steps in the lifecycle of a template instance's
            // DOM fragment:
            //  1. Clone - create the instance fragment
            //  2. Adopt - adopt into the main document
            //  3. Process - find part markers and create parts
            //  4. Upgrade - upgrade custom elements
            //  5. Update - set node, attribute, property, etc., values
            //  6. Connect - connect to the document. Optional and outside of this
            //     method.
            //
            // We have a few constraints on the ordering of these steps:
            //  * We need to upgrade before updating, so that property values will pass
            //    through any property setters.
            //  * We would like to process before upgrading so that we're sure that the
            //    cloned fragment is inert and not disturbed by self-modifying DOM.
            //  * We want custom elements to upgrade even in disconnected fragments.
            //
            // Given these constraints, with full custom elements support we would
            // prefer the order: Clone, Process, Adopt, Upgrade, Update, Connect
            //
            // But Safari does not implement CustomElementRegistry#upgrade, so we
            // can not implement that order and still have upgrade-before-update and
            // upgrade disconnected fragments. So we instead sacrifice the
            // process-before-upgrade constraint, since in Custom Elements v1 elements
            // must not modify their light DOM in the constructor. We still have issues
            // when co-existing with CEv0 elements like Polymer 1, and with polyfills
            // that don't strictly adhere to the no-modification rule because shadow
            // DOM, which may be created in the constructor, is emulated by being placed
            // in the light DOM.
            //
            // The resulting order is on native is: Clone, Adopt, Upgrade, Process,
            // Update, Connect. document.importNode() performs Clone, Adopt, and Upgrade
            // in one step.
            //
            // The Custom Elements v1 polyfill supports upgrade(), so the order when
            // polyfilled is the more ideal: Clone, Process, Adopt, Upgrade, Update,
            // Connect.
            const fragment = isCEPolyfill ?
                this.template.element.content.cloneNode(true) :
                document.importNode(this.template.element.content, true);
            const stack = [];
            const parts = this.template.parts;
            // Edge needs all 4 parameters present; IE11 needs 3rd parameter to be null
            const walker = document.createTreeWalker(fragment, 133 /* NodeFilter.SHOW_{ELEMENT|COMMENT|TEXT} */, null, false);
            let partIndex = 0;
            let nodeIndex = 0;
            let part;
            let node = walker.nextNode();
            // Loop through all the nodes and parts of a template
            while (partIndex < parts.length) {
                part = parts[partIndex];
                if (!isTemplatePartActive(part)) {
                    this.__parts.push(undefined);
                    partIndex++;
                    continue;
                }
                // Progress the tree walker until we find our next part's node.
                // Note that multiple parts may share the same node (attribute parts
                // on a single element), so this loop may not run at all.
                while (nodeIndex < part.index) {
                    nodeIndex++;
                    if (node.nodeName === 'TEMPLATE') {
                        stack.push(node);
                        walker.currentNode = node.content;
                    }
                    if ((node = walker.nextNode()) === null) {
                        // We've exhausted the content inside a nested template element.
                        // Because we still have parts (the outer for-loop), we know:
                        // - There is a template in the stack
                        // - The walker will find a nextNode outside the template
                        walker.currentNode = stack.pop();
                        node = walker.nextNode();
                    }
                }
                // We've arrived at our part's node.
                if (part.type === 'node') {
                    const part = this.processor.handleTextExpression(this.options);
                    part.insertAfterNode(node.previousSibling);
                    this.__parts.push(part);
                }
                else {
                    this.__parts.push(...this.processor.handleAttributeExpressions(node, part.name, part.strings, this.options));
                }
                partIndex++;
            }
            if (isCEPolyfill) {
                document.adoptNode(fragment);
                customElements.upgrade(fragment);
            }
            return fragment;
        }
    }

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    const commentMarker = ` ${marker} `;
    /**
     * The return type of `html`, which holds a Template and the values from
     * interpolated expressions.
     */
    class TemplateResult {
        constructor(strings, values, type, processor) {
            this.strings = strings;
            this.values = values;
            this.type = type;
            this.processor = processor;
        }
        /**
         * Returns a string of HTML used to create a `<template>` element.
         */
        getHTML() {
            const l = this.strings.length - 1;
            let html = '';
            let isCommentBinding = false;
            for (let i = 0; i < l; i++) {
                const s = this.strings[i];
                // For each binding we want to determine the kind of marker to insert
                // into the template source before it's parsed by the browser's HTML
                // parser. The marker type is based on whether the expression is in an
                // attribute, text, or comment position.
                //   * For node-position bindings we insert a comment with the marker
                //     sentinel as its text content, like <!--{{lit-guid}}-->.
                //   * For attribute bindings we insert just the marker sentinel for the
                //     first binding, so that we support unquoted attribute bindings.
                //     Subsequent bindings can use a comment marker because multi-binding
                //     attributes must be quoted.
                //   * For comment bindings we insert just the marker sentinel so we don't
                //     close the comment.
                //
                // The following code scans the template source, but is *not* an HTML
                // parser. We don't need to track the tree structure of the HTML, only
                // whether a binding is inside a comment, and if not, if it appears to be
                // the first binding in an attribute.
                const commentOpen = s.lastIndexOf('<!--');
                // We're in comment position if we have a comment open with no following
                // comment close. Because <-- can appear in an attribute value there can
                // be false positives.
                isCommentBinding = (commentOpen > -1 || isCommentBinding) &&
                    s.indexOf('-->', commentOpen + 1) === -1;
                // Check to see if we have an attribute-like sequence preceding the
                // expression. This can match "name=value" like structures in text,
                // comments, and attribute values, so there can be false-positives.
                const attributeMatch = lastAttributeNameRegex.exec(s);
                if (attributeMatch === null) {
                    // We're only in this branch if we don't have a attribute-like
                    // preceding sequence. For comments, this guards against unusual
                    // attribute values like <div foo="<!--${'bar'}">. Cases like
                    // <!-- foo=${'bar'}--> are handled correctly in the attribute branch
                    // below.
                    html += s + (isCommentBinding ? commentMarker : nodeMarker);
                }
                else {
                    // For attributes we use just a marker sentinel, and also append a
                    // $lit$ suffix to the name to opt-out of attribute-specific parsing
                    // that IE and Edge do for style and certain SVG attributes.
                    html += s.substr(0, attributeMatch.index) + attributeMatch[1] +
                        attributeMatch[2] + boundAttributeSuffix + attributeMatch[3] +
                        marker;
                }
            }
            html += this.strings[l];
            return html;
        }
        getTemplateElement() {
            const template = document.createElement('template');
            template.innerHTML = this.getHTML();
            return template;
        }
    }

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    const isPrimitive = (value) => {
        return (value === null ||
            !(typeof value === 'object' || typeof value === 'function'));
    };
    const isIterable = (value) => {
        return Array.isArray(value) ||
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            !!(value && value[Symbol.iterator]);
    };
    /**
     * Writes attribute values to the DOM for a group of AttributeParts bound to a
     * single attribute. The value is only set once even if there are multiple parts
     * for an attribute.
     */
    class AttributeCommitter {
        constructor(element, name, strings) {
            this.dirty = true;
            this.element = element;
            this.name = name;
            this.strings = strings;
            this.parts = [];
            for (let i = 0; i < strings.length - 1; i++) {
                this.parts[i] = this._createPart();
            }
        }
        /**
         * Creates a single part. Override this to create a differnt type of part.
         */
        _createPart() {
            return new AttributePart(this);
        }
        _getValue() {
            const strings = this.strings;
            const l = strings.length - 1;
            let text = '';
            for (let i = 0; i < l; i++) {
                text += strings[i];
                const part = this.parts[i];
                if (part !== undefined) {
                    const v = part.value;
                    if (isPrimitive(v) || !isIterable(v)) {
                        text += typeof v === 'string' ? v : String(v);
                    }
                    else {
                        for (const t of v) {
                            text += typeof t === 'string' ? t : String(t);
                        }
                    }
                }
            }
            text += strings[l];
            return text;
        }
        commit() {
            if (this.dirty) {
                this.dirty = false;
                this.element.setAttribute(this.name, this._getValue());
            }
        }
    }
    /**
     * A Part that controls all or part of an attribute value.
     */
    class AttributePart {
        constructor(committer) {
            this.value = undefined;
            this.committer = committer;
        }
        setValue(value) {
            if (value !== noChange && (!isPrimitive(value) || value !== this.value)) {
                this.value = value;
                // If the value is a not a directive, dirty the committer so that it'll
                // call setAttribute. If the value is a directive, it'll dirty the
                // committer if it calls setValue().
                if (!isDirective(value)) {
                    this.committer.dirty = true;
                }
            }
        }
        commit() {
            while (isDirective(this.value)) {
                const directive = this.value;
                this.value = noChange;
                directive(this);
            }
            if (this.value === noChange) {
                return;
            }
            this.committer.commit();
        }
    }
    /**
     * A Part that controls a location within a Node tree. Like a Range, NodePart
     * has start and end locations and can set and update the Nodes between those
     * locations.
     *
     * NodeParts support several value types: primitives, Nodes, TemplateResults,
     * as well as arrays and iterables of those types.
     */
    class NodePart {
        constructor(options) {
            this.value = undefined;
            this.__pendingValue = undefined;
            this.options = options;
        }
        /**
         * Appends this part into a container.
         *
         * This part must be empty, as its contents are not automatically moved.
         */
        appendInto(container) {
            this.startNode = container.appendChild(createMarker());
            this.endNode = container.appendChild(createMarker());
        }
        /**
         * Inserts this part after the `ref` node (between `ref` and `ref`'s next
         * sibling). Both `ref` and its next sibling must be static, unchanging nodes
         * such as those that appear in a literal section of a template.
         *
         * This part must be empty, as its contents are not automatically moved.
         */
        insertAfterNode(ref) {
            this.startNode = ref;
            this.endNode = ref.nextSibling;
        }
        /**
         * Appends this part into a parent part.
         *
         * This part must be empty, as its contents are not automatically moved.
         */
        appendIntoPart(part) {
            part.__insert(this.startNode = createMarker());
            part.__insert(this.endNode = createMarker());
        }
        /**
         * Inserts this part after the `ref` part.
         *
         * This part must be empty, as its contents are not automatically moved.
         */
        insertAfterPart(ref) {
            ref.__insert(this.startNode = createMarker());
            this.endNode = ref.endNode;
            ref.endNode = this.startNode;
        }
        setValue(value) {
            this.__pendingValue = value;
        }
        commit() {
            if (this.startNode.parentNode === null) {
                return;
            }
            while (isDirective(this.__pendingValue)) {
                const directive = this.__pendingValue;
                this.__pendingValue = noChange;
                directive(this);
            }
            const value = this.__pendingValue;
            if (value === noChange) {
                return;
            }
            if (isPrimitive(value)) {
                if (value !== this.value) {
                    this.__commitText(value);
                }
            }
            else if (value instanceof TemplateResult) {
                this.__commitTemplateResult(value);
            }
            else if (value instanceof Node) {
                this.__commitNode(value);
            }
            else if (isIterable(value)) {
                this.__commitIterable(value);
            }
            else if (value === nothing) {
                this.value = nothing;
                this.clear();
            }
            else {
                // Fallback, will render the string representation
                this.__commitText(value);
            }
        }
        __insert(node) {
            this.endNode.parentNode.insertBefore(node, this.endNode);
        }
        __commitNode(value) {
            if (this.value === value) {
                return;
            }
            this.clear();
            this.__insert(value);
            this.value = value;
        }
        __commitText(value) {
            const node = this.startNode.nextSibling;
            value = value == null ? '' : value;
            // If `value` isn't already a string, we explicitly convert it here in case
            // it can't be implicitly converted - i.e. it's a symbol.
            const valueAsString = typeof value === 'string' ? value : String(value);
            if (node === this.endNode.previousSibling &&
                node.nodeType === 3 /* Node.TEXT_NODE */) {
                // If we only have a single text node between the markers, we can just
                // set its value, rather than replacing it.
                // TODO(justinfagnani): Can we just check if this.value is primitive?
                node.data = valueAsString;
            }
            else {
                this.__commitNode(document.createTextNode(valueAsString));
            }
            this.value = value;
        }
        __commitTemplateResult(value) {
            const template = this.options.templateFactory(value);
            if (this.value instanceof TemplateInstance &&
                this.value.template === template) {
                this.value.update(value.values);
            }
            else {
                // Make sure we propagate the template processor from the TemplateResult
                // so that we use its syntax extension, etc. The template factory comes
                // from the render function options so that it can control template
                // caching and preprocessing.
                const instance = new TemplateInstance(template, value.processor, this.options);
                const fragment = instance._clone();
                instance.update(value.values);
                this.__commitNode(fragment);
                this.value = instance;
            }
        }
        __commitIterable(value) {
            // For an Iterable, we create a new InstancePart per item, then set its
            // value to the item. This is a little bit of overhead for every item in
            // an Iterable, but it lets us recurse easily and efficiently update Arrays
            // of TemplateResults that will be commonly returned from expressions like:
            // array.map((i) => html`${i}`), by reusing existing TemplateInstances.
            // If _value is an array, then the previous render was of an
            // iterable and _value will contain the NodeParts from the previous
            // render. If _value is not an array, clear this part and make a new
            // array for NodeParts.
            if (!Array.isArray(this.value)) {
                this.value = [];
                this.clear();
            }
            // Lets us keep track of how many items we stamped so we can clear leftover
            // items from a previous render
            const itemParts = this.value;
            let partIndex = 0;
            let itemPart;
            for (const item of value) {
                // Try to reuse an existing part
                itemPart = itemParts[partIndex];
                // If no existing part, create a new one
                if (itemPart === undefined) {
                    itemPart = new NodePart(this.options);
                    itemParts.push(itemPart);
                    if (partIndex === 0) {
                        itemPart.appendIntoPart(this);
                    }
                    else {
                        itemPart.insertAfterPart(itemParts[partIndex - 1]);
                    }
                }
                itemPart.setValue(item);
                itemPart.commit();
                partIndex++;
            }
            if (partIndex < itemParts.length) {
                // Truncate the parts array so _value reflects the current state
                itemParts.length = partIndex;
                this.clear(itemPart && itemPart.endNode);
            }
        }
        clear(startNode = this.startNode) {
            removeNodes(this.startNode.parentNode, startNode.nextSibling, this.endNode);
        }
    }
    /**
     * Implements a boolean attribute, roughly as defined in the HTML
     * specification.
     *
     * If the value is truthy, then the attribute is present with a value of
     * ''. If the value is falsey, the attribute is removed.
     */
    class BooleanAttributePart {
        constructor(element, name, strings) {
            this.value = undefined;
            this.__pendingValue = undefined;
            if (strings.length !== 2 || strings[0] !== '' || strings[1] !== '') {
                throw new Error('Boolean attributes can only contain a single expression');
            }
            this.element = element;
            this.name = name;
            this.strings = strings;
        }
        setValue(value) {
            this.__pendingValue = value;
        }
        commit() {
            while (isDirective(this.__pendingValue)) {
                const directive = this.__pendingValue;
                this.__pendingValue = noChange;
                directive(this);
            }
            if (this.__pendingValue === noChange) {
                return;
            }
            const value = !!this.__pendingValue;
            if (this.value !== value) {
                if (value) {
                    this.element.setAttribute(this.name, '');
                }
                else {
                    this.element.removeAttribute(this.name);
                }
                this.value = value;
            }
            this.__pendingValue = noChange;
        }
    }
    /**
     * Sets attribute values for PropertyParts, so that the value is only set once
     * even if there are multiple parts for a property.
     *
     * If an expression controls the whole property value, then the value is simply
     * assigned to the property under control. If there are string literals or
     * multiple expressions, then the strings are expressions are interpolated into
     * a string first.
     */
    class PropertyCommitter extends AttributeCommitter {
        constructor(element, name, strings) {
            super(element, name, strings);
            this.single =
                (strings.length === 2 && strings[0] === '' && strings[1] === '');
        }
        _createPart() {
            return new PropertyPart(this);
        }
        _getValue() {
            if (this.single) {
                return this.parts[0].value;
            }
            return super._getValue();
        }
        commit() {
            if (this.dirty) {
                this.dirty = false;
                // eslint-disable-next-line @typescript-eslint/no-explicit-any
                this.element[this.name] = this._getValue();
            }
        }
    }
    class PropertyPart extends AttributePart {
    }
    // Detect event listener options support. If the `capture` property is read
    // from the options object, then options are supported. If not, then the third
    // argument to add/removeEventListener is interpreted as the boolean capture
    // value so we should only pass the `capture` property.
    let eventOptionsSupported = false;
    // Wrap into an IIFE because MS Edge <= v41 does not support having try/catch
    // blocks right into the body of a module
    (() => {
        try {
            const options = {
                get capture() {
                    eventOptionsSupported = true;
                    return false;
                }
            };
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            window.addEventListener('test', options, options);
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            window.removeEventListener('test', options, options);
        }
        catch (_e) {
            // event options not supported
        }
    })();
    class EventPart {
        constructor(element, eventName, eventContext) {
            this.value = undefined;
            this.__pendingValue = undefined;
            this.element = element;
            this.eventName = eventName;
            this.eventContext = eventContext;
            this.__boundHandleEvent = (e) => this.handleEvent(e);
        }
        setValue(value) {
            this.__pendingValue = value;
        }
        commit() {
            while (isDirective(this.__pendingValue)) {
                const directive = this.__pendingValue;
                this.__pendingValue = noChange;
                directive(this);
            }
            if (this.__pendingValue === noChange) {
                return;
            }
            const newListener = this.__pendingValue;
            const oldListener = this.value;
            const shouldRemoveListener = newListener == null ||
                oldListener != null &&
                    (newListener.capture !== oldListener.capture ||
                        newListener.once !== oldListener.once ||
                        newListener.passive !== oldListener.passive);
            const shouldAddListener = newListener != null && (oldListener == null || shouldRemoveListener);
            if (shouldRemoveListener) {
                this.element.removeEventListener(this.eventName, this.__boundHandleEvent, this.__options);
            }
            if (shouldAddListener) {
                this.__options = getOptions(newListener);
                this.element.addEventListener(this.eventName, this.__boundHandleEvent, this.__options);
            }
            this.value = newListener;
            this.__pendingValue = noChange;
        }
        handleEvent(event) {
            if (typeof this.value === 'function') {
                this.value.call(this.eventContext || this.element, event);
            }
            else {
                this.value.handleEvent(event);
            }
        }
    }
    // We copy options because of the inconsistent behavior of browsers when reading
    // the third argument of add/removeEventListener. IE11 doesn't support options
    // at all. Chrome 41 only reads `capture` if the argument is an object.
    const getOptions = (o) => o &&
        (eventOptionsSupported ?
            { capture: o.capture, passive: o.passive, once: o.once } :
            o.capture);

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    /**
     * The default TemplateFactory which caches Templates keyed on
     * result.type and result.strings.
     */
    function templateFactory(result) {
        let templateCache = templateCaches.get(result.type);
        if (templateCache === undefined) {
            templateCache = {
                stringsArray: new WeakMap(),
                keyString: new Map()
            };
            templateCaches.set(result.type, templateCache);
        }
        let template = templateCache.stringsArray.get(result.strings);
        if (template !== undefined) {
            return template;
        }
        // If the TemplateStringsArray is new, generate a key from the strings
        // This key is shared between all templates with identical content
        const key = result.strings.join(marker);
        // Check if we already have a Template for this key
        template = templateCache.keyString.get(key);
        if (template === undefined) {
            // If we have not seen this key before, create a new Template
            template = new Template(result, result.getTemplateElement());
            // Cache the Template for this key
            templateCache.keyString.set(key, template);
        }
        // Cache all future queries for this TemplateStringsArray
        templateCache.stringsArray.set(result.strings, template);
        return template;
    }
    const templateCaches = new Map();

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    const parts = new WeakMap();
    /**
     * Renders a template result or other value to a container.
     *
     * To update a container with new values, reevaluate the template literal and
     * call `render` with the new result.
     *
     * @param result Any value renderable by NodePart - typically a TemplateResult
     *     created by evaluating a template tag like `html` or `svg`.
     * @param container A DOM parent to render to. The entire contents are either
     *     replaced, or efficiently updated if the same result type was previous
     *     rendered there.
     * @param options RenderOptions for the entire render tree rendered to this
     *     container. Render options must *not* change between renders to the same
     *     container, as those changes will not effect previously rendered DOM.
     */
    const render = (result, container, options) => {
        let part = parts.get(container);
        if (part === undefined) {
            removeNodes(container, container.firstChild);
            parts.set(container, part = new NodePart(Object.assign({ templateFactory }, options)));
            part.appendInto(container);
        }
        part.setValue(result);
        part.commit();
    };

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    /**
     * Creates Parts when a template is instantiated.
     */
    class DefaultTemplateProcessor {
        /**
         * Create parts for an attribute-position binding, given the event, attribute
         * name, and string literals.
         *
         * @param element The element containing the binding
         * @param name  The attribute name
         * @param strings The string literals. There are always at least two strings,
         *   event for fully-controlled bindings with a single expression.
         */
        handleAttributeExpressions(element, name, strings, options) {
            const prefix = name[0];
            if (prefix === '.') {
                const committer = new PropertyCommitter(element, name.slice(1), strings);
                return committer.parts;
            }
            if (prefix === '@') {
                return [new EventPart(element, name.slice(1), options.eventContext)];
            }
            if (prefix === '?') {
                return [new BooleanAttributePart(element, name.slice(1), strings)];
            }
            const committer = new AttributeCommitter(element, name, strings);
            return committer.parts;
        }
        /**
         * Create parts for a text-position binding.
         * @param templateFactory
         */
        handleTextExpression(options) {
            return new NodePart(options);
        }
    }
    const defaultTemplateProcessor = new DefaultTemplateProcessor();

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    // IMPORTANT: do not change the property name or the assignment expression.
    // This line will be used in regexes to search for lit-html usage.
    // TODO(justinfagnani): inject version number at build time
    if (typeof window !== 'undefined') {
        (window['litHtmlVersions'] || (window['litHtmlVersions'] = [])).push('1.2.1');
    }
    /**
     * Interprets a template literal as an HTML template that can efficiently
     * render to and update a container.
     */
    const html = (strings, ...values) => new TemplateResult(strings, values, 'html', defaultTemplateProcessor);

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    // Get a key to lookup in `templateCaches`.
    const getTemplateCacheKey = (type, scopeName) => `${type}--${scopeName}`;
    let compatibleShadyCSSVersion = true;
    if (typeof window.ShadyCSS === 'undefined') {
        compatibleShadyCSSVersion = false;
    }
    else if (typeof window.ShadyCSS.prepareTemplateDom === 'undefined') {
        console.warn(`Incompatible ShadyCSS version detected. ` +
            `Please update to at least @webcomponents/webcomponentsjs@2.0.2 and ` +
            `@webcomponents/shadycss@1.3.1.`);
        compatibleShadyCSSVersion = false;
    }
    /**
     * Template factory which scopes template DOM using ShadyCSS.
     * @param scopeName {string}
     */
    const shadyTemplateFactory = (scopeName) => (result) => {
        const cacheKey = getTemplateCacheKey(result.type, scopeName);
        let templateCache = templateCaches.get(cacheKey);
        if (templateCache === undefined) {
            templateCache = {
                stringsArray: new WeakMap(),
                keyString: new Map()
            };
            templateCaches.set(cacheKey, templateCache);
        }
        let template = templateCache.stringsArray.get(result.strings);
        if (template !== undefined) {
            return template;
        }
        const key = result.strings.join(marker);
        template = templateCache.keyString.get(key);
        if (template === undefined) {
            const element = result.getTemplateElement();
            if (compatibleShadyCSSVersion) {
                window.ShadyCSS.prepareTemplateDom(element, scopeName);
            }
            template = new Template(result, element);
            templateCache.keyString.set(key, template);
        }
        templateCache.stringsArray.set(result.strings, template);
        return template;
    };
    const TEMPLATE_TYPES = ['html', 'svg'];
    /**
     * Removes all style elements from Templates for the given scopeName.
     */
    const removeStylesFromLitTemplates = (scopeName) => {
        TEMPLATE_TYPES.forEach((type) => {
            const templates = templateCaches.get(getTemplateCacheKey(type, scopeName));
            if (templates !== undefined) {
                templates.keyString.forEach((template) => {
                    const { element: { content } } = template;
                    // IE 11 doesn't support the iterable param Set constructor
                    const styles = new Set();
                    Array.from(content.querySelectorAll('style')).forEach((s) => {
                        styles.add(s);
                    });
                    removeNodesFromTemplate(template, styles);
                });
            }
        });
    };
    const shadyRenderSet = new Set();
    /**
     * For the given scope name, ensures that ShadyCSS style scoping is performed.
     * This is done just once per scope name so the fragment and template cannot
     * be modified.
     * (1) extracts styles from the rendered fragment and hands them to ShadyCSS
     * to be scoped and appended to the document
     * (2) removes style elements from all lit-html Templates for this scope name.
     *
     * Note, <style> elements can only be placed into templates for the
     * initial rendering of the scope. If <style> elements are included in templates
     * dynamically rendered to the scope (after the first scope render), they will
     * not be scoped and the <style> will be left in the template and rendered
     * output.
     */
    const prepareTemplateStyles = (scopeName, renderedDOM, template) => {
        shadyRenderSet.add(scopeName);
        // If `renderedDOM` is stamped from a Template, then we need to edit that
        // Template's underlying template element. Otherwise, we create one here
        // to give to ShadyCSS, which still requires one while scoping.
        const templateElement = !!template ? template.element : document.createElement('template');
        // Move styles out of rendered DOM and store.
        const styles = renderedDOM.querySelectorAll('style');
        const { length } = styles;
        // If there are no styles, skip unnecessary work
        if (length === 0) {
            // Ensure prepareTemplateStyles is called to support adding
            // styles via `prepareAdoptedCssText` since that requires that
            // `prepareTemplateStyles` is called.
            //
            // ShadyCSS will only update styles containing @apply in the template
            // given to `prepareTemplateStyles`. If no lit Template was given,
            // ShadyCSS will not be able to update uses of @apply in any relevant
            // template. However, this is not a problem because we only create the
            // template for the purpose of supporting `prepareAdoptedCssText`,
            // which doesn't support @apply at all.
            window.ShadyCSS.prepareTemplateStyles(templateElement, scopeName);
            return;
        }
        const condensedStyle = document.createElement('style');
        // Collect styles into a single style. This helps us make sure ShadyCSS
        // manipulations will not prevent us from being able to fix up template
        // part indices.
        // NOTE: collecting styles is inefficient for browsers but ShadyCSS
        // currently does this anyway. When it does not, this should be changed.
        for (let i = 0; i < length; i++) {
            const style = styles[i];
            style.parentNode.removeChild(style);
            condensedStyle.textContent += style.textContent;
        }
        // Remove styles from nested templates in this scope.
        removeStylesFromLitTemplates(scopeName);
        // And then put the condensed style into the "root" template passed in as
        // `template`.
        const content = templateElement.content;
        if (!!template) {
            insertNodeIntoTemplate(template, condensedStyle, content.firstChild);
        }
        else {
            content.insertBefore(condensedStyle, content.firstChild);
        }
        // Note, it's important that ShadyCSS gets the template that `lit-html`
        // will actually render so that it can update the style inside when
        // needed (e.g. @apply native Shadow DOM case).
        window.ShadyCSS.prepareTemplateStyles(templateElement, scopeName);
        const style = content.querySelector('style');
        if (window.ShadyCSS.nativeShadow && style !== null) {
            // When in native Shadow DOM, ensure the style created by ShadyCSS is
            // included in initially rendered output (`renderedDOM`).
            renderedDOM.insertBefore(style.cloneNode(true), renderedDOM.firstChild);
        }
        else if (!!template) {
            // When no style is left in the template, parts will be broken as a
            // result. To fix this, we put back the style node ShadyCSS removed
            // and then tell lit to remove that node from the template.
            // There can be no style in the template in 2 cases (1) when Shady DOM
            // is in use, ShadyCSS removes all styles, (2) when native Shadow DOM
            // is in use ShadyCSS removes the style if it contains no content.
            // NOTE, ShadyCSS creates its own style so we can safely add/remove
            // `condensedStyle` here.
            content.insertBefore(condensedStyle, content.firstChild);
            const removes = new Set();
            removes.add(condensedStyle);
            removeNodesFromTemplate(template, removes);
        }
    };
    /**
     * Extension to the standard `render` method which supports rendering
     * to ShadowRoots when the ShadyDOM (https://github.com/webcomponents/shadydom)
     * and ShadyCSS (https://github.com/webcomponents/shadycss) polyfills are used
     * or when the webcomponentsjs
     * (https://github.com/webcomponents/webcomponentsjs) polyfill is used.
     *
     * Adds a `scopeName` option which is used to scope element DOM and stylesheets
     * when native ShadowDOM is unavailable. The `scopeName` will be added to
     * the class attribute of all rendered DOM. In addition, any style elements will
     * be automatically re-written with this `scopeName` selector and moved out
     * of the rendered DOM and into the document `<head>`.
     *
     * It is common to use this render method in conjunction with a custom element
     * which renders a shadowRoot. When this is done, typically the element's
     * `localName` should be used as the `scopeName`.
     *
     * In addition to DOM scoping, ShadyCSS also supports a basic shim for css
     * custom properties (needed only on older browsers like IE11) and a shim for
     * a deprecated feature called `@apply` that supports applying a set of css
     * custom properties to a given location.
     *
     * Usage considerations:
     *
     * * Part values in `<style>` elements are only applied the first time a given
     * `scopeName` renders. Subsequent changes to parts in style elements will have
     * no effect. Because of this, parts in style elements should only be used for
     * values that will never change, for example parts that set scope-wide theme
     * values or parts which render shared style elements.
     *
     * * Note, due to a limitation of the ShadyDOM polyfill, rendering in a
     * custom element's `constructor` is not supported. Instead rendering should
     * either done asynchronously, for example at microtask timing (for example
     * `Promise.resolve()`), or be deferred until the first time the element's
     * `connectedCallback` runs.
     *
     * Usage considerations when using shimmed custom properties or `@apply`:
     *
     * * Whenever any dynamic changes are made which affect
     * css custom properties, `ShadyCSS.styleElement(element)` must be called
     * to update the element. There are two cases when this is needed:
     * (1) the element is connected to a new parent, (2) a class is added to the
     * element that causes it to match different custom properties.
     * To address the first case when rendering a custom element, `styleElement`
     * should be called in the element's `connectedCallback`.
     *
     * * Shimmed custom properties may only be defined either for an entire
     * shadowRoot (for example, in a `:host` rule) or via a rule that directly
     * matches an element with a shadowRoot. In other words, instead of flowing from
     * parent to child as do native css custom properties, shimmed custom properties
     * flow only from shadowRoots to nested shadowRoots.
     *
     * * When using `@apply` mixing css shorthand property names with
     * non-shorthand names (for example `border` and `border-width`) is not
     * supported.
     */
    const render$1 = (result, container, options) => {
        if (!options || typeof options !== 'object' || !options.scopeName) {
            throw new Error('The `scopeName` option is required.');
        }
        const scopeName = options.scopeName;
        const hasRendered = parts.has(container);
        const needsScoping = compatibleShadyCSSVersion &&
            container.nodeType === 11 /* Node.DOCUMENT_FRAGMENT_NODE */ &&
            !!container.host;
        // Handle first render to a scope specially...
        const firstScopeRender = needsScoping && !shadyRenderSet.has(scopeName);
        // On first scope render, render into a fragment; this cannot be a single
        // fragment that is reused since nested renders can occur synchronously.
        const renderContainer = firstScopeRender ? document.createDocumentFragment() : container;
        render(result, renderContainer, Object.assign({ templateFactory: shadyTemplateFactory(scopeName) }, options));
        // When performing first scope render,
        // (1) We've rendered into a fragment so that there's a chance to
        // `prepareTemplateStyles` before sub-elements hit the DOM
        // (which might cause them to render based on a common pattern of
        // rendering in a custom element's `connectedCallback`);
        // (2) Scope the template with ShadyCSS one time only for this scope.
        // (3) Render the fragment into the container and make sure the
        // container knows its `part` is the one we just rendered. This ensures
        // DOM will be re-used on subsequent renders.
        if (firstScopeRender) {
            const part = parts.get(renderContainer);
            parts.delete(renderContainer);
            // ShadyCSS might have style sheets (e.g. from `prepareAdoptedCssText`)
            // that should apply to `renderContainer` even if the rendered value is
            // not a TemplateInstance. However, it will only insert scoped styles
            // into the document if `prepareTemplateStyles` has already been called
            // for the given scope name.
            const template = part.value instanceof TemplateInstance ?
                part.value.template :
                undefined;
            prepareTemplateStyles(scopeName, renderContainer, template);
            removeNodes(container, container.firstChild);
            container.appendChild(renderContainer);
            parts.set(container, part);
        }
        // After elements have hit the DOM, update styling if this is the
        // initial render to this container.
        // This is needed whenever dynamic changes are made so it would be
        // safest to do every render; however, this would regress performance
        // so we leave it up to the user to call `ShadyCSS.styleElement`
        // for dynamic changes.
        if (!hasRendered && needsScoping) {
            window.ShadyCSS.styleElement(container.host);
        }
    };

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    var _a;
    /**
     * When using Closure Compiler, JSCompiler_renameProperty(property, object) is
     * replaced at compile time by the munged name for object[property]. We cannot
     * alias this function, so we have to use a small shim that has the same
     * behavior when not compiling.
     */
    window.JSCompiler_renameProperty =
        (prop, _obj) => prop;
    const defaultConverter = {
        toAttribute(value, type) {
            switch (type) {
                case Boolean:
                    return value ? '' : null;
                case Object:
                case Array:
                    // if the value is `null` or `undefined` pass this through
                    // to allow removing/no change behavior.
                    return value == null ? value : JSON.stringify(value);
            }
            return value;
        },
        fromAttribute(value, type) {
            switch (type) {
                case Boolean:
                    return value !== null;
                case Number:
                    return value === null ? null : Number(value);
                case Object:
                case Array:
                    return JSON.parse(value);
            }
            return value;
        }
    };
    /**
     * Change function that returns true if `value` is different from `oldValue`.
     * This method is used as the default for a property's `hasChanged` function.
     */
    const notEqual = (value, old) => {
        // This ensures (old==NaN, value==NaN) always returns false
        return old !== value && (old === old || value === value);
    };
    const defaultPropertyDeclaration = {
        attribute: true,
        type: String,
        converter: defaultConverter,
        reflect: false,
        hasChanged: notEqual
    };
    const STATE_HAS_UPDATED = 1;
    const STATE_UPDATE_REQUESTED = 1 << 2;
    const STATE_IS_REFLECTING_TO_ATTRIBUTE = 1 << 3;
    const STATE_IS_REFLECTING_TO_PROPERTY = 1 << 4;
    /**
     * The Closure JS Compiler doesn't currently have good support for static
     * property semantics where "this" is dynamic (e.g.
     * https://github.com/google/closure-compiler/issues/3177 and others) so we use
     * this hack to bypass any rewriting by the compiler.
     */
    const finalized = 'finalized';
    /**
     * Base element class which manages element properties and attributes. When
     * properties change, the `update` method is asynchronously called. This method
     * should be supplied by subclassers to render updates as desired.
     */
    class UpdatingElement extends HTMLElement {
        constructor() {
            super();
            this._updateState = 0;
            this._instanceProperties = undefined;
            // Initialize to an unresolved Promise so we can make sure the element has
            // connected before first update.
            this._updatePromise = new Promise((res) => this._enableUpdatingResolver = res);
            /**
             * Map with keys for any properties that have changed since the last
             * update cycle with previous values.
             */
            this._changedProperties = new Map();
            /**
             * Map with keys of properties that should be reflected when updated.
             */
            this._reflectingProperties = undefined;
            this.initialize();
        }
        /**
         * Returns a list of attributes corresponding to the registered properties.
         * @nocollapse
         */
        static get observedAttributes() {
            // note: piggy backing on this to ensure we're finalized.
            this.finalize();
            const attributes = [];
            // Use forEach so this works even if for/of loops are compiled to for loops
            // expecting arrays
            this._classProperties.forEach((v, p) => {
                const attr = this._attributeNameForProperty(p, v);
                if (attr !== undefined) {
                    this._attributeToPropertyMap.set(attr, p);
                    attributes.push(attr);
                }
            });
            return attributes;
        }
        /**
         * Ensures the private `_classProperties` property metadata is created.
         * In addition to `finalize` this is also called in `createProperty` to
         * ensure the `@property` decorator can add property metadata.
         */
        /** @nocollapse */
        static _ensureClassProperties() {
            // ensure private storage for property declarations.
            if (!this.hasOwnProperty(JSCompiler_renameProperty('_classProperties', this))) {
                this._classProperties = new Map();
                // NOTE: Workaround IE11 not supporting Map constructor argument.
                const superProperties = Object.getPrototypeOf(this)._classProperties;
                if (superProperties !== undefined) {
                    superProperties.forEach((v, k) => this._classProperties.set(k, v));
                }
            }
        }
        /**
         * Creates a property accessor on the element prototype if one does not exist
         * and stores a PropertyDeclaration for the property with the given options.
         * The property setter calls the property's `hasChanged` property option
         * or uses a strict identity check to determine whether or not to request
         * an update.
         *
         * This method may be overridden to customize properties; however,
         * when doing so, it's important to call `super.createProperty` to ensure
         * the property is setup correctly. This method calls
         * `getPropertyDescriptor` internally to get a descriptor to install.
         * To customize what properties do when they are get or set, override
         * `getPropertyDescriptor`. To customize the options for a property,
         * implement `createProperty` like this:
         *
         * static createProperty(name, options) {
         *   options = Object.assign(options, {myOption: true});
         *   super.createProperty(name, options);
         * }
         *
         * @nocollapse
         */
        static createProperty(name, options = defaultPropertyDeclaration) {
            // Note, since this can be called by the `@property` decorator which
            // is called before `finalize`, we ensure storage exists for property
            // metadata.
            this._ensureClassProperties();
            this._classProperties.set(name, options);
            // Do not generate an accessor if the prototype already has one, since
            // it would be lost otherwise and that would never be the user's intention;
            // Instead, we expect users to call `requestUpdate` themselves from
            // user-defined accessors. Note that if the super has an accessor we will
            // still overwrite it
            if (options.noAccessor || this.prototype.hasOwnProperty(name)) {
                return;
            }
            const key = typeof name === 'symbol' ? Symbol() : `__${name}`;
            const descriptor = this.getPropertyDescriptor(name, key, options);
            if (descriptor !== undefined) {
                Object.defineProperty(this.prototype, name, descriptor);
            }
        }
        /**
         * Returns a property descriptor to be defined on the given named property.
         * If no descriptor is returned, the property will not become an accessor.
         * For example,
         *
         *   class MyElement extends LitElement {
         *     static getPropertyDescriptor(name, key, options) {
         *       const defaultDescriptor =
         *           super.getPropertyDescriptor(name, key, options);
         *       const setter = defaultDescriptor.set;
         *       return {
         *         get: defaultDescriptor.get,
         *         set(value) {
         *           setter.call(this, value);
         *           // custom action.
         *         },
         *         configurable: true,
         *         enumerable: true
         *       }
         *     }
         *   }
         *
         * @nocollapse
         */
        static getPropertyDescriptor(name, key, _options) {
            return {
                // tslint:disable-next-line:no-any no symbol in index
                get() {
                    return this[key];
                },
                set(value) {
                    const oldValue = this[name];
                    this[key] = value;
                    this._requestUpdate(name, oldValue);
                },
                configurable: true,
                enumerable: true
            };
        }
        /**
         * Returns the property options associated with the given property.
         * These options are defined with a PropertyDeclaration via the `properties`
         * object or the `@property` decorator and are registered in
         * `createProperty(...)`.
         *
         * Note, this method should be considered "final" and not overridden. To
         * customize the options for a given property, override `createProperty`.
         *
         * @nocollapse
         * @final
         */
        static getPropertyOptions(name) {
            return this._classProperties && this._classProperties.get(name) ||
                defaultPropertyDeclaration;
        }
        /**
         * Creates property accessors for registered properties and ensures
         * any superclasses are also finalized.
         * @nocollapse
         */
        static finalize() {
            // finalize any superclasses
            const superCtor = Object.getPrototypeOf(this);
            if (!superCtor.hasOwnProperty(finalized)) {
                superCtor.finalize();
            }
            this[finalized] = true;
            this._ensureClassProperties();
            // initialize Map populated in observedAttributes
            this._attributeToPropertyMap = new Map();
            // make any properties
            // Note, only process "own" properties since this element will inherit
            // any properties defined on the superClass, and finalization ensures
            // the entire prototype chain is finalized.
            if (this.hasOwnProperty(JSCompiler_renameProperty('properties', this))) {
                const props = this.properties;
                // support symbols in properties (IE11 does not support this)
                const propKeys = [
                    ...Object.getOwnPropertyNames(props),
                    ...(typeof Object.getOwnPropertySymbols === 'function') ?
                        Object.getOwnPropertySymbols(props) :
                        []
                ];
                // This for/of is ok because propKeys is an array
                for (const p of propKeys) {
                    // note, use of `any` is due to TypeSript lack of support for symbol in
                    // index types
                    // tslint:disable-next-line:no-any no symbol in index
                    this.createProperty(p, props[p]);
                }
            }
        }
        /**
         * Returns the property name for the given attribute `name`.
         * @nocollapse
         */
        static _attributeNameForProperty(name, options) {
            const attribute = options.attribute;
            return attribute === false ?
                undefined :
                (typeof attribute === 'string' ?
                    attribute :
                    (typeof name === 'string' ? name.toLowerCase() : undefined));
        }
        /**
         * Returns true if a property should request an update.
         * Called when a property value is set and uses the `hasChanged`
         * option for the property if present or a strict identity check.
         * @nocollapse
         */
        static _valueHasChanged(value, old, hasChanged = notEqual) {
            return hasChanged(value, old);
        }
        /**
         * Returns the property value for the given attribute value.
         * Called via the `attributeChangedCallback` and uses the property's
         * `converter` or `converter.fromAttribute` property option.
         * @nocollapse
         */
        static _propertyValueFromAttribute(value, options) {
            const type = options.type;
            const converter = options.converter || defaultConverter;
            const fromAttribute = (typeof converter === 'function' ? converter : converter.fromAttribute);
            return fromAttribute ? fromAttribute(value, type) : value;
        }
        /**
         * Returns the attribute value for the given property value. If this
         * returns undefined, the property will *not* be reflected to an attribute.
         * If this returns null, the attribute will be removed, otherwise the
         * attribute will be set to the value.
         * This uses the property's `reflect` and `type.toAttribute` property options.
         * @nocollapse
         */
        static _propertyValueToAttribute(value, options) {
            if (options.reflect === undefined) {
                return;
            }
            const type = options.type;
            const converter = options.converter;
            const toAttribute = converter && converter.toAttribute ||
                defaultConverter.toAttribute;
            return toAttribute(value, type);
        }
        /**
         * Performs element initialization. By default captures any pre-set values for
         * registered properties.
         */
        initialize() {
            this._saveInstanceProperties();
            // ensures first update will be caught by an early access of
            // `updateComplete`
            this._requestUpdate();
        }
        /**
         * Fixes any properties set on the instance before upgrade time.
         * Otherwise these would shadow the accessor and break these properties.
         * The properties are stored in a Map which is played back after the
         * constructor runs. Note, on very old versions of Safari (<=9) or Chrome
         * (<=41), properties created for native platform properties like (`id` or
         * `name`) may not have default values set in the element constructor. On
         * these browsers native properties appear on instances and therefore their
         * default value will overwrite any element default (e.g. if the element sets
         * this.id = 'id' in the constructor, the 'id' will become '' since this is
         * the native platform default).
         */
        _saveInstanceProperties() {
            // Use forEach so this works even if for/of loops are compiled to for loops
            // expecting arrays
            this.constructor
                ._classProperties.forEach((_v, p) => {
                if (this.hasOwnProperty(p)) {
                    const value = this[p];
                    delete this[p];
                    if (!this._instanceProperties) {
                        this._instanceProperties = new Map();
                    }
                    this._instanceProperties.set(p, value);
                }
            });
        }
        /**
         * Applies previously saved instance properties.
         */
        _applyInstanceProperties() {
            // Use forEach so this works even if for/of loops are compiled to for loops
            // expecting arrays
            // tslint:disable-next-line:no-any
            this._instanceProperties.forEach((v, p) => this[p] = v);
            this._instanceProperties = undefined;
        }
        connectedCallback() {
            // Ensure first connection completes an update. Updates cannot complete
            // before connection.
            this.enableUpdating();
        }
        enableUpdating() {
            if (this._enableUpdatingResolver !== undefined) {
                this._enableUpdatingResolver();
                this._enableUpdatingResolver = undefined;
            }
        }
        /**
         * Allows for `super.disconnectedCallback()` in extensions while
         * reserving the possibility of making non-breaking feature additions
         * when disconnecting at some point in the future.
         */
        disconnectedCallback() {
        }
        /**
         * Synchronizes property values when attributes change.
         */
        attributeChangedCallback(name, old, value) {
            if (old !== value) {
                this._attributeToProperty(name, value);
            }
        }
        _propertyToAttribute(name, value, options = defaultPropertyDeclaration) {
            const ctor = this.constructor;
            const attr = ctor._attributeNameForProperty(name, options);
            if (attr !== undefined) {
                const attrValue = ctor._propertyValueToAttribute(value, options);
                // an undefined value does not change the attribute.
                if (attrValue === undefined) {
                    return;
                }
                // Track if the property is being reflected to avoid
                // setting the property again via `attributeChangedCallback`. Note:
                // 1. this takes advantage of the fact that the callback is synchronous.
                // 2. will behave incorrectly if multiple attributes are in the reaction
                // stack at time of calling. However, since we process attributes
                // in `update` this should not be possible (or an extreme corner case
                // that we'd like to discover).
                // mark state reflecting
                this._updateState = this._updateState | STATE_IS_REFLECTING_TO_ATTRIBUTE;
                if (attrValue == null) {
                    this.removeAttribute(attr);
                }
                else {
                    this.setAttribute(attr, attrValue);
                }
                // mark state not reflecting
                this._updateState = this._updateState & ~STATE_IS_REFLECTING_TO_ATTRIBUTE;
            }
        }
        _attributeToProperty(name, value) {
            // Use tracking info to avoid deserializing attribute value if it was
            // just set from a property setter.
            if (this._updateState & STATE_IS_REFLECTING_TO_ATTRIBUTE) {
                return;
            }
            const ctor = this.constructor;
            // Note, hint this as an `AttributeMap` so closure clearly understands
            // the type; it has issues with tracking types through statics
            // tslint:disable-next-line:no-unnecessary-type-assertion
            const propName = ctor._attributeToPropertyMap.get(name);
            if (propName !== undefined) {
                const options = ctor.getPropertyOptions(propName);
                // mark state reflecting
                this._updateState = this._updateState | STATE_IS_REFLECTING_TO_PROPERTY;
                this[propName] =
                    // tslint:disable-next-line:no-any
                    ctor._propertyValueFromAttribute(value, options);
                // mark state not reflecting
                this._updateState = this._updateState & ~STATE_IS_REFLECTING_TO_PROPERTY;
            }
        }
        /**
         * This private version of `requestUpdate` does not access or return the
         * `updateComplete` promise. This promise can be overridden and is therefore
         * not free to access.
         */
        _requestUpdate(name, oldValue) {
            let shouldRequestUpdate = true;
            // If we have a property key, perform property update steps.
            if (name !== undefined) {
                const ctor = this.constructor;
                const options = ctor.getPropertyOptions(name);
                if (ctor._valueHasChanged(this[name], oldValue, options.hasChanged)) {
                    if (!this._changedProperties.has(name)) {
                        this._changedProperties.set(name, oldValue);
                    }
                    // Add to reflecting properties set.
                    // Note, it's important that every change has a chance to add the
                    // property to `_reflectingProperties`. This ensures setting
                    // attribute + property reflects correctly.
                    if (options.reflect === true &&
                        !(this._updateState & STATE_IS_REFLECTING_TO_PROPERTY)) {
                        if (this._reflectingProperties === undefined) {
                            this._reflectingProperties = new Map();
                        }
                        this._reflectingProperties.set(name, options);
                    }
                }
                else {
                    // Abort the request if the property should not be considered changed.
                    shouldRequestUpdate = false;
                }
            }
            if (!this._hasRequestedUpdate && shouldRequestUpdate) {
                this._updatePromise = this._enqueueUpdate();
            }
        }
        /**
         * Requests an update which is processed asynchronously. This should
         * be called when an element should update based on some state not triggered
         * by setting a property. In this case, pass no arguments. It should also be
         * called when manually implementing a property setter. In this case, pass the
         * property `name` and `oldValue` to ensure that any configured property
         * options are honored. Returns the `updateComplete` Promise which is resolved
         * when the update completes.
         *
         * @param name {PropertyKey} (optional) name of requesting property
         * @param oldValue {any} (optional) old value of requesting property
         * @returns {Promise} A Promise that is resolved when the update completes.
         */
        requestUpdate(name, oldValue) {
            this._requestUpdate(name, oldValue);
            return this.updateComplete;
        }
        /**
         * Sets up the element to asynchronously update.
         */
        async _enqueueUpdate() {
            this._updateState = this._updateState | STATE_UPDATE_REQUESTED;
            try {
                // Ensure any previous update has resolved before updating.
                // This `await` also ensures that property changes are batched.
                await this._updatePromise;
            }
            catch (e) {
                // Ignore any previous errors. We only care that the previous cycle is
                // done. Any error should have been handled in the previous update.
            }
            const result = this.performUpdate();
            // If `performUpdate` returns a Promise, we await it. This is done to
            // enable coordinating updates with a scheduler. Note, the result is
            // checked to avoid delaying an additional microtask unless we need to.
            if (result != null) {
                await result;
            }
            return !this._hasRequestedUpdate;
        }
        get _hasRequestedUpdate() {
            return (this._updateState & STATE_UPDATE_REQUESTED);
        }
        get hasUpdated() {
            return (this._updateState & STATE_HAS_UPDATED);
        }
        /**
         * Performs an element update. Note, if an exception is thrown during the
         * update, `firstUpdated` and `updated` will not be called.
         *
         * You can override this method to change the timing of updates. If this
         * method is overridden, `super.performUpdate()` must be called.
         *
         * For instance, to schedule updates to occur just before the next frame:
         *
         * ```
         * protected async performUpdate(): Promise<unknown> {
         *   await new Promise((resolve) => requestAnimationFrame(() => resolve()));
         *   super.performUpdate();
         * }
         * ```
         */
        performUpdate() {
            // Mixin instance properties once, if they exist.
            if (this._instanceProperties) {
                this._applyInstanceProperties();
            }
            let shouldUpdate = false;
            const changedProperties = this._changedProperties;
            try {
                shouldUpdate = this.shouldUpdate(changedProperties);
                if (shouldUpdate) {
                    this.update(changedProperties);
                }
                else {
                    this._markUpdated();
                }
            }
            catch (e) {
                // Prevent `firstUpdated` and `updated` from running when there's an
                // update exception.
                shouldUpdate = false;
                // Ensure element can accept additional updates after an exception.
                this._markUpdated();
                throw e;
            }
            if (shouldUpdate) {
                if (!(this._updateState & STATE_HAS_UPDATED)) {
                    this._updateState = this._updateState | STATE_HAS_UPDATED;
                    this.firstUpdated(changedProperties);
                }
                this.updated(changedProperties);
            }
        }
        _markUpdated() {
            this._changedProperties = new Map();
            this._updateState = this._updateState & ~STATE_UPDATE_REQUESTED;
        }
        /**
         * Returns a Promise that resolves when the element has completed updating.
         * The Promise value is a boolean that is `true` if the element completed the
         * update without triggering another update. The Promise result is `false` if
         * a property was set inside `updated()`. If the Promise is rejected, an
         * exception was thrown during the update.
         *
         * To await additional asynchronous work, override the `_getUpdateComplete`
         * method. For example, it is sometimes useful to await a rendered element
         * before fulfilling this Promise. To do this, first await
         * `super._getUpdateComplete()`, then any subsequent state.
         *
         * @returns {Promise} The Promise returns a boolean that indicates if the
         * update resolved without triggering another update.
         */
        get updateComplete() {
            return this._getUpdateComplete();
        }
        /**
         * Override point for the `updateComplete` promise.
         *
         * It is not safe to override the `updateComplete` getter directly due to a
         * limitation in TypeScript which means it is not possible to call a
         * superclass getter (e.g. `super.updateComplete.then(...)`) when the target
         * language is ES5 (https://github.com/microsoft/TypeScript/issues/338).
         * This method should be overridden instead. For example:
         *
         *   class MyElement extends LitElement {
         *     async _getUpdateComplete() {
         *       await super._getUpdateComplete();
         *       await this._myChild.updateComplete;
         *     }
         *   }
         */
        _getUpdateComplete() {
            return this._updatePromise;
        }
        /**
         * Controls whether or not `update` should be called when the element requests
         * an update. By default, this method always returns `true`, but this can be
         * customized to control when to update.
         *
         * @param _changedProperties Map of changed properties with old values
         */
        shouldUpdate(_changedProperties) {
            return true;
        }
        /**
         * Updates the element. This method reflects property values to attributes.
         * It can be overridden to render and keep updated element DOM.
         * Setting properties inside this method will *not* trigger
         * another update.
         *
         * @param _changedProperties Map of changed properties with old values
         */
        update(_changedProperties) {
            if (this._reflectingProperties !== undefined &&
                this._reflectingProperties.size > 0) {
                // Use forEach so this works even if for/of loops are compiled to for
                // loops expecting arrays
                this._reflectingProperties.forEach((v, k) => this._propertyToAttribute(k, this[k], v));
                this._reflectingProperties = undefined;
            }
            this._markUpdated();
        }
        /**
         * Invoked whenever the element is updated. Implement to perform
         * post-updating tasks via DOM APIs, for example, focusing an element.
         *
         * Setting properties inside this method will trigger the element to update
         * again after this update cycle completes.
         *
         * @param _changedProperties Map of changed properties with old values
         */
        updated(_changedProperties) {
        }
        /**
         * Invoked when the element is first updated. Implement to perform one time
         * work on the element after update.
         *
         * Setting properties inside this method will trigger the element to update
         * again after this update cycle completes.
         *
         * @param _changedProperties Map of changed properties with old values
         */
        firstUpdated(_changedProperties) {
        }
    }
    _a = finalized;
    /**
     * Marks class as having finished creating properties.
     */
    UpdatingElement[_a] = true;

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    const legacyCustomElement = (tagName, clazz) => {
        window.customElements.define(tagName, clazz);
        // Cast as any because TS doesn't recognize the return type as being a
        // subtype of the decorated class when clazz is typed as
        // `Constructor<HTMLElement>` for some reason.
        // `Constructor<HTMLElement>` is helpful to make sure the decorator is
        // applied to elements however.
        // tslint:disable-next-line:no-any
        return clazz;
    };
    const standardCustomElement = (tagName, descriptor) => {
        const { kind, elements } = descriptor;
        return {
            kind,
            elements,
            // This callback is called once the class is otherwise fully defined
            finisher(clazz) {
                window.customElements.define(tagName, clazz);
            }
        };
    };
    /**
     * Class decorator factory that defines the decorated class as a custom element.
     *
     * ```
     * @customElement('my-element')
     * class MyElement {
     *   render() {
     *     return html``;
     *   }
     * }
     * ```
     *
     * @param tagName The name of the custom element to define.
     */
    const customElement = (tagName) => (classOrDescriptor) => (typeof classOrDescriptor === 'function') ?
        legacyCustomElement(tagName, classOrDescriptor) :
        standardCustomElement(tagName, classOrDescriptor);
    const standardProperty = (options, element) => {
        // When decorating an accessor, pass it through and add property metadata.
        // Note, the `hasOwnProperty` check in `createProperty` ensures we don't
        // stomp over the user's accessor.
        if (element.kind === 'method' && element.descriptor &&
            !('value' in element.descriptor)) {
            return Object.assign(Object.assign({}, element), { finisher(clazz) {
                    clazz.createProperty(element.key, options);
                } });
        }
        else {
            // createProperty() takes care of defining the property, but we still
            // must return some kind of descriptor, so return a descriptor for an
            // unused prototype field. The finisher calls createProperty().
            return {
                kind: 'field',
                key: Symbol(),
                placement: 'own',
                descriptor: {},
                // When @babel/plugin-proposal-decorators implements initializers,
                // do this instead of the initializer below. See:
                // https://github.com/babel/babel/issues/9260 extras: [
                //   {
                //     kind: 'initializer',
                //     placement: 'own',
                //     initializer: descriptor.initializer,
                //   }
                // ],
                initializer() {
                    if (typeof element.initializer === 'function') {
                        this[element.key] = element.initializer.call(this);
                    }
                },
                finisher(clazz) {
                    clazz.createProperty(element.key, options);
                }
            };
        }
    };
    const legacyProperty = (options, proto, name) => {
        proto.constructor
            .createProperty(name, options);
    };
    /**
     * A property decorator which creates a LitElement property which reflects a
     * corresponding attribute value. A `PropertyDeclaration` may optionally be
     * supplied to configure property features.
     *
     * This decorator should only be used for public fields. Private or protected
     * fields should use the internalProperty decorator.
     *
     * @example
     *
     *     class MyElement {
     *       @property({ type: Boolean })
     *       clicked = false;
     *     }
     *
     * @ExportDecoratedItems
     */
    function property(options) {
        // tslint:disable-next-line:no-any decorator
        return (protoOrDescriptor, name) => (name !== undefined) ?
            legacyProperty(options, protoOrDescriptor, name) :
            standardProperty(options, protoOrDescriptor);
    }
    /**
     * A property decorator that converts a class property into a getter that
     * executes a querySelector on the element's renderRoot.
     *
     * @param selector A DOMString containing one or more selectors to match.
     *
     * See: https://developer.mozilla.org/en-US/docs/Web/API/Document/querySelector
     *
     * @example
     *
     *     class MyElement {
     *       @query('#first')
     *       first;
     *
     *       render() {
     *         return html`
     *           <div id="first"></div>
     *           <div id="second"></div>
     *         `;
     *       }
     *     }
     *
     */
    function query(selector) {
        return (protoOrDescriptor, 
        // tslint:disable-next-line:no-any decorator
        name) => {
            const descriptor = {
                get() {
                    return this.renderRoot.querySelector(selector);
                },
                enumerable: true,
                configurable: true,
            };
            return (name !== undefined) ?
                legacyQuery(descriptor, protoOrDescriptor, name) :
                standardQuery(descriptor, protoOrDescriptor);
        };
    }
    const legacyQuery = (descriptor, proto, name) => {
        Object.defineProperty(proto, name, descriptor);
    };
    const standardQuery = (descriptor, element) => ({
        kind: 'method',
        placement: 'prototype',
        key: element.key,
        descriptor,
    });

    /**
    @license
    Copyright (c) 2019 The Polymer Project Authors. All rights reserved.
    This code may only be used under the BSD style license found at
    http://polymer.github.io/LICENSE.txt The complete set of authors may be found at
    http://polymer.github.io/AUTHORS.txt The complete set of contributors may be
    found at http://polymer.github.io/CONTRIBUTORS.txt Code distributed by Google as
    part of the polymer project is also subject to an additional IP rights grant
    found at http://polymer.github.io/PATENTS.txt
    */
    const supportsAdoptingStyleSheets = ('adoptedStyleSheets' in Document.prototype) &&
        ('replace' in CSSStyleSheet.prototype);
    const constructionToken = Symbol();
    class CSSResult {
        constructor(cssText, safeToken) {
            if (safeToken !== constructionToken) {
                throw new Error('CSSResult is not constructable. Use `unsafeCSS` or `css` instead.');
            }
            this.cssText = cssText;
        }
        // Note, this is a getter so that it's lazy. In practice, this means
        // stylesheets are not created until the first element instance is made.
        get styleSheet() {
            if (this._styleSheet === undefined) {
                // Note, if `adoptedStyleSheets` is supported then we assume CSSStyleSheet
                // is constructable.
                if (supportsAdoptingStyleSheets) {
                    this._styleSheet = new CSSStyleSheet();
                    this._styleSheet.replaceSync(this.cssText);
                }
                else {
                    this._styleSheet = null;
                }
            }
            return this._styleSheet;
        }
        toString() {
            return this.cssText;
        }
    }
    const textFromCSSResult = (value) => {
        if (value instanceof CSSResult) {
            return value.cssText;
        }
        else if (typeof value === 'number') {
            return value;
        }
        else {
            throw new Error(`Value passed to 'css' function must be a 'css' function result: ${value}. Use 'unsafeCSS' to pass non-literal values, but
            take care to ensure page security.`);
        }
    };
    /**
     * Template tag which which can be used with LitElement's `style` property to
     * set element styles. For security reasons, only literal string values may be
     * used. To incorporate non-literal values `unsafeCSS` may be used inside a
     * template string part.
     */
    const css = (strings, ...values) => {
        const cssText = values.reduce((acc, v, idx) => acc + textFromCSSResult(v) + strings[idx + 1], strings[0]);
        return new CSSResult(cssText, constructionToken);
    };

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    // IMPORTANT: do not change the property name or the assignment expression.
    // This line will be used in regexes to search for LitElement usage.
    // TODO(justinfagnani): inject version number at build time
    (window['litElementVersions'] || (window['litElementVersions'] = []))
        .push('2.3.1');
    /**
     * Sentinal value used to avoid calling lit-html's render function when
     * subclasses do not implement `render`
     */
    const renderNotImplemented = {};
    class LitElement extends UpdatingElement {
        /**
         * Return the array of styles to apply to the element.
         * Override this method to integrate into a style management system.
         *
         * @nocollapse
         */
        static getStyles() {
            return this.styles;
        }
        /** @nocollapse */
        static _getUniqueStyles() {
            // Only gather styles once per class
            if (this.hasOwnProperty(JSCompiler_renameProperty('_styles', this))) {
                return;
            }
            // Take care not to call `this.getStyles()` multiple times since this
            // generates new CSSResults each time.
            // TODO(sorvell): Since we do not cache CSSResults by input, any
            // shared styles will generate new stylesheet objects, which is wasteful.
            // This should be addressed when a browser ships constructable
            // stylesheets.
            const userStyles = this.getStyles();
            if (userStyles === undefined) {
                this._styles = [];
            }
            else if (Array.isArray(userStyles)) {
                // De-duplicate styles preserving the _last_ instance in the set.
                // This is a performance optimization to avoid duplicated styles that can
                // occur especially when composing via subclassing.
                // The last item is kept to try to preserve the cascade order with the
                // assumption that it's most important that last added styles override
                // previous styles.
                const addStyles = (styles, set) => styles.reduceRight((set, s) => 
                // Note: On IE set.add() does not return the set
                Array.isArray(s) ? addStyles(s, set) : (set.add(s), set), set);
                // Array.from does not work on Set in IE, otherwise return
                // Array.from(addStyles(userStyles, new Set<CSSResult>())).reverse()
                const set = addStyles(userStyles, new Set());
                const styles = [];
                set.forEach((v) => styles.unshift(v));
                this._styles = styles;
            }
            else {
                this._styles = [userStyles];
            }
        }
        /**
         * Performs element initialization. By default this calls `createRenderRoot`
         * to create the element `renderRoot` node and captures any pre-set values for
         * registered properties.
         */
        initialize() {
            super.initialize();
            this.constructor._getUniqueStyles();
            this.renderRoot =
                this.createRenderRoot();
            // Note, if renderRoot is not a shadowRoot, styles would/could apply to the
            // element's getRootNode(). While this could be done, we're choosing not to
            // support this now since it would require different logic around de-duping.
            if (window.ShadowRoot && this.renderRoot instanceof window.ShadowRoot) {
                this.adoptStyles();
            }
        }
        /**
         * Returns the node into which the element should render and by default
         * creates and returns an open shadowRoot. Implement to customize where the
         * element's DOM is rendered. For example, to render into the element's
         * childNodes, return `this`.
         * @returns {Element|DocumentFragment} Returns a node into which to render.
         */
        createRenderRoot() {
            return this.attachShadow({ mode: 'open' });
        }
        /**
         * Applies styling to the element shadowRoot using the `static get styles`
         * property. Styling will apply using `shadowRoot.adoptedStyleSheets` where
         * available and will fallback otherwise. When Shadow DOM is polyfilled,
         * ShadyCSS scopes styles and adds them to the document. When Shadow DOM
         * is available but `adoptedStyleSheets` is not, styles are appended to the
         * end of the `shadowRoot` to [mimic spec
         * behavior](https://wicg.github.io/construct-stylesheets/#using-constructed-stylesheets).
         */
        adoptStyles() {
            const styles = this.constructor._styles;
            if (styles.length === 0) {
                return;
            }
            // There are three separate cases here based on Shadow DOM support.
            // (1) shadowRoot polyfilled: use ShadyCSS
            // (2) shadowRoot.adoptedStyleSheets available: use it.
            // (3) shadowRoot.adoptedStyleSheets polyfilled: append styles after
            // rendering
            if (window.ShadyCSS !== undefined && !window.ShadyCSS.nativeShadow) {
                window.ShadyCSS.ScopingShim.prepareAdoptedCssText(styles.map((s) => s.cssText), this.localName);
            }
            else if (supportsAdoptingStyleSheets) {
                this.renderRoot.adoptedStyleSheets =
                    styles.map((s) => s.styleSheet);
            }
            else {
                // This must be done after rendering so the actual style insertion is done
                // in `update`.
                this._needsShimAdoptedStyleSheets = true;
            }
        }
        connectedCallback() {
            super.connectedCallback();
            // Note, first update/render handles styleElement so we only call this if
            // connected after first update.
            if (this.hasUpdated && window.ShadyCSS !== undefined) {
                window.ShadyCSS.styleElement(this);
            }
        }
        /**
         * Updates the element. This method reflects property values to attributes
         * and calls `render` to render DOM via lit-html. Setting properties inside
         * this method will *not* trigger another update.
         * @param _changedProperties Map of changed properties with old values
         */
        update(changedProperties) {
            // Setting properties in `render` should not trigger an update. Since
            // updates are allowed after super.update, it's important to call `render`
            // before that.
            const templateResult = this.render();
            super.update(changedProperties);
            // If render is not implemented by the component, don't call lit-html render
            if (templateResult !== renderNotImplemented) {
                this.constructor
                    .render(templateResult, this.renderRoot, { scopeName: this.localName, eventContext: this });
            }
            // When native Shadow DOM is used but adoptedStyles are not supported,
            // insert styling after rendering to ensure adoptedStyles have highest
            // priority.
            if (this._needsShimAdoptedStyleSheets) {
                this._needsShimAdoptedStyleSheets = false;
                this.constructor._styles.forEach((s) => {
                    const style = document.createElement('style');
                    style.textContent = s.cssText;
                    this.renderRoot.appendChild(style);
                });
            }
        }
        /**
         * Invoked on each update to perform rendering tasks. This method may return
         * any value renderable by lit-html's NodePart - typically a TemplateResult.
         * Setting properties inside this method will *not* trigger the element to
         * update.
         */
        render() {
            return renderNotImplemented;
        }
    }
    /**
     * Ensure this class is marked as `finalized` as an optimization ensuring
     * it will not needlessly try to `finalize`.
     *
     * Note this property name is a string to prevent breaking Closure JS Compiler
     * optimizations. See updating-element.ts for more information.
     */
    LitElement['finalized'] = true;
    /**
     * Render method used to render the value to the element's DOM.
     * @param result The value to render.
     * @param container Node into which to render.
     * @param options Element name.
     * @nocollapse
     */
    LitElement.render = render$1;

    function hass() {
      return document.querySelector('home-assistant').hass
    }
    function load_lovelace() {
      if(customElements.get("hui-view")) return true;

      const res = document.createElement("partial-panel-resolver");
      res.hass = hass();
      res.route = {path: "/lovelace/"};
      // res._updateRoutes();
      try {
        document.querySelector("home-assistant").appendChild(res).catch((error) => {});
      } catch (error) {
        document.querySelector("home-assistant").removeChild(res);
      }
      if(customElements.get("hui-view")) return true;
      return false;

    }

    const getConfiguration = async (hass) => {
        const response = await hass.connection.sendMessagePromise({
            type: "hacs/config",
        });
        return response;
    };
    const getRepositories = async (hass) => {
        const response = await hass.connection.sendMessagePromise({
            type: "hacs/repositories",
        });
        return response;
    };
    const getCritical = async (hass) => {
        const response = await hass.connection.sendMessagePromise({
            type: "hacs/get_critical",
        });
        return response;
    };
    const getStatus = async (hass) => {
        const response = await hass.connection.sendMessagePromise({
            type: "hacs/status",
        });
        return response;
    };
    const repositoryInstall = async (hass, repository) => {
        await hass.connection.sendMessagePromise({
            type: "hacs/repository",
            action: "install",
            repository: repository,
        });
    };
    const repositoryUninstall = async (hass, repository) => {
        await hass.connection.sendMessagePromise({
            type: "hacs/repository",
            action: "uninstall",
            repository: repository,
        });
    };
    const repositoryToggleBeta = async (hass, repository) => {
        await hass.connection.sendMessagePromise({
            type: "hacs/repository",
            action: "toggle_beta",
            repository: repository,
        });
    };
    const repositoryInstallVersion = async (hass, repository, version) => {
        await hass.connection.sendMessagePromise({
            type: "hacs/repository/data",
            action: "install",
            repository: repository,
            data: version,
        });
    };
    const repositoryAdd = async (hass, repository, category) => {
        await hass.connection.sendMessagePromise({
            type: "hacs/repository/data",
            action: "add",
            repository: repository,
            data: category,
        });
    };
    const repositorySetNotNew = async (hass, repository) => {
        await hass.connection.sendMessagePromise({
            type: "hacs/repository",
            action: "not_new",
            repository: repository,
        });
    };
    const repositoryUpdate = async (hass, repository) => {
        await hass.connection.sendMessagePromise({
            type: "hacs/repository",
            action: "update",
            repository: repository,
        });
    };
    const repositoryDelete = async (hass, repository) => {
        await hass.connection.sendMessagePromise({
            type: "hacs/repository",
            action: "delete",
            repository: repository,
        });
    };
    const settingsClearAllNewRepositories = async (hass) => {
        await hass.connection.sendMessagePromise({
            type: "hacs/settings",
            action: "clear_new",
        });
    };
    const getLovelaceConfiguration = async (hass) => {
        try {
            const response = await hass.connection.sendMessagePromise({
                type: "lovelace/resources",
            });
            return response;
        }
        catch (e) {
            return null;
        }
    };
    const fetchResources = (hass) => hass.connection.sendMessagePromise({
        type: "lovelace/resources",
    });
    const createResource = (hass, values) => hass.callWS(Object.assign({ type: "lovelace/resources/create" }, values));
    const updateResource = (hass, values) => hass.callWS(Object.assign({ type: "lovelace/resources/update" }, values));
    const deleteResource = (hass, id) => hass.callWS({
        type: "lovelace/resources/delete",
        resource_id: id,
    });

    var common = {
    	about: "Om",
    	add: "tilføj",
    	appdaemon_apps: "AppDaemon-apps",
    	appdaemon_plural: "AppDaemon-apps",
    	background_task: "Baggrundsopgave kører. Denne side vil genindlæses automatisk.",
    	check_log_file: "Tjek din logfil for flere detaljer.",
    	"continue": "Fortsæt",
    	disabled: "Deaktiveret",
    	documentation: "Dokumentation",
    	element: "element",
    	hacs_is_disabled: "HACS er deaktiveret",
    	install: "Installer",
    	installed: "installeret",
    	integration: "Integration",
    	integration_plural: "Integrationer",
    	integrations: "Integrationer",
    	lovelace: "Lovelace",
    	lovelace_element: "Lovelace-element",
    	lovelace_elements: "Lovelace-elementer",
    	manage: "administrer",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDaemon-apps",
    	netdaemon_plural: "NetDaemon-apps",
    	plugin: "Lovelace",
    	plugin_plural: "Lovelace-elementer",
    	plugins: "Lovelace-elementer",
    	python_script: "Python-script",
    	python_script_plural: "Python-scripts",
    	python_scripts: "Python-scripts",
    	repositories: "Repositories",
    	repository: "Repository",
    	settings: "indstillinger",
    	theme: "Tema",
    	theme_plural: "Temaer",
    	themes: "Temaer",
    	uninstall: "Afinstaller",
    	version: "Version"
    };
    var config = {
    	abort: {
    		single_instance_allowed: "Kun en enkelt konfiguration af HACS er tilladt."
    	},
    	error: {
    		auth: "Personlig adgangstoken er ikke korrekt."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Aktiver opdagelse & sporing af AppDaemon-apps",
    				netdaemon: "Aktiver opdagelse og sporing af NetDaemon-apps",
    				python_script: "Aktivér opdagelse og sporing af python_scripts",
    				sidepanel_icon: "Sidepanelikon",
    				sidepanel_title: "Sidepanelets titel",
    				theme: "Aktivér opdagelse og sporing af temaer",
    				token: "GitHub personlig adgangstoken"
    			},
    			description: "Hvis du har brug for hjælp til konfigurationen, så kig her: https://hacs.xyz/docs/configuration/start"
    		}
    	}
    };
    var confirm = {
    	add_to_lovelace: "Er du sikker på, at du vil tilføje dette til dine Lovelace-ressourcer?",
    	bg_task: "Handlingen er deaktiveret, mens baggrundsopgaver kører.",
    	cancel: "Annuller",
    	"continue": "Er du sikker på, at du vil fortsætte?",
    	"delete": "Er du sikker på, at du vil slette '{Item}'?",
    	delete_installed: "'{item}' er installeret. Du skal afinstallere det, før du kan slette det.",
    	exist: "{item} findes allerede",
    	generic: "Er du sikker?",
    	home_assistant_is_restarting: "Vent venligst - Home Assistant genstarter nu.",
    	home_assistant_version_not_correct: "Du kører Home Assistant version '{haversion}', men dette repository kræver som minimum version '{minversion}'.",
    	no: "Nej",
    	no_upgrades: "Der er ingen opdateringer tilgængelig",
    	ok: "OK",
    	overwrite: "Dette vil overskrive den.",
    	reload_data: "Dette genindlæser data fra alle repositories, som HACS kender til. Dette vil tage nogen tid at fuldføre.",
    	restart_home_assistant: "Er du sikker på, at du vil genstarte Home Assistant?",
    	uninstall: "Er du sikker på, at du vil afinstallere '{Item}'?",
    	upgrade_all: "Dette vil opdatere alle repositories. Sørg for at du har læst udgivelsesnoterne for dem alle, inden du fortsætter.",
    	yes: "Ja"
    };
    var dialog_about = {
    	frontend_version: "Frontend-version",
    	installed_repositories: "Installerede repositories",
    	integration_version: "Integrationsversion",
    	useful_links: "Nyttige links"
    };
    var dialog_add_repo = {
    	limit: "Kun de første 100 repositories vises. Brug søgningen til at filtrere, hvad du har brug for",
    	no_match: "Der blev ikke fundet nogen repositories, der matcher dit filter",
    	sort_by: "Sorter efter",
    	title: "Tilføj repository"
    };
    var dialog_custom_repositories = {
    	category: "Kategori",
    	no_category: "Manglende kategori",
    	no_repository: "Manglende repository",
    	title: "Brugerdefinerede repositories",
    	url_placeholder: "Tilføj brugerdefineret repository-webadresse"
    };
    var dialog_info = {
    	author: "Udvikler",
    	downloads: "Downloads",
    	install: "Installer dette repository i HACS",
    	loading: "Indlæser oplysninger...",
    	no_info: "Udvikleren har ikke givet flere oplysninger om dette repository",
    	open_repo: "Åbn repository",
    	stars: "Stjerner",
    	version_installed: "Installeret version"
    };
    var dialog_install = {
    	restart: "Husk, at du skal genstarte Home Assistant, før ændringer af integrationer (custom_components) træder i kræft.",
    	select_version: "Vælg version",
    	show_beta: "Vis betaversioner",
    	type: "Type",
    	url: "Webadresse"
    };
    var dialog_update = {
    	available_version: "Tilgængelig version",
    	changelog: "Udgivelsesnoter",
    	installed_version: "Installeret version",
    	title: "Ventende opdatering"
    };
    var entry = {
    	information: "Oplysninger",
    	intro: "Opdateringer og vigtige meddelelser vises her, hvis der er nogen",
    	messages: {
    		disabled: {
    			content: "Tjek din logfil for flere detaljer",
    			title: "HACS er deaktiveret"
    		},
    		has_pending_tasks: {
    			content: "Nogle repositories vises muligvis ikke, før dette er fuldført",
    			title: "Baggrundsopgaver venter"
    		},
    		startup: {
    			content: "HACS starter op. Der kan i dette tidsrum mangle nogle oplysninger, eller de kan være ukorekte.",
    			title: "HACS starter op"
    		}
    	},
    	pending_updates: "Ventende opdateringer"
    };
    var menu = {
    	about: "Om HACS",
    	clear: "Ryd alle nye",
    	custom_repositories: "Brugerdefinerede repositories",
    	dismiss: "Afvis alle nye repositories",
    	documentation: "Dokumentation",
    	open_issue: "Opret issue",
    	reload: "Genindlæs vindue"
    };
    var options = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Aktiver opdagelse og sporing af AppDaemon-apps",
    				country: "Filtrer med landekode.",
    				debug: "Aktiver debug.",
    				experimental: "Aktivér eksperimentelle funktioner",
    				netdaemon: "Aktiver opdagelse og sporing af NetDaemon-apps",
    				not_in_use: "Ikke i brug med YAML",
    				release_limit: "Antal udgivelser, der skal vises.",
    				sidepanel_icon: "Sidepanelikon",
    				sidepanel_title: "Sidepanelets titel"
    			}
    		}
    	}
    };
    var repository_banner = {
    	config_flow: "Denne integration understøtter config_flow. Det betyder, at du nu kan gå til integrationssektionen i din brugerflade for at konfigurere den.",
    	config_flow_title: "Brugerfladekonfiguration understøttet",
    	integration_not_loaded: "Denne integration er ikke indlæst i Home Assistant.",
    	no_restart_required: "Ingen genstart påkrævet",
    	not_loaded: "Ikke indlæst",
    	plugin_not_loaded: "Dette element er ikke føjet til dine Lovelace-ressourcer.",
    	restart: "Du skal genstarte Home Assistant.",
    	restart_pending: "Afventer genstart"
    };
    var repository_card = {
    	dismiss: "Afvis",
    	hide: "Skjul",
    	information: "Oplysninger",
    	new_repository: "Nyt repository",
    	open_issue: "Opret issue",
    	pending_update: "Ventende opdatering",
    	reinstall: "Geninstaller",
    	report: "Rapporter til fjernelse"
    };
    var repository = {
    	add_to_lovelace: "Tilføj til Lovelace",
    	authors: "Forfattere",
    	available: "Tilgængelig",
    	back_to: "Tilbage til",
    	changelog: "Udgivelsesnoter",
    	downloads: "Downloads",
    	flag_this: "Marker denne",
    	frontend_version: "Frontend-version",
    	github_stars: "GitHub-stjerner",
    	goto_integrations: "Gå til integrationer",
    	hide: "Skjul",
    	hide_beta: "Skjul beta",
    	install: "Installer",
    	installed: "Installeret",
    	lovelace_copy_example: "Kopiér eksemplet til din Udklipsholder",
    	lovelace_instruction: "Tilføj dette til din lovelace-konfiguration",
    	lovelace_no_js_type: "Kunne ikke afgøre typen af dette element, tjek venligst repository'et.",
    	newest: "nyeste",
    	note_appdaemon: "Du skal stadig føje den til filen 'apps.yaml'",
    	note_installed: "Når det er installeret, vil dette være placeret i",
    	note_integration: "du skal stadig føje den til filen 'configuration.yaml'",
    	note_plugin: "du skal stadig tilføje det til din lovelace-konfiguration ('ui-lovelace.yaml' eller Tekstbaseret redigering)",
    	note_plugin_post_107: "du skal stadig tilføje det til din lovelace-konfiguration ('configuration.yaml' eller ressourceeditoren '/config/lovelace/resources')",
    	open_issue: "Opret issue",
    	open_plugin: "Åbn element",
    	reinstall: "Geninstaller",
    	repository: "Repository",
    	restart_home_assistant: "Genstart Home Assistant",
    	show_beta: "Vis beta",
    	uninstall: "Afinstaller",
    	update_information: "Opdater oplysninger",
    	upgrade: "Opdater"
    };
    var search = {
    	placeholder: "Søg efter repository"
    };
    var sections = {
    	about: {
    		description: "Vis information om HACS",
    		title: "Om"
    	},
    	automation: {
    		description: "Det er her, du finder python_scripts, AppDaemon-apps og NetDaemon-apps",
    		title: "Automatisering"
    	},
    	frontend: {
    		description: "Det er her, du finder temaer, brugerdefinerede kort og andre elementer til lovelace",
    		title: "Frontend"
    	},
    	integrations: {
    		description: "Det er her, du finder brugerdefinerede integrationer (custom_components)",
    		title: "Integrationer"
    	},
    	pending_repository_upgrade: "Du kører version {installed}, version {available} er tilgængelig"
    };
    var settings = {
    	add_custom_repository: "TILFØJ ET BRUGERDEFINERET REPOSITORY",
    	adding_new_repo: "Tilføjer nyt repository '{repo}'",
    	adding_new_repo_category: "Med kategorien '{category}'.",
    	bg_task_custom: "Brugerdefinerede repositories er skjult, mens opgaver i baggrunden kører.",
    	category: "Kategori",
    	compact_mode: "Kompakt tilstand",
    	custom_repositories: "BRUGERDEFINEREDE REPOSITORIES",
    	"delete": "Slet",
    	display: "Visning",
    	grid: "Gitter",
    	hacs_repo: "HACS-repo",
    	hidden_repositories: "Skjulte repositories",
    	missing_category: "Du skal vælge en kategori",
    	open_repository: "Åbn repository",
    	reload_data: "Genindlæs data",
    	reload_window: "Genindlæs vindue",
    	repository_configuration: "Konfiguration af repository",
    	save: "Gem",
    	table: "Tabel",
    	table_view: "Tabelvisning",
    	unhide: "Vis",
    	upgrade_all: "Opdater alle"
    };
    var store = {
    	ascending: "stigende",
    	clear_new: "Marker alle som set",
    	descending: "faldende",
    	last_updated: "Sidst opdateret",
    	name: "Navn",
    	new_repositories: "Nye repositories",
    	new_repositories_note: "Du har over 10 nye repositories, der vises her, hvis du vil rydde dem alle, skal du klikke på de 3 prikker i øverste højre hjørne og afvise dem alle.",
    	no_repositories: "Ingen repositories",
    	no_repositories_desc1: "Det ser ud til, at du ikke har nogen repositories installeret i denne sektion endnu.",
    	no_repositories_desc2: "Klik på + i nederste hjørne for at tilføje dit første!",
    	pending_upgrades: "Afventende opdateringer",
    	placeholder_search: "Indtast en søgeterm...",
    	sort: "sorter",
    	stars: "Stjerner",
    	status: "Status"
    };
    var time = {
    	ago: "siden",
    	day: "dag",
    	days: "dage",
    	hour: "time",
    	hours: "timer",
    	minute: "minut",
    	minutes: "minutter",
    	month: "måned",
    	months: "måneder",
    	one: "Et",
    	one_day_ago: "en dag siden",
    	one_hour_ago: "en time siden",
    	one_minute_ago: "et minut siden",
    	one_month_ago: "en måned siden",
    	one_second_ago: "et sekund siden",
    	one_year_ago: "et år siden",
    	second: "sekund",
    	seconds: "sekunder",
    	x_days_ago: "{x} dage siden",
    	x_hours_ago: "{x} timer siden",
    	x_minutes_ago: "{x} minutter siden",
    	x_months_ago: "{x} måneder siden",
    	x_seconds_ago: "{x} sekunder siden",
    	x_years_ago: "{x} år siden",
    	year: "år",
    	years: "år"
    };
    var da = {
    	common: common,
    	config: config,
    	confirm: confirm,
    	dialog_about: dialog_about,
    	dialog_add_repo: dialog_add_repo,
    	dialog_custom_repositories: dialog_custom_repositories,
    	dialog_info: dialog_info,
    	dialog_install: dialog_install,
    	dialog_update: dialog_update,
    	entry: entry,
    	menu: menu,
    	options: options,
    	repository_banner: repository_banner,
    	repository_card: repository_card,
    	repository: repository,
    	search: search,
    	sections: sections,
    	settings: settings,
    	store: store,
    	time: time
    };

    var da$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common,
        config: config,
        confirm: confirm,
        dialog_about: dialog_about,
        dialog_add_repo: dialog_add_repo,
        dialog_custom_repositories: dialog_custom_repositories,
        dialog_info: dialog_info,
        dialog_install: dialog_install,
        dialog_update: dialog_update,
        entry: entry,
        menu: menu,
        options: options,
        repository_banner: repository_banner,
        repository_card: repository_card,
        repository: repository,
        search: search,
        sections: sections,
        settings: settings,
        store: store,
        time: time,
        'default': da
    });

    var common$1 = {
    	about: "Über",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "AppDaemon Apps",
    	appdaemon_plural: "AppDaemon Apps",
    	background_task: "Hintergrundprozess läuft. Diese Seite lädt neu, sobald dieser fertig ist.",
    	check_log_file: "Überprüfe die Logdatei für weitere Informationen.",
    	"continue": "Fortfahren",
    	disabled: "Deaktiviert",
    	documentation: "Dokumentation",
    	element: "Element",
    	hacs_is_disabled: "HACS ist deaktiviert",
    	installed: "Installiert",
    	integration: "Integration",
    	integration_plural: "Integrationen",
    	integrations: "Integrationen",
    	lovelace: "Lovelace",
    	lovelace_element: "Lovelace-Element",
    	lovelace_elements: "Lovelace-Elemente",
    	manage: "verwalten",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDaemon Apps",
    	netdaemon_plural: "NetDaemon Apps",
    	plugin: "Lovelace",
    	plugin_plural: "Lovelace-Elemente",
    	plugins: "Lovelace-Elemente",
    	python_script: "Python Skript",
    	python_script_plural: "Python Skripte",
    	python_scripts: "Python Skripte",
    	repositories: "Repositories",
    	settings: "Einstellungen",
    	theme: "Theme",
    	theme_plural: "Themes",
    	themes: "Themes",
    	version: "Version"
    };
    var config$1 = {
    	abort: {
    		single_instance_allowed: "Es ist nur eine einzelne HACS-Instanz erlaubt."
    	},
    	error: {
    		auth: "Persönlicher Zugriffstoken ist falsch."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "AppDaemon App-Entdeckung & Nachverfolgung aktivieren",
    				netdaemon: "NetDaemon App-Entdeckung & Nachverfolgung aktivieren",
    				python_script: "Python Script-Entdeckung & Nachverfolgung aktivieren",
    				sidepanel_icon: "Sidepanel Symbol",
    				sidepanel_title: "Sidepanel Titel",
    				theme: "Theme-Entdeckung & Nachverfolgung aktivieren",
    				token: "Persönlicher GitHub Zugriffstoken"
    			},
    			description: "Wenn du Hilfe mit den Einstellungen brauchst, kannst du hier nachsehen: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$1 = {
    	add_to_lovelace: "Möchtest du dies wirklich zu deinen Lovelace-Ressourcen hinzufügen?",
    	bg_task: "Die Aktion ist deaktiviert, während Hintergrundprozesse ausgeführt werden.",
    	cancel: "Abbrechen",
    	"continue": "Bist du dir sicher, dass du fortfahren möchtest?",
    	"delete": "Möchtest du '{item}' wirklich löschen?",
    	delete_installed: "'{item}' ist installiert. Du musst es deinstallieren, bevor du es löschen kannst.",
    	exist: "{item} existiert bereits",
    	generic: "Bist du dir sicher?",
    	home_assistant_is_restarting: "Bitte warte, Home Assistant wird jetzt neu gestartet.",
    	home_assistant_version_not_correct: "Du benutzt die Home Assistant-Version '{haversion}', für dieses Repository muss jedoch die Mindestversion '{minversion}' installiert sein.",
    	no: "Nein",
    	no_upgrades: "Keine Upgrades ausstehend",
    	ok: "OK",
    	overwrite: "Dadurch wird die Datei überschrieben.",
    	reload_data: "Hierdurch werden die Daten aller Repositories die HACS kennt neu geladen. Dies wird einige Zeit in Anspruch nehmen.",
    	restart_home_assistant: "Bist du dir sicher, dass du Home Assistant neu starten möchtest?",
    	uninstall: "Möchtest du '{item}' wirklich deinstallieren?",
    	upgrade_all: "Hierdurch werden all diese Repositories aktualisiert. Stelle sicher, dass du die Versionshinweise vorher gelesen hast.",
    	yes: "Ja"
    };
    var dialog_add_repo$1 = {
    	sort_by: "Sortiere nach"
    };
    var dialog_install$1 = {
    	restart: "Denken Sie daran, dass Sie Home Assistant neu starten müssen, bevor Änderungen an Integrationen (custom_components) angewendet werden."
    };
    var menu$1 = {
    	dismiss: "Alle neuen Repositories ausblenden"
    };
    var options$1 = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "AppDaemon App-Entdeckung & Nachverfolgung aktivieren",
    				country: "Nach Ländercode filtern.",
    				debug: "Debug aktivieren.",
    				experimental: "Experimentelle Funktionen aktivieren",
    				netdaemon: "NetDaemon App-Entdeckung & Nachverfolgung aktivieren",
    				not_in_use: "Nicht in Verwendung mit YAML",
    				release_limit: "Anzahl anzuzeigender Releases.",
    				sidepanel_icon: "Sidepanel Symbol",
    				sidepanel_title: "Sidepanel Titel"
    			}
    		}
    	}
    };
    var repository_banner$1 = {
    	config_flow: "Diese Integration unterstützt config_flow. Das bedeutet, dass du jetzt in den Integrationsbereich deiner Benutzeroberfläche wechseln kannst, um es zu konfigurieren.",
    	config_flow_title: "UI-Konfiguration unterstützt",
    	integration_not_loaded: "Diese Integration ist in Home Assistant nicht geladen.",
    	no_restart_required: "Kein Neustart erforderlich",
    	not_loaded: "Nicht geladen",
    	plugin_not_loaded: "Dieses Element wird nicht zu Ihren Lovelace-Ressourcen hinzugefügt.",
    	restart: "Du musst Home Assistant neu starten.",
    	restart_pending: "Neustart ausstehend"
    };
    var repository_card$1 = {
    	dismiss: "Ausblenden",
    	open_source: "Quelldateien öffnen",
    	report: "Melden zur Entfernung des Repositorys"
    };
    var repository$1 = {
    	add_to_lovelace: "Zu Lovelace hinzufügen",
    	authors: "Autoren",
    	available: "Verfügbar",
    	back_to: "Zurück zu",
    	changelog: "Änderungsprotokoll",
    	downloads: "Downloads",
    	flag_this: "Melden",
    	frontend_version: "Frontend Version",
    	github_stars: "GitHub Sterne",
    	goto_integrations: "Gehe zu Integrationen",
    	hide: "Verstecken",
    	hide_beta: "Beta verstecken",
    	install: "Installieren",
    	installed: "Installiert",
    	lovelace_copy_example: "Beispiel in die Zwischenablage kopieren",
    	lovelace_instruction: "Zum Hinzufügen zu deinen Lovelace-Einstellungen, benutze Folgendes",
    	lovelace_no_js_type: "Der Typ dieses Elements konnte nicht ermittelt werden. Überprüfen Sie das Repository.",
    	newest: "neueste",
    	note_appdaemon: "du musst es dann noch in die Datei 'apps.yaml' hinzufügen",
    	note_installed: "Wird installiert nach",
    	note_integration: "du musst es dann noch in die Datei 'configuration.yaml' hinzufügen",
    	note_plugin: "du musst es dann noch in deine Lovelace-Einstellungen ('ui-lovelace.yaml' oder im Raw-Konfigurationseditor) hinzufügen",
    	note_plugin_post_107: "Du musst es noch zu deiner Lovelace-Konfiguration hinzufügen ('configuration.yaml' oder der Ressourceneditor '/config/lovelace/resources')",
    	open_issue: "Problem melden",
    	open_plugin: "Element öffnen",
    	reinstall: "Neu installieren",
    	repository: "Repository",
    	restart_home_assistant: "Home Assistant neu starten",
    	show_beta: "Beta anzeigen",
    	uninstall: "Deinstallieren",
    	update_information: "Aktualisierungsinformationen",
    	upgrade: "Aktualisieren"
    };
    var sections$1 = {
    	about: {
    		description: "Informationen zu HACS anzeigen",
    		title: "Über"
    	},
    	automation: {
    		description: "Hier finden Sie python_scripts, AppDaemon-Apps und NetDaemon-Apps",
    		title: "Automatisierung"
    	},
    	frontend: {
    		description: "Hier finden Sie Themen, individuelle Karten und andere Elemente für Lovelace",
    		title: "Frontend"
    	},
    	integrations: {
    		description: "Hier finden Sie benutzerdefinierte Integrationen (custom_components)",
    		title: "Integrationen"
    	},
    	pending_repository_upgrade: "Sie verwenden Version {installed}, Version {available} ist verfügbar"
    };
    var settings$1 = {
    	add_custom_repository: "BENUTZERDEFINIERTES REPOSITORY HINZUFÜGEN",
    	adding_new_repo: "Hinzufügen eines neuen Repository '{repo}'",
    	adding_new_repo_category: "Mit der Kategorie '{category}'.",
    	bg_task_custom: "Custom Repositorys werden ausgeblendet, während Hintergrundaufgaben ausgeführt werden.",
    	category: "Kategorie",
    	compact_mode: "Kompakter Modus",
    	custom_repositories: "BENUTZERDEFINIERTE REPOSITORIES",
    	"delete": "Löschen",
    	display: "Anzeige",
    	grid: "Gitter",
    	hacs_repo: "HACS repo",
    	hidden_repositories: "versteckte Repositories",
    	missing_category: "Du musst eine Kategorie auswählen.",
    	open_repository: "Repository öffnen",
    	reload_data: "Daten neu laden",
    	reload_window: "Fenster neu laden",
    	repository_configuration: "Repository Konfiguration",
    	save: "Speichern",
    	table: "Tabelle",
    	table_view: "Tabellenansicht",
    	unhide: "einblenden",
    	upgrade_all: "Alles aktualisieren"
    };
    var store$1 = {
    	ascending: "Aufsteigend",
    	clear_new: "Alle neuen Repositories als gesehen markieren",
    	descending: "Absteigend",
    	last_updated: "Zuletzt aktualisiert",
    	name: "Name",
    	new_repositories: "Neue Repositories",
    	new_repositories_note: "Hier werden über 10 neue Repositorys angezeigt. Wenn Sie alle als gelesen markieren möchten, klicken Sie auf die 3 Punkte in der oberen rechten Ecke und blenden sie alle aus.",
    	pending_upgrades: "Ausstehende Upgrades",
    	placeholder_search: "Suchbegriff eingeben…",
    	sort: "Sortieren",
    	stars: "Sterne",
    	status: "Status"
    };
    var time$1 = {
    	ago: "vor",
    	day: "Tag",
    	days: "Tage",
    	hour: "Stunde",
    	hours: "Stunden",
    	minute: "Minute",
    	minutes: "Minuten",
    	month: "Monat",
    	months: "Monate",
    	one: "Eins",
    	one_day_ago: "vor einem Tag",
    	one_hour_ago: "vor einer Stunde",
    	one_minute_ago: "vor einer Minute",
    	one_month_ago: "vor einem Monat",
    	one_second_ago: "vor einer Sekunde",
    	one_year_ago: "vor einem Jahr",
    	second: "Sekunde",
    	seconds: "Sekunden",
    	x_days_ago: "vor {x} Tagen",
    	x_hours_ago: "vor {x} Stunden",
    	x_minutes_ago: "vor {x} Minuten",
    	x_months_ago: "vor {x} Monaten",
    	x_seconds_ago: "vor {x} Sekunden",
    	x_years_ago: "vor {x} Jahren",
    	year: "Jahr",
    	years: "Jahre"
    };
    var de = {
    	common: common$1,
    	config: config$1,
    	confirm: confirm$1,
    	dialog_add_repo: dialog_add_repo$1,
    	dialog_install: dialog_install$1,
    	menu: menu$1,
    	options: options$1,
    	repository_banner: repository_banner$1,
    	repository_card: repository_card$1,
    	repository: repository$1,
    	sections: sections$1,
    	settings: settings$1,
    	store: store$1,
    	time: time$1
    };

    var de$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$1,
        config: config$1,
        confirm: confirm$1,
        dialog_add_repo: dialog_add_repo$1,
        dialog_install: dialog_install$1,
        menu: menu$1,
        options: options$1,
        repository_banner: repository_banner$1,
        repository_card: repository_card$1,
        repository: repository$1,
        sections: sections$1,
        settings: settings$1,
        store: store$1,
        time: time$1,
        'default': de
    });

    var common$2 = {
    	about: "Σχετικά με",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "AppDaemon Apps",
    	background_task: "Τρέχει μια διεργασία στο παρασκήνιο, η σελίδα θα ανανεωθεί μόλις αυτό ολοκληρωθεί.",
    	check_log_file: "Ελέγξτε το αρχείο καταγραφής για περισσότερες λεπτομέρειες.",
    	"continue": "Να συνεχίσει",
    	disabled: "Απενεργοποιημένο",
    	documentation: "Τεκμηρίωση",
    	hacs_is_disabled: "Το HACS είναι απενεργοποιημένο",
    	installed: "εγκατεστημένο",
    	integration: "Ενσωμάτωση",
    	integrations: "Ενσωματωμένα",
    	manage: "διαχειρίζονται",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDaemon Apps",
    	plugin: "Πρόσθετο",
    	plugins: "Πρόσθετα",
    	python_script: "Πρόγραμμα Python",
    	python_scripts: "Προγράμματα Python",
    	repositories: "Αποθετήρια",
    	settings: "ρυθμίσεις",
    	theme: "Θέμα",
    	themes: "Θέματα",
    	version: "Έκδοση"
    };
    var config$2 = {
    	abort: {
    		single_instance_allowed: "Μονάχα μία ρύθμιση των παραμέτρων του HACS επιτρέπεται."
    	},
    	error: {
    		auth: "Το διακριτικό πρόσβασης δεν είναι σωστό."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Ενεργοποίηση εύρεσης & παρακολούθησης για το AppDaemon",
    				netdaemon: "Ενεργοποίηση εύρεσης & παρακολούθησης για το NetDaemon",
    				python_script: "Ενεργοποίηση εύρεσης & παρακολούθησης για τα python_scripts",
    				sidepanel_icon: "Εικονίδιο πλαϊνού πάνελ",
    				sidepanel_title: "Τίτλος πλαϊνού πάνελ",
    				theme: "Ενεργοποίηση εύρεσης & παρακολούθησης για τα θεμάτων",
    				token: "Διακριτικό πρόσβασης του GitHub"
    			},
    			description: "Εάν χρειαστείτε βοήθεια με τις ρυθμίσεις ανατρέξτε εδώ: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$2 = {
    	add_to_lovelace: "Είστε βέβαιοι ότι θέλετε να προσθέσετε αυτό στους πόρους του Lovelace;",
    	bg_task: "Η ενέργεια είναι απενεργοποιημένη όσο εκτελούνται εργασίες στο παρασκήνιο",
    	cancel: "Ακύρωση",
    	"continue": "Είστε βέβαιοι ότι θέλετε να συνεχίσετε;",
    	"delete": "Είστε σίγουροι ότι θέλετε να διαγράψετε το '{item}';",
    	delete_installed: "Το '{item}' είναι εγκατεστημένο, πρέπει να το απεγκαταστήσετε πριν να το διαγράψετε.",
    	exist: "{item} υπάρχει ήδη",
    	generic: "Είστε βέβαιοι;",
    	home_assistant_is_restarting: "Περιμένετε, το Home Assistant επανεκκινείται τώρα.",
    	no: "Οχι",
    	no_upgrades: "Δεν υπάρχουν αναβαθμίσεις σε εκκρεμότητα",
    	ok: "Εντάξει",
    	overwrite: "Αυτό θα το αντικαταστήσει.",
    	restart_home_assistant: "Είστε βέβαιοι ότι θέλετε να κάνετε επανεκκίνηση του Home Assistant;",
    	uninstall: "Είστε βέβαιοι ότι θέλετε να απεγκαταστήσετε το '{item}';",
    	upgrade_all: "Αυτό θα αναβαθμίσει όλα αυτά τα αποθετήρια, βεβαιωθείτε ότι έχετε διαβάσει τις σημειώσεις έκδοσης για όλα πριν προχωρήσετε.",
    	yes: "Ναι"
    };
    var options$2 = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Ενεργοποίηση εύρεσης & παρακολούθησης για το AppDaemon",
    				country: "Κριτήριο με βάση τον κωδικό χώρας.",
    				debug: "Ενεργοποίηση εντοπισμού σφαλμάτων.",
    				experimental: "Ενεργοποίση πειραματικών λειτουργιών",
    				netdaemon: "Ενεργοποίηση εύρεσης & παρακολούθησης για το NetDaemon",
    				not_in_use: "Δεν χρησιμοποιείται με το YAML",
    				release_limit: "Αριθμός εκδόσεων που να παραθέτονται.",
    				sidepanel_icon: "Εικονίδιο πλαϊνού πάνελ",
    				sidepanel_title: "Τίτλος πλαϊνού πάνελ"
    			}
    		}
    	}
    };
    var repository_banner$2 = {
    	integration_not_loaded: "Αυτή η ενσωμάτωση δεν φορτώθηκε στο Home Assistant.",
    	no_restart_required: "Δεν απαιτείται επανεκκίνηση",
    	not_loaded: "Δεν έχει φορτωθεί",
    	plugin_not_loaded: "Αυτό το πρόσθετο δεν προστέθηκε στους πόρους του Lovelace.",
    	restart: "Πρέπει να κάνετε επανεκκίνηση του Home Assistant.",
    	restart_pending: "Επανεκκίνηση σε εκκρεμότητα"
    };
    var repository$2 = {
    	add_to_lovelace: "Προσθήκη στο Lovelace",
    	authors: "Συγγραφείς",
    	available: "Διαθέσιμο",
    	back_to: "Πίσω στο",
    	changelog: "Σημειώσεις των αλλαγών",
    	downloads: "Λήψεις",
    	flag_this: "Σημείωσε αυτό",
    	frontend_version: "Έκδοση Frontend",
    	github_stars: "GitHub αστέρια",
    	goto_integrations: "Μετάβαση στις ενσωματώσεις",
    	hide: "Απόκρυψη",
    	hide_beta: "Απόκριση του beta",
    	install: "Εγκατάσταση",
    	installed: "Εγκατεστημένο",
    	lovelace_copy_example: "Αντίγραψε το παράδειγμα στο πρόχειρο",
    	lovelace_instruction: "Όταν το προσθέσετε στις ρυθμίσεις του lovelace χρησιμοποιήστε τούτο",
    	lovelace_no_js_type: "Δεν καταφέραμε να προσδιορίσουμε τον τύπο αυτού του προσθέτου, ελέξτε το αποθετήριο.",
    	newest: "νεότερο",
    	note_appdaemon: "εξακολουθεί να χρειάζεται να το προσθέσετε στο αρχείο 'apps.yaml'",
    	note_installed: "Όταν εγκατασταθεί, θα προστεθεί στο",
    	note_integration: "εξακολουθεί να χρειάζεται να το προσθέσετε στο αρχείο 'configuration.yaml'",
    	note_plugin: "εξακολουθεί να χρειάζετε να το προσθέσετε στις ρυθμίσεις του lovelace ('ui-lovelace.yaml' ή μέσω του γραφικού επεξεργαστή των ρυθμίσεων)",
    	open_issue: "Εκκρεμόν ζήτημα",
    	open_plugin: "Ανοιχτό πρόσθετο",
    	reinstall: "Επανεγκατάσταση",
    	repository: "Αποθετήριο",
    	restart_home_assistant: "Επανεκκίνηση του Home Assistant",
    	show_beta: "Εμφάνιση του beta",
    	uninstall: "Απεγκατάσταση",
    	update_information: "Ενημέρωση πληροφοριών",
    	upgrade: "Ενημέρωση"
    };
    var settings$2 = {
    	add_custom_repository: "ΠΡΟΣΘΕΣΤΕ ΕΝΑ ΕΙΔΙΚΟ ΑΠΟΘΕΤΗΡΙΟ",
    	adding_new_repo: "Προσθήκη νέου αποθετηρίου '{repo}'",
    	adding_new_repo_category: "Με κατηγορία '{category}'.",
    	category: "Κατηγορία",
    	compact_mode: "Συμπαγής λειτουργία",
    	custom_repositories: "ΕΙΔΙΚΑ ΑΠΟΘΕΤΗΡΙΑ",
    	"delete": "Διαγραφή",
    	display: "Εμφάνιση",
    	grid: "Πλέγμα",
    	hacs_repo: "Αποθετήριο του HACS",
    	hidden_repositories: "κρυφά αποθετήρια",
    	missing_category: "Πρέπει να επιλέξετε μια κατηγορία",
    	open_repository: "Ανοίξτε το αποθετήριο",
    	reload_data: "Επαναφόρτωση δεδομένων",
    	reload_window: "Επαναφόρτωση του παραθύρου",
    	repository_configuration: "Διαμόρφωση αποθετηρίου",
    	save: "Αποθήκευση",
    	table: "Πίνακας",
    	table_view: "Προβολή πίνακα",
    	unhide: "αποκρύψω",
    	upgrade_all: "Αναβάθμιση όλων"
    };
    var store$2 = {
    	ascending: "αύξουσα",
    	clear_new: "Απαλοιφή όλων των νέων αποθετηρίων",
    	descending: "φθίνουσα",
    	last_updated: "Τελευταία ενημέρωση",
    	name: "Ονομα",
    	new_repositories: "Νέα αποθετήρια",
    	pending_upgrades: "Εκκρεμείς αναβαθμίσεις",
    	placeholder_search: "Παρακαλώ πληκτρολογήστε έναν όρο προς αναζήτηση...",
    	sort: "είδος",
    	stars: "Αστέρια",
    	status: "Κατάσταση"
    };
    var time$2 = {
    	ago: "πριν",
    	day: "ημέρα",
    	days: "ημέρες",
    	hour: "ώρα",
    	hours: "ώρες",
    	minute: "λεπτό",
    	minutes: "λεπτά",
    	month: "μήνας",
    	months: "μήνες",
    	one: "Ένα",
    	one_day_ago: "πριν από μία ημέρα",
    	one_hour_ago: "πριν από μια ώρα",
    	one_minute_ago: "πριν από ένα λεπτό",
    	one_month_ago: "πριν από ένα μήνα",
    	one_second_ago: "πριν από ένα δευτερόλεπτο",
    	one_year_ago: "πριν από ένα χρόνο",
    	second: "δευτερόλεπτο",
    	seconds: "δευτερόλεπτα",
    	x_days_ago: "{x} ημέρες πριν",
    	x_hours_ago: "{x} ώρες πριν",
    	x_minutes_ago: "{x} λεπτά πριν",
    	x_months_ago: "{x} μήνες πριν",
    	x_seconds_ago: "{x} δευτερόλεπτα πριν",
    	x_years_ago: "{x} χρόνια πριν",
    	year: "έτος",
    	years: "χρόνια"
    };
    var el = {
    	common: common$2,
    	config: config$2,
    	confirm: confirm$2,
    	options: options$2,
    	repository_banner: repository_banner$2,
    	repository: repository$2,
    	settings: settings$2,
    	store: store$2,
    	time: time$2
    };

    var el$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$2,
        config: config$2,
        confirm: confirm$2,
        options: options$2,
        repository_banner: repository_banner$2,
        repository: repository$2,
        settings: settings$2,
        store: store$2,
        time: time$2,
        'default': el
    });

    var common$3 = {
    	about: "About",
    	add: "add",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "AppDaemon Apps",
    	appdaemon_plural: "AppDaemon Apps",
    	background_task: "Background task running, this page will reload when it's done.",
    	check_log_file: "Check your log file for more details.",
    	"continue": "Continue",
    	disabled: "Disabled",
    	documentation: "Documentation",
    	element: "element",
    	hacs_is_disabled: "HACS is disabled",
    	install: "Install",
    	installed: "installed",
    	integration: "Integration",
    	integration_plural: "Integrations",
    	integrations: "Integrations",
    	lovelace: "Lovelace",
    	lovelace_element: "Lovelace element",
    	lovelace_elements: "Lovelace elements",
    	manage: "manage",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDaemon Apps",
    	netdaemon_plural: "NetDaemon Apps",
    	plugin: "Lovelace",
    	plugin_plural: "Lovelace elements",
    	plugins: "Lovelace elements",
    	python_script: "Python Script",
    	python_script_plural: "Python Scripts",
    	python_scripts: "Python Scripts",
    	repositories: "Repositories",
    	repository: "Repository",
    	settings: "settings",
    	theme: "Theme",
    	theme_plural: "Themes",
    	themes: "Themes",
    	uninstall: "Uninstall",
    	update: "Update",
    	version: "Version"
    };
    var config$3 = {
    	abort: {
    		single_instance_allowed: "Only a single configuration of HACS is allowed."
    	},
    	error: {
    		auth: "Personal Access Token is not correct."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Enable AppDaemon apps discovery & tracking",
    				netdaemon: "Enable NetDaemon apps discovery & tracking",
    				python_script: "Enable python_scripts discovery & tracking",
    				sidepanel_icon: "Side panel icon",
    				sidepanel_title: "Side panel title",
    				theme: "Enable Themes discovery & tracking",
    				token: "GitHub Personal Access Token"
    			},
    			description: "If you need help with the configuration have a look here: https://hacs.xyz/docs/configuration/start/",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$3 = {
    	add_to_lovelace: "Are you sure you want to add this to your Lovelace resources?",
    	bg_task: "Action is disabled while background tasks is running.",
    	cancel: "Cancel",
    	"continue": "Are you sure you want to continue?",
    	"delete": "Are you sure you want to delete \"{item}\"?",
    	delete_installed: "'{item}' is installed, you need to uninstall it before you can delete it.",
    	exist: "{item} already exists",
    	generic: "Are you sure?",
    	home_assistant_is_restarting: "Hold on, Home Assistant is now restarting.",
    	home_assistant_version_not_correct: "You are running Home Assistant version '{haversion}', but this repository requires minimum version '{minversion}' to be installed.",
    	no: "No",
    	no_upgrades: "No upgrades pending",
    	ok: "OK",
    	overwrite: "Doing this will overwrite it.",
    	reload_data: "This reloads the data of all repositories HACS knows about, this will take some time to finish.",
    	restart_home_assistant: "Are you sure you want to restart Home Assistant?",
    	uninstall: "Are you sure you want to uninstall \"{item}\"?",
    	upgrade_all: "This will upgrade all of these repositories, make sure that you have read the release notes for all of them before proceeding.",
    	yes: "Yes"
    };
    var dialog_about$1 = {
    	frontend_version: "Frontend version",
    	installed_repositories: "Installed repositories",
    	integration_version: "Integration version",
    	useful_links: "Useful links"
    };
    var dialog_add_repo$2 = {
    	limit: "Only the first 100 repositories are shown, use the search to filter what you need",
    	no_match: "No repositories found matching your filter",
    	sort_by: "Sort by",
    	title: "Add repository"
    };
    var dialog_custom_repositories$1 = {
    	category: "Category",
    	no_category: "Missing category",
    	no_repository: "Missing repository",
    	title: "Custom repositories",
    	url_placeholder: "Add custom repository URL"
    };
    var dialog_info$1 = {
    	author: "Author",
    	downloads: "Downloads",
    	install: "Install this repository in HACS",
    	loading: "Loading information...",
    	no_info: "The developer has not provided any more information for this repository",
    	open_issues: "Open issues",
    	open_repo: "Open repository",
    	stars: "Stars",
    	version_installed: "Version installed"
    };
    var dialog_install$2 = {
    	restart: "Remember that you need to restart Home Assistant before changes to integrations (custom_components) are applied.",
    	select_version: "Select version",
    	show_beta: "Show beta versions",
    	type: "Type",
    	url: "URL"
    };
    var dialog_update$1 = {
    	available_version: "Available version",
    	changelog: "Changelog",
    	installed_version: "Installed version",
    	title: "Update pending"
    };
    var entry$1 = {
    	information: "Information",
    	intro: "Updates and important messages will show here if there are any",
    	messages: {
    		disabled: {
    			content: "Check your log file for more details",
    			title: "HACS is disabled"
    		},
    		has_pending_tasks: {
    			content: "Some repositories might not show untill this is completed",
    			title: "Background tasks pending"
    		},
    		startup: {
    			content: "HACS is starting up, during this time some information might be missing or incorrect",
    			title: "HACS is starting up"
    		}
    	},
    	pending_updates: "Pending updates"
    };
    var menu$2 = {
    	about: "About HACS",
    	clear: "Clear all new",
    	custom_repositories: "Custom repositories",
    	dismiss: "Dismiss all new repositories",
    	documentation: "Documentation",
    	open_issue: "Open issue",
    	reload: "Reload window"
    };
    var options$3 = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Enable AppDaemon apps discovery & tracking",
    				country: "Filter with country code.",
    				debug: "Enable debug.",
    				experimental: "Enable experimental features",
    				netdaemon: "Enable NetDaemon apps discovery & tracking",
    				not_in_use: "Not in use with YAML",
    				release_limit: "Number of releases to show.",
    				sidepanel_icon: "Side panel icon",
    				sidepanel_title: "Side panel title"
    			}
    		}
    	}
    };
    var repository_banner$3 = {
    	config_flow: "This integration supports config_flow, that means that you now can go to the integration section of your UI to configure it.",
    	config_flow_title: "UI Configuration supported",
    	integration_not_loaded: "This integration is not loaded in Home Assistant.",
    	no_restart_required: "No restart required",
    	not_loaded: "Not loaded",
    	plugin_not_loaded: "This plugin is not added to your Lovelace resources.",
    	restart: "You need to restart Home Assistant.",
    	restart_pending: "Restart pending"
    };
    var repository_card$2 = {
    	dismiss: "dismiss",
    	hide: "Hide",
    	information: "Information",
    	new_repository: "New repository",
    	open_issue: "Open issue",
    	open_source: "Open source",
    	pending_update: "Pending update",
    	reinstall: "Reinstall",
    	report: "Report for removal",
    	update_information: "Update information"
    };
    var repository$3 = {
    	add_to_lovelace: "Add to Lovelace",
    	authors: "Authors",
    	available: "Available",
    	back_to: "Back to",
    	changelog: "Change log",
    	downloads: "Downloads",
    	flag_this: "Flag this",
    	frontend_version: "Frontend version",
    	github_stars: "GitHub stars",
    	goto_integrations: "Go to integrations",
    	hide: "Hide",
    	hide_beta: "Hide beta",
    	install: "Install",
    	installed: "Installed",
    	lovelace_copy_example: "Copy the example to your clipboard",
    	lovelace_instruction: "When you add this to your lovelace configuration use this",
    	lovelace_no_js_type: "Could not determine the type of this plugin, check the repository.",
    	newest: "newest",
    	note_appdaemon: "you still need to add it to your 'apps.yaml' file",
    	note_installed: "When installed, this will be located in",
    	note_integration: "you still need to add it to your 'configuration.yaml' file",
    	note_plugin: "you still need to add it to your lovelace configuration ('ui-lovelace.yaml' or the raw UI config editor)",
    	note_plugin_post_107: "you still need to add it to your lovelace configuration ('configuration.yaml' or the resource editor '/config/lovelace/resources')",
    	open_issue: "Open issue",
    	open_plugin: "Open element",
    	reinstall: "Reinstall",
    	repository: "Repository",
    	restart_home_assistant: "Restart Home Assistant",
    	show_beta: "Show beta",
    	uninstall: "Uninstall",
    	update_information: "Update information",
    	upgrade: "Update"
    };
    var search$1 = {
    	placeholder: "Search for repository"
    };
    var sections$2 = {
    	about: {
    		description: "Show information about HACS",
    		title: "About"
    	},
    	automation: {
    		description: "This is where you find python_scripts, AppDaemon apps and NetDaemon apps",
    		title: "Automation"
    	},
    	frontend: {
    		description: "This is where you find themes, custom cards and other elements for lovelace",
    		title: "Frontend"
    	},
    	integrations: {
    		description: "This is where you find custom integrations (custom_components)",
    		title: "Integrations"
    	},
    	pending_repository_upgrade: "You are running version {installed}, version {available} is available"
    };
    var settings$3 = {
    	add_custom_repository: "ADD CUSTOM REPOSITORY",
    	adding_new_repo: "Adding new repository '{repo}'",
    	adding_new_repo_category: "With category '{category}'.",
    	bg_task_custom: "Custom repositories are hidden while background tasks is running.",
    	category: "Category",
    	compact_mode: "Compact mode",
    	custom_repositories: "CUSTOM REPOSITORIES",
    	"delete": "Delete",
    	display: "Display",
    	grid: "Grid",
    	hacs_repo: "HACS repo",
    	hidden_repositories: "hidden repositories",
    	missing_category: "You need to select a category",
    	open_repository: "Open repository",
    	reload_data: "Reload data",
    	reload_window: "Reload window",
    	repository_configuration: "Repository configuration",
    	save: "Save",
    	table: "Table",
    	table_view: "Table view",
    	unhide: "unhide",
    	upgrade_all: "Upgrade all"
    };
    var store$3 = {
    	ascending: "ascending",
    	clear_new: "Clear all new repositories",
    	descending: "descending",
    	last_updated: "Last updated",
    	name: "Name",
    	new_repositories: "New Repositories",
    	new_repositories_note: "You have over 10 new repositories showing here, if you want to clear them all click the 3 dots in the top right corner and dismiss all of them.",
    	no_repositories: "No repositories",
    	no_repositories_desc1: "It seems like you don't have any repositories installed in this section yet.",
    	no_repositories_desc2: "Click on the + in the bottom corner to add your first!",
    	pending_upgrades: "Pending upgrades",
    	placeholder_search: "Please enter a search term...",
    	sort: "sort",
    	stars: "Stars",
    	status: "Status"
    };
    var time$3 = {
    	ago: "ago",
    	day: "day",
    	days: "days",
    	hour: "hour",
    	hours: "hours",
    	minute: "minute",
    	minutes: "minutes",
    	month: "month",
    	months: "months",
    	one: "One",
    	one_day_ago: "one day ago",
    	one_hour_ago: "one hour ago",
    	one_minute_ago: "one minute ago",
    	one_month_ago: "one month ago",
    	one_second_ago: "one second ago",
    	one_year_ago: "one year ago",
    	second: "second",
    	seconds: "seconds",
    	x_days_ago: "{x} days ago",
    	x_hours_ago: "{x} hours ago",
    	x_minutes_ago: "{x} minutes ago",
    	x_months_ago: "{x} months ago",
    	x_seconds_ago: "{x} seconds ago",
    	x_years_ago: "{x} years ago",
    	year: "year",
    	years: "years"
    };
    var en = {
    	common: common$3,
    	config: config$3,
    	confirm: confirm$3,
    	dialog_about: dialog_about$1,
    	dialog_add_repo: dialog_add_repo$2,
    	dialog_custom_repositories: dialog_custom_repositories$1,
    	dialog_info: dialog_info$1,
    	dialog_install: dialog_install$2,
    	dialog_update: dialog_update$1,
    	entry: entry$1,
    	menu: menu$2,
    	options: options$3,
    	repository_banner: repository_banner$3,
    	repository_card: repository_card$2,
    	repository: repository$3,
    	search: search$1,
    	sections: sections$2,
    	settings: settings$3,
    	store: store$3,
    	time: time$3
    };

    var en$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$3,
        config: config$3,
        confirm: confirm$3,
        dialog_about: dialog_about$1,
        dialog_add_repo: dialog_add_repo$2,
        dialog_custom_repositories: dialog_custom_repositories$1,
        dialog_info: dialog_info$1,
        dialog_install: dialog_install$2,
        dialog_update: dialog_update$1,
        entry: entry$1,
        menu: menu$2,
        options: options$3,
        repository_banner: repository_banner$3,
        repository_card: repository_card$2,
        repository: repository$3,
        search: search$1,
        sections: sections$2,
        settings: settings$3,
        store: store$3,
        time: time$3,
        'default': en
    });

    var common$4 = {
    	about: "Acerca de",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "AppDaemon Apps",
    	appdaemon_plural: "AppDaemon Apps",
    	background_task: "Ejecutando tareas en segundo plano. Se refrescará automaticamente esta página al finalizar.",
    	check_log_file: "Compruebe el archivo de registro para obtener más detalles.",
    	"continue": "Continuar",
    	disabled: "Deshabilitado",
    	documentation: "Documentación",
    	element: "elemento",
    	hacs_is_disabled: "HACS está deshabilitado",
    	installed: "instalado",
    	integration: "Integración",
    	integration_plural: "Integraciones",
    	integrations: "Integraciones",
    	lovelace: "Lovelace",
    	lovelace_element: "Elemento de Lovelace",
    	lovelace_elements: "Elementos de Lovelace",
    	manage: "Administrar",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDaemon Apps",
    	netdaemon_plural: "Aplicaciones NetDaemon",
    	plugin: "Plugin",
    	plugin_plural: "Elementos de Lovelace",
    	plugins: "Plugins",
    	python_script: "Python Script",
    	python_script_plural: "Python Scripts",
    	python_scripts: "Python Scripts",
    	repositories: "Repositorios",
    	settings: "configuraciones",
    	theme: "Tema",
    	theme_plural: "Temas",
    	themes: "Temas",
    	version: "Versión"
    };
    var config$4 = {
    	abort: {
    		single_instance_allowed: "Sólo se permite una única configuración de HACS."
    	},
    	error: {
    		auth: "El token de acceso personal no es correcto."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Habilitar el descubrimiento y seguimiento de las aplicaciones de AppDaemon",
    				netdaemon: "Habilitar el descubrimiento y seguimiento de las aplicaciones de NetDaemon",
    				python_script: "Habilitar el descubrimiento y seguimiento en python_scripts",
    				sidepanel_icon: "Ícono del panel lateral",
    				sidepanel_title: "Título del panel lateral",
    				theme: "Habilitar el descubrimiento y seguimiento de temas",
    				token: "Token de acceso personal de GitHub"
    			},
    			description: "Si necesitas ayuda con la configuración, visita la siguiente pagina: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$4 = {
    	add_to_lovelace: "¿Está seguro de que desea agregar esto a sus recursos de Lovelace?",
    	bg_task: "La acción está deshabilitada mientras se ejecutan tareas en segundo plano.",
    	cancel: "Cancelar",
    	"continue": "Estás seguro de que quieres continuar?",
    	"delete": "¿Seguro que quieres eliminar '{item}'?",
    	delete_installed: "'{item}' está instalado, debe desinstalarlo antes de poder eliminarlo.",
    	exist: "{item} ya existe",
    	generic: "¿Estás seguro?",
    	home_assistant_is_restarting: "Espera, Home Assistant se está reiniciando.",
    	home_assistant_version_not_correct: "Está ejecutando la versión '{haversion}' de Home Assistant, pero este repositorio requiere la instalación de la versión '{minversion}' mínima.",
    	no: "No",
    	no_upgrades: "No hay actualizaciones pendientes",
    	ok: "OK",
    	overwrite: "Si haces esto, se sobrescribirá.",
    	reload_data: "Esto recarga los datos de todos los repositorios que HACS conoce, esto tardará algún tiempo en finalizar.",
    	restart_home_assistant: "¿Está seguro de que desea reiniciar Home Assistant?",
    	uninstall: "¿Está seguro de que deseas desinstalar '{item}'?",
    	upgrade_all: "Esto actualizará todos estos repositorios, asegúrese de que ha leído las notas de la versión de todos ellos antes de continuar.",
    	yes: "Si"
    };
    var options$4 = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Habilitar el descubrimiento y seguimiento de las aplicaciones de AppDaemon",
    				country: "Filtrar por el código de país.",
    				debug: "Habilitar depuración.",
    				experimental: "Habilitar funciones experimentales",
    				netdaemon: "Habilitar el descubrimiento y seguimiento de las aplicaciones de NetDaemon",
    				not_in_use: "No usarse con YAML",
    				release_limit: "Número de versiones a mostrar.",
    				sidepanel_icon: "Icono del panel lateral",
    				sidepanel_title: "Título del panel lateral"
    			}
    		}
    	}
    };
    var repository_banner$4 = {
    	config_flow: "Esta integración soporta config_flow, lo que significa que ahora puede ir a la sección de integración de su UI para configurarlo.",
    	config_flow_title: "Configuración de UI soportada",
    	integration_not_loaded: "Esta integración no se carga en Home Assistant.",
    	no_restart_required: "No es necesario reiniciar",
    	not_loaded: "No está cargado",
    	plugin_not_loaded: "Este plugin aun no se ha añadido a sus recursos de Lovelace.",
    	restart: "Es necesario reiniciar Home Assistant.",
    	restart_pending: "Reinicio pendiente"
    };
    var repository$4 = {
    	add_to_lovelace: "Añadir a Lovelace",
    	authors: "Autores",
    	available: "Disponible",
    	back_to: "Volver a",
    	changelog: "Registro de cambios",
    	downloads: "Descargas",
    	flag_this: "Marcar esto",
    	frontend_version: "Versión del frontend",
    	github_stars: "Estrellas de GitHub",
    	goto_integrations: "Ir a integraciones",
    	hide: "Ocultar",
    	hide_beta: "Ocultar beta",
    	install: "Instalar",
    	installed: "Instalado",
    	lovelace_copy_example: "Copiar el ejemplo al clipboard",
    	lovelace_instruction: "Agregue lo siguiente en su configuración de lovelace",
    	lovelace_no_js_type: "No se pudo determinar el tipo de plugin, revise el repositorio.",
    	newest: "más nuevo",
    	note_appdaemon: "deberá agregar esto a su archivo 'apps.yaml'",
    	note_installed: "Cuando esté instalado, se ubicará en",
    	note_integration: "deberá agregar esto a su archivo 'configuration.yaml'",
    	note_plugin: "deberá agregar esto a su configuración de lovelace ('ui-lovelace.yaml' o en el editor UI de lovelace)",
    	note_plugin_post_107: "todavía necesita agregarlo a su configuración de lovelace ('configuration.yaml' o al editor de recursos '/config/lovelace/resources')",
    	open_issue: "Abrir issue",
    	open_plugin: "Abrir plugin",
    	reinstall: "Reinstalar",
    	repository: "Repositorio",
    	restart_home_assistant: "Reiniciar Home Assistant",
    	show_beta: "Mostrar beta",
    	uninstall: "Desinstalar",
    	update_information: "Actualizar información",
    	upgrade: "Actualizar"
    };
    var sections$3 = {
    	about: {
    		description: "Mostrar información sobre HACS",
    		title: "Acerca de"
    	},
    	automation: {
    		description: "Aquí es donde se encuentran python_scripts, aplicaciones AppDaemon y aplicaciones NetDaemon",
    		title: "Automatización"
    	},
    	frontend: {
    		description: "Aquí es donde encontrarás temas, tarjetas personalizadas y otros elementos para lovelace",
    		title: "Interfaz"
    	},
    	integrations: {
    		description: "Aquí es donde se encuentran las integraciones personalizadas (custom_components)",
    		title: "Integraciones"
    	},
    	pending_repository_upgrade: "Está ejecutando la versión {installed}, la versión {available} está disponible"
    };
    var settings$4 = {
    	add_custom_repository: "AGREGAR REPOSITORIO PERSONALIZADO",
    	adding_new_repo: "Añadiendo un nuevo repositorio '{repo}'.",
    	adding_new_repo_category: "Con la categoría '{category}'.",
    	bg_task_custom: "Los repositorios personalizados están ocultos mientras se ejecutan las tareas en segundo plano.",
    	category: "Categoría",
    	compact_mode: "Modo compacto",
    	custom_repositories: "REPOSITORIOS PERSONALIZADOS",
    	"delete": "Eliminar",
    	display: "Mostrar",
    	grid: "Cuadrícula",
    	hacs_repo: "HACS repo",
    	hidden_repositories: "repositorios ocultos",
    	missing_category: "Es necesario seleccionar una categoría",
    	open_repository: "Abrir repositorio",
    	reload_data: "Recargar datos",
    	reload_window: "Recargar ventana",
    	repository_configuration: "Configuración del repositorio",
    	save: "Guardar",
    	table: "Tabla",
    	table_view: "Vista de la tabla",
    	unhide: "mostrar",
    	upgrade_all: "Actualizar todo"
    };
    var store$4 = {
    	ascending: "ascendente",
    	clear_new: "Eliminar la lista los nuevos repositorios",
    	descending: "descendente",
    	last_updated: "Última actualización",
    	name: "Nombre",
    	new_repositories: "Nuevos Repositorios",
    	pending_upgrades: "Actualizaciones pendientes",
    	placeholder_search: "Por favor escriba una palabra clave de búsqueda...",
    	sort: "ordenar",
    	stars: "Estrellas",
    	status: "Estado"
    };
    var time$4 = {
    	ago: "hace",
    	day: "dia",
    	days: "dias",
    	hour: "hora",
    	hours: "horas",
    	minute: "minuto",
    	minutes: "minutos",
    	month: "mes",
    	months: "meses",
    	one: "Uno",
    	one_day_ago: "hace un día",
    	one_hour_ago: "hace una hora",
    	one_minute_ago: "hace un minuto",
    	one_month_ago: "hace un mes",
    	one_second_ago: "hace un segundo",
    	one_year_ago: "hace un año",
    	second: "segundo",
    	seconds: "segundos",
    	x_days_ago: "hace {x} dias",
    	x_hours_ago: "hace {x} horas",
    	x_minutes_ago: "hace {x} minutos",
    	x_months_ago: "hace {x} meses",
    	x_seconds_ago: "hace {x} segundos",
    	x_years_ago: "hace {x} años",
    	year: "año",
    	years: "años"
    };
    var es = {
    	common: common$4,
    	config: config$4,
    	confirm: confirm$4,
    	options: options$4,
    	repository_banner: repository_banner$4,
    	repository: repository$4,
    	sections: sections$3,
    	settings: settings$4,
    	store: store$4,
    	time: time$4
    };

    var es$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$4,
        config: config$4,
        confirm: confirm$4,
        options: options$4,
        repository_banner: repository_banner$4,
        repository: repository$4,
        sections: sections$3,
        settings: settings$4,
        store: store$4,
        time: time$4,
        'default': es
    });

    var common$5 = {
    	about: "À propos de",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "Applications AppDaemon",
    	background_task: "Tache de fond en cours, cette page se rechargera une fois terminée",
    	check_log_file: "Consultez votre fichier journal pour plus de détails.",
    	"continue": "Continuer",
    	disabled: "Désactivé",
    	documentation: "Documentation",
    	hacs_is_disabled: "HACS est désactivé",
    	installed: "installés",
    	integration: "Intégration",
    	integrations: "Intégrations",
    	manage: "gérer",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "Applications NetDaemon",
    	plugin: "Plugin",
    	plugins: "Plugins",
    	python_script: "Script Python",
    	python_scripts: "Scripts Python",
    	repositories: "Dépôts",
    	settings: "paramètres",
    	theme: "Thème",
    	themes: "Thèmes",
    	version: "Version"
    };
    var config$5 = {
    	abort: {
    		single_instance_allowed: "Une seule configuration de HACS est autorisée."
    	},
    	error: {
    		auth: "Le jeton personnel d'accès est invalide."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Activer la découverte et le suivi des applications AppDaemon",
    				netdaemon: "Activer la découverte et le suivi des applications NetDaemon",
    				python_script: "Activer la découverte et le suivi des scripts python",
    				sidepanel_icon: "Icône de la barre latérale",
    				sidepanel_title: "Titre de la barre latérale",
    				theme: "Activer la découverte et le suivi des thèmes",
    				token: "Jeton d'accès personnel GitHub"
    			},
    			description: "Si vous avez besoin d'aide pour la configuration, jetez un œil ici: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$5 = {
    	add_to_lovelace: "Êtes-vous sûr de vouloir l'ajouter à vos ressources Lovelace ?",
    	bg_task: "L'action est désactivée pendant l'exécution de tâches en arrière-plan.",
    	cancel: "Annuler",
    	"continue": "Es-tu sur de vouloir continuer?",
    	"delete": "Êtes-vous sûr de vouloir supprimer '{item}'?",
    	delete_installed: "'{item}' est installé, vous devez le désinstaller avant de pouvoir le supprimer.",
    	exist: "{item} existe déjà",
    	generic: "Êtes-vous sûr?",
    	home_assistant_is_restarting: "Attendez, Home Assistant redémarre maintenant.",
    	home_assistant_version_not_correct: "Vous utilisez la version '{haversion}' de Home Assistant, mais ce référentiel nécessite l'installation de la version minimale '{minversion}'.",
    	no: "Non",
    	no_upgrades: "Aucune mise à niveau en attente",
    	ok: "OK",
    	overwrite: "En faisant cela, cela l'écrasera.",
    	reload_data: "Cela recharge les données de tous les dépôts dont HACS a connaissance, cela prendra un certain temps à terminer.",
    	restart_home_assistant: "Voulez-vous vraiment redémarrer Home Assistant ?",
    	uninstall: "Êtes-vous sûr de vouloir désinstaller '{item}'?",
    	upgrade_all: "Cela mettra à niveau tous ces dépôts, assurez-vous d'avoir lu les notes de publication pour chacun d'entre eux avant de continuer.",
    	yes: "Oui"
    };
    var options$5 = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Activer la découverte et le suivi des applications AppDaemon",
    				country: "Filtrer par code pays.",
    				debug: "Activez le débogage.",
    				experimental: "Activer les fonctionnalités expérimentales",
    				netdaemon: "Activer la découverte et le suivi des applications NetDaemon",
    				not_in_use: "Non utilisé avec YAML",
    				release_limit: "Nombre de recensés à afficher.",
    				sidepanel_icon: "Icône de la barre latérale",
    				sidepanel_title: "Titre de la barre latérale"
    			}
    		}
    	}
    };
    var repository_banner$5 = {
    	config_flow: "Cette intégration prend en charge config_flow, ce qui signifie que vous pouvez maintenant accéder à la section d'intégration de votre interface utilisateur pour le configurer.",
    	config_flow_title: "Configuration de l'interface utilisateur prise en charge",
    	integration_not_loaded: "Cette intégration n'est pas chargée dans Home Assistant.",
    	no_restart_required: "Aucun redémarrage requis",
    	not_loaded: "Non chargé",
    	plugin_not_loaded: "Ce plugin n'est pas ajouté à vos ressources Lovelace.",
    	restart: "Vous devez redémarrer Home Assistant.",
    	restart_pending: "Redémarrage en attente"
    };
    var repository$5 = {
    	add_to_lovelace: "Ajouter à Lovelace",
    	authors: "Auteurs",
    	available: "Disponible",
    	back_to: "Retour",
    	changelog: "Change log",
    	downloads: "Téléchargements",
    	flag_this: "Marquer",
    	frontend_version: "Version de l'interface",
    	github_stars: "Étoiles GitHub",
    	goto_integrations: "Aller aux intégrations",
    	hide: "Masquer",
    	hide_beta: "Masquer les bêta",
    	install: "Installer",
    	installed: "Installé",
    	lovelace_copy_example: "Copier cet exemple dans le presse-papier",
    	lovelace_instruction: "Quand vous l'ajoutez à votre configuration lovelace, utilisez",
    	lovelace_no_js_type: "Impossible de déterminer le type de plugin, veuillez vérifier le dépôt",
    	newest: "nouveau",
    	note_appdaemon: "vous devez toujours l'ajouter à votre fichier 'apps.yaml'",
    	note_installed: "Une fois installé, il se trouvera dans",
    	note_integration: "Vous devez toujours l'ajouter à votre fichier 'configuration.yaml'",
    	note_plugin: "Vous devez toujours l'ajouter à votre configuration lovelace ('ui-lovelace.yaml' ou l'éditeur de configuration de l'interface)",
    	note_plugin_post_107: "Vous devez toujours l'ajouter à votre configuration lovelace ('configuration.yaml' ou l'éditeur de configuration de l'interface '/config/lovelace/resources')",
    	open_issue: "Ouvrir un ticket",
    	open_plugin: "Ouvrir le plugin",
    	reinstall: "Réinstaller",
    	repository: "Dépôt",
    	restart_home_assistant: "Redémarrer Home Assistant",
    	show_beta: "Afficher les bêta",
    	uninstall: "Désinstaller",
    	update_information: "Mettre à jour les informations",
    	upgrade: "Mettre à jour"
    };
    var settings$5 = {
    	add_custom_repository: "AJOUTER UN DÉPÔT PERSONNALISÉ",
    	adding_new_repo: "Ajout d'un nouveau dépôt '{repo}'",
    	adding_new_repo_category: "Avec la catégorie '{category}'.",
    	bg_task_custom: "Les dépôts personnalisés sont masqués pendant l'exécution de tâches en arrière-plan.",
    	category: "Catégorie",
    	compact_mode: "Mode compact",
    	custom_repositories: "DÉPÔTS PERSONNALISÉS",
    	"delete": "Supprimer",
    	display: "Affichage",
    	grid: "Grille",
    	hacs_repo: "Dépôt HACS",
    	hidden_repositories: "dépôts cachés",
    	missing_category: "Vous devez sélectionner une catégorie",
    	open_repository: "Ouvrir dépôt",
    	reload_data: "Recharger les données",
    	reload_window: "Recharger la fenêtre",
    	repository_configuration: "Configuration de dépôt",
    	save: "Enregistrer",
    	table: "Tableau",
    	table_view: "Vue table",
    	unhide: "Afficher",
    	upgrade_all: "Tout mettre à jour"
    };
    var store$5 = {
    	ascending: "ascendant",
    	clear_new: "Effacer tous les nouveaux dépôts",
    	descending: "descendant",
    	last_updated: "Dernière mise à jour",
    	name: "Nom",
    	new_repositories: "Nouveaux dépôts",
    	pending_upgrades: "Mises à niveau en attente",
    	placeholder_search: "Veuillez entrer un terme de recherche...",
    	sort: "Trier",
    	stars: "Étoiles",
    	status: "Statut"
    };
    var time$5 = {
    	ago: "il y a",
    	day: "jour",
    	days: "jours",
    	hour: "heure",
    	hours: "heures",
    	minute: "minute",
    	minutes: "minutes",
    	month: "mois",
    	months: "mois",
    	one: "Un",
    	one_day_ago: "il y a 1 jour",
    	one_hour_ago: "il y a 1 heure",
    	one_minute_ago: "il y a 1 minute",
    	one_month_ago: "il y a 1 mois",
    	one_second_ago: "il y a 1 seconde",
    	one_year_ago: "il y a 1 an",
    	second: "seconde",
    	seconds: "secondes",
    	x_days_ago: "il y a {x} jours",
    	x_hours_ago: "il y a {x} heures",
    	x_minutes_ago: "il y a {x} minutes",
    	x_months_ago: "il y a {x} mois",
    	x_seconds_ago: "il y a {x} secondes",
    	x_years_ago: "il y a {x} ans",
    	year: "an",
    	years: "ans"
    };
    var fr = {
    	common: common$5,
    	config: config$5,
    	confirm: confirm$5,
    	options: options$5,
    	repository_banner: repository_banner$5,
    	repository: repository$5,
    	settings: settings$5,
    	store: store$5,
    	time: time$5
    };

    var fr$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$5,
        config: config$5,
        confirm: confirm$5,
        options: options$5,
        repository_banner: repository_banner$5,
        repository: repository$5,
        settings: settings$5,
        store: store$5,
        time: time$5,
        'default': fr
    });

    var common$6 = {
    	about: "Névjegy",
    	add: "hozzáadás",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "AppDaemon Appok",
    	appdaemon_plural: "AppDaemon appok",
    	background_task: "Éppen háttérfeladat fut, ez az oldal frissülni fog, ha kész.",
    	check_log_file: "További részletekért ellenőrizd a naplófájlt.",
    	"continue": "Folytatás",
    	disabled: "Tiltva",
    	documentation: "Dokumentáció",
    	element: "bővítmény",
    	hacs_is_disabled: "A HACS le van tiltva",
    	install: "Telepítés",
    	installed: "Telepített",
    	integration: "Integráció",
    	integration_plural: "Integrációk",
    	integrations: "Integrációk",
    	lovelace: "Lovelace",
    	lovelace_element: "Lovelace bővítmény",
    	lovelace_elements: "Lovelace bővítmények",
    	manage: "kezelés",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDaemon Appok",
    	netdaemon_plural: "NetDaemon appok",
    	plugin: "Lovelace",
    	plugin_plural: "Lovelace bővítmények",
    	plugins: "Lovelace bővítmények",
    	python_script: "Python Szkript",
    	python_script_plural: "Python szkriptek",
    	python_scripts: "Python Szkriptek",
    	repositories: "Tárolók",
    	repository: "Tároló",
    	settings: "beállítások",
    	theme: "Téma",
    	theme_plural: "Témák",
    	themes: "Témák",
    	uninstall: "Eltávolítás",
    	update: "Frissítés",
    	version: "Verzió"
    };
    var config$6 = {
    	abort: {
    		single_instance_allowed: "Csak egyetlen HACS-konfiguráció megengedett."
    	},
    	error: {
    		auth: "A Személyes Hozzáférési Token nem megfelelő."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "AppDaemon appok felfedezésének és nyomon követésének engedélyezése",
    				netdaemon: "NetDaemon appok felfedezésének és nyomon követésének engedélyezése",
    				python_script: "Python szkriptek felfedezésének és nyomon követésének engedélyezése",
    				sidepanel_icon: "Oldalsó panel ikon",
    				sidepanel_title: "Oldalsó panel cím",
    				theme: "Témák felfedezésének és nyomon követésének engedélyezése",
    				token: "GitHub Személyes Hozzáférési Token"
    			},
    			description: "Ha segítségre van szükséged a konfigurációval kapcsolatban, akkor tekintsd meg ezt az oldalt: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$6 = {
    	add_to_lovelace: "Biztosan hozzá szeretnéd ezt adni a Lovelace erőforrásokhoz?",
    	bg_task: "A művelet le van tiltva, amíg háttérfeladat fut.",
    	cancel: "Mégse",
    	"continue": "Biztosan folytatni szeretnéd?",
    	"delete": "Biztosan törölni szeretnéd a(z) '{item}'-t?",
    	delete_installed: "A(z) '{item}' telepítve van, törlés előtt el kell távolítanod.",
    	exist: "{item} már létezik",
    	generic: "Biztos vagy benne?",
    	home_assistant_is_restarting: "Várj, a Home Assistant éppen újraindul.",
    	home_assistant_version_not_correct: "A Home Assistant '{haversion}' verzióját használod, de ehhez a tárolóhoz legalább a(z) '{minversion}' verzióra van szükség.",
    	no: "Nem",
    	no_upgrades: "Nincsenek elérhető frissítések",
    	ok: "OK",
    	overwrite: "Ezzel felül fogod írni.",
    	reload_data: "Ez újratölti a HACS által ismert összes tároló adatát, ami némi időbe telhet.",
    	restart_home_assistant: "Biztosan újraindítod a Home Assistant programot?",
    	uninstall: "Biztosan el szeretnéd távolítani a(z) '{item}'-t?",
    	upgrade_all: "Ez frissíteni fogja az összes tárolót. Győzödj meg róla, hogy elolvastad az összes kiadási megjegyzést, mielőtt továbblépnél.",
    	yes: "Igen"
    };
    var dialog_about$2 = {
    	frontend_version: "Frontend verzió",
    	installed_repositories: "Telepített tárolók",
    	integration_version: "Integráció verzió",
    	useful_links: "Hasznos linkek"
    };
    var dialog_add_repo$3 = {
    	limit: "Csak az első 100 tároló jelenik meg, használd a keresőt a találatok szűkítéséhez",
    	no_match: "Nincs a szűrésnek megfelelő tároló",
    	sort_by: "Rendezés",
    	title: "Tároló hozzáadása"
    };
    var dialog_custom_repositories$2 = {
    	category: "Kategória",
    	no_category: "Hiányzó kategória",
    	no_repository: "Hiányzó tároló",
    	title: "Egyéni tárolók",
    	url_placeholder: "Egyéni tároló URL címének hozzáadása"
    };
    var dialog_info$2 = {
    	author: "Szerző",
    	downloads: "Letöltések",
    	install: "Tároló telepítése HACS-ben",
    	loading: "Információ betöltése...",
    	no_info: "A fejlesztő nem adott meg több információt ehhez a tárolóhoz",
    	open_issues: "Jelentett problémák",
    	open_repo: "Tároló megnyitása",
    	stars: "Csillagok",
    	version_installed: "Telepített verzió"
    };
    var dialog_install$3 = {
    	restart: "Ne feledd, hogy az egyéni integrációk (custom_components) módosításainak alkalmazásához újra kell indítanod a Home Assistant alkalmazást.",
    	select_version: "Verzió kiválasztása",
    	show_beta: "Béta verziók megjelenítése",
    	type: "Típus",
    	url: "URL"
    };
    var dialog_update$2 = {
    	available_version: "Elérhető verzió",
    	changelog: "Változási napló",
    	installed_version: "Telepített verzió",
    	title: "Frissítés érhető el"
    };
    var entry$2 = {
    	information: "Információ",
    	intro: "A frissítések és a fontos üzenetek itt jelennek meg, ha vannak",
    	messages: {
    		disabled: {
    			content: "További részletek a naplófájlban",
    			title: "A HACS le van tiltva"
    		},
    		has_pending_tasks: {
    			content: "Előfordulhat, hogy egyes tárolók nem jelennek meg, amíg ez be nem fejeződik",
    			title: "Függőben lévő háttérfeladatok"
    		},
    		startup: {
    			content: "A HACS éppen indul, ezidő alatt egyes információk hiányozhatnak vagy helytelenek lehetnek",
    			title: "A HACS éppen indul"
    		}
    	},
    	pending_updates: "Frissítések érhetők el"
    };
    var menu$3 = {
    	about: "HACS névjegye",
    	clear: "Új jelölések törlése",
    	custom_repositories: "Egyéni tárolók",
    	dismiss: "Minden új tároló elvetése",
    	documentation: "Dokumentáció",
    	open_issue: "Probléma jelentése",
    	reload: "Ablak újratöltése"
    };
    var options$6 = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "AppDaemon appok felfedezésének és nyomon követésének engedélyezése",
    				country: "Szűrés országkóddal.",
    				debug: "Hibakeresés engedélyezése.",
    				experimental: "Kísérleti funkciók engedélyezése",
    				netdaemon: "NetDaemon appok felfedezésének és nyomon követésének engedélyezése",
    				not_in_use: "YAML-lel nem használható",
    				release_limit: "Megjelenítendő kiadások száma.",
    				sidepanel_icon: "Oldalsó panel ikon",
    				sidepanel_title: "Oldalsó panel cím"
    			}
    		}
    	}
    };
    var repository_banner$6 = {
    	config_flow: "Ez az integráció támogatja a config_flow-t, vagyis a felhasználói felületen az integrációk menüben lehet konfigurálni.",
    	config_flow_title: "A felhasználói felület konfigurációja támogatott",
    	integration_not_loaded: "Ez az integráció nincs betöltve a Home Assistantban.",
    	no_restart_required: "Nincs szükség újraindításra",
    	not_loaded: "Nincs betöltve",
    	plugin_not_loaded: "Ez a bővítmény nincs hozzáadva a Lovelace erőforrásaidhoz.",
    	restart: "Indítsd újra a Home Assistant programot.",
    	restart_pending: "Újraindítás függőben"
    };
    var repository_card$3 = {
    	dismiss: "elvetés",
    	hide: "Elrejtés",
    	information: "Információ",
    	new_repository: "Új tároló",
    	open_issue: "Probléma jelentése",
    	open_source: "Forrás megnyitása",
    	pending_update: "Frissítés érhető el",
    	reinstall: "Újratelepítés",
    	report: "Jelentés eltávolításra",
    	update_information: "Frissítési információ"
    };
    var repository$6 = {
    	add_to_lovelace: "Hozzáadás a Lovelace-hez",
    	authors: "Szerzők",
    	available: "Elérhető",
    	back_to: "Vissza -",
    	changelog: "Változási napló",
    	downloads: "Letöltések",
    	flag_this: "Megjelölés",
    	frontend_version: "Frontend verzió",
    	github_stars: "GitHub csillagok",
    	goto_integrations: "Ugrás az integrációkhoz",
    	hide: "Elrejtés",
    	hide_beta: "Béta elrejtése",
    	install: "Telepítés",
    	installed: "Telepített",
    	lovelace_copy_example: "Példa másolása a vágólapra",
    	lovelace_instruction: "Amikor hozzáadod ezt a lovelace konfigurációdhoz, használd ezt",
    	lovelace_no_js_type: "Nem sikerült meghatározni a bővítmény típusát, ellenőrizd a tárolót.",
    	newest: "legújabb",
    	note_appdaemon: "de még hozzá kell adnod az 'apps.yaml' fájlhoz",
    	note_installed: "Telepítéskor a következő helyre kerül:",
    	note_integration: "de még hozzá kell adnod a 'configuration.yaml' fájlhoz",
    	note_plugin: "de még hozzá kell adnod a lovelace konfigurációhoz (az 'ui-lovelace.yaml' fájlban vagy a Lovelace felületen a konfiguráció szerkesztőben)",
    	note_plugin_post_107: "de még hozzá kell adnod a lovelace konfigurációhoz ('configuration.yaml' vagy az erőforrás szerkesztőben '/config/lovelace/resources')",
    	open_issue: "Probléma jelentése",
    	open_plugin: "Bővítmény megnyitása",
    	reinstall: "Újratelepítés",
    	repository: "Tároló",
    	restart_home_assistant: "Home Assistant újraindítása",
    	show_beta: "Béta megjelenítése",
    	uninstall: "Eltávolítás",
    	update_information: "Frissítési információk",
    	upgrade: "Frissítés"
    };
    var search$2 = {
    	placeholder: "Tároló keresése"
    };
    var sections$4 = {
    	about: {
    		description: "Információk megjelenítése a HACS-ről",
    		title: "Névjegy"
    	},
    	automation: {
    		description: "Itt Python szkripteket, AppDaemon és NetDaemon appokat találsz",
    		title: "Automatizálás"
    	},
    	frontend: {
    		description: "Itt témákat, egyéni kártyákat és más bővítményeket találsz a Lovelace-hez",
    		title: "Frontend"
    	},
    	integrations: {
    		description: "Itt találod az egyéni integrációkat (custom_components)",
    		title: "Integrációk"
    	},
    	pending_repository_upgrade: "A(z) {installed} verziót futtatod, a(z) {available} verzió már elérhető"
    };
    var settings$6 = {
    	add_custom_repository: "EGYÉNI TÁROLÓ HOZZÁADÁSA",
    	adding_new_repo: "Új tároló hozzáadása '{repo}'",
    	adding_new_repo_category: "A '{category}' kategóriával.",
    	bg_task_custom: "Az egyéni tárolók rejtve vannak, amíg háttérfeladat fut.",
    	category: "Kategória",
    	compact_mode: "Kompakt mód",
    	custom_repositories: "EGYÉNI TÁROLÓK",
    	"delete": "Törlés",
    	display: "Megjelenítés",
    	grid: "Rács",
    	hacs_repo: "HACS tároló",
    	hidden_repositories: "rejtett tárolók",
    	missing_category: "Ki kell választanod egy kategóriát",
    	open_repository: "Tároló megnyitása",
    	reload_data: "Adatok újratöltése",
    	reload_window: "Ablak újratöltése",
    	repository_configuration: "Tároló konfiguráció",
    	save: "Mentés",
    	table: "Táblázat",
    	table_view: "Táblázat nézet",
    	unhide: "felfedés",
    	upgrade_all: "Minden frissítése"
    };
    var store$6 = {
    	ascending: "növekvő",
    	clear_new: "Új tárolók megjelölése látottként",
    	descending: "csökkenő",
    	last_updated: "Utolsó frissítés",
    	name: "Név",
    	new_repositories: "Új tárolók",
    	new_repositories_note: "Több mint 10 új tároló látható. Ha törölni szeretnéd őket, akkor kattints a jobb felső sarokban lévő 3 pontra, és válaszd ki a 'Minden új tároló elvetése' menüpontot.",
    	no_repositories: "Nincsenek tárolók",
    	no_repositories_desc1: "Úgy tűnik, még nincsenek telepítve tárolók ebben a szekcióban.",
    	no_repositories_desc2: "Kattints az alsó sarokban található + jelre az első hozzáadásához!",
    	pending_upgrades: "Frissítések érhetők el",
    	placeholder_search: "Kérlek adj meg egy keresési kifejezést...",
    	sort: "rendezés",
    	stars: "Csillag",
    	status: "Állapot"
    };
    var time$6 = {
    	ago: "ezelőtt",
    	day: "nap",
    	days: "nap",
    	hour: "óra",
    	hours: "óra",
    	minute: "perc",
    	minutes: "perc",
    	month: "hónap",
    	months: "hónap",
    	one: "Egy",
    	one_day_ago: "egy nappal ezelőtt",
    	one_hour_ago: "egy órával ezelőtt",
    	one_minute_ago: "egy perccel ezelőtt",
    	one_month_ago: "egy hónappal ezelőtt",
    	one_second_ago: "egy másodperccel ezelőtt",
    	one_year_ago: "egy évvel ezelőtt",
    	second: "másodperc",
    	seconds: "másodperc",
    	x_days_ago: "{x} nappal ezelőtt",
    	x_hours_ago: "{x} órával ezelőtt",
    	x_minutes_ago: "{x} perccel ezelőtt",
    	x_months_ago: "{x} hónappal ezelőtt",
    	x_seconds_ago: "{x} másodperccel ezelőtt",
    	x_years_ago: "{x} évvel ezelőtt",
    	year: "év",
    	years: "év"
    };
    var hu = {
    	common: common$6,
    	config: config$6,
    	confirm: confirm$6,
    	dialog_about: dialog_about$2,
    	dialog_add_repo: dialog_add_repo$3,
    	dialog_custom_repositories: dialog_custom_repositories$2,
    	dialog_info: dialog_info$2,
    	dialog_install: dialog_install$3,
    	dialog_update: dialog_update$2,
    	entry: entry$2,
    	menu: menu$3,
    	options: options$6,
    	repository_banner: repository_banner$6,
    	repository_card: repository_card$3,
    	repository: repository$6,
    	search: search$2,
    	sections: sections$4,
    	settings: settings$6,
    	store: store$6,
    	time: time$6
    };

    var hu$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$6,
        config: config$6,
        confirm: confirm$6,
        dialog_about: dialog_about$2,
        dialog_add_repo: dialog_add_repo$3,
        dialog_custom_repositories: dialog_custom_repositories$2,
        dialog_info: dialog_info$2,
        dialog_install: dialog_install$3,
        dialog_update: dialog_update$2,
        entry: entry$2,
        menu: menu$3,
        options: options$6,
        repository_banner: repository_banner$6,
        repository_card: repository_card$3,
        repository: repository$6,
        search: search$2,
        sections: sections$4,
        settings: settings$6,
        store: store$6,
        time: time$6,
        'default': hu
    });

    var common$7 = {
    	about: "Informazioni su",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "Applicazioni AppDaemon",
    	background_task: "Attività in esecuzione, questa pagina sarà ricaricata al termine.",
    	check_log_file: "Controlla il tuo file di registro per maggiori dettagli.",
    	"continue": "Continua",
    	disabled: "Disabilitato",
    	documentation: "Documentazione",
    	hacs_is_disabled: "HACS è disabilitato",
    	installed: "Installati",
    	integration: "Integrazione",
    	integrations: "Integrazioni",
    	manage: "gestione",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "Applicazioni NetDaemon",
    	plugin: "Plugin",
    	plugins: "Plugin",
    	python_script: "Script python",
    	python_scripts: "Script python",
    	repositories: "Repository",
    	settings: "Impostazioni",
    	theme: "Tema",
    	themes: "Temi",
    	version: "Versione"
    };
    var config$7 = {
    	abort: {
    		single_instance_allowed: "È consentita una sola configurazione di HACS."
    	},
    	error: {
    		auth: "Il token di accesso personale non è corretto."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Abilita il rilevamento e il monitoraggio delle applicazioni AppDaemon",
    				netdaemon: "Abilita il rilevamento e il monitoraggio delle applicazioni NetDaemon",
    				python_script: "Abilita il rilevamento e il monitoraggio dei python_scripts",
    				sidepanel_icon: "Icona nel pannello laterale",
    				sidepanel_title: "Titolo nel pannello laterale",
    				theme: "Abilita individuazione e tracciamento dei temi",
    				token: "Token di accesso personale GitHub"
    			},
    			description: "Se hai bisogno di aiuto con la configurazione dai un'occhiata qui: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$7 = {
    	add_to_lovelace: "Sei sicuro di voler aggiungerlo alle tue risorse Lovelace?",
    	bg_task: "L'azione è disabilitata mentre sono in esecuzione attività in background.",
    	cancel: "Annulla",
    	"continue": "Sei sicuro di voler continuare?",
    	"delete": "Sei sicuro di voler disinstallare '{item}'?",
    	delete_installed: "'{item}' è installato, è necessario disinstallarlo prima di poterlo eliminare.",
    	exist: "{item} esiste già",
    	generic: "Sei sicuro?",
    	home_assistant_is_restarting: "Aspetta, Home Assistant si sta riavviando.",
    	home_assistant_version_not_correct: "Stai eseguendo la versione Home Assistant '{haversion}', ma questo repository richiede l'installazione della versione minima '{minversion}'.",
    	no: "No",
    	no_upgrades: "Nessun aggiornamento in sospeso",
    	ok: "OK",
    	overwrite: "In questo modo lo sovrascriverà.",
    	reload_data: "Questo ricarica i dati di tutte le repository di cui HACS è a conoscenza, ci vorrà del tempo per finire.",
    	restart_home_assistant: "Sei sicuro di voler riavviare Home Assistant?",
    	uninstall: "Sei sicuro di voler disinstallare '{item}'?",
    	upgrade_all: "Questa azione aggiornerà tutti i repository, assicurati di aver letto le note di rilascio prima di procedere.",
    	yes: "Sì"
    };
    var options$7 = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Abilita il rilevamento e il monitoraggio delle applicazioni AppDaemon",
    				country: "Filtra con prefisso internazionale.",
    				debug: "Abilita debug.",
    				experimental: "Abilita funzionalità sperimentali",
    				netdaemon: "Abilita il rilevamento e il monitoraggio delle applicazioni NetDaemon",
    				not_in_use: "Non in uso con YAML",
    				release_limit: "Numero di versioni da mostrare.",
    				sidepanel_icon: "Icona nel pannello laterale",
    				sidepanel_title: "Titolo nel pannello laterale"
    			}
    		}
    	}
    };
    var repository_banner$7 = {
    	config_flow: "Questa integrazione supporta config_flow, questo significa che è ora possibile passare alla sezione \"IntegrazionI\" dell'interfaccia utente per la configurazione.",
    	config_flow_title: "Configurazione dell'interfaccia utente supportata",
    	integration_not_loaded: "Questa integrazione non è caricata in Home Assistant.",
    	no_restart_required: "Non è necessario riavviare",
    	not_loaded: "Non caricato",
    	plugin_not_loaded: "Questo plugin non è stato aggiunto alle risorse di Lovelace.",
    	restart: "È necessario riavviare Home Assistant.",
    	restart_pending: "Riavvio in attesa"
    };
    var repository$7 = {
    	add_to_lovelace: "Aggiungi a Lovelace",
    	authors: "Autori",
    	available: "Disponibile",
    	back_to: "Torna a",
    	changelog: "Change log",
    	downloads: "Downloads",
    	flag_this: "Spunta questo",
    	frontend_version: "Frontend versione",
    	github_stars: "GitHub stelle",
    	goto_integrations: "Vai alle Integrazioni",
    	hide: "Nascondi",
    	hide_beta: "Nascondi beta",
    	install: "Installa",
    	installed: "Installato",
    	lovelace_copy_example: "Copia l'esempio negli appunti",
    	lovelace_instruction: "Quando lo aggiungi nella configurazione di lovelace, usa questo",
    	lovelace_no_js_type: "Impossibile determinare il tipo di plugin, verificare il repository.",
    	newest: "Più recente",
    	note_appdaemon: "dovrai aggiungerlo nel file 'apps.yaml'",
    	note_installed: "Una volta installato, si troverà in",
    	note_integration: "dovrai aggiungerlo nel file 'configuration.yaml'",
    	note_plugin: "devi aggiungere la configurazione nel file 'ui-lovelace.yaml' oppure via Editor RAW della UI.",
    	note_plugin_post_107: "devi aggiungere la configurazione nel file 'ui-lovelace.yaml' oppure via Editor RAW della UI.",
    	open_issue: "Segnala anomalia",
    	open_plugin: "Apri plugin",
    	reinstall: "Reinstalla",
    	repository: "Archivio Software (Repository)",
    	restart_home_assistant: "Riavvia Home Assistant",
    	show_beta: "Visualizza beta",
    	uninstall: "Rimuovi",
    	update_information: "Aggiorna informazioni",
    	upgrade: "Aggiorna"
    };
    var settings$7 = {
    	add_custom_repository: "AGGIUNGI REPOSITORY PERSONALIZZATA",
    	adding_new_repo: "Aggiunta di un nuovo repository '{repo}'",
    	adding_new_repo_category: "Con la categoria '{category}'.",
    	bg_task_custom: "I repository personalizzati sono nascosti mentre sono in esecuzione attività in background.",
    	category: "Categoria",
    	compact_mode: "Modalità compatta",
    	custom_repositories: "REPOSITORY PERSONALIZZATE",
    	"delete": "Cancella",
    	display: "Visualizza",
    	grid: "Griglia",
    	hacs_repo: "HACS repo",
    	hidden_repositories: "repository nascosti",
    	missing_category: "Devi selezionare una categoria",
    	open_repository: "Apri il repository",
    	reload_data: "Ricarica i dati",
    	reload_window: "Ricarica la finestra",
    	repository_configuration: "Configurazione del repository",
    	save: "Salva",
    	table: "Tabella",
    	table_view: "Vista tabella",
    	unhide: "Mostra",
    	upgrade_all: "Aggiorna tutto"
    };
    var store$7 = {
    	ascending: "ascendente",
    	clear_new: "Ripulisci i nuovi repository",
    	descending: "discendente",
    	last_updated: "Ultimo aggiornamento",
    	name: "Nome",
    	new_repositories: "Nuovi repository",
    	pending_upgrades: "Aggiornamenti in sospeso",
    	placeholder_search: "Inserire un termine di ricerca",
    	sort: "Ordinare",
    	stars: "Stelle",
    	status: "Stato"
    };
    var time$7 = {
    	ago: "fa",
    	day: "giorno",
    	days: "giorni",
    	hour: "ora",
    	hours: "ore",
    	minute: "minuto",
    	minutes: "minuti",
    	month: "mese",
    	months: "mesi",
    	one: "Un",
    	one_day_ago: "un giorno fa",
    	one_hour_ago: "un'ora fa",
    	one_minute_ago: "un minuto fa",
    	one_month_ago: "un mese fa",
    	one_second_ago: "un secondo fa",
    	one_year_ago: "un anno fa",
    	second: "secondo",
    	seconds: "secondi",
    	x_days_ago: "{x} giorni fa",
    	x_hours_ago: "{x} ore fa",
    	x_minutes_ago: "{x} minuti fa",
    	x_months_ago: "{x} mesi fa",
    	x_seconds_ago: "{x} secondi fa",
    	x_years_ago: "{x} anni fa",
    	year: "anno",
    	years: "anni"
    };
    var it = {
    	common: common$7,
    	config: config$7,
    	confirm: confirm$7,
    	options: options$7,
    	repository_banner: repository_banner$7,
    	repository: repository$7,
    	settings: settings$7,
    	store: store$7,
    	time: time$7
    };

    var it$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$7,
        config: config$7,
        confirm: confirm$7,
        options: options$7,
        repository_banner: repository_banner$7,
        repository: repository$7,
        settings: settings$7,
        store: store$7,
        time: time$7,
        'default': it
    });

    var common$8 = {
    	about: "Om",
    	add: "legg til",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "AppDaemon Apper",
    	appdaemon_plural: "AppDaemon-apper",
    	background_task: "Bakgrunnsoppgaven kjører. Denne siden lastes inn på nytt når den er ferdig.",
    	check_log_file: "Sjekk loggfilen din for mer informasjon.",
    	"continue": "Fortsett",
    	disabled: "Deaktivert",
    	documentation: "dokumentasjon",
    	element: "element",
    	hacs_is_disabled: "HACS er deaktivert",
    	install: "Installer",
    	installed: "Installert",
    	integration: "Integrasjon",
    	integration_plural: "Integrasjoner",
    	integrations: "Integrasjoner",
    	lovelace: "Lovelace",
    	lovelace_element: "Lovelace-element",
    	lovelace_elements: "Lovelace-elementer",
    	manage: "manage",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDaemon Apper",
    	netdaemon_plural: "NetDaemon-apper",
    	plugin: "Lovelace",
    	plugin_plural: "Lovelace-elementer",
    	plugins: "Lovelace-elementer",
    	python_script: "Python-skript",
    	python_script_plural: "Python-skript",
    	python_scripts: "Python-skript",
    	repositories: "Repositories",
    	repository: "Repository",
    	settings: "Innstillinger",
    	theme: "Tema",
    	theme_plural: "Temaer",
    	themes: "Temaer",
    	uninstall: "Avinstaller",
    	update: "Oppdater",
    	version: "Versjon"
    };
    var config$8 = {
    	abort: {
    		single_instance_allowed: "Bare en konfigurasjon av HACS er tillatt."
    	},
    	error: {
    		auth: "Personlig tilgangstoken er ikke korrekt."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Aktiver oppdagelse og sporing av AppDaemon-apper",
    				netdaemon: "Aktiver oppdagelse og sporing av NetDaemon-apper",
    				python_script: "Aktiver oppdagelse og sporing av python_scripts",
    				sidepanel_icon: "Sidepanel ikon",
    				sidepanel_title: "Sidepanel tittel",
    				theme: "Aktiver oppdagelse og sporing av temaer",
    				token: "GitHub Personal Access Token"
    			},
    			description: "Hvis du trenger hjelp med konfigurasjonen, ta en titt her: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$8 = {
    	add_to_lovelace: "Er du sikker på at du vil legge dette til i dine Lovelace resources?",
    	bg_task: "Handlingen er deaktivert mens bakgrunnsoppgaver kjører.",
    	cancel: "Avbryt",
    	"continue": "Er du sikker på at du vil fortsette?",
    	"delete": "Er du sikker på at du vil fjerne '{item}'?",
    	delete_installed: "'{item}' er installert, du må avinstallere det før du kan slette det.",
    	exist: "{item} eksisterer allerede",
    	generic: "Er du sikker?",
    	home_assistant_is_restarting: "Vent, Home Assistant starter nå på nytt.",
    	home_assistant_version_not_correct: "Du kjører Home Assistant '{haversion}', men denne krever minimum versjon '{minversion}' for å bli installert.",
    	no: "Nei",
    	no_upgrades: "Ingen oppgraderinger tilgjengelig",
    	ok: "OK",
    	overwrite: "Å gjøre dette vil overskrive det.",
    	reload_data: "Dette laster inn dataene til alle repositories HACS vet om, dette vil ta litt tid å fullføre.",
    	restart_home_assistant: "Er du sikker på at du vil starte Home Assistant på nytt?",
    	uninstall: "Er du sikker på at du vil avinstallere '{item}'?",
    	upgrade_all: "Dette vil oppgradere alle disse repositorene, sørg for at du har lest utgivelses notatene for dem alle før du fortsetter.",
    	yes: "Ja"
    };
    var dialog_about$3 = {
    	frontend_version: "Frontend versjon",
    	installed_repositories: "Installerte repositories",
    	integration_version: "Integrasjonsversjon",
    	useful_links: "Nyttige lenker"
    };
    var dialog_add_repo$4 = {
    	limit: "Bare de første 100 elementene vises, bruk søket til å filtrere det du trenger",
    	no_match: "Ingen elementer funnet som samsvarer med filteret ditt",
    	title: "Legg til repository"
    };
    var dialog_custom_repositories$3 = {
    	category: "Kategori",
    	no_category: "Mangler kategori",
    	no_repository: "Mangler repository",
    	title: "Custom repositories",
    	url_placeholder: "Legg til custom repository"
    };
    var dialog_info$3 = {
    	author: "Utgiver",
    	downloads: "Nedlastinger",
    	install: "Installer dette elementet i HACS",
    	loading: "Laster inn informasjon ...",
    	no_info: "Utvikleren har ikke gitt mer informasjon for dette elementet",
    	open_issues: "Åpne problemer",
    	open_repo: "Åpne repository",
    	stars: "Stjerner",
    	version_installed: "Versjon installert"
    };
    var dialog_install$4 = {
    	select_version: "Velg versjon",
    	show_beta: "Vis betaversjoner",
    	type: "Type",
    	url: "URL"
    };
    var dialog_update$3 = {
    	available_version: "Tilgjengelig versjon",
    	changelog: "Endringslogg",
    	installed_version: "Installert versjon",
    	title: "Oppdatering venter"
    };
    var entry$3 = {
    	information: "Informasjon",
    	intro: "Oppdateringer og viktige meldinger vises her hvis det er noen",
    	messages: {
    		disabled: {
    			content: "Sjekk loggfilen din for mer informasjon",
    			title: "HACS er deaktivert"
    		},
    		has_pending_tasks: {
    			content: "Noen elementer vises kanskje ikke før dette er fullført",
    			title: "Venter på bakgrunnsoppgaver"
    		},
    		startup: {
    			content: "HACS starter opp, i løpet av denne tiden kan det hende at noe informasjon mangler eller er feil",
    			title: "HACS starter opp"
    		}
    	},
    	pending_updates: "Oppdateringer er klare"
    };
    var menu$4 = {
    	about: "Om HACS",
    	clear: "Fjern alt nytt",
    	custom_repositories: "Custom repositories",
    	documentation: "Dokumentasjon",
    	open_issue: "Meld et problem",
    	reload: "Last inn vinduet på nytt"
    };
    var options$8 = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Aktiver oppdagelse og sporing av AppDaemon-apper",
    				country: "Filtrer med landskode.",
    				debug: "Aktiver debug",
    				experimental: "Aktiver eksperimentelle funksjoner",
    				netdaemon: "Aktiver oppdagelse og sporing av NetDaemon-apper",
    				not_in_use: "Ikke i bruk med YAML",
    				release_limit: "Antall utgivelser som skal vises.",
    				sidepanel_icon: "Sidepanel ikon",
    				sidepanel_title: "Sidepanel tittel"
    			}
    		}
    	}
    };
    var repository_banner$8 = {
    	config_flow: "Denne integrasjonen støtter config_flow, det betyr at du nå kan gå til integrasjoner i brukergrensesnittet for å konfigurere den.",
    	config_flow_title: "UI konfigurasjon støttet",
    	integration_not_loaded: "Integrasjonen er ikke lastet inn i Home Assistant.",
    	no_restart_required: "Ingen omstart kreves",
    	not_loaded: "Ikke lastet inn",
    	plugin_not_loaded: "Dette elementet er ikke lagt til i lovelace under \"resource\" delen av konfigurasjonen.",
    	restart: "Du må restart Home Assistant",
    	restart_pending: "Restart er nødvendig"
    };
    var repository_card$4 = {
    	hide: "Skjul",
    	information: "Informasjon",
    	new_repository: "Ny",
    	open_issue: "Meld et problem",
    	open_source: "Åpne kilde",
    	pending_update: "Oppdatering venter",
    	reinstall: "Installer på nytt",
    	report: "Rapporter for fjerning",
    	update_information: "Oppdater informasjon"
    };
    var repository$8 = {
    	add_to_lovelace: "Legg til i Lovelace",
    	authors: "Laget av",
    	available: "Tilgjengelig",
    	back_to: "Tilbake til",
    	changelog: "Endringslogg",
    	downloads: "Nedlastinger",
    	flag_this: "Flag dette",
    	frontend_version: "Frontend versjon",
    	github_stars: "GitHub-stjerner",
    	goto_integrations: "Gå til integrasjoner",
    	hide: "Skjul",
    	hide_beta: "Skjul beta",
    	installed: "Installert",
    	lovelace_copy_example: "Kopier eksemplet til utklippstavlen",
    	lovelace_instruction: "Når du legger til dette i lovelace-konfigurasjonen din, bruk dette",
    	lovelace_no_js_type: "Kunne ikke bestemme typen for dettte elementet, sjekk repository.",
    	newest: "Nyeste",
    	note_appdaemon: "du må fortsatt legge den til i 'apps.yaml' filen",
    	note_installed: "Når det er installert, vil dette ligge i",
    	note_integration: "du må fortsatt legge den til 'configuration.yaml' filen",
    	note_plugin: "du må fortsatt legge den til i lovelace-konfigurasjonen ('ui-lovelace.yaml' eller den rå UI-konfigurasjonsredigereren)",
    	note_plugin_post_107: "du må fortsatt legge den til i lovelace konfigurasjonen ('configuration.yaml' eller via resource behanleren i grensesnittet '/config/lovelace/resources')",
    	open_issue: "Meld et problem",
    	open_plugin: "Åpne kilde",
    	reinstall: "Installer på nytt",
    	repository: "Repository",
    	restart_home_assistant: "Start Home Assistant på nytt",
    	show_beta: "Vis beta",
    	uninstall: "Avinstaller",
    	update_information: "Oppdater informasjon",
    	upgrade: "Oppdater"
    };
    var search$3 = {
    	placeholder: "Søk etter repository"
    };
    var sections$5 = {
    	about: {
    		description: "Vis informasjon om HACS",
    		title: "Om"
    	},
    	automation: {
    		description: "Det er her du finner python_scripts, AppDaemon-apper og NetDaemon-apper",
    		title: "Automasjon"
    	},
    	frontend: {
    		description: "Det er her du finner temaer, tilpassede kort og andre elementer for lovelace",
    		title: "Frontend"
    	},
    	integrations: {
    		description: "Det er her du finner tilpassede integrasjoner (custom_components)",
    		title: "Integrasjoner"
    	},
    	pending_repository_upgrade: "Du kjører versjon {installed} , versjon {available} er tilgjengelig"
    };
    var settings$8 = {
    	add_custom_repository: "LEGG TIL REPOSITORY",
    	adding_new_repo: "Legger til ny repository '{repo}'",
    	adding_new_repo_category: "Med kategori '{category}'.",
    	bg_task_custom: "Custom repositories er skjult mens bakgrunnsoppgaver kjører.",
    	category: "Kategori",
    	compact_mode: "Kompakt modus",
    	custom_repositories: "TILPASSEDE REPOSITORIER",
    	"delete": "Slett",
    	display: "Vise",
    	grid: "Nett",
    	hacs_repo: "HACS repo",
    	hidden_repositories: "Gjemte repositories",
    	missing_category: "Du må velge en kategori",
    	open_repository: "Åpne repository",
    	reload_data: "Last inn data på nytt",
    	reload_window: "Last inn vinduet på nytt",
    	repository_configuration: "Repository konfigurasjon",
    	save: "Lagre",
    	table: "Tabell",
    	table_view: "Tabellvisning",
    	unhide: "Vis igjen",
    	upgrade_all: "Oppgradere alle"
    };
    var store$8 = {
    	ascending: "stigende",
    	clear_new: "Tøm nye repositories",
    	descending: "synkende",
    	last_updated: "Sist oppdatert",
    	name: "Navn",
    	new_repositories: "Nye repositories",
    	no_repositories: "Ingen repositories",
    	no_repositories_desc1: "Det virker som om du ikke har noen elementer installert i denne delen ennå.",
    	no_repositories_desc2: "Klikk på + i nederste hjørne for å legge til din første!",
    	pending_upgrades: "Venter på oppgradering",
    	placeholder_search: "Skriv inn et søkeord ...",
    	sort: "sorter",
    	stars: "Stjerner",
    	status: "Status"
    };
    var time$8 = {
    	ago: "siden",
    	day: "dag",
    	days: "dager",
    	hour: "time",
    	hours: "timer",
    	minute: "minutt",
    	minutes: "minutter",
    	month: "måned",
    	months: "måneder",
    	one: "En",
    	one_day_ago: "for en dag siden",
    	one_hour_ago: "en time siden",
    	one_minute_ago: "ett minutt siden",
    	one_month_ago: "en måned siden",
    	one_second_ago: "ett sekund siden",
    	one_year_ago: "ett år siden",
    	second: "sekund",
    	seconds: "sekunder",
    	x_days_ago: "{x} dager siden",
    	x_hours_ago: "{x} timer siden",
    	x_minutes_ago: "{x} minutter siden",
    	x_months_ago: "{x} måneder siden",
    	x_seconds_ago: "{x} sekunder siden",
    	x_years_ago: "{x} år siden",
    	year: "år",
    	years: "år"
    };
    var nb = {
    	common: common$8,
    	config: config$8,
    	confirm: confirm$8,
    	dialog_about: dialog_about$3,
    	dialog_add_repo: dialog_add_repo$4,
    	dialog_custom_repositories: dialog_custom_repositories$3,
    	dialog_info: dialog_info$3,
    	dialog_install: dialog_install$4,
    	dialog_update: dialog_update$3,
    	entry: entry$3,
    	menu: menu$4,
    	options: options$8,
    	repository_banner: repository_banner$8,
    	repository_card: repository_card$4,
    	repository: repository$8,
    	search: search$3,
    	sections: sections$5,
    	settings: settings$8,
    	store: store$8,
    	time: time$8
    };

    var nb$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$8,
        config: config$8,
        confirm: confirm$8,
        dialog_about: dialog_about$3,
        dialog_add_repo: dialog_add_repo$4,
        dialog_custom_repositories: dialog_custom_repositories$3,
        dialog_info: dialog_info$3,
        dialog_install: dialog_install$4,
        dialog_update: dialog_update$3,
        entry: entry$3,
        menu: menu$4,
        options: options$8,
        repository_banner: repository_banner$8,
        repository_card: repository_card$4,
        repository: repository$8,
        search: search$3,
        sections: sections$5,
        settings: settings$8,
        store: store$8,
        time: time$8,
        'default': nb
    });

    var common$9 = {
    	about: "Over",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "AppDaemon Apps",
    	background_task: "Achtergrond taak is draaiende, de pagina herhaalt zichzelf wanneer dit klaar is.",
    	check_log_file: "Controleer het logbestand voor meer details.",
    	"continue": "Doorgaan",
    	disabled: "Uitgeschakeld",
    	documentation: "Documentatie",
    	hacs_is_disabled: "HACS is uitgeschakeld",
    	installed: "geinstalleerd",
    	integration: "Integratie",
    	integrations: "Integraties",
    	manage: "beheer",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDaemon Apps",
    	plugin: "Plugin",
    	plugins: "Plugins",
    	python_script: "Python Script",
    	python_scripts: "Python Scripts",
    	repositories: "Repositories",
    	settings: "instellingen",
    	theme: "Thema",
    	themes: "Themas",
    	version: "Versie"
    };
    var config$9 = {
    	abort: {
    		single_instance_allowed: "Je kunt maar een enkele configuratie van HACS tegelijk hebben."
    	},
    	error: {
    		auth: "Persoonlijke Toegang Token is niet correct."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Zet AppDaemon apps ontdekken & traceren aan",
    				netdaemon: "Zet NetDaemon apps ontdekken & traceren aan",
    				python_script: "Zet python_scripts ontdekken & traceren aan",
    				sidepanel_icon: "Zijpaneel icoon",
    				sidepanel_title: "Zijpaneel titel",
    				theme: "Zet Themes ontdekken & traceren aan",
    				token: "GitHub Persoonlijke Toegang Token"
    			},
    			description: "Als je hulp nodig hebt met de configuratie, kun je hier verder kijken: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$9 = {
    	add_to_lovelace: "Weet u zeker dat u dit wilt toevoegen aan uw Lovelace bronnen?",
    	bg_task: "Actie is geblokkeerd terwijl achtergrondtaken actief zijn.",
    	cancel: "Annuleer",
    	"continue": "Weet je zeker dat je wilt doorgaan?",
    	"delete": "Weet u zeker dat u '{item}' wilt verwijderen?",
    	delete_installed: "'{item}' is geïnstalleerd, je dient het eerst te deïnstalleren voordat je het kan verwijderen.",
    	exist: "{item} bestaat al.",
    	generic: "Weet je het zeker?",
    	home_assistant_is_restarting: "Een moment alstublieft, Home Assistant is aan het herstarten.",
    	home_assistant_version_not_correct: "Je gebruikt Home Assistant versie '{haversion}', echter deze repository vereist dat minimaal versie '{minversion}' is geïnstalleerd.",
    	no: "Nee",
    	no_upgrades: "Geen upgrades in afwachting.",
    	ok: "Oké",
    	overwrite: "Door dit te doen, wordt het overschreven.",
    	reload_data: "Dit zal alle bekende data herladen van alle repositories van HACS. Dit kan even duren",
    	restart_home_assistant: "Weet u zeker dat u Home Assistant opnieuw wilt starten?",
    	uninstall: "Weet u zeker dat u '{item}' wilt verwijderen?",
    	upgrade_all: "Hiermee worden al deze repositories geüpgraded. Zorg ervoor dat u de release-opmerkingen van allen heeft gelezen voordat u doorgaat.",
    	yes: "Ja"
    };
    var options$9 = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Zet AppDaemon apps ontdekken & traceren aan",
    				country: "Filter met land code.",
    				debug: "Schakel debug in.",
    				experimental: "Zet experimentele functies aan",
    				netdaemon: "Zet NetDaemon apps ontdekken & traceren aan",
    				not_in_use: "Niet in gebruik met YAML",
    				release_limit: "Aantal releases om te laten zien.",
    				sidepanel_icon: "Zijpaneel icoon",
    				sidepanel_title: "Zijpaneel titel"
    			}
    		}
    	}
    };
    var repository_banner$9 = {
    	config_flow: "Deze integratie ondersteunt config_flow, wat betekent dat u via uw \"Instellingen\" naar \"Integraties\" kunt gaan om het te configureren.",
    	config_flow_title: "UI-configuratie ondersteund",
    	integration_not_loaded: "Deze integratie wordt niet geladen in Home Assistant.",
    	no_restart_required: "Geen herstart vereist",
    	not_loaded: "Niet geladen",
    	plugin_not_loaded: "Deze plugin wordt niet toegevoegd aan je Lovelace resources.",
    	restart: "U moet Home Assistant opnieuw starten.",
    	restart_pending: "Wachten op herstart"
    };
    var repository$9 = {
    	add_to_lovelace: "Toevoegen aan Lovelace",
    	authors: "Auteurs",
    	available: "Beschikbaar",
    	back_to: "Terug naar",
    	changelog: "Changelog",
    	downloads: "Downloads",
    	flag_this: "Vlag dit",
    	frontend_version: "Frontend versie",
    	github_stars: "GitHub-sterren",
    	goto_integrations: "Ga naar integraties",
    	hide: "Verberg",
    	hide_beta: "Verberg beta",
    	install: "Installeer",
    	installed: "Geinstalleerd",
    	lovelace_copy_example: "Kopier het voorbeeld naar je klembord",
    	lovelace_instruction: "Wanneer je dit gaat toevoegen aan je lovelace configuratie gebruik dit",
    	lovelace_no_js_type: "Kon niet achterhalen welk type plugin dit is, controleer de repository van de plugin.",
    	newest: "nieuwste",
    	note_appdaemon: "je moet het nog steeds toevoegen aan je 'apps.yaml' bestand",
    	note_installed: "Wanneer geïnstalleerd, staat het in",
    	note_integration: "je moet het nog steeds toevoegen aan je 'configuration.yaml' bestand",
    	note_plugin: "je moet het nog steeds toevoegen aan je lovelace configuratie ('ui-lovelace.yaml') of raw UI config editor.",
    	note_plugin_post_107: "je moet het nog steeds toevoegen aan je lovelace configuratie ('configuration.yaml' of de resource editor '/config/lovelace/resources')",
    	open_issue: "Meld probleem",
    	open_plugin: "Open plugin",
    	reinstall: "Herinstalleer",
    	repository: "Repository",
    	restart_home_assistant: "Start Home Assistant opnieuw",
    	show_beta: "Laat beta zien",
    	uninstall: "Verwijder",
    	update_information: "Update informatie",
    	upgrade: "Update"
    };
    var settings$9 = {
    	add_custom_repository: "VOEG EIGEN REPOSITORY TOE",
    	adding_new_repo: "Nieuwe repository '{repo}' toevoegen",
    	adding_new_repo_category: "Met categorie '{category}'.",
    	bg_task_custom: "Aangepaste repositories zijn verborgen terwijl de achtergrondtaken actief zijn.",
    	category: "Categorie",
    	compact_mode: "Compacte modus",
    	custom_repositories: "EIGEN REPOSITORIES",
    	"delete": "Verwijder",
    	display: "Weergave",
    	grid: "Rooster",
    	hacs_repo: "HACS repo",
    	hidden_repositories: "verborgen repositories",
    	missing_category: "Je moet een categorie selecteren.",
    	open_repository: "Open repository",
    	reload_data: "Herlaad data",
    	reload_window: "Herlaad venster",
    	repository_configuration: "Repository configuratie",
    	save: "Opslaan",
    	table: "Tabel",
    	table_view: "Tabelweergave",
    	unhide: "zichtbaar maken",
    	upgrade_all: "Upgrade alles"
    };
    var store$9 = {
    	ascending: "oplopend",
    	clear_new: "Wissen van alle nieuwe repositories",
    	descending: "Aflopend",
    	last_updated: "Laatste update",
    	name: "Naam",
    	new_repositories: "Nieuwe Repositories",
    	pending_upgrades: "Upgrades in afwachting",
    	placeholder_search: "Typ iets om te zoeken...",
    	sort: "sorteer",
    	stars: "Sterren",
    	status: "Status"
    };
    var time$9 = {
    	ago: "geleden",
    	day: "dag",
    	days: "dagen",
    	hour: "uur",
    	hours: "uren",
    	minute: "minuut",
    	minutes: "minuten",
    	month: "maand",
    	months: "maanden",
    	one: "Eén",
    	one_day_ago: "een dag geleden",
    	one_hour_ago: "een uur geleden",
    	one_minute_ago: "een minuut geleden",
    	one_month_ago: "een maand geleden",
    	one_second_ago: "een seconde geleden",
    	one_year_ago: "een jaar geleden",
    	second: "seconde",
    	seconds: "seconden",
    	x_days_ago: "{x} dagen geleden",
    	x_hours_ago: "{x} uur geleden",
    	x_minutes_ago: "{x} minuten geleden",
    	x_months_ago: "{x} maanden geleden",
    	x_seconds_ago: "{x} seconden geleden",
    	x_years_ago: "{x} jaar geleden",
    	year: "jaar",
    	years: "jaren"
    };
    var nl = {
    	common: common$9,
    	config: config$9,
    	confirm: confirm$9,
    	options: options$9,
    	repository_banner: repository_banner$9,
    	repository: repository$9,
    	settings: settings$9,
    	store: store$9,
    	time: time$9
    };

    var nl$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$9,
        config: config$9,
        confirm: confirm$9,
        options: options$9,
        repository_banner: repository_banner$9,
        repository: repository$9,
        settings: settings$9,
        store: store$9,
        time: time$9,
        'default': nl
    });

    var common$a = {
    	about: "Om",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "AppDeamon-appar",
    	appdaemon_plural: "AppDaemon-appar",
    	background_task: "Bakgrunnsoppgåve køyrer. Denne sida kjem til å laste seg omatt når ho er ferdig.",
    	check_log_file: "Sjå i loggfila di for meir detaljar.",
    	"continue": "Hald fram",
    	disabled: "Deaktivert",
    	documentation: "Dokumentasjon",
    	element: "element",
    	hacs_is_disabled: "HACS er deaktivert",
    	installed: "Installert",
    	integration: "Integrasjon",
    	integration_plural: "Integrasjonar",
    	integrations: "Integrasjonar",
    	lovelace: "Lovelace",
    	lovelace_element: "Lovelace-element",
    	lovelace_elements: "Lovelace-element",
    	manage: "Handtere",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDeamon-appar",
    	netdaemon_plural: "NetDaemon-appar",
    	plugin: "Tillegg",
    	plugin_plural: "Lovelace-element",
    	plugins: "Tillegg",
    	python_script: "Pythonskript",
    	python_script_plural: "Pythonskript",
    	python_scripts: "Pythonskript",
    	repositories: "Repositories",
    	settings: "innstillingar",
    	theme: "Tema",
    	theme_plural: "Tema",
    	themes: "Tema",
    	version: "Versjon"
    };
    var config$a = {
    	abort: {
    		single_instance_allowed: "Berre éin enkelt konfigurasjon av HACS er tillete."
    	},
    	error: {
    		auth: "Personleg tilgangsnøkkel er ikkje korrekt."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Aktiver AppDeamon-appar-oppdaging og sporing",
    				netdaemon: "Aktiver NetDeamon-appar-oppdaging og sporing",
    				python_script: "Aktiver pythonscript-oppdaging og sporing",
    				sidepanel_icon: "Sidepanelikon",
    				sidepanel_title: "Sidepaneltittel",
    				theme: "Aktiver tema-oppdaging og sporing",
    				token: "Personleg GitHub tilgangsnøkkel"
    			},
    			description: "Dersom du treng hjelp med konfigurasjonen, ta ein kik her: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$a = {
    	add_to_lovelace: "Er du sikker på at du vil legge til dette i Lovelace-ressursane dine?",
    	bg_task: "Handlinga er deaktivert medan bakgrunnsoppgåveene køyrer.",
    	cancel: "Avbryt",
    	"continue": "Er du sikker på at du vil halde fram?",
    	"delete": "Er du sikker på at du vil slette '{item}'?",
    	delete_installed: "'{item}' er installert. Du må avinstallere det før du kan slette det.",
    	exist: "{item} eksisterer allereie",
    	generic: "Er du sikker?",
    	home_assistant_is_restarting: "Vent... Home Assistant starter på nytt no.",
    	home_assistant_version_not_correct: "Du køyrer Home Assistant-versjonen '{haversion}', men dette kodedepoet krev minst versjon '{minversion}' for å bli installert.",
    	no: "Nei",
    	no_upgrades: "Ingen ventande oppgradringer",
    	ok: "OK",
    	overwrite: "Ved å gjere dette kjem du til å overskrive.",
    	reload_data: "Dette laster inn dataa til depota HACS veit om, og dette vil ta litt tid å fullføre.",
    	restart_home_assistant: "Er du sikker på at du vil starte Home Assistant på nytt?",
    	uninstall: "Er du sikker på at du vil avinstallere '{item}'?",
    	upgrade_all: "Dette kjem til å oppgradere alle depota. Ver sikker på at du har lest alle versjonsmerknadene før du held fram.",
    	yes: "Ja"
    };
    var options$a = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Aktiver AppDeamon-appar-oppdaging og sporing",
    				country: "Filterer med landskode",
    				debug: "Aktiver debug.",
    				experimental: "Aktiver ekspreimentelle funksjonar",
    				netdaemon: "Aktiver NetDeamon-appar-oppdaging og sporing",
    				not_in_use: "Kan ikkje brukast saman med YAML",
    				release_limit: "Talet på utgivingar",
    				sidepanel_icon: "Sidepanelikon",
    				sidepanel_title: "Sidepaneltittel"
    			}
    		}
    	}
    };
    var repository_banner$a = {
    	config_flow: "Denne integrasjonen støttar config_flow, som betyr at du no kan gå til integrasjonssida i brukargrensesnittet for å konfigurere den.",
    	config_flow_title: "UI-konfigurasjon støtta",
    	integration_not_loaded: "Integrasjonen er ikkje lasta inn i Home Assistant.",
    	no_restart_required: "Ingen omstart kravd",
    	not_loaded: "Ikkje lasta",
    	plugin_not_loaded: "Tillegget er ikkje lagt til i Lovelace-ressursane dine.",
    	restart: "Du må starte Home Assistant på nytt",
    	restart_pending: "Omstart venter"
    };
    var repository$a = {
    	add_to_lovelace: "Legg til i Lovelace",
    	authors: "Forfatter(e)",
    	available: "Tilgjengeleg",
    	back_to: "Tilbake til",
    	changelog: "Endre logg",
    	downloads: "Nedlastinger",
    	flag_this: "Marker dette",
    	frontend_version: "Frontend-versjon",
    	github_stars: "GitHub-stjerner",
    	goto_integrations: "Gå til integrasjonar",
    	hide: "Gøym",
    	hide_beta: "Gøym beta",
    	install: "Installer",
    	installed: "Installert",
    	lovelace_copy_example: "Kopier eksempelet til utklippsbreittet ditt",
    	lovelace_instruction: "Når du legg til dette i Lovelace-konfigurasjonen din, bruk dette",
    	lovelace_no_js_type: "Kunne ikkje slå fast typen til dette tilegget. Sjå i repositoryet.",
    	newest: "nyaste",
    	note_appdaemon: "du må framleis legge dette til i \"apps.yaml\"-fila di",
    	note_installed: "Når dette er installert, kjem den til å vere plassert i",
    	note_integration: "du må framleis legge dette til i \"configuration.yaml\"-fila di",
    	note_plugin: "du må framleis dette til i Lovelace-konfigurasjonen (\"ui-lovelace.yaml\" eller i rå-brukargrensesnittredigeraren",
    	note_plugin_post_107: "du må framleis legge dette til i lovelace-konfigurasjonen ('configuration.yaml' eller i kjelderedigeraren ''/config/lovelace/resources')",
    	open_issue: "Opne problem",
    	open_plugin: "Opne tillegg",
    	reinstall: "Installer på nytt",
    	repository: "Repository",
    	restart_home_assistant: "Start Home Assistant på nytt",
    	show_beta: "Vis beta",
    	uninstall: "Avinstaller",
    	update_information: "Oppdater informasjonen",
    	upgrade: "Oppdater"
    };
    var sections$6 = {
    	about: {
    		description: "Vis informasjon om HACS",
    		title: "Om"
    	},
    	automation: {
    		description: "Her finn du python_scripts, AppDaemon-appar og NetDaemon-appar",
    		title: "Automasjon"
    	},
    	frontend: {
    		description: "Her finn du tema, eigendefinerte kort og andre element for lovelace",
    		title: "Frontend"
    	},
    	integrations: {
    		description: "Her finn du eigendefinerte ingtegrasjonar (custom_components)",
    		title: "Integrasjonar"
    	},
    	pending_repository_upgrade: "Du køyrer versjon {installed}, og versjon {available} er tilgjengeleg"
    };
    var settings$a = {
    	add_custom_repository: "LEGG TIL EIN ANNAN REPOSITORY",
    	adding_new_repo: "Legger til ny repository '{repo}'",
    	adding_new_repo_category: "Med kategori '{category}'.",
    	bg_task_custom: "Custom repositories er skjult medan bakgrunnsoppgaver køyrer.",
    	category: "Kategori",
    	compact_mode: "Kompaktmodus",
    	custom_repositories: "VANLEG REPOSITORY",
    	"delete": "Slett",
    	display: "Vis",
    	grid: "rutenett",
    	hacs_repo: "HACS repo",
    	hidden_repositories: "gøymde repositories",
    	missing_category: "Du må velje éin kategori",
    	open_repository: "Opne repository",
    	reload_data: "Last om dataa",
    	reload_window: "Last inn vindauget på nytt",
    	repository_configuration: "Repository-konfigurasjon",
    	save: "Lagre",
    	table: "Tavle",
    	table_view: "Tabellvisning",
    	unhide: "vis",
    	upgrade_all: "Oppdater alle"
    };
    var store$a = {
    	ascending: "stigande",
    	clear_new: "Fjern alle nye repositories",
    	descending: "synkande",
    	last_updated: "Sist oppdatert",
    	name: "Namn",
    	new_repositories: "Ny repository",
    	pending_upgrades: "Ventande oppgraderinger",
    	placeholder_search: "Ver vennleg og skriv inn ei søkefrase",
    	sort: "Sorter",
    	stars: "Stjerner",
    	status: "Status"
    };
    var time$a = {
    	ago: "sidan",
    	day: "dag",
    	days: "dagar",
    	hour: "time",
    	hours: "timar",
    	minute: "minutt",
    	minutes: "minutt",
    	month: "månad",
    	months: "månadar",
    	one: "Éin",
    	one_day_ago: "for éin dag sidan",
    	one_hour_ago: "éin time sidan",
    	one_minute_ago: "eitt minutt sidan",
    	one_month_ago: "ein månad sidan",
    	one_second_ago: "eitt sekund sidan",
    	one_year_ago: "eitt år sidan",
    	second: "sekund",
    	seconds: "sekund",
    	x_days_ago: "{x} dagar siden",
    	x_hours_ago: "{x} timer sidan",
    	x_minutes_ago: "{x} minutt sidan",
    	x_months_ago: "{x} månadar sidan",
    	x_seconds_ago: "{x} sekund sidan",
    	x_years_ago: "{x} år sidan",
    	year: "år",
    	years: "år"
    };
    var nn = {
    	common: common$a,
    	config: config$a,
    	confirm: confirm$a,
    	options: options$a,
    	repository_banner: repository_banner$a,
    	repository: repository$a,
    	sections: sections$6,
    	settings: settings$a,
    	store: store$a,
    	time: time$a
    };

    var nn$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$a,
        config: config$a,
        confirm: confirm$a,
        options: options$a,
        repository_banner: repository_banner$a,
        repository: repository$a,
        sections: sections$6,
        settings: settings$a,
        store: store$a,
        time: time$a,
        'default': nn
    });

    var common$b = {
    	about: "O",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "Aplikacje AppDaemon",
    	appdaemon_plural: "Aplikacje AppDaemon",
    	background_task: "Wykonywanie zadania w tle, ta strona zostanie odświeżona, gdy zadanie zostanie ukończone.",
    	check_log_file: "Sprawdź plik dziennika, aby uzyskać więcej informacji.",
    	"continue": "Kontynuuj",
    	disabled: "Wyłączony",
    	documentation: "Dokumentacja",
    	hacs_is_disabled: "HACS jest wyłączony",
    	installed: "zainstalowane",
    	integration: "Integracja",
    	integration_plural: "Integracje",
    	integrations: "Integracje",
    	lovelace: "Lovelace",
    	lovelace_element: "Element Lovelace",
    	lovelace_elements: "Elementy Lovelace",
    	manage: "zarządzaj",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "Aplikacje NetDaemon",
    	netdaemon_plural: "Aplikacje NetDaemon",
    	plugin: "Wtyczka",
    	plugin_plural: "Elementy Lovelace",
    	plugins: "Wtyczki",
    	python_script: "Skrypt Python",
    	python_script_plural: "Skrypty języka Python",
    	python_scripts: "Skrypty Python",
    	repositories: "Repozytoria",
    	settings: "ustawienia",
    	theme: "Motyw",
    	theme_plural: "Motywy",
    	themes: "Motywy",
    	version: "Wersja"
    };
    var config$b = {
    	abort: {
    		single_instance_allowed: "Dozwolona jest tylko jedna konfiguracja HACS."
    	},
    	error: {
    		auth: "Osobisty token dostępu jest nieprawidłowy."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Włącz wykrywanie i śledzenie aplikacji AppDaemon",
    				netdaemon: "Włącz wykrywanie i śledzenie aplikacji NetDaemon",
    				python_script: "Włącz wykrywanie i śledzenie skryptów Python",
    				sidepanel_icon: "Ikona w panelu bocznym",
    				sidepanel_title: "Tytuł w panelu bocznym",
    				theme: "Włącz wykrywanie i śledzenie motywów",
    				token: "Osobisty token dostępu GitHub"
    			},
    			description: "Jeśli potrzebujesz pomocy w konfiguracji, przejdź na stronę: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$b = {
    	add_to_lovelace: "Na pewno chcesz dodać to do swoich zasobów Lovelace?",
    	bg_task: "Akcja jest wyłączona podczas wykonywania zadań w tle.",
    	cancel: "Anuluj",
    	"continue": "Na pewno chcesz kontynuować?",
    	"delete": "Na pewno chcesz usunąć '{item}'?",
    	delete_installed: "'{item}' jest zainstalowany, musisz go odinstalować zanim będziesz mógł go usunąć.",
    	exist: "{item} już istnieje",
    	generic: "Jesteś pewny?",
    	home_assistant_is_restarting: "Poczekaj, Home Assistant jest teraz ponownie uruchamiany.",
    	home_assistant_version_not_correct: "Używasz Home Assistant'a w wersji '{haversion}', a to repozytorium wymaga wersji minimum '{minversion}'.",
    	no: "Nie",
    	no_upgrades: "Brak oczekujących aktualizacji",
    	ok: "Ok",
    	overwrite: "Spowoduje to zastąpienie istniejącej kopii.",
    	reload_data: "To przeładowuje dane wszystkich repozytoriów, o których wie HACS, może to trochę potrwać.",
    	restart_home_assistant: "Na pewno chcesz ponownie uruchomić Home Assistant'a?",
    	uninstall: "Na pewno chcesz odinstalować '{item}'?",
    	upgrade_all: "To uaktualni wszystkie te repozytoria, upewnij się, że przeczytałeś uwagi do wydania dla wszystkich z nich przed kontynuacją.",
    	yes: "Tak"
    };
    var options$b = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Włącz wykrywanie i śledzenie aplikacji AppDaemon",
    				country: "Filtruj według kodu kraju",
    				debug: "Włącz debugowanie.",
    				experimental: "Włącz funkcje eksperymentalne",
    				netdaemon: "Włącz wykrywanie i śledzenie aplikacji NetDaemon",
    				not_in_use: "Nieużywany z YAML",
    				release_limit: "Liczba wydań do wyświetlenia",
    				sidepanel_icon: "Ikona w panelu bocznym",
    				sidepanel_title: "Tytuł w panelu bocznym"
    			}
    		}
    	}
    };
    var repository_banner$b = {
    	config_flow: "Ta integracja obsługuje config_flow, co oznacza, że możesz teraz przejść do sekcji integracji w interfejsie użytkownika, aby ją skonfigurować.",
    	config_flow_title: "Obsługiwana konfiguracja poprzez interfejs użytkownika",
    	integration_not_loaded: "Ta integracja nie jest załadowana do Home Assistant'a.",
    	no_restart_required: "Ponowne uruchomienie nie jest wymagane",
    	not_loaded: "Nie załadowano",
    	plugin_not_loaded: "Ta wtyczka nie jest dodana do zasobów Lovelace.",
    	restart: "Musisz ponownie uruchomić Home Assistant'a.",
    	restart_pending: "Oczekiwanie na ponowne uruchomienie"
    };
    var repository$b = {
    	add_to_lovelace: "Dodaj do Lovelace",
    	authors: "Autorzy",
    	available: "Dostępna",
    	back_to: "Wróć do",
    	changelog: "Lista zmian",
    	downloads: "Ilość pobrań",
    	flag_this: "Oflaguj",
    	frontend_version: "Wersja frontendu",
    	github_stars: "Gwiazdki GitHub",
    	goto_integrations: "Przejdź do integracji",
    	hide: "Ukryj",
    	hide_beta: "Ukryj wydania beta",
    	install: "Zainstaluj",
    	installed: "Zainstalowano",
    	lovelace_copy_example: "Skopiuj przykład do schowka",
    	lovelace_instruction: "Interfejs użytkownika użyje tej wtyczki po dodaniu konfiguracji",
    	lovelace_no_js_type: "Nie można określić typu tej wtyczki, sprawdź repozytorium.",
    	newest: "najnowsza",
    	note_appdaemon: "musisz jeszcze dodać aplikację do pliku 'apps.yaml'",
    	note_installed: "Po zainstalowaniu dodatek będzie znajdować się w",
    	note_integration: "musisz jeszcze dodać integrację do pliku 'configuration.yaml'",
    	note_plugin: "musisz jeszcze dodać wtyczkę do konfiguracji interfejsu użytkownika (plik 'ui-lovelace.yaml' lub edytor interfejsu użytkownika)",
    	note_plugin_post_107: "nadal musisz dodać go do konfiguracji Lovelace ('configuration.yaml' lub edytora zasobów '/config/lovelace/resources')",
    	open_issue: "Powiadom o problemie",
    	open_plugin: "Otwórz dodatek",
    	reinstall: "Przeinstaluj",
    	repository: "Repozytorium",
    	restart_home_assistant: "Uruchom ponownie Home Assistant'a",
    	show_beta: "Wyświetl wydania beta",
    	uninstall: "Odinstaluj",
    	update_information: "Uaktualnij informacje",
    	upgrade: "Uaktualnij"
    };
    var sections$7 = {
    	about: {
    		description: "Pokaż informacje o HACS"
    	},
    	automation: {
    		description: "Tutaj znajdziesz skrypty Pythona, aplikacje AppDaemon i NetDaemon",
    		title: "Automatyzacja"
    	},
    	frontend: {
    		description: "Tutaj znajdziesz motywy, niestandardowe karty i inne elementy dla Lovelace"
    	},
    	integrations: {
    		description: "Tutaj znajdziesz niestandardowe integracje (custom_components)",
    		title: "Integracje"
    	}
    };
    var settings$b = {
    	add_custom_repository: "DODAJ REPOZYTORIUM NIESTANDARDOWE",
    	adding_new_repo: "Dodawanie nowego repozytorium '{repo}'",
    	adding_new_repo_category: "Z kategorią '{category}'.",
    	bg_task_custom: "Niestandardowe repozytoria są ukryte podczas wykonywania zadań w tle.",
    	category: "Kategoria",
    	compact_mode: "Tryb kompaktowy",
    	custom_repositories: "REPOZYTORIA NIESTANDARDOWE",
    	"delete": "Usuń",
    	display: "Sposób wyświetlania",
    	grid: "Siatka",
    	hacs_repo: "Repozytorium HACS",
    	hidden_repositories: "ukryte repozytoria",
    	missing_category: "Musisz wybrać kategorię",
    	open_repository: "Otwórz repozytorium",
    	reload_data: "Wczytaj ponownie dane",
    	reload_window: "Załaduj ponownie okno",
    	repository_configuration: "Konfiguracja repozytorium",
    	save: "Zapisz",
    	table: "tabela",
    	table_view: "Widok tabeli",
    	unhide: "pokaż",
    	upgrade_all: "Uaktualnij wszystkie"
    };
    var store$b = {
    	ascending: "rosnąco",
    	clear_new: "Wyczyść wszystkie nowe repozytoria",
    	descending: "malejąco",
    	last_updated: "Ostatnia aktualizacja",
    	name: "Nazwa",
    	new_repositories: "Nowe repozytoria",
    	pending_upgrades: "Oczekujące aktualizacje",
    	placeholder_search: "Wprowadź wyszukiwane hasło...",
    	sort: "sortowanie",
    	stars: "Gwiazdki",
    	status: "Status"
    };
    var time$b = {
    	ago: "temu",
    	day: "dzień",
    	days: "dni",
    	hour: "godzina",
    	hours: "godziny",
    	minute: "minuta",
    	minutes: "minuty",
    	month: "miesiąc",
    	months: "miesięcy",
    	one: "Jeden",
    	one_day_ago: "jeden dzień temu",
    	one_hour_ago: "jedna godzina temu",
    	one_minute_ago: "jedna minuta temu",
    	one_month_ago: "jeden miesiąc temu",
    	one_second_ago: "jedna sekunda temu",
    	one_year_ago: "jeden rok temu",
    	second: "sekunda",
    	seconds: "sekundy",
    	x_days_ago: "{x} dni temu",
    	x_hours_ago: "{x} godzin(y) temu",
    	x_minutes_ago: "{x} minut(y) temu",
    	x_months_ago: "{x} miesi(ące/ęcy) temu",
    	x_seconds_ago: "{x} sekund(y) temu",
    	x_years_ago: "{x} lat(a) temu",
    	year: "rok",
    	years: "lata"
    };
    var pl = {
    	common: common$b,
    	config: config$b,
    	confirm: confirm$b,
    	options: options$b,
    	repository_banner: repository_banner$b,
    	repository: repository$b,
    	sections: sections$7,
    	settings: settings$b,
    	store: store$b,
    	time: time$b
    };

    var pl$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$b,
        config: config$b,
        confirm: confirm$b,
        options: options$b,
        repository_banner: repository_banner$b,
        repository: repository$b,
        sections: sections$7,
        settings: settings$b,
        store: store$b,
        time: time$b,
        'default': pl
    });

    var common$c = {
    	about: "Sobre",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "AppDaemon Apps",
    	background_task: "Tarefa em segundo plano em execução, esta página será recarregada quando terminar.",
    	check_log_file: "Verifique seu arquivo de log para obter mais detalhes.",
    	"continue": "Continuar",
    	disabled: "Desativado",
    	documentation: "Documentação",
    	hacs_is_disabled: "HACS está desativado",
    	installed: "instalado",
    	integration: "Integração",
    	integrations: "Integrações",
    	manage: "gerenciar",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDaemon Apps",
    	plugin: "Plugin",
    	plugins: "Plugins",
    	python_script: "Python Script",
    	python_scripts: "Python Scripts",
    	repositories: "Repositórios",
    	settings: "configurações",
    	theme: "Tema",
    	themes: "Temas",
    	version: "Versão"
    };
    var config$c = {
    	abort: {
    		single_instance_allowed: "Apenas uma configuração do HACS é permitida."
    	},
    	error: {
    		auth: "Token de acesso pessoal incorreto."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Habilitar AppDaemon apps descoberta & rastreamento",
    				netdaemon: "Habilitar NetDaemon apps descoberta & rastreamento",
    				python_script: "Habilitar python_scripts descoberta & rastreamento",
    				sidepanel_icon: "Icone do painel lateral",
    				sidepanel_title: "Titulo do painel lateral",
    				theme: "Habilitar temas descoberta & rastreamento",
    				token: "GitHub Token de acesso pessoal"
    			},
    			description: "Se você preciar de ajuda com a configuração olhe aqui: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$c = {
    	add_to_lovelace: "Tem certeza de que deseja adicionar isso aos seus recursos do Lovelace?",
    	bg_task: "A ação é desativada enquanto as tarefas de fundo estão sendo executadas.",
    	cancel: "Cancelar",
    	"continue": "Tem certeza que quer continuar?",
    	"delete": "Tem certeza de que deseja excluir '{item}'?",
    	delete_installed: "'{item}' está instalado, é necessário desinstalá-lo para poder excluí-lo.",
    	exist: "{item} já existe",
    	generic: "Tem certeza?",
    	home_assistant_is_restarting: "Espere, o Home Assistant está agora a reiniciar.",
    	home_assistant_version_not_correct: "Você está executando a versão Home Assistant '{haversion}', mas este repositório requer que a versão mínima '{minversion}' esteja instalada.",
    	no: "Não",
    	no_upgrades: "Não há atualizações pendentes",
    	ok: "OK",
    	overwrite: "Fazer isso irá substituí-lo.",
    	reload_data: "Isso recarrega os dados de todos os repositórios que o HACS conhece e levará algum tempo para concluir.",
    	restart_home_assistant: "Tem certeza de que deseja reiniciar o Home Assistant?",
    	uninstall: "Tem certeza de que deseja desinstalar '{item}'?",
    	upgrade_all: "Isso atualizará todos esses repositórios, verifique se você leu as notas de versão de todos eles antes de continuar.",
    	yes: "Sim"
    };
    var options$c = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Habilitar AppDaemon apps descoberta & rastreamento",
    				country: "Filtro pelo código do país.",
    				debug: "Ative a depuração.",
    				experimental: "Ativar recursos experimentais",
    				netdaemon: "Habilitar NetDaemon apps descoberta & rastreamento",
    				not_in_use: "Não está em uso com o YAML",
    				release_limit: "Número de lançamentos a serem exibidos.",
    				sidepanel_icon: "Icone do painel lateral",
    				sidepanel_title: "Titulo do painel lateral"
    			}
    		}
    	}
    };
    var repository_banner$c = {
    	config_flow: "Essa integração oferece suporte ao config_flow, o que significa que agora você pode acessar a seção de integração da sua interface do usuário para configurá-lo.",
    	config_flow_title: "Configuração de interface do usuário suportada",
    	integration_not_loaded: "Esta integração não é carregada no Home Assistant.",
    	no_restart_required: "Não é necessário reiniciar",
    	not_loaded: "Não carregado",
    	plugin_not_loaded: "Este plugin não foi adicionado aos seus recursos do Lovelace.",
    	restart: "Você precisa reiniciar o Home Assistant.",
    	restart_pending: "Reiniciar pendente"
    };
    var repository$c = {
    	add_to_lovelace: "Adicionar a Lovelace",
    	authors: "Autores",
    	available: "Disponível",
    	back_to: "Voltar para",
    	changelog: "Changelog",
    	downloads: "Downloads",
    	flag_this: "Sinalizar isso",
    	frontend_version: "Versão Frontend",
    	github_stars: "Estrelas de GitHub",
    	goto_integrations: "Ir para integrações",
    	hide: "Esconder",
    	hide_beta: "Esconder beta",
    	install: "Instalar",
    	installed: "Instalado",
    	lovelace_copy_example: "Copie este exemplo para seu clipboard",
    	lovelace_instruction: "Quando você adicionar isso à sua configuração do lovelace, use este",
    	lovelace_no_js_type: "Não foi possível determinar o tipo deste plug-in, verifique o repositório.",
    	newest: "O mais novo",
    	note_appdaemon: "Você ainda precisa adicioná-lo ao seu arquivo 'apps.yaml'",
    	note_installed: "Quando instalado, ele estará localizado em",
    	note_integration: "Você ainda precisa adicioná-lo ao seu arquivo 'configuration.yaml'",
    	note_plugin: "você ainda precisará adicioná-lo à sua configuração do lovelace ('ui-lovelace.yaml' ou o editor de configuração da interface do usuário)",
    	note_plugin_post_107: "você ainda precisará adicioná-lo à sua configuração do lovelace ('configuration.yaml' ou o editor de recursos '/config /lovelace/resources')",
    	open_issue: "Open issue",
    	open_plugin: "Plugin aberto",
    	reinstall: "Reinstalar",
    	repository: "Repositório",
    	restart_home_assistant: "Reiniciar Home Assistant",
    	show_beta: "Mostrar beta",
    	uninstall: "Desinstalar",
    	update_information: "Atualizar informação",
    	upgrade: "Melhorar"
    };
    var settings$c = {
    	add_custom_repository: "ADICIONAR REPOSITÓRIO PERSONALIZADO",
    	adding_new_repo: "Adicionando novo repositório '{repo}'",
    	adding_new_repo_category: "Com a categoria '{category}'.",
    	bg_task_custom: "Os repositórios personalizados ficam ocultas enquanto as tarefas de fundo estão em execução.",
    	category: "Categoria",
    	compact_mode: "Modo compacto",
    	custom_repositories: "REPOSITÓRIOS PERSONALIZADOS",
    	"delete": "Deletar",
    	display: "Display",
    	grid: "Grade",
    	hacs_repo: "HACS repo",
    	hidden_repositories: "repositórios ocultos",
    	missing_category: "Você precisa selecionar uma categoria",
    	open_repository: "Repositório aberto",
    	reload_data: "Recarregar dados",
    	reload_window: "Recarregar janela",
    	repository_configuration: "Configuração do Repositório",
    	save: "Salvar",
    	table: "Tabela",
    	table_view: "Vista de mesa",
    	unhide: "reexibir",
    	upgrade_all: "Atualizar tudo"
    };
    var store$c = {
    	ascending: "ascendente",
    	clear_new: "Limpar todos os novos repositórios",
    	descending: "descendente",
    	last_updated: "Última atualização",
    	name: "Nome",
    	new_repositories: "Novos Repositórios",
    	pending_upgrades: "Atualizações pendentes",
    	placeholder_search: "Por favor insira um termo de pesquisa...",
    	sort: "ordenar",
    	stars: "Estrelas",
    	status: "Status"
    };
    var time$c = {
    	ago: "atrás",
    	day: "dia",
    	days: "dias",
    	hour: "hora",
    	hours: "horas",
    	minute: "minuto",
    	minutes: "minutos",
    	month: "mês",
    	months: "meses",
    	one: "Um",
    	one_day_ago: "um dia atrás",
    	one_hour_ago: "uma hora atrás",
    	one_minute_ago: "um minuto atrás",
    	one_month_ago: "um mês atrás",
    	one_second_ago: "um segundo atrás",
    	one_year_ago: "um ano atrás",
    	second: "segundo",
    	seconds: "segundos",
    	x_days_ago: "{x} dias atrás",
    	x_hours_ago: "{x} horas atrás",
    	x_minutes_ago: "{x} minutos atrás",
    	x_months_ago: "{x} meses atrás",
    	x_seconds_ago: "{x} segundos atrás",
    	x_years_ago: "{x} anos atrás",
    	year: "ano",
    	years: "anos"
    };
    var ptBR = {
    	common: common$c,
    	config: config$c,
    	confirm: confirm$c,
    	options: options$c,
    	repository_banner: repository_banner$c,
    	repository: repository$c,
    	settings: settings$c,
    	store: store$c,
    	time: time$c
    };

    var pt_BR = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$c,
        config: config$c,
        confirm: confirm$c,
        options: options$c,
        repository_banner: repository_banner$c,
        repository: repository$c,
        settings: settings$c,
        store: store$c,
        time: time$c,
        'default': ptBR
    });

    var common$d = {
    	about: "Despre",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "Aplicații AppDaemon",
    	background_task: "Activitatea de fundal se execută, această pagină se va reîncărca atunci când este gata.",
    	check_log_file: "Verificați log-ul pentru mai multe detalii.",
    	"continue": "Continua",
    	disabled: "Dezactivat",
    	documentation: "Documentație",
    	hacs_is_disabled: "HACS este dezactivat",
    	installed: "instalat",
    	integration: "Integrare",
    	integrations: "Integrări",
    	manage: "administra",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "Aplicații NetDaemon",
    	plugin: "Plugin",
    	plugins: "Plugin-uri",
    	python_script: "Script Python",
    	python_scripts: "Scripturi Python",
    	repositories: "Depozite",
    	settings: "setări",
    	theme: "Temă",
    	themes: "Teme",
    	version: "Versiune"
    };
    var config$d = {
    	abort: {
    		single_instance_allowed: "Doar o singură configurație pentru HACS este permisă."
    	},
    	error: {
    		auth: "Token-ul de acces personal nu este corect."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Activați descoperirea și urmărirea aplicațiilor AppDaemon",
    				netdaemon: "Activați descoperirea și urmărirea aplicațiilor NetDaemon",
    				python_script: "Activați descoperirea și urmărirea python_scripts",
    				sidepanel_icon: "Pictogramă Panou lateral",
    				sidepanel_title: "Titlu panou lateral",
    				theme: "Activați descoperirea și urmărirea temelor",
    				token: "Token de acces personal GitHub"
    			},
    			description: "Dacă aveți nevoie de ajutor pentru configurare, uitați-vă aici: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$d = {
    	add_to_lovelace: "Ești sigur că vrei să adaugi asta la resursele tale Lovelace?",
    	bg_task: "Acțiunea este dezactivată în timp ce activitățile de fundal se execută.",
    	cancel: "Anulare",
    	"continue": "Esti sigur ca vrei sa continui?",
    	"delete": "Sigur doriți să ștergeți '{item}'?",
    	exist: "{item} există deja",
    	generic: "Ești sigur?",
    	home_assistant_is_restarting: "Asteptati, Home Assistant repornește.",
    	no: "Nu",
    	no_upgrades: "Nu există actualizări în curs",
    	ok: "OK",
    	overwrite: "Făcând acest lucru, îl va suprascrie.",
    	restart_home_assistant: "Sigur doriți să reporniți Home Assistant?",
    	uninstall: "Sigur doriți să dezinstalați '{item}'?",
    	yes: "Da"
    };
    var options$d = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Activați descoperirea și urmărirea aplicațiilor AppDaemon",
    				country: "Filtrează cu codul țării.",
    				debug: "Activează depanarea.",
    				experimental: "Activați funcțiile experimentale",
    				netdaemon: "Activați descoperirea și urmărirea aplicațiilor NetDaemon",
    				not_in_use: "Nu este utilizat cu YAML",
    				release_limit: "Număr de versiuni afișate.",
    				sidepanel_icon: "Pictogramă Panou lateral",
    				sidepanel_title: "Titlu panou lateral"
    			}
    		}
    	}
    };
    var repository_banner$d = {
    	integration_not_loaded: "Această integrare nu este încărcată în Home Assistant.",
    	no_restart_required: "Nu este necesară repornirea",
    	not_loaded: "Neîncărcat",
    	plugin_not_loaded: "Acest plugin nu este adăugat la resursele Lovelace.",
    	restart: "Trebuie să reporniți Home Assistant.",
    	restart_pending: "Reporniți în așteptare"
    };
    var repository$d = {
    	add_to_lovelace: "Adăugați la Lovelace",
    	authors: "Autori",
    	available: "Disponibil",
    	back_to: "Înapoi la",
    	changelog: "Jurnal modificari",
    	downloads: "Descărcări",
    	flag_this: "Semnalizează",
    	frontend_version: "Versiune frontend",
    	github_stars: "Stele GitHub",
    	goto_integrations: "Mergi la integrări",
    	hide: "Ascunde",
    	hide_beta: "Ascundere beta",
    	install: "Instalează",
    	installed: "Instalat",
    	lovelace_copy_example: "Copiați exemplul în clipboard",
    	lovelace_instruction: "Când adăugați acest lucru la configurația lovelace, folosiți acest",
    	lovelace_no_js_type: "Nu s-a putut determina tipul acestui plugin, verificați \"repository\".",
    	newest: "cele mai noi",
    	note_appdaemon: "mai trebuie să o adăugați in fișierul „apps.yaml”",
    	note_installed: "Când este instalat, acesta va fi localizat în",
    	note_integration: "mai trebuie să o adăugați in fișierul „configuration.yaml”",
    	note_plugin: "mai trebuie să o adăugați la configurația lovelace ('ui-lovelace.yaml' sau in editorul UI)",
    	open_issue: "Deschidere problemă",
    	open_plugin: "Deschide plugin",
    	reinstall: "Reinstalare",
    	repository: "Depozit",
    	restart_home_assistant: "Reporniți Home Assistant",
    	show_beta: "Afișare beta",
    	uninstall: "Dezinstalare",
    	update_information: "Actualizare informație",
    	upgrade: "Actualizare"
    };
    var settings$d = {
    	add_custom_repository: "ADĂUGARE DEPOZIT PERSONALIZAT",
    	adding_new_repo: "Adăugarea unui depozit nou '{repo}'",
    	adding_new_repo_category: "Cu categoria '{categoria}'.",
    	category: "Categorie",
    	compact_mode: "Modul compact",
    	custom_repositories: "DEPOZITE PERSONALIZATE",
    	"delete": "Șterge",
    	display: "Afişeaza",
    	grid: "Grilă",
    	hacs_repo: "HACS repo",
    	hidden_repositories: "depozite ascunse",
    	missing_category: "Trebuie să selectați o categorie",
    	open_repository: "Deschideți depozitul",
    	reload_data: "Reîncărcați datele",
    	reload_window: "Reîncărcați fereastra",
    	repository_configuration: "Configurația depozitului",
    	save: "Salveaza",
    	table: "Tabel",
    	table_view: "Vizualizare tabel",
    	unhide: "Unhide",
    	upgrade_all: "Actualizați toate"
    };
    var store$d = {
    	ascending: "ascendent",
    	clear_new: "Ștergeți toate depozitele noi",
    	descending: "descendent",
    	last_updated: "Ultima actualizare",
    	name: "Nume",
    	new_repositories: "Noi depozite",
    	pending_upgrades: "Actualizări în așteptare",
    	placeholder_search: "Vă rugăm să introduceți un termen de căutare ...",
    	sort: "fel",
    	stars: "Stele",
    	status: "Starea"
    };
    var time$d = {
    	ago: "în urmă",
    	day: "zi",
    	days: "zi",
    	hour: "oră",
    	hours: "ore",
    	minute: "minut",
    	minutes: "minute",
    	month: "lună",
    	months: "luni",
    	one: "Unu",
    	one_day_ago: "acum o zi",
    	one_hour_ago: "acum o oră",
    	one_minute_ago: "acum un minut",
    	one_month_ago: "acum o luna",
    	one_second_ago: "acum o secunda",
    	one_year_ago: "acum un an",
    	second: "al doilea",
    	seconds: "secunde",
    	x_days_ago: "{x} zile în urmă",
    	x_hours_ago: "{x} ore în urmă",
    	x_minutes_ago: "{x} minute în urmă",
    	x_months_ago: "{x} luni în urmă",
    	x_seconds_ago: "{x} secunde în urmă",
    	x_years_ago: "{x} ani în urmă",
    	year: "an",
    	years: "ani"
    };
    var ro = {
    	common: common$d,
    	config: config$d,
    	confirm: confirm$d,
    	options: options$d,
    	repository_banner: repository_banner$d,
    	repository: repository$d,
    	settings: settings$d,
    	store: store$d,
    	time: time$d
    };

    var ro$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$d,
        config: config$d,
        confirm: confirm$d,
        options: options$d,
        repository_banner: repository_banner$d,
        repository: repository$d,
        settings: settings$d,
        store: store$d,
        time: time$d,
        'default': ro
    });

    var common$e = {
    	about: "О проекте",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "Приложения AppDaemon",
    	background_task: "Выполняется фоновая задача, страница перезагрузится по готовности.",
    	check_log_file: "Проверьте логи для получения более подробной информации.",
    	"continue": "Продолжить",
    	disabled: "Отключено",
    	documentation: "Документация",
    	hacs_is_disabled: "HACS отключен",
    	installed: "установлено",
    	integration: "Интеграция",
    	integrations: "Интеграции",
    	manage: "управлять",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "Приложения NetDaemon",
    	plugin: "Плагин",
    	plugins: "Плагины",
    	python_script: "Скрипт Python",
    	python_scripts: "Скрипты Python",
    	repositories: "Репозитории",
    	settings: "настройки",
    	theme: "Тема",
    	themes: "Темы",
    	version: "Версия"
    };
    var config$e = {
    	abort: {
    		single_instance_allowed: "Разрешена только одна настройка HACS."
    	},
    	error: {
    		auth: "Неверный токен персонального доступа."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Включить поиск и отслеживание приложений AppDaemon",
    				netdaemon: "Включить поиск и отслеживание приложений NetDaemon",
    				python_script: "Включить поиск и отслеживание скриптов Python",
    				sidepanel_icon: "Иконка в боковом меню",
    				sidepanel_title: "Название в боковом меню",
    				theme: "Включить поиск и отслеживание тем",
    				token: "Персональный токен доступа GitHub"
    			},
    			description: "Если вам нужна помощь с настройкой, посмотрите: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$e = {
    	add_to_lovelace: "Вы уверены, что хотите добавить это в ресурсы Lovelace?",
    	bg_task: "Действие отключено во время выполнения фоновых задач.",
    	cancel: "Отмена",
    	"continue": "Вы уверены, что хотите продолжить?",
    	"delete": "Вы уверены, что хотите удалить '{item}'?",
    	delete_installed: "'{item}' установлен, вам нужно нажать 'Удалить', чтобы удалить его.",
    	exist: "{item} уже существует.",
    	generic: "Вы уверены?",
    	home_assistant_is_restarting: "Подожди, теперь Home Assistant перезагружается.",
    	no: "Нет",
    	no_upgrades: "Нет обновлений",
    	ok: "ОК",
    	overwrite: "После подтверждения файлы будут перезаписаны.",
    	reload_data: "Выполняется перезагрузка данных всех репозиториев в HACS, это займет некоторое время.",
    	restart_home_assistant: "Вы уверены, что хотите перезапустить Home Assistant?",
    	uninstall: "Вы уверены, что хотите удалить '{item}'?",
    	yes: "Да"
    };
    var options$e = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Включить поиск и отслеживание приложений AppDaemon",
    				country: "Фильтр по стране.",
    				debug: "Включить отладку.",
    				experimental: "Вкл. экспериментальные функции",
    				netdaemon: "Включить поиск и отслеживание приложений NetDaemon",
    				not_in_use: "Не используется с YAML",
    				release_limit: "Число доступных версий.",
    				sidepanel_icon: "Иконка в боковом меню",
    				sidepanel_title: "Название в боковом меню"
    			}
    		}
    	}
    };
    var repository_banner$e = {
    	integration_not_loaded: "Эта интеграция не загружена в Home Assistant.",
    	no_restart_required: "Перезагрузка не требуется",
    	not_loaded: "Не загружено",
    	plugin_not_loaded: "Этот плагин не добавлен к ресурсам Lovelace.",
    	restart: "Вам нужно перезапустить Home Assistant.",
    	restart_pending: "Ожидается перезапуск"
    };
    var repository$e = {
    	add_to_lovelace: "Добавить в Lovelace",
    	authors: "Авторы",
    	available: "Доступно",
    	back_to: "Назад к",
    	changelog: "Изменения",
    	downloads: "Загрузки",
    	flag_this: "Пожаловаться",
    	frontend_version: "Версия",
    	github_stars: "Звезды GitHub",
    	goto_integrations: "Перейти к интеграции",
    	hide: "Скрыть",
    	hide_beta: "Скрыть бета",
    	install: "Установить",
    	installed: "Установлено",
    	lovelace_copy_example: "Скопируйте пример в буфер обмена",
    	lovelace_instruction: "Для добавления в конфигурацию Lovelace, используйте",
    	lovelace_no_js_type: "Не удалось определить тип этого плагина, проверьте репозиторий.",
    	newest: "новейшая",
    	note_appdaemon: "вам всё ещё нужно добавить код для настройки приложения в файл 'apps.yaml'",
    	note_installed: "После установки, файлы будут расположены в",
    	note_integration: "вам всё ещё нужно добавить код для настройки интеграции в файл 'configuration.yaml'",
    	note_plugin: "вам всё ещё нужно добавить код для настройки плагина в конфигурацию Lovelace ('ui-lovelace.yaml')",
    	open_issue: "Сообщить о проблеме",
    	open_plugin: "Открыть плагин",
    	reinstall: "Переустановить",
    	repository: "Репозиторий",
    	restart_home_assistant: "Перезагрузка Home Assistant",
    	show_beta: "Показать бета",
    	uninstall: "Удалить",
    	update_information: "Обновить информацию",
    	upgrade: "Обновить"
    };
    var settings$e = {
    	add_custom_repository: "ДОБАВИТЬ СВОЙ РЕПОЗИТОРИЙ",
    	adding_new_repo: "Добавление нового репозитория '{repo}'",
    	adding_new_repo_category: "С категорией '{category}'.",
    	bg_task_custom: "Свои репозитории скрыты во время выполнения фоновых задач.",
    	category: "Категория",
    	compact_mode: "Компактный режим",
    	custom_repositories: "СВОИ РЕПОЗИТОРИИ",
    	"delete": "Удалить",
    	display: "Вид",
    	grid: "Сетка",
    	hacs_repo: "Репозиторий HACS",
    	hidden_repositories: "Скрытые репозитории",
    	missing_category: "Вы должны выбрать категорию",
    	open_repository: "Открыть репозиторий",
    	reload_data: "Перезагрузить",
    	reload_window: "Перезагрузить окно",
    	repository_configuration: "Конфигурация репозитория",
    	save: "Сохранить",
    	table: "Таблица",
    	table_view: "Таблица",
    	unhide: "Показать",
    	upgrade_all: "Обновить всё"
    };
    var store$e = {
    	ascending: "по возрастанию",
    	clear_new: "Очистить все новые репозитории",
    	descending: "по убыванию",
    	last_updated: "Последнее обновление",
    	name: "Название",
    	new_repositories: "Новые репозитории",
    	pending_upgrades: "Ожидается обновление",
    	placeholder_search: "Пожалуйста, введите условие для поиска...",
    	sort: "Сортировка",
    	stars: "Звезды",
    	status: "Статус"
    };
    var time$e = {
    	ago: "назад",
    	day: "день",
    	days: "дней",
    	hour: "час",
    	hours: "часов",
    	minute: "минута",
    	minutes: "минут",
    	month: "месяц",
    	months: "месяца",
    	one: "Одна",
    	one_day_ago: "один день назад",
    	one_hour_ago: "час назад",
    	one_minute_ago: "минуту назад",
    	one_month_ago: "месяц назад",
    	one_second_ago: "одну секунду назад",
    	one_year_ago: "один год назад",
    	second: "секунда",
    	seconds: "секунд",
    	x_days_ago: "{x} дней назад",
    	x_hours_ago: "{x} часов назад",
    	x_minutes_ago: "{x} минут назад",
    	x_months_ago: "{x} месяцев назад",
    	x_seconds_ago: "{x} секунд назад",
    	x_years_ago: "{x} лет назад",
    	year: "год",
    	years: "лет"
    };
    var ru = {
    	common: common$e,
    	config: config$e,
    	confirm: confirm$e,
    	options: options$e,
    	repository_banner: repository_banner$e,
    	repository: repository$e,
    	settings: settings$e,
    	store: store$e,
    	time: time$e
    };

    var ru$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$e,
        config: config$e,
        confirm: confirm$e,
        options: options$e,
        repository_banner: repository_banner$e,
        repository: repository$e,
        settings: settings$e,
        store: store$e,
        time: time$e,
        'default': ru
    });

    var common$f = {
    	about: "O tem",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "AppDaemon Aplikacije",
    	background_task: "V ozadju se izvaja naloga, ko se konča, se bo stran osvežila.",
    	check_log_file: "Za več podrobnosti preverite datoteko dnevnika.",
    	"continue": "Nadaljuj",
    	disabled: "Onemogočeno",
    	documentation: "Dokumentacija",
    	hacs_is_disabled: "HACS je onemogočen",
    	installed: "Nameščeno",
    	integration: "Integracija",
    	integrations: "Integracije",
    	manage: "upravljanje",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDaemon Aplikacije",
    	plugin: "Vtičnik",
    	plugins: "Vtičniki",
    	python_script: "Python skripta",
    	python_scripts: "Python skripte",
    	repositories: "Repozitoriji",
    	settings: "nastavitve",
    	theme: "Tema",
    	themes: "Teme",
    	version: "Različica"
    };
    var config$f = {
    	abort: {
    		single_instance_allowed: "Dovoljena je samo ena konfiguracija HACS."
    	},
    	error: {
    		auth: "Osebni dostopni žeton ni pravilen."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Omogoči odkrivanje in sledenje aplikacij AppDaemon",
    				netdaemon: "Omogoči odkrivanje in sledenje aplikacij NetDaemon",
    				python_script: "Omogoči odkrivanje in sledenje python_scripts",
    				sidepanel_icon: "Ikona stranske plošče",
    				sidepanel_title: "Naslov stranske plošče",
    				theme: "Omogoči odkrivanje in sledenje tem",
    				token: "GitHub žeton za osebni dostop"
    			},
    			description: "Če potrebujete pomoč pri konfiguraciji, poglejte tukaj: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$f = {
    	add_to_lovelace: "Ali si prepričan, da želiš to dodati med svoje vire Lovelace?",
    	bg_task: "Dejanje je onemogočeno, medtem ko se izvajajo opravila v ozadju.",
    	cancel: "Prekliči",
    	"continue": "Ali ste prepričani, da želite nadaljevati?",
    	"delete": "Ali ste prepričani, da želite izbrisati '{item}'?",
    	delete_installed: "'{item}' je nameščen, pred odstranitvijo ga morate odstraniti.",
    	exist: "{item} že obstaja",
    	generic: "Ali si prepričan?",
    	home_assistant_is_restarting: "Počakaj, Home Assistant se zdaj znova zaganja.",
    	home_assistant_version_not_correct: "Uporabljate Home Assistant verzije '{haversion}', vendar to skladišče zahteva nameščeno najmanj različico '{minversion}'.",
    	no: "Ne",
    	no_upgrades: "Ni nadgradenj",
    	ok: "V redu",
    	overwrite: "Če to storite, bo to prepisano.",
    	reload_data: "Ponovno se naložijo podatki vseh shramb HA, to bo trajalo nekaj časa.",
    	restart_home_assistant: "Ali ste prepričani, da želite znova zagnati Home Assistant?",
    	uninstall: "Ali ste prepričani, da želite odstraniti '{item}'?",
    	upgrade_all: "To bo nadgradil vse te repozitorije, preden nadaljujete poskrbite, da ste prebrali vse opombe ob izdaji.",
    	yes: "Da"
    };
    var options$f = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Omogoči odkrivanje in sledenje aplikacij AppDaemon",
    				country: "Filtrirajte s kodo države.",
    				debug: "Omogoči odpravljanje napak.",
    				experimental: "Omogočite poskusne funkcije",
    				netdaemon: "Omogoči odkrivanje in sledenje aplikacij NetDaemon",
    				not_in_use: "Ni v uporabi z YAML",
    				release_limit: "Število izdaj, ki jih želite prikazati.",
    				sidepanel_icon: "Ikona stranske plošče",
    				sidepanel_title: "Naslov stranske plošče"
    			}
    		}
    	}
    };
    var repository_banner$f = {
    	config_flow: "Ta integracija podpira config_flow, kar pomeni, da lahko zdaj greste na odsek integracije vašega uporabniškega vmesnika, da ga konfigurirate.",
    	config_flow_title: "Konfiguracija uporabniškega vmesnika je podprta",
    	integration_not_loaded: "Ta integracija ni naložena v programu Home Assistant.",
    	no_restart_required: "Ponovni zagon ni potreben",
    	not_loaded: "Ni naloženo",
    	plugin_not_loaded: "Ta vtičnik ni dodan vašim virom Lovelace.",
    	restart: "Znova morate zagnati Home Assistant.",
    	restart_pending: "Vnovični zagon v čakanju"
    };
    var repository$f = {
    	add_to_lovelace: "Dodaj v Lovelace",
    	authors: "Avtorji",
    	available: "Na voljo",
    	back_to: "Nazaj na",
    	changelog: "Dnevnik sprememb",
    	downloads: "Prenosi",
    	flag_this: "Označite to",
    	frontend_version: "Frontend različica",
    	github_stars: "GitHub zvezde",
    	goto_integrations: "Pojdite na integracije",
    	hide: "Skrij",
    	hide_beta: "Skrij različico beta",
    	install: "Namestite",
    	installed: "Nameščeno",
    	lovelace_copy_example: "Kopirajte primer v odložišče",
    	lovelace_instruction: "Ko ga boste dodajali v svojo lovelace konfiguracijo uporabite to:",
    	lovelace_no_js_type: "Ni bilo mogoče določiti vrsto tega vtičnika, preverite repozitorij.",
    	newest: "najnovejše",
    	note_appdaemon: "Še vedno ga morate dodati v svojo 'apps.yaml' datoteko",
    	note_installed: "Ko bo nameščen, se bo nahajal v",
    	note_integration: "Še vedno ga morate dodati v svojo 'configuration.yaml' datoteko",
    	note_plugin: "vendar ga še vedno morate dodati v svojo lovelace konfiguracijo ('ui-lovelace.yaml' ali \"raw\" UI config urejevalnik)",
    	note_plugin_post_107: "še vedno ga morate dodati v svojo konfiguracijo lovelace ('config.yaml' ali urejevalnik virov '/config/lovelace/resources')",
    	open_issue: "Odprite težavo",
    	open_plugin: "Odprite vtičnik",
    	reinstall: "Znova namestite",
    	repository: "Repozitorij",
    	restart_home_assistant: "Znova zaženite Home Assistant",
    	show_beta: "Pokaži različico beta",
    	uninstall: "Odstranite",
    	update_information: "Posodobite podatke",
    	upgrade: "Nadgradnja"
    };
    var settings$f = {
    	add_custom_repository: "DODAJTE SVOJ REPOZITORIJ",
    	adding_new_repo: "Dodajanje novega repozitorija '{repo}'",
    	adding_new_repo_category: "S kategorijo '{category}'.",
    	bg_task_custom: "Med izvajanjem nalog v ozadju so repozitoriji po meri skriti.",
    	category: "Kategorija",
    	compact_mode: "Kompaktni način",
    	custom_repositories: "VAŠI REPOZITORIJI",
    	"delete": "Izbriši",
    	display: "Prikaz",
    	grid: "Mreža",
    	hacs_repo: "HACS repo",
    	hidden_repositories: "skriti repozitoriji",
    	missing_category: "Izbrati moraš kategorijo",
    	open_repository: "Odprite skladišče",
    	reload_data: "Ponovno naloži podatke",
    	reload_window: "Ponovno naloži okno",
    	repository_configuration: "Konfiguracija skladišč",
    	save: "Shranite",
    	table: "Tabela",
    	table_view: "Pogled tabele",
    	unhide: "Razkrij",
    	upgrade_all: "Nadgradite vse"
    };
    var store$f = {
    	ascending: "naraščajoče",
    	clear_new: "Počisti vse nove repozitorije",
    	descending: "padajoče",
    	last_updated: "Nazadnje posodobljeno",
    	name: "Ime",
    	new_repositories: "Novi repozitoriji",
    	pending_upgrades: "Nadgradnje na čakanju",
    	placeholder_search: "Prosim vnesite iskalni izraz ...",
    	sort: "razvrsti",
    	stars: "Zvezd",
    	status: "Stanje"
    };
    var time$f = {
    	ago: "nazaj",
    	day: "dan",
    	days: "Dni",
    	hour: "ura",
    	hours: "Ur",
    	minute: "minuta",
    	minutes: "minut",
    	month: "mesec",
    	months: "mesecev",
    	one: "Eno",
    	one_day_ago: "pred enim dnevom",
    	one_hour_ago: "pred eno uro",
    	one_minute_ago: "pred eno minuto",
    	one_month_ago: "pred enim mesecem",
    	one_second_ago: "pred sekundo",
    	one_year_ago: "pred enim letom",
    	second: "drugič",
    	seconds: "sekund",
    	x_days_ago: "{x} dni nazaj",
    	x_hours_ago: "{x} ur nazaj",
    	x_minutes_ago: "{x} minut nazaj",
    	x_months_ago: "{x} mesecev nazaj",
    	x_seconds_ago: "{x} sekund nazaj",
    	x_years_ago: "{x} let nazaj",
    	year: "leto",
    	years: "Let"
    };
    var sl = {
    	common: common$f,
    	config: config$f,
    	confirm: confirm$f,
    	options: options$f,
    	repository_banner: repository_banner$f,
    	repository: repository$f,
    	settings: settings$f,
    	store: store$f,
    	time: time$f
    };

    var sl$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$f,
        config: config$f,
        confirm: confirm$f,
        options: options$f,
        repository_banner: repository_banner$f,
        repository: repository$f,
        settings: settings$f,
        store: store$f,
        time: time$f,
        'default': sl
    });

    var common$g = {
    	about: "Om",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "Appdaemon Applikationer",
    	background_task: "Bakgrundsjobb körs, denna sida kommer att laddas igen när det är klart.",
    	check_log_file: "Kontrollera din loggfil för mer information.",
    	"continue": "Fortsätta",
    	disabled: "Inaktiverad",
    	documentation: "Dokumentation",
    	hacs_is_disabled: "HACS är inaktiverat",
    	installed: "installerad",
    	integration: "Integration",
    	integrations: "Integrationer",
    	manage: "hantera",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDaemon Applikationer",
    	plugin: "Plugin",
    	plugins: "Plugins",
    	python_script: "Python skript",
    	python_scripts: "Python skript",
    	repositories: "Repositories",
    	settings: "inställningar",
    	theme: "Tema",
    	themes: "Teman",
    	version: "Version"
    };
    var config$g = {
    	abort: {
    		single_instance_allowed: "Endast en konfiguration kan användas i HACS."
    	},
    	error: {
    		auth: "Personal Access Token är inte korrekt."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Upptäck och följ Appdaemon applikationer",
    				netdaemon: "Upptäck och följ NetDaemon applikationer",
    				python_script: "Upptäck och följ python_scripts",
    				sidepanel_icon: "Ikon för sidpanel",
    				sidepanel_title: "Rubrik för sidpanel",
    				theme: "Upptäck och följ teman",
    				token: "GitHub Personal Access Token"
    			},
    			description: "Om du behöver hjälp med konfigurationen, se här: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$g = {
    	add_to_lovelace: "Är du säker på att du vill lägga till detta till dina Lovelace-resurser?",
    	bg_task: "Åtgärden är inaktiverad medan aktiviteter i bakgrunden körs",
    	cancel: "Avbryt",
    	"continue": "Är du säker på att du vill fortsätta?",
    	"delete": "Är du säker på att du vill ta bort '{item}'?",
    	delete_installed: "'{item}' är installerat, du måste avinstallera det innan du kan ta bort det.",
    	exist: "{item} existerar redan",
    	generic: "Är du säker?",
    	home_assistant_is_restarting: "Vänta, Home Assistant startar nu om.",
    	home_assistant_version_not_correct: "Du kör Home Assistant-versionen '{haversion}', men detta repository kräver att lägsta versionen '{minversion}' måste installeras.",
    	no: "Nej",
    	no_upgrades: "Inga uppgraderingar väntar",
    	ok: "OK",
    	overwrite: "Detta kommer att skriva över den.",
    	reload_data: "Detta laddar om data för alla repositories HACS vet om, kommer detta att ta lite tid att slutföra.",
    	restart_home_assistant: "Är du säker på att du vill starta om Home Assistant?",
    	uninstall: "Är du säker på att du vill avinstallera '{item}'?",
    	upgrade_all: "Detta kommer uppgradera alla dessa repositories, säkerhetsställ att du läst release anteckningarna för dem alla innan du fortsätter",
    	yes: "Ja"
    };
    var options$g = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Upptäck och följ Appdaemon applikationer",
    				country: "Filtrera på landskod.",
    				debug: "Aktivera felsökning",
    				experimental: "Använd experimentella funktioner",
    				netdaemon: "Upptäck och följ NetDaemon applikationer",
    				not_in_use: "Används inte med YAML",
    				release_limit: "Antalet releaser som visas.",
    				sidepanel_icon: "Ikon för sidpanel",
    				sidepanel_title: "Rubrik för sidpanel"
    			}
    		}
    	}
    };
    var repository_banner$g = {
    	config_flow: "Den här integreringen stöder config_flow, det innebär att du nu kan gå till integrationsdelen i användargränssnittet för att konfigurera det.",
    	config_flow_title: "UI-konfiguration stöds",
    	integration_not_loaded: "Denna integration inte laddats i Hem Assistent.",
    	no_restart_required: "Ingen omstart krävs",
    	not_loaded: "Ej laddad",
    	plugin_not_loaded: "Detta plugin är inte tillagt till din Lovelaceresurs.",
    	restart: "Du måste starta om Home Assistant.",
    	restart_pending: "Omstart väntar"
    };
    var repository$g = {
    	add_to_lovelace: "Lägg till i Lovelace",
    	authors: "Författare",
    	available: "Tillgänglig",
    	back_to: "Tillbaka till",
    	changelog: "Ändringslogg",
    	downloads: "Nedladdningar",
    	flag_this: "Flagga denna",
    	frontend_version: "Frontend-version",
    	github_stars: "GitHub stjärnor",
    	goto_integrations: "Gå till integrationer",
    	hide: "Göm",
    	hide_beta: "Göm betaversioner",
    	install: "Installera",
    	installed: "Installerad",
    	lovelace_copy_example: "Kopiera exemplet till urklipp",
    	lovelace_instruction: "När du lägger till denna till din lovelace konfiguration, använd",
    	lovelace_no_js_type: "Kan inte avgöra villken typ av plugin, kontrollera i GIT \nrepository",
    	newest: "nyaste",
    	note_appdaemon: "du behöver fortfarande lägga till den till filen 'apps.yaml'",
    	note_installed: "När den är installerad kommer den finnas i",
    	note_integration: "du behöver fortfarande lägga den till filen 'configuration.yaml'",
    	note_plugin: "du behöver fortfarande lägga till den till din lovelace konfiguration ('ui-lovelace.yaml' eller raw UI config redigerare)",
    	note_plugin_post_107: "du behöver fortfarande lägga till den i din lovelace-konfiguration ('configuration.yaml' eller resursredigeraren /config/lovelace/resources')",
    	open_issue: "Rapportera problem",
    	open_plugin: "Öppna plugin",
    	reinstall: "Ominstallera",
    	repository: "Repository",
    	restart_home_assistant: "Starta om Home Assistant",
    	show_beta: "Visa betaversioner",
    	uninstall: "Avinstallera",
    	update_information: "Uppdatera information",
    	upgrade: "Uppdatera"
    };
    var settings$g = {
    	add_custom_repository: "LÄGG TILL ETT REPOSITORY",
    	adding_new_repo: "Lägger till nytt repository '{repo}'",
    	adding_new_repo_category: "Med kategori '{kategori}'.",
    	bg_task_custom: "Anpassade repositories är dolda under tiden aktiviteter i bakgrunden körs",
    	category: "Kategori",
    	compact_mode: "Kompakt läge",
    	custom_repositories: "REPOSITORIES",
    	"delete": "Ta bort",
    	display: "Visa",
    	grid: "Rutnät",
    	hacs_repo: "HACS repo",
    	hidden_repositories: "dolda förråd",
    	missing_category: "Du behöver välja en kategori",
    	open_repository: "Öppna Repository",
    	reload_data: "Ladda om data",
    	reload_window: "Ladda om fönstret",
    	repository_configuration: "Konfiguration av repository",
    	save: "Spara",
    	table: "Tabell",
    	table_view: "Tabellvy",
    	unhide: "ta fram",
    	upgrade_all: "Uppgradera alla"
    };
    var store$g = {
    	ascending: "stigande",
    	clear_new: "Rensa alla nya förvar",
    	descending: "fallande",
    	last_updated: "Senast uppdaterad",
    	name: "Namn",
    	new_repositories: "Nya förvar",
    	pending_upgrades: "Väntande uppgraderingar",
    	placeholder_search: "Skriv ett sökord...",
    	sort: "sortera",
    	stars: "Stjärnor",
    	status: "Status"
    };
    var time$g = {
    	ago: "sedan",
    	day: "dag",
    	days: "dagar",
    	hour: "timme",
    	hours: "timmar",
    	minute: "minut",
    	minutes: "minuter",
    	month: "månad",
    	months: "månader",
    	one: "Ett",
    	one_day_ago: "för en dag sedan",
    	one_hour_ago: "för en timme sedan",
    	one_minute_ago: "en minut sedan",
    	one_month_ago: "en månad sedan",
    	one_second_ago: "för en sekund sedan",
    	one_year_ago: "ett år sedan",
    	second: "andra",
    	seconds: "sekunder",
    	x_days_ago: "{x} dagar sedan",
    	x_hours_ago: "{x} timmar sedan",
    	x_minutes_ago: "{x} minuter sedan",
    	x_months_ago: "{x} månader sedan",
    	x_seconds_ago: "{x} sekunder sedan",
    	x_years_ago: "{x} år sedan",
    	year: "år",
    	years: "år"
    };
    var sv = {
    	common: common$g,
    	config: config$g,
    	confirm: confirm$g,
    	options: options$g,
    	repository_banner: repository_banner$g,
    	repository: repository$g,
    	settings: settings$g,
    	store: store$g,
    	time: time$g
    };

    var sv$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$g,
        config: config$g,
        confirm: confirm$g,
        options: options$g,
        repository_banner: repository_banner$g,
        repository: repository$g,
        settings: settings$g,
        store: store$g,
        time: time$g,
        'default': sv
    });

    var common$h = {
    	about: "Thông tin",
    	appdaemon_apps: "Ứng dụng AppDaemon",
    	background_task: "Tác vụ nền đang chạy, trang này sẽ được tải lại khi hoàn thành.",
    	check_log_file: "Kiểm tra tệp nhật ký để biết thêm chi tiết.",
    	"continue": "Tiếp tục",
    	disabled: "Bị vô hiệu hoá",
    	documentation: "Tài liệu",
    	hacs_is_disabled: "HACS đã bị vô hiệu hoá",
    	installed: "đã cài đặt",
    	integration: "Tích Hợp",
    	integrations: "Tích Hợp",
    	manage: "quản lý",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "Ứng dụng NetDaemon",
    	plugin: "Bổ Sung",
    	plugins: "Bổ Sung",
    	python_script: "Chương trình Python",
    	python_scripts: "Chương trình Python",
    	repositories: "Các kho ứng dụng",
    	settings: "thiết lập",
    	theme: "Chủ đề",
    	themes: "Chủ đề",
    	version: "Phiên bản"
    };
    var config$h = {
    	abort: {
    		single_instance_allowed: "Chỉ cho phép một cấu hình HACS duy nhất."
    	},
    	error: {
    		auth: "Mã Truy Cập Cá Nhân không đúng."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Kích hoạt phát hiện & theo dõi các ứng dụng AppDaemon",
    				netdaemon: "Kích hoạt phát hiện & theo dõi các ứng dụng NetDaemon",
    				python_script: "Kích hoạt phát hiện & theo dõi các Chương trình Python",
    				sidepanel_icon: "Biểu tượng bảng điều khiển cạnh bên",
    				sidepanel_title: "Tiêu đề bảng điều khiển cạnh bên",
    				theme: "Kích hoạt phát hiện & theo dõi các Chủ Đề",
    				token: "Mã Truy Cập Cá Nhân từ GitHub"
    			},
    			description: "Nếu bạn cần giúp đỡ về các cấu hình, hãy xem ở đây:\nhttps://hacs.xyz/docs/configuration/start"
    		}
    	}
    };
    var confirm$h = {
    	add_to_lovelace: "Bạn có chắc muốn thêm vào tài nguyên Lovelace của mình không?",
    	bg_task: "Hành động bị vô hiệu hoá trong khi các tác vụ nền đang chạy.",
    	cancel: "Huỷ bỏ",
    	"continue": "Bạn có chắc chắn muốn tiếp tục?",
    	"delete": "Bạn có chắc muốn xoá '{item}'?",
    	delete_installed: "'{item}' đã được cài đặt, bạn cần gỡ cài đặt nó trước khi có thể xoá.",
    	exist: "{item} đang tồn tại",
    	generic: "Bạn có chắc không?",
    	home_assistant_is_restarting: "Khoan đã, Home Assistant đang khởi động lại.",
    	home_assistant_version_not_correct: "Bạn đang chạy phiên bản Home Assistant '{haversion}' nhưng kho ứng dụng này yêu cầu phiên bản thấp nhất '{minversion}' để cài đặt.",
    	no: "Không",
    	no_upgrades: "Không có cập nhật nào đang xử lý",
    	ok: "OK",
    	overwrite: "Tiếp tục sẽ khiến card này bị ghi đè lên.",
    	reload_data: "Điều này tải lại dữ liệu của tất cả các kho ứng dụng trong HACS, sẽ mất một lúc để hoàn tất.",
    	restart_home_assistant: "Bạn có muốn khởi động lại Home Assistant?",
    	uninstall: "Bạn có chắc muốn gỡ cài đặt '{item}'?",
    	upgrade_all: "Điều này sẽ nâng cấp tất cả các kho ứng dụng này, đảm bảo rằng bạn đã đọc tất cả các lưu ý phát hành trước khi tiếp tục.",
    	yes: "Có"
    };
    var options$h = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Kích hoạt phát hiện & theo dõi các ứng dụng AppDaemon",
    				country: "Lọc với mã quốc gia.",
    				debug: "Kích hoạt Trình gỡ lỗi",
    				experimental: "Kích hoạt các tính năng thử nghiệm",
    				netdaemon: "Kích hoạt phát hiện & theo dõi các ứng dụng NetDaemon",
    				not_in_use: "Không sử dụng với YAML",
    				release_limit: "Số lượng phiên bản hiển thị.",
    				sidepanel_icon: "Biểu tượng bảng điều khiển cạnh bên",
    				sidepanel_title: "Tiêu đề bảng điều khiển cạnh bên"
    			}
    		}
    	}
    };
    var repository_banner$h = {
    	config_flow: "Tích hợp này hỗ trợ config_flow, điều này có nghĩa là bây giờ bạn có thể chuyển đến khu vực Tích hợp trên giao diện để cấu hình nó.",
    	config_flow_title: "Hỗ trợ cấu hình qua giao diện người dùng",
    	integration_not_loaded: "Tích hợp này chưa được tải trong Home Assistant.",
    	no_restart_required: "Không cần khởi động lại",
    	not_loaded: "Chưa được tải",
    	plugin_not_loaded: "Bổ sung này chưa được thêm vào tài nguyên Lovelace của bạn.",
    	restart: "Bạn cần khởi động lại Home Assistant.",
    	restart_pending: "Đang chờ khởi động lại"
    };
    var repository$h = {
    	add_to_lovelace: "Thêm vào Lovelace",
    	authors: "Tác giả",
    	available: "Có sẵn",
    	back_to: "Quay lại",
    	changelog: "Nhật ký thay đổi",
    	downloads: "Tải xuống",
    	flag_this: "Đánh dấu mục này",
    	frontend_version: "Phiên bản Frontend",
    	github_stars: "Số Sao trên GitHub",
    	goto_integrations: "Đi đến Tích hợp",
    	hide: "Ẩn",
    	hide_beta: "Ẩn phiên bản beta",
    	install: "Cài đặt",
    	installed: "Đã cài đặt",
    	lovelace_copy_example: "Sao chép ví dụ vào bộ nhớ tạm",
    	lovelace_instruction: "Khi bạn thêm mục này vào cấu hình lovelace, sử dụng",
    	lovelace_no_js_type: "Không thể xác định loại Bổ sung này, kiểm tra lại kho ứng dụng.",
    	newest: "mới nhất",
    	note_appdaemon: "bạn vẫn cần thêm vào tập tin 'apps.yaml'",
    	note_installed: "Một khi được cài đặt, mục nằm sẽ nằm ở",
    	note_integration: "bạn vẫn cần thêm vào tập tin 'configuration.yaml'",
    	note_plugin: "bạn vẫn cần thêm vào cấu hình lovelace ('ui-lovelace.yaml' hoặc trình soạn thảo giao diện)",
    	note_plugin_post_107: "bạn vẫn cần phải thêm nó vào cấu hình Lovelace ('configuration.yaml' hoặc trình biên tập tài nguyên '/config/Lovelace/Resources ')",
    	open_issue: "Báo cáo vấn đề",
    	open_plugin: "Mở Bổ sung",
    	reinstall: "Cài đặt lại",
    	repository: "Kho ứng dụng",
    	restart_home_assistant: "Khởi động lại Home Assistant",
    	show_beta: "Hiển thị phiên bản beta",
    	uninstall: "Gỡ cài đặt",
    	update_information: "Cập nhật thông tin",
    	upgrade: "Cập nhật"
    };
    var settings$h = {
    	add_custom_repository: "THÊM KHO ỨNG DỤNG TUỲ CHỈNH",
    	adding_new_repo: "Thêm kho ứng dụng mới '{repo}'",
    	adding_new_repo_category: "Với danh mục '{category}'",
    	bg_task_custom: "Các kho ứng dụng tuỳ chỉnh bị ẩn khi tác vụ nền đang chạy.",
    	category: "Danh mục",
    	compact_mode: "Chế độ thu gọn",
    	custom_repositories: "KHO ỨNG DỤNG TUỲ CHỈNH",
    	"delete": "Xoá",
    	display: "Hiển thị",
    	grid: "Lưới",
    	hacs_repo: "Kho ứng dụng HACS",
    	hidden_repositories: "các kho ứng dụng bị ẩn",
    	missing_category: "Bạn cần chọn một danh mục",
    	open_repository: "Mở Kho ứng dụng",
    	reload_data: "Tải lại dữ liệu",
    	reload_window: "Tải lại cửa sổ",
    	repository_configuration: "Cấu hình Kho ứng dụng",
    	save: "Lưu",
    	table: "Bảng",
    	table_view: "Xem dạng bảng",
    	unhide: "thôi ẩn",
    	upgrade_all: "Nâng cấp tất cả"
    };
    var store$h = {
    	ascending: "tăng dần",
    	clear_new: "Xoá tất cả kho ứng dụng mới",
    	descending: "giảm dần",
    	last_updated: "Cập nhật lần cuối",
    	name: "Tên",
    	new_repositories: "Kho ứng dụng Mới",
    	pending_upgrades: "Nâng cấp chờ xử lý",
    	placeholder_search: "Vui lòng nhập cụm từ cần tìm",
    	sort: "sắp xếp",
    	stars: "Sao",
    	status: "Trạng thái"
    };
    var time$h = {
    	ago: "trước",
    	day: "ngày",
    	days: "ngày",
    	hour: "giờ",
    	hours: "giờ",
    	minute: "phút",
    	minutes: "phút",
    	month: "tháng",
    	months: "tháng",
    	one: "Một",
    	one_day_ago: "một ngày trước",
    	one_hour_ago: "một giờ trước",
    	one_minute_ago: "một phút trước",
    	one_month_ago: "một tháng trước",
    	one_second_ago: "một giây trước",
    	one_year_ago: "một năm trước",
    	second: "giây",
    	seconds: "giây",
    	x_days_ago: "{x} ngày trước",
    	x_hours_ago: "{x} giờ trước",
    	x_minutes_ago: "{x} phút trước",
    	x_months_ago: "{x} tháng trước",
    	x_seconds_ago: "{x} giây trước",
    	x_years_ago: "{x} năm trước",
    	year: "năm",
    	years: "năm"
    };
    var vi = {
    	common: common$h,
    	config: config$h,
    	confirm: confirm$h,
    	options: options$h,
    	repository_banner: repository_banner$h,
    	repository: repository$h,
    	settings: settings$h,
    	store: store$h,
    	time: time$h
    };

    var vi$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$h,
        config: config$h,
        confirm: confirm$h,
        options: options$h,
        repository_banner: repository_banner$h,
        repository: repository$h,
        settings: settings$h,
        store: store$h,
        time: time$h,
        'default': vi
    });

    var common$i = {
    	about: "关于",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "AppDaemon应用",
    	background_task: "后台任务正在运行，完成后将重新加载此页面。",
    	check_log_file: "请查看日志文件以了解更多信息。",
    	"continue": "继续",
    	disabled: "禁用",
    	documentation: "文档",
    	hacs_is_disabled: "HACS已禁用",
    	installed: "已安装",
    	integration: "自定义组件",
    	integrations: "自定义组件",
    	manage: "管理",
    	netdaemon: "NetDaemon应用",
    	netdaemon_apps: "NetDaemon应用",
    	plugin: "Lovelace插件",
    	plugins: "Lovelace插件",
    	python_script: "Python脚本",
    	python_scripts: "Python脚本",
    	repositories: "仓库数量",
    	settings: "设置",
    	theme: "主题",
    	themes: "主题",
    	version: "版本"
    };
    var config$i = {
    	abort: {
    		single_instance_allowed: "仅允许单个HACS配置。"
    	},
    	error: {
    		auth: "个人访问令牌不正确。"
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "启用AppDaemon应用发现和跟踪",
    				netdaemon: "启用NetDaemon应用发现和跟踪",
    				python_script: "启用python_scripts发现和跟踪",
    				sidepanel_icon: "侧面板图标",
    				sidepanel_title: "侧面板标题",
    				theme: "启用主题发现和跟踪",
    				token: "GitHub个人访问令牌"
    			},
    			description: "如果您需要有关配置的帮助，请在此处查看：https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$i = {
    	add_to_lovelace: "您确定要将此添加到Lovelace资源中吗？",
    	bg_task: "后台任务正在运行时，操作被禁用。",
    	cancel: "取消",
    	"continue": "你确定你要继续吗？",
    	"delete": "是否确实要删除\"{item}\"？",
    	delete_installed: "已安装 '{item}'，需要先将其卸载，然后才能将其删除。",
    	exist: "{item}已经存在",
    	generic: "你确定吗？",
    	home_assistant_is_restarting: "请等待，Home Assistant现在正在重新启动。",
    	home_assistant_version_not_correct: "您正在运行Home Assistant的版本为'{haversion}'，但是这个需要安装最低版本是'{minversion}'。",
    	no: "取消",
    	no_upgrades: "暂无升级",
    	ok: "确定",
    	overwrite: "这样做会覆盖它。",
    	reload_data: "这将重新加载HACS知道的所有仓库的数据，这需要一些时间才能完成。",
    	restart_home_assistant: "您确定要重新启动Home Assistant吗？",
    	uninstall: "您确定要卸载 '{item}' 吗？",
    	upgrade_all: "这将升级所有这些仓库，请确保在继续之前已阅读所有仓库的发行说明。",
    	yes: "确定"
    };
    var options$i = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "启用AppDaemon应用发现和跟踪",
    				country: "用国家代码过滤。",
    				debug: "启用调试。",
    				experimental: "启用实验功能",
    				netdaemon: "启用NetDaemon应用发现和跟踪",
    				not_in_use: "不适用于 YAML",
    				release_limit: "要显示的发行数量。",
    				sidepanel_icon: "侧面板图标",
    				sidepanel_title: "侧面板标题"
    			}
    		}
    	}
    };
    var repository_banner$i = {
    	config_flow: "此组件支持config_flow，这意味着您现在可以转到UI的“集成”部分进行配置。",
    	config_flow_title: "支持UI配置",
    	integration_not_loaded: "此集成未加载到Home Assistant中。",
    	no_restart_required: "无需重启",
    	not_loaded: "未加载",
    	plugin_not_loaded: "该插件未添加到您的Lovelace资源中。",
    	restart: "您需要重新启动Home Assistant。",
    	restart_pending: "重新启动待处理"
    };
    var repository$i = {
    	add_to_lovelace: "添加到Lovelace",
    	authors: "作者",
    	available: "版本号",
    	back_to: "返回",
    	changelog: "更新日志",
    	downloads: "下载",
    	flag_this: "标记",
    	frontend_version: "前端版本",
    	github_stars: "GitHub星级",
    	goto_integrations: "转到组件",
    	hide: "隐藏",
    	hide_beta: "隐藏测试版",
    	install: "安装",
    	installed: "已安装版本",
    	lovelace_copy_example: "复制样例代码到你的剪贴板",
    	lovelace_instruction: "您仍然需要将下列代码添加到lovelace配置中（“ ui-lovelace.yaml”或原始配置编辑器）:",
    	lovelace_no_js_type: "无法确定此插件的类型，请检查仓库。",
    	newest: "最新",
    	note_appdaemon: "您仍然需要将其添加到“ apps.yaml”文件中",
    	note_installed: "安装后，它将位于",
    	note_integration: "您仍然需要将其添加到“ configuration.yaml”文件中",
    	note_plugin: "您仍然需要将其添加到lovelace配置中（“ ui-lovelace.yaml”或原始配置编辑器）ui-lovelace.yaml",
    	note_plugin_post_107: "您仍然需要将其添加到lovelace配置中（“ configuration.yaml”或资源编辑器“/config/lovelace/resources”）",
    	open_issue: "提交问题",
    	open_plugin: "打开插件",
    	reinstall: "重新安装",
    	repository: "仓库",
    	restart_home_assistant: "重新启动Home Assistant",
    	show_beta: "显示测试版",
    	uninstall: "卸载",
    	update_information: "更新信息",
    	upgrade: "升级"
    };
    var settings$i = {
    	add_custom_repository: "添加自定义仓库",
    	adding_new_repo: "添加新的仓库 '{repo}'",
    	adding_new_repo_category: "类别为 '{category}'。",
    	bg_task_custom: "自定义仓库在后台任务运行时被隐藏。",
    	category: "类别",
    	compact_mode: "紧凑模式",
    	custom_repositories: "自定义仓库",
    	"delete": "删除",
    	display: "视图方式",
    	grid: "网格",
    	hacs_repo: "HACS仓库",
    	hidden_repositories: "隐藏的仓库",
    	missing_category: "您需要选择一个类别",
    	open_repository: "打开仓库",
    	reload_data: "重载数据",
    	reload_window: "重新加载窗口",
    	repository_configuration: "仓库数据配置",
    	save: "保存",
    	table: "列表",
    	table_view: "表视图",
    	unhide: "取消隐藏",
    	upgrade_all: "升级全部"
    };
    var store$i = {
    	ascending: "升序",
    	clear_new: "清除所有新仓库标记",
    	descending: "降序",
    	last_updated: "最近更新时间",
    	name: "名称",
    	new_repositories: "新仓库",
    	pending_upgrades: "待升级",
    	placeholder_search: "搜索项目...",
    	sort: "排序",
    	stars: "星级",
    	status: "状态"
    };
    var time$i = {
    	ago: "过去",
    	day: "天",
    	days: "天",
    	hour: "小时",
    	hours: "小时",
    	minute: "分",
    	minutes: "分",
    	month: "月",
    	months: "月",
    	one: "一",
    	one_day_ago: "一天前",
    	one_hour_ago: "一个小时之前",
    	one_minute_ago: "一分钟前",
    	one_month_ago: "一个月前",
    	one_second_ago: "一秒钟前",
    	one_year_ago: "一年前",
    	second: "秒",
    	seconds: "秒",
    	x_days_ago: "{x}天前",
    	x_hours_ago: "{x}小时前",
    	x_minutes_ago: "{x}分钟前",
    	x_months_ago: "{x}个月前",
    	x_seconds_ago: "{x}秒前",
    	x_years_ago: "{x}年前",
    	year: "年",
    	years: "年"
    };
    var zhHans = {
    	common: common$i,
    	config: config$i,
    	confirm: confirm$i,
    	options: options$i,
    	repository_banner: repository_banner$i,
    	repository: repository$i,
    	settings: settings$i,
    	store: store$i,
    	time: time$i
    };

    var zh_Hans = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$i,
        config: config$i,
        confirm: confirm$i,
        options: options$i,
        repository_banner: repository_banner$i,
        repository: repository$i,
        settings: settings$i,
        store: store$i,
        time: time$i,
        'default': zhHans
    });

    function localize(string, search = undefined, replace = undefined) {
        const languages = {
            da: da$1,
            de: de$1,
            el: el$1,
            en: en$1,
            es: es$1,
            fr: fr$1,
            hu: hu$1,
            it: it$1,
            nb: nb$1,
            nl: nl$1,
            nn: nn$1,
            pl: pl$1,
            pt_BR: pt_BR,
            ro: ro$1,
            ru: ru$1,
            sl: sl$1,
            sv: sv$1,
            vi: vi$1,
            zh_Hans: zh_Hans,
        };
        let translated;
        const split = string.split(".");
        const lang = (localStorage.getItem("selectedLanguage") || "en")
            .replace(/['"]+/g, "")
            .replace("-", "_");
        try {
            translated = languages[lang];
            split.forEach((section) => {
                translated = translated[section];
            });
        }
        catch (e) {
            translated = languages["en"];
            split.forEach((section) => {
                translated = translated[section];
            });
        }
        if (translated === undefined) {
            translated = languages["en"];
            split.forEach((section) => {
                translated = translated[section];
            });
        }
        if (search !== undefined && replace !== undefined) {
            translated = translated.replace(search, replace);
        }
        return translated;
    }

    var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

    function unwrapExports (x) {
    	return x && x.__esModule && Object.prototype.hasOwnProperty.call(x, 'default') ? x['default'] : x;
    }

    function createCommonjsModule(fn, module) {
    	return module = { exports: {} }, fn(module, module.exports), module.exports;
    }

    function getCjsExportFromNamespace (n) {
    	return n && n['default'] || n;
    }

    /**
     * lodash (Custom Build) <https://lodash.com/>
     * Build: `lodash modularize exports="npm" -o ./`
     * Copyright jQuery Foundation and other contributors <https://jquery.org/>
     * Released under MIT license <https://lodash.com/license>
     * Based on Underscore.js 1.8.3 <http://underscorejs.org/LICENSE>
     * Copyright Jeremy Ashkenas, DocumentCloud and Investigative Reporters & Editors
     */

    /** Used as references for various `Number` constants. */
    var MAX_SAFE_INTEGER = 9007199254740991;

    /** `Object#toString` result references. */
    var argsTag = '[object Arguments]',
        funcTag = '[object Function]',
        genTag = '[object GeneratorFunction]',
        mapTag = '[object Map]',
        objectTag = '[object Object]',
        promiseTag = '[object Promise]',
        setTag = '[object Set]',
        stringTag = '[object String]',
        weakMapTag = '[object WeakMap]';

    var dataViewTag = '[object DataView]';

    /**
     * Used to match `RegExp`
     * [syntax characters](http://ecma-international.org/ecma-262/7.0/#sec-patterns).
     */
    var reRegExpChar = /[\\^$.*+?()[\]{}|]/g;

    /** Used to detect host constructors (Safari). */
    var reIsHostCtor = /^\[object .+?Constructor\]$/;

    /** Used to detect unsigned integer values. */
    var reIsUint = /^(?:0|[1-9]\d*)$/;

    /** Used to compose unicode character classes. */
    var rsAstralRange = '\\ud800-\\udfff',
        rsComboMarksRange = '\\u0300-\\u036f\\ufe20-\\ufe23',
        rsComboSymbolsRange = '\\u20d0-\\u20f0',
        rsVarRange = '\\ufe0e\\ufe0f';

    /** Used to compose unicode capture groups. */
    var rsAstral = '[' + rsAstralRange + ']',
        rsCombo = '[' + rsComboMarksRange + rsComboSymbolsRange + ']',
        rsFitz = '\\ud83c[\\udffb-\\udfff]',
        rsModifier = '(?:' + rsCombo + '|' + rsFitz + ')',
        rsNonAstral = '[^' + rsAstralRange + ']',
        rsRegional = '(?:\\ud83c[\\udde6-\\uddff]){2}',
        rsSurrPair = '[\\ud800-\\udbff][\\udc00-\\udfff]',
        rsZWJ = '\\u200d';

    /** Used to compose unicode regexes. */
    var reOptMod = rsModifier + '?',
        rsOptVar = '[' + rsVarRange + ']?',
        rsOptJoin = '(?:' + rsZWJ + '(?:' + [rsNonAstral, rsRegional, rsSurrPair].join('|') + ')' + rsOptVar + reOptMod + ')*',
        rsSeq = rsOptVar + reOptMod + rsOptJoin,
        rsSymbol = '(?:' + [rsNonAstral + rsCombo + '?', rsCombo, rsRegional, rsSurrPair, rsAstral].join('|') + ')';

    /** Used to match [string symbols](https://mathiasbynens.be/notes/javascript-unicode). */
    var reUnicode = RegExp(rsFitz + '(?=' + rsFitz + ')|' + rsSymbol + rsSeq, 'g');

    /** Used to detect strings with [zero-width joiners or code points from the astral planes](http://eev.ee/blog/2015/09/12/dark-corners-of-unicode/). */
    var reHasUnicode = RegExp('[' + rsZWJ + rsAstralRange  + rsComboMarksRange + rsComboSymbolsRange + rsVarRange + ']');

    /** Detect free variable `global` from Node.js. */
    var freeGlobal = typeof commonjsGlobal == 'object' && commonjsGlobal && commonjsGlobal.Object === Object && commonjsGlobal;

    /** Detect free variable `self`. */
    var freeSelf = typeof self == 'object' && self && self.Object === Object && self;

    /** Used as a reference to the global object. */
    var root = freeGlobal || freeSelf || Function('return this')();

    /**
     * A specialized version of `_.map` for arrays without support for iteratee
     * shorthands.
     *
     * @private
     * @param {Array} [array] The array to iterate over.
     * @param {Function} iteratee The function invoked per iteration.
     * @returns {Array} Returns the new mapped array.
     */
    function arrayMap(array, iteratee) {
      var index = -1,
          length = array ? array.length : 0,
          result = Array(length);

      while (++index < length) {
        result[index] = iteratee(array[index], index, array);
      }
      return result;
    }

    /**
     * Converts an ASCII `string` to an array.
     *
     * @private
     * @param {string} string The string to convert.
     * @returns {Array} Returns the converted array.
     */
    function asciiToArray(string) {
      return string.split('');
    }

    /**
     * The base implementation of `_.times` without support for iteratee shorthands
     * or max array length checks.
     *
     * @private
     * @param {number} n The number of times to invoke `iteratee`.
     * @param {Function} iteratee The function invoked per iteration.
     * @returns {Array} Returns the array of results.
     */
    function baseTimes(n, iteratee) {
      var index = -1,
          result = Array(n);

      while (++index < n) {
        result[index] = iteratee(index);
      }
      return result;
    }

    /**
     * The base implementation of `_.values` and `_.valuesIn` which creates an
     * array of `object` property values corresponding to the property names
     * of `props`.
     *
     * @private
     * @param {Object} object The object to query.
     * @param {Array} props The property names to get values for.
     * @returns {Object} Returns the array of property values.
     */
    function baseValues(object, props) {
      return arrayMap(props, function(key) {
        return object[key];
      });
    }

    /**
     * Gets the value at `key` of `object`.
     *
     * @private
     * @param {Object} [object] The object to query.
     * @param {string} key The key of the property to get.
     * @returns {*} Returns the property value.
     */
    function getValue(object, key) {
      return object == null ? undefined : object[key];
    }

    /**
     * Checks if `string` contains Unicode symbols.
     *
     * @private
     * @param {string} string The string to inspect.
     * @returns {boolean} Returns `true` if a symbol is found, else `false`.
     */
    function hasUnicode(string) {
      return reHasUnicode.test(string);
    }

    /**
     * Checks if `value` is a host object in IE < 9.
     *
     * @private
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is a host object, else `false`.
     */
    function isHostObject(value) {
      // Many host objects are `Object` objects that can coerce to strings
      // despite having improperly defined `toString` methods.
      var result = false;
      if (value != null && typeof value.toString != 'function') {
        try {
          result = !!(value + '');
        } catch (e) {}
      }
      return result;
    }

    /**
     * Converts `iterator` to an array.
     *
     * @private
     * @param {Object} iterator The iterator to convert.
     * @returns {Array} Returns the converted array.
     */
    function iteratorToArray(iterator) {
      var data,
          result = [];

      while (!(data = iterator.next()).done) {
        result.push(data.value);
      }
      return result;
    }

    /**
     * Converts `map` to its key-value pairs.
     *
     * @private
     * @param {Object} map The map to convert.
     * @returns {Array} Returns the key-value pairs.
     */
    function mapToArray(map) {
      var index = -1,
          result = Array(map.size);

      map.forEach(function(value, key) {
        result[++index] = [key, value];
      });
      return result;
    }

    /**
     * Creates a unary function that invokes `func` with its argument transformed.
     *
     * @private
     * @param {Function} func The function to wrap.
     * @param {Function} transform The argument transform.
     * @returns {Function} Returns the new function.
     */
    function overArg(func, transform) {
      return function(arg) {
        return func(transform(arg));
      };
    }

    /**
     * Converts `set` to an array of its values.
     *
     * @private
     * @param {Object} set The set to convert.
     * @returns {Array} Returns the values.
     */
    function setToArray(set) {
      var index = -1,
          result = Array(set.size);

      set.forEach(function(value) {
        result[++index] = value;
      });
      return result;
    }

    /**
     * Converts `string` to an array.
     *
     * @private
     * @param {string} string The string to convert.
     * @returns {Array} Returns the converted array.
     */
    function stringToArray(string) {
      return hasUnicode(string)
        ? unicodeToArray(string)
        : asciiToArray(string);
    }

    /**
     * Converts a Unicode `string` to an array.
     *
     * @private
     * @param {string} string The string to convert.
     * @returns {Array} Returns the converted array.
     */
    function unicodeToArray(string) {
      return string.match(reUnicode) || [];
    }

    /** Used for built-in method references. */
    var funcProto = Function.prototype,
        objectProto = Object.prototype;

    /** Used to detect overreaching core-js shims. */
    var coreJsData = root['__core-js_shared__'];

    /** Used to detect methods masquerading as native. */
    var maskSrcKey = (function() {
      var uid = /[^.]+$/.exec(coreJsData && coreJsData.keys && coreJsData.keys.IE_PROTO || '');
      return uid ? ('Symbol(src)_1.' + uid) : '';
    }());

    /** Used to resolve the decompiled source of functions. */
    var funcToString = funcProto.toString;

    /** Used to check objects for own properties. */
    var hasOwnProperty = objectProto.hasOwnProperty;

    /**
     * Used to resolve the
     * [`toStringTag`](http://ecma-international.org/ecma-262/7.0/#sec-object.prototype.tostring)
     * of values.
     */
    var objectToString = objectProto.toString;

    /** Used to detect if a method is native. */
    var reIsNative = RegExp('^' +
      funcToString.call(hasOwnProperty).replace(reRegExpChar, '\\$&')
      .replace(/hasOwnProperty|(function).*?(?=\\\()| for .+?(?=\\\])/g, '$1.*?') + '$'
    );

    /** Built-in value references. */
    var Symbol$1 = root.Symbol,
        iteratorSymbol = Symbol$1 ? Symbol$1.iterator : undefined,
        propertyIsEnumerable = objectProto.propertyIsEnumerable;

    /* Built-in method references for those with the same name as other `lodash` methods. */
    var nativeKeys = overArg(Object.keys, Object);

    /* Built-in method references that are verified to be native. */
    var DataView = getNative(root, 'DataView'),
        Map$1 = getNative(root, 'Map'),
        Promise$1 = getNative(root, 'Promise'),
        Set$1 = getNative(root, 'Set'),
        WeakMap$1 = getNative(root, 'WeakMap');

    /** Used to detect maps, sets, and weakmaps. */
    var dataViewCtorString = toSource(DataView),
        mapCtorString = toSource(Map$1),
        promiseCtorString = toSource(Promise$1),
        setCtorString = toSource(Set$1),
        weakMapCtorString = toSource(WeakMap$1);

    /**
     * Creates an array of the enumerable property names of the array-like `value`.
     *
     * @private
     * @param {*} value The value to query.
     * @param {boolean} inherited Specify returning inherited property names.
     * @returns {Array} Returns the array of property names.
     */
    function arrayLikeKeys(value, inherited) {
      // Safari 8.1 makes `arguments.callee` enumerable in strict mode.
      // Safari 9 makes `arguments.length` enumerable in strict mode.
      var result = (isArray(value) || isArguments(value))
        ? baseTimes(value.length, String)
        : [];

      var length = result.length,
          skipIndexes = !!length;

      for (var key in value) {
        if ((inherited || hasOwnProperty.call(value, key)) &&
            !(skipIndexes && (key == 'length' || isIndex(key, length)))) {
          result.push(key);
        }
      }
      return result;
    }

    /**
     * The base implementation of `getTag`.
     *
     * @private
     * @param {*} value The value to query.
     * @returns {string} Returns the `toStringTag`.
     */
    function baseGetTag(value) {
      return objectToString.call(value);
    }

    /**
     * The base implementation of `_.isNative` without bad shim checks.
     *
     * @private
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is a native function,
     *  else `false`.
     */
    function baseIsNative(value) {
      if (!isObject(value) || isMasked(value)) {
        return false;
      }
      var pattern = (isFunction(value) || isHostObject(value)) ? reIsNative : reIsHostCtor;
      return pattern.test(toSource(value));
    }

    /**
     * The base implementation of `_.keys` which doesn't treat sparse arrays as dense.
     *
     * @private
     * @param {Object} object The object to query.
     * @returns {Array} Returns the array of property names.
     */
    function baseKeys(object) {
      if (!isPrototype(object)) {
        return nativeKeys(object);
      }
      var result = [];
      for (var key in Object(object)) {
        if (hasOwnProperty.call(object, key) && key != 'constructor') {
          result.push(key);
        }
      }
      return result;
    }

    /**
     * Copies the values of `source` to `array`.
     *
     * @private
     * @param {Array} source The array to copy values from.
     * @param {Array} [array=[]] The array to copy values to.
     * @returns {Array} Returns `array`.
     */
    function copyArray(source, array) {
      var index = -1,
          length = source.length;

      array || (array = Array(length));
      while (++index < length) {
        array[index] = source[index];
      }
      return array;
    }

    /**
     * Gets the native function at `key` of `object`.
     *
     * @private
     * @param {Object} object The object to query.
     * @param {string} key The key of the method to get.
     * @returns {*} Returns the function if it's native, else `undefined`.
     */
    function getNative(object, key) {
      var value = getValue(object, key);
      return baseIsNative(value) ? value : undefined;
    }

    /**
     * Gets the `toStringTag` of `value`.
     *
     * @private
     * @param {*} value The value to query.
     * @returns {string} Returns the `toStringTag`.
     */
    var getTag = baseGetTag;

    // Fallback for data views, maps, sets, and weak maps in IE 11,
    // for data views in Edge < 14, and promises in Node.js.
    if ((DataView && getTag(new DataView(new ArrayBuffer(1))) != dataViewTag) ||
        (Map$1 && getTag(new Map$1) != mapTag) ||
        (Promise$1 && getTag(Promise$1.resolve()) != promiseTag) ||
        (Set$1 && getTag(new Set$1) != setTag) ||
        (WeakMap$1 && getTag(new WeakMap$1) != weakMapTag)) {
      getTag = function(value) {
        var result = objectToString.call(value),
            Ctor = result == objectTag ? value.constructor : undefined,
            ctorString = Ctor ? toSource(Ctor) : undefined;

        if (ctorString) {
          switch (ctorString) {
            case dataViewCtorString: return dataViewTag;
            case mapCtorString: return mapTag;
            case promiseCtorString: return promiseTag;
            case setCtorString: return setTag;
            case weakMapCtorString: return weakMapTag;
          }
        }
        return result;
      };
    }

    /**
     * Checks if `value` is a valid array-like index.
     *
     * @private
     * @param {*} value The value to check.
     * @param {number} [length=MAX_SAFE_INTEGER] The upper bounds of a valid index.
     * @returns {boolean} Returns `true` if `value` is a valid index, else `false`.
     */
    function isIndex(value, length) {
      length = length == null ? MAX_SAFE_INTEGER : length;
      return !!length &&
        (typeof value == 'number' || reIsUint.test(value)) &&
        (value > -1 && value % 1 == 0 && value < length);
    }

    /**
     * Checks if `func` has its source masked.
     *
     * @private
     * @param {Function} func The function to check.
     * @returns {boolean} Returns `true` if `func` is masked, else `false`.
     */
    function isMasked(func) {
      return !!maskSrcKey && (maskSrcKey in func);
    }

    /**
     * Checks if `value` is likely a prototype object.
     *
     * @private
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is a prototype, else `false`.
     */
    function isPrototype(value) {
      var Ctor = value && value.constructor,
          proto = (typeof Ctor == 'function' && Ctor.prototype) || objectProto;

      return value === proto;
    }

    /**
     * Converts `func` to its source code.
     *
     * @private
     * @param {Function} func The function to process.
     * @returns {string} Returns the source code.
     */
    function toSource(func) {
      if (func != null) {
        try {
          return funcToString.call(func);
        } catch (e) {}
        try {
          return (func + '');
        } catch (e) {}
      }
      return '';
    }

    /**
     * Checks if `value` is likely an `arguments` object.
     *
     * @static
     * @memberOf _
     * @since 0.1.0
     * @category Lang
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is an `arguments` object,
     *  else `false`.
     * @example
     *
     * _.isArguments(function() { return arguments; }());
     * // => true
     *
     * _.isArguments([1, 2, 3]);
     * // => false
     */
    function isArguments(value) {
      // Safari 8.1 makes `arguments.callee` enumerable in strict mode.
      return isArrayLikeObject(value) && hasOwnProperty.call(value, 'callee') &&
        (!propertyIsEnumerable.call(value, 'callee') || objectToString.call(value) == argsTag);
    }

    /**
     * Checks if `value` is classified as an `Array` object.
     *
     * @static
     * @memberOf _
     * @since 0.1.0
     * @category Lang
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is an array, else `false`.
     * @example
     *
     * _.isArray([1, 2, 3]);
     * // => true
     *
     * _.isArray(document.body.children);
     * // => false
     *
     * _.isArray('abc');
     * // => false
     *
     * _.isArray(_.noop);
     * // => false
     */
    var isArray = Array.isArray;

    /**
     * Checks if `value` is array-like. A value is considered array-like if it's
     * not a function and has a `value.length` that's an integer greater than or
     * equal to `0` and less than or equal to `Number.MAX_SAFE_INTEGER`.
     *
     * @static
     * @memberOf _
     * @since 4.0.0
     * @category Lang
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is array-like, else `false`.
     * @example
     *
     * _.isArrayLike([1, 2, 3]);
     * // => true
     *
     * _.isArrayLike(document.body.children);
     * // => true
     *
     * _.isArrayLike('abc');
     * // => true
     *
     * _.isArrayLike(_.noop);
     * // => false
     */
    function isArrayLike(value) {
      return value != null && isLength(value.length) && !isFunction(value);
    }

    /**
     * This method is like `_.isArrayLike` except that it also checks if `value`
     * is an object.
     *
     * @static
     * @memberOf _
     * @since 4.0.0
     * @category Lang
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is an array-like object,
     *  else `false`.
     * @example
     *
     * _.isArrayLikeObject([1, 2, 3]);
     * // => true
     *
     * _.isArrayLikeObject(document.body.children);
     * // => true
     *
     * _.isArrayLikeObject('abc');
     * // => false
     *
     * _.isArrayLikeObject(_.noop);
     * // => false
     */
    function isArrayLikeObject(value) {
      return isObjectLike(value) && isArrayLike(value);
    }

    /**
     * Checks if `value` is classified as a `Function` object.
     *
     * @static
     * @memberOf _
     * @since 0.1.0
     * @category Lang
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is a function, else `false`.
     * @example
     *
     * _.isFunction(_);
     * // => true
     *
     * _.isFunction(/abc/);
     * // => false
     */
    function isFunction(value) {
      // The use of `Object#toString` avoids issues with the `typeof` operator
      // in Safari 8-9 which returns 'object' for typed array and other constructors.
      var tag = isObject(value) ? objectToString.call(value) : '';
      return tag == funcTag || tag == genTag;
    }

    /**
     * Checks if `value` is a valid array-like length.
     *
     * **Note:** This method is loosely based on
     * [`ToLength`](http://ecma-international.org/ecma-262/7.0/#sec-tolength).
     *
     * @static
     * @memberOf _
     * @since 4.0.0
     * @category Lang
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is a valid length, else `false`.
     * @example
     *
     * _.isLength(3);
     * // => true
     *
     * _.isLength(Number.MIN_VALUE);
     * // => false
     *
     * _.isLength(Infinity);
     * // => false
     *
     * _.isLength('3');
     * // => false
     */
    function isLength(value) {
      return typeof value == 'number' &&
        value > -1 && value % 1 == 0 && value <= MAX_SAFE_INTEGER;
    }

    /**
     * Checks if `value` is the
     * [language type](http://www.ecma-international.org/ecma-262/7.0/#sec-ecmascript-language-types)
     * of `Object`. (e.g. arrays, functions, objects, regexes, `new Number(0)`, and `new String('')`)
     *
     * @static
     * @memberOf _
     * @since 0.1.0
     * @category Lang
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is an object, else `false`.
     * @example
     *
     * _.isObject({});
     * // => true
     *
     * _.isObject([1, 2, 3]);
     * // => true
     *
     * _.isObject(_.noop);
     * // => true
     *
     * _.isObject(null);
     * // => false
     */
    function isObject(value) {
      var type = typeof value;
      return !!value && (type == 'object' || type == 'function');
    }

    /**
     * Checks if `value` is object-like. A value is object-like if it's not `null`
     * and has a `typeof` result of "object".
     *
     * @static
     * @memberOf _
     * @since 4.0.0
     * @category Lang
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is object-like, else `false`.
     * @example
     *
     * _.isObjectLike({});
     * // => true
     *
     * _.isObjectLike([1, 2, 3]);
     * // => true
     *
     * _.isObjectLike(_.noop);
     * // => false
     *
     * _.isObjectLike(null);
     * // => false
     */
    function isObjectLike(value) {
      return !!value && typeof value == 'object';
    }

    /**
     * Checks if `value` is classified as a `String` primitive or object.
     *
     * @static
     * @since 0.1.0
     * @memberOf _
     * @category Lang
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is a string, else `false`.
     * @example
     *
     * _.isString('abc');
     * // => true
     *
     * _.isString(1);
     * // => false
     */
    function isString(value) {
      return typeof value == 'string' ||
        (!isArray(value) && isObjectLike(value) && objectToString.call(value) == stringTag);
    }

    /**
     * Converts `value` to an array.
     *
     * @static
     * @since 0.1.0
     * @memberOf _
     * @category Lang
     * @param {*} value The value to convert.
     * @returns {Array} Returns the converted array.
     * @example
     *
     * _.toArray({ 'a': 1, 'b': 2 });
     * // => [1, 2]
     *
     * _.toArray('abc');
     * // => ['a', 'b', 'c']
     *
     * _.toArray(1);
     * // => []
     *
     * _.toArray(null);
     * // => []
     */
    function toArray(value) {
      if (!value) {
        return [];
      }
      if (isArrayLike(value)) {
        return isString(value) ? stringToArray(value) : copyArray(value);
      }
      if (iteratorSymbol && value[iteratorSymbol]) {
        return iteratorToArray(value[iteratorSymbol]());
      }
      var tag = getTag(value),
          func = tag == mapTag ? mapToArray : (tag == setTag ? setToArray : values);

      return func(value);
    }

    /**
     * Creates an array of the own enumerable property names of `object`.
     *
     * **Note:** Non-object values are coerced to objects. See the
     * [ES spec](http://ecma-international.org/ecma-262/7.0/#sec-object.keys)
     * for more details.
     *
     * @static
     * @since 0.1.0
     * @memberOf _
     * @category Object
     * @param {Object} object The object to query.
     * @returns {Array} Returns the array of property names.
     * @example
     *
     * function Foo() {
     *   this.a = 1;
     *   this.b = 2;
     * }
     *
     * Foo.prototype.c = 3;
     *
     * _.keys(new Foo);
     * // => ['a', 'b'] (iteration order is not guaranteed)
     *
     * _.keys('hi');
     * // => ['0', '1']
     */
    function keys(object) {
      return isArrayLike(object) ? arrayLikeKeys(object) : baseKeys(object);
    }

    /**
     * Creates an array of the own enumerable string keyed property values of `object`.
     *
     * **Note:** Non-object values are coerced to objects.
     *
     * @static
     * @since 0.1.0
     * @memberOf _
     * @category Object
     * @param {Object} object The object to query.
     * @returns {Array} Returns the array of property values.
     * @example
     *
     * function Foo() {
     *   this.a = 1;
     *   this.b = 2;
     * }
     *
     * Foo.prototype.c = 3;
     *
     * _.values(new Foo);
     * // => [1, 2] (iteration order is not guaranteed)
     *
     * _.values('hi');
     * // => ['h', 'i']
     */
    function values(object) {
      return object ? baseValues(object, keys(object)) : [];
    }

    var lodash_toarray = toArray;

    var umbrella_with_rain_drops = "☔";
    var coffee = "☕";
    var aries = "♈";
    var taurus = "♉";
    var sagittarius = "♐";
    var capricorn = "♑";
    var aquarius = "♒";
    var pisces = "♓";
    var anchor = "⚓";
    var white_check_mark = "✅";
    var sparkles = "✨";
    var question = "❓";
    var grey_question = "❔";
    var grey_exclamation = "❕";
    var exclamation = "❗";
    var heavy_exclamation_mark = "❗";
    var heavy_plus_sign = "➕";
    var heavy_minus_sign = "➖";
    var heavy_division_sign = "➗";
    var hash = "#️⃣";
    var keycap_star = "*️⃣";
    var zero = "0️⃣";
    var one = "1️⃣";
    var two = "2️⃣";
    var three = "3️⃣";
    var four = "4️⃣";
    var five = "5️⃣";
    var six = "6️⃣";
    var seven = "7️⃣";
    var eight = "8️⃣";
    var nine = "9️⃣";
    var copyright = "©️";
    var registered = "®️";
    var mahjong = "🀄";
    var black_joker = "🃏";
    var a = "🅰️";
    var b = "🅱️";
    var o2 = "🅾️";
    var parking = "🅿️";
    var ab = "🆎";
    var cl = "🆑";
    var cool = "🆒";
    var free = "🆓";
    var id = "🆔";
    var ng = "🆖";
    var ok = "🆗";
    var sos = "🆘";
    var up = "🆙";
    var vs = "🆚";
    var koko = "🈁";
    var sa = "🈂️";
    var u7121 = "🈚";
    var u6307 = "🈯";
    var u7981 = "🈲";
    var u7a7a = "🈳";
    var u5408 = "🈴";
    var u6e80 = "🈵";
    var u6709 = "🈶";
    var u6708 = "🈷️";
    var u7533 = "🈸";
    var u5272 = "🈹";
    var u55b6 = "🈺";
    var ideograph_advantage = "🉐";
    var accept = "🉑";
    var cyclone = "🌀";
    var foggy = "🌁";
    var closed_umbrella = "🌂";
    var night_with_stars = "🌃";
    var sunrise_over_mountains = "🌄";
    var sunrise = "🌅";
    var city_sunset = "🌆";
    var city_sunrise = "🌇";
    var rainbow = "🌈";
    var bridge_at_night = "🌉";
    var ocean = "🌊";
    var volcano = "🌋";
    var milky_way = "🌌";
    var earth_africa = "🌍";
    var earth_americas = "🌎";
    var earth_asia = "🌏";
    var globe_with_meridians = "🌐";
    var new_moon = "🌑";
    var waxing_crescent_moon = "🌒";
    var first_quarter_moon = "🌓";
    var moon = "🌔";
    var waxing_gibbous_moon = "🌔";
    var full_moon = "🌕";
    var waning_gibbous_moon = "🌖";
    var last_quarter_moon = "🌗";
    var waning_crescent_moon = "🌘";
    var crescent_moon = "🌙";
    var new_moon_with_face = "🌚";
    var first_quarter_moon_with_face = "🌛";
    var last_quarter_moon_with_face = "🌜";
    var full_moon_with_face = "🌝";
    var sun_with_face = "🌞";
    var star2 = "🌟";
    var stars = "🌠";
    var thermometer = "🌡️";
    var mostly_sunny = "🌤️";
    var sun_small_cloud = "🌤️";
    var barely_sunny = "🌥️";
    var sun_behind_cloud = "🌥️";
    var partly_sunny_rain = "🌦️";
    var sun_behind_rain_cloud = "🌦️";
    var rain_cloud = "🌧️";
    var snow_cloud = "🌨️";
    var lightning = "🌩️";
    var lightning_cloud = "🌩️";
    var tornado = "🌪️";
    var tornado_cloud = "🌪️";
    var fog = "🌫️";
    var wind_blowing_face = "🌬️";
    var hotdog = "🌭";
    var taco = "🌮";
    var burrito = "🌯";
    var chestnut = "🌰";
    var seedling = "🌱";
    var evergreen_tree = "🌲";
    var deciduous_tree = "🌳";
    var palm_tree = "🌴";
    var cactus = "🌵";
    var hot_pepper = "🌶️";
    var tulip = "🌷";
    var cherry_blossom = "🌸";
    var rose = "🌹";
    var hibiscus = "🌺";
    var sunflower = "🌻";
    var blossom = "🌼";
    var corn = "🌽";
    var ear_of_rice = "🌾";
    var herb = "🌿";
    var four_leaf_clover = "🍀";
    var maple_leaf = "🍁";
    var fallen_leaf = "🍂";
    var leaves = "🍃";
    var mushroom = "🍄";
    var tomato = "🍅";
    var eggplant = "🍆";
    var grapes = "🍇";
    var melon = "🍈";
    var watermelon = "🍉";
    var tangerine = "🍊";
    var lemon = "🍋";
    var banana = "🍌";
    var pineapple = "🍍";
    var apple = "🍎";
    var green_apple = "🍏";
    var pear = "🍐";
    var peach = "🍑";
    var cherries = "🍒";
    var strawberry = "🍓";
    var hamburger = "🍔";
    var pizza = "🍕";
    var meat_on_bone = "🍖";
    var poultry_leg = "🍗";
    var rice_cracker = "🍘";
    var rice_ball = "🍙";
    var rice = "🍚";
    var curry = "🍛";
    var ramen = "🍜";
    var spaghetti = "🍝";
    var bread = "🍞";
    var fries = "🍟";
    var sweet_potato = "🍠";
    var dango = "🍡";
    var oden = "🍢";
    var sushi = "🍣";
    var fried_shrimp = "🍤";
    var fish_cake = "🍥";
    var icecream = "🍦";
    var shaved_ice = "🍧";
    var ice_cream = "🍨";
    var doughnut = "🍩";
    var cookie = "🍪";
    var chocolate_bar = "🍫";
    var candy = "🍬";
    var lollipop = "🍭";
    var custard = "🍮";
    var honey_pot = "🍯";
    var cake = "🍰";
    var bento = "🍱";
    var stew = "🍲";
    var fried_egg = "🍳";
    var cooking = "🍳";
    var fork_and_knife = "🍴";
    var tea = "🍵";
    var sake = "🍶";
    var wine_glass = "🍷";
    var cocktail = "🍸";
    var tropical_drink = "🍹";
    var beer = "🍺";
    var beers = "🍻";
    var baby_bottle = "🍼";
    var knife_fork_plate = "🍽️";
    var champagne = "🍾";
    var popcorn = "🍿";
    var ribbon = "🎀";
    var gift = "🎁";
    var birthday = "🎂";
    var jack_o_lantern = "🎃";
    var christmas_tree = "🎄";
    var santa = "🎅";
    var fireworks = "🎆";
    var sparkler = "🎇";
    var balloon = "🎈";
    var tada = "🎉";
    var confetti_ball = "🎊";
    var tanabata_tree = "🎋";
    var crossed_flags = "🎌";
    var bamboo = "🎍";
    var dolls = "🎎";
    var flags = "🎏";
    var wind_chime = "🎐";
    var rice_scene = "🎑";
    var school_satchel = "🎒";
    var mortar_board = "🎓";
    var medal = "🎖️";
    var reminder_ribbon = "🎗️";
    var studio_microphone = "🎙️";
    var level_slider = "🎚️";
    var control_knobs = "🎛️";
    var film_frames = "🎞️";
    var admission_tickets = "🎟️";
    var carousel_horse = "🎠";
    var ferris_wheel = "🎡";
    var roller_coaster = "🎢";
    var fishing_pole_and_fish = "🎣";
    var microphone = "🎤";
    var movie_camera = "🎥";
    var cinema = "🎦";
    var headphones = "🎧";
    var art = "🎨";
    var tophat = "🎩";
    var circus_tent = "🎪";
    var ticket = "🎫";
    var clapper = "🎬";
    var performing_arts = "🎭";
    var video_game = "🎮";
    var dart = "🎯";
    var slot_machine = "🎰";
    var game_die = "🎲";
    var bowling = "🎳";
    var flower_playing_cards = "🎴";
    var musical_note = "🎵";
    var notes = "🎶";
    var saxophone = "🎷";
    var guitar = "🎸";
    var musical_keyboard = "🎹";
    var trumpet = "🎺";
    var violin = "🎻";
    var musical_score = "🎼";
    var running_shirt_with_sash = "🎽";
    var tennis = "🎾";
    var ski = "🎿";
    var basketball = "🏀";
    var checkered_flag = "🏁";
    var snowboarder = "🏂";
    var runner = "🏃‍♂️";
    var running = "🏃‍♂️";
    var surfer = "🏄‍♂️";
    var sports_medal = "🏅";
    var trophy = "🏆";
    var horse_racing = "🏇";
    var football = "🏈";
    var rugby_football = "🏉";
    var swimmer = "🏊‍♂️";
    var weight_lifter = "🏋️‍♂️";
    var golfer = "🏌️‍♂️";
    var racing_motorcycle = "🏍️";
    var racing_car = "🏎️";
    var cricket_bat_and_ball = "🏏";
    var volleyball = "🏐";
    var field_hockey_stick_and_ball = "🏑";
    var ice_hockey_stick_and_puck = "🏒";
    var table_tennis_paddle_and_ball = "🏓";
    var snow_capped_mountain = "🏔️";
    var camping = "🏕️";
    var beach_with_umbrella = "🏖️";
    var building_construction = "🏗️";
    var house_buildings = "🏘️";
    var cityscape = "🏙️";
    var derelict_house_building = "🏚️";
    var classical_building = "🏛️";
    var desert = "🏜️";
    var desert_island = "🏝️";
    var national_park = "🏞️";
    var stadium = "🏟️";
    var house = "🏠";
    var house_with_garden = "🏡";
    var office = "🏢";
    var post_office = "🏣";
    var european_post_office = "🏤";
    var hospital = "🏥";
    var bank = "🏦";
    var atm = "🏧";
    var hotel = "🏨";
    var love_hotel = "🏩";
    var convenience_store = "🏪";
    var school = "🏫";
    var department_store = "🏬";
    var factory = "🏭";
    var izakaya_lantern = "🏮";
    var lantern = "🏮";
    var japanese_castle = "🏯";
    var european_castle = "🏰";
    var waving_white_flag = "🏳️";
    var waving_black_flag = "🏴";
    var rosette = "🏵️";
    var label = "🏷️";
    var badminton_racquet_and_shuttlecock = "🏸";
    var bow_and_arrow = "🏹";
    var amphora = "🏺";
    var rat = "🐀";
    var mouse2 = "🐁";
    var ox = "🐂";
    var water_buffalo = "🐃";
    var cow2 = "🐄";
    var tiger2 = "🐅";
    var leopard = "🐆";
    var rabbit2 = "🐇";
    var cat2 = "🐈";
    var dragon = "🐉";
    var crocodile = "🐊";
    var whale2 = "🐋";
    var snail = "🐌";
    var snake = "🐍";
    var racehorse = "🐎";
    var ram = "🐏";
    var goat = "🐐";
    var sheep = "🐑";
    var monkey = "🐒";
    var rooster = "🐓";
    var chicken = "🐔";
    var dog2 = "🐕";
    var pig2 = "🐖";
    var boar = "🐗";
    var elephant = "🐘";
    var octopus = "🐙";
    var shell = "🐚";
    var bug = "🐛";
    var ant = "🐜";
    var bee = "🐝";
    var honeybee = "🐝";
    var beetle = "🐞";
    var fish = "🐟";
    var tropical_fish = "🐠";
    var blowfish = "🐡";
    var turtle = "🐢";
    var hatching_chick = "🐣";
    var baby_chick = "🐤";
    var hatched_chick = "🐥";
    var bird = "🐦";
    var penguin = "🐧";
    var koala = "🐨";
    var poodle = "🐩";
    var dromedary_camel = "🐪";
    var camel = "🐫";
    var dolphin = "🐬";
    var flipper = "🐬";
    var mouse = "🐭";
    var cow = "🐮";
    var tiger = "🐯";
    var rabbit = "🐰";
    var cat = "🐱";
    var dragon_face = "🐲";
    var whale = "🐳";
    var horse = "🐴";
    var monkey_face = "🐵";
    var dog = "🐶";
    var pig = "🐷";
    var frog = "🐸";
    var hamster = "🐹";
    var wolf = "🐺";
    var bear = "🐻";
    var panda_face = "🐼";
    var pig_nose = "🐽";
    var feet = "🐾";
    var paw_prints = "🐾";
    var chipmunk = "🐿️";
    var eyes = "👀";
    var eye = "👁️";
    var ear = "👂";
    var nose = "👃";
    var lips = "👄";
    var tongue = "👅";
    var point_up_2 = "👆";
    var point_down = "👇";
    var point_left = "👈";
    var point_right = "👉";
    var facepunch = "👊";
    var punch = "👊";
    var wave = "👋";
    var ok_hand = "👌";
    var thumbsup = "👍";
    var thumbsdown = "👎";
    var clap = "👏";
    var open_hands = "👐";
    var crown = "👑";
    var womans_hat = "👒";
    var eyeglasses = "👓";
    var necktie = "👔";
    var shirt = "👕";
    var tshirt = "👕";
    var jeans = "👖";
    var dress = "👗";
    var kimono = "👘";
    var bikini = "👙";
    var womans_clothes = "👚";
    var purse = "👛";
    var handbag = "👜";
    var pouch = "👝";
    var mans_shoe = "👞";
    var shoe = "👞";
    var athletic_shoe = "👟";
    var high_heel = "👠";
    var sandal = "👡";
    var boot = "👢";
    var footprints = "👣";
    var bust_in_silhouette = "👤";
    var busts_in_silhouette = "👥";
    var boy = "👦";
    var girl = "👧";
    var family = "👨‍👩‍👦";
    var man = "👨";
    var couple_with_heart = "👩‍❤️‍👨";
    var couplekiss = "👩‍❤️‍💋‍👨";
    var woman = "👩";
    var couple = "👫";
    var man_and_woman_holding_hands = "👫";
    var two_men_holding_hands = "👬";
    var two_women_holding_hands = "👭";
    var cop = "👮‍♂️";
    var dancers = "👯‍♀️";
    var bride_with_veil = "👰";
    var person_with_blond_hair = "👱‍♂️";
    var man_with_gua_pi_mao = "👲";
    var man_with_turban = "👳‍♂️";
    var older_man = "👴";
    var older_woman = "👵";
    var baby = "👶";
    var construction_worker = "👷‍♂️";
    var princess = "👸";
    var japanese_ogre = "👹";
    var japanese_goblin = "👺";
    var ghost = "👻";
    var angel = "👼";
    var alien = "👽";
    var space_invader = "👾";
    var imp = "👿";
    var skull = "💀";
    var information_desk_person = "💁‍♀️";
    var guardsman = "💂‍♂️";
    var dancer = "💃";
    var lipstick = "💄";
    var nail_care = "💅";
    var massage = "💆‍♀️";
    var haircut = "💇‍♀️";
    var barber = "💈";
    var syringe = "💉";
    var pill = "💊";
    var kiss = "💋";
    var love_letter = "💌";
    var ring = "💍";
    var gem = "💎";
    var bouquet = "💐";
    var wedding = "💒";
    var heartbeat = "💓";
    var broken_heart = "💔";
    var two_hearts = "💕";
    var sparkling_heart = "💖";
    var heartpulse = "💗";
    var cupid = "💘";
    var blue_heart = "💙";
    var green_heart = "💚";
    var yellow_heart = "💛";
    var purple_heart = "💜";
    var gift_heart = "💝";
    var revolving_hearts = "💞";
    var heart_decoration = "💟";
    var diamond_shape_with_a_dot_inside = "💠";
    var bulb = "💡";
    var anger = "💢";
    var bomb = "💣";
    var zzz = "💤";
    var boom = "💥";
    var collision = "💥";
    var sweat_drops = "💦";
    var droplet = "💧";
    var dash = "💨";
    var poop = "💩";
    var muscle = "💪";
    var dizzy = "💫";
    var speech_balloon = "💬";
    var thought_balloon = "💭";
    var white_flower = "💮";
    var moneybag = "💰";
    var currency_exchange = "💱";
    var heavy_dollar_sign = "💲";
    var credit_card = "💳";
    var yen = "💴";
    var dollar = "💵";
    var euro = "💶";
    var pound = "💷";
    var money_with_wings = "💸";
    var chart = "💹";
    var seat = "💺";
    var computer = "💻";
    var briefcase = "💼";
    var minidisc = "💽";
    var floppy_disk = "💾";
    var cd = "💿";
    var dvd = "📀";
    var file_folder = "📁";
    var open_file_folder = "📂";
    var page_with_curl = "📃";
    var page_facing_up = "📄";
    var date = "📅";
    var calendar = "📆";
    var card_index = "📇";
    var chart_with_upwards_trend = "📈";
    var chart_with_downwards_trend = "📉";
    var bar_chart = "📊";
    var clipboard = "📋";
    var pushpin = "📌";
    var round_pushpin = "📍";
    var paperclip = "📎";
    var straight_ruler = "📏";
    var triangular_ruler = "📐";
    var bookmark_tabs = "📑";
    var ledger = "📒";
    var notebook = "📓";
    var notebook_with_decorative_cover = "📔";
    var closed_book = "📕";
    var book = "📖";
    var open_book = "📖";
    var green_book = "📗";
    var blue_book = "📘";
    var orange_book = "📙";
    var books = "📚";
    var name_badge = "📛";
    var scroll = "📜";
    var memo = "📝";
    var pencil = "📝";
    var telephone_receiver = "📞";
    var pager = "📟";
    var fax = "📠";
    var satellite_antenna = "📡";
    var loudspeaker = "📢";
    var mega = "📣";
    var outbox_tray = "📤";
    var inbox_tray = "📥";
    var incoming_envelope = "📨";
    var envelope_with_arrow = "📩";
    var mailbox_closed = "📪";
    var mailbox = "📫";
    var mailbox_with_mail = "📬";
    var mailbox_with_no_mail = "📭";
    var postbox = "📮";
    var postal_horn = "📯";
    var newspaper = "📰";
    var iphone = "📱";
    var calling = "📲";
    var vibration_mode = "📳";
    var mobile_phone_off = "📴";
    var no_mobile_phones = "📵";
    var signal_strength = "📶";
    var camera = "📷";
    var camera_with_flash = "📸";
    var video_camera = "📹";
    var tv = "📺";
    var radio = "📻";
    var vhs = "📼";
    var film_projector = "📽️";
    var prayer_beads = "📿";
    var twisted_rightwards_arrows = "🔀";
    var repeat = "🔁";
    var repeat_one = "🔂";
    var arrows_clockwise = "🔃";
    var arrows_counterclockwise = "🔄";
    var low_brightness = "🔅";
    var high_brightness = "🔆";
    var mute = "🔇";
    var speaker = "🔈";
    var sound = "🔉";
    var loud_sound = "🔊";
    var battery = "🔋";
    var electric_plug = "🔌";
    var mag = "🔍";
    var mag_right = "🔎";
    var lock_with_ink_pen = "🔏";
    var closed_lock_with_key = "🔐";
    var key = "🔑";
    var lock = "🔒";
    var unlock = "🔓";
    var bell = "🔔";
    var no_bell = "🔕";
    var bookmark = "🔖";
    var link = "🔗";
    var radio_button = "🔘";
    var back = "🔙";
    var end = "🔚";
    var on = "🔛";
    var soon = "🔜";
    var underage = "🔞";
    var keycap_ten = "🔟";
    var capital_abcd = "🔠";
    var abcd = "🔡";
    var symbols = "🔣";
    var abc = "🔤";
    var fire = "🔥";
    var flashlight = "🔦";
    var wrench = "🔧";
    var hammer = "🔨";
    var nut_and_bolt = "🔩";
    var hocho = "🔪";
    var knife = "🔪";
    var gun = "🔫";
    var microscope = "🔬";
    var telescope = "🔭";
    var crystal_ball = "🔮";
    var six_pointed_star = "🔯";
    var beginner = "🔰";
    var trident = "🔱";
    var black_square_button = "🔲";
    var white_square_button = "🔳";
    var red_circle = "🔴";
    var large_blue_circle = "🔵";
    var large_orange_diamond = "🔶";
    var large_blue_diamond = "🔷";
    var small_orange_diamond = "🔸";
    var small_blue_diamond = "🔹";
    var small_red_triangle = "🔺";
    var small_red_triangle_down = "🔻";
    var arrow_up_small = "🔼";
    var arrow_down_small = "🔽";
    var om_symbol = "🕉️";
    var dove_of_peace = "🕊️";
    var kaaba = "🕋";
    var mosque = "🕌";
    var synagogue = "🕍";
    var menorah_with_nine_branches = "🕎";
    var candle = "🕯️";
    var mantelpiece_clock = "🕰️";
    var hole = "🕳️";
    var man_in_business_suit_levitating = "🕴️";
    var sleuth_or_spy = "🕵️‍♂️";
    var dark_sunglasses = "🕶️";
    var spider = "🕷️";
    var spider_web = "🕸️";
    var joystick = "🕹️";
    var man_dancing = "🕺";
    var linked_paperclips = "🖇️";
    var lower_left_ballpoint_pen = "🖊️";
    var lower_left_fountain_pen = "🖋️";
    var lower_left_paintbrush = "🖌️";
    var lower_left_crayon = "🖍️";
    var raised_hand_with_fingers_splayed = "🖐️";
    var middle_finger = "🖕";
    var reversed_hand_with_middle_finger_extended = "🖕";
    var black_heart = "🖤";
    var desktop_computer = "🖥️";
    var printer = "🖨️";
    var three_button_mouse = "🖱️";
    var trackball = "🖲️";
    var frame_with_picture = "🖼️";
    var card_index_dividers = "🗂️";
    var card_file_box = "🗃️";
    var file_cabinet = "🗄️";
    var wastebasket = "🗑️";
    var spiral_note_pad = "🗒️";
    var spiral_calendar_pad = "🗓️";
    var compression = "🗜️";
    var old_key = "🗝️";
    var rolled_up_newspaper = "🗞️";
    var dagger_knife = "🗡️";
    var speaking_head_in_silhouette = "🗣️";
    var left_speech_bubble = "🗨️";
    var right_anger_bubble = "🗯️";
    var ballot_box_with_ballot = "🗳️";
    var world_map = "🗺️";
    var mount_fuji = "🗻";
    var tokyo_tower = "🗼";
    var statue_of_liberty = "🗽";
    var japan = "🗾";
    var moyai = "🗿";
    var grinning = "😀";
    var grin = "😁";
    var joy = "😂";
    var smiley = "😃";
    var smile = "😄";
    var sweat_smile = "😅";
    var laughing = "😆";
    var satisfied = "😆";
    var innocent = "😇";
    var smiling_imp = "😈";
    var wink = "😉";
    var blush = "😊";
    var yum = "😋";
    var relieved = "😌";
    var heart_eyes = "😍";
    var sunglasses = "😎";
    var smirk = "😏";
    var neutral_face = "😐";
    var expressionless = "😑";
    var unamused = "😒";
    var sweat = "😓";
    var pensive = "😔";
    var confused = "😕";
    var confounded = "😖";
    var kissing = "😗";
    var kissing_heart = "😘";
    var kissing_smiling_eyes = "😙";
    var kissing_closed_eyes = "😚";
    var stuck_out_tongue = "😛";
    var stuck_out_tongue_winking_eye = "😜";
    var stuck_out_tongue_closed_eyes = "😝";
    var disappointed = "😞";
    var worried = "😟";
    var angry = "😠";
    var rage = "😡";
    var cry = "😢";
    var persevere = "😣";
    var triumph = "😤";
    var disappointed_relieved = "😥";
    var frowning = "😦";
    var anguished = "😧";
    var fearful = "😨";
    var weary = "😩";
    var sleepy = "😪";
    var tired_face = "😫";
    var grimacing = "😬";
    var sob = "😭";
    var open_mouth = "😮";
    var hushed = "😯";
    var cold_sweat = "😰";
    var scream = "😱";
    var astonished = "😲";
    var flushed = "😳";
    var sleeping = "😴";
    var dizzy_face = "😵";
    var no_mouth = "😶";
    var mask = "😷";
    var smile_cat = "😸";
    var joy_cat = "😹";
    var smiley_cat = "😺";
    var heart_eyes_cat = "😻";
    var smirk_cat = "😼";
    var kissing_cat = "😽";
    var pouting_cat = "😾";
    var crying_cat_face = "😿";
    var scream_cat = "🙀";
    var slightly_frowning_face = "🙁";
    var slightly_smiling_face = "🙂";
    var upside_down_face = "🙃";
    var face_with_rolling_eyes = "🙄";
    var no_good = "🙅‍♀️";
    var ok_woman = "🙆‍♀️";
    var bow = "🙇‍♂️";
    var see_no_evil = "🙈";
    var hear_no_evil = "🙉";
    var speak_no_evil = "🙊";
    var raising_hand = "🙋‍♀️";
    var raised_hands = "🙌";
    var person_frowning = "🙍‍♀️";
    var person_with_pouting_face = "🙎‍♀️";
    var pray = "🙏";
    var rocket = "🚀";
    var helicopter = "🚁";
    var steam_locomotive = "🚂";
    var railway_car = "🚃";
    var bullettrain_side = "🚄";
    var bullettrain_front = "🚅";
    var train2 = "🚆";
    var metro = "🚇";
    var light_rail = "🚈";
    var station = "🚉";
    var tram = "🚊";
    var train = "🚋";
    var bus = "🚌";
    var oncoming_bus = "🚍";
    var trolleybus = "🚎";
    var busstop = "🚏";
    var minibus = "🚐";
    var ambulance = "🚑";
    var fire_engine = "🚒";
    var police_car = "🚓";
    var oncoming_police_car = "🚔";
    var taxi = "🚕";
    var oncoming_taxi = "🚖";
    var car = "🚗";
    var red_car = "🚗";
    var oncoming_automobile = "🚘";
    var blue_car = "🚙";
    var truck = "🚚";
    var articulated_lorry = "🚛";
    var tractor = "🚜";
    var monorail = "🚝";
    var mountain_railway = "🚞";
    var suspension_railway = "🚟";
    var mountain_cableway = "🚠";
    var aerial_tramway = "🚡";
    var ship = "🚢";
    var rowboat = "🚣‍♂️";
    var speedboat = "🚤";
    var traffic_light = "🚥";
    var vertical_traffic_light = "🚦";
    var construction = "🚧";
    var rotating_light = "🚨";
    var triangular_flag_on_post = "🚩";
    var door = "🚪";
    var no_entry_sign = "🚫";
    var smoking = "🚬";
    var no_smoking = "🚭";
    var put_litter_in_its_place = "🚮";
    var do_not_litter = "🚯";
    var potable_water = "🚰";
    var bike = "🚲";
    var no_bicycles = "🚳";
    var bicyclist = "🚴‍♂️";
    var mountain_bicyclist = "🚵‍♂️";
    var walking = "🚶‍♂️";
    var no_pedestrians = "🚷";
    var children_crossing = "🚸";
    var mens = "🚹";
    var womens = "🚺";
    var restroom = "🚻";
    var baby_symbol = "🚼";
    var toilet = "🚽";
    var wc = "🚾";
    var shower = "🚿";
    var bath = "🛀";
    var bathtub = "🛁";
    var passport_control = "🛂";
    var customs = "🛃";
    var baggage_claim = "🛄";
    var left_luggage = "🛅";
    var couch_and_lamp = "🛋️";
    var sleeping_accommodation = "🛌";
    var shopping_bags = "🛍️";
    var bellhop_bell = "🛎️";
    var bed = "🛏️";
    var place_of_worship = "🛐";
    var octagonal_sign = "🛑";
    var shopping_trolley = "🛒";
    var hammer_and_wrench = "🛠️";
    var shield = "🛡️";
    var oil_drum = "🛢️";
    var motorway = "🛣️";
    var railway_track = "🛤️";
    var motor_boat = "🛥️";
    var small_airplane = "🛩️";
    var airplane_departure = "🛫";
    var airplane_arriving = "🛬";
    var satellite = "🛰️";
    var passenger_ship = "🛳️";
    var scooter = "🛴";
    var motor_scooter = "🛵";
    var canoe = "🛶";
    var sled = "🛷";
    var flying_saucer = "🛸";
    var zipper_mouth_face = "🤐";
    var money_mouth_face = "🤑";
    var face_with_thermometer = "🤒";
    var nerd_face = "🤓";
    var thinking_face = "🤔";
    var face_with_head_bandage = "🤕";
    var robot_face = "🤖";
    var hugging_face = "🤗";
    var the_horns = "🤘";
    var sign_of_the_horns = "🤘";
    var call_me_hand = "🤙";
    var raised_back_of_hand = "🤚";
    var handshake = "🤝";
    var crossed_fingers = "🤞";
    var hand_with_index_and_middle_fingers_crossed = "🤞";
    var i_love_you_hand_sign = "🤟";
    var face_with_cowboy_hat = "🤠";
    var clown_face = "🤡";
    var nauseated_face = "🤢";
    var rolling_on_the_floor_laughing = "🤣";
    var drooling_face = "🤤";
    var lying_face = "🤥";
    var face_palm = "🤦";
    var sneezing_face = "🤧";
    var face_with_raised_eyebrow = "🤨";
    var face_with_one_eyebrow_raised = "🤨";
    var grinning_face_with_star_eyes = "🤩";
    var zany_face = "🤪";
    var grinning_face_with_one_large_and_one_small_eye = "🤪";
    var shushing_face = "🤫";
    var face_with_finger_covering_closed_lips = "🤫";
    var face_with_symbols_on_mouth = "🤬";
    var serious_face_with_symbols_covering_mouth = "🤬";
    var face_with_hand_over_mouth = "🤭";
    var smiling_face_with_smiling_eyes_and_hand_covering_mouth = "🤭";
    var face_vomiting = "🤮";
    var face_with_open_mouth_vomiting = "🤮";
    var exploding_head = "🤯";
    var shocked_face_with_exploding_head = "🤯";
    var pregnant_woman = "🤰";
    var palms_up_together = "🤲";
    var selfie = "🤳";
    var prince = "🤴";
    var man_in_tuxedo = "🤵";
    var mrs_claus = "🤶";
    var mother_christmas = "🤶";
    var shrug = "🤷";
    var person_doing_cartwheel = "🤸";
    var juggling = "🤹";
    var fencer = "🤺";
    var wrestlers = "🤼";
    var water_polo = "🤽";
    var handball = "🤾";
    var wilted_flower = "🥀";
    var drum_with_drumsticks = "🥁";
    var clinking_glasses = "🥂";
    var tumbler_glass = "🥃";
    var spoon = "🥄";
    var goal_net = "🥅";
    var first_place_medal = "🥇";
    var second_place_medal = "🥈";
    var third_place_medal = "🥉";
    var boxing_glove = "🥊";
    var martial_arts_uniform = "🥋";
    var curling_stone = "🥌";
    var croissant = "🥐";
    var avocado = "🥑";
    var cucumber = "🥒";
    var bacon = "🥓";
    var potato = "🥔";
    var carrot = "🥕";
    var baguette_bread = "🥖";
    var green_salad = "🥗";
    var shallow_pan_of_food = "🥘";
    var stuffed_flatbread = "🥙";
    var egg = "🥚";
    var glass_of_milk = "🥛";
    var peanuts = "🥜";
    var kiwifruit = "🥝";
    var pancakes = "🥞";
    var dumpling = "🥟";
    var fortune_cookie = "🥠";
    var takeout_box = "🥡";
    var chopsticks = "🥢";
    var bowl_with_spoon = "🥣";
    var cup_with_straw = "🥤";
    var coconut = "🥥";
    var broccoli = "🥦";
    var pie = "🥧";
    var pretzel = "🥨";
    var cut_of_meat = "🥩";
    var sandwich = "🥪";
    var canned_food = "🥫";
    var crab = "🦀";
    var lion_face = "🦁";
    var scorpion = "🦂";
    var turkey = "🦃";
    var unicorn_face = "🦄";
    var eagle = "🦅";
    var duck = "🦆";
    var bat = "🦇";
    var shark = "🦈";
    var owl = "🦉";
    var fox_face = "🦊";
    var butterfly = "🦋";
    var deer = "🦌";
    var gorilla = "🦍";
    var lizard = "🦎";
    var rhinoceros = "🦏";
    var shrimp = "🦐";
    var squid = "🦑";
    var giraffe_face = "🦒";
    var zebra_face = "🦓";
    var hedgehog = "🦔";
    var sauropod = "🦕";
    var cricket = "🦗";
    var cheese_wedge = "🧀";
    var face_with_monocle = "🧐";
    var adult = "🧑";
    var child = "🧒";
    var older_adult = "🧓";
    var bearded_person = "🧔";
    var person_with_headscarf = "🧕";
    var woman_in_steamy_room = "🧖‍♀️";
    var man_in_steamy_room = "🧖‍♂️";
    var person_in_steamy_room = "🧖‍♂️";
    var woman_climbing = "🧗‍♀️";
    var person_climbing = "🧗‍♀️";
    var man_climbing = "🧗‍♂️";
    var woman_in_lotus_position = "🧘‍♀️";
    var person_in_lotus_position = "🧘‍♀️";
    var man_in_lotus_position = "🧘‍♂️";
    var female_mage = "🧙‍♀️";
    var mage = "🧙‍♀️";
    var male_mage = "🧙‍♂️";
    var female_fairy = "🧚‍♀️";
    var fairy = "🧚‍♀️";
    var male_fairy = "🧚‍♂️";
    var female_vampire = "🧛‍♀️";
    var vampire = "🧛‍♀️";
    var male_vampire = "🧛‍♂️";
    var mermaid = "🧜‍♀️";
    var merman = "🧜‍♂️";
    var merperson = "🧜‍♂️";
    var female_elf = "🧝‍♀️";
    var male_elf = "🧝‍♂️";
    var elf = "🧝‍♂️";
    var female_genie = "🧞‍♀️";
    var male_genie = "🧞‍♂️";
    var genie = "🧞‍♂️";
    var female_zombie = "🧟‍♀️";
    var male_zombie = "🧟‍♂️";
    var zombie = "🧟‍♂️";
    var brain = "🧠";
    var orange_heart = "🧡";
    var billed_cap = "🧢";
    var scarf = "🧣";
    var gloves = "🧤";
    var coat = "🧥";
    var socks = "🧦";
    var bangbang = "‼️";
    var interrobang = "⁉️";
    var tm = "™️";
    var information_source = "ℹ️";
    var left_right_arrow = "↔️";
    var arrow_up_down = "↕️";
    var arrow_upper_left = "↖️";
    var arrow_upper_right = "↗️";
    var arrow_lower_right = "↘️";
    var arrow_lower_left = "↙️";
    var leftwards_arrow_with_hook = "↩️";
    var arrow_right_hook = "↪️";
    var watch = "⌚";
    var hourglass = "⌛";
    var keyboard = "⌨️";
    var eject = "⏏️";
    var fast_forward = "⏩";
    var rewind = "⏪";
    var arrow_double_up = "⏫";
    var arrow_double_down = "⏬";
    var black_right_pointing_double_triangle_with_vertical_bar = "⏭️";
    var black_left_pointing_double_triangle_with_vertical_bar = "⏮️";
    var black_right_pointing_triangle_with_double_vertical_bar = "⏯️";
    var alarm_clock = "⏰";
    var stopwatch = "⏱️";
    var timer_clock = "⏲️";
    var hourglass_flowing_sand = "⏳";
    var double_vertical_bar = "⏸️";
    var black_square_for_stop = "⏹️";
    var black_circle_for_record = "⏺️";
    var m = "Ⓜ️";
    var black_small_square = "▪️";
    var white_small_square = "▫️";
    var arrow_forward = "▶️";
    var arrow_backward = "◀️";
    var white_medium_square = "◻️";
    var black_medium_square = "◼️";
    var white_medium_small_square = "◽";
    var black_medium_small_square = "◾";
    var sunny = "☀️";
    var cloud = "☁️";
    var umbrella = "☂️";
    var snowman = "☃️";
    var comet = "☄️";
    var phone = "☎️";
    var telephone = "☎️";
    var ballot_box_with_check = "☑️";
    var shamrock = "☘️";
    var point_up = "☝️";
    var skull_and_crossbones = "☠️";
    var radioactive_sign = "☢️";
    var biohazard_sign = "☣️";
    var orthodox_cross = "☦️";
    var star_and_crescent = "☪️";
    var peace_symbol = "☮️";
    var yin_yang = "☯️";
    var wheel_of_dharma = "☸️";
    var white_frowning_face = "☹️";
    var relaxed = "☺️";
    var female_sign = "♀️";
    var male_sign = "♂️";
    var gemini = "♊";
    var cancer = "♋";
    var leo = "♌";
    var virgo = "♍";
    var libra = "♎";
    var scorpius = "♏";
    var spades = "♠️";
    var clubs = "♣️";
    var hearts = "♥️";
    var diamonds = "♦️";
    var hotsprings = "♨️";
    var recycle = "♻️";
    var wheelchair = "♿";
    var hammer_and_pick = "⚒️";
    var crossed_swords = "⚔️";
    var medical_symbol = "⚕️";
    var staff_of_aesculapius = "⚕️";
    var scales = "⚖️";
    var alembic = "⚗️";
    var gear = "⚙️";
    var atom_symbol = "⚛️";
    var fleur_de_lis = "⚜️";
    var warning = "⚠️";
    var zap = "⚡";
    var white_circle = "⚪";
    var black_circle = "⚫";
    var coffin = "⚰️";
    var funeral_urn = "⚱️";
    var soccer = "⚽";
    var baseball = "⚾";
    var snowman_without_snow = "⛄";
    var partly_sunny = "⛅";
    var thunder_cloud_and_rain = "⛈️";
    var ophiuchus = "⛎";
    var pick = "⛏️";
    var helmet_with_white_cross = "⛑️";
    var chains = "⛓️";
    var no_entry = "⛔";
    var shinto_shrine = "⛩️";
    var church = "⛪";
    var mountain = "⛰️";
    var umbrella_on_ground = "⛱️";
    var fountain = "⛲";
    var golf = "⛳";
    var ferry = "⛴️";
    var boat = "⛵";
    var sailboat = "⛵";
    var skier = "⛷️";
    var ice_skate = "⛸️";
    var person_with_ball = "⛹️‍♂️";
    var tent = "⛺";
    var fuelpump = "⛽";
    var scissors = "✂️";
    var airplane = "✈️";
    var email = "✉️";
    var envelope = "✉️";
    var fist = "✊";
    var hand = "✋";
    var raised_hand = "✋";
    var v = "✌️";
    var writing_hand = "✍️";
    var pencil2 = "✏️";
    var black_nib = "✒️";
    var heavy_check_mark = "✔️";
    var heavy_multiplication_x = "✖️";
    var latin_cross = "✝️";
    var star_of_david = "✡️";
    var eight_spoked_asterisk = "✳️";
    var eight_pointed_black_star = "✴️";
    var snowflake = "❄️";
    var sparkle = "❇️";
    var x = "❌";
    var negative_squared_cross_mark = "❎";
    var heavy_heart_exclamation_mark_ornament = "❣️";
    var heart = "❤️";
    var arrow_right = "➡️";
    var curly_loop = "➰";
    var loop = "➿";
    var arrow_heading_up = "⤴️";
    var arrow_heading_down = "⤵️";
    var arrow_left = "⬅️";
    var arrow_up = "⬆️";
    var arrow_down = "⬇️";
    var black_large_square = "⬛";
    var white_large_square = "⬜";
    var star = "⭐";
    var o = "⭕";
    var wavy_dash = "〰️";
    var part_alternation_mark = "〽️";
    var congratulations = "㊗️";
    var secret = "㊙️";
    var emoji = {
    	"100": "💯",
    	"1234": "🔢",
    	umbrella_with_rain_drops: umbrella_with_rain_drops,
    	coffee: coffee,
    	aries: aries,
    	taurus: taurus,
    	sagittarius: sagittarius,
    	capricorn: capricorn,
    	aquarius: aquarius,
    	pisces: pisces,
    	anchor: anchor,
    	white_check_mark: white_check_mark,
    	sparkles: sparkles,
    	question: question,
    	grey_question: grey_question,
    	grey_exclamation: grey_exclamation,
    	exclamation: exclamation,
    	heavy_exclamation_mark: heavy_exclamation_mark,
    	heavy_plus_sign: heavy_plus_sign,
    	heavy_minus_sign: heavy_minus_sign,
    	heavy_division_sign: heavy_division_sign,
    	hash: hash,
    	keycap_star: keycap_star,
    	zero: zero,
    	one: one,
    	two: two,
    	three: three,
    	four: four,
    	five: five,
    	six: six,
    	seven: seven,
    	eight: eight,
    	nine: nine,
    	copyright: copyright,
    	registered: registered,
    	mahjong: mahjong,
    	black_joker: black_joker,
    	a: a,
    	b: b,
    	o2: o2,
    	parking: parking,
    	ab: ab,
    	cl: cl,
    	cool: cool,
    	free: free,
    	id: id,
    	"new": "🆕",
    	ng: ng,
    	ok: ok,
    	sos: sos,
    	up: up,
    	vs: vs,
    	koko: koko,
    	sa: sa,
    	u7121: u7121,
    	u6307: u6307,
    	u7981: u7981,
    	u7a7a: u7a7a,
    	u5408: u5408,
    	u6e80: u6e80,
    	u6709: u6709,
    	u6708: u6708,
    	u7533: u7533,
    	u5272: u5272,
    	u55b6: u55b6,
    	ideograph_advantage: ideograph_advantage,
    	accept: accept,
    	cyclone: cyclone,
    	foggy: foggy,
    	closed_umbrella: closed_umbrella,
    	night_with_stars: night_with_stars,
    	sunrise_over_mountains: sunrise_over_mountains,
    	sunrise: sunrise,
    	city_sunset: city_sunset,
    	city_sunrise: city_sunrise,
    	rainbow: rainbow,
    	bridge_at_night: bridge_at_night,
    	ocean: ocean,
    	volcano: volcano,
    	milky_way: milky_way,
    	earth_africa: earth_africa,
    	earth_americas: earth_americas,
    	earth_asia: earth_asia,
    	globe_with_meridians: globe_with_meridians,
    	new_moon: new_moon,
    	waxing_crescent_moon: waxing_crescent_moon,
    	first_quarter_moon: first_quarter_moon,
    	moon: moon,
    	waxing_gibbous_moon: waxing_gibbous_moon,
    	full_moon: full_moon,
    	waning_gibbous_moon: waning_gibbous_moon,
    	last_quarter_moon: last_quarter_moon,
    	waning_crescent_moon: waning_crescent_moon,
    	crescent_moon: crescent_moon,
    	new_moon_with_face: new_moon_with_face,
    	first_quarter_moon_with_face: first_quarter_moon_with_face,
    	last_quarter_moon_with_face: last_quarter_moon_with_face,
    	full_moon_with_face: full_moon_with_face,
    	sun_with_face: sun_with_face,
    	star2: star2,
    	stars: stars,
    	thermometer: thermometer,
    	mostly_sunny: mostly_sunny,
    	sun_small_cloud: sun_small_cloud,
    	barely_sunny: barely_sunny,
    	sun_behind_cloud: sun_behind_cloud,
    	partly_sunny_rain: partly_sunny_rain,
    	sun_behind_rain_cloud: sun_behind_rain_cloud,
    	rain_cloud: rain_cloud,
    	snow_cloud: snow_cloud,
    	lightning: lightning,
    	lightning_cloud: lightning_cloud,
    	tornado: tornado,
    	tornado_cloud: tornado_cloud,
    	fog: fog,
    	wind_blowing_face: wind_blowing_face,
    	hotdog: hotdog,
    	taco: taco,
    	burrito: burrito,
    	chestnut: chestnut,
    	seedling: seedling,
    	evergreen_tree: evergreen_tree,
    	deciduous_tree: deciduous_tree,
    	palm_tree: palm_tree,
    	cactus: cactus,
    	hot_pepper: hot_pepper,
    	tulip: tulip,
    	cherry_blossom: cherry_blossom,
    	rose: rose,
    	hibiscus: hibiscus,
    	sunflower: sunflower,
    	blossom: blossom,
    	corn: corn,
    	ear_of_rice: ear_of_rice,
    	herb: herb,
    	four_leaf_clover: four_leaf_clover,
    	maple_leaf: maple_leaf,
    	fallen_leaf: fallen_leaf,
    	leaves: leaves,
    	mushroom: mushroom,
    	tomato: tomato,
    	eggplant: eggplant,
    	grapes: grapes,
    	melon: melon,
    	watermelon: watermelon,
    	tangerine: tangerine,
    	lemon: lemon,
    	banana: banana,
    	pineapple: pineapple,
    	apple: apple,
    	green_apple: green_apple,
    	pear: pear,
    	peach: peach,
    	cherries: cherries,
    	strawberry: strawberry,
    	hamburger: hamburger,
    	pizza: pizza,
    	meat_on_bone: meat_on_bone,
    	poultry_leg: poultry_leg,
    	rice_cracker: rice_cracker,
    	rice_ball: rice_ball,
    	rice: rice,
    	curry: curry,
    	ramen: ramen,
    	spaghetti: spaghetti,
    	bread: bread,
    	fries: fries,
    	sweet_potato: sweet_potato,
    	dango: dango,
    	oden: oden,
    	sushi: sushi,
    	fried_shrimp: fried_shrimp,
    	fish_cake: fish_cake,
    	icecream: icecream,
    	shaved_ice: shaved_ice,
    	ice_cream: ice_cream,
    	doughnut: doughnut,
    	cookie: cookie,
    	chocolate_bar: chocolate_bar,
    	candy: candy,
    	lollipop: lollipop,
    	custard: custard,
    	honey_pot: honey_pot,
    	cake: cake,
    	bento: bento,
    	stew: stew,
    	fried_egg: fried_egg,
    	cooking: cooking,
    	fork_and_knife: fork_and_knife,
    	tea: tea,
    	sake: sake,
    	wine_glass: wine_glass,
    	cocktail: cocktail,
    	tropical_drink: tropical_drink,
    	beer: beer,
    	beers: beers,
    	baby_bottle: baby_bottle,
    	knife_fork_plate: knife_fork_plate,
    	champagne: champagne,
    	popcorn: popcorn,
    	ribbon: ribbon,
    	gift: gift,
    	birthday: birthday,
    	jack_o_lantern: jack_o_lantern,
    	christmas_tree: christmas_tree,
    	santa: santa,
    	fireworks: fireworks,
    	sparkler: sparkler,
    	balloon: balloon,
    	tada: tada,
    	confetti_ball: confetti_ball,
    	tanabata_tree: tanabata_tree,
    	crossed_flags: crossed_flags,
    	bamboo: bamboo,
    	dolls: dolls,
    	flags: flags,
    	wind_chime: wind_chime,
    	rice_scene: rice_scene,
    	school_satchel: school_satchel,
    	mortar_board: mortar_board,
    	medal: medal,
    	reminder_ribbon: reminder_ribbon,
    	studio_microphone: studio_microphone,
    	level_slider: level_slider,
    	control_knobs: control_knobs,
    	film_frames: film_frames,
    	admission_tickets: admission_tickets,
    	carousel_horse: carousel_horse,
    	ferris_wheel: ferris_wheel,
    	roller_coaster: roller_coaster,
    	fishing_pole_and_fish: fishing_pole_and_fish,
    	microphone: microphone,
    	movie_camera: movie_camera,
    	cinema: cinema,
    	headphones: headphones,
    	art: art,
    	tophat: tophat,
    	circus_tent: circus_tent,
    	ticket: ticket,
    	clapper: clapper,
    	performing_arts: performing_arts,
    	video_game: video_game,
    	dart: dart,
    	slot_machine: slot_machine,
    	"8ball": "🎱",
    	game_die: game_die,
    	bowling: bowling,
    	flower_playing_cards: flower_playing_cards,
    	musical_note: musical_note,
    	notes: notes,
    	saxophone: saxophone,
    	guitar: guitar,
    	musical_keyboard: musical_keyboard,
    	trumpet: trumpet,
    	violin: violin,
    	musical_score: musical_score,
    	running_shirt_with_sash: running_shirt_with_sash,
    	tennis: tennis,
    	ski: ski,
    	basketball: basketball,
    	checkered_flag: checkered_flag,
    	snowboarder: snowboarder,
    	"woman-running": "🏃‍♀️",
    	"man-running": "🏃‍♂️",
    	runner: runner,
    	running: running,
    	"woman-surfing": "🏄‍♀️",
    	"man-surfing": "🏄‍♂️",
    	surfer: surfer,
    	sports_medal: sports_medal,
    	trophy: trophy,
    	horse_racing: horse_racing,
    	football: football,
    	rugby_football: rugby_football,
    	"woman-swimming": "🏊‍♀️",
    	"man-swimming": "🏊‍♂️",
    	swimmer: swimmer,
    	"woman-lifting-weights": "🏋️‍♀️",
    	"man-lifting-weights": "🏋️‍♂️",
    	weight_lifter: weight_lifter,
    	"woman-golfing": "🏌️‍♀️",
    	"man-golfing": "🏌️‍♂️",
    	golfer: golfer,
    	racing_motorcycle: racing_motorcycle,
    	racing_car: racing_car,
    	cricket_bat_and_ball: cricket_bat_and_ball,
    	volleyball: volleyball,
    	field_hockey_stick_and_ball: field_hockey_stick_and_ball,
    	ice_hockey_stick_and_puck: ice_hockey_stick_and_puck,
    	table_tennis_paddle_and_ball: table_tennis_paddle_and_ball,
    	snow_capped_mountain: snow_capped_mountain,
    	camping: camping,
    	beach_with_umbrella: beach_with_umbrella,
    	building_construction: building_construction,
    	house_buildings: house_buildings,
    	cityscape: cityscape,
    	derelict_house_building: derelict_house_building,
    	classical_building: classical_building,
    	desert: desert,
    	desert_island: desert_island,
    	national_park: national_park,
    	stadium: stadium,
    	house: house,
    	house_with_garden: house_with_garden,
    	office: office,
    	post_office: post_office,
    	european_post_office: european_post_office,
    	hospital: hospital,
    	bank: bank,
    	atm: atm,
    	hotel: hotel,
    	love_hotel: love_hotel,
    	convenience_store: convenience_store,
    	school: school,
    	department_store: department_store,
    	factory: factory,
    	izakaya_lantern: izakaya_lantern,
    	lantern: lantern,
    	japanese_castle: japanese_castle,
    	european_castle: european_castle,
    	"rainbow-flag": "🏳️‍🌈",
    	waving_white_flag: waving_white_flag,
    	"flag-england": "🏴󠁧󠁢󠁥󠁮󠁧󠁿",
    	"flag-scotland": "🏴󠁧󠁢󠁳󠁣󠁴󠁿",
    	"flag-wales": "🏴󠁧󠁢󠁷󠁬󠁳󠁿",
    	waving_black_flag: waving_black_flag,
    	rosette: rosette,
    	label: label,
    	badminton_racquet_and_shuttlecock: badminton_racquet_and_shuttlecock,
    	bow_and_arrow: bow_and_arrow,
    	amphora: amphora,
    	rat: rat,
    	mouse2: mouse2,
    	ox: ox,
    	water_buffalo: water_buffalo,
    	cow2: cow2,
    	tiger2: tiger2,
    	leopard: leopard,
    	rabbit2: rabbit2,
    	cat2: cat2,
    	dragon: dragon,
    	crocodile: crocodile,
    	whale2: whale2,
    	snail: snail,
    	snake: snake,
    	racehorse: racehorse,
    	ram: ram,
    	goat: goat,
    	sheep: sheep,
    	monkey: monkey,
    	rooster: rooster,
    	chicken: chicken,
    	dog2: dog2,
    	pig2: pig2,
    	boar: boar,
    	elephant: elephant,
    	octopus: octopus,
    	shell: shell,
    	bug: bug,
    	ant: ant,
    	bee: bee,
    	honeybee: honeybee,
    	beetle: beetle,
    	fish: fish,
    	tropical_fish: tropical_fish,
    	blowfish: blowfish,
    	turtle: turtle,
    	hatching_chick: hatching_chick,
    	baby_chick: baby_chick,
    	hatched_chick: hatched_chick,
    	bird: bird,
    	penguin: penguin,
    	koala: koala,
    	poodle: poodle,
    	dromedary_camel: dromedary_camel,
    	camel: camel,
    	dolphin: dolphin,
    	flipper: flipper,
    	mouse: mouse,
    	cow: cow,
    	tiger: tiger,
    	rabbit: rabbit,
    	cat: cat,
    	dragon_face: dragon_face,
    	whale: whale,
    	horse: horse,
    	monkey_face: monkey_face,
    	dog: dog,
    	pig: pig,
    	frog: frog,
    	hamster: hamster,
    	wolf: wolf,
    	bear: bear,
    	panda_face: panda_face,
    	pig_nose: pig_nose,
    	feet: feet,
    	paw_prints: paw_prints,
    	chipmunk: chipmunk,
    	eyes: eyes,
    	"eye-in-speech-bubble": "👁️‍🗨️",
    	eye: eye,
    	ear: ear,
    	nose: nose,
    	lips: lips,
    	tongue: tongue,
    	point_up_2: point_up_2,
    	point_down: point_down,
    	point_left: point_left,
    	point_right: point_right,
    	facepunch: facepunch,
    	punch: punch,
    	wave: wave,
    	ok_hand: ok_hand,
    	"+1": "👍",
    	thumbsup: thumbsup,
    	"-1": "👎",
    	thumbsdown: thumbsdown,
    	clap: clap,
    	open_hands: open_hands,
    	crown: crown,
    	womans_hat: womans_hat,
    	eyeglasses: eyeglasses,
    	necktie: necktie,
    	shirt: shirt,
    	tshirt: tshirt,
    	jeans: jeans,
    	dress: dress,
    	kimono: kimono,
    	bikini: bikini,
    	womans_clothes: womans_clothes,
    	purse: purse,
    	handbag: handbag,
    	pouch: pouch,
    	mans_shoe: mans_shoe,
    	shoe: shoe,
    	athletic_shoe: athletic_shoe,
    	high_heel: high_heel,
    	sandal: sandal,
    	boot: boot,
    	footprints: footprints,
    	bust_in_silhouette: bust_in_silhouette,
    	busts_in_silhouette: busts_in_silhouette,
    	boy: boy,
    	girl: girl,
    	"male-farmer": "👨‍🌾",
    	"male-cook": "👨‍🍳",
    	"male-student": "👨‍🎓",
    	"male-singer": "👨‍🎤",
    	"male-artist": "👨‍🎨",
    	"male-teacher": "👨‍🏫",
    	"male-factory-worker": "👨‍🏭",
    	"man-boy-boy": "👨‍👦‍👦",
    	"man-boy": "👨‍👦",
    	"man-girl-boy": "👨‍👧‍👦",
    	"man-girl-girl": "👨‍👧‍👧",
    	"man-girl": "👨‍👧",
    	"man-man-boy": "👨‍👨‍👦",
    	"man-man-boy-boy": "👨‍👨‍👦‍👦",
    	"man-man-girl": "👨‍👨‍👧",
    	"man-man-girl-boy": "👨‍👨‍👧‍👦",
    	"man-man-girl-girl": "👨‍👨‍👧‍👧",
    	"man-woman-boy": "👨‍👩‍👦",
    	family: family,
    	"man-woman-boy-boy": "👨‍👩‍👦‍👦",
    	"man-woman-girl": "👨‍👩‍👧",
    	"man-woman-girl-boy": "👨‍👩‍👧‍👦",
    	"man-woman-girl-girl": "👨‍👩‍👧‍👧",
    	"male-technologist": "👨‍💻",
    	"male-office-worker": "👨‍💼",
    	"male-mechanic": "👨‍🔧",
    	"male-scientist": "👨‍🔬",
    	"male-astronaut": "👨‍🚀",
    	"male-firefighter": "👨‍🚒",
    	"male-doctor": "👨‍⚕️",
    	"male-judge": "👨‍⚖️",
    	"male-pilot": "👨‍✈️",
    	"man-heart-man": "👨‍❤️‍👨",
    	"man-kiss-man": "👨‍❤️‍💋‍👨",
    	man: man,
    	"female-farmer": "👩‍🌾",
    	"female-cook": "👩‍🍳",
    	"female-student": "👩‍🎓",
    	"female-singer": "👩‍🎤",
    	"female-artist": "👩‍🎨",
    	"female-teacher": "👩‍🏫",
    	"female-factory-worker": "👩‍🏭",
    	"woman-boy-boy": "👩‍👦‍👦",
    	"woman-boy": "👩‍👦",
    	"woman-girl-boy": "👩‍👧‍👦",
    	"woman-girl-girl": "👩‍👧‍👧",
    	"woman-girl": "👩‍👧",
    	"woman-woman-boy": "👩‍👩‍👦",
    	"woman-woman-boy-boy": "👩‍👩‍👦‍👦",
    	"woman-woman-girl": "👩‍👩‍👧",
    	"woman-woman-girl-boy": "👩‍👩‍👧‍👦",
    	"woman-woman-girl-girl": "👩‍👩‍👧‍👧",
    	"female-technologist": "👩‍💻",
    	"female-office-worker": "👩‍💼",
    	"female-mechanic": "👩‍🔧",
    	"female-scientist": "👩‍🔬",
    	"female-astronaut": "👩‍🚀",
    	"female-firefighter": "👩‍🚒",
    	"female-doctor": "👩‍⚕️",
    	"female-judge": "👩‍⚖️",
    	"female-pilot": "👩‍✈️",
    	"woman-heart-man": "👩‍❤️‍👨",
    	couple_with_heart: couple_with_heart,
    	"woman-heart-woman": "👩‍❤️‍👩",
    	"woman-kiss-man": "👩‍❤️‍💋‍👨",
    	couplekiss: couplekiss,
    	"woman-kiss-woman": "👩‍❤️‍💋‍👩",
    	woman: woman,
    	couple: couple,
    	man_and_woman_holding_hands: man_and_woman_holding_hands,
    	two_men_holding_hands: two_men_holding_hands,
    	two_women_holding_hands: two_women_holding_hands,
    	"female-police-officer": "👮‍♀️",
    	"male-police-officer": "👮‍♂️",
    	cop: cop,
    	"woman-with-bunny-ears-partying": "👯‍♀️",
    	dancers: dancers,
    	"man-with-bunny-ears-partying": "👯‍♂️",
    	bride_with_veil: bride_with_veil,
    	"blond-haired-woman": "👱‍♀️",
    	"blond-haired-man": "👱‍♂️",
    	person_with_blond_hair: person_with_blond_hair,
    	man_with_gua_pi_mao: man_with_gua_pi_mao,
    	"woman-wearing-turban": "👳‍♀️",
    	"man-wearing-turban": "👳‍♂️",
    	man_with_turban: man_with_turban,
    	older_man: older_man,
    	older_woman: older_woman,
    	baby: baby,
    	"female-construction-worker": "👷‍♀️",
    	"male-construction-worker": "👷‍♂️",
    	construction_worker: construction_worker,
    	princess: princess,
    	japanese_ogre: japanese_ogre,
    	japanese_goblin: japanese_goblin,
    	ghost: ghost,
    	angel: angel,
    	alien: alien,
    	space_invader: space_invader,
    	imp: imp,
    	skull: skull,
    	"woman-tipping-hand": "💁‍♀️",
    	information_desk_person: information_desk_person,
    	"man-tipping-hand": "💁‍♂️",
    	"female-guard": "💂‍♀️",
    	"male-guard": "💂‍♂️",
    	guardsman: guardsman,
    	dancer: dancer,
    	lipstick: lipstick,
    	nail_care: nail_care,
    	"woman-getting-massage": "💆‍♀️",
    	massage: massage,
    	"man-getting-massage": "💆‍♂️",
    	"woman-getting-haircut": "💇‍♀️",
    	haircut: haircut,
    	"man-getting-haircut": "💇‍♂️",
    	barber: barber,
    	syringe: syringe,
    	pill: pill,
    	kiss: kiss,
    	love_letter: love_letter,
    	ring: ring,
    	gem: gem,
    	bouquet: bouquet,
    	wedding: wedding,
    	heartbeat: heartbeat,
    	broken_heart: broken_heart,
    	two_hearts: two_hearts,
    	sparkling_heart: sparkling_heart,
    	heartpulse: heartpulse,
    	cupid: cupid,
    	blue_heart: blue_heart,
    	green_heart: green_heart,
    	yellow_heart: yellow_heart,
    	purple_heart: purple_heart,
    	gift_heart: gift_heart,
    	revolving_hearts: revolving_hearts,
    	heart_decoration: heart_decoration,
    	diamond_shape_with_a_dot_inside: diamond_shape_with_a_dot_inside,
    	bulb: bulb,
    	anger: anger,
    	bomb: bomb,
    	zzz: zzz,
    	boom: boom,
    	collision: collision,
    	sweat_drops: sweat_drops,
    	droplet: droplet,
    	dash: dash,
    	poop: poop,
    	muscle: muscle,
    	dizzy: dizzy,
    	speech_balloon: speech_balloon,
    	thought_balloon: thought_balloon,
    	white_flower: white_flower,
    	moneybag: moneybag,
    	currency_exchange: currency_exchange,
    	heavy_dollar_sign: heavy_dollar_sign,
    	credit_card: credit_card,
    	yen: yen,
    	dollar: dollar,
    	euro: euro,
    	pound: pound,
    	money_with_wings: money_with_wings,
    	chart: chart,
    	seat: seat,
    	computer: computer,
    	briefcase: briefcase,
    	minidisc: minidisc,
    	floppy_disk: floppy_disk,
    	cd: cd,
    	dvd: dvd,
    	file_folder: file_folder,
    	open_file_folder: open_file_folder,
    	page_with_curl: page_with_curl,
    	page_facing_up: page_facing_up,
    	date: date,
    	calendar: calendar,
    	card_index: card_index,
    	chart_with_upwards_trend: chart_with_upwards_trend,
    	chart_with_downwards_trend: chart_with_downwards_trend,
    	bar_chart: bar_chart,
    	clipboard: clipboard,
    	pushpin: pushpin,
    	round_pushpin: round_pushpin,
    	paperclip: paperclip,
    	straight_ruler: straight_ruler,
    	triangular_ruler: triangular_ruler,
    	bookmark_tabs: bookmark_tabs,
    	ledger: ledger,
    	notebook: notebook,
    	notebook_with_decorative_cover: notebook_with_decorative_cover,
    	closed_book: closed_book,
    	book: book,
    	open_book: open_book,
    	green_book: green_book,
    	blue_book: blue_book,
    	orange_book: orange_book,
    	books: books,
    	name_badge: name_badge,
    	scroll: scroll,
    	memo: memo,
    	pencil: pencil,
    	telephone_receiver: telephone_receiver,
    	pager: pager,
    	fax: fax,
    	satellite_antenna: satellite_antenna,
    	loudspeaker: loudspeaker,
    	mega: mega,
    	outbox_tray: outbox_tray,
    	inbox_tray: inbox_tray,
    	"package": "📦",
    	"e-mail": "📧",
    	incoming_envelope: incoming_envelope,
    	envelope_with_arrow: envelope_with_arrow,
    	mailbox_closed: mailbox_closed,
    	mailbox: mailbox,
    	mailbox_with_mail: mailbox_with_mail,
    	mailbox_with_no_mail: mailbox_with_no_mail,
    	postbox: postbox,
    	postal_horn: postal_horn,
    	newspaper: newspaper,
    	iphone: iphone,
    	calling: calling,
    	vibration_mode: vibration_mode,
    	mobile_phone_off: mobile_phone_off,
    	no_mobile_phones: no_mobile_phones,
    	signal_strength: signal_strength,
    	camera: camera,
    	camera_with_flash: camera_with_flash,
    	video_camera: video_camera,
    	tv: tv,
    	radio: radio,
    	vhs: vhs,
    	film_projector: film_projector,
    	prayer_beads: prayer_beads,
    	twisted_rightwards_arrows: twisted_rightwards_arrows,
    	repeat: repeat,
    	repeat_one: repeat_one,
    	arrows_clockwise: arrows_clockwise,
    	arrows_counterclockwise: arrows_counterclockwise,
    	low_brightness: low_brightness,
    	high_brightness: high_brightness,
    	mute: mute,
    	speaker: speaker,
    	sound: sound,
    	loud_sound: loud_sound,
    	battery: battery,
    	electric_plug: electric_plug,
    	mag: mag,
    	mag_right: mag_right,
    	lock_with_ink_pen: lock_with_ink_pen,
    	closed_lock_with_key: closed_lock_with_key,
    	key: key,
    	lock: lock,
    	unlock: unlock,
    	bell: bell,
    	no_bell: no_bell,
    	bookmark: bookmark,
    	link: link,
    	radio_button: radio_button,
    	back: back,
    	end: end,
    	on: on,
    	soon: soon,
    	underage: underage,
    	keycap_ten: keycap_ten,
    	capital_abcd: capital_abcd,
    	abcd: abcd,
    	symbols: symbols,
    	abc: abc,
    	fire: fire,
    	flashlight: flashlight,
    	wrench: wrench,
    	hammer: hammer,
    	nut_and_bolt: nut_and_bolt,
    	hocho: hocho,
    	knife: knife,
    	gun: gun,
    	microscope: microscope,
    	telescope: telescope,
    	crystal_ball: crystal_ball,
    	six_pointed_star: six_pointed_star,
    	beginner: beginner,
    	trident: trident,
    	black_square_button: black_square_button,
    	white_square_button: white_square_button,
    	red_circle: red_circle,
    	large_blue_circle: large_blue_circle,
    	large_orange_diamond: large_orange_diamond,
    	large_blue_diamond: large_blue_diamond,
    	small_orange_diamond: small_orange_diamond,
    	small_blue_diamond: small_blue_diamond,
    	small_red_triangle: small_red_triangle,
    	small_red_triangle_down: small_red_triangle_down,
    	arrow_up_small: arrow_up_small,
    	arrow_down_small: arrow_down_small,
    	om_symbol: om_symbol,
    	dove_of_peace: dove_of_peace,
    	kaaba: kaaba,
    	mosque: mosque,
    	synagogue: synagogue,
    	menorah_with_nine_branches: menorah_with_nine_branches,
    	candle: candle,
    	mantelpiece_clock: mantelpiece_clock,
    	hole: hole,
    	man_in_business_suit_levitating: man_in_business_suit_levitating,
    	"female-detective": "🕵️‍♀️",
    	"male-detective": "🕵️‍♂️",
    	sleuth_or_spy: sleuth_or_spy,
    	dark_sunglasses: dark_sunglasses,
    	spider: spider,
    	spider_web: spider_web,
    	joystick: joystick,
    	man_dancing: man_dancing,
    	linked_paperclips: linked_paperclips,
    	lower_left_ballpoint_pen: lower_left_ballpoint_pen,
    	lower_left_fountain_pen: lower_left_fountain_pen,
    	lower_left_paintbrush: lower_left_paintbrush,
    	lower_left_crayon: lower_left_crayon,
    	raised_hand_with_fingers_splayed: raised_hand_with_fingers_splayed,
    	middle_finger: middle_finger,
    	reversed_hand_with_middle_finger_extended: reversed_hand_with_middle_finger_extended,
    	"spock-hand": "🖖",
    	black_heart: black_heart,
    	desktop_computer: desktop_computer,
    	printer: printer,
    	three_button_mouse: three_button_mouse,
    	trackball: trackball,
    	frame_with_picture: frame_with_picture,
    	card_index_dividers: card_index_dividers,
    	card_file_box: card_file_box,
    	file_cabinet: file_cabinet,
    	wastebasket: wastebasket,
    	spiral_note_pad: spiral_note_pad,
    	spiral_calendar_pad: spiral_calendar_pad,
    	compression: compression,
    	old_key: old_key,
    	rolled_up_newspaper: rolled_up_newspaper,
    	dagger_knife: dagger_knife,
    	speaking_head_in_silhouette: speaking_head_in_silhouette,
    	left_speech_bubble: left_speech_bubble,
    	right_anger_bubble: right_anger_bubble,
    	ballot_box_with_ballot: ballot_box_with_ballot,
    	world_map: world_map,
    	mount_fuji: mount_fuji,
    	tokyo_tower: tokyo_tower,
    	statue_of_liberty: statue_of_liberty,
    	japan: japan,
    	moyai: moyai,
    	grinning: grinning,
    	grin: grin,
    	joy: joy,
    	smiley: smiley,
    	smile: smile,
    	sweat_smile: sweat_smile,
    	laughing: laughing,
    	satisfied: satisfied,
    	innocent: innocent,
    	smiling_imp: smiling_imp,
    	wink: wink,
    	blush: blush,
    	yum: yum,
    	relieved: relieved,
    	heart_eyes: heart_eyes,
    	sunglasses: sunglasses,
    	smirk: smirk,
    	neutral_face: neutral_face,
    	expressionless: expressionless,
    	unamused: unamused,
    	sweat: sweat,
    	pensive: pensive,
    	confused: confused,
    	confounded: confounded,
    	kissing: kissing,
    	kissing_heart: kissing_heart,
    	kissing_smiling_eyes: kissing_smiling_eyes,
    	kissing_closed_eyes: kissing_closed_eyes,
    	stuck_out_tongue: stuck_out_tongue,
    	stuck_out_tongue_winking_eye: stuck_out_tongue_winking_eye,
    	stuck_out_tongue_closed_eyes: stuck_out_tongue_closed_eyes,
    	disappointed: disappointed,
    	worried: worried,
    	angry: angry,
    	rage: rage,
    	cry: cry,
    	persevere: persevere,
    	triumph: triumph,
    	disappointed_relieved: disappointed_relieved,
    	frowning: frowning,
    	anguished: anguished,
    	fearful: fearful,
    	weary: weary,
    	sleepy: sleepy,
    	tired_face: tired_face,
    	grimacing: grimacing,
    	sob: sob,
    	open_mouth: open_mouth,
    	hushed: hushed,
    	cold_sweat: cold_sweat,
    	scream: scream,
    	astonished: astonished,
    	flushed: flushed,
    	sleeping: sleeping,
    	dizzy_face: dizzy_face,
    	no_mouth: no_mouth,
    	mask: mask,
    	smile_cat: smile_cat,
    	joy_cat: joy_cat,
    	smiley_cat: smiley_cat,
    	heart_eyes_cat: heart_eyes_cat,
    	smirk_cat: smirk_cat,
    	kissing_cat: kissing_cat,
    	pouting_cat: pouting_cat,
    	crying_cat_face: crying_cat_face,
    	scream_cat: scream_cat,
    	slightly_frowning_face: slightly_frowning_face,
    	slightly_smiling_face: slightly_smiling_face,
    	upside_down_face: upside_down_face,
    	face_with_rolling_eyes: face_with_rolling_eyes,
    	"woman-gesturing-no": "🙅‍♀️",
    	no_good: no_good,
    	"man-gesturing-no": "🙅‍♂️",
    	"woman-gesturing-ok": "🙆‍♀️",
    	ok_woman: ok_woman,
    	"man-gesturing-ok": "🙆‍♂️",
    	"woman-bowing": "🙇‍♀️",
    	"man-bowing": "🙇‍♂️",
    	bow: bow,
    	see_no_evil: see_no_evil,
    	hear_no_evil: hear_no_evil,
    	speak_no_evil: speak_no_evil,
    	"woman-raising-hand": "🙋‍♀️",
    	raising_hand: raising_hand,
    	"man-raising-hand": "🙋‍♂️",
    	raised_hands: raised_hands,
    	"woman-frowning": "🙍‍♀️",
    	person_frowning: person_frowning,
    	"man-frowning": "🙍‍♂️",
    	"woman-pouting": "🙎‍♀️",
    	person_with_pouting_face: person_with_pouting_face,
    	"man-pouting": "🙎‍♂️",
    	pray: pray,
    	rocket: rocket,
    	helicopter: helicopter,
    	steam_locomotive: steam_locomotive,
    	railway_car: railway_car,
    	bullettrain_side: bullettrain_side,
    	bullettrain_front: bullettrain_front,
    	train2: train2,
    	metro: metro,
    	light_rail: light_rail,
    	station: station,
    	tram: tram,
    	train: train,
    	bus: bus,
    	oncoming_bus: oncoming_bus,
    	trolleybus: trolleybus,
    	busstop: busstop,
    	minibus: minibus,
    	ambulance: ambulance,
    	fire_engine: fire_engine,
    	police_car: police_car,
    	oncoming_police_car: oncoming_police_car,
    	taxi: taxi,
    	oncoming_taxi: oncoming_taxi,
    	car: car,
    	red_car: red_car,
    	oncoming_automobile: oncoming_automobile,
    	blue_car: blue_car,
    	truck: truck,
    	articulated_lorry: articulated_lorry,
    	tractor: tractor,
    	monorail: monorail,
    	mountain_railway: mountain_railway,
    	suspension_railway: suspension_railway,
    	mountain_cableway: mountain_cableway,
    	aerial_tramway: aerial_tramway,
    	ship: ship,
    	"woman-rowing-boat": "🚣‍♀️",
    	"man-rowing-boat": "🚣‍♂️",
    	rowboat: rowboat,
    	speedboat: speedboat,
    	traffic_light: traffic_light,
    	vertical_traffic_light: vertical_traffic_light,
    	construction: construction,
    	rotating_light: rotating_light,
    	triangular_flag_on_post: triangular_flag_on_post,
    	door: door,
    	no_entry_sign: no_entry_sign,
    	smoking: smoking,
    	no_smoking: no_smoking,
    	put_litter_in_its_place: put_litter_in_its_place,
    	do_not_litter: do_not_litter,
    	potable_water: potable_water,
    	"non-potable_water": "🚱",
    	bike: bike,
    	no_bicycles: no_bicycles,
    	"woman-biking": "🚴‍♀️",
    	"man-biking": "🚴‍♂️",
    	bicyclist: bicyclist,
    	"woman-mountain-biking": "🚵‍♀️",
    	"man-mountain-biking": "🚵‍♂️",
    	mountain_bicyclist: mountain_bicyclist,
    	"woman-walking": "🚶‍♀️",
    	"man-walking": "🚶‍♂️",
    	walking: walking,
    	no_pedestrians: no_pedestrians,
    	children_crossing: children_crossing,
    	mens: mens,
    	womens: womens,
    	restroom: restroom,
    	baby_symbol: baby_symbol,
    	toilet: toilet,
    	wc: wc,
    	shower: shower,
    	bath: bath,
    	bathtub: bathtub,
    	passport_control: passport_control,
    	customs: customs,
    	baggage_claim: baggage_claim,
    	left_luggage: left_luggage,
    	couch_and_lamp: couch_and_lamp,
    	sleeping_accommodation: sleeping_accommodation,
    	shopping_bags: shopping_bags,
    	bellhop_bell: bellhop_bell,
    	bed: bed,
    	place_of_worship: place_of_worship,
    	octagonal_sign: octagonal_sign,
    	shopping_trolley: shopping_trolley,
    	hammer_and_wrench: hammer_and_wrench,
    	shield: shield,
    	oil_drum: oil_drum,
    	motorway: motorway,
    	railway_track: railway_track,
    	motor_boat: motor_boat,
    	small_airplane: small_airplane,
    	airplane_departure: airplane_departure,
    	airplane_arriving: airplane_arriving,
    	satellite: satellite,
    	passenger_ship: passenger_ship,
    	scooter: scooter,
    	motor_scooter: motor_scooter,
    	canoe: canoe,
    	sled: sled,
    	flying_saucer: flying_saucer,
    	zipper_mouth_face: zipper_mouth_face,
    	money_mouth_face: money_mouth_face,
    	face_with_thermometer: face_with_thermometer,
    	nerd_face: nerd_face,
    	thinking_face: thinking_face,
    	face_with_head_bandage: face_with_head_bandage,
    	robot_face: robot_face,
    	hugging_face: hugging_face,
    	the_horns: the_horns,
    	sign_of_the_horns: sign_of_the_horns,
    	call_me_hand: call_me_hand,
    	raised_back_of_hand: raised_back_of_hand,
    	"left-facing_fist": "🤛",
    	"right-facing_fist": "🤜",
    	handshake: handshake,
    	crossed_fingers: crossed_fingers,
    	hand_with_index_and_middle_fingers_crossed: hand_with_index_and_middle_fingers_crossed,
    	i_love_you_hand_sign: i_love_you_hand_sign,
    	face_with_cowboy_hat: face_with_cowboy_hat,
    	clown_face: clown_face,
    	nauseated_face: nauseated_face,
    	rolling_on_the_floor_laughing: rolling_on_the_floor_laughing,
    	drooling_face: drooling_face,
    	lying_face: lying_face,
    	"woman-facepalming": "🤦‍♀️",
    	"man-facepalming": "🤦‍♂️",
    	face_palm: face_palm,
    	sneezing_face: sneezing_face,
    	face_with_raised_eyebrow: face_with_raised_eyebrow,
    	face_with_one_eyebrow_raised: face_with_one_eyebrow_raised,
    	"star-struck": "🤩",
    	grinning_face_with_star_eyes: grinning_face_with_star_eyes,
    	zany_face: zany_face,
    	grinning_face_with_one_large_and_one_small_eye: grinning_face_with_one_large_and_one_small_eye,
    	shushing_face: shushing_face,
    	face_with_finger_covering_closed_lips: face_with_finger_covering_closed_lips,
    	face_with_symbols_on_mouth: face_with_symbols_on_mouth,
    	serious_face_with_symbols_covering_mouth: serious_face_with_symbols_covering_mouth,
    	face_with_hand_over_mouth: face_with_hand_over_mouth,
    	smiling_face_with_smiling_eyes_and_hand_covering_mouth: smiling_face_with_smiling_eyes_and_hand_covering_mouth,
    	face_vomiting: face_vomiting,
    	face_with_open_mouth_vomiting: face_with_open_mouth_vomiting,
    	exploding_head: exploding_head,
    	shocked_face_with_exploding_head: shocked_face_with_exploding_head,
    	pregnant_woman: pregnant_woman,
    	"breast-feeding": "🤱",
    	palms_up_together: palms_up_together,
    	selfie: selfie,
    	prince: prince,
    	man_in_tuxedo: man_in_tuxedo,
    	mrs_claus: mrs_claus,
    	mother_christmas: mother_christmas,
    	"woman-shrugging": "🤷‍♀️",
    	"man-shrugging": "🤷‍♂️",
    	shrug: shrug,
    	"woman-cartwheeling": "🤸‍♀️",
    	"man-cartwheeling": "🤸‍♂️",
    	person_doing_cartwheel: person_doing_cartwheel,
    	"woman-juggling": "🤹‍♀️",
    	"man-juggling": "🤹‍♂️",
    	juggling: juggling,
    	fencer: fencer,
    	"woman-wrestling": "🤼‍♀️",
    	"man-wrestling": "🤼‍♂️",
    	wrestlers: wrestlers,
    	"woman-playing-water-polo": "🤽‍♀️",
    	"man-playing-water-polo": "🤽‍♂️",
    	water_polo: water_polo,
    	"woman-playing-handball": "🤾‍♀️",
    	"man-playing-handball": "🤾‍♂️",
    	handball: handball,
    	wilted_flower: wilted_flower,
    	drum_with_drumsticks: drum_with_drumsticks,
    	clinking_glasses: clinking_glasses,
    	tumbler_glass: tumbler_glass,
    	spoon: spoon,
    	goal_net: goal_net,
    	first_place_medal: first_place_medal,
    	second_place_medal: second_place_medal,
    	third_place_medal: third_place_medal,
    	boxing_glove: boxing_glove,
    	martial_arts_uniform: martial_arts_uniform,
    	curling_stone: curling_stone,
    	croissant: croissant,
    	avocado: avocado,
    	cucumber: cucumber,
    	bacon: bacon,
    	potato: potato,
    	carrot: carrot,
    	baguette_bread: baguette_bread,
    	green_salad: green_salad,
    	shallow_pan_of_food: shallow_pan_of_food,
    	stuffed_flatbread: stuffed_flatbread,
    	egg: egg,
    	glass_of_milk: glass_of_milk,
    	peanuts: peanuts,
    	kiwifruit: kiwifruit,
    	pancakes: pancakes,
    	dumpling: dumpling,
    	fortune_cookie: fortune_cookie,
    	takeout_box: takeout_box,
    	chopsticks: chopsticks,
    	bowl_with_spoon: bowl_with_spoon,
    	cup_with_straw: cup_with_straw,
    	coconut: coconut,
    	broccoli: broccoli,
    	pie: pie,
    	pretzel: pretzel,
    	cut_of_meat: cut_of_meat,
    	sandwich: sandwich,
    	canned_food: canned_food,
    	crab: crab,
    	lion_face: lion_face,
    	scorpion: scorpion,
    	turkey: turkey,
    	unicorn_face: unicorn_face,
    	eagle: eagle,
    	duck: duck,
    	bat: bat,
    	shark: shark,
    	owl: owl,
    	fox_face: fox_face,
    	butterfly: butterfly,
    	deer: deer,
    	gorilla: gorilla,
    	lizard: lizard,
    	rhinoceros: rhinoceros,
    	shrimp: shrimp,
    	squid: squid,
    	giraffe_face: giraffe_face,
    	zebra_face: zebra_face,
    	hedgehog: hedgehog,
    	sauropod: sauropod,
    	"t-rex": "🦖",
    	cricket: cricket,
    	cheese_wedge: cheese_wedge,
    	face_with_monocle: face_with_monocle,
    	adult: adult,
    	child: child,
    	older_adult: older_adult,
    	bearded_person: bearded_person,
    	person_with_headscarf: person_with_headscarf,
    	woman_in_steamy_room: woman_in_steamy_room,
    	man_in_steamy_room: man_in_steamy_room,
    	person_in_steamy_room: person_in_steamy_room,
    	woman_climbing: woman_climbing,
    	person_climbing: person_climbing,
    	man_climbing: man_climbing,
    	woman_in_lotus_position: woman_in_lotus_position,
    	person_in_lotus_position: person_in_lotus_position,
    	man_in_lotus_position: man_in_lotus_position,
    	female_mage: female_mage,
    	mage: mage,
    	male_mage: male_mage,
    	female_fairy: female_fairy,
    	fairy: fairy,
    	male_fairy: male_fairy,
    	female_vampire: female_vampire,
    	vampire: vampire,
    	male_vampire: male_vampire,
    	mermaid: mermaid,
    	merman: merman,
    	merperson: merperson,
    	female_elf: female_elf,
    	male_elf: male_elf,
    	elf: elf,
    	female_genie: female_genie,
    	male_genie: male_genie,
    	genie: genie,
    	female_zombie: female_zombie,
    	male_zombie: male_zombie,
    	zombie: zombie,
    	brain: brain,
    	orange_heart: orange_heart,
    	billed_cap: billed_cap,
    	scarf: scarf,
    	gloves: gloves,
    	coat: coat,
    	socks: socks,
    	bangbang: bangbang,
    	interrobang: interrobang,
    	tm: tm,
    	information_source: information_source,
    	left_right_arrow: left_right_arrow,
    	arrow_up_down: arrow_up_down,
    	arrow_upper_left: arrow_upper_left,
    	arrow_upper_right: arrow_upper_right,
    	arrow_lower_right: arrow_lower_right,
    	arrow_lower_left: arrow_lower_left,
    	leftwards_arrow_with_hook: leftwards_arrow_with_hook,
    	arrow_right_hook: arrow_right_hook,
    	watch: watch,
    	hourglass: hourglass,
    	keyboard: keyboard,
    	eject: eject,
    	fast_forward: fast_forward,
    	rewind: rewind,
    	arrow_double_up: arrow_double_up,
    	arrow_double_down: arrow_double_down,
    	black_right_pointing_double_triangle_with_vertical_bar: black_right_pointing_double_triangle_with_vertical_bar,
    	black_left_pointing_double_triangle_with_vertical_bar: black_left_pointing_double_triangle_with_vertical_bar,
    	black_right_pointing_triangle_with_double_vertical_bar: black_right_pointing_triangle_with_double_vertical_bar,
    	alarm_clock: alarm_clock,
    	stopwatch: stopwatch,
    	timer_clock: timer_clock,
    	hourglass_flowing_sand: hourglass_flowing_sand,
    	double_vertical_bar: double_vertical_bar,
    	black_square_for_stop: black_square_for_stop,
    	black_circle_for_record: black_circle_for_record,
    	m: m,
    	black_small_square: black_small_square,
    	white_small_square: white_small_square,
    	arrow_forward: arrow_forward,
    	arrow_backward: arrow_backward,
    	white_medium_square: white_medium_square,
    	black_medium_square: black_medium_square,
    	white_medium_small_square: white_medium_small_square,
    	black_medium_small_square: black_medium_small_square,
    	sunny: sunny,
    	cloud: cloud,
    	umbrella: umbrella,
    	snowman: snowman,
    	comet: comet,
    	phone: phone,
    	telephone: telephone,
    	ballot_box_with_check: ballot_box_with_check,
    	shamrock: shamrock,
    	point_up: point_up,
    	skull_and_crossbones: skull_and_crossbones,
    	radioactive_sign: radioactive_sign,
    	biohazard_sign: biohazard_sign,
    	orthodox_cross: orthodox_cross,
    	star_and_crescent: star_and_crescent,
    	peace_symbol: peace_symbol,
    	yin_yang: yin_yang,
    	wheel_of_dharma: wheel_of_dharma,
    	white_frowning_face: white_frowning_face,
    	relaxed: relaxed,
    	female_sign: female_sign,
    	male_sign: male_sign,
    	gemini: gemini,
    	cancer: cancer,
    	leo: leo,
    	virgo: virgo,
    	libra: libra,
    	scorpius: scorpius,
    	spades: spades,
    	clubs: clubs,
    	hearts: hearts,
    	diamonds: diamonds,
    	hotsprings: hotsprings,
    	recycle: recycle,
    	wheelchair: wheelchair,
    	hammer_and_pick: hammer_and_pick,
    	crossed_swords: crossed_swords,
    	medical_symbol: medical_symbol,
    	staff_of_aesculapius: staff_of_aesculapius,
    	scales: scales,
    	alembic: alembic,
    	gear: gear,
    	atom_symbol: atom_symbol,
    	fleur_de_lis: fleur_de_lis,
    	warning: warning,
    	zap: zap,
    	white_circle: white_circle,
    	black_circle: black_circle,
    	coffin: coffin,
    	funeral_urn: funeral_urn,
    	soccer: soccer,
    	baseball: baseball,
    	snowman_without_snow: snowman_without_snow,
    	partly_sunny: partly_sunny,
    	thunder_cloud_and_rain: thunder_cloud_and_rain,
    	ophiuchus: ophiuchus,
    	pick: pick,
    	helmet_with_white_cross: helmet_with_white_cross,
    	chains: chains,
    	no_entry: no_entry,
    	shinto_shrine: shinto_shrine,
    	church: church,
    	mountain: mountain,
    	umbrella_on_ground: umbrella_on_ground,
    	fountain: fountain,
    	golf: golf,
    	ferry: ferry,
    	boat: boat,
    	sailboat: sailboat,
    	skier: skier,
    	ice_skate: ice_skate,
    	"woman-bouncing-ball": "⛹️‍♀️",
    	"man-bouncing-ball": "⛹️‍♂️",
    	person_with_ball: person_with_ball,
    	tent: tent,
    	fuelpump: fuelpump,
    	scissors: scissors,
    	airplane: airplane,
    	email: email,
    	envelope: envelope,
    	fist: fist,
    	hand: hand,
    	raised_hand: raised_hand,
    	v: v,
    	writing_hand: writing_hand,
    	pencil2: pencil2,
    	black_nib: black_nib,
    	heavy_check_mark: heavy_check_mark,
    	heavy_multiplication_x: heavy_multiplication_x,
    	latin_cross: latin_cross,
    	star_of_david: star_of_david,
    	eight_spoked_asterisk: eight_spoked_asterisk,
    	eight_pointed_black_star: eight_pointed_black_star,
    	snowflake: snowflake,
    	sparkle: sparkle,
    	x: x,
    	negative_squared_cross_mark: negative_squared_cross_mark,
    	heavy_heart_exclamation_mark_ornament: heavy_heart_exclamation_mark_ornament,
    	heart: heart,
    	arrow_right: arrow_right,
    	curly_loop: curly_loop,
    	loop: loop,
    	arrow_heading_up: arrow_heading_up,
    	arrow_heading_down: arrow_heading_down,
    	arrow_left: arrow_left,
    	arrow_up: arrow_up,
    	arrow_down: arrow_down,
    	black_large_square: black_large_square,
    	white_large_square: white_large_square,
    	star: star,
    	o: o,
    	wavy_dash: wavy_dash,
    	part_alternation_mark: part_alternation_mark,
    	congratulations: congratulations,
    	secret: secret
    };

    var emoji$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        umbrella_with_rain_drops: umbrella_with_rain_drops,
        coffee: coffee,
        aries: aries,
        taurus: taurus,
        sagittarius: sagittarius,
        capricorn: capricorn,
        aquarius: aquarius,
        pisces: pisces,
        anchor: anchor,
        white_check_mark: white_check_mark,
        sparkles: sparkles,
        question: question,
        grey_question: grey_question,
        grey_exclamation: grey_exclamation,
        exclamation: exclamation,
        heavy_exclamation_mark: heavy_exclamation_mark,
        heavy_plus_sign: heavy_plus_sign,
        heavy_minus_sign: heavy_minus_sign,
        heavy_division_sign: heavy_division_sign,
        hash: hash,
        keycap_star: keycap_star,
        zero: zero,
        one: one,
        two: two,
        three: three,
        four: four,
        five: five,
        six: six,
        seven: seven,
        eight: eight,
        nine: nine,
        copyright: copyright,
        registered: registered,
        mahjong: mahjong,
        black_joker: black_joker,
        a: a,
        b: b,
        o2: o2,
        parking: parking,
        ab: ab,
        cl: cl,
        cool: cool,
        free: free,
        id: id,
        ng: ng,
        ok: ok,
        sos: sos,
        up: up,
        vs: vs,
        koko: koko,
        sa: sa,
        u7121: u7121,
        u6307: u6307,
        u7981: u7981,
        u7a7a: u7a7a,
        u5408: u5408,
        u6e80: u6e80,
        u6709: u6709,
        u6708: u6708,
        u7533: u7533,
        u5272: u5272,
        u55b6: u55b6,
        ideograph_advantage: ideograph_advantage,
        accept: accept,
        cyclone: cyclone,
        foggy: foggy,
        closed_umbrella: closed_umbrella,
        night_with_stars: night_with_stars,
        sunrise_over_mountains: sunrise_over_mountains,
        sunrise: sunrise,
        city_sunset: city_sunset,
        city_sunrise: city_sunrise,
        rainbow: rainbow,
        bridge_at_night: bridge_at_night,
        ocean: ocean,
        volcano: volcano,
        milky_way: milky_way,
        earth_africa: earth_africa,
        earth_americas: earth_americas,
        earth_asia: earth_asia,
        globe_with_meridians: globe_with_meridians,
        new_moon: new_moon,
        waxing_crescent_moon: waxing_crescent_moon,
        first_quarter_moon: first_quarter_moon,
        moon: moon,
        waxing_gibbous_moon: waxing_gibbous_moon,
        full_moon: full_moon,
        waning_gibbous_moon: waning_gibbous_moon,
        last_quarter_moon: last_quarter_moon,
        waning_crescent_moon: waning_crescent_moon,
        crescent_moon: crescent_moon,
        new_moon_with_face: new_moon_with_face,
        first_quarter_moon_with_face: first_quarter_moon_with_face,
        last_quarter_moon_with_face: last_quarter_moon_with_face,
        full_moon_with_face: full_moon_with_face,
        sun_with_face: sun_with_face,
        star2: star2,
        stars: stars,
        thermometer: thermometer,
        mostly_sunny: mostly_sunny,
        sun_small_cloud: sun_small_cloud,
        barely_sunny: barely_sunny,
        sun_behind_cloud: sun_behind_cloud,
        partly_sunny_rain: partly_sunny_rain,
        sun_behind_rain_cloud: sun_behind_rain_cloud,
        rain_cloud: rain_cloud,
        snow_cloud: snow_cloud,
        lightning: lightning,
        lightning_cloud: lightning_cloud,
        tornado: tornado,
        tornado_cloud: tornado_cloud,
        fog: fog,
        wind_blowing_face: wind_blowing_face,
        hotdog: hotdog,
        taco: taco,
        burrito: burrito,
        chestnut: chestnut,
        seedling: seedling,
        evergreen_tree: evergreen_tree,
        deciduous_tree: deciduous_tree,
        palm_tree: palm_tree,
        cactus: cactus,
        hot_pepper: hot_pepper,
        tulip: tulip,
        cherry_blossom: cherry_blossom,
        rose: rose,
        hibiscus: hibiscus,
        sunflower: sunflower,
        blossom: blossom,
        corn: corn,
        ear_of_rice: ear_of_rice,
        herb: herb,
        four_leaf_clover: four_leaf_clover,
        maple_leaf: maple_leaf,
        fallen_leaf: fallen_leaf,
        leaves: leaves,
        mushroom: mushroom,
        tomato: tomato,
        eggplant: eggplant,
        grapes: grapes,
        melon: melon,
        watermelon: watermelon,
        tangerine: tangerine,
        lemon: lemon,
        banana: banana,
        pineapple: pineapple,
        apple: apple,
        green_apple: green_apple,
        pear: pear,
        peach: peach,
        cherries: cherries,
        strawberry: strawberry,
        hamburger: hamburger,
        pizza: pizza,
        meat_on_bone: meat_on_bone,
        poultry_leg: poultry_leg,
        rice_cracker: rice_cracker,
        rice_ball: rice_ball,
        rice: rice,
        curry: curry,
        ramen: ramen,
        spaghetti: spaghetti,
        bread: bread,
        fries: fries,
        sweet_potato: sweet_potato,
        dango: dango,
        oden: oden,
        sushi: sushi,
        fried_shrimp: fried_shrimp,
        fish_cake: fish_cake,
        icecream: icecream,
        shaved_ice: shaved_ice,
        ice_cream: ice_cream,
        doughnut: doughnut,
        cookie: cookie,
        chocolate_bar: chocolate_bar,
        candy: candy,
        lollipop: lollipop,
        custard: custard,
        honey_pot: honey_pot,
        cake: cake,
        bento: bento,
        stew: stew,
        fried_egg: fried_egg,
        cooking: cooking,
        fork_and_knife: fork_and_knife,
        tea: tea,
        sake: sake,
        wine_glass: wine_glass,
        cocktail: cocktail,
        tropical_drink: tropical_drink,
        beer: beer,
        beers: beers,
        baby_bottle: baby_bottle,
        knife_fork_plate: knife_fork_plate,
        champagne: champagne,
        popcorn: popcorn,
        ribbon: ribbon,
        gift: gift,
        birthday: birthday,
        jack_o_lantern: jack_o_lantern,
        christmas_tree: christmas_tree,
        santa: santa,
        fireworks: fireworks,
        sparkler: sparkler,
        balloon: balloon,
        tada: tada,
        confetti_ball: confetti_ball,
        tanabata_tree: tanabata_tree,
        crossed_flags: crossed_flags,
        bamboo: bamboo,
        dolls: dolls,
        flags: flags,
        wind_chime: wind_chime,
        rice_scene: rice_scene,
        school_satchel: school_satchel,
        mortar_board: mortar_board,
        medal: medal,
        reminder_ribbon: reminder_ribbon,
        studio_microphone: studio_microphone,
        level_slider: level_slider,
        control_knobs: control_knobs,
        film_frames: film_frames,
        admission_tickets: admission_tickets,
        carousel_horse: carousel_horse,
        ferris_wheel: ferris_wheel,
        roller_coaster: roller_coaster,
        fishing_pole_and_fish: fishing_pole_and_fish,
        microphone: microphone,
        movie_camera: movie_camera,
        cinema: cinema,
        headphones: headphones,
        art: art,
        tophat: tophat,
        circus_tent: circus_tent,
        ticket: ticket,
        clapper: clapper,
        performing_arts: performing_arts,
        video_game: video_game,
        dart: dart,
        slot_machine: slot_machine,
        game_die: game_die,
        bowling: bowling,
        flower_playing_cards: flower_playing_cards,
        musical_note: musical_note,
        notes: notes,
        saxophone: saxophone,
        guitar: guitar,
        musical_keyboard: musical_keyboard,
        trumpet: trumpet,
        violin: violin,
        musical_score: musical_score,
        running_shirt_with_sash: running_shirt_with_sash,
        tennis: tennis,
        ski: ski,
        basketball: basketball,
        checkered_flag: checkered_flag,
        snowboarder: snowboarder,
        runner: runner,
        running: running,
        surfer: surfer,
        sports_medal: sports_medal,
        trophy: trophy,
        horse_racing: horse_racing,
        football: football,
        rugby_football: rugby_football,
        swimmer: swimmer,
        weight_lifter: weight_lifter,
        golfer: golfer,
        racing_motorcycle: racing_motorcycle,
        racing_car: racing_car,
        cricket_bat_and_ball: cricket_bat_and_ball,
        volleyball: volleyball,
        field_hockey_stick_and_ball: field_hockey_stick_and_ball,
        ice_hockey_stick_and_puck: ice_hockey_stick_and_puck,
        table_tennis_paddle_and_ball: table_tennis_paddle_and_ball,
        snow_capped_mountain: snow_capped_mountain,
        camping: camping,
        beach_with_umbrella: beach_with_umbrella,
        building_construction: building_construction,
        house_buildings: house_buildings,
        cityscape: cityscape,
        derelict_house_building: derelict_house_building,
        classical_building: classical_building,
        desert: desert,
        desert_island: desert_island,
        national_park: national_park,
        stadium: stadium,
        house: house,
        house_with_garden: house_with_garden,
        office: office,
        post_office: post_office,
        european_post_office: european_post_office,
        hospital: hospital,
        bank: bank,
        atm: atm,
        hotel: hotel,
        love_hotel: love_hotel,
        convenience_store: convenience_store,
        school: school,
        department_store: department_store,
        factory: factory,
        izakaya_lantern: izakaya_lantern,
        lantern: lantern,
        japanese_castle: japanese_castle,
        european_castle: european_castle,
        waving_white_flag: waving_white_flag,
        waving_black_flag: waving_black_flag,
        rosette: rosette,
        label: label,
        badminton_racquet_and_shuttlecock: badminton_racquet_and_shuttlecock,
        bow_and_arrow: bow_and_arrow,
        amphora: amphora,
        rat: rat,
        mouse2: mouse2,
        ox: ox,
        water_buffalo: water_buffalo,
        cow2: cow2,
        tiger2: tiger2,
        leopard: leopard,
        rabbit2: rabbit2,
        cat2: cat2,
        dragon: dragon,
        crocodile: crocodile,
        whale2: whale2,
        snail: snail,
        snake: snake,
        racehorse: racehorse,
        ram: ram,
        goat: goat,
        sheep: sheep,
        monkey: monkey,
        rooster: rooster,
        chicken: chicken,
        dog2: dog2,
        pig2: pig2,
        boar: boar,
        elephant: elephant,
        octopus: octopus,
        shell: shell,
        bug: bug,
        ant: ant,
        bee: bee,
        honeybee: honeybee,
        beetle: beetle,
        fish: fish,
        tropical_fish: tropical_fish,
        blowfish: blowfish,
        turtle: turtle,
        hatching_chick: hatching_chick,
        baby_chick: baby_chick,
        hatched_chick: hatched_chick,
        bird: bird,
        penguin: penguin,
        koala: koala,
        poodle: poodle,
        dromedary_camel: dromedary_camel,
        camel: camel,
        dolphin: dolphin,
        flipper: flipper,
        mouse: mouse,
        cow: cow,
        tiger: tiger,
        rabbit: rabbit,
        cat: cat,
        dragon_face: dragon_face,
        whale: whale,
        horse: horse,
        monkey_face: monkey_face,
        dog: dog,
        pig: pig,
        frog: frog,
        hamster: hamster,
        wolf: wolf,
        bear: bear,
        panda_face: panda_face,
        pig_nose: pig_nose,
        feet: feet,
        paw_prints: paw_prints,
        chipmunk: chipmunk,
        eyes: eyes,
        eye: eye,
        ear: ear,
        nose: nose,
        lips: lips,
        tongue: tongue,
        point_up_2: point_up_2,
        point_down: point_down,
        point_left: point_left,
        point_right: point_right,
        facepunch: facepunch,
        punch: punch,
        wave: wave,
        ok_hand: ok_hand,
        thumbsup: thumbsup,
        thumbsdown: thumbsdown,
        clap: clap,
        open_hands: open_hands,
        crown: crown,
        womans_hat: womans_hat,
        eyeglasses: eyeglasses,
        necktie: necktie,
        shirt: shirt,
        tshirt: tshirt,
        jeans: jeans,
        dress: dress,
        kimono: kimono,
        bikini: bikini,
        womans_clothes: womans_clothes,
        purse: purse,
        handbag: handbag,
        pouch: pouch,
        mans_shoe: mans_shoe,
        shoe: shoe,
        athletic_shoe: athletic_shoe,
        high_heel: high_heel,
        sandal: sandal,
        boot: boot,
        footprints: footprints,
        bust_in_silhouette: bust_in_silhouette,
        busts_in_silhouette: busts_in_silhouette,
        boy: boy,
        girl: girl,
        family: family,
        man: man,
        couple_with_heart: couple_with_heart,
        couplekiss: couplekiss,
        woman: woman,
        couple: couple,
        man_and_woman_holding_hands: man_and_woman_holding_hands,
        two_men_holding_hands: two_men_holding_hands,
        two_women_holding_hands: two_women_holding_hands,
        cop: cop,
        dancers: dancers,
        bride_with_veil: bride_with_veil,
        person_with_blond_hair: person_with_blond_hair,
        man_with_gua_pi_mao: man_with_gua_pi_mao,
        man_with_turban: man_with_turban,
        older_man: older_man,
        older_woman: older_woman,
        baby: baby,
        construction_worker: construction_worker,
        princess: princess,
        japanese_ogre: japanese_ogre,
        japanese_goblin: japanese_goblin,
        ghost: ghost,
        angel: angel,
        alien: alien,
        space_invader: space_invader,
        imp: imp,
        skull: skull,
        information_desk_person: information_desk_person,
        guardsman: guardsman,
        dancer: dancer,
        lipstick: lipstick,
        nail_care: nail_care,
        massage: massage,
        haircut: haircut,
        barber: barber,
        syringe: syringe,
        pill: pill,
        kiss: kiss,
        love_letter: love_letter,
        ring: ring,
        gem: gem,
        bouquet: bouquet,
        wedding: wedding,
        heartbeat: heartbeat,
        broken_heart: broken_heart,
        two_hearts: two_hearts,
        sparkling_heart: sparkling_heart,
        heartpulse: heartpulse,
        cupid: cupid,
        blue_heart: blue_heart,
        green_heart: green_heart,
        yellow_heart: yellow_heart,
        purple_heart: purple_heart,
        gift_heart: gift_heart,
        revolving_hearts: revolving_hearts,
        heart_decoration: heart_decoration,
        diamond_shape_with_a_dot_inside: diamond_shape_with_a_dot_inside,
        bulb: bulb,
        anger: anger,
        bomb: bomb,
        zzz: zzz,
        boom: boom,
        collision: collision,
        sweat_drops: sweat_drops,
        droplet: droplet,
        dash: dash,
        poop: poop,
        muscle: muscle,
        dizzy: dizzy,
        speech_balloon: speech_balloon,
        thought_balloon: thought_balloon,
        white_flower: white_flower,
        moneybag: moneybag,
        currency_exchange: currency_exchange,
        heavy_dollar_sign: heavy_dollar_sign,
        credit_card: credit_card,
        yen: yen,
        dollar: dollar,
        euro: euro,
        pound: pound,
        money_with_wings: money_with_wings,
        chart: chart,
        seat: seat,
        computer: computer,
        briefcase: briefcase,
        minidisc: minidisc,
        floppy_disk: floppy_disk,
        cd: cd,
        dvd: dvd,
        file_folder: file_folder,
        open_file_folder: open_file_folder,
        page_with_curl: page_with_curl,
        page_facing_up: page_facing_up,
        date: date,
        calendar: calendar,
        card_index: card_index,
        chart_with_upwards_trend: chart_with_upwards_trend,
        chart_with_downwards_trend: chart_with_downwards_trend,
        bar_chart: bar_chart,
        clipboard: clipboard,
        pushpin: pushpin,
        round_pushpin: round_pushpin,
        paperclip: paperclip,
        straight_ruler: straight_ruler,
        triangular_ruler: triangular_ruler,
        bookmark_tabs: bookmark_tabs,
        ledger: ledger,
        notebook: notebook,
        notebook_with_decorative_cover: notebook_with_decorative_cover,
        closed_book: closed_book,
        book: book,
        open_book: open_book,
        green_book: green_book,
        blue_book: blue_book,
        orange_book: orange_book,
        books: books,
        name_badge: name_badge,
        scroll: scroll,
        memo: memo,
        pencil: pencil,
        telephone_receiver: telephone_receiver,
        pager: pager,
        fax: fax,
        satellite_antenna: satellite_antenna,
        loudspeaker: loudspeaker,
        mega: mega,
        outbox_tray: outbox_tray,
        inbox_tray: inbox_tray,
        incoming_envelope: incoming_envelope,
        envelope_with_arrow: envelope_with_arrow,
        mailbox_closed: mailbox_closed,
        mailbox: mailbox,
        mailbox_with_mail: mailbox_with_mail,
        mailbox_with_no_mail: mailbox_with_no_mail,
        postbox: postbox,
        postal_horn: postal_horn,
        newspaper: newspaper,
        iphone: iphone,
        calling: calling,
        vibration_mode: vibration_mode,
        mobile_phone_off: mobile_phone_off,
        no_mobile_phones: no_mobile_phones,
        signal_strength: signal_strength,
        camera: camera,
        camera_with_flash: camera_with_flash,
        video_camera: video_camera,
        tv: tv,
        radio: radio,
        vhs: vhs,
        film_projector: film_projector,
        prayer_beads: prayer_beads,
        twisted_rightwards_arrows: twisted_rightwards_arrows,
        repeat: repeat,
        repeat_one: repeat_one,
        arrows_clockwise: arrows_clockwise,
        arrows_counterclockwise: arrows_counterclockwise,
        low_brightness: low_brightness,
        high_brightness: high_brightness,
        mute: mute,
        speaker: speaker,
        sound: sound,
        loud_sound: loud_sound,
        battery: battery,
        electric_plug: electric_plug,
        mag: mag,
        mag_right: mag_right,
        lock_with_ink_pen: lock_with_ink_pen,
        closed_lock_with_key: closed_lock_with_key,
        key: key,
        lock: lock,
        unlock: unlock,
        bell: bell,
        no_bell: no_bell,
        bookmark: bookmark,
        link: link,
        radio_button: radio_button,
        back: back,
        end: end,
        on: on,
        soon: soon,
        underage: underage,
        keycap_ten: keycap_ten,
        capital_abcd: capital_abcd,
        abcd: abcd,
        symbols: symbols,
        abc: abc,
        fire: fire,
        flashlight: flashlight,
        wrench: wrench,
        hammer: hammer,
        nut_and_bolt: nut_and_bolt,
        hocho: hocho,
        knife: knife,
        gun: gun,
        microscope: microscope,
        telescope: telescope,
        crystal_ball: crystal_ball,
        six_pointed_star: six_pointed_star,
        beginner: beginner,
        trident: trident,
        black_square_button: black_square_button,
        white_square_button: white_square_button,
        red_circle: red_circle,
        large_blue_circle: large_blue_circle,
        large_orange_diamond: large_orange_diamond,
        large_blue_diamond: large_blue_diamond,
        small_orange_diamond: small_orange_diamond,
        small_blue_diamond: small_blue_diamond,
        small_red_triangle: small_red_triangle,
        small_red_triangle_down: small_red_triangle_down,
        arrow_up_small: arrow_up_small,
        arrow_down_small: arrow_down_small,
        om_symbol: om_symbol,
        dove_of_peace: dove_of_peace,
        kaaba: kaaba,
        mosque: mosque,
        synagogue: synagogue,
        menorah_with_nine_branches: menorah_with_nine_branches,
        candle: candle,
        mantelpiece_clock: mantelpiece_clock,
        hole: hole,
        man_in_business_suit_levitating: man_in_business_suit_levitating,
        sleuth_or_spy: sleuth_or_spy,
        dark_sunglasses: dark_sunglasses,
        spider: spider,
        spider_web: spider_web,
        joystick: joystick,
        man_dancing: man_dancing,
        linked_paperclips: linked_paperclips,
        lower_left_ballpoint_pen: lower_left_ballpoint_pen,
        lower_left_fountain_pen: lower_left_fountain_pen,
        lower_left_paintbrush: lower_left_paintbrush,
        lower_left_crayon: lower_left_crayon,
        raised_hand_with_fingers_splayed: raised_hand_with_fingers_splayed,
        middle_finger: middle_finger,
        reversed_hand_with_middle_finger_extended: reversed_hand_with_middle_finger_extended,
        black_heart: black_heart,
        desktop_computer: desktop_computer,
        printer: printer,
        three_button_mouse: three_button_mouse,
        trackball: trackball,
        frame_with_picture: frame_with_picture,
        card_index_dividers: card_index_dividers,
        card_file_box: card_file_box,
        file_cabinet: file_cabinet,
        wastebasket: wastebasket,
        spiral_note_pad: spiral_note_pad,
        spiral_calendar_pad: spiral_calendar_pad,
        compression: compression,
        old_key: old_key,
        rolled_up_newspaper: rolled_up_newspaper,
        dagger_knife: dagger_knife,
        speaking_head_in_silhouette: speaking_head_in_silhouette,
        left_speech_bubble: left_speech_bubble,
        right_anger_bubble: right_anger_bubble,
        ballot_box_with_ballot: ballot_box_with_ballot,
        world_map: world_map,
        mount_fuji: mount_fuji,
        tokyo_tower: tokyo_tower,
        statue_of_liberty: statue_of_liberty,
        japan: japan,
        moyai: moyai,
        grinning: grinning,
        grin: grin,
        joy: joy,
        smiley: smiley,
        smile: smile,
        sweat_smile: sweat_smile,
        laughing: laughing,
        satisfied: satisfied,
        innocent: innocent,
        smiling_imp: smiling_imp,
        wink: wink,
        blush: blush,
        yum: yum,
        relieved: relieved,
        heart_eyes: heart_eyes,
        sunglasses: sunglasses,
        smirk: smirk,
        neutral_face: neutral_face,
        expressionless: expressionless,
        unamused: unamused,
        sweat: sweat,
        pensive: pensive,
        confused: confused,
        confounded: confounded,
        kissing: kissing,
        kissing_heart: kissing_heart,
        kissing_smiling_eyes: kissing_smiling_eyes,
        kissing_closed_eyes: kissing_closed_eyes,
        stuck_out_tongue: stuck_out_tongue,
        stuck_out_tongue_winking_eye: stuck_out_tongue_winking_eye,
        stuck_out_tongue_closed_eyes: stuck_out_tongue_closed_eyes,
        disappointed: disappointed,
        worried: worried,
        angry: angry,
        rage: rage,
        cry: cry,
        persevere: persevere,
        triumph: triumph,
        disappointed_relieved: disappointed_relieved,
        frowning: frowning,
        anguished: anguished,
        fearful: fearful,
        weary: weary,
        sleepy: sleepy,
        tired_face: tired_face,
        grimacing: grimacing,
        sob: sob,
        open_mouth: open_mouth,
        hushed: hushed,
        cold_sweat: cold_sweat,
        scream: scream,
        astonished: astonished,
        flushed: flushed,
        sleeping: sleeping,
        dizzy_face: dizzy_face,
        no_mouth: no_mouth,
        mask: mask,
        smile_cat: smile_cat,
        joy_cat: joy_cat,
        smiley_cat: smiley_cat,
        heart_eyes_cat: heart_eyes_cat,
        smirk_cat: smirk_cat,
        kissing_cat: kissing_cat,
        pouting_cat: pouting_cat,
        crying_cat_face: crying_cat_face,
        scream_cat: scream_cat,
        slightly_frowning_face: slightly_frowning_face,
        slightly_smiling_face: slightly_smiling_face,
        upside_down_face: upside_down_face,
        face_with_rolling_eyes: face_with_rolling_eyes,
        no_good: no_good,
        ok_woman: ok_woman,
        bow: bow,
        see_no_evil: see_no_evil,
        hear_no_evil: hear_no_evil,
        speak_no_evil: speak_no_evil,
        raising_hand: raising_hand,
        raised_hands: raised_hands,
        person_frowning: person_frowning,
        person_with_pouting_face: person_with_pouting_face,
        pray: pray,
        rocket: rocket,
        helicopter: helicopter,
        steam_locomotive: steam_locomotive,
        railway_car: railway_car,
        bullettrain_side: bullettrain_side,
        bullettrain_front: bullettrain_front,
        train2: train2,
        metro: metro,
        light_rail: light_rail,
        station: station,
        tram: tram,
        train: train,
        bus: bus,
        oncoming_bus: oncoming_bus,
        trolleybus: trolleybus,
        busstop: busstop,
        minibus: minibus,
        ambulance: ambulance,
        fire_engine: fire_engine,
        police_car: police_car,
        oncoming_police_car: oncoming_police_car,
        taxi: taxi,
        oncoming_taxi: oncoming_taxi,
        car: car,
        red_car: red_car,
        oncoming_automobile: oncoming_automobile,
        blue_car: blue_car,
        truck: truck,
        articulated_lorry: articulated_lorry,
        tractor: tractor,
        monorail: monorail,
        mountain_railway: mountain_railway,
        suspension_railway: suspension_railway,
        mountain_cableway: mountain_cableway,
        aerial_tramway: aerial_tramway,
        ship: ship,
        rowboat: rowboat,
        speedboat: speedboat,
        traffic_light: traffic_light,
        vertical_traffic_light: vertical_traffic_light,
        construction: construction,
        rotating_light: rotating_light,
        triangular_flag_on_post: triangular_flag_on_post,
        door: door,
        no_entry_sign: no_entry_sign,
        smoking: smoking,
        no_smoking: no_smoking,
        put_litter_in_its_place: put_litter_in_its_place,
        do_not_litter: do_not_litter,
        potable_water: potable_water,
        bike: bike,
        no_bicycles: no_bicycles,
        bicyclist: bicyclist,
        mountain_bicyclist: mountain_bicyclist,
        walking: walking,
        no_pedestrians: no_pedestrians,
        children_crossing: children_crossing,
        mens: mens,
        womens: womens,
        restroom: restroom,
        baby_symbol: baby_symbol,
        toilet: toilet,
        wc: wc,
        shower: shower,
        bath: bath,
        bathtub: bathtub,
        passport_control: passport_control,
        customs: customs,
        baggage_claim: baggage_claim,
        left_luggage: left_luggage,
        couch_and_lamp: couch_and_lamp,
        sleeping_accommodation: sleeping_accommodation,
        shopping_bags: shopping_bags,
        bellhop_bell: bellhop_bell,
        bed: bed,
        place_of_worship: place_of_worship,
        octagonal_sign: octagonal_sign,
        shopping_trolley: shopping_trolley,
        hammer_and_wrench: hammer_and_wrench,
        shield: shield,
        oil_drum: oil_drum,
        motorway: motorway,
        railway_track: railway_track,
        motor_boat: motor_boat,
        small_airplane: small_airplane,
        airplane_departure: airplane_departure,
        airplane_arriving: airplane_arriving,
        satellite: satellite,
        passenger_ship: passenger_ship,
        scooter: scooter,
        motor_scooter: motor_scooter,
        canoe: canoe,
        sled: sled,
        flying_saucer: flying_saucer,
        zipper_mouth_face: zipper_mouth_face,
        money_mouth_face: money_mouth_face,
        face_with_thermometer: face_with_thermometer,
        nerd_face: nerd_face,
        thinking_face: thinking_face,
        face_with_head_bandage: face_with_head_bandage,
        robot_face: robot_face,
        hugging_face: hugging_face,
        the_horns: the_horns,
        sign_of_the_horns: sign_of_the_horns,
        call_me_hand: call_me_hand,
        raised_back_of_hand: raised_back_of_hand,
        handshake: handshake,
        crossed_fingers: crossed_fingers,
        hand_with_index_and_middle_fingers_crossed: hand_with_index_and_middle_fingers_crossed,
        i_love_you_hand_sign: i_love_you_hand_sign,
        face_with_cowboy_hat: face_with_cowboy_hat,
        clown_face: clown_face,
        nauseated_face: nauseated_face,
        rolling_on_the_floor_laughing: rolling_on_the_floor_laughing,
        drooling_face: drooling_face,
        lying_face: lying_face,
        face_palm: face_palm,
        sneezing_face: sneezing_face,
        face_with_raised_eyebrow: face_with_raised_eyebrow,
        face_with_one_eyebrow_raised: face_with_one_eyebrow_raised,
        grinning_face_with_star_eyes: grinning_face_with_star_eyes,
        zany_face: zany_face,
        grinning_face_with_one_large_and_one_small_eye: grinning_face_with_one_large_and_one_small_eye,
        shushing_face: shushing_face,
        face_with_finger_covering_closed_lips: face_with_finger_covering_closed_lips,
        face_with_symbols_on_mouth: face_with_symbols_on_mouth,
        serious_face_with_symbols_covering_mouth: serious_face_with_symbols_covering_mouth,
        face_with_hand_over_mouth: face_with_hand_over_mouth,
        smiling_face_with_smiling_eyes_and_hand_covering_mouth: smiling_face_with_smiling_eyes_and_hand_covering_mouth,
        face_vomiting: face_vomiting,
        face_with_open_mouth_vomiting: face_with_open_mouth_vomiting,
        exploding_head: exploding_head,
        shocked_face_with_exploding_head: shocked_face_with_exploding_head,
        pregnant_woman: pregnant_woman,
        palms_up_together: palms_up_together,
        selfie: selfie,
        prince: prince,
        man_in_tuxedo: man_in_tuxedo,
        mrs_claus: mrs_claus,
        mother_christmas: mother_christmas,
        shrug: shrug,
        person_doing_cartwheel: person_doing_cartwheel,
        juggling: juggling,
        fencer: fencer,
        wrestlers: wrestlers,
        water_polo: water_polo,
        handball: handball,
        wilted_flower: wilted_flower,
        drum_with_drumsticks: drum_with_drumsticks,
        clinking_glasses: clinking_glasses,
        tumbler_glass: tumbler_glass,
        spoon: spoon,
        goal_net: goal_net,
        first_place_medal: first_place_medal,
        second_place_medal: second_place_medal,
        third_place_medal: third_place_medal,
        boxing_glove: boxing_glove,
        martial_arts_uniform: martial_arts_uniform,
        curling_stone: curling_stone,
        croissant: croissant,
        avocado: avocado,
        cucumber: cucumber,
        bacon: bacon,
        potato: potato,
        carrot: carrot,
        baguette_bread: baguette_bread,
        green_salad: green_salad,
        shallow_pan_of_food: shallow_pan_of_food,
        stuffed_flatbread: stuffed_flatbread,
        egg: egg,
        glass_of_milk: glass_of_milk,
        peanuts: peanuts,
        kiwifruit: kiwifruit,
        pancakes: pancakes,
        dumpling: dumpling,
        fortune_cookie: fortune_cookie,
        takeout_box: takeout_box,
        chopsticks: chopsticks,
        bowl_with_spoon: bowl_with_spoon,
        cup_with_straw: cup_with_straw,
        coconut: coconut,
        broccoli: broccoli,
        pie: pie,
        pretzel: pretzel,
        cut_of_meat: cut_of_meat,
        sandwich: sandwich,
        canned_food: canned_food,
        crab: crab,
        lion_face: lion_face,
        scorpion: scorpion,
        turkey: turkey,
        unicorn_face: unicorn_face,
        eagle: eagle,
        duck: duck,
        bat: bat,
        shark: shark,
        owl: owl,
        fox_face: fox_face,
        butterfly: butterfly,
        deer: deer,
        gorilla: gorilla,
        lizard: lizard,
        rhinoceros: rhinoceros,
        shrimp: shrimp,
        squid: squid,
        giraffe_face: giraffe_face,
        zebra_face: zebra_face,
        hedgehog: hedgehog,
        sauropod: sauropod,
        cricket: cricket,
        cheese_wedge: cheese_wedge,
        face_with_monocle: face_with_monocle,
        adult: adult,
        child: child,
        older_adult: older_adult,
        bearded_person: bearded_person,
        person_with_headscarf: person_with_headscarf,
        woman_in_steamy_room: woman_in_steamy_room,
        man_in_steamy_room: man_in_steamy_room,
        person_in_steamy_room: person_in_steamy_room,
        woman_climbing: woman_climbing,
        person_climbing: person_climbing,
        man_climbing: man_climbing,
        woman_in_lotus_position: woman_in_lotus_position,
        person_in_lotus_position: person_in_lotus_position,
        man_in_lotus_position: man_in_lotus_position,
        female_mage: female_mage,
        mage: mage,
        male_mage: male_mage,
        female_fairy: female_fairy,
        fairy: fairy,
        male_fairy: male_fairy,
        female_vampire: female_vampire,
        vampire: vampire,
        male_vampire: male_vampire,
        mermaid: mermaid,
        merman: merman,
        merperson: merperson,
        female_elf: female_elf,
        male_elf: male_elf,
        elf: elf,
        female_genie: female_genie,
        male_genie: male_genie,
        genie: genie,
        female_zombie: female_zombie,
        male_zombie: male_zombie,
        zombie: zombie,
        brain: brain,
        orange_heart: orange_heart,
        billed_cap: billed_cap,
        scarf: scarf,
        gloves: gloves,
        coat: coat,
        socks: socks,
        bangbang: bangbang,
        interrobang: interrobang,
        tm: tm,
        information_source: information_source,
        left_right_arrow: left_right_arrow,
        arrow_up_down: arrow_up_down,
        arrow_upper_left: arrow_upper_left,
        arrow_upper_right: arrow_upper_right,
        arrow_lower_right: arrow_lower_right,
        arrow_lower_left: arrow_lower_left,
        leftwards_arrow_with_hook: leftwards_arrow_with_hook,
        arrow_right_hook: arrow_right_hook,
        watch: watch,
        hourglass: hourglass,
        keyboard: keyboard,
        eject: eject,
        fast_forward: fast_forward,
        rewind: rewind,
        arrow_double_up: arrow_double_up,
        arrow_double_down: arrow_double_down,
        black_right_pointing_double_triangle_with_vertical_bar: black_right_pointing_double_triangle_with_vertical_bar,
        black_left_pointing_double_triangle_with_vertical_bar: black_left_pointing_double_triangle_with_vertical_bar,
        black_right_pointing_triangle_with_double_vertical_bar: black_right_pointing_triangle_with_double_vertical_bar,
        alarm_clock: alarm_clock,
        stopwatch: stopwatch,
        timer_clock: timer_clock,
        hourglass_flowing_sand: hourglass_flowing_sand,
        double_vertical_bar: double_vertical_bar,
        black_square_for_stop: black_square_for_stop,
        black_circle_for_record: black_circle_for_record,
        m: m,
        black_small_square: black_small_square,
        white_small_square: white_small_square,
        arrow_forward: arrow_forward,
        arrow_backward: arrow_backward,
        white_medium_square: white_medium_square,
        black_medium_square: black_medium_square,
        white_medium_small_square: white_medium_small_square,
        black_medium_small_square: black_medium_small_square,
        sunny: sunny,
        cloud: cloud,
        umbrella: umbrella,
        snowman: snowman,
        comet: comet,
        phone: phone,
        telephone: telephone,
        ballot_box_with_check: ballot_box_with_check,
        shamrock: shamrock,
        point_up: point_up,
        skull_and_crossbones: skull_and_crossbones,
        radioactive_sign: radioactive_sign,
        biohazard_sign: biohazard_sign,
        orthodox_cross: orthodox_cross,
        star_and_crescent: star_and_crescent,
        peace_symbol: peace_symbol,
        yin_yang: yin_yang,
        wheel_of_dharma: wheel_of_dharma,
        white_frowning_face: white_frowning_face,
        relaxed: relaxed,
        female_sign: female_sign,
        male_sign: male_sign,
        gemini: gemini,
        cancer: cancer,
        leo: leo,
        virgo: virgo,
        libra: libra,
        scorpius: scorpius,
        spades: spades,
        clubs: clubs,
        hearts: hearts,
        diamonds: diamonds,
        hotsprings: hotsprings,
        recycle: recycle,
        wheelchair: wheelchair,
        hammer_and_pick: hammer_and_pick,
        crossed_swords: crossed_swords,
        medical_symbol: medical_symbol,
        staff_of_aesculapius: staff_of_aesculapius,
        scales: scales,
        alembic: alembic,
        gear: gear,
        atom_symbol: atom_symbol,
        fleur_de_lis: fleur_de_lis,
        warning: warning,
        zap: zap,
        white_circle: white_circle,
        black_circle: black_circle,
        coffin: coffin,
        funeral_urn: funeral_urn,
        soccer: soccer,
        baseball: baseball,
        snowman_without_snow: snowman_without_snow,
        partly_sunny: partly_sunny,
        thunder_cloud_and_rain: thunder_cloud_and_rain,
        ophiuchus: ophiuchus,
        pick: pick,
        helmet_with_white_cross: helmet_with_white_cross,
        chains: chains,
        no_entry: no_entry,
        shinto_shrine: shinto_shrine,
        church: church,
        mountain: mountain,
        umbrella_on_ground: umbrella_on_ground,
        fountain: fountain,
        golf: golf,
        ferry: ferry,
        boat: boat,
        sailboat: sailboat,
        skier: skier,
        ice_skate: ice_skate,
        person_with_ball: person_with_ball,
        tent: tent,
        fuelpump: fuelpump,
        scissors: scissors,
        airplane: airplane,
        email: email,
        envelope: envelope,
        fist: fist,
        hand: hand,
        raised_hand: raised_hand,
        v: v,
        writing_hand: writing_hand,
        pencil2: pencil2,
        black_nib: black_nib,
        heavy_check_mark: heavy_check_mark,
        heavy_multiplication_x: heavy_multiplication_x,
        latin_cross: latin_cross,
        star_of_david: star_of_david,
        eight_spoked_asterisk: eight_spoked_asterisk,
        eight_pointed_black_star: eight_pointed_black_star,
        snowflake: snowflake,
        sparkle: sparkle,
        x: x,
        negative_squared_cross_mark: negative_squared_cross_mark,
        heavy_heart_exclamation_mark_ornament: heavy_heart_exclamation_mark_ornament,
        heart: heart,
        arrow_right: arrow_right,
        curly_loop: curly_loop,
        loop: loop,
        arrow_heading_up: arrow_heading_up,
        arrow_heading_down: arrow_heading_down,
        arrow_left: arrow_left,
        arrow_up: arrow_up,
        arrow_down: arrow_down,
        black_large_square: black_large_square,
        white_large_square: white_large_square,
        star: star,
        o: o,
        wavy_dash: wavy_dash,
        part_alternation_mark: part_alternation_mark,
        congratulations: congratulations,
        secret: secret,
        'default': emoji
    });

    var emojiByName = getCjsExportFromNamespace(emoji$1);

    /**
     * regex to parse emoji in a string - finds emoji, e.g. :coffee:
     */
    var emojiNameRegex = /:([a-zA-Z0-9_\-\+]+):/g;

    /**
     * regex to trim whitespace
     * use instead of String.prototype.trim() for IE8 support
     */
    var trimSpaceRegex = /^[\s\uFEFF\xA0]+|[\s\uFEFF\xA0]+$/g;

    /**
     * Removes colons on either side
     * of the string if present
     * @param  {string} str
     * @return {string}
     */
    function stripColons(str) {
      var colonIndex = str.indexOf(':');
      if (colonIndex > -1) {
        // :emoji: (http://www.emoji-cheat-sheet.com/)
        if (colonIndex === str.length - 1) {
          str = str.substring(0, colonIndex);
          return stripColons(str);
        } else {
          str = str.substr(colonIndex + 1);
          return stripColons(str);
        }
      }

      return str;
    }

    /**
     * Adds colons to either side
     * of the string
     * @param {string} str
     * @return {string}
     */
    function wrapColons(str) {
      return (typeof str === 'string' && str.length > 0) ? ':' + str + ':' : str;
    }

    /**
     * Ensure that the word is wrapped in colons
     * by only adding them, if they are not there.
     * @param {string} str
     * @return {string}
     */
    function ensureColons(str) {
      return (typeof str === 'string' && str[0] !== ':') ? wrapColons(str) : str;
    }

    // Remove the non-spacing-mark from the code, never send a stripped version
    // to the client, as it kills graphical emoticons.
    function stripNSB(code) {
      return code;
    }
    // Reversed hash table, where as emojiByName contains a { heart: '❤' }
    // dictionary emojiByCode contains { ❤: 'heart' }. The codes are normalized
    // to the text version.
    var emojiByCode = Object.keys(emojiByName).reduce(function (h, k) {
      h[stripNSB(emojiByName[k])] = k;
      return h;
    }, {});

    /**
     * Emoji namespace
     */
    var Emoji = {
      emoji: emojiByName,
    };

    /**
     * get emoji code from name
     * @param  {string} emoji
     * @return {string}
     */
    Emoji._get = function _get(emoji) {
      if (emojiByName.hasOwnProperty(emoji)) {
        return emojiByName[emoji];
      }

      return ensureColons(emoji);
    };

    /**
     * get emoji code from :emoji: string or name
     * @param  {string} emoji
     * @return {string}
     */
    Emoji.get = function get(emoji) {
      emoji = stripColons(emoji);

      return Emoji._get(emoji);
    };

    /**
     * find the emoji by either code or name
     * @param {string} nameOrCode The emoji to find, either `coffee`, `:coffee:` or `☕`;
     * @return {object}
     */
    Emoji.find = function find(nameOrCode) {
      return Emoji.findByName(nameOrCode) || Emoji.findByCode(nameOrCode);
    };

    /**
     * find the emoji by name
     * @param {string} name The emoji to find either `coffee` or `:coffee:`;
     * @return {object}
     */
    Emoji.findByName = function findByName(name) {
      var stripped = stripColons(name);
      var emoji = emojiByName[stripped];

      return emoji ? ({ emoji: emoji, key: stripped }) : undefined;
    };

    /**
     * find the emoji by code (emoji)
     * @param {string} code The emoji to find; for example `☕` or `☔`
     * @return {object}
     */
    Emoji.findByCode = function findByCode(code) {
      var stripped = stripNSB(code);
      var name = emojiByCode[stripped];

      // lookup emoji to ensure the Variant Form is returned
      return name ? ({ emoji: emojiByName[name], key: name }) : undefined;
    };


    /**
     * Check if an emoji is known by this library
     * @param {string} nameOrCode The emoji to validate, either `coffee`, `:coffee:` or `☕`;
     * @return {object}
     */
    Emoji.hasEmoji = function hasEmoji(nameOrCode) {
      return Emoji.hasEmojiByName(nameOrCode) || Emoji.hasEmojiByCode(nameOrCode);
    };

    /**
     * Check if an emoji with given name is known by this library
     * @param {string} name The emoji to validate either `coffee` or `:coffee:`;
     * @return {object}
     */
    Emoji.hasEmojiByName = function hasEmojiByName(name) {
      var result = Emoji.findByName(name);
      return !!result && result.key === stripColons(name);
    };

    /**
     * Check if a given emoji is known by this library
     * @param {string} code The emoji to validate; for example `☕` or `☔`
     * @return {object}
     */
    Emoji.hasEmojiByCode = function hasEmojiByCode(code) {
      var result = Emoji.findByCode(code);
      return !!result && stripNSB(result.emoji) === stripNSB(code);
    };

    /**
     * get emoji name from code
     * @param  {string} emoji
     * @param  {boolean} includeColons should the result include the ::
     * @return {string}
     */
    Emoji.which = function which(emoji_code, includeColons) {
      var code = stripNSB(emoji_code);
      var word = emojiByCode[code];

      return includeColons ? wrapColons(word) : word;
    };

    /**
     * emojify a string (replace :emoji: with an emoji)
     * @param  {string} str
     * @param  {function} on_missing (gets emoji name without :: and returns a proper emoji if no emoji was found)
     * @param  {function} format (wrap the returned emoji in a custom element)
     * @return {string}
     */
    Emoji.emojify = function emojify(str, on_missing, format) {
      if (!str) return '';

      return str.split(emojiNameRegex) // parse emoji via regex
        .map(function parseEmoji(s, i) {
          // every second element is an emoji, e.g. "test :fast_forward:" -> [ "test ", "fast_forward" ]
          if (i % 2 === 0) return s;
          var emoji = Emoji._get(s);
          var isMissing = emoji.indexOf(':') > -1;

          if (isMissing && typeof on_missing === 'function') {
            return on_missing(s);
          }

          if (!isMissing && typeof format === 'function') {
            return format(emoji, s);
          }

          return emoji;
        })
        .join('') // convert back to string
        ;
    };

    /**
     * return a random emoji
     * @return {string}
     */
    Emoji.random = function random() {
      var emojiKeys = Object.keys(emojiByName);
      var randomIndex = Math.floor(Math.random() * emojiKeys.length);
      var key = emojiKeys[randomIndex];
      var emoji = Emoji._get(key);
      return { key: key, emoji: emoji };
    };

    /**
     *  return an collection of potential emoji matches
     *  @param {string} str
     *  @return {Array.<Object>}
     */
    Emoji.search = function search(str) {
      var emojiKeys = Object.keys(emojiByName);
      var matcher = stripColons(str);
      var matchingKeys = emojiKeys.filter(function (key) {
        return key.toString().indexOf(matcher) === 0;
      });
      return matchingKeys.map(function (key) {
        return {
          key: key,
          emoji: Emoji._get(key),
        };
      });
    };

    /**
     * unemojify a string (replace emoji with :emoji:)
     * @param  {string} str
     * @return {string}
     */
    Emoji.unemojify = function unemojify(str) {
      if (!str) return '';
      var words = lodash_toarray(str);

      return words.map(function (word) {
        return Emoji.which(word, true) || word;
      }).join('');
    };

    /**
     * replace emojis with replacement value
     * @param {string} str
     * @param {function|string} the string or callback function to replace the emoji with
     * @param {boolean} should trailing whitespaces be cleaned? Defaults false
     * @return {string}
     */
    Emoji.replace = function replace(str, replacement, cleanSpaces) {
      if (!str) return '';

      var replace = typeof replacement === 'function' ? replacement : function () { return replacement; };
      var words = lodash_toarray(str);

      var replaced = words.map(function (word, idx) {
        var emoji = Emoji.findByCode(word);
        
        if (emoji && cleanSpaces && words[idx + 1] === ' ') {
          words[idx + 1] = '';
        }

        return emoji ? replace(emoji) : word;
      }).join('');

      return cleanSpaces ? replaced.replace(trimSpaceRegex, '') : replaced;
    };


    /**
     * remove all emojis from a string
     * @param {string} str
     * @return {string}
     */
    Emoji.strip = function strip(str) {
      return Emoji.replace(str, '', true);
    };

    var emoji$2 = Emoji;

    var nodeEmoji = emoji$2;

    var defaults = createCommonjsModule(function (module) {
    function getDefaults() {
      return {
        baseUrl: null,
        breaks: false,
        gfm: true,
        headerIds: true,
        headerPrefix: '',
        highlight: null,
        langPrefix: 'language-',
        mangle: true,
        pedantic: false,
        renderer: null,
        sanitize: false,
        sanitizer: null,
        silent: false,
        smartLists: false,
        smartypants: false,
        tokenizer: null,
        walkTokens: null,
        xhtml: false
      };
    }

    function changeDefaults(newDefaults) {
      module.exports.defaults = newDefaults;
    }

    module.exports = {
      defaults: getDefaults(),
      getDefaults,
      changeDefaults
    };
    });
    var defaults_1 = defaults.defaults;
    var defaults_2 = defaults.getDefaults;
    var defaults_3 = defaults.changeDefaults;

    /**
     * Helpers
     */
    const escapeTest = /[&<>"']/;
    const escapeReplace = /[&<>"']/g;
    const escapeTestNoEncode = /[<>"']|&(?!#?\w+;)/;
    const escapeReplaceNoEncode = /[<>"']|&(?!#?\w+;)/g;
    const escapeReplacements = {
      '&': '&amp;',
      '<': '&lt;',
      '>': '&gt;',
      '"': '&quot;',
      "'": '&#39;'
    };
    const getEscapeReplacement = (ch) => escapeReplacements[ch];
    function escape(html, encode) {
      if (encode) {
        if (escapeTest.test(html)) {
          return html.replace(escapeReplace, getEscapeReplacement);
        }
      } else {
        if (escapeTestNoEncode.test(html)) {
          return html.replace(escapeReplaceNoEncode, getEscapeReplacement);
        }
      }

      return html;
    }

    const unescapeTest = /&(#(?:\d+)|(?:#x[0-9A-Fa-f]+)|(?:\w+));?/ig;

    function unescape$1(html) {
      // explicitly match decimal, hex, and named HTML entities
      return html.replace(unescapeTest, (_, n) => {
        n = n.toLowerCase();
        if (n === 'colon') return ':';
        if (n.charAt(0) === '#') {
          return n.charAt(1) === 'x'
            ? String.fromCharCode(parseInt(n.substring(2), 16))
            : String.fromCharCode(+n.substring(1));
        }
        return '';
      });
    }

    const caret = /(^|[^\[])\^/g;
    function edit(regex, opt) {
      regex = regex.source || regex;
      opt = opt || '';
      const obj = {
        replace: (name, val) => {
          val = val.source || val;
          val = val.replace(caret, '$1');
          regex = regex.replace(name, val);
          return obj;
        },
        getRegex: () => {
          return new RegExp(regex, opt);
        }
      };
      return obj;
    }

    const nonWordAndColonTest = /[^\w:]/g;
    const originIndependentUrl = /^$|^[a-z][a-z0-9+.-]*:|^[?#]/i;
    function cleanUrl(sanitize, base, href) {
      if (sanitize) {
        let prot;
        try {
          prot = decodeURIComponent(unescape$1(href))
            .replace(nonWordAndColonTest, '')
            .toLowerCase();
        } catch (e) {
          return null;
        }
        if (prot.indexOf('javascript:') === 0 || prot.indexOf('vbscript:') === 0 || prot.indexOf('data:') === 0) {
          return null;
        }
      }
      if (base && !originIndependentUrl.test(href)) {
        href = resolveUrl(base, href);
      }
      try {
        href = encodeURI(href).replace(/%25/g, '%');
      } catch (e) {
        return null;
      }
      return href;
    }

    const baseUrls = {};
    const justDomain = /^[^:]+:\/*[^/]*$/;
    const protocol = /^([^:]+:)[\s\S]*$/;
    const domain = /^([^:]+:\/*[^/]*)[\s\S]*$/;

    function resolveUrl(base, href) {
      if (!baseUrls[' ' + base]) {
        // we can ignore everything in base after the last slash of its path component,
        // but we might need to add _that_
        // https://tools.ietf.org/html/rfc3986#section-3
        if (justDomain.test(base)) {
          baseUrls[' ' + base] = base + '/';
        } else {
          baseUrls[' ' + base] = rtrim(base, '/', true);
        }
      }
      base = baseUrls[' ' + base];
      const relativeBase = base.indexOf(':') === -1;

      if (href.substring(0, 2) === '//') {
        if (relativeBase) {
          return href;
        }
        return base.replace(protocol, '$1') + href;
      } else if (href.charAt(0) === '/') {
        if (relativeBase) {
          return href;
        }
        return base.replace(domain, '$1') + href;
      } else {
        return base + href;
      }
    }

    const noopTest = { exec: function noopTest() {} };

    function merge(obj) {
      let i = 1,
        target,
        key;

      for (; i < arguments.length; i++) {
        target = arguments[i];
        for (key in target) {
          if (Object.prototype.hasOwnProperty.call(target, key)) {
            obj[key] = target[key];
          }
        }
      }

      return obj;
    }

    function splitCells(tableRow, count) {
      // ensure that every cell-delimiting pipe has a space
      // before it to distinguish it from an escaped pipe
      const row = tableRow.replace(/\|/g, (match, offset, str) => {
          let escaped = false,
            curr = offset;
          while (--curr >= 0 && str[curr] === '\\') escaped = !escaped;
          if (escaped) {
            // odd number of slashes means | is escaped
            // so we leave it alone
            return '|';
          } else {
            // add space before unescaped |
            return ' |';
          }
        }),
        cells = row.split(/ \|/);
      let i = 0;

      if (cells.length > count) {
        cells.splice(count);
      } else {
        while (cells.length < count) cells.push('');
      }

      for (; i < cells.length; i++) {
        // leading or trailing whitespace is ignored per the gfm spec
        cells[i] = cells[i].trim().replace(/\\\|/g, '|');
      }
      return cells;
    }

    // Remove trailing 'c's. Equivalent to str.replace(/c*$/, '').
    // /c*$/ is vulnerable to REDOS.
    // invert: Remove suffix of non-c chars instead. Default falsey.
    function rtrim(str, c, invert) {
      const l = str.length;
      if (l === 0) {
        return '';
      }

      // Length of suffix matching the invert condition.
      let suffLen = 0;

      // Step left until we fail to match the invert condition.
      while (suffLen < l) {
        const currChar = str.charAt(l - suffLen - 1);
        if (currChar === c && !invert) {
          suffLen++;
        } else if (currChar !== c && invert) {
          suffLen++;
        } else {
          break;
        }
      }

      return str.substr(0, l - suffLen);
    }

    function findClosingBracket(str, b) {
      if (str.indexOf(b[1]) === -1) {
        return -1;
      }
      const l = str.length;
      let level = 0,
        i = 0;
      for (; i < l; i++) {
        if (str[i] === '\\') {
          i++;
        } else if (str[i] === b[0]) {
          level++;
        } else if (str[i] === b[1]) {
          level--;
          if (level < 0) {
            return i;
          }
        }
      }
      return -1;
    }

    function checkSanitizeDeprecation(opt) {
      if (opt && opt.sanitize && !opt.silent) {
        console.warn('marked(): sanitize and sanitizer parameters are deprecated since version 0.7.0, should not be used and will be removed in the future. Read more here: https://marked.js.org/#/USING_ADVANCED.md#options');
      }
    }

    var helpers = {
      escape,
      unescape: unescape$1,
      edit,
      cleanUrl,
      resolveUrl,
      noopTest,
      merge,
      splitCells,
      rtrim,
      findClosingBracket,
      checkSanitizeDeprecation
    };

    const { defaults: defaults$1 } = defaults;
    const {
      rtrim: rtrim$1,
      splitCells: splitCells$1,
      escape: escape$1,
      findClosingBracket: findClosingBracket$1
    } = helpers;

    function outputLink(cap, link, raw) {
      const href = link.href;
      const title = link.title ? escape$1(link.title) : null;

      if (cap[0].charAt(0) !== '!') {
        return {
          type: 'link',
          raw,
          href,
          title,
          text: cap[1]
        };
      } else {
        return {
          type: 'image',
          raw,
          text: escape$1(cap[1]),
          href,
          title
        };
      }
    }

    function indentCodeCompensation(raw, text) {
      const matchIndentToCode = raw.match(/^(\s+)(?:```)/);

      if (matchIndentToCode === null) {
        return text;
      }

      const indentToCode = matchIndentToCode[1];

      return text
        .split('\n')
        .map(node => {
          const matchIndentInNode = node.match(/^\s+/);
          if (matchIndentInNode === null) {
            return node;
          }

          const [indentInNode] = matchIndentInNode;

          if (indentInNode.length >= indentToCode.length) {
            return node.slice(indentToCode.length);
          }

          return node;
        })
        .join('\n');
    }

    /**
     * Tokenizer
     */
    var Tokenizer_1 = class Tokenizer {
      constructor(options) {
        this.options = options || defaults$1;
      }

      space(src) {
        const cap = this.rules.block.newline.exec(src);
        if (cap) {
          if (cap[0].length > 1) {
            return {
              type: 'space',
              raw: cap[0]
            };
          }
          return { raw: '\n' };
        }
      }

      code(src, tokens) {
        const cap = this.rules.block.code.exec(src);
        if (cap) {
          const lastToken = tokens[tokens.length - 1];
          // An indented code block cannot interrupt a paragraph.
          if (lastToken && lastToken.type === 'paragraph') {
            return {
              raw: cap[0],
              text: cap[0].trimRight()
            };
          }

          const text = cap[0].replace(/^ {4}/gm, '');
          return {
            type: 'code',
            raw: cap[0],
            codeBlockStyle: 'indented',
            text: !this.options.pedantic
              ? rtrim$1(text, '\n')
              : text
          };
        }
      }

      fences(src) {
        const cap = this.rules.block.fences.exec(src);
        if (cap) {
          const raw = cap[0];
          const text = indentCodeCompensation(raw, cap[3] || '');

          return {
            type: 'code',
            raw,
            lang: cap[2] ? cap[2].trim() : cap[2],
            text
          };
        }
      }

      heading(src) {
        const cap = this.rules.block.heading.exec(src);
        if (cap) {
          return {
            type: 'heading',
            raw: cap[0],
            depth: cap[1].length,
            text: cap[2]
          };
        }
      }

      nptable(src) {
        const cap = this.rules.block.nptable.exec(src);
        if (cap) {
          const item = {
            type: 'table',
            header: splitCells$1(cap[1].replace(/^ *| *\| *$/g, '')),
            align: cap[2].replace(/^ *|\| *$/g, '').split(/ *\| */),
            cells: cap[3] ? cap[3].replace(/\n$/, '').split('\n') : [],
            raw: cap[0]
          };

          if (item.header.length === item.align.length) {
            let l = item.align.length;
            let i;
            for (i = 0; i < l; i++) {
              if (/^ *-+: *$/.test(item.align[i])) {
                item.align[i] = 'right';
              } else if (/^ *:-+: *$/.test(item.align[i])) {
                item.align[i] = 'center';
              } else if (/^ *:-+ *$/.test(item.align[i])) {
                item.align[i] = 'left';
              } else {
                item.align[i] = null;
              }
            }

            l = item.cells.length;
            for (i = 0; i < l; i++) {
              item.cells[i] = splitCells$1(item.cells[i], item.header.length);
            }

            return item;
          }
        }
      }

      hr(src) {
        const cap = this.rules.block.hr.exec(src);
        if (cap) {
          return {
            type: 'hr',
            raw: cap[0]
          };
        }
      }

      blockquote(src) {
        const cap = this.rules.block.blockquote.exec(src);
        if (cap) {
          const text = cap[0].replace(/^ *> ?/gm, '');

          return {
            type: 'blockquote',
            raw: cap[0],
            text
          };
        }
      }

      list(src) {
        const cap = this.rules.block.list.exec(src);
        if (cap) {
          let raw = cap[0];
          const bull = cap[2];
          const isordered = bull.length > 1;

          const list = {
            type: 'list',
            raw,
            ordered: isordered,
            start: isordered ? +bull : '',
            loose: false,
            items: []
          };

          // Get each top-level item.
          const itemMatch = cap[0].match(this.rules.block.item);

          let next = false,
            item,
            space,
            b,
            addBack,
            loose,
            istask,
            ischecked;

          const l = itemMatch.length;
          for (let i = 0; i < l; i++) {
            item = itemMatch[i];
            raw = item;

            // Remove the list item's bullet
            // so it is seen as the next token.
            space = item.length;
            item = item.replace(/^ *([*+-]|\d+\.) */, '');

            // Outdent whatever the
            // list item contains. Hacky.
            if (~item.indexOf('\n ')) {
              space -= item.length;
              item = !this.options.pedantic
                ? item.replace(new RegExp('^ {1,' + space + '}', 'gm'), '')
                : item.replace(/^ {1,4}/gm, '');
            }

            // Determine whether the next list item belongs here.
            // Backpedal if it does not belong in this list.
            if (i !== l - 1) {
              b = this.rules.block.bullet.exec(itemMatch[i + 1])[0];
              if (bull.length > 1 ? b.length === 1
                : (b.length > 1 || (this.options.smartLists && b !== bull))) {
                addBack = itemMatch.slice(i + 1).join('\n');
                list.raw = list.raw.substring(0, list.raw.length - addBack.length);
                i = l - 1;
              }
            }

            // Determine whether item is loose or not.
            // Use: /(^|\n)(?! )[^\n]+\n\n(?!\s*$)/
            // for discount behavior.
            loose = next || /\n\n(?!\s*$)/.test(item);
            if (i !== l - 1) {
              next = item.charAt(item.length - 1) === '\n';
              if (!loose) loose = next;
            }

            if (loose) {
              list.loose = true;
            }

            // Check for task list items
            istask = /^\[[ xX]\] /.test(item);
            ischecked = undefined;
            if (istask) {
              ischecked = item[1] !== ' ';
              item = item.replace(/^\[[ xX]\] +/, '');
            }

            list.items.push({
              type: 'list_item',
              raw,
              task: istask,
              checked: ischecked,
              loose: loose,
              text: item
            });
          }

          return list;
        }
      }

      html(src) {
        const cap = this.rules.block.html.exec(src);
        if (cap) {
          return {
            type: this.options.sanitize
              ? 'paragraph'
              : 'html',
            raw: cap[0],
            pre: !this.options.sanitizer
              && (cap[1] === 'pre' || cap[1] === 'script' || cap[1] === 'style'),
            text: this.options.sanitize ? (this.options.sanitizer ? this.options.sanitizer(cap[0]) : escape$1(cap[0])) : cap[0]
          };
        }
      }

      def(src) {
        const cap = this.rules.block.def.exec(src);
        if (cap) {
          if (cap[3]) cap[3] = cap[3].substring(1, cap[3].length - 1);
          const tag = cap[1].toLowerCase().replace(/\s+/g, ' ');
          return {
            tag,
            raw: cap[0],
            href: cap[2],
            title: cap[3]
          };
        }
      }

      table(src) {
        const cap = this.rules.block.table.exec(src);
        if (cap) {
          const item = {
            type: 'table',
            header: splitCells$1(cap[1].replace(/^ *| *\| *$/g, '')),
            align: cap[2].replace(/^ *|\| *$/g, '').split(/ *\| */),
            cells: cap[3] ? cap[3].replace(/\n$/, '').split('\n') : []
          };

          if (item.header.length === item.align.length) {
            item.raw = cap[0];

            let l = item.align.length;
            let i;
            for (i = 0; i < l; i++) {
              if (/^ *-+: *$/.test(item.align[i])) {
                item.align[i] = 'right';
              } else if (/^ *:-+: *$/.test(item.align[i])) {
                item.align[i] = 'center';
              } else if (/^ *:-+ *$/.test(item.align[i])) {
                item.align[i] = 'left';
              } else {
                item.align[i] = null;
              }
            }

            l = item.cells.length;
            for (i = 0; i < l; i++) {
              item.cells[i] = splitCells$1(
                item.cells[i].replace(/^ *\| *| *\| *$/g, ''),
                item.header.length);
            }

            return item;
          }
        }
      }

      lheading(src) {
        const cap = this.rules.block.lheading.exec(src);
        if (cap) {
          return {
            type: 'heading',
            raw: cap[0],
            depth: cap[2].charAt(0) === '=' ? 1 : 2,
            text: cap[1]
          };
        }
      }

      paragraph(src) {
        const cap = this.rules.block.paragraph.exec(src);
        if (cap) {
          return {
            type: 'paragraph',
            raw: cap[0],
            text: cap[1].charAt(cap[1].length - 1) === '\n'
              ? cap[1].slice(0, -1)
              : cap[1]
          };
        }
      }

      text(src, tokens) {
        const cap = this.rules.block.text.exec(src);
        if (cap) {
          const lastToken = tokens[tokens.length - 1];
          if (lastToken && lastToken.type === 'text') {
            return {
              raw: cap[0],
              text: cap[0]
            };
          }

          return {
            type: 'text',
            raw: cap[0],
            text: cap[0]
          };
        }
      }

      escape(src) {
        const cap = this.rules.inline.escape.exec(src);
        if (cap) {
          return {
            type: 'escape',
            raw: cap[0],
            text: escape$1(cap[1])
          };
        }
      }

      tag(src, inLink, inRawBlock) {
        const cap = this.rules.inline.tag.exec(src);
        if (cap) {
          if (!inLink && /^<a /i.test(cap[0])) {
            inLink = true;
          } else if (inLink && /^<\/a>/i.test(cap[0])) {
            inLink = false;
          }
          if (!inRawBlock && /^<(pre|code|kbd|script)(\s|>)/i.test(cap[0])) {
            inRawBlock = true;
          } else if (inRawBlock && /^<\/(pre|code|kbd|script)(\s|>)/i.test(cap[0])) {
            inRawBlock = false;
          }

          return {
            type: this.options.sanitize
              ? 'text'
              : 'html',
            raw: cap[0],
            inLink,
            inRawBlock,
            text: this.options.sanitize
              ? (this.options.sanitizer
                ? this.options.sanitizer(cap[0])
                : escape$1(cap[0]))
              : cap[0]
          };
        }
      }

      link(src) {
        const cap = this.rules.inline.link.exec(src);
        if (cap) {
          const lastParenIndex = findClosingBracket$1(cap[2], '()');
          if (lastParenIndex > -1) {
            const start = cap[0].indexOf('!') === 0 ? 5 : 4;
            const linkLen = start + cap[1].length + lastParenIndex;
            cap[2] = cap[2].substring(0, lastParenIndex);
            cap[0] = cap[0].substring(0, linkLen).trim();
            cap[3] = '';
          }
          let href = cap[2];
          let title = '';
          if (this.options.pedantic) {
            const link = /^([^'"]*[^\s])\s+(['"])(.*)\2/.exec(href);

            if (link) {
              href = link[1];
              title = link[3];
            } else {
              title = '';
            }
          } else {
            title = cap[3] ? cap[3].slice(1, -1) : '';
          }
          href = href.trim().replace(/^<([\s\S]*)>$/, '$1');
          const token = outputLink(cap, {
            href: href ? href.replace(this.rules.inline._escapes, '$1') : href,
            title: title ? title.replace(this.rules.inline._escapes, '$1') : title
          }, cap[0]);
          return token;
        }
      }

      reflink(src, links) {
        let cap;
        if ((cap = this.rules.inline.reflink.exec(src))
            || (cap = this.rules.inline.nolink.exec(src))) {
          let link = (cap[2] || cap[1]).replace(/\s+/g, ' ');
          link = links[link.toLowerCase()];
          if (!link || !link.href) {
            const text = cap[0].charAt(0);
            return {
              type: 'text',
              raw: text,
              text
            };
          }
          const token = outputLink(cap, link, cap[0]);
          return token;
        }
      }

      strong(src) {
        const cap = this.rules.inline.strong.exec(src);
        if (cap) {
          return {
            type: 'strong',
            raw: cap[0],
            text: cap[4] || cap[3] || cap[2] || cap[1]
          };
        }
      }

      em(src) {
        const cap = this.rules.inline.em.exec(src);
        if (cap) {
          return {
            type: 'em',
            raw: cap[0],
            text: cap[6] || cap[5] || cap[4] || cap[3] || cap[2] || cap[1]
          };
        }
      }

      codespan(src) {
        const cap = this.rules.inline.code.exec(src);
        if (cap) {
          let text = cap[2].replace(/\n/g, ' ');
          const hasNonSpaceChars = /[^ ]/.test(text);
          const hasSpaceCharsOnBothEnds = text.startsWith(' ') && text.endsWith(' ');
          if (hasNonSpaceChars && hasSpaceCharsOnBothEnds) {
            text = text.substring(1, text.length - 1);
          }
          text = escape$1(text, true);
          return {
            type: 'codespan',
            raw: cap[0],
            text
          };
        }
      }

      br(src) {
        const cap = this.rules.inline.br.exec(src);
        if (cap) {
          return {
            type: 'br',
            raw: cap[0]
          };
        }
      }

      del(src) {
        const cap = this.rules.inline.del.exec(src);
        if (cap) {
          return {
            type: 'del',
            raw: cap[0],
            text: cap[1]
          };
        }
      }

      autolink(src, mangle) {
        const cap = this.rules.inline.autolink.exec(src);
        if (cap) {
          let text, href;
          if (cap[2] === '@') {
            text = escape$1(this.options.mangle ? mangle(cap[1]) : cap[1]);
            href = 'mailto:' + text;
          } else {
            text = escape$1(cap[1]);
            href = text;
          }

          return {
            type: 'link',
            raw: cap[0],
            text,
            href,
            tokens: [
              {
                type: 'text',
                raw: text,
                text
              }
            ]
          };
        }
      }

      url(src, mangle) {
        let cap;
        if (cap = this.rules.inline.url.exec(src)) {
          let text, href;
          if (cap[2] === '@') {
            text = escape$1(this.options.mangle ? mangle(cap[0]) : cap[0]);
            href = 'mailto:' + text;
          } else {
            // do extended autolink path validation
            let prevCapZero;
            do {
              prevCapZero = cap[0];
              cap[0] = this.rules.inline._backpedal.exec(cap[0])[0];
            } while (prevCapZero !== cap[0]);
            text = escape$1(cap[0]);
            if (cap[1] === 'www.') {
              href = 'http://' + text;
            } else {
              href = text;
            }
          }
          return {
            type: 'link',
            raw: cap[0],
            text,
            href,
            tokens: [
              {
                type: 'text',
                raw: text,
                text
              }
            ]
          };
        }
      }

      inlineText(src, inRawBlock, smartypants) {
        const cap = this.rules.inline.text.exec(src);
        if (cap) {
          let text;
          if (inRawBlock) {
            text = this.options.sanitize ? (this.options.sanitizer ? this.options.sanitizer(cap[0]) : escape$1(cap[0])) : cap[0];
          } else {
            text = escape$1(this.options.smartypants ? smartypants(cap[0]) : cap[0]);
          }
          return {
            type: 'text',
            raw: cap[0],
            text
          };
        }
      }
    };

    const {
      noopTest: noopTest$1,
      edit: edit$1,
      merge: merge$1
    } = helpers;

    /**
     * Block-Level Grammar
     */
    const block = {
      newline: /^\n+/,
      code: /^( {4}[^\n]+\n*)+/,
      fences: /^ {0,3}(`{3,}(?=[^`\n]*\n)|~{3,})([^\n]*)\n(?:|([\s\S]*?)\n)(?: {0,3}\1[~`]* *(?:\n+|$)|$)/,
      hr: /^ {0,3}((?:- *){3,}|(?:_ *){3,}|(?:\* *){3,})(?:\n+|$)/,
      heading: /^ {0,3}(#{1,6}) +([^\n]*?)(?: +#+)? *(?:\n+|$)/,
      blockquote: /^( {0,3}> ?(paragraph|[^\n]*)(?:\n|$))+/,
      list: /^( {0,3})(bull) [\s\S]+?(?:hr|def|\n{2,}(?! )(?!\1bull )\n*|\s*$)/,
      html: '^ {0,3}(?:' // optional indentation
        + '<(script|pre|style)[\\s>][\\s\\S]*?(?:</\\1>[^\\n]*\\n+|$)' // (1)
        + '|comment[^\\n]*(\\n+|$)' // (2)
        + '|<\\?[\\s\\S]*?\\?>\\n*' // (3)
        + '|<![A-Z][\\s\\S]*?>\\n*' // (4)
        + '|<!\\[CDATA\\[[\\s\\S]*?\\]\\]>\\n*' // (5)
        + '|</?(tag)(?: +|\\n|/?>)[\\s\\S]*?(?:\\n{2,}|$)' // (6)
        + '|<(?!script|pre|style)([a-z][\\w-]*)(?:attribute)*? */?>(?=[ \\t]*(?:\\n|$))[\\s\\S]*?(?:\\n{2,}|$)' // (7) open tag
        + '|</(?!script|pre|style)[a-z][\\w-]*\\s*>(?=[ \\t]*(?:\\n|$))[\\s\\S]*?(?:\\n{2,}|$)' // (7) closing tag
        + ')',
      def: /^ {0,3}\[(label)\]: *\n? *<?([^\s>]+)>?(?:(?: +\n? *| *\n *)(title))? *(?:\n+|$)/,
      nptable: noopTest$1,
      table: noopTest$1,
      lheading: /^([^\n]+)\n {0,3}(=+|-+) *(?:\n+|$)/,
      // regex template, placeholders will be replaced according to different paragraph
      // interruption rules of commonmark and the original markdown spec:
      _paragraph: /^([^\n]+(?:\n(?!hr|heading|lheading|blockquote|fences|list|html)[^\n]+)*)/,
      text: /^[^\n]+/
    };

    block._label = /(?!\s*\])(?:\\[\[\]]|[^\[\]])+/;
    block._title = /(?:"(?:\\"?|[^"\\])*"|'[^'\n]*(?:\n[^'\n]+)*\n?'|\([^()]*\))/;
    block.def = edit$1(block.def)
      .replace('label', block._label)
      .replace('title', block._title)
      .getRegex();

    block.bullet = /(?:[*+-]|\d{1,9}\.)/;
    block.item = /^( *)(bull) ?[^\n]*(?:\n(?!\1bull ?)[^\n]*)*/;
    block.item = edit$1(block.item, 'gm')
      .replace(/bull/g, block.bullet)
      .getRegex();

    block.list = edit$1(block.list)
      .replace(/bull/g, block.bullet)
      .replace('hr', '\\n+(?=\\1?(?:(?:- *){3,}|(?:_ *){3,}|(?:\\* *){3,})(?:\\n+|$))')
      .replace('def', '\\n+(?=' + block.def.source + ')')
      .getRegex();

    block._tag = 'address|article|aside|base|basefont|blockquote|body|caption'
      + '|center|col|colgroup|dd|details|dialog|dir|div|dl|dt|fieldset|figcaption'
      + '|figure|footer|form|frame|frameset|h[1-6]|head|header|hr|html|iframe'
      + '|legend|li|link|main|menu|menuitem|meta|nav|noframes|ol|optgroup|option'
      + '|p|param|section|source|summary|table|tbody|td|tfoot|th|thead|title|tr'
      + '|track|ul';
    block._comment = /<!--(?!-?>)[\s\S]*?-->/;
    block.html = edit$1(block.html, 'i')
      .replace('comment', block._comment)
      .replace('tag', block._tag)
      .replace('attribute', / +[a-zA-Z:_][\w.:-]*(?: *= *"[^"\n]*"| *= *'[^'\n]*'| *= *[^\s"'=<>`]+)?/)
      .getRegex();

    block.paragraph = edit$1(block._paragraph)
      .replace('hr', block.hr)
      .replace('heading', ' {0,3}#{1,6} ')
      .replace('|lheading', '') // setex headings don't interrupt commonmark paragraphs
      .replace('blockquote', ' {0,3}>')
      .replace('fences', ' {0,3}(?:`{3,}(?=[^`\\n]*\\n)|~{3,})[^\\n]*\\n')
      .replace('list', ' {0,3}(?:[*+-]|1[.)]) ') // only lists starting from 1 can interrupt
      .replace('html', '</?(?:tag)(?: +|\\n|/?>)|<(?:script|pre|style|!--)')
      .replace('tag', block._tag) // pars can be interrupted by type (6) html blocks
      .getRegex();

    block.blockquote = edit$1(block.blockquote)
      .replace('paragraph', block.paragraph)
      .getRegex();

    /**
     * Normal Block Grammar
     */

    block.normal = merge$1({}, block);

    /**
     * GFM Block Grammar
     */

    block.gfm = merge$1({}, block.normal, {
      nptable: '^ *([^|\\n ].*\\|.*)\\n' // Header
        + ' *([-:]+ *\\|[-| :]*)' // Align
        + '(?:\\n((?:(?!\\n|hr|heading|blockquote|code|fences|list|html).*(?:\\n|$))*)\\n*|$)', // Cells
      table: '^ *\\|(.+)\\n' // Header
        + ' *\\|?( *[-:]+[-| :]*)' // Align
        + '(?:\\n *((?:(?!\\n|hr|heading|blockquote|code|fences|list|html).*(?:\\n|$))*)\\n*|$)' // Cells
    });

    block.gfm.nptable = edit$1(block.gfm.nptable)
      .replace('hr', block.hr)
      .replace('heading', ' {0,3}#{1,6} ')
      .replace('blockquote', ' {0,3}>')
      .replace('code', ' {4}[^\\n]')
      .replace('fences', ' {0,3}(?:`{3,}(?=[^`\\n]*\\n)|~{3,})[^\\n]*\\n')
      .replace('list', ' {0,3}(?:[*+-]|1[.)]) ') // only lists starting from 1 can interrupt
      .replace('html', '</?(?:tag)(?: +|\\n|/?>)|<(?:script|pre|style|!--)')
      .replace('tag', block._tag) // tables can be interrupted by type (6) html blocks
      .getRegex();

    block.gfm.table = edit$1(block.gfm.table)
      .replace('hr', block.hr)
      .replace('heading', ' {0,3}#{1,6} ')
      .replace('blockquote', ' {0,3}>')
      .replace('code', ' {4}[^\\n]')
      .replace('fences', ' {0,3}(?:`{3,}(?=[^`\\n]*\\n)|~{3,})[^\\n]*\\n')
      .replace('list', ' {0,3}(?:[*+-]|1[.)]) ') // only lists starting from 1 can interrupt
      .replace('html', '</?(?:tag)(?: +|\\n|/?>)|<(?:script|pre|style|!--)')
      .replace('tag', block._tag) // tables can be interrupted by type (6) html blocks
      .getRegex();

    /**
     * Pedantic grammar (original John Gruber's loose markdown specification)
     */

    block.pedantic = merge$1({}, block.normal, {
      html: edit$1(
        '^ *(?:comment *(?:\\n|\\s*$)'
        + '|<(tag)[\\s\\S]+?</\\1> *(?:\\n{2,}|\\s*$)' // closed tag
        + '|<tag(?:"[^"]*"|\'[^\']*\'|\\s[^\'"/>\\s]*)*?/?> *(?:\\n{2,}|\\s*$))')
        .replace('comment', block._comment)
        .replace(/tag/g, '(?!(?:'
          + 'a|em|strong|small|s|cite|q|dfn|abbr|data|time|code|var|samp|kbd|sub'
          + '|sup|i|b|u|mark|ruby|rt|rp|bdi|bdo|span|br|wbr|ins|del|img)'
          + '\\b)\\w+(?!:|[^\\w\\s@]*@)\\b')
        .getRegex(),
      def: /^ *\[([^\]]+)\]: *<?([^\s>]+)>?(?: +(["(][^\n]+[")]))? *(?:\n+|$)/,
      heading: /^ *(#{1,6}) *([^\n]+?) *(?:#+ *)?(?:\n+|$)/,
      fences: noopTest$1, // fences not supported
      paragraph: edit$1(block.normal._paragraph)
        .replace('hr', block.hr)
        .replace('heading', ' *#{1,6} *[^\n]')
        .replace('lheading', block.lheading)
        .replace('blockquote', ' {0,3}>')
        .replace('|fences', '')
        .replace('|list', '')
        .replace('|html', '')
        .getRegex()
    });

    /**
     * Inline-Level Grammar
     */
    const inline = {
      escape: /^\\([!"#$%&'()*+,\-./:;<=>?@\[\]\\^_`{|}~])/,
      autolink: /^<(scheme:[^\s\x00-\x1f<>]*|email)>/,
      url: noopTest$1,
      tag: '^comment'
        + '|^</[a-zA-Z][\\w:-]*\\s*>' // self-closing tag
        + '|^<[a-zA-Z][\\w-]*(?:attribute)*?\\s*/?>' // open tag
        + '|^<\\?[\\s\\S]*?\\?>' // processing instruction, e.g. <?php ?>
        + '|^<![a-zA-Z]+\\s[\\s\\S]*?>' // declaration, e.g. <!DOCTYPE html>
        + '|^<!\\[CDATA\\[[\\s\\S]*?\\]\\]>', // CDATA section
      link: /^!?\[(label)\]\(\s*(href)(?:\s+(title))?\s*\)/,
      reflink: /^!?\[(label)\]\[(?!\s*\])((?:\\[\[\]]?|[^\[\]\\])+)\]/,
      nolink: /^!?\[(?!\s*\])((?:\[[^\[\]]*\]|\\[\[\]]|[^\[\]])*)\](?:\[\])?/,
      strong: /^__([^\s_])__(?!_)|^\*\*([^\s*])\*\*(?!\*)|^__([^\s][\s\S]*?[^\s])__(?!_)|^\*\*([^\s][\s\S]*?[^\s])\*\*(?!\*)/,
      em: /^_([^\s_])_(?!_)|^_([^\s_<][\s\S]*?[^\s_])_(?!_|[^\s,punctuation])|^_([^\s_<][\s\S]*?[^\s])_(?!_|[^\s,punctuation])|^\*([^\s*<\[])\*(?!\*)|^\*([^\s<"][\s\S]*?[^\s\[\*])\*(?![\]`punctuation])|^\*([^\s*"<\[][\s\S]*[^\s])\*(?!\*)/,
      code: /^(`+)([^`]|[^`][\s\S]*?[^`])\1(?!`)/,
      br: /^( {2,}|\\)\n(?!\s*$)/,
      del: noopTest$1,
      text: /^(`+|[^`])(?:[\s\S]*?(?:(?=[\\<!\[`*]|\b_|$)|[^ ](?= {2,}\n))|(?= {2,}\n))/
    };

    // list of punctuation marks from common mark spec
    // without ` and ] to workaround Rule 17 (inline code blocks/links)
    // without , to work around example 393
    inline._punctuation = '!"#$%&\'()*+\\-./:;<=>?@\\[^_{|}~';
    inline.em = edit$1(inline.em).replace(/punctuation/g, inline._punctuation).getRegex();

    inline._escapes = /\\([!"#$%&'()*+,\-./:;<=>?@\[\]\\^_`{|}~])/g;

    inline._scheme = /[a-zA-Z][a-zA-Z0-9+.-]{1,31}/;
    inline._email = /[a-zA-Z0-9.!#$%&'*+/=?^_`{|}~-]+(@)[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(?:\.[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)+(?![-_])/;
    inline.autolink = edit$1(inline.autolink)
      .replace('scheme', inline._scheme)
      .replace('email', inline._email)
      .getRegex();

    inline._attribute = /\s+[a-zA-Z:_][\w.:-]*(?:\s*=\s*"[^"]*"|\s*=\s*'[^']*'|\s*=\s*[^\s"'=<>`]+)?/;

    inline.tag = edit$1(inline.tag)
      .replace('comment', block._comment)
      .replace('attribute', inline._attribute)
      .getRegex();

    inline._label = /(?:\[[^\[\]]*\]|\\.|`[^`]*`|[^\[\]\\`])*?/;
    inline._href = /<(?:\\[<>]?|[^\s<>\\])*>|[^\s\x00-\x1f]*/;
    inline._title = /"(?:\\"?|[^"\\])*"|'(?:\\'?|[^'\\])*'|\((?:\\\)?|[^)\\])*\)/;

    inline.link = edit$1(inline.link)
      .replace('label', inline._label)
      .replace('href', inline._href)
      .replace('title', inline._title)
      .getRegex();

    inline.reflink = edit$1(inline.reflink)
      .replace('label', inline._label)
      .getRegex();

    /**
     * Normal Inline Grammar
     */

    inline.normal = merge$1({}, inline);

    /**
     * Pedantic Inline Grammar
     */

    inline.pedantic = merge$1({}, inline.normal, {
      strong: /^__(?=\S)([\s\S]*?\S)__(?!_)|^\*\*(?=\S)([\s\S]*?\S)\*\*(?!\*)/,
      em: /^_(?=\S)([\s\S]*?\S)_(?!_)|^\*(?=\S)([\s\S]*?\S)\*(?!\*)/,
      link: edit$1(/^!?\[(label)\]\((.*?)\)/)
        .replace('label', inline._label)
        .getRegex(),
      reflink: edit$1(/^!?\[(label)\]\s*\[([^\]]*)\]/)
        .replace('label', inline._label)
        .getRegex()
    });

    /**
     * GFM Inline Grammar
     */

    inline.gfm = merge$1({}, inline.normal, {
      escape: edit$1(inline.escape).replace('])', '~|])').getRegex(),
      _extended_email: /[A-Za-z0-9._+-]+(@)[a-zA-Z0-9-_]+(?:\.[a-zA-Z0-9-_]*[a-zA-Z0-9])+(?![-_])/,
      url: /^((?:ftp|https?):\/\/|www\.)(?:[a-zA-Z0-9\-]+\.?)+[^\s<]*|^email/,
      _backpedal: /(?:[^?!.,:;*_~()&]+|\([^)]*\)|&(?![a-zA-Z0-9]+;$)|[?!.,:;*_~)]+(?!$))+/,
      del: /^~+(?=\S)([\s\S]*?\S)~+/,
      text: /^(`+|[^`])(?:[\s\S]*?(?:(?=[\\<!\[`*~]|\b_|https?:\/\/|ftp:\/\/|www\.|$)|[^ ](?= {2,}\n)|[^a-zA-Z0-9.!#$%&'*+\/=?_`{\|}~-](?=[a-zA-Z0-9.!#$%&'*+\/=?_`{\|}~-]+@))|(?= {2,}\n|[a-zA-Z0-9.!#$%&'*+\/=?_`{\|}~-]+@))/
    });

    inline.gfm.url = edit$1(inline.gfm.url, 'i')
      .replace('email', inline.gfm._extended_email)
      .getRegex();
    /**
     * GFM + Line Breaks Inline Grammar
     */

    inline.breaks = merge$1({}, inline.gfm, {
      br: edit$1(inline.br).replace('{2,}', '*').getRegex(),
      text: edit$1(inline.gfm.text)
        .replace('\\b_', '\\b_| {2,}\\n')
        .replace(/\{2,\}/g, '*')
        .getRegex()
    });

    var rules = {
      block,
      inline
    };

    const { defaults: defaults$2 } = defaults;
    const { block: block$1, inline: inline$1 } = rules;

    /**
     * smartypants text replacement
     */
    function smartypants(text) {
      return text
        // em-dashes
        .replace(/---/g, '\u2014')
        // en-dashes
        .replace(/--/g, '\u2013')
        // opening singles
        .replace(/(^|[-\u2014/(\[{"\s])'/g, '$1\u2018')
        // closing singles & apostrophes
        .replace(/'/g, '\u2019')
        // opening doubles
        .replace(/(^|[-\u2014/(\[{\u2018\s])"/g, '$1\u201c')
        // closing doubles
        .replace(/"/g, '\u201d')
        // ellipses
        .replace(/\.{3}/g, '\u2026');
    }

    /**
     * mangle email addresses
     */
    function mangle(text) {
      let out = '',
        i,
        ch;

      const l = text.length;
      for (i = 0; i < l; i++) {
        ch = text.charCodeAt(i);
        if (Math.random() > 0.5) {
          ch = 'x' + ch.toString(16);
        }
        out += '&#' + ch + ';';
      }

      return out;
    }

    /**
     * Block Lexer
     */
    var Lexer_1 = class Lexer {
      constructor(options) {
        this.tokens = [];
        this.tokens.links = Object.create(null);
        this.options = options || defaults$2;
        this.options.tokenizer = this.options.tokenizer || new Tokenizer_1();
        this.tokenizer = this.options.tokenizer;
        this.tokenizer.options = this.options;

        const rules = {
          block: block$1.normal,
          inline: inline$1.normal
        };

        if (this.options.pedantic) {
          rules.block = block$1.pedantic;
          rules.inline = inline$1.pedantic;
        } else if (this.options.gfm) {
          rules.block = block$1.gfm;
          if (this.options.breaks) {
            rules.inline = inline$1.breaks;
          } else {
            rules.inline = inline$1.gfm;
          }
        }
        this.tokenizer.rules = rules;
      }

      /**
       * Expose Rules
       */
      static get rules() {
        return {
          block: block$1,
          inline: inline$1
        };
      }

      /**
       * Static Lex Method
       */
      static lex(src, options) {
        const lexer = new Lexer(options);
        return lexer.lex(src);
      }

      /**
       * Preprocessing
       */
      lex(src) {
        src = src
          .replace(/\r\n|\r/g, '\n')
          .replace(/\t/g, '    ');

        this.blockTokens(src, this.tokens, true);

        this.inline(this.tokens);

        return this.tokens;
      }

      /**
       * Lexing
       */
      blockTokens(src, tokens = [], top = true) {
        src = src.replace(/^ +$/gm, '');
        let token, i, l, lastToken;

        while (src) {
          // newline
          if (token = this.tokenizer.space(src)) {
            src = src.substring(token.raw.length);
            if (token.type) {
              tokens.push(token);
            }
            continue;
          }

          // code
          if (token = this.tokenizer.code(src, tokens)) {
            src = src.substring(token.raw.length);
            if (token.type) {
              tokens.push(token);
            } else {
              lastToken = tokens[tokens.length - 1];
              lastToken.raw += '\n' + token.raw;
              lastToken.text += '\n' + token.text;
            }
            continue;
          }

          // fences
          if (token = this.tokenizer.fences(src)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // heading
          if (token = this.tokenizer.heading(src)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // table no leading pipe (gfm)
          if (token = this.tokenizer.nptable(src)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // hr
          if (token = this.tokenizer.hr(src)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // blockquote
          if (token = this.tokenizer.blockquote(src)) {
            src = src.substring(token.raw.length);
            token.tokens = this.blockTokens(token.text, [], top);
            tokens.push(token);
            continue;
          }

          // list
          if (token = this.tokenizer.list(src)) {
            src = src.substring(token.raw.length);
            l = token.items.length;
            for (i = 0; i < l; i++) {
              token.items[i].tokens = this.blockTokens(token.items[i].text, [], false);
            }
            tokens.push(token);
            continue;
          }

          // html
          if (token = this.tokenizer.html(src)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // def
          if (top && (token = this.tokenizer.def(src))) {
            src = src.substring(token.raw.length);
            if (!this.tokens.links[token.tag]) {
              this.tokens.links[token.tag] = {
                href: token.href,
                title: token.title
              };
            }
            continue;
          }

          // table (gfm)
          if (token = this.tokenizer.table(src)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // lheading
          if (token = this.tokenizer.lheading(src)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // top-level paragraph
          if (top && (token = this.tokenizer.paragraph(src))) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // text
          if (token = this.tokenizer.text(src, tokens)) {
            src = src.substring(token.raw.length);
            if (token.type) {
              tokens.push(token);
            } else {
              lastToken = tokens[tokens.length - 1];
              lastToken.raw += '\n' + token.raw;
              lastToken.text += '\n' + token.text;
            }
            continue;
          }

          if (src) {
            const errMsg = 'Infinite loop on byte: ' + src.charCodeAt(0);
            if (this.options.silent) {
              console.error(errMsg);
              break;
            } else {
              throw new Error(errMsg);
            }
          }
        }

        return tokens;
      }

      inline(tokens) {
        let i,
          j,
          k,
          l2,
          row,
          token;

        const l = tokens.length;
        for (i = 0; i < l; i++) {
          token = tokens[i];
          switch (token.type) {
            case 'paragraph':
            case 'text':
            case 'heading': {
              token.tokens = [];
              this.inlineTokens(token.text, token.tokens);
              break;
            }
            case 'table': {
              token.tokens = {
                header: [],
                cells: []
              };

              // header
              l2 = token.header.length;
              for (j = 0; j < l2; j++) {
                token.tokens.header[j] = [];
                this.inlineTokens(token.header[j], token.tokens.header[j]);
              }

              // cells
              l2 = token.cells.length;
              for (j = 0; j < l2; j++) {
                row = token.cells[j];
                token.tokens.cells[j] = [];
                for (k = 0; k < row.length; k++) {
                  token.tokens.cells[j][k] = [];
                  this.inlineTokens(row[k], token.tokens.cells[j][k]);
                }
              }

              break;
            }
            case 'blockquote': {
              this.inline(token.tokens);
              break;
            }
            case 'list': {
              l2 = token.items.length;
              for (j = 0; j < l2; j++) {
                this.inline(token.items[j].tokens);
              }
              break;
            }
          }
        }

        return tokens;
      }

      /**
       * Lexing/Compiling
       */
      inlineTokens(src, tokens = [], inLink = false, inRawBlock = false) {
        let token;

        while (src) {
          // escape
          if (token = this.tokenizer.escape(src)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // tag
          if (token = this.tokenizer.tag(src, inLink, inRawBlock)) {
            src = src.substring(token.raw.length);
            inLink = token.inLink;
            inRawBlock = token.inRawBlock;
            tokens.push(token);
            continue;
          }

          // link
          if (token = this.tokenizer.link(src)) {
            src = src.substring(token.raw.length);
            if (token.type === 'link') {
              token.tokens = this.inlineTokens(token.text, [], true, inRawBlock);
            }
            tokens.push(token);
            continue;
          }

          // reflink, nolink
          if (token = this.tokenizer.reflink(src, this.tokens.links)) {
            src = src.substring(token.raw.length);
            if (token.type === 'link') {
              token.tokens = this.inlineTokens(token.text, [], true, inRawBlock);
            }
            tokens.push(token);
            continue;
          }

          // strong
          if (token = this.tokenizer.strong(src)) {
            src = src.substring(token.raw.length);
            token.tokens = this.inlineTokens(token.text, [], inLink, inRawBlock);
            tokens.push(token);
            continue;
          }

          // em
          if (token = this.tokenizer.em(src)) {
            src = src.substring(token.raw.length);
            token.tokens = this.inlineTokens(token.text, [], inLink, inRawBlock);
            tokens.push(token);
            continue;
          }

          // code
          if (token = this.tokenizer.codespan(src)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // br
          if (token = this.tokenizer.br(src)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // del (gfm)
          if (token = this.tokenizer.del(src)) {
            src = src.substring(token.raw.length);
            token.tokens = this.inlineTokens(token.text, [], inLink, inRawBlock);
            tokens.push(token);
            continue;
          }

          // autolink
          if (token = this.tokenizer.autolink(src, mangle)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // url (gfm)
          if (!inLink && (token = this.tokenizer.url(src, mangle))) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // text
          if (token = this.tokenizer.inlineText(src, inRawBlock, smartypants)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          if (src) {
            const errMsg = 'Infinite loop on byte: ' + src.charCodeAt(0);
            if (this.options.silent) {
              console.error(errMsg);
              break;
            } else {
              throw new Error(errMsg);
            }
          }
        }

        return tokens;
      }
    };

    const { defaults: defaults$3 } = defaults;
    const {
      cleanUrl: cleanUrl$1,
      escape: escape$2
    } = helpers;

    /**
     * Renderer
     */
    var Renderer_1 = class Renderer {
      constructor(options) {
        this.options = options || defaults$3;
      }

      code(code, infostring, escaped) {
        const lang = (infostring || '').match(/\S*/)[0];
        if (this.options.highlight) {
          const out = this.options.highlight(code, lang);
          if (out != null && out !== code) {
            escaped = true;
            code = out;
          }
        }

        if (!lang) {
          return '<pre><code>'
            + (escaped ? code : escape$2(code, true))
            + '</code></pre>\n';
        }

        return '<pre><code class="'
          + this.options.langPrefix
          + escape$2(lang, true)
          + '">'
          + (escaped ? code : escape$2(code, true))
          + '</code></pre>\n';
      }

      blockquote(quote) {
        return '<blockquote>\n' + quote + '</blockquote>\n';
      }

      html(html) {
        return html;
      }

      heading(text, level, raw, slugger) {
        if (this.options.headerIds) {
          return '<h'
            + level
            + ' id="'
            + this.options.headerPrefix
            + slugger.slug(raw)
            + '">'
            + text
            + '</h'
            + level
            + '>\n';
        }
        // ignore IDs
        return '<h' + level + '>' + text + '</h' + level + '>\n';
      }

      hr() {
        return this.options.xhtml ? '<hr/>\n' : '<hr>\n';
      }

      list(body, ordered, start) {
        const type = ordered ? 'ol' : 'ul',
          startatt = (ordered && start !== 1) ? (' start="' + start + '"') : '';
        return '<' + type + startatt + '>\n' + body + '</' + type + '>\n';
      }

      listitem(text) {
        return '<li>' + text + '</li>\n';
      }

      checkbox(checked) {
        return '<input '
          + (checked ? 'checked="" ' : '')
          + 'disabled="" type="checkbox"'
          + (this.options.xhtml ? ' /' : '')
          + '> ';
      }

      paragraph(text) {
        return '<p>' + text + '</p>\n';
      }

      table(header, body) {
        if (body) body = '<tbody>' + body + '</tbody>';

        return '<table>\n'
          + '<thead>\n'
          + header
          + '</thead>\n'
          + body
          + '</table>\n';
      }

      tablerow(content) {
        return '<tr>\n' + content + '</tr>\n';
      }

      tablecell(content, flags) {
        const type = flags.header ? 'th' : 'td';
        const tag = flags.align
          ? '<' + type + ' align="' + flags.align + '">'
          : '<' + type + '>';
        return tag + content + '</' + type + '>\n';
      }

      // span level renderer
      strong(text) {
        return '<strong>' + text + '</strong>';
      }

      em(text) {
        return '<em>' + text + '</em>';
      }

      codespan(text) {
        return '<code>' + text + '</code>';
      }

      br() {
        return this.options.xhtml ? '<br/>' : '<br>';
      }

      del(text) {
        return '<del>' + text + '</del>';
      }

      link(href, title, text) {
        href = cleanUrl$1(this.options.sanitize, this.options.baseUrl, href);
        if (href === null) {
          return text;
        }
        let out = '<a href="' + escape$2(href) + '"';
        if (title) {
          out += ' title="' + title + '"';
        }
        out += '>' + text + '</a>';
        return out;
      }

      image(href, title, text) {
        href = cleanUrl$1(this.options.sanitize, this.options.baseUrl, href);
        if (href === null) {
          return text;
        }

        let out = '<img src="' + href + '" alt="' + text + '"';
        if (title) {
          out += ' title="' + title + '"';
        }
        out += this.options.xhtml ? '/>' : '>';
        return out;
      }

      text(text) {
        return text;
      }
    };

    /**
     * TextRenderer
     * returns only the textual part of the token
     */
    var TextRenderer_1 = class TextRenderer {
      // no need for block level renderers
      strong(text) {
        return text;
      }

      em(text) {
        return text;
      }

      codespan(text) {
        return text;
      }

      del(text) {
        return text;
      }

      html(text) {
        return text;
      }

      text(text) {
        return text;
      }

      link(href, title, text) {
        return '' + text;
      }

      image(href, title, text) {
        return '' + text;
      }

      br() {
        return '';
      }
    };

    /**
     * Slugger generates header id
     */
    var Slugger_1 = class Slugger {
      constructor() {
        this.seen = {};
      }

      /**
       * Convert string to unique id
       */
      slug(value) {
        let slug = value
          .toLowerCase()
          .trim()
          // remove html tags
          .replace(/<[!\/a-z].*?>/ig, '')
          // remove unwanted chars
          .replace(/[\u2000-\u206F\u2E00-\u2E7F\\'!"#$%&()*+,./:;<=>?@[\]^`{|}~]/g, '')
          .replace(/\s/g, '-');

        if (this.seen.hasOwnProperty(slug)) {
          const originalSlug = slug;
          do {
            this.seen[originalSlug]++;
            slug = originalSlug + '-' + this.seen[originalSlug];
          } while (this.seen.hasOwnProperty(slug));
        }
        this.seen[slug] = 0;

        return slug;
      }
    };

    const { defaults: defaults$4 } = defaults;
    const {
      unescape: unescape$2
    } = helpers;

    /**
     * Parsing & Compiling
     */
    var Parser_1 = class Parser {
      constructor(options) {
        this.options = options || defaults$4;
        this.options.renderer = this.options.renderer || new Renderer_1();
        this.renderer = this.options.renderer;
        this.renderer.options = this.options;
        this.textRenderer = new TextRenderer_1();
        this.slugger = new Slugger_1();
      }

      /**
       * Static Parse Method
       */
      static parse(tokens, options) {
        const parser = new Parser(options);
        return parser.parse(tokens);
      }

      /**
       * Parse Loop
       */
      parse(tokens, top = true) {
        let out = '',
          i,
          j,
          k,
          l2,
          l3,
          row,
          cell,
          header,
          body,
          token,
          ordered,
          start,
          loose,
          itemBody,
          item,
          checked,
          task,
          checkbox;

        const l = tokens.length;
        for (i = 0; i < l; i++) {
          token = tokens[i];
          switch (token.type) {
            case 'space': {
              continue;
            }
            case 'hr': {
              out += this.renderer.hr();
              continue;
            }
            case 'heading': {
              out += this.renderer.heading(
                this.parseInline(token.tokens),
                token.depth,
                unescape$2(this.parseInline(token.tokens, this.textRenderer)),
                this.slugger);
              continue;
            }
            case 'code': {
              out += this.renderer.code(token.text,
                token.lang,
                token.escaped);
              continue;
            }
            case 'table': {
              header = '';

              // header
              cell = '';
              l2 = token.header.length;
              for (j = 0; j < l2; j++) {
                cell += this.renderer.tablecell(
                  this.parseInline(token.tokens.header[j]),
                  { header: true, align: token.align[j] }
                );
              }
              header += this.renderer.tablerow(cell);

              body = '';
              l2 = token.cells.length;
              for (j = 0; j < l2; j++) {
                row = token.tokens.cells[j];

                cell = '';
                l3 = row.length;
                for (k = 0; k < l3; k++) {
                  cell += this.renderer.tablecell(
                    this.parseInline(row[k]),
                    { header: false, align: token.align[k] }
                  );
                }

                body += this.renderer.tablerow(cell);
              }
              out += this.renderer.table(header, body);
              continue;
            }
            case 'blockquote': {
              body = this.parse(token.tokens);
              out += this.renderer.blockquote(body);
              continue;
            }
            case 'list': {
              ordered = token.ordered;
              start = token.start;
              loose = token.loose;
              l2 = token.items.length;

              body = '';
              for (j = 0; j < l2; j++) {
                item = token.items[j];
                checked = item.checked;
                task = item.task;

                itemBody = '';
                if (item.task) {
                  checkbox = this.renderer.checkbox(checked);
                  if (loose) {
                    if (item.tokens.length > 0 && item.tokens[0].type === 'text') {
                      item.tokens[0].text = checkbox + ' ' + item.tokens[0].text;
                      if (item.tokens[0].tokens && item.tokens[0].tokens.length > 0 && item.tokens[0].tokens[0].type === 'text') {
                        item.tokens[0].tokens[0].text = checkbox + ' ' + item.tokens[0].tokens[0].text;
                      }
                    } else {
                      item.tokens.unshift({
                        type: 'text',
                        text: checkbox
                      });
                    }
                  } else {
                    itemBody += checkbox;
                  }
                }

                itemBody += this.parse(item.tokens, loose);
                body += this.renderer.listitem(itemBody, task, checked);
              }

              out += this.renderer.list(body, ordered, start);
              continue;
            }
            case 'html': {
              // TODO parse inline content if parameter markdown=1
              out += this.renderer.html(token.text);
              continue;
            }
            case 'paragraph': {
              out += this.renderer.paragraph(this.parseInline(token.tokens));
              continue;
            }
            case 'text': {
              body = token.tokens ? this.parseInline(token.tokens) : token.text;
              while (i + 1 < l && tokens[i + 1].type === 'text') {
                token = tokens[++i];
                body += '\n' + (token.tokens ? this.parseInline(token.tokens) : token.text);
              }
              out += top ? this.renderer.paragraph(body) : body;
              continue;
            }
            default: {
              const errMsg = 'Token with "' + token.type + '" type was not found.';
              if (this.options.silent) {
                console.error(errMsg);
                return;
              } else {
                throw new Error(errMsg);
              }
            }
          }
        }

        return out;
      }

      /**
       * Parse Inline Tokens
       */
      parseInline(tokens, renderer) {
        renderer = renderer || this.renderer;
        let out = '',
          i,
          token;

        const l = tokens.length;
        for (i = 0; i < l; i++) {
          token = tokens[i];
          switch (token.type) {
            case 'escape': {
              out += renderer.text(token.text);
              break;
            }
            case 'html': {
              out += renderer.html(token.text);
              break;
            }
            case 'link': {
              out += renderer.link(token.href, token.title, this.parseInline(token.tokens, renderer));
              break;
            }
            case 'image': {
              out += renderer.image(token.href, token.title, token.text);
              break;
            }
            case 'strong': {
              out += renderer.strong(this.parseInline(token.tokens, renderer));
              break;
            }
            case 'em': {
              out += renderer.em(this.parseInline(token.tokens, renderer));
              break;
            }
            case 'codespan': {
              out += renderer.codespan(token.text);
              break;
            }
            case 'br': {
              out += renderer.br();
              break;
            }
            case 'del': {
              out += renderer.del(this.parseInline(token.tokens, renderer));
              break;
            }
            case 'text': {
              out += renderer.text(token.text);
              break;
            }
            default: {
              const errMsg = 'Token with "' + token.type + '" type was not found.';
              if (this.options.silent) {
                console.error(errMsg);
                return;
              } else {
                throw new Error(errMsg);
              }
            }
          }
        }
        return out;
      }
    };

    const {
      merge: merge$2,
      checkSanitizeDeprecation: checkSanitizeDeprecation$1,
      escape: escape$3
    } = helpers;
    const {
      getDefaults,
      changeDefaults,
      defaults: defaults$5
    } = defaults;

    /**
     * Marked
     */
    function marked(src, opt, callback) {
      // throw error in case of non string input
      if (typeof src === 'undefined' || src === null) {
        throw new Error('marked(): input parameter is undefined or null');
      }
      if (typeof src !== 'string') {
        throw new Error('marked(): input parameter is of type '
          + Object.prototype.toString.call(src) + ', string expected');
      }

      if (typeof opt === 'function') {
        callback = opt;
        opt = null;
      }

      opt = merge$2({}, marked.defaults, opt || {});
      checkSanitizeDeprecation$1(opt);

      if (callback) {
        const highlight = opt.highlight;
        let tokens;

        try {
          tokens = Lexer_1.lex(src, opt);
        } catch (e) {
          return callback(e);
        }

        const done = function(err) {
          let out;

          if (!err) {
            try {
              out = Parser_1.parse(tokens, opt);
            } catch (e) {
              err = e;
            }
          }

          opt.highlight = highlight;

          return err
            ? callback(err)
            : callback(null, out);
        };

        if (!highlight || highlight.length < 3) {
          return done();
        }

        delete opt.highlight;

        if (!tokens.length) return done();

        let pending = 0;
        marked.walkTokens(tokens, function(token) {
          if (token.type === 'code') {
            pending++;
            highlight(token.text, token.lang, function(err, code) {
              if (err) {
                return done(err);
              }
              if (code != null && code !== token.text) {
                token.text = code;
                token.escaped = true;
              }

              pending--;
              if (pending === 0) {
                done();
              }
            });
          }
        });

        if (pending === 0) {
          done();
        }

        return;
      }

      try {
        const tokens = Lexer_1.lex(src, opt);
        if (opt.walkTokens) {
          marked.walkTokens(tokens, opt.walkTokens);
        }
        return Parser_1.parse(tokens, opt);
      } catch (e) {
        e.message += '\nPlease report this to https://github.com/markedjs/marked.';
        if (opt.silent) {
          return '<p>An error occurred:</p><pre>'
            + escape$3(e.message + '', true)
            + '</pre>';
        }
        throw e;
      }
    }

    /**
     * Options
     */

    marked.options =
    marked.setOptions = function(opt) {
      merge$2(marked.defaults, opt);
      changeDefaults(marked.defaults);
      return marked;
    };

    marked.getDefaults = getDefaults;

    marked.defaults = defaults$5;

    /**
     * Use Extension
     */

    marked.use = function(extension) {
      const opts = merge$2({}, extension);
      if (extension.renderer) {
        const renderer = marked.defaults.renderer || new Renderer_1();
        for (const prop in extension.renderer) {
          const prevRenderer = renderer[prop];
          renderer[prop] = (...args) => {
            let ret = extension.renderer[prop].apply(renderer, args);
            if (ret === false) {
              ret = prevRenderer.apply(renderer, args);
            }
            return ret;
          };
        }
        opts.renderer = renderer;
      }
      if (extension.tokenizer) {
        const tokenizer = marked.defaults.tokenizer || new Tokenizer_1();
        for (const prop in extension.tokenizer) {
          const prevTokenizer = tokenizer[prop];
          tokenizer[prop] = (...args) => {
            let ret = extension.tokenizer[prop].apply(tokenizer, args);
            if (ret === false) {
              ret = prevTokenizer.apply(tokenizer, args);
            }
            return ret;
          };
        }
        opts.tokenizer = tokenizer;
      }
      if (extension.walkTokens) {
        const walkTokens = marked.defaults.walkTokens;
        opts.walkTokens = (token) => {
          extension.walkTokens(token);
          if (walkTokens) {
            walkTokens(token);
          }
        };
      }
      marked.setOptions(opts);
    };

    /**
     * Run callback for every token
     */

    marked.walkTokens = function(tokens, callback) {
      for (const token of tokens) {
        callback(token);
        switch (token.type) {
          case 'table': {
            for (const cell of token.tokens.header) {
              marked.walkTokens(cell, callback);
            }
            for (const row of token.tokens.cells) {
              for (const cell of row) {
                marked.walkTokens(cell, callback);
              }
            }
            break;
          }
          case 'list': {
            marked.walkTokens(token.items, callback);
            break;
          }
          default: {
            if (token.tokens) {
              marked.walkTokens(token.tokens, callback);
            }
          }
        }
      }
    };

    /**
     * Expose
     */

    marked.Parser = Parser_1;
    marked.parser = Parser_1.parse;

    marked.Renderer = Renderer_1;
    marked.TextRenderer = TextRenderer_1;

    marked.Lexer = Lexer_1;
    marked.lexer = Lexer_1.lex;

    marked.Tokenizer = Tokenizer_1;

    marked.Slugger = Slugger_1;

    marked.parse = marked;

    var marked_1 = marked;

    /*! @license DOMPurify | (c) Cure53 and other contributors | Released under the Apache license 2.0 and Mozilla Public License 2.0 | github.com/cure53/DOMPurify/blob/2.0.8/LICENSE */

    function _toConsumableArray(arr) { if (Array.isArray(arr)) { for (var i = 0, arr2 = Array(arr.length); i < arr.length; i++) { arr2[i] = arr[i]; } return arr2; } else { return Array.from(arr); } }

    var hasOwnProperty$1 = Object.hasOwnProperty,
        setPrototypeOf = Object.setPrototypeOf,
        isFrozen = Object.isFrozen,
        objectKeys = Object.keys;
    var freeze = Object.freeze,
        seal = Object.seal; // eslint-disable-line import/no-mutable-exports

    var _ref = typeof Reflect !== 'undefined' && Reflect,
        apply = _ref.apply,
        construct = _ref.construct;

    if (!apply) {
      apply = function apply(fun, thisValue, args) {
        return fun.apply(thisValue, args);
      };
    }

    if (!freeze) {
      freeze = function freeze(x) {
        return x;
      };
    }

    if (!seal) {
      seal = function seal(x) {
        return x;
      };
    }

    if (!construct) {
      construct = function construct(Func, args) {
        return new (Function.prototype.bind.apply(Func, [null].concat(_toConsumableArray(args))))();
      };
    }

    var arrayForEach = unapply(Array.prototype.forEach);
    var arrayIndexOf = unapply(Array.prototype.indexOf);
    var arrayJoin = unapply(Array.prototype.join);
    var arrayPop = unapply(Array.prototype.pop);
    var arrayPush = unapply(Array.prototype.push);
    var arraySlice = unapply(Array.prototype.slice);

    var stringToLowerCase = unapply(String.prototype.toLowerCase);
    var stringMatch = unapply(String.prototype.match);
    var stringReplace = unapply(String.prototype.replace);
    var stringIndexOf = unapply(String.prototype.indexOf);
    var stringTrim = unapply(String.prototype.trim);

    var regExpTest = unapply(RegExp.prototype.test);
    var regExpCreate = unconstruct(RegExp);

    var typeErrorCreate = unconstruct(TypeError);

    function unapply(func) {
      return function (thisArg) {
        for (var _len = arguments.length, args = Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
          args[_key - 1] = arguments[_key];
        }

        return apply(func, thisArg, args);
      };
    }

    function unconstruct(func) {
      return function () {
        for (var _len2 = arguments.length, args = Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
          args[_key2] = arguments[_key2];
        }

        return construct(func, args);
      };
    }

    /* Add properties to a lookup table */
    function addToSet(set, array) {
      if (setPrototypeOf) {
        // Make 'in' and truthy checks like Boolean(set.constructor)
        // independent of any properties defined on Object.prototype.
        // Prevent prototype setters from intercepting set as a this value.
        setPrototypeOf(set, null);
      }

      var l = array.length;
      while (l--) {
        var element = array[l];
        if (typeof element === 'string') {
          var lcElement = stringToLowerCase(element);
          if (lcElement !== element) {
            // Config presets (e.g. tags.js, attrs.js) are immutable.
            if (!isFrozen(array)) {
              array[l] = lcElement;
            }

            element = lcElement;
          }
        }

        set[element] = true;
      }

      return set;
    }

    /* Shallow clone an object */
    function clone(object) {
      var newObject = {};

      var property = void 0;
      for (property in object) {
        if (apply(hasOwnProperty$1, object, [property])) {
          newObject[property] = object[property];
        }
      }

      return newObject;
    }

    var html$1 = freeze(['a', 'abbr', 'acronym', 'address', 'area', 'article', 'aside', 'audio', 'b', 'bdi', 'bdo', 'big', 'blink', 'blockquote', 'body', 'br', 'button', 'canvas', 'caption', 'center', 'cite', 'code', 'col', 'colgroup', 'content', 'data', 'datalist', 'dd', 'decorator', 'del', 'details', 'dfn', 'dir', 'div', 'dl', 'dt', 'element', 'em', 'fieldset', 'figcaption', 'figure', 'font', 'footer', 'form', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'head', 'header', 'hgroup', 'hr', 'html', 'i', 'img', 'input', 'ins', 'kbd', 'label', 'legend', 'li', 'main', 'map', 'mark', 'marquee', 'menu', 'menuitem', 'meter', 'nav', 'nobr', 'ol', 'optgroup', 'option', 'output', 'p', 'picture', 'pre', 'progress', 'q', 'rp', 'rt', 'ruby', 's', 'samp', 'section', 'select', 'shadow', 'small', 'source', 'spacer', 'span', 'strike', 'strong', 'style', 'sub', 'summary', 'sup', 'table', 'tbody', 'td', 'template', 'textarea', 'tfoot', 'th', 'thead', 'time', 'tr', 'track', 'tt', 'u', 'ul', 'var', 'video', 'wbr']);

    // SVG
    var svg = freeze(['svg', 'a', 'altglyph', 'altglyphdef', 'altglyphitem', 'animatecolor', 'animatemotion', 'animatetransform', 'audio', 'canvas', 'circle', 'clippath', 'defs', 'desc', 'ellipse', 'filter', 'font', 'g', 'glyph', 'glyphref', 'hkern', 'image', 'line', 'lineargradient', 'marker', 'mask', 'metadata', 'mpath', 'path', 'pattern', 'polygon', 'polyline', 'radialgradient', 'rect', 'stop', 'style', 'switch', 'symbol', 'text', 'textpath', 'title', 'tref', 'tspan', 'video', 'view', 'vkern']);

    var svgFilters = freeze(['feBlend', 'feColorMatrix', 'feComponentTransfer', 'feComposite', 'feConvolveMatrix', 'feDiffuseLighting', 'feDisplacementMap', 'feDistantLight', 'feFlood', 'feFuncA', 'feFuncB', 'feFuncG', 'feFuncR', 'feGaussianBlur', 'feMerge', 'feMergeNode', 'feMorphology', 'feOffset', 'fePointLight', 'feSpecularLighting', 'feSpotLight', 'feTile', 'feTurbulence']);

    var mathMl = freeze(['math', 'menclose', 'merror', 'mfenced', 'mfrac', 'mglyph', 'mi', 'mlabeledtr', 'mmultiscripts', 'mn', 'mo', 'mover', 'mpadded', 'mphantom', 'mroot', 'mrow', 'ms', 'mspace', 'msqrt', 'mstyle', 'msub', 'msup', 'msubsup', 'mtable', 'mtd', 'mtext', 'mtr', 'munder', 'munderover']);

    var text = freeze(['#text']);

    var html$1$1 = freeze(['accept', 'action', 'align', 'alt', 'autocapitalize', 'autocomplete', 'autopictureinpicture', 'autoplay', 'background', 'bgcolor', 'border', 'capture', 'cellpadding', 'cellspacing', 'checked', 'cite', 'class', 'clear', 'color', 'cols', 'colspan', 'controls', 'controlslist', 'coords', 'crossorigin', 'datetime', 'decoding', 'default', 'dir', 'disabled', 'disablepictureinpicture', 'disableremoteplayback', 'download', 'draggable', 'enctype', 'enterkeyhint', 'face', 'for', 'headers', 'height', 'hidden', 'high', 'href', 'hreflang', 'id', 'inputmode', 'integrity', 'ismap', 'kind', 'label', 'lang', 'list', 'loading', 'loop', 'low', 'max', 'maxlength', 'media', 'method', 'min', 'minlength', 'multiple', 'muted', 'name', 'noshade', 'novalidate', 'nowrap', 'open', 'optimum', 'pattern', 'placeholder', 'playsinline', 'poster', 'preload', 'pubdate', 'radiogroup', 'readonly', 'rel', 'required', 'rev', 'reversed', 'role', 'rows', 'rowspan', 'spellcheck', 'scope', 'selected', 'shape', 'size', 'sizes', 'span', 'srclang', 'start', 'src', 'srcset', 'step', 'style', 'summary', 'tabindex', 'title', 'translate', 'type', 'usemap', 'valign', 'value', 'width', 'xmlns']);

    var svg$1 = freeze(['accent-height', 'accumulate', 'additive', 'alignment-baseline', 'ascent', 'attributename', 'attributetype', 'azimuth', 'basefrequency', 'baseline-shift', 'begin', 'bias', 'by', 'class', 'clip', 'clip-path', 'clip-rule', 'color', 'color-interpolation', 'color-interpolation-filters', 'color-profile', 'color-rendering', 'cx', 'cy', 'd', 'dx', 'dy', 'diffuseconstant', 'direction', 'display', 'divisor', 'dur', 'edgemode', 'elevation', 'end', 'fill', 'fill-opacity', 'fill-rule', 'filter', 'filterunits', 'flood-color', 'flood-opacity', 'font-family', 'font-size', 'font-size-adjust', 'font-stretch', 'font-style', 'font-variant', 'font-weight', 'fx', 'fy', 'g1', 'g2', 'glyph-name', 'glyphref', 'gradientunits', 'gradienttransform', 'height', 'href', 'id', 'image-rendering', 'in', 'in2', 'k', 'k1', 'k2', 'k3', 'k4', 'kerning', 'keypoints', 'keysplines', 'keytimes', 'lang', 'lengthadjust', 'letter-spacing', 'kernelmatrix', 'kernelunitlength', 'lighting-color', 'local', 'marker-end', 'marker-mid', 'marker-start', 'markerheight', 'markerunits', 'markerwidth', 'maskcontentunits', 'maskunits', 'max', 'mask', 'media', 'method', 'mode', 'min', 'name', 'numoctaves', 'offset', 'operator', 'opacity', 'order', 'orient', 'orientation', 'origin', 'overflow', 'paint-order', 'path', 'pathlength', 'patterncontentunits', 'patterntransform', 'patternunits', 'points', 'preservealpha', 'preserveaspectratio', 'primitiveunits', 'r', 'rx', 'ry', 'radius', 'refx', 'refy', 'repeatcount', 'repeatdur', 'restart', 'result', 'rotate', 'scale', 'seed', 'shape-rendering', 'specularconstant', 'specularexponent', 'spreadmethod', 'startoffset', 'stddeviation', 'stitchtiles', 'stop-color', 'stop-opacity', 'stroke-dasharray', 'stroke-dashoffset', 'stroke-linecap', 'stroke-linejoin', 'stroke-miterlimit', 'stroke-opacity', 'stroke', 'stroke-width', 'style', 'surfacescale', 'tabindex', 'targetx', 'targety', 'transform', 'text-anchor', 'text-decoration', 'text-rendering', 'textlength', 'type', 'u1', 'u2', 'unicode', 'values', 'viewbox', 'visibility', 'version', 'vert-adv-y', 'vert-origin-x', 'vert-origin-y', 'width', 'word-spacing', 'wrap', 'writing-mode', 'xchannelselector', 'ychannelselector', 'x', 'x1', 'x2', 'xmlns', 'y', 'y1', 'y2', 'z', 'zoomandpan']);

    var mathMl$1 = freeze(['accent', 'accentunder', 'align', 'bevelled', 'close', 'columnsalign', 'columnlines', 'columnspan', 'denomalign', 'depth', 'dir', 'display', 'displaystyle', 'encoding', 'fence', 'frame', 'height', 'href', 'id', 'largeop', 'length', 'linethickness', 'lspace', 'lquote', 'mathbackground', 'mathcolor', 'mathsize', 'mathvariant', 'maxsize', 'minsize', 'movablelimits', 'notation', 'numalign', 'open', 'rowalign', 'rowlines', 'rowspacing', 'rowspan', 'rspace', 'rquote', 'scriptlevel', 'scriptminsize', 'scriptsizemultiplier', 'selection', 'separator', 'separators', 'stretchy', 'subscriptshift', 'supscriptshift', 'symmetric', 'voffset', 'width', 'xmlns']);

    var xml = freeze(['xlink:href', 'xml:id', 'xlink:title', 'xml:space', 'xmlns:xlink']);

    // eslint-disable-next-line unicorn/better-regex
    var MUSTACHE_EXPR = seal(/\{\{[\s\S]*|[\s\S]*\}\}/gm); // Specify template detection regex for SAFE_FOR_TEMPLATES mode
    var ERB_EXPR = seal(/<%[\s\S]*|[\s\S]*%>/gm);
    var DATA_ATTR = seal(/^data-[\-\w.\u00B7-\uFFFF]/); // eslint-disable-line no-useless-escape
    var ARIA_ATTR = seal(/^aria-[\-\w]+$/); // eslint-disable-line no-useless-escape
    var IS_ALLOWED_URI = seal(/^(?:(?:(?:f|ht)tps?|mailto|tel|callto|cid|xmpp):|[^a-z]|[a-z+.\-]+(?:[^a-z+.\-:]|$))/i // eslint-disable-line no-useless-escape
    );
    var IS_SCRIPT_OR_DATA = seal(/^(?:\w+script|data):/i);
    var ATTR_WHITESPACE = seal(/[\u0000-\u0020\u00A0\u1680\u180E\u2000-\u2029\u205f\u3000]/g // eslint-disable-line no-control-regex
    );

    var _typeof = typeof Symbol === "function" && typeof Symbol.iterator === "symbol" ? function (obj) { return typeof obj; } : function (obj) { return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj; };

    function _toConsumableArray$1(arr) { if (Array.isArray(arr)) { for (var i = 0, arr2 = Array(arr.length); i < arr.length; i++) { arr2[i] = arr[i]; } return arr2; } else { return Array.from(arr); } }

    var getGlobal = function getGlobal() {
      return typeof window === 'undefined' ? null : window;
    };

    /**
     * Creates a no-op policy for internal use only.
     * Don't export this function outside this module!
     * @param {?TrustedTypePolicyFactory} trustedTypes The policy factory.
     * @param {Document} document The document object (to determine policy name suffix)
     * @return {?TrustedTypePolicy} The policy created (or null, if Trusted Types
     * are not supported).
     */
    var _createTrustedTypesPolicy = function _createTrustedTypesPolicy(trustedTypes, document) {
      if ((typeof trustedTypes === 'undefined' ? 'undefined' : _typeof(trustedTypes)) !== 'object' || typeof trustedTypes.createPolicy !== 'function') {
        return null;
      }

      // Allow the callers to control the unique policy name
      // by adding a data-tt-policy-suffix to the script element with the DOMPurify.
      // Policy creation with duplicate names throws in Trusted Types.
      var suffix = null;
      var ATTR_NAME = 'data-tt-policy-suffix';
      if (document.currentScript && document.currentScript.hasAttribute(ATTR_NAME)) {
        suffix = document.currentScript.getAttribute(ATTR_NAME);
      }

      var policyName = 'dompurify' + (suffix ? '#' + suffix : '');

      try {
        return trustedTypes.createPolicy(policyName, {
          createHTML: function createHTML(html$$1) {
            return html$$1;
          }
        });
      } catch (_) {
        // Policy creation failed (most likely another DOMPurify script has
        // already run). Skip creating the policy, as this will only cause errors
        // if TT are enforced.
        console.warn('TrustedTypes policy ' + policyName + ' could not be created.');
        return null;
      }
    };

    function createDOMPurify() {
      var window = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : getGlobal();

      var DOMPurify = function DOMPurify(root) {
        return createDOMPurify(root);
      };

      /**
       * Version label, exposed for easier checks
       * if DOMPurify is up to date or not
       */
      DOMPurify.version = '2.0.11';

      /**
       * Array of elements that DOMPurify removed during sanitation.
       * Empty if nothing was removed.
       */
      DOMPurify.removed = [];

      if (!window || !window.document || window.document.nodeType !== 9) {
        // Not running in a browser, provide a factory function
        // so that you can pass your own Window
        DOMPurify.isSupported = false;

        return DOMPurify;
      }

      var originalDocument = window.document;
      var removeTitle = false;

      var document = window.document;
      var DocumentFragment = window.DocumentFragment,
          HTMLTemplateElement = window.HTMLTemplateElement,
          Node = window.Node,
          NodeFilter = window.NodeFilter,
          _window$NamedNodeMap = window.NamedNodeMap,
          NamedNodeMap = _window$NamedNodeMap === undefined ? window.NamedNodeMap || window.MozNamedAttrMap : _window$NamedNodeMap,
          Text = window.Text,
          Comment = window.Comment,
          DOMParser = window.DOMParser,
          trustedTypes = window.trustedTypes;

      // As per issue #47, the web-components registry is inherited by a
      // new document created via createHTMLDocument. As per the spec
      // (http://w3c.github.io/webcomponents/spec/custom/#creating-and-passing-registries)
      // a new empty registry is used when creating a template contents owner
      // document, so we use that as our parent document to ensure nothing
      // is inherited.

      if (typeof HTMLTemplateElement === 'function') {
        var template = document.createElement('template');
        if (template.content && template.content.ownerDocument) {
          document = template.content.ownerDocument;
        }
      }

      var trustedTypesPolicy = _createTrustedTypesPolicy(trustedTypes, originalDocument);
      var emptyHTML = trustedTypesPolicy ? trustedTypesPolicy.createHTML('') : '';

      var _document = document,
          implementation = _document.implementation,
          createNodeIterator = _document.createNodeIterator,
          getElementsByTagName = _document.getElementsByTagName,
          createDocumentFragment = _document.createDocumentFragment;
      var importNode = originalDocument.importNode;


      var hooks = {};

      /**
       * Expose whether this browser supports running the full DOMPurify.
       */
      DOMPurify.isSupported = implementation && typeof implementation.createHTMLDocument !== 'undefined' && document.documentMode !== 9;

      var MUSTACHE_EXPR$$1 = MUSTACHE_EXPR,
          ERB_EXPR$$1 = ERB_EXPR,
          DATA_ATTR$$1 = DATA_ATTR,
          ARIA_ATTR$$1 = ARIA_ATTR,
          IS_SCRIPT_OR_DATA$$1 = IS_SCRIPT_OR_DATA,
          ATTR_WHITESPACE$$1 = ATTR_WHITESPACE;
      var IS_ALLOWED_URI$$1 = IS_ALLOWED_URI;

      /**
       * We consider the elements and attributes below to be safe. Ideally
       * don't add any new ones but feel free to remove unwanted ones.
       */

      /* allowed element names */

      var ALLOWED_TAGS = null;
      var DEFAULT_ALLOWED_TAGS = addToSet({}, [].concat(_toConsumableArray$1(html$1), _toConsumableArray$1(svg), _toConsumableArray$1(svgFilters), _toConsumableArray$1(mathMl), _toConsumableArray$1(text)));

      /* Allowed attribute names */
      var ALLOWED_ATTR = null;
      var DEFAULT_ALLOWED_ATTR = addToSet({}, [].concat(_toConsumableArray$1(html$1$1), _toConsumableArray$1(svg$1), _toConsumableArray$1(mathMl$1), _toConsumableArray$1(xml)));

      /* Explicitly forbidden tags (overrides ALLOWED_TAGS/ADD_TAGS) */
      var FORBID_TAGS = null;

      /* Explicitly forbidden attributes (overrides ALLOWED_ATTR/ADD_ATTR) */
      var FORBID_ATTR = null;

      /* Decide if ARIA attributes are okay */
      var ALLOW_ARIA_ATTR = true;

      /* Decide if custom data attributes are okay */
      var ALLOW_DATA_ATTR = true;

      /* Decide if unknown protocols are okay */
      var ALLOW_UNKNOWN_PROTOCOLS = false;

      /* Output should be safe for jQuery's $() factory? */
      var SAFE_FOR_JQUERY = false;

      /* Output should be safe for common template engines.
       * This means, DOMPurify removes data attributes, mustaches and ERB
       */
      var SAFE_FOR_TEMPLATES = false;

      /* Decide if document with <html>... should be returned */
      var WHOLE_DOCUMENT = false;

      /* Track whether config is already set on this instance of DOMPurify. */
      var SET_CONFIG = false;

      /* Decide if all elements (e.g. style, script) must be children of
       * document.body. By default, browsers might move them to document.head */
      var FORCE_BODY = false;

      /* Decide if a DOM `HTMLBodyElement` should be returned, instead of a html
       * string (or a TrustedHTML object if Trusted Types are supported).
       * If `WHOLE_DOCUMENT` is enabled a `HTMLHtmlElement` will be returned instead
       */
      var RETURN_DOM = false;

      /* Decide if a DOM `DocumentFragment` should be returned, instead of a html
       * string  (or a TrustedHTML object if Trusted Types are supported) */
      var RETURN_DOM_FRAGMENT = false;

      /* If `RETURN_DOM` or `RETURN_DOM_FRAGMENT` is enabled, decide if the returned DOM
       * `Node` is imported into the current `Document`. If this flag is not enabled the
       * `Node` will belong (its ownerDocument) to a fresh `HTMLDocument`, created by
       * DOMPurify. */
      var RETURN_DOM_IMPORT = false;

      /* Try to return a Trusted Type object instead of a string, retrun a string in
       * case Trusted Types are not supported  */
      var RETURN_TRUSTED_TYPE = false;

      /* Output should be free from DOM clobbering attacks? */
      var SANITIZE_DOM = true;

      /* Keep element content when removing element? */
      var KEEP_CONTENT = true;

      /* If a `Node` is passed to sanitize(), then performs sanitization in-place instead
       * of importing it into a new Document and returning a sanitized copy */
      var IN_PLACE = false;

      /* Allow usage of profiles like html, svg and mathMl */
      var USE_PROFILES = {};

      /* Tags to ignore content of when KEEP_CONTENT is true */
      var FORBID_CONTENTS = addToSet({}, ['annotation-xml', 'audio', 'colgroup', 'desc', 'foreignobject', 'head', 'iframe', 'math', 'mi', 'mn', 'mo', 'ms', 'mtext', 'noembed', 'noframes', 'plaintext', 'script', 'style', 'svg', 'template', 'thead', 'title', 'video', 'xmp']);

      /* Tags that are safe for data: URIs */
      var DATA_URI_TAGS = null;
      var DEFAULT_DATA_URI_TAGS = addToSet({}, ['audio', 'video', 'img', 'source', 'image', 'track']);

      /* Attributes safe for values like "javascript:" */
      var URI_SAFE_ATTRIBUTES = null;
      var DEFAULT_URI_SAFE_ATTRIBUTES = addToSet({}, ['alt', 'class', 'for', 'id', 'label', 'name', 'pattern', 'placeholder', 'summary', 'title', 'value', 'style', 'xmlns']);

      /* Keep a reference to config to pass to hooks */
      var CONFIG = null;

      /* Ideally, do not touch anything below this line */
      /* ______________________________________________ */

      var formElement = document.createElement('form');

      /**
       * _parseConfig
       *
       * @param  {Object} cfg optional config literal
       */
      // eslint-disable-next-line complexity
      var _parseConfig = function _parseConfig(cfg) {
        if (CONFIG && CONFIG === cfg) {
          return;
        }

        /* Shield configuration object from tampering */
        if (!cfg || (typeof cfg === 'undefined' ? 'undefined' : _typeof(cfg)) !== 'object') {
          cfg = {};
        }

        /* Set configuration parameters */
        ALLOWED_TAGS = 'ALLOWED_TAGS' in cfg ? addToSet({}, cfg.ALLOWED_TAGS) : DEFAULT_ALLOWED_TAGS;
        ALLOWED_ATTR = 'ALLOWED_ATTR' in cfg ? addToSet({}, cfg.ALLOWED_ATTR) : DEFAULT_ALLOWED_ATTR;
        URI_SAFE_ATTRIBUTES = 'ADD_URI_SAFE_ATTR' in cfg ? addToSet(clone(DEFAULT_URI_SAFE_ATTRIBUTES), cfg.ADD_URI_SAFE_ATTR) : DEFAULT_URI_SAFE_ATTRIBUTES;
        DATA_URI_TAGS = 'ADD_DATA_URI_TAGS' in cfg ? addToSet(clone(DEFAULT_DATA_URI_TAGS), cfg.ADD_DATA_URI_TAGS) : DEFAULT_DATA_URI_TAGS;
        FORBID_TAGS = 'FORBID_TAGS' in cfg ? addToSet({}, cfg.FORBID_TAGS) : {};
        FORBID_ATTR = 'FORBID_ATTR' in cfg ? addToSet({}, cfg.FORBID_ATTR) : {};
        USE_PROFILES = 'USE_PROFILES' in cfg ? cfg.USE_PROFILES : false;
        ALLOW_ARIA_ATTR = cfg.ALLOW_ARIA_ATTR !== false; // Default true
        ALLOW_DATA_ATTR = cfg.ALLOW_DATA_ATTR !== false; // Default true
        ALLOW_UNKNOWN_PROTOCOLS = cfg.ALLOW_UNKNOWN_PROTOCOLS || false; // Default false
        SAFE_FOR_JQUERY = cfg.SAFE_FOR_JQUERY || false; // Default false
        SAFE_FOR_TEMPLATES = cfg.SAFE_FOR_TEMPLATES || false; // Default false
        WHOLE_DOCUMENT = cfg.WHOLE_DOCUMENT || false; // Default false
        RETURN_DOM = cfg.RETURN_DOM || false; // Default false
        RETURN_DOM_FRAGMENT = cfg.RETURN_DOM_FRAGMENT || false; // Default false
        RETURN_DOM_IMPORT = cfg.RETURN_DOM_IMPORT || false; // Default false
        RETURN_TRUSTED_TYPE = cfg.RETURN_TRUSTED_TYPE || false; // Default false
        FORCE_BODY = cfg.FORCE_BODY || false; // Default false
        SANITIZE_DOM = cfg.SANITIZE_DOM !== false; // Default true
        KEEP_CONTENT = cfg.KEEP_CONTENT !== false; // Default true
        IN_PLACE = cfg.IN_PLACE || false; // Default false
        IS_ALLOWED_URI$$1 = cfg.ALLOWED_URI_REGEXP || IS_ALLOWED_URI$$1;
        if (SAFE_FOR_TEMPLATES) {
          ALLOW_DATA_ATTR = false;
        }

        if (RETURN_DOM_FRAGMENT) {
          RETURN_DOM = true;
        }

        /* Parse profile info */
        if (USE_PROFILES) {
          ALLOWED_TAGS = addToSet({}, [].concat(_toConsumableArray$1(text)));
          ALLOWED_ATTR = [];
          if (USE_PROFILES.html === true) {
            addToSet(ALLOWED_TAGS, html$1);
            addToSet(ALLOWED_ATTR, html$1$1);
          }

          if (USE_PROFILES.svg === true) {
            addToSet(ALLOWED_TAGS, svg);
            addToSet(ALLOWED_ATTR, svg$1);
            addToSet(ALLOWED_ATTR, xml);
          }

          if (USE_PROFILES.svgFilters === true) {
            addToSet(ALLOWED_TAGS, svgFilters);
            addToSet(ALLOWED_ATTR, svg$1);
            addToSet(ALLOWED_ATTR, xml);
          }

          if (USE_PROFILES.mathMl === true) {
            addToSet(ALLOWED_TAGS, mathMl);
            addToSet(ALLOWED_ATTR, mathMl$1);
            addToSet(ALLOWED_ATTR, xml);
          }
        }

        /* Merge configuration parameters */
        if (cfg.ADD_TAGS) {
          if (ALLOWED_TAGS === DEFAULT_ALLOWED_TAGS) {
            ALLOWED_TAGS = clone(ALLOWED_TAGS);
          }

          addToSet(ALLOWED_TAGS, cfg.ADD_TAGS);
        }

        if (cfg.ADD_ATTR) {
          if (ALLOWED_ATTR === DEFAULT_ALLOWED_ATTR) {
            ALLOWED_ATTR = clone(ALLOWED_ATTR);
          }

          addToSet(ALLOWED_ATTR, cfg.ADD_ATTR);
        }

        if (cfg.ADD_URI_SAFE_ATTR) {
          addToSet(URI_SAFE_ATTRIBUTES, cfg.ADD_URI_SAFE_ATTR);
        }

        /* Add #text in case KEEP_CONTENT is set to true */
        if (KEEP_CONTENT) {
          ALLOWED_TAGS['#text'] = true;
        }

        /* Add html, head and body to ALLOWED_TAGS in case WHOLE_DOCUMENT is true */
        if (WHOLE_DOCUMENT) {
          addToSet(ALLOWED_TAGS, ['html', 'head', 'body']);
        }

        /* Add tbody to ALLOWED_TAGS in case tables are permitted, see #286, #365 */
        if (ALLOWED_TAGS.table) {
          addToSet(ALLOWED_TAGS, ['tbody']);
          delete FORBID_TAGS.tbody;
        }

        // Prevent further manipulation of configuration.
        // Not available in IE8, Safari 5, etc.
        if (freeze) {
          freeze(cfg);
        }

        CONFIG = cfg;
      };

      /**
       * _forceRemove
       *
       * @param  {Node} node a DOM node
       */
      var _forceRemove = function _forceRemove(node) {
        arrayPush(DOMPurify.removed, { element: node });
        try {
          // eslint-disable-next-line unicorn/prefer-node-remove
          node.parentNode.removeChild(node);
        } catch (_) {
          node.outerHTML = emptyHTML;
        }
      };

      /**
       * _removeAttribute
       *
       * @param  {String} name an Attribute name
       * @param  {Node} node a DOM node
       */
      var _removeAttribute = function _removeAttribute(name, node) {
        try {
          arrayPush(DOMPurify.removed, {
            attribute: node.getAttributeNode(name),
            from: node
          });
        } catch (_) {
          arrayPush(DOMPurify.removed, {
            attribute: null,
            from: node
          });
        }

        node.removeAttribute(name);
      };

      /**
       * _initDocument
       *
       * @param  {String} dirty a string of dirty markup
       * @return {Document} a DOM, filled with the dirty markup
       */
      var _initDocument = function _initDocument(dirty) {
        /* Create a HTML document */
        var doc = void 0;
        var leadingWhitespace = void 0;

        if (FORCE_BODY) {
          dirty = '<remove></remove>' + dirty;
        } else {
          /* If FORCE_BODY isn't used, leading whitespace needs to be preserved manually */
          var matches = stringMatch(dirty, /^[\r\n\t ]+/);
          leadingWhitespace = matches && matches[0];
        }

        var dirtyPayload = trustedTypesPolicy ? trustedTypesPolicy.createHTML(dirty) : dirty;
        /* Use the DOMParser API by default, fallback later if needs be */
        try {
          doc = new DOMParser().parseFromString(dirtyPayload, 'text/html');
        } catch (_) {}

        /* Remove title to fix a mXSS bug in older MS Edge */
        if (removeTitle) {
          addToSet(FORBID_TAGS, ['title']);
        }

        /* Use createHTMLDocument in case DOMParser is not available */
        if (!doc || !doc.documentElement) {
          doc = implementation.createHTMLDocument('');
          var _doc = doc,
              body = _doc.body;

          body.parentNode.removeChild(body.parentNode.firstElementChild);
          body.outerHTML = dirtyPayload;
        }

        if (dirty && leadingWhitespace) {
          doc.body.insertBefore(document.createTextNode(leadingWhitespace), doc.body.childNodes[0] || null);
        }

        /* Work on whole document or just its body */
        return getElementsByTagName.call(doc, WHOLE_DOCUMENT ? 'html' : 'body')[0];
      };

      /* Here we test for a broken feature in Edge that might cause mXSS */
      if (DOMPurify.isSupported) {
        (function () {
          try {
            var doc = _initDocument('<x/><title>&lt;/title&gt;&lt;img&gt;');
            if (regExpTest(/<\/title/, doc.querySelector('title').innerHTML)) {
              removeTitle = true;
            }
          } catch (_) {}
        })();
      }

      /**
       * _createIterator
       *
       * @param  {Document} root document/fragment to create iterator for
       * @return {Iterator} iterator instance
       */
      var _createIterator = function _createIterator(root) {
        return createNodeIterator.call(root.ownerDocument || root, root, NodeFilter.SHOW_ELEMENT | NodeFilter.SHOW_COMMENT | NodeFilter.SHOW_TEXT, function () {
          return NodeFilter.FILTER_ACCEPT;
        }, false);
      };

      /**
       * _isClobbered
       *
       * @param  {Node} elm element to check for clobbering attacks
       * @return {Boolean} true if clobbered, false if safe
       */
      var _isClobbered = function _isClobbered(elm) {
        if (elm instanceof Text || elm instanceof Comment) {
          return false;
        }

        if (typeof elm.nodeName !== 'string' || typeof elm.textContent !== 'string' || typeof elm.removeChild !== 'function' || !(elm.attributes instanceof NamedNodeMap) || typeof elm.removeAttribute !== 'function' || typeof elm.setAttribute !== 'function' || typeof elm.namespaceURI !== 'string') {
          return true;
        }

        return false;
      };

      /**
       * _isNode
       *
       * @param  {Node} obj object to check whether it's a DOM node
       * @return {Boolean} true is object is a DOM node
       */
      var _isNode = function _isNode(object) {
        return (typeof Node === 'undefined' ? 'undefined' : _typeof(Node)) === 'object' ? object instanceof Node : object && (typeof object === 'undefined' ? 'undefined' : _typeof(object)) === 'object' && typeof object.nodeType === 'number' && typeof object.nodeName === 'string';
      };

      /**
       * _executeHook
       * Execute user configurable hooks
       *
       * @param  {String} entryPoint  Name of the hook's entry point
       * @param  {Node} currentNode node to work on with the hook
       * @param  {Object} data additional hook parameters
       */
      var _executeHook = function _executeHook(entryPoint, currentNode, data) {
        if (!hooks[entryPoint]) {
          return;
        }

        arrayForEach(hooks[entryPoint], function (hook) {
          hook.call(DOMPurify, currentNode, data, CONFIG);
        });
      };

      /**
       * _sanitizeElements
       *
       * @protect nodeName
       * @protect textContent
       * @protect removeChild
       *
       * @param   {Node} currentNode to check for permission to exist
       * @return  {Boolean} true if node was killed, false if left alive
       */
      // eslint-disable-next-line complexity
      var _sanitizeElements = function _sanitizeElements(currentNode) {
        var content = void 0;

        /* Execute a hook if present */
        _executeHook('beforeSanitizeElements', currentNode, null);

        /* Check if element is clobbered or can clobber */
        if (_isClobbered(currentNode)) {
          _forceRemove(currentNode);
          return true;
        }

        /* Now let's check the element's type and name */
        var tagName = stringToLowerCase(currentNode.nodeName);

        /* Execute a hook if present */
        _executeHook('uponSanitizeElement', currentNode, {
          tagName: tagName,
          allowedTags: ALLOWED_TAGS
        });

        /* Take care of an mXSS pattern using p, br inside svg, math */
        if ((tagName === 'svg' || tagName === 'math') && currentNode.querySelectorAll('p, br').length !== 0) {
          _forceRemove(currentNode);
          return true;
        }

        /* Remove element if anything forbids its presence */
        if (!ALLOWED_TAGS[tagName] || FORBID_TAGS[tagName]) {
          /* Keep content except for black-listed elements */
          if (KEEP_CONTENT && !FORBID_CONTENTS[tagName] && typeof currentNode.insertAdjacentHTML === 'function') {
            try {
              var htmlToInsert = currentNode.innerHTML;
              currentNode.insertAdjacentHTML('AfterEnd', trustedTypesPolicy ? trustedTypesPolicy.createHTML(htmlToInsert) : htmlToInsert);
            } catch (_) {}
          }

          _forceRemove(currentNode);
          return true;
        }

        /* Remove in case a noscript/noembed XSS is suspected */
        if (tagName === 'noscript' && regExpTest(/<\/noscript/i, currentNode.innerHTML)) {
          _forceRemove(currentNode);
          return true;
        }

        if (tagName === 'noembed' && regExpTest(/<\/noembed/i, currentNode.innerHTML)) {
          _forceRemove(currentNode);
          return true;
        }

        /* Convert markup to cover jQuery behavior */
        if (SAFE_FOR_JQUERY && !currentNode.firstElementChild && (!currentNode.content || !currentNode.content.firstElementChild) && regExpTest(/</g, currentNode.textContent)) {
          arrayPush(DOMPurify.removed, { element: currentNode.cloneNode() });
          if (currentNode.innerHTML) {
            currentNode.innerHTML = stringReplace(currentNode.innerHTML, /</g, '&lt;');
          } else {
            currentNode.innerHTML = stringReplace(currentNode.textContent, /</g, '&lt;');
          }
        }

        /* Sanitize element content to be template-safe */
        if (SAFE_FOR_TEMPLATES && currentNode.nodeType === 3) {
          /* Get the element's text content */
          content = currentNode.textContent;
          content = stringReplace(content, MUSTACHE_EXPR$$1, ' ');
          content = stringReplace(content, ERB_EXPR$$1, ' ');
          if (currentNode.textContent !== content) {
            arrayPush(DOMPurify.removed, { element: currentNode.cloneNode() });
            currentNode.textContent = content;
          }
        }

        /* Execute a hook if present */
        _executeHook('afterSanitizeElements', currentNode, null);

        return false;
      };

      /**
       * _isValidAttribute
       *
       * @param  {string} lcTag Lowercase tag name of containing element.
       * @param  {string} lcName Lowercase attribute name.
       * @param  {string} value Attribute value.
       * @return {Boolean} Returns true if `value` is valid, otherwise false.
       */
      // eslint-disable-next-line complexity
      var _isValidAttribute = function _isValidAttribute(lcTag, lcName, value) {
        /* Make sure attribute cannot clobber */
        if (SANITIZE_DOM && (lcName === 'id' || lcName === 'name') && (value in document || value in formElement)) {
          return false;
        }

        /* Allow valid data-* attributes: At least one character after "-"
            (https://html.spec.whatwg.org/multipage/dom.html#embedding-custom-non-visible-data-with-the-data-*-attributes)
            XML-compatible (https://html.spec.whatwg.org/multipage/infrastructure.html#xml-compatible and http://www.w3.org/TR/xml/#d0e804)
            We don't need to check the value; it's always URI safe. */
        if (ALLOW_DATA_ATTR && regExpTest(DATA_ATTR$$1, lcName)) ; else if (ALLOW_ARIA_ATTR && regExpTest(ARIA_ATTR$$1, lcName)) ; else if (!ALLOWED_ATTR[lcName] || FORBID_ATTR[lcName]) {
          return false;

          /* Check value is safe. First, is attr inert? If so, is safe */
        } else if (URI_SAFE_ATTRIBUTES[lcName]) ; else if (regExpTest(IS_ALLOWED_URI$$1, stringReplace(value, ATTR_WHITESPACE$$1, ''))) ; else if ((lcName === 'src' || lcName === 'xlink:href' || lcName === 'href') && lcTag !== 'script' && stringIndexOf(value, 'data:') === 0 && DATA_URI_TAGS[lcTag]) ; else if (ALLOW_UNKNOWN_PROTOCOLS && !regExpTest(IS_SCRIPT_OR_DATA$$1, stringReplace(value, ATTR_WHITESPACE$$1, ''))) ; else if (!value) ; else {
          return false;
        }

        return true;
      };

      /**
       * _sanitizeAttributes
       *
       * @protect attributes
       * @protect nodeName
       * @protect removeAttribute
       * @protect setAttribute
       *
       * @param  {Node} currentNode to sanitize
       */
      // eslint-disable-next-line complexity
      var _sanitizeAttributes = function _sanitizeAttributes(currentNode) {
        var attr = void 0;
        var value = void 0;
        var lcName = void 0;
        var idAttr = void 0;
        var l = void 0;
        /* Execute a hook if present */
        _executeHook('beforeSanitizeAttributes', currentNode, null);

        var attributes = currentNode.attributes;

        /* Check if we have attributes; if not we might have a text node */

        if (!attributes) {
          return;
        }

        var hookEvent = {
          attrName: '',
          attrValue: '',
          keepAttr: true,
          allowedAttributes: ALLOWED_ATTR
        };
        l = attributes.length;

        /* Go backwards over all attributes; safely remove bad ones */
        while (l--) {
          attr = attributes[l];
          var _attr = attr,
              name = _attr.name,
              namespaceURI = _attr.namespaceURI;

          value = stringTrim(attr.value);
          lcName = stringToLowerCase(name);

          /* Execute a hook if present */
          hookEvent.attrName = lcName;
          hookEvent.attrValue = value;
          hookEvent.keepAttr = true;
          hookEvent.forceKeepAttr = undefined; // Allows developers to see this is a property they can set
          _executeHook('uponSanitizeAttribute', currentNode, hookEvent);
          value = hookEvent.attrValue;
          /* Did the hooks approve of the attribute? */
          if (hookEvent.forceKeepAttr) {
            continue;
          }

          /* Remove attribute */
          // Safari (iOS + Mac), last tested v8.0.5, crashes if you try to
          // remove a "name" attribute from an <img> tag that has an "id"
          // attribute at the time.
          if (lcName === 'name' && currentNode.nodeName === 'IMG' && attributes.id) {
            idAttr = attributes.id;
            attributes = arraySlice(attributes, []);
            _removeAttribute('id', currentNode);
            _removeAttribute(name, currentNode);
            if (arrayIndexOf(attributes, idAttr) > l) {
              currentNode.setAttribute('id', idAttr.value);
            }
          } else if (
          // This works around a bug in Safari, where input[type=file]
          // cannot be dynamically set after type has been removed
          currentNode.nodeName === 'INPUT' && lcName === 'type' && value === 'file' && hookEvent.keepAttr && (ALLOWED_ATTR[lcName] || !FORBID_ATTR[lcName])) {
            continue;
          } else {
            // This avoids a crash in Safari v9.0 with double-ids.
            // The trick is to first set the id to be empty and then to
            // remove the attribute
            if (name === 'id') {
              currentNode.setAttribute(name, '');
            }

            _removeAttribute(name, currentNode);
          }

          /* Did the hooks approve of the attribute? */
          if (!hookEvent.keepAttr) {
            continue;
          }

          /* Work around a security issue in jQuery 3.0 */
          if (SAFE_FOR_JQUERY && regExpTest(/\/>/i, value)) {
            _removeAttribute(name, currentNode);
            continue;
          }

          /* Take care of an mXSS pattern using namespace switches */
          if (regExpTest(/svg|math/i, currentNode.namespaceURI) && regExpTest(regExpCreate('</(' + arrayJoin(objectKeys(FORBID_CONTENTS), '|') + ')', 'i'), value)) {
            _removeAttribute(name, currentNode);
            continue;
          }

          /* Sanitize attribute content to be template-safe */
          if (SAFE_FOR_TEMPLATES) {
            value = stringReplace(value, MUSTACHE_EXPR$$1, ' ');
            value = stringReplace(value, ERB_EXPR$$1, ' ');
          }

          /* Is `value` valid for this attribute? */
          var lcTag = currentNode.nodeName.toLowerCase();
          if (!_isValidAttribute(lcTag, lcName, value)) {
            continue;
          }

          /* Handle invalid data-* attribute set by try-catching it */
          try {
            if (namespaceURI) {
              currentNode.setAttributeNS(namespaceURI, name, value);
            } else {
              /* Fallback to setAttribute() for browser-unrecognized namespaces e.g. "x-schema". */
              currentNode.setAttribute(name, value);
            }

            arrayPop(DOMPurify.removed);
          } catch (_) {}
        }

        /* Execute a hook if present */
        _executeHook('afterSanitizeAttributes', currentNode, null);
      };

      /**
       * _sanitizeShadowDOM
       *
       * @param  {DocumentFragment} fragment to iterate over recursively
       */
      var _sanitizeShadowDOM = function _sanitizeShadowDOM(fragment) {
        var shadowNode = void 0;
        var shadowIterator = _createIterator(fragment);

        /* Execute a hook if present */
        _executeHook('beforeSanitizeShadowDOM', fragment, null);

        while (shadowNode = shadowIterator.nextNode()) {
          /* Execute a hook if present */
          _executeHook('uponSanitizeShadowNode', shadowNode, null);

          /* Sanitize tags and elements */
          if (_sanitizeElements(shadowNode)) {
            continue;
          }

          /* Deep shadow DOM detected */
          if (shadowNode.content instanceof DocumentFragment) {
            _sanitizeShadowDOM(shadowNode.content);
          }

          /* Check attributes, sanitize if necessary */
          _sanitizeAttributes(shadowNode);
        }

        /* Execute a hook if present */
        _executeHook('afterSanitizeShadowDOM', fragment, null);
      };

      /**
       * Sanitize
       * Public method providing core sanitation functionality
       *
       * @param {String|Node} dirty string or DOM node
       * @param {Object} configuration object
       */
      // eslint-disable-next-line complexity
      DOMPurify.sanitize = function (dirty, cfg) {
        var body = void 0;
        var importedNode = void 0;
        var currentNode = void 0;
        var oldNode = void 0;
        var returnNode = void 0;
        /* Make sure we have a string to sanitize.
          DO NOT return early, as this will return the wrong type if
          the user has requested a DOM object rather than a string */
        if (!dirty) {
          dirty = '<!-->';
        }

        /* Stringify, in case dirty is an object */
        if (typeof dirty !== 'string' && !_isNode(dirty)) {
          // eslint-disable-next-line no-negated-condition
          if (typeof dirty.toString !== 'function') {
            throw typeErrorCreate('toString is not a function');
          } else {
            dirty = dirty.toString();
            if (typeof dirty !== 'string') {
              throw typeErrorCreate('dirty is not a string, aborting');
            }
          }
        }

        /* Check we can run. Otherwise fall back or ignore */
        if (!DOMPurify.isSupported) {
          if (_typeof(window.toStaticHTML) === 'object' || typeof window.toStaticHTML === 'function') {
            if (typeof dirty === 'string') {
              return window.toStaticHTML(dirty);
            }

            if (_isNode(dirty)) {
              return window.toStaticHTML(dirty.outerHTML);
            }
          }

          return dirty;
        }

        /* Assign config vars */
        if (!SET_CONFIG) {
          _parseConfig(cfg);
        }

        /* Clean up removed elements */
        DOMPurify.removed = [];

        /* Check if dirty is correctly typed for IN_PLACE */
        if (typeof dirty === 'string') {
          IN_PLACE = false;
        }

        if (IN_PLACE) ; else if (dirty instanceof Node) {
          /* If dirty is a DOM element, append to an empty document to avoid
             elements being stripped by the parser */
          body = _initDocument('<!-->');
          importedNode = body.ownerDocument.importNode(dirty, true);
          if (importedNode.nodeType === 1 && importedNode.nodeName === 'BODY') {
            /* Node is already a body, use as is */
            body = importedNode;
          } else if (importedNode.nodeName === 'HTML') {
            body = importedNode;
          } else {
            // eslint-disable-next-line unicorn/prefer-node-append
            body.appendChild(importedNode);
          }
        } else {
          /* Exit directly if we have nothing to do */
          if (!RETURN_DOM && !SAFE_FOR_TEMPLATES && !WHOLE_DOCUMENT && RETURN_TRUSTED_TYPE &&
          // eslint-disable-next-line unicorn/prefer-includes
          dirty.indexOf('<') === -1) {
            return trustedTypesPolicy ? trustedTypesPolicy.createHTML(dirty) : dirty;
          }

          /* Initialize the document to work on */
          body = _initDocument(dirty);

          /* Check we have a DOM node from the data */
          if (!body) {
            return RETURN_DOM ? null : emptyHTML;
          }
        }

        /* Remove first element node (ours) if FORCE_BODY is set */
        if (body && FORCE_BODY) {
          _forceRemove(body.firstChild);
        }

        /* Get node iterator */
        var nodeIterator = _createIterator(IN_PLACE ? dirty : body);

        /* Now start iterating over the created document */
        while (currentNode = nodeIterator.nextNode()) {
          /* Fix IE's strange behavior with manipulated textNodes #89 */
          if (currentNode.nodeType === 3 && currentNode === oldNode) {
            continue;
          }

          /* Sanitize tags and elements */
          if (_sanitizeElements(currentNode)) {
            continue;
          }

          /* Shadow DOM detected, sanitize it */
          if (currentNode.content instanceof DocumentFragment) {
            _sanitizeShadowDOM(currentNode.content);
          }

          /* Check attributes, sanitize if necessary */
          _sanitizeAttributes(currentNode);

          oldNode = currentNode;
        }

        oldNode = null;

        /* If we sanitized `dirty` in-place, return it. */
        if (IN_PLACE) {
          return dirty;
        }

        /* Return sanitized string or DOM */
        if (RETURN_DOM) {
          if (RETURN_DOM_FRAGMENT) {
            returnNode = createDocumentFragment.call(body.ownerDocument);

            while (body.firstChild) {
              // eslint-disable-next-line unicorn/prefer-node-append
              returnNode.appendChild(body.firstChild);
            }
          } else {
            returnNode = body;
          }

          if (RETURN_DOM_IMPORT) {
            /*
              AdoptNode() is not used because internal state is not reset
              (e.g. the past names map of a HTMLFormElement), this is safe
              in theory but we would rather not risk another attack vector.
              The state that is cloned by importNode() is explicitly defined
              by the specs.
            */
            returnNode = importNode.call(originalDocument, returnNode, true);
          }

          return returnNode;
        }

        var serializedHTML = WHOLE_DOCUMENT ? body.outerHTML : body.innerHTML;

        /* Sanitize final string template-safe */
        if (SAFE_FOR_TEMPLATES) {
          serializedHTML = stringReplace(serializedHTML, MUSTACHE_EXPR$$1, ' ');
          serializedHTML = stringReplace(serializedHTML, ERB_EXPR$$1, ' ');
        }

        return trustedTypesPolicy && RETURN_TRUSTED_TYPE ? trustedTypesPolicy.createHTML(serializedHTML) : serializedHTML;
      };

      /**
       * Public method to set the configuration once
       * setConfig
       *
       * @param {Object} cfg configuration object
       */
      DOMPurify.setConfig = function (cfg) {
        _parseConfig(cfg);
        SET_CONFIG = true;
      };

      /**
       * Public method to remove the configuration
       * clearConfig
       *
       */
      DOMPurify.clearConfig = function () {
        CONFIG = null;
        SET_CONFIG = false;
      };

      /**
       * Public method to check if an attribute value is valid.
       * Uses last set config, if any. Otherwise, uses config defaults.
       * isValidAttribute
       *
       * @param  {string} tag Tag name of containing element.
       * @param  {string} attr Attribute name.
       * @param  {string} value Attribute value.
       * @return {Boolean} Returns true if `value` is valid. Otherwise, returns false.
       */
      DOMPurify.isValidAttribute = function (tag, attr, value) {
        /* Initialize shared config vars if necessary. */
        if (!CONFIG) {
          _parseConfig({});
        }

        var lcTag = stringToLowerCase(tag);
        var lcName = stringToLowerCase(attr);
        return _isValidAttribute(lcTag, lcName, value);
      };

      /**
       * AddHook
       * Public method to add DOMPurify hooks
       *
       * @param {String} entryPoint entry point for the hook to add
       * @param {Function} hookFunction function to execute
       */
      DOMPurify.addHook = function (entryPoint, hookFunction) {
        if (typeof hookFunction !== 'function') {
          return;
        }

        hooks[entryPoint] = hooks[entryPoint] || [];
        arrayPush(hooks[entryPoint], hookFunction);
      };

      /**
       * RemoveHook
       * Public method to remove a DOMPurify hook at a given entryPoint
       * (pops it from the stack of hooks if more are present)
       *
       * @param {String} entryPoint entry point for the hook to remove
       */
      DOMPurify.removeHook = function (entryPoint) {
        if (hooks[entryPoint]) {
          arrayPop(hooks[entryPoint]);
        }
      };

      /**
       * RemoveHooks
       * Public method to remove all DOMPurify hooks at a given entryPoint
       *
       * @param  {String} entryPoint entry point for the hooks to remove
       */
      DOMPurify.removeHooks = function (entryPoint) {
        if (hooks[entryPoint]) {
          hooks[entryPoint] = [];
        }
      };

      /**
       * RemoveAllHooks
       * Public method to remove all DOMPurify hooks
       *
       */
      DOMPurify.removeAllHooks = function () {
        hooks = {};
      };

      return DOMPurify;
    }

    var purify = createDOMPurify();

    // https://github.com/substack/deep-freeze/blob/master/index.js
    function deepFreeze (o) {
      Object.freeze(o);

      var objIsFunction = typeof o === 'function';

      Object.getOwnPropertyNames(o).forEach(function (prop) {
        if (o.hasOwnProperty(prop)
        && o[prop] !== null
        && (typeof o[prop] === "object" || typeof o[prop] === "function")
        // IE11 fix: https://github.com/highlightjs/highlight.js/issues/2318
        // TODO: remove in the future
        && (objIsFunction ? prop !== 'caller' && prop !== 'callee' && prop !== 'arguments' : true)
        && !Object.isFrozen(o[prop])) {
          deepFreeze(o[prop]);
        }
      });

      return o;
    }

    function escapeHTML(value) {
      return value.replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;');
    }


    /**
     * performs a shallow merge of multiple objects into one
     *
     * @arguments list of objects with properties to merge
     * @returns a single new object
     */
    function inherit(parent) {  // inherit(parent, override_obj, override_obj, ...)
      var key;
      var result = {};
      var objects = Array.prototype.slice.call(arguments, 1);

      for (key in parent)
        result[key] = parent[key];
      objects.forEach(function(obj) {
        for (key in obj)
          result[key] = obj[key];
      });
      return result;
    }

    /* Stream merging */


    function tag(node) {
      return node.nodeName.toLowerCase();
    }


    function nodeStream(node) {
      var result = [];
      (function _nodeStream(node, offset) {
        for (var child = node.firstChild; child; child = child.nextSibling) {
          if (child.nodeType === 3)
            offset += child.nodeValue.length;
          else if (child.nodeType === 1) {
            result.push({
              event: 'start',
              offset: offset,
              node: child
            });
            offset = _nodeStream(child, offset);
            // Prevent void elements from having an end tag that would actually
            // double them in the output. There are more void elements in HTML
            // but we list only those realistically expected in code display.
            if (!tag(child).match(/br|hr|img|input/)) {
              result.push({
                event: 'stop',
                offset: offset,
                node: child
              });
            }
          }
        }
        return offset;
      })(node, 0);
      return result;
    }

    function mergeStreams(original, highlighted, value) {
      var processed = 0;
      var result = '';
      var nodeStack = [];

      function selectStream() {
        if (!original.length || !highlighted.length) {
          return original.length ? original : highlighted;
        }
        if (original[0].offset !== highlighted[0].offset) {
          return (original[0].offset < highlighted[0].offset) ? original : highlighted;
        }

        /*
        To avoid starting the stream just before it should stop the order is
        ensured that original always starts first and closes last:

        if (event1 == 'start' && event2 == 'start')
          return original;
        if (event1 == 'start' && event2 == 'stop')
          return highlighted;
        if (event1 == 'stop' && event2 == 'start')
          return original;
        if (event1 == 'stop' && event2 == 'stop')
          return highlighted;

        ... which is collapsed to:
        */
        return highlighted[0].event === 'start' ? original : highlighted;
      }

      function open(node) {
        function attr_str(a) {
          return ' ' + a.nodeName + '="' + escapeHTML(a.value).replace(/"/g, '&quot;') + '"';
        }
        result += '<' + tag(node) + [].map.call(node.attributes, attr_str).join('') + '>';
      }

      function close(node) {
        result += '</' + tag(node) + '>';
      }

      function render(event) {
        (event.event === 'start' ? open : close)(event.node);
      }

      while (original.length || highlighted.length) {
        var stream = selectStream();
        result += escapeHTML(value.substring(processed, stream[0].offset));
        processed = stream[0].offset;
        if (stream === original) {
          /*
          On any opening or closing tag of the original markup we first close
          the entire highlighted node stack, then render the original tag along
          with all the following original tags at the same offset and then
          reopen all the tags on the highlighted stack.
          */
          nodeStack.reverse().forEach(close);
          do {
            render(stream.splice(0, 1)[0]);
            stream = selectStream();
          } while (stream === original && stream.length && stream[0].offset === processed);
          nodeStack.reverse().forEach(open);
        } else {
          if (stream[0].event === 'start') {
            nodeStack.push(stream[0].node);
          } else {
            nodeStack.pop();
          }
          render(stream.splice(0, 1)[0]);
        }
      }
      return result + escapeHTML(value.substr(processed));
    }

    var utils = /*#__PURE__*/Object.freeze({
      __proto__: null,
      escapeHTML: escapeHTML,
      inherit: inherit,
      nodeStream: nodeStream,
      mergeStreams: mergeStreams
    });

    const SPAN_CLOSE = '</span>';

    const emitsWrappingTags = (node) => {
      return !!node.kind;
    };

    class HTMLRenderer {
      constructor(tree, options) {
        this.buffer = "";
        this.classPrefix = options.classPrefix;
        tree.walk(this);
      }

      // renderer API

      addText(text) {
        this.buffer += escapeHTML(text);
      }

      openNode(node) {
        if (!emitsWrappingTags(node)) return;

        let className = node.kind;
        if (!node.sublanguage)
          className = `${this.classPrefix}${className}`;
        this.span(className);
      }

      closeNode(node) {
        if (!emitsWrappingTags(node)) return;

        this.buffer += SPAN_CLOSE;
      }

      // helpers

      span(className) {
        this.buffer += `<span class="${className}">`;
      }

      value() {
        return this.buffer;
      }
    }

    class TokenTree {
      constructor() {
        this.rootNode = { children: [] };
        this.stack = [ this.rootNode ];
      }

      get top() {
        return this.stack[this.stack.length - 1];
      }

      get root() { return this.rootNode };

      add(node) {
        this.top.children.push(node);
      }

      openNode(kind) {
        let node = { kind, children: [] };
        this.add(node);
        this.stack.push(node);
      }

      closeNode() {
        if (this.stack.length > 1)
          return this.stack.pop();
      }

      closeAllNodes() {
        while (this.closeNode());
      }

      toJSON() {
        return JSON.stringify(this.rootNode, null, 4);
      }

      walk(builder) {
        return this.constructor._walk(builder, this.rootNode);
      }

      static _walk(builder, node) {
        if (typeof node === "string") {
          builder.addText(node);
        } else if (node.children) {
          builder.openNode(node);
          node.children.forEach((child) => this._walk(builder, child));
          builder.closeNode(node);
        }
        return builder;
      }

      static _collapse(node) {
        if (!node.children) {
          return;
        }
        if (node.children.every(el => typeof el === "string")) {
          node.text = node.children.join("");
          delete node["children"];
        } else {
          node.children.forEach((child) => {
            if (typeof child === "string") return;
            TokenTree._collapse(child);
          });
        }
      }
    }

    /**
      Currently this is all private API, but this is the minimal API necessary
      that an Emitter must implement to fully support the parser.

      Minimal interface:

      - addKeyword(text, kind)
      - addText(text)
      - addSublanguage(emitter, subLangaugeName)
      - finalize()
      - openNode(kind)
      - closeNode()
      - closeAllNodes()
      - toHTML()

    */
    class TokenTreeEmitter extends TokenTree {
      constructor(options) {
        super();
        this.options = options;
      }

      addKeyword(text, kind) {
        if (text === "") { return; }

        this.openNode(kind);
        this.addText(text);
        this.closeNode();
      }

      addText(text) {
        if (text === "") { return; }

        this.add(text);
      }

      addSublanguage(emitter, name) {
        let node = emitter.root;
        node.kind = name;
        node.sublanguage = true;
        this.add(node);
      }

      toHTML() {
        let renderer = new HTMLRenderer(this, this.options);
        return renderer.value();
      }

      finalize() {
        return;
      }

    }

    function escape$4(value) {
      return new RegExp(value.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&'), 'm');
    }

    function source(re) {
      // if it's a regex get it's source,
      // otherwise it's a string already so just return it
      return (re && re.source) || re;
    }

    function countMatchGroups(re) {
      return (new RegExp(re.toString() + '|')).exec('').length - 1;
    }

    function startsWith(re, lexeme) {
      var match = re && re.exec(lexeme);
      return match && match.index === 0;
    }

    // join logically computes regexps.join(separator), but fixes the
    // backreferences so they continue to match.
    // it also places each individual regular expression into it's own
    // match group, keeping track of the sequencing of those match groups
    // is currently an exercise for the caller. :-)
    function join(regexps, separator) {
      // backreferenceRe matches an open parenthesis or backreference. To avoid
      // an incorrect parse, it additionally matches the following:
      // - [...] elements, where the meaning of parentheses and escapes change
      // - other escape sequences, so we do not misparse escape sequences as
      //   interesting elements
      // - non-matching or lookahead parentheses, which do not capture. These
      //   follow the '(' with a '?'.
      var backreferenceRe = /\[(?:[^\\\]]|\\.)*\]|\(\??|\\([1-9][0-9]*)|\\./;
      var numCaptures = 0;
      var ret = '';
      for (var i = 0; i < regexps.length; i++) {
        numCaptures += 1;
        var offset = numCaptures;
        var re = source(regexps[i]);
        if (i > 0) {
          ret += separator;
        }
        ret += "(";
        while (re.length > 0) {
          var match = backreferenceRe.exec(re);
          if (match == null) {
            ret += re;
            break;
          }
          ret += re.substring(0, match.index);
          re = re.substring(match.index + match[0].length);
          if (match[0][0] == '\\' && match[1]) {
            // Adjust the backreference.
            ret += '\\' + String(Number(match[1]) + offset);
          } else {
            ret += match[0];
            if (match[0] == '(') {
              numCaptures++;
            }
          }
        }
        ret += ")";
      }
      return ret;
    }

    // Common regexps
    const IDENT_RE = '[a-zA-Z]\\w*';
    const UNDERSCORE_IDENT_RE = '[a-zA-Z_]\\w*';
    const NUMBER_RE = '\\b\\d+(\\.\\d+)?';
    const C_NUMBER_RE = '(-?)(\\b0[xX][a-fA-F0-9]+|(\\b\\d+(\\.\\d*)?|\\.\\d+)([eE][-+]?\\d+)?)'; // 0x..., 0..., decimal, float
    const BINARY_NUMBER_RE = '\\b(0b[01]+)'; // 0b...
    const RE_STARTERS_RE = '!|!=|!==|%|%=|&|&&|&=|\\*|\\*=|\\+|\\+=|,|-|-=|/=|/|:|;|<<|<<=|<=|<|===|==|=|>>>=|>>=|>=|>>>|>>|>|\\?|\\[|\\{|\\(|\\^|\\^=|\\||\\|=|\\|\\||~';

    // Common modes
    const BACKSLASH_ESCAPE = {
      begin: '\\\\[\\s\\S]', relevance: 0
    };
    const APOS_STRING_MODE = {
      className: 'string',
      begin: '\'', end: '\'',
      illegal: '\\n',
      contains: [BACKSLASH_ESCAPE]
    };
    const QUOTE_STRING_MODE = {
      className: 'string',
      begin: '"', end: '"',
      illegal: '\\n',
      contains: [BACKSLASH_ESCAPE]
    };
    const PHRASAL_WORDS_MODE = {
      begin: /\b(a|an|the|are|I'm|isn't|don't|doesn't|won't|but|just|should|pretty|simply|enough|gonna|going|wtf|so|such|will|you|your|they|like|more)\b/
    };
    const COMMENT = function (begin, end, inherits) {
      var mode = inherit(
        {
          className: 'comment',
          begin: begin, end: end,
          contains: []
        },
        inherits || {}
      );
      mode.contains.push(PHRASAL_WORDS_MODE);
      mode.contains.push({
        className: 'doctag',
        begin: '(?:TODO|FIXME|NOTE|BUG|XXX):',
        relevance: 0
      });
      return mode;
    };
    const C_LINE_COMMENT_MODE = COMMENT('//', '$');
    const C_BLOCK_COMMENT_MODE = COMMENT('/\\*', '\\*/');
    const HASH_COMMENT_MODE = COMMENT('#', '$');
    const NUMBER_MODE = {
      className: 'number',
      begin: NUMBER_RE,
      relevance: 0
    };
    const C_NUMBER_MODE = {
      className: 'number',
      begin: C_NUMBER_RE,
      relevance: 0
    };
    const BINARY_NUMBER_MODE = {
      className: 'number',
      begin: BINARY_NUMBER_RE,
      relevance: 0
    };
    const CSS_NUMBER_MODE = {
      className: 'number',
      begin: NUMBER_RE + '(' +
        '%|em|ex|ch|rem'  +
        '|vw|vh|vmin|vmax' +
        '|cm|mm|in|pt|pc|px' +
        '|deg|grad|rad|turn' +
        '|s|ms' +
        '|Hz|kHz' +
        '|dpi|dpcm|dppx' +
        ')?',
      relevance: 0
    };
    const REGEXP_MODE = {
      // this outer rule makes sure we actually have a WHOLE regex and not simply
      // an expression such as:
      //
      //     3 / something
      //
      // (which will then blow up when regex's `illegal` sees the newline)
      begin: /(?=\/[^\/\n]*\/)/,
      contains: [{
        className: 'regexp',
        begin: /\//, end: /\/[gimuy]*/,
        illegal: /\n/,
        contains: [
          BACKSLASH_ESCAPE,
          {
            begin: /\[/, end: /\]/,
            relevance: 0,
            contains: [BACKSLASH_ESCAPE]
          }
        ]
      }]
    };
    const TITLE_MODE = {
      className: 'title',
      begin: IDENT_RE,
      relevance: 0
    };
    const UNDERSCORE_TITLE_MODE = {
      className: 'title',
      begin: UNDERSCORE_IDENT_RE,
      relevance: 0
    };
    const METHOD_GUARD = {
      // excludes method names from keyword processing
      begin: '\\.\\s*' + UNDERSCORE_IDENT_RE,
      relevance: 0
    };

    var MODES = /*#__PURE__*/Object.freeze({
      __proto__: null,
      IDENT_RE: IDENT_RE,
      UNDERSCORE_IDENT_RE: UNDERSCORE_IDENT_RE,
      NUMBER_RE: NUMBER_RE,
      C_NUMBER_RE: C_NUMBER_RE,
      BINARY_NUMBER_RE: BINARY_NUMBER_RE,
      RE_STARTERS_RE: RE_STARTERS_RE,
      BACKSLASH_ESCAPE: BACKSLASH_ESCAPE,
      APOS_STRING_MODE: APOS_STRING_MODE,
      QUOTE_STRING_MODE: QUOTE_STRING_MODE,
      PHRASAL_WORDS_MODE: PHRASAL_WORDS_MODE,
      COMMENT: COMMENT,
      C_LINE_COMMENT_MODE: C_LINE_COMMENT_MODE,
      C_BLOCK_COMMENT_MODE: C_BLOCK_COMMENT_MODE,
      HASH_COMMENT_MODE: HASH_COMMENT_MODE,
      NUMBER_MODE: NUMBER_MODE,
      C_NUMBER_MODE: C_NUMBER_MODE,
      BINARY_NUMBER_MODE: BINARY_NUMBER_MODE,
      CSS_NUMBER_MODE: CSS_NUMBER_MODE,
      REGEXP_MODE: REGEXP_MODE,
      TITLE_MODE: TITLE_MODE,
      UNDERSCORE_TITLE_MODE: UNDERSCORE_TITLE_MODE,
      METHOD_GUARD: METHOD_GUARD
    });

    // keywords that should have no default relevance value
    var COMMON_KEYWORDS = 'of and for in not or if then'.split(' ');

    // compilation

    function compileLanguage(language) {

      function langRe(value, global) {
        return new RegExp(
          source(value),
          'm' + (language.case_insensitive ? 'i' : '') + (global ? 'g' : '')
        );
      }

      /**
        Stores multiple regular expressions and allows you to quickly search for
        them all in a string simultaneously - returning the first match.  It does
        this by creating a huge (a|b|c) regex - each individual item wrapped with ()
        and joined by `|` - using match groups to track position.  When a match is
        found checking which position in the array has content allows us to figure
        out which of the original regexes / match groups triggered the match.

        The match object itself (the result of `Regex.exec`) is returned but also
        enhanced by merging in any meta-data that was registered with the regex.
        This is how we keep track of which mode matched, and what type of rule
        (`illegal`, `begin`, end, etc).
      */
      class MultiRegex {
        constructor() {
          this.matchIndexes = {};
          this.regexes = [];
          this.matchAt = 1;
          this.position = 0;
        }

        addRule(re, opts) {
          opts.position = this.position++;
          this.matchIndexes[this.matchAt] = opts;
          this.regexes.push([opts, re]);
          this.matchAt += countMatchGroups(re) + 1;
        }

        compile() {
          if (this.regexes.length === 0) {
            // avoids the need to check length every time exec is called
            this.exec = () => null;
          }
          let terminators = this.regexes.map(el => el[1]);
          this.matcherRe = langRe(join(terminators, '|'), true);
          this.lastIndex = 0;
        }

        exec(s) {
          this.matcherRe.lastIndex = this.lastIndex;
          let match = this.matcherRe.exec(s);
          if (!match) { return null; }

          let i = match.findIndex((el, i) => i>0 && el!=undefined);
          let matchData = this.matchIndexes[i];

          return Object.assign(match, matchData);
        }
      }

      /*
        Created to solve the key deficiently with MultiRegex - there is no way to
        test for multiple matches at a single location.  Why would we need to do
        that?  In the future a more dynamic engine will allow certain matches to be
        ignored.  An example: if we matched say the 3rd regex in a large group but
        decided to ignore it - we'd need to started testing again at the 4th
        regex... but MultiRegex itself gives us no real way to do that.

        So what this class creates MultiRegexs on the fly for whatever search
        position they are needed.

        NOTE: These additional MultiRegex objects are created dynamically.  For most
        grammars most of the time we will never actually need anything more than the
        first MultiRegex - so this shouldn't have too much overhead.

        Say this is our search group, and we match regex3, but wish to ignore it.

          regex1 | regex2 | regex3 | regex4 | regex5    ' ie, startAt = 0

        What we need is a new MultiRegex that only includes the remaining
        possibilities:

          regex4 | regex5                               ' ie, startAt = 3

        This class wraps all that complexity up in a simple API... `startAt` decides
        where in the array of expressions to start doing the matching. It
        auto-increments, so if a match is found at position 2, then startAt will be
        set to 3.  If the end is reached startAt will return to 0.

        MOST of the time the parser will be setting startAt manually to 0.
      */
      class ResumableMultiRegex {
        constructor() {
          this.rules = [];
          this.multiRegexes = [];
          this.count = 0;

          this.lastIndex = 0;
          this.regexIndex = 0;
        }

        getMatcher(index) {
          if (this.multiRegexes[index]) return this.multiRegexes[index];

          let matcher = new MultiRegex();
          this.rules.slice(index).forEach(([re, opts])=> matcher.addRule(re,opts));
          matcher.compile();
          this.multiRegexes[index] = matcher;
          return matcher;
        }

        considerAll() {
          this.regexIndex = 0;
        }

        addRule(re, opts) {
          this.rules.push([re, opts]);
          if (opts.type==="begin") this.count++;
        }

        exec(s) {
          let m = this.getMatcher(this.regexIndex);
          m.lastIndex = this.lastIndex;
          let result = m.exec(s);
          if (result) {
            this.regexIndex += result.position + 1;
            if (this.regexIndex === this.count) // wrap-around
              this.regexIndex = 0;
          }

          // this.regexIndex = 0;
          return result;
        }
      }

      function buildModeRegex(mode) {

        let mm = new ResumableMultiRegex();

        mode.contains.forEach(term => mm.addRule(term.begin, {rule: term, type: "begin" }));

        if (mode.terminator_end)
          mm.addRule(mode.terminator_end, {type: "end"} );
        if (mode.illegal)
          mm.addRule(mode.illegal, {type: "illegal"} );

        return mm;
      }

      // TODO: We need negative look-behind support to do this properly
      function skipIfhasPrecedingOrTrailingDot(match) {
        let before = match.input[match.index-1];
        let after = match.input[match.index + match[0].length];
        if (before === "." || after === ".") {
          return {ignoreMatch: true };
        }
      }

      /** skip vs abort vs ignore
       *
       * @skip   - The mode is still entered and exited normally (and contains rules apply),
       *           but all content is held and added to the parent buffer rather than being
       *           output when the mode ends.  Mostly used with `sublanguage` to build up
       *           a single large buffer than can be parsed by sublanguage.
       *
       *             - The mode begin ands ends normally.
       *             - Content matched is added to the parent mode buffer.
       *             - The parser cursor is moved forward normally.
       *
       * @abort  - A hack placeholder until we have ignore.  Aborts the mode (as if it
       *           never matched) but DOES NOT continue to match subsequent `contains`
       *           modes.  Abort is bad/suboptimal because it can result in modes
       *           farther down not getting applied because an earlier rule eats the
       *           content but then aborts.
       *
       *             - The mode does not begin.
       *             - Content matched by `begin` is added to the mode buffer.
       *             - The parser cursor is moved forward accordingly.
       *
       * @ignore - Ignores the mode (as if it never matched) and continues to match any
       *           subsequent `contains` modes.  Ignore isn't technically possible with
       *           the current parser implementation.
       *
       *             - The mode does not begin.
       *             - Content matched by `begin` is ignored.
       *             - The parser cursor is not moved forward.
       */

      function compileMode(mode, parent) {
        if (mode.compiled)
          return;
        mode.compiled = true;

        // __onBegin is considered private API, internal use only
        mode.__onBegin = null;

        mode.keywords = mode.keywords || mode.beginKeywords;
        if (mode.keywords)
          mode.keywords = compileKeywords(mode.keywords, language.case_insensitive);

        mode.lexemesRe = langRe(mode.lexemes || /\w+/, true);

        if (parent) {
          if (mode.beginKeywords) {
            // for languages with keywords that include non-word characters checking for
            // a word boundary is not sufficient, so instead we check for a word boundary
            // or whitespace - this does no harm in any case since our keyword engine
            // doesn't allow spaces in keywords anyways and we still check for the boundary
            // first
            mode.begin = '\\b(' + mode.beginKeywords.split(' ').join('|') + ')(?=\\b|\\s)';
            mode.__onBegin = skipIfhasPrecedingOrTrailingDot;
          }
          if (!mode.begin)
            mode.begin = /\B|\b/;
          mode.beginRe = langRe(mode.begin);
          if (mode.endSameAsBegin)
            mode.end = mode.begin;
          if (!mode.end && !mode.endsWithParent)
            mode.end = /\B|\b/;
          if (mode.end)
            mode.endRe = langRe(mode.end);
          mode.terminator_end = source(mode.end) || '';
          if (mode.endsWithParent && parent.terminator_end)
            mode.terminator_end += (mode.end ? '|' : '') + parent.terminator_end;
        }
        if (mode.illegal)
          mode.illegalRe = langRe(mode.illegal);
        if (mode.relevance == null)
          mode.relevance = 1;
        if (!mode.contains) {
          mode.contains = [];
        }
        mode.contains = [].concat(...mode.contains.map(function(c) {
          return expand_or_clone_mode(c === 'self' ? mode : c);
        }));
        mode.contains.forEach(function(c) {compileMode(c, mode);});

        if (mode.starts) {
          compileMode(mode.starts, parent);
        }

        mode.matcher = buildModeRegex(mode);
      }

      // self is not valid at the top-level
      if (language.contains && language.contains.includes('self')) {
        throw new Error("ERR: contains `self` is not supported at the top-level of a language.  See documentation.")
      }
      compileMode(language);
    }

    function dependencyOnParent(mode) {
      if (!mode) return false;

      return mode.endsWithParent || dependencyOnParent(mode.starts);
    }

    function expand_or_clone_mode(mode) {
      if (mode.variants && !mode.cached_variants) {
        mode.cached_variants = mode.variants.map(function(variant) {
          return inherit(mode, {variants: null}, variant);
        });
      }

      // EXPAND
      // if we have variants then essentially "replace" the mode with the variants
      // this happens in compileMode, where this function is called from
      if (mode.cached_variants)
        return mode.cached_variants;

      // CLONE
      // if we have dependencies on parents then we need a unique
      // instance of ourselves, so we can be reused with many
      // different parents without issue
      if (dependencyOnParent(mode))
        return inherit(mode, { starts: mode.starts ? inherit(mode.starts) : null });

      if (Object.isFrozen(mode))
        return inherit(mode);

      // no special dependency issues, just return ourselves
      return mode;
    }


    // keywords

    function compileKeywords(rawKeywords, case_insensitive) {
      var compiled_keywords = {};

      if (typeof rawKeywords === 'string') { // string
        splitAndCompile('keyword', rawKeywords);
      } else {
        Object.keys(rawKeywords).forEach(function (className) {
          splitAndCompile(className, rawKeywords[className]);
        });
      }
    return compiled_keywords;

    // ---

    function splitAndCompile(className, str) {
      if (case_insensitive) {
        str = str.toLowerCase();
      }
      str.split(' ').forEach(function(keyword) {
        var pair = keyword.split('|');
        compiled_keywords[pair[0]] = [className, scoreForKeyword(pair[0], pair[1])];
      });
    }
    }

    function scoreForKeyword(keyword, providedScore) {
    // manual scores always win over common keywords
    // so you can force a score of 1 if you really insist
    if (providedScore)
      return Number(providedScore);

    return commonKeyword(keyword) ? 0 : 1;
    }

    function commonKeyword(word) {
    return COMMON_KEYWORDS.includes(word.toLowerCase());
    }

    var version = "10.0.3";

    /*
    Syntax highlighting with language autodetection.
    https://highlightjs.org/
    */

    const escape$1$1 = escapeHTML;
    const inherit$1 = inherit;

    const { nodeStream: nodeStream$1, mergeStreams: mergeStreams$1 } = utils;


    const HLJS = function(hljs) {

      // Convenience variables for build-in objects
      var ArrayProto = [];

      // Global internal variables used within the highlight.js library.
      var languages = {},
          aliases   = {},
          plugins   = [];

      // safe/production mode - swallows more errors, tries to keep running
      // even if a single syntax or parse hits a fatal error
      var SAFE_MODE = true;

      // Regular expressions used throughout the highlight.js library.
      var fixMarkupRe      = /((^(<[^>]+>|\t|)+|(?:\n)))/gm;

      var LANGUAGE_NOT_FOUND = "Could not find the language '{}', did you forget to load/include a language module?";

      // Global options used when within external APIs. This is modified when
      // calling the `hljs.configure` function.
      var options = {
        noHighlightRe: /^(no-?highlight)$/i,
        languageDetectRe: /\blang(?:uage)?-([\w-]+)\b/i,
        classPrefix: 'hljs-',
        tabReplace: null,
        useBR: false,
        languages: undefined,
        // beta configuration options, subject to change, welcome to discuss
        // https://github.com/highlightjs/highlight.js/issues/1086
        __emitter: TokenTreeEmitter
      };

      /* Utility functions */

      function shouldNotHighlight(language) {
        return options.noHighlightRe.test(language);
      }

      function blockLanguage(block) {
        var match;
        var classes = block.className + ' ';

        classes += block.parentNode ? block.parentNode.className : '';

        // language-* takes precedence over non-prefixed class names.
        match = options.languageDetectRe.exec(classes);
        if (match) {
          var language = getLanguage(match[1]);
          if (!language) {
            console.warn(LANGUAGE_NOT_FOUND.replace("{}", match[1]));
            console.warn("Falling back to no-highlight mode for this block.", block);
          }
          return language ? match[1] : 'no-highlight';
        }

        return classes
          .split(/\s+/)
          .find((_class) => shouldNotHighlight(_class) || getLanguage(_class));
      }

      /**
       * Core highlighting function.
       *
       * @param {string} languageName - the language to use for highlighting
       * @param {string} code - the code to highlight
       * @param {boolean} ignore_illegals - whether to ignore illegal matches, default is to bail
       * @param {array<mode>} continuation - array of continuation modes
       *
       * @returns an object that represents the result
       * @property {string} language - the language name
       * @property {number} relevance - the relevance score
       * @property {string} value - the highlighted HTML code
       * @property {string} code - the original raw code
       * @property {mode} top - top of the current mode stack
       * @property {boolean} illegal - indicates whether any illegal matches were found
      */
      function highlight(languageName, code, ignore_illegals, continuation) {
        var context = {
          code,
          language: languageName
        };
        // the plugin can change the desired language or the code to be highlighted
        // just be changing the object it was passed
        fire("before:highlight", context);

        // a before plugin can usurp the result completely by providing it's own
        // in which case we don't even need to call highlight
        var result = context.result ?
          context.result :
          _highlight(context.language, context.code, ignore_illegals, continuation);

        result.code = context.code;
        // the plugin can change anything in result to suite it
        fire("after:highlight", result);

        return result;
      }

      // private highlight that's used internally and does not fire callbacks
      function _highlight(languageName, code, ignore_illegals, continuation) {
        var codeToHighlight = code;

        function endOfMode(mode, lexeme) {
          if (startsWith(mode.endRe, lexeme)) {
            while (mode.endsParent && mode.parent) {
              mode = mode.parent;
            }
            return mode;
          }
          if (mode.endsWithParent) {
            return endOfMode(mode.parent, lexeme);
          }
        }

        function keywordMatch(mode, match) {
          var match_str = language.case_insensitive ? match[0].toLowerCase() : match[0];
          return mode.keywords.hasOwnProperty(match_str) && mode.keywords[match_str];
        }

        function processKeywords() {
          var keyword_match, last_index, match, buf;

          if (!top.keywords) {
            emitter.addText(mode_buffer);
            return;
          }

          last_index = 0;
          top.lexemesRe.lastIndex = 0;
          match = top.lexemesRe.exec(mode_buffer);
          buf = "";

          while (match) {
            buf += mode_buffer.substring(last_index, match.index);
            keyword_match = keywordMatch(top, match);
            var kind = null;
            if (keyword_match) {
              emitter.addText(buf);
              buf = "";

              relevance += keyword_match[1];
              kind = keyword_match[0];
              emitter.addKeyword(match[0], kind);
            } else {
              buf += match[0];
            }
            last_index = top.lexemesRe.lastIndex;
            match = top.lexemesRe.exec(mode_buffer);
          }
          buf += mode_buffer.substr(last_index);
          emitter.addText(buf);
        }

        function processSubLanguage() {
          if (mode_buffer === "") return;

          var explicit = typeof top.subLanguage === 'string';

          if (explicit && !languages[top.subLanguage]) {
            emitter.addText(mode_buffer);
            return;
          }

          var result = explicit ?
                       _highlight(top.subLanguage, mode_buffer, true, continuations[top.subLanguage]) :
                       highlightAuto(mode_buffer, top.subLanguage.length ? top.subLanguage : undefined);

          // Counting embedded language score towards the host language may be disabled
          // with zeroing the containing mode relevance. Use case in point is Markdown that
          // allows XML everywhere and makes every XML snippet to have a much larger Markdown
          // score.
          if (top.relevance > 0) {
            relevance += result.relevance;
          }
          if (explicit) {
            continuations[top.subLanguage] = result.top;
          }
          emitter.addSublanguage(result.emitter, result.language);
        }

        function processBuffer() {
          if (top.subLanguage != null)
            processSubLanguage();
          else
            processKeywords();
          mode_buffer = '';
        }

        function startNewMode(mode) {
          if (mode.className) {
            emitter.openNode(mode.className);
          }
          top = Object.create(mode, {parent: {value: top}});
        }

        function doIgnore(lexeme) {
          if (top.matcher.regexIndex === 0) {
            // no more regexs to potentially match here, so we move the cursor forward one
            // space
            mode_buffer += lexeme[0];
            return 1;
          } else {
            // no need to move the cursor, we still have additional regexes to try and
            // match at this very spot
            continueScanAtSamePosition = true;
            return 0;
          }
        }

        function doBeginMatch(match) {
          var lexeme = match[0];
          var new_mode = match.rule;

          if (new_mode.__onBegin) {
            let res = new_mode.__onBegin(match) || {};
            if (res.ignoreMatch)
              return doIgnore(lexeme);
          }

          if (new_mode && new_mode.endSameAsBegin) {
            new_mode.endRe = escape$4( lexeme );
          }

          if (new_mode.skip) {
            mode_buffer += lexeme;
          } else {
            if (new_mode.excludeBegin) {
              mode_buffer += lexeme;
            }
            processBuffer();
            if (!new_mode.returnBegin && !new_mode.excludeBegin) {
              mode_buffer = lexeme;
            }
          }
          startNewMode(new_mode);
          return new_mode.returnBegin ? 0 : lexeme.length;
        }

        function doEndMatch(match) {
          var lexeme = match[0];
          var matchPlusRemainder = codeToHighlight.substr(match.index);
          var end_mode = endOfMode(top, matchPlusRemainder);
          if (!end_mode) { return; }

          var origin = top;
          if (origin.skip) {
            mode_buffer += lexeme;
          } else {
            if (!(origin.returnEnd || origin.excludeEnd)) {
              mode_buffer += lexeme;
            }
            processBuffer();
            if (origin.excludeEnd) {
              mode_buffer = lexeme;
            }
          }
          do {
            if (top.className) {
              emitter.closeNode();
            }
            if (!top.skip && !top.subLanguage) {
              relevance += top.relevance;
            }
            top = top.parent;
          } while (top !== end_mode.parent);
          if (end_mode.starts) {
            if (end_mode.endSameAsBegin) {
              end_mode.starts.endRe = end_mode.endRe;
            }
            startNewMode(end_mode.starts);
          }
          return origin.returnEnd ? 0 : lexeme.length;
        }

        function processContinuations() {
          var list = [];
          for(var current = top; current !== language; current = current.parent) {
            if (current.className) {
              list.unshift(current.className);
            }
          }
          list.forEach(item => emitter.openNode(item));
        }

        var lastMatch = {};
        function processLexeme(text_before_match, match) {

          var err;
          var lexeme = match && match[0];

          // add non-matched text to the current mode buffer
          mode_buffer += text_before_match;

          if (lexeme == null) {
            processBuffer();
            return 0;
          }



          // we've found a 0 width match and we're stuck, so we need to advance
          // this happens when we have badly behaved rules that have optional matchers to the degree that
          // sometimes they can end up matching nothing at all
          // Ref: https://github.com/highlightjs/highlight.js/issues/2140
          if (lastMatch.type=="begin" && match.type=="end" && lastMatch.index == match.index && lexeme === "") {
            // spit the "skipped" character that our regex choked on back into the output sequence
            mode_buffer += codeToHighlight.slice(match.index, match.index + 1);
            if (!SAFE_MODE) {
              err = new Error('0 width match regex');
              err.languageName = languageName;
              err.badRule = lastMatch.rule;
              throw(err);
            }
            return 1;
          }
          lastMatch = match;

          if (match.type==="begin") {
            return doBeginMatch(match);
          } else if (match.type==="illegal" && !ignore_illegals) {
            // illegal match, we do not continue processing
            err = new Error('Illegal lexeme "' + lexeme + '" for mode "' + (top.className || '<unnamed>') + '"');
            err.mode = top;
            throw err;
          } else if (match.type==="end") {
            var processed = doEndMatch(match);
            if (processed != undefined)
              return processed;
          }

          // edge case for when illegal matches $ (end of line) which is technically
          // a 0 width match but not a begin/end match so it's not caught by the
          // first handler (when ignoreIllegals is true)
          if (match.type === "illegal" && lexeme === "") {
            // advance so we aren't stuck in an infinite loop
            return 1;
          }

          // infinite loops are BAD, this is a last ditch catch all. if we have a
          // decent number of iterations yet our index (cursor position in our
          // parsing) still 3x behind our index then something is very wrong
          // so we bail
          if (iterations > 100000 && iterations > match.index * 3) {
            const err = new Error('potential infinite loop, way more iterations than matches');
            throw err;
          }

          /*
          Why might be find ourselves here?  Only one occasion now.  An end match that was
          triggered but could not be completed.  When might this happen?  When an `endSameasBegin`
          rule sets the end rule to a specific match.  Since the overall mode termination rule that's
          being used to scan the text isn't recompiled that means that any match that LOOKS like
          the end (but is not, because it is not an exact match to the beginning) will
          end up here.  A definite end match, but when `doEndMatch` tries to "reapply"
          the end rule and fails to match, we wind up here, and just silently ignore the end.

          This causes no real harm other than stopping a few times too many.
          */

          mode_buffer += lexeme;
          return lexeme.length;
        }

        var language = getLanguage(languageName);
        if (!language) {
          console.error(LANGUAGE_NOT_FOUND.replace("{}", languageName));
          throw new Error('Unknown language: "' + languageName + '"');
        }

        compileLanguage(language);
        var top = continuation || language;
        var continuations = {}; // keep continuations for sub-languages
        var result;
        var emitter = new options.__emitter(options);
        processContinuations();
        var mode_buffer = '';
        var relevance = 0;
        var match;
        var processedCount;
        var index = 0;
        var iterations = 0;
        var continueScanAtSamePosition = false;

        try {
          top.matcher.considerAll();

          for (;;) {
            iterations++;
            if (continueScanAtSamePosition) {
              continueScanAtSamePosition = false;
              // only regexes not matched previously will now be
              // considered for a potential match
            } else {
              top.matcher.lastIndex = index;
              top.matcher.considerAll();
            }
            match = top.matcher.exec(codeToHighlight);
            // console.log("match", match[0], match.rule && match.rule.begin)
            if (!match)
              break;
            let beforeMatch = codeToHighlight.substring(index, match.index);
            processedCount = processLexeme(beforeMatch, match);
            index = match.index + processedCount;
          }
          processLexeme(codeToHighlight.substr(index));
          emitter.closeAllNodes();
          emitter.finalize();
          result = emitter.toHTML();

          return {
            relevance: relevance,
            value: result,
            language: languageName,
            illegal: false,
            emitter: emitter,
            top: top
          };
        } catch (err) {
          if (err.message && err.message.includes('Illegal')) {
            return {
              illegal: true,
              illegalBy: {
                msg: err.message,
                context: codeToHighlight.slice(index-100,index+100),
                mode: err.mode
              },
              sofar: result,
              relevance: 0,
              value: escape$1$1(codeToHighlight),
              emitter: emitter,
            };
          } else if (SAFE_MODE) {
            return {
              relevance: 0,
              value: escape$1$1(codeToHighlight),
              emitter: emitter,
              language: languageName,
              top: top,
              errorRaised: err
            };
          } else {
            throw err;
          }
        }
      }

      // returns a valid highlight result, without actually
      // doing any actual work, auto highlight starts with
      // this and it's possible for small snippets that
      // auto-detection may not find a better match
      function justTextHighlightResult(code) {
        const result = {
          relevance: 0,
          emitter: new options.__emitter(options),
          value: escape$1$1(code),
          illegal: false,
          top: PLAINTEXT_LANGUAGE
        };
        result.emitter.addText(code);
        return result;
      }

      /*
      Highlighting with language detection. Accepts a string with the code to
      highlight. Returns an object with the following properties:

      - language (detected language)
      - relevance (int)
      - value (an HTML string with highlighting markup)
      - second_best (object with the same structure for second-best heuristically
        detected language, may be absent)

      */
      function highlightAuto(code, languageSubset) {
        languageSubset = languageSubset || options.languages || Object.keys(languages);
        var result = justTextHighlightResult(code);
        var second_best = result;
        languageSubset.filter(getLanguage).filter(autoDetection).forEach(function(name) {
          var current = _highlight(name, code, false);
          current.language = name;
          if (current.relevance > second_best.relevance) {
            second_best = current;
          }
          if (current.relevance > result.relevance) {
            second_best = result;
            result = current;
          }
        });
        if (second_best.language) {
          result.second_best = second_best;
        }
        return result;
      }

      /*
      Post-processing of the highlighted markup:

      - replace TABs with something more useful
      - replace real line-breaks with '<br>' for non-pre containers

      */
      function fixMarkup(value) {
        if (!(options.tabReplace || options.useBR)) {
          return value;
        }

        return value.replace(fixMarkupRe, function(match, p1) {
            if (options.useBR && match === '\n') {
              return '<br>';
            } else if (options.tabReplace) {
              return p1.replace(/\t/g, options.tabReplace);
            }
            return '';
        });
      }

      function buildClassName(prevClassName, currentLang, resultLang) {
        var language = currentLang ? aliases[currentLang] : resultLang,
            result   = [prevClassName.trim()];

        if (!prevClassName.match(/\bhljs\b/)) {
          result.push('hljs');
        }

        if (!prevClassName.includes(language)) {
          result.push(language);
        }

        return result.join(' ').trim();
      }

      /*
      Applies highlighting to a DOM node containing code. Accepts a DOM node and
      two optional parameters for fixMarkup.
      */
      function highlightBlock(block) {
        var node, originalStream, result, resultNode, text;
        var language = blockLanguage(block);

        if (shouldNotHighlight(language))
            return;

        fire("before:highlightBlock",
          { block: block, language: language});

        if (options.useBR) {
          node = document.createElement('div');
          node.innerHTML = block.innerHTML.replace(/\n/g, '').replace(/<br[ \/]*>/g, '\n');
        } else {
          node = block;
        }
        text = node.textContent;
        result = language ? highlight(language, text, true) : highlightAuto(text);

        originalStream = nodeStream$1(node);
        if (originalStream.length) {
          resultNode = document.createElement('div');
          resultNode.innerHTML = result.value;
          result.value = mergeStreams$1(originalStream, nodeStream$1(resultNode), text);
        }
        result.value = fixMarkup(result.value);

        fire("after:highlightBlock", { block: block, result: result});

        block.innerHTML = result.value;
        block.className = buildClassName(block.className, language, result.language);
        block.result = {
          language: result.language,
          re: result.relevance
        };
        if (result.second_best) {
          block.second_best = {
            language: result.second_best.language,
            re: result.second_best.relevance
          };
        }
      }

      /*
      Updates highlight.js global options with values passed in the form of an object.
      */
      function configure(user_options) {
        options = inherit$1(options, user_options);
      }

      /*
      Applies highlighting to all <pre><code>..</code></pre> blocks on a page.
      */
      function initHighlighting() {
        if (initHighlighting.called)
          return;
        initHighlighting.called = true;

        var blocks = document.querySelectorAll('pre code');
        ArrayProto.forEach.call(blocks, highlightBlock);
      }

      /*
      Attaches highlighting to the page load event.
      */
      function initHighlightingOnLoad() {
        window.addEventListener('DOMContentLoaded', initHighlighting, false);
      }

      const PLAINTEXT_LANGUAGE = { disableAutodetect: true, name: 'Plain text' };

      function registerLanguage(name, language) {
        var lang;
        try { lang = language(hljs); }
        catch (error) {
          console.error("Language definition for '{}' could not be registered.".replace("{}", name));
          // hard or soft error
          if (!SAFE_MODE) { throw error; } else { console.error(error); }
          // languages that have serious errors are replaced with essentially a
          // "plaintext" stand-in so that the code blocks will still get normal
          // css classes applied to them - and one bad language won't break the
          // entire highlighter
          lang = PLAINTEXT_LANGUAGE;
        }
        // give it a temporary name if it doesn't have one in the meta-data
        if (!lang.name)
          lang.name = name;
        languages[name] = lang;
        lang.rawDefinition = language.bind(null,hljs);

        if (lang.aliases) {
          lang.aliases.forEach(function(alias) {aliases[alias] = name;});
        }
      }

      function listLanguages() {
        return Object.keys(languages);
      }

      /*
        intended usage: When one language truly requires another

        Unlike `getLanguage`, this will throw when the requested language
        is not available.
      */
      function requireLanguage(name) {
        var lang = getLanguage(name);
        if (lang) { return lang; }

        var err = new Error('The \'{}\' language is required, but not loaded.'.replace('{}',name));
        throw err;
      }

      function getLanguage(name) {
        name = (name || '').toLowerCase();
        return languages[name] || languages[aliases[name]];
      }

      function autoDetection(name) {
        var lang = getLanguage(name);
        return lang && !lang.disableAutodetect;
      }

      function addPlugin(plugin, options) {
        plugins.push(plugin);
      }

      function fire(event, args) {
        var cb = event;
        plugins.forEach(function (plugin) {
          if (plugin[cb]) {
            plugin[cb](args);
          }
        });
      }

      /* Interface definition */

      Object.assign(hljs,{
        highlight,
        highlightAuto,
        fixMarkup,
        highlightBlock,
        configure,
        initHighlighting,
        initHighlightingOnLoad,
        registerLanguage,
        listLanguages,
        getLanguage,
        requireLanguage,
        autoDetection,
        inherit: inherit$1,
        addPlugin
      });

      hljs.debugMode = function() { SAFE_MODE = false; };
      hljs.safeMode = function() { SAFE_MODE = true; };
      hljs.versionString = version;

      for (const key in MODES) {
        if (typeof MODES[key] === "object")
          deepFreeze(MODES[key]);
      }

      // merge all the modes/regexs into our main object
      Object.assign(hljs, MODES);

      return hljs;
    };

    // export an "instance" of the highlighter
    var highlight = HLJS({});

    var core = highlight;

    /*
    Language: YAML
    Description: Yet Another Markdown Language
    Author: Stefan Wienert <stwienert@gmail.com>
    Contributors: Carl Baxter <carl@cbax.tech>
    Requires: ruby.js
    Website: https://yaml.org
    Category: common, config
    */
    function yaml(hljs) {
      var LITERALS = 'true false yes no null';

      // Define keys as starting with a word character
      // ...containing word chars, spaces, colons, forward-slashes, hyphens and periods
      // ...and ending with a colon followed immediately by a space, tab or newline.
      // The YAML spec allows for much more than this, but this covers most use-cases.
      var KEY = {
        className: 'attr',
        variants: [
          { begin: '\\w[\\w :\\/.-]*:(?=[ \t]|$)' },
          { begin: '"\\w[\\w :\\/.-]*":(?=[ \t]|$)' }, //double quoted keys
          { begin: '\'\\w[\\w :\\/.-]*\':(?=[ \t]|$)' } //single quoted keys
        ]
      };

      var TEMPLATE_VARIABLES = {
        className: 'template-variable',
        variants: [
          { begin: '\{\{', end: '\}\}' }, // jinja templates Ansible
          { begin: '%\{', end: '\}' } // Ruby i18n
        ]
      };
      var STRING = {
        className: 'string',
        relevance: 0,
        variants: [
          {begin: /'/, end: /'/},
          {begin: /"/, end: /"/},
          {begin: /\S+/}
        ],
        contains: [
          hljs.BACKSLASH_ESCAPE,
          TEMPLATE_VARIABLES
        ]
      };

      var DATE_RE = '[0-9]{4}(-[0-9][0-9]){0,2}';
      var TIME_RE = '([Tt \\t][0-9][0-9]?(:[0-9][0-9]){2})?';
      var FRACTION_RE = '(\\.[0-9]*)?';
      var ZONE_RE = '([ \\t])*(Z|[-+][0-9][0-9]?(:[0-9][0-9])?)?';
      var TIMESTAMP = {
        className: 'number',
        begin: '\\b' + DATE_RE + TIME_RE + FRACTION_RE + ZONE_RE + '\\b',
      };

      return {
        name: 'YAML',
        case_insensitive: true,
        aliases: ['yml', 'YAML'],
        contains: [
          KEY,
          {
            className: 'meta',
            begin: '^---\s*$',
            relevance: 10
          },
          { // multi line string
            // Blocks start with a | or > followed by a newline
            //
            // Indentation of subsequent lines must be the same to
            // be considered part of the block
            className: 'string',
            begin: '[\\|>]([0-9]?[+-])?[ ]*\\n( *)[\\S ]+\\n(\\2[\\S ]+\\n?)*',
          },
          { // Ruby/Rails erb
            begin: '<%[%=-]?', end: '[%-]?%>',
            subLanguage: 'ruby',
            excludeBegin: true,
            excludeEnd: true,
            relevance: 0
          },
          { // local tags
            className: 'type',
            begin: '!' + hljs.UNDERSCORE_IDENT_RE,
          },
          { // data type
            className: 'type',
            begin: '!!' + hljs.UNDERSCORE_IDENT_RE,
          },
          { // fragment id &ref
            className: 'meta',
            begin: '&' + hljs.UNDERSCORE_IDENT_RE + '$',
          },
          { // fragment reference *ref
            className: 'meta',
            begin: '\\*' + hljs.UNDERSCORE_IDENT_RE + '$'
          },
          { // array listing
            className: 'bullet',
          // TODO: remove |$ hack when we have proper look-ahead support
          begin: '\\-(?=[ ]|$)',
            relevance: 0
          },
          hljs.HASH_COMMENT_MODE,
          {
            beginKeywords: LITERALS,
            keywords: {literal: LITERALS}
          },
          TIMESTAMP,
          // numbers are any valid C-style number that
          // sit isolated from other words
          {
            className: 'number',
            begin: hljs.C_NUMBER_RE + '\\b'
          },
          STRING
        ]
      };
    }

    var yaml_1 = yaml;

    /*
    Language: JavaScript
    Description: JavaScript (JS) is a lightweight, interpreted, or just-in-time compiled programming language with first-class functions.
    Category: common, scripting
    Website: https://developer.mozilla.org/en-US/docs/Web/JavaScript
    */

    function javascript(hljs) {
      var FRAGMENT = {
        begin: '<>',
        end: '</>'
      };
      var XML_TAG = {
        begin: /<[A-Za-z0-9\\._:-]+/,
        end: /\/[A-Za-z0-9\\._:-]+>|\/>/
      };
      var IDENT_RE = '[A-Za-z$_][0-9A-Za-z$_]*';
      var KEYWORDS = {
        keyword:
          'in of if for while finally var new function do return void else break catch ' +
          'instanceof with throw case default try this switch continue typeof delete ' +
          'let yield const export super debugger as async await static ' +
          // ECMAScript 6 modules import
          'import from as'
        ,
        literal:
          'true false null undefined NaN Infinity',
        built_in:
          'eval isFinite isNaN parseFloat parseInt decodeURI decodeURIComponent ' +
          'encodeURI encodeURIComponent escape unescape Object Function Boolean Error ' +
          'EvalError InternalError RangeError ReferenceError StopIteration SyntaxError ' +
          'TypeError URIError Number Math Date String RegExp Array Float32Array ' +
          'Float64Array Int16Array Int32Array Int8Array Uint16Array Uint32Array ' +
          'Uint8Array Uint8ClampedArray ArrayBuffer DataView JSON Intl arguments require ' +
          'module console window document Symbol Set Map WeakSet WeakMap Proxy Reflect ' +
          'Promise'
      };
      var NUMBER = {
        className: 'number',
        variants: [
          { begin: '\\b(0[bB][01]+)n?' },
          { begin: '\\b(0[oO][0-7]+)n?' },
          { begin: hljs.C_NUMBER_RE + 'n?' }
        ],
        relevance: 0
      };
      var SUBST = {
        className: 'subst',
        begin: '\\$\\{', end: '\\}',
        keywords: KEYWORDS,
        contains: []  // defined later
      };
      var HTML_TEMPLATE = {
        begin: 'html`', end: '',
        starts: {
          end: '`', returnEnd: false,
          contains: [
            hljs.BACKSLASH_ESCAPE,
            SUBST
          ],
          subLanguage: 'xml',
        }
      };
      var CSS_TEMPLATE = {
        begin: 'css`', end: '',
        starts: {
          end: '`', returnEnd: false,
          contains: [
            hljs.BACKSLASH_ESCAPE,
            SUBST
          ],
          subLanguage: 'css',
        }
      };
      var TEMPLATE_STRING = {
        className: 'string',
        begin: '`', end: '`',
        contains: [
          hljs.BACKSLASH_ESCAPE,
          SUBST
        ]
      };
      SUBST.contains = [
        hljs.APOS_STRING_MODE,
        hljs.QUOTE_STRING_MODE,
        HTML_TEMPLATE,
        CSS_TEMPLATE,
        TEMPLATE_STRING,
        NUMBER,
        hljs.REGEXP_MODE
      ];
      var PARAMS_CONTAINS = SUBST.contains.concat([
        hljs.C_BLOCK_COMMENT_MODE,
        hljs.C_LINE_COMMENT_MODE
      ]);
      var PARAMS = {
        className: 'params',
        begin: /\(/, end: /\)/,
        excludeBegin: true,
        excludeEnd: true,
        contains: PARAMS_CONTAINS
      };

      return {
        name: 'JavaScript',
        aliases: ['js', 'jsx', 'mjs', 'cjs'],
        keywords: KEYWORDS,
        contains: [
          {
            className: 'meta',
            relevance: 10,
            begin: /^\s*['"]use (strict|asm)['"]/
          },
          {
            className: 'meta',
            begin: /^#!/, end: /$/
          },
          hljs.APOS_STRING_MODE,
          hljs.QUOTE_STRING_MODE,
          HTML_TEMPLATE,
          CSS_TEMPLATE,
          TEMPLATE_STRING,
          hljs.C_LINE_COMMENT_MODE,
          hljs.COMMENT(
            '/\\*\\*',
            '\\*/',
            {
              relevance : 0,
              contains : [
                {
                  className : 'doctag',
                  begin : '@[A-Za-z]+',
                  contains : [
                    {
                      className: 'type',
                      begin: '\\{',
                      end: '\\}',
                      relevance: 0
                    },
                    {
                      className: 'variable',
                      begin: IDENT_RE + '(?=\\s*(-)|$)',
                      endsParent: true,
                      relevance: 0
                    },
                    // eat spaces (not newlines) so we can find
                    // types or variables
                    {
                      begin: /(?=[^\n])\s/,
                      relevance: 0
                    },
                  ]
                }
              ]
            }
          ),
          hljs.C_BLOCK_COMMENT_MODE,
          NUMBER,
          { // object attr container
            begin: /[{,\n]\s*/, relevance: 0,
            contains: [
              {
                begin: IDENT_RE + '\\s*:', returnBegin: true,
                relevance: 0,
                contains: [{className: 'attr', begin: IDENT_RE, relevance: 0}]
              }
            ]
          },
          { // "value" container
            begin: '(' + hljs.RE_STARTERS_RE + '|\\b(case|return|throw)\\b)\\s*',
            keywords: 'return throw case',
            contains: [
              hljs.C_LINE_COMMENT_MODE,
              hljs.C_BLOCK_COMMENT_MODE,
              hljs.REGEXP_MODE,
              {
                className: 'function',
                begin: '(\\(.*?\\)|' + IDENT_RE + ')\\s*=>', returnBegin: true,
                end: '\\s*=>',
                contains: [
                  {
                    className: 'params',
                    variants: [
                      {
                        begin: IDENT_RE
                      },
                      {
                        begin: /\(\s*\)/,
                      },
                      {
                        begin: /\(/, end: /\)/,
                        excludeBegin: true, excludeEnd: true,
                        keywords: KEYWORDS,
                        contains: PARAMS_CONTAINS
                      }
                    ]
                  }
                ]
              },
              { // could be a comma delimited list of params to a function call
                begin: /,/, relevance: 0,
              },
              {
                className: '',
                begin: /\s/,
                end: /\s*/,
                skip: true,
              },
              { // JSX
                variants: [
                  { begin: FRAGMENT.begin, end: FRAGMENT.end },
                  { begin: XML_TAG.begin, end: XML_TAG.end }
                ],
                subLanguage: 'xml',
                contains: [
                  {
                    begin: XML_TAG.begin, end: XML_TAG.end, skip: true,
                    contains: ['self']
                  }
                ]
              },
            ],
            relevance: 0
          },
          {
            className: 'function',
            beginKeywords: 'function', end: /\{/, excludeEnd: true,
            contains: [
              hljs.inherit(hljs.TITLE_MODE, {begin: IDENT_RE}),
              PARAMS
            ],
            illegal: /\[|%/
          },
          {
            begin: /\$[(.]/ // relevance booster for a pattern common to JS libs: `$(something)` and `$.something`
          },

          hljs.METHOD_GUARD,
          { // ES6 class
            className: 'class',
            beginKeywords: 'class', end: /[{;=]/, excludeEnd: true,
            illegal: /[:"\[\]]/,
            contains: [
              {beginKeywords: 'extends'},
              hljs.UNDERSCORE_TITLE_MODE
            ]
          },
          {
            beginKeywords: 'constructor', end: /\{/, excludeEnd: true
          },
          {
            begin:'(get|set)\\s+(?=' + IDENT_RE+ '\\()',
            end: /{/,
            keywords: "get set",
            contains: [
              hljs.inherit(hljs.TITLE_MODE, {begin: IDENT_RE}),
              { begin: /\(\)/ }, // eat to avoid empty params
              PARAMS
            ]

          }
        ],
        illegal: /#(?!!)/
      };
    }

    var javascript_1 = javascript;

    /*
    Language: JSON
    Description: JSON (JavaScript Object Notation) is a lightweight data-interchange format.
    Author: Ivan Sagalaev <maniac@softwaremaniacs.org>
    Website: http://www.json.org
    Category: common, protocols
    */

    function json(hljs) {
      var LITERALS = {literal: 'true false null'};
      var ALLOWED_COMMENTS = [
        hljs.C_LINE_COMMENT_MODE,
        hljs.C_BLOCK_COMMENT_MODE
      ];
      var TYPES = [
        hljs.QUOTE_STRING_MODE,
        hljs.C_NUMBER_MODE
      ];
      var VALUE_CONTAINER = {
        end: ',', endsWithParent: true, excludeEnd: true,
        contains: TYPES,
        keywords: LITERALS
      };
      var OBJECT = {
        begin: '{', end: '}',
        contains: [
          {
            className: 'attr',
            begin: /"/, end: /"/,
            contains: [hljs.BACKSLASH_ESCAPE],
            illegal: '\\n',
          },
          hljs.inherit(VALUE_CONTAINER, {begin: /:/})
        ].concat(ALLOWED_COMMENTS),
        illegal: '\\S'
      };
      var ARRAY = {
        begin: '\\[', end: '\\]',
        contains: [hljs.inherit(VALUE_CONTAINER)], // inherit is a workaround for a bug that makes shared modes with endsWithParent compile only the ending of one of the parents
        illegal: '\\S'
      };
      TYPES.push(OBJECT, ARRAY);
      ALLOWED_COMMENTS.forEach(function(rule) {
        TYPES.push(rule);
      });
      return {
        name: 'JSON',
        contains: TYPES,
        keywords: LITERALS,
        illegal: '\\S'
      };
    }

    var json_1 = json;

    // from https://raw.githubusercontent.com/highlightjs/highlight.js/master/src/styles/github.css
    const HLJS$1 = `
  /*

github.com style (c) Vasily Polovnyov <vast@whiteants.net>

*/

  .hljs {
    display: block;
    overflow-x: auto;
    padding: 0.5em;
    color: #333;
    background: #f8f8f8;
  }

  .hljs-comment,
  .hljs-quote {
    color: #998;
    font-style: italic;
  }

  .hljs-keyword,
  .hljs-selector-tag,
  .hljs-subst {
    color: #333;
    font-weight: bold;
  }

  .hljs-number,
  .hljs-literal,
  .hljs-variable,
  .hljs-template-variable,
  .hljs-tag .hljs-attr {
    color: #008080;
  }

  .hljs-string,
  .hljs-doctag {
    color: #d14;
  }

  .hljs-title,
  .hljs-section,
  .hljs-selector-id {
    color: #900;
    font-weight: bold;
  }

  .hljs-subst {
    font-weight: normal;
  }

  .hljs-type,
  .hljs-class .hljs-title {
    color: #458;
    font-weight: bold;
  }

  .hljs-tag,
  .hljs-name,
  .hljs-attribute {
    color: #000080;
    font-weight: normal;
  }

  .hljs-regexp,
  .hljs-link {
    color: #009926;
  }

  .hljs-symbol,
  .hljs-bullet {
    color: #990073;
  }

  .hljs-built_in,
  .hljs-builtin-name {
    color: #0086b3;
  }

  .hljs-meta {
    color: #999;
    font-weight: bold;
  }

  .hljs-deletion {
    background: #fdd;
  }

  .hljs-addition {
    background: #dfd;
  }

  .hljs-emphasis {
    font-style: italic;
  }

  .hljs-strong {
    font-weight: bold;
  }
`;
    // From https://github.com/sindresorhus/github-markdown-css/blob/gh-pages/github-markdown.css
    const GFM1 = `
  @font-face {
    font-family: octicons-link;
    src: url(data:font/woff;charset=utf-8;base64,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)
      format("woff");
  }
`;
    const GFM2 = `
  .markdown-body .octicon {
    display: inline-block;
    fill: currentColor;
    vertical-align: text-bottom;
  }

  .markdown-body .anchor {
    float: left;
    line-height: 1;
    margin-left: -20px;
    padding-right: 4px;
  }

  .markdown-body .anchor:focus {
    outline: none;
  }

  .markdown-body h1 .octicon-link,
  .markdown-body h2 .octicon-link,
  .markdown-body h3 .octicon-link,
  .markdown-body h4 .octicon-link,
  .markdown-body h5 .octicon-link,
  .markdown-body h6 .octicon-link {
    color: #1b1f23;
    vertical-align: middle;
    visibility: hidden;
  }

  .markdown-body h1:hover .anchor,
  .markdown-body h2:hover .anchor,
  .markdown-body h3:hover .anchor,
  .markdown-body h4:hover .anchor,
  .markdown-body h5:hover .anchor,
  .markdown-body h6:hover .anchor {
    text-decoration: none;
  }

  .markdown-body h1:hover .anchor .octicon-link,
  .markdown-body h2:hover .anchor .octicon-link,
  .markdown-body h3:hover .anchor .octicon-link,
  .markdown-body h4:hover .anchor .octicon-link,
  .markdown-body h5:hover .anchor .octicon-link,
  .markdown-body h6:hover .anchor .octicon-link {
    visibility: visible;
  }
  .markdown-body {
    -ms-text-size-adjust: 100%;
    -webkit-text-size-adjust: 100%;
    line-height: 1.5;
    font-family: -apple-system, BlinkMacSystemFont, Segoe UI, Helvetica, Arial,
      sans-serif, Apple Color Emoji, Segoe UI Emoji;
    font-size: 16px;
    line-height: 1.5;
    word-wrap: break-word;
  }

  .markdown-body .pl-c {
    color: #6a737d;
  }

  .markdown-body .pl-c1,
  .markdown-body .pl-s .pl-v {
    color: #005cc5;
  }

  .markdown-body .pl-e,
  .markdown-body .pl-en {
    color: #6f42c1;
  }

  .markdown-body .pl-s .pl-s1,
  .markdown-body .pl-smi {
    color: #24292e;
  }

  .markdown-body .pl-ent {
    color: #22863a;
  }

  .markdown-body .pl-k {
    color: #d73a49;
  }

  .markdown-body .pl-pds,
  .markdown-body .pl-s,
  .markdown-body .pl-s .pl-pse .pl-s1,
  .markdown-body .pl-sr,
  .markdown-body .pl-sr .pl-cce,
  .markdown-body .pl-sr .pl-sra,
  .markdown-body .pl-sr .pl-sre {
    color: #032f62;
  }

  .markdown-body .pl-smw,
  .markdown-body .pl-v {
    color: #e36209;
  }

  .markdown-body .pl-bu {
    color: #b31d28;
  }

  .markdown-body .pl-ii {
    color: #fafbfc;
    background-color: #b31d28;
  }

  .markdown-body .pl-c2 {
    color: #fafbfc;
    background-color: #d73a49;
  }

  .markdown-body .pl-c2:before {
    content: "^M";
  }

  .markdown-body .pl-sr .pl-cce {
    font-weight: 700;
    color: #22863a;
  }

  .markdown-body .pl-ml {
    color: #735c0f;
  }

  .markdown-body .pl-mh,
  .markdown-body .pl-mh .pl-en,
  .markdown-body .pl-ms {
    font-weight: 700;
    color: #005cc5;
  }

  .markdown-body .pl-mi {
    font-style: italic;
    color: #24292e;
  }

  .markdown-body .pl-mb {
    font-weight: 700;
    color: #24292e;
  }

  .markdown-body .pl-md {
    color: #b31d28;
    background-color: #ffeef0;
  }

  .markdown-body .pl-mi1 {
    color: #22863a;
    background-color: #f0fff4;
  }

  .markdown-body .pl-mc {
    color: #e36209;
    background-color: #ffebda;
  }

  .markdown-body .pl-mi2 {
    color: #f6f8fa;
    background-color: #005cc5;
  }

  .markdown-body .pl-mdr {
    font-weight: 700;
    color: #6f42c1;
  }

  .markdown-body .pl-ba {
    color: #586069;
  }

  .markdown-body .pl-sg {
    color: #959da5;
  }

  .markdown-body .pl-corl {
    text-decoration: underline;
    color: #032f62;
  }

  .markdown-body details {
    display: block;
  }

  .markdown-body summary {
    display: list-item;
  }

  .markdown-body a {
    background-color: initial;
  }

  .markdown-body a:active,
  .markdown-body a:hover {
    outline-width: 0;
  }

  .markdown-body strong {
    font-weight: inherit;
    font-weight: bolder;
  }

  .markdown-body h1 {
    font-size: 2em;
    margin: 0.67em 0;
  }

  .markdown-body img {
    border-style: none;
  }

  .markdown-body code,
  .markdown-body kbd,
  .markdown-body pre {
    font-family: monospace, monospace;
    font-size: 1em;
    background-color: var(--material-secondary-background-color);
  }

  .markdown-body hr {
    box-sizing: initial;
    height: 0;
    overflow: visible;
  }

  .markdown-body input {
    font: inherit;
    margin: 0;
  }

  .markdown-body input {
    overflow: visible;
  }

  .markdown-body [type="checkbox"] {
    box-sizing: border-box;
    padding: 0;
  }

  .markdown-body * {
    box-sizing: border-box;
  }

  .markdown-body input {
    font-family: inherit;
    font-size: inherit;
    line-height: inherit;
  }

  .markdown-body a {
    color: var(--link-text-color, var(--accent-color));
    font-weight: 600;
    text-decoration: none;
  }

  .markdown-body a:hover {
    text-decoration: underline;
  }

  .markdown-body strong {
    font-weight: 600;
  }

  .markdown-body hr {
    height: 0;
    margin: 15px 0;
    overflow: hidden;
    background: transparent;
    border: 0;
    border-bottom: 1px solid #dfe2e5;
  }

  .markdown-body hr:after,
  .markdown-body hr:before {
    display: table;
    content: "";
  }

  .markdown-body hr:after {
    clear: both;
  }

  .markdown-body table {
    border-spacing: 0;
    border-collapse: collapse;
  }

  .markdown-body td,
  .markdown-body th {
    padding: 0;
  }

  .markdown-body details summary {
    cursor: pointer;
  }

  .markdown-body kbd {
    display: inline-block;
    padding: 3px 5px;
    font: 11px SFMono-Regular, Consolas, Liberation Mono, Menlo, monospace;
    line-height: 10px;
    color: #444d56;
    vertical-align: middle;
    background-color: var(--material-secondary-background-color);
    border: 1px solid #d1d5da;
    border-radius: 3px;
    box-shadow: inset 0 -1px 0 #d1d5da;
  }

  .markdown-body h1,
  .markdown-body h2,
  .markdown-body h3,
  .markdown-body h4,
  .markdown-body h5,
  .markdown-body h6 {
    margin-top: 0;
    margin-bottom: 0;
  }

  .markdown-body h1 {
    font-size: 32px;
  }

  .markdown-body h1,
  .markdown-body h2 {
    font-weight: 600;
  }

  .markdown-body h2 {
    font-size: 24px;
  }

  .markdown-body h3 {
    font-size: 20px;
  }

  .markdown-body h3,
  .markdown-body h4 {
    font-weight: 600;
  }

  .markdown-body h4 {
    font-size: 16px;
  }

  .markdown-body h5 {
    font-size: 14px;
  }

  .markdown-body h5,
  .markdown-body h6 {
    font-weight: 600;
  }

  .markdown-body h6 {
    font-size: 12px;
  }

  .markdown-body p {
    margin-top: 0;
    margin-bottom: 10px;
  }

  .markdown-body blockquote {
    margin: 0;
  }

  .markdown-body ol,
  .markdown-body ul {
    padding-left: 0;
    margin-top: 0;
    margin-bottom: 0;
  }

  .markdown-body ol ol,
  .markdown-body ul ol {
    list-style-type: lower-roman;
  }

  .markdown-body ol ol ol,
  .markdown-body ol ul ol,
  .markdown-body ul ol ol,
  .markdown-body ul ul ol {
    list-style-type: lower-alpha;
  }

  .markdown-body dd {
    margin-left: 0;
  }

  .markdown-body code,
  .markdown-body pre {
    font-family: SFMono-Regular, Consolas, Liberation Mono, Menlo, monospace;
    font-size: 12px;
  }

  .markdown-body pre {
    margin-top: 0;
    margin-bottom: 0;
  }

  .markdown-body input::-webkit-inner-spin-button,
  .markdown-body input::-webkit-outer-spin-button {
    margin: 0;
    -webkit-appearance: none;
    appearance: none;
  }

  .markdown-body .border {
    border: 1px solid #e1e4e8 !important;
  }

  .markdown-body .border-0 {
    border: 0 !important;
  }

  .markdown-body .border-bottom {
    border-bottom: 1px solid #e1e4e8 !important;
  }

  .markdown-body .rounded-1 {
    border-radius: 3px !important;
  }

  .markdown-body .bg-white {
    background-color: #fff !important;
  }

  .markdown-body .bg-gray-light {
    background-color: #fafbfc !important;
  }

  .markdown-body .text-gray-light {
    color: #6a737d !important;
  }

  .markdown-body .mb-0 {
    margin-bottom: 0 !important;
  }

  .markdown-body .my-2 {
    margin-top: 8px !important;
    margin-bottom: 8px !important;
  }

  .markdown-body .pl-0 {
    padding-left: 0 !important;
  }

  .markdown-body .py-0 {
    padding-top: 0 !important;
    padding-bottom: 0 !important;
  }

  .markdown-body .pl-1 {
    padding-left: 4px !important;
  }

  .markdown-body .pl-2 {
    padding-left: 8px !important;
  }

  .markdown-body .py-2 {
    padding-top: 8px !important;
    padding-bottom: 8px !important;
  }

  .markdown-body .pl-3,
  .markdown-body .px-3 {
    padding-left: 16px !important;
  }

  .markdown-body .px-3 {
    padding-right: 16px !important;
  }

  .markdown-body .pl-4 {
    padding-left: 24px !important;
  }

  .markdown-body .pl-5 {
    padding-left: 32px !important;
  }

  .markdown-body .pl-6 {
    padding-left: 40px !important;
  }

  .markdown-body .f6 {
    font-size: 12px !important;
  }

  .markdown-body .lh-condensed {
    line-height: 1.25 !important;
  }

  .markdown-body .text-bold {
    font-weight: 600 !important;
  }

  .markdown-body .pl-7 {
    padding-left: 48px !important;
  }

  .markdown-body .pl-8 {
    padding-left: 64px !important;
  }

  .markdown-body .pl-9 {
    padding-left: 80px !important;
  }

  .markdown-body .pl-10 {
    padding-left: 96px !important;
  }

  .markdown-body .pl-11 {
    padding-left: 112px !important;
  }

  .markdown-body .pl-12 {
    padding-left: 128px !important;
  }

  .markdown-body hr {
    border-bottom-color: #eee;
  }

  .markdown-body kbd {
    display: inline-block;
    padding: 3px 5px;
    font: 11px SFMono-Regular, Consolas, Liberation Mono, Menlo, monospace;
    line-height: 10px;
    color: #444d56;
    vertical-align: middle;
    background-color: var(--material-secondary-background-color);
    border: 1px solid #d1d5da;
    border-radius: 3px;
    box-shadow: inset 0 -1px 0 #d1d5da;
  }

  .markdown-body:after,
  .markdown-body:before {
    display: table;
    content: "";
  }

  .markdown-body:after {
    clear: both;
  }

  .markdown-body > :first-child {
    margin-top: 0 !important;
  }

  .markdown-body > :last-child {
    margin-bottom: 0 !important;
  }

  .markdown-body a:not([href]) {
    color: inherit;
    text-decoration: none;
  }
`;
    const GFM3 = `
  .markdown-body blockquote,
  .markdown-body details,
  .markdown-body dl,
  .markdown-body ol,
  .markdown-body p,
  .markdown-body pre,
  .markdown-body table,
  .markdown-body ul {
    margin-top: 0;
    margin-bottom: 16px;
  }

  .markdown-body hr {
    height: 0.25em;
    padding: 0;
    margin: 24px 0;
    background-color: #e1e4e8;
    border: 0;
  }

  .markdown-body blockquote {
    padding: 0 1em;
    color: #6a737d;
    border-left: 0.25em solid #dfe2e5;
  }

  .markdown-body blockquote > :first-child {
    margin-top: 0;
  }

  .markdown-body blockquote > :last-child {
    margin-bottom: 0;
  }

  .markdown-body h1,
  .markdown-body h2,
  .markdown-body h3,
  .markdown-body h4,
  .markdown-body h5,
  .markdown-body h6 {
    margin-top: 24px;
    margin-bottom: 16px;
    font-weight: 600;
    line-height: 1.25;
  }

  .markdown-body h1 {
    font-size: 2em;
  }

  .markdown-body h1,
  .markdown-body h2 {
    padding-bottom: 0.3em;
    border-bottom: 1px solid #eaecef;
  }

  .markdown-body h2 {
    font-size: 1.5em;
  }

  .markdown-body h3 {
    font-size: 1.25em;
  }

  .markdown-body h4 {
    font-size: 1em;
  }

  .markdown-body h5 {
    font-size: 0.875em;
  }

  .markdown-body h6 {
    font-size: 0.85em;
    color: #6a737d;
  }

  .markdown-body ol,
  .markdown-body ul {
    padding-left: 2em;
  }

  .markdown-body ol ol,
  .markdown-body ol ul,
  .markdown-body ul ol,
  .markdown-body ul ul {
    margin-top: 0;
    margin-bottom: 0;
  }

  .markdown-body li {
    word-wrap: break-all;
  }

  .markdown-body li > p {
    margin-top: 16px;
  }

  .markdown-body li + li {
    margin-top: 0.25em;
  }

  .markdown-body dl {
    padding: 0;
  }

  .markdown-body dl dt {
    padding: 0;
    margin-top: 16px;
    font-size: 1em;
    font-style: italic;
    font-weight: 600;
  }

  .markdown-body dl dd {
    padding: 0 16px;
    margin-bottom: 16px;
  }

  .markdown-body table {
    display: block;
    width: 100%;
    overflow: auto;
  }

  .markdown-body table th {
    font-weight: 600;
  }

  .markdown-body table td,
  .markdown-body table th {
    padding: 6px 13px;
    border: 1px solid #dfe2e5;
  }

  .markdown-body table th {
    background-color: var(--primary-background-color, #fff);
  }

  .markdown-body table tr {
    background-color: hsla(var(--primary-background-color, #fff), 0.1);
    border-top: 1px solid #c6cbd1;
  }

  .markdown-body table tr:nth-child(2n) {
    background-color: var(--primary-background-color, #fff);
  }

  .markdown-body img {
    max-width: 100%;
    box-sizing: initial;
    background-color: #fff;
    border-radius: var(--ha-card-border-radius);
  }

  .markdown-body img[align="right"] {
    padding-left: 20px;
  }

  .markdown-body img[align="left"] {
    padding-right: 20px;
  }

  .markdown-body code {
    padding: 0.2em 0.4em;
    margin: 0;
    font-size: 85%;
    background-color: var(--material-secondary-background-color);
    border-radius: 3px;
  }

  .markdown-body pre {
    word-wrap: normal;
  }

  .markdown-body pre > code {
    padding: 0;
    margin: 0;
    font-size: 100%;
    word-break: normal;
    white-space: pre;
    background: transparent;
    border: 0;
  }

  .markdown-body .highlight {
    margin-bottom: 16px;
  }

  .markdown-body .highlight pre {
    margin-bottom: 0;
    word-break: normal;
  }

  .markdown-body .highlight pre,
  .markdown-body pre {
    padding: 16px;
    overflow: auto;
    font-size: 85%;
    line-height: 1.45;
    background-color: var(--material-secondary-background-color, #f6f8fa);
    border-radius: var(--ha-card-border-radius, 3px);
  }

  .markdown-body pre code {
    display: inline;
    max-width: auto;
    padding: 0;
    margin: 0;
    overflow: visible;
    line-height: inherit;
    word-wrap: normal;
    background-color: initial;
    border: 0;
  }

  .markdown-body .commit-tease-sha {
    display: inline-block;
    font-family: SFMono-Regular, Consolas, Liberation Mono, Menlo, monospace;
    font-size: 90%;
    color: #444d56;
  }

  .markdown-body .full-commit .btn-outline:not(:disabled):hover {
    color: #005cc5;
    border-color: #005cc5;
  }

  .markdown-body .blob-wrapper {
    overflow-x: auto;
    overflow-y: hidden;
  }

  .markdown-body .blob-wrapper-embedded {
    max-height: 240px;
    overflow-y: auto;
  }

  .markdown-body .blob-num {
    width: 1%;
    min-width: 50px;
    padding-right: 10px;
    padding-left: 10px;
    font-family: SFMono-Regular, Consolas, Liberation Mono, Menlo, monospace;
    font-size: 12px;
    line-height: 20px;
    color: rgba(27, 31, 35, 0.3);
    text-align: right;
    white-space: nowrap;
    vertical-align: top;
    cursor: pointer;
    -webkit-user-select: none;
    -moz-user-select: none;
    -ms-user-select: none;
    user-select: none;
  }

  .markdown-body .blob-num:hover {
    color: rgba(27, 31, 35, 0.6);
  }

  .markdown-body .blob-num:before {
    content: attr(data-line-number);
  }

  .markdown-body .blob-code {
    position: relative;
    padding-right: 10px;
    padding-left: 10px;
    line-height: 20px;
    vertical-align: top;
  }

  .markdown-body .blob-code-inner {
    overflow: visible;
    font-family: SFMono-Regular, Consolas, Liberation Mono, Menlo, monospace;
    font-size: 12px;
    color: #24292e;
    word-wrap: normal;
    white-space: pre;
  }

  .markdown-body .pl-token.active,
  .markdown-body .pl-token:hover {
    cursor: pointer;
    background: #ffea7f;
  }

  .markdown-body :checked + .radio-label {
    position: relative;
    z-index: 1;
    border-color: var(--link-text-color, var(--accent-color));
  }

  .markdown-body
    .select-menu-item
    input[type="radio"]:not(:checked)
    + .octicon-check,
  .markdown-body
    .select-menu-item
    input[type="radio"]:not(:checked)
    + .octicon-circle-slash {
    visibility: hidden;
  }

  .markdown-body .pl-7 {
    padding-left: 48px !important;
  }

  .markdown-body .pl-8 {
    padding-left: 64px !important;
  }

  .markdown-body .pl-9 {
    padding-left: 80px !important;
  }

  .markdown-body .pl-10 {
    padding-left: 96px !important;
  }

  .markdown-body .pl-11 {
    padding-left: 112px !important;
  }

  .markdown-body .pl-12 {
    padding-left: 128px !important;
  }

  .markdown-body .tab-size[data-tab-size="1"] {
    -moz-tab-size: 1;
    tab-size: 1;
  }

  .markdown-body .tab-size[data-tab-size="2"] {
    -moz-tab-size: 2;
    tab-size: 2;
  }

  .markdown-body .tab-size[data-tab-size="3"] {
    -moz-tab-size: 3;
    tab-size: 3;
  }

  .markdown-body .tab-size[data-tab-size="4"] {
    -moz-tab-size: 4;
    tab-size: 4;
  }

  .markdown-body .tab-size[data-tab-size="5"] {
    -moz-tab-size: 5;
    tab-size: 5;
  }

  .markdown-body .tab-size[data-tab-size="6"] {
    -moz-tab-size: 6;
    tab-size: 6;
  }

  .markdown-body .tab-size[data-tab-size="7"] {
    -moz-tab-size: 7;
    tab-size: 7;
  }

  .markdown-body .tab-size[data-tab-size="8"] {
    -moz-tab-size: 8;
    tab-size: 8;
  }

  .markdown-body .tab-size[data-tab-size="9"] {
    -moz-tab-size: 9;
    tab-size: 9;
  }

  .markdown-body .tab-size[data-tab-size="10"] {
    -moz-tab-size: 10;
    tab-size: 10;
  }

  .markdown-body .tab-size[data-tab-size="11"] {
    -moz-tab-size: 11;
    tab-size: 11;
  }

  .markdown-body .tab-size[data-tab-size="12"] {
    -moz-tab-size: 12;
    tab-size: 12;
  }

  .markdown-body .task-list-item {
    list-style-type: none;
  }

  .markdown-body .task-list-item + .task-list-item {
    margin-top: 3px;
  }

  .markdown-body .task-list-item input {
    margin: 0 0.2em 0.25em -1.6em;
    vertical-align: middle;
  }
`;
    const GFM = `${GFM1}${GFM2}${GFM3}`;

    core.registerLanguage("yaml", yaml_1);
    core.registerLanguage("javascript", javascript_1);
    core.registerLanguage("json", json_1);
    const marked$1 = marked_1;
    marked$1.setOptions({
        highlight: function (code, lang) {
            if (lang && core.getLanguage(lang)) {
                return core.highlight(lang, code, true).value;
            }
            else {
                return core.highlightAuto(code).value;
            }
        },
        breaks: true,
        gfm: true,
        tables: true,
        langPrefix: "",
    });
    class markdown {
        static convert(input) {
            return marked$1(input);
        }
        static html(input, repo) {
            // Convert emoji short codes to real emojis
            input = nodeEmoji.emojify(input);
            // Handle convertion to raw GitHub URL
            input = input.replace(/(https:\/\/github\.com\/.*.\/blob*.[^\s]+)/g, function (x) {
                let url = x
                    .replace("https://github.com/", "https://raw.githubusercontent.com/")
                    .replace("/blob/", "/");
                return url;
            });
            // Handle relative links
            input = input.replace(/\!\[*.*\]\(\w*\.\w*\)/g, function (x) {
                let url = x
                    .replace("(", `(https://raw.githubusercontent.com/${repo === null || repo === void 0 ? void 0 : repo.full_name}/master/`)
                    .replace("/blob/", "/");
                return url;
            });
            const content = document.createElement("div");
            content.className = "markdown-body";
            content.innerHTML = purify.sanitize(marked$1(input), {
                css: false,
            }).replace(/\<a href="http\w:\/\/.*.\">.*.\<\/a>\W/g, function (x) {
                return x
                    .replace(/<a href=/gm, "<hacs-link url=")
                    .replace(/<\/a>/gm, "</hacs-link>");
            });
            const style = document.createElement("style");
            style.innerText = `${HLJS$1}${GFM}`;
            return html `${style}${content} `;
        }
    }

    /*
    Simple logger class to make logging consistent

    Usage:

    Basic: const logger = new Logger()
    Advanced: const logger = new Logger("main")

    logger.info("My message")
    logger.error("My message")
    logger.warning("My message")

    const obj = {test: "string"}
    logger.info(obj)

    */
    class Logger {
        constructor(section) {
            const name = "HACS";
            this.section = section;
            if (section === undefined)
                this.prefix = name;
            else
                this.prefix = `${name}.${section}`;
        }
        info(content, object) {
            if (content instanceof Object)
                console.log(`[${this.prefix}] `, content);
            else
                console.log(`[${this.prefix}] ${content}`, object || "");
        }
        warning(content, object) {
            if (content instanceof Object)
                console.warn(`[${this.prefix}] `, content);
            else
                console.warn(`[${this.prefix}] ${content}`, object || "");
        }
        error(content, object) {
            if (content instanceof Object)
                console.error(`[${this.prefix}] `, content);
            else
                console.error(`[${this.prefix}] ${content}`, object || "");
        }
    }

    function RepositoryWebSocketAction(hass, repository, Action, Data = undefined) {
        let message;
        if (Data !== undefined) {
            message = {
                type: "hacs/repository/data",
                action: Action,
                repository: repository,
                data: Data
            };
        }
        else {
            message = {
                type: "hacs/repository",
                action: Action,
                repository: repository
            };
        }
        hass.connection.sendMessage(message);
    }
    function scrollToTarget(element, target) {
        const top = 0;
        const scroller = target;
        const easingFn = function easeOutQuad(t, b, c, d) {
            /* eslint-disable no-param-reassign, space-infix-ops, no-mixed-operators */
            t /= d;
            return -c * t * (t - 2) + b;
            /* eslint-enable no-param-reassign, space-infix-ops, no-mixed-operators */
        };
        const animationId = Math.random();
        const duration = 200;
        const startTime = Date.now();
        const currentScrollTop = scroller.scrollTop;
        const deltaScrollTop = top - currentScrollTop;
        element._currentAnimationId = animationId;
        (function updateFrame() {
            const now = Date.now();
            const elapsedTime = now - startTime;
            if (elapsedTime > duration) {
                scroller.scrollTop = top;
            }
            else if (element._currentAnimationId === animationId) {
                scroller.scrollTop = easingFn(elapsedTime, currentScrollTop, deltaScrollTop, duration);
                requestAnimationFrame(updateFrame.bind(element));
            }
        }.call(element));
    }
    const navigate = (_node, path) => {
        history.pushState(null, "", path);
    };
    function isnullorempty(test) {
        if (test === undefined)
            return true;
        if (test === null)
            return true;
        if (test === "")
            return true;
        if (test === 0)
            return true;
        if (String(test).length === 0)
            return true;
        return false;
    }

    class Hacs {
        constructor(configuration, repositories, status) {
            this.logger = new Logger();
            this.localize = function (str, s, r) {
                return localize(str, s, r);
            };
            this.emojify = function (string) {
                return nodeEmoji.emojify(string);
            };
            this.scrollToTarget = function (element, target) {
                scrollToTarget(element, target);
            };
            this.navigate = (_node, path) => {
                navigate(_node, path);
            };
            this.markdown = function (input) {
                return markdown.convert(input);
            };
            this.isnullorempty = function (test) {
                return isnullorempty(test);
            };
            this.RepositoryWebSocketAction = function (hass, repository, action, data) {
                RepositoryWebSocketAction(hass, repository, action, data);
            };
            this.configuration = configuration;
            this.repositories = repositories;
            this.status = status;
        }
        RelativeTimeSince(input) {
            let current = new Date().getTime();
            let target = Date.parse(input);
            var value;
            const msPerMinute = 60 * 1000;
            const msPerHour = msPerMinute * 60;
            const msPerDay = msPerHour * 24;
            const msPerMonth = msPerDay * 30;
            const msPerYear = msPerDay * 365;
            const elapsed = current - target;
            if (elapsed < msPerMinute) {
                value = Math.round(elapsed / 1000);
                return `${value === 1
                ? this.localize(`time.one_second_ago`)
                : this.localize("time.x_seconds_ago", "{x}", String(value))}`;
            }
            else if (elapsed < msPerHour) {
                value = Math.round(elapsed / msPerMinute);
                return `${value === 1
                ? this.localize(`time.one_minute_ago`)
                : this.localize("time.x_minutes_ago", "{x}", String(value))}`;
            }
            else if (elapsed < msPerDay) {
                value = Math.round(elapsed / msPerHour);
                return `${value === 1
                ? this.localize(`time.one_hour_ago`)
                : this.localize("time.x_hours_ago", "{x}", String(value))}`;
            }
            else if (elapsed < msPerMonth) {
                value = Math.round(elapsed / msPerDay);
                return `${value === 1
                ? this.localize(`time.one_day_ago`)
                : this.localize("time.x_days_ago", "{x}", String(value))}`;
            }
            else if (elapsed < msPerYear) {
                value = Math.round(elapsed / msPerMonth);
                return `${value === 1
                ? this.localize(`time.one_month_ago`)
                : this.localize("time.x_months_ago", "{x}", String(value))}`;
            }
            else {
                value = Math.round(elapsed / msPerYear);
                return `${value === 1
                ? this.localize(`time.one_year_ago`)
                : this.localize("time.x_years_ago", "{x}", String(value))}`;
            }
        }
    }

    const _haStyle = css `
  :host {
    @apply --paper-font-body1;
  }

  app-header-layout,
  ha-app-layout {
    background-color: var(--primary-background-color);
  }

  app-header,
  app-toolbar {
    background-color: var(--app-header-background-color, var(--primary-color));
    font-weight: 400;
    color: var(--app-header-text-color, var(--text-primary-color, white));
  }

  app-toolbar ha-menu-button + [main-title],
  app-toolbar ha-paper-icon-button-arrow-prev + [main-title],
  app-toolbar ha-icon-button + [main-title] {
    margin-left: 24px;
  }

  button.link {
    background: none;
    color: inherit;
    border: none;
    padding: 0;
    font: inherit;
    text-align: left;
    text-decoration: underline;
    cursor: pointer;
  }

  .card-actions a {
    text-decoration: none;
  }

  .card-actions .warning {
    --mdc-theme-primary: var(--google-red-500);
  }
`;

    const _navStyle = css `
  :host {
    font-family: var(--paper-font-body1_-_font-family);
    -webkit-font-smoothing: var(--paper-font-body1_-_-webkit-font-smoothing);
    font-size: var(--paper-font-body1_-_font-size);
    font-weight: var(--paper-font-body1_-_font-weight);
    line-height: var(--paper-font-body1_-_line-height);
  }

  app-header-layout {
    background-color: var(--primary-background-color);
  }

  app-header,
  app-toolbar,
  paper-tabs {
    background-color: var(--app-header-background-color, var(--primary-color));
    color: var(--app-header-text-color, var(--text-primary-color, white));
    font-weight: 400;
    text-transform: uppercase;
  }

  paper-tabs {
    --paper-tabs-selection-bar-color: #fff;
    padding-left: 12px;
  }

  app-toolbar ha-menu-button + [main-title],
  app-toolbar ha-paper-icon-button-arrow-prev + [main-title],
  app-toolbar ha-icon-button + [main-title] {
    margin-left: 24px;
  }
  .tabs {
    position: relative;
    z-index: -1;
  }
  div#contentContainer {
    z-index: 0;
  }
`;

    const baseHacsStyles = css `
  :root {
    font-family: var(--paper-font-body1_-_font-family);
    -webkit-font-smoothing: var(--paper-font-body1_-_-webkit-font-smoothing);
    font-size: var(--paper-font-body1_-_font-size);
    font-weight: var(--paper-font-body1_-_font-weight);
    line-height: var(--paper-font-body1_-_line-height);
  }
  a {
    text-decoration: none;
    color: var(--link-text-color, var(--accent-color));
  }
  h1 {
    font-family: var(--paper-font-title_-_font-family);
    -webkit-font-smoothing: var(--paper-font-title_-_-webkit-font-smoothing);
    white-space: var(--paper-font-title_-_white-space);
    overflow: var(--paper-font-title_-_overflow);
    text-overflow: var(--paper-font-title_-_text-overflow);
    font-size: var(--paper-font-title_-_font-size);
    font-weight: var(--paper-font-title_-_font-weight);
    line-height: var(--paper-font-title_-_line-height);
    @apply --paper-font-title;
  }
  .title {
    margin-bottom: 8px;
    padding-top: 4px;
    color: var(--primary-text-color);
    white-space: nowrap;
    text-overflow: ellipsis;
    overflow: hidden;
    width: calc(100% - 48px) !important;
  }
  .addition {
    color: var(--secondary-text-color);
    position: relative;
    width: 100% !important;
    height: auto;
    line-height: 1.2em;
    text-overflow: ellipsis;
    overflow: hidden;
  }
  ha-card {
    margin: 8px;
  }
  ha-icon {
    height: 24px;
    width: 24px;
    margin-right: 16px;
    float: left;
    color: var(--primary-text-color);
  }
  .installed {
    color: var(--hacs-status-installed, #126e15);
  }
  .pending-upgrade {
    color: var(--hacs-status-pending-update, #ffab40);
  }
  .pending-restart {
    color: var(--hacs-status-pending-restart, var(--google-red-500));
  }
  .not-loaded {
    color: var(--hacs-status-not-loaded, var(--google-red-500));
  }
  .new {
    color: var(--hacs-badge-color, var(--primary-color));
  }
  .card-group {
    margin-top: 24px;
  }
  .two-line-list-body {
    min-height: 0 !important;
    display: contents;
  }
  .card-group {
    color: var(--primary-text-color);
    margin-bottom: 12px;
  }

  ha-card.compact {
    height: 80px !important;
  }

  ha-card.compact .addition {
    height: 1.2em;
    white-space: preliine !important;
  }

  ha-card.compact .title {
    height: 1.2em;
    white-space: preliine !important;
  }

  paper-item.compact {
    margin-bottom: 2px !important;
    white-space: nowrap !important;
  }

  .card-group ha-card {
    --card-group-columns: 5;
    width: calc(
      (100% - 12px * var(--card-group-columns)) / var(--card-group-columns)
    );
    margin: 4px;
    vertical-align: top;
    height: 136px;
  }

  @media screen and (max-width: 2400px) and (min-width: 1801px) {
    .card-group ha-card {
      --card-group-columns: 4;
    }
  }

  @media screen and (max-width: 1800px) and (min-width: 1201px) {
    .card-group ha-card {
      --card-group-columns: 3;
    }
  }

  @media screen and (max-width: 1200px) and (min-width: 601px) {
    .card-group ha-card {
      --card-group-columns: 2;
    }
  }

  @media screen and (max-width: 600px) and (min-width: 0) {
    .card-group ha-card {
      width: 100%;
      margin: 4px 0;
    }
    .content {
      padding: 0;
    }
  }
  hr {
    border-color: var(--accent-color);
  }
  .leftspace {
    margin-left: 8px !important;
  }
  .disabled-button {
    --mdc-theme-primary: var(--disabled-text-color);
  }
  .uninstall-button {
    --mdc-theme-primary: var(--google-red-500);
    color: var(--google-red-500);
  }
  .red-button {
    --mdc-theme-primary: var(--google-red-500);
  }
  .link {
    color: var(--link-text-color, var(--accent-color));
    font-weight: 600;
    text-decoration: none;
  }
  paper-item {
    border-radius: var(--ha-card-border-radius);
    background: var(--primary-background-color);
    display: block;
    cursor: pointer;
    margin-bottom: 12px;
  }
  paper-item.list {
    margin-bottom: 24px;
  }
  paper-item:hover {
    outline: 0;
    background: var(--table-row-alternative-background-color);
  }
  .right {
    float: right;
  }
  .griddy {
    display: grid;
  }
  .flexy {
    display: flex;
  }
  .flowroot {
    display: flow-root;
  }
`;
    const mobileHacsStyles = css `
  @media screen and (max-width: 600px) and (min-width: 0) {
    .MobileHide {
      display: none !important;
    }
    .MobileGrid {
      display: grid !important;
      text-align: center;
      position: initial !important;
      width: 100% !important;
      padding-left: 0px !important;
      padding-right: 0px !important;
    }
  }
`;
    const HacsStyle = [
        _haStyle,
        _navStyle,
        baseHacsStyles,
        mobileHacsStyles,
    ];

    const AllCategories = [
        "integrations",
        "plugins",
        "appdaemon_apps",
        "netdaemon_apps",
        "python_scripts",
        "themes",
    ];
    const getConfiguration$1 = async (hass) => {
        const response = await hass.connection.sendMessagePromise({
            type: "hacs/config",
        });
        return response;
    };
    const getRepositories$1 = async (hass) => {
        const response = await hass.connection.sendMessagePromise({
            type: "hacs/repositories",
        });
        return response;
    };
    const getCritical$1 = async (hass) => {
        const response = await hass.connection.sendMessagePromise({
            type: "hacs/get_critical",
        });
        return response;
    };
    const getStatus$1 = async (hass) => {
        const response = await hass.connection.sendMessagePromise({
            type: "hacs/status",
        });
        return response;
    };
    const getLovelaceConfiguration$1 = async (hass) => {
        try {
            const response = await hass.connection.sendMessagePromise({
                type: "lovelace/resources",
            });
            return response;
        }
        catch (e) {
            return null;
        }
    };

    let HacsFrontendBase = /** @class */ (() => {
        let HacsFrontendBase = class HacsFrontendBase extends LitElement {
            constructor() {
                super(...arguments);
                this.repository_view = false;
                this.logger = new Logger();
            }
            update(changedProperties) {
                super.update(changedProperties);
                if (this.hacs.configuration && this.hacs.configuration.debug)
                    this.hacs.logger.info("Changed properties", changedProperties);
            }
            connectedCallback() {
                /* I have no idea why this is done, but without it shit breaks */
                super.connectedCallback();
                /* Create the HACS object */
                this.hacs = new Hacs(this.configuration, this.repositories, this.status);
                /* Add handlers for custom HACS browser events */
                this.addEventListener("hacs-location-change", this.locationChanged);
                this.addEventListener("hacs-repository-action", this.RepositoryAction);
                this.addEventListener("hacs-onboarding-done", this.onboardingDone);
                this.addEventListener("hacs-recreate", this._recreatehacs);
                this.addEventListener("hacs-force-reload", this._reload);
                /* "steal" Lovelace elements */
                load_lovelace();
                /* Backend event subscription */
                this.hass.connection.subscribeEvents(() => this.updateProperty("configuration"), "hacs/config");
                this.hass.connection.subscribeEvents(() => this.updateProperty("status"), "hacs/status");
                this.hass.connection.subscribeEvents((e) => this._reload(e), "hacs/reload");
                this.hass.connection.subscribeEvents(() => this.updateProperty("repositories"), "hacs/repository");
                this.hass.connection.subscribeEvents(() => this.updateProperty("lovelaceconfig"), "lovelace_updated");
                /* Reset local storage */
                localStorage.setItem("hacs-search", "");
                localStorage.setItem("hacs-sort", "name-desc");
            }
            _recreatehacs() {
                var configuration = this.configuration;
                var repositories = this.repositories;
                var status = this.status;
                if (this.hacs.isnullorempty(configuration))
                    configuration = this.hacs.configuration;
                if (this.hacs.isnullorempty(repositories))
                    repositories = this.hacs.repositories;
                if (this.hacs.isnullorempty(status))
                    status = this.hacs.status;
                this.hacs = new Hacs(configuration, repositories, status);
                this.requestUpdate();
            }
            RepositoryAction(ev) {
                if (this.configuration.debug)
                    this.hacs.logger.info(ev.detail);
                const evdata = ev.detail;
                this.hacs.RepositoryWebSocketAction(this.hass, evdata.repo, evdata.action, evdata.data);
            }
            async firstUpdated() {
                window.onpopstate = function () {
                    if (window.location.pathname.includes("hacs")) {
                        window.location.reload();
                    }
                };
                const [repositories, configuration, status, critical, lovelaceconfig,] = await Promise.all([
                    getRepositories$1(this.hass),
                    getConfiguration$1(this.hass),
                    getStatus$1(this.hass),
                    getCritical$1(this.hass),
                    getLovelaceConfiguration$1(this.hass),
                ]);
                this.repositories = repositories;
                this.configuration = configuration;
                this.status = status;
                this.critical = critical;
                this.lovelaceconfig = lovelaceconfig;
                this._recreatehacs();
            }
            async updateProperty(property) {
                if (property === "repositories")
                    this.repositories = await getRepositories$1(this.hass);
                else if (property === "configuration")
                    this.configuration = await getConfiguration$1(this.hass);
                else if (property === "status")
                    this.status = await getStatus$1(this.hass);
                else if (property === "critical")
                    this.critical = await getCritical$1(this.hass);
                else if (property === "lovelaceconfig")
                    this.lovelaceconfig = await getLovelaceConfiguration$1(this.hass);
                this._recreatehacs();
            }
            _reload(e) {
                const force = (e.data && e.data.force) || (e.detail && e.detail.force) || false;
                window.location.reload(force);
            }
            render() {
                // Handle access to root
                if (this.route.path === "" || this.route.path === undefined) {
                    this.hacs.navigate(this, `${this.route.prefix}/installed`);
                    this.route.path = "/installed";
                }
                if (this.repositories === undefined ||
                    this.configuration === undefined ||
                    this.status === undefined) {
                    return html `
        <div class="loader"><paper-spinner active></paper-spinner></div>
      `;
                }
                if ((!this.configuration.onboarding_done && !this.status.disabled) ||
                    false) {
                    return html `
        <hacs-onboarding
          .hacs=${this.hacs}
          .hass=${this.hass}
          .narrow=${this.narrow}
        ></hacs-onboarding>
      `;
                }
                return html `
      <app-header-layout has-scrolling-region>
        <app-header slot="header" fixed>
          <div id="contentContainer">
            <app-toolbar>
              <ha-menu-button
                .hass="${this.hass}"
                .narrow="${this.narrow}"
              ></ha-menu-button>
              <div main-title class="fulltitle">
                Home Assistant Community Store
                ${this._rootPath === "hacs_dev" ? html ` (DEVELOPMENT) ` : ""}
              </div>
              <div main-title class="mobiletitle">
                HACS
                ${this._rootPath === "hacs_dev" ? html ` (DEVELOPMENT) ` : ""}
              </div>
              <hacs-menu
                .location=${window.location.pathname}
                .hacs=${this.hacs}
                .hass=${this.hass}
                .configuration=${this.configuration}
                .status=${this.status}
                .repositories=${this.repositories}
              ></hacs-menu>
            </app-toolbar>

            <paper-tabs
              scrollable
              autoselect
              class="tabs"
              attr-for-selected="page-name"
              .selected=${this._activeTab}
              @iron-activate=${this.handlePageSelected}
            >
              <paper-tab page-name="installed"
                >${this.hacs.localize(`common.installed`)}</paper-tab
              >

              <paper-tab page-name="integration"
                >${this.hacs.localize(`common.integrations`)}</paper-tab
              >

              <paper-tab page-name="plugin"
                >${this.hacs.localize(`common.lovelace_elements`)}</paper-tab
              >

              ${this.configuration.categories.includes("appdaemon")
                ? html `
                    <paper-tab page-name="appdaemon">
                      ${this.hacs.localize(`common.appdaemon_apps`)}
                    </paper-tab>
                  `
                : ""}
              ${this.configuration.categories.includes("netdaemon")
                ? html `
                    <paper-tab page-name="netdaemon">
                      ${this.hacs.localize(`common.netdaemon_apps`)}
                    </paper-tab>
                  `
                : ""}
              ${this.configuration.categories.includes("python_script")
                ? html `
                    <paper-tab page-name="python_script">
                      ${this.hacs.localize(`common.python_scripts`)}
                    </paper-tab>
                  `
                : ""}
              ${this.configuration.categories.includes("theme")
                ? html `
                    <paper-tab page-name="theme">
                      ${this.hacs.localize(`common.themes`)}
                    </paper-tab>
                  `
                : ""}
              <paper-tab page-name="settings"
                >${this.hacs.localize("common.settings")}</paper-tab
              >
            </paper-tabs>
          </div>
        </app-header>

        <hacs-progressbar
          .active=${this.status.background_task || this.status.disabled}
        ></hacs-progressbar>
        ${this.status.disabled
                ? html `
              <ha-card header="${this.hacs
                    .localize("common.disabled")
                    .toUpperCase()}!">
                <div class="card-content">
                  ${this.hacs.localize("common.hacs_is_disabled")}! </br> ${this.hacs.localize("common.check_log_file")}
                </div>
              </ha-card>
            `
                : ""}
        <hacs-body>
          <hacs-critical
            .hacs=${this.hacs}
            .hass=${this.hass}
            .critical=${this.critical}
          ></hacs-critical>
          <hacs-error .hacs=${this.hacs} .hass=${this.hass}></hacs-error>

          ${this.route.path === "/installed"
                ? html `
                <hacs-installed
                  .hacs=${this.hacs}
                  .hass=${this.hass}
                  .route=${this.route}
                  .lovelaceconfig=${this.lovelaceconfig}
                  .repositories=${this.repositories}
                  .configuration=${this.configuration}
                  .status=${this.status}
                ></hacs-installed>
              `
                : this.route.path === "/settings"
                    ? html `
                <hacs-settings
                  .hacs=${this.hacs}
                  .hass=${this.hass}
                  .route=${this.route}
                  .status=${this.status}
                  .lovelaceconfig=${this.lovelaceconfig}
                  .configuration=${this.configuration}
                >
                </hacs-settings>
              `
                    : this.route.path.includes("/repository")
                        ? html `
                <hacs-repository
                  .hacs=${this.hacs}
                  .repository=${this._get_repository}
                  .hass=${this.hass}
                  .route=${this.route}
                  .lovelaceconfig=${this.lovelaceconfig}
                  .repositories=${this.repositories}
                  .configuration=${this.configuration}
                  .status=${this.status}
                ></hacs-repository>
              `
                        : html `
                <hacs-store
                  .hacs=${this.hacs}
                  .store=${this._get_store}
                  .hass=${this.hass}
                  .route=${this.route}
                  .repositories=${this.repositories}
                  .configuration=${this.configuration}
                  .status=${this.status}
                ></hacs-store>
              `}
        </hacs-body>
      </app-header-layout>
    `;
            }
            get _get_store() {
                return this.route.path.split("/")[1];
            }
            get _get_repository() {
                return this.route.path.split("/")[2];
            }
            locationChanged(ev) {
                if (this.configuration.debug)
                    this.hacs.logger.info(ev.type, ev.detail);
                const new_path = `${ev.detail.value}`;
                if (new_path.startsWith("/")) {
                    this.route.prefix = new_path;
                    this.route.path = "";
                }
                else {
                    this.route.path = `/${ev.detail.value}`;
                }
                const force = ev.detail.force;
                this.hacs.navigate(this, `${this.route.prefix}${this.route.path}`);
                if (force)
                    window.location.reload();
                else
                    this.requestUpdate();
                this.hacs.scrollToTarget(this, 
                // @ts-ignore
                this.shadowRoot.querySelector("app-header-layout").header.scrollTarget);
            }
            onboardingDone() {
                this.configuration.onboarding_done = true;
                this.hass.connection.sendMessage({
                    type: "hacs/settings",
                    action: "onboarding_done",
                });
                this.requestUpdate();
            }
            handlePageSelected(e) {
                this.dispatchEvent(new CustomEvent("hacs-location-change", {
                    detail: { value: e.detail.selected },
                    bubbles: true,
                    composed: true,
                }));
            }
            get _activeTab() {
                if (this.route.path.split("/")[1] === undefined)
                    return "installed";
                return this.route.path.split("/")[1];
            }
            get _rootPath() {
                if (this.route.prefix.split("/")[1] === undefined)
                    return "hacs";
                return this.route.prefix.split("/")[1];
            }
            static get styles() {
                return [
                    HacsStyle,
                    css `
        app-header-layout {
          background: var(
            --lovelace-background,
            var(--primary-background-color)
          );
        }
        .loader {
          background-color: var(--primary-background-color);
          height: 100%;
          width: 100%;
        }
        paper-spinner {
          position: absolute;
          top: 30%;
          left: 50%;
          transform: translate(-50%, -50%);
          z-index: 99;
          width: 150px;
          height: 150px;
        }
        paper-tab {
          cursor: pointer;
        }

        .margin {
          width: 20%;
        }
        hacs-progressbar {
          top: 0;
          position: sticky;
          display: block;
          z-index: 1337;
        }
        .fulltitle {
          display: block;
        }
        .mobiletitle {
          display: none;
        }
        @media screen and (max-width: 612px) and (min-width: 0) {
          .fulltitle {
            display: none;
          }
          .mobiletitle {
            display: block;
            margin-left: 24px;
          }
        }
      `,
                ];
            }
        };
        __decorate([
            property()
        ], HacsFrontendBase.prototype, "critical", void 0);
        __decorate([
            property()
        ], HacsFrontendBase.prototype, "narrow", void 0);
        __decorate([
            property()
        ], HacsFrontendBase.prototype, "repository_view", void 0);
        __decorate([
            property()
        ], HacsFrontendBase.prototype, "hacs", void 0);
        __decorate([
            property()
        ], HacsFrontendBase.prototype, "hass", void 0);
        __decorate([
            property()
        ], HacsFrontendBase.prototype, "lovelaceconfig", void 0);
        __decorate([
            property()
        ], HacsFrontendBase.prototype, "route", void 0);
        HacsFrontendBase = __decorate([
            customElement("hacs-frontendbase")
        ], HacsFrontendBase);
        return HacsFrontendBase;
    })();

    function AddedToLovelace(repository, lovelaceconfig, status) {
        if (status.lovelace_mode === "yaml")
            return true;
        if (lovelaceconfig) {
            var loaded = false;
            var URL1 = `/community_plugin/${repository.full_name.split("/")[1]}/${repository.file_name}`;
            var URL2 = `/hacsfiles/${repository.full_name.split("/")[1]}/${repository.file_name}`;
            let resources = lovelaceconfig;
            if (lovelaceconfig.hasOwnProperty("resources")) {
                resources = lovelaceconfig.resources;
            }
            else if (lovelaceconfig.hasOwnProperty("views")) {
                resources = [];
            }
            if (resources) {
                resources.forEach((item) => {
                    if (item.url === URL1 || item.url === URL2)
                        loaded = true;
                });
            }
            return loaded;
        }
        else
            return true;
    }

    class OviewItemBuilder {
        constructor(hass, configuration, lovelaceconfig, status, route) {
            this.hass = hass;
            this.configuration = configuration;
            this.lovelaceconfig = lovelaceconfig;
            this.status = status;
            this.route = route;
        }
        render_card(repository) {
            return html `
      <ha-card
        @click="${this.ShowRepository}"
        .RepoID="${repository.id}"
        class="${this.configuration.frontend_compact ? "compact" : ""}"
      >
        <div class="card-content">
          <div>
            <ha-icon
              icon=${repository.new ? "mdi:new-box" : "mdi:cube"}
              class="${this.StatusAndDescription(repository).status}"
              title="${this.StatusAndDescription(repository).description}"
            >
            </ha-icon>
            <div class="title">
              ${nodeEmoji.emojify(repository.name || "")}
            </div>
            <div class="addition">
              ${nodeEmoji.emojify(repository.description || "")}
            </div>
          </div>
        </div>
      </ha-card>
    `;
        }
        render_list_line(repository) {
            return html `
      <paper-item
        .RepoID=${repository.id}
        @click="${this.ShowRepository}"
        class="${this.configuration.frontend_compact ? "compact" : ""}"
      >
        <div class="icon">
          <ha-icon
            icon=${repository.new ? "mdi:new-box" : "mdi:cube"}
            class="${this.StatusAndDescription(repository).status}"
            title="${this.StatusAndDescription(repository).description}"
          >
          </ha-icon>
        </div>
        <paper-item-body two-line class="two-line-list-body">
          <div>
            ${nodeEmoji.emojify(repository.name || "")}
            ${this.route.path === "/installed"
            ? html `
                  <div class="MobileHide right flexy">
                    <div>${repository.installed_version}</div>
                    &nbsp; (
                    <div
                      class="${repository.pending_upgrade
                ? this.StatusAndDescription(repository).status
                : ""}"
                    >
                      ${repository.available_version}
                    </div>
                    )
                  </div>
                `
            : ""}
          </div>
          <div class="addition">
            ${nodeEmoji.emojify(repository.description || "")}
          </div>
        </paper-item-body>
      </paper-item>
    `;
        }
        render(repository) {
            if (this.configuration.frontend_mode === "Grid")
                return this.render_card(repository);
            return this.render_list_line(repository);
        }
        ShowRepository(ev) {
            var RepoID;
            ev.composedPath().forEach((item) => {
                if (item.RepoID) {
                    RepoID = item.RepoID;
                }
            });
            this.dispatchEvent(new CustomEvent("hacs-location-change", {
                detail: { value: `repository/${RepoID}` },
                bubbles: true,
                composed: true,
            }));
        }
        StatusAndDescription(repository) {
            var status = repository.status;
            if (this.status.startup && repository.installed)
                status = "installed";
            var description = repository.status_description;
            if (repository.installed && !this.status.background_task) {
                if (repository.category === "plugin" &&
                    !AddedToLovelace(repository, this.lovelaceconfig, this.status)) {
                    status = "not-loaded";
                    description = "Not loaded in lovelace";
                }
            }
            return { status: status, description: description };
        }
    }

    let HacsInstalled = /** @class */ (() => {
        let HacsInstalled = class HacsInstalled extends LitElement {
            render() {
                if (this.repositories === undefined)
                    return html `
        <hacs-progressbar></hacs-progressbar>
      `;
                const builder = new OviewItemBuilder(this.hass, this.configuration, this.lovelaceconfig, this.status, this.route);
                var categories = {
                    integrations: [],
                    plugins: [],
                    appdaemon_apps: [],
                    netdaemon_apps: [],
                    python_scripts: [],
                    themes: []
                };
                var installed_repositories = this.repositories.filter(function (repository) {
                    if (repository.installed) {
                        if (repository.category === "integration")
                            categories.integrations.push(repository);
                        if (repository.category === "plugin")
                            categories.plugins.push(repository);
                        if (repository.category === "appdaemon")
                            categories.appdaemon_apps.push(repository);
                        if (repository.category === "netdaemon")
                            categories.netdaemon_apps.push(repository);
                        if (repository.category === "python_script")
                            categories.python_scripts.push(repository);
                        if (repository.category === "theme")
                            categories.themes.push(repository);
                        return true;
                    }
                    return false;
                });
                var updatable_repositories = installed_repositories.filter(function (repository) {
                    if (repository.pending_upgrade)
                        return true;
                    return false;
                });
                return html `
      <hacs-body>
        ${updatable_repositories.length !== 0 && !this.status.startup
                ? html `
              <div class="card-group">
                <div class="leftspace grouptitle">
                  ${this.hacs.localize("store.pending_upgrades")}
                </div>
                ${updatable_repositories
                    .sort((a, b) => a.name > b.name ? 1 : -1)
                    .map(repository => {
                    return builder.render(repository);
                })}
              </div>
              <hr noshade />
            `
                : ""}
        ${AllCategories.map(category => {
                if (categories[category] === undefined ||
                    categories[category].length === 0)
                    return html ``;
                return html `
            <div class="card-group">
              <div class="leftspace grouptitle">
                ${this.hacs.localize(`common.${category}`)}
              </div>
              ${categories[category]
                    .sort((a, b) => a.name.toLowerCase() > b.name.toLowerCase() ? 1 : -1)
                    .map(repository => {
                    return builder.render(repository);
                })}
            </div>
          `;
            })}
      </hacs-body>
    `;
            }
            static get styles() {
                return [
                    HacsStyle,
                    css `
        .loader {
          background-color: var(--primary-background-color);
          height: 100%;
          width: 100%;
        }
        ha-card {
          display: inline-block;
          cursor: pointer;
        }
        .grouptitle {
          margin-bottom: 12px;
          font-size: larger;
        }
      `
                ];
            }
        };
        __decorate([
            property()
        ], HacsInstalled.prototype, "repositories", void 0);
        __decorate([
            property()
        ], HacsInstalled.prototype, "configuration", void 0);
        __decorate([
            property()
        ], HacsInstalled.prototype, "hacs", void 0);
        __decorate([
            property()
        ], HacsInstalled.prototype, "hass", void 0);
        __decorate([
            property()
        ], HacsInstalled.prototype, "lovelaceconfig", void 0);
        __decorate([
            property()
        ], HacsInstalled.prototype, "route", void 0);
        __decorate([
            property()
        ], HacsInstalled.prototype, "status", void 0);
        HacsInstalled = __decorate([
            customElement("hacs-installed")
        ], HacsInstalled);
        return HacsInstalled;
    })();

    const version$1 = "202005282138";

    let HacsRepository = /** @class */ (() => {
        let HacsRepository = class HacsRepository extends LitElement {
            constructor() {
                super(...arguments);
                this.repository_view = false;
            }
            update(changedProperties) {
                super.update(changedProperties);
                if (this.hacs.configuration.debug)
                    this.hacs.logger.info(changedProperties);
            }
            shouldUpdate(changedProperties) {
                changedProperties.forEach((_oldValue, propName) => {
                    if (propName === "hacs") {
                        const _repository = this.repository;
                        const _repositories = this.hacs.repositories.filter(function (repo) {
                            return String(repo.id) === String(_repository);
                        });
                        const repo = _repositories[0];
                        if (!this.repo || JSON.stringify(repo) !== JSON.stringify(this.repo)) {
                            this.repo = repo;
                        }
                    }
                });
                return (changedProperties.has("repo") || changedProperties.has("lovelaceconfig"));
            }
            firstUpdated() {
                var _a;
                if (!((_a = this.repo) === null || _a === void 0 ? void 0 : _a.updated_info)) {
                    this.dispatchEvent(new CustomEvent("hacs-repository-action", {
                        detail: {
                            repo: this.repository,
                            action: "set_state",
                            data: "other",
                        },
                        bubbles: true,
                        composed: true,
                    }));
                    this.dispatchEvent(new CustomEvent("hacs-repository-action", {
                        detail: { repo: this.repository, action: "update" },
                        bubbles: true,
                        composed: true,
                    }));
                }
            }
            render() {
                if (this.repo === undefined)
                    return html `
        <div class="loader"><paper-spinner active></paper-spinner></div>
      `;
                if (this.repo.installed) {
                    var back = this.hacs.localize(`common.installed`);
                }
                else {
                    if (this.repo.category === "appdaemon") {
                        var FE_cat = "appdaemon_apps";
                    }
                    else if (this.repo.category === "netdaemon") {
                        var FE_cat = "netdaemon_apps";
                    }
                    else {
                        FE_cat = `${this.repo.category}s`;
                    }
                    var back = this.hacs.localize(`common.${FE_cat}`);
                }
                return html `
      <hacs-body>
        <div class="getBack">
          <mwc-button @click=${this.GoBackToStore} title="${back}">
            <ha-icon icon="mdi:arrow-left"></ha-icon>
            ${this.hacs.localize(`repository.back_to`)} ${back}
          </mwc-button>
          ${this.repo.state === "other"
                ? html `
                <div class="loader"><paper-spinner active></paper-spinner></div>
              `
                : ""}
        </div>

        <hacs-repository-banner-note
          .hacs=${this.hacs}
          .hass=${this.hass}
          .status=${this.status}
          .repository=${this.repo}
          .route=${this.route}
          .lovelaceconfig=${this.lovelaceconfig}
          .configuration=${this.configuration}
        >
        </hacs-repository-banner-note>

        <ha-card>
          <div class="repotitle">
            ${nodeEmoji.emojify(this.repo.name || "")}
          </div>
          <hacs-repository-menu
            .hass=${this.hass}
            .route=${this.route}
            .repository=${this.repo}
          ></hacs-repository-menu>

          <div class="card-content">
            <div class="description">
              ${nodeEmoji.emojify(this.repo.description || "")}
            </div>

            <div class="information">
              <hacs-authors
                .hass=${this.hass}
                .authors=${this.repo.authors}
              ></hacs-authors>
              ${!this.hacs.isnullorempty(this.repo.stars)
                ? html `
                    <div class="stars">
                      <b>${this.hacs.localize(`repository.github_stars`)}: </b>
                      ${this.repo.stars}
                    </div>
                  `
                : ""}
              ${!this.hacs.isnullorempty(this.repo.downloads)
                ? html `
                    <div class="downloads">
                      <b>${this.hacs.localize(`repository.downloads`)}: </b>
                      ${this.repo.downloads}
                    </div>
                  `
                : ""}
              ${!this.hacs.isnullorempty(this.repo.last_updated)
                ? html `
                    <div class="last_updated">
                      <b>${this.hacs.localize(`store.last_updated`)}: </b>
                      ${this.hacs.RelativeTimeSince(this.repo.last_updated)}
                    </div>
                  `
                : ""}
              ${this.repo.full_name === "hacs/integration"
                ? html `
                    <div class="frontend-version">
                      <b
                        >${this.hacs.localize(`repository.frontend_version`)}:
                      </b>
                      ${version$1}
                    </div>
                  `
                : ""}
              ${this.repo.installed
                ? html `
                    <div class="version-installed">
                      <b>${this.hacs.localize(`repository.installed`)}: </b>
                      ${this.repo.installed_version}
                    </div>
                  `
                : ""}
              ${String(this.repo.releases.length) === "0"
                ? html `
                    <div class="version-available">
                      <b>${this.hacs.localize(`repository.available`)}: </b>
                      ${this.repo.available_version}
                    </div>
                  `
                : html `
                    <div class="version-available-dropdown">
                      <paper-dropdown-menu
                        @value-changed="${this.setRepositoryVersion}"
                        label="${this.hacs.localize(`repository.available`)}:
                     (${this.hacs.localize(`repository.newest`)}: ${this.repo
                    .releases[0]})"
                      >
                        <paper-listbox
                          slot="dropdown-content"
                          selected="${this.repo.selected_tag}"
                          attr-for-selected="value"
                        >
                          ${this.repo.releases.map((release) => {
                    return html `
                              <paper-item value="${release}"
                                >${release}</paper-item
                              >
                            `;
                })}
                          ${this.repo.full_name === "hacs/integration" ||
                    this.repo.hide_default_branch
                    ? ""
                    : html `
                                <paper-item value="${this.repo.default_branch}"
                                  >${this.repo.default_branch}</paper-item
                                >
                              `}
                        </paper-listbox>
                      </paper-dropdown-menu>
                    </div>
                  `}
            </div>
          </div>

          <div class="card-actions MobileGrid">
            <hacs-button-main-action
              .hass=${this.hass}
              .repository=${this.repo}
              .status=${this.status}
            ></hacs-button-main-action>
            <hacs-button-changelog
              .hass=${this.hass}
              .repository=${this.repo}
            ></hacs-button-changelog>
            <hacs-button-open-repository
              .hass=${this.hass}
              .repository=${this.repo}
            ></hacs-button-open-repository>
            ${this.repo.category === "plugin"
                ? html `
                  <hacs-button-open-plugin
                    .hass=${this.hass}
                    .repository=${this.repo}
                  ></hacs-button-open-plugin>
                `
                : ""}
            <hacs-button-uninstall
              class="right"
              .hass=${this.hass}
              .repository=${this.repo}
              .status=${this.status}
            ></hacs-button-uninstall>
          </div>
        </ha-card>
        ${this.repo.updated_info
                ? html `
              <ha-card class="additional">
                <div class="card-content">
                  <div class="more_info markdown-body">
                    ${markdown.html(this.repo.additional_info || "", this.repo)}
                  </div>
                  <hacs-repository-note
                    .hass=${this.hass}
                    .lovelaceconfig=${this.lovelaceconfig}
                    .status=${this.status}
                    .configuration=${this.configuration}
                    .repository=${this.repo}
                  ></hacs-repository-note>
                </div>
              </ha-card>
            `
                : ""}
      </hacs-body>
    `;
            }
            setRepositoryVersion(e) {
                if (e.detail.value.length > 0) {
                    this.dispatchEvent(new CustomEvent("hacs-repository-action", {
                        detail: { repo: this.repository, action: "set_state", data: "other" },
                        bubbles: true,
                        composed: true,
                    }));
                    this.dispatchEvent(new CustomEvent("hacs-repository-action", {
                        detail: {
                            repo: this.repository,
                            action: "set_version",
                            data: e.detail.value,
                        },
                        bubbles: true,
                        composed: true,
                    }));
                }
            }
            GoBackToStore() {
                this.repository = undefined;
                if (this.repo.installed) {
                    this.panel = "installed";
                }
                else {
                    this.panel = this.repo.category;
                }
                this.dispatchEvent(new CustomEvent("hacs-location-change", {
                    detail: { value: this.panel },
                    bubbles: true,
                    composed: true,
                }));
            }
            static get styles() {
                return [
                    HacsStyle,
                    css `
        .loader {
          background-color: var(--primary-background-color);
          height: 100%;
          width: 100%;
        }
        paper-spinner {
          position: absolute;
          top: 30%;
          left: 50%;
          transform: translate(-50%, -50%);
          z-index: 99;
          width: 150px;
          height: 150px;
        }
        paper-dropdown-menu {
          width: 50%;
        }
        @media screen and (max-width: 600px) and (min-width: 0) {
          paper-dropdown-menu {
            width: 100%;
          }
          .getBack {
            margin-top: 0px !important;
          }
        }
        paper-item {
          display: flex;
          background-color: var(
            --paper-listbox-background-color,
            var(--primary-background-color)
          );
        }
        .description {
          width: 100%;
          margin-bottom: 8px;
          color: var(--secondary-text-color);
          text-align: left !important;
        }
        .version {
          padding-bottom: 8px;
        }
        .version-installed,
        .version-available {
          margin-top: 12px;
        }

        .options {
          float: right;
          width: 40%;
        }
        .information {
          width: 100%;
        }
        .additional {
          margin-bottom: 108px;
        }
        .getBack {
          margin-top: -12px;
          margin-bottom: 4px;
          margin-left: 5%;
        }
        .right {
          float: right;
        }
        .loading {
          text-align: center;
          width: 100%;
        }
        ha-card {
          width: 90%;
          margin-left: 5%;
        }
        .link-icon {
          color: var(--dark-primary-color);
          margin-right: 8px;
        }
        .repotitle {
          padding: 24px 24px 24px 18px;
          width: calc(100% - (24px + 4px + 24px + 56px));
          font-size: 1.5em;
          white-space: nowrap;
          text-overflow: ellipsis;
          overflow: hidden;
        }
      `,
                ];
            }
        };
        __decorate([
            property({ attribute: false })
        ], HacsRepository.prototype, "repositories", void 0);
        __decorate([
            property({ type: Boolean })
        ], HacsRepository.prototype, "repository_view", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsRepository.prototype, "repo", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsRepository.prototype, "configuration", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsRepository.prototype, "hacs", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsRepository.prototype, "hass", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsRepository.prototype, "lovelaceconfig", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsRepository.prototype, "route", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsRepository.prototype, "status", void 0);
        HacsRepository = __decorate([
            customElement("hacs-repository")
        ], HacsRepository);
        return HacsRepository;
    })();

    let HacsSettings = /** @class */ (() => {
        let HacsSettings = class HacsSettings extends LitElement {
            render() {
                if (this.hacs === undefined) {
                    return html `
        <hacs-progressbar></hacs-progressbar>
      `;
                }
                return html `
      <hacs-body>
        <hacs-legacy-url-for-plugins
          .hass=${this.hass}
          .status=${this.status}
          .lovelaceconfig=${this.lovelaceconfig}
        >
        </hacs-legacy-url-for-plugins>
        <hacs-custom-repositories .hacs=${this.hacs} .route=${this.route}>
        </hacs-custom-repositories>
        <hacs-hidden-repositories .hacs=${this.hacs}>
        </hacs-hidden-repositories>
      </hacs-body>
    `;
            }
        };
        __decorate([
            property()
        ], HacsSettings.prototype, "hacs", void 0);
        __decorate([
            property()
        ], HacsSettings.prototype, "hass", void 0);
        __decorate([
            property()
        ], HacsSettings.prototype, "lovelaceconfig", void 0);
        __decorate([
            property()
        ], HacsSettings.prototype, "route", void 0);
        __decorate([
            property()
        ], HacsSettings.prototype, "status", void 0);
        HacsSettings = __decorate([
            customElement("hacs-settings")
        ], HacsSettings);
        return HacsSettings;
    })();

    let HacsStore = /** @class */ (() => {
        let HacsStore = class HacsStore extends LitElement {
            constructor() {
                super(...arguments);
                this.search = "";
                this.sort = "name-desc";
            }
            SortRepo(a, b) {
                const sorttype = this.sort.split("-")[1];
                const sortkey = this.sort.split("-")[0];
                if (sorttype === "desc") {
                    if (sortkey === "stars")
                        return a.stars < b.stars;
                    if (sortkey === "last_updated") {
                        return Date.parse(a.last_updated) < Date.parse(b.last_updated);
                    }
                    return (String(a[sortkey]).toLowerCase() > String(b[sortkey]).toLowerCase());
                }
                else {
                    if (sortkey === "stars")
                        return a.stars > b.stars;
                    if (sortkey === "last_updated") {
                        return Date.parse(a.last_updated) > Date.parse(b.last_updated);
                    }
                    return (String(a[sortkey]).toLowerCase() < String(b[sortkey]).toLowerCase());
                }
            }
            SetSortKey(ev) {
                const attr = ev.detail.item.getAttribute("key");
                if (attr.length === 0)
                    return;
                this.sort = attr;
                localStorage.setItem("hacs-sort", this.sort);
            }
            DoSearch(ev) {
                this.search = ev.composedPath()[0].value;
                localStorage.setItem("hacs-search", this.search);
            }
            clearSearch() {
                this.search = "";
                localStorage.setItem("hacs-search", this.search);
            }
            render() {
                if (this.repositories === undefined)
                    return html ` <hacs-progressbar></hacs-progressbar> `;
                const builder = new OviewItemBuilder(this.hass, this.configuration, this.lovelaceconfig, this.status, this.route);
                var new_repositories = [];
                this.search = localStorage.getItem("hacs-search");
                this.sort = localStorage.getItem("hacs-sort");
                var repositories = this.repositories.filter((repository) => {
                    // Hide hidden repos from the store
                    if (repository.hide)
                        return false;
                    // Check contry restrictions
                    if (this.configuration.country !== "ALL" &&
                        !this.hacs.isnullorempty(repository.country)) {
                        if (this.configuration.country !== repository.country)
                            return false;
                    }
                    // Check if repository category matches store
                    if (repository.category === this.store) {
                        // Hide HACS from stores
                        if (String(repository.id) === "172733314")
                            return false;
                        // Search filter
                        if (this.search !== "") {
                            if (repository.name.toLowerCase().includes(this.search.toLowerCase()))
                                return true;
                            if (String(repository.description)
                                .toLowerCase()
                                .includes(this.search.toLowerCase()))
                                return true;
                            if (repository.full_name
                                .toLowerCase()
                                .includes(this.search.toLowerCase()))
                                return true;
                            if (String(repository.authors)
                                .toLowerCase()
                                .includes(this.search.toLowerCase()))
                                return true;
                            if (String(repository.topics)
                                .toLowerCase()
                                .includes(this.search.toLowerCase()))
                                return true;
                            return false;
                        }
                        // Is this new?
                        if (repository.new)
                            new_repositories.push(repository);
                        return true;
                    }
                    return false;
                });
                return html `
      <hacs-body>
        <div class="store-top">
          <paper-input
            class="search-bar padder"
            type="text"
            id="Search"
            @input=${this.DoSearch}
            placeholder="  ${this.hacs.localize("store.placeholder_search")}."
            autofocus
            .value=${this.search}
          >
            ${this.search.length > 0
                ? html `
                  <ha-icon
                    slot="suffix"
                    icon="mdi:close"
                    @click="${this.clearSearch}"
                  ></ha-icon>
                `
                : ""}
          </paper-input>
          <paper-dropdown-menu
            @iron-select="${this.SetSortKey}"
            class="sort padder"
            label="Sort"
          >
            <paper-listbox
              slot="dropdown-content"
              class="list"
              selected="${this.sort}"
              attr-for-selected="key"
              dir="rtl"
            >
              <paper-item class="listitem" key="last_updated-desc"
                >${this.hacs.localize("store.last_updated")}&nbsp;(${this.hacs.localize("store.descending")})</paper-item
              >
              <paper-item class="listitem" key="last_updated-asc"
                >${this.hacs.localize("store.last_updated")}&nbsp;(${this.hacs.localize("store.ascending")})</paper-item
              >
              <paper-item class="listitem" key="name-desc"
                >${this.hacs.localize("store.name")}&nbsp;(${this.hacs.localize("store.descending")})</paper-item
              >
              <paper-item class="listitem" key="name-asc"
                >${this.hacs.localize("store.name")}&nbsp;(${this.hacs.localize("store.ascending")})</paper-item
              >
              <paper-item class="listitem" key="stars-desc"
                >${this.hacs.localize("store.stars")}&nbsp;(${this.hacs.localize("store.descending")})</paper-item
              >
              <paper-item class="listitem" key="stars-asc"
                >${this.hacs.localize("store.stars")}&nbsp;(${this.hacs.localize("store.ascending")})</paper-item
              >
            </paper-listbox>
          </paper-dropdown-menu>
        </div>

        ${new_repositories.length !== 0
                ? html `
              <div class="card-group">
                <div class="leftspace grouptitle">
                  ${this.hacs.localize("store.new_repositories")}
                </div>
                ${new_repositories
                    .sort((a, b) => (this.SortRepo(a, b) ? 1 : -1))
                    .map((repository) => {
                    return builder.render(repository);
                })}
                <div class="card-group">
                  <hacs-button-clear-new
                    .hass=${this.hass}
                    .category=${this.store}
                  ></hacs-button-clear-new>
                </div>
              </div>
              <hr noshade />
            `
                : ""}

        <div class="card-group">
          ${repositories
                .sort((a, b) => (this.SortRepo(a, b) ? 1 : -1))
                .map((repository) => {
                return builder.render(repository);
            })}
        </div>
      </hacs-body>
    `;
            }
            static get styles() {
                return [
                    HacsStyle,
                    css `
        .loader {
          background-color: var(--primary-background-color);
          height: 100%;
          width: 100%;
        }
        ha-card {
          display: inline-block;
          cursor: pointer;
        }
        .padder {
          padding-left: 8px;
          padding-right: 8px;
        }
        .search-bar {
          display: block;
          width: 60%;
          background-color: var(--primary-background-color);
          color: var(--primary-text-color);
          line-height: 32px;
          border-color: var(--dark-primary-color);
          border-width: inherit;
          border-bottom-width: thin;
          border-radius: var(--ha-card-border-radius);
        }

        .sort {
          display: block;
          width: 30%;
          margin-left: 10%;
          background-color: var(--primary-background-color);
          color: var(--primary-text-color);
          line-height: 32px;
          border-color: var(--dark-primary-color);
          border-width: inherit;
          border-bottom-width: thin;
          border-radius: var(--ha-card-border-radius);
        }

        .store-top {
          display: flex;
          margin-top: -12px;
          max-width: 100%;
        }
        .card-content {
          width: calc(100% - 32px) !important;
        }
        paper-item.listitem {
          display: flex;
          background-color: var(
            --paper-listbox-background-color,
            var(--primary-background-color)
          );
        }
      `,
                ];
            }
        };
        __decorate([
            property()
        ], HacsStore.prototype, "repositories", void 0);
        __decorate([
            property()
        ], HacsStore.prototype, "configuration", void 0);
        __decorate([
            property()
        ], HacsStore.prototype, "hacs", void 0);
        __decorate([
            property()
        ], HacsStore.prototype, "hass", void 0);
        __decorate([
            property()
        ], HacsStore.prototype, "lovelaceconfig", void 0);
        __decorate([
            property()
        ], HacsStore.prototype, "route", void 0);
        __decorate([
            property()
        ], HacsStore.prototype, "status", void 0);
        __decorate([
            property()
        ], HacsStore.prototype, "store", void 0);
        __decorate([
            property()
        ], HacsStore.prototype, "search", void 0);
        __decorate([
            property()
        ], HacsStore.prototype, "sort", void 0);
        HacsStore = __decorate([
            customElement("hacs-store")
        ], HacsStore);
        return HacsStore;
    })();

    var sweetalert_min = createCommonjsModule(function (module, exports) {
    !function(t,e){module.exports=e();}(commonjsGlobal,function(){return function(t){function e(o){if(n[o])return n[o].exports;var r=n[o]={i:o,l:!1,exports:{}};return t[o].call(r.exports,r,r.exports,e),r.l=!0,r.exports}var n={};return e.m=t,e.c=n,e.d=function(t,n,o){e.o(t,n)||Object.defineProperty(t,n,{configurable:!1,enumerable:!0,get:o});},e.n=function(t){var n=t&&t.__esModule?function(){return t.default}:function(){return t};return e.d(n,"a",n),n},e.o=function(t,e){return Object.prototype.hasOwnProperty.call(t,e)},e.p="",e(e.s=8)}([function(t,e,n){Object.defineProperty(e,"__esModule",{value:!0});var o="swal-button";e.CLASS_NAMES={MODAL:"swal-modal",OVERLAY:"swal-overlay",SHOW_MODAL:"swal-overlay--show-modal",MODAL_TITLE:"swal-title",MODAL_TEXT:"swal-text",ICON:"swal-icon",ICON_CUSTOM:"swal-icon--custom",CONTENT:"swal-content",FOOTER:"swal-footer",BUTTON_CONTAINER:"swal-button-container",BUTTON:o,CONFIRM_BUTTON:o+"--confirm",CANCEL_BUTTON:o+"--cancel",DANGER_BUTTON:o+"--danger",BUTTON_LOADING:o+"--loading",BUTTON_LOADER:o+"__loader"},e.default=e.CLASS_NAMES;},function(t,e,n){Object.defineProperty(e,"__esModule",{value:!0}),e.getNode=function(t){var e="."+t;return document.querySelector(e)},e.stringToNode=function(t){var e=document.createElement("div");return e.innerHTML=t.trim(),e.firstChild},e.insertAfter=function(t,e){var n=e.nextSibling;e.parentNode.insertBefore(t,n);},e.removeNode=function(t){t.parentElement.removeChild(t);},e.throwErr=function(t){throw t=t.replace(/ +(?= )/g,""),"SweetAlert: "+(t=t.trim())},e.isPlainObject=function(t){if("[object Object]"!==Object.prototype.toString.call(t))return !1;var e=Object.getPrototypeOf(t);return null===e||e===Object.prototype},e.ordinalSuffixOf=function(t){var e=t%10,n=t%100;return 1===e&&11!==n?t+"st":2===e&&12!==n?t+"nd":3===e&&13!==n?t+"rd":t+"th"};},function(t,e,n){function o(t){for(var n in t)e.hasOwnProperty(n)||(e[n]=t[n]);}Object.defineProperty(e,"__esModule",{value:!0}),o(n(25));var r=n(26);e.overlayMarkup=r.default,o(n(27)),o(n(28)),o(n(29));var i=n(0),a=i.default.MODAL_TITLE,s=i.default.MODAL_TEXT,c=i.default.ICON,l=i.default.FOOTER;e.iconMarkup='\n  <div class="'+c+'"></div>',e.titleMarkup='\n  <div class="'+a+'"></div>\n',e.textMarkup='\n  <div class="'+s+'"></div>',e.footerMarkup='\n  <div class="'+l+'"></div>\n';},function(t,e,n){Object.defineProperty(e,"__esModule",{value:!0});var o=n(1);e.CONFIRM_KEY="confirm",e.CANCEL_KEY="cancel";var r={visible:!0,text:null,value:null,className:"",closeModal:!0},i=Object.assign({},r,{visible:!1,text:"Cancel",value:null}),a=Object.assign({},r,{text:"OK",value:!0});e.defaultButtonList={cancel:i,confirm:a};var s=function(t){switch(t){case e.CONFIRM_KEY:return a;case e.CANCEL_KEY:return i;default:var n=t.charAt(0).toUpperCase()+t.slice(1);return Object.assign({},r,{text:n,value:t})}},c=function(t,e){var n=s(t);return !0===e?Object.assign({},n,{visible:!0}):"string"==typeof e?Object.assign({},n,{visible:!0,text:e}):o.isPlainObject(e)?Object.assign({visible:!0},n,e):Object.assign({},n,{visible:!1})},l=function(t){for(var e={},n=0,o=Object.keys(t);n<o.length;n++){var r=o[n],a=t[r],s=c(r,a);e[r]=s;}return e.cancel||(e.cancel=i),e},u=function(t){var n={};switch(t.length){case 1:n[e.CANCEL_KEY]=Object.assign({},i,{visible:!1});break;case 2:n[e.CANCEL_KEY]=c(e.CANCEL_KEY,t[0]),n[e.CONFIRM_KEY]=c(e.CONFIRM_KEY,t[1]);break;default:o.throwErr("Invalid number of 'buttons' in array ("+t.length+").\n      If you want more than 2 buttons, you need to use an object!");}return n};e.getButtonListOpts=function(t){var n=e.defaultButtonList;return "string"==typeof t?n[e.CONFIRM_KEY]=c(e.CONFIRM_KEY,t):Array.isArray(t)?n=u(t):o.isPlainObject(t)?n=l(t):!0===t?n=u([!0,!0]):!1===t?n=u([!1,!1]):void 0===t&&(n=e.defaultButtonList),n};},function(t,e,n){Object.defineProperty(e,"__esModule",{value:!0});var o=n(1),r=n(2),i=n(0),a=i.default.MODAL,s=i.default.OVERLAY,c=n(30),l=n(31),u=n(32),f=n(33);e.injectElIntoModal=function(t){var e=o.getNode(a),n=o.stringToNode(t);return e.appendChild(n),n};var d=function(t){t.className=a,t.textContent="";},p=function(t,e){d(t);var n=e.className;n&&t.classList.add(n);};e.initModalContent=function(t){var e=o.getNode(a);p(e,t),c.default(t.icon),l.initTitle(t.title),l.initText(t.text),f.default(t.content),u.default(t.buttons,t.dangerMode);};var m=function(){var t=o.getNode(s),e=o.stringToNode(r.modalMarkup);t.appendChild(e);};e.default=m;},function(t,e,n){Object.defineProperty(e,"__esModule",{value:!0});var o=n(3),r={isOpen:!1,promise:null,actions:{},timer:null},i=Object.assign({},r);e.resetState=function(){i=Object.assign({},r);},e.setActionValue=function(t){if("string"==typeof t)return a(o.CONFIRM_KEY,t);for(var e in t)a(e,t[e]);};var a=function(t,e){i.actions[t]||(i.actions[t]={}),Object.assign(i.actions[t],{value:e});};e.setActionOptionsFor=function(t,e){var n=(void 0===e?{}:e).closeModal,o=void 0===n||n;Object.assign(i.actions[t],{closeModal:o});},e.default=i;},function(t,e,n){Object.defineProperty(e,"__esModule",{value:!0});var o=n(1),r=n(3),i=n(0),a=i.default.OVERLAY,s=i.default.SHOW_MODAL,c=i.default.BUTTON,l=i.default.BUTTON_LOADING,u=n(5);e.openModal=function(){o.getNode(a).classList.add(s),u.default.isOpen=!0;};var f=function(){o.getNode(a).classList.remove(s),u.default.isOpen=!1;};e.onAction=function(t){void 0===t&&(t=r.CANCEL_KEY);var e=u.default.actions[t],n=e.value;if(!1===e.closeModal){var i=c+"--"+t;o.getNode(i).classList.add(l);}else f();u.default.promise.resolve(n);},e.getState=function(){var t=Object.assign({},u.default);return delete t.promise,delete t.timer,t},e.stopLoading=function(){for(var t=document.querySelectorAll("."+c),e=0;e<t.length;e++){t[e].classList.remove(l);}};},function(t,e){var n;n=function(){return this}();try{n=n||Function("return this")()||(0,eval)("this");}catch(t){"object"==typeof window&&(n=window);}t.exports=n;},function(t,e,n){(function(e){t.exports=e.sweetAlert=n(9);}).call(e,n(7));},function(t,e,n){(function(e){t.exports=e.swal=n(10);}).call(e,n(7));},function(t,e,n){"undefined"!=typeof window&&n(11),n(16);var o=n(23).default;t.exports=o;},function(t,e,n){var o=n(12);"string"==typeof o&&(o=[[t.i,o,""]]);var r={insertAt:"top"};r.transform=void 0;n(14)(o,r);o.locals&&(t.exports=o.locals);},function(t,e,n){e=t.exports=n(13)(void 0),e.push([t.i,'.swal-icon--error{border-color:#f27474;-webkit-animation:animateErrorIcon .5s;animation:animateErrorIcon .5s}.swal-icon--error__x-mark{position:relative;display:block;-webkit-animation:animateXMark .5s;animation:animateXMark .5s}.swal-icon--error__line{position:absolute;height:5px;width:47px;background-color:#f27474;display:block;top:37px;border-radius:2px}.swal-icon--error__line--left{-webkit-transform:rotate(45deg);transform:rotate(45deg);left:17px}.swal-icon--error__line--right{-webkit-transform:rotate(-45deg);transform:rotate(-45deg);right:16px}@-webkit-keyframes animateErrorIcon{0%{-webkit-transform:rotateX(100deg);transform:rotateX(100deg);opacity:0}to{-webkit-transform:rotateX(0deg);transform:rotateX(0deg);opacity:1}}@keyframes animateErrorIcon{0%{-webkit-transform:rotateX(100deg);transform:rotateX(100deg);opacity:0}to{-webkit-transform:rotateX(0deg);transform:rotateX(0deg);opacity:1}}@-webkit-keyframes animateXMark{0%{-webkit-transform:scale(.4);transform:scale(.4);margin-top:26px;opacity:0}50%{-webkit-transform:scale(.4);transform:scale(.4);margin-top:26px;opacity:0}80%{-webkit-transform:scale(1.15);transform:scale(1.15);margin-top:-6px}to{-webkit-transform:scale(1);transform:scale(1);margin-top:0;opacity:1}}@keyframes animateXMark{0%{-webkit-transform:scale(.4);transform:scale(.4);margin-top:26px;opacity:0}50%{-webkit-transform:scale(.4);transform:scale(.4);margin-top:26px;opacity:0}80%{-webkit-transform:scale(1.15);transform:scale(1.15);margin-top:-6px}to{-webkit-transform:scale(1);transform:scale(1);margin-top:0;opacity:1}}.swal-icon--warning{border-color:#f8bb86;-webkit-animation:pulseWarning .75s infinite alternate;animation:pulseWarning .75s infinite alternate}.swal-icon--warning__body{width:5px;height:47px;top:10px;border-radius:2px;margin-left:-2px}.swal-icon--warning__body,.swal-icon--warning__dot{position:absolute;left:50%;background-color:#f8bb86}.swal-icon--warning__dot{width:7px;height:7px;border-radius:50%;margin-left:-4px;bottom:-11px}@-webkit-keyframes pulseWarning{0%{border-color:#f8d486}to{border-color:#f8bb86}}@keyframes pulseWarning{0%{border-color:#f8d486}to{border-color:#f8bb86}}.swal-icon--success{border-color:#a5dc86}.swal-icon--success:after,.swal-icon--success:before{content:"";border-radius:50%;position:absolute;width:60px;height:120px;background:#fff;-webkit-transform:rotate(45deg);transform:rotate(45deg)}.swal-icon--success:before{border-radius:120px 0 0 120px;top:-7px;left:-33px;-webkit-transform:rotate(-45deg);transform:rotate(-45deg);-webkit-transform-origin:60px 60px;transform-origin:60px 60px}.swal-icon--success:after{border-radius:0 120px 120px 0;top:-11px;left:30px;-webkit-transform:rotate(-45deg);transform:rotate(-45deg);-webkit-transform-origin:0 60px;transform-origin:0 60px;-webkit-animation:rotatePlaceholder 4.25s ease-in;animation:rotatePlaceholder 4.25s ease-in}.swal-icon--success__ring{width:80px;height:80px;border:4px solid hsla(98,55%,69%,.2);border-radius:50%;box-sizing:content-box;position:absolute;left:-4px;top:-4px;z-index:2}.swal-icon--success__hide-corners{width:5px;height:90px;background-color:#fff;padding:1px;position:absolute;left:28px;top:8px;z-index:1;-webkit-transform:rotate(-45deg);transform:rotate(-45deg)}.swal-icon--success__line{height:5px;background-color:#a5dc86;display:block;border-radius:2px;position:absolute;z-index:2}.swal-icon--success__line--tip{width:25px;left:14px;top:46px;-webkit-transform:rotate(45deg);transform:rotate(45deg);-webkit-animation:animateSuccessTip .75s;animation:animateSuccessTip .75s}.swal-icon--success__line--long{width:47px;right:8px;top:38px;-webkit-transform:rotate(-45deg);transform:rotate(-45deg);-webkit-animation:animateSuccessLong .75s;animation:animateSuccessLong .75s}@-webkit-keyframes rotatePlaceholder{0%{-webkit-transform:rotate(-45deg);transform:rotate(-45deg)}5%{-webkit-transform:rotate(-45deg);transform:rotate(-45deg)}12%{-webkit-transform:rotate(-405deg);transform:rotate(-405deg)}to{-webkit-transform:rotate(-405deg);transform:rotate(-405deg)}}@keyframes rotatePlaceholder{0%{-webkit-transform:rotate(-45deg);transform:rotate(-45deg)}5%{-webkit-transform:rotate(-45deg);transform:rotate(-45deg)}12%{-webkit-transform:rotate(-405deg);transform:rotate(-405deg)}to{-webkit-transform:rotate(-405deg);transform:rotate(-405deg)}}@-webkit-keyframes animateSuccessTip{0%{width:0;left:1px;top:19px}54%{width:0;left:1px;top:19px}70%{width:50px;left:-8px;top:37px}84%{width:17px;left:21px;top:48px}to{width:25px;left:14px;top:45px}}@keyframes animateSuccessTip{0%{width:0;left:1px;top:19px}54%{width:0;left:1px;top:19px}70%{width:50px;left:-8px;top:37px}84%{width:17px;left:21px;top:48px}to{width:25px;left:14px;top:45px}}@-webkit-keyframes animateSuccessLong{0%{width:0;right:46px;top:54px}65%{width:0;right:46px;top:54px}84%{width:55px;right:0;top:35px}to{width:47px;right:8px;top:38px}}@keyframes animateSuccessLong{0%{width:0;right:46px;top:54px}65%{width:0;right:46px;top:54px}84%{width:55px;right:0;top:35px}to{width:47px;right:8px;top:38px}}.swal-icon--info{border-color:#c9dae1}.swal-icon--info:before{width:5px;height:29px;bottom:17px;border-radius:2px;margin-left:-2px}.swal-icon--info:after,.swal-icon--info:before{content:"";position:absolute;left:50%;background-color:#c9dae1}.swal-icon--info:after{width:7px;height:7px;border-radius:50%;margin-left:-3px;top:19px}.swal-icon{width:80px;height:80px;border-width:4px;border-style:solid;border-radius:50%;padding:0;position:relative;box-sizing:content-box;margin:20px auto}.swal-icon:first-child{margin-top:32px}.swal-icon--custom{width:auto;height:auto;max-width:100%;border:none;border-radius:0}.swal-icon img{max-width:100%;max-height:100%}.swal-title{color:rgba(0,0,0,.65);font-weight:600;text-transform:none;position:relative;display:block;padding:13px 16px;font-size:27px;line-height:normal;text-align:center;margin-bottom:0}.swal-title:first-child{margin-top:26px}.swal-title:not(:first-child){padding-bottom:0}.swal-title:not(:last-child){margin-bottom:13px}.swal-text{font-size:16px;position:relative;float:none;line-height:normal;vertical-align:top;text-align:left;display:inline-block;margin:0;padding:0 10px;font-weight:400;color:rgba(0,0,0,.64);max-width:calc(100% - 20px);overflow-wrap:break-word;box-sizing:border-box}.swal-text:first-child{margin-top:45px}.swal-text:last-child{margin-bottom:45px}.swal-footer{text-align:right;padding-top:13px;margin-top:13px;padding:13px 16px;border-radius:inherit;border-top-left-radius:0;border-top-right-radius:0}.swal-button-container{margin:5px;display:inline-block;position:relative}.swal-button{background-color:#7cd1f9;color:#fff;border:none;box-shadow:none;border-radius:5px;font-weight:600;font-size:14px;padding:10px 24px;margin:0;cursor:pointer}.swal-button:not([disabled]):hover{background-color:#78cbf2}.swal-button:active{background-color:#70bce0}.swal-button:focus{outline:none;box-shadow:0 0 0 1px #fff,0 0 0 3px rgba(43,114,165,.29)}.swal-button[disabled]{opacity:.5;cursor:default}.swal-button::-moz-focus-inner{border:0}.swal-button--cancel{color:#555;background-color:#efefef}.swal-button--cancel:not([disabled]):hover{background-color:#e8e8e8}.swal-button--cancel:active{background-color:#d7d7d7}.swal-button--cancel:focus{box-shadow:0 0 0 1px #fff,0 0 0 3px rgba(116,136,150,.29)}.swal-button--danger{background-color:#e64942}.swal-button--danger:not([disabled]):hover{background-color:#df4740}.swal-button--danger:active{background-color:#cf423b}.swal-button--danger:focus{box-shadow:0 0 0 1px #fff,0 0 0 3px rgba(165,43,43,.29)}.swal-content{padding:0 20px;margin-top:20px;font-size:medium}.swal-content:last-child{margin-bottom:20px}.swal-content__input,.swal-content__textarea{-webkit-appearance:none;background-color:#fff;border:none;font-size:14px;display:block;box-sizing:border-box;width:100%;border:1px solid rgba(0,0,0,.14);padding:10px 13px;border-radius:2px;transition:border-color .2s}.swal-content__input:focus,.swal-content__textarea:focus{outline:none;border-color:#6db8ff}.swal-content__textarea{resize:vertical}.swal-button--loading{color:transparent}.swal-button--loading~.swal-button__loader{opacity:1}.swal-button__loader{position:absolute;height:auto;width:43px;z-index:2;left:50%;top:50%;-webkit-transform:translateX(-50%) translateY(-50%);transform:translateX(-50%) translateY(-50%);text-align:center;pointer-events:none;opacity:0}.swal-button__loader div{display:inline-block;float:none;vertical-align:baseline;width:9px;height:9px;padding:0;border:none;margin:2px;opacity:.4;border-radius:7px;background-color:hsla(0,0%,100%,.9);transition:background .2s;-webkit-animation:swal-loading-anim 1s infinite;animation:swal-loading-anim 1s infinite}.swal-button__loader div:nth-child(3n+2){-webkit-animation-delay:.15s;animation-delay:.15s}.swal-button__loader div:nth-child(3n+3){-webkit-animation-delay:.3s;animation-delay:.3s}@-webkit-keyframes swal-loading-anim{0%{opacity:.4}20%{opacity:.4}50%{opacity:1}to{opacity:.4}}@keyframes swal-loading-anim{0%{opacity:.4}20%{opacity:.4}50%{opacity:1}to{opacity:.4}}.swal-overlay{position:fixed;top:0;bottom:0;left:0;right:0;text-align:center;font-size:0;overflow-y:auto;background-color:rgba(0,0,0,.4);z-index:10000;pointer-events:none;opacity:0;transition:opacity .3s}.swal-overlay:before{content:" ";display:inline-block;vertical-align:middle;height:100%}.swal-overlay--show-modal{opacity:1;pointer-events:auto}.swal-overlay--show-modal .swal-modal{opacity:1;pointer-events:auto;box-sizing:border-box;-webkit-animation:showSweetAlert .3s;animation:showSweetAlert .3s;will-change:transform}.swal-modal{width:478px;opacity:0;pointer-events:none;background-color:#fff;text-align:center;border-radius:5px;position:static;margin:20px auto;display:inline-block;vertical-align:middle;-webkit-transform:scale(1);transform:scale(1);-webkit-transform-origin:50% 50%;transform-origin:50% 50%;z-index:10001;transition:opacity .2s,-webkit-transform .3s;transition:transform .3s,opacity .2s;transition:transform .3s,opacity .2s,-webkit-transform .3s}@media (max-width:500px){.swal-modal{width:calc(100% - 20px)}}@-webkit-keyframes showSweetAlert{0%{-webkit-transform:scale(1);transform:scale(1)}1%{-webkit-transform:scale(.5);transform:scale(.5)}45%{-webkit-transform:scale(1.05);transform:scale(1.05)}80%{-webkit-transform:scale(.95);transform:scale(.95)}to{-webkit-transform:scale(1);transform:scale(1)}}@keyframes showSweetAlert{0%{-webkit-transform:scale(1);transform:scale(1)}1%{-webkit-transform:scale(.5);transform:scale(.5)}45%{-webkit-transform:scale(1.05);transform:scale(1.05)}80%{-webkit-transform:scale(.95);transform:scale(.95)}to{-webkit-transform:scale(1);transform:scale(1)}}',""]);},function(t,e){function n(t,e){var n=t[1]||"",r=t[3];if(!r)return n;if(e&&"function"==typeof btoa){var i=o(r);return [n].concat(r.sources.map(function(t){return "/*# sourceURL="+r.sourceRoot+t+" */"})).concat([i]).join("\n")}return [n].join("\n")}function o(t){return "/*# sourceMappingURL=data:application/json;charset=utf-8;base64,"+btoa(unescape(encodeURIComponent(JSON.stringify(t))))+" */"}t.exports=function(t){var e=[];return e.toString=function(){return this.map(function(e){var o=n(e,t);return e[2]?"@media "+e[2]+"{"+o+"}":o}).join("")},e.i=function(t,n){"string"==typeof t&&(t=[[null,t,""]]);for(var o={},r=0;r<this.length;r++){var i=this[r][0];"number"==typeof i&&(o[i]=!0);}for(r=0;r<t.length;r++){var a=t[r];"number"==typeof a[0]&&o[a[0]]||(n&&!a[2]?a[2]=n:n&&(a[2]="("+a[2]+") and ("+n+")"),e.push(a));}},e};},function(t,e,n){function o(t,e){for(var n=0;n<t.length;n++){var o=t[n],r=m[o.id];if(r){r.refs++;for(var i=0;i<r.parts.length;i++)r.parts[i](o.parts[i]);for(;i<o.parts.length;i++)r.parts.push(u(o.parts[i],e));}else {for(var a=[],i=0;i<o.parts.length;i++)a.push(u(o.parts[i],e));m[o.id]={id:o.id,refs:1,parts:a};}}}function r(t,e){for(var n=[],o={},r=0;r<t.length;r++){var i=t[r],a=e.base?i[0]+e.base:i[0],s=i[1],c=i[2],l=i[3],u={css:s,media:c,sourceMap:l};o[a]?o[a].parts.push(u):n.push(o[a]={id:a,parts:[u]});}return n}function i(t,e){var n=v(t.insertInto);if(!n)throw new Error("Couldn't find a style target. This probably means that the value for the 'insertInto' parameter is invalid.");var o=w[w.length-1];if("top"===t.insertAt)o?o.nextSibling?n.insertBefore(e,o.nextSibling):n.appendChild(e):n.insertBefore(e,n.firstChild),w.push(e);else {if("bottom"!==t.insertAt)throw new Error("Invalid value for parameter 'insertAt'. Must be 'top' or 'bottom'.");n.appendChild(e);}}function a(t){if(null===t.parentNode)return !1;t.parentNode.removeChild(t);var e=w.indexOf(t);e>=0&&w.splice(e,1);}function s(t){var e=document.createElement("style");return t.attrs.type="text/css",l(e,t.attrs),i(t,e),e}function c(t){var e=document.createElement("link");return t.attrs.type="text/css",t.attrs.rel="stylesheet",l(e,t.attrs),i(t,e),e}function l(t,e){Object.keys(e).forEach(function(n){t.setAttribute(n,e[n]);});}function u(t,e){var n,o,r,i;if(e.transform&&t.css){if(!(i=e.transform(t.css)))return function(){};t.css=i;}if(e.singleton){var l=h++;n=g||(g=s(e)),o=f.bind(null,n,l,!1),r=f.bind(null,n,l,!0);}else t.sourceMap&&"function"==typeof URL&&"function"==typeof URL.createObjectURL&&"function"==typeof URL.revokeObjectURL&&"function"==typeof Blob&&"function"==typeof btoa?(n=c(e),o=p.bind(null,n,e),r=function(){a(n),n.href&&URL.revokeObjectURL(n.href);}):(n=s(e),o=d.bind(null,n),r=function(){a(n);});return o(t),function(e){if(e){if(e.css===t.css&&e.media===t.media&&e.sourceMap===t.sourceMap)return;o(t=e);}else r();}}function f(t,e,n,o){var r=n?"":o.css;if(t.styleSheet)t.styleSheet.cssText=x(e,r);else {var i=document.createTextNode(r),a=t.childNodes;a[e]&&t.removeChild(a[e]),a.length?t.insertBefore(i,a[e]):t.appendChild(i);}}function d(t,e){var n=e.css,o=e.media;if(o&&t.setAttribute("media",o),t.styleSheet)t.styleSheet.cssText=n;else {for(;t.firstChild;)t.removeChild(t.firstChild);t.appendChild(document.createTextNode(n));}}function p(t,e,n){var o=n.css,r=n.sourceMap,i=void 0===e.convertToAbsoluteUrls&&r;(e.convertToAbsoluteUrls||i)&&(o=y(o)),r&&(o+="\n/*# sourceMappingURL=data:application/json;base64,"+btoa(unescape(encodeURIComponent(JSON.stringify(r))))+" */");var a=new Blob([o],{type:"text/css"}),s=t.href;t.href=URL.createObjectURL(a),s&&URL.revokeObjectURL(s);}var m={},b=function(t){var e;return function(){return void 0===e&&(e=t.apply(this,arguments)),e}}(function(){return window&&document&&document.all&&!window.atob}),v=function(t){var e={};return function(n){return void 0===e[n]&&(e[n]=t.call(this,n)),e[n]}}(function(t){return document.querySelector(t)}),g=null,h=0,w=[],y=n(15);t.exports=function(t,e){if("undefined"!=typeof DEBUG&&DEBUG&&"object"!=typeof document)throw new Error("The style-loader cannot be used in a non-browser environment");e=e||{},e.attrs="object"==typeof e.attrs?e.attrs:{},e.singleton||(e.singleton=b()),e.insertInto||(e.insertInto="head"),e.insertAt||(e.insertAt="bottom");var n=r(t,e);return o(n,e),function(t){for(var i=[],a=0;a<n.length;a++){var s=n[a],c=m[s.id];c.refs--,i.push(c);}if(t){o(r(t,e),e);}for(var a=0;a<i.length;a++){var c=i[a];if(0===c.refs){for(var l=0;l<c.parts.length;l++)c.parts[l]();delete m[c.id];}}}};var x=function(){var t=[];return function(e,n){return t[e]=n,t.filter(Boolean).join("\n")}}();},function(t,e){t.exports=function(t){var e="undefined"!=typeof window&&window.location;if(!e)throw new Error("fixUrls requires window.location");if(!t||"string"!=typeof t)return t;var n=e.protocol+"//"+e.host,o=n+e.pathname.replace(/\/[^\/]*$/,"/");return t.replace(/url\s*\(((?:[^)(]|\((?:[^)(]+|\([^)(]*\))*\))*)\)/gi,function(t,e){var r=e.trim().replace(/^"(.*)"$/,function(t,e){return e}).replace(/^'(.*)'$/,function(t,e){return e});if(/^(#|data:|http:\/\/|https:\/\/|file:\/\/\/)/i.test(r))return t;var i;return i=0===r.indexOf("//")?r:0===r.indexOf("/")?n+r:o+r.replace(/^\.\//,""),"url("+JSON.stringify(i)+")"})};},function(t,e,n){var o=n(17);"undefined"==typeof window||window.Promise||(window.Promise=o),n(21),String.prototype.includes||(String.prototype.includes=function(t,e){return "number"!=typeof e&&(e=0),!(e+t.length>this.length)&&-1!==this.indexOf(t,e)}),Array.prototype.includes||Object.defineProperty(Array.prototype,"includes",{value:function(t,e){if(null==this)throw new TypeError('"this" is null or not defined');var n=Object(this),o=n.length>>>0;if(0===o)return !1;for(var r=0|e,i=Math.max(r>=0?r:o-Math.abs(r),0);i<o;){if(function(t,e){return t===e||"number"==typeof t&&"number"==typeof e&&isNaN(t)&&isNaN(e)}(n[i],t))return !0;i++;}return !1}}),"undefined"!=typeof window&&function(t){t.forEach(function(t){t.hasOwnProperty("remove")||Object.defineProperty(t,"remove",{configurable:!0,enumerable:!0,writable:!0,value:function(){this.parentNode.removeChild(this);}});});}([Element.prototype,CharacterData.prototype,DocumentType.prototype]);},function(t,e,n){(function(e){!function(n){function o(){}function r(t,e){return function(){t.apply(e,arguments);}}function i(t){if("object"!=typeof this)throw new TypeError("Promises must be constructed via new");if("function"!=typeof t)throw new TypeError("not a function");this._state=0,this._handled=!1,this._value=void 0,this._deferreds=[],f(t,this);}function a(t,e){for(;3===t._state;)t=t._value;if(0===t._state)return void t._deferreds.push(e);t._handled=!0,i._immediateFn(function(){var n=1===t._state?e.onFulfilled:e.onRejected;if(null===n)return void(1===t._state?s:c)(e.promise,t._value);var o;try{o=n(t._value);}catch(t){return void c(e.promise,t)}s(e.promise,o);});}function s(t,e){try{if(e===t)throw new TypeError("A promise cannot be resolved with itself.");if(e&&("object"==typeof e||"function"==typeof e)){var n=e.then;if(e instanceof i)return t._state=3,t._value=e,void l(t);if("function"==typeof n)return void f(r(n,e),t)}t._state=1,t._value=e,l(t);}catch(e){c(t,e);}}function c(t,e){t._state=2,t._value=e,l(t);}function l(t){2===t._state&&0===t._deferreds.length&&i._immediateFn(function(){t._handled||i._unhandledRejectionFn(t._value);});for(var e=0,n=t._deferreds.length;e<n;e++)a(t,t._deferreds[e]);t._deferreds=null;}function u(t,e,n){this.onFulfilled="function"==typeof t?t:null,this.onRejected="function"==typeof e?e:null,this.promise=n;}function f(t,e){var n=!1;try{t(function(t){n||(n=!0,s(e,t));},function(t){n||(n=!0,c(e,t));});}catch(t){if(n)return;n=!0,c(e,t);}}var d=setTimeout;i.prototype.catch=function(t){return this.then(null,t)},i.prototype.then=function(t,e){var n=new this.constructor(o);return a(this,new u(t,e,n)),n},i.all=function(t){var e=Array.prototype.slice.call(t);return new i(function(t,n){function o(i,a){try{if(a&&("object"==typeof a||"function"==typeof a)){var s=a.then;if("function"==typeof s)return void s.call(a,function(t){o(i,t);},n)}e[i]=a,0==--r&&t(e);}catch(t){n(t);}}if(0===e.length)return t([]);for(var r=e.length,i=0;i<e.length;i++)o(i,e[i]);})},i.resolve=function(t){return t&&"object"==typeof t&&t.constructor===i?t:new i(function(e){e(t);})},i.reject=function(t){return new i(function(e,n){n(t);})},i.race=function(t){return new i(function(e,n){for(var o=0,r=t.length;o<r;o++)t[o].then(e,n);})},i._immediateFn="function"==typeof e&&function(t){e(t);}||function(t){d(t,0);},i._unhandledRejectionFn=function(t){"undefined"!=typeof console&&console&&console.warn("Possible Unhandled Promise Rejection:",t);},i._setImmediateFn=function(t){i._immediateFn=t;},i._setUnhandledRejectionFn=function(t){i._unhandledRejectionFn=t;},void 0!==t&&t.exports?t.exports=i:n.Promise||(n.Promise=i);}(this);}).call(e,n(18).setImmediate);},function(t,e,n){function o(t,e){this._id=t,this._clearFn=e;}var r=Function.prototype.apply;e.setTimeout=function(){return new o(r.call(setTimeout,window,arguments),clearTimeout)},e.setInterval=function(){return new o(r.call(setInterval,window,arguments),clearInterval)},e.clearTimeout=e.clearInterval=function(t){t&&t.close();},o.prototype.unref=o.prototype.ref=function(){},o.prototype.close=function(){this._clearFn.call(window,this._id);},e.enroll=function(t,e){clearTimeout(t._idleTimeoutId),t._idleTimeout=e;},e.unenroll=function(t){clearTimeout(t._idleTimeoutId),t._idleTimeout=-1;},e._unrefActive=e.active=function(t){clearTimeout(t._idleTimeoutId);var e=t._idleTimeout;e>=0&&(t._idleTimeoutId=setTimeout(function(){t._onTimeout&&t._onTimeout();},e));},n(19),e.setImmediate=setImmediate,e.clearImmediate=clearImmediate;},function(t,e,n){(function(t,e){!function(t,n){function o(t){"function"!=typeof t&&(t=new Function(""+t));for(var e=new Array(arguments.length-1),n=0;n<e.length;n++)e[n]=arguments[n+1];var o={callback:t,args:e};return l[c]=o,s(c),c++}function r(t){delete l[t];}function i(t){var e=t.callback,o=t.args;switch(o.length){case 0:e();break;case 1:e(o[0]);break;case 2:e(o[0],o[1]);break;case 3:e(o[0],o[1],o[2]);break;default:e.apply(n,o);}}function a(t){if(u)setTimeout(a,0,t);else {var e=l[t];if(e){u=!0;try{i(e);}finally{r(t),u=!1;}}}}if(!t.setImmediate){var s,c=1,l={},u=!1,f=t.document,d=Object.getPrototypeOf&&Object.getPrototypeOf(t);d=d&&d.setTimeout?d:t,"[object process]"==={}.toString.call(t.process)?function(){s=function(t){e.nextTick(function(){a(t);});};}():function(){if(t.postMessage&&!t.importScripts){var e=!0,n=t.onmessage;return t.onmessage=function(){e=!1;},t.postMessage("","*"),t.onmessage=n,e}}()?function(){var e="setImmediate$"+Math.random()+"$",n=function(n){n.source===t&&"string"==typeof n.data&&0===n.data.indexOf(e)&&a(+n.data.slice(e.length));};t.addEventListener?t.addEventListener("message",n,!1):t.attachEvent("onmessage",n),s=function(n){t.postMessage(e+n,"*");};}():t.MessageChannel?function(){var t=new MessageChannel;t.port1.onmessage=function(t){a(t.data);},s=function(e){t.port2.postMessage(e);};}():f&&"onreadystatechange"in f.createElement("script")?function(){var t=f.documentElement;s=function(e){var n=f.createElement("script");n.onreadystatechange=function(){a(e),n.onreadystatechange=null,t.removeChild(n),n=null;},t.appendChild(n);};}():function(){s=function(t){setTimeout(a,0,t);};}(),d.setImmediate=o,d.clearImmediate=r;}}("undefined"==typeof self?void 0===t?this:t:self);}).call(e,n(7),n(20));},function(t,e){function n(){throw new Error("setTimeout has not been defined")}function o(){throw new Error("clearTimeout has not been defined")}function r(t){if(u===setTimeout)return setTimeout(t,0);if((u===n||!u)&&setTimeout)return u=setTimeout,setTimeout(t,0);try{return u(t,0)}catch(e){try{return u.call(null,t,0)}catch(e){return u.call(this,t,0)}}}function i(t){if(f===clearTimeout)return clearTimeout(t);if((f===o||!f)&&clearTimeout)return f=clearTimeout,clearTimeout(t);try{return f(t)}catch(e){try{return f.call(null,t)}catch(e){return f.call(this,t)}}}function a(){b&&p&&(b=!1,p.length?m=p.concat(m):v=-1,m.length&&s());}function s(){if(!b){var t=r(a);b=!0;for(var e=m.length;e;){for(p=m,m=[];++v<e;)p&&p[v].run();v=-1,e=m.length;}p=null,b=!1,i(t);}}function c(t,e){this.fun=t,this.array=e;}function l(){}var u,f,d=t.exports={};!function(){try{u="function"==typeof setTimeout?setTimeout:n;}catch(t){u=n;}try{f="function"==typeof clearTimeout?clearTimeout:o;}catch(t){f=o;}}();var p,m=[],b=!1,v=-1;d.nextTick=function(t){var e=new Array(arguments.length-1);if(arguments.length>1)for(var n=1;n<arguments.length;n++)e[n-1]=arguments[n];m.push(new c(t,e)),1!==m.length||b||r(s);},c.prototype.run=function(){this.fun.apply(null,this.array);},d.title="browser",d.browser=!0,d.env={},d.argv=[],d.version="",d.versions={},d.on=l,d.addListener=l,d.once=l,d.off=l,d.removeListener=l,d.removeAllListeners=l,d.emit=l,d.prependListener=l,d.prependOnceListener=l,d.listeners=function(t){return []},d.binding=function(t){throw new Error("process.binding is not supported")},d.cwd=function(){return "/"},d.chdir=function(t){throw new Error("process.chdir is not supported")},d.umask=function(){return 0};},function(t,e,n){n(22).polyfill();},function(t,e,n){function o(t,e){if(void 0===t||null===t)throw new TypeError("Cannot convert first argument to object");for(var n=Object(t),o=1;o<arguments.length;o++){var r=arguments[o];if(void 0!==r&&null!==r)for(var i=Object.keys(Object(r)),a=0,s=i.length;a<s;a++){var c=i[a],l=Object.getOwnPropertyDescriptor(r,c);void 0!==l&&l.enumerable&&(n[c]=r[c]);}}return n}function r(){Object.assign||Object.defineProperty(Object,"assign",{enumerable:!1,configurable:!0,writable:!0,value:o});}t.exports={assign:o,polyfill:r};},function(t,e,n){Object.defineProperty(e,"__esModule",{value:!0});var o=n(24),r=n(6),i=n(5),a=n(36),s=function(){for(var t=[],e=0;e<arguments.length;e++)t[e]=arguments[e];if("undefined"!=typeof window){var n=a.getOpts.apply(void 0,t);return new Promise(function(t,e){i.default.promise={resolve:t,reject:e},o.default(n),setTimeout(function(){r.openModal();});})}};s.close=r.onAction,s.getState=r.getState,s.setActionValue=i.setActionValue,s.stopLoading=r.stopLoading,s.setDefaults=a.setDefaults,e.default=s;},function(t,e,n){Object.defineProperty(e,"__esModule",{value:!0});var o=n(1),r=n(0),i=r.default.MODAL,a=n(4),s=n(34),c=n(35),l=n(1);e.init=function(t){o.getNode(i)||(document.body||l.throwErr("You can only use SweetAlert AFTER the DOM has loaded!"),s.default(),a.default()),a.initModalContent(t),c.default(t);},e.default=e.init;},function(t,e,n){Object.defineProperty(e,"__esModule",{value:!0});var o=n(0),r=o.default.MODAL;e.modalMarkup='\n  <div class="'+r+'" role="dialog" aria-modal="true"></div>',e.default=e.modalMarkup;},function(t,e,n){Object.defineProperty(e,"__esModule",{value:!0});var o=n(0),r=o.default.OVERLAY,i='<div \n    class="'+r+'"\n    tabIndex="-1">\n  </div>';e.default=i;},function(t,e,n){Object.defineProperty(e,"__esModule",{value:!0});var o=n(0),r=o.default.ICON;e.errorIconMarkup=function(){var t=r+"--error",e=t+"__line";return '\n    <div class="'+t+'__x-mark">\n      <span class="'+e+" "+e+'--left"></span>\n      <span class="'+e+" "+e+'--right"></span>\n    </div>\n  '},e.warningIconMarkup=function(){var t=r+"--warning";return '\n    <span class="'+t+'__body">\n      <span class="'+t+'__dot"></span>\n    </span>\n  '},e.successIconMarkup=function(){var t=r+"--success";return '\n    <span class="'+t+"__line "+t+'__line--long"></span>\n    <span class="'+t+"__line "+t+'__line--tip"></span>\n\n    <div class="'+t+'__ring"></div>\n    <div class="'+t+'__hide-corners"></div>\n  '};},function(t,e,n){Object.defineProperty(e,"__esModule",{value:!0});var o=n(0),r=o.default.CONTENT;e.contentMarkup='\n  <div class="'+r+'">\n\n  </div>\n';},function(t,e,n){Object.defineProperty(e,"__esModule",{value:!0});var o=n(0),r=o.default.BUTTON_CONTAINER,i=o.default.BUTTON,a=o.default.BUTTON_LOADER;e.buttonMarkup='\n  <div class="'+r+'">\n\n    <button\n      class="'+i+'"\n    ></button>\n\n    <div class="'+a+'">\n      <div></div>\n      <div></div>\n      <div></div>\n    </div>\n\n  </div>\n';},function(t,e,n){Object.defineProperty(e,"__esModule",{value:!0});var o=n(4),r=n(2),i=n(0),a=i.default.ICON,s=i.default.ICON_CUSTOM,c=["error","warning","success","info"],l={error:r.errorIconMarkup(),warning:r.warningIconMarkup(),success:r.successIconMarkup()},u=function(t,e){var n=a+"--"+t;e.classList.add(n);var o=l[t];o&&(e.innerHTML=o);},f=function(t,e){e.classList.add(s);var n=document.createElement("img");n.src=t,e.appendChild(n);},d=function(t){if(t){var e=o.injectElIntoModal(r.iconMarkup);c.includes(t)?u(t,e):f(t,e);}};e.default=d;},function(t,e,n){Object.defineProperty(e,"__esModule",{value:!0});var o=n(2),r=n(4),i=function(t){navigator.userAgent.includes("AppleWebKit")&&(t.style.display="none",t.offsetHeight,t.style.display="");};e.initTitle=function(t){if(t){var e=r.injectElIntoModal(o.titleMarkup);e.textContent=t,i(e);}},e.initText=function(t){if(t){var e=document.createDocumentFragment();t.split("\n").forEach(function(t,n,o){e.appendChild(document.createTextNode(t)),n<o.length-1&&e.appendChild(document.createElement("br"));});var n=r.injectElIntoModal(o.textMarkup);n.appendChild(e),i(n);}};},function(t,e,n){Object.defineProperty(e,"__esModule",{value:!0});var o=n(1),r=n(4),i=n(0),a=i.default.BUTTON,s=i.default.DANGER_BUTTON,c=n(3),l=n(2),u=n(6),f=n(5),d=function(t,e,n){var r=e.text,i=e.value,d=e.className,p=e.closeModal,m=o.stringToNode(l.buttonMarkup),b=m.querySelector("."+a),v=a+"--"+t;if(b.classList.add(v),d){(Array.isArray(d)?d:d.split(" ")).filter(function(t){return t.length>0}).forEach(function(t){b.classList.add(t);});}n&&t===c.CONFIRM_KEY&&b.classList.add(s),b.textContent=r;var g={};return g[t]=i,f.setActionValue(g),f.setActionOptionsFor(t,{closeModal:p}),b.addEventListener("click",function(){return u.onAction(t)}),m},p=function(t,e){var n=r.injectElIntoModal(l.footerMarkup);for(var o in t){var i=t[o],a=d(o,i,e);i.visible&&n.appendChild(a);}0===n.children.length&&n.remove();};e.default=p;},function(t,e,n){Object.defineProperty(e,"__esModule",{value:!0});var o=n(3),r=n(4),i=n(2),a=n(5),s=n(6),c=n(0),l=c.default.CONTENT,u=function(t){t.addEventListener("input",function(t){var e=t.target,n=e.value;a.setActionValue(n);}),t.addEventListener("keyup",function(t){if("Enter"===t.key)return s.onAction(o.CONFIRM_KEY)}),setTimeout(function(){t.focus(),a.setActionValue("");},0);},f=function(t,e,n){var o=document.createElement(e),r=l+"__"+e;o.classList.add(r);for(var i in n){var a=n[i];o[i]=a;}"input"===e&&u(o),t.appendChild(o);},d=function(t){if(t){var e=r.injectElIntoModal(i.contentMarkup),n=t.element,o=t.attributes;"string"==typeof n?f(e,n,o):e.appendChild(n);}};e.default=d;},function(t,e,n){Object.defineProperty(e,"__esModule",{value:!0});var o=n(1),r=n(2),i=function(){var t=o.stringToNode(r.overlayMarkup);document.body.appendChild(t);};e.default=i;},function(t,e,n){Object.defineProperty(e,"__esModule",{value:!0});var o=n(5),r=n(6),i=n(1),a=n(3),s=n(0),c=s.default.MODAL,l=s.default.BUTTON,u=s.default.OVERLAY,f=function(t){t.preventDefault(),v();},d=function(t){t.preventDefault(),g();},p=function(t){if(o.default.isOpen)switch(t.key){case"Escape":return r.onAction(a.CANCEL_KEY)}},m=function(t){if(o.default.isOpen)switch(t.key){case"Tab":return f(t)}},b=function(t){if(o.default.isOpen)return "Tab"===t.key&&t.shiftKey?d(t):void 0},v=function(){var t=i.getNode(l);t&&(t.tabIndex=0,t.focus());},g=function(){var t=i.getNode(c),e=t.querySelectorAll("."+l),n=e.length-1,o=e[n];o&&o.focus();},h=function(t){t[t.length-1].addEventListener("keydown",m);},w=function(t){t[0].addEventListener("keydown",b);},y=function(){var t=i.getNode(c),e=t.querySelectorAll("."+l);e.length&&(h(e),w(e));},x=function(t){if(i.getNode(u)===t.target)return r.onAction(a.CANCEL_KEY)},_=function(t){var e=i.getNode(u);e.removeEventListener("click",x),t&&e.addEventListener("click",x);},k=function(t){o.default.timer&&clearTimeout(o.default.timer),t&&(o.default.timer=window.setTimeout(function(){return r.onAction(a.CANCEL_KEY)},t));},O=function(t){t.closeOnEsc?document.addEventListener("keyup",p):document.removeEventListener("keyup",p),t.dangerMode?v():g(),y(),_(t.closeOnClickOutside),k(t.timer);};e.default=O;},function(t,e,n){Object.defineProperty(e,"__esModule",{value:!0});var o=n(1),r=n(3),i=n(37),a=n(38),s={title:null,text:null,icon:null,buttons:r.defaultButtonList,content:null,className:null,closeOnClickOutside:!0,closeOnEsc:!0,dangerMode:!1,timer:null},c=Object.assign({},s);e.setDefaults=function(t){c=Object.assign({},s,t);};var l=function(t){var e=t&&t.button,n=t&&t.buttons;return void 0!==e&&void 0!==n&&o.throwErr("Cannot set both 'button' and 'buttons' options!"),void 0!==e?{confirm:e}:n},u=function(t){return o.ordinalSuffixOf(t+1)},f=function(t,e){o.throwErr(u(e)+" argument ('"+t+"') is invalid");},d=function(t,e){var n=t+1,r=e[n];o.isPlainObject(r)||void 0===r||o.throwErr("Expected "+u(n)+" argument ('"+r+"') to be a plain object");},p=function(t,e){var n=t+1,r=e[n];void 0!==r&&o.throwErr("Unexpected "+u(n)+" argument ("+r+")");},m=function(t,e,n,r){var i=typeof e,a="string"===i,s=e instanceof Element;if(a){if(0===n)return {text:e};if(1===n)return {text:e,title:r[0]};if(2===n)return d(n,r),{icon:e};f(e,n);}else {if(s&&0===n)return d(n,r),{content:e};if(o.isPlainObject(e))return p(n,r),e;f(e,n);}};e.getOpts=function(){for(var t=[],e=0;e<arguments.length;e++)t[e]=arguments[e];var n={};t.forEach(function(e,o){var r=m(0,e,o,t);Object.assign(n,r);});var o=l(n);n.buttons=r.getButtonListOpts(o),delete n.button,n.content=i.getContentOpts(n.content);var u=Object.assign({},s,c,n);return Object.keys(u).forEach(function(t){a.DEPRECATED_OPTS[t]&&a.logDeprecation(t);}),u};},function(t,e,n){Object.defineProperty(e,"__esModule",{value:!0});var o=n(1),r={element:"input",attributes:{placeholder:""}};e.getContentOpts=function(t){var e={};return o.isPlainObject(t)?Object.assign(e,t):t instanceof Element?{element:t}:"input"===t?r:null};},function(t,e,n){Object.defineProperty(e,"__esModule",{value:!0}),e.logDeprecation=function(t){var n=e.DEPRECATED_OPTS[t],o=n.onlyRename,r=n.replacement,i=n.subOption,a=n.link,s=o?"renamed":"deprecated",c='SweetAlert warning: "'+t+'" option has been '+s+".";if(r){c+=" Please use"+(i?' "'+i+'" in ':" ")+'"'+r+'" instead.';}var l="https://sweetalert.js.org";c+=a?" More details: "+l+a:" More details: "+l+"/guides/#upgrading-from-1x",console.warn(c);},e.DEPRECATED_OPTS={type:{replacement:"icon",link:"/docs/#icon"},imageUrl:{replacement:"icon",link:"/docs/#icon"},customClass:{replacement:"className",onlyRename:!0,link:"/docs/#classname"},imageSize:{},showCancelButton:{replacement:"buttons",link:"/docs/#buttons"},showConfirmButton:{replacement:"button",link:"/docs/#button"},confirmButtonText:{replacement:"button",link:"/docs/#button"},confirmButtonColor:{},cancelButtonText:{replacement:"buttons",link:"/docs/#buttons"},closeOnConfirm:{replacement:"button",subOption:"closeModal",link:"/docs/#button"},closeOnCancel:{replacement:"buttons",subOption:"closeModal",link:"/docs/#buttons"},showLoaderOnConfirm:{replacement:"buttons"},animation:{},inputType:{replacement:"content",link:"/docs/#content"},inputValue:{replacement:"content",link:"/docs/#content"},inputPlaceholder:{replacement:"content",link:"/docs/#content"},html:{replacement:"content",link:"/docs/#content"},allowEscapeKey:{replacement:"closeOnEsc",onlyRename:!0,link:"/docs/#closeonesc"},allowClickOutside:{replacement:"closeOnClickOutside",onlyRename:!0,link:"/docs/#closeonclickoutside"}};}])});
    });

    var swal = unwrapExports(sweetalert_min);
    var sweetalert_min_1 = sweetalert_min.swal;

    let HacsRepositoryButton = /** @class */ (() => {
        class HacsRepositoryButton extends LitElement {
            static get styles() {
                return HacsStyle;
            }
        }
        __decorate([
            property()
        ], HacsRepositoryButton.prototype, "hass", void 0);
        __decorate([
            property()
        ], HacsRepositoryButton.prototype, "repository", void 0);
        __decorate([
            property()
        ], HacsRepositoryButton.prototype, "status", void 0);
        return HacsRepositoryButton;
    })();

    let HacsButtonAddToLovelace = /** @class */ (() => {
        let HacsButtonAddToLovelace = class HacsButtonAddToLovelace extends HacsRepositoryButton {
            constructor() {
                super(...arguments);
                this.logger = new Logger("add_to_lovelace");
            }
            render() {
                if (!this.repository.installed)
                    return html ``;
                return html `
      <mwc-button @click=${this.RepositoryAddToLovelace}>
        ${localize("repository.add_to_lovelace")}
      </mwc-button>
    `;
            }
            async RepositoryAddToLovelace() {
                const value = await swal(localize("confirm.add_to_lovelace"), {
                    buttons: [localize("confirm.no"), localize("confirm.yes")],
                });
                if (value !== null) {
                    this.hass.callWS({
                        type: "lovelace/resources/create",
                        res_type: "module",
                        url: `/hacsfiles/${this.repository.full_name.split("/")[1]}/${this.repository.file_name}`,
                    });
                    this.dispatchEvent(new CustomEvent("hacs-force-reload", { bubbles: true, composed: true }));
                }
            }
        };
        HacsButtonAddToLovelace = __decorate([
            customElement("hacs-button-add-to-lovelace")
        ], HacsButtonAddToLovelace);
        return HacsButtonAddToLovelace;
    })();

    let HacsButtonChangelog = /** @class */ (() => {
        let HacsButtonChangelog = class HacsButtonChangelog extends HacsRepositoryButton {
            render() {
                if (!this.repository.pending_upgrade)
                    return html ``;
                var URL = `https://github.com/${this.repository.full_name}/releases`;
                if (this.repository.version_or_commit === "commit") {
                    URL = `https://github.com/${this.repository.full_name}/compare/${this.repository.installed_version}...${this.repository.available_version}`;
                }
                return html `
      <hacs-link .url="${String(URL)}">
        <mwc-button>
          ${localize(`repository.changelog`)}
        </mwc-button>
      </hacs-link>
    `;
            }
            RepositoryInstall() {
                RepositoryWebSocketAction(this.hass, this.repository.id, "set_state", "installing");
                RepositoryWebSocketAction(this.hass, this.repository.id, "uninstall");
            }
        };
        HacsButtonChangelog = __decorate([
            customElement("hacs-button-changelog")
        ], HacsButtonChangelog);
        return HacsButtonChangelog;
    })();

    let HacsButtonClearNew = /** @class */ (() => {
        let HacsButtonClearNew = class HacsButtonClearNew extends HacsRepositoryButton {
            render() {
                return html `
      <mwc-button @click=${this.ExecuteAction}>
        ${localize(`store.clear_new`)}
      </mwc-button>
    `;
            }
            ExecuteAction() {
                var message = {
                    type: "hacs/settings",
                    action: "clear_new",
                    category: this.category,
                };
                this.hass.connection.sendMessage(message);
            }
        };
        __decorate([
            property()
        ], HacsButtonClearNew.prototype, "category", void 0);
        HacsButtonClearNew = __decorate([
            customElement("hacs-button-clear-new")
        ], HacsButtonClearNew);
        return HacsButtonClearNew;
    })();

    let HacsButtonGoToIntegrations = /** @class */ (() => {
        let HacsButtonGoToIntegrations = class HacsButtonGoToIntegrations extends HacsRepositoryButton {
            render() {
                return html `
      <mwc-button @click=${this.GoToIntegrations}>
        ${this.hacs.localize("repository.goto_integrations")}
      </mwc-button>
    `;
            }
            GoToIntegrations() {
                this.dispatchEvent(new CustomEvent("hacs-location-change", {
                    detail: { value: "/config/integrations/dashboard", force: true },
                    bubbles: true,
                    composed: true
                }));
            }
        };
        __decorate([
            property()
        ], HacsButtonGoToIntegrations.prototype, "hacs", void 0);
        HacsButtonGoToIntegrations = __decorate([
            customElement("hacs-button-goto-integrations")
        ], HacsButtonGoToIntegrations);
        return HacsButtonGoToIntegrations;
    })();

    let HacsButtonMainAction = /** @class */ (() => {
        let HacsButtonMainAction = class HacsButtonMainAction extends HacsRepositoryButton {
            constructor() {
                super(...arguments);
                this.pathExists = false;
                this.logger = new Logger("main_action");
            }
            updated() {
                let path = this.repository.local_path;
                this.hass.connection
                    .sendMessagePromise({
                    type: "hacs/check_path",
                    path: path,
                })
                    .then((resp) => {
                    this.pathExists = resp["exist"];
                }, (err) => {
                    this.logger.error("(hacs/check_path) Message failed!");
                    this.logger.error(err);
                });
            }
            render() {
                const label = localize(`repository.${this.repository.main_action.toLowerCase()}`);
                if (this.status.background_task) {
                    return html `
        <mwc-button
          class="disabled-button"
          title="${localize("confirm.bg_task")}"
          @click=${this.disabledAction}
        >
          ${label}
        </mwc-button>
      `;
                }
                if (!this.repository.can_install && this.repository.installed) {
                    return html `
        <mwc-button class="disabled-button" @click=${this.RepositoryInstall}>
          ${localize(`repository.upgrade`)}
        </mwc-button>
      `;
                }
                return html `
      <mwc-button @click=${this.RepositoryInstall}>
        ${this.repository.state == "installing"
                ? html ` <paper-spinner active></paper-spinner> `
                : html ` ${label} `}
      </mwc-button>
    `;
            }
            disabledAction() {
                swal(localize("confirm.bg_task"));
            }
            RepositoryInstall() {
                RepositoryWebSocketAction(this.hass, this.repository.id, "set_state", "installing");
                if (this.pathExists && !this.repository.installed) {
                    let path = this.repository.local_path;
                    if (this.repository.category === "python_script") {
                        path = `${path}/${this.repository.file_name}`;
                    }
                    swal(localize("confirm.exist", "{item}", path) +
                        "\n" +
                        localize("confirm.overwrite") +
                        "\n" +
                        localize("confirm.continue"), {
                        buttons: [localize("confirm.no"), localize("confirm.yes")],
                    }).then((value) => {
                        if (value !== null) {
                            this.ExecuteAction();
                        }
                        else {
                            RepositoryWebSocketAction(this.hass, this.repository.id, "set_state", "");
                        }
                    });
                }
                else if (!this.repository.can_install) {
                    swal(localize("confirm.home_assistant_version_not_correct")
                        .replace("{haversion}", this.hass.config.version)
                        .replace("{minversion}", this.repository.homeassistant));
                    RepositoryWebSocketAction(this.hass, this.repository.id, "set_state", "");
                }
                else
                    this.ExecuteAction();
            }
            ExecuteAction() {
                RepositoryWebSocketAction(this.hass, this.repository.id, "install");
            }
        };
        __decorate([
            property()
        ], HacsButtonMainAction.prototype, "pathExists", void 0);
        HacsButtonMainAction = __decorate([
            customElement("hacs-button-main-action")
        ], HacsButtonMainAction);
        return HacsButtonMainAction;
    })();

    let HacsButtonOpenPlugin = /** @class */ (() => {
        let HacsButtonOpenPlugin = class HacsButtonOpenPlugin extends HacsRepositoryButton {
            render() {
                if (this.repository.category != "plugin")
                    return html ``;
                if (!this.repository.installed)
                    return html ``;
                const path = this.repository.local_path.split("/");
                return html `
      <a
        href="/hacsfiles/${path.pop()}/${this.repository.file_name}"
        target="_blank"
      >
        <mwc-button>
          ${localize(`repository.open_plugin`)}
        </mwc-button>
      </a>
    `;
            }
        };
        HacsButtonOpenPlugin = __decorate([
            customElement("hacs-button-open-plugin")
        ], HacsButtonOpenPlugin);
        return HacsButtonOpenPlugin;
    })();

    let HacsButtonOpenRepository = /** @class */ (() => {
        let HacsButtonOpenRepository = class HacsButtonOpenRepository extends HacsRepositoryButton {
            render() {
                return html `
      <a
        href="https://github.com/${this.repository.full_name}"
        rel="noreferrer"
        target="_blank"
      >
        <mwc-button>
          ${localize(`repository.repository`)}
        </mwc-button>
      </a>
    `;
            }
        };
        HacsButtonOpenRepository = __decorate([
            customElement("hacs-button-open-repository")
        ], HacsButtonOpenRepository);
        return HacsButtonOpenRepository;
    })();

    let HacsButtonRestartHomeAssistant = /** @class */ (() => {
        let HacsButtonRestartHomeAssistant = class HacsButtonRestartHomeAssistant extends HacsRepositoryButton {
            render() {
                return html `
      <mwc-button @click=${this.RestartHomeAssistant}>
        ${this.hacs.localize("repository.restart_home_assistant")}
      </mwc-button>
    `;
            }
            static get styles() {
                return [
                    css `
        mwc-button {
          --mdc-theme-primary: var(--google-red-500);
        }
      `
                ];
            }
            RestartHomeAssistant() {
                swal(this.hacs.localize("confirm.restart_home_assistant"), {
                    buttons: [
                        this.hacs.localize("confirm.no"),
                        this.hacs.localize("confirm.yes")
                    ]
                }).then(value => {
                    if (value !== null) {
                        this.hass.callService("homeassistant", "restart");
                        swal(this.hacs.localize("confirm.home_assistant_is_restarting"));
                    }
                });
            }
        };
        __decorate([
            property()
        ], HacsButtonRestartHomeAssistant.prototype, "hacs", void 0);
        __decorate([
            property()
        ], HacsButtonRestartHomeAssistant.prototype, "route", void 0);
        HacsButtonRestartHomeAssistant = __decorate([
            customElement("hacs-button-restart-home-assistant")
        ], HacsButtonRestartHomeAssistant);
        return HacsButtonRestartHomeAssistant;
    })();

    let HacsButtonUninstall = /** @class */ (() => {
        let HacsButtonUninstall = class HacsButtonUninstall extends HacsRepositoryButton {
            render() {
                if (!this.repository.installed)
                    return html ``;
                const label = localize("repository.uninstall");
                if (this.status.background_task) {
                    return html `
        <mwc-button
          class="disabled-button"
          title="${localize("confirm.bg_task")}"
          @click=${this.disabledAction}
        >
          ${label}
        </mwc-button>
      `;
                }
                return html `
      <mwc-button @click=${this.RepositoryUnInstall}>
        ${this.repository.state == "uninstalling"
                ? html ` <paper-spinner active></paper-spinner> `
                : html ` ${label} `}
      </mwc-button>
    `;
            }
            static get styles() {
                return [
                    HacsStyle,
                    css `
        mwc-button {
          --mdc-theme-primary: var(--google-red-500);
        }
      `,
                ];
            }
            disabledAction() {
                swal(localize("confirm.bg_task"));
            }
            async RepositoryUnInstall() {
                const value = await swal(localize("confirm.uninstall", "{item}", this.repository.name), {
                    buttons: [localize("confirm.no"), localize("confirm.yes")],
                });
                if (value !== null) {
                    await this.ExecuteAction();
                }
            }
            async ExecuteAction() {
                RepositoryWebSocketAction(this.hass, this.repository.id, "set_state", "uninstalling");
                if (this.repository.category === "plugin" &&
                    this.status.lovelace_mode === "storage") {
                    await this.RepositoryRemoveFromLovelace();
                }
                RepositoryWebSocketAction(this.hass, this.repository.id, "uninstall");
            }
            async RepositoryRemoveFromLovelace() {
                const url1 = `/community_plugin/${this.repository.full_name.split("/")[1]}/${this.repository.file_name}`;
                const url2 = `/hacsfiles/${this.repository.full_name.split("/")[1]}/${this.repository.file_name}`;
                const resources = await this.hass.callWS({
                    type: "lovelace/resources",
                });
                const resource = resources.find(function (element) {
                    return element.url === url1 || element.url === url2;
                });
                if (resource) {
                    this.hass.callWS({
                        type: "lovelace/resources/delete",
                        resource_id: resource.id,
                    });
                }
            }
        };
        HacsButtonUninstall = __decorate([
            customElement("hacs-button-uninstall")
        ], HacsButtonUninstall);
        return HacsButtonUninstall;
    })();

    let HacsRepositoryBanner = /** @class */ (() => {
        let HacsRepositoryBanner = class HacsRepositoryBanner extends LitElement {
            static get styles() {
                return [
                    HacsStyle,
                    css `
        ha-card {
          width: 90%;
          margin-left: 5%;
        }
        ha-card.alert {
          background-color: var(
            --hacs-status-pending-restart,
            var(--google-red-500)
          );
          color: var(--text-primary-color);
        }
        ha-card.warning {
          background-color: var(--hacs-status-pending-update);
          color: var(--primary-text-color);
        }
        ha-card.info {
          background-color: var(--primary-background-color);
          color: var(--primary-text-color);
        }
      `
                ];
            }
        };
        __decorate([
            property()
        ], HacsRepositoryBanner.prototype, "hacs", void 0);
        __decorate([
            property()
        ], HacsRepositoryBanner.prototype, "configuration", void 0);
        __decorate([
            property()
        ], HacsRepositoryBanner.prototype, "hass", void 0);
        __decorate([
            property()
        ], HacsRepositoryBanner.prototype, "lovelaceconfig", void 0);
        __decorate([
            property()
        ], HacsRepositoryBanner.prototype, "repository", void 0);
        __decorate([
            property()
        ], HacsRepositoryBanner.prototype, "route", void 0);
        __decorate([
            property()
        ], HacsRepositoryBanner.prototype, "status", void 0);
        HacsRepositoryBanner = __decorate([
            customElement("hacs-repository-banner")
        ], HacsRepositoryBanner);
        return HacsRepositoryBanner;
    })();

    let RepositoryBannerIntegrationFirstInstall = /** @class */ (() => {
        let RepositoryBannerIntegrationFirstInstall = class RepositoryBannerIntegrationFirstInstall extends HacsRepositoryBanner {
            render() {
                const title = this.hacs.localize("repository_banner.config_flow_title");
                return html `
      <ha-card class="info" .header="${title}">
        <div class="card-content">
          ${this.hacs.localize("repository_banner.config_flow")}
          </br>
          ${this.hacs.localize("repository_banner.no_restart_required")} 🎉
        </div>
        <div class="card-actions">
          <hacs-button-goto-integrations
            .hacs=${this.hacs}
            .route=${this.route}
          ></hacs-button-goto-integrations>
        </div>
      </ha-card>
    `;
            }
        };
        RepositoryBannerIntegrationFirstInstall = __decorate([
            customElement("hacs-repository-banner-integration-first-install")
        ], RepositoryBannerIntegrationFirstInstall);
        return RepositoryBannerIntegrationFirstInstall;
    })();

    let RepositoryBannerIntegrationPendingRestart = /** @class */ (() => {
        let RepositoryBannerIntegrationPendingRestart = class RepositoryBannerIntegrationPendingRestart extends HacsRepositoryBanner {
            render() {
                const title = this.hacs.localize("repository_banner.restart_pending");
                return html `
      <ha-card class="info" .header="${title}">
        <div class="card-content">
          ${this.hacs.localize("repository_banner.restart")}
        </div>
        <div class="card-actions">
          <hacs-button-restart-home-assistant
            .hacs=${this.hacs}
            .hass=${this.hass}
            .repository=${this.repository}
            .route=${this.route}
          ></hacs-button-restart-home-assistant>
        </div>
      </ha-card>
    `;
            }
        };
        RepositoryBannerIntegrationPendingRestart = __decorate([
            customElement("hacs-repository-banner-integration-pending-restart")
        ], RepositoryBannerIntegrationPendingRestart);
        return RepositoryBannerIntegrationPendingRestart;
    })();

    let HacsLegacyUrlForPlugins = /** @class */ (() => {
        let HacsLegacyUrlForPlugins = class HacsLegacyUrlForPlugins extends HacsRepositoryBanner {
            constructor() {
                super(...arguments);
                this._wrongURL = [];
            }
            render() {
                if (this.status.lovelace_mode === "yaml")
                    return html ``;
                const title = "Legacy URL's detected";
                const resources = this.lovelaceconfig;
                if (!resources)
                    return html ``;
                this._wrongURL = resources === null || resources === void 0 ? void 0 : resources.filter((resource) => {
                    return resource.url.startsWith("/community_plugin");
                });
                if (this._wrongURL.length === 0)
                    return html ``;
                return html `
      <ha-card class="warning" .header="🚨 ${title}">
        <div class="card-content">
          You have plugin resources in your lovelace configuration that still
          uses the old "/community_plugin" URL and not "/hacsfiles"
        </div>
        <div class="card-actions">
          <mwc-button @click=${this.UpdateResources}>
            Update resources
          </mwc-button>
        </div>
      </ha-card>
    `;
            }
            async UpdateResources() {
                this._wrongURL.forEach((resource) => {
                    const url = resource.url.replace("/community_plugin", "/hacsfiles");
                    this.hass.callWS({
                        type: "lovelace/resources/update",
                        resource_id: resource.id,
                        url,
                    });
                });
                this.dispatchEvent(new CustomEvent("hacs-force-reload", { bubbles: true, composed: true }));
            }
        };
        __decorate([
            property()
        ], HacsLegacyUrlForPlugins.prototype, "_wrongURL", void 0);
        HacsLegacyUrlForPlugins = __decorate([
            customElement("hacs-legacy-url-for-plugins")
        ], HacsLegacyUrlForPlugins);
        return HacsLegacyUrlForPlugins;
    })();

    let RepositoryBannerPluginNotLoaded = /** @class */ (() => {
        let RepositoryBannerPluginNotLoaded = class RepositoryBannerPluginNotLoaded extends HacsRepositoryBanner {
            render() {
                const title = this.hacs.localize("repository_banner.not_loaded");
                const loaded = AddedToLovelace(this.repository, this.lovelaceconfig, this.status);
                if (loaded)
                    return html ``;
                return html `
      <ha-card class="info" .header="${title}">
        <div class="card-content">
          ${this.hacs.localize("repository_banner.plugin_not_loaded")}
        </div>
        <div class="card-actions">
          <hacs-button-add-to-lovelace
            .hacs=${this.hacs}
            .hass=${this.hass}
            .repository=${this.repository}
            .route=${this.route}
          ></hacs-button-add-to-lovelace>
        </div>
      </ha-card>
    `;
            }
        };
        RepositoryBannerPluginNotLoaded = __decorate([
            customElement("hacs-repository-banner-plugin-not-loaded")
        ], RepositoryBannerPluginNotLoaded);
        return RepositoryBannerPluginNotLoaded;
    })();

    let HacsBody = /** @class */ (() => {
        let HacsBody = class HacsBody extends LitElement {
            render() {
                return html `
      <div class="hacs-body">
        <slot></slot>
      </div>
    `;
            }
            static get styles() {
                return [
                    HacsStyle,
                    css `
        .hacs-body {
          width: 95%;
          margin-left: 2.5%;
          margin-top: 2.5%;
          margin-bottom: 2.5%;
        }
      `
                ];
            }
        };
        HacsBody = __decorate([
            customElement("hacs-body")
        ], HacsBody);
        return HacsBody;
    })();

    let HacsLink = /** @class */ (() => {
        let HacsLink = class HacsLink extends LitElement {
            constructor() {
                super(...arguments);
                this.newtab = false;
            }
            render() {
                const external = this.url.includes("http");
                if (external) {
                    return html `
        <a href="${this.url}" target="_blank" rel="noreferrer">
          <slot></slot>
        </a>
      `;
                }
                if (this.newtab) {
                    return html `
        <a href="${this.url}" target="_blank">
          <slot></slot>
        </a>
      `;
                }
                return html `
      <a href="${this.url}" target="_top">
        <slot></slot>
      </a>
    `;
            }
            static get styles() {
                return [
                    css `
        a {
          color: var(--hacs-link-text-color, var(--accent-color));
          text-decoration: var(--hacs-link-text-decoration, none);
        }
      `,
                ];
            }
        };
        __decorate([
            property({ type: Boolean })
        ], HacsLink.prototype, "newtab", void 0);
        __decorate([
            property()
        ], HacsLink.prototype, "url", void 0);
        HacsLink = __decorate([
            customElement("hacs-link")
        ], HacsLink);
        return HacsLink;
    })();

    let HacsProgressbar = /** @class */ (() => {
        let HacsProgressbar = class HacsProgressbar extends LitElement {
            constructor() {
                super(...arguments);
                this.active = true;
            }
            render() {
                if (!this.active)
                    return html ``;
                return html `
      <paper-progress indeterminate></paper-progress>
    `;
            }
            static get styles() {
                return [
                    HacsStyle,
                    css `
        paper-progress {
          width: 100%;
          --paper-progress-active-color: var(--accent-color);
        }
      `
                ];
            }
        };
        __decorate([
            property()
        ], HacsProgressbar.prototype, "active", void 0);
        HacsProgressbar = __decorate([
            customElement("hacs-progressbar")
        ], HacsProgressbar);
        return HacsProgressbar;
    })();

    let Spinner = /** @class */ (() => {
        let Spinner = class Spinner extends LitElement {
            constructor() {
                super(...arguments);
                this.primary = "#faa179";
                this.secondary = "black";
                this.active = false;
            }
            render() {
                if (!this.active)
                    return html ``;
                return html `
          <svg class="spin spinner13" viewBox="0 0 66 66" xmlns="http://www.w3.org/2000/svg">
            <circle class="stroke stroke30" fill="transparent" stroke-width="1" cx="33" cy="33" r="30" stroke="red"/>
            </circle>
          </svg>
          <svg class="spin reversespinner13" viewBox="0 0 66 66" xmlns="http://www.w3.org/2000/svg">
            <circle class="stroke stroke25" fill="transparent" stroke-width="1" cx="33" cy="33" r="25" stroke="blue"/>
            </circle>
          </svg>
          <svg class="spin spinner14" viewBox="0 0 66 66" xmlns="http://www.w3.org/2000/svg">
            <circle class="stroke stroke20" fill="transparent" stroke-width="1" cx="33" cy="33" r="20" stroke="red"/>
            </circle>
          </svg>
          <svg class="spin reversespinner14" viewBox="0 0 66 66" xmlns="http://www.w3.org/2000/svg">
            <circle class="stroke stroke15" fill="transparent" stroke-width="1" cx="33" cy="33" r="15" stroke="blue"/>
            </circle>
          </svg>
          <svg class="spin spinner15" viewBox="0 0 66 66" xmlns="http://www.w3.org/2000/svg">
            <circle class="stroke stroke10" fill="transparent" stroke-width="1" cx="33" cy="33" r="10" stroke="red"/>
            </circle>
          </svg>
          <svg class="spin reversespinner15" viewBox="0 0 66 66" xmlns="http://www.w3.org/2000/svg">
            <circle class="stroke stroke5" fill="transparent" stroke-width="1" cx="33" cy="33" r="5" stroke="blue"/>
            </circle>
          </svg>
    `;
            }
            static get styles() {
                return [
                    css `
        .stroke {
          stroke-dasharray: 200;
        }
        .stroke30 {
          stroke-dashoffset: 50;
        }
        .stroke25 {
          stroke-dashoffset: 75;
        }
        .stroke20 {
          stroke-dashoffset: 100;
        }
        .stroke15 {
          stroke-dashoffset: 125;
        }
        .stroke10 {
          stroke-dashoffset: 150;
        }
        .stroke5 {
          stroke-dashoffset: 175;
        }

        .spin {
          position: absolute;
          z-index: 1337;
          width: auto;
        }

        .spinner13 {
          animation: rotate 1.3s linear infinite;
          -webkit-animation: rotate 1.3s linear infinite;
          -moz-animation: rotate 1.3s linear infinite;
        }

        .reversespinner13 {
          animation: reverserotate 1.3s linear infinite;
          -webkit-animation: reverserotate 1.3s linear infinite;
          -moz-animation: reverserotate 1.3s linear infinite;
        }

        .spinner14 {
          animation: rotate 1.4s linear infinite;
          -webkit-animation: rotate 1.4s linear infinite;
          -moz-animation: rotate 1.4s linear infinite;
        }

        .reversespinner14 {
          animation: reverserotate 1.4s linear infinite;
          -webkit-animation: reverserotate 1.4s linear infinite;
          -moz-animation: reverserotate 1.4s linear infinite;
        }

        .spinner15 {
          animation: rotate 1.5s linear infinite;
          -webkit-animation: rotate 1.5s linear infinite;
          -moz-animation: rotate 1.5s linear infinite;
        }

        .reversespinner15 {
          animation: reverserotate 1.5s linear infinite;
          -webkit-animation: reverserotate 1.5s linear infinite;
          -moz-animation: reverserotate 1.5s linear infinite;
        }

        @keyframes rotate {
          to {
            transform: rotate(360deg);
          }
        }

        @-webkit-keyframes rotate {
          to {
            -webkit-transform: rotate(360deg);
          }
        }

        @-moz-keyframes rotate {
          to {
            transform: rotate(360deg);
          }
        }

        @keyframes reverserotate {
          to {
            transform: rotate(-360deg);
          }
        }

        @-webkit-keyframes reverserotate {
          to {
            -webkit-transform: rotate(-360deg);
          }
        }

        @-moz-keyframes reverserotate {
          to {
            transform: rotate(-360deg);
          }
        }
      `
                ];
            }
        };
        __decorate([
            property()
        ], Spinner.prototype, "primary", void 0);
        __decorate([
            property()
        ], Spinner.prototype, "secondary", void 0);
        __decorate([
            property()
        ], Spinner.prototype, "active", void 0);
        Spinner = __decorate([
            customElement("multi-spinner")
        ], Spinner);
        return Spinner;
    })();

    function AboutHacs(hacs, repositories, configuration) {
        const content = document.createElement("div");
        content.style.color = `var(--primary-text-color)`;
        content.innerHTML = hacs.markdown(`
### HACS (Home Assistant Community Store)

**Integration ${hacs.localize("common.version").toLowerCase()}:**| ${configuration.version}
-- | --
**Frontend ${hacs
        .localize("common.version")
        .toLowerCase()}:** | ${version$1}
**${hacs.localize("common.repositories")}:** | ${repositories.length || 0}
  `);
        return content;
    }

    let HacsMenu = /** @class */ (() => {
        let HacsMenu = class HacsMenu extends LitElement {
            render() {
                return html `
      <paper-menu-button
        horizontal-align="right"
        role="group"
        aria-haspopup="true"
        vertical-align="top"
        aria-disabled="false"
      >
        <ha-icon-button
          icon="mdi:dots-vertical"
          slot="dropdown-trigger"
          role="button"
        ></ha-icon-button>
        <paper-listbox
          slot="dropdown-content"
          role="listbox"
          tabindex="0"
          dir="rtl"
        >
          <paper-item @click=${this.openHelp}
            >${this.hacs.localize("common.documentation")}
          </paper-item>

          <paper-item @click=${this.ReloadWindow}>
            ${this.hacs.localize("settings.reload_window")}
          </paper-item>

          ${this.location.includes("installed")
                ? html `
                <paper-item @click=${this.UpgradeAll} class="uninstall-button">
                  ${this.hacs.localize("settings.upgrade_all")}
                </paper-item>
              `
                : ""}

          <hacs-link url="https://github.com/hacs">
            <paper-item>
              ${this.hacs.localize(`settings.hacs_repo`)}
            </paper-item>
          </hacs-link>

          <hacs-link url="https://hacs.xyz/docs/issues">
            <paper-item>
              ${this.hacs.localize(`repository.open_issue`)}
            </paper-item>
          </hacs-link>
          ${!this.location.includes("settings") &&
                !this.location.includes("repository")
                ? html `
                <paper-item>
                  <ha-switch
                    .checked=${this.configuration.frontend_mode === "Table"}
                    @change=${this.SetFeStyle}
                    ><div class="switch-text">Table</div></ha-switch
                  >
                </paper-item>
                <paper-item>
                  <ha-switch
                    .checked=${this.configuration.frontend_compact}
                    @change=${this.SetFeCompact}
                    ><div class="switch-text">Compact</div>
                  </ha-switch>
                </paper-item>
              `
                : ""}

          <paper-item @click=${this.openAbout}
            >${this.hacs.localize("common.about")}
          </paper-item>
        </paper-listbox>
      </paper-menu-button>
    `;
            }
            disabledAction() {
                swal(this.hacs.localize("confirm.bg_task"));
            }
            SetFeStyle() {
                this.hass.connection.sendMessage({
                    type: "hacs/settings",
                    action: `set_fe_${this.configuration.frontend_mode !== "Table" ? "table" : "grid"}`,
                });
            }
            SetFeCompact() {
                this.hass.connection.sendMessage({
                    type: "hacs/settings",
                    action: `set_fe_compact_${String(this.configuration.frontend_compact).toLocaleLowerCase()}`,
                });
            }
            ReloadWindow() {
                this.dispatchEvent(new CustomEvent("hacs-force-reload", {
                    detail: { force: true },
                    bubbles: true,
                    composed: true,
                }));
            }
            ReloadData() {
                if (this.status.background_task) {
                    this.disabledAction();
                    return;
                }
                swal(`${this.hacs.localize(`confirm.reload_data`)}\n${this.hacs.localize("confirm.continue")}`, {
                    buttons: [
                        this.hacs.localize("confirm.no"),
                        this.hacs.localize("confirm.yes"),
                    ],
                }).then((value) => {
                    if (value !== null) {
                        this.hass.connection.sendMessage({
                            type: "hacs/settings",
                            action: "reload_data",
                        });
                    }
                });
            }
            UpgradeAll() {
                if (this.status.background_task) {
                    this.disabledAction();
                    return;
                }
                var elements = [];
                this.repositories.forEach((element) => {
                    if (element.pending_upgrade)
                        elements.push(element);
                });
                if (elements.length > 0) {
                    var msg = this.hacs.localize(`confirm.upgrade_all`) + "\n\n";
                    elements.forEach((element) => {
                        msg += `${element.name}: ${element.installed_version} -> ${element.available_version}\n`;
                    });
                    msg += `\n${this.hacs.localize("confirm.continue")}`;
                    swal(msg, {
                        buttons: [
                            this.hacs.localize("confirm.no"),
                            this.hacs.localize("confirm.yes"),
                        ],
                    }).then((value) => {
                        if (value !== null) {
                            this.hass.connection.sendMessage({
                                type: "hacs/settings",
                                action: "upgrade_all",
                            });
                        }
                    });
                }
                else {
                    swal(this.hacs.localize(`confirm.no_upgrades`), {
                        buttons: [this.hacs.localize("confirm.ok")],
                    });
                }
            }
            openAbout() {
                swal({
                    buttons: [false],
                    content: AboutHacs(this.hacs, this.repositories, this.configuration),
                });
            }
            openHelp() {
                const base = "https://hacs.xyz/docs/navigation/";
                const fallback = "https://hacs.xyz/docs/basic/getting_started/";
                var location = window.location.pathname.split("/")[2];
                if (this.hacs.isnullorempty(location)) {
                    window.open(fallback, "Documentation", "noreferrer");
                    return;
                }
                if (["integration", "plugin", "appdaemon", "python_script", "theme"].includes(location))
                    location = "stores";
                window.open(`${base}${location}`, "Documentation", "noreferrer");
            }
            static get styles() {
                return [
                    HacsStyle,
                    css `
        paper-listbox {
          top: 0px;
          overflow: hidden;
        }
        a {
          color: var(--primary-text-color);
        }
        ha-icon {
          cursor: pointer;
          float: right;
        }
        ha-switch {
          right: -24px;
        }
        .switch-text {
          margin-right: 8px;
        }
        .dropdown-content {
          overflow: hidden;
          right: 0;
          top: 0;
        }
        .extended {
          display: block !important;
        }
        paper-item {
          display: flex;
          font-size: 14px;
          background-color: var(
            --paper-listbox-background-color,
            var(--primary-background-color)
          );
        }
      `,
                ];
            }
        };
        __decorate([
            property()
        ], HacsMenu.prototype, "location", void 0);
        __decorate([
            property()
        ], HacsMenu.prototype, "hacs", void 0);
        __decorate([
            property()
        ], HacsMenu.prototype, "hass", void 0);
        __decorate([
            property()
        ], HacsMenu.prototype, "configuration", void 0);
        __decorate([
            property()
        ], HacsMenu.prototype, "status", void 0);
        __decorate([
            property()
        ], HacsMenu.prototype, "repositories", void 0);
        HacsMenu = __decorate([
            customElement("hacs-menu")
        ], HacsMenu);
        return HacsMenu;
    })();

    let CustomRepositories = /** @class */ (() => {
        let CustomRepositories = class CustomRepositories extends LitElement {
            constructor() {
                super(...arguments);
                this.background_task = true;
            }
            update(changedProperties) {
                super.update(changedProperties);
                if (this.hacs.configuration.debug)
                    this.hacs.logger.info(changedProperties);
            }
            shouldUpdate(changedProperties) {
                changedProperties.forEach((_oldValue, propName) => {
                    if (propName === "hacs") {
                        this.background_task = this.hacs.status.background_task;
                        const customrepositories = this.getCustomRepositories();
                        if (!this.custom || this.custom.length !== customrepositories.length) {
                            this.custom = customrepositories;
                        }
                    }
                });
                return (changedProperties.has("custom") ||
                    changedProperties.has("background_task"));
            }
            getCustomRepositories() {
                return this.hacs.repositories
                    .sort((a, b) => (a.full_name > b.full_name ? 1 : -1))
                    .filter((repo) => {
                    if (repo.custom)
                        return true;
                    else
                        return false;
                });
            }
            Delete(ev) {
                let RepoID;
                let RepoFullName;
                ev.composedPath().forEach((item) => {
                    if (item.RepoID) {
                        RepoID = item.RepoID;
                        RepoFullName = item.RepoFullName;
                    }
                });
                swal(this.hacs.localize("confirm.delete", "{item}", RepoFullName), {
                    buttons: [
                        this.hacs.localize("confirm.no"),
                        this.hacs.localize("confirm.yes"),
                    ],
                }).then((value) => {
                    if (!this.hacs.isnullorempty(value)) {
                        this.dispatchEvent(new CustomEvent("hacs-repository-action", {
                            detail: { repo: RepoID, action: "delete" },
                            bubbles: true,
                            composed: true,
                        }));
                    }
                });
            }
            DeleteInstalled(ev) {
                let RepoFullName;
                ev.composedPath().forEach((item) => {
                    if (item.RepoFullName) {
                        RepoFullName = item.RepoFullName;
                    }
                });
                swal(this.hacs.localize("confirm.delete_installed", "{item}", RepoFullName));
            }
            Save(ev) {
                const elems = ev.composedPath()[4].children;
                var selected = elems[1].selectedItem;
                if (selected === undefined) {
                    swal(this.hacs.localize("settings.missing_category"));
                    return;
                }
                var category = selected.category;
                var repo = elems[0].value;
                this.dispatchEvent(new CustomEvent("hacs-repository-action", {
                    detail: { repo: repo, action: "add", data: category },
                    bubbles: true,
                    composed: true,
                }));
                swal(this.hacs.localize("settings.adding_new_repo", "{repo}", repo) +
                    "\n" +
                    this.hacs.localize("settings.adding_new_repo_category", "{category}", category));
            }
            ShowRepository(ev) {
                let RepoID;
                ev.composedPath().forEach((item) => {
                    if (item.RepoID) {
                        RepoID = item.RepoID;
                    }
                });
                this.dispatchEvent(new CustomEvent("hacs-location-change", {
                    detail: { value: `repository/${RepoID}` },
                    bubbles: true,
                    composed: true,
                }));
            }
            render() {
                if (this.hacs.repositories === undefined) {
                    this.dispatchEvent(new CustomEvent("hacs-recreate", { bubbles: true, composed: true }));
                    return html ``;
                }
                return html `
      <ha-card header="${this.hacs.localize("settings.custom_repositories")}">
        <div class="card-content">
          <div class="custom-repositories-list">
            ${this.background_task
                ? html `
                  <i class="addition"
                    >${this.hacs.localize("settings.bg_task_custom")}</i
                  >
                `
                : html `
                  ${this.custom &&
                    this.custom.map((repo) => html `
                        <div
                          class="row"
                          .RepoID=${repo.id}
                          .RepoFullName=${repo.full_name}
                        >
                          <paper-item class="customlistitem">
                            <div
                              @click=${this.ShowRepository}
                              class="link flexy"
                              title="${this.hacs.localize("settings.open_repository")}"
                            >
                              <div class="MobileHide">
                                [${repo.category}]&nbsp;
                              </div>
                              ${repo.full_name}
                            </div>
                            ${repo.installed
                        ? html `
                                  <ha-icon
                                    title="${this.hacs.localize("settings.delete")}"
                                    class="listicon disabled"
                                    icon="mdi:delete"
                                    @click=${this.DeleteInstalled}
                                  ></ha-icon>
                                `
                        : html `
                                  <ha-icon
                                    title="${this.hacs.localize("settings.delete")}"
                                    class="listicon"
                                    icon="mdi:delete"
                                    @click=${this.Delete}
                                  ></ha-icon>
                                `}
                          </paper-item>
                        </div>
                      `)}
                `}
          </div>
        </div>
        ${this.hacs.status.background_task
                ? ""
                : html `
              <div class="card-actions">
                <paper-input
                  class="inputfield MobileGrid"
                  placeholder=${this.hacs.localize("settings.add_custom_repository")}
                  type="text"
                ></paper-input>
                <paper-dropdown-menu
                  class="category MobileGrid"
                  label="${this.hacs.localize(`settings.category`)}"
                >
                  <paper-listbox slot="dropdown-content" selected="-1">
                    ${this.hacs.configuration.categories.map((category) => html `
                        <paper-item class="categoryitem" .category=${category}>
                          ${this.hacs.localize(`common.${category}`)}
                        </paper-item>
                      `)}
                  </paper-listbox>
                </paper-dropdown-menu>

                <div class="save">
                  <ha-icon
                    title="${this.hacs.localize("settings.save")}"
                    icon="mdi:content-save"
                    class="saveicon MobileGrid"
                    @click=${this.Save}
                  >
                  </ha-icon>
                </div>
              </div>
            `}
      </ha-card>
    `;
            }
            static get styles() {
                return [
                    HacsStyle,
                    css `
        .inputfield {
          width: 60%;
        }
        .category {
          position: absolute;
          width: 30%;
          right: 54px;
          bottom: 5px;
        }
        .saveicon {
          color: var(--hacs-badge-color, --accent-color);
          position: absolute;
          right: 0;
          bottom: 24px;
          cursor: pointer;
        }
        .listicon {
          color: var(--hacs-badge-color, --accent-color);
          right: 0px;
          position: absolute;
          cursor: pointer;
        }
        .loading {
          position: absolute;
          right: 10px;
          bottom: 22px;
        }

        @media screen and (max-width: 600px) and (min-width: 0) {
          .saveicon {
            height: 64px;
          }
          .save {
            padding-bottom: 64px;
          }
        }
        paper-item.categoryitem {
          display: flex;
          background-color: var(
            --paper-listbox-background-color,
            var(--primary-background-color)
          );
        }

        paper-item.customlistitem {
          display: flex;
        }
      `,
                ];
            }
        };
        __decorate([
            property()
        ], CustomRepositories.prototype, "hacs", void 0);
        __decorate([
            property()
        ], CustomRepositories.prototype, "route", void 0);
        __decorate([
            property()
        ], CustomRepositories.prototype, "background_task", void 0);
        __decorate([
            property()
        ], CustomRepositories.prototype, "custom", void 0);
        CustomRepositories = __decorate([
            customElement("hacs-custom-repositories")
        ], CustomRepositories);
        return CustomRepositories;
    })();

    let HacsCritical = /** @class */ (() => {
        let HacsCritical = class HacsCritical extends LitElement {
            async Acknowledge(ev) {
                var repository = ev.composedPath()[3].repository;
                const resp = await this.hass.connection.sendMessagePromise({
                    type: "hacs/critical",
                    repository: repository,
                });
                this.critical = resp.data;
            }
            render() {
                if (this.critical === undefined)
                    return html ``;
                var _critical = [];
                this.critical.forEach((element) => {
                    if (!element.acknowledged)
                        _critical.push(element);
                });
                return html `
      ${_critical.map((repo) => html `
            <ha-card header="Critical Issue!" class="alert">
                <div class="card-content">
                    The repository "${repo.repository}" has been flagged as a critical repository.</br>
                    The repository has now been uninstalled and removed.</br>
                    For information about how and why these are handled, see
                    <hacs-link url="https://hacs.xyz/docs/developer/maintainer#critical-repositories">
                        https://hacs.xyz/docs/developer/maintainer#critical-repositories
                    </hacs-link></br>
                    As a result of this Home Assistant was also restarted.</br></br>

                    <b>Reason: </b>${repo.reason}
                </div>
                <div class="card-actions">
                    <mwc-button @click=${this.Acknowledge} .repository=${repo.repository}>
                        Acknowledge
                    </mwc-button>
                    <hacs-link .url="${repo.link}">
                        <mwc-button>
                            More information about this incident
                        </mwc-button>
                    </hacs-link>
                </div>
            </ha-card>`)}
    `;
            }
            static get styles() {
                return [
                    HacsStyle,
                    css `
        ha-card {
          width: 90%;
          margin-left: 5%;
        }
        .alert {
          background-color: var(
            --hacs-status-pending-restart,
            var(--google-red-500)
          );
          color: var(--text-primary-color);
        }
      `,
                ];
            }
        };
        __decorate([
            property()
        ], HacsCritical.prototype, "critical", void 0);
        __decorate([
            property()
        ], HacsCritical.prototype, "hass", void 0);
        HacsCritical = __decorate([
            customElement("hacs-critical")
        ], HacsCritical);
        return HacsCritical;
    })();

    let HacsError = /** @class */ (() => {
        let HacsError = class HacsError extends LitElement {
            constructor() {
                super(...arguments);
                this.error = undefined;
            }
            clearError() {
                this.error = undefined;
            }
            firstUpdated() {
                this.hass.connection.subscribeEvents(msg => (this.error = msg.data), "hacs/error");
            }
            render() {
                if (this.error === undefined)
                    return html ``;
                var message = this.error.message;
                var additional = "";
                if (this.error.action === "add_repository") {
                    additional =
                        "Could not add this repository, make sure it is compliant with HACS.";
                }
                return html `
            <ha-card header="An error occurred while processing" class="alert">
                <div class="card-content">
                  ${this.error.exception !== undefined
                ? html `
                          <b>Exception: </b> ${this.error.exception} </br>
                        `
                : ""}
                  ${this.error.exception !== undefined
                ? html `
                          <b>Message: </b>
                        `
                : ""}${this.error.message} </br>
                    ${additional}
                </div>
            <div class="card-actions">
                <mwc-button @click=${this.clearError}>
                    Acknowledge
                </mwc-button>
            ${this.error.action === "add_repository"
                ? html `
                    <a
                      href="https://hacs.xyz/docs/publish/start"
                      rel="noreferrer"
                      target="_blank"
                    >
                      <mwc-button>
                        Documentation
                      </mwc-button>
                    </a>
                  `
                : ""}
            </div>
            </ha-card>
            `;
            }
            static get styles() {
                return [
                    HacsStyle,
                    css `
        ha-card {
          width: 90%;
          margin-left: 5%;
        }
        .alert {
          background-color: var(
            --hacs-status-pending-restart,
            var(--google-red-500)
          );
          color: var(--text-primary-color);
        }
      `
                ];
            }
        };
        __decorate([
            property()
        ], HacsError.prototype, "hass", void 0);
        __decorate([
            property()
        ], HacsError.prototype, "error", void 0);
        HacsError = __decorate([
            customElement("hacs-error")
        ], HacsError);
        return HacsError;
    })();

    const step = html `
  <h1>
    Welcome to HACS!👋
  </h1>
  <hr />
  <p>
    Congratulations, you have successfully installed HACS! 🎉 </br> This onboarding
    process will hopefully help you to understand what HACS is and what it does.
  </p>
  <p>To start off here are a few useful links:</p>
  <li><hacs-link url="https://hacs.xyz/">HACS Documentation</hacs-link></li>
  <li><hacs-link url="https://github.com/hacs">HACS @ GitHub</hacs-link></li>
  <li><hacs-link url="https://discord.gg/apgchf8">HACS @ Discord</hacs-link></li>
  </br>
  <p><b>NB!: You will only see this onboarding once, so take your time to read it all.</b></p>
`;

    const step$1 = html `
  <h1>
    Introduction
  </h1>
  <hr />
  <p>
    HACS, in short, is really just a glorified downloader, but no one wants
    that so....
  </p>
  <p>
    HACS works as a distribution platform for extensions to Home Assistant that
    are not part of Home Assistant core.
  </p>
  <p>
    Extensions, or "categories", in this context are:
  </p>
  <li>Custom integrations (custom_components)</li>
  <li>
    <hacs-link url="https://www.home-assistant.io/lovelace"
      >Plugins (Lovelace elements)</hacs-link
    >
  </li>
  <li>
    <hacs-link url="https://appdaemon.readthedocs.io/en/latest/"
      >AppDaemon apps</hacs-link
    >
  </li>
  <li>
    <hacs-link url="https://www.home-assistant.io/integrations/python_script/"
      >python scripts</hacs-link
    >
  </li>
  <li>
    <hacs-link
      url="https://www.home-assistant.io/integrations/frontend/#defining-themes"
      >Themes</hacs-link
    >
  </li>
  <p>
    Having HACS as a distribution platform allows the developer(s)/author(s) of
    these extensions to reach a broader audience and allows for fast deployment
    of new versions.
  </p>
  <p>
  That's fine for developers/authors, but how about users?</br>
  The nice part for users is that most of the custom extensions for Home Assistant now are in HACS, which makes it super easy to find, install, and upgrade.
  </p>
`;

    const step$2 = html `
  <h1>
    Background tasks
  </h1>
  <hr />
  <p>
    If you see this progressbar it means that HACS is working with something in
    the background. Upgrade and uninstall are disabled while this is running.
  </p>
  <hacs-progressbar .active=${true}></hacs-progressbar>
  <p>This is just an example, you can click "Continue" now.</p>
`;

    const step$3 = html `
  <h1>
    First startup
  </h1>
  <hr />
  <p>
    The first time you start Home Assistant with HACS enabled it needs to get information about every repository it knows.</br> It is important that you let it finish this process before you restart Home Assistant, if you restart before it's finished it will need to start over.
  </p>
  <p>When it's done you will no longer see the progressbar.</p>
  <hacs-progressbar .active=${true}></hacs-progressbar>
  <p>This process can take several minutes depending on your system and network.</p>
  <p>Again this progressbar is just an example, you can click "Continue" now.</p>
`;

    const step$4 = html `
  <h1>
    Automation
  </h1>
  <hr />
  <p>
    Automation is at the core of Home Assistant, and parts of HACS may also be
    automated.
  </p>
  <p>Some examples on what can be automated:</p>
  <li>
    <hacs-link
      url="https://hacs.xyz/docs/basic/automation#new-repositories-added"
      >New repository was added to HACS</hacs-link
    >
  </li>
  <li>
    <hacs-link url="https://hacs.xyz/docs/basic/automation#updates-pending"
      >Update pending in HACS</hacs-link
    >
  </li>
`;

    const step$5 = html `
  <h1>
    🚨 Warning
  </h1>
  <hr />
  <p>
  Everything you find in HACS is <b>not</b> tested by Home Assistant, that includes HACS itself.
  </br></br>
  Custom extentions can cause harm to your system and/or contain bugs.</b>
  If you have issues with a custom extension you found in HACS, report it to the author of <i>that</i> custom extension.</b></br>
  You can easily get to the issue tracker of anything you find in HACS with the menu (located at the top right) on every repository you find inside HACS.
  </br></br>
  The HACS and Home Assistant teams do not support <i>anything</i> you find here.
  </p>
`;

    const step$6 = html `
  <h1>
    Last words
  </h1>
  <hr />
  <p>
    HACS is mainly developed/maintained by a single person (<hacs-link
      url="https://github.com/ludeeus"
      >@ludeeus</hacs-link
    >), because of this it can take some time before issues are looked at and
    pull requests are reviewed.
  </p>
  <p>
    While you have read through this onboarding section, HACS has started to get
    information about the repositories it knows, so you will already now see
    items in under the store pages.
  </p>
  <p>
    If you have issues/suggestions regarding HACS, report them here: 
    <hacs-link url="https://hacs.xyz/docs/issues">GitHub</hacs-link>
  </p>
  <p>
    If you have questions regarding HACS please join the
    <hacs-link url="https://discord.gg/apgchf8">Discord server</hacs-link>
  </p>

  </br>
  <p>If you find this custom integration useful please consider supporting me(<hacs-link
      url="https://github.com/ludeeus"
      >@ludeeus</hacs-link
    >)
    <hacs-link url="https://github.com/sponsors/ludeeus">by sponsoring me on GitHub❤️</hacs-link>
    or
    <hacs-link url="https://buymeacoffee.com/ludeeus">buying me ☕️/🍺</hacs-link>
    </p>
`;

    let HacsOnboarding = /** @class */ (() => {
        let HacsOnboarding = class HacsOnboarding extends LitElement {
            constructor() {
                super(...arguments);
                this.step = 0;
            }
            NextStep() {
                this.step += 1;
                if (this.step === 7) {
                    this.dispatchEvent(new CustomEvent("hacs-onboarding-done", {
                        bubbles: true,
                        composed: true
                    }));
                }
            }
            render() {
                return html `
      <app-header-layout has-scrolling-region>
        <app-header slot="header" fixed>
          <app-toolbar>
            <ha-menu-button
              .hass="${this.hass}"
              .narrow="${this.narrow}"
            ></ha-menu-button>
            <div main-title>
              Home Assistant Community Store
            </div>
          </app-toolbar>
        </app-header>
        <hacs-body>
          <ha-card>
            <div class="card-content">
              ${this.step === 0
                ? step
                : this.step === 1
                    ? step$1
                    : this.step === 2
                        ? step$2
                        : this.step === 3
                            ? step$3
                            : this.step === 4
                                ? step$4
                                : this.step === 5
                                    ? step$5
                                    : this.step === 6
                                        ? step$6
                                        : ""}
            </div>
            <div class="card-actions">
              <mwc-button
                raised
                title="${this.hacs.localize("common.continue")}"
                @click=${this.NextStep}
              >
                ${this.hacs.localize("common.continue")}
              </mwc-button>
            </div>
          </ha-card>
        </hacs-body>
      </app-header-layout>
    `;
            }
            static get styles() {
                return [
                    HacsStyle,
                    css `
        mwc-button {
          display: grid;
        }
      `
                ];
            }
        };
        __decorate([
            property()
        ], HacsOnboarding.prototype, "narrow", void 0);
        __decorate([
            property()
        ], HacsOnboarding.prototype, "hacs", void 0);
        __decorate([
            property()
        ], HacsOnboarding.prototype, "hass", void 0);
        __decorate([
            property()
        ], HacsOnboarding.prototype, "step", void 0);
        HacsOnboarding = __decorate([
            customElement("hacs-onboarding")
        ], HacsOnboarding);
        return HacsOnboarding;
    })();

    let HacsRepositoryMenu = /** @class */ (() => {
        let HacsRepositoryMenu = class HacsRepositoryMenu extends LitElement {
            render() {
                return html `
      <paper-menu-button
        no-animations
        horizontal-align="right"
        role="group"
        aria-haspopup="true"
        vertical-align="top"
        aria-disabled="false"
      >
        <ha-icon-button
          icon="hass:dots-vertical"
          slot="dropdown-trigger"
          role="button"
        ></ha-icon-button>
        <paper-listbox
          slot="dropdown-content"
          role="listbox"
          tabindex="0"
          dir="rtl"
        >
          <paper-item @click=${this.RepositoryReload}>
            ${localize(`repository.update_information`)}
          </paper-item>

          ${this.repository.version_or_commit === "version"
                ? html `
                <paper-item @click=${this.RepositoryBeta}>
                  ${this.repository.beta
                    ? localize(`repository.hide_beta`)
                    : localize(`repository.show_beta`)}
                </paper-item>
              `
                : ""}
          ${!this.repository.custom && !this.repository.installed
                ? html `
                <paper-item @click=${this.RepositoryHide}>
                  ${localize(`repository.hide`)}
                </paper-item>
              `
                : ""}

          <hacs-link .url="https://github.com/${this.repository.full_name}">
            <paper-item>
              ${localize(`repository.open_issue`)}
            </paper-item>
          </hacs-link>

          <hacs-link
            .url="https://github.com/hacs/integration/issues/new?assignees=ludeeus&labels=flag&template=flag.md&title=${this
                .repository.full_name}"
          >
            <paper-item>
              ${localize(`repository.flag_this`)}
            </paper-item>
          </hacs-link>
        </paper-listbox>
      </paper-menu-button>
    `;
            }
            static get styles() {
                return [
                    HacsStyle,
                    css `
        paper-dropdown-menu {
          width: 250px;
          margin-top: -24px;
        }
        paper-menu-button {
          float: right;
          top: -68px;
        }
        paper-item {
          color: var(--primary-text-color);
          display: flex;
          background-color: var(
            --paper-listbox-background-color,
            var(--primary-background-color)
          );
        }
      `,
                ];
            }
            RepositoryReload() {
                this.dispatchEvent(new CustomEvent("hacs-repository-action", {
                    detail: {
                        repo: this.repository.id,
                        action: "set_state",
                        data: "other",
                    },
                    bubbles: true,
                    composed: true,
                }));
                this.dispatchEvent(new CustomEvent("hacs-repository-action", {
                    detail: { repo: this.repository.id, action: "update" },
                    bubbles: true,
                    composed: true,
                }));
            }
            RepositoryBeta() {
                this.dispatchEvent(new CustomEvent("hacs-repository-action", {
                    detail: {
                        repo: this.repository.id,
                        action: "set_state",
                        data: "other",
                    },
                    bubbles: true,
                    composed: true,
                }));
                if (this.repository.beta) {
                    this.dispatchEvent(new CustomEvent("hacs-repository-action", {
                        detail: { repo: this.repository.id, action: "hide_beta" },
                        bubbles: true,
                        composed: true,
                    }));
                }
                else {
                    this.dispatchEvent(new CustomEvent("hacs-repository-action", {
                        detail: { repo: this.repository.id, action: "show_beta" },
                        bubbles: true,
                        composed: true,
                    }));
                }
            }
            RepositoryHide() {
                this.dispatchEvent(new CustomEvent("hacs-repository-action", {
                    detail: {
                        repo: this.repository.id,
                        action: "set_state",
                        data: "other",
                    },
                    bubbles: true,
                    composed: true,
                }));
                if (this.repository.hide) {
                    this.dispatchEvent(new CustomEvent("hacs-repository-action", {
                        detail: { repo: this.repository.id, action: "unhide" },
                        bubbles: true,
                        composed: true,
                    }));
                }
                else {
                    this.dispatchEvent(new CustomEvent("hacs-repository-action", {
                        detail: { repo: this.repository.id, action: "hide" },
                        bubbles: true,
                        composed: true,
                    }));
                }
                this.dispatchEvent(new CustomEvent("hacs-location-change", {
                    detail: { value: this.repository.category },
                    bubbles: true,
                    composed: true,
                }));
            }
        };
        __decorate([
            property({ attribute: false })
        ], HacsRepositoryMenu.prototype, "hass", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsRepositoryMenu.prototype, "repository", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsRepositoryMenu.prototype, "route", void 0);
        HacsRepositoryMenu = __decorate([
            customElement("hacs-repository-menu")
        ], HacsRepositoryMenu);
        return HacsRepositoryMenu;
    })();

    let HiddenRepositories = /** @class */ (() => {
        let HiddenRepositories = class HiddenRepositories extends LitElement {
            UnHide(ev) {
                var repo = ev.composedPath()[4].repoID;
                this.dispatchEvent(new CustomEvent("hacs-repository-action", {
                    detail: { repo: repo, action: "unhide" },
                    bubbles: true,
                    composed: true
                }));
            }
            render() {
                if (this.hacs.repositories === undefined) {
                    this.dispatchEvent(new CustomEvent("hacs-recreate", { bubbles: true, composed: true }));
                    return html ``;
                }
                this._hidden = this.hacs.repositories.filter(function (repo) {
                    return repo.hide;
                });
                if (this._hidden.length === 0)
                    return html ``;
                return html `
      <ha-card
        header="${this.hacs
                .localize("settings.hidden_repositories")
                .toUpperCase()}"
      >
        <div class="card-content">
          <div class="custom-repositories-list">
            ${this._hidden
                .sort((a, b) => (a.full_name > b.full_name ? 1 : -1))
                .map(repo => html `
                    <div class="row" .repoID=${repo.id}>
                      <paper-item>
                        <ha-icon
                          title="${this.hacs.localize("settings.unhide")}"
                          class="listicon"
                          icon="mdi:restore"
                          @click=${this.UnHide}
                        ></ha-icon>
                        ${repo.full_name}
                      </paper-item>
                    </div>
                  `)}
          </div>
        </div>
      </ha-card>
    `;
            }
            static get styles() {
                return [
                    HacsStyle,
                    css `
        paper-item {
          display: flex;
        }
        .listicon {
          color: var(--hacs-badge-color, --accent-color);
          left: 0px;
        }
      `
                ];
            }
        };
        __decorate([
            property()
        ], HiddenRepositories.prototype, "_hidden", void 0);
        __decorate([
            property()
        ], HiddenRepositories.prototype, "hacs", void 0);
        HiddenRepositories = __decorate([
            customElement("hacs-hidden-repositories")
        ], HiddenRepositories);
        return HiddenRepositories;
    })();

    let LoveLaceHint = /** @class */ (() => {
        let LoveLaceHint = class LoveLaceHint extends LitElement {
            render() {
                const pluginpath = `${this.repository.full_name.split("/")[1]}/${this.repository.file_name}`;
                return html `
      <div class="lovelace-hint markdown-body">
        <p class="example-title">
          ${localize(`repository.lovelace_instruction`)}:
        </p>
        <table class="llhint">
          <tr>
            <td>URL:</td>
            <td><code>/hacsfiles/${pluginpath}</code></td>
          </tr>
          <tr>
            <td>Type:</td>
            <td>
              module
            </td>
          </tr>
        </table>
      </div>
    `;
            }
            static get styles() {
                return [
                    HacsStyle,
                    css `
        .llhint tbody {
          width: 100%;
          display: inline-table;
        }
        .example-title {
          margin-block-end: 0em;
        }
        .yaml {
          display: inline-flex;
          width: calc(100% - 46px);
        }
      `,
                ];
            }
        };
        __decorate([
            property()
        ], LoveLaceHint.prototype, "configuration", void 0);
        __decorate([
            property()
        ], LoveLaceHint.prototype, "hass", void 0);
        __decorate([
            property()
        ], LoveLaceHint.prototype, "repository", void 0);
        __decorate([
            query("#LovelaceExample")
        ], LoveLaceHint.prototype, "hint", void 0);
        LoveLaceHint = __decorate([
            customElement("hacs-lovelace-hint")
        ], LoveLaceHint);
        return LoveLaceHint;
    })();

    let Authors = /** @class */ (() => {
        let Authors = class Authors extends LitElement {
            render() {
                if (String(this.authors.length) === "0")
                    return html ``;
                let authors = [];
                this.authors.forEach((author) => {
                    const seperateElement = document.createElement("div");
                    seperateElement.className = "seperator";
                    seperateElement.innerText = ",";
                    const authorElement = document.createElement("a");
                    authorElement.href = `https://github.com/${author.replace("@", "")}`;
                    authorElement.target = "_blank";
                    authorElement.className = "author";
                    authorElement.innerText = author.replace("@", "");
                    authors.push(authorElement);
                    authors.push(seperateElement);
                });
                authors.pop();
                return html `
      <div>
        <p>
        <div class="authors">
          <b>${localize("repository.authors")}: </b>
          ${authors}
          </div>
        </p>
      </div>
    `;
            }
            static get styles() {
                return [
                    HacsStyle,
                    css `
        .author {
          color: var(--link-text-color, var(--accent-color));
          margin-left: 4px;
        }
        .authors {
          display: flex;
        }
      `,
                ];
            }
        };
        __decorate([
            property()
        ], Authors.prototype, "hass", void 0);
        __decorate([
            property()
        ], Authors.prototype, "authors", void 0);
        Authors = __decorate([
            customElement("hacs-authors")
        ], Authors);
        return Authors;
    })();

    let RepositoryBannerNote = /** @class */ (() => {
        let RepositoryBannerNote = class RepositoryBannerNote extends LitElement {
            render() {
                if (!this.repository.installed)
                    return html ``;
                let banner;
                if (this.repository.category === "integration") {
                    if (this.repository.first_install && this.repository.config_flow) {
                        banner = document.createElement("hacs-repository-banner-integration-first-install");
                        banner.hacs = this.hacs;
                        banner.hass = this.hass;
                        banner.repository = this.repository;
                        banner.route = this.route;
                        return html `
          ${banner}
        `;
                    }
                    else if (this.repository.status === "pending-restart") {
                        banner = document.createElement("hacs-repository-banner-integration-pending-restart");
                        banner.hacs = this.hacs;
                        banner.hass = this.hass;
                        banner.repository = this.repository;
                        banner.route = this.route;
                        return html `
          ${banner}
        `;
                    }
                }
                if (this.repository.category === "plugin") {
                    if (this.lovelaceconfig !== undefined && !this.status.background_task) {
                        banner = document.createElement("hacs-repository-banner-plugin-not-loaded");
                        banner.hacs = this.hacs;
                        banner.hass = this.hass;
                        banner.configuration = this.configuration;
                        banner.lovelaceconfig = this.lovelaceconfig;
                        banner.status = this.status;
                        banner.repository = this.repository;
                        return html `
          ${banner}
        `;
                    }
                }
                return html ``;
            }
        };
        __decorate([
            property()
        ], RepositoryBannerNote.prototype, "hacs", void 0);
        __decorate([
            property()
        ], RepositoryBannerNote.prototype, "configuration", void 0);
        __decorate([
            property()
        ], RepositoryBannerNote.prototype, "hass", void 0);
        __decorate([
            property()
        ], RepositoryBannerNote.prototype, "lovelaceconfig", void 0);
        __decorate([
            property()
        ], RepositoryBannerNote.prototype, "repository", void 0);
        __decorate([
            property()
        ], RepositoryBannerNote.prototype, "route", void 0);
        __decorate([
            property()
        ], RepositoryBannerNote.prototype, "status", void 0);
        RepositoryBannerNote = __decorate([
            customElement("hacs-repository-banner-note")
        ], RepositoryBannerNote);
        return RepositoryBannerNote;
    })();

    let RepositoryNote = /** @class */ (() => {
        let RepositoryNote = class RepositoryNote extends LitElement {
            render() {
                const addedToLovelace = AddedToLovelace(this.repository, this.lovelaceconfig, this.status);
                let path = this.repository.local_path;
                if (this.repository.category === "theme") {
                    path = `${path}/${this.repository.file_name}`;
                }
                const Note = document.createElement("div");
                Note.className = "repository-note";
                const p = document.createElement("p");
                p.innerText = `${localize(`repository.note_installed`)} '${path}'`;
                if (["appdaemon", "integration"].includes(this.repository.category)) {
                    p.innerText += `, ${localize(`repository.note_${this.repository.category}`)}.`;
                }
                else if (this.repository.category === "plugin" && !addedToLovelace) {
                    p.innerHTML += `, ${localize("repository.note_plugin_post_107").replace("/config/lovelace/resources", "<hacs-link url='/config/lovelace/resources'>'/config/lovelace/resources'</hacs-link>")}`;
                }
                Note.appendChild(p);
                if (this.repository.category === "plugin" && !addedToLovelace) {
                    const LLHint = document.createElement("hacs-lovelace-hint");
                    LLHint.hass = this.hass;
                    LLHint.configuration = this.configuration;
                    LLHint.repository = this.repository;
                    Note.appendChild(LLHint);
                }
                return html ` ${Note} `;
            }
            static get styles() {
                return [
                    HacsStyle,
                    css `
        .repository-note {
          border-top: 1px solid var(--primary-text-color);
        }
        p {
          font-style: italic;
        }
      `,
                ];
            }
        };
        __decorate([
            property({ attribute: false })
        ], RepositoryNote.prototype, "configuration", void 0);
        __decorate([
            property({ attribute: false })
        ], RepositoryNote.prototype, "hass", void 0);
        __decorate([
            property({ attribute: false })
        ], RepositoryNote.prototype, "repository", void 0);
        __decorate([
            property({ attribute: false })
        ], RepositoryNote.prototype, "lovelaceconfig", void 0);
        __decorate([
            property({ attribute: false })
        ], RepositoryNote.prototype, "status", void 0);
        RepositoryNote = __decorate([
            customElement("hacs-repository-note")
        ], RepositoryNote);
        return RepositoryNote;
    })();

    function areInputsEqual(newInputs, lastInputs) {
        if (newInputs.length !== lastInputs.length) {
            return false;
        }
        for (var i = 0; i < newInputs.length; i++) {
            if (newInputs[i] !== lastInputs[i]) {
                return false;
            }
        }
        return true;
    }

    function memoizeOne(resultFn, isEqual) {
        if (isEqual === void 0) { isEqual = areInputsEqual; }
        var lastThis;
        var lastArgs = [];
        var lastResult;
        var calledOnce = false;
        function memoized() {
            var newArgs = [];
            for (var _i = 0; _i < arguments.length; _i++) {
                newArgs[_i] = arguments[_i];
            }
            if (calledOnce && lastThis === this && isEqual(newArgs, lastArgs)) {
                return lastResult;
            }
            lastResult = resultFn.apply(this, newArgs);
            calledOnce = true;
            lastThis = this;
            lastArgs = newArgs;
            return lastResult;
        }
        return memoized;
    }

    /*
    Simple logger class to make logging consistent

    Usage:

    Basic: const logger = new Logger()
    Advanced: const logger = new Logger("main")

    logger.info("My message")
    logger.error("My message")
    logger.warning("My message")

    const obj = {test: "string"}
    logger.info(obj)

    */
    class HacsLogger {
        constructor(section) {
            const name = "HACS";
            this.section = section;
            if (section === undefined)
                this.prefix = name;
            else
                this.prefix = `${name}.${section}`;
        }
        debug(content, object) {
            if (content instanceof Object)
                console.debug(`[${this.prefix}] `, content);
            else
                console.debug(`[${this.prefix}] ${content}`, object || "");
        }
        info(content, object) {
            if (content instanceof Object)
                console.info(`[${this.prefix}] `, content);
            else
                console.info(`[${this.prefix}] ${content}`, object || "");
        }
        warning(content, object) {
            if (content instanceof Object)
                console.warn(`[${this.prefix}] `, content);
            else
                console.warn(`[${this.prefix}] ${content}`, object || "");
        }
        error(content, object) {
            if (content instanceof Object)
                console.error(`[${this.prefix}] `, content);
            else
                console.error(`[${this.prefix}] ${content}`, object || "");
        }
    }

    const navigate$1 = (_node, path) => {
        history.pushState(null, "", path);
    };

    const sortRepositoriesByName = (repositories) => {
        return repositories === null || repositories === void 0 ? void 0 : repositories.sort((a, b) => a.name.toLowerCase() > b.name.toLowerCase() ? 1 : -1);
    };

    /**
     * @license
     * Copyright (c) 2018 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    // IE11 doesn't support classList on SVG elements, so we emulate it with a Set
    class ClassList {
        constructor(element) {
            this.classes = new Set();
            this.changed = false;
            this.element = element;
            const classList = (element.getAttribute('class') || '').split(/\s+/);
            for (const cls of classList) {
                this.classes.add(cls);
            }
        }
        add(cls) {
            this.classes.add(cls);
            this.changed = true;
        }
        remove(cls) {
            this.classes.delete(cls);
            this.changed = true;
        }
        commit() {
            if (this.changed) {
                let classString = '';
                this.classes.forEach((cls) => classString += cls + ' ');
                this.element.setAttribute('class', classString);
            }
        }
    }
    /**
     * Stores the ClassInfo object applied to a given AttributePart.
     * Used to unset existing values when a new ClassInfo object is applied.
     */
    const previousClassesCache = new WeakMap();
    /**
     * A directive that applies CSS classes. This must be used in the `class`
     * attribute and must be the only part used in the attribute. It takes each
     * property in the `classInfo` argument and adds the property name to the
     * element's `class` if the property value is truthy; if the property value is
     * falsey, the property name is removed from the element's `class`. For example
     * `{foo: bar}` applies the class `foo` if the value of `bar` is truthy.
     * @param classInfo {ClassInfo}
     */
    const classMap = directive((classInfo) => (part) => {
        if (!(part instanceof AttributePart) || (part instanceof PropertyPart) ||
            part.committer.name !== 'class' || part.committer.parts.length > 1) {
            throw new Error('The `classMap` directive must be used in the `class` attribute ' +
                'and must be the only part in the attribute.');
        }
        const { committer } = part;
        const { element } = committer;
        let previousClasses = previousClassesCache.get(part);
        if (previousClasses === undefined) {
            // Write static classes once
            // Use setAttribute() because className isn't a string on SVG elements
            element.setAttribute('class', committer.strings.join(' '));
            previousClassesCache.set(part, previousClasses = new Set());
        }
        const classList = (element.classList || new ClassList(element));
        // Remove old classes that no longer apply
        // We use forEach() instead of for-of so that re don't require down-level
        // iteration.
        previousClasses.forEach((name) => {
            if (!(name in classInfo)) {
                classList.remove(name);
                previousClasses.delete(name);
            }
        });
        // Add or remove classes based on their classMap value
        for (const name in classInfo) {
            const value = classInfo[name];
            if (value != previousClasses.has(name)) {
                // We explicitly want a loose truthy check of `value` because it seems
                // more convenient that '' and 0 are skipped.
                if (value) {
                    classList.add(name);
                    previousClasses.add(name);
                }
                else {
                    classList.remove(name);
                    previousClasses.delete(name);
                }
            }
        }
        if (typeof classList.commit === 'function') {
            classList.commit();
        }
    });

    let HacsSinglePageLayout = /** @class */ (() => {
        let HacsSinglePageLayout = class HacsSinglePageLayout extends LitElement {
            constructor() {
                super(...arguments);
                this.isWide = false;
            }
            render() {
                this.isWide =
                    window.localStorage.getItem("dockedSidebar") === '"always_hidden"';
                return html `
      <div
        class="content ${classMap({
                narrow: !this.isWide,
            })}"
      >
        <div class="header-group">
          ${this.narrow || this.isWide
                ? html `<ha-menu-button
                class="${classMap({
                    sidebarhidden: this.isWide && !this.narrow,
                })}"
                .hass=${this.hass}
                .narrow=${this.narrow}
              ></ha-menu-button>`
                : ""}

          <div class="header">${this.header || ""}</div>
        </div>
        <div
          class="together layout ${classMap({
                narrow: this.narrow,
                vertical: !this.isWide || this.narrow,
                horizontal: this.isWide && !this.narrow,
            })}"
        >
          <div class="intro">${this.intro || ""}</div>
          <div class="panel flex-auto"><slot></slot></div>
        </div>
      </div>
    `;
            }
            static get styles() {
                return css `
      :host {
        display: block;
      }
      .header-group {
        display: flex;
        align-items: center;
      }
      .content {
        padding: 28px 20px 0;
        max-width: 1040px;
        margin: 0 auto;
      }
      .layout {
        display: flex;
      }
      .horizontal {
        flex-direction: row;
      }
      .vertical {
        flex-direction: column;
      }
      .flex-auto {
        flex: 1 1 auto;
      }
      .header {
        font-family: var(--paper-font-headline_-_font-family);
        -webkit-font-smoothing: var(
          --paper-font-headline_-_-webkit-font-smoothing
        );
        font-size: var(--paper-font-headline_-_font-size);
        font-weight: var(--paper-font-headline_-_font-weight);
        letter-spacing: var(--paper-font-headline_-_letter-spacing);
        line-height: var(--paper-font-headline_-_line-height);
        opacity: var(--dark-primary-opacity);
      }
      .together {
        margin-top: 32px;
      }
      .intro {
        font-family: var(--paper-font-subhead_-_font-family);
        -webkit-font-smoothing: var(
          --paper-font-subhead_-_-webkit-font-smoothing
        );
        font-weight: var(--paper-font-subhead_-_font-weight);
        line-height: var(--paper-font-subhead_-_line-height);
        width: 100%;
        max-width: 400px;
        margin-right: 40px;
        opacity: var(--dark-primary-opacity);
        font-size: 14px;
        padding-bottom: 20px;
      }
      .panel {
        margin-top: -24px;
      }
      .panel ::slotted(*) {
        margin-top: 24px;
        display: block;
      }
      .narrow.content {
        max-width: 640px;
      }
      .narrow .together {
        margin-top: 0;
      }
      .narrow .intro {
        padding-bottom: 20px;
        margin-right: 0;
        max-width: 500px;
      }
      ha-menu-button {
        --app-header-background-color: var(--primary-background-color);
      }

      ha-menu-button.sidebarhidden {
        position: absolute;
        top: 0;
        left: 0;
      }
    `;
            }
        };
        __decorate([
            property({ attribute: false })
        ], HacsSinglePageLayout.prototype, "hass", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsSinglePageLayout.prototype, "narrow", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsSinglePageLayout.prototype, "route", void 0);
        __decorate([
            property({ type: Boolean })
        ], HacsSinglePageLayout.prototype, "isWide", void 0);
        __decorate([
            property()
        ], HacsSinglePageLayout.prototype, "header", void 0);
        __decorate([
            property()
        ], HacsSinglePageLayout.prototype, "intro", void 0);
        HacsSinglePageLayout = __decorate([
            customElement("hacs-single-page-layout")
        ], HacsSinglePageLayout);
        return HacsSinglePageLayout;
    })();

    const HacsCommonStyle = css `
  :root {
    font-family: var(--paper-font-body1_-_font-family);
    -webkit-font-smoothing: var(--paper-font-body1_-_-webkit-font-smoothing);
    font-size: var(--paper-font-body1_-_font-size);
    font-weight: var(--paper-font-body1_-_font-weight);
    line-height: var(--paper-font-body1_-_line-height);
  }

  a {
    text-decoration: none;
    color: var(--link-text-color, var(--accent-color));
  }
  .pending_upgrade,
  .warning {
    color: orange;
  }

  .header {
    font-size: var(--paper-font-headline_-_font-size);
    opacity: var(--dark-primary-opacity);
    padding: 8px 0 4px 16px;
  }
  mwc-button[raised] {
    border-radius: 10px;
  }
`;

    const sections$8 = {
        updates: [],
        panels: [
            {
                icon: "mdi:puzzle",
                id: "integrations",
                categories: ["integration"],
            },
            {
                icon: "mdi:palette",
                id: "frontend",
                categories: ["plugin", "theme"],
            },
            {
                icon: "mdi:robot",
                id: "automation",
                categories: ["python_script", "appdaemon", "netdaemon"],
            },
        ],
    };

    //import "../components/hacs-link";
    let HacsEntryPanel = /** @class */ (() => {
        let HacsEntryPanel = class HacsEntryPanel extends LitElement {
            constructor() {
                super(...arguments);
                this._panelsEnabled = memoizeOne((sections, config) => {
                    return sections.panels.filter((panel) => {
                        const categories = panel.categories;
                        if (categories === undefined)
                            return true;
                        return (categories.filter((c) => config === null || config === void 0 ? void 0 : config.categories.includes(c)).length !== 0);
                    });
                });
                this._getMessages = memoizeOne((status) => {
                    const messages = [];
                    if (status === null || status === void 0 ? void 0 : status.startup) {
                        messages.push({
                            title: localize("entry.messages.startup.title"),
                            content: localize("entry.messages.startup.content"),
                            severity: "information",
                        });
                    }
                    if (status === null || status === void 0 ? void 0 : status.has_pending_tasks) {
                        messages.push({
                            title: localize("entry.messages.has_pending_tasks.title"),
                            content: localize("entry.messages.has_pending_tasks.content"),
                            severity: "warning",
                        });
                    }
                    if (status === null || status === void 0 ? void 0 : status.disabled) {
                        messages.push({
                            title: localize("entry.messages.disabled.title"),
                            content: localize("entry.messages.disabled.content"),
                            severity: "error",
                        });
                    }
                    return messages;
                });
            }
            render() {
                var _a;
                const messages = this._getMessages(this.status);
                this.isWide =
                    window.localStorage.getItem("dockedSidebar") === '"always_hidden"';
                sections$8.updates = []; // reset so we don't get duplicates
                (_a = this.repositories) === null || _a === void 0 ? void 0 : _a.forEach((repo) => {
                    if (repo.pending_upgrade) {
                        sections$8.updates.push(repo);
                    }
                });
                const enabledSections = this._panelsEnabled(sections$8, this.configuration);
                return html `
      <hacs-single-page-layout
        .hass=${this.hass}
        .route=${this.route}
        .narrow=${this.narrow}
        .header=${this.narrow ? "HACS" : "Home Assistant Community Store"}
        intro="${this.isWide ||
                (sections$8.updates.length === 0 && messages.length === 0)
                ? localize("entry.intro")
                : ""}"
      >
        ${messages.length !== 0
                ? html ` <ha-card>
              <div class="header">${localize("entry.information")}</div>
              ${messages.map((message) => html `
                    <paper-icon-item>
                      <ha-icon
                        class="${message.severity}"
                        icon="mdi:alert-circle"
                        slot="item-icon"
                      ></ha-icon>
                      <paper-item-body two-line>
                        ${message.title}
                        <div secondary>
                          ${message.content}
                        </div>
                      </paper-item-body>
                    </paper-icon-item>
                  `)}
            </ha-card>`
                : ""}
        ${sections$8.updates.length !== 0
                ? html ` <ha-card>
              <div class="header">${localize("entry.pending_updates")}</div>
              ${sortRepositoriesByName(sections$8.updates).map((repository) => html `
                    <paper-icon-item
                      @click="${() => this._openUpdateDialog(repository)}"
                    >
                      <ha-icon
                        class="pending_upgrade"
                        icon="mdi:arrow-up-circle"
                        slot="item-icon"
                      ></ha-icon>
                      <paper-item-body two-line>
                        ${repository.name}
                        <div secondary>
                          ${localize("sections.pending_repository_upgrade")
                    .replace("{installed}", repository.installed_version)
                    .replace("{available}", repository.available_version)}
                        </div>
                      </paper-item-body>
                    </paper-icon-item>
                  `)}
            </ha-card>`
                : ""}
        <ha-card>
          ${enabledSections.map((panel) => html `
                <paper-icon-item @click=${() => this._changeLocation(panel.id)}>
                  <ha-icon .icon=${panel.icon} slot="item-icon"></ha-icon>
                  <paper-item-body two-line>
                    ${localize(`sections.${panel.id}.title`)}
                    <div secondary>
                      ${localize(`sections.${panel.id}.description`)}
                    </div>
                  </paper-item-body>
                  <ha-icon-button icon="mdi:chevron-right"></ha-icon-button>
                </paper-icon-item>
              `)}
          <paper-icon-item @click=${this._openAboutDialog}>
            <ha-icon icon="mdi:information" slot="item-icon"></ha-icon>
            <paper-item-body two-line>
              ${localize(`sections.about.title`)}
              <div secondary>
                ${localize(`sections.about.description`)}
              </div>
            </paper-item-body>
          </paper-icon-item>
        </ha-card>
      </hacs-single-page-layout>
    `;
            }
            _openUpdateDialog(repository) {
                this.dispatchEvent(new CustomEvent("hacs-dialog", {
                    detail: {
                        type: "update",
                        repository: repository.id,
                    },
                    bubbles: true,
                    composed: true,
                }));
            }
            _openAboutDialog() {
                this.dispatchEvent(new CustomEvent("hacs-dialog", {
                    detail: {
                        type: "about",
                        configuration: this.configuration,
                        repositories: this.repositories,
                    },
                    bubbles: true,
                    composed: true,
                }));
            }
            static get styles() {
                return [
                    HacsCommonStyle,
                    css `
        a {
          text-decoration: none;
          color: var(--primary-text-color);
          position: relative;
          display: block;
          outline: 0;
        }
        ha-icon-button,
        ha-icon {
          color: var(--secondary-text-color);
        }

        .iron-selected paper-item::before {
          position: absolute;
          top: 0;
          right: 0;
          bottom: 0;
          left: 0;
          pointer-events: none;
          content: "";
          transition: opacity 15ms linear;
          will-change: opacity;
        }
        paper-icon-item {
          cursor: pointer;
        }
        .iron-selected paper-item:focus::before,
        .iron-selected:focus paper-item::before {
          opacity: 0.2;
        }

        paper-item-body {
          width: 100%;
          min-height: var(--paper-item-body-two-line-min-height, 72px);
          display: var(--layout-vertical_-_display);
          flex-direction: var(--layout-vertical_-_flex-direction);
          justify-content: var(--layout-center-justified_-_justify-content);
        }
        paper-item-body div {
          font-size: 14px;
          color: var(--secondary-text-color);
        }
      `,
                ];
            }
            _changeLocation(id) {
                this.route.path = `/${id}`;
                this.dispatchEvent(new CustomEvent("hacs-location-changed", {
                    detail: { route: this.route },
                    bubbles: true,
                    composed: true,
                }));
            }
        };
        __decorate([
            property({ attribute: false })
        ], HacsEntryPanel.prototype, "configuration", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsEntryPanel.prototype, "hass", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsEntryPanel.prototype, "lovelace", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsEntryPanel.prototype, "repositories", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsEntryPanel.prototype, "route", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsEntryPanel.prototype, "status", void 0);
        __decorate([
            property({ type: Boolean })
        ], HacsEntryPanel.prototype, "isWide", void 0);
        __decorate([
            property({ type: Boolean })
        ], HacsEntryPanel.prototype, "narrow", void 0);
        HacsEntryPanel = __decorate([
            customElement("hacs-entry-panel")
        ], HacsEntryPanel);
        return HacsEntryPanel;
    })();

    let HacsTabbedLayout = /** @class */ (() => {
        let HacsTabbedLayout = class HacsTabbedLayout extends LitElement {
            render() {
                return html `
      <div class="main">
        <div class="toolbar">
        <ha-icon-button icon="mdi:chevron-left" @click=${this._goBack}></ha-icon-button>
                ${this.narrow
                ? html `
                        <div class="main-title">
                          ${localize(`sections.${this.selected}.title`)}
                        </div>
                      `
                : ""}
          <div id="tabbar" class=${classMap({ "bottom-bar": this.narrow })}>
            ${this.tabs.map((tab) => html `
                <div
                  class="${classMap({
                "toolbar-button": true,
                selected: this.selected === tab.id,
            })}"
                  @click=${() => this._ChangeTabAction(tab.id)}
                >
                  ${this.narrow
                ? html `<ha-icon-button .icon=${tab.icon}></ha-icon-button
                        ><span class="name"
                          >${localize(`sections.${tab.id}.title`)}</span
                        >`
                : localize(`sections.${tab.id}.title`)}
                </div>
              `)}
          </div>
          <div id="toolbar-icon">
          <slot name="toolbar-icon"></slot>
          </div>
        </div>
        </div>
        <div class="content"><slot></slot></div>
      </div>
    `;
            }
            _ChangeTabAction(tab) {
                window.scrollTo(0, 0);
                this.selected = tab;
                this.route.path = `/${tab}`;
                this._locationChanged();
            }
            _goBack() {
                this.route.path = "";
                this._locationChanged();
            }
            _locationChanged() {
                this.dispatchEvent(new CustomEvent("hacs-location-changed", {
                    detail: { route: this.route },
                    bubbles: true,
                    composed: true,
                }));
            }
            static get styles() {
                return css `
      :host {
        display: block;
        height: 100vh;
        min-height: -webkit-fill-available;
        background-color: var(--primary-background-color);
      }
      ha-menu-button {
        margin-right: 24px;
      }
      .name {
        white-space: nowrap;
      }
      .toolbar {
        display: flex;
        align-items: center;
        font-size: 20px;
        height: 65px;
        background-color: var(--sidebar-background-color);
        font-weight: 400;
        color: var(--sidebar-text-color);
        border-bottom: 1px solid var(--divider-color);
        font-family: var(--paper-font-body1_-_font-family);
        padding: 0 16px;
        box-sizing: border-box;
      }
      .searchbar {
        display: flex;
        align-items: center;
        font-size: 20px;
        top: 65px;
        height: 65px;
        background-color: var(--sidebar-background-color);
        border-bottom: 1px solid var(--divider-color);
        padding: 0 16px;
        box-sizing: border-box;
      }
      #tabbar {
        display: flex;
        font-size: 14px;
      }
      #tabbar.bottom-bar {
        position: absolute;
        bottom: 0;
        left: 0;
        padding: 0 16px;
        box-sizing: border-box;
        background-color: var(--sidebar-background-color);
        border-top: 1px solid var(--divider-color);
        justify-content: space-between;
        z-index: 1;
        font-size: 12px;
        width: 100%;
      }
      #tabbar:not(.bottom-bar) {
        flex: 1;
        justify-content: center;
      }

      :host(:not([narrow])) #toolbar-icon {
        min-width: 40px;
      }
      ha-menu-button,
      ha-icon-button-arrow-prev,
      ::slotted([slot="toolbar-icon"]) {
        flex-shrink: 0;
        pointer-events: auto;
        color: var(--sidebar-icon-color);
      }

      .main-title {
        -webkit-font-smoothing: var(
          --paper-font-headline_-_-webkit-font-smoothing
        );
        flex: 1;
        font-family: var(--paper-font-headline_-_font-family);
        font-size: var(--paper-font-headline_-_font-size);
        font-weight: var(--paper-font-headline_-_font-weight);
        letter-spacing: var(--paper-font-headline_-_letter-spacing);
        line-height: 26px;
        max-height: 26px;
        opacity: var(--dark-primary-opacity);
        overflow: hidden;
        text-overflow: ellipsis;
      }

      .content {
        position: relative;
        width: 100%;
        height: calc(100vh - 65px);
        overflow-y: auto;
        overflow: auto;
        -webkit-overflow-scrolling: touch;
      }

      :host([narrow]) .content {
        height: calc(100vh - 128px);
      }

      .toolbar-button {
        padding: 0 32px;
        display: flex;
        flex-direction: column;
        text-align: center;
        align-items: center;
        justify-content: center;
        height: 64px;
        cursor: pointer;
        position: relative;
        outline: none;
        box-sizing: border-box;
      }

      ha-icon-button {
        cursor: pointer;
      }
      .selected {
        color: var(--primary-color);
        border-bottom: 2px solid var(--primary-color);
      }
      .search-input {
        width: calc(100% - 48px);
        height: 40px;
        border: 0;
        padding: 0 16px;
        font-size: initial;
        color: var(--sidebar-text-color);
        font-family: var(--paper-font-body1_-_font-family);
      }
      input:focus {
        outline-offset: 0;
        outline: 0;
      }
    `;
            }
        };
        __decorate([
            property({ attribute: false })
        ], HacsTabbedLayout.prototype, "hass", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsTabbedLayout.prototype, "route", void 0);
        __decorate([
            property({ type: Boolean })
        ], HacsTabbedLayout.prototype, "narrow", void 0);
        __decorate([
            property()
        ], HacsTabbedLayout.prototype, "selected", void 0);
        __decorate([
            property()
        ], HacsTabbedLayout.prototype, "tabs", void 0);
        HacsTabbedLayout = __decorate([
            customElement("hacs-tabbed-layout")
        ], HacsTabbedLayout);
        return HacsTabbedLayout;
    })();

    let HacsRepositoryCard = /** @class */ (() => {
        let HacsRepositoryCard = class HacsRepositoryCard extends LitElement {
            render() {
                const path = this.repository.local_path.split("/");
                return html `
      <ha-card
        class=${classMap({
                "status-border": this.repository.new || this.repository.pending_upgrade,
                "status-new": this.repository.new && !this.repository.installed,
                "status-update": this.repository.pending_upgrade,
            })}
      >
        <div class="card-content">
          <div class="group-header">
            ${this.repository.pending_upgrade
                ? html `
                  <div class="status-header update-header">
                    ${localize("repository_card.pending_update")}
                  </div>
                `
                : this.repository.new && !this.repository.installed
                    ? html `
                  <div class="status-header new-header">
                    ${localize("repository_card.new_repository")}
                  </div>
                `
                    : html `<div class="status-header default-header"></div>`}

            <div class="title">
              <h2 class="pointer" @click=${this._showReopsitoryInfo}>
                ${this.repository.name}
              </h2>
              ${this.repository.category !== "integration"
                ? html ` <hacs-chip
                    icon="hacs:hacs"
                    .value=${localize(`common.${this.repository.category}`)}
                  ></hacs-chip>`
                : ""}
            </div>
          </div>
          <paper-item>
            <paper-item-body
              >${this.repository.description}</paper-item-body
            ></paper-item
          >
        </div>
        <div class="card-actions">
          ${this.repository.new && !this.repository.installed
                ? html `<div>
                  <mwc-button @click=${this._installRepository}
                    >${localize("common.install")}</mwc-button
                  >
                </div>
                <div>
                  <mwc-button @click=${this._showReopsitoryInfo}
                    >${localize("repository_card.information")}</mwc-button
                  >
                </div>
                <div>
                  <hacs-link
                    .url="https://github.com/${this.repository.full_name}"
                    ><mwc-button
                      >${localize("common.repository")}</mwc-button
                    ></hacs-link
                  >
                </div>
                <div>
                  <mwc-button @click=${this._setNotNew}
                    >${localize("repository_card.dismiss")}</mwc-button
                  >
                </div>`
                : this.repository.pending_upgrade
                    ? html `<div>
                <mwc-button
                  class="update-header"
                  @click=${this._updateRepository}
                  raised
                  >${localize("common.update")}</mwc-button
                >
              </div>`
                    : html `<div>
                <hacs-link
                  .url="https://github.com/${this.repository.full_name}"
                  ><mwc-button
                    >${localize("common.repository")}</mwc-button
                  ></hacs-link
                >
              </div>`}
          ${this.repository.installed
                ? html ` <paper-menu-button
                horizontal-align="right"
                vertical-align="top"
                vertical-offset="40"
                close-on-activate
              >
                <ha-icon-button
                  icon="hass:dots-vertical"
                  slot="dropdown-trigger"
                ></ha-icon-button>
                <paper-listbox slot="dropdown-content">
                  <paper-item class="pointer" @click=${this._showReopsitoryInfo}
                    >${localize("repository_card.information")}</paper-item
                  >

                  <paper-item
                    class="pointer"
                    @click=${this._updateReopsitoryInfo}
                    >${localize("repository_card.update_information")}</paper-item
                  >

                  <paper-item @click=${this._installRepository}
                    >${localize("repository_card.reinstall")}</paper-item
                  >

                  ${this.repository.category === "plugin"
                    ? html `<hacs-link
                        .url="/hacsfiles/${path.pop()}/${this.repository
                        .file_name}"
                        newtab
                        ><paper-item class="pointer"
                          >${localize("repository_card.open_source")}</paper-item
                        ></hacs-link
                      >`
                    : ""}

                  <hacs-link
                    .url="https://github.com/${this.repository
                    .full_name}/issues"
                    ><paper-item class="pointer"
                      >${localize("repository_card.open_issue")}</paper-item
                    ></hacs-link
                  >

                  ${String(this.repository.id) !== "172733314"
                    ? html `<hacs-link
                          .url="https://github.com/hacs/integration/issues/new?assignees=ludeeus&labels=flag&template=flag.md&title=${this
                        .repository.full_name}"
                          ><paper-item class="pointer uninstall"
                            >${localize("repository_card.report")}</paper-item
                          ></hacs-link
                        >
                        <paper-item
                          class="pointer uninstall"
                          @click=${this._uninstallRepository}
                          >${localize("common.uninstall")}</paper-item
                        >`
                    : ""}
                </paper-listbox>
              </paper-menu-button>`
                : ""}
        </div>
      </ha-card>
    `;
            }
            async _updateReopsitoryInfo() {
                await repositoryUpdate(this.hass, this.repository.id);
            }
            async _showReopsitoryInfo() {
                this.dispatchEvent(new CustomEvent("hacs-dialog", {
                    detail: {
                        type: "repository-info",
                        repository: this.repository.id,
                    },
                    bubbles: true,
                    composed: true,
                }));
            }
            _lovelaceUrl() {
                var _a, _b;
                return `/hacsfiles/${(_a = this.repository) === null || _a === void 0 ? void 0 : _a.full_name.split("/")[1]}/${(_b = this.repository) === null || _b === void 0 ? void 0 : _b.file_name}`;
            }
            async _updateRepository() {
                this.dispatchEvent(new CustomEvent("hacs-dialog", {
                    detail: {
                        type: "update",
                        repository: this.repository.id,
                    },
                    bubbles: true,
                    composed: true,
                }));
            }
            async _setNotNew() {
                await repositorySetNotNew(this.hass, this.repository.id);
            }
            _installRepository() {
                this.dispatchEvent(new CustomEvent("hacs-dialog", {
                    detail: {
                        type: "install",
                        repository: this.repository.id,
                    },
                    bubbles: true,
                    composed: true,
                }));
            }
            async _uninstallRepository() {
                await repositoryUninstall(this.hass, this.repository.id);
                if (this.repository.category === "plugin" &&
                    this.status.lovelace_mode !== "yaml") {
                    const resources = await fetchResources(this.hass);
                    resources
                        .filter((resource) => resource.url === this._lovelaceUrl())
                        .forEach((resource) => {
                        deleteResource(this.hass, String(resource.id));
                    });
                }
            }
            static get styles() {
                return [
                    HacsCommonStyle,
                    css `
        :host {
          max-width: 500px;
        }
        ha-card {
          display: flex;
          flex-direction: column;
          height: 100%;
        }
        hacs-chip {
          margin: 8px 4px 0 0;
        }
        .title {
          display: flex;
          justify-content: space-between;
        }
        .card-content {
          padding: 0 0 3px 0;
          height: 100%;
        }
        .card-actions {
          border-top: none;
          bottom: 0;
          display: flex;
          justify-content: space-between;
          align-items: center;
          padding: 5px;
        }
        .group-header {
          height: auto;
          align-content: center;
        }
        .group-header h2 {
          margin: 0;
          padding: 8px 16px;
        }
        h2 {
          margin-top: 0;
          min-height: 24px;
        }
        paper-menu-button {
          color: var(--secondary-text-color);
          padding: 0;
          float: right;
        }

        .pointer {
          cursor: pointer;
        }
        paper-item-body {
          opacity: var(--dark-primary-opacity);
        }

        .status-new {
          border-color: var(--hacs-new-color, var(--google-blue-500));
        }

        .status-update {
          border-color: var(--hacs-update-color, var(--google-yellow-500));
        }

        .new-header {
          background-color: var(--hacs-new-color, var(--google-blue-500));
          color: var(--hacs-new-text-color, var(--text-primary-color));
        }

        .update-header {
          background-color: var(--hacs-update-color, var(--google-yellow-500));
          color: var(--hacs-update-text-color, var(--text-primary-color));
        }

        .default-header {
          padding: 10px 0 !important;
        }

        mwc-button.update-header {
          --mdc-theme-primary: var(
            --hacs-update-color,
            var(--google-yellow-500)
          );
          --mdc-theme-on-primary: var(
            --hacs-update-text-color,
            var(--text-primary-color)
          );
        }

        .status-border {
          border-style: solid;
          border-width: 1px;
        }

        .status-header {
          top: 0;
          padding: 6px 1px;
          margin: -1px;
          width: 100%;
          font-weight: 300;
          text-align: center;
          left: 0;
          border-top-left-radius: var(--ha-card-border-radius);
          border-top-right-radius: var(--ha-card-border-radius);
        }
        .uninstall {
          color: var(--hacs-error-color, var(--google-red-500));
        }
      `,
                ];
            }
        };
        __decorate([
            property({ attribute: false })
        ], HacsRepositoryCard.prototype, "hass", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsRepositoryCard.prototype, "repository", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsRepositoryCard.prototype, "status", void 0);
        __decorate([
            property({ type: Boolean })
        ], HacsRepositoryCard.prototype, "narrow", void 0);
        HacsRepositoryCard = __decorate([
            customElement("hacs-repository-card")
        ], HacsRepositoryCard);
        return HacsRepositoryCard;
    })();

    let HacsSearch = /** @class */ (() => {
        let HacsSearch = class HacsSearch extends LitElement {
            render() {
                return html `
      <div class="searchbar">
        <ha-icon icon="mdi:magnify" role="button"></ha-icon>
        <input
          id="search-input"
          class="search-input"
          placeholder="${localize("search.placeholder")}"
          .value=${this.input || ""}
          @input=${this._inputValueChanged}
        />
        ${this.input
                ? html `
              <ha-icon-button
                icon="mdi:close"
                role="button"
                @click=${this._clearInput}
              ></ha-icon-button>
            `
                : ""}
      </div>
    `;
            }
            _inputValueChanged() {
                var _a;
                this.input = (_a = this._filterInput) === null || _a === void 0 ? void 0 : _a.value;
            }
            _clearInput() {
                this.input = "";
                this.dispatchEvent(new CustomEvent("input", {
                    bubbles: true,
                    composed: true,
                }));
            }
            static get styles() {
                return [
                    HacsCommonStyle,
                    css `
        .searchbar {
          display: flex;
          align-items: center;
          font-size: 20px;
          top: 65px;
          height: 65px;
          background-color: var(--sidebar-background-color);
          border-bottom: 1px solid var(--divider-color);
          padding: 0 16px;
          box-sizing: border-box;
        }
        .search-input {
          width: calc(100% - 48px);
          height: 40px;
          border: 0;
          padding: 0 16px;
          font-size: initial;
          color: var(--sidebar-text-color);
          font-family: var(--paper-font-body1_-_font-family);
        }
        input:focus {
          outline-offset: 0;
          outline: 0;
        }
        input {
          background-color: var(--sidebar-background-color);
        }
      `,
                ];
            }
        };
        __decorate([
            property()
        ], HacsSearch.prototype, "input", void 0);
        __decorate([
            query("#search-input")
        ], HacsSearch.prototype, "_filterInput", void 0);
        HacsSearch = __decorate([
            customElement("hacs-search")
        ], HacsSearch);
        return HacsSearch;
    })();

    let HacsTabbedMenu = /** @class */ (() => {
        let HacsTabbedMenu = class HacsTabbedMenu extends LitElement {
            render() {
                var _a, _b, _c;
                return html `<paper-menu-button
      slot="toolbar-icon"
      horizontal-align="right"
      vertical-align="top"
      vertical-offset="40"
      close-on-activate
    >
      <ha-icon-button
        icon="hass:dots-vertical"
        slot="dropdown-trigger"
      ></ha-icon-button>
      <paper-listbox slot="dropdown-content">
        <hacs-link url="https://hacs.xyz/"
          ><paper-item>${localize("menu.documentation")}</paper-item></hacs-link
        >
        <paper-item @click=${() => window.location.reload(true)}
          >${localize("menu.reload")}</paper-item
        >
        ${((_a = this.repositories) === null || _a === void 0 ? void 0 : _a.filter((repo) => repo.new).length) !== 0
                ? html ` <paper-item @click=${this._clearAllNewRepositories}
              >${localize("menu.dismiss")}</paper-item
            >`
                : ""}

        <hacs-link url="https://github.com/hacs"
          ><paper-item>GitHub</paper-item></hacs-link
        >
        <hacs-link url="https://hacs.xyz/docs/issues"
          ><paper-item>${localize("menu.open_issue")}</paper-item></hacs-link
        >
        ${!((_b = this.status) === null || _b === void 0 ? void 0 : _b.disabled) && !((_c = this.status) === null || _c === void 0 ? void 0 : _c.background_task)
                ? html `<paper-item @click=${this._showCustomRepositoriesDialog}
              >${localize("menu.custom_repositories")}</paper-item
            >`
                : ""}

        <paper-item @click=${this._showAboutDialog}
          >${localize("menu.about")}</paper-item
        >
      </paper-listbox>
    </paper-menu-button>`;
            }
            async _clearAllNewRepositories() {
                await settingsClearAllNewRepositories(this.hass);
            }
            _showAboutDialog() {
                this.dispatchEvent(new CustomEvent("hacs-dialog", {
                    detail: {
                        type: "about",
                        configuration: this.configuration,
                        repositories: this.repositories,
                    },
                    bubbles: true,
                    composed: true,
                }));
            }
            _showCustomRepositoriesDialog() {
                this.dispatchEvent(new CustomEvent("hacs-dialog", {
                    detail: {
                        type: "custom-repositories",
                        repositories: this.repositories,
                    },
                    bubbles: true,
                    composed: true,
                }));
            }
            static get styles() {
                return css `
      paper-menu-button {
        color: var(--secondary-text-color);
        padding: 0;
      }
      paper-item {
        cursor: pointer;
      }
      paper-item-body {
        opacity: var(--dark-primary-opacity);
      }
    `;
            }
        };
        __decorate([
            property({ attribute: false })
        ], HacsTabbedMenu.prototype, "configuration", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsTabbedMenu.prototype, "hass", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsTabbedMenu.prototype, "narrow", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsTabbedMenu.prototype, "route", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsTabbedMenu.prototype, "repositories", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsTabbedMenu.prototype, "lovelace", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsTabbedMenu.prototype, "status", void 0);
        HacsTabbedMenu = __decorate([
            customElement("hacs-tabbed-menu")
        ], HacsTabbedMenu);
        return HacsTabbedMenu;
    })();

    let HacsStorePanel = /** @class */ (() => {
        let HacsStorePanel = class HacsStorePanel extends LitElement {
            constructor() {
                super(...arguments);
                this._repositoriesInActiveSection = memoizeOne((repositories, sections, section) => {
                    const installedRepositories = repositories === null || repositories === void 0 ? void 0 : repositories.filter((repo) => {
                        var _a;
                        return ((_a = sections.panels
                            .find((panel) => panel.id === section)
                            .categories) === null || _a === void 0 ? void 0 : _a.includes(repo.category)) && repo.installed;
                    });
                    const newRepositories = repositories === null || repositories === void 0 ? void 0 : repositories.filter((repo) => {
                        var _a;
                        return ((_a = sections.panels
                            .find((panel) => panel.id === section)
                            .categories) === null || _a === void 0 ? void 0 : _a.includes(repo.category)) &&
                            repo.new &&
                            !repo.installed;
                    });
                    return [installedRepositories || [], newRepositories || []];
                });
                this._panelsEnabled = memoizeOne((sections, config) => {
                    return sections.panels.filter((panel) => {
                        const categories = panel.categories;
                        if (categories === undefined)
                            return true;
                        return (categories.filter((c) => config === null || config === void 0 ? void 0 : config.categories.includes(c)).length !== 0);
                    });
                });
            }
            render() {
                var _a;
                const [InstalledRepositories, newRepositories,] = this._repositoriesInActiveSection(this.repositories, sections$8, this.section);
                const tabs = this._panelsEnabled(sections$8, this.configuration);
                return html `<hacs-tabbed-layout
      .hass=${this.hass}
      .tabs=${tabs}
      .route=${this.route}
      .narrow=${this.narrow}
      .selected=${this.section}
      ><hacs-tabbed-menu
        slot="toolbar-icon"
        .hass=${this.hass}
        .route=${this.route}
        .narrow=${this.narrow}
        .configuration=${this.configuration}
        .lovelace=${this.lovelace}
        .status=${this.status}
        .repositories=${this.repositories}
      >
      </hacs-tabbed-menu>

      ${(newRepositories === null || newRepositories === void 0 ? void 0 : newRepositories.length) > 10
                ? html `<div class="new-repositories">
            ${localize("store.new_repositories_note")}
          </div>`
                : ""}
      <div class="content">
        ${(newRepositories === null || newRepositories === void 0 ? void 0 : newRepositories.concat(InstalledRepositories).length) !== 0
                ? (_a = newRepositories === null || newRepositories === void 0 ? void 0 : newRepositories.concat(InstalledRepositories)) === null || _a === void 0 ? void 0 : _a.map((repo) => html `<hacs-repository-card
                    .hass=${this.hass}
                    .repository=${repo}
                    .narrow=${this.narrow}
                    .status=${this.status}
                  ></hacs-repository-card>`) : html `<ha-card class="no-repositories">
              <div class="header">${localize("store.no_repositories")} 😕</div>
              <p>
                ${localize("store.no_repositories_desc1")}<br />${localize("store.no_repositories_desc2")}
              </p>
            </ha-card>`}
      </div>
      <mwc-fab
        ?narrow="${this.narrow}"
        title="Add integration"
        @click=${this._addIntegration}
      >
        <ha-icon icon="mdi:plus"></ha-icon>
      </mwc-fab>
    </hacs-tabbed-layout>`;
            }
            _addIntegration() {
                this.dispatchEvent(new CustomEvent("hacs-dialog", {
                    detail: {
                        type: "add-repository",
                        repositories: this.repositories,
                        section: this.section,
                    },
                    bubbles: true,
                    composed: true,
                }));
            }
            static get styles() {
                return [
                    HacsCommonStyle,
                    css `
        hacs-repository-card {
          max-width: 500px;
          display: flex;
          flex-direction: column;
          justify-content: space-between;
        }
        .content {
          display: grid;
          grid-template-columns: repeat(auto-fit, minmax(360px, 1fr));
          grid-gap: 16px 16px;
          padding: 8px 16px 16px;
          margin-bottom: 64px;
        }
        .no-repositories {
          width: 100%;
          text-align: center;
          margin-top: 12px;
        }
        .new-repositories {
          margin: 4px 16px 0 16px;
        }
        paper-item {
          cursor: pointer;
        }

        mwc-fab[narrow] {
          margin-bottom: 65px;
        }
        mwc-fab {
          position: fixed;
          bottom: 16px;
          right: 16px;
          filter: blur(0.4px);
          z-index: 1;
          display: inline-flex;
          align-items: center;
          justify-content: center;
          box-sizing: border-box;
          width: 56px;
          height: 56px;
          fill: currentcolor;
          cursor: pointer;
          user-select: none;
          -webkit-appearance: none;
          background-color: var(--accent-color);
          color: var(--text-primary-color) !important;
          box-shadow: var(
            --mdc-fab-box-shadow-active,
            0px 7px 8px -4px rgba(0, 0, 0, 0.2),
            0px 12px 17px 2px rgba(0, 0, 0, 0.14),
            0px 5px 22px 4px rgba(0, 0, 0, 0.12)
          );
          border-radius: 50%;
          --mdc-ripple-fg-opacity: 0.24;
          --mdc-ripple-fg-size: 32px;
          --mdc-ripple-fg-scale: 1.75;
          --mdc-ripple-left: 12px;
          --mdc-ripple-top: 12px;
        }
      `,
                ];
            }
        };
        __decorate([
            property({ attribute: false })
        ], HacsStorePanel.prototype, "configuration", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsStorePanel.prototype, "hass", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsStorePanel.prototype, "narrow", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsStorePanel.prototype, "route", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsStorePanel.prototype, "repositories", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsStorePanel.prototype, "lovelace", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsStorePanel.prototype, "status", void 0);
        __decorate([
            property()
        ], HacsStorePanel.prototype, "section", void 0);
        HacsStorePanel = __decorate([
            customElement("hacs-store-panel")
        ], HacsStorePanel);
        return HacsStorePanel;
    })();

    let HacsDialogBase = /** @class */ (() => {
        class HacsDialogBase extends LitElement {
            constructor() {
                super(...arguments);
                this.active = false;
                this.secondary = false;
                this.loading = true;
            }
            shouldUpdate(changedProperties) {
                changedProperties.forEach((_oldValue, propName) => {
                    if (propName === "hass") {
                        this.sidebarDocked =
                            window.localStorage.getItem("dockedSidebar") === '"docked"';
                    }
                });
                return (changedProperties.has("sidebarDocked") ||
                    changedProperties.has("narrow") ||
                    changedProperties.has("active") ||
                    changedProperties.has("params") ||
                    changedProperties.has("_updating"));
            }
            connectedCallback() {
                super.connectedCallback();
                this.sidebarDocked =
                    window.localStorage.getItem("dockedSidebar") === '"docked"';
            }
        }
        __decorate([
            property({ attribute: false })
        ], HacsDialogBase.prototype, "configuration", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsDialogBase.prototype, "critical", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsDialogBase.prototype, "hass", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsDialogBase.prototype, "lovelace", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsDialogBase.prototype, "repositories", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsDialogBase.prototype, "route", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsDialogBase.prototype, "status", void 0);
        __decorate([
            property({ type: Boolean })
        ], HacsDialogBase.prototype, "active", void 0);
        __decorate([
            property({ type: Boolean })
        ], HacsDialogBase.prototype, "secondary", void 0);
        __decorate([
            property({ type: Boolean })
        ], HacsDialogBase.prototype, "loading", void 0);
        __decorate([
            property({ type: Boolean })
        ], HacsDialogBase.prototype, "narrow", void 0);
        __decorate([
            property({ type: Boolean })
        ], HacsDialogBase.prototype, "sidebarDocked", void 0);
        return HacsDialogBase;
    })();

    let HacsGenericDialog = /** @class */ (() => {
        let HacsGenericDialog = class HacsGenericDialog extends HacsDialogBase {
            constructor() {
                super(...arguments);
                this.markdown = false;
                this._getRepository = memoizeOne((repositories, repository) => repositories === null || repositories === void 0 ? void 0 : repositories.find((repo) => repo.id === repository));
            }
            render() {
                if (!this.active)
                    return html ``;
                const repository = this._getRepository(this.repositories, this.repository);
                return html `
      <hacs-dialog
        .active=${this.active}
        .narrow=${this.narrow}
        .hass=${this.hass}
      >
        <div slot="header">${this.header || ""}</div>
        ${this.markdown
                ? this.repository
                    ? markdown.html(this.content || "", repository)
                    : markdown.html(this.content || "")
                : this.content || ""}
      </hacs-dialog>
    `;
            }
        };
        __decorate([
            property({ type: Boolean })
        ], HacsGenericDialog.prototype, "markdown", void 0);
        __decorate([
            property()
        ], HacsGenericDialog.prototype, "repository", void 0);
        __decorate([
            property()
        ], HacsGenericDialog.prototype, "header", void 0);
        __decorate([
            property()
        ], HacsGenericDialog.prototype, "content", void 0);
        HacsGenericDialog = __decorate([
            customElement("hacs-generic-dialog")
        ], HacsGenericDialog);
        return HacsGenericDialog;
    })();

    let HacsDialog = /** @class */ (() => {
        let HacsDialog = class HacsDialog extends HacsDialogBase {
            constructor() {
                super(...arguments);
                this.noActions = false;
                this.hasContent = true;
                this.dynamicHeight = false;
                this.hasFilter = false;
            }
            render() {
                if (!this.active) {
                    return html ``;
                }
                const sidebar = window.localStorage.getItem("dockedSidebar");
                const sidebarDocked = sidebar === '"docked"';
                return html `
      <div
        class=${classMap({
                backdrop: true,
                docked: sidebarDocked,
                "full-width": sidebar === '"always_hidden"',
                narrow: this.narrow,
            })}
      >
        <ha-card
          class=${classMap({
                dialog: true,
                "has-content": this.hasContent,
                "dynamic-height": this.dynamicHeight,
                narrow: this.narrow,
            })}
        >
          <div class="header-group">
            <ha-icon-button
              class="close"
              icon="mdi:close"
              @click=${this._close}
            ></ha-icon-button>

            <div
              class=${classMap({
                header: true,
                "narrow-header": this.narrow,
            })}
            >
              <slot name="header"></slot>
            </div>
          </div>
          <slot name="filter"></slot>
          <div
            @scroll=${this._scrollEvent}
            class=${classMap({
                "card-content": true,
                noactions: this.noActions && !this.hasFilter,
                "dynamic-height": !this.narrow && this.dynamicHeight,
                "narrow-content": this.narrow,
            })}
          >
            <div class="content">
              <slot></slot>
            </div>
          </div>
          ${!this.noActions
                ? html ` <div
                class=${classMap({
                    "card-actions": true,
                    "narrow-actions": this.narrow,
                })}
              >
                <slot name="actions"></slot>
              </div>`
                : ""}
        </ha-card>
      </div>
    `;
            }
            _scrollEvent(ev) {
                this.dispatchEvent(new CustomEvent("scroll", {
                    detail: {
                        target: ev.target,
                    },
                    bubbles: true,
                    composed: true,
                }));
            }
            _close() {
                this.dispatchEvent(new Event(this.secondary ? "hacs-secondary-dialog-closed" : "hacs-dialog-closed", { bubbles: true, composed: true }));
            }
            static get styles() {
                return [
                    HacsCommonStyle,
                    css `
        ha-card {
          background-color: var(
            --paper-dialog-background-color,
            var(--card-background-color)
          );
          transition: none;
        }
        .header-group {
          position: absolute;
          width: 100%;
          display: flex;
          align-items: center;
          font-size: 20px;
          height: 65px;
          background-color: var(--primary-background-color);
          border-bottom: 1px solid var(--divider-color);
          font-weight: 400;
          font-family: var(--paper-font-body1_-_font-family);
          padding: 12px 16px;
          box-sizing: border-box;
        }
        .header {
          -webkit-font-smoothing: var(
            --paper-font-headline_-_-webkit-font-smoothing
          );
          font-family: var(--paper-font-headline_-_font-family);
          font-size: var(--paper-font-headline_-_font-size);
          font-weight: var(--paper-font-headline_-_font-weight);
          letter-spacing: var(--paper-font-headline_-_letter-spacing);
          line-height: 28px;
          max-height: 26px;
          opacity: var(--dark-primary-opacity);
          overflow: hidden;
          padding: 4px;
          text-overflow: ellipsis;
        }
        .close {
          pointer-events: auto;
          color: var(--sidebar-icon-color);
          --mdc-icon-size: 32px;
        }

        .card-content {
          position: relative;
          padding: 0;
          width: 100%;
          padding: 0;
          height: calc(100% - 136px);
          overflow-y: auto;
          overflow-x: hidden;
          -webkit-overflow-scrolling: touch;
          margin-top: 65px;
        }
        ha-card.dynamic-height .noactions {
          margin-bottom: -65px;
        }
        .noactions {
          height: calc(100% - 68px);
        }
        .content {
          padding: 8px;
          height: fit-content;
        }

        ::slotted([slot="actions"]) {
          position: absolute;
          width: calc(100% - 32px);
          display: flex;
          justify-content: space-between;
          align-items: center;
          font-size: 20px;
          height: 65px;
          background-color: var(--sidebar-background-color);
          font-weight: 400;
          color: var(--sidebar-text-color);
          font-family: var(--paper-font-body1_-_font-family);
          padding: 12px 16px;
          box-sizing: border-box;
        }

        .backdrop {
          background-color: rgba(0, 0, 0, 0.75);
          height: 100%;
          position: fixed;
          z-index: 1;
          top: 0;
          left: var(--app-drawer-width);
          width: calc(100% - var(--app-drawer-width));
        }

        .narrow {
          top: 0 !important;
          left: 0 !important;
          bottom: 0 !important;
          right: 0 !important;
          height: 100% !important;
          width: 100% !important;
          max-height: 100% !important;
          max-width: 100% !important;
        }

        .dialog {
          height: 300px;
          margin: auto;
          max-height: calc(100% - 130px);
          max-width: 90%;
          min-width: 350px;
          overflow-x: hidden;
          padding-bottom: 65px;
          padding: unset;
          top: 64px;
          width: fit-content;
          z-index: 2;
        }
        .has-content {
          height: var(--hacs-dialog-height, -webkit-fill-available);
          height: var(--hacs-dialog-height, 100%);
        }

        .dynamic-height {
          height: auto;
        }
        ha-card.dynamic-height {
          padding-bottom: 65px;
        }

        .full-width {
          --app-drawer-width: 0%;
        }
      `,
                ];
            }
        };
        __decorate([
            property({ type: Boolean })
        ], HacsDialog.prototype, "noActions", void 0);
        __decorate([
            property({ type: Boolean })
        ], HacsDialog.prototype, "hasContent", void 0);
        __decorate([
            property({ type: Boolean })
        ], HacsDialog.prototype, "dynamicHeight", void 0);
        __decorate([
            property({ type: Boolean })
        ], HacsDialog.prototype, "hasFilter", void 0);
        HacsDialog = __decorate([
            customElement("hacs-dialog")
        ], HacsDialog);
        return HacsDialog;
    })();

    let HacsAboutDialog = /** @class */ (() => {
        let HacsAboutDialog = class HacsAboutDialog extends HacsDialogBase {
            render() {
                if (!this.active)
                    return html ``;
                return html `
      <hacs-dialog
        .active=${this.active}
        .narrow=${this.narrow}
        .hass=${this.hass}
        noActions
        dynamicHeight
      >
        <div slot="header">
          ${this.narrow ? "HACS" : "Home Assistant Community Store"}
        </div>
        <div class="content">
          ${markdown.html(`
**${localize("dialog_about.integration_version")}:** | ${this.configuration.version}
--|--
**${localize("dialog_about.frontend_version")}:** | ${version$1}
**${localize("common.repositories")}:** | ${this.repositories.length}
**${localize("dialog_about.installed_repositories")}:** | ${this.repositories.filter((repo) => repo.installed).length}

**${localize("dialog_about.useful_links")}:**

- [General documentation](https://hacs.xyz/)
- [Configuration](https://hacs.xyz/docs/configuration/start)
- [FAQ](https://hacs.xyz/docs/faq/what)
- [GitHub](https://github.com/hacs)
- [Discord](https://discord.gg/apgchf8)
- [Become a GitHub sponsor? ❤️](https://github.com/sponsors/ludeeus)
- [BuyMe~~Coffee~~Beer? 🍺🙈](https://buymeacoffee.com/ludeeus)

***

_Everything you find in HACS is **not** tested by Home Assistant, that includes HACS itself._
_The HACS and Home Assistant teams do not support **anything** you find here._
        `)}
        </div>
      </hacs-dialog>
    `;
            }
            static get styles() {
                return [
                    css `
        .content {
          font-size: 16px;
          padding-top: 16px;
        }
      `,
                ];
            }
        };
        HacsAboutDialog = __decorate([
            customElement("hacs-about-dialog")
        ], HacsAboutDialog);
        return HacsAboutDialog;
    })();

    async function updateLovelaceResources(hass, repository) {
        const resources = await fetchResources(hass);
        const namespace = `/hacsfiles/${repository.full_name.split("/")[1]}`;
        const url = `${namespace}/${repository.file_name}`;
        const exsisting = resources.find((resource) => resource.url.includes(namespace));
        if (exsisting && exsisting.url !== url) {
            console.debug(`Updating exsusting resource for ${namespace}`);
            updateResource(hass, {
                url,
                resource_id: exsisting.id,
                res_type: exsisting.type,
            });
        }
        else if (!resources.map((resource) => resource.url).includes(url)) {
            console.debug(`Adding ${url} to Lovelace resources`);
            createResource(hass, {
                url,
                res_type: "module",
            });
        }
    }

    let HacsUpdateDialog = /** @class */ (() => {
        let HacsUpdateDialog = class HacsUpdateDialog extends HacsDialogBase {
            constructor() {
                super(...arguments);
                this._updating = false;
                this._getRepository = memoizeOne((repositories, repository) => repositories === null || repositories === void 0 ? void 0 : repositories.find((repo) => repo.id === repository));
            }
            render() {
                if (!this.active)
                    return html ``;
                const repository = this._getRepository(this.repositories, this.repository);
                return html `
      <hacs-dialog
        .active=${this.active}
        .narrow=${this.narrow}
        .hass=${this.hass}
        dynamicHeight
      >
        <div slot="header">${localize("dialog_update.title")}</div>
        <div class="content">
          ${repository.name}
          <p>
            <b>${localize("dialog_update.installed_version")}:</b>
            ${repository.installed_version}
          </p>
          <p>
            <b>${localize("dialog_update.available_version")}:</b>
            ${repository.available_version}
          </p>
          ${!repository.can_install
                ? html `<p class="error">
                ${localize("confirm.home_assistant_version_not_correct")
                    .replace("{haversion}", this.hass.config.version)
                    .replace("{minversion}", repository.homeassistant)}
              </p>`
                : ""}
          ${repository.category === "integration"
                ? html `<p>${localize("dialog_install.restart")}</p>`
                : ""}
        </div>
        <div slot="actions">
          <mwc-button
            ?disabled=${!repository.can_install}
            @click=${this._updateRepository}
            >${this._updating
                ? html `<paper-spinner active></paper-spinner>`
                : localize("common.update")}</mwc-button
          >
          <hacs-link .url=${this._getChanglogURL()}
            ><mwc-button
              >${localize("dialog_update.changelog")}</mwc-button
            ></hacs-link
          >
          <hacs-link .url="https://github.com/${repository.full_name}"
            ><mwc-button
              >${localize("common.repository")}</mwc-button
            ></hacs-link
          >
        </div>
      </hacs-dialog>
    `;
            }
            async _updateRepository() {
                this._updating = true;
                const repository = this._getRepository(this.repositories, this.repository);
                if (repository.version_or_commit !== "commit") {
                    await repositoryInstallVersion(this.hass, repository.id, repository.available_version);
                }
                else {
                    await repositoryInstall(this.hass, repository.id);
                }
                if (repository.category === "plugin") {
                    if (this.status.lovelace_mode !== "yaml") {
                        await updateLovelaceResources(this.hass, repository);
                    }
                    window.location.reload(true);
                }
                this._updating = false;
                this.dispatchEvent(new Event("hacs-dialog-closed", { bubbles: true, composed: true }));
            }
            _getChanglogURL() {
                const repository = this._getRepository(this.repositories, this.repository);
                if (repository.version_or_commit === "commit") {
                    return `https://github.com/${repository.full_name}/compare/${repository.installed_version}...${repository.available_version}`;
                }
                return `https://github.com/${repository.full_name}/releases/${repository.available_version}`;
            }
            static get styles() {
                return [
                    css `
        .content {
          padding: 32px 8px;
        }
        .error {
          color: var(--hacs-error-color, var(--google-red-500));
        }
      `,
                ];
            }
        };
        __decorate([
            property()
        ], HacsUpdateDialog.prototype, "repository", void 0);
        __decorate([
            property()
        ], HacsUpdateDialog.prototype, "_updating", void 0);
        HacsUpdateDialog = __decorate([
            customElement("hacs-update-dialog")
        ], HacsUpdateDialog);
        return HacsUpdateDialog;
    })();

    let HacsChip = /** @class */ (() => {
        let HacsChip = class HacsChip extends LitElement {
            render() {
                return html `
      <div class="chip">
        <div class="icon"><ha-icon .icon=${this.icon}></ha-icon></div>
        <div class="value">${String(this.value) || ""}</div>
      </div>
    `;
            }
            static get styles() {
                return [
                    css `
        .chip {
          background-color: var(
            --hacs-chip-background-color,
            var(--accent-color)
          );
          height: 24px;
          color: var(--hacs-chip-text-color, var(--text-primary-color));
          max-width: fit-content;
          display: flex;
          border-radius: 50px;
          padding: 0 4px;
          box-shadow: 2px 2px 8px 1px rgba(0, 0, 0, 0.3);
        }
        .icon {
          margin: auto;
          color: var(--hacs-chip-background-color, var(--accent-color));
          height: 20px;
          width: 20px;
          line-height: 20px;
          text-align: center;

          margin-left: -2px;
          background-color: var(
            --hacs-chip-text-color,
            var(--text-primary-color)
          );
          border-radius: 50px;
        }
        .value {
          width: max-content;
          margin: auto;
          margin-left: 5px;
          margin-right: 5px;
        }
        ha-icon {
          --mdc-icon-size: 16px;
        }
      `,
                ];
            }
        };
        __decorate([
            property()
        ], HacsChip.prototype, "icon", void 0);
        __decorate([
            property()
        ], HacsChip.prototype, "value", void 0);
        HacsChip = __decorate([
            customElement("hacs-chip")
        ], HacsChip);
        return HacsChip;
    })();

    let HacsRepositoryDialog = /** @class */ (() => {
        let HacsRepositoryDialog = class HacsRepositoryDialog extends HacsDialogBase {
            constructor() {
                super(...arguments);
                this._getRepository = memoizeOne((repositories, repository) => repositories === null || repositories === void 0 ? void 0 : repositories.find((repo) => repo.id === repository));
                this._getAuthors = memoizeOne((repository) => {
                    const authors = [];
                    if (!repository.authors)
                        return authors;
                    repository.authors.forEach((author) => authors.push(author.replace("@", "")));
                    if (authors.length === 0) {
                        const author = repository.full_name.split("/")[0];
                        if ([
                            "custom-cards",
                            "custom-components",
                            "home-assistant-community-themes",
                        ].includes(author)) {
                            return authors;
                        }
                        authors.push(author);
                    }
                    return authors;
                });
            }
            shouldUpdate(changedProperties) {
                changedProperties.forEach((_oldValue, propName) => {
                    if (propName === "hass") {
                        this.sidebarDocked =
                            window.localStorage.getItem("dockedSidebar") === '"docked"';
                    }
                    if (propName === "repositories") {
                        this._repository = this._getRepository(this.repositories, this.repository);
                    }
                });
                return (changedProperties.has("sidebarDocked") ||
                    changedProperties.has("narrow") ||
                    changedProperties.has("active") ||
                    changedProperties.has("_repository"));
            }
            async firstUpdated() {
                this._repository = this._getRepository(this.repositories, this.repository);
                if (!this._repository.updated_info) {
                    await repositoryUpdate(this.hass, this._repository.id);
                    this.repositories = await getRepositories(this.hass);
                }
            }
            render() {
                var _a, _b, _c, _d, _e, _f;
                if (!this.active)
                    return html ``;
                const authors = this._getAuthors(this._repository);
                return html `
      <hacs-dialog
        .active=${this.active}
        .narrow=${this.narrow}
        .hass=${this.hass}
        .secondary=${this.secondary}
        ?hasContent=${((_b = (_a = this._repository) === null || _a === void 0 ? void 0 : _a.additional_info) === null || _b === void 0 ? void 0 : _b.length) > 100}
        ?noActions=${(_c = this._repository) === null || _c === void 0 ? void 0 : _c.installed}
        ?dynamicHeight=${((_d = this._repository) === null || _d === void 0 ? void 0 : _d.installed) &&
                ((_f = (_e = this._repository) === null || _e === void 0 ? void 0 : _e.additional_info) === null || _f === void 0 ? void 0 : _f.length) < 100}
      >
        <div slot="header">${this._repository.name || ""}</div>
        <div class="chips">
          ${this._repository.installed
                ? html `<hacs-chip
                title="${localize("dialog_info.version_installed")}"
                icon="mdi:cube"
                .value=${this._repository.installed_version}
              ></hacs-chip>`
                : ""}
          ${authors
                ? authors.map((author) => html `<hacs-link .url="https://github.com/${author}"
                  ><hacs-chip
                    title="${localize("dialog_info.author")}"
                    icon="mdi:account"
                    .value="@${author}"
                  ></hacs-chip
                ></hacs-link>`)
                : ""}
          ${this._repository.downloads
                ? html ` <hacs-chip
                title="${localize("dialog_info.downloads")}"
                icon="mdi:arrow-down-bold"
                .value=${this._repository.downloads}
              ></hacs-chip>`
                : ""}
          <hacs-chip
            title="${localize("dialog_info.stars")}"
            icon="mdi:star"
            .value=${this._repository.stars}
          ></hacs-chip>
          <hacs-link
            .url="https://github.com/${this._repository.full_name}/issues"
          >
            <hacs-chip
              title="${localize("dialog_info.open_issues")}"
              icon="mdi:exclamation-thick"
              .value=${this._repository.issues}
            ></hacs-chip
          ></hacs-link>
        </div>
        ${this._repository.updated_info
                ? markdown.html(this._repository.additional_info ||
                    localize("dialog_info.no_info"), this._repository)
                : localize("dialog_info.loading")}
        ${!this._repository.installed && this._repository.updated_info
                ? html ` <div slot="actions">
              <mwc-button @click=${this._installRepository} raised
                >${localize("dialog_info.install")}</mwc-button
              ><hacs-link
                .url="https://github.com/${this._repository.full_name}"
                ><mwc-button
                  >${localize("dialog_info.open_repo")}</mwc-button
                ></hacs-link
              >
            </div>`
                : ""}
      </hacs-dialog>
    `;
            }
            static get styles() {
                return css `
      img {
        max-width: 100%;
      }
      .chips {
        margin: 0 -4px;
        display: flex;
        padding-bottom: 8px;
      }
      hacs-chip {
        margin: 0 4px;
      }
      div.chips hacs-link {
        margin: -17px 4px;
      }
    `;
            }
            async _installRepository() {
                this.dispatchEvent(new CustomEvent("hacs-dialog-secondary", {
                    detail: {
                        type: "install",
                        repository: this._repository.id,
                    },
                    bubbles: true,
                    composed: true,
                }));
            }
        };
        __decorate([
            property()
        ], HacsRepositoryDialog.prototype, "repository", void 0);
        __decorate([
            property()
        ], HacsRepositoryDialog.prototype, "_repository", void 0);
        HacsRepositoryDialog = __decorate([
            customElement("hacs-repository-info-dialog")
        ], HacsRepositoryDialog);
        return HacsRepositoryDialog;
    })();

    let HacsCustomRepositoriesDialog = /** @class */ (() => {
        let HacsCustomRepositoriesDialog = class HacsCustomRepositoriesDialog extends HacsDialogBase {
            shouldUpdate(changedProperties) {
                changedProperties.forEach((_oldValue, propName) => {
                    if (propName === "hass") {
                        this.sidebarDocked =
                            window.localStorage.getItem("dockedSidebar") === '"docked"';
                    }
                });
                return (changedProperties.has("sidebarDocked") ||
                    changedProperties.has("narrow") ||
                    changedProperties.has("active") ||
                    changedProperties.has("_error") ||
                    changedProperties.has("repositories"));
            }
            render() {
                var _a;
                if (!this.active)
                    return html ``;
                const repositories = (_a = this.repositories) === null || _a === void 0 ? void 0 : _a.filter((repo) => repo.custom);
                return html `
      <hacs-dialog
        .active=${this.active}
        .narrow=${this.narrow}
        .hass=${this.hass}
        noActions
        dynamicHeight
      >
        <div slot="header">${localize("dialog_custom_repositories.title")}</div>
        <div class="content">
          ${this._error
                ? html `<div class="error">${this._error.message}</div>`
                : ""}
          <div class="list">
            ${repositories === null || repositories === void 0 ? void 0 : repositories.map((repo) => html `<paper-icon-item>
                ${repo.category === "integration"
                ? html `
                      <img
                        src="https://brands.home-assistant.io/_/${repo.domain}/icon.png"
                        referrerpolicy="no-referrer"
                        @error=${this._onImageError}
                        @load=${this._onImageLoad}
                      />
                    `
                : html `<ha-icon
                      icon="mdi:github-circle"
                      slot="item-icon"
                    ></ha-icon>`}
                <paper-item-body
                  @click=${() => this._showReopsitoryInfo(String(repo.id))}
                  three-line
                  >${repo.name}
                  <div secondary>${repo.description}</div>
                  <div secondary>
                    Category: ${repo.category}
                  </div></paper-item-body
                ><ha-icon-button
                  class="delete"
                  icon="mdi:delete"
                  @click=${() => this._removeRepository(repo.id)}
                ></ha-icon-button>
              </paper-icon-item>`)}
          </div>
          <div class="add">
            <input
              id="add-input"
              class="add-input"
              placeholder="${localize("dialog_custom_repositories.url_placeholder")}"
              .value=${this._inputRepository || ""}
              @input=${this._inputValueChanged}
            />
          </div>
          <div class="add-actions">
            <paper-dropdown-menu
              class="category"
              label="${localize("dialog_custom_repositories.category")}"
            >
              <paper-listbox
                id="category"
                slot="dropdown-content"
                selected="-1"
              >
                ${this.configuration.categories.map((category) => html `
                    <paper-item class="categoryitem" .category=${category}>
                      ${localize(`common.${category}`)}
                    </paper-item>
                  `)}
              </paper-listbox>
            </paper-dropdown-menu>
            <mwc-button raised @click=${this._addRepository}
              >${localize("common.add")}</mwc-button
            >
          </div>
        </div>
      </hacs-dialog>
    `;
            }
            firstUpdated() {
                this.hass.connection.subscribeEvents((msg) => (this._error = msg.data), "hacs/error");
            }
            _inputValueChanged() {
                var _a;
                this._inputRepository = (_a = this._addInput) === null || _a === void 0 ? void 0 : _a.value;
            }
            async _addRepository() {
                var _a, _b;
                this._error = undefined;
                const repository = this._inputRepository;
                const category = (_b = (_a = this._addCategory) === null || _a === void 0 ? void 0 : _a.selectedItem) === null || _b === void 0 ? void 0 : _b.category;
                if (!category) {
                    this._error = {
                        message: localize("dialog_custom_repositories.no_category"),
                    };
                    return;
                }
                if (!repository) {
                    this._error = {
                        message: localize("dialog_custom_repositories.no_repository"),
                    };
                    return;
                }
                await repositoryAdd(this.hass, repository, category);
                this.repositories = await getRepositories(this.hass);
            }
            async _removeRepository(repository) {
                this._error = undefined;
                await repositoryDelete(this.hass, repository);
                this.repositories = await getRepositories(this.hass);
            }
            _onImageLoad(ev) {
                ev.target.style.visibility = "initial";
            }
            _onImageError(ev) {
                ev.target.outerHTML = `<ha-icon
      icon="mdi:github-circle"
      slot="item-icon"
    ></ha-icon>`;
            }
            async _showReopsitoryInfo(repository) {
                this.dispatchEvent(new CustomEvent("hacs-dialog-secondary", {
                    detail: {
                        type: "repository-info",
                        repository,
                    },
                    bubbles: true,
                    composed: true,
                }));
            }
            static get styles() {
                return css `
      .content {
        width: 1024px;
        display: contents;
      }
      .list {
        margin-top: 16px;
      }
      ha-icon-button,
      ha-icon {
        color: var(--secondary-text-color);
      }
      ha-icon {
        --mdc-icon-size: 36px;
      }
      img {
        align-items: center;
        display: block;
        justify-content: center;
        margin-bottom: 16px;
        max-height: 36px;
        max-width: 36px;
        position: absolute;
      }
      .delete,
      paper-item-body {
        cursor: pointer;
      }
      .error {
        line-height: 0px;
        margin: 12px;
        color: var(--hacs-error-color, var(--google-red-500));
      }

      paper-item-body {
        width: 100%;
        min-height: var(--paper-item-body-two-line-min-height, 72px);
        display: var(--layout-vertical_-_display);
        flex-direction: var(--layout-vertical_-_flex-direction);
        justify-content: var(--layout-center-justified_-_justify-content);
      }
      paper-item-body div {
        font-size: 14px;
        color: var(--secondary-text-color);
      }
      .add {
        border-top: 1px solid var(--divider-color);
        margin-top: 32px;
      }
      .add-actions {
        justify-content: space-between;
      }
      .add,
      .add-actions {
        display: flex;
        align-items: center;
        font-size: 20px;
        height: 65px;
        background-color: var(--sidebar-background-color);
        border-bottom: 1px solid var(--divider-color);
        padding: 0 16px;
        box-sizing: border-box;
      }
      .add-input {
        width: calc(100% - 80px);
        height: 40px;
        border: 0;
        padding: 0 16px;
        font-size: initial;
        color: var(--sidebar-text-color);
        font-family: var(--paper-font-body1_-_font-family);
      }
      input:focus {
        outline-offset: 0;
        outline: 0;
      }
      input {
        background-color: var(--sidebar-background-color);
      }
      paper-dropdown-menu {
        width: 75%;
      }
    `;
            }
        };
        __decorate([
            property()
        ], HacsCustomRepositoriesDialog.prototype, "_inputRepository", void 0);
        __decorate([
            property()
        ], HacsCustomRepositoriesDialog.prototype, "_error", void 0);
        __decorate([
            query("#add-input")
        ], HacsCustomRepositoriesDialog.prototype, "_addInput", void 0);
        __decorate([
            query("#category")
        ], HacsCustomRepositoriesDialog.prototype, "_addCategory", void 0);
        HacsCustomRepositoriesDialog = __decorate([
            customElement("hacs-custom-repositories-dialog")
        ], HacsCustomRepositoriesDialog);
        return HacsCustomRepositoriesDialog;
    })();

    let HacsAddRepositoryDialog = /** @class */ (() => {
        let HacsAddRepositoryDialog = class HacsAddRepositoryDialog extends HacsDialogBase {
            constructor() {
                super(...arguments);
                this._load = 30;
                this._top = 0;
                this._searchInput = "";
                this._sortBy = "stars";
                this._repositoriesInActiveCategory = memoizeOne((repositories, categories) => repositories === null || repositories === void 0 ? void 0 : repositories.filter((repo) => {
                    var _a;
                    return !repo.installed && ((_a = sections$8 === null || sections$8 === void 0 ? void 0 : sections$8.panels.find((panel) => panel.id === this.section).categories) === null || _a === void 0 ? void 0 : _a.includes(repo.category)) &&
                        !repo.installed && (categories === null || categories === void 0 ? void 0 : categories.includes(repo.category));
                }));
                this._filterRepositories = memoizeOne((repositories, filter) => repositories === null || repositories === void 0 ? void 0 : repositories.filter((repo) => {
                    var _a, _b, _c, _d;
                    return ((_a = repo.name) === null || _a === void 0 ? void 0 : _a.toLocaleLowerCase().includes(filter)) || ((_b = repo.description) === null || _b === void 0 ? void 0 : _b.toLocaleLowerCase().includes(filter)) ||
                        repo.category.toLocaleLowerCase().includes(filter) ||
                        repo.full_name.toLocaleLowerCase().includes(filter) || ((_c = String(repo.authors)) === null || _c === void 0 ? void 0 : _c.toLocaleLowerCase().includes(filter)) || ((_d = repo.domain) === null || _d === void 0 ? void 0 : _d.toLocaleLowerCase().includes(filter));
                }));
            }
            shouldUpdate(changedProperties) {
                changedProperties.forEach((_oldValue, propName) => {
                    if (propName === "hass") {
                        this.sidebarDocked =
                            window.localStorage.getItem("dockedSidebar") === '"docked"';
                    }
                });
                return (changedProperties.has("sidebarDocked") ||
                    changedProperties.has("narrow") ||
                    changedProperties.has("active") ||
                    changedProperties.has("_searchInput") ||
                    changedProperties.has("_load") ||
                    changedProperties.has("_sortBy"));
            }
            render() {
                var _a, _b;
                this._searchInput = window.localStorage.getItem("hacs-search") || "";
                if (!this.active)
                    return html ``;
                const repositories = this._filterRepositories(this._repositoriesInActiveCategory(this.repositories, (_a = this.configuration) === null || _a === void 0 ? void 0 : _a.categories), (_b = this._searchInput) === null || _b === void 0 ? void 0 : _b.toLocaleLowerCase());
                return html `
      <hacs-dialog
        .active=${this.active}
        .narrow=${this.narrow}
        .hass=${this.hass}
        @scroll=${this._loadMore}
        noActions
        hasFilter
      >
        <div slot="header">
          ${localize("dialog_add_repo.title")}
        </div>
        <div slot="filter" class="filter">
          <hacs-search
            .input=${this._searchInput}
            @input=${this._inputValueChanged}
          ></hacs-search>
          <paper-dropdown-menu label="${localize("dialog_add_repo.sort_by")}">
            <paper-listbox slot="dropdown-content" selected="0">
              <paper-item @click=${() => (this._sortBy = "stars")}
                >${localize("store.stars")}</paper-item
              >
              <paper-item @click=${() => (this._sortBy = "name")}
                >${localize("store.name")}</paper-item
              >
              <paper-item @click=${() => (this._sortBy = "last_updated")}
                >${localize("store.last_updated")}</paper-item
              >
            </paper-listbox>
          </paper-dropdown-menu>
        </div>
        <div class=${classMap({ content: true, narrow: this.narrow })}>
          <div class=${classMap({ list: true, narrow: this.narrow })}>
            ${repositories
                .sort((a, b) => {
                if (this._sortBy === "name") {
                    return a.name.toLocaleLowerCase() < b.name.toLocaleLowerCase()
                        ? -1
                        : 1;
                }
                return a[this._sortBy] > b[this._sortBy] ? -1 : 1;
            })
                .slice(0, this._load)
                .map((repo) => html `<paper-icon-item
                  class=${classMap({ narrow: this.narrow })}
                  @click=${() => this._openInformation(repo)}
                >
                  ${repo.category === "integration"
                ? html `
                        <img
                          src="https://brands.home-assistant.io/_/${repo.domain}/icon.png"
                          referrerpolicy="no-referrer"
                          @error=${this._onImageError}
                          @load=${this._onImageLoad}
                        />
                      `
                : html `<ha-icon
                        icon="mdi:github-circle"
                        slot="item-icon"
                      ></ha-icon>`}
                  <paper-item-body two-line
                    >${repo.name}
                    <div class="category-chip">
                      <hacs-chip
                        icon="hacs:hacs"
                        .value=${localize(`common.${repo.category}`)}
                      ></hacs-chip>
                    </div>
                    <div secondary>${repo.description}</div>
                  </paper-item-body>
                </paper-icon-item>`)}
            ${repositories.length === 0
                ? html `<p>${localize("dialog_add_repo.no_match")}</p>`
                : ""}
          </div>
        </div>
      </hacs-dialog>
    `;
            }
            _loadMore(ev) {
                const top = ev.detail.target.scrollTop;
                if (top >= this._top) {
                    this._load += 1;
                }
                else {
                    this._load -= 1;
                }
                this._top = top;
            }
            _inputValueChanged(ev) {
                this._searchInput = ev.target.input;
                window.localStorage.setItem("hacs-search", this._searchInput);
            }
            _openInformation(repo) {
                this.dispatchEvent(new CustomEvent("hacs-dialog-secondary", {
                    detail: {
                        type: "repository-info",
                        repository: repo.id,
                    },
                    bubbles: true,
                    composed: true,
                }));
            }
            _onImageLoad(ev) {
                ev.target.style.visibility = "initial";
            }
            _onImageError(ev) {
                if (ev.target) {
                    ev.target.outerHTML = `<ha-icon
    icon="mdi:github-circle"
    slot="item-icon"
  ></ha-icon>`;
                }
            }
            static get styles() {
                return css `
      .content {
        width: 100%;
        margin-bottom: -65px;
      }
      .filter {
        margin-bottom: -65px;
        margin-top: 65px;
        display: flex;
      }
      .narrow {
        min-width: unset !important;
        width: 100% !important;
      }
      .list {
        margin-top: 16px;
        width: 1024px;
        max-width: 100%;
      }
      .category-chip {
        position: absolute;
        top: 8px;
        right: 8px;
      }
      ha-icon-button,
      ha-icon {
        color: var(--secondary-text-color);
      }
      ha-icon {
        --mdc-icon-size: 36px;
      }
      img {
        align-items: center;
        display: block;
        justify-content: center;
        margin-bottom: 16px;
        max-height: 36px;
        max-width: 36px;
        position: absolute;
      }

      paper-icon-item:focus {
        background-color: var(--divider-color);
      }

      paper-icon-item {
        cursor: pointer;
        padding: 2px 0;
      }

      paper-dropdown-menu {
        max-width: 30%;
        margin: 11px 4px -5px;
      }

      paper-item-body {
        width: 100%;
        min-height: var(--paper-item-body-two-line-min-height, 72px);
        display: var(--layout-vertical_-_display);
        flex-direction: var(--layout-vertical_-_flex-direction);
        justify-content: var(--layout-center-justified_-_justify-content);
      }
      paper-icon-item.narrow {
        border-bottom: 1px solid var(--divider-color);
        padding: 8px 0;
      }
      paper-item-body div {
        font-size: 14px;
        color: var(--secondary-text-color);
      }
      .add {
        border-top: 1px solid var(--divider-color);
        margin-top: 32px;
      }
      .add-actions {
        justify-content: space-between;
      }
      .add,
      .add-actions {
        display: flex;
        align-items: center;
        font-size: 20px;
        height: 65px;
        background-color: var(--sidebar-background-color);
        border-bottom: 1px solid var(--divider-color);
        padding: 0 16px;
        box-sizing: border-box;
      }
      .add-input {
        width: calc(100% - 80px);
        height: 40px;
        border: 0;
        padding: 0 16px;
        font-size: initial;
        color: var(--sidebar-text-color);
        font-family: var(--paper-font-body1_-_font-family);
      }
      input:focus {
        outline-offset: 0;
        outline: 0;
      }
      input {
        background-color: var(--sidebar-background-color);
      }
      paper-dropdown-menu {
        width: 75%;
      }
      hacs-search {
        width: 100%;
      }
    `;
            }
        };
        __decorate([
            property()
        ], HacsAddRepositoryDialog.prototype, "_load", void 0);
        __decorate([
            property()
        ], HacsAddRepositoryDialog.prototype, "_top", void 0);
        __decorate([
            property()
        ], HacsAddRepositoryDialog.prototype, "_searchInput", void 0);
        __decorate([
            property()
        ], HacsAddRepositoryDialog.prototype, "_sortBy", void 0);
        __decorate([
            property()
        ], HacsAddRepositoryDialog.prototype, "section", void 0);
        HacsAddRepositoryDialog = __decorate([
            customElement("hacs-add-repository-dialog")
        ], HacsAddRepositoryDialog);
        return HacsAddRepositoryDialog;
    })();

    let HacsInstallDialog = /** @class */ (() => {
        let HacsInstallDialog = class HacsInstallDialog extends HacsDialogBase {
            constructor() {
                super(...arguments);
                this._toggle = true;
                this._installing = false;
                this._getRepository = memoizeOne((repositories, repository) => repositories === null || repositories === void 0 ? void 0 : repositories.find((repo) => repo.id === repository));
                this._getInstallPath = memoizeOne((repository) => {
                    let path = repository.local_path;
                    if (repository.category === "theme") {
                        path = `${path}/${repository.file_name}`;
                    }
                    return path;
                });
            }
            shouldUpdate(changedProperties) {
                changedProperties.forEach((_oldValue, propName) => {
                    if (propName === "hass") {
                        this.sidebarDocked =
                            window.localStorage.getItem("dockedSidebar") === '"docked"';
                    }
                    if (propName === "repositories") {
                        this._repository = this._getRepository(this.repositories, this.repository);
                    }
                });
                return (changedProperties.has("sidebarDocked") ||
                    changedProperties.has("narrow") ||
                    changedProperties.has("active") ||
                    changedProperties.has("_toggle") ||
                    changedProperties.has("_repository") ||
                    changedProperties.has("_installing"));
            }
            async firstUpdated() {
                this._repository = this._getRepository(this.repositories, this.repository);
                if (!this._repository.updated_info) {
                    await repositoryUpdate(this.hass, this._repository.id);
                    this.repositories = await getRepositories(this.hass);
                }
                this._toggle = false;
            }
            render() {
                if (!this.active)
                    return html ``;
                const installPath = this._getInstallPath(this._repository);
                return html `
      <hacs-dialog
        .active=${this.active}
        .narrow=${this.narrow}
        .hass=${this.hass}
        .secondary=${this.secondary}
        dynamicHeight
      >
        <div slot="header">${this._repository.name}</div>
        <div class="content">
          ${this._repository.version_or_commit === "version"
                ? html `<div class="beta-container">
                  <ha-switch
                    ?disabled=${this._toggle}
                    .checked=${this._repository.beta}
                    @change=${this._toggleBeta}
                    >${localize("dialog_install.show_beta")}</ha-switch
                  >
                </div>
                <div class="version-select-container">
                  <paper-dropdown-menu
                    ?disabled=${this._toggle}
                    class="version-select-dropdown"
                    label="${localize("dialog_install.select_version")}"
                  >
                    <paper-listbox
                      id="version"
                      class="version-select-list"
                      slot="dropdown-content"
                      selected="0"
                    >
                      ${this._repository.releases.map((release) => {
                    return html `<paper-item
                          .version=${release}
                          class="version-select-item"
                          >${release}</paper-item
                        >`;
                })}
                      ${this._repository.full_name === "hacs/integration" ||
                    this._repository.hide_default_branch
                    ? ""
                    : html `
                            <paper-item
                              .version=${this._repository.default_branch}
                              class="version-select-item"
                              >${this._repository.default_branch}</paper-item
                            >
                          `}
                    </paper-listbox>
                  </paper-dropdown-menu>
                </div>`
                : ""}
          ${!this._repository.can_install
                ? html `<p class="error">
                ${localize("confirm.home_assistant_version_not_correct")
                    .replace("{haversion}", this.hass.config.version)
                    .replace("{minversion}", this._repository.homeassistant)}
              </p>`
                : ""}
          <div class="note">
            ${localize(`repository.note_installed`)}
            <code>'${installPath}'</code>
            ${this._repository.category === "plugin" &&
                this.status.lovelace_mode === "yaml"
                ? html ` <table class="lovelace">
                  <tr>
                    <td>${localize("dialog_install.url")}:</td>
                    <td>
                      <code>${this._lovelaceUrl()}</code>
                    </td>
                  </tr>
                  <tr>
                    <td>${localize("dialog_install.type")}:</td>
                    <td>
                      module
                    </td>
                  </tr>
                </table>`
                : ""}
            ${this._repository.category === "integration"
                ? html `<p>${localize("dialog_install.restart")}</p>`
                : ""}
          </div>
        </div>
        <div slot="actions">
          <mwc-button
            ?disabled=${!this._repository.can_install || this._toggle}
            @click=${this._installRepository}
            >${this._installing
                ? html `<paper-spinner active></paper-spinner>`
                : localize("common.install")}</mwc-button
          >

          <hacs-link .url="https://github.com/${this._repository.full_name}"
            ><mwc-button
              >${localize("common.repository")}</mwc-button
            ></hacs-link
          >
        </div>
      </hacs-dialog>
    `;
            }
            _lovelaceUrl() {
                var _a, _b;
                return `/hacsfiles/${(_a = this._repository) === null || _a === void 0 ? void 0 : _a.full_name.split("/")[1]}/${(_b = this._repository) === null || _b === void 0 ? void 0 : _b.file_name}`;
            }
            async _toggleBeta() {
                this._toggle = true;
                await repositoryToggleBeta(this.hass, this.repository);
                this.repositories = await getRepositories(this.hass);
                this._toggle = false;
            }
            async _installRepository() {
                var _a, _b;
                this._installing = true;
                if (this._repository.version_or_commit !== "commit") {
                    const selectedVersion = ((_b = (_a = this._version) === null || _a === void 0 ? void 0 : _a.selectedItem) === null || _b === void 0 ? void 0 : _b.version) ||
                        this._repository.available_version ||
                        this._repository.default_branch;
                    await repositoryInstallVersion(this.hass, this._repository.id, selectedVersion);
                }
                else {
                    await repositoryInstall(this.hass, this._repository.id);
                }
                if (this._repository.category === "plugin" &&
                    this.status.lovelace_mode !== "yaml") {
                    await updateLovelaceResources(this.hass, this._repository);
                }
                this._installing = false;
                if (this._repository.category === "plugin") {
                    window.location.reload(true);
                }
                this.dispatchEvent(new Event("hacs-secondary-dialog-closed", {
                    bubbles: true,
                    composed: true,
                }));
                this.dispatchEvent(new Event("hacs-dialog-closed", {
                    bubbles: true,
                    composed: true,
                }));
            }
            static get styles() {
                return [
                    css `
        .version-select-dropdown {
          width: 100%;
        }
        .content {
          padding: 32px 8px;
        }
        .note {
          margin-bottom: -32px;
          margin-top: 12px;
        }
        .lovelace {
          margin-top: 8px;
        }
        .error {
          color: var(--hacs-error-color, var(--google-red-500));
        }
        paper-menu-button {
          color: var(--secondary-text-color);
          padding: 0;
        }
        paper-item {
          cursor: pointer;
        }
        paper-item-body {
          opacity: var(--dark-primary-opacity);
        }
      `,
                ];
            }
        };
        __decorate([
            property()
        ], HacsInstallDialog.prototype, "repository", void 0);
        __decorate([
            property()
        ], HacsInstallDialog.prototype, "_repository", void 0);
        __decorate([
            property()
        ], HacsInstallDialog.prototype, "_toggle", void 0);
        __decorate([
            property()
        ], HacsInstallDialog.prototype, "_installing", void 0);
        __decorate([
            query("#version")
        ], HacsInstallDialog.prototype, "_version", void 0);
        HacsInstallDialog = __decorate([
            customElement("hacs-install-dialog")
        ], HacsInstallDialog);
        return HacsInstallDialog;
    })();

    let HacsEventDialog = /** @class */ (() => {
        let HacsEventDialog = class HacsEventDialog extends HacsDialogBase {
            render() {
                if (!this.active)
                    return html ``;
                const el = document.createElement(`hacs-${this.params.type || "generic"}-dialog`);
                el.active = true;
                el.hass = this.hass;
                el.narrow = this.narrow;
                el.configuration = this.configuration;
                el.lovelace = this.lovelace;
                el.secondary = this.secondary;
                el.repositories = this.repositories;
                el.route = this.route;
                el.status = this.status;
                if (this.params) {
                    for (let [key, value] of Object.entries(this.params)) {
                        el[key] = value;
                    }
                }
                return html `${el}`;
            }
        };
        __decorate([
            property({ attribute: false })
        ], HacsEventDialog.prototype, "params", void 0);
        HacsEventDialog = __decorate([
            customElement("hacs-event-dialog")
        ], HacsEventDialog);
        return HacsEventDialog;
    })();

    let HacsResolver = /** @class */ (() => {
        let HacsResolver = class HacsResolver extends LitElement {
            constructor() {
                super(...arguments);
                this.logger = new HacsLogger();
                this._sortRepositoriesByName = memoizeOne((repositories) => repositories.sort((a, b) => a.name.toLowerCase() > b.name.toLowerCase() ? 1 : -1));
            }
            connectedCallback() {
                super.connectedCallback();
                this.addEventListener("hacs-location-changed", (e) => this._setRoute(e));
                this.addEventListener("hacs-dialog", (e) => this._showDialog(e));
                this.addEventListener("hacs-dialog-secondary", (e) => this._showDialogSecondary(e));
            }
            async firstUpdated() {
                window.onpopstate = function () {
                    if (window.location.pathname.includes("hacs")) {
                        window.location.reload();
                    }
                };
                /* Backend event subscription */
                this.hass.connection.subscribeEvents(() => this._updateProperties(), "hacs/config");
                this.hass.connection.subscribeEvents(() => this._updateProperties(), "hacs/status");
                this.hass.connection.subscribeEvents(() => this._updateProperties(), "hacs/repository");
                this.hass.connection.subscribeEvents(() => this._updateProperties(), "lovelace_updated");
                await this._updateProperties();
            }
            async _updateProperties() {
                const [repositories, configuration, status, critical, lovelace,] = await Promise.all([
                    getRepositories(this.hass),
                    getConfiguration(this.hass),
                    getStatus(this.hass),
                    getCritical(this.hass),
                    getLovelaceConfiguration(this.hass),
                ]);
                this.configuration = configuration;
                this.status = status;
                this.critical = critical;
                this.lovelace = lovelace;
                this.configuration = configuration;
                this.repositories = this._sortRepositoriesByName(repositories);
            }
            render() {
                if (this.route.path === "" || this.route.path === "/") {
                    this.route.path = "/dashboard";
                }
                return html `${["/integrations", "/frontend", "/automation"].includes(this.route.path)
                ? html `<hacs-store-panel
            .hass=${this.hass}
            .route=${this.route}
            .narrow=${this.narrow}
            .configuration=${this.configuration}
            .lovelace=${this.lovelace}
            .repositories=${this.repositories}
            .status=${this.status}
            .section=${this.route.path.split("/")[1]}
          ></hacs-store-panel>`
                : html `<hacs-entry-panel
            .hass=${this.hass}
            .route=${this.route}
            .narrow=${this.narrow}
            .configuration=${this.configuration}
            .lovelace=${this.lovelace}
            .status=${this.status}
            .repositories=${this.repositories}
          ></hacs-entry-panel>`}
      <hacs-event-dialog
        .hass=${this.hass}
        .route=${this.route}
        .narrow=${this.narrow}
        .configuration=${this.configuration}
        .lovelace=${this.lovelace}
        .status=${this.status}
        .repositories=${this.repositories}
        id="hacs-dialog"
      ></hacs-event-dialog>
      <hacs-event-dialog
        .hass=${this.hass}
        .route=${this.route}
        .narrow=${this.narrow}
        .configuration=${this.configuration}
        .lovelace=${this.lovelace}
        .status=${this.status}
        .repositories=${this.repositories}
        id="hacs-dialog-secondary"
      ></hacs-event-dialog>`;
            }
            _showDialog(ev) {
                const dialogParams = ev.detail;
                this._hacsDialog.active = true;
                this._hacsDialog.params = dialogParams;
                this.addEventListener("hacs-dialog-closed", () => (this._hacsDialog.active = false));
            }
            _showDialogSecondary(ev) {
                const dialogParams = ev.detail;
                this._hacsDialogSecondary.active = true;
                this._hacsDialogSecondary.secondary = true;
                this._hacsDialogSecondary.params = dialogParams;
                this.addEventListener("hacs-secondary-dialog-closed", () => (this._hacsDialogSecondary.active = false));
            }
            _setRoute(ev) {
                this.route = ev.detail.route;
                navigate$1(this, this.route.prefix + this.route.path);
                this.requestUpdate();
            }
        };
        __decorate([
            property({ attribute: false })
        ], HacsResolver.prototype, "configuration", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsResolver.prototype, "critical", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsResolver.prototype, "hass", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsResolver.prototype, "lovelace", void 0);
        __decorate([
            property({ type: Boolean })
        ], HacsResolver.prototype, "narrow", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsResolver.prototype, "repositories", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsResolver.prototype, "route", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsResolver.prototype, "status", void 0);
        __decorate([
            query("#hacs-dialog")
        ], HacsResolver.prototype, "_hacsDialog", void 0);
        __decorate([
            query("#hacs-dialog-secondary")
        ], HacsResolver.prototype, "_hacsDialogSecondary", void 0);
        HacsResolver = __decorate([
            customElement("hacs-resolver")
        ], HacsResolver);
        return HacsResolver;
    })();

    let HacsFrontend = /** @class */ (() => {
        let HacsFrontend = class HacsFrontend extends LitElement {
            _setModalCSS() {
                if (document.getElementById("modal-style"))
                    return;
                var element = document.body;
                var style = document.createElement("style");
                style.id = "modal-style";
                style.innerHTML = `
      .swal-modal {
          background-color: var(--primary-background-color) !important;
          text-align: left;
      }
      .swal-text {
          color: var(--primary-text-color) !important;
      }
      .swal-button {
          background-color: transparent !important;
          color: var(--mdc-theme-primary, --primary-text-color) !important;
      }
      .swal-text:first-child {
        margin: 16px 0px 0px 8px;
      }
      .swal-button:focus {
        outline: none;
        box-shadow: none;
      }
      .swal-button:hover {
        outline: none;
        box-shadow: 0 0 0 1px #fff, 0 0 0 3px;
      }`;
                element.appendChild(style);
            }
            async connectedCallback() {
                super.connectedCallback();
                this._setModalCSS();
                load_lovelace();
                this.configuration = await getConfiguration(this.hass);
            }
            render() {
                if (!this.hass || !this.configuration)
                    return html ``;
                return html `
      <hacs-resolver
        .hass=${this.hass}
        .route=${this.route}
        .narrow=${this.narrow}
        .configuration=${this.configuration}
      ></hacs-resolver>
    `;
            }
        };
        __decorate([
            property({ attribute: false })
        ], HacsFrontend.prototype, "hass", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsFrontend.prototype, "narrow", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsFrontend.prototype, "route", void 0);
        __decorate([
            property({ attribute: false })
        ], HacsFrontend.prototype, "configuration", void 0);
        HacsFrontend = __decorate([
            customElement("hacs-frontend")
        ], HacsFrontend);
        return HacsFrontend;
    })();

}());
