# ANKML - 高效恶意软件检测库

ANKML是一个基于机器学习的恶意软件检测库，提供简单易用的API来检测文件是否为恶意软件。

## 安装

```bash
pip install ankml
```

## 快速开始

### 1. 配置服务器地址

**重要：** ANKML库需要连接到ANKML服务器来下载和更新模型。使用前必须先配置服务器地址：

```python
from ankml.config import ANKMLConfig

# 设置服务器地址
config = ANKMLConfig()
config.set('server_url', 'https://your-ankml-server.com/api')
# 或者设置为本地测试服务器
config.set('server_url', 'http://127.0.0.1:5000')
```

### 2. 基本使用

```python
from ankml.predictor import ANKPredictor

# 创建预测器实例
predictor = ANKPredictor(model_type='tall')

# 扫描单个文件
result = predictor.predict('/path/to/file.exe')
print(f"文件是否为恶意软件: {result['is_malware']}")
print(f"置信度: {result['confidence']:.2f}")
```

## 功能特性

- **多种模型支持**: 支持grande、tall、short三种不同精度的模型
- **自动模型管理**: 自动下载、缓存和更新模型
- **高效特征提取**: 快速提取文件特征用于检测
- **批量扫描**: 支持批量文件扫描
- **命令行工具**: 提供便捷的CLI接口

## API文档

### ANKPredictor类

主要的预测器类，用于恶意软件检测。

#### 构造函数

```python
ANKPredictor(model_type='tall', config_file=None)
```

参数：
- `model_type`: 模型类型，可选值：'grande'、'tall'、'short'
- `config_file`: 配置文件路径（可选）

#### 方法

##### predict(file_path)

检测单个文件是否为恶意软件。

```python
result = predictor.predict('/path/to/file.exe')
```

返回值：
```python
{
    'is_malware': bool,      # 是否为恶意软件
    'confidence': float,     # 置信度 (0.0-1.0)
    'model_type': str,       # 使用的模型类型
    'file_path': str         # 文件路径
}
```

### ModelLoader类

模型加载和管理类。

```python
from ankml.loader import ModelLoader

loader = ModelLoader()

# 下载模型
loader.download_model('tall')

# 检查模型更新
loader.check_for_updates('tall')

# 获取模型信息
info = loader.get_model_info('tall')
```

### FeatureExtractor类

文件特征提取类。

```python
from ankml.features import FeatureExtractor

extractor = FeatureExtractor()

# 提取文件特征
features = extractor.extract_features('/path/to/file.exe')
```

### 配置管理

```python
from ankml.config import ANKMLConfig

config = ANKMLConfig()

# 设置配置
config.set('server_url', 'https://your-server.com/api')
config.set('cache_dir', './my_cache')
config.set('timeout', 60)

# 获取配置
server_url = config.get('server_url')
```

## 命令行工具

ANKML提供了便捷的命令行工具：

```bash
# 扫描单个文件
ankml scan /path/to/file.exe

# 扫描多个文件
ankml scan /path/to/file1.exe /path/to/file2.exe

# 使用指定模型
ankml scan --model grande /path/to/file.exe

# 更新模型
ankml update --model tall

# 查看模型信息
ankml info --model tall
```

## 配置文件

可以创建配置文件来管理设置：

```json
{
    "server_url": "https://your-ankml-server.com/api",
    "cache_dir": "./ankml_cache",
    "timeout": 30,
    "max_retries": 3,
    "log_level": "INFO"
}
```

## 错误处理

```python
from ankml.exceptions import ANKMLError, ModelNotFoundError, NetworkError

try:
    predictor = ANKPredictor(model_type='tall')
    result = predictor.predict('/path/to/file.exe')
except ModelNotFoundError:
    print("模型未找到，请检查模型是否已下载")
except NetworkError:
    print("网络连接错误，请检查服务器地址和网络连接")
except ANKMLError as e:
    print(f"ANKML错误: {e}")
```

## 性能优化

- 使用缓存：模型和特征会自动缓存以提高性能
- 批量处理：对于大量文件，建议使用批量扫描功能
- 模型选择：根据需求选择合适的模型（grande精度最高但速度较慢，short速度最快但精度较低）

## 许可证

MIT License

## 支持

如有问题或建议，请访问我们的官网或提交Issue。