"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ComponentBuilder = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const imagebuilder = require("aws-cdk-lib/aws-imagebuilder");
const yaml = require("yaml");
const YamlValidator = require("yaml-validator");
const component_version_bump_1 = require("./component_version_bump");
const index_1 = require("./index");
const utils = require("./utils");
class ComponentBuilder {
    constructor(scope, componentDefs) {
        this.cacheDir = 'component_cache';
        this.componentDeps = componentDefs;
        this.scope = scope;
        if (!fs.existsSync(this.cacheDir)) {
            fs.mkdirSync(this.cacheDir);
        }
        this.componentDependenciesMap = this.createComponentDependenciesMap();
    }
    createComponentDependenciesMap() {
        const components = this.componentDeps.map(c => this.createComponent(c)).filter(c => c);
        return components.reduce((a, el) => ({
            ...a,
            [el.name]: el,
        }), {}); // remove the prefix
    }
    createComponent(componentConfig) {
        let componentYaml;
        switch (componentConfig.type) {
            case 'local':
                componentYaml = this.assembleComponent(componentConfig, 'local_components');
                break;
            case 'git':
                componentYaml = this.assembleComponent(componentConfig, this.cacheDir);
                break;
            case 'aws_arn':
                return new index_1.ArnComponentRef(componentConfig.arn, componentConfig.name);
            default:
                componentYaml = '';
        }
        if (componentConfig && componentYaml) {
            return new imagebuilder.CfnComponent(this.scope, `${utils.pascalize(componentConfig.name)}Component`, componentYaml);
        }
        return undefined;
    }
    assembleComponent(componentConfig, basedir) {
        let componentPath = componentConfig.name;
        let metadataPath = componentConfig.name;
        if (componentConfig.path) {
            componentPath = `${componentPath}/${componentConfig.path}`;
            metadataPath = `${metadataPath}/${componentConfig.path}`;
        }
        // Check if there is an environment specific component yaml
        if ('ENVIRONMENT' in process.env
            && process.env.ENVIRONMENT
            && fs.existsSync(path.join(basedir, componentPath, process.env.ENVIRONMENT || '', 'component.yaml'))) {
            componentPath = path.join(componentPath, process.env.ENVIRONMENT || '');
        }
        if (this.checkComponentYaml(basedir, componentPath, metadataPath)) {
            // Get metadata YAML
            const metadata = fs.readFileSync(path.join(basedir, metadataPath, 'metadata.yaml'), 'utf-8');
            const metayaml = yaml.parse(metadata);
            // Get base component.yaml
            const baseComponent = fs.readFileSync(path.join(basedir, componentPath, 'component.yaml'), 'utf-8');
            const baseYaml = yaml.parse(baseComponent);
            let assembledYaml = baseYaml;
            if (metayaml.distribution_configs) {
                const distributionconfigcomp = metayaml.distribution_configs[componentConfig.platform];
                // if there is a platform specific component yaml specified, load it and add steps to the assembled component
                if (distributionconfigcomp) {
                    const componentExtension = fs.readFileSync(path.join(basedir, componentPath, distributionconfigcomp), 'utf-8');
                    const componentExtYaml = yaml.parse(componentExtension);
                    const phase_extensions = componentExtYaml.phase_extensions;
                    // Add phase extensions to base component
                    this.addPhaseExtensions(phase_extensions, assembledYaml);
                }
            }
            this.addDependencyConstants(componentConfig, assembledYaml);
            const name = componentConfig.name;
            const componentVersionBump = new component_version_bump_1.ComponentVersionBump(this.scope, `${utils.pascalize(componentConfig.name)}VersionBump`, name);
            return {
                name: name,
                platform: metayaml.platform,
                version: componentVersionBump.nextVersion,
                data: yaml.stringify(assembledYaml),
                tags: {
                    ShortName: utils.pascalize(componentConfig.name),
                },
            };
        }
        else {
            throw `${componentConfig.name} does not validate YAML checks`;
        }
    }
    addDependencyConstants(componentConfig, assembledYaml) {
        if ('constants' in componentConfig) {
            let constantsDict = componentConfig.constants;
            let compConstants = Object.keys(constantsDict).map((k) => {
                let constantObj = {};
                constantObj[k] = {
                    type: 'string',
                    value: constantsDict[k],
                };
                return constantObj;
            });
            if ('constants' in assembledYaml) {
                // Override the keys from the component itself with the ones from the component dependencies.
                let compConstantsKeys = Object.keys(constantsDict);
                assembledYaml.constants = assembledYaml.constants.filter(function (c) {
                    // See if their is an intersection between the keys from component dependency and
                    // the ones defined in the original component.yaml file and remove those from component.yaml file.
                    return Object.keys(c).filter(d => compConstantsKeys.includes(d)).length == 0;
                });
                assembledYaml.constants.splice(assembledYaml.constants.length, 0, ...compConstants);
            }
            else {
                assembledYaml.constants = compConstants;
            }
        }
    }
    addPhaseExtensions(phase_extensions, assembledYaml) {
        for (let phase_extension of phase_extensions) {
            // For prebuild phases, copy the steps before the build step
            if (phase_extension.name === 'prebuild') {
                const buildphase = assembledYaml.phases.find((el) => el.name === 'build');
                if (buildphase) {
                    // Insert the prebuild before the build steps.
                    buildphase.steps.splice(0, 0, ...phase_extension.steps);
                }
                else {
                    // In case the original componen yaml doesn't include a build phase,
                    // copy the phase from the extension directly in the assembled YAML
                    const copied_phase = { ...phase_extension };
                    copied_phase.name = 'build';
                    assembledYaml.phases.push(copied_phase);
                }
            }
            else { // Else just copy the build phase.
                const extphase = assembledYaml.phases.find((el) => el.name === phase_extension.name);
                if (extphase) {
                    extphase.steps.push(...phase_extension.steps);
                }
            }
        }
    }
    checkComponentYaml(baseDir, componentPath, metadataPath) {
        const metadatafile = path.join(baseDir, metadataPath, 'metadata.yaml');
        const componentfile = path.join(baseDir, componentPath, 'component.yaml');
        const metadataExists = fs.existsSync(metadatafile);
        const componentYamlExists = fs.existsSync(componentfile);
        // Ignore false error alert
        // @ts-ignore
        const yamlValidator = new YamlValidator({});
        yamlValidator.validate([
            metadatafile,
            componentfile,
        ]);
        return metadataExists && componentYamlExists && yamlValidator.report() == 0;
    }
}
exports.ComponentBuilder = ComponentBuilder;
_a = JSII_RTTI_SYMBOL_1;
ComponentBuilder[_a] = { fqn: "halloumi-ami-pipelines.ComponentBuilder", version: "0.0.37" };
class PhaseExtension {
}
class Step {
}
class Inputs {
}
//# sourceMappingURL=data:application/json;base64,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