"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MicroAppsEdgeToOrigin = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/* eslint-disable @typescript-eslint/indent */
const crypto = require("crypto");
const fs_1 = require("fs");
const os = require("os");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cf = require("aws-cdk-lib/aws-cloudfront");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const lambdaNodejs = require("aws-cdk-lib/aws-lambda-nodejs");
const logs = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
/**
 * Create a new MicroApps Edge to Origin Function w/ `config.yml`
 */
class MicroAppsEdgeToOrigin extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props === undefined) {
            throw new Error('props must be set');
        }
        const { addXForwardedHostHeader = true, assetNameRoot, assetNameSuffix, originRegion, setupApiGatewayPermissions = false, signingMode = 'sign', removalPolicy, rootPathPrefix, replaceHostHeader = true, tableRulesArn, } = props;
        // Create the edge function config file from the construct options
        const edgeToOriginConfigYaml = MicroAppsEdgeToOrigin.generateEdgeToOriginConfig({
            originRegion: originRegion || aws_cdk_lib_1.Aws.REGION,
            addXForwardedHostHeader,
            replaceHostHeader,
            signingMode: signingMode === 'none' ? '' : signingMode,
            rootPathPrefix,
            ...(tableRulesArn
                ? {
                    tableName: tableRulesArn,
                }
                : {}),
        });
        //
        // Create the Edge to Origin Function
        //
        const edgeToOriginFuncProps = {
            functionName: assetNameRoot ? `${assetNameRoot}-edge-to-origin${assetNameSuffix}` : undefined,
            memorySize: 1769,
            logRetention: logs.RetentionDays.ONE_MONTH,
            runtime: lambda.Runtime.NODEJS_16_X,
            timeout: aws_cdk_lib_1.Duration.seconds(5),
            initialPolicy: [
                // This can't have a reference to the httpApi because it would mean
                // the parent stack (this stack) has to be created before the us-east-1
                // child stack for the Edge Lambda Function.
                // That's why we use a tag-based policy to allow the Edge Function
                // to invoke any API Gateway API that we apply a tag to
                // We allow the edge function to sign for all regions since
                // we may use custom closest region in the future.
                ...(setupApiGatewayPermissions
                    ? [
                        new iam.PolicyStatement({
                            actions: ['execute-api:Invoke'],
                            resources: [`arn:aws:execute-api:*:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:*/*/*/*`],
                        }),
                    ]
                    : []),
                //
                // Grant permission to invoke tagged Function URLs
                //
                new iam.PolicyStatement({
                    actions: ['lambda:InvokeFunctionUrl'],
                    resources: [`arn:aws:lambda:*:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:*`],
                    conditions: {
                        StringEquals: { 'aws:ResourceTag/microapp-managed': 'true' },
                    },
                }),
            ],
            ...(removalPolicy ? { removalPolicy } : {}),
        };
        const rootDistPath = path.join(__dirname, '..', '..', 'microapps-edge-to-origin', 'dist');
        const rootDistExists = fs_1.existsSync(path.join(rootDistPath, 'index.js'));
        const localDistPath = path.join(__dirname, 'microapps-edge-to-origin');
        const localDistExists = fs_1.existsSync(path.join(localDistPath, 'index.js'));
        if (process.env.NODE_ENV === 'test' && rootDistExists) {
            // This is for tests run under jest - Prefer root dist bundle
            // This is also for anytime when the edge function has already been bundled
            this._edgeToOriginFunction = this.createEdgeFunction(rootDistPath, edgeToOriginConfigYaml, edgeToOriginFuncProps);
        }
        else if (localDistExists) {
            // Prefer local dist above root dist if both exist (when building for distribution)
            this._edgeToOriginFunction = this.createEdgeFunction(localDistPath, edgeToOriginConfigYaml, edgeToOriginFuncProps);
        }
        else if (rootDistExists) {
            // Use local dist if it exists (when deploying from CDK in this repo)
            this._edgeToOriginFunction = this.createEdgeFunction(rootDistPath, edgeToOriginConfigYaml, edgeToOriginFuncProps);
        }
        else {
            // This is used when bundling the app and building the CDK module
            // for distribution.
            fs_1.writeFileSync(path.join(__dirname, '..', '..', 'microapps-edge-to-origin', 'config.yml'), edgeToOriginConfigYaml);
            // Copy the appFrame.html to the place where the bundling will find it
            fs_1.copyFileSync(path.join(__dirname, '..', '..', 'microapps-router', 'appFrame.html'), path.join(__dirname, '..', '..', 'microapps-edge-to-origin', 'appFrame.html'));
            // This builds the function for distribution with the CDK Construct
            // and will be used during local builds and PR builds of microapps-core
            // if the microapps-edge-to-origin function is not already bundled.
            // This will fail to deploy in any region other than us-east-1
            this._edgeToOriginFunction = new lambdaNodejs.NodejsFunction(this, 'edge-to-apigwy-func', {
                entry: path.join(__dirname, '..', '..', 'microapps-edge-to-origin', 'src', 'index.ts'),
                handler: 'handler',
                bundling: {
                    minify: true,
                    sourceMap: true,
                    commandHooks: {
                        beforeInstall: () => [],
                        beforeBundling: () => [],
                        afterBundling: (_inputDir, outputDir) => {
                            // 2022-10-02 - Note that this is ignoring the generated config
                            // file above and including the default template config file
                            return [
                                `${os.platform() === 'win32' ? 'copy' : 'cp'} ${path.join(__dirname, '..', '..', '..', 'configs', 'microapps-edge-to-origin', 'config.yml')} ${outputDir}`,
                                `${os.platform() === 'win32' ? 'copy' : 'cp'} ${path.join(__dirname, '..', '..', 'microapps-router', 'appFrame.html')} ${outputDir}`,
                            ];
                        },
                    },
                },
                ...edgeToOriginFuncProps,
            });
        }
        this._edgeToOriginLambdas = [
            {
                eventType: cf.LambdaEdgeEventType.ORIGIN_REQUEST,
                functionVersion: this._edgeToOriginFunction.currentVersion,
                includeBody: true,
            },
        ];
        // Grant access to the rules table
        if (tableRulesArn) {
            const tableRules = dynamodb.Table.fromTableName(this, 'tableRules', tableRulesArn);
            tableRules.grantReadData(this._edgeToOriginFunction);
        }
    }
    /**
     * Generate the yaml config for the edge lambda
     * @param props
     * @returns
     */
    static generateEdgeToOriginConfig(props) {
        return `originRegion: ${props.originRegion}
${props.signingMode === '' ? '' : `signingMode: ${props.signingMode}`}
addXForwardedHostHeader: ${props.addXForwardedHostHeader}
replaceHostHeader: ${props.replaceHostHeader}
${props.tableName ? `tableName: '${props.tableName}'` : ''}
${props.rootPathPrefix ? `rootPathPrefix: '${props.rootPathPrefix}'` : ''}`;
    }
    get edgeToOriginFunction() {
        return this._edgeToOriginFunction;
    }
    get edgeToOriginLambdas() {
        return this._edgeToOriginLambdas;
    }
    /**
     * Hash the stack name to make the EdgeFunction parameter name unique
     *
     * @param stack
     * @returns
     */
    hashStackName() {
        return crypto.createHash('sha1').update(aws_cdk_lib_1.Stack.of(this).stackName).digest('hex').substring(0, 8);
    }
    createEdgeFunction(distPath, edgeToOriginConfigYaml, edgeToOriginFuncProps) {
        var _b;
        fs_1.writeFileSync(path.join(distPath, 'config.yml'), edgeToOriginConfigYaml);
        // Skip the copy of appFrame.html on deployed modules
        if (!__dirname.includes('node_modules')) {
            fs_1.copyFileSync(path.join(__dirname, '..', '..', 'microapps-router', 'appFrame.html'), path.join(distPath, 'appFrame.html'));
        }
        // The exclude varying per stack name is a kludge to get the asset bundled
        // with the stack-specifc config.yml file, otherwise they all get the file
        // generated for the first instance of the construct within any stack
        // in the app.
        const code = lambda.Code.fromAsset(distPath, { exclude: [`**/${aws_cdk_lib_1.Stack.of(this)}`] });
        const stackHash = (_b = this.hashStackName()) !== null && _b !== void 0 ? _b : '';
        // EdgeFunction has a bug where it will generate the same parameter
        // name across multiple stacks in the same region if the id param is constant
        const edge = new cf.experimental.EdgeFunction(this, `edge-to-apigwy-func-${stackHash}`, {
            stackId: `microapps-edge-to-origin-${stackHash}`,
            code,
            functionName: `microapps-edge-to-origin-${stackHash}`,
            handler: 'index.handler',
            ...edgeToOriginFuncProps,
        });
        aws_cdk_lib_1.Tags.of(edge).add('Name', aws_cdk_lib_1.Stack.of(this).stackName);
        return edge;
    }
}
exports.MicroAppsEdgeToOrigin = MicroAppsEdgeToOrigin;
_a = JSII_RTTI_SYMBOL_1;
MicroAppsEdgeToOrigin[_a] = { fqn: "@pwrdrvr/microapps-cdk.MicroAppsEdgeToOrigin", version: "0.3.5-alpha.1" };
//# sourceMappingURL=data:application/json;base64,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