"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MicroAppsAPIGwy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const apigwy = require("@aws-cdk/aws-apigatewayv2-alpha");
const apigwyAuth = require("@aws-cdk/aws-apigatewayv2-authorizers-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const logs = require("aws-cdk-lib/aws-logs");
const r53 = require("aws-cdk-lib/aws-route53");
const r53targets = require("aws-cdk-lib/aws-route53-targets");
const constructs_1 = require("constructs");
/**
 * Create a new MicroApps API Gateway HTTP API endpoint, optionally
 * requiring IAM authorization
 */
class MicroAppsAPIGwy extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props === undefined) {
            throw new Error('props cannot be undefined');
        }
        if ((props.r53Zone === undefined && props.domainNameEdge !== undefined) ||
            (props.r53Zone !== undefined && props.domainNameEdge === undefined)) {
            throw new Error('If either of r53Zone or domainNameEdge are set then the other must be set');
        }
        if (props.domainNameOrigin !== undefined && props.certOrigin === undefined) {
            throw new Error('If domainNameOrigin is set then certOrigin must be set');
        }
        if ((props.domainNameEdge === undefined && props.certOrigin !== undefined) ||
            (props.domainNameEdge !== undefined && props.certOrigin === undefined)) {
            throw new Error('If either of domainNameEdge or certOrigin are set then the other must be set');
        }
        const { r53Zone, domainNameEdge, domainNameOrigin, certOrigin, removalPolicy, assetNameRoot, assetNameSuffix, rootPathPrefix, requireIAMAuthorization = true, } = props;
        // API Gateway uses the `id` string as the gateway name without
        // any randomization... we have to make sure the name is unique-ish
        const apigatewayName = assetNameRoot
            ? `${assetNameRoot}${assetNameSuffix}`
            : `${aws_cdk_lib_1.Stack.of(this).stackName}-microapps`;
        //
        // APIGateway domain names for CloudFront and origin
        //
        this._httpApi = new apigwy.HttpApi(this, 'gwy', {
            apiName: apigatewayName,
            createDefaultStage: false,
            defaultAuthorizer: requireIAMAuthorization ? new apigwyAuth.HttpIamAuthorizer() : undefined,
        });
        if (removalPolicy !== undefined) {
            this._httpApi.applyRemovalPolicy(removalPolicy);
        }
        // This allows the Lambda @ Edge function to execute this api
        aws_cdk_lib_1.Tags.of(this._httpApi).add('microapp-managed', 'true');
        // Create the stage
        const stage = new apigwy.HttpStage(this, 'stage', {
            httpApi: this._httpApi,
            autoDeploy: true,
            // If rootPathPrefix is not defined this will be the $default stage
            stageName: rootPathPrefix,
        });
        // This allows the Lambda @ Edge function to execute this api
        aws_cdk_lib_1.Tags.of(stage).add('microapp-managed', 'true');
        if (domainNameEdge !== undefined && certOrigin !== undefined) {
            // Create Custom Domains for API Gateway
            const dnAppsEdge = new apigwy.DomainName(this, 'microapps-apps-edge-dn', {
                domainName: domainNameEdge,
                certificate: certOrigin,
            });
            if (removalPolicy !== undefined) {
                dnAppsEdge.applyRemovalPolicy(removalPolicy);
            }
            // Create the edge domain name mapping for the API
            const apiMapping = new apigwy.ApiMapping(this, 'mapping', {
                api: this._httpApi,
                domainName: dnAppsEdge,
                stage,
            });
            // 2022-01-16 - CDK is still generating CloudFormation with no dependency
            // between the R53 RecordSet and the Mapping
            apiMapping.node.addDependency(dnAppsEdge);
        }
        if (domainNameOrigin !== undefined && certOrigin !== undefined) {
            this._dnAppsOrigin = new apigwy.DomainName(this, 'origin-dn', {
                domainName: domainNameOrigin,
                certificate: certOrigin,
            });
            if (removalPolicy !== undefined) {
                this._dnAppsOrigin.applyRemovalPolicy(removalPolicy);
            }
        }
        // Enable access logs on API Gateway
        const apiAccessLogs = new logs.LogGroup(this, 'logs', {
            logGroupName: apigatewayName
                ? `/aws/apigwy/${apigatewayName}`
                : `/aws/apigwy/${this.httpApi.httpApiName}`,
            retention: logs.RetentionDays.TWO_WEEKS,
        });
        if (removalPolicy !== undefined) {
            apiAccessLogs.applyRemovalPolicy(removalPolicy);
        }
        const cfnStage = stage.node.defaultChild;
        cfnStage.accessLogSettings = {
            destinationArn: apiAccessLogs.logGroupArn,
            format: JSON.stringify({
                requestId: '$context.requestId',
                userAgent: '$context.identity.userAgent',
                sourceIp: '$context.identity.sourceIp',
                requestTime: '$context.requestTime',
                requestTimeEpoch: '$context.requestTimeEpoch',
                httpMethod: '$context.httpMethod',
                path: '$context.path',
                status: '$context.status',
                protocol: '$context.protocol',
                responseLength: '$context.responseLength',
                domainName: '$context.domainName',
                authorizerError: '$context.authorizer.error',
            }),
        };
        // Create a logging role
        // Tips: https://github.com/aws/aws-cdk/issues/11100
        const apiGwyLogRole = new iam.Role(this, 'logs-role', {
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonAPIGatewayPushToCloudWatchLogs'),
            ],
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        });
        apiAccessLogs.grantWrite(apiGwyLogRole);
        //
        // Let API Gateway accept requests using domainNameOrigin
        // That is the origin URI that CloudFront uses for this gateway.
        // The gateway will refuse the traffic if it doesn't have the
        // domain name registered.
        //
        if (this._dnAppsOrigin !== undefined) {
            const mappingAppsApis = new apigwy.ApiMapping(this, 'api-map-origin', {
                api: this._httpApi,
                domainName: this._dnAppsOrigin,
                stage,
            });
            // 2022-01-16 - CDK is still generating CloudFormation with no dependency
            // between the R53 RecordSet and the Mapping
            mappingAppsApis.node.addDependency(this._dnAppsOrigin);
            if (removalPolicy !== undefined) {
                mappingAppsApis.applyRemovalPolicy(removalPolicy);
            }
        }
        //
        // Create the origin name for API Gateway
        //
        if (r53Zone !== undefined && this._dnAppsOrigin) {
            const rrAppsOrigin = new r53.ARecord(this, 'origin-arecord', {
                zone: r53Zone,
                recordName: domainNameOrigin,
                target: r53.RecordTarget.fromAlias(new r53targets.ApiGatewayv2DomainProperties(this._dnAppsOrigin.regionalDomainName, this._dnAppsOrigin.regionalHostedZoneId)),
            });
            if (removalPolicy !== undefined) {
                rrAppsOrigin.applyRemovalPolicy(removalPolicy);
            }
        }
    }
    get dnAppsOrigin() {
        return this._dnAppsOrigin;
    }
    get httpApi() {
        return this._httpApi;
    }
}
exports.MicroAppsAPIGwy = MicroAppsAPIGwy;
_a = JSII_RTTI_SYMBOL_1;
MicroAppsAPIGwy[_a] = { fqn: "@pwrdrvr/microapps-cdk.MicroAppsAPIGwy", version: "0.3.5-alpha.1" };
//# sourceMappingURL=data:application/json;base64,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