# Copyright (c) 2020, NVIDIA CORPORATION.  All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# Copyright 2017 Johns Hopkins University (Shinji Watanabe)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from dataclasses import dataclass, field
from typing import Any, Dict, List, Optional, Union

import torch


@dataclass
class Hypothesis:
    """Hypothesis class for beam search algorithms.

    Args:
        score: A float score obtained from an AbstractRNNTDecoder module's score_hypothesis method.
        y_sequence: Either a sequence of integer ids pointing to some vocabulary, or a packed torch.Tensor
            behaving in the same manner. dtype must be torch.Long in the latter case.
        dec_state: A list (or list of list) of LSTM-RNN decoder states. Can be None.
        y: (Unused) A list of torch.Tensors representing the list of hypotheses.
        lm_state: (Unused) A dictionary state cache used by an external Language Model.
        lm_scores: (Unused) Score of the external Language Model.
        tokens: (Optional) List of decoded tokens.
        text: Decoded transcript of the acoustic input.
        timestep: (Optional) List of int timesteps where tokens were predicted.
        length: (Optional) int which represents the length of the decoded tokens / text.
    """

    score: float
    y_sequence: Union[List[int], torch.Tensor]
    dec_state: Optional[Union[List[List[torch.Tensor]], List[torch.Tensor]]] = None
    y: List[torch.tensor] = None
    lm_state: Union[Dict[str, Any], List[Any]] = None
    lm_scores: torch.Tensor = None
    tokens: Optional[Union[List[int], torch.Tensor]] = None
    text: str = None
    timestep: Union[List[int], torch.Tensor] = field(default_factory=list)
    length: int = 0


@dataclass
class NBestHypotheses:
    """List of N best hypotheses

    Args:
        n_best_hypotheses: An optional list of :class:`Hypothesis` objects.
    """

    n_best_hypotheses: Optional[List[Hypothesis]]
