# -*- coding=utf-8 -*-

"""Input data module."""

from pathlib import Path
from typing import Iterator, Literal

from revsymg.index_lib import FORWARD_INT, REVERSE_INT, OrT


# ============================================================================ #
#                                     TYPES                                    #
# ============================================================================ #
# ---------------------------------------------------------------------------- #
#                                    Contigs                                   #
# ---------------------------------------------------------------------------- #
IdCT = str
"""Contig ID type."""

MultT = int
"""Multiplicity type."""

PresScoreT = float
"""Contig's presence score type."""

# ---------------------------------------------------------------------------- #
#                                     Links                                    #
# ---------------------------------------------------------------------------- #
OrStrT = Literal['-', '+']
LinkStrT = tuple[str, IdCT, OrStrT, IdCT, OrStrT]
LinkT = tuple[IdCT, IdCT, OrT, IdCT, OrT]


# ============================================================================ #
#                                   CONSTANTS                                  #
# ============================================================================ #
# ---------------------------------------------------------------------------- #
#                                    Solvers                                   #
# ---------------------------------------------------------------------------- #
SOLVER_CBC = 'CBC'
SOLVER_GUROBI = 'GUROBI'

# ---------------------------------------------------------------------------- #
#                                Default Values                                #
# ---------------------------------------------------------------------------- #
# DOC: docstrings for default values
MULT_UPB_DEF = 4
"""
"""

PRESSCORE_UPB_DEF = 1.0
OUTDEBUG_DEF = False
OUTDIR_DEF = Path('./')
INSTANCE_NAME_DEF = 'khloraascaf'

# ---------------------------------------------------------------------------- #
#                                 Contigs Links                                #
# ---------------------------------------------------------------------------- #
#
# Orientation strings
#
FORWARD_STR: OrStrT = '+'
REVERSE_STR: OrStrT = '-'

STR_ORIENT: dict[OrStrT, OrT] = {'+': FORWARD_INT, '-': REVERSE_INT}


# ============================================================================ #
#                                   FUNCTIONS                                  #
# ============================================================================ #
def read_contigs_attributes(
        contigs_attrs_path: Path) -> Iterator[tuple[IdCT, MultT, PresScoreT]]:
    """Iterate over contigs attributes.

    Parameters
    ----------
    contigs_attrs_path : Path
        Contigs attributes filepath

    Yields
    ------
    IdCT
        Contig's identifier
    MultT
        Contig's multiplicity
    PresScoreT
        Contig's presence score
    """
    with open(contigs_attrs_path, 'r', encoding='utf-8') as cmult_fin:
        for line in cmult_fin:
            split_line = line.split()
            if split_line:
                contig_id, mult_str, presscore_str = split_line
                yield contig_id, MultT(mult_str), PresScoreT(presscore_str)


def read_contigs_links_file(contigs_links_path: Path) -> Iterator[LinkT]:
    """Iterate over contigs links.

    Parameters
    ----------
    contigs_links_path : Path
        Contigs liniks filepath

    Yields
    ------
    IdCT
        Link's identifier
    IdCT
        Fisrt contig's identifier
    OrT
        First contig's orientation
    IdCT
        Second contig's identifier
    OrT
        Second contig's orientation
    """
    with open(contigs_links_path, 'r', encoding='utf-8') as clinks_fin:
        for line in clinks_fin:
            l_id, c_id, c_or_str, d_id, d_or_str = line.split()

            yield (
                l_id,
                c_id, STR_ORIENT[c_or_str],  # type: ignore
                d_id, STR_ORIENT[d_or_str],  # type: ignore
            )
