"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @see https://docs.aws.amazon.com/athena/latest/ug/data-types.html
 */
class Schema {
    /**
     * Creates a decimal type.
     *
     * TODO: Bounds
     *
     * @param precision the total number of digits
     * @param scale the number of digits in fractional part, the default is 0
     */
    static decimal(precision, scale) {
        return {
            isPrimitive: true,
            inputString: scale !== undefined ? `decimal(${precision},${scale})` : `decimal(${precision})`
        };
    }
    /**
     * Fixed length character data, with a specified length between 1 and 255.
     *
     * @param length length between 1 and 255
     */
    static char(length) {
        if (length <= 0 || length > 255) {
            throw new Error(`char length must be (inclusively) between 1 and 255, but was ${length}`);
        }
        if (length % 1 !== 0) {
            throw new Error(`char length must be a positive integer, was ${length}`);
        }
        return {
            isPrimitive: true,
            inputString: `char(${length})`
        };
    }
    /**
     * Variable length character data, with a specified length between 1 and 65535.
     *
     * @param length length between 1 and 65535.
     */
    static varchar(length) {
        if (length <= 0 || length > 65535) {
            throw new Error(`varchar length must be (inclusively) between 1 and 65535, but was ${length}`);
        }
        if (length % 1 !== 0) {
            throw new Error(`varchar length must be a positive integer, was ${length}`);
        }
        return {
            isPrimitive: true,
            inputString: `varchar(${length})`
        };
    }
    /**
     * Creates an array of some other type.
     *
     * @param itemType type contained by the array.
     */
    static array(itemType) {
        return {
            isPrimitive: false,
            inputString: `array<${itemType.inputString}>`
        };
    }
    /**
     * Creates a map of some primitive key type to some value type.
     *
     * @param keyType type of key, must be a primitive.
     * @param valueType type fo the value indexed by the key.
     */
    static map(keyType, valueType) {
        if (!keyType.isPrimitive) {
            throw new Error(`the key type of a 'map' must be a primitive, but was ${keyType.inputString}`);
        }
        return {
            isPrimitive: false,
            inputString: `map<${keyType.inputString},${valueType.inputString}>`
        };
    }
    /**
     * Creates a nested structure containing individually named and typed columns.
     *
     * @param columns the columns of the structure.
     */
    static struct(columns) {
        return {
            isPrimitive: false,
            inputString: `struct<${columns.map(column => {
                if (column.comment === undefined) {
                    return `${column.name}:${column.type.inputString}`;
                }
                else {
                    return `${column.name}:${column.type.inputString} COMMENT '${column.comment}'`;
                }
            }).join(',')}>`
        };
    }
}
exports.Schema = Schema;
Schema.BOOLEAN = {
    isPrimitive: true,
    inputString: 'boolean'
};
Schema.BINARY = {
    isPrimitive: true,
    inputString: 'binary'
};
/**
 * A 64-bit signed INTEGER in two’s complement format, with a minimum value of -2^63 and a maximum value of 2^63-1.
 */
Schema.BIG_INT = {
    isPrimitive: true,
    inputString: 'bigint'
};
Schema.DOUBLE = {
    isPrimitive: true,
    inputString: 'double'
};
Schema.FLOAT = {
    isPrimitive: true,
    inputString: 'float'
};
/**
 * A 32-bit signed INTEGER in two’s complement format, with a minimum value of -2^31 and a maximum value of 2^31-1.
 */
Schema.INTEGER = {
    isPrimitive: true,
    inputString: 'int'
};
/**
 * A 16-bit signed INTEGER in two’s complement format, with a minimum value of -2^15 and a maximum value of 2^15-1.
 */
Schema.SMALL_INT = {
    isPrimitive: true,
    inputString: 'smallint'
};
/**
 * A 8-bit signed INTEGER in two’s complement format, with a minimum value of -2^7 and a maximum value of 2^7-1
 */
Schema.TINY_INT = {
    isPrimitive: true,
    inputString: 'tinyint'
};
/**
 * Date type.
 */
Schema.DATE = {
    isPrimitive: true,
    inputString: 'date'
};
/**
 * Timestamp type (date and time).
 */
Schema.TIMESTAMP = {
    isPrimitive: true,
    inputString: 'timestamp'
};
/**
 * Arbitrary-length string type.
 */
Schema.STRING = {
    isPrimitive: true,
    inputString: 'string'
};
//# sourceMappingURL=data:application/json;base64,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