# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/10_deepzoom.ipynb (unless otherwise specified).

__all__ = ['dzi_html_template', 'dzi_js_template', 'viewer_div_template', 'callback_js_template',
           'callback_div_template', 'upload', 'make_html', 'connect_b2_bucket', 'dzp_save', 'dzi_to_js',
           'dzp_thumbnail', 'list_names', 'stack_image_json', 'viewer_div', 'callback_viewer', 'DeepZoomStore',
           'ROOTDIR', 'VIEWERSDIR']

# Cell
import pyvips
import os
import shutil
import glob
import re
from bs4 import BeautifulSoup
import sys
import time
import b2sdk.v1 as b2
import numpy as np
import json
import requests
import textwrap
import uuid

# CONSTANTS

ROOTDIR = 'deepzoom'
VIEWERSDIR = 'viewers'

# TEMPLATES

def dzi_html_template(name):
    '''Create viewer html page for single deep zoom image *name*. '''

    html = textwrap.dedent('''\
    <!DOCTYPE html>
    <html>
    <body style="width: 800px; height: 600px; background-color: snow">

    <div id="openseadragon1" style="width: 100%; height: 100%; background-color: gray"></div>

    <script src="https://cdnjs.cloudflare.com/ajax/libs/openseadragon/2.4.2/openseadragon.min.js"
        integrity="sha512-qvQYH6mPuE46uFcWLI8BdGaJpB5taX4lltbSIw5GF4iODh2xIgyz5ii1WpuzPFUknHCps0mi4mFGR44bjdZlZg=="
        crossorigin="anonymous">
    </script>

    <script type="text/javascript">
    var viewer = OpenSeadragon({{
        id: "openseadragon1",
        prefixUrl: "https://cdnjs.cloudflare.com/ajax/libs/openseadragon/2.4.2/images/",
        tileSources: "{name}.dzi",
     }});
    </script>


    </body>
    </html>
    ''').format(name=name)

    return html


def dzi_js_template(name, url, Format, Overlap, TileSize, Height, Width):
    '''Create single deep zoom image .js file.

    Created from deep zoom image data.'''

    js = textwrap.dedent('''\
    OpenSeadragon({{
        "id": "{name}",
        "prefixUrl": "https://cdnjs.cloudflare.com/ajax/libs/openseadragon/2.4.2/images/",
        "tileSources": {{
            "Image": {{
               "xmlns":    "http://schemas.microsoft.com/deepzoom/2008",
               "Url":      "{url}",
               "Format": "{Format}",
               "Overlap": "{Overlap}",
               "TileSize": "{TileSize}",
               "Size": {{
                   "Height": "{Height}",
                   "Width": "{Width}"
                }}
            }}
        }}
    }});
    ''').format(name=name, url=url, Format=Format, Overlap=Overlap,
                TileSize=TileSize, Height=Height, Width=Width)

    return js


def viewer_div_template(div_id, multi_tileSources,
                        sequenceMode='true', preserveViewport='true',
                        showReferenceStrip='true', showNavigator='true'):
    '''Create stand alone div code snippet with expanded tileSources array.'''

    div = textwrap.dedent('''\
    <div id="{div_id}" style="width: 800px; height: 600px; background-color: snow"></div>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/openseadragon/2.4.2/openseadragon.min.js"
        integrity="sha512-qvQYH6mPuE46uFcWLI8BdGaJpB5taX4lltbSIw5GF4iODh2xIgyz5ii1WpuzPFUknHCps0mi4mFGR44bjdZlZg=="
        crossorigin="anonymous">
    </script>
    <script type="text/javascript">
    var viewer = OpenSeadragon({{
        id: "{div_id}",
        prefixUrl: "https://cdnjs.cloudflare.com/ajax/libs/openseadragon/2.4.2/images/",
        tileSources: [{multi_tileSources}],
        sequenceMode: {sequenceMode},
        preserveViewport: {preserveViewport},
        showReferenceStrip: {showReferenceStrip},
        showNavigator: {showNavigator}
     }});
    </script>
    ''').format(div_id=div_id, multi_tileSources=multi_tileSources, sequenceMode=sequenceMode,
                preserveViewport=preserveViewport, showReferenceStrip=showReferenceStrip,
                showNavigator=showNavigator)

    return div


def callback_js_template(viewer_name, multi_tileSources, sequenceMode='true', preserveViewport='true',
                     showReferenceStrip='true', showNavigator='true'):
    '''Multi image .js template. '''

    viewer_js = textwrap.dedent('''\
    OpenSeadragon({{
        id: "{viewer_name}",
        prefixUrl: "https://cdnjs.cloudflare.com/ajax/libs/openseadragon/2.4.2/images/",
        tileSources: [{multi_tileSources}],
        sequenceMode: {sequenceMode},
        preserveViewport: {preserveViewport},
        showReferenceStrip: {showReferenceStrip},
        showNavigator: {showNavigator}
     }});
    ''').format(viewer_name=viewer_name, multi_tileSources=multi_tileSources,
                sequenceMode=sequenceMode, preserveViewport=preserveViewport,
                     showReferenceStrip=showReferenceStrip, showNavigator=showNavigator)

    return viewer_js

def callback_div_template(base_url, bucket_name, viewer_name):
    '''Creates JSONP callback code snippet to load multi image viewer .js file'''



    callback = textwrap.dedent('''\
    <script src="https://cdnjs.cloudflare.com/ajax/libs/openseadragon/2.4.2/openseadragon.min.js"
        integrity="sha512-qvQYH6mPuE46uFcWLI8BdGaJpB5taX4lltbSIw5GF4iODh2xIgyz5ii1WpuzPFUknHCps0mi4mFGR44bjdZlZg=="
        crossorigin="anonymous">
    </script>

    <div id="{viewer_name}" style="width: 800px; height: 400px; background-color: snow"></div>

    <script src="{base_url}{bucket_name}/{ROOTDIR}/{VIEWERSDIR}/{viewer_name}.js?callback=OpenSeadragon"></script>
    ''').format(base_url=base_url, bucket_name=bucket_name, viewer_name=viewer_name,
                ROOTDIR=ROOTDIR, VIEWERSDIR=VIEWERSDIR)

    return callback



# Functions

def upload(b2_api, bucket_name):
    '''Upload newly created files in local directory './deepzoom/' to b2 bucket.'''

    source = os.path.join('.', ROOTDIR)
    destination = f'b2://{bucket_name}/{ROOTDIR}'

    source = b2.parse_sync_folder(source, b2_api)
    destination = b2.parse_sync_folder(destination, b2_api)

    policies_manager = b2.ScanPoliciesManager(exclude_all_symlinks=True)

    synchronizer = b2.Synchronizer(
            max_workers=10,
            policies_manager=policies_manager,
            dry_run=False,
            allow_empty_source=True,
        )

    no_progress = False

    # need to run this to actually synchronize

    with b2.SyncReport(sys.stdout, no_progress) as reporter:
            synchronizer.sync_folders(
                source_folder=source,
                dest_folder=destination,
                now_millis=int(round(time.time() * 1000)),
                reporter=reporter,
            )




def make_html(name):
    '''Create standalone html viewer page for single image. '''

    html_fname = os.path.join('.', ROOTDIR, f'dzp_{name}', f'{name}_view.html')

    if not os.path.exists(html_fname):

        html = dzi_html_template(name)

        with open(html_fname, 'w') as fh:

            print(f'Saving: {html_fname}')
            fh.writelines(html)

    else:
        print(f'Not overwriting existing files: {html_fname}')




def connect_b2_bucket(application_key_id, application_key, bucket_name):
    '''Creates upload connection to your Backblaze b2 bucket.'''

    info = b2.InMemoryAccountInfo()
    b2_api = b2.B2Api(info)
    b2_api.authorize_account("production", application_key_id, application_key)

    bucket = b2_api.get_bucket_by_name(bucket_name)

    filenames = [file_info.file_name for file_info, _ in bucket.ls(show_versions=False, recursive=False)]
    urls = [b2_api.get_download_url_for_file_name(bucket_name, f) for f in filenames]

    base_url = re.split(bucket_name, urls[0])[0] # 'https://f002.backblazeb2.com/file/'

    return b2_api, base_url


def dzp_save(fname):
    '''Create local dzp_folder and use pyvips to convert image file *fname* to deepzoom pyramid. '''

    name, ext = os.path.splitext(os.path.basename(fname))
    dzp_folder = os.path.join('.', ROOTDIR, f'dzp_{name}')
    dzi_fname = os.path.join('.', ROOTDIR, f'dzp_{name}', f'{name}.dzi')
    dzi_base = os.path.join('.', ROOTDIR, f'dzp_{name}', f'{name}')

    os.makedirs(dzp_folder, exist_ok=True)

    if not os.path.exists(dzi_fname):
        v_img = pyvips.Image.new_from_file(fname)
        v_img.dzsave(dzi_base)
    else:
        print(f'Not overwriting existing files: {dzi_base}')

    return name


def dzi_to_js(name, base_url, bucket_name):
    '''Parse (xml) .dzi file and convert into (jsonp) .js file.

    '''

    url = f'{base_url}{bucket_name}/{ROOTDIR}/dzp_{name}/{name}_files/'
    dzi_fname = os.path.join('.', ROOTDIR, f'dzp_{name}', f'{name}.dzi')
    jsonp_fname = os.path.join('.', ROOTDIR, f'dzp_{name}', f'{name}.js')

    # for testing overwrite always
    #if not os.path.exists(jsonp_fname):
    if True:

        with open(dzi_fname, 'r') as fh:
            lines = fh.read()

        soup = BeautifulSoup(lines, 'xml')

        Format = soup.Image.attrs['Format']
        Overlap = soup.Image.attrs['Overlap']
        TileSize = soup.Image.attrs['TileSize']
        Height = soup.Image.Size.attrs['Height']
        Width = soup.Image.Size.attrs['Width']

        jsonp = dzi_js_template(name, url, Format, Overlap, TileSize, Height, Width)

        with open(jsonp_fname, 'w') as fh:
            print(f'Saving: {jsonp_fname}')
            fh.writelines(jsonp)

    else:
        print(f'Not overwriting existing file: {jsonp_fname}')


def dzp_thumbnail(fname, tn_height=500):
    '''Creates thumbnail file in dzp_ directory'''

    # create directory if needed
    name, ext = os.path.splitext(os.path.basename(fname))
    dzp_fname = os.path.join('.', ROOTDIR, f'dzp_{name}')
    os.makedirs(dzp_fname, exist_ok=True)

    # construct standard thumbnail filename
    thumb_fname = os.path.join('.', ROOTDIR, f'dzp_{name}', f'tn_{name}.png')

    # resize
    tn_height = 500
    v_img = pyvips.Image.new_from_file(fname)
    scale = tn_height / v_img.height
    v_thumb = v_img.resize(scale)

    # write to file
    v_thumb.write_to_file(thumb_fname)



def list_names(application_key_id, application_key, bucket_name):
    '''List image names and js urls for all uploaded images'''

    # connect
    b2_api, base_url =connect_b2_bucket(application_key_id, application_key, bucket_name)
    bucket = b2_api.get_bucket_by_name(bucket_name)

    # list file names in deepzoom bucket folder
    filenames = [file_info.file_name for file_info, _ in
                 bucket.ls(folder_to_list=ROOTDIR,
                           show_versions=False, recursive=False)]

    # extract image names
    ptrn = f'{ROOTDIR}/dzp_(?P<name>[^/]+).*'

    names = []
    for f in filenames:
        m = re.match(ptrn, f)
        if m:
            names.append(m.groups()[0])

    return names


def stack_image_json(names, application_key_id, application_key, bucket_name):
    '''Downloads and combines json data for selected images *names*.'''

    # need to (re)connect just to to get base_url for bucket
    _, base_url =connect_b2_bucket(application_key_id, application_key, bucket_name)

    # infer js urls from names
    js_urls = [f'{base_url}{bucket_name}/{ROOTDIR}/dzp_{n}/{n}.js' for n in names]

    # download and parse .js files and append into multi tileSources string
    jsonp_strings = []

    # expects a single image tileSources value
    ptrn = '.*"tileSources":(?P<tsources>.*)\s*\}\s*\);'
    repl = '\g<tsources>'

    for u in js_urls:
        # download jsonp.js file
        s = requests.get(u).content.decode()
        # extract tileSources substring
        s = re.sub(ptrn, repl, s, flags=re.DOTALL)
        # check valid json and improve formatting
        s = json.dumps(json.loads(s), indent=2)

        jsonp_strings.append(s)


    # add [] brackets for multiple image json?
    multi_tileSources = ',\n'.join(jsonp_strings)

    return multi_tileSources


def viewer_div(names, application_key_id, application_key, bucket_name,
                 sequenceMode='true', preserveViewport='true', showReferenceStrip='true',
                 showNavigator='true', div_id='auto'):
    '''Creates stand alone viewer html div snippet for selected images *names*.

    Can be pasted directly into web page. '''


    # create random hash for div id to avoid name collisons on single html page
    if div_id is 'auto':
        div_id = str(uuid.uuid4())[:8]

    # downloading and combining json for image list *names*
    multi_tileSources = stack_image_json(names, application_key_id, application_key, bucket_name)

    div = viewer_div_template(div_id, multi_tileSources, sequenceMode=sequenceMode,
                              preserveViewport=preserveViewport, showReferenceStrip=showReferenceStrip,
                              showNavigator=showNavigator)

    return div


def callback_viewer(names, viewer_name, application_key_id, application_key, bucket_name,
                 sequenceMode='true', preserveViewport='true', showReferenceStrip='true',
                 showNavigator='true', print_callback=True):
    '''Creates online callback multi image viewer .js file and prints callback code snippet.

    Composes jsonp call back script that is uploaded to bucket'''

    viewers_dir = os.path.join('.', ROOTDIR, VIEWERSDIR)
    js_fname = os.path.join('.', ROOTDIR, VIEWERSDIR, f'{viewer_name}.js')

    # downloading and combining json for image list *names*
    multi_tileSources = stack_image_json(names, application_key_id, application_key, bucket_name)

    # create multi image viewer .js file
    js = callback_js_template(viewer_name, multi_tileSources,
                              sequenceMode=sequenceMode, preserveViewport=preserveViewport,
                              showReferenceStrip=showReferenceStrip, showNavigator=showNavigator)

    # saving js viewer locally
    with open(js_fname, 'w') as fh:
        print(f'Saving: {js_fname}')
        fh.writelines(js)

    # then uploading to bucket
    b2_api, base_url = connect_b2_bucket(application_key_id, application_key, bucket_name)
    upload(b2_api, bucket_name)


    # print jsonp callback code snippet
    div = callback_div_template(base_url, bucket_name, viewer_name)

    if print_callback:
        print('Paste this callback code snippet into your html:\n')
        print(div)


# Class

class DeepZoomStore:
    '''Create DeepZoomStore object. '''

    def __init__(self, application_key_id, application_key, bucket_name):
        '''Create DeepZoomStore object for your B2 bucket'''

        # create local directory structure in current working directory
        store_dir = os.path.join('.', ROOTDIR)
        os.makedirs(store_dir, exist_ok=True)

        viewers_dir = os.path.join('.', ROOTDIR, VIEWERSDIR)
        os.makedirs(viewers_dir, exist_ok=True)

        # connect to b2 bucket
        self.application_key_id = application_key_id
        self.application_key = application_key
        self.bucket_name = bucket_name

        self.b2_api, self.base_url = connect_b2_bucket(application_key_id, application_key, bucket_name)

        # initialize image list for this session
        # why again?
        self.images = []


    def make_deepzoom(self, fname):

        # convert image file to to local deepzoom directory
        name = dzp_save(fname)
        self.images.append(name)

        # also make thumnail
        dzp_thumbnail(fname)

        # create viewer pages
        dzi_to_js(name, self.base_url, self.bucket_name)
        make_html(name, self.bucket_name)

        # and sync new files to b2 bucket
        upload(self.b2_api, self.bucket_name)



    def list_names(self):
        '''List all image names in bucket'''

        names = list_names(self.application_key_id, self.application_key, self.bucket_name)

        return names

    def make_viewer(self, names, viewer_name, sequenceMode='true', preserveViewport='true',
                        showReferenceStrip='true', showNavigator='true', print_callback=True):
        '''Creates and uploads jsonp js file for a multi image viewer that can be loaded with a
        callback query.'''


        callback_viewer(names, viewer_name,
                        self.application_key_id, self.application_key, self.bucket_name,
                        sequenceMode=sequenceMode, preserveViewport=preserveViewport,
                        showReferenceStrip=showReferenceStrip,
                        showNavigator=showNavigator,
                        print_callback=print_callback)

