"""
This module provides methods for compiling the plugin
"""


import os
import importlib

from collections import namedtuple
from imp import load_source

import bootstrap4c4d
from bootstrap4c4d.classes.fp import Either, Left, Right, pipe, encase, chain

from bootstrap4c4d.reducers.res import reduce_resource
from bootstrap4c4d.reducers.h import reduce_header
from bootstrap4c4d.reducers.str import reduce_strings

from bootstrap4c4d.render.res import render_resource
from bootstrap4c4d.render.h import render_header
from bootstrap4c4d.render.str import render_strings

from bootstrap4c4d.utilities.path import assert_directories

COMMENT_C = "// "
"""
Comment indicator for C-Type files
"""

COMMENT_PYTHON = "# "
"""
Comment indicator for python files
"""

PREFIX = "generated by bootstrap4c4d for c4d version {}".format(
    bootstrap4c4d.__version__
)
"""
Prefix to prepend to compiled files
"""


class Config(object):
    """
    This class represents a container of variables
    that are used to create the plugin
    """

    def __init__(self, path, name, destination, root_name, root, module):
        self.path = path
        self.name = name
        self.destination = destination
        self.root_name = root_name
        self.root = root
        self.module = module


def write_resource(config: Config) -> Config:
    """
    This method compiles the description to a resource file
    """
    destination_file = os.path.join(
        config.destination,
        "res/description",
        "{}.res".format(config.name)
    )

    contents = render_resource(reduce_resource(config.root))

    contents = "\n".join([COMMENT_C + PREFIX, contents])

    assert_directories(destination_file, True)

    with open(destination_file, "w") as f:
        f.write(contents)

    return config


def write_header(config: Config) -> Config:
    """
    This method compiles the description to a header file
    """
    destination_file = os.path.join(
        config.destination,
        "res/description",
        "{}.h".format(config.name)
    )

    contents = render_header(reduce_header(config.root))

    contents = "\n".join([COMMENT_C + PREFIX, contents])

    assert_directories(destination_file, True)

    with open(destination_file, "w") as f:
        f.write(contents)

    return config


def write_locales(config: Config) -> Config:
    """
    This method compiles the description to string files
    """
    strings_rendered = render_strings(reduce_strings(config.root))

    destination_files = []

    for key, contents in strings_rendered.items():
        destination_file = os.path.join(
            config.destination,
            "res",
            key,
            "description",
            "{}.str".format(config.name)
        )

        contents = "\n".join([COMMENT_C + PREFIX, contents])

        assert_directories(destination_file, True)

        with open(destination_file, "w") as f:
            f.write(contents)

        destination_files.append(destination_file)

    return config


def write_plugin(config: Config) -> Config:
    """
    This method compiles the python plugin to a cinema 4d pyp file
    """
    with open(config.path, "r") as input_file:
        lines = input_file.read().split("\n")

        lines_computed = []

        ignore_lines = False
        id_section = False

        for line in lines:
            # id section
            if line.startswith("#----begin_id_section----"):
                id_section = True

                lines_computed.append(COMMENT_PYTHON + PREFIX)

                continue

            if line.startswith("#----end_id_section----"):
                id_section = False

                continue

            if id_section:
                if line.startswith("#"):
                    continue

                if not line:
                    continue

                variables = [x.strip() for x in line.split("=")]

                if variables:
                    variable_name = variables[0]

                    lines_computed.append(
                        "{} = {}".format(
                            variable_name,
                            getattr(config.module, variable_name)
                        )
                    )

                continue

            # skip bootstrap4c4d lines
            if line.startswith("#----begin"):
                ignore_lines = True

                continue

            if line.startswith("#----end"):
                ignore_lines = False

                continue

            if not ignore_lines:
                lines_computed.append(line)

        compiled_plugin_file = os.path.join(
            config.destination, "{}.pyp".format(config.name)
        )

        assert_directories(compiled_plugin_file, True)

        with open(compiled_plugin_file, "w") as output_file:
            output_file.write("\n".join(lines_computed))

    return config


def assert_plugin_path(config: Config) -> Config:
    """
    This method asserts the existence of the python plugin
    on the filesystem

    Raises:
        Exception
    """
    if (
        os.path.isfile(config.path) and
        config.path.endswith(".py")
    ):
        return config

    raise Exception(
        "{} is not a valid python file".format(config.path)
    )


def assert_python_module(config: Config) -> Config:
    """
    This method loads the provided python plugin as a module
    and sets the module attribute on the config instance

    Raises:
        Exception
    """
    spec = importlib.util.spec_from_file_location(
        config.name,
        config.path
    )

    if spec:
        config.module = importlib.util.module_from_spec(spec)
        spec.loader.exec_module(config.module)

        return config

    raise Exception("{} is not a valid python module".format(plugin))


def assert_root_attribute(config: Config) -> Config:
    """
    This method sets the root attribute on the config instance
    """
    config.root = getattr(config.module, config.root_name)

    return config


def assert_root_attribute_type(config: Config) -> Config:
    """
    This method asserts that the root attribute
    is an instance of 'bootstrap4c4d.Description'

    Raises:
        Exception
    """
    if isinstance(config.root, bootstrap4c4d.Description):
        return config

    raise Exception((
        "plugin must define variable 'root' ",
        "of type 'bootstrap4c4d.Description'"
    ))


def assert_destination(config: Config) -> Config:
    """
    This method asserts the existence of the destination directory
    """
    assert_directories(config.destination)

    return config


# Right -> Either
assert_plugin_config = pipe([
    encase(assert_plugin_path),
    encase(assert_destination),
    encase(assert_python_module),
    encase(assert_root_attribute),
    encase(assert_root_attribute_type)
])
"""
This creates a pipe for asserting the correctness of the Config instance
"""


# Right -> Either
create_plugin = pipe([
    encase(write_resource),
    encase(write_header),
    encase(write_locales),
    encase(write_plugin)
])
"""
This creates a pipe for writing the plugin files
"""


# Right -> Either
build_plugin = pipe([
    chain(assert_plugin_config),
    chain(create_plugin)
])
"""
This creates a pipe for creating the plugin
"""
