"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const secretsmanager = require("../lib");
module.exports = {
    'default secret'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new secretsmanager.Secret(stack, 'Secret');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret', {
            GenerateSecretString: {}
        }));
        test.done();
    },
    'secret with generate secret string options'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new secretsmanager.Secret(stack, 'Secret', {
            generateSecretString: {
                excludeUppercase: true,
                passwordLength: 20
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret', {
            GenerateSecretString: {
                ExcludeUppercase: true,
                PasswordLength: 20
            }
        }));
        test.done();
    },
    'templated secret string'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new secretsmanager.Secret(stack, 'Secret', {
            generateSecretString: {
                secretStringTemplate: JSON.stringify({ username: 'username' }),
                generateStringKey: 'password'
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret', {
            GenerateSecretString: {
                SecretStringTemplate: '{"username":"username"}',
                GenerateStringKey: 'password'
            }
        }));
        test.done();
    },
    'grantRead'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const key = new kms.Key(stack, 'KMS');
        const secret = new secretsmanager.Secret(stack, 'Secret', { encryptionKey: key });
        const role = new iam.Role(stack, 'Role', { assumedBy: new iam.AccountRootPrincipal() });
        // WHEN
        secret.grantRead(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: 'secretsmanager:GetSecretValue',
                        Effect: 'Allow',
                        Resource: { Ref: 'SecretA720EF05' },
                    }]
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [{
                        Action: [
                            "kms:Create*",
                            "kms:Describe*",
                            "kms:Enable*",
                            "kms:List*",
                            "kms:Put*",
                            "kms:Update*",
                            "kms:Revoke*",
                            "kms:Disable*",
                            "kms:Get*",
                            "kms:Delete*",
                            "kms:ScheduleKeyDeletion",
                            "kms:CancelKeyDeletion",
                            "kms:GenerateDataKey",
                            "kms:TagResource",
                            "kms:UntagResource"
                        ],
                        Effect: "Allow",
                        Principal: {
                            AWS: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":iam::",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":root"
                                    ]
                                ]
                            }
                        },
                        Resource: "*"
                    }, {
                        Action: "kms:Decrypt",
                        Condition: {
                            StringEquals: {
                                "kms:ViaService": {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "secretsmanager.",
                                            {
                                                Ref: "AWS::Region"
                                            },
                                            ".amazonaws.com"
                                        ]
                                    ]
                                }
                            }
                        },
                        Effect: "Allow",
                        Principal: {
                            AWS: {
                                "Fn::GetAtt": [
                                    "Role1ABCC5F0",
                                    "Arn"
                                ]
                            }
                        },
                        Resource: "*"
                    }
                ],
                Version: "2012-10-17"
            }
        }));
        test.done();
    },
    'grantRead with version label constraint'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const key = new kms.Key(stack, 'KMS');
        const secret = new secretsmanager.Secret(stack, 'Secret', { encryptionKey: key });
        const role = new iam.Role(stack, 'Role', { assumedBy: new iam.AccountRootPrincipal() });
        // WHEN
        secret.grantRead(role, ['FOO', 'bar']);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: 'secretsmanager:GetSecretValue',
                        Effect: 'Allow',
                        Resource: { Ref: 'SecretA720EF05' },
                        Condition: {
                            'ForAnyValue:StringEquals': {
                                'secretsmanager:VersionStage': ['FOO', 'bar'],
                            },
                        },
                    }]
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [{
                        Action: [
                            "kms:Create*",
                            "kms:Describe*",
                            "kms:Enable*",
                            "kms:List*",
                            "kms:Put*",
                            "kms:Update*",
                            "kms:Revoke*",
                            "kms:Disable*",
                            "kms:Get*",
                            "kms:Delete*",
                            "kms:ScheduleKeyDeletion",
                            "kms:CancelKeyDeletion",
                            "kms:GenerateDataKey",
                            "kms:TagResource",
                            "kms:UntagResource"
                        ],
                        Effect: "Allow",
                        Principal: {
                            AWS: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":iam::",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":root"
                                    ]
                                ]
                            }
                        },
                        Resource: "*"
                    }, {
                        Action: "kms:Decrypt",
                        Condition: {
                            StringEquals: {
                                "kms:ViaService": {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "secretsmanager.",
                                            {
                                                Ref: "AWS::Region"
                                            },
                                            ".amazonaws.com"
                                        ]
                                    ]
                                }
                            }
                        },
                        Effect: "Allow",
                        Principal: {
                            AWS: {
                                "Fn::GetAtt": [
                                    "Role1ABCC5F0",
                                    "Arn"
                                ]
                            }
                        },
                        Resource: "*"
                    }
                ],
                Version: "2012-10-17"
            }
        }));
        test.done();
    },
    'secretValue'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const key = new kms.Key(stack, 'KMS');
        const secret = new secretsmanager.Secret(stack, 'Secret', { encryptionKey: key });
        // WHEN
        new cdk.CfnResource(stack, 'FakeResource', {
            type: 'CDK::Phony::Resource',
            properties: {
                value: secret.secretValue
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('CDK::Phony::Resource', {
            value: {
                'Fn::Join': ['', [
                        '{{resolve:secretsmanager:',
                        { Ref: 'SecretA720EF05' },
                        ':SecretString:::}}'
                    ]]
            }
        }));
        test.done();
    },
    'import'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const encryptionKey = new kms.Key(stack, 'KMS');
        const secretArn = 'arn::of::a::secret';
        // WHEN
        const secret = secretsmanager.Secret.fromSecretAttributes(stack, 'Secret', {
            secretArn, encryptionKey
        });
        // THEN
        test.equals(secret.secretArn, secretArn);
        test.same(secret.encryptionKey, encryptionKey);
        test.deepEqual(stack.resolve(secret.secretValue), '{{resolve:secretsmanager:arn::of::a::secret:SecretString:::}}');
        test.deepEqual(stack.resolve(secret.secretValueFromJson('password')), '{{resolve:secretsmanager:arn::of::a::secret:SecretString:password::}}');
        test.done();
    },
    'can attach a secret with attach()'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const secret = new secretsmanager.Secret(stack, 'Secret');
        // WHEN
        secret.attach({
            asSecretAttachmentTarget: () => ({
                targetId: 'target-id',
                targetType: 'target-type'
            })
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::SecretTargetAttachment', {
            SecretId: {
                Ref: 'SecretA720EF05'
            },
            TargetId: 'target-id',
            TargetType: 'target-type'
        }));
        test.done();
    },
    'throws when trying to attach a target multiple times to a secret'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const secret = new secretsmanager.Secret(stack, 'Secret');
        const target = {
            asSecretAttachmentTarget: () => ({
                targetId: 'target-id',
                targetType: 'target-type'
            })
        };
        secret.attach(target);
        // THEN
        test.throws(() => secret.attach(target), /Secret is already attached to a target/);
        test.done();
    },
    'add a rotation schedule to an attached secret'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const secret = new secretsmanager.Secret(stack, 'Secret');
        const attachedSecret = secret.attach({
            asSecretAttachmentTarget: () => ({
                targetId: 'target-id',
                targetType: 'target-type'
            })
        });
        const rotationLambda = new lambda.Function(stack, 'Lambda', {
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('export.handler = event => event;'),
            handler: 'index.handler'
        });
        // WHEN
        attachedSecret.addRotationSchedule('RotationSchedule', {
            rotationLambda
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::RotationSchedule', {
            SecretId: {
                Ref: 'SecretAttachment2E1B7C3B' // The secret returned by the attachment, not the secret itself.
            }
        }));
        test.done();
    },
    'throws when specifying secretStringTemplate but not generateStringKey'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => new secretsmanager.Secret(stack, 'Secret', {
            generateSecretString: {
                secretStringTemplate: JSON.stringify({ username: 'username' })
            }
        }), /`secretStringTemplate`.+`generateStringKey`/);
        test.done();
    },
    'throws when specifying generateStringKey but not secretStringTemplate'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => new secretsmanager.Secret(stack, 'Secret', {
            generateSecretString: {
                generateStringKey: 'password'
            }
        }), /`secretStringTemplate`.+`generateStringKey`/);
        test.done();
    },
    'equivalence of SecretValue and Secret.fromSecretAttributes'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const imported = secretsmanager.Secret.fromSecretAttributes(stack, 'Imported', { secretArn: 'my-secret-arn' }).secretValueFromJson('password');
        const value = cdk.SecretValue.secretsManager('my-secret-arn', { jsonField: 'password' });
        // THEN
        test.deepEqual(stack.resolve(imported), stack.resolve(value));
        test.done();
    },
    'can add to the resource policy of a secret'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const secret = new secretsmanager.Secret(stack, 'Secret');
        // WHEN
        secret.addToResourcePolicy(new iam.PolicyStatement({
            actions: ['secretsmanager:GetSecretValue'],
            resources: ['*'],
            principals: [new iam.ArnPrincipal('arn:aws:iam::123456789012:user/cool-user')]
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::ResourcePolicy', {
            ResourcePolicy: {
                Statement: [
                    {
                        Action: 'secretsmanager:GetSecretValue',
                        Effect: 'Allow',
                        Principal: {
                            AWS: 'arn:aws:iam::123456789012:user/cool-user'
                        },
                        Resource: '*'
                    }
                ],
                Version: '2012-10-17'
            },
            SecretId: {
                Ref: 'SecretA720EF05'
            }
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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