"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const lambda = require("@aws-cdk/aws-lambda");
const serverless = require("@aws-cdk/aws-sam");
const core_1 = require("@aws-cdk/core");
/**
 * A secret rotation serverless application.
 */
class SecretRotationApplication {
    constructor(applicationId, semanticVersion, options) {
        this.applicationId = `arn:aws:serverlessrepo:us-east-1:297356227824:applications/${applicationId}`;
        this.semanticVersion = semanticVersion;
        this.isMultiUser = options && options.isMultiUser;
    }
}
exports.SecretRotationApplication = SecretRotationApplication;
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MariaDB using the single user rotation scheme
 */
SecretRotationApplication.MARIADB_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSMariaDBRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MariaDB using the multi user rotation scheme
 */
SecretRotationApplication.MARIADB_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSMariaDBRotationMultiUser', '1.1.3', {
    isMultiUser: true
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MySQL using the single user rotation scheme
 */
SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSMySQLRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MySQL using the multi user rotation scheme
 */
SecretRotationApplication.MYSQL_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSMySQLRotationMultiUser', '1.1.3', {
    isMultiUser: true
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS Oracle using the single user rotation scheme
 */
SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSOracleRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS Oracle using the multi user rotation scheme
 */
SecretRotationApplication.ORACLE_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSOracleRotationMultiUser', '1.1.3', {
    isMultiUser: true
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS PostgreSQL using the single user rotation scheme
 */
SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSPostgreSQLRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS PostgreSQL using the multi user rotation scheme
 */
SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSPostgreSQLRotationMultiUser', '1.1.3', {
    isMultiUser: true
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS SQL Server using the single user rotation scheme
 */
SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSSQLServerRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS SQL Server using the multi user rotation scheme
 */
SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSSQLServerRotationMultiUser', '1.1.3', {
    isMultiUser: true
});
/**
 * Conducts an AWS SecretsManager secret rotation for Amazon Redshift using the single user rotation scheme
 */
SecretRotationApplication.REDSHIFT_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRedshiftRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for Amazon Redshift using the multi user rotation scheme
 */
SecretRotationApplication.REDSHIFT_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRedshiftRotationMultiUser', '1.1.3', {
    isMultiUser: true
});
/**
 * Conducts an AWS SecretsManager secret rotation for MongoDB using the single user rotation scheme
 */
SecretRotationApplication.MONGODB_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerMongoDBRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for MongoDB using the multi user rotation scheme
 */
SecretRotationApplication.MONGODB_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerMongoDBRotationMultiUser', '1.1.3', {
    isMultiUser: true
});
/**
 * Secret rotation for a service or database
 */
class SecretRotation extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.target.connections.defaultPort) {
            throw new Error('The `target` connections must have a default port range.');
        }
        if (props.application.isMultiUser && !props.masterSecret) {
            throw new Error('The `masterSecret` must be specified for application using the multi user scheme.');
        }
        const rotationFunctionName = this.node.uniqueId;
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.vpc
        });
        props.target.connections.allowDefaultPortFrom(securityGroup);
        const parameters = {
            endpoint: `https://secretsmanager.${core_1.Stack.of(this).region}.${core_1.Stack.of(this).urlSuffix}`,
            functionName: rotationFunctionName,
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds.join(','),
            vpcSecurityGroupIds: securityGroup.securityGroupId,
        };
        if (props.secret.encryptionKey) {
            parameters.kmsKeyArn = props.secret.encryptionKey.keyArn;
        }
        if (props.masterSecret) {
            parameters.masterSecretArn = props.masterSecret.secretArn;
            if (props.masterSecret.encryptionKey) {
                parameters.masterSecretKmsKeyArn = props.masterSecret.encryptionKey.keyArn;
            }
        }
        const application = new serverless.CfnApplication(this, 'Resource', {
            location: props.application,
            parameters,
        });
        // This creates a CF a dependency between the rotation schedule and the
        // serverless application. This is needed because it's the application
        // that creates the Lambda permission to invoke the function.
        // See https://docs.aws.amazon.com/secretsmanager/latest/userguide/integrating_cloudformation.html
        const rotationLambda = lambda.Function.fromFunctionArn(this, 'RotationLambda', core_1.Token.asString(application.getAtt('Outputs.RotationLambdaARN')));
        props.secret.addRotationSchedule('RotationSchedule', {
            rotationLambda,
            automaticallyAfter: props.automaticallyAfter
        });
        // Prevent secrets deletions when rotation is in place
        props.secret.denyAccountRootDelete();
        if (props.masterSecret) {
            props.masterSecret.denyAccountRootDelete();
        }
    }
}
exports.SecretRotation = SecretRotation;
//# sourceMappingURL=data:application/json;base64,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