"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_cloudformation_1 = require("@aws-cdk/aws-cloudformation");
const codebuild = require("@aws-cdk/aws-codebuild");
const codecommit = require("@aws-cdk/aws-codecommit");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const cpactions = require("../../lib");
/**
 * A test stack with a half-prepared pipeline ready to add CloudFormation actions to
 */
class TestFixture extends cdk.Stack {
    constructor() {
        super();
        this.pipeline = new codepipeline.Pipeline(this, 'Pipeline');
        this.sourceStage = this.pipeline.addStage({ stageName: 'Source' });
        this.deployStage = this.pipeline.addStage({ stageName: 'Deploy' });
        this.repo = new codecommit.Repository(this, 'MyVeryImportantRepo', {
            repositoryName: 'my-very-important-repo',
        });
        this.sourceOutput = new codepipeline.Artifact('SourceArtifact');
        const source = new cpactions.CodeCommitSourceAction({
            actionName: 'Source',
            output: this.sourceOutput,
            repository: this.repo,
        });
        this.sourceStage.addAction(source);
    }
}
module.exports = {
    'CreateChangeSetAction can be used to make a change set from a CodePipeline'(test) {
        const stack = new cdk.Stack();
        const pipeline = new codepipeline.Pipeline(stack, 'MagicPipeline');
        const changeSetExecRole = new aws_iam_1.Role(stack, 'ChangeSetRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('cloudformation.amazonaws.com'),
        });
        /** Source! */
        const repo = new codecommit.Repository(stack, 'MyVeryImportantRepo', {
            repositoryName: 'my-very-important-repo',
        });
        const sourceOutput = new codepipeline.Artifact('SourceArtifact');
        const source = new cpactions.CodeCommitSourceAction({
            actionName: 'source',
            output: sourceOutput,
            repository: repo,
            trigger: cpactions.CodeCommitTrigger.POLL,
        });
        pipeline.addStage({
            stageName: 'source',
            actions: [source]
        });
        /** Build! */
        const project = new codebuild.PipelineProject(stack, 'MyBuildProject');
        const buildOutput = new codepipeline.Artifact('OutputYo');
        const buildAction = new cpactions.CodeBuildAction({
            actionName: 'build',
            project,
            input: sourceOutput,
            outputs: [buildOutput],
        });
        pipeline.addStage({
            stageName: 'build',
            actions: [buildAction],
        });
        /** Deploy! */
        // To execute a change set - yes, you probably do need *:* 🤷‍♀️
        changeSetExecRole.addToPolicy(new aws_iam_1.PolicyStatement({ resources: ['*'], actions: ['*'] }));
        const stackName = 'BrelandsStack';
        const changeSetName = 'MyMagicalChangeSet';
        pipeline.addStage({
            stageName: 'prod',
            actions: [
                new cpactions.CloudFormationCreateReplaceChangeSetAction({
                    actionName: 'BuildChangeSetProd',
                    stackName,
                    changeSetName,
                    deploymentRole: changeSetExecRole,
                    templatePath: new codepipeline.ArtifactPath(buildOutput, 'template.yaml'),
                    templateConfiguration: new codepipeline.ArtifactPath(buildOutput, 'templateConfig.json'),
                    adminPermissions: false,
                }),
                new cpactions.CloudFormationExecuteChangeSetAction({
                    actionName: 'ExecuteChangeSetProd',
                    stackName,
                    changeSetName,
                }),
            ],
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'ArtifactStore': {
                'Location': {
                    'Ref': 'MagicPipelineArtifactsBucket212FE7BF'
                },
                'Type': 'S3'
            }, 'RoleArn': {
                'Fn::GetAtt': ['MagicPipelineRoleFB2BD6DE',
                    'Arn'
                ]
            },
            'Stages': [{
                    'Actions': [
                        {
                            'ActionTypeId': {
                                'Category': 'Source',
                                'Owner': 'AWS', 'Provider': 'CodeCommit', 'Version': '1'
                            },
                            'Configuration': {
                                'RepositoryName': {
                                    'Fn::GetAtt': [
                                        'MyVeryImportantRepo11BC3EBD',
                                        'Name'
                                    ]
                                },
                                'BranchName': 'master',
                                'PollForSourceChanges': true
                            },
                            'Name': 'source',
                            'OutputArtifacts': [
                                {
                                    'Name': 'SourceArtifact'
                                }
                            ],
                            'RunOrder': 1
                        }
                    ],
                    'Name': 'source'
                },
                {
                    'Actions': [
                        {
                            'ActionTypeId': {
                                'Category': 'Build',
                                'Owner': 'AWS',
                                'Provider': 'CodeBuild',
                                'Version': '1'
                            },
                            'Configuration': {
                                'ProjectName': {
                                    'Ref': 'MyBuildProject30DB9D6E'
                                }
                            },
                            'InputArtifacts': [
                                {
                                    'Name': 'SourceArtifact'
                                }
                            ],
                            'Name': 'build',
                            'OutputArtifacts': [
                                {
                                    'Name': 'OutputYo'
                                }
                            ],
                            'RunOrder': 1
                        }
                    ],
                    'Name': 'build'
                },
                {
                    'Actions': [
                        {
                            'ActionTypeId': {
                                'Category': 'Deploy',
                                'Owner': 'AWS',
                                'Provider': 'CloudFormation',
                                'Version': '1'
                            },
                            'Configuration': {
                                'ActionMode': 'CHANGE_SET_REPLACE',
                                'ChangeSetName': 'MyMagicalChangeSet',
                                'RoleArn': {
                                    'Fn::GetAtt': [
                                        'ChangeSetRole0BCF99E6',
                                        'Arn'
                                    ]
                                },
                                'StackName': 'BrelandsStack',
                                'TemplatePath': 'OutputYo::template.yaml',
                                'TemplateConfiguration': 'OutputYo::templateConfig.json'
                            },
                            'InputArtifacts': [{ 'Name': 'OutputYo' }],
                            'Name': 'BuildChangeSetProd',
                            'RunOrder': 1
                        },
                        {
                            'ActionTypeId': {
                                'Category': 'Deploy',
                                'Owner': 'AWS',
                                'Provider': 'CloudFormation',
                                'Version': '1'
                            },
                            'Configuration': {
                                'ActionMode': 'CHANGE_SET_EXECUTE',
                                'ChangeSetName': 'MyMagicalChangeSet'
                            },
                            'Name': 'ExecuteChangeSetProd',
                            'RunOrder': 1
                        }
                    ],
                    'Name': 'prod'
                }
            ]
        }));
        test.done();
    },
    'fullPermissions leads to admin role and full IAM capabilities with pipeline bucket+key read permissions'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            adminPermissions: true,
        }));
        const roleId = 'PipelineDeployCreateUpdateRole515CB7D4';
        // THEN: Action in Pipeline has named IAM capabilities
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                { 'Name': 'Source' /* don't care about the rest */ },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'Configuration': {
                                'Capabilities': 'CAPABILITY_NAMED_IAM',
                                'RoleArn': { 'Fn::GetAtt': [roleId, 'Arn'] },
                                'ActionMode': 'CREATE_UPDATE',
                                'StackName': 'MyStack',
                                'TemplatePath': 'SourceArtifact::template.yaml'
                            },
                            'InputArtifacts': [{ 'Name': 'SourceArtifact' }],
                            'Name': 'CreateUpdate',
                        },
                    ],
                }
            ]
        }));
        // THEN: Role is created with full permissions
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        'Action': [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                        ],
                        'Effect': 'Allow',
                    },
                    {
                        'Action': [
                            'kms:Decrypt',
                            'kms:DescribeKey',
                        ],
                        'Effect': 'Allow',
                    },
                    {
                        Action: '*',
                        Effect: 'Allow',
                        Resource: '*'
                    }
                ],
            },
            Roles: [{ Ref: roleId }]
        }));
        test.done();
    },
    'outputFileName leads to creation of output artifact'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            outputFileName: 'CreateResponse.json',
            adminPermissions: false,
        }));
        // THEN: Action has output artifacts
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                { 'Name': 'Source' /* don't care about the rest */ },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'OutputArtifacts': [{ 'Name': 'CreateUpdate_MyStack_Artifact' }],
                            'Name': 'CreateUpdate',
                        },
                    ],
                }
            ]
        }));
        test.done();
    },
    'replaceOnFailure switches action type'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            replaceOnFailure: true,
            adminPermissions: false,
        }));
        // THEN: Action has output artifacts
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                { 'Name': 'Source' /* don't care about the rest */ },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'Configuration': {
                                'ActionMode': 'REPLACE_ON_FAILURE',
                            },
                            'Name': 'CreateUpdate',
                        },
                    ],
                }
            ]
        }));
        test.done();
    },
    'parameterOverrides are serialized as a string'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            adminPermissions: false,
            parameterOverrides: {
                RepoName: stack.repo.repositoryName
            }
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                { 'Name': 'Source' /* don't care about the rest */ },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'Configuration': {
                                'ParameterOverrides': { 'Fn::Join': ['', [
                                            '{"RepoName":"',
                                            { 'Fn::GetAtt': ['MyVeryImportantRepo11BC3EBD', 'Name'] },
                                            '"}'
                                        ]] }
                            },
                            'Name': 'CreateUpdate',
                        },
                    ],
                }
            ]
        }));
        test.done();
    },
    'Action service role is passed to template'(test) {
        const stack = new TestFixture();
        const importedRole = aws_iam_1.Role.fromRoleArn(stack, 'ImportedRole', 'arn:aws:iam::000000000000:role/action-role');
        const freshRole = new aws_iam_1.Role(stack, 'FreshRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('magicservice')
        });
        stack.deployStage.addAction(new cpactions.CloudFormationExecuteChangeSetAction({
            actionName: 'ImportedRoleAction',
            role: importedRole,
            changeSetName: 'magicSet',
            stackName: 'magicStack',
        }));
        stack.deployStage.addAction(new cpactions.CloudFormationExecuteChangeSetAction({
            actionName: 'FreshRoleAction',
            role: freshRole,
            changeSetName: 'magicSet',
            stackName: 'magicStack',
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                {
                    'Name': 'Source' /* don't care about the rest */
                },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'Name': 'ImportedRoleAction',
                            'RoleArn': 'arn:aws:iam::000000000000:role/action-role'
                        },
                        {
                            'Name': 'FreshRoleAction',
                            'RoleArn': {
                                'Fn::GetAtt': [
                                    'FreshRole472F6E18',
                                    'Arn'
                                ]
                            }
                        }
                    ]
                }
            ]
        }));
        test.done();
    },
    'Single capability is passed to template'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            adminPermissions: false,
            capabilities: [
                aws_cloudformation_1.CloudFormationCapabilities.NAMED_IAM
            ]
        }));
        const roleId = 'PipelineDeployCreateUpdateRole515CB7D4';
        // THEN: Action in Pipeline has named IAM capabilities
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                { 'Name': 'Source' /* don't care about the rest */ },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'Configuration': {
                                'Capabilities': 'CAPABILITY_NAMED_IAM',
                                'RoleArn': { 'Fn::GetAtt': [roleId, 'Arn'] },
                                'ActionMode': 'CREATE_UPDATE',
                                'StackName': 'MyStack',
                                'TemplatePath': 'SourceArtifact::template.yaml'
                            },
                            'InputArtifacts': [{ 'Name': 'SourceArtifact' }],
                            'Name': 'CreateUpdate',
                        },
                    ],
                }
            ]
        }));
        test.done();
    },
    'Multiple capabilities are passed to template'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            adminPermissions: false,
            capabilities: [
                aws_cloudformation_1.CloudFormationCapabilities.NAMED_IAM,
                aws_cloudformation_1.CloudFormationCapabilities.AUTO_EXPAND
            ]
        }));
        const roleId = 'PipelineDeployCreateUpdateRole515CB7D4';
        // THEN: Action in Pipeline has named IAM and AUTOEXPAND capabilities
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                { 'Name': 'Source' /* don't care about the rest */ },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'Configuration': {
                                'Capabilities': 'CAPABILITY_NAMED_IAM,CAPABILITY_AUTO_EXPAND',
                                'RoleArn': { 'Fn::GetAtt': [roleId, 'Arn'] },
                                'ActionMode': 'CREATE_UPDATE',
                                'StackName': 'MyStack',
                                'TemplatePath': 'SourceArtifact::template.yaml'
                            },
                            'InputArtifacts': [{ 'Name': 'SourceArtifact' }],
                            'Name': 'CreateUpdate',
                        },
                    ],
                }
            ]
        }));
        test.done();
    },
    'Empty capabilities is not passed to template'(test) {
        // GIVEN
        const stack = new TestFixture();
        // WHEN
        stack.deployStage.addAction(new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'CreateUpdate',
            stackName: 'MyStack',
            templatePath: stack.sourceOutput.atPath('template.yaml'),
            adminPermissions: false,
            capabilities: [
                aws_cloudformation_1.CloudFormationCapabilities.NONE
            ]
        }));
        const roleId = 'PipelineDeployCreateUpdateRole515CB7D4';
        // THEN: Action in Pipeline has no capabilities
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                { 'Name': 'Source' /* don't care about the rest */ },
                {
                    'Name': 'Deploy',
                    'Actions': [
                        {
                            'Configuration': {
                                'RoleArn': { 'Fn::GetAtt': [roleId, 'Arn'] },
                                'ActionMode': 'CREATE_UPDATE',
                                'StackName': 'MyStack',
                                'TemplatePath': 'SourceArtifact::template.yaml'
                            },
                            'InputArtifacts': [{ 'Name': 'SourceArtifact' }],
                            'Name': 'CreateUpdate',
                        },
                    ],
                }
            ]
        }));
        test.done();
    },
    'cross-account CFN Pipeline': {
        'correctly creates the deployment Role in the other account'(test) {
            const app = new cdk.App();
            const pipelineStack = new cdk.Stack(app, 'PipelineStack', {
                env: {
                    account: '234567890123',
                    region: 'us-west-2',
                },
            });
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new cpactions.CodeCommitSourceAction({
                                actionName: 'CodeCommit',
                                repository: codecommit.Repository.fromRepositoryName(pipelineStack, 'Repo', 'RepoName'),
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Deploy',
                        actions: [
                            new cpactions.CloudFormationCreateUpdateStackAction({
                                actionName: 'CFN',
                                stackName: 'MyStack',
                                adminPermissions: true,
                                templatePath: sourceOutput.atPath('template.yaml'),
                                account: '123456789012',
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Name': 'Source',
                    },
                    {
                        'Name': 'Deploy',
                        'Actions': [
                            {
                                'Name': 'CFN',
                                'RoleArn': { 'Fn::Join': ['', ['arn:', { 'Ref': 'AWS::Partition' },
                                            ':iam::123456789012:role/pipelinestack-support-123loycfnactionrole56af64af3590f311bc50',
                                        ]],
                                },
                                'Configuration': {
                                    'RoleArn': {
                                        'Fn::Join': ['', ['arn:', { 'Ref': 'AWS::Partition' },
                                                ':iam::123456789012:role/pipelinestack-support-123fndeploymentrole4668d9b5a30ce3dc4508',
                                            ]],
                                    },
                                },
                            },
                        ],
                    },
                ],
            }));
            // the pipeline's BucketPolicy should trust both CFN roles
            assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Action': [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            'Effect': 'Allow',
                            'Principal': {
                                'AWS': {
                                    'Fn::Join': ['', ['arn:', { 'Ref': 'AWS::Partition' },
                                            ':iam::123456789012:role/pipelinestack-support-123fndeploymentrole4668d9b5a30ce3dc4508',
                                        ]],
                                },
                            },
                        },
                        {
                            'Action': [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            'Effect': 'Allow',
                            'Principal': {
                                'AWS': {
                                    'Fn::Join': ['', ['arn:', { 'Ref': 'AWS::Partition' },
                                            ':iam::123456789012:role/pipelinestack-support-123loycfnactionrole56af64af3590f311bc50',
                                        ]],
                                },
                            },
                        },
                    ],
                },
            }));
            const otherStack = app.node.findChild('cross-account-support-stack-123456789012');
            assert_1.expect(otherStack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                'RoleName': 'pipelinestack-support-123loycfnactionrole56af64af3590f311bc50',
            }));
            assert_1.expect(otherStack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                'RoleName': 'pipelinestack-support-123fndeploymentrole4668d9b5a30ce3dc4508',
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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