"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codepipeline = require("@aws-cdk/aws-codepipeline");
const targets = require("@aws-cdk/aws-events-targets");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * How should the S3 Action detect changes.
 * This is the type of the {@link S3SourceAction.trigger} property.
 */
var S3Trigger;
(function (S3Trigger) {
    /**
     * The Action will never detect changes -
     * the Pipeline it's part of will only begin a run when explicitly started.
     */
    S3Trigger["NONE"] = "None";
    /**
     * CodePipeline will poll S3 to detect changes.
     * This is the default method of detecting changes.
     */
    S3Trigger["POLL"] = "Poll";
    /**
     * CodePipeline will use CloudWatch Events to be notified of changes.
     * Note that the Bucket that the Action uses needs to be part of a CloudTrail Trail
     * for the events to be delivered.
     */
    S3Trigger["EVENTS"] = "Events";
})(S3Trigger = exports.S3Trigger || (exports.S3Trigger = {}));
/**
 * Source that is provided by a specific Amazon S3 object.
 *
 * Will trigger the pipeline as soon as the S3 object changes, but only if there is
 * a CloudTrail Trail in the account that captures the S3 event.
 */
class S3SourceAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.bucket,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'S3',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        if (props.bucketKey.length === 0) {
            throw new Error('Property bucketKey cannot be an empty string');
        }
        this.props = props;
    }
    /** The variables emitted by this action. */
    get variables() {
        return {
            versionId: this.variableExpression('VersionId'),
            eTag: this.variableExpression('ETag'),
        };
    }
    bound(_scope, stage, options) {
        if (this.props.trigger === S3Trigger.EVENTS) {
            const id = stage.pipeline.node.uniqueId + 'SourceEventRule' + this.props.bucketKey;
            if (this.props.bucket.node.tryFindChild(id)) {
                // this means a duplicate path for the same bucket - error out
                throw new Error(`S3 source action with path '${this.props.bucketKey}' is already present in the pipeline for this source bucket`);
            }
            this.props.bucket.onCloudTrailWriteObject(id, {
                target: new targets.CodePipeline(stage.pipeline),
                paths: [this.props.bucketKey]
            });
        }
        // we need to read from the source bucket...
        this.props.bucket.grantRead(options.role);
        // ...and write to the Pipeline bucket
        options.bucket.grantWrite(options.role);
        return {
            configuration: {
                S3Bucket: this.props.bucket.bucketName,
                S3ObjectKey: this.props.bucketKey,
                PollForSourceChanges: this.props.trigger && this.props.trigger === S3Trigger.POLL,
            },
        };
    }
}
exports.S3SourceAction = S3SourceAction;
//# sourceMappingURL=data:application/json;base64,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