"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codepipeline = require("@aws-cdk/aws-codepipeline");
const cdk = require("@aws-cdk/core");
const custom_action_registration_1 = require("../custom-action-registration");
class BaseJenkinsProvider extends cdk.Construct {
    constructor(scope, id, version) {
        super(scope, id);
        this.version = version || '1';
    }
}
exports.BaseJenkinsProvider = BaseJenkinsProvider;
/**
 * A class representing Jenkins providers.
 *
 * @see #import
 */
class JenkinsProvider extends BaseJenkinsProvider {
    constructor(scope, id, props) {
        super(scope, id, props.version);
        this.buildIncluded = false;
        this.testIncluded = false;
        this.providerName = props.providerName;
        this.serverUrl = props.serverUrl;
        if (props.forBuild === true) {
            this._registerBuildProvider();
        }
        if (props.forTest === true) {
            this._registerTestProvider();
        }
    }
    /**
     * Import a Jenkins provider registered either outside the CDK,
     * or in a different CDK Stack.
     *
     * @param scope the parent Construct for the new provider
     * @param id the identifier of the new provider Construct
     * @param attrs the properties used to identify the existing provider
     * @returns a new Construct representing a reference to an existing Jenkins provider
     */
    static fromJenkinsProviderAttributes(scope, id, attrs) {
        return new ImportedJenkinsProvider(scope, id, attrs);
    }
    /**
     * @internal
     */
    _registerBuildProvider() {
        if (this.buildIncluded) {
            return;
        }
        this.buildIncluded = true;
        this.registerJenkinsCustomAction('JenkinsBuildProviderResource', codepipeline.ActionCategory.BUILD);
    }
    /**
     * @internal
     */
    _registerTestProvider() {
        if (this.testIncluded) {
            return;
        }
        this.testIncluded = true;
        this.registerJenkinsCustomAction('JenkinsTestProviderResource', codepipeline.ActionCategory.TEST);
    }
    registerJenkinsCustomAction(id, category) {
        new custom_action_registration_1.CustomActionRegistration(this, id, {
            category,
            artifactBounds: exports.jenkinsArtifactsBounds,
            provider: this.providerName,
            version: this.version,
            entityUrl: appendToUrl(this.serverUrl, 'job/{Config:ProjectName}'),
            executionUrl: appendToUrl(this.serverUrl, 'job/{Config:ProjectName}/{ExternalExecutionId}'),
            actionProperties: [
                {
                    name: 'ProjectName',
                    required: true,
                    key: true,
                    queryable: true,
                },
            ],
        });
    }
}
exports.JenkinsProvider = JenkinsProvider;
class ImportedJenkinsProvider extends BaseJenkinsProvider {
    constructor(scope, id, props) {
        super(scope, id, props.version);
        this.providerName = props.providerName;
        this.serverUrl = props.serverUrl;
    }
    _registerBuildProvider() {
        // do nothing
    }
    _registerTestProvider() {
        // do nothing
    }
}
function appendToUrl(baseUrl, path) {
    return baseUrl.endsWith('/') ? baseUrl + path : `${baseUrl}/${path}`;
}
exports.jenkinsArtifactsBounds = {
    minInputs: 0,
    maxInputs: 5,
    minOutputs: 0,
    maxOutputs: 5
};
//# sourceMappingURL=data:application/json;base64,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