"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StepScalingPolicy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_autoscaling_common_1 = require("@aws-cdk/aws-autoscaling-common");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const step_scaling_action_1 = require("./step-scaling-action");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
/**
 * Define a scaling strategy which scales depending on absolute values of some metric.
 *
 * You can specify the scaling behavior for various values of the metric.
 *
 * Implemented using one or more CloudWatch alarms and Step Scaling Policies.
 *
 * @stability stable
 */
class StepScalingPolicy extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e;
        super(scope, id);
        if (props.scalingSteps.length < 2) {
            throw new Error('You must supply at least 2 intervals for autoscaling');
        }
        const adjustmentType = props.adjustmentType || step_scaling_action_1.AdjustmentType.CHANGE_IN_CAPACITY;
        const changesAreAbsolute = adjustmentType === step_scaling_action_1.AdjustmentType.EXACT_CAPACITY;
        const intervals = aws_autoscaling_common_1.normalizeIntervals(props.scalingSteps, changesAreAbsolute);
        const alarms = aws_autoscaling_common_1.findAlarmThresholds(intervals);
        if (alarms.lowerAlarmIntervalIndex !== undefined) {
            const threshold = intervals[alarms.lowerAlarmIntervalIndex].upper;
            this.lowerAction = new step_scaling_action_1.StepScalingAction(this, 'LowerPolicy', {
                adjustmentType,
                cooldown: props.cooldown,
                metricAggregationType: (_b = props.metricAggregationType) !== null && _b !== void 0 ? _b : aggregationTypeFromMetric(props.metric),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                scalingTarget: props.scalingTarget,
            });
            for (let i = alarms.lowerAlarmIntervalIndex; i >= 0; i--) {
                this.lowerAction.addAdjustment({
                    adjustment: intervals[i].change,
                    lowerBound: i !== 0 ? intervals[i].lower - threshold : undefined,
                    upperBound: intervals[i].upper - threshold,
                });
            }
            this.lowerAlarm = new cloudwatch.Alarm(this, 'LowerAlarm', {
                // Recommended by AutoScaling
                metric: props.metric,
                alarmDescription: 'Lower threshold scaling alarm',
                comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: (_c = props.evaluationPeriods) !== null && _c !== void 0 ? _c : 1,
                threshold,
            });
            this.lowerAlarm.addAlarmAction(new StepScalingAlarmAction(this.lowerAction));
        }
        if (alarms.upperAlarmIntervalIndex !== undefined) {
            const threshold = intervals[alarms.upperAlarmIntervalIndex].lower;
            this.upperAction = new step_scaling_action_1.StepScalingAction(this, 'UpperPolicy', {
                adjustmentType,
                cooldown: props.cooldown,
                metricAggregationType: (_d = props.metricAggregationType) !== null && _d !== void 0 ? _d : aggregationTypeFromMetric(props.metric),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                scalingTarget: props.scalingTarget,
            });
            for (let i = alarms.upperAlarmIntervalIndex; i < intervals.length; i++) {
                this.upperAction.addAdjustment({
                    adjustment: intervals[i].change,
                    lowerBound: intervals[i].lower - threshold,
                    upperBound: i !== intervals.length - 1 ? intervals[i].upper - threshold : undefined,
                });
            }
            this.upperAlarm = new cloudwatch.Alarm(this, 'UpperAlarm', {
                // Recommended by AutoScaling
                metric: props.metric,
                alarmDescription: 'Upper threshold scaling alarm',
                comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: (_e = props.evaluationPeriods) !== null && _e !== void 0 ? _e : 1,
                threshold,
            });
            this.upperAlarm.addAlarmAction(new StepScalingAlarmAction(this.upperAction));
        }
    }
}
exports.StepScalingPolicy = StepScalingPolicy;
_a = JSII_RTTI_SYMBOL_1;
StepScalingPolicy[_a] = { fqn: "@aws-cdk/aws-applicationautoscaling.StepScalingPolicy", version: "1.106.1" };
function aggregationTypeFromMetric(metric) {
    var _b;
    const statistic = (_b = metric.toMetricConfig().metricStat) === null || _b === void 0 ? void 0 : _b.statistic;
    if (statistic == null) {
        return undefined;
    } // Math expression, don't know aggregation, leave default
    switch (statistic) {
        case 'Average':
            return step_scaling_action_1.MetricAggregationType.AVERAGE;
        case 'Minimum':
            return step_scaling_action_1.MetricAggregationType.MINIMUM;
        case 'Maximum':
            return step_scaling_action_1.MetricAggregationType.MAXIMUM;
        default:
            return step_scaling_action_1.MetricAggregationType.AVERAGE;
    }
}
/**
 * Use a StepScalingAction as an Alarm Action
 *
 * This class is here and not in aws-cloudwatch-actions because this library
 * needs to use the class, and otherwise we'd have a circular dependency:
 *
 * aws-autoscaling -> aws-cloudwatch-actions (for using the Action)
 * aws-cloudwatch-actions -> aws-autoscaling (for the definition of IStepScalingAction)
 */
class StepScalingAlarmAction {
    constructor(stepScalingAction) {
        this.stepScalingAction = stepScalingAction;
    }
    bind(_scope, _alarm) {
        return { alarmActionArn: this.stepScalingAction.scalingPolicyArn };
    }
}
//# sourceMappingURL=data:application/json;base64,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