# AUTOGENERATED! DO NOT EDIT! File to edit: Widgets.ipynb (unless otherwise specified).

__all__ = ['css_style', 'dark_colors', 'light_colors', 'get_files_gui', 'get_input_gui', 'click_data', 'tabulate_data',
           'save_data', 'color_toggle', 'generate_summary', 'VasprunApp']

# Cell
import os
import json
from time import sleep

# Widgets Imports
from IPython.display import display, Markdown
import ipywidgets as ipw
from ipywidgets import Layout,Label,Button, Box,HBox,VBox, Dropdown, Text
from ipywidgets.embed import embed_minimal_html, dependency_state

# More exports
import numpy as np
import pandas as pd
import plotly.graph_objects as go

# Inside packages import to work both with package and jupyter notebook.
try:
    from . import g_utils as gu
    from . import vr_parser as vp
    from . import i_plots as ip
except:
    import pivotpy.g_utils as gu
    import pivotpy.vr_parser as vp
    import pivotpy.i_plots as ip

# Cell
def css_style(colors_dict):
    """Return style based on colors_dict available as pp.light_colors, pp.dark_colors etc"""
    return """<style>
    .widget-text input {{
        background-color: {input_bg} !important;
        border-radius:20px !important;
        padding: 0px 10px 0px 10px !important;
        border: 1px solid {input_border} !important;
        color: {input_fg} !important;
        }}
    .widget-text input:focus {{
        border: 1px solid {input_hover} !important;
        }}
    .widget-text input:hover {{
        border: 1px solid {input_hover} !important;
        }}
    .widget-dropdown > select {{
        background-color: {dd_select_bg} !important;
        border:none !important;
        border-bottom: 1px solid {input_hover} !important;
        box-shadow: inset 0 -20px 10px -20px {input_hover};
        color: {dd_select_fg} !important;
    }}
    .widget-dropdown > select:hover {{
        background-color: {dd_hover} !important;
    }}
    .widget-dropdown > select > option {{
        color: {dd_opt_fg} !important;
        background-color: {dd_opt_bg} !important;
    }}
    .widget-dropdown > select > option:focus {{
        background-color: {dd_focus_bg} !important;
    }}
    .widget-label {{
        color: {text} !important;
    }}
    .widget-html {{
        color: {text} !important;
    }}
    .widget-box {{
        background-color: {box_bg} !important;
        border-radius:5px !important;
        padding:1px !important;
        border: 1px solid {box_border} !important;
        box-shadow: 1px 1px 1px 1px {box_border} !important;
    }}
    .borderless {{
        border: 1px solid transparent !important;
        box-shadow: none !important;
        border-radius: 4px !important;
        margin:4px !important;
    }}
    .marginless {{
        margin: 0px !important;
        border-radius: 0px !important;
    }}
    .output {{
        color: {text} !important;
        background-color: inherit !important;
    }}
    .widget-tab {{
        background-color: {tab_bg} !important;
        border: none !important;
        box-shadow: 1px 1px 1px 1px {tab_shadow} !important;
        padding: 0px 2px 2px 2px !important;
    }}
    .widget-tab-contents, .widget-tab > .widget-tab-contents {{
        width: 100%;
        box-sizing: border-box;
        margin: 0px !important;
        padding: 0px !important;
        flex-grow: 1;
        overflow: auto;
        border: none !important;
        background-color: {tab_bg} !important;
    }}
    .widget-tab > .p-TabBar .p-TabBar-tab {{
        background-color:{tab_bg} !important;
        border: none !important;
        color: {tab_fg} !important;
        font-weight: bold !important;
        font-size: 16px !important;
        font-family: "Times","serif" !important;
        text-align: center !important;
    }}
    table {{
        color: {table_fg} !important;
        }}
    tr:nth-child(odd) {{
        background-color: {tr_odd_bg} !important;
        }}
    tr:nth-child(even) {{
        background-color: {tr_even_bg} !important;
        }}
    .widget-button,.widget-toggle-button {{
        color: 	{btn_fg} !important;
        min-width: max-content !important;
        background-color: {btn_bg};
        border-radius: 5px !important;
    }}
    tr:hover {{
        background-color: {tr_hover_bg} !important;
        }}
    </style>
    """.format(**colors_dict)

dark_colors = {
 'table_fg': '#ABB2BF',
 'tr_odd_bg': '#282C34',
 'tr_even_bg': '#21252B',
 'tr_hover_bg': '#414855',
 'btn_fg': '#ABB2BF',
 'btn_bg': '#3D4450',
 'tab_bg': '#21252B',
 'tab_fg': '#61AFEF',
 'tab_shadow': '#282C34',
 'box_bg': '#21252B',
 'box_border': '#282C34',
 'text': '#ABB2BF',
 'input_bg': '#282C34',
 'input_fg': '#ABB2BF',
 'input_border': '#282C34',
 'input_hover': '#414855', # For focus and hover on border, also for dd se
 'dd_select_fg': '#ABB2BF',
 'dd_select_bg': 'transparent',
 'dd_hover': '#3D4450',
 'dd_opt_bg': '#282C34',
 'dd_opt_fg': 'whitesmoke',
 'dd_focus_bg': 'green'}

light_colors = {
    'table_fg'    : 'black',
    'tr_odd_bg'   : '#eaf0f0',
    'tr_even_bg'  : 'white',
    'tr_hover_bg' : '#abe4ff',
    'btn_fg'      : 'black',
    'btn_bg'      : '#c3d4d4',
    'tab_bg'      : '#F3F3F3',
    'tab_fg'      : 'black',
    'tab_shadow'  : 'gray',
    'box_bg'      : '#F3F3F3',
    'box_border'  : 'whitesmoke',
    'text'        : 'black',
    'input_bg'    : 'white',
    'input_fg'    : 'gray',
    'input_border': '#e0e8e8',
    'input_hover' : 'skyblue', # For focus and hover on border, also for dd select
    'dd_select_fg': 'skyblue',
    'dd_select_bg': 'transparent',
    'dd_hover'    : 'white',
    'dd_opt_bg'   : '#eaf0f0',
    'dd_opt_fg'   : 'gray',
    'dd_focus_bg' : 'red'
}

# Cell
def get_files_gui(auto_fill = 'vasprun.xml',html_style=None,height=320):
    """
    - Creates a GUI interface for files/folders filtering.
    - **Parmeters**
        - auto_fill  : Default is `vasprun.xml`, any file/folder.
        - html_style : None,Output of `css_style`.
        - height     : Height of Grid box.
    - **Returns**
        - Tuple(GUI_gridbox,Files_Dropdown). Access second one by item itself.
    """
    files_w = ipw.Dropdown(continuous_update=False)
    pw = ipw.Text(value=os.getcwd())

    incldue_w = ipw.Text(value=auto_fill)

    excldue_w = ipw.Text()
    d_layout = Layout(width='30%')
    l_layout = Layout(width='19%')
    depth_w = ipw.Dropdown(options=[None,1,2,3,4,5],value=4,layout=d_layout)
    item_w = ipw.Dropdown(options=['Both','Files','Folders'],value='Files',layout=d_layout)
    item_box = ipw.HBox([ipw.Label('Depth: ',layout=l_layout),depth_w,ipw.Label('Type: ',layout=l_layout),item_w])
    item_box.add_class('borderless').add_class('marginless')

    applybtn_w = ipw.Button(description='Apply Filters')
    gci_output = ipw.Output(layout=Layout(height='{}px'.format(height-70)))
    label_head = ipw.HTML("<h3>Your Filtered Files List</h3>")


    def filter_gci(applybtn_w):
        applybtn_w.description = 'Applying...'
        applybtn_w.disabled = True
        if os.path.isdir(pw.value):
            path = pw.value
        else:
            with gci_output:
                print("Given path does not exists.")
                print("Falling back to PWD: {}".format(os.getcwd()))
            path = os.getcwd()
            pw.value = path
        gci = vp.Dict2Data({'children':[],'parent':path})

        if 'Files' in item_w.value:
            file_type = dict(filesOnly=True)
        elif 'Folders' in item_w.value:
            file_type = dict(dirsOnly=True)
        else:
            file_type = {}
        try:
            gci = gu.get_child_items(path=path, **file_type,
                           include= incldue_w.value,
                           exclude= excldue_w.value,
                           depth=depth_w.value)
        except:
            with gci_output:
                print('Something went wrong')
        # Enable before any error occur.
        applybtn_w.disabled = False
        files_w.options = {name: os.path.join(gci.parent,name) for name in gci.children}

        applybtn_w.description = 'Successful!'
        label_head.value = "<h3>From: {}</h3>".format(gci.parent)
        with gci_output:
            display(ipw.HTML("<h4>{} files found.</h4>".format(len(gci.children))))
            display(ipw.HTML("<ol>{}<ol>".format(''.join(['<li>{}</li>'.format(i) for i in gci.children]))))


        applybtn_w.description = 'Apply Filters'
        gci_output.clear_output(wait=True)

    applybtn_w.on_click(filter_gci)
    out_box = ipw.Box([gci_output])
    right_box = ipw.VBox([label_head,out_box])
    out_box.add_class('borderless')
    right_box.add_class('borderless')
    i_layout = Layout(width='99%')
    incldue_w.layout = i_layout
    excldue_w.layout = i_layout
    pw.layout = i_layout
    input_box = ipw.VBox([
        ipw.Label('Path to Project Folder',layout=i_layout),pw,
        ipw.Label('Items to Include (separate by |)',layout=i_layout),incldue_w,
        ipw.Label('Items to Exclude (separate by |)',layout=i_layout),excldue_w,
        item_box,
        applybtn_w],layout=Layout(width='330px'))
    if not html_style:
        html_style = ''
    full_box = ipw.HBox([ipw.HTML(html_style),input_box, right_box],
                        layout=Layout(height='{}px'.format(height)))
    full_box.add_class('borderless')
    full_box.add_class('marginless')
    return full_box, files_w


# Cell
def get_input_gui(rgb=True,sys_info=None,html_style=None,height=400):
    """
    - Creates a GUI interface for input/selection of orbitals/ions projection.
    - **Parmeters**
        - rgb        : Default is `True` and generates input for `plotly(quick)_rgb_lines`, if `False` creates input for `quick(plotly)_dos(color)_lines`
        - html_style : None,Output of `css_style`.
        - height     : Height of Grid box.
    - **Returns**
        - Tuple(GUI_gridbox,json_in_HTML). Access second one by item.value.
    """
    if not html_style:
        html_style = ''
    if sys_info ==None:
        sys_info = vp.Dict2Data({'fields':['s'],'ElemIndex':[0,1],'ElemName':['A']})
    layout = Layout(width='30%')
    orbs_w  = ipw.Dropdown(options={'s':0},value=0,layout=layout)
    orbi_w  = ipw.Text(layout=layout)
    ions_w  = ipw.Dropdown(options={'All':[0,1,1]},value=[0,1,1],layout=layout)
    ioni_w  = ipw.Text(layout=layout)
    label_w = ipw.Text(layout=layout)
    rgb_w   = ipw.Dropdown(options={'Red':0,'Green':1,'Blue':2},value=0,layout=layout)
    rgbl_w  = Label('Color: ')
    click_w= ipw.Button(layout=Layout(width='max-content'),icon='fa-hand-o-up',tooltip='Read Projection Input')
    # Uniform label widths
    l_width = Layout(width='20%')
    rgbl_w.layout=l_width

    inpro_w = ipw.HTML(layout=Layout(height='20px'))
    if not rgb:
        layout = Layout(width='32px')
        add_w = ipw.Button(description='',layout=layout,icon='fa-plus-circle')
        del_w = ipw.Button(description='',layout=layout,icon='fa-minus-circle')
        add_w.style.button_color='#5AD5D1'
        del_w.style.button_color='#5AD5D1'
        read_box = [Label(u"Line \u00B1: "),add_w,del_w,Label(),Label(),click_w]
    else:
        read_box = [click_w]

    in_w = VBox([ipw.HTML(html_style),
                ipw.HTML("<h3>Projections</h3>"),
                HBox([rgbl_w,rgb_w,Label('Label: ',layout=l_width),label_w
                    ]).add_class('borderless').add_class('marginless'),
                HBox([Label('Ions: ',layout=l_width),ions_w,Label('::>>:: ',layout=l_width),ioni_w
                    ]).add_class('borderless').add_class('marginless'),
                HBox([Label('Orbs: ',layout=l_width),orbs_w,Label('::>>:: ',layout=l_width),orbi_w
                    ]).add_class('borderless').add_class('marginless'),
                HBox(read_box).add_class('borderless').add_class('marginless')
                ],layout=Layout(height="{}px".format(height))
                ).add_class('marginless')

    orbs_w.options= {str(i)+': '+item:str(i) for i,item in enumerate(sys_info.fields)}
    ipw.dlink((orbs_w,'value'), (orbi_w,'value'))

    inds = sys_info.ElemIndex
    ions_w.options = {"{}-{}: {}".format(inds[i],inds[i+1]-1,item):"{}-{}".format(inds[i],inds[i+1]-1) for i,item                             in enumerate(sys_info.ElemName)}

    ipw.dlink((ions_w,'value'), (ioni_w,'value'))


    def read_pro(ions_w,orbi_w,label_w):
        orbs_l = []
        if orbi_w.value:
            for o in orbi_w.value.split(","):
                if '-' in o:
                    strt,stop = o.split("-")
                    [orbs_l.append(i) for i in range(int(strt),int(stop)+1)]
                else:
                    orbs_l.append(int(o))

        ions_l = []
        if ioni_w.value:
            for o in ioni_w.value.split(","):
                if '-' in o:
                    strt,stop = o.split("-")
                    [ions_l.append(i) for i in range(int(strt),int(stop)+1)]
                else:
                    ions_l.append(int(o))
        if label_w.value:
            label_l = label_w.value
        else:
            label_l = "{}:{}".format(ioni_w.value,orbi_w.value)
        return ions_l,orbs_l,label_l
    if rgb:
        elements,orbs,labels = [[],[],[]],[[],[],[]],['','','']
    else:
        rgb_w.options = {"Line 0":0}
        rgb_w.value = 0
        rgbl_w.value = 'Line: '
        elements,orbs, labels = [[],],[[],],['',] # For DOS


    def read_lines(click_w):

        # Read Now
        index=rgb_w.value
        r_p = read_pro(ions_w,orbi_w,label_w)
        # update
        try:
            elements[index] = r_p[0]
            orbs[index]     = r_p[1]
            labels[index]   = r_p[2]
        except:
            elements.append(r_p[0]) # In case new line added.
            orbs.append(r_p[1])
            labels.append(r_p[2])

        max_ind = len(rgb_w.options) # in case a line is deleted.
        _input_ = dict(elements=elements[:max_ind],orbs=orbs[:max_ind],labels=labels[:max_ind])
        inpro_w.value = json.dumps(_input_,indent=2)

        # Button feedback.
        click_w.description = "Got {}".format(rgb_w.label)
        click_w.icon = 'check'
        sleep(1)
        click_w.description = "Read Input"
        click_w.icon = 'fa-hand-o-up'
        click_w.tooltip = "Current Input\n{!r}".format(vp.Dict2Data(_input_))

    click_w.on_click(read_lines)

    # Observe output of a line right in box.
    def see_input(change):
        #if change:
        x = rgb_w.value
        try: #Avoid None and prior update
            ioni_w.value = ','.join([str(i) for i in elements[x]])
            orbi_w.value = ','.join([str(i) for i in orbs[x]])
            label_w.value = labels[x]
            click_w.description = "{}".format(rgb_w.label) # Triggers when line +/- as well.
            click_w.icon = 'fa-refresh'
        except: pass

    rgb_w.observe(see_input,'value')

    # Add and delete lines
    if not rgb:
        def add_line(add_w):
            l_opts = len(rgb_w.options)+1
            opts = {'Line {}'.format(i):i for i in range(l_opts)}
            rgb_w.options = opts
            rgb_w.value = l_opts - 1

        add_w.on_click(add_line)

        def del_line(del_w):
            l_opts = len(rgb_w.options)-1
            if l_opts > 0: # Do not delete last line. just update,otherwise issues.
                opts = {'Line {}'.format(i):i for i in range(l_opts)}
                rgb_w.options = opts
                rgb_w.value = l_opts - 1

        del_w.on_click(del_line)
    # Finally Link Line number to input button
    ipw.dlink((rgb_w,'label'),(click_w,'description')) # Link line to input buuton.
    click_w.description='Read Input' # After link is important

    return in_w, inpro_w

# Cell
#mouse event handler
def click_data(sel_en_w,fermi_w,data_dict,fig):
    def handle_click(trace, points, state):
        if(points.ys!=[]):
            e_fermi = (float(fermi_w.value) if fermi_w.value else 0)
            val = np.round(float(points.ys[0]) + e_fermi,4) #exact value
            for key in sel_en_w.options:
                if key in sel_en_w.value and key != 'None':
                    data_dict[key] = val # Assign value back
                if 'Fermi' in sel_en_w.value:
                    fermi_w.value = str(val) # change fermi
            # Shift Graph for Fermi value
            if 'Fermi' in sel_en_w.value:
                with fig.batch_animate():
                    for trace in fig.data: # SHift graph as Fermi Changes
                        trace.y = [y - data_dict['Fermi'] + e_fermi for y in trace.y]

            # Update Fermi, SO etc
            if data_dict['VBM'] and data_dict['CBM']:
                data_dict['E_gap'] = np.round(data_dict['CBM'] - data_dict['VBM'], 4)
            if data_dict['so_max'] and data_dict['so_min']:
                data_dict['Δ_SO'] = np.round(data_dict['so_max'] - data_dict['so_min'], 4)
            # Cycle energy types on graph click and chnage table as well unless it is None.
            if sel_en_w.value == 'CBM': # Avoid accidental SO calculation
                sel_en_w.value = 'None'
            if sel_en_w.value != 'None': # Keep usually as None
                _this = sel_en_w.options.index(sel_en_w.value)
                _next = _this + 1 if _this < len(sel_en_w.options) - 1 else 0
                sel_en_w.value = sel_en_w.options[_next] #To simulate as it changes

    for i in range(len(fig.data)):
        trace=fig.data[i]
        trace.on_click(handle_click)

# Display Table
def tabulate_data(data_dict):
    new_dict = {k:v for k,v in data_dict.items() if v != '' and k not in ['sys','so_max','so_min','Fermi']}
    ls = list(new_dict.keys())
    ds = list(new_dict.values())

    if len(ls) % 2 != 0:
        ls.append('')
        ds.append('')

    tab_data = [ls[:int(len(ls)/2)],ds[:int(len(ls)/2)],ls[int(len(ls)/2):],ds[int(len(ls)/2):]]

    htm_string = """<style>table {border-collapse: collapse !important;
      min-width: 100% !important;
      margin: 1px 1px 1px 1px !important;
      font-size: small !important;
      font-family: "Times New Roman", "Times", "serif" !important;}
      th, td {text-align: center !important;
      padding: 0px 8px 0px 8px !important;}
      tr { width: 100% !important;}
      tr:nth-child(odd) {font-weight:bold !important;}
      </style>"""
    htm_string += "<table><tr>{}</tr></table>".format( '</tr><tr>'.join(
                   '<td>{}</td>'.format('</td><td>'.join(str(_) for _ in row)) for row in tab_data) )
    return htm_string

# Send Data
def save_data(out_w1,data_dict):
    out_f = os.path.join(os.path.split(out_w1.value)[0],'result.json')
    vp.dump_dict(data_dict,dump_to='json',outfile=out_f)


# Cell
def color_toggle(tog_w,fig,rd_btn):
    if tog_w.icon == 'toggle-off':
        tog_w.icon = 'toggle-on'
        with fig.batch_animate():
            if rd_btn.value == 'DOS':
                for trace in fig.data:
                    trace.fill='tozeroy'
            else:
                for trace in fig.data[:1]:
                    trace.mode='markers+lines'
                    trace.line.color='rgba(222,222,220,0.1)'
    else:
        tog_w.icon = 'toggle-off'
        with fig.batch_animate():
            if rd_btn.value == 'DOS':
                for trace in fig.data:
                    trace.fill=None
            else:
                for trace in fig.data[:1]:
                    trace.mode='lines'
                    trace.line.width=1.5
                    trace.line.color='skyblue'


# Cell
def generate_summary(paths_list=None):
    # Make Data Frame
    result_paths = []
    common_prefix = '' #placeholder
    if paths_list:
        common_prefix = os.path.commonprefix(paths_list)
        for item in paths_list:
            if item and os.path.isdir(item):
                result_paths.append(os.path.join(item,'result.json'))
            elif item and os.path.isfile(item):
                result_paths.append(os.path.join(os.path.split(item)[0],'result.json'))
    result_dicts = []
    for path in result_paths:
        try: _p_ = os.path.split(path)[0].split(common_prefix)[1]
        except: _p_ = '' #In current directory
        try:
            f = open(path,'r')
            l_d = json.load(f)
            l_d.update({'rel_path':_p_})
            result_dicts.append(l_d)
            f.close()
        except: pass
    out_dict = {} # placeholder
    if result_dicts:
        out_dict.update({k:[v] if v else np.nan for k,v in result_dicts[0].items()})
        for i,d in enumerate(result_dicts):
            if i != 0:
                for k,v in d.items():
                    v = np.nan if not v else v
                    try: out_dict[k].append(v)
                    except:
                        out_dict.update({k:[np.nan for l in range(i)]}) #if not key before, add to all previous
                        out_dict[k].append(v) # Then append for current value
            # If next dictionary does not have key
            for k in out_dict.keys():
                if k not in d.keys():
                    out_dict[k].append(np.nan)
    try: out_dict.pop('Fermi',None) # Remove Fermi as not necessary
    except: pass

    df = pd.DataFrame(out_dict)
    if common_prefix:
        return df.style.set_caption("Root Path: {}".format(common_prefix)) # return with header
    return df #return simple


# Cell
class VasprunApp:
    # capture output
    output = ipw.Output().add_class('output')
    def __init__(self,height=600):
        self.height = height
        tab = ipw.Tab(layout = ipw.Layout(min_height=f'{height}px', max_height='100vh',
                                           min_width='700px', max_width='100vw')
            ).add_class('marginless').add_class('borderless') # Main Tab

        for i,item in enumerate(['Home','Graphs','STD(out/err)']):
            tab.set_title(i,item)

        self.tab    = tab # Main layout
        self.data   = None # Export vasprun object.
        self.__path = None # current path
        self.fig    = go.FigureWidget() # plotly's figure widget
        self.fig.update_layout(autosize=True)
        self.df     = None # Summary DataFrame
        self.result = {'sys':'','V':'','a':'','b':'','c':'','Fermi': None,
                     'VBM':'','CBM':'','so_max':'','so_min':''} # Table widget value

        self.files_gui,self.files_dd = get_files_gui()
        self.input_gui,self.input_dd = get_input_gui(height=None)
        self.input  = {} # Dictionary for input
        self.fig_gui = HBox() # Middle Tab
        self.theme_colors = light_colors.copy() # Avoid Modification

        l_btn = ipw.Layout(width='max-content')
        self.buttons = {'load_data' : Button(description='Load Data',layout=l_btn,tooltip='Load and Cache Data'),
                        'load_graph': Button(description='Load Graph',layout=l_btn,tooltip='Create Graph'),
                        'confirm'   : Button(description='Confirm Delete',layout=l_btn,icon='trash'),
                        'summary'   : Button(description='Project Summary',layout=l_btn,tootltip='Make DataFrame'),
                        'expand'    : Button(icon = "fa-expand",layout=l_btn,tooltip='Expand Fig'),
                        'toggle'    : Button(description='RGB',layout=l_btn,icon='toggle-on',tooltip='Toggle Colors'),
                        'save_fig'  : Button(description='Save Fig',icon='download',layout=l_btn,tooltip='')
                        }

        b_out = Layout(width='30%')
        en_options = ['Fermi','VBM','CBM','so_max','so_min','None']
        self.dds   = {'band_dos': Dropdown(options=['Bands','DOS'],value='Bands',
                                                layout= Layout(width='80px')),
                      'en_type' : Dropdown(options = en_options,value='None',layout=b_out),
                      'cache'   : Dropdown(options=['Table Data','PWD Cache','All Cache','None'],
                                            value='None',layout=b_out),
                      'theme'   : Dropdown(options=['Default','Light','Dark','Custom'],
                                            value='Light',layout=l_btn),
                      'style'   : Dropdown(options=["plotly", "plotly_white", "plotly_dark",
                                    "ggplot2", "seaborn", "simple_white", "none"],layout=l_btn),
                                    }

        self.texts = {'kticks': Text(value='',layout=b_out),
                      'ktickv': Text(value='',layout=b_out),
                      'kjoin' : Text(value='',layout=b_out),
                      'elim'  : Text(value='',layout=b_out),
                      'fermi' : Text(value='',layout=b_out),
                      'xyt'   : Text(value='')
                      }

        self.htmls = {'theme': ipw.HTML(css_style(light_colors)),
                      'table': ipw.HTML()}

        # Observing
        self.input_dd.observe(self.__update_input,"value")
        self.dds['theme'].observe(self.__update_theme,"value")
        self.dds['style'].observe(self.__update_plot_style,"value")
        self.files_dd.observe(self.__load_previous,"value")
        self.buttons['load_data'].on_click(self.__on_load)
        self.dds['band_dos'].observe(self.__figure_tab,"value")
        self.files_dd.observe(self.__update_table,'value')
        self.buttons['load_data'].observe(self.__update_table,'value')
        self.buttons['summary'].on_click(self.__df_out)
        self.buttons['confirm'].on_click(self.__deleter)
        self.texts['xyt'].observe(self.__update_xyt)
        self.buttons['toggle'].on_click(self.__tog_b)
        self.buttons['save_fig'].on_click(self.__save_connected)
        self.buttons['expand'].on_click(self.__expand_fig)
        self.buttons['load_graph'].on_click(self.__update_graph)
        self.dds['en_type'].observe(self.__update_table,"value") # This works from click_data


    def __figure_tab(self,change):
        l_out = Layout(width='20%')
        summary_box = HBox([self.buttons['summary']]
                        ).add_class('marginless')
        cache_box = HBox([Label('Delete Cache:'),self.dds['cache'],self.buttons['confirm']]
                        ).add_class('marginless')
        upper_box = VBox([
                HBox([Label('File:',layout=Layout(width='50px')),self.files_dd
                    ]).add_class('borderless').add_class('marginless'),
                HBox([Label('View:',layout=Layout(width='50px')),
                        self.dds['band_dos'],self.buttons['load_data']
                    ]).add_class('borderless').add_class('marginless')
                ]).add_class('marginless').add_class('borderless')

        points_box = HBox([Box([Label('E Type:',layout=l_out),
                                self.dds['en_type'],
                                Label('E-Fermi:',layout=l_out),
                                self.texts['fermi']
                    ]).add_class('marginless').add_class('borderless'),
                    self.buttons['load_graph']
                    ],layout=Layout(width='100%')).add_class('marginless')
        in_box = VBox([self.input_gui]).add_class('marginless').add_class('borderless')
        top_right = HBox([self.buttons['load_graph'],
                          Label('Style:'),
                          self.dds['style'],
                          Label('Theme:'),
                          self.dds['theme'],
                          self.buttons['expand']
                    ]).add_class('marginless')
        fig_box = Box([self.fig],layout=Layout(min_height='380px')).add_class('marginless')
        right_box = VBox([top_right,fig_box,self.htmls['table']
                 ],layout=Layout(min_width='60%')).add_class('marginless').add_class('borderless')


        if 'Bands' in self.dds['band_dos'].value:
            in_box.children = [self.input_gui,
                      VBox([HBox([Label('Ticks At: ',layout=l_out),
                                  self.texts['kticks'],
                                  Label('Labels: ',layout=l_out),
                                  self.texts['ktickv']
                                 ]).add_class('borderless').add_class('marginless'),
                      HBox([Label('Join At: ',layout=l_out),
                            self.texts['kjoin'],
                            Label('E Range: ',layout=l_out),
                            self.texts['elim']
                           ]).add_class('marginless').add_class('borderless')
                      ]).add_class('marginless')]
            right_box.children = [top_right,fig_box,points_box,self.htmls['table']]
        else:
            in_box.children = [self.input_gui,
                      HBox([Label('E Range:',layout=l_out),
                      self.texts['elim'],
                      Label('E-Fermi:',layout=l_out),
                      self.texts['fermi']
                      ]).add_class('marginless')]
            right_box.children = [top_right,fig_box,self.htmls['table']]
            self.dds['en_type'].value = 'None' # no scatter collection in DOS.


        left_box = VBox([upper_box,
                        in_box,
                        HBox([Label('X, Y, Title'),self.texts['xyt']]).add_class('borderless'),
                        HBox([Label('Options:'),self.buttons['save_fig'],self.buttons['toggle']]),
                        cache_box,
                        summary_box],layout=Layout(max_width='40%'),
                        ).add_class('marginless').add_class('borderless')
        self.fig_gui.children = (left_box,right_box)
        return self.fig_gui # Return for use in show



    @output.capture()
    def show(self):
        intro_html = ipw.HTML("<h2>Pivotpy</h2><p>Filter files here and switch tab to Graphs. You can create cache ahead of time to load quickly while working. If anything does not seem to work, see the error in STD(out/err) tab. For large files, do `Export-VR` in Powershell to access fast.</p><marquee style='color:red'>Pivotpy GUI based on ipywidgets!</marquee>")
        header_box = HBox([intro_html,
                           Label('Theme:',layout=Layout(width='80px')),
                           self.dds['theme']
                    ]).add_class('marginless').add_class('borderless')
        summary_gui = HBox([ipw.Label(),
                            self.buttons['summary']]
                            ).add_class('borderless')
        intro_box = VBox([self.htmls['theme'],
                      header_box,
                      self.files_gui,
                      summary_gui]
                      ).add_class('marginless').add_class('borderless').add_class('marginless')
        self.fig_gui = self.__figure_tab(1) #Start
        self.tab.children = (intro_box,
                             self.fig_gui,
                             VBox([HBox([ipw.HTML("<h3>Functions Logging/Output</h3>"),
                                          Box([
                                          Label('Theme:',layout=Layout(width='80px')),
                                          self.dds['theme']],layout=Layout(left='50%')).add_class('borderless')
                                          ]).add_class('borderless').add_class('marginless'),
                                    VasprunApp.output])
                             )
        self.dds['style'].value = 'plotly' # to trigger first callback on graph.
        return self.tab

    def __update_theme(self,change):
        if self.dds['theme'].value == 'Dark':
            self.htmls['theme'].value = css_style(dark_colors)
            self.fig.update_layout(template='plotly_dark')
            self.dds['style'].value = 'plotly_dark'
        elif self.dds['theme'].value == 'Light':
            self.htmls['theme'].value = css_style(light_colors)
            self.fig.update_layout(template='ggplot2')
            self.dds['style'].value = 'ggplot2'
        elif self.dds['theme'].value == 'Custom':
            self.htmls['theme'].value = css_style(self.theme_colors)
        else:
            self.htmls['theme'].value = ''
            self.fig.update_layout(template='plotly')
            self.dds['style'].value = 'plotly'

    def __update_plot_style(self,change):
        self.fig.update_layout(template = self.dds['style'].value)

    @output.capture(clear_output=True,wait=True)
    def __load_previous(self,change):
        path = self.files_dd.value
        try:
            _dir = os.path.split(path)[0]
            r_f = os.path.join(_dir,'result.json')
            self.result = vp.load_from_dump(r_f,keep_as_dict=True)
            print('Previous Analysis loaded in Table for {}'.format(path))
        except:
            print('Previous Analysis does not exist for {}'.format(path))

    @output.capture(clear_output=True,wait=True)
    def __read_data(self,poscar=None,sys_info=None):
        if sys_info != None:
            self.result["sys"] = sys_info.SYSTEM
        if poscar != None:
            self.result["V"] = np.round(poscar.volume,5)
            a,b,c = np.round(np.linalg.norm(poscar.basis,axis=1),5)
            self.result["a"] = a
            self.result["b"] = b
            self.result["c"] = c

    @output.capture(clear_output=True,wait=True)
    def __on_load(self,button):
        self.__load_previous(change=button) # previous calculations.
        self.tab.selected_index = 2
        self.buttons['load_data'].description='Loading ...'
        _dir = os.path.split(self.files_dd.value)[0] # directory
        try:
            sys_info = vp.load_from_dump(os.path.join(_dir,'sys_info.pickle'))
            self.data = vp.load_from_dump(os.path.join(_dir,'vasprun.pickle'))
            print('Cache Loaded')
        except:
            print('Trying Loading from Python ...')
            self.data = vp.export_vasprun(self.files_dd.value)
            print('Caching From: {}'.format(self.files_dd.value)) #Cache result
            ifile = os.path.join(_dir,'sys_info.pickle')
            vfile = os.path.join(_dir,'vasprun.pickle')
            vp.dump_dict(self.data.sys_info,outfile=ifile)
            vp.dump_dict(self.data,outfile=vfile)
            sys_info = self.data.sys_info # required here.

        _ = self.__read_data(self.data.poscar,sys_info) # Update Table data on load
        print('Done')

        self.tab.selected_index = 1
        if self.dds['band_dos'].value=='DOS':
            tmp_ui,__ = get_input_gui(rgb=False,sys_info=sys_info,height=None)
            self.dds['en_type'].value = 'None' #Avoid data ,modification
        else:
            tmp_ui,__ = get_input_gui(rgb=True,sys_info=sys_info,height=None)

        self.input_gui.children = tmp_ui.children
        __.value = self.input_dd.value # keep values
        #self.input = json.loads(__.value)
        ipw.dlink((__,'value'),(self.input_dd,'value'))
        self.buttons['load_data'].description='Load Data'
        self.__path = self.files_dd.value # Update in __on_load or graph to make sure data loads once
        self.buttons['load_data'].tooltip = "Current System\n{!r}".format(self.data.sys_info)

    def __update_input(self,change):
        self.input = json.loads(self.input_dd.value)

    @output.capture(clear_output=True,wait=True)
    def __update_table(self,change):
        self.htmls['table'].value = tabulate_data(self.result)
        save_data(self.files_dd,self.result) # save data as well.

    @output.capture(clear_output=True,wait=True)
    def __df_out(self,btn):
        self.tab.selected_index = 2
        self.buttons['summary'].description = 'See STD(out/err) Tab'
        paths = [v for k,v in self.files_dd.options.items()]
        df = generate_summary(paths_list=paths)
        display(df)
        self.df = df # Assign
        print('Get above DataFrame by app_name.df\nNote: app_name is variable name assigned to VasprunApp()')
        print('==================== OR =========================')
        _code = "import pivotpy as pp\npaths = ['{}']\ndf = pp.generate_summary(paths_list=paths)\ndf".format("',\n         '".join([str(p).replace('\\','/') for p in paths]))
        _code += "\n#ax = pp.init_figure()\n#df.plot(ax=ax,x='sys',y=['V','a'])"
        display(Markdown("```python\n{}\n```".format(_code)))
        self.buttons['summary'].description = 'Project Summary'

    @output.capture(clear_output=True,wait=True)
    def __deleter(self,btn):
        self.buttons['confirm'].description = 'Deleting ...'
        if self.files_dd.value:
            print('Deleting Selected Cache...')
            self.__clear_cache() # Deleting
            print('Done')
        self.__update_table(1) #Update when delete data
        self.buttons['confirm'].description='Confirm Delete'

    @output.capture(clear_output=True,wait=True)
    def __update_xyt(self,change):
        if self.texts['xyt'].value:
            xyt_text = self.texts['xyt'].value.split(',')
            try:
                self.fig.update_xaxes(title=xyt_text[0])
                self.fig.update_yaxes(title=xyt_text[1])
                self.fig.update_layout(title=xyt_text[2])
            except: pass #do nothing else

    @output.capture(clear_output=True,wait=True)
    def __tog_b(self,tog_w):
        color_toggle(self.buttons['toggle'],self.fig,self.dds['band_dos'])

    @output.capture(clear_output=True,wait=True)
    def __save_connected(self,btn):
        s_p = os.path.split(self.files_dd.value)[0]
        filename = os.path.join(s_p,'ConnectedFig.html')
        self.buttons['save_fig'].description = 'Saving...'
        views = VBox([self.htmls['theme'],self.fig,self.htmls['table']],
                layout=Layout(width='500px',height='490px')).add_class('borderless')
        embed_minimal_html(filename, views=[views], state=dependency_state([views]))
        self.buttons['save_fig'].description = 'Save Fig'
        self.buttons['save_fig'].tooltip = 'Recently Saved\n{!r}'.format(filename)

    @output.capture(clear_output=True,wait=True)
    def __expand_fig(self,btn):
        self.tab.selected_index = 2
        self.dds['en_type'] = 'None' # To avoid accidental clicks
        display(self.fig)
    # Garph
    @output.capture(clear_output=True,wait=True)
    def __update_graph(self,btn):
        path = self.files_dd.value
        if path:
            self.tab.selected_index = 2
            self.fig.data = []
            if self.data and path == self.__path: # Same load and data exists, heeps in fast
                print('Data already loaded')
            else:
                try:
                    self.buttons['load_graph'].description = 'loading pickle...'
                    print('Trying to Load Cache for Graph ...')
                    file = os.path.join(os.path.split(path)[0],'vasprun.pickle')
                    self.buttons['load_graph'].description = file
                    self.data = vp.load_from_dump(file)
                    self.buttons['load_graph'].description = 'Load Graph'
                except:
                    self.buttons['load_graph'].description = 'loading export...'
                    print('No cache found. Loading from file {} ...'.format(path))
                    self.data = vp.export_vasprun(path)
                self.buttons['load_graph'].description = "Load Graph"

            print('Done')

            self.__path  = self.files_dd.value #update here or __on_load. Useful to match things
            self.buttons['load_graph'].description = 'Load Graph'
            _ = self.__read_data(self.data.poscar,self.data.sys_info) # Update Table data
            if not self.texts['fermi'].value: #Read Only if not in box.
                self.texts['fermi'].value = str(self.data.sys_info.E_Fermi) # E_Fermi
            # Args of Graph function
            argdict = self.input
            elim_str  = self.texts['elim'].value
            fermi_str = self.texts['fermi'].value

            argdict.update({'E_Fermi':(float(fermi_str) if fermi_str else None)})
            argdict.update({'elim':([float(v) for v in elim_str.split(',')] if elim_str else None)})

            if self.dds['band_dos'].value == 'Bands':
                kjoin_str = self.texts['kjoin'].value
                kticks_str = self.texts['kticks'].value
                ktickv_str = self.texts['ktickv'].value
                argdict.update({'joinPathAt':([int(v) for v in kjoin_str.split(',')] if kjoin_str else None)})
                argdict.update({'xt_indices':([int(v) for v in kticks_str.split(',')] if kticks_str else [0,-1])})
                argdict.update({'xt_labels':([v for v in ktickv_str.split(',')] if ktickv_str else ['A','B'])})
                fig_data = ip.plotly_rgb_lines(path_evr=self.data,**argdict)
            else:
                self.dds['en_type'].value = 'None' # Avoid random clicks
                fig_data = ip.plotly_dos_lines(path_evr=self.data,**argdict)

            self.tab.selected_index = 1
            with self.fig.batch_animate():
                for d in fig_data.data:
                    self.fig.add_trace(d)
                fig_data.layout.template = self.dds['style'].value # before layout to avoid color blink
                self.fig.layout = fig_data.layout

            click_data(self.dds['en_type'],self.texts['fermi'],self.result,self.fig)
            self.buttons['load_graph'].tooltip = "Current System\n{!r}".format(self.data.sys_info)

    @output.capture(clear_output=True,wait=True)
    def __clear_cache(self):
        self.tab.selected_index = 2
        _dir = os.path.split(self.files_dd.value)[0]
        if 'Table' in self.dds['cache'].value:
            for k in self.result.keys(): # Avoid deleting V,a,b,Fermi
                if k not in ['sys','V','a','b','c','Fermi']:
                    self.result[k] = ''
        if 'PWD' in self.dds['cache'].value:
            _files = [os.path.join(_dir,f) for f in ['sys_info.pickle','vasprun.pickle']]
            _ = [[print("Deleting", _file),os.remove(_file)] for _file in _files if os.path.isfile(_file)]
        if 'All' in self.dds['cache'].value:
            for key, value in self.files_dd.options.items():
                _dir = os.path.split(value)[0]
                _files = [os.path.join(_dir,f) for f in ['sys_info.pickle','vasprun.pickle']]
                _ = [[print("Deleting", _file),os.remove(_file)] for _file in _files if os.path.isfile(_file)]
        self.tab.selected_index = 1
