"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const eks = require("../lib");
const kubectl_layer_1 = require("../lib/kubectl-layer");
const spot_interrupt_handler_1 = require("../lib/spot-interrupt-handler");
const util_1 = require("./util");
module.exports = {
    'a default cluster spans all subnets'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: false, defaultCapacity: 0 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EKS::Cluster', {
            ResourcesVpcConfig: {
                SubnetIds: [
                    { Ref: "VPCPublicSubnet1SubnetB4246D30" },
                    { Ref: "VPCPublicSubnet2Subnet74179F39" },
                    { Ref: "VPCPrivateSubnet1Subnet8BCA10E0" },
                    { Ref: "VPCPrivateSubnet2SubnetCFCDAA7A" },
                ]
            }
        }));
        test.done();
    },
    'create custom cluster correctly in any aws region'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'stack', { env: { region: 'us-east-1' } });
        // WHEN
        const vpc = new ec2.Vpc(stack, 'VPC');
        new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: true, defaultCapacity: 0 });
        const layer = kubectl_layer_1.KubectlLayer.getOrCreate(stack, {});
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-Cluster'));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Serverless::Application', {
            Location: {
                ApplicationId: 'arn:aws:serverlessrepo:us-east-1:903779448426:applications/lambda-layer-kubectl',
            }
        }));
        test.equal(layer.isChina(), false);
        test.done();
    },
    'create custom cluster correctly in any aws region in china'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'stack', { env: { region: 'cn-north-1' } });
        // WHEN
        const vpc = new ec2.Vpc(stack, 'VPC');
        new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: true, defaultCapacity: 0 });
        new kubectl_layer_1.KubectlLayer(stack, 'NewLayer');
        const layer = kubectl_layer_1.KubectlLayer.getOrCreate(stack);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-Cluster'));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Serverless::Application', {
            Location: {
                ApplicationId: 'arn:aws-cn:serverlessrepo:cn-north-1:487369736442:applications/lambda-layer-kubectl',
            }
        }));
        test.equal(layer.isChina(), true);
        test.done();
    },
    'if "vpc" is not specified, vpc with default configuration will be created'(test) {
        // GIVEN
        const { stack } = util_1.testFixtureNoVpc();
        // WHEN
        new eks.Cluster(stack, 'cluster');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC'));
        test.done();
    },
    'default capacity': {
        'x2 m5.large by default'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            // WHEN
            const cluster = new eks.Cluster(stack, 'cluster');
            // THEN
            test.ok(cluster.defaultCapacity);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', { MinSize: '2', MaxSize: '2' }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::LaunchConfiguration', { InstanceType: 'm5.large' }));
            test.done();
        },
        'quantity and type can be customized'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            // WHEN
            const cluster = new eks.Cluster(stack, 'cluster', {
                defaultCapacity: 10,
                defaultCapacityInstance: new ec2.InstanceType('m2.xlarge')
            });
            // THEN
            test.ok(cluster.defaultCapacity);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', { MinSize: '10', MaxSize: '10' }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::LaunchConfiguration', { InstanceType: 'm2.xlarge' }));
            test.done();
        },
        'defaultCapacity=0 will not allocate at all'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            // WHEN
            const cluster = new eks.Cluster(stack, 'cluster', { defaultCapacity: 0 });
            // THEN
            test.ok(!cluster.defaultCapacity);
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::AutoScaling::LaunchConfiguration'));
            test.done();
        }
    },
    'creating a cluster tags the private VPC subnets'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: false, defaultCapacity: 0 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
            Tags: [
                { Key: "Name", Value: "Stack/VPC/PrivateSubnet1" },
                { Key: "aws-cdk:subnet-name", Value: "Private" },
                { Key: "aws-cdk:subnet-type", Value: "Private" },
                { Key: "kubernetes.io/role/internal-elb", Value: "1" }
            ]
        }));
        test.done();
    },
    'creating a cluster tags the public VPC subnets'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: false, defaultCapacity: 0 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
            MapPublicIpOnLaunch: true,
            Tags: [
                { Key: "Name", Value: "Stack/VPC/PublicSubnet1" },
                { Key: "aws-cdk:subnet-name", Value: "Public" },
                { Key: "aws-cdk:subnet-type", Value: "Public" },
                { Key: "kubernetes.io/role/elb", Value: "1" }
            ]
        }));
        test.done();
    },
    'adding capacity creates an ASG with tags'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: false, defaultCapacity: 0 });
        // WHEN
        cluster.addCapacity('Default', {
            instanceType: new ec2.InstanceType('t2.medium'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
            Tags: [
                {
                    Key: "Name",
                    PropagateAtLaunch: true,
                    Value: "Stack/Cluster/Default"
                },
                {
                    Key: { "Fn::Join": ["", ["kubernetes.io/cluster/", { Ref: "ClusterEB0386A7" }]] },
                    PropagateAtLaunch: true,
                    Value: "owned"
                }
            ]
        }));
        test.done();
    },
    'exercise export/import'(test) {
        // GIVEN
        const { stack: stack1, vpc, app } = util_1.testFixture();
        const stack2 = new cdk.Stack(app, 'stack2', { env: { region: 'us-east-1' } });
        const cluster = new eks.Cluster(stack1, 'Cluster', { vpc, kubectlEnabled: false, defaultCapacity: 0 });
        // WHEN
        const imported = eks.Cluster.fromClusterAttributes(stack2, 'Imported', {
            clusterArn: cluster.clusterArn,
            vpc: cluster.vpc,
            clusterEndpoint: cluster.clusterEndpoint,
            clusterName: cluster.clusterName,
            securityGroups: cluster.connections.securityGroups,
            clusterCertificateAuthorityData: cluster.clusterCertificateAuthorityData
        });
        // this should cause an export/import
        new cdk.CfnOutput(stack2, 'ClusterARN', { value: imported.clusterArn });
        // THEN
        assert_1.expect(stack2).toMatch({
            Outputs: {
                ClusterARN: {
                    Value: {
                        "Fn::ImportValue": "Stack:ExportsOutputFnGetAttClusterEB0386A7Arn2F2E3C3F"
                    }
                }
            }
        });
        test.done();
    },
    'disabled features when kubectl is disabled'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: false, defaultCapacity: 0 });
        test.throws(() => cluster.awsAuth, /Cannot define aws-auth mappings if kubectl is disabled/);
        test.throws(() => cluster.addResource('foo', {}), /Unable to perform this operation since kubectl is not enabled for this cluster/);
        test.throws(() => cluster.addCapacity('boo', { instanceType: new ec2.InstanceType('r5d.24xlarge'), mapRole: true }), /Cannot map instance IAM role to RBAC if kubectl is disabled for the cluster/);
        test.throws(() => new eks.HelmChart(stack, 'MyChart', { cluster, chart: 'chart' }), /Unable to perform this operation since kubectl is not enabled for this cluster/);
        test.done();
    },
    'mastersRole can be used to map an IAM role to "system:masters" (required kubectl)'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const role = new iam.Role(stack, 'role', { assumedBy: new iam.AnyPrincipal() });
        // WHEN
        new eks.Cluster(stack, 'Cluster', { vpc, mastersRole: role, defaultCapacity: 0 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE, {
            Manifest: {
                "Fn::Join": [
                    "",
                    [
                        "[{\"apiVersion\":\"v1\",\"kind\":\"ConfigMap\",\"metadata\":{\"name\":\"aws-auth\",\"namespace\":\"kube-system\"},\"data\":{\"mapRoles\":\"[{\\\"rolearn\\\":\\\"",
                        {
                            "Fn::GetAtt": [
                                "roleC7B7E775",
                                "Arn"
                            ]
                        },
                        "\\\",\\\"username\\\":\\\"",
                        {
                            "Fn::GetAtt": [
                                "roleC7B7E775",
                                "Arn"
                            ]
                        },
                        "\\\",\\\"groups\\\":[\\\"system:masters\\\"]}]\",\"mapUsers\":\"[]\",\"mapAccounts\":\"[]\"}}]"
                    ]
                ]
            }
        }));
        test.done();
    },
    'addResource can be used to apply k8s manifests on this cluster'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc, defaultCapacity: 0 });
        // WHEN
        cluster.addResource('manifest1', { foo: 123 });
        cluster.addResource('manifest2', { bar: 123 }, { boor: [1, 2, 3] });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE, {
            Manifest: "[{\"foo\":123}]"
        }));
        assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE, {
            Manifest: "[{\"bar\":123},{\"boor\":[1,2,3]}]"
        }));
        test.done();
    },
    'when kubectl is enabled (default) adding capacity will automatically map its IAM role'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc, defaultCapacity: 0 });
        // WHEN
        cluster.addCapacity('default', {
            instanceType: new ec2.InstanceType('t2.nano'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE, {
            Manifest: {
                "Fn::Join": [
                    "",
                    [
                        "[{\"apiVersion\":\"v1\",\"kind\":\"ConfigMap\",\"metadata\":{\"name\":\"aws-auth\",\"namespace\":\"kube-system\"},\"data\":{\"mapRoles\":\"[{\\\"rolearn\\\":\\\"",
                        {
                            "Fn::GetAtt": [
                                "ClusterdefaultInstanceRoleF20A29CD",
                                "Arn"
                            ]
                        },
                        "\\\",\\\"username\\\":\\\"system:node:{{EC2PrivateDNSName}}\\\",\\\"groups\\\":[\\\"system:bootstrappers\\\",\\\"system:nodes\\\"]}]\",\"mapUsers\":\"[]\",\"mapAccounts\":\"[]\"}}]"
                    ]
                ]
            }
        }));
        test.done();
    },
    'addCapacity will *not* map the IAM role if mapRole is false'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc, defaultCapacity: 0 });
        // WHEN
        cluster.addCapacity('default', {
            instanceType: new ec2.InstanceType('t2.nano'),
            mapRole: false
        });
        // THEN
        assert_1.expect(stack).to(assert_1.not(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE)));
        test.done();
    },
    'addCapacity will *not* map the IAM role if kubectl is disabled'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: false, defaultCapacity: 0 });
        // WHEN
        cluster.addCapacity('default', {
            instanceType: new ec2.InstanceType('t2.nano')
        });
        // THEN
        assert_1.expect(stack).to(assert_1.not(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE)));
        test.done();
    },
    'outputs': {
        'aws eks update-kubeconfig is the only output synthesized by default'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.Cluster(stack, 'Cluster');
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.deepEqual(template.Outputs, {
                ClusterConfigCommand43AAE40F: { Value: { 'Fn::Join': ['', ['aws eks update-kubeconfig --name ', { Ref: 'Cluster9EE0221C' }, ' --region us-east-1']] } },
                ClusterGetTokenCommand06AE992E: { Value: { 'Fn::Join': ['', ['aws eks get-token --cluster-name ', { Ref: 'Cluster9EE0221C' }, ' --region us-east-1']] } }
            });
            test.done();
        },
        'if masters role is defined, it should be included in the config command'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            const mastersRole = new iam.Role(stack, 'masters', { assumedBy: new iam.AccountRootPrincipal() });
            new eks.Cluster(stack, 'Cluster', { mastersRole });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.deepEqual(template.Outputs, {
                ClusterConfigCommand43AAE40F: { Value: { 'Fn::Join': ['', ['aws eks update-kubeconfig --name ', { Ref: 'Cluster9EE0221C' }, ' --region us-east-1 --role-arn ', { 'Fn::GetAtt': ['masters0D04F23D', 'Arn'] }]] } },
                ClusterGetTokenCommand06AE992E: { Value: { 'Fn::Join': ['', ['aws eks get-token --cluster-name ', { Ref: 'Cluster9EE0221C' }, ' --region us-east-1 --role-arn ', { 'Fn::GetAtt': ['masters0D04F23D', 'Arn'] }]] } }
            });
            test.done();
        },
        'if `outputConfigCommand=false` will disabled the output'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            const mastersRole = new iam.Role(stack, 'masters', { assumedBy: new iam.AccountRootPrincipal() });
            new eks.Cluster(stack, 'Cluster', {
                mastersRole,
                outputConfigCommand: false,
            });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.ok(!template.Outputs); // no outputs
            test.done();
        },
        '`outputClusterName` can be used to synthesize an output with the cluster name'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.Cluster(stack, 'Cluster', {
                outputConfigCommand: false,
                outputClusterName: true
            });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.deepEqual(template.Outputs, {
                ClusterClusterNameEB26049E: { Value: { Ref: 'Cluster9EE0221C' } }
            });
            test.done();
        },
        '`outputMastersRoleArn` can be used to synthesize an output with the arn of the masters role if defined'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.Cluster(stack, 'Cluster', {
                outputConfigCommand: false,
                outputMastersRoleArn: true,
                mastersRole: new iam.Role(stack, 'masters', { assumedBy: new iam.AccountRootPrincipal() })
            });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.deepEqual(template.Outputs, {
                ClusterMastersRoleArnB15964B1: { Value: { 'Fn::GetAtt': ['masters0D04F23D', 'Arn'] } }
            });
            test.done();
        },
        'when adding capacity, instance role ARN will not be outputed only if we do not auto-map aws-auth'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.Cluster(stack, 'Cluster', {
                outputConfigCommand: false,
                kubectlEnabled: false
            });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.deepEqual(template.Outputs, {
                ClusterDefaultCapacityInstanceRoleARN7DADF219: {
                    Value: { 'Fn::GetAtt': ['ClusterDefaultCapacityInstanceRole3E209969', 'Arn'] }
                }
            });
            test.done();
        },
    },
    'boostrap user-data': {
        'rendered by default for ASGs'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0 });
            // WHEN
            cluster.addCapacity('MyCapcity', { instanceType: new ec2.InstanceType('m3.xlargs') });
            // THEN
            const template = app.synth().getStackByName(stack.stackName).template;
            const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
            test.deepEqual(userData, { 'Fn::Base64': { 'Fn::Join': ['', ['#!/bin/bash\nset -o xtrace\n/etc/eks/bootstrap.sh ', { Ref: 'Cluster9EE0221C' }, ' --kubelet-extra-args "--node-labels lifecycle=OnDemand" --use-max-pods true\n/opt/aws/bin/cfn-signal --exit-code $? --stack Stack --resource ClusterMyCapcityASGD4CD8B97 --region us-east-1']] } });
            test.done();
        },
        'not rendered if bootstrap is disabled'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0 });
            // WHEN
            cluster.addCapacity('MyCapcity', {
                instanceType: new ec2.InstanceType('m3.xlargs'),
                bootstrapEnabled: false
            });
            // THEN
            const template = app.synth().getStackByName(stack.stackName).template;
            const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
            test.deepEqual(userData, { "Fn::Base64": "#!/bin/bash" });
            test.done();
        },
        // cursory test for options: see test.user-data.ts for full suite
        'bootstrap options'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0 });
            // WHEN
            cluster.addCapacity('MyCapcity', {
                instanceType: new ec2.InstanceType('m3.xlargs'),
                bootstrapOptions: {
                    kubeletExtraArgs: '--node-labels FOO=42'
                }
            });
            // THEN
            const template = app.synth().getStackByName(stack.stackName).template;
            const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
            test.deepEqual(userData, { 'Fn::Base64': { 'Fn::Join': ['', ['#!/bin/bash\nset -o xtrace\n/etc/eks/bootstrap.sh ', { Ref: 'Cluster9EE0221C' }, ' --kubelet-extra-args "--node-labels lifecycle=OnDemand  --node-labels FOO=42" --use-max-pods true\n/opt/aws/bin/cfn-signal --exit-code $? --stack Stack --resource ClusterMyCapcityASGD4CD8B97 --region us-east-1']] } });
            test.done();
        },
        'spot instances': {
            'nodes labeled an tainted accordingly'(test) {
                // GIVEN
                const { app, stack } = util_1.testFixtureNoVpc();
                const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0 });
                // WHEN
                cluster.addCapacity('MyCapcity', {
                    instanceType: new ec2.InstanceType('m3.xlargs'),
                    spotPrice: '0.01'
                });
                // THEN
                const template = app.synth().getStackByName(stack.stackName).template;
                const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
                test.deepEqual(userData, { 'Fn::Base64': { 'Fn::Join': ['', ['#!/bin/bash\nset -o xtrace\n/etc/eks/bootstrap.sh ', { Ref: 'Cluster9EE0221C' }, ' --kubelet-extra-args "--node-labels lifecycle=Ec2Spot --register-with-taints=spotInstance=true:PreferNoSchedule" --use-max-pods true\n/opt/aws/bin/cfn-signal --exit-code $? --stack Stack --resource ClusterMyCapcityASGD4CD8B97 --region us-east-1']] } });
                test.done();
            },
            'if kubectl is enabled, the interrupt handler is added'(test) {
                // GIVEN
                const { stack } = util_1.testFixtureNoVpc();
                const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0 });
                // WHEN
                cluster.addCapacity('MyCapcity', {
                    instanceType: new ec2.InstanceType('m3.xlargs'),
                    spotPrice: '0.01'
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE, { Manifest: JSON.stringify(spot_interrupt_handler_1.spotInterruptHandler()) }));
                test.done();
            },
            'if kubectl is disabled, interrupt handler is not added'(test) {
                // GIVEN
                const { stack } = util_1.testFixtureNoVpc();
                const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, kubectlEnabled: false });
                // WHEN
                cluster.addCapacity('MyCapcity', {
                    instanceType: new ec2.InstanceType('m3.xlargs'),
                    spotPrice: '0.01'
                });
                // THEN
                assert_1.expect(stack).notTo(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE));
                test.done();
            }
        }
    },
    'if bootstrap is disabled cannot specify options'(test) {
        // GIVEN
        const { stack } = util_1.testFixtureNoVpc();
        const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0 });
        // THEN
        test.throws(() => cluster.addCapacity('MyCapcity', {
            instanceType: new ec2.InstanceType('m3.xlargs'),
            bootstrapEnabled: false,
            bootstrapOptions: { awsApiRetryAttempts: 10 }
        }), /Cannot specify "bootstrapOptions" if "bootstrapEnabled" is false/);
        test.done();
    },
    'EKS-Optimized AMI with GPU support'(test) {
        // GIVEN
        const { app, stack } = util_1.testFixtureNoVpc();
        // WHEN
        new eks.Cluster(stack, 'cluster', {
            defaultCapacity: 2,
            defaultCapacityInstance: new ec2.InstanceType('g4dn.xlarge'),
        });
        // THEN
        const assembly = app.synth();
        const parameters = assembly.getStackByName(stack.stackName).template.Parameters;
        test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') && v.Default.includes('amazon-linux2-gpu')), 'EKS AMI with GPU should be in ssm parameters');
        test.done();
    },
    'when using custom resource a creation role & policy is defined'(test) {
        // GIVEN
        const { stack } = util_1.testFixture();
        // WHEN
        new eks.Cluster(stack, 'MyCluster', {
            clusterName: 'my-cluster-name'
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-Cluster', {
            Config: {
                name: "my-cluster-name",
                roleArn: { "Fn::GetAtt": ["MyClusterRoleBA20FE72", "Arn"] },
                resourcesVpcConfig: {
                    securityGroupIds: [{ "Fn::GetAtt": ["MyClusterControlPlaneSecurityGroup6B658F79", "GroupId"] }],
                    subnetIds: [
                        { Ref: "MyClusterDefaultVpcPublicSubnet1SubnetFAE5A9B6" },
                        { Ref: "MyClusterDefaultVpcPublicSubnet2SubnetF6D028A0" },
                        { Ref: "MyClusterDefaultVpcPrivateSubnet1SubnetE1D0DCDB" },
                        { Ref: "MyClusterDefaultVpcPrivateSubnet2Subnet11FEA8D0" }
                    ]
                }
            }
        }));
        // role can be assumed by 3 lambda handlers (2 for the cluster resource and 1 for the kubernetes resource)
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: "sts:AssumeRole",
                        Effect: "Allow",
                        Principal: {
                            AWS: [
                                {
                                    "Fn::GetAtt": [
                                        "awscdkawseksClusterResourceProviderNestedStackawscdkawseksClusterResourceProviderNestedStackResource9827C454",
                                        "Outputs.StackawscdkawseksClusterResourceProviderOnEventHandlerServiceRole3AEE0A43Arn"
                                    ]
                                },
                                {
                                    "Fn::GetAtt": [
                                        "awscdkawseksClusterResourceProviderNestedStackawscdkawseksClusterResourceProviderNestedStackResource9827C454",
                                        "Outputs.StackawscdkawseksClusterResourceProviderIsCompleteHandlerServiceRole8E7F1C11Arn"
                                    ]
                                }
                            ]
                        }
                    },
                    {
                        Action: "sts:AssumeRole",
                        Effect: "Allow",
                        Principal: {
                            AWS: {
                                "Fn::GetAtt": [
                                    "awscdkawseksKubectlProviderNestedStackawscdkawseksKubectlProviderNestedStackResourceA7AEBA6B",
                                    "Outputs.StackawscdkawseksKubectlProviderHandlerServiceRole2C52B3ECArn"
                                ]
                            }
                        }
                    }
                ],
                Version: "2012-10-17"
            }
        }));
        // policy allows creation role to pass the cluster role and to interact with the cluster (given we know the explicit cluster name)
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: "iam:PassRole",
                        Effect: "Allow",
                        Resource: {
                            "Fn::GetAtt": [
                                "MyClusterRoleBA20FE72",
                                "Arn"
                            ]
                        }
                    },
                    {
                        Action: "ec2:DescribeSubnets",
                        Effect: "Allow",
                        Resource: "*",
                    },
                    {
                        Action: [
                            "eks:CreateCluster",
                            "eks:DescribeCluster",
                            "eks:DeleteCluster",
                            "eks:UpdateClusterVersion",
                            "eks:UpdateClusterConfig",
                            "eks:CreateFargateProfile"
                        ],
                        Effect: "Allow",
                        Resource: [{
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":eks:us-east-1:",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":cluster/my-cluster-name"
                                    ]
                                ]
                            }, {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":eks:us-east-1:",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":cluster/my-cluster-name/*"
                                    ]
                                ]
                            }]
                    },
                    {
                        Action: [
                            "eks:DescribeFargateProfile",
                            "eks:DeleteFargateProfile"
                        ],
                        Effect: "Allow",
                        Resource: {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        Ref: "AWS::Partition"
                                    },
                                    ":eks:us-east-1:",
                                    {
                                        Ref: "AWS::AccountId"
                                    },
                                    ":fargateprofile/my-cluster-name/*"
                                ]
                            ]
                        }
                    },
                    {
                        Action: "iam:GetRole",
                        Effect: "Allow",
                        Resource: "*"
                    },
                    {
                        Action: "iam:CreateServiceLinkedRole",
                        Effect: "Allow",
                        Resource: "*"
                    }
                ],
                Version: "2012-10-17"
            }
        }));
        test.done();
    },
    'if an explicit cluster name is not provided, the creation role policy is wider (allows interacting with all clusters)'(test) {
        // GIVEN
        const { stack } = util_1.testFixture();
        // WHEN
        new eks.Cluster(stack, 'MyCluster');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: "iam:PassRole",
                        Effect: "Allow",
                        Resource: {
                            "Fn::GetAtt": [
                                "MyClusterRoleBA20FE72",
                                "Arn"
                            ]
                        }
                    },
                    {
                        Action: "ec2:DescribeSubnets",
                        Effect: "Allow",
                        Resource: "*",
                    },
                    {
                        Action: [
                            "eks:CreateCluster",
                            "eks:DescribeCluster",
                            "eks:DeleteCluster",
                            "eks:UpdateClusterVersion",
                            "eks:UpdateClusterConfig",
                            "eks:CreateFargateProfile"
                        ],
                        Effect: "Allow",
                        Resource: ["*"]
                    },
                    {
                        Action: [
                            "eks:DescribeFargateProfile",
                            "eks:DeleteFargateProfile"
                        ],
                        Effect: "Allow",
                        Resource: "*"
                    },
                    {
                        Action: "iam:GetRole",
                        Effect: "Allow",
                        Resource: "*"
                    },
                    {
                        Action: "iam:CreateServiceLinkedRole",
                        Effect: "Allow",
                        Resource: "*"
                    }
                ],
                Version: "2012-10-17"
            }
        }));
        test.done();
    },
    'if helm charts are used, its resource provider is allowed to assume the creation role'(test) {
        // GIVEN
        const { stack } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'MyCluster', {
            clusterName: 'my-cluster-name'
        });
        // WHEN
        cluster.addChart('MyChart', {
            chart: 'foo'
        });
        // THEN
        // role can be assumed by 4 principals: two for the cluster resource, one
        // for kubernetes resource and one for the helm resource.
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: "sts:AssumeRole",
                        Effect: "Allow",
                        Principal: {
                            AWS: [
                                {
                                    "Fn::GetAtt": [
                                        "awscdkawseksClusterResourceProviderNestedStackawscdkawseksClusterResourceProviderNestedStackResource9827C454",
                                        "Outputs.StackawscdkawseksClusterResourceProviderOnEventHandlerServiceRole3AEE0A43Arn"
                                    ]
                                },
                                {
                                    "Fn::GetAtt": [
                                        "awscdkawseksClusterResourceProviderNestedStackawscdkawseksClusterResourceProviderNestedStackResource9827C454",
                                        "Outputs.StackawscdkawseksClusterResourceProviderIsCompleteHandlerServiceRole8E7F1C11Arn"
                                    ]
                                }
                            ]
                        }
                    },
                    {
                        Action: "sts:AssumeRole",
                        Effect: "Allow",
                        Principal: {
                            AWS: {
                                "Fn::GetAtt": [
                                    "awscdkawseksKubectlProviderNestedStackawscdkawseksKubectlProviderNestedStackResourceA7AEBA6B",
                                    "Outputs.StackawscdkawseksKubectlProviderHandlerServiceRole2C52B3ECArn"
                                ]
                            }
                        }
                    }
                ],
                Version: "2012-10-17"
            }
        }));
        test.done();
    },
    'coreDnsComputeType will patch the coreDNS configuration to use a "fargate" compute type and restore to "ec2" upon removal'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new eks.Cluster(stack, 'MyCluster', {
            coreDnsComputeType: eks.CoreDnsComputeType.FARGATE
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-KubernetesPatch', {
            ResourceName: "deployment/coredns",
            ResourceNamespace: "kube-system",
            ApplyPatchJson: "{\"spec\":{\"template\":{\"metadata\":{\"annotations\":{\"eks.amazonaws.com/compute-type\":\"fargate\"}}}}}",
            RestorePatchJson: "{\"spec\":{\"template\":{\"metadata\":{\"annotations\":{\"eks.amazonaws.com/compute-type\":\"ec2\"}}}}}",
            ClusterName: {
                Ref: "MyCluster8AD82BF8"
            },
            RoleArn: {
                "Fn::GetAtt": [
                    "MyClusterCreationRoleB5FA4FF3",
                    "Arn"
                ]
            }
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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