"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateSchema = exports.bump = exports.update = exports.SCHEMAS = void 0;
const fs = require("fs");
const path = require("path");
const semver = require("semver");
// eslint-disable-next-line import/no-extraneous-dependencies
const tjs = require("typescript-json-schema");
function log(message) {
    // eslint-disable-next-line no-console
    console.log(message);
}
/**
 * Where schemas are committed.
 */
const SCHEMA_DIR = path.resolve(__dirname, '../schema');
const SCHEMA_DEFINITIONS = {
    'assets': { rootTypeName: 'AssetManifest' },
    'cloud-assembly': { rootTypeName: 'AssemblyManifest' },
};
exports.SCHEMAS = Object.keys(SCHEMA_DEFINITIONS);
function update() {
    for (const s of exports.SCHEMAS) {
        generateSchema(s);
    }
    bump();
}
exports.update = update;
function bump() {
    const versionFile = path.join(SCHEMA_DIR, 'cloud-assembly.version.json');
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const metadata = require(versionFile);
    const oldVersion = metadata.version;
    const newVersion = semver.inc(oldVersion, 'major');
    log(`Updating schema version: ${oldVersion} -> ${newVersion}`);
    fs.writeFileSync(versionFile, JSON.stringify({ version: newVersion }));
}
exports.bump = bump;
/**
 * Generates a schema from typescript types.
 * @returns JSON schema
 * @param schemaName the schema to generate
 * @param shouldBump writes a new version of the schema and bumps the major version
 */
function generateSchema(schemaName, saveToFile = true) {
    const spec = SCHEMA_DEFINITIONS[schemaName];
    const out = saveToFile ? path.join(SCHEMA_DIR, `${schemaName}.schema.json`) : '';
    const settings = {
        required: true,
        ref: true,
        topRef: true,
        noExtraProps: false,
        out,
    };
    const compilerOptions = {
        strictNullChecks: true,
    };
    const program = tjs.getProgramFromFiles([path.join(__dirname, '../lib/index.d.ts')], compilerOptions);
    const schema = tjs.generateSchema(program, spec.rootTypeName, settings);
    augmentDescription(schema);
    addAnyMetadataEntry(schema);
    if (out) {
        log(`Generating schema to ${out}`);
        fs.writeFileSync(out, JSON.stringify(schema, null, 4));
    }
    return schema;
}
exports.generateSchema = generateSchema;
/**
 * Remove 'default' from the schema since its generated
 * from the tsdocs, which are not necessarily actual values,
 * but rather descriptive behavior.
 *
 * To keep this inforamtion in the schema, we append it to the
 * 'description' of the property.
 */
function augmentDescription(schema) {
    function _recurse(o) {
        for (const prop in o) {
            if (prop === 'description' && typeof o[prop] === 'string') {
                const description = o[prop];
                const defaultValue = o.default;
                if (!defaultValue) {
                    // property doesn't have a default value
                    // skip
                    continue;
                }
                const descriptionWithDefault = `${description} (Default ${defaultValue})`;
                delete o.default;
                o[prop] = descriptionWithDefault;
            }
            else if (typeof o[prop] === 'object') {
                _recurse(o[prop]);
            }
        }
    }
    _recurse(schema);
}
/**
 * Patch the properties of MetadataEntry to allow
 * specifying any free form data. This is needed since source
 * code doesn't allow this in order to enforce stricter jsii
 * compatibility checks.
 */
function addAnyMetadataEntry(schema) {
    var _a;
    (_a = schema.definitions.MetadataEntry) === null || _a === void 0 ? void 0 : _a.properties.data.anyOf.push({ description: 'Free form data.' });
}
//# sourceMappingURL=data:application/json;base64,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