# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['cacheme', 'cacheme.storages']

package_data = \
{'': ['*'], 'cacheme.storages': ['scripts/*']}

install_requires = \
['msgpack>=1.0.4,<2.0.0',
 'pydantic>=1.10.4,<2.0.0',
 'theine>=0.1.1,<0.2.0',
 'typing-extensions>=4.4.0,<5.0.0']

extras_require = \
{'aiomysql': ['aiomysql>=0.1.1,<0.2.0'],
 'asyncpg': ['asyncpg>=0.27.0,<0.28.0'],
 'motor': ['motor>=3.1.1,<4.0.0'],
 'redis': ['redis>=4.4.2,<5.0.0']}

setup_kwargs = {
    'name': 'cacheme',
    'version': '0.2.3',
    'description': 'async caching framework',
    'long_description': '# Cacheme\n\nAsyncio cache framework with multiple cache storages. [中文文档](README_ZH.md)\n\n- **Better cache management:** Cache configuration with node, you can apply different strategies on different nodes.\n- **Multiple cache storages:** in-memory/redis/mongodb/postgres..., also support chain storages.\n- **Multiple serializers:** Pickle/Json/Msgpack serializers.\n- **Type annotated:** All cacheme API are type annotated with generics.\n- **High hit ratio in-memory cache:** TinyLFU written in Rust with little memory overhead.\n- **Thundering herd protection:** Simultaneously requests to same key are blocked by asyncio Event and only load from source once.\n- **Cache stats API:** Stats of each node and colected automatically.\n\nRelated projects:\n- High performance in-memory cache: https://github.com/Yiling-J/theine\n- Benchmark(auto updated): https://github.com/Yiling-J/cacheme-benchmark\n\n## Table of Contents\n\n- [Installation](#installation)\n- [Add Node](#add-node)\n- [Register Storage](#register-storage)\n- [Cacheme API](#cacheme-api)\n- [Cache Node](#cache-node)\n    + [Key](#key)\n    + [Meta Class](#meta-class)\n    + [Serializers](#serializers)\n    + [DoorKeeper](#doorkeeper)\n- [Cache Storage](#cache-storage)\n    + [Local Storage](#local-storage)\n    + [Redis Storage](#redis-storage)\n    + [MongoDB Storage](#mongodb-storage)\n    + [Sqlite Storage](#sqlite-storage)\n    + [PostgreSQL Storage](#postgresql-storage)\n    + [MySQL Storage](#mysql-storage)\n- [Benchmarks](#benchmarks)\n\n## Requirements\nPython 3.7+\n\n## Installation\n\n```\npip install cacheme\n```\n\nMultiple storages are supported by drivers. You can install the required drivers with:\n```\npip install cacheme[redis]\npip install cacheme[aiomysql]\npip install cacheme[motor]\npip install cacheme[asyncpg]\n```\n\n## Add Node\nNode is the core part of cache. Each node has its own key function, load function and storage options. Stats of each node are collected independently. You can place all node definations into one package/module, so everyone knows exactly what is cached now and how they are cached. All cacheme API are based on node.\n\nEach node contains:\n- Key attritubes and `key` method,  which are used to generate cache key. Here the `UserInfoNode` is a dataclass, so `__init__` method is generated automatically.\n- Async `load` method, which will be called to load data from data source on cache missing. This method can be omitted if you use `Memoize` decorator only.\n- `Meta` class, node cache configurations. See [Cache Node](#cache-node)\n\n```python\nimport cacheme\nfrom dataclasses import dataclass\nfrom cacheme.serializer import MsgPackSerializer\n\n@dataclass\nclass UserInfoNode(cacheme.Node):\n    user_id: int\n\n    def key(self) -> str:\n        return f"user:{self.user_id}:info"\n\n    async def load(self) -> Dict:\n        user = get_user_from_db(self.user_id)\n        return serialize(user)\n\n    class Meta(cacheme.Node.Meta):\n        version = "v1"\n        caches = [cacheme.Cache(storage="my-redis", ttl=None)]\n        serializer = MsgPackSerializer()\n```\nThis simple example use a cache storage called "my-redis", which will be registered next step. Also we use `MsgPackSerializer` here to dump and load data from redis. See [Cache Node] for more details.\n\n## Register Storage\n\nRegister a redis storage called "my-redis", which you can use in node meta data. The `register_storage` is asynchronous and will try to establish connection to cache store.\nSee [Cache Storage] for more details.\n\n```python\nimport cacheme\n\nawait cacheme.register_storage("my-redis", cacheme.Storage(url="redis://localhost:6379"))\n```\n\n## Cacheme API\n\n`get`: get data from single node.\n```python\nuser = await cacheme.get(UserInfoNode(user_id=1))\n```\n\n`get_all`: get data from multiple nodes, same node type.\n```python\nusers = await cacheme.get_all([UserInfoNode(user_id=1), UserInfoNode(user_id=2)])\n```\n\n`invalidate`: invalidate a node, remove data from cache.\n```python\nawait cacheme.invalidate(UserInfoNode(user_id=1))\n```\n\n`refresh`: reload node data using `load` method.\n```python\nawait cacheme.refresh(UserInfoNode(user_id=1))\n```\n\n`Memoize`: memoize function with this decorator.\n\nDecorate your function with `cacheme.Memoize` decorator and cache node. Cacheme will load data using the decorated function and ignore `load` method.\nBecause your function may contain variable number of args/kwargs, we need one more step to map between args/kwargs to node. The decorated map function should have same input signature as memoized function, and return a cache node.\n\n```python\n@cacheme.Memoize(UserInfoNode)\nasync def get_user_info(user_id: int) -> Dict:\n    return {}\n\n# function name is not important, so just use _ here\n@get_user_info.to_node\ndef _(user_id: int) -> UserInfoNode:\n    return UserInfoNode(user_id=user_id)\n```\n\n`nodes`: list all nodes.\n```python\nnodes = cacheme.nodes()\n```\n\n`stats`: get node stats.\n```\nmetrics = cacheme.stats(UserInfoNode)\n\nmetrics.request_count() # total request count\nmetrics.hit_count() # total hit count\nmetrics.hit_rate() # hit_count/request_count\nmetrics.miss_count() # (request_count - hit_count)/request_count\nmetrics.miss_rate() # miss_count/request_count\nmetric.load_success_count() # total load success count\nmetrics.load_failure_count() # total load fail count\nmetrics.load_failure_rate() # load_failure_count/load_count\nmetrics.load_count() # total load count\nmetrics.total_load_time() # total load time in nanoseconds\nmetrics.average_load_time() # total_load_time/load_count\n```\n\n`set_prefix`: set prefix for all keys. Default prefix is `cacheme`. Change prefix will invalid all keys, because prefix is part of the key.\n```python\ncacheme.set_prefix("mycache")\n```\n\n\n## Cache Node\n\n#### Key\nGenerated cache key will be: `{prefix}:{key()}:{Meta.version}`. So change `version` will invalid all keys automatically.\n\n#### Meta Class\n- `version[str]`: Version of node, will be used as suffix of cache key.\n- `caches[List[Cache]]`: Caches for node. Each `Cache` has 2 attributes, `storage[str]` and `ttl[Optional[timedelta]]`. `storage` is the name you registered with `register_storage` and `ttl` is how long this cache will live. Cacheme will try to get data from each cache from left to right. In most cases, use single cache or [local, remote] combination.\n- `serializer[Optional[Serializer]]`: Serializer used to dump/load data. If storage type is `local`, serializer is ignored. See [Serializers](#serializers).\n- `doorkeeper[Optional[DoorKeeper]]`: See [DoorKeeper](#doorkeeper).\n\nMultiple caches example. Local cache is not synchronized, so set a much shorter ttl compared to redis one. Then we don\'t need to worry too much about stale data.\n\n```python\nimport cacheme\nfrom dataclasses import dataclass\nfrom datetime import timedelta\nfrom cacheme.serializer import MsgPackSerializer\n\n@dataclass\nclass UserInfoNode(cacheme.Node):\n    user_id: int\n\n    def key(self) -> str:\n        return f"user:{self.user_id}:info"\n\n    async def load(self) -> Dict:\n        user = get_user_from_db(self.user_id)\n        return serialize(user)\n\n    class Meta(cacheme.Node.Meta):\n        version = "v1"\n        caches = [\n            cacheme.Cache(storage="local", ttl=timedelta(seconds=30)),\n            cacheme.Cache(storage="my-redis", ttl=timedelta(days=10))\n        ]\n        serializer = MsgPackSerializer()\n```\n\nCacheme also support creating Node dynamically, you can use this together with `Memoize` decorator:\n\n```python\n@Memoize(cacheme.build_node("TestNodeDynamic", "v1", [Cache(storage="local", ttl=None)]))\nasync def fn(a: int) -> int:\n    return 1\n\n\n@fn.to_node\ndef _(a: int) -> cacheme.DynamicNode:\n    return DynamicNode(key=f"bar:{a}")\n```\nHere we use `DynamicNode`, which only support one param: `key`\n\n#### Serializers\nCacheme provides serveral builtin serializers, you can also write your own serializer.\n\n- `PickleSerializer`: All python objects.\n- `JSONSerializer`: Use `pydantic_encoder` and `json`, support python primitive types, dataclass, pydantic model. See [pydantic types](https://docs.pydantic.dev/usage/types/).\n- `MsgPackSerializer`: Use `pydantic_encoder` and `msgpack`, support python primitive types, dataclass, pydantic model. See [pydantic types](https://docs.pydantic.dev/usage/types/).\n\nserializer with compression, use zlib level-3\n\n- `CompressedPickleSerializer`\n- `CompressedJSONSerializer`\n- `CompressedMsgPackSerializer`\n\n#### DoorKeeper\nIdea from [TinyLfu paper](https://arxiv.org/pdf/1512.00727.pdf).\n\n*The Doorkeeper is a regular Bloom filter placed in front of the cahce. Upon\nitem arrival, we first check if the item is contained in the Doorkeeper. If it is not contained in the\nDoorkeeper (as is expected with first timers and tail items), the item is inserted to the Doorkeeper and\notherwise, it is inserted to the cache.*\n\n```python\nfrom cacheme import BloomFilter\n\n@dataclass\nclass UserInfoNode(cacheme.Node):\n\n    class Meta(cacheme.Node.Meta):\n        # size 100000, false positive probability 0.01\n        doorkeeper = BloomFilter(100000, 0.01)\n```\nBloomFilter is cleared automatically when requests count == size.\n\n\n## Cache Storage\n\n#### Local Storage\nLocal storage uses dictionary to store data. A policy is used to evicate keys when cache is full.\n```python\n# lru policy\nStorage(url="local://lru", size=10000)\n\n# tinylfu policy\nStorage(url="local://tlfu", size=10000)\n\n```\nParameters:\n\n- `url`: `local://{policy}`. 2 policies are currently supported:\n  - `lru`\n  - `tlfu`: TinyLfu policy, see https://arxiv.org/pdf/1512.00727.pdf\n\n- `size`: size of the storage. Policy will be used to evicate key when cache is full.\n\n#### Redis Storage\n```python\nStorage(url="redis://localhost:6379")\n\n# cluster\nStorage(url="redis://localhost:6379", cluster=True)\n```\nParameters:\n\n- `url`: redis connection url.\n- `cluster`: bool, cluster or not, default False.\n- `pool_size`: connection pool size, default 100.\n\n#### MongoDB Storage\nTo use mongodb storage, create index first. See [mongo.js](cacheme/storages/scripts/mongo.js)\n```python\nStorage(url="mongodb://test:password@localhost:27017",database="test",collection="cache")\n```\nParameters:\n\n- `url`: mongodb connection url.\n- `database`: mongodb database name.\n- `collection`: mongodb collection name.\n- `pool_size`: connection pool size, default 50.\n\n#### Sqlite Storage\nTo use sqlite storage, create table and index first. See [sqlite.sql](cacheme/storages/scripts/sqlite.sql)\n```python\nStorage(url="sqlite:///test", table="cache")\n```\nParameters:\n\n- `url`: sqlite connection url.\n- `table`: cache table name.\n- `pool_size`: connection pool size, default 50.\n\n#### PostgreSQL Storage\nTo use postgres storage, create table and index first. See [postgresql.sql](cacheme/storages/scripts/postgresql.sql)\n```python\nStorage(url="postgresql://username:password@127.0.0.1:5432/test", table="cache")\n```\nParameters:\n\n- `url`: postgres connection url.\n- `table`: cache table name.\n- `pool_size`: connection pool size, default 50.\n\n#### MySQL Storage\nTo use mysql storage, create table and index first. See [mysql.sql](cacheme/storages/scripts/mysql.sql)\n```python\nStorage("mysql://username:password@localhost:3306/test", table="cache")\n```\nParameters:\n\n- `url`: mysql connection url.\n- `table`: cache table name.\n- `pool_size`: connection pool size, default 50.\n\n## Benchmarks\n- Local Storage Hit Ratios(hit_count/request_count)\n  ![hit ratios](benchmarks/hit_ratio.png)\n  [source code](benchmarks/tlfu_hit.py)\n\n- Throughput Benchmark of different storages\n\n  See [benchmark]( https://github.com/Yiling-J/cacheme-benchmark)\n',
    'author': 'Yiling-J',
    'author_email': 'njjyl723@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
