"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CreateStackSimulation = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const url_1 = require("url");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const data_type_1 = require("../../domain/data-type");
const response_code_1 = require("../../domain/response-code");
const stack_status_1 = require("../../domain/stack-status");
const string_variable_1 = require("../../interface/variables/string-variable");
const aws_api_step_1 = require("../../parent-steps/automation/aws-api-step");
const delete_stack_step_1 = require("../../parent-steps/automation/delete-stack-step");
const sleep_step_1 = require("../../parent-steps/automation/sleep-step");
const wait_for_and_assert_1 = require("../../parent-steps/automation/sub-steps/wait-for-and-assert");
const prune_and_transform_record_1 = require("../../utils/prune-and-transform-record");
const automation_step_simulation_1 = require("../automation-step-simulation");
const automation_simulation_base_1 = require("./automation-simulation-base");
/**
 * AutomationStep implementation for aws:createStack
 * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-createstack.html
 */
class CreateStackSimulation extends automation_simulation_base_1.AutomationSimulationBase {
    constructor(step, props) {
        super(step);
        this.action = 'aws:createStack';
        this.creatStackStep = step;
        this.props = props;
    }
    executeStep(inputs) {
        this.verifyUrlProps(inputs);
        const stackName = this.creatStackStep.stackName.resolveToString(inputs);
        console.log(`CreateStack: Checking that ${stackName} is ready to be created`);
        this.preVerifyStackSubStep(stackName);
        console.log(`CreateStack: Creating ${stackName}`);
        const stackId = this.createStack(inputs);
        console.log(`CreateStack: Waiting for ${stackName} completion`);
        this.waitForStackCompletion(stackName);
        console.log('CreateStack: Stack is created');
        const stack = this.getStackState(stackName);
        return {
            StackId: stackId,
            StackStatus: stack?.StackStatus,
            StackStatusReason: stack?.StackStatusReason ?? '',
        };
    }
    verifyUrlProps(inputs) {
        return this.verifyUrlProp(this.creatStackStep.templateUrl, inputs)
            && this.verifyUrlProp(this.creatStackStep.stackPolicyUrl, inputs);
    }
    verifyUrlProp(urlToVerify, inputs) {
        return urlToVerify === undefined || new url_1.URL(urlToVerify.resolveToString(inputs));
    }
    preVerifyStackSubStep(stackName) {
        while (!this.preVerifyStack(stackName)) {
            new automation_step_simulation_1.AutomationStepSimulation(new sleep_step_1.SleepStep(new aws_cdk_lib_1.Stack(), 'sleep', {
                sleepSeconds: 2,
            }), this.props).invoke({});
        }
    }
    preVerifyStack(stackName) {
        const state = this.getStackState(stackName);
        if (this.isStackOperationInProgress(state)) {
            return false;
        }
        if (state?.StackStatus !== 'ROLLBACK_COMPLETE') {
            return true;
        }
        console.log(`Deleting stack ${stackName} because we are retrying a create on an existing stack in ROLLBACK_COMPLETE state.`);
        new automation_step_simulation_1.AutomationStepSimulation(new delete_stack_step_1.DeleteStackStep(new aws_cdk_lib_1.Stack(), 'deleteStack', {
            stackNameVariable: new string_variable_1.HardCodedString(stackName),
        }), this.props).invoke({});
        return false;
    }
    isStackOperationInProgress(state) {
        return state !== null && CreateStackSimulation.InProgressStatuses.includes(stack_status_1.StackStatus[state.StackStatus]);
    }
    createStack(inputs) {
        const apiParams = prune_and_transform_record_1.pruneAndTransformRecord({
            StackName: this.creatStackStep.stackName,
            Capabilities: this.creatStackStep.capabilities,
            ClientRequestToken: this.creatStackStep.clientRequestToken,
            NotificationARNs: this.creatStackStep.notificationARNs,
            OnFailure: this.creatStackStep.onStackFailure,
            ResourceTypes: this.creatStackStep.resourceTypes,
            RoleARN: this.creatStackStep.roleArn,
            StackPolicyBody: this.creatStackStep.stackPolicyBody,
            StackPolicyURL: this.creatStackStep.stackPolicyUrl,
            Tags: this.creatStackStep.tags,
            TemplateBody: this.creatStackStep.templateBody,
            TemplateURL: this.creatStackStep.templateUrl,
            TimeoutInMinutes: this.creatStackStep.timeoutInMinutes,
        }, x => x.resolve(inputs));
        const parameters = this.resolveSecureStringsInParametersInput(this.creatStackStep.parameters?.resolveToMapList(inputs));
        if (parameters) {
            apiParams.Parameters = parameters;
        }
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'createStack', {
            service: 'CloudFormation',
            pascalCaseApi: 'CreateStack',
            apiParams: apiParams,
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.STRING,
                    selector: '$.StackId',
                    name: 'StackId',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Create stack failed for ${apiParams.StackName}: ${result.stackTrace}`);
        }
        return result.outputs?.['createStack.StackId'];
    }
    resolveSecureStringsInParametersInput(parameters) {
        if (!parameters) {
            return null;
        }
        for (const parameter of parameters) {
            const value = parameter.ParameterValue;
            if (!value) {
                continue;
            }
            parameter.ParameterValue = this.props.parameterResolver.resolve(value);
        }
        return parameters;
    }
    getStackState(stackName) {
        const describeResponse = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'describeStack', {
            service: 'CloudFormation',
            pascalCaseApi: 'DescribeStacks',
            apiParams: {
                StackName: stackName,
            },
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.STRING_MAP,
                    name: 'Stack',
                    selector: '$.Stacks[0]',
                }],
        }), this.props).invoke({});
        return describeResponse.outputs?.['describeStack.Stack'] ?? null;
    }
    waitForStackCompletion(stackName) {
        wait_for_and_assert_1.waitForAndAssertStackStatus({
            stackName: stackName,
            waitForStatus: [
                stack_status_1.StackStatus.CREATE_FAILED,
                stack_status_1.StackStatus.CREATE_COMPLETE,
                stack_status_1.StackStatus.ROLLBACK_FAILED,
                stack_status_1.StackStatus.ROLLBACK_COMPLETE,
                stack_status_1.StackStatus.DELETE_IN_PROGRESS,
                stack_status_1.StackStatus.DELETE_FAILED,
                stack_status_1.StackStatus.UPDATE_IN_PROGRESS,
                stack_status_1.StackStatus.UPDATE_COMPLETE_CLEANUP_IN_PROGRESS,
                stack_status_1.StackStatus.UPDATE_COMPLETE,
                stack_status_1.StackStatus.UPDATE_FAILED,
                stack_status_1.StackStatus.UPDATE_ROLLBACK_IN_PROGRESS,
                stack_status_1.StackStatus.UPDATE_ROLLBACK_FAILED,
                stack_status_1.StackStatus.UPDATE_ROLLBACK_COMPLETE_CLEANUP_IN_PROGRESS,
                stack_status_1.StackStatus.UPDATE_ROLLBACK_COMPLETE,
                stack_status_1.StackStatus.REVIEW_IN_PROGRESS,
                stack_status_1.StackStatus.IMPORT_IN_PROGRESS,
                stack_status_1.StackStatus.IMPORT_COMPLETE,
                stack_status_1.StackStatus.IMPORT_ROLLBACK_IN_PROGRESS,
                stack_status_1.StackStatus.IMPORT_ROLLBACK_FAILED,
                stack_status_1.StackStatus.IMPORT_ROLLBACK_COMPLETE,
            ],
            assertStatus: stack_status_1.StackStatus.CREATE_COMPLETE,
        }, this.props);
    }
}
exports.CreateStackSimulation = CreateStackSimulation;
_a = JSII_RTTI_SYMBOL_1;
CreateStackSimulation[_a] = { fqn: "@cdklabs/cdk-ssm-documents.CreateStackSimulation", version: "0.0.17" };
CreateStackSimulation.InProgressStatuses = [
    stack_status_1.StackStatus.ROLLBACK_IN_PROGRESS,
    stack_status_1.StackStatus.DELETE_IN_PROGRESS,
    stack_status_1.StackStatus.UPDATE_IN_PROGRESS,
    stack_status_1.StackStatus.UPDATE_COMPLETE_CLEANUP_IN_PROGRESS,
    stack_status_1.StackStatus.UPDATE_ROLLBACK_IN_PROGRESS,
    stack_status_1.StackStatus.UPDATE_ROLLBACK_COMPLETE_CLEANUP_IN_PROGRESS,
    stack_status_1.StackStatus.REVIEW_IN_PROGRESS,
];
//# sourceMappingURL=data:application/json;base64,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