"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SsmDocument = exports.DocumentFormat = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const os_1 = require("os");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ssm_1 = require("aws-cdk-lib/aws-ssm");
const constructs_1 = require("constructs");
const string_variable_1 = require("../interface/variables/string-variable");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const yaml = require('js-yaml');
var DocumentFormat;
(function (DocumentFormat) {
    DocumentFormat[DocumentFormat["JSON"] = 0] = "JSON";
    DocumentFormat[DocumentFormat["YAML"] = 1] = "YAML";
})(DocumentFormat = exports.DocumentFormat || (exports.DocumentFormat = {}));
class SsmDocument extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.documentName = props.documentName ?? id;
        this.description = props.description ?? this.documentName;
        this.header = props.header;
        this.assumeRole = props.assumeRole;
        this.docOutputs = props.docOutputs ?? [];
        this.docInputs = props.docInputs ?? [];
        if (this.assumeRole && this.assumeRole instanceof string_variable_1.StringVariable &&
            !this.docInputs.map(i => i.name).includes(this.assumeRole.reference)) {
            throw new Error('Assume role specified but not provided in inputs: ' + this.assumeRole);
        }
        this.props = props;
        const isYaml = this.props.documentFormat == DocumentFormat.YAML;
        new aws_ssm_1.CfnDocument(this, 'Resource', {
            ...this.props,
            ...{
                content: aws_cdk_lib_1.Lazy.any({
                    produce: () => {
                        const doc = this.buildSsmDocument();
                        return JSON.parse(JSON.stringify(doc));
                    },
                }),
                documentFormat: isYaml ? 'YAML' : 'JSON',
                documentType: this.documentType(),
                tags: [{ key: 'CdkGenerated', value: 'true' }, ...(this.props.tags || [])],
            },
        });
    }
    /**
     * The SSM Document inputs optionally specify a number of parameters including allowedValues, minItems etc.
     * This function builds an object containing the relevant (declared) input parameters.
     * The return object will be used to build the yaml/json representation of the document.
     */
    formatInputs() {
        const ssmInputs = {};
        this.docInputs.forEach(inp => {
            ssmInputs[inp.name] = inp.toSsm();
        });
        return ssmInputs;
    }
    /**
       * Synthesize before calling this function!
       * You can use this to Synthesize: cdk.SynthUtils.synthesize(stack);
       *
       * Converts the objects define in the SSM Document (including all of the steps) to an SSM document string.
       * The format is dependency on the documentFormat property provided to the class.
       * The yaml can be used as is and will behave (or at least should behave) as was simulated in the runSimulation().
       * @returns a string representation of this document as an SSM formatted yaml/json.
       */
    print() {
        const isYaml = this.props.documentFormat == DocumentFormat.YAML;
        if (isYaml) {
            // Prepend the document with the header if defined
            return this.headerWithComments() + yaml.dump(this.buildSsmDocument());
        }
        else {
            // There is no way to provide the header comment in a json document
            return JSON.stringify(this.buildSsmDocument());
        }
    }
    /**
       * Prepend header line with # if it does not start with #.
       * If no header is provided, this method will return an emptry string.
       * Headers are only available in yaml format.
       */
    headerWithComments() {
        if (this.header == undefined) {
            return '';
        }
        return this.header.split(os_1.EOL).map(line => line.trim().startsWith('#') ? line : '# ' + line).join(os_1.EOL) +
            // "---" demarks the end of the header and the start of the document.
            '\n---\n';
    }
}
exports.SsmDocument = SsmDocument;
_a = JSII_RTTI_SYMBOL_1;
SsmDocument[_a] = { fqn: "@cdklabs/cdk-ssm-documents.SsmDocument", version: "0.0.17" };
//# sourceMappingURL=data:application/json;base64,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