"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabRunnerAutoscaling = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const cdk = require("aws-cdk-lib");
const asg = require("aws-cdk-lib/aws-autoscaling");
const aws_autoscaling_hooktargets_1 = require("aws-cdk-lib/aws-autoscaling-hooktargets");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const assets = require("aws-cdk-lib/aws-s3-assets");
const sns = require("aws-cdk-lib/aws-sns");
const subscriptions = require("aws-cdk-lib/aws-sns-subscriptions");
const cr = require("aws-cdk-lib/custom-resources");
// eslint-disable-next-line import/no-extraneous-dependencies
const compare_versions_1 = require("compare-versions");
const constructs_1 = require("constructs");
/**
 * GitlabRunnerAutoscaling Construct for create Autoscaling Gitlab Runner.
 */
class GitlabRunnerAutoscaling extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const defaultProps = {
            instanceType: 't3.micro',
            tags: ['gitlab', 'awscdk', 'runner'],
            gitlabUrl: 'https://gitlab.com/',
            gitlabRunnerImage: 'public.ecr.aws/gitlab/gitlab-runner:latest',
            alarms: [
                {
                    AlarmName: 'GitlabRunnerDiskUsage',
                    MetricName: 'disk_used_percent',
                },
            ],
        };
        const runnerProps = { ...defaultProps, ...props };
        if ((0, compare_versions_1.compare)(props.gitlabRunnerVersion, '15.10', '>=') && props.gitlabToken.includes('glrt-') === false) {
            throw new Error('If gitlabRunnerVersion >= 15.10, gitlabtoken please give glrt-xxxxxxx @see https://docs.gitlab.com/ee/ci/runners/new_creation_workflow.html');
        }
        const asset = new assets.Asset(this, 'GitlabRunnerUserDataAsset', {
            path: path.join(__dirname, '../assets/userdata/amazon-cloudwatch-agent.json'),
        });
        const userData = ec2.UserData.forLinux();
        userData.addS3DownloadCommand({
            bucket: asset.bucket,
            bucketKey: asset.s3ObjectKey,
            localFile: '/opt/aws/amazon-cloudwatch-agent/etc/amazon-cloudwatch-agent.json',
        });
        userData.addCommands(...this.createUserData(runnerProps));
        this.instanceRole =
            runnerProps.instanceRole ??
                new iam.Role(this, 'GitlabRunnerInstanceRole', {
                    assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
                    description: 'For EC2 Instance (Gitlab Runner) Role',
                    managedPolicies: [
                        iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
                        iam.ManagedPolicy.fromAwsManagedPolicyName('CloudWatchAgentServerPolicy'),
                        iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonS3ReadOnlyAccess'),
                    ],
                });
        this.vpc = runnerProps.vpc ?? new ec2.Vpc(this, 'VPC');
        this.securityGroup = new ec2.SecurityGroup(this, 'GitlabRunnerSecurityGroup', {
            vpc: this.vpc,
        });
        const instanceProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.instanceRole.roleName],
        });
        const lt = new ec2.CfnLaunchTemplate(this, 'GitlabRunnerLaunchTemplate', {
            launchTemplateData: {
                imageId: ec2.MachineImage.latestAmazonLinux2().getImage(this).imageId,
                instanceType: runnerProps.instanceType,
                instanceMarketOptions: {
                    marketType: runnerProps.spotInstance ? 'spot' : undefined,
                    spotOptions: runnerProps.spotInstance ? {
                        spotInstanceType: 'one-time',
                    } : undefined,
                },
                userData: cdk.Fn.base64(userData.render()),
                blockDeviceMappings: [
                    {
                        deviceName: '/dev/xvda',
                        ebs: {
                            volumeSize: runnerProps.ebsSize ?? 60,
                        },
                    },
                ],
                iamInstanceProfile: {
                    arn: instanceProfile.attrArn,
                },
                securityGroupIds: this.securityGroup.connections.securityGroups.map((m) => m.securityGroupId),
            },
        });
        this.autoscalingGroup = new asg.AutoScalingGroup(this, 'GitlabRunnerAutoscalingGroup', {
            instanceType: new ec2.InstanceType(runnerProps.instanceType),
            autoScalingGroupName: `Gitlab Runners (${runnerProps.instanceType})`,
            vpc: this.vpc,
            vpcSubnets: runnerProps.vpcSubnet,
            machineImage: ec2.MachineImage.latestAmazonLinux2(),
            minCapacity: runnerProps.minCapacity,
            maxCapacity: runnerProps.maxCapacity,
            desiredCapacity: runnerProps.desiredCapacity,
        });
        const cfnAsg = this.autoscalingGroup.node.tryFindChild('ASG');
        cfnAsg.addPropertyDeletionOverride('LaunchConfigurationName');
        cfnAsg.addPropertyOverride('LaunchTemplate', {
            LaunchTemplateId: lt.ref,
            Version: lt.attrLatestVersionNumber,
        });
        this.autoscalingGroup.node.tryRemoveChild('LaunchConfig');
        this.topicAlarm = new sns.Topic(this, 'GitlabRunnerAlarm');
        const alarms = JSON.stringify(runnerProps.alarms);
        // Put alarms at launch
        const registerFunction = new lambda.Function(this, 'GitlabRunnerRegisterFunction', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'autoscaling_events.register',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: cdk.Duration.seconds(60),
            logRetention: logs.RetentionDays.ONE_DAY,
            environment: {
                ALARMS: alarms,
                SNS_TOPIC_ARN: this.topicAlarm.topicArn,
            },
        });
        registerFunction.role?.addToPrincipalPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: ['*'],
            actions: [
                'cloudwatch:PutMetricAlarm',
            ],
        }));
        this.autoscalingGroup.addLifecycleHook('GitlabRunnerLifeCycleHookLaunching', {
            lifecycleTransition: asg.LifecycleTransition.INSTANCE_LAUNCHING,
            notificationTarget: new aws_autoscaling_hooktargets_1.FunctionHook(registerFunction),
            defaultResult: asg.DefaultResult.CONTINUE,
            heartbeatTimeout: cdk.Duration.seconds(60),
        });
        // Add an alarm action to terminate invalid instances
        const alarmAction = new lambda.Function(this, 'GitlabRunnerAlarmAction', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'autoscaling_events.on_alarm',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: cdk.Duration.seconds(60),
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        alarmAction.role?.addToPrincipalPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: ['*'],
            actions: [
                'autoscaling:SetInstanceHealth',
            ],
        }));
        const alarmSubscription = new subscriptions.LambdaSubscription(alarmAction);
        this.topicAlarm.addSubscription(alarmSubscription);
        // Unregister gitlab runners and remove alarms on instance termination or CFn stack deletion
        const unregisterRole = new iam.Role(this, 'GitlabRunnerUnregisterRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            description: 'For Gitlab Runner Unregistering Function Role',
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        unregisterRole.addToPrincipalPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: ['*'],
            actions: [
                'ssm:SendCommand',
                'autoscaling:DescribeAutoScalingGroups',
                'cloudwatch:DeleteAlarms',
            ],
        }));
        const unregisterFunction = new lambda.Function(this, 'GitlabRunnerUnregisterFunction', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'autoscaling_events.unregister',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: cdk.Duration.seconds(60),
            role: unregisterRole,
            logRetention: logs.RetentionDays.ONE_DAY,
            environment: {
                ALARMS: alarms,
            },
        });
        this.autoscalingGroup.addLifecycleHook('GitlabRunnerLifeCycleHookTerminating', {
            lifecycleTransition: asg.LifecycleTransition.INSTANCE_TERMINATING,
            notificationTarget: new aws_autoscaling_hooktargets_1.FunctionHook(unregisterFunction),
            defaultResult: asg.DefaultResult.CONTINUE,
            heartbeatTimeout: cdk.Duration.seconds(60),
        });
        const unregisterCustomResource = new lambda.Function(this, 'GitlabRunnerUnregisterCustomResource', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'autoscaling_events.on_event',
            runtime: lambda.Runtime.PYTHON_3_8,
            role: unregisterRole,
            logRetention: logs.RetentionDays.ONE_DAY,
            environment: {
                ALARMS: alarms,
            },
        });
        const unregisterProvider = new cr.Provider(this, 'GitlabRunnerUnregisterProvider', {
            onEventHandler: unregisterCustomResource,
        });
        const customResource = new cdk.CustomResource(this, 'GitlabRunnerCustomResource', {
            serviceToken: unregisterProvider.serviceToken,
            properties: {
                AutoScalingGroupNames: [this.autoscalingGroup.autoScalingGroupName],
            },
        });
        customResource.node.addDependency(unregisterProvider);
        new cdk.CfnOutput(this, 'GitlabRunnerAutoScalingGroupArn', {
            value: this.autoscalingGroup.autoScalingGroupArn,
        });
    }
    dockerVolumesList(dockerVolume) {
        let tempString = '--docker-volumes "/var/run/docker.sock:/var/run/docker.sock"';
        if (dockerVolume) {
            let tempList = [];
            dockerVolume.forEach(e => {
                tempList.push(`"${e.hostPath}:${e.containerPath}"`);
            });
            tempList.forEach(e => {
                tempString = `${tempString} --docker-volumes ${e}`;
            });
        }
        return tempString;
    }
    /**
     * @param props
     * @returns Array.
     */
    createUserData(props) {
        return [
            'yum update -y',
            'sleep 15 && amazon-linux-extras install docker && yum install -y amazon-cloudwatch-agent && systemctl start docker && usermod -aG docker ec2-user && chmod 777 /var/run/docker.sock',
            'systemctl restart docker && systemctl enable docker && systemctl start amazon-cloudwatch-agent && systemctl enable amazon-cloudwatch-agent',
            `docker run -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock \
      --name gitlab-runner-register ${props.gitlabRunnerImage} register --non-interactive --url ${props.gitlabUrl} ${(0, compare_versions_1.compare)(props.gitlabRunnerVersion, '15.10', '>=') ? '--token' : '--registration-token'} ${props.gitlabToken} \
      --docker-pull-policy if-not-present ${this.dockerVolumesList(props?.dockerVolumes)} \
      --executor docker --docker-image "alpine:latest" --description "A Runner on EC2 Instance (${props.instanceType})" \
      ${(0, compare_versions_1.compare)(props.gitlabRunnerVersion, '15.10', '>=') ? undefined : `--tag-list "${props.tags?.join(',')}" `} --docker-privileged`,
            `sleep 2 && docker run --restart always -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock --name gitlab-runner ${props.gitlabRunnerImage}`,
        ];
    }
}
exports.GitlabRunnerAutoscaling = GitlabRunnerAutoscaling;
_a = JSII_RTTI_SYMBOL_1;
GitlabRunnerAutoscaling[_a] = { fqn: "cdk-gitlab-runner.GitlabRunnerAutoscaling", version: "2.2.374" };
//# sourceMappingURL=data:application/json;base64,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