"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const batch_generated_1 = require("./batch.generated");
/**
 * Batch Job Queue.
 *
 * Defines a batch job queue to define how submitted batch jobs
 * should be ran based on specified batch compute environments.
 */
class JobQueue extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.jobQueueName,
        });
        if (props.computeEnvironments.length === 0) {
            throw new Error('computeEnvironments must be non-empty');
        }
        const jobQueue = new batch_generated_1.CfnJobQueue(this, 'Resource', {
            computeEnvironmentOrder: props.computeEnvironments.map(cp => ({
                computeEnvironment: cp.computeEnvironment.computeEnvironmentArn,
                order: cp.order,
            })),
            jobQueueName: this.physicalName,
            priority: props.priority || 1,
            state: props.enabled === undefined ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
        });
        this.jobQueueArn = this.getResourceArnAttribute(jobQueue.ref, {
            service: 'batch',
            resource: 'job-queue',
            resourceName: this.physicalName,
        });
        this.jobQueueName = this.getResourceNameAttribute(jobQueue.ref);
    }
    /**
     * Fetches an existing batch job queue by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param jobQueueArn
     */
    static fromJobQueueArn(scope, id, jobQueueArn) {
        const stack = core_1.Stack.of(scope);
        const jobQueueName = stack.parseArn(jobQueueArn).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.jobQueueArn = jobQueueArn;
                this.jobQueueName = jobQueueName;
            }
        }
        return new Import(scope, id);
    }
}
exports.JobQueue = JobQueue;
//# sourceMappingURL=data:application/json;base64,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