from typing import List, Dict, Any, Union
from csle_common.dao.emulation_config.vulnerability_type import VulnType
from csle_common.dao.emulation_config.transport_protocol import TransportProtocol
from csle_common.dao.emulation_config.credential import Credential
from csle_common.util.general_util import GeneralUtil


class NodeVulnerabilityConfig:
    """
    A DTO object representing a vulnerability of a container in the emulation environment
    """

    def __init__(self, ip: str, vuln_type: VulnType, name: str, port: Union[int, None],
                 protocol: TransportProtocol, credentials: List[Credential] = None,
                 cvss: float = 2.0, cve: str = None, service: str = None, root: bool = False,
                 docker_gw_bridge_ip: str = "", physical_host_ip: str = ""):
        """
        Initializes the DTO

        :param ip: the ip of the node
        :param vuln_type: the vulnerability type
        :param name: the name of the vulnerability
        :param port: the port of the service of the vulnerability
        :param protocol: the protocol of the service of the vulnerability
        :param credentials: the credentials of the vulnerability
        :param cvss: the CVSS of the vulnerability
        :param cve: the CVE of the vulnerability
        :param service: the service of the vulnerability
        :param root: if the vulnerability gives root or not
        :param docker_gw_bridge_ip: IP to reach the container from the host network
        :param physical_host_ip: IP of the physical host where the container is running
        """
        self.ip = ip
        self.vuln_type = vuln_type
        self.name = name
        self.port = port
        self.protocol = protocol
        self.credentials = credentials
        self.cvss = cvss
        self.cve = cve
        self.service = service
        self.root = root
        self.docker_gw_bridge_ip = docker_gw_bridge_ip
        self.physical_host_ip = physical_host_ip

    def to_dict(self) -> Dict[str, Any]:
        """
        :return: a dict representation of the object
        """
        d = {}
        d["ip"] = self.ip
        d["vuln_type"] = self.vuln_type
        d["name"] = self.name
        d["port"] = self.port
        d["protocol"] = self.protocol
        d["credentials"] = list(map(lambda x: x.to_dict(), self.credentials))
        d["root"] = self.root
        d["cve"] = self.cve
        d["docker_gw_bridge_ip"] = self.docker_gw_bridge_ip
        d["physical_host_ip"] = self.physical_host_ip
        return d

    @staticmethod
    def from_dict(d: Dict[str, Any]) -> "NodeVulnerabilityConfig":
        """
        Convert a dict representation to a DTO representation

        :return: a dto representation of the object
        """
        dto = NodeVulnerabilityConfig(ip=d["ip"], vuln_type=d["vuln_type"],
                                      name=d["name"], port=d["port"],
                                      protocol=d["protocol"],
                                      credentials=list(map(lambda x: Credential.from_dict(x), d["credentials"])),
                                      root=d["root"], cve=d["cve"],
                                      docker_gw_bridge_ip=d["docker_gw_bridge_ip"],
                                      physical_host_ip=d["physical_host_ip"])
        return dto

    def __str__(self) -> str:
        """
        :return: a string representation of the object
        """
        return f"node_internal_ip:{self.ip}, vuln_type:{self.vuln_type}, name:{self.name}, port:{self.port}, " \
               f"protocol: {self.protocol}, credentials: {list(map(lambda x: str(x), self.credentials))}, " \
               f"cvss: {self.cvss}, cve: {self.cve}, service: {self.service}, root:{self.root}, " \
               f"docker_gw_bridge_ip: {self.docker_gw_bridge_ip}, physical_host_ip: {self.physical_host_ip}"

    def to_json_str(self) -> str:
        """
        Converts the DTO into a json string

        :return: the json string representation of the DTO
        """
        import json
        json_str = json.dumps(self.to_dict(), indent=4, sort_keys=True)
        return json_str

    def to_json_file(self, json_file_path: str) -> None:
        """
        Saves the DTO to a json file

        :param json_file_path: the json file path to save  the DTO to
        :return: None
        """
        import io
        json_str = self.to_json_str()
        with io.open(json_file_path, 'w', encoding='utf-8') as f:
            f.write(json_str)

    @staticmethod
    def from_json_file(json_file_path: str) -> "NodeVulnerabilityConfig":
        """
        Reads a json file and converts it to a DTO

        :param json_file_path: the json file path
        :return: the converted DTO
        """
        import io
        import json
        with io.open(json_file_path, 'r') as f:
            json_str = f.read()
        return NodeVulnerabilityConfig.from_dict(json.loads(json_str))

    def copy(self) -> "NodeVulnerabilityConfig":
        """
        :return: a copy of the DTO
        """
        return NodeVulnerabilityConfig.from_dict(self.to_dict())

    def create_execution_config(self, ip_first_octet: int) -> "NodeVulnerabilityConfig":
        """
        Creates a new config for an execution

        :param ip_first_octet: the first octet of the IP of the new execution
        :return: the new config
        """
        config = self.copy()
        config.ip = GeneralUtil.replace_first_octet_of_ip(ip=config.ip, ip_first_octet=ip_first_octet)
        return config
