"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SonarCodeScanner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const pdk_nag_1 = require("@aws-prototyping-sdk/pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const sonarqube_commands_1 = require("./sonarqube-commands");
const unpackSourceAndArtifacts = (includeGlobsForScan) => [
    'export BUILT_ARTIFACT_URI=`aws codebuild batch-get-builds --ids $SYNTH_BUILD_ID | jq -r \'.builds[0].secondaryArtifacts[] | select(.artifactIdentifier == "Synth__") | .location\' | awk \'{sub("arn:aws:s3:::","s3://")}1\' $1`',
    "export SYNTH_SOURCE_URI=`aws codebuild batch-get-builds --ids $SYNTH_BUILD_ID | jq -r '.builds[0].sourceVersion' | awk '{sub(\"arn:aws:s3:::\",\"s3://\")}1' $1`",
    "aws s3 cp $SYNTH_SOURCE_URI source.zip",
    "aws s3 cp $BUILT_ARTIFACT_URI built.zip",
    "unzip source.zip -d src",
    "unzip built.zip -d built",
    "rm source.zip built.zip",
    `rsync -a built/* src --include="*/" ${includeGlobsForScan
        ? includeGlobsForScan.map((g) => `--include ${g}`).join(" ")
        : ""} --include="**/coverage/**" --include="**/cdk.out/**" --exclude="**/node_modules/**/*" --exclude="**/.env/**" --exclude="*" --prune-empty-dirs`,
];
const owaspScan = () => `npx owasp-dependency-check --format HTML --out src/reports --exclude '**/.git/**/*' --scan src --enableExperimental --bin /tmp/dep-check --disableRetireJS`;
const cfnNagScan = (cdkOutDir, cfnNagIgnorePath) => cdkOutDir
    ? `cfn_nag ${cfnNagIgnorePath ? `--deny-list-path=${cfnNagIgnorePath}` : ""} built/${cdkOutDir}/**/*.template.json --output-format=json > src/reports/cfn-nag-report.json`
    : 'echo "skipping cfn_nag as no cdkOutDir was specified.';
class SonarCodeScanner extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const sonarQubeToken = new aws_secretsmanager_1.Secret(this, "SonarQubeToken");
        const synthBuildProject = aws_codebuild_1.Project.fromProjectArn(this, "SynthBuildProject", props.synthBuildArn);
        const validationProject = new aws_codebuild_1.Project(this, "ValidationProject", {
            environment: {
                buildImage: aws_codebuild_1.LinuxBuildImage.STANDARD_5_0,
            },
            environmentVariables: {
                SONARQUBE_TOKEN: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.SECRETS_MANAGER,
                    value: sonarQubeToken.secretArn,
                },
                SONARQUBE_ENDPOINT: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.PLAINTEXT,
                    value: props.sonarqubeEndpoint,
                },
                PROJECT_NAME: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.PLAINTEXT,
                    value: props.sonarqubeProjectName,
                },
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: "0.2",
                env: {
                    shell: "bash",
                },
                phases: {
                    install: {
                        commands: ["npm install -g aws-cdk", "gem install cfn-nag"],
                    },
                    build: {
                        commands: [
                            "export RESOLVED_SOURCE_VERSION=`aws codebuild batch-get-builds --ids $SYNTH_BUILD_ID | jq -r '.builds[0].resolvedSourceVersion'`",
                            ...unpackSourceAndArtifacts(props.includeGlobsForScan),
                            ...sonarqube_commands_1.createSonarqubeProject(props),
                            "mkdir -p src/reports",
                            owaspScan(),
                            cfnNagScan(props.cdkOutDir, props.cfnNagIgnorePath),
                            "cd src",
                            sonarqube_commands_1.sonarqubeScanner(props.excludeGlobsForScan),
                            ...sonarqube_commands_1.generateSonarqubeReports(),
                            ...(props.preArchiveCommands || []),
                        ],
                    },
                },
            }),
        });
        validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ["codebuild:BatchGetBuilds"],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [synthBuildProject.projectArn],
        }));
        validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ["s3:GetObject*"],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [props.artifactBucketArn, `${props.artifactBucketArn}/**`],
        }));
        props.artifactBucketKeyArn &&
            validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
                actions: ["kms:Decrypt", "kms:DescribeKey"],
                effect: aws_iam_1.Effect.ALLOW,
                resources: [props.artifactBucketKeyArn],
            }));
        synthBuildProject.onBuildSucceeded("OnSynthSuccess", {
            target: new aws_events_targets_1.CodeBuildProject(validationProject, {
                event: aws_events_1.RuleTargetInput.fromObject({
                    environmentVariablesOverride: [
                        {
                            name: "SYNTH_BUILD_ID",
                            type: "PLAINTEXT",
                            value: aws_events_1.EventField.fromPath("$.detail.build-id"),
                        },
                    ],
                }),
            }),
        });
        new aws_cdk_lib_1.CfnOutput(this, "SonarqubeSecretArn", {
            exportName: "SonarqubeSecretArn",
            value: sonarQubeToken.secretArn,
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(sonarQubeToken, [
            {
                id: "AwsSolutions-SMG4",
                reason: "Key rotation is not possible as a user token needs to be generated from Sonarqube",
            },
        ]);
        const stack = aws_cdk_lib_1.Stack.of(this);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(validationProject.role, [
            {
                id: "AwsSolutions-IAM5",
                reason: "Validation CodeBuild project requires access to the ArtifactsBucket and ability to create logs.",
                appliesTo: [
                    {
                        regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:logs:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:log-group:/aws/codebuild/<.*SonarCodeScannerValidationProject.*>:\\*$/g`,
                    },
                    {
                        regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:codebuild:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:report-group/<.*SonarCodeScannerValidationProject.*>-\\*$/g`,
                    },
                    {
                        regex: `/^Action::s3:GetObject\\*$/g`,
                    },
                    {
                        regex: "/^Resource::<ArtifactsBucket.*.Arn>/\\*\\*$/g",
                    },
                ],
            },
        ], true);
    }
}
exports.SonarCodeScanner = SonarCodeScanner;
_a = JSII_RTTI_SYMBOL_1;
SonarCodeScanner[_a] = { fqn: "@aws-prototyping-sdk/pipeline.SonarCodeScanner", version: "0.12.6" };
//# sourceMappingURL=data:application/json;base64,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