# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['unipressed', 'unipressed.types']

package_data = \
{'': ['*']}

install_requires = \
['requests>=2.28.1,<3.0.0', 'typing-extensions>=4.3.0,<5.0.0']

setup_kwargs = {
    'name': 'unipressed',
    'version': '0.2.0',
    'description': 'Comprehensive Python client for the Uniprot REST API',
    'long_description': '# Unipressed\n\n**Please visit the [project website](https://multimeric.github.io/Unipressed/) for more comprehensive documentation.**\n\n## Introduction\n\nUnipressed (Uniprot REST) is an API client for the protein database [Uniprot](https://www.uniprot.org/).\nIt provides thoroughly typed and documented code to ensure your use of the library is easy, fast, and correct!\n\n### Example\nLet\'s say we\'re interested in very long proteins that are encoded within a chloroplast, in any organism:\n```python\nimport json\nfrom unipressed import UniprotkbSearch\n\nfor record in UniprotkbSearch(\n    query={\n        "and_": [\n            {"organelle": "chloroplast"},\n            {"length": (5000, "*")}\n        ]\n    },\n    fields=["length", "gene_names"]\n).each_record():\n    print(json.dumps(record, indent=4))\n```\n\nThis will print:\n```json\n{\n    "primaryAccession": "A0A088CK67",\n    "genes": [\n        {\n            "geneName": {\n                "evidences": [\n                    {\n                        "evidenceCode": "ECO:0000313",\n                        "source": "EMBL",\n                        "id": "AID67672.1"\n                    }\n                ],\n                "value": "ftsH"\n            }\n        }\n    ],\n    "sequence": {\n        "length": 5242\n    }\n}\n```\n\n### Advantages\n\n* Detailed type hints for autocompleting queries as you type\n* Autocompletion for return fields\n* Documentation for each field\n* Automatic results parsing, for `json`, `tsv`, `list`, and `xml`\n* Built-in pagination, so you don\'t have to handle any of that yourself!\n* Most of the API is automatically generated, ensuring very rapid updates whenever the API changes\n* Thoroughly tested, with 41 unit tests and counting!\n\n## Usage\n\n### Installation\n\nIf you\'re using poetry:\n```bash\npoetry add unipressed\n```\n\nOtherwise:\n```bash\npip install unipressed\n```\n\n\n### Query Syntax\n\nYou can\'t go wrong by following the type hints.\nI strongly recommend using something like [`pylance`](https://marketplace.visualstudio.com/items?itemName=ms-python.vscode-pylance) for [Visual Studio Code](https://code.visualstudio.com/), which will provide automatic completions and warn you when you have used the wrong syntax.\n\nIf you already know how to use the Uniprot query language, you can always just input your queries as strings:\n```python\nUniprotkbSearch(query="(gene:BRCA*) AND (organism_id:10090)")\n```\n\n\nHowever, if you want some built-in query validation and code completion using Python\'s type system, then you can instead use a dictionary.\nThe simplest query is a dictionary with a single key: \n```python\nUniprotkbSearch(query={ "family": "kinase"})\n```\nFor brevity, for the rest of this section we will omit everything but the value of the `query` argument.\n\nYou can compile more complex queries using the `and_`, `or_` and `not_` keys.\nThese first two operators take a list of query dictionaries: \n```python\n{\n    "and_": [\n        {"family": "kinase"},\n        {"organism_id": "9606"},\n    ]\n}\n```\n\nMost "leaf" nodes of the query tree (ie those that aren\'t operators like `and_`) are strings, integers or floats, which you input as normal Python literals as you can see above.\nFor string fields, you also have access to wildcards, namely the `*` character. \nFor example, if you want every human protein belonging to a gene whose name starts with `PRO`, you could use:\n```python\n{\n    "and_": [\n        {"gene": "PRO*"},\n        {"organism_id": "9606"},\n    ]\n}\n```\n\nA few query fields are *ranges*, which you input using a tuple with two elements, indicating the start and end of the range.\nIf you use the literal `"*"` then you can leave the range open at one end. \nFor example, this query returns any protein that is in the range $(5000, \\infty)$\n```python\n{"length": (5000, "*")}\n```\n\nFinally, a few query fields take dates.\nThese you input as a Python `datetime.date` object.\nFor example, to find proteins added to UniProt since July 2022, we would do:\n```python\nfrom datetime import date\n\nUniprotkbSearch(query={"date_created": (date(2022, 7, 1), "*")})\n```\n\n### Use with Visual Studio Code\nTo get VS Code to offer suggestions, press the `Trigger Suggest` shortcut which is usually bound to `Ctrl + Space`.\nIn particular, code completion generally won\'t work *until* you open a string literal using a quotation mark.\n\nSecondly, to get live access to the documentation, you can either use the `Show Hover` shortcut, which is usually bound to `Ctrl + K, Ctrl + I`, or you can install the [`docs-view`](https://marketplace.visualstudio.com/items?itemName=bierner.docs-view) extension, which lets you view the docstrings in the sidebar without interfering with your code.\n\n## Changelog\n\n### 0.2.0\n\n**Note, if you are using Visual Studio Code, please update Pylance to at least version 2022.8.20.\nA bug in earlier versions will give you false errors with this new release of `unipressed`**.\n\n#### Added\n* Also allow strings within the query dictionary, so that e.g. this is now allowed:\n    ```python\n    {\n        "and_": [\n            "foo*",\n            "*bar"\n        ]\n    }\n    ```\n    This will search for all proteins that have any field that starts with `foo` and any field that ends with `bar`.\n\n* Auto generated docstrings for all fields\n* Examples to the documentation of each field\n* Certain missing query fields for the `arba` dataset:\n    * `cc_scl_term`\n* Certain missing query fields for the `proteomes` dataset:\n    * `organism_id`\n    * `taxonomy_id`\n* Certain missing query fields for the `unirule` dataset:\n    * `cc_scl_term`\n* Certain missing query fields for the `uniparc` dataset:\n    * `taxonomy_id`\n* Certain missing query fields for the `uniprotkb` dataset:\n    * `organism_id`\n    * `taxonomy_id`\n    * `virus_host_id`\n\n#### Removed\n* Uniprot seem to have removed certain `uniprokb` query fields, so these are now not part of the accepted query type:\n    * `ft_metal`\n    * `ftlen_metal`\n    * `ft_ca_bind`\n    * `ftlen_ca_bind`\n    * `ft_np_bind`\n    * `ftlen_np_bind`\n* Likewise, some `uniprotkb` return fields have been removed:\n    * `ft_ca_bind`\n    * `ft_metal`\n    * `ft_np_bind`\n\n#### Internal\n* Move from `pyhumps` to `inflection` for code generation\n* Add a test for the date field\n* Added tests for all datasets\n* Add types for code generation API',
    'author': 'Michael Milton',
    'author_email': 'michael.r.milton@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://multimeric.github.io/Unipressed',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
