"""
---
    Copyright (c) 2018 Baskar Ganapathysubramanian, Balaji Sesha Sarath Pokuri

    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in all
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
    SOFTWARE.
---
"""
"""
Example 0: Getting started with bayesian optimization using PARyOpt
1- dimensional

What you will learn:
* setting up logger
* initializing the optimization
* updating iterations
* getting optimization related data such as function evaluations, existing data points, ..
* visualization
"""
import logging
import time

import numpy as np
import examples.examples_all_functions as exf
from PARyOpt import BayesOpt
from PARyOpt.evaluators import FunctionEvaluator

if __name__ == "__main__":
    # logging setup -- if this is not done, it will be streamed to stdout
    logger = logging.getLogger()
    logger.setLevel(logging.INFO)  # either NOTSET, INFO, DEBUG, WARNING, ERROR, CRITICAL -- different levels of log
    log_file_name = 'example0_{}.log'.format(time.strftime("%Y.%m.%d-%H%M%S"))
    fh = logging.FileHandler(log_file_name, mode='a')

    # log file format
    formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
    fh.setFormatter(formatter)
    logger.addHandler(fh)

    # first line of the log
    logger.info('Example run for Bayesian optimization')

    # let us define basic parameters here

    # dimensionality of problem
    n_dim = 1

    # bounds, as a numpy array
    # l_bound[0] is the lower bound for dimension 0, etc.
    l_bound = np.asarray([-12.])
    u_bound = np.asarray([12.])

    # number of cost function calls to create the initial surrogate
    n_init = 2

    # maximum number of iterations
    max_iter = 10

    # cost function : simple parabola
    cost_function = exf.parabolic_cost_function
    # how to evaluate the cost function - we will use the serial, in-script evaluator
    evaluator = FunctionEvaluator(cost_function)

    # kappa strategy : defines exploration vs exploitation .. we will use a pure exploration here
    # large values of kappa are exploratory (prioritize areas where confidence is low),
    # small values are exploitatory (prioritize areas where the surrogate seems optimal)
    # the following definition returns a value of 1000 for any iteration value
    def my_kappa(iteration: int) -> float:
        """
        return a constant kappa value of 1000.
        :param iteration: iteration
        :return:
        """
        return 1000.0


    # instantiate the optimizer with cost function, bounds, initial evaluation points,
    # type of kernel function (squared exponential), acquisition function (lower confidence bound)
    # and a kappa strategy (constant here)

    b_opt = BayesOpt(cost_function=evaluator,
                     l_bound=l_bound, u_bound=u_bound, n_dim=n_dim,
                     n_init=n_init,
                     kern_function='sqr_exp',
                     acq_func='LCB',
                     kappa_strategy=my_kappa,
                     if_restart=False)

    logger.info('BO initialized')

    # visualize the surrogate generated by the optimizer, with no hyper-parameter optimization
    exf.visualize_fit(b_opt)
    # estimate best kernel parameters: perform hyper parameter optimization
    b_opt.estimate_best_kernel_parameters(theta_bounds=[[0.1, 10]])
    exf.visualize_fit(b_opt)

    # an example of how to evaluate the surrogate at any point and see how far it is from the actual cost function
    pt_to_evaluate = np.asarray([0.0])
    mean, variance = b_opt.evaluate_surrogate_at(pt_to_evaluate)
    true_cost_function = exf.parabolic_cost_function(pt_to_evaluate)

    print('At {}, surrogate mean:{}, variance:{}, true value:{}'.format(pt_to_evaluate, mean, variance,
                                                                        true_cost_function))

    # update iterations
    for curr_iter in range(max_iter):
        # update_iter finds acquisition function optima and updates the prior to get the posterior
        # (optimize acquisition function to pick the next set of points,
        # evaluate the cost function at those points, and update the surrogate with the new values)
        b_opt.update_iter()
        # estimate and set best kernel parameters
        b_opt.estimate_best_kernel_parameters(theta_bounds=[[0.1, 10]])
        # visualize it every other iteration
        if curr_iter % 2:
            exf.visualize_fit(b_opt)

    # get the population and respective function values from the optimizer
    total_population, function_values = b_opt.get_total_population()
    # get current best evaluated value
    best_location, best_value = b_opt.get_current_best()

    result_txt = 'Optimization done for {} iterations, best evaluation is at {} with cost: {}'. \
        format(b_opt.get_current_iteration(), best_location, best_value)

    logger.info(result_txt)
    print(result_txt)

    csv_file = './temp/example_0_data.csv'
    b_opt.export_csv(csv_file)
    logger.info('Data exported to {}'.format(csv_file))
