import subprocess
from typing import List

import os
import json
import re
import hcl
import logging
from src.logging import configure_logging

configure_logging()
logger = logging.getLogger()


def run_cmd(cmd):
    p = subprocess.Popen(cmd, shell=True)
    p.wait()
    return p.returncode


def create_command(arguments_entered: List[str]) -> str:
    cloud_providers = ['aws', 'azure', 'gcloud']
    logger.debug("arguments_entered: {0}".format(arguments_entered))
    if '-cloud' in arguments_entered:
        arguments_entered.remove('-cloud')
    for i in cloud_providers:
        if i in arguments_entered:
            arguments_entered.remove(i)
    return ' '.join(arguments_entered)


def build_remote_backend_tf_file(storage_type):
    if os.path.exists("remote_backend.tf"):
        logger.info("remote_backend.tf exists, not generating")
        return True
    logger.debug("remote_backend.tf not found, generating")
    logger.debug("storage_type: {}".format(storage_type))
    try:
        with open('remote_backend.tf', 'w') as f:
            f.write("# file generated by wrapper script to configure backend\n")
            f.write("# do not edit or delete!\n")
            f.write("\n")
            f.write("terraform {\n\tbackend \"%s\" {\n\t}\n}\n" % storage_type)
        return True
    except IOError as e:
        logger.error("error creating file: {0}".format('remote_backend.tf'))


def build_tf_state_path(required_vars, var_data):
    for var in required_vars:
        if var in var_data["inline_vars"]:
            required_vars[var] = var_data["inline_vars"][var]
        elif var in var_data["tfvars"]:
            required_vars[var] = var_data["tfvars"][var]
        elif var in var_data["variables_tf"]:
            required_vars[var] = var_data["variables_tf"][var]
        else:
            raise Exception("ERROR: required var %s not defined" % var)
    else:
        path = "terraform/%s/%s/terraform.tfstate" % (required_vars['teamid'], required_vars['prjid'])
        logger.debug("terraform path: %s" % path)
        return path


def parse_vars(var_data, args):
    """
    function to parse variables
    """
    var_data["inline_vars"] = parse_inline_vars(args)
    var_data["tfvars"] = parse_tfvar_files(args)
    var_data["variables_tf"] = parse_var_file("variables.tf")
    logger.debug("parsed variables: %s" % (json.dumps(var_data, indent=2, sort_keys=True)))


def parse_inline_vars(args):
    """
    parse variables defined on the command line (-var foo=bar)
    """
    logger.debug("parsing inline variables")
    results = {}
    if vars(args)['inline_vars'] is None:
        return results
    for var in vars(args)['inline_vars']:
        logger.debug(var)
        match = re.split("\s*=\s*", var, maxsplit=1)
        key = match[0]
        value = match[1]
        results[key] = value
    return results


def parse_tfvar_files(args):
    """
    parse variables defined in terraform.tfvars and files defined in command line (-var-file foo.tfvars)
    """
    tfvar_files = vars(args)['tfvar_files']
    if tfvar_files is None:
        logger.error("Please use .tfvars file to specify the custom parameters.  eg: \"-var-file custom.tfvars\"")
        exit(1)
    if (os.path.isfile('terraform.tfvars')) and ('terraform.tfvars' not in tfvar_files):
        tfvar_files.insert(0, 'terraform.tfvars')
    for file in tfvar_files:
        with open(file) as fh:
            obj = hcl.load(fh)
            return obj
    #         for line in fh:
    #             line = line.rstrip("\r\n")
    #             line = re.sub("^\s+", "", line)
    #             if re.search("^#", line) or re.search("^$", line):
    #                 continue
    #             else:
    #                 match = re.split("\s*=\s*", line, maxsplit=1)
    #                 key = match[0]
    #                 value = re.sub("^\"|^'|\"$|'$", "", match[1])
    #                 results[key] = value
    # return results


def parse_var_file(file):
    """
    parse the variables defined in .tf file
    eg:
    variable name {
        foo = "bar"
    }
    """
    results = {}
    with open(file) as fh:
        data = hcl.load(fh)
    for var in data["variable"]:
        if "default" in data["variable"][var]:
            results[var] = data["variable"][var]["default"]
    return results
