# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['xbotlib']
install_requires = \
['humanize>=3.2.0,<4.0.0', 'slixmpp>=1.6.0,<2.0.0']

extras_require = \
{'redis': ['redis>=3.5.3,<4.0.0'],
 'web': ['aiohttp>=3.7.3,<4.0.0', 'Jinja2>=2.11.2,<3.0.0']}

setup_kwargs = {
    'name': 'xbotlib',
    'version': '0.16.0',
    'description': 'XMPP bots for humans',
    'long_description': '# xbotlib\n\n[![PyPI version](https://badge.fury.io/py/xbotlib.svg)](https://badge.fury.io/py/xbotlib)\n[![Build Status](https://drone.autonomic.zone/api/badges/decentral1se/xbotlib/status.svg?ref=refs/heads/main)](https://drone.autonomic.zone/decentral1se/xbotlib)\n\n## XMPP bots for humans\n\n> status: experimental\n\nA friendly lightweight wrapper around\n[slixmpp](https://slixmpp.readthedocs.io/) for writing XMPP bots in Python. The\ngoal is to make writing and running XMPP bots easy and fun. `xbotlib` is a\n[single file implementation](./xbotlib.py) which can easily be understood and\nextended. The `xbotlib` source code and ideas are largely borrowed from the\nXMPP bot experiments going on in\n[Varia](https://git.vvvvvvaria.org/explore/repos?tab=&sort=recentupdate&q=bots).\n\n- [Install](#install)\n- [Getting Started](#getting-started)\n- [Working with your bot](#working-with-your-bot)\n  - [Documentation](#documentation)\n  - [Commands](#commands)\n  - [Avatars](#avatars)\n  - [Configuration](#configuration)\n    - [Using the `.conf` configuration file](#using-the-conf-configuration-file)\n    - [Using the command-line interface](#using-the-command-line-interface)\n    - [Using the environment](#using-the-environment)\n  - [Storage back-end](#storage-back-end)\n  - [Loading Plugins](#loading-plugins)\n  - [Serving HTTP](#serving-http)\n  - [Invitations](#invitations)\n  - [API Reference](#api-reference)\n    - [Bot.direct(message)](#bot-direct-message)\n    - [Bot.group(message)](#bot-group-message)\n    - [Bot.serve(request)](#bot-serve-request)\n    - [Bot.routes()](#bot-routes)\n    - [Bot.setup()](#bot-setup)\n    - [SimpleMessage](#simplemessage)\n    - [Bot](#bot)\n- [Chatroom](#chatroom)\n- [More Examples](#more-examples)\n- [Deploy your bots](#deploy-your-bots)\n- [Roadmap](#roadmap)\n- [Changes](#changes)\n- [License](#license)\n- [Contributing](#contributing)\n\n## Install\n\n```sh\n$ pip install xbotlib\n```\n\n## Getting Started\n\nPut the following in a `echo.py` file. This bot echoes back whatever message\nyou send it in both direct messages and group messages. In group chats, you\nneed to message the bot directly (e.g. `echobot: hi`) (see [the commands\nsection](#commands) for more).\n\n```python\nfrom xbotlib import Bot\n\nclass EchoBot(Bot):\n    def direct(self, message):\n        self.reply(message.text, to=message.sender)\n\n    def group(self, message):\n        self.reply(message.content, room=message.room)\n\nEchoBot()\n```\n\nAnd then `python echo.py`. You will be asked a few questions in order to load\nthe account details that your bot will be using. This will generate an\n`echobot.conf` file in the same working directory for further use. See the\n[configuration](#configure-your-bot) section for more.\n\nThat\'s it! If you want to go further, continue reading [here](#working-with-your-bot).\n\n## Working with your bot\n\nThe following sections try to cover all the ways you can configure and extend\nyour bot using `xbotlib`. If anything is unclear, please let us know [on the\nchat](#chatroom).\n\n### Documentation\n\nAdd a `help = "my help"` to your `Bot` class like so.\n\n```python\nclass MyBot(Bot):\n    help = """This is my cool help.\n\n    I can list some commands too:\n\n      @foo - some command\n    """\n```\n\nYour bot will then respond to `mybot: @help` invocations. This string can be a\nmulti-line string with indentation. `xbotlib` will try to format this sensibly\nfor showing in chat clients.\n\nSee more in the [commands](#commands) section for more.\n\n### Commands\n\nUsing `@<command>` in direct messages and `<nick>, @<command>` (the `,` is\noptional, anything will be accepted here and there doesn\'t seem to be a\nconsensus on what is most common way to "at" another user in XMPP) in group chats,\nhere are the supported commands.\n\n- `@uptime`: how long the bot has been running\n- `@help`: the help text for what the bot does\n\nThere are also more general status commands which all bots respond to.\n\n- `@bots`: status check on who is a bot in the group chat\n\nThese commands will be detected in any part of the message sent to the bot. So\nyou can write `echobot, can we see your @uptime`, or `I\'d love to know which @bots are here.`\n\n### Avatars\n\nBy default, `xbotlib` will look for an `avatar.png` (so far tested with `.png`\nbut other file types may work) file alongside your Python script which contains\nyour bot implementation. You can also specify another path using the `--avatar`\noption on the command-line interface. The images should ideally have a height\nof `64` and a width of `64` pixels each.\n\n### Configuration\n\nAll the ways you can pass configuration details to your bot. There are three\nways to configure your bot, the configuration file, command-line interface and\nthe environment. Use whichever one suits you best. The values are loaded in the\nfollowing order: command-line > configuration file > environment. This means\nyou can override everything from the command-line easily.\n\n#### Using the `.conf` configuration file\n\nIf you run simply run your Python script which contains the bot then `xbotlib`\nwill generate a configuration for you by asking a few questions. This is the\nsimplest way to run your bot locally.\n\n- **account**: the account of the bot\n- **password**: the password of the bot account\n- **nick**: the nickname of the bot\n- **avatar**: the avatar of the bot (default: `avatar.png`)\n- **redis_url**: the Redis connection URL\n- **rooms**: a list of rooms to automatically join (comma separated)\n- **no_auto_join**: disable auto-join when invited (default: `False`)\n- **template**: the port to serve from (default: `index.html.j2`)\n- **serve**: turn on the web server (default: `False`)\n- **port**: the port to serve from (default: `8080`)\n- **storage**: storage back-end (default: `file`)\n- **output**: path to the output directory (default: `./`)\n\n#### Using the command-line interface\n\nEvery bot accepts a number of comand-line arguments to load configuration. You\ncan use the `--help` option to see what is available (e.g. `python bot.py --help`).\n\n- **-h, --help**: show this help message and exit\n- **-d, --debug**: enable verbose debug logs\n- **-a ACCOUNT, --account ACCOUNT**: account for the bot account\n- **-p PASSWORD, --password PASSWORD**: password for the bot account\n- **-n NICK, --nick NICK**: nickname for the bot account\n- **-av AVATAR, --avatar AVATAR**: avatar for the bot account (default: `avatar.png`)\n- **-ru REDIS_URL, --redis-url REDIS_URL**: redis storage connection URL\n- **-r ROOMS [ROOMS ...], --rooms ROOMS [ROOMS ...]**: Rooms to automatically join\n- **-naj, --no-auto-join**: disable automatically joining rooms when invited (default: `False`)\n- **-pt PORT, --port PORT**: the port to serve from (default: `8080`)\n- **-t TEMPLATE, --template TEMPLATE**: the template to render (default: `index.html.j2`)\n- **-s, --serve**: turn on the web server (default: `False`)\n- **-st {file,redis}, --storage {file,redis}**: choice of storage back-end (default: `file`)\n- **-o OUTPUT, --output OUTPUT**: path to the output directory (default: `./`)\n\n#### Using the environment\n\n`xbotlib` will try to read the following configuration values from the\nenvironment if it cannot read them from a configuration file or the\ncommand-line interface. This can be useful when doing remote server\ndeployments.\n\n- **XBOT_ACCOUNT**: The bot account\n- **XBOT_PASSWORD**: The bot password\n- **XBOT_NICK**: The bot nickname\n- **XBOT_AVATAR**: The bot avatar icon (default: `avatar.png`)\n- **XBOT_REDIS_URL**: Redis key store connection URL\n- **XBOT_ROOMS**: The rooms to automatically join\n- **XBOT_NO_AUTO_JOIN**: Disable auto-joining on invite (default: `False`)\n- **XBOT_TEMPLATE**: the template to render (default: `index.html.j2`)\n- **XBOT_SERVE**: Turn on the web server (default: `False`)\n- **XBOT_PORT**: The port to serve from (default: `8080`)\n- **XBOT_STORAGE**: choice of storage back-end (default: `file`)\n- **XBOT_OUTPUT**: path to the output directory (default: `./`)\n\n### Storage back-end\n\nIn order to store data you can make use of the `self.db` attribute of the `Bot`\nclass. It is a Python dictionary which will be saved to disk automatically for\nyou as a `<nick>.json` in your current working directory. The name and path to\nthis file can be configured using the output option (e.g. `python bot.py --output /var/www/html`)\n\n```python\ndef group(self, message):\n    if not message.room in self.db.keys():\n        self.db[message.room] = "visited"\n```\n\nIf you want to inspect the database when the bot is not running, you can look\nin the file directly.\n\n```bash\n$ cat <nick>.json\n```\n\nFor more advanced use cases, `xbotlib` also supports [Redis](https://redis.io/)\nas a storage back-end. You\'ll need to configure this (e.g. `--storage redis`)\nas the default uses the filesystem approach mentioned above. The same `self.db`\nwill then be passed as a Redis connection object. You will also need to install\nadditional dependencies using `pip install xbotlib[redis]`.\n\n### Loading Plugins\n\nYou can specify a `plugins = [...]` on your bot definition and they will be\nautomatically loaded when you start your bot.\n\n```python\nclass MyBot(Bot):\n    plugins = ["xep_0066"]\n```\n\nSee [here](https://slixmpp.readthedocs.io/xeps.html) for the list of supported plugins.\n\n### Serving HTTP\n\nFirstly, you\'ll need to install additional dependencies.\n\n```bash\n$ pip install xbotlib[web]\n```\n\nYour bot will run a web server if you configure it to do so. Use the `--serve`\noption on the command-line, the `serve = True` configuration option or the\n`XBOT_SERVE=True` environment variable.\n\nIf you\'re running your bot locally, just visit\n[0.0.0.0:8080](http://0.0.0.0:8080) to see. The default response is just some\nplaceholder text. You can write your own responses using the\n[Bot.serve](#bot-serve-request) function.\n\n`xbotlib` provides a small wrapper API for\n[Jinja2](https://jinja.palletsprojects.com/en/2.11.x/) which allows you to\neasily template and generate HTML. The web server is provided by\n[aiohttp](https://docs.aiohttp.org/).\n\nThe default template search path is `index.html.j2` in the current working\ndirectory. This can be configured through the usual configuration entrypoints.\n\nHere\'s a small example that renders a random ASCII letter and uses a stylesheet.\n\n> index.html.j2\n\n```jinja\n<html>\n<head>\n  <style> h1 { color: red; } </style>\n</head>\n<body>\n  <h1>{{ letter }}</h1>\n</body>\n</html>\n```\n\n> bot.py\n\n```python\nfrom string import ascii_letters\n\nasync def serve(self, request):\n    letter = choice(ascii_letters)\n    rendered = self.template.render(letter=letter)\n    return self.respond(rendered)\n```\n\nPlease note the use of the `return` keyword here. The `serve` function must\nreturn a response that will be passed to the web server. Also, the `async`\nkeyword. This function can handle asynchronous operations and must be marked as\nsuch. You can return any content type that you might find on the web (e.g.\nHTML, XML, JSON, audio and maybe even video) but you must specify the\n`content_type=...` keyword argument for `respond`.\n\nSee the [list of available content\ntypes](https://developer.mozilla.org/en-US/docs/Web/HTTP/Basics_of_HTTP/MIME_types#types)\nfor more.\n\nIf you want to pass data from your `direct`/`group` functions to the `serve`\nfunction, you\'ll need to make use of [some type of persistent\nstorage](#storage-back-end). Your `serve` function can read from the storage\nback-end and then respond. This is usually as simple as accessing the `self.db`\nattribute.\n\n### Invitations\n\nAs long as the `--no-auto-join` option is not set (via the configuration file\nor environment also), then your bot will automatically join any room it is\ninvited to. Rooms that your bot has been invited to will be stored in the\n`.xbotlib/data.json` file. If your bot is turned off or fails for some reason\nthen it will read this file when turned back on to see what rooms it should\nre-join automatically. The `data.json` file can be edited freely by hand.\n\n### API Reference\n\nWhen writing your own bot, you always sub-class the `Bot` class provided from\n`xbotlib`. Then if you want to respond to a direct message, you write a\n[direct](#botdirectmessage) function. If you want to respond to a group chat\nmessage, you write a [group](#botgroupmessage) function. That\'s it for the\nbasics.\n\n#### Bot.direct(message)\n\nRespond to direct messages.\n\nArguments:\n\n- **message**: received message (see [SimpleMessage](#simplemessage) below for available attributes)\n\n#### Bot.group(message)\n\nRespond to a message in a group chat.\n\nArguments:\n\n- **message**: received message (see [SimpleMessage](#simplemessage) below for available attributes)\n\n#### Bot.serve(request)\n\nServe requests via the built-in web server.\n\nSee [this section](#serving-http) for more.\n\nArguments:\n\n- **request**: the web request\n\n#### Bot.routes()\n\nRegister additional routes for web serving.\n\nSee [this section](#serving-http) for more.\n\nThis is an advanced feature. Use `self.web.add_routes`.\n\n```python\nfrom aiohttp.web import get\n\ndef routes(self):\n    self.web.add_routes([get("/foo", self.foo)])\n```\n\n#### Bot.setup()\n\nRun some setup logic before starting your bot.\n\n#### SimpleMessage\n\nA simple message interface.\n\nAttributes:\n\n- **text**: the entire text of the message\n- **content**: the text of the message after the nick\n- **sender**: the user the message came from\n- **room**: the room the message came from\n- **receiver**: the receiver of the message\n- **nick**: the nickname of the sender\n- **type**: the type of message\n- **url**: The URL of a sent file\n\n#### Bot\n\n> Bot.reply(message, to=None, room=None)\n\nSend a reply back.\n\nArguments:\n\n- **message**: the message that is sent\n- **to**: the user to send the reply to\n- **room**: the room to send the reply to\n\n> Bot.respond(response, content_type="text/html")\n\nReturn a response via the web server.\n\nArguments:\n\n- **response**: the text of the response\n- **content_type**: the type of response\n\nOther useful attributes on the `Bot` class are:\n\n- **self.db**: The [storage back-end](#storage-back-end)\n\n## Chatroom\n\nWe\'re lurking in\n[xbotlibtest@muc.vvvvvvaria.org](xmpp:xbotlibtest@muc.vvvvvvaria.org?join) if\nyou want to chat or just invite your bots for testing.\n\n## More Examples\n\nSee more examples on [git.vvvvvvaria.org](https://git.vvvvvvaria.org/explore/repos?q=xbotlib&topic=1).\n\n## Deploy your bots\n\nSee [bots.varia.zone](https://bots.varia.zone/).\n\n## Roadmap\n\nSee the [issue tracker](https://git.autonomic.zone/decentral1se/xbotlib/issues).\n\n## Changes\n\nSee the [CHANGELOG.md](./CHANGELOG.md).\n\n## License\n\nSee the [LICENSE](./LICENSE.md).\n\n## Contributing\n\nAny and all contributions most welcome! Happy to hear how you use the library\nor what could be improved from a usability perspective.\n\nTo test, install [Tox](https://tox.readthedocs.io/en/latest/) (`pip install tox`) and run `tox` to run the test suite locally.\n',
    'author': 'decentral1se',
    'author_email': 'lukewm@riseup.net',
    'maintainer': 'decentral1se',
    'maintainer_email': 'lukewm@riseup.net',
    'url': 'https://git.autonomic.zone/decentral1se/xbotlib',
    'py_modules': modules,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
