from __future__ import annotations
from functools import wraps as functools_wraps
from typing import Any, Callable, TYPE_CHECKING, Iterable, TypeVar, overload
from typing_extensions import _AnnotatedAlias
import inspect
from enum import Enum
import warnings
from collections.abc import MutableSequence

from magicgui.events import Signal
from magicgui.signature import MagicParameter
from magicgui.widgets import FunctionGui, FileEdit, EmptyWidget, Widget, Container
from magicgui.widgets._bases import ValueWidget
from macrokit import Macro, Expr, Head, Symbol, symbol

from .keybinding import as_shortcut
from .mgui_ext import AbstractAction, FunctionGuiPlus, PushButtonPlus
from .utils import get_parameters

from ..utils import get_signature, iter_members, extract_tooltip, screen_center
from ..widgets import Separator, MacroEdit
from ..fields import MagicField
from ..signature import MagicMethodSignature, get_additional_option
from ..wrappers import upgrade_signature

if TYPE_CHECKING:
    import napari
    from qtpy.QtWidgets import QDockWidget

class PopUpMode(Enum):
    popup = "popup"
    first = "first"
    last = "last"
    above = "above"
    below = "below"
    dock = "dock"
    parentlast = "parentlast"

class ErrorMode(Enum):
    msgbox = "msgbox"
    stderr = "stderr"
    
defaults = {"popup_mode": PopUpMode.popup,
            "error_mode": ErrorMode.msgbox,
            "close_on_run": True,
            }

_RESERVED = {"__magicclass_parent__", "__magicclass_children__", "_close_on_run", 
             "_error_mode", "_popup_mode", "_recorded_macro", "_my_symbol", "_macro_instance", 
             "annotation", "enabled", "gui_only", "height", "label_changed", "label", "layout",
             "labels", "margins", "max_height", "max_width", "min_height", "min_width", "name",
             "options", "param_kind", "parent_changed", "tooltip", "visible", "widget_type", 
             "width", "parent_viewer", "parent_dock_widget", "objectName", "create_macro", "wraps",
             "_unwrap_method", "_search_parent_magicclass", "_iter_child_magicclasses",
             }

UI = Symbol("ui")

def check_override(cls: type):
    subclass_members = set(cls.__dict__.keys())
    collision = subclass_members & _RESERVED
    if collision:
        raise AttributeError(f"Cannot override magic class reserved attributes: {collision}")

class MagicTemplate:    
    __magicclass_parent__: None | MagicTemplate
    __magicclass_children__: list[MagicTemplate]
    _close_on_run: bool
    _component_class: type[AbstractAction | Widget]
    _error_mode: ErrorMode
    _macro_instance: Macro
    _my_symbol: Symbol
    _popup_mode: PopUpMode
    annotation: Any
    changed: Signal
    enabled: bool
    gui_only: bool
    height: int
    label_changed: Signal
    label: str
    layout: str
    labels: bool
    margins: tuple[int, int, int, int]
    max_height: int
    max_width: int
    min_height: int
    min_width: int
    name: str
    options: dict
    param_kind: inspect._ParameterKind
    parent: Widget
    parent_changed: Signal
    tooltip: str
    visible: bool
    widget_type: str
    width: int
    
    __init_subclass__ = check_override
    
    def show(self) -> None:
        raise NotImplementedError()
    
    def hide(self) -> None:
        raise NotImplementedError()
    
    def close(self) -> None:
        raise NotImplementedError()
    
    @overload
    def __getitem__(self, key: int | str) -> Widget: ...

    @overload
    def __getitem__(self, key: slice) -> MutableSequence[Widget]: ...

    def __getitem__(self, key):
        raise NotImplementedError()
    
    def index(self, value: Any, start: int, stop: int) -> int:
        raise NotImplementedError()
    
    def remove(self, value: Widget | str):
        raise NotImplementedError()
    
    def append(self, widget: Widget) -> None:
        return self.insert(len(self, widget))
        
    def insert(self, key: int, widget: Widget) -> None:
        raise NotImplementedError()
    
    @property
    def _recorded_macro(self):
        if self.__magicclass_parent__ is None:
            return self._macro_instance
        else:
            return self.__magicclass_parent__._recorded_macro
    
    @property
    def parent_viewer(self) -> "napari.Viewer" | None:
        """
        Return napari.Viewer if magic class is a dock widget of a viewer.
        """
        parent_self = self._search_parent_magicclass()
        try:
            viewer = parent_self.parent.parent().qt_viewer.viewer
        except AttributeError:
            viewer = None
        return viewer
    
    @property
    def parent_dock_widget(self) -> "QDockWidget" | None:
        """
        Return dock widget object if magic class is a dock widget of a main window widget,
        such as a napari Viewer.
        """
        parent_self = self._search_parent_magicclass()
        try:
            dock = parent_self.native.parent()
        except AttributeError:
            dock = None
        return dock
    
    def objectName(self) -> str:
        """
        This function makes the object name discoverable by napari's 
        `viewer.window.add_dock_widget` function.
        """        
        return self.native.objectName()
    
    def create_macro(self, show: bool = False) -> str:
        """
        Create executable Python scripts from the recorded macro object.

        Parameters
        ----------
        show : bool, default is False
            Launch a TextEdit window that shows recorded macro.
        """
        
        out = str(self._recorded_macro)
                    
        if show:
            win = MacroEdit(name="macro")
            win.value = out
            
            win.native.setParent(self.native, win.native.windowFlags())
            win.native.move(screen_center() - win.native.rect().center())
            win.show()
            
        return out
    
    @classmethod
    def wraps(cls, 
              method: Callable | None = None,
              *, 
              template: Callable | None = None) -> Callable:
        """
        Wrap a parent method in a child magic-class. Wrapped method will appear in the
        child widget but behaves as if it is in the parent widget.
        
        Basically, this function is used as a wrapper like below.
        
        .. code-block:: python
        @magicclass
        class C:
            @magicclass
            class D: 
                def func(self, ...): ... # pre-definition
            @D.wraps
            def func(self, ...): ...

        Parameters
        ----------
        method : Callable, optional
            Method of parent class.
        template : Callable, optional
            Function template for signature.
            
        Returns
        -------
        Callable
            Same method as input, but has updated signature to hide the button.
        """      
            
        def wrapper(method: Callable):
            # Base function to get access to the original function
            if isinstance(method, FunctionGui):
                func = method._function
            else:
                func = method
            if template is not None:
                func = wraps(template)(func)
            if hasattr(cls, method.__name__):
                getattr(cls, method.__name__).__signature__ = get_signature(func)
            
            upgrade_signature(func, additional_options={"into": cls.__name__})
            return func
        
        return wrapper if method is None else wrapper(method)
    
    def _unwrap_method(self, child_clsname: str, name: str, widget: FunctionGui | PushButtonPlus):
        """
        This private method converts class methods that are wrapped by its child widget class
        into widget in child widget. Practically same widget is shared between parent and child,
        but only visible in the child side.

        Parameters
        ----------
        child_clsname : str
            Name of child widget class name.
        name : str
            Name of method.
        widget : FunctionGui
            Widget to be added.

        Raises
        ------
        RuntimeError
            If ``child_clsname`` was not found in child widget list. This error will NEVER be raised
            in the user's side.
        """        
        for child_instance in self._iter_child_magicclasses():
            if child_instance.__class__.__name__ == child_clsname:
                # get the position of predefined child widget
                try:
                    index = _get_index(child_instance, name)
                    new = False
                except ValueError:
                    index = len(child_instance)
                    new = True
                
                self.append(widget)
                
                if isinstance(widget, FunctionGui):
                    if new:
                        child_instance.append(widget)
                    else:
                        del child_instance[index]
                        child_instance.insert(index, widget)
                else:
                    widget.visible = False
                    if new:
                        widget = child_instance._create_widget_from_method(lambda x: None)
                        child_instance.append(widget)
                    else:
                        child_widget: PushButtonPlus | AbstractAction = child_instance[index]
                        child_widget.changed.disconnect()
                        child_widget.changed.connect(widget.changed)
                        child_widget.tooltip = widget.tooltip
                break
        else:
            raise RuntimeError(f"{child_clsname} not found in class {self.__class__.__name__}")
    
    def _convert_attributes_into_widgets(self):
        """
        This function is called in dynamically created __init__. Methods, fields and nested
        classes are converted to magicgui widgets.
        """        
        raise NotImplementedError()
    
    def _create_widget_from_field(self, name: str, fld: MagicField):
        """
        This function is called when magic-class encountered a MagicField in its definition.

        Parameters
        ----------
        name : str
            Name of variable
        fld : MagicField
            A field object that describes what type of widget it should be.
        """        
        raise NotImplementedError()
    
    def _create_widget_from_method(self, obj: Callable):
        text = obj.__name__.replace("_", " ")
        widget = self._component_class(name=obj.__name__, text=text, gui_only=True)

        # Wrap function to deal with errors in a right way.
        if self._error_mode == ErrorMode.msgbox:
            wrapper = _raise_error_in_msgbox
        elif self._error_mode == ErrorMode.stderr:
            wrapper = _identity_wrapper
        else:
            raise ValueError(self._error_mode)
        
        # Wrap function to block macro recording.
        _inner_func = wrapper(obj, parent=self)
        if _need_record(obj):
            @functools_wraps(obj)
            def func(*args, **kwargs):
                with self._recorded_macro.blocked():
                    out = _inner_func(*args, **kwargs)
                return out
        else:
            @functools_wraps(obj)
            def func(*args, **kwargs):
                return _inner_func(*args, **kwargs)
        
        # Signature must be updated like this. Otherwise, already wrapped member function
        # will have signature with "self".
        func.__signature__ = inspect.signature(obj)
        
        # Prepare a button or action
        widget.tooltip = extract_tooltip(func)
        
        # Get the number of parameters except for empty widgets.
        # With these lines, "bind" method of magicgui works inside magicclass.
        fgui = FunctionGuiPlus.from_callable(obj)
        n_empty = len([_widget for _widget in fgui if isinstance(_widget, EmptyWidget)])
        nparams = _n_parameters(func) - n_empty
        
        # This block enables instance methods in "bind" method of ValueWidget.
        all_params = []
        for param in func.__signature__.parameters.values():
            if isinstance(param.annotation, _AnnotatedAlias):
                param = MagicParameter.from_parameter(param)
            if isinstance(param, MagicParameter):
                bound_value = param.options.get("bind", None)
                
                # If bound method is a class method, use self.method(widget) as the value.
                # "_n_parameters(bound_value) == 2" seems a indirect way to determine that
                # "bound_value" is a class method but "_method_as_getter" raises ValueError
                # if "bound_value" is defined in a wrong namespace.
                if isinstance(bound_value, Callable) and _n_parameters(bound_value) == 2:
                    param.options["bind"] = _method_as_getter(self, bound_value)
                
                # If a MagicFiled is bound, bind the value of the connected widget.
                elif isinstance(bound_value, MagicField):
                    param.options["bind"] = _field_as_getter(self, bound_value)
                
            
            all_params.append(param)
        
        func.__signature__ = func.__signature__.replace(
            parameters=all_params
            )
        
        obj_sig = get_signature(obj)
        if isinstance(func.__signature__, MagicMethodSignature):
            # NOTE: I don't know the reason why "additional_options" is lost. 
            func.__signature__.additional_options = getattr(obj_sig, "additional_options", {})
            
        if nparams == 0:
            # We don't want a dialog with a single widget "Run" to show up.
            def run_function():
                # NOTE: callback must be defined inside function. Magic class must be
                # "compiled" otherwise function wrappings are not ready!
                mgui = _build_mgui(widget, func)
                mgui.native.setParent(self.native, mgui.native.windowFlags())
                if mgui.call_count == 0 and len(mgui.called._slots) == 0 and _need_record(func):
                    callback = _temporal_function_gui_callback(self, mgui, widget)
                    mgui.called.connect(callback)
                
                out = mgui()
                
                return out
            
        elif nparams == 1 and isinstance(fgui[0], FileEdit):
            # We don't want to open a magicgui dialog and again open a file dialog.
            def run_function():
                mgui = _build_mgui(widget, func)
                mgui.native.setParent(self.native, mgui.native.windowFlags())
                if mgui.call_count == 0 and len(mgui.called._slots) == 0 and _need_record(func):
                    callback = _temporal_function_gui_callback(self, mgui, widget)
                    mgui.called.connect(callback)
                
                fdialog: FileEdit = mgui[0]
                result = fdialog._show_file_dialog(
                    fdialog.mode,
                    caption=fdialog._btn_text,
                    start_path=str(fdialog.value),
                    filter=fdialog.filter,
                )

                if result:
                    fdialog.value = result
                    out = mgui(result)
                else:
                    out = None
                return out
            
        else:                
            def run_function():
                mgui = _build_mgui(widget, func)
                if mgui.call_count == 0 and len(mgui.called._slots) == 0:
                    mgui.native.setParent(self.native, mgui.native.windowFlags())
                    mgui.native.move(screen_center() - mgui.native.rect().center())
                    
                    # deal with popup mode.
                    if self._popup_mode not in (PopUpMode.popup, PopUpMode.dock):
                        mgui.label = ""
                        mgui.name = f"mgui-{id(mgui._function)}" # to avoid name collision
                        mgui.margins = (0, 0, 0, 0)
                        title = Separator(orientation="horizontal", text=text, button=True)
                        title.btn_text = "-"
                        title.btn_clicked.connect(mgui.hide)
                        mgui.insert(0, title)
                        mgui.append(Separator(orientation="horizontal"))
                        
                        if self._popup_mode == PopUpMode.parentlast:
                            parent_self = self._search_parent_magicclass()
                            parent_self.append(mgui)
                        elif self._popup_mode == PopUpMode.first:
                            self.insert(0, mgui)
                        elif self._popup_mode == PopUpMode.last:
                            self.append(mgui)
                        elif self._popup_mode == PopUpMode.above:
                            name = _get_widget_name(widget)
                            i = _get_index(self, name)
                            self.insert(i, mgui)
                        elif self._popup_mode == PopUpMode.below:
                            name = _get_widget_name(widget)
                            i = _get_index(self, name)
                            self.insert(i+1, mgui)
                            
                    elif self._popup_mode == PopUpMode.dock:
                        parent_self = self._search_parent_magicclass()
                        viewer = parent_self.parent_viewer
                        if viewer is None:
                            if not hasattr(parent_self.native, "addDockWidget"):
                                msg = "Cannot add dock widget to a normal container. Please use\n" \
                                      ">>> @magicclass(widget_type='mainwindow')\n" \
                                      "to create main window widget, or add the container as a dock "\
                                      "widget in napari."
                                warnings.warn(msg, UserWarning)
                            
                            else:    
                                from qtpy.QtWidgets import QDockWidget
                                from qtpy.QtCore import Qt
                                dock = QDockWidget(_get_widget_name(widget), self.native)
                                dock.setWidget(mgui.native)
                                parent_self.native.addDockWidget(
                                    Qt.DockWidgetArea.RightDockWidgetArea, dock
                                    )
                        else:
                            dock = viewer.window.add_dock_widget(
                                mgui, name=_get_widget_name(widget), area="right"
                                )
                    
                    if self._close_on_run:
                        if self._popup_mode != PopUpMode.dock:
                            mgui.called.connect(mgui.hide)
                        else:
                            # If FunctioGui is docked, we should close QDockWidget.
                            mgui.called.connect(lambda: mgui.parent.hide())
                    
                    if _need_record(func):
                        callback = _temporal_function_gui_callback(self, mgui, widget)
                        mgui.called.connect(callback)
                                    
                if self._popup_mode != PopUpMode.dock:
                    widget.mgui.show()
                else:
                    mgui.parent.show()
                                
                return None
            
        widget.changed.connect(run_function)
        
        # If design is given, load the options.
        widget.from_options(obj)
        
        # keybinding
        keybinding = get_additional_option(func, "keybinding", None)
        if keybinding is not None:
            if obj.__name__.startswith("_"):
                from qtpy.QtWidgets import QShortcut
                shortcut = QShortcut(as_shortcut(keybinding), self.native)
                shortcut.activated.connect(widget.changed)
            else:
                shortcut = as_shortcut(keybinding)
                widget.set_shortcut(shortcut)
            
        return widget
    
    def _search_parent_magicclass(self) -> MagicTemplate:
        current_self = self
        while getattr(current_self, "__magicclass_parent__", None) is not None:
            current_self = current_self.__magicclass_parent__
        return current_self
    
    def _iter_child_magicclasses(self) -> Iterable[MagicTemplate]:
        for child in self.__magicclass_children__:
            yield child
            yield from child.__magicclass_children__
    
class BaseGui(MagicTemplate):
    def __init__(self, close_on_run, popup_mode, error_mode):
        self._macro_instance = Macro(flags={"Get": False, "Return": False})
        self.__magicclass_parent__: None | BaseGui = None
        self.__magicclass_children__: list[MagicTemplate] = []
        self._close_on_run = close_on_run
        self._popup_mode = popup_mode
        self._error_mode = error_mode
        self._my_symbol = UI

def _get_widget_name(widget: Widget):
    # To escape reference
    return widget.name
    
def _temporal_function_gui_callback(bgui: MagicTemplate, fgui: FunctionGuiPlus, widget: PushButtonPlus):
    if isinstance(fgui, FunctionGui):
        _function = fgui._function
    else:
        raise TypeError("fgui must be FunctionGui object.")
        
    def _after_run():
        bound = fgui._previous_bound
        return_type = fgui.return_annotation
        result_required = return_type is not inspect._empty
        result = Symbol("result")
        # Standard button will be connected with two callbacks.
        # 1. Build FunctionGui
        # 2. Emit value changed signal.
        # But if there are more, they also have to be called.
        if len(widget.changed._slots) > 2:
            b = Expr(head=Head.getitem, args=[symbol(bgui), widget.name])
            ev = Expr(head=Head.getattr, args=[b, Symbol("changed")])
            line = Expr(head=Head.call, args=[ev])
            if result_required:
                line = Expr(head=Head.assign, args=[result, line])
            bgui._recorded_macro.append(line)
        else:
            kwargs = {k: v for k, v in bound.arguments.items()}
            line = Expr.parse_method(bgui, _function, (), kwargs)
            if result_required:
                line = Expr(Head.assign, [result, line])
            bgui._recorded_macro.append(line)
        
        # Deal with return annotation
        
        if result_required:
            from magicgui.type_map import _type2callback

            for callback in _type2callback(return_type):
                b = Expr(head=Head.getitem, args=[symbol(bgui), widget.name])
                _gui = Expr(head=Head.getattr, args=[b, Symbol("mgui")])
                line = Expr.parse_call(callback, (_gui, result, return_type), {})
                bgui._recorded_macro.append(line)
        
        return None
    
    return _after_run

def _build_mgui(widget_, func):
    if widget_.mgui is not None:
        return widget_.mgui
    try:
        mgui = FunctionGuiPlus(func)
    except Exception as e:
        msg = f"Exception was raised during building magicgui from method {func.__name__}.\n" \
            f"{e.__class__.__name__}: {e}"
        raise type(e)(msg)
    
    widget_.mgui = mgui
    mgui.native.setWindowTitle(widget_.name)
    return mgui
        
_C = TypeVar("_C", Callable, type)

def wraps(template: Callable | inspect.Signature) -> Callable[[_C], _C]:
    """
    Update signature using a template. If class is wrapped, then all the methods
    except for those start with "__" will be wrapped.

    Parameters
    ----------
    template : Callable or inspect.Signature object
        Template function or its signature.

    Returns
    -------
    Callable
        A wrapper which take a function or class as an input and returns same
        function or class with updated signature(s).
    """    
    def wrapper(f: _C) -> _C:
        if isinstance(f, type):
            for name, attr in iter_members(f):
                if callable(attr) or isinstance(attr, type):
                    wrapper(attr)
            return f
        
        Param = inspect.Parameter
        old_signature = inspect.signature(f)
            
        old_params = old_signature.parameters
        
        if callable(template):
            template_signature = inspect.signature(template)
        elif isinstance(template, inspect.Signature):
            template_signature = template
        else:
            raise TypeError("template must be a callable object or signature, "
                           f"but got {type(template)}.")
        
        # update empty signatures
        template_params = template_signature.parameters
        new_params: list[Param] = []
        
        for k, v in old_params.items():
            if v.annotation is inspect._empty and v.default is inspect._empty:
                new_params.append(
                    template_params.get(k, 
                                        Param(k, Param.POSITIONAL_OR_KEYWORD)
                                        )
                    )
            else:
                new_params.append(v)
        
        # update empty return annotation
        if old_signature.return_annotation is inspect._empty:
            return_annotation = template_signature.return_annotation
        else:
            return_annotation = old_signature.return_annotation
        
        f.__signature__ = inspect.Signature(
            parameters=new_params,
            return_annotation=return_annotation
            )
        return f
    return wrapper

def _raise_error_in_msgbox(_func: Callable, parent: Widget = None):
    """
    If exception happened inside function, then open a message box.
    """    
    def wrapped_func(*args, **kwargs):
        from qtpy.QtWidgets import QMessageBox
        try:
            out = _func(*args, **kwargs)
        except Exception as e:
            QMessageBox.critical(parent.native, e.__class__.__name__, str(e), QMessageBox.Ok)
            out = e
        return out
    
    return wrapped_func

def _identity_wrapper(_func: Callable, parent: Widget = None):
    """
    Do nothing.
    """    
    def wrapped_func(*args, **kwargs):
        return _func(*args, **kwargs)
    return wrapped_func

def _n_parameters(func: Callable):
    """
    Count the number of parameters of a callable object.
    """    
    return len(inspect.signature(func).parameters)

def _get_index(container: Container, widget_or_name: Widget | str):
    """
    Identical to container[widget_or_name], which sometimes doesn't work
    in magic-class.
    """    
    if isinstance(widget_or_name, str):
        name = widget_or_name
    else:
        name = widget_or_name.name
    for i, widget in enumerate(container):
        if widget.name == name:
            break
    else:
        raise ValueError(f"{widget_or_name} not found in {container}")
    return i

def _method_as_getter(self, bound_value: Callable):
    *clsnames, funcname = bound_value.__qualname__.split(".")
    
    def _func(w):
        ins = self
        while clsnames[0] != ins.__class__.__name__:
            ins = getattr(ins, "__magicclass_parent__", None)
            if ins is None:
                raise ValueError(f"Method {bound_value.__qualname__} is invisible"
                                 f"from magicclass {self.__class__.__qualname__}")
        
        for clsname in clsnames[1:]:
            ins = getattr(ins, clsname)
        return getattr(ins, funcname)(w)
    return _func

def _field_as_getter(self, bound_value: MagicField):
    def _func(w):
        namespace = bound_value.parent_class.__qualname__
        clsnames = namespace.split(".")
        ins = self
        while type(ins).__name__ not in clsnames:
            ins = getattr(ins, "__magicclass_parent__", None)
            if ins is None:
                raise ValueError(f"MagicField {namespace}.{bound_value.name} is invisible"
                                 f"from magicclass {self.__class__.__qualname__}")
        i = clsnames.index(type(ins).__name__)
        for clsname in clsnames[i:]:
            ins = getattr(ins, clsname, ins)
            
        _field_widget = bound_value.get_widget(ins)
        if not hasattr(_field_widget, "value"):
            raise TypeError(f"MagicField {bound_value.name} does not return ValueWidget "
                            "thus cannot be used as a bound value.")
        return bound_value.as_getter(ins)(w)
    return _func

def _need_record(func: Callable):
    return get_additional_option(func, "record", True)

def value_widget_callback(gui: MagicTemplate, widget: ValueWidget, name: str, getvalue: bool = True):
    sym_name = Symbol(name)
    sym_value = Symbol("value")
    def _set_value():
        if not widget.enabled:
            # If widget is read only, it means that value is set in script (not manually).
            # Thus this event should not be recorded as a macro.
            return None
        
        gui.changed.emit(gui)
        
        if getvalue:
            sub = Expr(head=Head.getattr, args=[sym_name, sym_value]) # name.value
        else:
            sub = Expr(head=Head.value, args=[sym_name])
        
        # Make an expression of
        # >>> x.name.value = value
        # or
        # >>> x.name = value
        expr = Expr(head=Head.assign, 
                    args=[Expr(head=Head.getattr, 
                               args=[symbol(gui), sub]), 
                          widget.value])
        
        last_expr = gui._recorded_macro[-1]
        if (len(gui._recorded_macro) > 1 and 
            last_expr.head == expr.head and 
            last_expr.args[0].args[1].head == expr.args[0].args[1].head and
            last_expr.args[0].args[1].args[0] == expr.args[0].args[1].args[0]):
            gui._recorded_macro[-1] = expr
        else:
            gui._recorded_macro.append(expr)
        return None
    return _set_value

def nested_function_gui_callback(gui: MagicTemplate, fgui: FunctionGui):
    fgui_name = Symbol(fgui.name)
    def _after_run():
        inputs = get_parameters(fgui)
        args = [Expr(head=Head.kw, args=[Symbol(k), v]) for k, v in inputs.items()]
        # args[0] is self
        sub = Expr(head=Head.getattr, args=[symbol(gui), fgui_name]) # {x}.func
        expr = Expr(head=Head.call, args=[sub] + args[1:]) # {x}.func(args...)

        if fgui._auto_call:
            # Auto-call will cause many redundant macros. To avoid this, only the last input
            # will be recorded in magic-class.
            last_expr = gui._recorded_macro[-1]
            if last_expr.head == Head.call and last_expr.args[0].head == Head.getattr and \
                last_expr.args[0].args[1] == expr.args[0].args[1]:
                gui._recorded_macro.pop()

        gui._recorded_macro.append(expr)
    return _after_run