import numpy as np
import pytest
import stk

from .case_data import CaseData

_complex = stk.ConstructedMolecule(
    topology_graph=stk.metal_complex.OctahedralDelta(
        metals=stk.BuildingBlock(
            smiles="[Fe+2]",
            functional_groups=(
                stk.SingleAtom(stk.Fe(0, charge=2)) for _ in range(6)
            ),
            position_matrix=np.array([[0, 0, 0]]),
        ),
        ligands=stk.BuildingBlock(
            smiles="C1=NC(C=NBr)=CC=C1",
            functional_groups=[
                stk.SmartsFunctionalGroupFactory(
                    smarts="[#6]~[#7X2]~[#35]",
                    bonders=(1,),
                    deleters=(),
                ),
                stk.SmartsFunctionalGroupFactory(
                    smarts="[#6]~[#7X2]~[#6]",
                    bonders=(1,),
                    deleters=(),
                ),
            ],
        ),
        optimizer=stk.MCHammer(),
    ),
)


@pytest.fixture(
    scope="session",
    params=(
        lambda name: CaseData(
            molecule=stk.ConstructedMolecule(
                topology_graph=stk.cage.FourPlusSix(
                    building_blocks=(
                        stk.BuildingBlock(
                            smiles="NCCN",
                            functional_groups=[stk.PrimaryAminoFactory()],
                        ),
                        stk.BuildingBlock(
                            smiles="O=CC(C=O)C=O",
                            functional_groups=[stk.AldehydeFactory()],
                        ),
                    ),
                    optimizer=stk.MCHammer(),
                ),
            ),
            metal_atom_distances={},
            metal_centroid_angles={},
            min_centoid_distance=3.2503016359941537,
            avg_centoid_distance=(5.375102295051954, 1.0292999378578707),
            radius_gyration=5.472767402726612,
            min_atom_atom_distance=1.0766369344893167,
            max_diameter=19.258604184686856,
            bonds={
                ("C", "N"): [
                    1.436896929428691,
                    1.403176580562134,
                    1.4368969294286889,
                    1.403176580562135,
                    1.436896929428689,
                    1.4031765805621377,
                    1.4368969294286882,
                    1.4031765805621368,
                    1.4368969294286884,
                    1.4031765805621361,
                    1.4368969294286882,
                    1.4031765805621381,
                    1.5902246281229657,
                    1.6194515573867505,
                    1.7061915836634536,
                    1.6127976358950904,
                    1.87542128153158,
                    1.6672029255189698,
                    1.819309222281951,
                    1.5475374503448591,
                    1.538740427772366,
                    1.6493618746994871,
                    1.8748048559156474,
                    1.62165636998199,
                ],
            },
            angles={
                ("C", "N", "C"): [
                    155.8291350925575,
                    164.1838583460421,
                    164.33352446928768,
                    175.2709718731384,
                    152.88587580126188,
                    163.52096227010736,
                    163.96647416455838,
                    170.0472949749414,
                    153.00343469023233,
                    159.12168687703831,
                    161.3331758772922,
                    154.28289869982567,
                ],
                ("C", "C", "N"): [
                    159.20540617729856,
                    171.63256187136975,
                    166.88244479617734,
                    161.52135862514666,
                    158.7288110225135,
                    157.34792765939977,
                    166.1344449918735,
                    167.32205677849387,
                    172.18088529713526,
                    172.44024765163527,
                    175.27146085376444,
                    168.34482782671077,
                    111.06913950132393,
                    114.09759412296782,
                    111.069139501324,
                    114.097594122968,
                    111.06913950132397,
                    114.0975941229678,
                    111.0691395013241,
                    114.09759412296775,
                    111.06913950132403,
                    114.09759412296776,
                    111.06913950132416,
                    114.09759412296782,
                ],
                ("H", "C", "N"): [
                    77.08637313883762,
                    65.94403831338589,
                    49.94357621729925,
                    73.13057709284602,
                    79.51585381779873,
                    49.01416813785448,
                    71.97882664879151,
                    70.2729680784005,
                    60.69031555903245,
                    64.85201503036426,
                    66.38804095524253,
                    52.17830343092516,
                    110.2958531097986,
                    107.05803628987213,
                    108.17401982811982,
                    111.57350243791369,
                    110.2958531097985,
                    107.05803628987222,
                    108.17401982811974,
                    111.57350243791355,
                    110.29585310979853,
                    107.05803628987218,
                    108.17401982811987,
                    111.57350243791355,
                    110.29585310979863,
                    107.05803628987215,
                    108.1740198281198,
                    111.57350243791365,
                    110.29585310979863,
                    107.05803628987215,
                    108.17401982811985,
                    111.57350243791356,
                    110.29585310979837,
                    107.05803628987218,
                    108.17401982811981,
                    111.57350243791369,
                ],
                ("H", "C", "H"): [
                    111.84941191236,
                    108.75245084958551,
                    111.84941191236004,
                    108.75245084958533,
                    111.84941191236013,
                    108.75245084958546,
                    111.84941191235995,
                    108.75245084958546,
                    111.84941191236008,
                    108.75245084958547,
                    111.84941191236018,
                    108.75245084958537,
                ],
            },
            torsions={
                ("C", "C", "C", "N"): [
                    -58.8264567977308,
                    123.46510333924043,
                    92.44090518825405,
                    -113.74319437749429,
                    -143.35746457534185,
                    177.84151693924497,
                    -155.4445108941948,
                    116.84405798772066,
                    159.4570093266152,
                    -120.36423972901386,
                    -76.34136043698051,
                    81.2234628427808,
                    -70.44705601119321,
                    58.545038008638954,
                    110.02901947104306,
                    -178.6632597080956,
                    -125.76935029255267,
                    166.22091772578239,
                    -124.97539100919866,
                    -114.89973276906152,
                    89.0214177186589,
                    7.891969514204105,
                    -146.77695204493673,
                    111.69258272777682,
                ],
                ("C", "N", "C", "C"): [
                    126.89570089838334,
                    113.4393942122361,
                    -31.092419337821692,
                    62.280235461277236,
                    -167.74724312219794,
                    150.36882686768246,
                    -157.97057329227346,
                    -24.247569116958886,
                    122.26875112043311,
                    102.85544992966203,
                    130.67670822902008,
                    114.8022156564823,
                    145.09998871879145,
                    79.84427671383055,
                    -28.465576254300263,
                    115.42152484687077,
                    -158.6927194486306,
                    162.06794008708675,
                    162.7225162766428,
                    83.84442183680272,
                    51.0654592694045,
                    -115.39515640605002,
                    175.16428807126385,
                    -149.5217138616757,
                ],
                ("C", "N", "C", "H"): [
                    4.384880589859175,
                    -117.52209264377026,
                    69.74193656927802,
                    -52.16503666435147,
                    27.858006559158348,
                    -94.04896667447125,
                    -90.37939555043477,
                    -153.4700766246677,
                    -2.8865784173753277,
                    85.99447247223941,
                    -33.5861864813812,
                    8.165887920495951,
                    -113.74108531313365,
                    -7.7086046520419504,
                    -129.6155778856714,
                    -169.84549921663208,
                    -89.00873432046444,
                    23.055513420233627,
                    29.0650344833041,
                    -90.51562447031613,
                    85.2723263158822,
                    -34.308332637738324,
                    39.55711977856277,
                    -82.34985345506679,
                    -109.06744430679083,
                    -163.09663702951957,
                    -10.035590377713207,
                    46.68756204115553,
                    -72.89309691246497,
                    59.12933383577668,
                    -60.45132511784375,
                    94.44333190283712,
                    -25.137327050783398,
                    -125.08780115089405,
                    -133.71605467214908,
                    -42.43664484908919,
                ],
                ("H", "C", "C", "H"): [
                    -177.44155760053528,
                    178.68085277466832,
                    60.61297031314438,
                    -177.44155760053545,
                    178.68085277466838,
                    60.61297031314449,
                    -177.44155760053528,
                    178.68085277466827,
                    60.61297031314459,
                    -177.4415576005353,
                    178.6808527746683,
                    60.61297031314446,
                    -179.3941880369845,
                    -63.081302906596584,
                    -58.03054623521211,
                    58.282338895175826,
                    -179.3941880369845,
                    -63.08130290659668,
                    -58.03054623521207,
                    58.28233889517573,
                    -179.3941880369845,
                    -63.081302906596555,
                    -58.03054623521203,
                    58.28233889517592,
                    -179.39418803698442,
                    -63.081302906596456,
                    -58.03054623521216,
                    58.28233889517581,
                    -179.39418803698447,
                    -63.08130290659649,
                    -58.03054623521209,
                    58.28233889517589,
                    -179.39418803698462,
                    -63.08130290659678,
                    -58.03054623521212,
                    58.28233889517571,
                ],
                ("H", "C", "C", "N"): [
                    -25.000478859710935,
                    59.852270214291025,
                    4.075629712851543,
                    42.015625278650184,
                    -36.76578388217302,
                    -2.5454156386680244,
                    -7.41236457692188,
                    48.23167100082843,
                    -60.84443561774971,
                    -28.419966329305968,
                    -6.29666399717691,
                    125.71079360454975,
                    58.05348764092604,
                    174.36637277131396,
                    62.54996497822907,
                    -176.08639321999854,
                    58.05348764092621,
                    174.366372771314,
                    62.549964978229006,
                    -176.0863932199986,
                    58.053487640926114,
                    174.36637277131408,
                    62.549964978228964,
                    -176.08639321999857,
                    58.053487640926015,
                    174.36637277131396,
                    62.54996497822917,
                    -176.0863932199986,
                    58.053487640926065,
                    174.36637277131402,
                    62.549964978229056,
                    -176.0863932199986,
                    58.05348764092616,
                    174.366372771314,
                    62.549964978228964,
                    -176.08639321999857,
                ],
                ("N", "C", "C", "N"): [
                    -60.002359343860384,
                    -60.002359343860334,
                    -60.0023593438604,
                    -60.00235934386041,
                    -60.00235934386043,
                    -60.002359343860284,
                ],
            },
            name=name,
        ),
        lambda name: CaseData(
            molecule=stk.ConstructedMolecule(
                topology_graph=stk.cage.M4L6TetrahedronSpacer(
                    building_blocks={
                        stk.BuildingBlock.init_from_molecule(
                            molecule=_complex,
                            functional_groups=[stk.BromoFactory()],
                        ): (0, 1, 2, 3),
                        stk.BuildingBlock(
                            smiles=("C1=CC(=CC=C1C2=CC=C(C=C2)Br)Br"),
                            functional_groups=[stk.BromoFactory()],
                        ): (4, 5, 6, 7, 8, 9),
                    },
                    optimizer=stk.MCHammer(),
                ),
            ),
            metal_atom_distances={
                (0, 40): 14.496255156029374,
                (0, 80): 14.307002353533244,
                (0, 120): 14.038436435377056,
                (40, 80): 14.884618345466887,
                (40, 120): 15.61630115867283,
                (80, 120): 14.55076063437277,
            },
            metal_centroid_angles={
                (0, 40): 107.6286801648586,
                (0, 80): 110.75873900673137,
                (0, 120): 107.13254641909572,
                (40, 80): 107.6718474455571,
                (40, 120): 115.18531816062348,
                (80, 120): 108.47202937098922,
            },
            min_centoid_distance=3.725088482208902,
            avg_centoid_distance=(8.90970162696734, 2.5231311926122526),
            radius_gyration=9.260074194989876,
            min_atom_atom_distance=1.0694746499945305,
            max_diameter=39.95166595631727,
            bonds={
                ("C", "N"): [
                    1.3483660995252742,
                    1.3429161748830707,
                    1.290134284490675,
                    1.3483660995252758,
                    1.3429161748830745,
                    1.2901342844906762,
                    1.3483660995252773,
                    1.3429161748830676,
                    1.2901342844906751,
                    1.3483660995252738,
                    1.342916174883073,
                    1.2901342844906716,
                    1.3483660995252738,
                    1.3429161748830727,
                    1.2901342844906751,
                    1.3483660995252804,
                    1.34291617488307,
                    1.2901342844906758,
                    1.3483660995252749,
                    1.3429161748830691,
                    1.290134284490677,
                    1.3483660995252718,
                    1.3429161748830705,
                    1.2901342844906738,
                    1.3483660995252797,
                    1.3429161748830734,
                    1.2901342844906782,
                    1.3483660995252735,
                    1.3429161748830678,
                    1.290134284490678,
                    1.3483660995252773,
                    1.3429161748830745,
                    1.290134284490679,
                    1.348366099525273,
                    1.3429161748830731,
                    1.2901342844906767,
                    1.9490771016251838,
                    2.0063329665379244,
                    2.382919669451856,
                    1.9781861848442748,
                    2.0190814930773047,
                    2.3422834415792324,
                    1.8357571189116264,
                    2.3560765865457074,
                    2.324042571234281,
                    2.0802576950462526,
                    2.7515814931061304,
                    1.83616693623852,
                ],
                ("C", "C"): [
                    1.4562504667425482,
                    1.367174505640155,
                    1.3814902308192545,
                    1.381674443852996,
                    1.389180447787276,
                    1.4562504667425424,
                    1.3671745056401547,
                    1.3814902308192596,
                    1.3816744438529973,
                    1.3891804477872771,
                    1.4562504667425458,
                    1.3671745056401567,
                    1.38149023081926,
                    1.3816744438529946,
                    1.3891804477872793,
                    1.4562504667425469,
                    1.367174505640152,
                    1.3814902308192583,
                    1.3816744438529955,
                    1.389180447787276,
                    1.4562504667425402,
                    1.367174505640147,
                    1.3814902308192591,
                    1.381674443852995,
                    1.3891804477872711,
                    1.4562504667425418,
                    1.367174505640152,
                    1.3814902308192596,
                    1.3816744438530042,
                    1.389180447787274,
                    1.4562504667425427,
                    1.3671745056401519,
                    1.3814902308192571,
                    1.3816744438529975,
                    1.3891804477872745,
                    1.456250466742543,
                    1.367174505640148,
                    1.3814902308192583,
                    1.381674443852997,
                    1.3891804477872771,
                    1.4562504667425384,
                    1.3671745056401545,
                    1.3814902308192534,
                    1.3816744438530009,
                    1.389180447787272,
                    1.4562504667425458,
                    1.3671745056401574,
                    1.3814902308192571,
                    1.3816744438529935,
                    1.3891804477872804,
                    1.4562504667425438,
                    1.367174505640154,
                    1.3814902308192594,
                    1.3816744438529995,
                    1.389180447787279,
                    1.456250466742541,
                    1.3671745056401565,
                    1.3814902308192574,
                    1.3816744438529953,
                    1.3891804477872767,
                    1.3819266304083873,
                    1.3797421675628239,
                    1.3795878525785152,
                    1.3905088360087436,
                    1.3794031252256436,
                    1.465923773445827,
                    1.3692671880205043,
                    1.393230464112883,
                    1.3716636820565455,
                    1.3729596207077222,
                    1.3658145691526875,
                    1.3786027653941018,
                    1.3688488252389202,
                    1.3819266304083848,
                    1.379742167562822,
                    1.3795878525785141,
                    1.3905088360087456,
                    1.3794031252256425,
                    1.4659237734458284,
                    1.369267188020506,
                    1.3932304641128843,
                    1.3716636820565447,
                    1.3729596207077255,
                    1.365814569152687,
                    1.3786027653941024,
                    1.368848825238921,
                    1.3819266304083806,
                    1.379742167562824,
                    1.3795878525785148,
                    1.3905088360087428,
                    1.3794031252256427,
                    1.4659237734458253,
                    1.3692671880205025,
                    1.3932304641128832,
                    1.3716636820565462,
                    1.372959620707722,
                    1.3658145691526884,
                    1.3786027653941029,
                    1.3688488252389213,
                    1.3819266304083848,
                    1.3797421675628236,
                    1.3795878525785152,
                    1.390508836008745,
                    1.3794031252256445,
                    1.465923773445829,
                    1.3692671880205074,
                    1.393230464112883,
                    1.3716636820565458,
                    1.3729596207077261,
                    1.3658145691526904,
                    1.378602765394104,
                    1.368848825238921,
                    1.3819266304083861,
                    1.3797421675628248,
                    1.3795878525785155,
                    1.3905088360087436,
                    1.3794031252256422,
                    1.4659237734458295,
                    1.369267188020508,
                    1.3932304641128828,
                    1.3716636820565467,
                    1.372959620707725,
                    1.36581456915269,
                    1.3786027653941044,
                    1.3688488252389215,
                    1.3819266304083857,
                    1.3797421675628223,
                    1.3795878525785172,
                    1.3905088360087445,
                    1.379403125225643,
                    1.4659237734458281,
                    1.3692671880205067,
                    1.3932304641128808,
                    1.3716636820565453,
                    1.3729596207077237,
                    1.3658145691526897,
                    1.3786027653941044,
                    1.3688488252389182,
                ],
                ("C", "H"): [
                    1.0875101642010656,
                    1.0694746499945293,
                    1.0841103288551472,
                    1.085896577811898,
                    1.0833025054451522,
                    1.087510164201065,
                    1.0694746499945318,
                    1.0841103288551444,
                    1.0858965778118925,
                    1.0833025054451544,
                    1.0875101642010636,
                    1.069474649994531,
                    1.0841103288551437,
                    1.0858965778118952,
                    1.0833025054451535,
                    1.0875101642010634,
                    1.069474649994531,
                    1.0841103288551437,
                    1.0858965778118996,
                    1.0833025054451562,
                    1.087510164201067,
                    1.0694746499945351,
                    1.084110328855145,
                    1.0858965778118936,
                    1.0833025054451526,
                    1.0875101642010556,
                    1.069474649994533,
                    1.0841103288551477,
                    1.0858965778118939,
                    1.0833025054451553,
                    1.087510164201062,
                    1.06947464999453,
                    1.0841103288551452,
                    1.0858965778118952,
                    1.0833025054451513,
                    1.0875101642010654,
                    1.0694746499945345,
                    1.0841103288551466,
                    1.085896577811896,
                    1.0833025054451513,
                    1.087510164201061,
                    1.0694746499945371,
                    1.084110328855148,
                    1.0858965778118934,
                    1.0833025054451533,
                    1.0875101642010625,
                    1.0694746499945278,
                    1.084110328855144,
                    1.0858965778118925,
                    1.0833025054451464,
                    1.0875101642010685,
                    1.069474649994531,
                    1.0841103288551432,
                    1.085896577811893,
                    1.0833025054451457,
                    1.0875101642010656,
                    1.0694746499945336,
                    1.0841103288551481,
                    1.085896577811896,
                    1.083302505445154,
                    1.0941430411436526,
                    1.0806549765097455,
                    1.0758670677919857,
                    1.075485685258647,
                    1.080230643633542,
                    1.0906603908409305,
                    1.0950351669394272,
                    1.0898109150529411,
                    1.0941430411436506,
                    1.0806549765097482,
                    1.0758670677919877,
                    1.0754856852586483,
                    1.0802306436335434,
                    1.0906603908409331,
                    1.0950351669394276,
                    1.0898109150529385,
                    1.0941430411436521,
                    1.0806549765097457,
                    1.0758670677919857,
                    1.075485685258648,
                    1.0802306436335438,
                    1.0906603908409307,
                    1.0950351669394252,
                    1.089810915052939,
                    1.0941430411436517,
                    1.0806549765097457,
                    1.075867067791985,
                    1.0754856852586492,
                    1.0802306436335443,
                    1.0906603908409305,
                    1.0950351669394272,
                    1.0898109150529383,
                    1.0941430411436543,
                    1.0806549765097457,
                    1.075867067791984,
                    1.0754856852586494,
                    1.0802306436335452,
                    1.090660390840934,
                    1.0950351669394272,
                    1.0898109150529405,
                    1.0941430411436508,
                    1.0806549765097446,
                    1.0758670677919862,
                    1.0754856852586478,
                    1.0802306436335443,
                    1.090660390840931,
                    1.095035166939426,
                    1.0898109150529407,
                ],
                ("Fe", "N"): [
                    1.8523799327301929,
                    1.8624126213648504,
                    1.8071722438223656,
                    1.914780325627807,
                    1.7870238453502443,
                    1.8586656881617896,
                    1.8523799327301933,
                    1.8624126213648502,
                    1.807172243822362,
                    1.9147803256278069,
                    1.787023845350246,
                    1.8586656881617867,
                    1.8523799327301875,
                    1.862412621364849,
                    1.8071722438223639,
                    1.9147803256278109,
                    1.787023845350242,
                    1.8586656881617867,
                    1.852379932730189,
                    1.862412621364852,
                    1.8071722438223679,
                    1.9147803256278073,
                    1.787023845350245,
                    1.8586656881617918,
                ],
            },
            angles={
                ("Fe", "N", "C"): [
                    141.79618994216818,
                    95.1737168042075,
                    97.53803302660268,
                    169.42294267924294,
                    142.88082060395675,
                    94.25835846636132,
                    99.0915055270005,
                    146.2182010139865,
                    143.85634214414847,
                    92.15186518320556,
                    95.9416418023339,
                    168.21336546739641,
                    141.7961899421683,
                    95.17371680420753,
                    97.53803302660275,
                    160.93847540082854,
                    142.88082060395678,
                    94.25835846636136,
                    99.0915055270005,
                    174.78613042666666,
                    143.8563421441485,
                    92.15186518320573,
                    95.94164180233372,
                    176.49187115944915,
                    141.79618994216807,
                    95.17371680420764,
                    97.53803302660269,
                    161.4562231989404,
                    142.88082060395647,
                    94.25835846636129,
                    99.09150552700042,
                    156.68893233234357,
                    143.85634214414873,
                    92.1518651832057,
                    95.94164180233366,
                    172.84786032386125,
                    141.79618994216804,
                    95.1737168042075,
                    97.53803302660273,
                    156.838394639216,
                    142.88082060395672,
                    94.25835846636124,
                    99.09150552700036,
                    174.37687135139905,
                    143.85634214414864,
                    92.15186518320536,
                    95.94164180233378,
                    174.10339472438375,
                ],
                ("C", "N", "C"): [
                    122.7833490326383,
                    71.90371353708271,
                    122.78334903263813,
                    65.1797503237024,
                    122.7833490326382,
                    72.79104373480327,
                    122.78334903263814,
                    92.03478850696686,
                    122.78334903263801,
                    84.09676052087161,
                    122.78334903263813,
                    83.55288589391328,
                    122.78334903263817,
                    99.40048635457273,
                    122.7833490326384,
                    63.819309437331015,
                    122.78334903263774,
                    84.4239242101861,
                    122.7833490326384,
                    64.2202901177826,
                    122.78334903263817,
                    76.64625092359503,
                    122.78334903263833,
                    89.30187244577043,
                ],
                ("C", "C", "C"): [
                    115.9151751471022,
                    120.54273327821548,
                    119.16849384113098,
                    121.54869946876121,
                    115.91517514710215,
                    120.54273327821564,
                    119.16849384113095,
                    121.54869946876106,
                    115.91517514710205,
                    120.54273327821528,
                    119.16849384113075,
                    121.54869946876131,
                    115.91517514710233,
                    120.54273327821538,
                    119.1684938411309,
                    121.54869946876113,
                    115.91517514710215,
                    120.54273327821556,
                    119.16849384113092,
                    121.54869946876106,
                    115.91517514710237,
                    120.54273327821566,
                    119.1684938411309,
                    121.54869946876073,
                    115.91517514710208,
                    120.54273327821534,
                    119.1684938411309,
                    121.54869946876126,
                    115.91517514710218,
                    120.5427332782156,
                    119.16849384113092,
                    121.54869946876103,
                    115.9151751471021,
                    120.54273327821544,
                    119.16849384113105,
                    121.54869946876126,
                    115.91517514710202,
                    120.54273327821528,
                    119.1684938411308,
                    121.54869946876141,
                    115.91517514710185,
                    120.54273327821556,
                    119.16849384113085,
                    121.54869946876134,
                    115.91517514710215,
                    120.54273327821528,
                    119.16849384113083,
                    121.54869946876136,
                    122.05420472135116,
                    117.3863074559602,
                    120.53508515403524,
                    120.85966491652931,
                    117.22015234871778,
                    120.76826337611575,
                    121.71140700647265,
                    122.07860683482706,
                    123.55578042015456,
                    116.97565895393097,
                    119.51031266538259,
                    119.77732237608899,
                    119.46856057530319,
                    121.79997503934317,
                    116.59199443458226,
                    122.8518347467823,
                    122.05420472135123,
                    117.38630745596025,
                    120.53508515403527,
                    120.85966491652928,
                    117.22015234871782,
                    120.76826337611575,
                    121.71140700647253,
                    122.07860683482699,
                    123.55578042015443,
                    116.97565895393105,
                    119.5103126653825,
                    119.77732237608892,
                    119.46856057530326,
                    121.79997503934327,
                    116.59199443458216,
                    122.85183474678237,
                    122.05420472135127,
                    117.38630745596012,
                    120.53508515403531,
                    120.85966491652935,
                    117.22015234871769,
                    120.76826337611573,
                    121.71140700647265,
                    122.07860683482708,
                    123.55578042015456,
                    116.97565895393097,
                    119.51031266538263,
                    119.77732237608893,
                    119.46856057530319,
                    121.79997503934321,
                    116.59199443458219,
                    122.85183474678234,
                    122.05420472135113,
                    117.3863074559602,
                    120.53508515403531,
                    120.85966491652933,
                    117.220152348718,
                    120.76826337611557,
                    121.71140700647265,
                    122.078606834827,
                    123.55578042015443,
                    116.97565895393085,
                    119.5103126653824,
                    119.77732237608873,
                    119.46856057530344,
                    121.79997503934334,
                    116.59199443458219,
                    122.85183474678234,
                    122.05420472135103,
                    117.38630745596022,
                    120.53508515403522,
                    120.85966491652925,
                    117.220152348718,
                    120.76826337611557,
                    121.71140700647277,
                    122.07860683482706,
                    123.55578042015443,
                    116.975658953931,
                    119.51031266538236,
                    119.77732237608893,
                    119.46856057530329,
                    121.7999750393435,
                    116.5919944345821,
                    122.85183474678232,
                    122.0542047213513,
                    117.38630745596028,
                    120.53508515403519,
                    120.85966491652916,
                    117.22015234871778,
                    120.7682633761157,
                    121.71140700647261,
                    122.078606834827,
                    123.55578042015448,
                    116.97565895393092,
                    119.51031266538243,
                    119.77732237608896,
                    119.46856057530331,
                    121.79997503934342,
                    116.59199443458216,
                    122.85183474678227,
                ],
                ("C", "C", "H"): [
                    122.29030861963713,
                    119.2142305328367,
                    119.71702271967725,
                    121.92609993138096,
                    119.74024400064332,
                    121.79451623033533,
                    115.8622810666697,
                    116.52520059783744,
                    122.29030861963719,
                    119.21423053283692,
                    119.71702271967737,
                    121.92609993138075,
                    119.74024400064309,
                    121.79451623033536,
                    115.86228106666985,
                    116.52520059783782,
                    122.29030861963699,
                    119.21423053283682,
                    119.71702271967753,
                    121.9260999313807,
                    119.74024400064324,
                    121.79451623033565,
                    115.86228106666985,
                    116.52520059783762,
                    122.29030861963709,
                    119.21423053283657,
                    119.7170227196775,
                    121.92609993138088,
                    119.74024400064319,
                    121.79451623033529,
                    115.86228106667005,
                    116.52520059783762,
                    122.29030861963729,
                    119.21423053283682,
                    119.7170227196775,
                    121.92609993138096,
                    119.74024400064303,
                    121.79451623033526,
                    115.86228106666992,
                    116.52520059783762,
                    122.29030861963757,
                    119.214230532837,
                    119.7170227196773,
                    121.9260999313811,
                    119.74024400064309,
                    121.79451623033478,
                    115.86228106667016,
                    116.5252005978378,
                    122.29030861963739,
                    119.21423053283672,
                    119.71702271967725,
                    121.92609993138078,
                    119.74024400064347,
                    121.79451623033523,
                    115.86228106666985,
                    116.52520059783762,
                    122.29030861963696,
                    119.21423053283664,
                    119.7170227196775,
                    121.92609993138106,
                    119.74024400064299,
                    121.79451623033557,
                    115.86228106667004,
                    116.52520059783754,
                    122.29030861963726,
                    119.21423053283692,
                    119.7170227196771,
                    121.92609993138088,
                    119.74024400064354,
                    121.79451623033532,
                    115.86228106666991,
                    116.5252005978375,
                    122.29030861963709,
                    119.21423053283667,
                    119.7170227196773,
                    121.92609993138042,
                    119.7402440006435,
                    121.79451623033557,
                    115.86228106666978,
                    116.5252005978378,
                    122.29030861963722,
                    119.2142305328369,
                    119.7170227196774,
                    121.926099931381,
                    119.74024400064312,
                    121.79451623033562,
                    115.8622810666698,
                    116.5252005978373,
                    122.29030861963699,
                    119.21423053283688,
                    119.71702271967746,
                    121.92609993138068,
                    119.74024400064334,
                    121.79451623033555,
                    115.8622810666698,
                    116.5252005978376,
                    120.95722386707854,
                    121.71322749266749,
                    116.98857034173362,
                    117.30922307387934,
                    121.01392851901166,
                    121.92251233457456,
                    117.42710721133403,
                    117.2746635167577,
                    119.46946067933925,
                    120.83101605157228,
                    117.29103112973753,
                    121.02021857704496,
                    120.90899081353274,
                    119.92662179158995,
                    119.39166050829613,
                    117.22154279374847,
                    120.95722386707845,
                    121.71322749266746,
                    116.98857034173362,
                    117.30922307387937,
                    121.01392851901196,
                    121.92251233457456,
                    117.42710721133408,
                    117.27466351675754,
                    119.46946067933936,
                    120.83101605157235,
                    117.29103112973743,
                    121.02021857704496,
                    120.90899081353272,
                    119.92662179158988,
                    119.39166050829613,
                    117.22154279374847,
                    120.95722386707858,
                    121.71322749266758,
                    116.98857034173345,
                    117.30922307387938,
                    121.01392851901186,
                    121.92251233457456,
                    117.42710721133385,
                    117.27466351675751,
                    119.4694606793393,
                    120.83101605157232,
                    117.29103112973749,
                    121.02021857704489,
                    120.90899081353274,
                    119.92662179158992,
                    119.3916605082961,
                    117.22154279374844,
                    120.95722386707851,
                    121.71322749266754,
                    116.9885703417337,
                    117.30922307387954,
                    121.01392851901183,
                    121.92251233457456,
                    117.42710721133398,
                    117.27466351675754,
                    119.46946067933936,
                    120.83101605157245,
                    117.29103112973733,
                    121.02021857704503,
                    120.90899081353277,
                    119.92662179159002,
                    119.39166050829617,
                    117.22154279374834,
                    120.95722386707851,
                    121.71322749266749,
                    116.98857034173378,
                    117.30922307387944,
                    121.01392851901176,
                    121.92251233457466,
                    117.42710721133408,
                    117.27466351675749,
                    119.46946067933936,
                    120.8310160515723,
                    117.2910311297372,
                    121.02021857704511,
                    120.90899081353277,
                    119.92662179158998,
                    119.39166050829616,
                    117.22154279374838,
                    120.95722386707841,
                    121.71322749266767,
                    116.98857034173358,
                    117.30922307387934,
                    121.01392851901178,
                    121.92251233457463,
                    117.427107211334,
                    117.27466351675761,
                    119.4694606793393,
                    120.83101605157245,
                    117.29103112973733,
                    121.02021857704509,
                    120.90899081353272,
                    119.92662179158998,
                    119.39166050829597,
                    117.22154279374848,
                ],
                ("N", "Fe", "N"): [
                    101.51425661600788,
                    87.82221111672037,
                    170.50382174861656,
                    94.4619785863201,
                    96.39539934765769,
                    92.74307534174659,
                    81.1719935314287,
                    163.7357871814907,
                    77.22453450249262,
                    101.20576884059219,
                    84.77043121316645,
                    169.6972031435041,
                    83.54820513479493,
                    75.19823419392335,
                    104.19566045126716,
                    101.5142566160078,
                    87.82221111672037,
                    170.50382174861662,
                    94.46197858631996,
                    96.39539934765752,
                    92.74307534174646,
                    81.17199353142881,
                    163.73578718149093,
                    77.22453450249276,
                    101.20576884059214,
                    84.77043121316648,
                    169.69720314350423,
                    83.54820513479504,
                    75.19823419392357,
                    104.19566045126716,
                    101.5142566160078,
                    87.82221111672052,
                    170.50382174861653,
                    94.46197858632023,
                    96.39539934765752,
                    92.74307534174653,
                    81.17199353142875,
                    163.7357871814907,
                    77.22453450249266,
                    101.20576884059216,
                    84.77043121316636,
                    169.6972031435041,
                    83.54820513479486,
                    75.19823419392338,
                    104.19566045126722,
                    101.51425661600786,
                    87.82221111672038,
                    170.5038217486164,
                    94.46197858632,
                    96.39539934765763,
                    92.74307534174659,
                    81.17199353142877,
                    163.73578718149085,
                    77.22453450249262,
                    101.20576884059233,
                    84.77043121316633,
                    169.69720314350423,
                    83.54820513479503,
                    75.19823419392326,
                    104.19566045126726,
                ],
                ("N", "C", "C"): [
                    121.3671448702113,
                    122.98860795401468,
                    117.84289820217766,
                    122.29822496997878,
                    141.51392401877325,
                    100.96234431763277,
                    121.36714487021158,
                    122.98860795401473,
                    117.84289820217766,
                    122.29822496997875,
                    137.8491025632268,
                    90.16671882598631,
                    121.36714487021145,
                    122.9886079540147,
                    117.84289820217789,
                    122.29822496997865,
                    139.6501036357593,
                    103.10978845466688,
                    121.36714487021132,
                    122.98860795401455,
                    117.84289820217789,
                    122.2982249699789,
                    112.91301748641997,
                    130.02858221585262,
                    121.3671448702116,
                    122.9886079540146,
                    117.84289820217779,
                    122.29822496997885,
                    123.74266550812223,
                    118.77498679673526,
                    121.36714487021135,
                    122.98860795401447,
                    117.84289820217795,
                    122.29822496997897,
                    128.91564934210024,
                    113.78995404457038,
                    121.36714487021145,
                    122.98860795401453,
                    117.84289820217792,
                    122.29822496997885,
                    123.5233221381543,
                    117.27599110588693,
                    121.36714487021128,
                    122.98860795401468,
                    117.84289820217774,
                    122.29822496997897,
                    142.1071367696597,
                    99.65224603515638,
                    121.36714487021166,
                    122.9886079540143,
                    117.84289820217799,
                    122.29822496997907,
                    134.79387929427062,
                    108.35773958553878,
                    121.36714487021132,
                    122.98860795401478,
                    117.84289820217774,
                    122.29822496997865,
                    154.07679096195457,
                    88.86554099817049,
                    121.3671448702116,
                    122.9886079540148,
                    117.84289820217766,
                    122.29822496997872,
                    130.42990895601,
                    112.87017950123438,
                    121.36714487021138,
                    122.98860795401482,
                    117.8428982021777,
                    122.2982249699788,
                    128.1858057137998,
                    114.8873280232886,
                ],
                ("N", "C", "H"): [
                    122.770574063088,
                    118.48754448034599,
                    122.77057406308758,
                    118.48754448034576,
                    122.77057406308774,
                    118.48754448034596,
                    122.77057406308765,
                    118.48754448034596,
                    122.77057406308748,
                    118.48754448034576,
                    122.77057406308752,
                    118.48754448034545,
                    122.77057406308774,
                    118.48754448034586,
                    122.77057406308774,
                    118.48754448034583,
                    122.77057406308745,
                    118.48754448034542,
                    122.77057406308793,
                    118.48754448034612,
                    122.77057406308762,
                    118.4875444803458,
                    122.77057406308785,
                    118.48754448034576,
                ],
            },
            torsions={
                ("Fe", "N", "C", "C"): [
                    172.6366551245519,
                    4.56553183103943,
                    -175.43506909653223,
                    -4.613323766410115,
                    1.4924995892505266,
                    -171.39211748374527,
                    175.82210962386205,
                    2.5275204142630305,
                    -177.47308051330853,
                    -2.7066057482020245,
                    -87.98766890816695,
                    138.7912086784121,
                    163.2391751481051,
                    9.800634302836524,
                    -170.19996662473528,
                    -10.24784726257969,
                    22.851767067941637,
                    -155.3524656767434,
                    172.6366551245518,
                    4.565531831039441,
                    -175.4350690965323,
                    -4.613323766410007,
                    169.14334323465712,
                    -19.08702334032336,
                    175.8221096238621,
                    2.5275204142631518,
                    -177.4730805133086,
                    -2.7066057482019574,
                    131.31634582828843,
                    -42.6298866661259,
                    163.23917514810555,
                    9.800634302836182,
                    -170.19996662473528,
                    -10.247847262579663,
                    86.06954407549311,
                    -90.67074080635906,
                    172.63665512455145,
                    4.565531831039549,
                    -175.43506909653226,
                    -4.61332376641033,
                    -121.82867602372438,
                    77.13973417955268,
                    175.82210962386202,
                    2.5275204142628143,
                    -177.47308051330833,
                    -2.706605748202168,
                    59.95572022876722,
                    -133.1473388056142,
                    163.23917514810495,
                    9.800634302836373,
                    -170.19996662473517,
                    -10.247847262579905,
                    -47.86430662991563,
                    138.38076700734365,
                    172.6366551245517,
                    4.565531831039541,
                    -175.43506909653223,
                    -4.613323766410215,
                    18.189226140642877,
                    -172.26316510867846,
                    175.82210962386205,
                    2.5275204142630074,
                    -177.4730805133085,
                    -2.7066057482019152,
                    93.41799384175282,
                    -90.55261105163507,
                    163.2391751481056,
                    9.800634302835856,
                    -170.19996662473534,
                    -10.247847262580102,
                    -107.92901280623516,
                    79.00736314654941,
                ],
                ("Fe", "N", "C", "H"): [
                    -7.363278165174034,
                    175.3881818413472,
                    -4.177823665864181,
                    177.29489985955573,
                    -16.760758141621125,
                    169.75365834517763,
                    -7.363278165173931,
                    175.38818184134732,
                    -4.177823665864209,
                    177.29489985955547,
                    -16.76075814162008,
                    169.7536583451779,
                    -7.363278165174512,
                    175.38818184134755,
                    -4.177823665863971,
                    177.29489985955556,
                    -16.76075814162131,
                    169.7536583451777,
                    -7.363278165174524,
                    175.38818184134712,
                    -4.177823665864272,
                    177.29489985955558,
                    -16.760758141620347,
                    169.753658345178,
                ],
                ("C", "N", "Fe", "N"): [
                    -179.8147664965291,
                    -87.4575891906162,
                    74.60969110615142,
                    -2.8776658824336723,
                    101.9709725564199,
                    6.200133169774901,
                    2.844819292933247,
                    -2.8613200788318855,
                    173.6172461553436,
                    -158.18847810321554,
                    -99.8800699272201,
                    -81.80327870996014,
                    84.15552995695622,
                    -82.26122136728128,
                    -99.03621777317356,
                    -6.025853867739905,
                    86.33132343817302,
                    -111.60139626505936,
                    170.91124674635554,
                    -84.24011481479089,
                    -82.13432473062842,
                    176.94514498474564,
                    -162.8350970464979,
                    100.2354929708031,
                    98.568247240023,
                    -84.95318652580154,
                    94.5977505739999,
                    152.90615874999534,
                    87.42222266518154,
                    -106.61896866790214,
                    177.33724452106327,
                    160.562248115171,
                    -179.89292160008253,
                    -97.54529626115463,
                    111.59887210251685,
                    -85.48963860747007,
                    3.5317471265842943,
                    61.840155302579724,
                    5.593592557804412,
                    171.55240122472077,
                    163.98539253347155,
                    147.2103961275793,
                    -3.4143553659070482,
                    78.93326997302084,
                    -71.92256166330769,
                    -79.81195764924307,
                    173.62025234217563,
                    173.58983110790396,
                    107.52493566119502,
                    -86.51625567188864,
                    93.24055444730867,
                    76.4655580414164,
                    -179.5611069240373,
                    -166.19041092333953,
                    -21.503549473247634,
                    14.002432414036184,
                    -2.772563991856092,
                    -13.60229825712094,
                    96.88017909396144,
                    -128.07133948182894,
                    -179.8147664965291,
                    -87.45758919061628,
                    74.6096911061512,
                    -2.87766588243371,
                    101.97097255641984,
                    6.200133169774818,
                    -113.31410635979769,
                    -2.8613200788318838,
                    173.6172461553435,
                    -158.1884781032154,
                    74.40505262990159,
                    -81.80327870996084,
                    84.15552995695629,
                    -82.26122136728132,
                    -163.79101080048918,
                    -6.0258538677398645,
                    86.33132343817294,
                    -111.60139626505959,
                    170.91124674635552,
                    -84.24011481479096,
                    -82.13432473062848,
                    176.94514498474558,
                    -162.8350970464975,
                    100.23549297080288,
                    98.56824724002291,
                    -84.9531865258017,
                    94.59775057399978,
                    -32.80871869288323,
                    87.42222266518041,
                    -106.6189686679025,
                    177.33724452106327,
                    95.80745508785544,
                    -179.8929216000825,
                    -97.54529626115448,
                    111.59887210251591,
                    158.35143573979903,
                    3.5317471265842904,
                    -123.87472214029872,
                    5.59359255780374,
                    171.55240122472085,
                    163.98539253347246,
                    82.45560310026461,
                    -3.414355365907117,
                    78.93326997302091,
                    -71.9225616633087,
                    -79.81195764924314,
                    173.6202523421756,
                    57.43090545517308,
                    107.52493566119429,
                    -86.5162556718886,
                    93.24055444730863,
                    11.710765014100794,
                    -179.56110692403777,
                    77.65066342392998,
                    152.78157308387384,
                    14.002432414035956,
                    -67.52735701917187,
                    -13.602298257120653,
                    -19.278746558769633,
                    46.2137830752926,
                    -179.8147664965287,
                    -87.45758919061578,
                    74.60969110615225,
                    -2.8776658824333246,
                    101.97097255642032,
                    6.20013316977514,
                    162.05950512421194,
                    -2.8613200788319837,
                    173.61724615534374,
                    -158.188478103216,
                    161.0773791487536,
                    -81.80327870996,
                    84.15552995695604,
                    -82.2612213672813,
                    10.304096766324149,
                    -6.0258538677401265,
                    86.33132343817279,
                    -111.60139626505918,
                    170.91124674635523,
                    -84.24011481479111,
                    -82.13432473062835,
                    176.94514498474572,
                    -162.8350970464974,
                    100.23549297080318,
                    98.56824724002281,
                    -84.95318652580148,
                    94.59775057399986,
                    53.86360782596943,
                    87.42222266518144,
                    -106.61896866790254,
                    177.33724452106333,
                    -90.09743734533124,
                    -179.89292160008267,
                    -97.54529626115485,
                    111.59887210251674,
                    73.72504722380845,
                    3.5317471265842943,
                    -37.20239562144614,
                    5.593592557804718,
                    171.55240122472077,
                    163.98539253347155,
                    -103.44928933292302,
                    -3.414355365906937,
                    78.93326997302087,
                    -71.92256166330756,
                    -79.81195764924297,
                    173.6202523421756,
                    -27.195483060817494,
                    107.5249356611953,
                    -86.51625567188867,
                    93.24055444730865,
                    -174.1941274190859,
                    -179.561106924037,
                    -6.975725092060595,
                    -120.54610039727339,
                    14.00243241403627,
                    106.56775054764172,
                    -13.602298257120967,
                    -103.90513507476003,
                    132.88610959414518,
                    -179.814766496529,
                    -87.45758919061603,
                    74.60969110615166,
                    -2.8776658824336088,
                    101.97097255642007,
                    6.200133169775034,
                    42.48809095143339,
                    -2.8613200788317457,
                    173.61724615534362,
                    -158.18847810321574,
                    161.43860107080556,
                    -81.8032787099606,
                    84.1555299569565,
                    -82.26122136728118,
                    70.41799425353894,
                    -6.025853867740054,
                    86.3313234381729,
                    -111.60139626505942,
                    170.91124674635532,
                    -84.24011481479103,
                    -82.13432473062831,
                    176.94514498474558,
                    -162.83509704649734,
                    100.23549297080316,
                    98.5682472400231,
                    -84.95318652580154,
                    94.59775057399978,
                    54.224829748021094,
                    87.4222226651807,
                    -106.61896866790222,
                    177.33724452106344,
                    -29.983539858116483,
                    -179.89292160008236,
                    -97.54529626115439,
                    111.59887210251681,
                    -45.84636694896996,
                    3.531747126584187,
                    -36.84117369939451,
                    5.5935925578039924,
                    171.55240122472108,
                    163.98539253347184,
                    -43.335391845708045,
                    -3.4143553659069728,
                    78.93326997302098,
                    -71.92256166330783,
                    -79.81195764924307,
                    173.62025234217555,
                    -146.76689723359607,
                    107.52493566119473,
                    -86.51625567188819,
                    93.24055444730871,
                    -114.08022993187117,
                    -179.56110692403763,
                    -126.54713926483902,
                    -120.18487847522175,
                    14.002432414036189,
                    166.6816480348563,
                    -13.60229825712056,
                    136.5234507524615,
                    133.24733151619682,
                ],
                ("C", "N", "C", "C"): [
                    -179.99940642650972,
                    -7.354081368180032e-06,
                    0.0001920563538641725,
                    174.73908890276746,
                    -2.007088835389905,
                    -174.89170590838572,
                    -179.99940642650986,
                    -7.354081423880997e-06,
                    0.0001920563538629845,
                    -151.2890612809162,
                    -20.212456897302733,
                    -153.43357931072367,
                    -179.99940642650978,
                    -7.354081581700461e-06,
                    0.00019205635393141423,
                    166.2137528842581,
                    5.363107314793896,
                    -172.84112542989115,
                    -179.99940642650967,
                    -7.354081413677349e-06,
                    0.00019205635377570377,
                    158.8647081105756,
                    48.82761909044641,
                    -139.40274748453407,
                    -179.9994064265098,
                    -7.354081519036904e-06,
                    0.00019205635400926954,
                    173.13178341716178,
                    3.3664214429849353,
                    -170.5798110514294,
                    -179.99940642650958,
                    -7.354081021209289e-06,
                    0.00019205635345864413,
                    166.26022529872603,
                    4.162850387892602,
                    -172.5774344939596,
                    -179.9994064265101,
                    -7.354081870925882e-06,
                    0.00019205635419013945,
                    -177.0375493743867,
                    33.90603342245545,
                    -127.12555637426753,
                    -179.99940642650992,
                    -7.354081078070793e-06,
                    0.00019205635357034248,
                    160.5700870933938,
                    14.064839895845914,
                    -179.0382191385355,
                    -179.99940642651003,
                    -7.354081564293872e-06,
                    0.00019205635404051464,
                    176.93123607708304,
                    45.422312434934675,
                    -128.33261392780605,
                    -179.99940642650984,
                    -7.35408159303968e-06,
                    0.0001920563540274526,
                    -169.6312429020799,
                    58.84997298328846,
                    -131.60241826603286,
                    -179.99940642650995,
                    -7.354081442447992e-06,
                    0.00019205635398596404,
                    173.5530891743189,
                    52.316375530470175,
                    -131.65422936291773,
                    -179.9994064265099,
                    -7.3540810792311656e-06,
                    0.0001920563535785805,
                    172.45495666482913,
                    44.90691651507726,
                    -128.1567075321382,
                ],
                ("C", "C", "C", "C"): [
                    0.00032945507370343353,
                    -0.000165097600564429,
                    179.99941616640407,
                    0.0003294550736774856,
                    -0.00016509760059959242,
                    179.99941616640425,
                    0.0003294550735304601,
                    -0.00016509760054421235,
                    179.9994161664041,
                    0.000329455073487265,
                    -0.00016509760047972804,
                    179.99941616640405,
                    0.00032945507392613326,
                    -0.0001650976008215436,
                    179.99941616640422,
                    0.000329455073418029,
                    -0.00016509760031111357,
                    179.99941616640402,
                    0.0003294550733007597,
                    -0.00016509760033634938,
                    179.99941616640416,
                    0.00032945507356937876,
                    -0.00016509760042667846,
                    179.99941616640433,
                    0.0003294550735520848,
                    -0.00016509760044187548,
                    179.99941616640425,
                    0.00032945507349976843,
                    -0.0001650976004235236,
                    179.99941616640405,
                    0.00032945507362559394,
                    -0.00016509760044675095,
                    179.99941616640422,
                    0.00032945507418126593,
                    -0.00016509760105783044,
                    179.99941616640464,
                    0.0002122906635241481,
                    -0.0006923531062212631,
                    31.867515274283885,
                    -148.12987728089553,
                    0.003901782307120953,
                    -179.98770272850803,
                    0.003004627681365604,
                    -0.0037683439517079385,
                    -0.0028250890561241516,
                    179.99077346224945,
                    -148.1236869100805,
                    31.878920534740107,
                    -179.99579919920492,
                    179.99919436108686,
                    -0.0021124935899882466,
                    -0.004596892413272122,
                    0.0016901626516019387,
                    -0.0005654239678955942,
                    0.001531740082624222,
                    0.003947365452804336,
                    0.00021229066342454158,
                    -0.0006923531062234576,
                    31.86751527428429,
                    -148.12987728089556,
                    0.003901782307046536,
                    -179.98770272850828,
                    0.003004627681403572,
                    -0.0037683439515838386,
                    -0.002825089056111575,
                    179.99077346224962,
                    -148.12368691008027,
                    31.87892053473987,
                    -179.99579919920518,
                    179.99919436108704,
                    -0.002112493590050064,
                    -0.004596892413073086,
                    0.0016901626513307442,
                    -0.0005654239679133688,
                    0.0015317400828305727,
                    0.003947365452754055,
                    0.000212290663464385,
                    -0.0006923531061619563,
                    31.867515274284024,
                    -148.1298772808956,
                    0.0039017823069633652,
                    -179.98770272850814,
                    0.0030046276813384815,
                    -0.003768343951553959,
                    -0.0028250890561012884,
                    179.99077346224948,
                    -148.1236869100803,
                    31.878920534740054,
                    -179.995799199205,
                    179.9991943610869,
                    -0.002112493590113317,
                    -0.0045968924130931135,
                    0.0016901626513906676,
                    -0.0005654239678146441,
                    0.0015317400828025107,
                    0.003947365452688508,
                    0.00021229066346217088,
                    -0.0006923531059002958,
                    31.86751527428392,
                    -148.12987728089595,
                    0.0039017823066562715,
                    -179.98770272850817,
                    0.003004627681296918,
                    -0.003768343951368077,
                    -0.0028250890561881655,
                    179.99077346224945,
                    -148.1236869100801,
                    31.878920534740043,
                    -179.99579919920504,
                    179.99919436108678,
                    -0.0021124935900700935,
                    -0.004596892412890878,
                    0.0016901626510837397,
                    -0.000565423967974629,
                    0.0015317400829885045,
                    0.003947365452718449,
                    0.00021229066340240613,
                    -0.0006923531061136334,
                    31.867515274284088,
                    -148.12987728089544,
                    0.0039017823069151924,
                    -179.98770272850828,
                    0.0030046276813916373,
                    -0.0037683439514884388,
                    -0.0028250890561481593,
                    179.99077346224962,
                    -148.12368691008035,
                    31.878920534740143,
                    -179.99579919920492,
                    179.99919436108698,
                    -0.0021124935902257823,
                    -0.004596892413234391,
                    0.0016901626515788636,
                    -0.0005654239676486421,
                    0.0015317400827287713,
                    0.003947365452650643,
                    0.00021229066343865218,
                    -0.000692353106178704,
                    31.867515274284006,
                    -148.12987728089567,
                    0.003901782306923877,
                    -179.9877027285082,
                    0.003004627681310684,
                    -0.0037683439514898903,
                    -0.0028250890560458336,
                    179.99077346224945,
                    -148.12368691008035,
                    31.878920534739997,
                    -179.99579919920507,
                    179.9991943610869,
                    -0.0021124935900243055,
                    -0.004596892413044084,
                    0.00169016265135606,
                    -0.0005654239678980569,
                    0.0015317400827728003,
                    0.003947365452698106,
                ],
                ("C", "C", "C", "H"): [
                    179.99982180856676,
                    -179.999629933107,
                    -0.00014254498270375754,
                    0.000553506181497267,
                    -179.99906170853131,
                    179.99959512453114,
                    179.9993935116423,
                    179.99982180856665,
                    -179.99962993310692,
                    -0.00014254498273032435,
                    0.0005535061812328967,
                    -179.99906170853134,
                    179.99959512453145,
                    179.99939351164244,
                    179.99982180856645,
                    -179.9996299331068,
                    -0.000142544982831274,
                    0.0005535061815873507,
                    -179.99906170853131,
                    179.9995951245315,
                    179.99939351164247,
                    179.99982180856654,
                    -179.99962993310692,
                    -0.00014254498271172806,
                    0.0005535061814804978,
                    -179.99906170853131,
                    179.99959512453117,
                    179.99939351164235,
                    179.9998218085665,
                    -179.99962993310677,
                    -0.00014254498261077861,
                    0.0005535061815543073,
                    -179.9990617085313,
                    179.9995951245313,
                    179.99939351164207,
                    179.99982180856648,
                    -179.99962993310675,
                    -0.0001425449825979931,
                    0.0005535061812707156,
                    -179.99906170853166,
                    179.99959512453123,
                    179.99939351164235,
                    179.99982180856657,
                    -179.99962993310697,
                    -0.0001425449825629596,
                    0.0005535061811708186,
                    -179.99906170853157,
                    179.9995951245313,
                    179.99939351164247,
                    179.99982180856642,
                    -179.99962993310663,
                    -0.00014254498207946263,
                    0.000553506181124972,
                    -179.99906170853126,
                    179.99959512453142,
                    179.99939351164207,
                    179.99982180856662,
                    -179.99962993310686,
                    -0.0001425449826589285,
                    0.0005535061813292237,
                    -179.9990617085315,
                    179.99959512453142,
                    179.99939351164252,
                    179.9998218085663,
                    -179.9996299331066,
                    -0.00014254498250185815,
                    0.0005535061815333002,
                    -179.99906170853131,
                    179.99959512453145,
                    179.9993935116422,
                    179.99982180856665,
                    -179.9996299331068,
                    -0.00014254498246466654,
                    0.0005535061813368522,
                    -179.99906170853137,
                    179.99959512453145,
                    179.99939351164232,
                    179.99982180856668,
                    -179.9996299331069,
                    -0.00014254498258205374,
                    0.0005535061810338136,
                    -179.99906170853149,
                    179.99959512453103,
                    179.99939351164224,
                    179.9881542153336,
                    -179.98455201986937,
                    -179.99649832511,
                    -179.99125810422188,
                    -179.98798936838796,
                    179.99693406296777,
                    179.9839717905231,
                    179.9841483703456,
                    0.036558761209881184,
                    0.011133398244286625,
                    0.005329552152639229,
                    -0.02037996931074214,
                    179.97806883904155,
                    179.98598028479785,
                    -179.986370800191,
                    -179.98638431697918,
                    179.9650148100737,
                    179.98363676211912,
                    -179.98003750541395,
                    -179.96611029950256,
                    179.98815421533342,
                    -179.98455201986923,
                    -179.99649832510983,
                    -179.99125810422169,
                    -179.98798936838764,
                    179.99693406296765,
                    179.98397179052282,
                    179.98414837034545,
                    0.03655876120968245,
                    0.011133398244569739,
                    0.005329552152353919,
                    -0.020379969310734016,
                    179.97806883904124,
                    179.9859802847979,
                    -179.98637080019114,
                    -179.9863843169791,
                    179.9650148100736,
                    179.9836367621192,
                    -179.98003750541375,
                    -179.9661102995023,
                    179.98815421533365,
                    -179.98455201986923,
                    -179.99649832510988,
                    -179.99125810422186,
                    -179.98798936838773,
                    179.99693406296768,
                    179.983971790523,
                    179.98414837034542,
                    0.03655876120977218,
                    0.011133398244449921,
                    0.0053295521525873555,
                    -0.020379969310720687,
                    179.97806883904144,
                    179.9859802847979,
                    -179.98637080019105,
                    -179.98638431697918,
                    179.96501481007365,
                    179.98363676211918,
                    -179.9800375054139,
                    -179.96611029950245,
                    179.9881542153337,
                    -179.98455201986908,
                    -179.9964983251101,
                    -179.9912581042217,
                    -179.98798936838793,
                    179.99693406296777,
                    179.98397179052338,
                    179.98414837034514,
                    0.03655876120953189,
                    0.011133398244574613,
                    0.005329552152952591,
                    -0.02037996931094462,
                    179.9780688390413,
                    179.98598028479802,
                    -179.98637080019114,
                    -179.98638431697904,
                    179.96501481007388,
                    179.98363676211915,
                    -179.98003750541386,
                    -179.96611029950245,
                    179.98815421533357,
                    -179.98455201986923,
                    -179.99649832510997,
                    -179.99125810422174,
                    -179.9879893683878,
                    179.99693406296777,
                    179.98397179052301,
                    179.98414837034542,
                    0.03655876120966472,
                    0.011133398244293616,
                    0.0053295521525588046,
                    -0.02037996931070319,
                    179.97806883904144,
                    179.98598028479773,
                    -179.9863708001911,
                    -179.98638431697918,
                    179.96501481007368,
                    179.98363676211926,
                    -179.98003750541386,
                    -179.96611029950267,
                    179.98815421533368,
                    -179.98455201986923,
                    -179.99649832510997,
                    -179.99125810422183,
                    -179.9879893683877,
                    179.99693406296777,
                    179.983971790523,
                    179.98414837034545,
                    0.03655876120977276,
                    0.011133398244415907,
                    0.005329552152661196,
                    -0.02037996931068517,
                    179.9780688390414,
                    179.9859802847979,
                    -179.9863708001911,
                    -179.9863843169791,
                    179.96501481007368,
                    179.98363676211926,
                    -179.98003750541386,
                    -179.9661102995024,
                ],
                ("N", "C", "C", "C"): [
                    -0.00034253920884789644,
                    -6.5878538666573876e-06,
                    -179.99793032897978,
                    -172.13983767549473,
                    175.030020408156,
                    -0.0003425392087989911,
                    -6.587853790117146e-06,
                    -179.9979303289796,
                    -124.96858845131044,
                    146.64059605921423,
                    -0.0003425392087189634,
                    -6.587853691065135e-06,
                    -179.9979303289797,
                    -178.03294954042045,
                    178.69563385889057,
                    -0.00034253920863004536,
                    -6.587853813754573e-06,
                    -179.99793032897978,
                    172.95822046524844,
                    -171.51648597235703,
                    -0.000342539209014621,
                    -6.58785365729739e-06,
                    -179.9979303289796,
                    -174.03305792356701,
                    174.34346908747315,
                    -0.0003425392084837422,
                    -6.587854126246885e-06,
                    -179.99793032897978,
                    -176.64548395019764,
                    177.15121391575033,
                    -0.0003425392087078485,
                    -6.587853633659992e-06,
                    -179.99793032897955,
                    -161.1512001021295,
                    162.36397154635418,
                    -0.00034253920864115974,
                    -6.587854071514994e-06,
                    -179.99793032897975,
                    165.44415673735432,
                    -170.98864607173144,
                    -0.0003425392088395587,
                    -6.587853809815036e-06,
                    -179.99793032897963,
                    -173.37046611594545,
                    175.0515309741255,
                    -0.00034253920879565676,
                    -6.587853794619538e-06,
                    -179.99793032897972,
                    168.29635382232945,
                    -174.907681203505,
                    -0.0003425392088856854,
                    -6.587853907178635e-06,
                    -179.99793032897978,
                    175.9079793473282,
                    -176.6168308200302,
                    -0.0003425392089495981,
                    -6.587853760288971e-06,
                    -179.99793032897944,
                    -172.96346412728425,
                    173.91032433701184,
                ],
                ("N", "C", "C", "H"): [
                    179.9990453179417,
                    179.999944979995,
                    -179.99956529924063,
                    7.871960665453799,
                    -4.957536239394753,
                    179.99904531794178,
                    179.99994497999486,
                    -179.99956529924077,
                    55.04320988963837,
                    -33.34696058833633,
                    179.99904531794164,
                    179.99994497999495,
                    -179.99956529924063,
                    1.978848800528496,
                    -1.2919227886600424,
                    179.99904531794172,
                    179.99994497999492,
                    -179.99956529924057,
                    -7.021251616197605,
                    8.493182345211022,
                    179.99904531794175,
                    179.999944979995,
                    -179.9995652992405,
                    5.978740417381598,
                    -5.644087560077219,
                    179.99904531794212,
                    179.999944979995,
                    -179.99956529924074,
                    3.366314390751196,
                    -2.836342731800301,
                    179.9990453179417,
                    179.99994497999455,
                    -179.99956529924037,
                    18.86932781642469,
                    -17.62636013607766,
                    179.99904531794172,
                    179.99994497999515,
                    -179.99956529924052,
                    -14.544044921696822,
                    9.023797280718009,
                    179.99904531794175,
                    179.99994497999495,
                    -179.9995652992407,
                    6.650061802608668,
                    -4.93880070830623,
                    179.99904531794158,
                    179.99994497999495,
                    -179.99956529924037,
                    -11.683118259116616,
                    5.101987114063136,
                    179.99904531794166,
                    179.99994497999506,
                    -179.9995652992406,
                    -4.071492734118035,
                    3.392837497537974,
                    179.99904531794226,
                    179.99994497999506,
                    -179.99956529924097,
                    7.0570637912697896,
                    -6.080007345419953,
                ],
                ("N", "C", "C", "N"): [
                    0.001461144833742751,
                    0.0014611448340052296,
                    0.0014611448336942043,
                    0.0014611448336611917,
                    0.0014611448338459137,
                    0.0014611448339770768,
                    0.0014611448338165424,
                    0.0014611448342868997,
                    0.001461144833993899,
                    0.0014611448337233312,
                    0.0014611448339384742,
                    0.0014611448345713052,
                ],
                ("C", "N", "C", "H"): [
                    -179.99974123337208,
                    -5.259405489475201,
                    -179.9997412333724,
                    28.712444326841517,
                    -179.9997412333723,
                    -13.784741507984592,
                    -179.99974123337196,
                    -21.133786281667096,
                    -179.9997412333723,
                    -6.866710975080816,
                    -179.9997412333722,
                    -13.738269093516411,
                    -179.99974123337176,
                    2.963956233371154,
                    -179.99974123337242,
                    -19.42840729884847,
                    -179.9997412333722,
                    -3.0672583151593384,
                    -179.9997412333722,
                    10.370262705677469,
                    -179.99974123337233,
                    -6.445405217923631,
                    -179.9997412333724,
                    -7.543537727412778,
                ],
                ("H", "C", "C", "H"): [
                    -0.0010170183183288175,
                    -7.132386413978327e-05,
                    0.0004306449617503705,
                    -0.0010170183179668132,
                    -7.132386386757232e-05,
                    0.00043064496158682844,
                    -0.0010170183181123249,
                    -7.132386379603952e-05,
                    0.0004306449616081602,
                    -0.0010170183184973992,
                    -7.132386407379267e-05,
                    0.000430644961747105,
                    -0.001017018317943746,
                    -7.132386384958528e-05,
                    0.0004306449613201784,
                    -0.0010170183181589208,
                    -7.132386406279447e-05,
                    0.0004306449613930596,
                    -0.0010170183182915606,
                    -7.132386417827807e-05,
                    0.0004306449616971395,
                    -0.001017018318234769,
                    -7.132386412443129e-05,
                    0.0004306449615974942,
                    -0.001017018317991663,
                    -7.132386389231892e-05,
                    0.0004306449615548292,
                    -0.0010170183181886382,
                    -7.13238639674748e-05,
                    0.0004306449615006853,
                    -0.0010170183180094065,
                    -7.132386407482427e-05,
                    0.0004306449616401606,
                    -0.0010170183177626278,
                    -7.132386359169466e-05,
                    0.0004306449610037545,
                    -0.008758190635161547,
                    -0.004284644820170987,
                    -0.054803857294777006,
                    -0.02578606066978444,
                    -0.008758190635430704,
                    -0.00428464482010486,
                    -0.054803857295121036,
                    -0.025786060669811362,
                    -0.008758190635353302,
                    -0.004284644820325922,
                    -0.054803857294829124,
                    -0.025786060669833542,
                    -0.008758190635372193,
                    -0.004284644820380474,
                    -0.05480385729476865,
                    -0.025786060669949176,
                    -0.008758190635456511,
                    -0.004284644820169106,
                    -0.05480385729464078,
                    -0.025786060669797772,
                    -0.00875819063549297,
                    -0.004284644820321906,
                    -0.05480385729490334,
                    -0.025786060669835607,
                ],
            },
            name=name,
        ),
        lambda name: CaseData(
            molecule=stk.ConstructedMolecule(
                topology_graph=stk.cage.M6L12Cube(
                    building_blocks=(
                        stk.BuildingBlock(
                            smiles="[Pd+2]",
                            functional_groups=(
                                stk.SingleAtom(stk.Pd(0, charge=2))
                                for _ in range(4)
                            ),
                            position_matrix=np.array([[0, 0, 0]]),
                        ),
                        stk.BuildingBlock(
                            smiles=(
                                "C1=NC=CC(C2=CC=CC(C3=C" "C=NC=C3)=C2)=C1"
                            ),
                            functional_groups=[
                                stk.SmartsFunctionalGroupFactory(
                                    smarts="[#6]~[#7X2]~[#6]",
                                    bonders=(1,),
                                    deleters=(),
                                ),
                            ],
                        ),
                    ),
                    optimizer=stk.Collapser(),
                ),
            ),
            metal_atom_distances={
                (0, 1): 10.896944441369676,
                (0, 2): 15.412420828743615,
                (0, 3): 10.898227238903015,
                (0, 4): 10.89758581086942,
                (0, 5): 10.89758581086942,
                (1, 2): 10.898227013424108,
                (1, 3): 15.412420828751983,
                (1, 4): 10.89758569813956,
                (1, 5): 10.89758569813956,
                (2, 3): 10.899510587395719,
                (2, 4): 10.898868771096192,
                (2, 5): 10.898868771096192,
                (3, 4): 10.898868883845243,
                (3, 5): 10.898868883845243,
                (4, 5): 15.412420799323275,
            },
            metal_centroid_angles={
                (0, 1): 89.97683228781653,
                (0, 2): 179.97682458236295,
                (0, 3): 89.99999891399814,
                (0, 4): 89.98841325868929,
                (0, 5): 89.98841325868929,
                (1, 2): 89.99999229454916,
                (1, 3): 179.97683120183706,
                (1, 4): 89.988409950303,
                (1, 5): 89.988409950303,
                (2, 3): 90.02317650363617,
                (2, 4): 90.01158205422921,
                (2, 5): 90.01158205422921,
                (3, 4): 90.0115853652926,
                (3, 5): 90.0115853652926,
                (4, 5): 179.96722969198316,
            },
            min_centoid_distance=6.383408826308891,
            avg_centoid_distance=(8.60220669009043, 1.4865651154918351),
            radius_gyration=8.729709948310644,
            min_atom_atom_distance=0.44643102631473996,
            max_diameter=33.73047877339638,
            bonds={
                ("C", "N"): [
                    1.353058965423603,
                    1.3451283016975855,
                    1.3465440099589545,
                    1.3435751961267794,
                    1.353058965423606,
                    1.3451283016975866,
                    1.3465440099589543,
                    1.3435751961267801,
                    1.3530589654236038,
                    1.3451283016975855,
                    1.3465440099589518,
                    1.3435751961267812,
                    1.3530589654236038,
                    1.3451283016975877,
                    1.3465440099589547,
                    1.3435751961267768,
                    1.3530589654236043,
                    1.3451283016975895,
                    1.3465440099589574,
                    1.3435751961267766,
                    1.3530589654236056,
                    1.34512830169759,
                    1.3465440099589556,
                    1.3435751961267788,
                    1.3530589654236065,
                    1.3451283016975861,
                    1.3465440099589556,
                    1.343575196126776,
                    1.3530589654236065,
                    1.3451283016975881,
                    1.3465440099589545,
                    1.3435751961267786,
                    1.3530589654236032,
                    1.3451283016975886,
                    1.3465440099589578,
                    1.3435751961267792,
                    1.3530589654236083,
                    1.3451283016975855,
                    1.3465440099589543,
                    1.3435751961267772,
                    1.353058965423604,
                    1.3451283016975852,
                    1.3465440099589534,
                    1.3435751961267814,
                    1.3530589654236045,
                    1.3451283016975875,
                    1.3465440099589556,
                    1.3435751961267772,
                ],
                ("N", "Pd"): [
                    1.1462870375279308,
                    1.1475305900353587,
                    1.1469088095181883,
                    1.1469088095181912,
                    1.161990472392297,
                    1.147530425643655,
                    1.146908753463117,
                    1.1469087534631166,
                    1.1626430874302047,
                    1.1481571311671641,
                    1.147535254936645,
                    1.1475352549366422,
                    1.162643252571796,
                    1.163868059705465,
                    1.1475353110970166,
                    1.147535311097016,
                    1.1623167164933919,
                    1.1635414322797313,
                    1.1623166098662991,
                    1.1635415389134378,
                    1.1623167164933907,
                    1.1635414322797317,
                    1.1623166098662991,
                    1.1635415389134378,
                ],
            },
            angles={
                ("Pd", "N", "C"): [
                    145.3364161204098,
                    92.71566433965049,
                    145.35355533585238,
                    92.69066373740844,
                    145.34750870396223,
                    92.70174983337294,
                    145.3424706652124,
                    92.70457102697569,
                    92.9848316785696,
                    143.6882660203896,
                    145.35355657996374,
                    92.69066192199989,
                    145.34247379868617,
                    92.70456674767497,
                    145.347511188711,
                    92.70174591755409,
                    93.02895476171841,
                    143.65593509762243,
                    145.32224414757718,
                    92.73632319173113,
                    145.31113736931889,
                    92.75024650225262,
                    145.3161777521393,
                    92.74742165640887,
                    93.02895634265737,
                    143.6559339386069,
                    93.00289339845105,
                    143.67503539976832,
                    145.31617461662046,
                    92.74742593332886,
                    145.31113488253465,
                    92.75025041580395,
                    93.0054944468834,
                    143.67448567558878,
                    92.98224296239863,
                    143.68880992718314,
                    93.00830159119344,
                    143.66971673834388,
                    92.97944812961104,
                    143.69356020470838,
                    93.00829939731649,
                    143.66971817172484,
                    92.97945032030039,
                    143.6935587749178,
                    93.00549700075612,
                    143.67448363012494,
                    92.98224041109339,
                    143.68881196998996,
                ],
                ("C", "N", "C"): [
                    118.19129090481385,
                    120.25508586967823,
                    118.1912909048137,
                    120.25508586967857,
                    118.19129090481388,
                    120.25508586967835,
                    118.19129090481366,
                    120.25508586967821,
                    118.19129090481367,
                    120.25508586967818,
                    118.1912909048137,
                    120.25508586967823,
                    118.19129090481375,
                    120.25508586967818,
                    118.19129090481353,
                    120.25508586967831,
                    118.19129090481373,
                    120.25508586967825,
                    118.19129090481367,
                    120.25508586967828,
                    118.19129090481375,
                    120.25508586967838,
                    118.1912909048137,
                    120.25508586967831,
                ],
                ("N", "Pd", "N"): [
                    146.05077203329736,
                    85.11751169319207,
                    85.11751169319201,
                    147.47168093004834,
                    85.51711582194356,
                    85.51711582195776,
                    85.10356916634119,
                    85.10356916634086,
                    148.9898489867429,
                    85.54406044093516,
                    85.54406044093523,
                    146.05076337936742,
                    148.94063169866487,
                    85.9029392004031,
                    85.88930722388662,
                    148.94063169866487,
                    85.90293920041705,
                    85.88930722388649,
                    85.10356771364772,
                    85.10356771363362,
                    147.57180639857145,
                    85.54405917253885,
                    85.54405917253868,
                    85.13206384277788,
                    85.13206384277795,
                    85.52969470280868,
                    85.52969470280857,
                    146.15257104730188,
                    85.88930937763236,
                    85.87571674231248,
                    85.88930937761859,
                    85.87571674231253,
                    146.0507542460134,
                    148.94063170205763,
                    148.9406317020578,
                    146.15258017587263,
                ],
                ("N", "C", "C"): [
                    121.5534906677226,
                    122.20332532342933,
                    122.45456305486576,
                    118.34402198600888,
                    121.5534906677226,
                    122.20332532342938,
                    122.45456305486576,
                    118.3440219860084,
                    121.55349066772247,
                    122.20332532342918,
                    122.45456305486589,
                    118.34402198600843,
                    121.5534906677227,
                    122.20332532342941,
                    122.45456305486576,
                    118.34402198600894,
                    121.55349066772266,
                    122.20332532342931,
                    122.45456305486566,
                    118.34402198600894,
                    121.5534906677226,
                    122.20332532342935,
                    122.45456305486579,
                    118.34402198600873,
                    121.55349066772243,
                    122.20332532342935,
                    122.45456305486576,
                    118.34402198600888,
                    121.55349066772266,
                    122.20332532342948,
                    122.45456305486586,
                    118.34402198600868,
                    121.55349066772268,
                    122.2033253234291,
                    122.45456305486573,
                    118.34402198600858,
                    121.55349066772256,
                    122.20332532342921,
                    122.45456305486582,
                    118.34402198600868,
                    121.5534906677226,
                    122.20332532342935,
                    122.45456305486583,
                    118.3440219860085,
                    121.5534906677226,
                    122.20332532342935,
                    122.45456305486593,
                    118.34402198600871,
                ],
            },
            torsions={
                ("Pd", "N", "C", "C"): [
                    -150.4154059000122,
                    163.67441178347977,
                    -150.37677213869708,
                    163.66209621087182,
                    -150.39954395378984,
                    163.6711220405506,
                    -150.39263169379097,
                    163.66538225321577,
                    -164.7009925945828,
                    153.57871485868606,
                    -150.37676933211472,
                    163.66209531660206,
                    -150.39262580717937,
                    163.6653806044848,
                    -150.39953717631272,
                    163.67111965231177,
                    -164.72286976688872,
                    153.6413964043478,
                    -150.4473052217879,
                    163.68458899306444,
                    -150.46315025411545,
                    163.68787886215839,
                    -150.4700746999408,
                    163.69362602724752,
                    -164.72287055078775,
                    153.64139864852854,
                    -164.70994774314025,
                    153.60438501371547,
                    -150.47008057315244,
                    163.69362767513792,
                    -150.46315701775086,
                    163.687881249329,
                    -164.7147691600345,
                    153.61357653620703,
                    -164.6961928591633,
                    153.5695614849389,
                    -164.7090979131514,
                    153.60657336895784,
                    -164.70183924722951,
                    153.57653305805795,
                    -164.70909637208274,
                    153.60656954703143,
                    -164.70184078741656,
                    153.57653687980556,
                    -164.71476997303765,
                    153.61357945891274,
                    -164.69619204829434,
                    153.56955856400685,
                ],
                ("C", "N", "Pd", "N"): [
                    -52.023697298649715,
                    21.699200044816045,
                    -125.74659464211479,
                    -52.0554770743546,
                    153.80020282338373,
                    -125.77518598187828,
                    80.06133619985025,
                    21.6960113712456,
                    -132.46150978829857,
                    153.79962200110748,
                    -132.47748065542677,
                    80.07672465764242,
                    -153.65877361989558,
                    132.73002997657724,
                    -80.04757721637219,
                    -52.05547938326251,
                    153.80020286536376,
                    23.064139247026052,
                    -131.09337699259535,
                    -127.14332427503561,
                    78.6932035945271,
                    48.97976795495558,
                    -24.6314284485716,
                    122.59096435847897,
                    -125.78245977679221,
                    21.671505628082368,
                    21.674699665408177,
                    -132.48877815286332,
                    -125.75387427604096,
                    80.08860456441487,
                    80.07322012094615,
                    -132.47281447417933,
                    -153.6576259872287,
                    -80.00076565110801,
                    132.68551367665071,
                    -153.65830590157975,
                    48.95908451875478,
                    -127.12884532319063,
                    78.64859347792681,
                    23.165685039686913,
                    -131.05090747384213,
                    48.92926738852439,
                    122.5861277246451,
                    -24.727592947596207,
                    -52.02627750781451,
                    -52.05289710281819,
                    153.78958173763766,
                    153.810244673914,
                    -153.6694005311077,
                    -78.63442949554072,
                    131.3465958201631,
                    -153.64767204740934,
                    48.998495227550045,
                    131.3180286802176,
                    -26.06629937243698,
                    -78.63145315193209,
                    124.00880254606055,
                    48.94033189315578,
                    123.97530292872278,
                    -26.04367175557344,
                    -153.64700322651748,
                    131.31802573791893,
                    -78.66299957778838,
                    -153.6700808869437,
                    48.970171651193404,
                    -78.63443231434874,
                    123.9752977125674,
                    131.31370009533705,
                    -26.040130233793214,
                    48.96867188070658,
                    -26.066299154857003,
                    123.95267552943568,
                    21.67149897017749,
                    -125.78245773670662,
                    -125.75387755423995,
                    80.08860620613868,
                    21.674692526230338,
                    -132.48877960420697,
                    -132.47281878119625,
                    80.07322451191962,
                    -153.6576260286505,
                    132.68551803423546,
                    -80.00077009153658,
                    -51.997462168305745,
                    153.7991379768883,
                    23.16567813229571,
                    -131.05090870526027,
                    -127.12884880680001,
                    78.64859490287598,
                    48.92926919615123,
                    -24.727586740962828,
                    122.58612513326514,
                    21.77439278960927,
                    -125.76931712622056,
                    -125.74073765033152,
                    80.0426755121125,
                    21.777566975827256,
                    -132.44498931449678,
                    -132.42900706519674,
                    80.02728301897348,
                    132.6810094155372,
                    -79.99762121869672,
                    23.143257146829296,
                    -131.07930405205326,
                    -127.10641743033305,
                    78.67699005613794,
                    -24.70160016412828,
                    122.61976920163778,
                    -51.99494566405491,
                    -51.96822501044932,
                    153.80921869922668,
                    153.7884674983891,
                    131.31369785374284,
                    -78.6600207658817,
                    -78.66299721602995,
                    123.95267473131548,
                    131.34226489198522,
                    -26.017479410022126,
                    -26.040134871297788,
                    123.98614650907767,
                    -78.63145078809187,
                    131.3422678315288,
                    131.34659358189884,
                    -26.04367639118502,
                    -78.66001794858028,
                    123.98615172228948,
                    124.00880175004527,
                    -26.017479630334073,
                    -52.052900593679674,
                    -52.02628443433404,
                    153.81024343522864,
                    153.78958316669895,
                    -153.64700460697915,
                    -153.67007950614166,
                    48.97017619185099,
                    48.968667340366274,
                    -153.66939902717587,
                    -153.64767355042574,
                    48.99849612044402,
                    48.940330999740254,
                    -51.968221530535786,
                    -51.99493875296791,
                    153.78846873350307,
                    153.80921727058168,
                ],
                ("C", "N", "C", "C"): [
                    -9.522309657420727e-05,
                    0.0025246278671328224,
                    -0.0015368971508784505,
                    0.002584298224398102,
                    -9.52230966722856e-05,
                    0.0025246278673199786,
                    -0.00153689715077464,
                    0.0025842982241423345,
                    -9.522309671790354e-05,
                    0.0025246278674934406,
                    -0.001536897150638974,
                    0.002584298223964221,
                    -9.522309654569603e-05,
                    0.0025246278672252575,
                    -0.0015368971508088513,
                    0.002584298224265653,
                    -9.522309643735347e-05,
                    0.002524627867064338,
                    -0.0015368971512323703,
                    0.0025842982246584156,
                    -9.522309657420704e-05,
                    0.0025246278672537833,
                    -0.0015368971509492418,
                    0.002584298224377541,
                    -9.522309658104985e-05,
                    0.0025246278672811755,
                    -0.0015368971507309958,
                    0.002584298224233678,
                    -9.52230965399938e-05,
                    0.002524627867254923,
                    -0.0015368971506413355,
                    0.0025842982241286385,
                    -9.522309634269601e-05,
                    0.00252462786705977,
                    -0.0015368971508607594,
                    0.00258429822416516,
                    -9.522309658903231e-05,
                    0.0025246278674420893,
                    -0.0015368971508702002,
                    0.002584298224247376,
                    -9.522309663921279e-05,
                    0.0025246278672903075,
                    -0.0015368971506802625,
                    0.0025842982240509953,
                    -9.522309643849378e-05,
                    0.0025246278672081297,
                    -0.0015368971505858927,
                    0.0025842982239687854,
                ],
                ("N", "C", "C", "C"): [
                    -0.004104968945188446,
                    -0.000776501588310222,
                    -0.000765564372668298,
                    -0.0014392589680501313,
                    -0.004104968945313624,
                    -0.0007765015883803146,
                    -0.0007655643727098696,
                    -0.0014392589676829172,
                    -0.004104968945617624,
                    -0.0007765015883236585,
                    -0.0007655643725897747,
                    -0.001439258967715452,
                    -0.004104968945339422,
                    -0.0007765015884060139,
                    -0.0007655643726221057,
                    -0.0014392589679804101,
                    -0.004104968945197716,
                    -0.0007765015884457308,
                    -0.0007655643723634239,
                    -0.001439258968203527,
                    -0.004104968945322901,
                    -0.0007765015884176931,
                    -0.0007655643726498196,
                    -0.0014392589678827981,
                    -0.004104968945415348,
                    -0.0007765015883855709,
                    -0.0007655643726498223,
                    -0.001439258968031541,
                    -0.004104968945347805,
                    -0.000776501588477277,
                    -0.0007655643727699229,
                    -0.0014392589678734985,
                    -0.004104968945120059,
                    -0.0007765015887208398,
                    -0.0007655643725389565,
                    -0.0014392589677340532,
                    -0.004104968945585162,
                    -0.0007765015885088154,
                    -0.0007655643725990097,
                    -0.0014392589678223708,
                    -0.004104968945324643,
                    -0.0007765015883689258,
                    -0.0007655643727791601,
                    -0.0014392589676131952,
                    -0.004104968945286968,
                    -0.0007765015886390666,
                    -0.0007655643728114937,
                    -0.0014392589676131987,
                ],
            },
            name=name,
        ),
    ),
)
def case_data(request: pytest.FixtureRequest) -> CaseData:
    return request.param(
        f"{request.fixturename}{request.param_index}",
    )
