import posix_ipc
import mmap
import numpy as np
import uuid
import json

numpy_types = dict(bool=np.bool, bool8=np.bool8, bool_=np.bool_, uint8=np.uint8,
                   uint16=np.uint16, uint32=np.uint32, uint64=np.uint64,
                   int_=np.int_, intc=np.intc, intp=np.intp, int=np.int,
                   int8=np.int8, int16=np.int16, int32=np.int32, int64=np.int64,
                   float_=np.float_, float=np.float, float16=np.float16,
                   float32=np.float32, float64=np.float64, float128=np.float128,
                   complex64=np.complex64, complex128=np.complex128)


class SharedArray(object):
    """
    SharedArray class allows you to create numpy object in shared memory
    space, then you can attach that shared memory space into another process
    and read that numpy object directly from memory without need to send
    data from one process to another.
    By doing this SharedArray avoids expensive serialization/deserialization
    and transfer times.
    It uses IPC capabilities provided by operating system to efficiently share
    data between processes.
    """

    def __init__(self, data, shared_memory, memory_buffer):
        self.shared_memory = shared_memory
        self.memory_buffer = memory_buffer
        self.data = data

    def to_json(self):
        """
        Generate json_descriptor for a shared_memory_block that contains numpy
        array. This descriptor should be passed to another process, then that
        process can use from_json() function to attach to the shared_memory.

        @return: return json descriptor for shared numpy array.
        @rtype: str

        """
        metadata = {"shared_mem_uuid": self.shared_memory.name,
                    "shape": self.data.shape,
                    "data_type": self.data.dtype.name}
        return json.dumps(metadata)

    @classmethod
    def from_json(cls, json_string):
        """
        Attaches to shared_memory block and reads numpy array from there, by
        using json_descriptor of shared array. Json descriptor should be
        generated by to_json() method.

        @param json_string:
        @type str:
        @return: Return Shared Numpy Array located in the shared memory!
        @rtype: SharedArray

        """
        params = json.loads(json_string)
        shared_memory = posix_ipc.SharedMemory(name=params["shared_mem_uuid"],
                                               flags=posix_ipc.O_CREAT,
                                               read_only=False)
        memory_buffer = mmap.mmap(shared_memory.fd, shared_memory.size)
        data = np.ndarray(buffer=memory_buffer,
                          dtype=numpy_types[params["data_type"]],
                          shape=params["shape"])

        return cls(data=data,
                   shared_memory=shared_memory,
                   memory_buffer=memory_buffer)

    @classmethod
    def from_array(cls, array: np.ndarray):
        """

        Put numpy array into shared memory in order to read it in another
        process.
        @param array:
        @type np.ndarray:
        @return: Return SharedArray object that contain numpy array.
        @rtype: SharedArray

        """
        if type(array) is np.ndarray:
            shared_mem_uuid = str(uuid.uuid4())[16:]

            shared_memory = posix_ipc.SharedMemory(name=shared_mem_uuid,
                                                   flags=posix_ipc.O_CREX,
                                                   size=array.nbytes,
                                                   read_only=False)
            memory_buffer = mmap.mmap(shared_memory.fd, shared_memory.size)
            memory_buffer.write(array.data)

            data = np.ndarray(buffer=memory_buffer,
                              dtype=array.dtype,
                              shape=array.shape)
        else:
            raise ValueError("array should be of type np.ndarray "
                             "instead of {}".format(type(array)))
        return cls(data=data,
                   shared_memory=shared_memory,
                   memory_buffer=memory_buffer)

    def get_data(self):
        """
        Extract numpy array from SharedArray object.
        @return: Return numpy array.
        @rtype: np.ndarray
        """
        return np.copy(self.data)

    def __unlink(self):
        """
        Closes memory_buffer and memory_descriptor.
        @return: None
        @rtype: None
        """
        self.memory_buffer.close()
        self.shared_memory.close_fd()

    def destroy(self):
        """
        Order operating system to destroy allocated shared memory.
        Important you should always destroy shared_memory after use. Otherwise
        you will have memory leak. This should be called by master process,
        otherwise child processes can destroy it without waiting for other
        processes to read the information.
        @return: None
        @rtype: None
        """
        posix_ipc.unlink_shared_memory(self.shared_memory.name)

    def __del__(self):
        self.__unlink()

