import dill
import pytest

from graphtik.modifiers import (
    dep_renamed,
    dep_singularized,
    dep_stripped,
    mapped,
    optional,
    sfx,
    sfxed,
    sfxed_vararg,
    sfxed_varargs,
    vararg,
    varargs,
)


def test_dill_modifier():
    s = sfxed("foo", "gg")
    s == dill.loads(dill.dumps(s))


@pytest.mark.parametrize(
    "mod, exp",
    [
        (lambda: mapped("b", None), "b"),
        (lambda: mapped("b", ""), "b"),
        (lambda: mapped("b", "bb"), "b"),
        (lambda: optional("b"), "b"),
        (lambda: optional("b", "bb"), "b"),
        (lambda: vararg("c"), "c"),
        (lambda: varargs("d"), "d"),
        (lambda: sfx("e"), "sfx: 'e'"),
        (lambda: sfx("e", optional=1), "sfx: 'e'"),
        (lambda: sfxed("f", "a", "b"), "sfxed('f', 'a', 'b')"),
        (lambda: sfxed("f", "ff", fn_kwarg="F"), "sfxed('f', 'ff')",),
        (lambda: sfxed("f", "ff", optional=1, fn_kwarg="F"), "sfxed('f', 'ff')",),
        (lambda: sfxed("f", "ff", optional=1), "sfxed('f', 'ff')"),
    ],
)
def test_modifs_str(mod, exp):
    got = str(mod())
    print(got)
    assert got == exp


@pytest.mark.parametrize(
    "mod, exp",
    [
        (lambda: mapped("b", None), "'b'(>)"),
        (lambda: mapped("b", ""), "'b'(>)"),
        (lambda: mapped("b", "bb"), "'b'(>'bb')"),
        (lambda: optional("b"), "'b'(?)"),
        (lambda: optional("b", "bb"), "'b'(?>'bb')"),
        (lambda: vararg("c"), "'c'(*)"),
        (lambda: varargs("d"), "'d'(+)"),
        (lambda: sfx("e"), "sfx: 'e'"),
        (lambda: sfx("e", optional=1), "sfx(?): 'e'"),
        (lambda: sfxed("f", "a", "b"), "sfxed('f', 'a', 'b')"),
        (lambda: sfxed("f", "ff", fn_kwarg="F"), "sfxed('f'(>'F'), 'ff')",),
        (
            lambda: sfxed("f", "ff", optional=1, fn_kwarg="F"),
            "sfxed('f'(?>'F'), 'ff')",
        ),
        (lambda: sfxed("f", "ff", optional=1), "sfxed('f'(?), 'ff')"),
        (lambda: sfxed_vararg("f", "a"), "sfxed('f'(*), 'a')"),
        (lambda: sfxed_varargs("f", "a", "b"), "sfxed('f'(+), 'a', 'b')"),
    ],
)
def test_modifs_repr(mod, exp):
    got = repr(mod())
    print(got)
    assert got == exp


def test_recreation():
    assert optional(optional("a")) == optional("a")
    assert mapped(mapped("a", 1), 1) == mapped("a", 1)
    assert vararg(vararg("a")) == vararg("a")
    assert varargs(varargs("a")) == varargs("a")


def test_recreation_repr():
    assert repr(optional(optional("a"))) == repr(optional("a"))
    assert repr(mapped(mapped("a", 1), 1)) == repr(mapped("a", 1))
    assert repr(vararg(vararg("a"))) == repr(vararg("a"))
    assert repr(varargs(varargs("a"))) == repr(varargs("a"))


@pytest.mark.parametrize(
    "call, exp",
    [
        (lambda: sfx(sfx("a")), "^`sideffected` cannot"),
        (lambda: sfxed("a", sfx("a")), "^`sfx_list` cannot"),
        (lambda: sfxed(sfx("a"), "a"), "^`sideffected` cannot"),
        (lambda: sfxed_vararg(sfx("a"), "a"), "^`sideffected` cannot"),
        (lambda: sfxed_varargs(sfx("a"), "a"), "^`sideffected` cannot"),
    ],
)
def test_sideffected_bad(call, exp):
    with pytest.raises(ValueError, match=exp):
        call()


@pytest.mark.parametrize(
    "mod, exp",
    [
        ("b", "'p.b'"),
        (mapped("b", None), "'p.b'(>'b')"),
        (mapped("b", ""), "'p.b'(>'b')"),
        (mapped("b", "bb"), "'p.b'(>'bb')"),
        (optional("b"), "'p.b'(?>'b')"),
        (optional("b", "bb"), "'p.b'(?>'bb')"),
        (vararg("c"), "'p.c'(*)"),
        (varargs("d"), "'p.d'(+)"),
        (sfx("e"), "sfx: 'p.e'"),
        (sfx("e", optional=1), "sfx(?): 'p.e'"),
        (sfxed("f", "a", "b"), "sfxed('p.f', 'a', 'b')",),
        (sfxed("f", "ff", fn_kwarg="F"), "sfxed('p.f'(>'F'), 'ff')",),
        (sfxed("f", "ff", optional=1, fn_kwarg="F"), "sfxed('p.f'(?>'F'), 'ff')",),
        (sfxed("f", "ff", optional=1), "sfxed('p.f'(?>'f'), 'ff')",),
        (sfxed_vararg("f", "a", "b"), "sfxed('p.f'(*), 'a', 'b')"),
        (sfxed_varargs("f", "a"), "sfxed('p.f'(+), 'a')"),
    ],
)
def test_modifs_rename_fn(mod, exp):
    renamer = lambda n: f"p.{n}"
    got = repr(dep_renamed(mod, renamer))
    print(got)
    assert got == exp


@pytest.mark.parametrize(
    "mod, exp",
    [
        (lambda: "s", "'D'"),
        (lambda: mapped("b", "bb"), "'D'(>'bb')"),
        (lambda: optional("b"), "'D'(?>'b')"),
        (lambda: optional("b", "bb"), "'D'(?>'bb')"),
        (lambda: vararg("c"), "'D'(*)"),
        (lambda: varargs("d"), "'D'(+)"),
        (lambda: sfx("e"), "sfx: 'D'"),
        (lambda: sfxed("f", "a", "b", optional=1,), "sfxed('D'(?>'f'), 'a', 'b')",),
        (
            lambda: sfxed("f", "a", "b", optional=1, fn_kwarg="F"),
            "sfxed('D'(?>'F'), 'a', 'b')",
        ),
    ],
)
def test_modifs_rename_str(mod, exp):
    got = repr(dep_renamed(mod(), "D"))
    print(got)
    assert got == exp


@pytest.mark.parametrize(
    "mod, exp",
    [
        (varargs("d"), "'d'(+)"),
        (sfx("e", optional=1), "sfx(?): 'e'"),
        (sfxed("f", "a", "b"), "'f'"),
        (sfxed("f", "ff", fn_kwarg="F"), "'f'(>'F')",),
        (sfxed("f", "ff", optional=1, fn_kwarg="F"), "'f'(?>'F')",),
        (sfxed("f", "ff", optional=1), "'f'(?)"),
        (sfxed_vararg("f", "a"), "'f'(*)"),
        (sfxed_varargs("f", "a", "b"), "'f'(+)"),
    ],
)
def test_sideffected_strip(mod, exp):
    got = dep_stripped(mod)
    assert repr(got) == exp


@pytest.mark.parametrize(
    "mod, exp",
    [
        ("a", ["a"]),
        (sfx("a"), [sfx("a")]),
        (sfxed("a", "b"), [sfxed("a", "b")]),
        (sfxed("a", "b", "c"), [sfxed("a", "b"), sfxed("a", "c")]),
    ],
)
def test_sideffected_singularized(mod, exp):
    got = list(dep_singularized(mod))
    assert got == exp
