// Copyright (c) Mito
// Distributed under the terms of the Modified BSD License.
import React, { useState } from 'react';
import useSyncedParams from '../../../hooks/useSyncedParams';
import { StepType } from '../../../types';
import { getDisplayColumnHeader } from '../../../utils/columnHeaders';
import DropdownItem from '../../elements/DropdownItem';
import LoadingDots from '../../elements/LoadingDots';
import MultiToggleBox from '../../elements/MultiToggleBox';
import MultiToggleItem from '../../elements/MultiToggleItem';
import Select from '../../elements/Select';
import Col from '../../spacing/Col';
import Row from '../../spacing/Row';
import DefaultEmptyTaskpane from '../DefaultTaskpane/DefaultEmptyTaskpane';
import DefaultTaskpane from '../DefaultTaskpane/DefaultTaskpane';
import DefaultTaskpaneBody from '../DefaultTaskpane/DefaultTaskpaneBody';
import DefaultTaskpaneHeader from '../DefaultTaskpane/DefaultTaskpaneHeader';
import { TaskpaneType } from '../taskpanes';
// Millisecond delay between changing params, so that
// we don't load send too many drop duplicate messages when the user
// is just configuring. 
const SEND_MESSAGE_DELAY = 250;
export const getDefaultParams = (selectedSheetIndex, sheetDataArray) => {
    var _a, _b;
    if (sheetDataArray.length === 0) {
        return undefined;
    }
    return {
        sheet_index: selectedSheetIndex,
        column_ids: ((_b = (_a = sheetDataArray[selectedSheetIndex]) === null || _a === void 0 ? void 0 : _a.data) === null || _b === void 0 ? void 0 : _b.map(c => c.columnID)) || [],
        keep: 'first',
    };
};
/*
    A taskpane that allows a user to drop duplicates
    in a sheet
*/
const DropDuplicatesTaskpane = (props) => {
    var _a, _b;
    const { params, setParams, loading } = useSyncedParams(getDefaultParams(props.selectedSheetIndex, props.sheetDataArray), StepType.DropDuplicates, props.mitoAPI, props.analysisData, SEND_MESSAGE_DELAY);
    const [originalNumRows, setOriginalNumRows] = useState(((_a = props.sheetDataArray[props.selectedSheetIndex]) === null || _a === void 0 ? void 0 : _a.numRows) || 0);
    if (props.sheetDataArray.length === 0 || params === undefined) {
        return React.createElement(DefaultEmptyTaskpane, { setUIState: props.setUIState });
    }
    /*
        If the sheetDataArray doesn't contain params.sheet_index,
        just close the taskpane to avoid a sheet crashing bug.
        
        TODO: We should handle this in useSyncedParams to so we can move
        closer to not having to write any custom code for this step.
    */
    if (props.sheetDataArray[params.sheet_index] === undefined) {
        props.setUIState((prevUIState) => {
            return Object.assign(Object.assign({}, prevUIState), { currOpenTaskpane: { type: TaskpaneType.NONE } });
        });
        // Return the defaut taskpane while the taskpane is closing
        return React.createElement(DefaultEmptyTaskpane, { setUIState: props.setUIState });
    }
    const columnIDsAndHeaders = ((_b = props.sheetDataArray[params.sheet_index]) === null || _b === void 0 ? void 0 : _b.data.map(c => [c.columnID, c.columnHeader])) || [];
    return (React.createElement(DefaultTaskpane, null,
        React.createElement(DefaultTaskpaneHeader, { header: 'Drop Duplicates', setUIState: props.setUIState }),
        React.createElement(DefaultTaskpaneBody, null,
            React.createElement(Row, { justify: 'space-between', align: 'center' },
                React.createElement(Col, null,
                    React.createElement("p", { className: 'text-header-3' }, "Sheet to Deduplicate")),
                React.createElement(Col, null,
                    React.createElement(Select, { width: 'medium', value: props.sheetDataArray[params.sheet_index].dfName, onChange: (newDfName) => {
                            const newSheetIndex = props.dfNames.indexOf(newDfName);
                            setParams(dropDuplicateParams => {
                                return Object.assign(Object.assign({}, dropDuplicateParams), { sheet_index: newSheetIndex, column_ids: props.sheetDataArray[newSheetIndex].data.map(c => c.columnID) });
                            });
                            setOriginalNumRows(props.sheetDataArray[newSheetIndex].numRows);
                            props.setUIState(prevUIState => {
                                return Object.assign(Object.assign({}, prevUIState), { selectedSheetIndex: newSheetIndex });
                            });
                        } }, props.dfNames.map(dfName => {
                        return (React.createElement(DropdownItem, { key: dfName, title: dfName }));
                    })))),
            React.createElement(Row, { justify: 'space-between', align: 'center' },
                React.createElement(Col, null,
                    React.createElement("p", { className: 'text-header-3' }, "Duplicates to Keep")),
                React.createElement(Col, null,
                    React.createElement(Select, { width: 'medium', value: params.keep + '', onChange: (newKeep) => {
                            if (newKeep === 'false') {
                                newKeep = false;
                            }
                            setParams(dropDuplicateParams => {
                                return Object.assign(Object.assign({}, dropDuplicateParams), { keep: newKeep });
                            });
                        } },
                        React.createElement(DropdownItem, { title: 'first', subtext: 'Keep the first instance of the duplicated row.' }),
                        React.createElement(DropdownItem, { title: 'last', subtext: 'Keep the last instance of the duplicated row.' }),
                        React.createElement(DropdownItem, { id: 'false', title: 'none', subtext: 'Keep none of the duplicated rows.' })))),
            React.createElement("p", { className: 'text-header-3 mt-10px' }, "Columns to Deduplicate On"),
            React.createElement(MultiToggleBox, { searchable: true, toggleAllIndexes: (indexesToToggle, newToggle) => {
                    const columnIDsToToggle = indexesToToggle.map(index => columnIDsAndHeaders[index][0]);
                    setParams(oldDropDuplicateParams => {
                        const newSelectedColumnIDs = [...oldDropDuplicateParams.column_ids];
                        columnIDsToToggle.forEach(columnID => {
                            if (newToggle) {
                                if (!newSelectedColumnIDs.includes(columnID)) {
                                    newSelectedColumnIDs.push(columnID);
                                }
                            }
                            else {
                                if (newSelectedColumnIDs.includes(columnID)) {
                                    newSelectedColumnIDs.splice(newSelectedColumnIDs.indexOf(columnID), 1);
                                }
                            }
                        });
                        return Object.assign(Object.assign({}, oldDropDuplicateParams), { column_ids: newSelectedColumnIDs });
                    });
                }, height: 'large' }, columnIDsAndHeaders.map(([columnID, columnHeader], index) => {
                return (React.createElement(MultiToggleItem, { key: index, title: getDisplayColumnHeader(columnHeader), toggled: params.column_ids.includes(columnID), index: index, onToggle: () => {
                        setParams(oldDropDuplicateParams => {
                            const newSelectedColumnIDs = [...oldDropDuplicateParams.column_ids];
                            if (!newSelectedColumnIDs.includes(columnID)) {
                                newSelectedColumnIDs.push(columnID);
                            }
                            else {
                                newSelectedColumnIDs.splice(newSelectedColumnIDs.indexOf(columnID), 1);
                            }
                            return Object.assign(Object.assign({}, oldDropDuplicateParams), { column_ids: newSelectedColumnIDs });
                        });
                    } }));
            })),
            loading &&
                React.createElement(Row, { className: 'mt-5' },
                    React.createElement("p", { className: 'text-subtext-1' },
                        "Deduplicating ",
                        React.createElement(LoadingDots, null))),
            !loading &&
                React.createElement(Row, { className: 'mt-5' },
                    React.createElement("p", { className: 'text-subtext-1' },
                        "Removed ",
                        originalNumRows - props.sheetDataArray[params.sheet_index].numRows,
                        " rows")))));
};
export default DropDuplicatesTaskpane;
//# sourceMappingURL=DropDuplicates.js.map