"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPoolClient = exports.UserPoolClientIdentityProvider = exports.OAuthScope = void 0;
const core_1 = require("@aws-cdk/core");
const cognito_generated_1 = require("./cognito.generated");
/**
 * OAuth scopes that are allowed with this client.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
 */
class OAuthScope {
    constructor(scopeName) {
        this.scopeName = scopeName;
    }
    /**
     * Custom scope is one that you define for your own resource server in the Resource Servers.
     *
     * The format is 'resource-server-identifier/scope'.
     *
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-define-resource-servers.html
     */
    static custom(name) {
        return new OAuthScope(name);
    }
}
exports.OAuthScope = OAuthScope;
/**
 * Grants access to the 'phone_number' and 'phone_number_verified' claims.
 *
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PHONE = new OAuthScope('phone');
/**
 * Grants access to the 'email' and 'email_verified' claims.
 *
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.EMAIL = new OAuthScope('email');
/**
 * Returns all user attributes in the ID token that are readable by the client.
 */
OAuthScope.OPENID = new OAuthScope('openid');
/**
 * Grants access to all user attributes that are readable by the client Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PROFILE = new OAuthScope('profile');
/**
 * Grants access to Amazon Cognito User Pool API operations that require access tokens, such as UpdateUserAttributes and VerifyUserAttribute.
 */
OAuthScope.COGNITO_ADMIN = new OAuthScope('aws.cognito.signin.user.admin');
/**
 * Identity providers supported by the UserPoolClient.
 */
class UserPoolClientIdentityProvider {
    constructor(name) {
        this.name = name;
    }
    /**
     * Specify a provider not yet supported by the CDK.
     *
     * @param name name of the identity provider as recognized by CloudFormation property `SupportedIdentityProviders`.
     */
    static custom(name) {
        return new UserPoolClientIdentityProvider(name);
    }
}
exports.UserPoolClientIdentityProvider = UserPoolClientIdentityProvider;
/**
 * Allow users to sign in using 'Facebook Login'.
 *
 * A `UserPoolIdentityProviderFacebook` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.FACEBOOK = new UserPoolClientIdentityProvider('Facebook');
/**
 * Allow users to sign in using 'Google Login'.
 *
 * A `UserPoolIdentityProviderGoogle` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.GOOGLE = new UserPoolClientIdentityProvider('Google');
/**
 * Allow users to sign in using 'Login With Amazon'.
 *
 * A `UserPoolIdentityProviderAmazon` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.AMAZON = new UserPoolClientIdentityProvider('LoginWithAmazon');
/**
 * Allow users to sign in directly as a user of the User Pool.
 */
UserPoolClientIdentityProvider.COGNITO = new UserPoolClientIdentityProvider('COGNITO');
/**
 * Define a UserPool App Client.
 */
class UserPoolClient extends core_1.Resource {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        if (props.disableOAuth && props.oAuth) {
            throw new Error('OAuth settings cannot be specified when disableOAuth is set.');
        }
        this.oAuthFlows = (_b = (_a = props.oAuth) === null || _a === void 0 ? void 0 : _a.flows) !== null && _b !== void 0 ? _b : {
            implicitCodeGrant: true,
            authorizationCodeGrant: true,
        };
        let callbackUrls = (_c = props.oAuth) === null || _c === void 0 ? void 0 : _c.callbackUrls;
        if (this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) {
            if (callbackUrls === undefined) {
                callbackUrls = ['https://example.com'];
            }
            else if (callbackUrls.length === 0) {
                throw new Error('callbackUrl must not be empty when codeGrant or implicitGrant OAuth flows are enabled.');
            }
        }
        const resource = new cognito_generated_1.CfnUserPoolClient(this, 'Resource', {
            clientName: props.userPoolClientName,
            generateSecret: props.generateSecret,
            userPoolId: props.userPool.userPoolId,
            explicitAuthFlows: this.configureAuthFlows(props),
            allowedOAuthFlows: props.disableOAuth ? undefined : this.configureOAuthFlows(),
            allowedOAuthScopes: props.disableOAuth ? undefined : this.configureOAuthScopes(props.oAuth),
            callbackUrLs: callbackUrls && callbackUrls.length > 0 && !props.disableOAuth ? callbackUrls : undefined,
            logoutUrLs: (_d = props.oAuth) === null || _d === void 0 ? void 0 : _d.logoutUrls,
            allowedOAuthFlowsUserPoolClient: !props.disableOAuth,
            preventUserExistenceErrors: this.configurePreventUserExistenceErrors(props.preventUserExistenceErrors),
            supportedIdentityProviders: this.configureIdentityProviders(props),
        });
        this.userPoolClientId = resource.ref;
        this._userPoolClientName = props.userPoolClientName;
    }
    /**
     * Import a user pool client given its id.
     */
    static fromUserPoolClientId(scope, id, userPoolClientId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.userPoolClientId = userPoolClientId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The client name that was specified via the `userPoolClientName` property during initialization, throws an error otherwise.
     */
    get userPoolClientName() {
        if (this._userPoolClientName === undefined) {
            throw new Error('userPoolClientName is available only if specified on the UserPoolClient during initialization');
        }
        return this._userPoolClientName;
    }
    configureAuthFlows(props) {
        if (!props.authFlows)
            return undefined;
        const authFlows = [];
        if (props.authFlows.userPassword) {
            authFlows.push('ALLOW_USER_PASSWORD_AUTH');
        }
        if (props.authFlows.adminUserPassword) {
            authFlows.push('ALLOW_ADMIN_USER_PASSWORD_AUTH');
        }
        if (props.authFlows.custom) {
            authFlows.push('ALLOW_CUSTOM_AUTH');
        }
        if (props.authFlows.userSrp) {
            authFlows.push('ALLOW_USER_SRP_AUTH');
        }
        // refreshToken should always be allowed if authFlows are present
        if (authFlows.length > 0) {
            authFlows.push('ALLOW_REFRESH_TOKEN_AUTH');
        }
        if (authFlows.length === 0) {
            return undefined;
        }
        return authFlows;
    }
    configureOAuthFlows() {
        if ((this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) && this.oAuthFlows.clientCredentials) {
            throw new Error('clientCredentials OAuth flow cannot be selected along with codeGrant or implicitGrant.');
        }
        const oAuthFlows = [];
        if (this.oAuthFlows.clientCredentials) {
            oAuthFlows.push('client_credentials');
        }
        if (this.oAuthFlows.implicitCodeGrant) {
            oAuthFlows.push('implicit');
        }
        if (this.oAuthFlows.authorizationCodeGrant) {
            oAuthFlows.push('code');
        }
        if (oAuthFlows.length === 0) {
            return undefined;
        }
        return oAuthFlows;
    }
    configureOAuthScopes(oAuth) {
        var _a;
        const scopes = (_a = oAuth === null || oAuth === void 0 ? void 0 : oAuth.scopes) !== null && _a !== void 0 ? _a : [OAuthScope.PROFILE, OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.OPENID,
            OAuthScope.COGNITO_ADMIN];
        const scopeNames = new Set(scopes.map((x) => x.scopeName));
        const autoOpenIdScopes = [OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.PROFILE];
        if (autoOpenIdScopes.reduce((agg, s) => agg || scopeNames.has(s.scopeName), false)) {
            scopeNames.add(OAuthScope.OPENID.scopeName);
        }
        return Array.from(scopeNames);
    }
    configurePreventUserExistenceErrors(prevent) {
        if (prevent === undefined) {
            return undefined;
        }
        return prevent ? 'ENABLED' : 'LEGACY';
    }
    configureIdentityProviders(props) {
        let providers;
        if (!props.supportedIdentityProviders) {
            const providerSet = new Set(props.userPool.identityProviders.map((p) => p.providerName));
            providerSet.add('COGNITO');
            providers = Array.from(providerSet);
        }
        else {
            providers = props.supportedIdentityProviders.map((p) => p.name);
        }
        if (providers.length === 0) {
            return undefined;
        }
        return Array.from(providers);
    }
}
exports.UserPoolClient = UserPoolClient;
//# sourceMappingURL=data:application/json;base64,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