# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['dataclass_mapper']

package_data = \
{'': ['*']}

extras_require = \
{'pydantic': ['pydantic>=1.9.0,<2.0.0']}

setup_kwargs = {
    'name': 'dataclass-mapper',
    'version': '1.0.1',
    'description': 'Autogenerate mappings between dataclasses',
    'long_description': '# dataclass-mapper\n\n[![Stable Version](https://img.shields.io/pypi/v/dataclass-mapper?color=blue)](https://pypi.org/project/dataclass-mapper/)\n[![Build Status](https://github.com/dataclass-mapper/dataclass-mapper/actions/workflows/test.yml/badge.svg)](https://github.com/dataclass-mapper/dataclass-mapper/actions)\n[![Documentation Status](https://readthedocs.org/projects/dataclass-mapper/badge/?version=latest)](https://dataclass-mapper.readthedocs.io/en/latest/?badge=latest)\n\nWriting mapper methods between two similar dataclasses is boring and error-prone.\nMuch better to let a library auto-generate them for you.\n\nThis library makes sure that all fields of the target class are actually mapped to (already at the module import time), and also provides helper mappers for variables that don\'t change their names.\nIt supports Python\'s dataclasses and also Pydantic models.\n\n## Installation\n\n```bash\npip install dataclass-mapper\n# or for Pydantic support\npip install dataclass-mapper[pydantic]\n```\n\n## Example\n\nWe have the following target data structure, a class `WorkContract` that contains an attribute of type `Person`.\n\n```python\nfrom dataclasses import dataclass\n\n@dataclass\nclass Person:\n    first_name: str\n    second_name: str\n    full_name: str\n    age: int\n\n@dataclass\nclass WorkContract:\n    worker: Person\n    manager: Optional[Person]\n    salary: int\n    signable: bool\n```\n\nWe want to have a safe mapper from the source data structure - `SoftwareDeveloperContract` with the attribute `ContactInfo`.\nNotice that the attribute `second_name` of `Person` is called `surname` in `ContactInfo`.\nOther than that, all the attribute names are the same.\n\nInstead of writing:\n\n```python\n@dataclass\nclass ContactInfo:\n    first_name: str\n    surname: str\n    age: int\n\n    def to_Person(self) -> Person:\n        return Person(\n            first_name=self.first_name,\n            second_name=self.surname,\n            full_name=f"{self.first_name} {self.surname}",\n            age=self.age,\n        )\n      \n@dataclass\nclass SoftwareDeveloperContract:\n    worker: ContactInfo\n    manager: Optional[ContactInfo]\n    salary: int\n\n    def to_WorkContract(self) -> WorkContract:\n        return WorkContract(\n            worker=self.worker.to_Person(),\n            manager=(None if self.manager is None else self.manager.to_Person()),\n            salary=self.salary,\n            signable=True,\n        )\n\n\nsoftware_developer_contract: SoftwareDeveloperContract\nwork_contract = software_developer_contract.to_WorkContract()\n```\n\nyou can write:\n\n```python\nfrom dataclass_mapper import map_to, mapper\n\n@mapper(Person, {\n  "second_name": "surname",\n  "full_name": lambda self: f"{self.first_name} {self.surname}"\n})\n@dataclass\nclass ContactInfo:\n    first_name: str\n    surname: str\n    age: int\n      \n@mapper(WorkContract, {"signable": lambda: True})\n@dataclass\nclass SoftwareDeveloperContract:\n    worker: ContactInfo\n    manager: Optional[ContactInfo]\n    salary: int\n\nsoftware_developer_contract: SoftwareDeveloperContract\nwork_contract = map_to(software_developer_contract, WorkContract)\n```\n\n## Features\n\nThe current version has support for:\n\n- :heavy_check_mark: Python\'s `dataclass`\n- :heavy_check_mark: `pydantic` classes, if installed with `pip install dataclass-mapper[pydantic]`\n- :heavy_check_mark: Checking if all target fields are actually initialized.\n  Raises a `ValueError` at class definition time when the type is different.\n- :heavy_check_mark: Simple types (`str`, `int`, `float`, `datetime`, custom types) if the type on the target is the same.\n  Raises a `TypeError` at class definition time when the type is different.\n- :heavy_check_mark: `Optional` types.\n  Raises a `TypeError` at class definition time when an optional type is mapped to a non-optional type.\n- :heavy_check_mark: Recursive models\n- :heavy_check_mark: `List` types\n- :heavy_check_mark: Default values for simple types\n- :heavy_check_mark: Mapper in the other direction. Use the `mapper_from` decorator and the same `map_to` method.\n- :heavy_check_mark: Assign Values with lambdas (with `{"x": lambda: 42}`)\n- :heavy_check_mark: Assign Functions Calls with lambdas and `self` (with `{"x": lambda self: self.x}`)\n- :heavy_check_mark: `USE_DEFAULT` for values that you don\'t wanna set but have a default value/factory\n\nStill missing features:\n\n- :heavy_multiplication_x: `Union` types\n- :heavy_multiplication_x: `Dict` types\n- :heavy_multiplication_x: Aliases in `pydantic` classes\n- :heavy_multiplication_x: Checking if all source attributes were used\n- :heavy_multiplication_x: SQLAlchemy ORM\n',
    'author': 'Jakob Kogler',
    'author_email': 'jakob.kogler@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://dataclass-mapper.readthedocs.io',
    'packages': packages,
    'package_data': package_data,
    'extras_require': extras_require,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
