"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KongEksDataPlane = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
// import * as kong_core from '../../../../../../kong-core';
const kong_core = require("kong-core");
const defaultHelmProps = {
    repository: 'https://charts.konghq.com',
    chart: 'kong',
    release: 'kong',
};
class KongEksDataPlane extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.helmOptions = { ...defaultHelmProps, ...props.HelmOptions };
        const kong_namespace = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongNamespace', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'Namespace',
                    metadata: { name: props.namespace },
                },
            ],
        });
        const kong_certificate = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongCert', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'cert-manager.io/v1',
                    kind: 'Certificate',
                    metadata: {
                        name: kong_core.Tls.KONG_DP_CERTNAME,
                        namespace: props.namespace,
                    },
                    spec: {
                        commonName: props.endpoints.hostedZoneName,
                        dnsNames: [
                            props.endpoints.clusterDns,
                            props.endpoints.telemetryDns,
                            props.endpoints.hostedZoneName,
                        ],
                        duration: '2160h0m0s',
                        issuerRef: {
                            group: 'awspca.cert-manager.io',
                            kind: 'AWSPCAClusterIssuer',
                            name: kong_core.Tls.KONG_DP_CLUSTER_ISSUER_NAME,
                        },
                        renewBefore: '360h0m0s',
                        secretName: `${kong_core.Tls.KONG_DP_CLUSTER_ISSUER_NAME}-secret`,
                        usages: [
                            'server auth',
                            'client auth',
                        ],
                        privateKey: {
                            algorithm: 'RSA',
                            size: 2048,
                        },
                    },
                },
            ],
        });
        const kong_client_certificate = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongDpClientCert', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'cert-manager.io/v1',
                    kind: 'Certificate',
                    metadata: {
                        name: kong_core.Tls.KONG_ACME_CERTNAME,
                        namespace: props.namespace,
                    },
                    spec: {
                        commonName: props.endpoints.hostedZoneName,
                        dnsNames: [
                            props.endpoints.proxyDns,
                            props.endpoints.hostedZoneName,
                        ],
                        duration: '2160h0m0s',
                        issuerRef: {
                            // group: 'awspca.cert-manager.io',
                            kind: 'ClusterIssuer',
                            name: kong_core.Tls.KONG_ACME_CLUSTER_ISSUER_NAME,
                        },
                        renewBefore: '360h0m0s',
                        secretName: `${kong_core.Tls.KONG_ACME_CLUSTER_ISSUER_NAME}-secret`,
                        usages: [
                            'server auth',
                            'client auth',
                        ],
                        privateKey: {
                            algorithm: 'RSA',
                            size: 2048,
                        },
                    },
                },
            ],
        });
        kong_certificate.node.addDependency(kong_namespace);
        kong_client_certificate.node.addDependency(kong_namespace);
        const kong_dp_service_account = new aws_cdk_lib_1.aws_eks.ServiceAccount(this, 'KongDpSA', {
            cluster: props.cluster,
            name: 'kong-dp-service-account',
            namespace: props.namespace,
        });
        const aws_region = aws_cdk_lib_1.Stack.of(this).region;
        const aws_account = aws_cdk_lib_1.Stack.of(this).account;
        kong_dp_service_account.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            resources: [
                `arn:aws:secretsmanager:${aws_region}:${aws_account}:secret:${props.license_secret_name}-??????`,
            ],
            actions: [
                'secretsmanager:GetSecretValue',
                'secretsmanager:DescribeSecret',
            ],
        }));
        kong_dp_service_account.node.addDependency(kong_namespace);
        const secrets_crd_mount = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'SecretsMount', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'secrets-store.csi.x-k8s.io/v1',
                    kind: 'SecretProviderClass',
                    metadata: {
                        name: props.license_secret_name,
                        namespace: props.namespace,
                    },
                    spec: {
                        provider: 'aws',
                        secretObjects: [
                            {
                                secretName: props.license_secret_name,
                                type: 'opaque',
                                data: [
                                    {
                                        objectName: props.license_secret_name,
                                        key: 'license',
                                    },
                                ],
                            },
                        ],
                        parameters: {
                            objects: `- objectName: "${props.license_secret_name}"\n  objectType: "secretsmanager"\n`,
                        },
                    },
                },
            ],
        });
        secrets_crd_mount.node.addDependency(kong_namespace);
        const deploy_kong_dp_helm = new aws_cdk_lib_1.aws_eks.HelmChart(this, 'KongDpHelm', {
            ...this.helmOptions,
            cluster: props.cluster,
            namespace: props.namespace,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            wait: true,
            createNamespace: true,
            values: {
                resources: {
                    requests: {
                        cpu: '300m',
                        memory: '300Mi',
                    },
                    limits: {
                        cpu: '1200m',
                        memory: '800Mi',
                    },
                },
                autoscaling: {
                    enabled: true,
                    minReplicas: 1,
                    maxReplicas: 20,
                    metrics: [
                        {
                            type: 'Resource',
                            resource: {
                                name: 'cpu',
                                target: {
                                    type: 'Utilization',
                                    averageUtilization: 75,
                                },
                            },
                        },
                    ],
                },
                ingressController: {
                    enabled: false,
                },
                image: {
                    repository: 'kong/kong-gateway',
                    tag: '2.8',
                },
                env: {
                    database: 'off',
                    role: 'data_plane',
                    cluster_mtls: 'pki',
                    cluster_cert: `/etc/secrets/${kong_core.Tls.KONG_DP_CLUSTER_ISSUER_NAME}-secret/tls.crt`,
                    cluster_cert_key: `/etc/secrets/${kong_core.Tls.KONG_DP_CLUSTER_ISSUER_NAME}-secret/tls.key`,
                    cluster_ca_cert: `/etc/secrets/${kong_core.Tls.KONG_DP_CLUSTER_ISSUER_NAME}-secret/ca.crt`,
                    ssl_cert: `/etc/secrets/${kong_core.Tls.KONG_ACME_CLUSTER_ISSUER_NAME}-secret/tls.crt`,
                    ssl_cert_key: `/etc/secrets/${kong_core.Tls.KONG_ACME_CLUSTER_ISSUER_NAME}-secret/tls.key`,
                    cluster_server_name: props.endpoints.hostedZoneName,
                    cluster_control_plane: `${props.endpoints.clusterDns}:443`,
                    cluster_telemetry_endpoint: `${props.endpoints.telemetryDns}:443`,
                    proxy_url: `https://${props.endpoints.proxyDns}`,
                },
                proxy: {
                    http: {
                        enabled: false,
                    },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-nlb-target-type': 'ip',
                        // 'service.beta.kubernetes.io/aws-load-balancer-ssl-cert': telemetry_cert.certificateArn,
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'external',
                        'service.beta.kubernetes.io/aws-load-balancer-scheme': 'internet-facing',
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.proxyDns,
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-proxy`,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                deployment: {
                    serviceAccount: {
                        create: false,
                        name: kong_dp_service_account.serviceAccountName,
                    },
                    userDefinedVolumes: [
                        {
                            name: props.license_secret_name,
                            csi: {
                                driver: 'secrets-store.csi.k8s.io',
                                readOnly: true,
                                volumeAttributes: {
                                    secretProviderClass: props.license_secret_name,
                                },
                            },
                        },
                    ],
                    userDefinedVolumeMounts: [
                        {
                            name: props.license_secret_name,
                            mountPath: '/mnt/secrets',
                            readOnly: true,
                        },
                    ],
                },
                enterprise: {
                    enabled: true,
                    license_secret: props.license_secret_name,
                },
                manager: {
                    enabled: false,
                },
                portal: {
                    enabled: false,
                },
                portalapi: {
                    enabled: false,
                },
                secretVolumes: [
                    `${kong_core.Tls.KONG_DP_CLUSTER_ISSUER_NAME}-secret`,
                    `${kong_core.Tls.KONG_ACME_CLUSTER_ISSUER_NAME}-secret`,
                ],
            },
        });
        deploy_kong_dp_helm.node.addDependency(secrets_crd_mount, kong_client_certificate, kong_certificate);
    }
}
exports.KongEksDataPlane = KongEksDataPlane;
//# sourceMappingURL=data:application/json;base64,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