"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KongDataplaneTaskdefinition = void 0;
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
// import * as kong_core from '../../../../kong-core/lib';
// import * as kong_core from 'kong-core';
// import * as kong_core from '../../../../kong-core/';
const kong_core = require("kong-core");
class KongDataplaneTaskdefinition extends aws_ecs_1.FargateTaskDefinition {
    constructor(scope, id, props) {
        super(scope, id, props.kongTaskProps);
        const dataPlanePort = 8000;
        const clusterEndpoint = props.cluster_dns;
        const telemetryEndpoint = props.telemetry_dns;
        let env = {};
        env.KONG_DATABASE = 'off';
        env.KONG_LOG_LEVEL = 'debug';
        env.KONG_PROXY_ACCESS_LOG = '/dev/stdout';
        env.KONG_PROXY_ERROR_LOG = '/dev/stderr';
        env.KONG_ROLE = 'data_plane';
        env.KONG_CLUSTER_SERVER_NAME = 'kong-cp.internal';
        env.KONG_CLUSTER_CA_CERT = '/certs/ca.crt';
        env.KONG_CLUSTER_CERT = '/certs/cluster.crt';
        env.KONG_CLUSTER_CERT_KEY = '/certs/cluster.key';
        env.KONG_CLUSTER_CONTROL_PLANE = clusterEndpoint;
        env.KONG_CLUSTER_TELEMETRY_ENDPOINT = telemetryEndpoint;
        env.KONG_CLUSTER_MTLS = 'pki';
        this.addToExecutionRolePolicy(new aws_iam_1.PolicyStatement({
            resources: ['*'],
            actions: [
                'ecr:GetAuthorizationToken',
                'ecr:BatchCheckLayerAvailability',
                'ecr:GetDownloadUrlForLayer',
                'ecr:BatchGetImage',
            ],
        }));
        this.addToTaskRolePolicy(new aws_iam_1.PolicyStatement({
            resources: ['*'],
            actions: [
                'logs:CreateLogGroup',
                'logs:CreateLogStream',
                'logs:PutLogEvents',
                'acm-pca:Get*',
                'acm:ExportCertificate',
            ],
        }));
        const kongContainer = this.addContainer('KongDataPlaneContainer', {
            image: aws_ecs_1.ContainerImage.fromRegistry(props.image),
            environment: env,
            logging: aws_ecs_1.LogDrivers.firelens({
                options: {
                    'Name': 'cloudwatch',
                    'region': props.cluster.stack.region,
                    'log_group_name': props.cluster.clusterName + '-KongDatalPlaneContainer',
                    'auto_create_group': 'true',
                    'log_stream_prefix': 'from-data-plane-',
                    'log-driver-buffer-limit': '2097152',
                },
            }),
            secrets: props.secrets,
            essential: true,
            portMappings: [{ containerPort: dataPlanePort, hostPort: dataPlanePort, protocol: aws_ecs_1.Protocol.TCP }],
        });
        const kongCertImage = new kong_core.CustomImage(this, 'KongCustomImage', {
            imageName: 'docker.io/kong/kong-gateway:2.8.1.0-alpine',
        });
        const kongContainerCerts = this.addContainer('KongDataPlaneContainerCerts', {
            image: kongCertImage.kongCustomImage,
            essential: false,
            cpu: 256,
            environment: {
                certArn: props.certificates.certificate.ref,
                rootCAArn: props.certificates.certificate.certificateAuthorityArn || '',
            },
            logging: aws_ecs_1.LogDrivers.firelens({
                options: {
                    'Name': 'cloudwatch',
                    'region': props.cluster.stack.region,
                    'log_group_name': props.cluster.clusterName + '-KongDataPlaneContainerCerts',
                    'auto_create_group': 'true',
                    'log_stream_prefix': 'from-data-plane-',
                    'log-driver-buffer-limit': '2097152',
                },
            }),
            command: ['sh', '/scripts/getCertificates.sh'],
        });
        kongContainer.addContainerDependencies({
            container: kongContainerCerts,
            condition: aws_ecs_1.ContainerDependencyCondition.SUCCESS,
        });
        //Add the firelens log router.
        this.addFirelensLogRouter('KongFirelens', {
            image: aws_ecs_1.ContainerImage.fromRegistry('amazon/aws-for-fluent-bit'),
            firelensConfig: {
                type: aws_ecs_1.FirelensLogRouterType.FLUENTBIT,
            },
        });
        const mountpoints = {
            containerPath: '/certs',
            readOnly: false,
            sourceVolume: 'certs',
        };
        kongContainer.addMountPoints(mountpoints);
        kongContainerCerts.addMountPoints(mountpoints);
        this.addVolume({
            name: mountpoints.sourceVolume,
        });
    }
}
exports.KongDataplaneTaskdefinition = KongDataplaneTaskdefinition;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoia29uZy1kYXRhcGxhbmUtdGFza2RlZmluaXRpb24uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvcmVzb3VyY2VzL2Vjcy9rb25nLWRhdGFwbGFuZS10YXNrZGVmaW5pdGlvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxpREFLNkI7QUFDN0IsaURBQXNEO0FBSXRELDBEQUEwRDtBQUMxRCwwQ0FBMEM7QUFDMUMsdURBQXVEO0FBQ3ZELHVDQUF1QztBQWV2QyxNQUFhLDJCQUE0QixTQUFRLCtCQUFxQjtJQUNwRSxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQThCO1FBQ3RFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUV0QyxNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUM7UUFDM0IsTUFBTSxlQUFlLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQztRQUMxQyxNQUFNLGlCQUFpQixHQUFHLEtBQUssQ0FBQyxhQUFhLENBQUM7UUFFOUMsSUFBSSxHQUFHLEdBQThCLEVBQUUsQ0FBQztRQUV4QyxHQUFHLENBQUMsYUFBYSxHQUFHLEtBQUssQ0FBQztRQUMxQixHQUFHLENBQUMsY0FBYyxHQUFHLE9BQU8sQ0FBQztRQUM3QixHQUFHLENBQUMscUJBQXFCLEdBQUcsYUFBYSxDQUFDO1FBQzFDLEdBQUcsQ0FBQyxvQkFBb0IsR0FBRyxhQUFhLENBQUM7UUFDekMsR0FBRyxDQUFDLFNBQVMsR0FBRyxZQUFZLENBQUM7UUFDN0IsR0FBRyxDQUFDLHdCQUF3QixHQUFHLGtCQUFrQixDQUFDO1FBQ2xELEdBQUcsQ0FBQyxvQkFBb0IsR0FBRyxlQUFlLENBQUM7UUFDM0MsR0FBRyxDQUFDLGlCQUFpQixHQUFHLG9CQUFvQixDQUFDO1FBQzdDLEdBQUcsQ0FBQyxxQkFBcUIsR0FBRyxvQkFBb0IsQ0FBQztRQUNqRCxHQUFHLENBQUMsMEJBQTBCLEdBQUcsZUFBZSxDQUFDO1FBQ2pELEdBQUcsQ0FBQywrQkFBK0IsR0FBRyxpQkFBaUIsQ0FBQztRQUN4RCxHQUFHLENBQUMsaUJBQWlCLEdBQUcsS0FBSyxDQUFDO1FBRTlCLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxJQUFJLHlCQUFlLENBQUM7WUFDaEQsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO1lBQ2hCLE9BQU8sRUFBRTtnQkFDUCwyQkFBMkI7Z0JBQzNCLGlDQUFpQztnQkFDakMsNEJBQTRCO2dCQUM1QixtQkFBbUI7YUFDcEI7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLHlCQUFlLENBQUM7WUFDM0MsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO1lBQ2hCLE9BQU8sRUFBRTtnQkFDUCxxQkFBcUI7Z0JBQ3JCLHNCQUFzQjtnQkFDdEIsbUJBQW1CO2dCQUNuQixjQUFjO2dCQUNkLHVCQUF1QjthQUN4QjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBQ0osTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyx3QkFBd0IsRUFBRTtZQUNoRSxLQUFLLEVBQUUsd0JBQWMsQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQztZQUMvQyxXQUFXLEVBQUUsR0FBRztZQUNoQixPQUFPLEVBQUUsb0JBQVUsQ0FBQyxRQUFRLENBQUM7Z0JBQzNCLE9BQU8sRUFBRTtvQkFDUCxNQUFNLEVBQUUsWUFBWTtvQkFDcEIsUUFBUSxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLE1BQU07b0JBQ3BDLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsV0FBVyxHQUFDLDBCQUEwQjtvQkFDdEUsbUJBQW1CLEVBQUUsTUFBTTtvQkFDM0IsbUJBQW1CLEVBQUUsa0JBQWtCO29CQUN2Qyx5QkFBeUIsRUFBRSxTQUFTO2lCQUNyQzthQUNGLENBQUM7WUFDRixPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU87WUFDdEIsU0FBUyxFQUFFLElBQUk7WUFDZixZQUFZLEVBQUUsQ0FBQyxFQUFFLGFBQWEsRUFBRSxhQUFhLEVBQUUsUUFBUSxFQUFFLGFBQWEsRUFBRSxRQUFRLEVBQUUsa0JBQVEsQ0FBQyxHQUFHLEVBQUUsQ0FBQztTQUNsRyxDQUFDLENBQUM7UUFFSCxNQUFNLGFBQWEsR0FBRyxJQUFJLFNBQVMsQ0FBQyxXQUFXLENBQUMsSUFBSSxFQUFFLGlCQUFpQixFQUFFO1lBQ3ZFLFNBQVMsRUFBRSw0Q0FBNEM7U0FDeEQsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxrQkFBa0IsR0FBRyxJQUFJLENBQUMsWUFBWSxDQUFDLDZCQUE2QixFQUFFO1lBQzFFLEtBQUssRUFBRSxhQUFhLENBQUMsZUFBZTtZQUNwQyxTQUFTLEVBQUUsS0FBSztZQUNoQixHQUFHLEVBQUUsR0FBRztZQUNSLFdBQVcsRUFBRTtnQkFDWCxPQUFPLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxXQUFXLENBQUMsR0FBRztnQkFDM0MsU0FBUyxFQUFFLEtBQUssQ0FBQyxZQUFZLENBQUMsV0FBVyxDQUFDLHVCQUF1QixJQUFHLEVBQUU7YUFDdkU7WUFDRCxPQUFPLEVBQUUsb0JBQVUsQ0FBQyxRQUFRLENBQUM7Z0JBQzNCLE9BQU8sRUFBRTtvQkFDUCxNQUFNLEVBQUUsWUFBWTtvQkFDcEIsUUFBUSxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLE1BQU07b0JBQ3BDLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsV0FBVyxHQUFDLDhCQUE4QjtvQkFDMUUsbUJBQW1CLEVBQUUsTUFBTTtvQkFDM0IsbUJBQW1CLEVBQUUsa0JBQWtCO29CQUN2Qyx5QkFBeUIsRUFBRSxTQUFTO2lCQUNyQzthQUNGLENBQUM7WUFDRixPQUFPLEVBQUUsQ0FBQyxJQUFJLEVBQUUsNkJBQTZCLENBQUM7U0FDL0MsQ0FBQyxDQUFDO1FBRUgsYUFBYSxDQUFDLHdCQUF3QixDQUFDO1lBQ3JDLFNBQVMsRUFBRSxrQkFBa0I7WUFDN0IsU0FBUyxFQUFFLHNDQUE0QixDQUFDLE9BQU87U0FDaEQsQ0FBQyxDQUFDO1FBRUgsOEJBQThCO1FBQzlCLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxjQUFjLEVBQUU7WUFDeEMsS0FBSyxFQUFFLHdCQUFjLENBQUMsWUFBWSxDQUFDLDJCQUEyQixDQUFDO1lBQy9ELGNBQWMsRUFBRTtnQkFDZCxJQUFJLEVBQUUsK0JBQXFCLENBQUMsU0FBUzthQUN0QztTQUNGLENBQUMsQ0FBQztRQUVILE1BQU0sV0FBVyxHQUFHO1lBQ2xCLGFBQWEsRUFBRSxRQUFRO1lBQ3ZCLFFBQVEsRUFBRSxLQUFLO1lBQ2YsWUFBWSxFQUFFLE9BQU87U0FDdEIsQ0FBQztRQUVGLGFBQWEsQ0FBQyxjQUFjLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDMUMsa0JBQWtCLENBQUMsY0FBYyxDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBQy9DLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixJQUFJLEVBQUUsV0FBVyxDQUFDLFlBQVk7U0FDL0IsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztDQUNGO0FBOUdELGtFQThHQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7XG4gIENvbnRhaW5lckRlcGVuZGVuY3lDb25kaXRpb24sXG4gIEZhcmdhdGVUYXNrRGVmaW5pdGlvbiwgTG9nRHJpdmVycyxcbiAgUHJvdG9jb2wsXG4gIENvbnRhaW5lckltYWdlLCBTZWNyZXQsIElDbHVzdGVyLCBGaXJlbGVuc0xvZ1JvdXRlclR5cGUsIEZhcmdhdGVUYXNrRGVmaW5pdGlvblByb3BzLFxufSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZWNzJztcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1pYW0nO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG4vLyBpbXBvcnQgeyBDZXJ0aWZpY2F0ZXMgfSBmcm9tICcuLi8uLi8uLi8uLi9rb25nLWNvcmUvJztcbmltcG9ydCB7IENlcnRpZmljYXRlcyB9IGZyb20gJ2tvbmctY29yZSc7XG4vLyBpbXBvcnQgKiBhcyBrb25nX2NvcmUgZnJvbSAnLi4vLi4vLi4vLi4va29uZy1jb3JlL2xpYic7XG4vLyBpbXBvcnQgKiBhcyBrb25nX2NvcmUgZnJvbSAna29uZy1jb3JlJztcbi8vIGltcG9ydCAqIGFzIGtvbmdfY29yZSBmcm9tICcuLi8uLi8uLi8uLi9rb25nLWNvcmUvJztcbmltcG9ydCAqIGFzIGtvbmdfY29yZSBmcm9tICdrb25nLWNvcmUnO1xuXG5cbmV4cG9ydCBpbnRlcmZhY2UgS29uZ1Rhc2tEZWZpbml0aW9uUHJvcHMge1xuICByZWFkb25seSBjbHVzdGVyOiBJQ2x1c3RlcjtcbiAgcmVhZG9ubHkgaW1hZ2U6IHN0cmluZztcbiAgcmVhZG9ubHkgY2x1c3Rlcl9kbnM6IHN0cmluZztcbiAgcmVhZG9ubHkgdGVsZW1ldHJ5X2Ruczogc3RyaW5nO1xuICByZWFkb25seSBrb25nVGFza1Byb3BzOiBGYXJnYXRlVGFza0RlZmluaXRpb25Qcm9wcztcbiAgcmVhZG9ubHkgY2VydGlmaWNhdGVzOiBDZXJ0aWZpY2F0ZXM7XG4gIHJlYWRvbmx5IHNlY3JldHM6IHtcbiAgICBba2V5OiBzdHJpbmddOiBTZWNyZXQ7XG4gIH07XG59XG5cbmV4cG9ydCBjbGFzcyBLb25nRGF0YXBsYW5lVGFza2RlZmluaXRpb24gZXh0ZW5kcyBGYXJnYXRlVGFza0RlZmluaXRpb24ge1xuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogS29uZ1Rhc2tEZWZpbml0aW9uUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzLmtvbmdUYXNrUHJvcHMpO1xuXG4gICAgY29uc3QgZGF0YVBsYW5lUG9ydCA9IDgwMDA7XG4gICAgY29uc3QgY2x1c3RlckVuZHBvaW50ID0gcHJvcHMuY2x1c3Rlcl9kbnM7XG4gICAgY29uc3QgdGVsZW1ldHJ5RW5kcG9pbnQgPSBwcm9wcy50ZWxlbWV0cnlfZG5zO1xuXG4gICAgbGV0IGVudiA6IHsgW2tleTogc3RyaW5nXTogc3RyaW5nfSA9IHt9O1xuXG4gICAgZW52LktPTkdfREFUQUJBU0UgPSAnb2ZmJztcbiAgICBlbnYuS09OR19MT0dfTEVWRUwgPSAnZGVidWcnO1xuICAgIGVudi5LT05HX1BST1hZX0FDQ0VTU19MT0cgPSAnL2Rldi9zdGRvdXQnO1xuICAgIGVudi5LT05HX1BST1hZX0VSUk9SX0xPRyA9ICcvZGV2L3N0ZGVycic7XG4gICAgZW52LktPTkdfUk9MRSA9ICdkYXRhX3BsYW5lJztcbiAgICBlbnYuS09OR19DTFVTVEVSX1NFUlZFUl9OQU1FID0gJ2tvbmctY3AuaW50ZXJuYWwnO1xuICAgIGVudi5LT05HX0NMVVNURVJfQ0FfQ0VSVCA9ICcvY2VydHMvY2EuY3J0JztcbiAgICBlbnYuS09OR19DTFVTVEVSX0NFUlQgPSAnL2NlcnRzL2NsdXN0ZXIuY3J0JztcbiAgICBlbnYuS09OR19DTFVTVEVSX0NFUlRfS0VZID0gJy9jZXJ0cy9jbHVzdGVyLmtleSc7XG4gICAgZW52LktPTkdfQ0xVU1RFUl9DT05UUk9MX1BMQU5FID0gY2x1c3RlckVuZHBvaW50O1xuICAgIGVudi5LT05HX0NMVVNURVJfVEVMRU1FVFJZX0VORFBPSU5UID0gdGVsZW1ldHJ5RW5kcG9pbnQ7XG4gICAgZW52LktPTkdfQ0xVU1RFUl9NVExTID0gJ3BraSc7XG5cbiAgICB0aGlzLmFkZFRvRXhlY3V0aW9uUm9sZVBvbGljeShuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICBhY3Rpb25zOiBbXG4gICAgICAgICdlY3I6R2V0QXV0aG9yaXphdGlvblRva2VuJyxcbiAgICAgICAgJ2VjcjpCYXRjaENoZWNrTGF5ZXJBdmFpbGFiaWxpdHknLFxuICAgICAgICAnZWNyOkdldERvd25sb2FkVXJsRm9yTGF5ZXInLFxuICAgICAgICAnZWNyOkJhdGNoR2V0SW1hZ2UnLFxuICAgICAgXSxcbiAgICB9KSk7XG5cbiAgICB0aGlzLmFkZFRvVGFza1JvbGVQb2xpY3kobmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgYWN0aW9uczogW1xuICAgICAgICAnbG9nczpDcmVhdGVMb2dHcm91cCcsXG4gICAgICAgICdsb2dzOkNyZWF0ZUxvZ1N0cmVhbScsXG4gICAgICAgICdsb2dzOlB1dExvZ0V2ZW50cycsXG4gICAgICAgICdhY20tcGNhOkdldConLFxuICAgICAgICAnYWNtOkV4cG9ydENlcnRpZmljYXRlJyxcbiAgICAgIF0sXG4gICAgfSkpO1xuICAgIGNvbnN0IGtvbmdDb250YWluZXIgPSB0aGlzLmFkZENvbnRhaW5lcignS29uZ0RhdGFQbGFuZUNvbnRhaW5lcicsIHtcbiAgICAgIGltYWdlOiBDb250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkocHJvcHMuaW1hZ2UpLFxuICAgICAgZW52aXJvbm1lbnQ6IGVudixcbiAgICAgIGxvZ2dpbmc6IExvZ0RyaXZlcnMuZmlyZWxlbnMoe1xuICAgICAgICBvcHRpb25zOiB7XG4gICAgICAgICAgJ05hbWUnOiAnY2xvdWR3YXRjaCcsXG4gICAgICAgICAgJ3JlZ2lvbic6IHByb3BzLmNsdXN0ZXIuc3RhY2sucmVnaW9uLFxuICAgICAgICAgICdsb2dfZ3JvdXBfbmFtZSc6IHByb3BzLmNsdXN0ZXIuY2x1c3Rlck5hbWUrJy1Lb25nRGF0YWxQbGFuZUNvbnRhaW5lcicsXG4gICAgICAgICAgJ2F1dG9fY3JlYXRlX2dyb3VwJzogJ3RydWUnLFxuICAgICAgICAgICdsb2dfc3RyZWFtX3ByZWZpeCc6ICdmcm9tLWRhdGEtcGxhbmUtJyxcbiAgICAgICAgICAnbG9nLWRyaXZlci1idWZmZXItbGltaXQnOiAnMjA5NzE1MicsXG4gICAgICAgIH0sXG4gICAgICB9KSxcbiAgICAgIHNlY3JldHM6IHByb3BzLnNlY3JldHMsXG4gICAgICBlc3NlbnRpYWw6IHRydWUsXG4gICAgICBwb3J0TWFwcGluZ3M6IFt7IGNvbnRhaW5lclBvcnQ6IGRhdGFQbGFuZVBvcnQsIGhvc3RQb3J0OiBkYXRhUGxhbmVQb3J0LCBwcm90b2NvbDogUHJvdG9jb2wuVENQIH1dLFxuICAgIH0pO1xuXG4gICAgY29uc3Qga29uZ0NlcnRJbWFnZSA9IG5ldyBrb25nX2NvcmUuQ3VzdG9tSW1hZ2UodGhpcywgJ0tvbmdDdXN0b21JbWFnZScsIHtcbiAgICAgIGltYWdlTmFtZTogJ2RvY2tlci5pby9rb25nL2tvbmctZ2F0ZXdheToyLjguMS4wLWFscGluZScsXG4gICAgfSk7XG4gICAgY29uc3Qga29uZ0NvbnRhaW5lckNlcnRzID0gdGhpcy5hZGRDb250YWluZXIoJ0tvbmdEYXRhUGxhbmVDb250YWluZXJDZXJ0cycsIHtcbiAgICAgIGltYWdlOiBrb25nQ2VydEltYWdlLmtvbmdDdXN0b21JbWFnZSxcbiAgICAgIGVzc2VudGlhbDogZmFsc2UsXG4gICAgICBjcHU6IDI1NixcbiAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgIGNlcnRBcm46IHByb3BzLmNlcnRpZmljYXRlcy5jZXJ0aWZpY2F0ZS5yZWYsXG4gICAgICAgIHJvb3RDQUFybjogcHJvcHMuY2VydGlmaWNhdGVzLmNlcnRpZmljYXRlLmNlcnRpZmljYXRlQXV0aG9yaXR5QXJufHwgJycsXG4gICAgICB9LFxuICAgICAgbG9nZ2luZzogTG9nRHJpdmVycy5maXJlbGVucyh7XG4gICAgICAgIG9wdGlvbnM6IHtcbiAgICAgICAgICAnTmFtZSc6ICdjbG91ZHdhdGNoJyxcbiAgICAgICAgICAncmVnaW9uJzogcHJvcHMuY2x1c3Rlci5zdGFjay5yZWdpb24sXG4gICAgICAgICAgJ2xvZ19ncm91cF9uYW1lJzogcHJvcHMuY2x1c3Rlci5jbHVzdGVyTmFtZSsnLUtvbmdEYXRhUGxhbmVDb250YWluZXJDZXJ0cycsXG4gICAgICAgICAgJ2F1dG9fY3JlYXRlX2dyb3VwJzogJ3RydWUnLFxuICAgICAgICAgICdsb2dfc3RyZWFtX3ByZWZpeCc6ICdmcm9tLWRhdGEtcGxhbmUtJyxcbiAgICAgICAgICAnbG9nLWRyaXZlci1idWZmZXItbGltaXQnOiAnMjA5NzE1MicsXG4gICAgICAgIH0sXG4gICAgICB9KSxcbiAgICAgIGNvbW1hbmQ6IFsnc2gnLCAnL3NjcmlwdHMvZ2V0Q2VydGlmaWNhdGVzLnNoJ10sXG4gICAgfSk7XG5cbiAgICBrb25nQ29udGFpbmVyLmFkZENvbnRhaW5lckRlcGVuZGVuY2llcyh7XG4gICAgICBjb250YWluZXI6IGtvbmdDb250YWluZXJDZXJ0cyxcbiAgICAgIGNvbmRpdGlvbjogQ29udGFpbmVyRGVwZW5kZW5jeUNvbmRpdGlvbi5TVUNDRVNTLFxuICAgIH0pO1xuXG4gICAgLy9BZGQgdGhlIGZpcmVsZW5zIGxvZyByb3V0ZXIuXG4gICAgdGhpcy5hZGRGaXJlbGVuc0xvZ1JvdXRlcignS29uZ0ZpcmVsZW5zJywge1xuICAgICAgaW1hZ2U6IENvbnRhaW5lckltYWdlLmZyb21SZWdpc3RyeSgnYW1hem9uL2F3cy1mb3ItZmx1ZW50LWJpdCcpLFxuICAgICAgZmlyZWxlbnNDb25maWc6IHtcbiAgICAgICAgdHlwZTogRmlyZWxlbnNMb2dSb3V0ZXJUeXBlLkZMVUVOVEJJVCxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBjb25zdCBtb3VudHBvaW50cyA9IHtcbiAgICAgIGNvbnRhaW5lclBhdGg6ICcvY2VydHMnLFxuICAgICAgcmVhZE9ubHk6IGZhbHNlLFxuICAgICAgc291cmNlVm9sdW1lOiAnY2VydHMnLFxuICAgIH07XG5cbiAgICBrb25nQ29udGFpbmVyLmFkZE1vdW50UG9pbnRzKG1vdW50cG9pbnRzKTtcbiAgICBrb25nQ29udGFpbmVyQ2VydHMuYWRkTW91bnRQb2ludHMobW91bnRwb2ludHMpO1xuICAgIHRoaXMuYWRkVm9sdW1lKHtcbiAgICAgIG5hbWU6IG1vdW50cG9pbnRzLnNvdXJjZVZvbHVtZSxcbiAgICB9KTtcbiAgfVxufVxuXG5cbiJdfQ==