"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StreamViewType = exports.ProjectionType = exports.BillingMode = exports.AttributeType = exports.Table = exports.TableEncryption = void 0;
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const dynamodb_generated_1 = require("./dynamodb.generated");
const perms = require("./perms");
const replica_provider_1 = require("./replica-provider");
const scalable_table_attribute_1 = require("./scalable-table-attribute");
const HASH_KEY_TYPE = 'HASH';
const RANGE_KEY_TYPE = 'RANGE';
// https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
const MAX_LOCAL_SECONDARY_INDEX_COUNT = 5;
/**
 * What kind of server-side encryption to apply to this table.
 */
var TableEncryption;
(function (TableEncryption) {
    /**
     * Server-side KMS encryption with a master key owned by AWS.
     */
    TableEncryption["DEFAULT"] = "AWS_OWNED";
    /**
     * Server-side KMS encryption with a customer master key managed by customer.
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     */
    TableEncryption["CUSTOMER_MANAGED"] = "CUSTOMER_MANAGED";
    /**
     * Server-side KMS encryption with a master key managed by AWS.
     */
    TableEncryption["AWS_MANAGED"] = "AWS_MANAGED";
})(TableEncryption = exports.TableEncryption || (exports.TableEncryption = {}));
class TableBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        this.regionalArns = new Array();
    }
    /**
     * Adds an IAM policy statement associated with this table to an IAM
     * principal's policy.
     *
     * If `encryptionKey` is present, appropriate grants to the key needs to be added
     * separately using the `table.encryptionKey.grant*` methods.
     *
     * @param grantee The principal (no-op if undefined)
     * @param actions The set of actions to allow (i.e. "dynamodb:PutItem", "dynamodb:GetItem", ...)
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [
                this.tableArn,
                core_1.Lazy.stringValue({ produce: () => this.hasIndex ? `${this.tableArn}/index/*` : core_1.Aws.NO_VALUE }),
                ...this.regionalArns,
                ...this.regionalArns.map(arn => core_1.Lazy.stringValue({
                    produce: () => this.hasIndex ? `${arn}/index/*` : core_1.Aws.NO_VALUE,
                })),
            ],
            scope: this,
        });
    }
    /**
     * Adds an IAM policy statement associated with this table's stream to an
     * IAM principal's policy.
     *
     * If `encryptionKey` is present, appropriate grants to the key needs to be added
     * separately using the `table.encryptionKey.grant*` methods.
     *
     * @param grantee The principal (no-op if undefined)
     * @param actions The set of actions to allow (i.e. "dynamodb:DescribeStream", "dynamodb:GetRecords", ...)
     */
    grantStream(grantee, ...actions) {
        if (!this.tableStreamArn) {
            throw new Error(`DynamoDB Streams must be enabled on the table ${this.node.path}`);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.tableStreamArn],
            scope: this,
        });
    }
    /**
     * Permits an IAM principal all data read operations from this table:
     * BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to
     */
    grantReadData(grantee) {
        return this.combinedGrant(grantee, { keyActions: perms.KEY_READ_ACTIONS, tableActions: perms.READ_DATA_ACTIONS });
    }
    /**
     * Permits an IAM Principal to list streams attached to current dynamodb table.
     *
     * @param grantee The principal (no-op if undefined)
     */
    grantTableListStreams(grantee) {
        if (!this.tableStreamArn) {
            throw new Error(`DynamoDB Streams must be enabled on the table ${this.node.path}`);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['dynamodb:ListStreams'],
            resourceArns: [
                core_1.Lazy.stringValue({ produce: () => `${this.tableArn}/stream/*` }),
                ...this.regionalArns.map(arn => core_1.Lazy.stringValue({ produce: () => `${arn}/stream/*` })),
            ],
        });
    }
    /**
     * Permits an IAM principal all stream data read operations for this
     * table's stream:
     * DescribeStream, GetRecords, GetShardIterator, ListStreams.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to
     */
    grantStreamRead(grantee) {
        this.grantTableListStreams(grantee);
        return this.combinedGrant(grantee, { keyActions: perms.KEY_READ_ACTIONS, streamActions: perms.READ_STREAM_DATA_ACTIONS });
    }
    /**
     * Permits an IAM principal all data write operations to this table:
     * BatchWriteItem, PutItem, UpdateItem, DeleteItem.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to
     */
    grantWriteData(grantee) {
        return this.combinedGrant(grantee, { keyActions: perms.KEY_WRITE_ACTIONS, tableActions: perms.WRITE_DATA_ACTIONS });
    }
    /**
     * Permits an IAM principal to all data read/write operations to this table.
     * BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan,
     * BatchWriteItem, PutItem, UpdateItem, DeleteItem
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to
     */
    grantReadWriteData(grantee) {
        const tableActions = perms.READ_DATA_ACTIONS.concat(perms.WRITE_DATA_ACTIONS);
        const keyActions = perms.KEY_READ_ACTIONS.concat(perms.KEY_WRITE_ACTIONS);
        return this.combinedGrant(grantee, { keyActions, tableActions });
    }
    /**
     * Permits all DynamoDB operations ("dynamodb:*") to an IAM principal.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to
     */
    grantFullAccess(grantee) {
        const keyActions = perms.KEY_READ_ACTIONS.concat(perms.KEY_WRITE_ACTIONS);
        return this.combinedGrant(grantee, { keyActions, tableActions: ['dynamodb:*'] });
    }
    /**
     * Return the given named metric for this Table
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/DynamoDB',
            metricName,
            dimensions: {
                TableName: this.tableName,
            },
            ...props,
        });
    }
    /**
     * Metric for the consumed read capacity units this table
     *
     * @default sum over a minute
     */
    metricConsumedReadCapacityUnits(props) {
        return this.metric('ConsumedReadCapacityUnits', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the consumed write capacity units this table
     *
     * @default sum over a minute
     */
    metricConsumedWriteCapacityUnits(props) {
        return this.metric('ConsumedWriteCapacityUnits', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the system errors this table
     *
     * @default sum over a minute
     */
    metricSystemErrors(props) {
        return this.metric('SystemErrors', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the user errors this table
     *
     * @default sum over a minute
     */
    metricUserErrors(props) {
        return this.metric('UserErrors', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the conditional check failed requests this table
     *
     * @default sum over a minute
     */
    metricConditionalCheckFailedRequests(props) {
        return this.metric('ConditionalCheckFailedRequests', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the successful request latency this table
     *
     * @default avg over a minute
     */
    metricSuccessfulRequestLatency(props) {
        return this.metric('SuccessfulRequestLatency', { statistic: 'avg', ...props });
    }
    /**
     * Adds an IAM policy statement associated with this table to an IAM
     * principal's policy.
     * @param grantee The principal (no-op if undefined)
     * @param opts Options for keyActions, tableActions and streamActions
     */
    combinedGrant(grantee, opts) {
        if (opts.tableActions) {
            const resources = [this.tableArn,
                core_1.Lazy.stringValue({ produce: () => this.hasIndex ? `${this.tableArn}/index/*` : core_1.Aws.NO_VALUE }),
                ...this.regionalArns,
                ...this.regionalArns.map(arn => core_1.Lazy.stringValue({
                    produce: () => this.hasIndex ? `${arn}/index/*` : core_1.Aws.NO_VALUE,
                })),
            ];
            const ret = iam.Grant.addToPrincipal({
                grantee,
                actions: opts.tableActions,
                resourceArns: resources,
                scope: this,
            });
            if (this.encryptionKey && opts.keyActions) {
                this.encryptionKey.grant(grantee, ...opts.keyActions);
            }
            return ret;
        }
        if (opts.streamActions) {
            if (!this.tableStreamArn) {
                throw new Error(`DynamoDB Streams must be enabled on the table ${this.node.path}`);
            }
            const resources = [this.tableStreamArn];
            const ret = iam.Grant.addToPrincipal({
                grantee,
                actions: opts.streamActions,
                resourceArns: resources,
                scope: this,
            });
            return ret;
        }
        throw new Error(`Unexpected 'action', ${opts.tableActions || opts.streamActions}`);
    }
}
/**
 * Provides a DynamoDB table.
 */
class Table extends TableBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.tableName,
        });
        this.keySchema = new Array();
        this.attributeDefinitions = new Array();
        this.globalSecondaryIndexes = new Array();
        this.localSecondaryIndexes = new Array();
        this.secondaryIndexNames = new Set();
        this.nonKeyAttributes = new Set();
        this.tableScaling = {};
        this.indexScaling = new Map();
        const { sseSpecification, encryptionKey } = this.parseEncryption(props);
        this.billingMode = props.billingMode || BillingMode.PROVISIONED;
        this.validateProvisioning(props);
        let streamSpecification;
        if (props.replicationRegions) {
            if (props.stream && props.stream !== StreamViewType.NEW_AND_OLD_IMAGES) {
                throw new Error('`stream` must be set to `NEW_AND_OLD_IMAGES` when specifying `replicationRegions`');
            }
            streamSpecification = { streamViewType: StreamViewType.NEW_AND_OLD_IMAGES };
            if (props.billingMode && props.billingMode !== BillingMode.PAY_PER_REQUEST) {
                throw new Error('The `PAY_PER_REQUEST` billing mode must be used when specifying `replicationRegions`');
            }
            this.billingMode = BillingMode.PAY_PER_REQUEST;
        }
        else if (props.stream) {
            streamSpecification = { streamViewType: props.stream };
        }
        this.table = new dynamodb_generated_1.CfnTable(this, 'Resource', {
            tableName: this.physicalName,
            keySchema: this.keySchema,
            attributeDefinitions: this.attributeDefinitions,
            globalSecondaryIndexes: core_1.Lazy.anyValue({ produce: () => this.globalSecondaryIndexes }, { omitEmptyArray: true }),
            localSecondaryIndexes: core_1.Lazy.anyValue({ produce: () => this.localSecondaryIndexes }, { omitEmptyArray: true }),
            pointInTimeRecoverySpecification: props.pointInTimeRecovery ? { pointInTimeRecoveryEnabled: props.pointInTimeRecovery } : undefined,
            billingMode: this.billingMode === BillingMode.PAY_PER_REQUEST ? this.billingMode : undefined,
            provisionedThroughput: this.billingMode === BillingMode.PAY_PER_REQUEST ? undefined : {
                readCapacityUnits: props.readCapacity || 5,
                writeCapacityUnits: props.writeCapacity || 5,
            },
            sseSpecification,
            streamSpecification,
            timeToLiveSpecification: props.timeToLiveAttribute ? { attributeName: props.timeToLiveAttribute, enabled: true } : undefined,
        });
        this.table.applyRemovalPolicy(props.removalPolicy);
        this.encryptionKey = encryptionKey;
        if (props.tableName) {
            this.node.addMetadata('aws:cdk:hasPhysicalName', props.tableName);
        }
        this.tableArn = this.getResourceArnAttribute(this.table.attrArn, {
            service: 'dynamodb',
            resource: 'table',
            resourceName: this.physicalName,
        });
        this.tableName = this.getResourceNameAttribute(this.table.ref);
        this.tableStreamArn = streamSpecification ? this.table.attrStreamArn : undefined;
        this.scalingRole = this.makeScalingRole();
        this.addKey(props.partitionKey, HASH_KEY_TYPE);
        this.tablePartitionKey = props.partitionKey;
        if (props.sortKey) {
            this.addKey(props.sortKey, RANGE_KEY_TYPE);
            this.tableSortKey = props.sortKey;
        }
        if (props.replicationRegions && props.replicationRegions.length > 0) {
            this.createReplicaTables(props.replicationRegions);
        }
    }
    /**
     * Permits an IAM Principal to list all DynamoDB Streams.
     * @deprecated Use {@link #grantTableListStreams} for more granular permission
     * @param grantee The principal (no-op if undefined)
     */
    static grantListStreams(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['dynamodb:ListStreams'],
            resourceArns: ['*'],
        });
    }
    /**
     * Creates a Table construct that represents an external table via table name.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param tableName The table's name.
     */
    static fromTableName(scope, id, tableName) {
        return Table.fromTableAttributes(scope, id, { tableName });
    }
    /**
     * Creates a Table construct that represents an external table via table arn.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param tableArn The table's ARN.
     */
    static fromTableArn(scope, id, tableArn) {
        return Table.fromTableAttributes(scope, id, { tableArn });
    }
    /**
     * Creates a Table construct that represents an external table.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `TableAttributes` object.
     */
    static fromTableAttributes(scope, id, attrs) {
        class Import extends TableBase {
            constructor(_tableArn, tableName, tableStreamArn) {
                var _a, _b;
                super(scope, id);
                this.hasIndex = ((_a = attrs.globalIndexes) !== null && _a !== void 0 ? _a : []).length > 0 ||
                    ((_b = attrs.localIndexes) !== null && _b !== void 0 ? _b : []).length > 0;
                this.tableArn = _tableArn;
                this.tableName = tableName;
                this.tableStreamArn = tableStreamArn;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        let name;
        let arn;
        const stack = core_1.Stack.of(scope);
        if (!attrs.tableName) {
            if (!attrs.tableArn) {
                throw new Error('One of tableName or tableArn is required!');
            }
            arn = attrs.tableArn;
            const maybeTableName = stack.parseArn(attrs.tableArn).resourceName;
            if (!maybeTableName) {
                throw new Error('ARN for DynamoDB table must be in the form: ...');
            }
            name = maybeTableName;
        }
        else {
            if (attrs.tableArn) {
                throw new Error('Only one of tableArn or tableName can be provided');
            }
            name = attrs.tableName;
            arn = stack.formatArn({
                service: 'dynamodb',
                resource: 'table',
                resourceName: attrs.tableName,
            });
        }
        return new Import(arn, name, attrs.tableStreamArn);
    }
    /**
     * Add a global secondary index of table.
     *
     * @param props the property of global secondary index
     */
    addGlobalSecondaryIndex(props) {
        this.validateProvisioning(props);
        this.validateIndexName(props.indexName);
        // build key schema and projection for index
        const gsiKeySchema = this.buildIndexKeySchema(props.partitionKey, props.sortKey);
        const gsiProjection = this.buildIndexProjection(props);
        this.secondaryIndexNames.add(props.indexName);
        this.globalSecondaryIndexes.push({
            indexName: props.indexName,
            keySchema: gsiKeySchema,
            projection: gsiProjection,
            provisionedThroughput: this.billingMode === BillingMode.PAY_PER_REQUEST ? undefined : {
                readCapacityUnits: props.readCapacity || 5,
                writeCapacityUnits: props.writeCapacity || 5,
            },
        });
        this.indexScaling.set(props.indexName, {});
    }
    /**
     * Add a local secondary index of table.
     *
     * @param props the property of local secondary index
     */
    addLocalSecondaryIndex(props) {
        // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
        if (this.localSecondaryIndexes.length >= MAX_LOCAL_SECONDARY_INDEX_COUNT) {
            throw new RangeError(`a maximum number of local secondary index per table is ${MAX_LOCAL_SECONDARY_INDEX_COUNT}`);
        }
        this.validateIndexName(props.indexName);
        // build key schema and projection for index
        const lsiKeySchema = this.buildIndexKeySchema(this.tablePartitionKey, props.sortKey);
        const lsiProjection = this.buildIndexProjection(props);
        this.secondaryIndexNames.add(props.indexName);
        this.localSecondaryIndexes.push({
            indexName: props.indexName,
            keySchema: lsiKeySchema,
            projection: lsiProjection,
        });
    }
    /**
     * Enable read capacity scaling for this table
     *
     * @returns An object to configure additional AutoScaling settings
     */
    autoScaleReadCapacity(props) {
        if (this.tableScaling.scalableReadAttribute) {
            throw new Error('Read AutoScaling already enabled for this table');
        }
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        return this.tableScaling.scalableReadAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, 'ReadScaling', {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}`,
            dimension: 'dynamodb:table:ReadCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
    }
    /**
     * Enable write capacity scaling for this table
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     */
    autoScaleWriteCapacity(props) {
        if (this.tableScaling.scalableWriteAttribute) {
            throw new Error('Write AutoScaling already enabled for this table');
        }
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        return this.tableScaling.scalableWriteAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, 'WriteScaling', {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}`,
            dimension: 'dynamodb:table:WriteCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
    }
    /**
     * Enable read capacity scaling for the given GSI
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     */
    autoScaleGlobalSecondaryIndexReadCapacity(indexName, props) {
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        const attributePair = this.indexScaling.get(indexName);
        if (!attributePair) {
            throw new Error(`No global secondary index with name ${indexName}`);
        }
        if (attributePair.scalableReadAttribute) {
            throw new Error('Read AutoScaling already enabled for this index');
        }
        return attributePair.scalableReadAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, `${indexName}ReadScaling`, {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}/index/${indexName}`,
            dimension: 'dynamodb:index:ReadCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
    }
    /**
     * Enable write capacity scaling for the given GSI
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     */
    autoScaleGlobalSecondaryIndexWriteCapacity(indexName, props) {
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        const attributePair = this.indexScaling.get(indexName);
        if (!attributePair) {
            throw new Error(`No global secondary index with name ${indexName}`);
        }
        if (attributePair.scalableWriteAttribute) {
            throw new Error('Write AutoScaling already enabled for this index');
        }
        return attributePair.scalableWriteAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, `${indexName}WriteScaling`, {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}/index/${indexName}`,
            dimension: 'dynamodb:index:WriteCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
    }
    /**
     * Validate the table construct.
     *
     * @returns an array of validation error message
     */
    validate() {
        const errors = new Array();
        if (!this.tablePartitionKey) {
            errors.push('a partition key must be specified');
        }
        if (this.localSecondaryIndexes.length > 0 && !this.tableSortKey) {
            errors.push('a sort key of the table must be specified to add local secondary indexes');
        }
        return errors;
    }
    /**
     * Validate read and write capacity are not specified for on-demand tables (billing mode PAY_PER_REQUEST).
     *
     * @param props read and write capacity properties
     */
    validateProvisioning(props) {
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            if (props.readCapacity !== undefined || props.writeCapacity !== undefined) {
                throw new Error('you cannot provision read and write capacity for a table with PAY_PER_REQUEST billing mode');
            }
        }
    }
    /**
     * Validate index name to check if a duplicate name already exists.
     *
     * @param indexName a name of global or local secondary index
     */
    validateIndexName(indexName) {
        if (this.secondaryIndexNames.has(indexName)) {
            // a duplicate index name causes validation exception, status code 400, while trying to create CFN stack
            throw new Error(`a duplicate index name, ${indexName}, is not allowed`);
        }
        this.secondaryIndexNames.add(indexName);
    }
    /**
     * Validate non-key attributes by checking limits within secondary index, which may vary in future.
     *
     * @param nonKeyAttributes a list of non-key attribute names
     */
    validateNonKeyAttributes(nonKeyAttributes) {
        if (this.nonKeyAttributes.size + nonKeyAttributes.length > 100) {
            // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
            throw new RangeError('a maximum number of nonKeyAttributes across all of secondary indexes is 100');
        }
        // store all non-key attributes
        nonKeyAttributes.forEach(att => this.nonKeyAttributes.add(att));
    }
    buildIndexKeySchema(partitionKey, sortKey) {
        this.registerAttribute(partitionKey);
        const indexKeySchema = [
            { attributeName: partitionKey.name, keyType: HASH_KEY_TYPE },
        ];
        if (sortKey) {
            this.registerAttribute(sortKey);
            indexKeySchema.push({ attributeName: sortKey.name, keyType: RANGE_KEY_TYPE });
        }
        return indexKeySchema;
    }
    buildIndexProjection(props) {
        if (props.projectionType === ProjectionType.INCLUDE && !props.nonKeyAttributes) {
            // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-dynamodb-projectionobject.html
            throw new Error(`non-key attributes should be specified when using ${ProjectionType.INCLUDE} projection type`);
        }
        if (props.projectionType !== ProjectionType.INCLUDE && props.nonKeyAttributes) {
            // this combination causes validation exception, status code 400, while trying to create CFN stack
            throw new Error(`non-key attributes should not be specified when not using ${ProjectionType.INCLUDE} projection type`);
        }
        if (props.nonKeyAttributes) {
            this.validateNonKeyAttributes(props.nonKeyAttributes);
        }
        return {
            projectionType: props.projectionType ? props.projectionType : ProjectionType.ALL,
            nonKeyAttributes: props.nonKeyAttributes ? props.nonKeyAttributes : undefined,
        };
    }
    findKey(keyType) {
        return this.keySchema.find(prop => prop.keyType === keyType);
    }
    addKey(attribute, keyType) {
        const existingProp = this.findKey(keyType);
        if (existingProp) {
            throw new Error(`Unable to set ${attribute.name} as a ${keyType} key, because ${existingProp.attributeName} is a ${keyType} key`);
        }
        this.registerAttribute(attribute);
        this.keySchema.push({
            attributeName: attribute.name,
            keyType,
        });
        return this;
    }
    /**
     * Register the key attribute of table or secondary index to assemble attribute definitions of TableResourceProps.
     *
     * @param attribute the key attribute of table or secondary index
     */
    registerAttribute(attribute) {
        const { name, type } = attribute;
        const existingDef = this.attributeDefinitions.find(def => def.attributeName === name);
        if (existingDef && existingDef.attributeType !== type) {
            throw new Error(`Unable to specify ${name} as ${type} because it was already defined as ${existingDef.attributeType}`);
        }
        if (!existingDef) {
            this.attributeDefinitions.push({
                attributeName: name,
                attributeType: type,
            });
        }
    }
    /**
     * Return the role that will be used for AutoScaling
     */
    makeScalingRole() {
        // Use a Service Linked Role.
        // https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-service-linked-roles.html
        return iam.Role.fromRoleArn(this, 'ScalingRole', core_1.Stack.of(this).formatArn({
            service: 'iam',
            region: '',
            resource: 'role/aws-service-role/dynamodb.application-autoscaling.amazonaws.com',
            resourceName: 'AWSServiceRoleForApplicationAutoScaling_DynamoDBTable',
        }));
    }
    /**
     * Creates replica tables
     *
     * @param regions regions where to create tables
     */
    createReplicaTables(regions) {
        const stack = core_1.Stack.of(this);
        if (!core_1.Token.isUnresolved(stack.region) && regions.includes(stack.region)) {
            throw new Error('`replicationRegions` cannot include the region where this stack is deployed.');
        }
        const provider = replica_provider_1.ReplicaProvider.getOrCreate(this);
        // Documentation at https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/V2gt_IAM.html
        // is currently incorrect. AWS Support recommends `dynamodb:*` in both source and destination regions
        const onEventHandlerPolicy = new SourceTableAttachedPolicy(this, provider.onEventHandler.role);
        const isCompleteHandlerPolicy = new SourceTableAttachedPolicy(this, provider.isCompleteHandler.role);
        // Permissions in the source region
        this.grant(onEventHandlerPolicy, 'dynamodb:*');
        this.grant(isCompleteHandlerPolicy, 'dynamodb:DescribeTable');
        let previousRegion;
        for (const region of new Set(regions)) { // Remove duplicates
            // Use multiple custom resources because multiple create/delete
            // updates cannot be combined in a single API call.
            const currentRegion = new core_1.CustomResource(this, `Replica${region}`, {
                serviceToken: provider.provider.serviceToken,
                resourceType: 'Custom::DynamoDBReplica',
                properties: {
                    TableName: this.tableName,
                    Region: region,
                },
            });
            currentRegion.node.addDependency(onEventHandlerPolicy.policy, isCompleteHandlerPolicy.policy);
            // Deploy time check to prevent from creating a replica in the region
            // where this stack is deployed. Only needed for environment agnostic
            // stacks.
            if (core_1.Token.isUnresolved(stack.region)) {
                const createReplica = new core_1.CfnCondition(this, `StackRegionNotEquals${region}`, {
                    expression: core_1.Fn.conditionNot(core_1.Fn.conditionEquals(region, core_1.Aws.REGION)),
                });
                const cfnCustomResource = currentRegion.node.defaultChild;
                cfnCustomResource.cfnOptions.condition = createReplica;
            }
            // Save regional arns for grantXxx() methods
            this.regionalArns.push(stack.formatArn({
                region,
                service: 'dynamodb',
                resource: 'table',
                resourceName: this.tableName,
            }));
            // We need to create/delete regions sequentially because we cannot
            // have multiple table updates at the same time. The `isCompleteHandler`
            // of the provider waits until the replica is in an ACTIVE state.
            if (previousRegion) {
                currentRegion.node.addDependency(previousRegion);
            }
            previousRegion = currentRegion;
        }
        // Permissions in the destination regions (outside of the loop to
        // minimize statements in the policy)
        onEventHandlerPolicy.grantPrincipal.addToPolicy(new iam.PolicyStatement({
            actions: ['dynamodb:*'],
            resources: this.regionalArns,
        }));
    }
    /**
     * Whether this table has indexes
     */
    get hasIndex() {
        return this.globalSecondaryIndexes.length + this.localSecondaryIndexes.length > 0;
    }
    /**
     * Set up key properties and return the Table encryption property from the
     * user's configuration.
     */
    parseEncryption(props) {
        var _a;
        let encryptionType = props.encryption;
        if (encryptionType != null && props.serverSideEncryption != null) {
            throw new Error('Only one of encryption and serverSideEncryption can be specified, but both were provided');
        }
        if (props.serverSideEncryption && props.encryptionKey) {
            throw new Error('encryptionKey cannot be specified when serverSideEncryption is specified. Use encryption instead');
        }
        if (encryptionType === undefined) {
            encryptionType = props.encryptionKey != null
                // If there is a configured encyptionKey, the encryption is implicitly CUSTOMER_MANAGED
                ? TableEncryption.CUSTOMER_MANAGED
                // Otherwise, if severSideEncryption is enabled, it's AWS_MANAGED; else DEFAULT
                : props.serverSideEncryption ? TableEncryption.AWS_MANAGED : TableEncryption.DEFAULT;
        }
        if (encryptionType !== TableEncryption.CUSTOMER_MANAGED && props.encryptionKey) {
            throw new Error('`encryptionKey cannot be specified unless encryption is set to TableEncryption.CUSTOMER_MANAGED (it was set to ${encryptionType})`');
        }
        if (encryptionType === TableEncryption.CUSTOMER_MANAGED && props.replicationRegions) {
            throw new Error('TableEncryption.CUSTOMER_MANAGED is not supported by DynamoDB Global Tables (where replicationRegions was set)');
        }
        switch (encryptionType) {
            case TableEncryption.CUSTOMER_MANAGED:
                const encryptionKey = (_a = props.encryptionKey) !== null && _a !== void 0 ? _a : new kms.Key(this, 'Key', {
                    description: `Customer-managed key auto-created for encrypting DynamoDB table at ${this.node.path}`,
                    enableKeyRotation: true,
                });
                return {
                    sseSpecification: { sseEnabled: true, kmsMasterKeyId: encryptionKey.keyArn, sseType: 'KMS' },
                    encryptionKey,
                };
            case TableEncryption.AWS_MANAGED:
                // Not specifying "sseType: 'KMS'" here because it would cause phony changes to existing stacks.
                return { sseSpecification: { sseEnabled: true } };
            case TableEncryption.DEFAULT:
                // Not specifying "sseEnabled: false" here because it would cause phony changes to existing stacks.
                return { sseSpecification: undefined };
            default:
                throw new Error(`Unexpected 'encryptionType': ${encryptionType}`);
        }
    }
}
exports.Table = Table;
/**
 * Data types for attributes within a table
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes
 */
var AttributeType;
(function (AttributeType) {
    /** Up to 400KiB of binary data (which must be encoded as base64 before sending to DynamoDB) */
    AttributeType["BINARY"] = "B";
    /** Numeric values made of up to 38 digits (positive, negative or zero) */
    AttributeType["NUMBER"] = "N";
    /** Up to 400KiB of UTF-8 encoded text */
    AttributeType["STRING"] = "S";
})(AttributeType = exports.AttributeType || (exports.AttributeType = {}));
/**
 * DyanmoDB's Read/Write capacity modes.
 */
var BillingMode;
(function (BillingMode) {
    /**
     * Pay only for what you use. You don't configure Read/Write capacity units.
     */
    BillingMode["PAY_PER_REQUEST"] = "PAY_PER_REQUEST";
    /**
     * Explicitly specified Read/Write capacity units.
     */
    BillingMode["PROVISIONED"] = "PROVISIONED";
})(BillingMode = exports.BillingMode || (exports.BillingMode = {}));
/**
 * The set of attributes that are projected into the index
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Projection.html
 */
var ProjectionType;
(function (ProjectionType) {
    /** Only the index and primary keys are projected into the index. */
    ProjectionType["KEYS_ONLY"] = "KEYS_ONLY";
    /** Only the specified table attributes are projected into the index. The list of projected attributes is in `nonKeyAttributes`. */
    ProjectionType["INCLUDE"] = "INCLUDE";
    /** All of the table attributes are projected into the index. */
    ProjectionType["ALL"] = "ALL";
})(ProjectionType = exports.ProjectionType || (exports.ProjectionType = {}));
/**
 * When an item in the table is modified, StreamViewType determines what information
 * is written to the stream for this table.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_StreamSpecification.html
 */
var StreamViewType;
(function (StreamViewType) {
    /** The entire item, as it appears after it was modified, is written to the stream. */
    StreamViewType["NEW_IMAGE"] = "NEW_IMAGE";
    /** The entire item, as it appeared before it was modified, is written to the stream. */
    StreamViewType["OLD_IMAGE"] = "OLD_IMAGE";
    /** Both the new and the old item images of the item are written to the stream. */
    StreamViewType["NEW_AND_OLD_IMAGES"] = "NEW_AND_OLD_IMAGES";
    /** Only the key attributes of the modified item are written to the stream. */
    StreamViewType["KEYS_ONLY"] = "KEYS_ONLY";
})(StreamViewType = exports.StreamViewType || (exports.StreamViewType = {}));
/**
 * An inline policy that is logically bound to the source table of a DynamoDB Global Tables
 * "cluster". This is here to ensure permissions are removed as part of (and not before) the
 * CleanUp phase of a stack update, when a replica is removed (or the entire "cluster" gets
 * replaced).
 *
 * If statements are added directly to the handler roles (as opposed to in a separate inline
 * policy resource), new permissions are in effect before clean up happens, and so replicas that
 * need to be dropped can no longer be due to lack of permissions.
 */
class SourceTableAttachedPolicy extends core_1.Construct {
    constructor(sourceTable, role) {
        super(sourceTable, `SourceTableAttachedPolicy-${role.node.uniqueId}`);
        const policy = new iam.Policy(this, 'Resource', { roles: [role] });
        this.policy = policy;
        this.grantPrincipal = new SourceTableAttachedPrincipal(role, policy);
    }
}
/**
 * An `IPrincipal` entity that can be used as the target of `grant` calls, used by the
 * `SourceTableAttachedPolicy` class so it can act as an `IGrantable`.
 */
class SourceTableAttachedPrincipal extends iam.PrincipalBase {
    constructor(role, policy) {
        super();
        this.role = role;
        this.policy = policy;
    }
    get policyFragment() {
        return this.role.policyFragment;
    }
    addToPrincipalPolicy(statement) {
        this.policy.addStatements(statement);
        return {
            policyDependable: this.policy,
            statementAdded: true,
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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