(function () {
    'use strict';

    /*! *****************************************************************************
    Copyright (c) Microsoft Corporation.

    Permission to use, copy, modify, and/or distribute this software for any
    purpose with or without fee is hereby granted.

    THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH
    REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
    AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT,
    INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
    LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
    OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
    PERFORMANCE OF THIS SOFTWARE.
    ***************************************************************************** */

    function __decorate(decorators, target, key, desc) {
        var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
        if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
        else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
        return c > 3 && r && Object.defineProperty(target, key, r), r;
    }

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    /**
     * True if the custom elements polyfill is in use.
     */
    const isCEPolyfill = typeof window !== 'undefined' &&
        window.customElements != null &&
        window.customElements.polyfillWrapFlushCallback !==
            undefined;
    /**
     * Removes nodes, starting from `start` (inclusive) to `end` (exclusive), from
     * `container`.
     */
    const removeNodes = (container, start, end = null) => {
        while (start !== end) {
            const n = start.nextSibling;
            container.removeChild(start);
            start = n;
        }
    };

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    /**
     * An expression marker with embedded unique key to avoid collision with
     * possible text in templates.
     */
    const marker = `{{lit-${String(Math.random()).slice(2)}}}`;
    /**
     * An expression marker used text-positions, multi-binding attributes, and
     * attributes with markup-like text values.
     */
    const nodeMarker = `<!--${marker}-->`;
    const markerRegex = new RegExp(`${marker}|${nodeMarker}`);
    /**
     * Suffix appended to all bound attribute names.
     */
    const boundAttributeSuffix = '$lit$';
    /**
     * An updatable Template that tracks the location of dynamic parts.
     */
    class Template {
        constructor(result, element) {
            this.parts = [];
            this.element = element;
            const nodesToRemove = [];
            const stack = [];
            // Edge needs all 4 parameters present; IE11 needs 3rd parameter to be null
            const walker = document.createTreeWalker(element.content, 133 /* NodeFilter.SHOW_{ELEMENT|COMMENT|TEXT} */, null, false);
            // Keeps track of the last index associated with a part. We try to delete
            // unnecessary nodes, but we never want to associate two different parts
            // to the same index. They must have a constant node between.
            let lastPartIndex = 0;
            let index = -1;
            let partIndex = 0;
            const { strings, values: { length } } = result;
            while (partIndex < length) {
                const node = walker.nextNode();
                if (node === null) {
                    // We've exhausted the content inside a nested template element.
                    // Because we still have parts (the outer for-loop), we know:
                    // - There is a template in the stack
                    // - The walker will find a nextNode outside the template
                    walker.currentNode = stack.pop();
                    continue;
                }
                index++;
                if (node.nodeType === 1 /* Node.ELEMENT_NODE */) {
                    if (node.hasAttributes()) {
                        const attributes = node.attributes;
                        const { length } = attributes;
                        // Per
                        // https://developer.mozilla.org/en-US/docs/Web/API/NamedNodeMap,
                        // attributes are not guaranteed to be returned in document order.
                        // In particular, Edge/IE can return them out of order, so we cannot
                        // assume a correspondence between part index and attribute index.
                        let count = 0;
                        for (let i = 0; i < length; i++) {
                            if (endsWith(attributes[i].name, boundAttributeSuffix)) {
                                count++;
                            }
                        }
                        while (count-- > 0) {
                            // Get the template literal section leading up to the first
                            // expression in this attribute
                            const stringForPart = strings[partIndex];
                            // Find the attribute name
                            const name = lastAttributeNameRegex.exec(stringForPart)[2];
                            // Find the corresponding attribute
                            // All bound attributes have had a suffix added in
                            // TemplateResult#getHTML to opt out of special attribute
                            // handling. To look up the attribute value we also need to add
                            // the suffix.
                            const attributeLookupName = name.toLowerCase() + boundAttributeSuffix;
                            const attributeValue = node.getAttribute(attributeLookupName);
                            node.removeAttribute(attributeLookupName);
                            const statics = attributeValue.split(markerRegex);
                            this.parts.push({ type: 'attribute', index, name, strings: statics });
                            partIndex += statics.length - 1;
                        }
                    }
                    if (node.tagName === 'TEMPLATE') {
                        stack.push(node);
                        walker.currentNode = node.content;
                    }
                }
                else if (node.nodeType === 3 /* Node.TEXT_NODE */) {
                    const data = node.data;
                    if (data.indexOf(marker) >= 0) {
                        const parent = node.parentNode;
                        const strings = data.split(markerRegex);
                        const lastIndex = strings.length - 1;
                        // Generate a new text node for each literal section
                        // These nodes are also used as the markers for node parts
                        for (let i = 0; i < lastIndex; i++) {
                            let insert;
                            let s = strings[i];
                            if (s === '') {
                                insert = createMarker();
                            }
                            else {
                                const match = lastAttributeNameRegex.exec(s);
                                if (match !== null && endsWith(match[2], boundAttributeSuffix)) {
                                    s = s.slice(0, match.index) + match[1] +
                                        match[2].slice(0, -boundAttributeSuffix.length) + match[3];
                                }
                                insert = document.createTextNode(s);
                            }
                            parent.insertBefore(insert, node);
                            this.parts.push({ type: 'node', index: ++index });
                        }
                        // If there's no text, we must insert a comment to mark our place.
                        // Else, we can trust it will stick around after cloning.
                        if (strings[lastIndex] === '') {
                            parent.insertBefore(createMarker(), node);
                            nodesToRemove.push(node);
                        }
                        else {
                            node.data = strings[lastIndex];
                        }
                        // We have a part for each match found
                        partIndex += lastIndex;
                    }
                }
                else if (node.nodeType === 8 /* Node.COMMENT_NODE */) {
                    if (node.data === marker) {
                        const parent = node.parentNode;
                        // Add a new marker node to be the startNode of the Part if any of
                        // the following are true:
                        //  * We don't have a previousSibling
                        //  * The previousSibling is already the start of a previous part
                        if (node.previousSibling === null || index === lastPartIndex) {
                            index++;
                            parent.insertBefore(createMarker(), node);
                        }
                        lastPartIndex = index;
                        this.parts.push({ type: 'node', index });
                        // If we don't have a nextSibling, keep this node so we have an end.
                        // Else, we can remove it to save future costs.
                        if (node.nextSibling === null) {
                            node.data = '';
                        }
                        else {
                            nodesToRemove.push(node);
                            index--;
                        }
                        partIndex++;
                    }
                    else {
                        let i = -1;
                        while ((i = node.data.indexOf(marker, i + 1)) !== -1) {
                            // Comment node has a binding marker inside, make an inactive part
                            // The binding won't work, but subsequent bindings will
                            // TODO (justinfagnani): consider whether it's even worth it to
                            // make bindings in comments work
                            this.parts.push({ type: 'node', index: -1 });
                            partIndex++;
                        }
                    }
                }
            }
            // Remove text binding nodes after the walk to not disturb the TreeWalker
            for (const n of nodesToRemove) {
                n.parentNode.removeChild(n);
            }
        }
    }
    const endsWith = (str, suffix) => {
        const index = str.length - suffix.length;
        return index >= 0 && str.slice(index) === suffix;
    };
    const isTemplatePartActive = (part) => part.index !== -1;
    // Allows `document.createComment('')` to be renamed for a
    // small manual size-savings.
    const createMarker = () => document.createComment('');
    /**
     * This regex extracts the attribute name preceding an attribute-position
     * expression. It does this by matching the syntax allowed for attributes
     * against the string literal directly preceding the expression, assuming that
     * the expression is in an attribute-value position.
     *
     * See attributes in the HTML spec:
     * https://www.w3.org/TR/html5/syntax.html#elements-attributes
     *
     * " \x09\x0a\x0c\x0d" are HTML space characters:
     * https://www.w3.org/TR/html5/infrastructure.html#space-characters
     *
     * "\0-\x1F\x7F-\x9F" are Unicode control characters, which includes every
     * space character except " ".
     *
     * So an attribute is:
     *  * The name: any character except a control character, space character, ('),
     *    ("), ">", "=", or "/"
     *  * Followed by zero or more space characters
     *  * Followed by "="
     *  * Followed by zero or more space characters
     *  * Followed by:
     *    * Any character except space, ('), ("), "<", ">", "=", (`), or
     *    * (") then any non-("), or
     *    * (') then any non-(')
     */
    const lastAttributeNameRegex = 
    // eslint-disable-next-line no-control-regex
    /([ \x09\x0a\x0c\x0d])([^\0-\x1F\x7F-\x9F "'>=/]+)([ \x09\x0a\x0c\x0d]*=[ \x09\x0a\x0c\x0d]*(?:[^ \x09\x0a\x0c\x0d"'`<>=]*|"[^"]*|'[^']*))$/;

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    const walkerNodeFilter = 133 /* NodeFilter.SHOW_{ELEMENT|COMMENT|TEXT} */;
    /**
     * Removes the list of nodes from a Template safely. In addition to removing
     * nodes from the Template, the Template part indices are updated to match
     * the mutated Template DOM.
     *
     * As the template is walked the removal state is tracked and
     * part indices are adjusted as needed.
     *
     * div
     *   div#1 (remove) <-- start removing (removing node is div#1)
     *     div
     *       div#2 (remove)  <-- continue removing (removing node is still div#1)
     *         div
     * div <-- stop removing since previous sibling is the removing node (div#1,
     * removed 4 nodes)
     */
    function removeNodesFromTemplate(template, nodesToRemove) {
        const { element: { content }, parts } = template;
        const walker = document.createTreeWalker(content, walkerNodeFilter, null, false);
        let partIndex = nextActiveIndexInTemplateParts(parts);
        let part = parts[partIndex];
        let nodeIndex = -1;
        let removeCount = 0;
        const nodesToRemoveInTemplate = [];
        let currentRemovingNode = null;
        while (walker.nextNode()) {
            nodeIndex++;
            const node = walker.currentNode;
            // End removal if stepped past the removing node
            if (node.previousSibling === currentRemovingNode) {
                currentRemovingNode = null;
            }
            // A node to remove was found in the template
            if (nodesToRemove.has(node)) {
                nodesToRemoveInTemplate.push(node);
                // Track node we're removing
                if (currentRemovingNode === null) {
                    currentRemovingNode = node;
                }
            }
            // When removing, increment count by which to adjust subsequent part indices
            if (currentRemovingNode !== null) {
                removeCount++;
            }
            while (part !== undefined && part.index === nodeIndex) {
                // If part is in a removed node deactivate it by setting index to -1 or
                // adjust the index as needed.
                part.index = currentRemovingNode !== null ? -1 : part.index - removeCount;
                // go to the next active part.
                partIndex = nextActiveIndexInTemplateParts(parts, partIndex);
                part = parts[partIndex];
            }
        }
        nodesToRemoveInTemplate.forEach((n) => n.parentNode.removeChild(n));
    }
    const countNodes = (node) => {
        let count = (node.nodeType === 11 /* Node.DOCUMENT_FRAGMENT_NODE */) ? 0 : 1;
        const walker = document.createTreeWalker(node, walkerNodeFilter, null, false);
        while (walker.nextNode()) {
            count++;
        }
        return count;
    };
    const nextActiveIndexInTemplateParts = (parts, startIndex = -1) => {
        for (let i = startIndex + 1; i < parts.length; i++) {
            const part = parts[i];
            if (isTemplatePartActive(part)) {
                return i;
            }
        }
        return -1;
    };
    /**
     * Inserts the given node into the Template, optionally before the given
     * refNode. In addition to inserting the node into the Template, the Template
     * part indices are updated to match the mutated Template DOM.
     */
    function insertNodeIntoTemplate(template, node, refNode = null) {
        const { element: { content }, parts } = template;
        // If there's no refNode, then put node at end of template.
        // No part indices need to be shifted in this case.
        if (refNode === null || refNode === undefined) {
            content.appendChild(node);
            return;
        }
        const walker = document.createTreeWalker(content, walkerNodeFilter, null, false);
        let partIndex = nextActiveIndexInTemplateParts(parts);
        let insertCount = 0;
        let walkerIndex = -1;
        while (walker.nextNode()) {
            walkerIndex++;
            const walkerNode = walker.currentNode;
            if (walkerNode === refNode) {
                insertCount = countNodes(node);
                refNode.parentNode.insertBefore(node, refNode);
            }
            while (partIndex !== -1 && parts[partIndex].index === walkerIndex) {
                // If we've inserted the node, simply adjust all subsequent parts
                if (insertCount > 0) {
                    while (partIndex !== -1) {
                        parts[partIndex].index += insertCount;
                        partIndex = nextActiveIndexInTemplateParts(parts, partIndex);
                    }
                    return;
                }
                partIndex = nextActiveIndexInTemplateParts(parts, partIndex);
            }
        }
    }

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    const directives = new WeakMap();
    /**
     * Brands a function as a directive factory function so that lit-html will call
     * the function during template rendering, rather than passing as a value.
     *
     * A _directive_ is a function that takes a Part as an argument. It has the
     * signature: `(part: Part) => void`.
     *
     * A directive _factory_ is a function that takes arguments for data and
     * configuration and returns a directive. Users of directive usually refer to
     * the directive factory as the directive. For example, "The repeat directive".
     *
     * Usually a template author will invoke a directive factory in their template
     * with relevant arguments, which will then return a directive function.
     *
     * Here's an example of using the `repeat()` directive factory that takes an
     * array and a function to render an item:
     *
     * ```js
     * html`<ul><${repeat(items, (item) => html`<li>${item}</li>`)}</ul>`
     * ```
     *
     * When `repeat` is invoked, it returns a directive function that closes over
     * `items` and the template function. When the outer template is rendered, the
     * return directive function is called with the Part for the expression.
     * `repeat` then performs it's custom logic to render multiple items.
     *
     * @param f The directive factory function. Must be a function that returns a
     * function of the signature `(part: Part) => void`. The returned function will
     * be called with the part object.
     *
     * @example
     *
     * import {directive, html} from 'lit-html';
     *
     * const immutable = directive((v) => (part) => {
     *   if (part.value !== v) {
     *     part.setValue(v)
     *   }
     * });
     */
    const directive = (f) => ((...args) => {
        const d = f(...args);
        directives.set(d, true);
        return d;
    });
    const isDirective = (o) => {
        return typeof o === 'function' && directives.has(o);
    };

    /**
     * @license
     * Copyright (c) 2018 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    /**
     * A sentinel value that signals that a value was handled by a directive and
     * should not be written to the DOM.
     */
    const noChange = {};
    /**
     * A sentinel value that signals a NodePart to fully clear its content.
     */
    const nothing = {};

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    /**
     * An instance of a `Template` that can be attached to the DOM and updated
     * with new values.
     */
    class TemplateInstance {
        constructor(template, processor, options) {
            this.__parts = [];
            this.template = template;
            this.processor = processor;
            this.options = options;
        }
        update(values) {
            let i = 0;
            for (const part of this.__parts) {
                if (part !== undefined) {
                    part.setValue(values[i]);
                }
                i++;
            }
            for (const part of this.__parts) {
                if (part !== undefined) {
                    part.commit();
                }
            }
        }
        _clone() {
            // There are a number of steps in the lifecycle of a template instance's
            // DOM fragment:
            //  1. Clone - create the instance fragment
            //  2. Adopt - adopt into the main document
            //  3. Process - find part markers and create parts
            //  4. Upgrade - upgrade custom elements
            //  5. Update - set node, attribute, property, etc., values
            //  6. Connect - connect to the document. Optional and outside of this
            //     method.
            //
            // We have a few constraints on the ordering of these steps:
            //  * We need to upgrade before updating, so that property values will pass
            //    through any property setters.
            //  * We would like to process before upgrading so that we're sure that the
            //    cloned fragment is inert and not disturbed by self-modifying DOM.
            //  * We want custom elements to upgrade even in disconnected fragments.
            //
            // Given these constraints, with full custom elements support we would
            // prefer the order: Clone, Process, Adopt, Upgrade, Update, Connect
            //
            // But Safari does not implement CustomElementRegistry#upgrade, so we
            // can not implement that order and still have upgrade-before-update and
            // upgrade disconnected fragments. So we instead sacrifice the
            // process-before-upgrade constraint, since in Custom Elements v1 elements
            // must not modify their light DOM in the constructor. We still have issues
            // when co-existing with CEv0 elements like Polymer 1, and with polyfills
            // that don't strictly adhere to the no-modification rule because shadow
            // DOM, which may be created in the constructor, is emulated by being placed
            // in the light DOM.
            //
            // The resulting order is on native is: Clone, Adopt, Upgrade, Process,
            // Update, Connect. document.importNode() performs Clone, Adopt, and Upgrade
            // in one step.
            //
            // The Custom Elements v1 polyfill supports upgrade(), so the order when
            // polyfilled is the more ideal: Clone, Process, Adopt, Upgrade, Update,
            // Connect.
            const fragment = isCEPolyfill ?
                this.template.element.content.cloneNode(true) :
                document.importNode(this.template.element.content, true);
            const stack = [];
            const parts = this.template.parts;
            // Edge needs all 4 parameters present; IE11 needs 3rd parameter to be null
            const walker = document.createTreeWalker(fragment, 133 /* NodeFilter.SHOW_{ELEMENT|COMMENT|TEXT} */, null, false);
            let partIndex = 0;
            let nodeIndex = 0;
            let part;
            let node = walker.nextNode();
            // Loop through all the nodes and parts of a template
            while (partIndex < parts.length) {
                part = parts[partIndex];
                if (!isTemplatePartActive(part)) {
                    this.__parts.push(undefined);
                    partIndex++;
                    continue;
                }
                // Progress the tree walker until we find our next part's node.
                // Note that multiple parts may share the same node (attribute parts
                // on a single element), so this loop may not run at all.
                while (nodeIndex < part.index) {
                    nodeIndex++;
                    if (node.nodeName === 'TEMPLATE') {
                        stack.push(node);
                        walker.currentNode = node.content;
                    }
                    if ((node = walker.nextNode()) === null) {
                        // We've exhausted the content inside a nested template element.
                        // Because we still have parts (the outer for-loop), we know:
                        // - There is a template in the stack
                        // - The walker will find a nextNode outside the template
                        walker.currentNode = stack.pop();
                        node = walker.nextNode();
                    }
                }
                // We've arrived at our part's node.
                if (part.type === 'node') {
                    const part = this.processor.handleTextExpression(this.options);
                    part.insertAfterNode(node.previousSibling);
                    this.__parts.push(part);
                }
                else {
                    this.__parts.push(...this.processor.handleAttributeExpressions(node, part.name, part.strings, this.options));
                }
                partIndex++;
            }
            if (isCEPolyfill) {
                document.adoptNode(fragment);
                customElements.upgrade(fragment);
            }
            return fragment;
        }
    }

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    const commentMarker = ` ${marker} `;
    /**
     * The return type of `html`, which holds a Template and the values from
     * interpolated expressions.
     */
    class TemplateResult {
        constructor(strings, values, type, processor) {
            this.strings = strings;
            this.values = values;
            this.type = type;
            this.processor = processor;
        }
        /**
         * Returns a string of HTML used to create a `<template>` element.
         */
        getHTML() {
            const l = this.strings.length - 1;
            let html = '';
            let isCommentBinding = false;
            for (let i = 0; i < l; i++) {
                const s = this.strings[i];
                // For each binding we want to determine the kind of marker to insert
                // into the template source before it's parsed by the browser's HTML
                // parser. The marker type is based on whether the expression is in an
                // attribute, text, or comment position.
                //   * For node-position bindings we insert a comment with the marker
                //     sentinel as its text content, like <!--{{lit-guid}}-->.
                //   * For attribute bindings we insert just the marker sentinel for the
                //     first binding, so that we support unquoted attribute bindings.
                //     Subsequent bindings can use a comment marker because multi-binding
                //     attributes must be quoted.
                //   * For comment bindings we insert just the marker sentinel so we don't
                //     close the comment.
                //
                // The following code scans the template source, but is *not* an HTML
                // parser. We don't need to track the tree structure of the HTML, only
                // whether a binding is inside a comment, and if not, if it appears to be
                // the first binding in an attribute.
                const commentOpen = s.lastIndexOf('<!--');
                // We're in comment position if we have a comment open with no following
                // comment close. Because <-- can appear in an attribute value there can
                // be false positives.
                isCommentBinding = (commentOpen > -1 || isCommentBinding) &&
                    s.indexOf('-->', commentOpen + 1) === -1;
                // Check to see if we have an attribute-like sequence preceding the
                // expression. This can match "name=value" like structures in text,
                // comments, and attribute values, so there can be false-positives.
                const attributeMatch = lastAttributeNameRegex.exec(s);
                if (attributeMatch === null) {
                    // We're only in this branch if we don't have a attribute-like
                    // preceding sequence. For comments, this guards against unusual
                    // attribute values like <div foo="<!--${'bar'}">. Cases like
                    // <!-- foo=${'bar'}--> are handled correctly in the attribute branch
                    // below.
                    html += s + (isCommentBinding ? commentMarker : nodeMarker);
                }
                else {
                    // For attributes we use just a marker sentinel, and also append a
                    // $lit$ suffix to the name to opt-out of attribute-specific parsing
                    // that IE and Edge do for style and certain SVG attributes.
                    html += s.substr(0, attributeMatch.index) + attributeMatch[1] +
                        attributeMatch[2] + boundAttributeSuffix + attributeMatch[3] +
                        marker;
                }
            }
            html += this.strings[l];
            return html;
        }
        getTemplateElement() {
            const template = document.createElement('template');
            template.innerHTML = this.getHTML();
            return template;
        }
    }

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    const isPrimitive = (value) => {
        return (value === null ||
            !(typeof value === 'object' || typeof value === 'function'));
    };
    const isIterable = (value) => {
        return Array.isArray(value) ||
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            !!(value && value[Symbol.iterator]);
    };
    /**
     * Writes attribute values to the DOM for a group of AttributeParts bound to a
     * single attribute. The value is only set once even if there are multiple parts
     * for an attribute.
     */
    class AttributeCommitter {
        constructor(element, name, strings) {
            this.dirty = true;
            this.element = element;
            this.name = name;
            this.strings = strings;
            this.parts = [];
            for (let i = 0; i < strings.length - 1; i++) {
                this.parts[i] = this._createPart();
            }
        }
        /**
         * Creates a single part. Override this to create a differnt type of part.
         */
        _createPart() {
            return new AttributePart(this);
        }
        _getValue() {
            const strings = this.strings;
            const l = strings.length - 1;
            let text = '';
            for (let i = 0; i < l; i++) {
                text += strings[i];
                const part = this.parts[i];
                if (part !== undefined) {
                    const v = part.value;
                    if (isPrimitive(v) || !isIterable(v)) {
                        text += typeof v === 'string' ? v : String(v);
                    }
                    else {
                        for (const t of v) {
                            text += typeof t === 'string' ? t : String(t);
                        }
                    }
                }
            }
            text += strings[l];
            return text;
        }
        commit() {
            if (this.dirty) {
                this.dirty = false;
                this.element.setAttribute(this.name, this._getValue());
            }
        }
    }
    /**
     * A Part that controls all or part of an attribute value.
     */
    class AttributePart {
        constructor(committer) {
            this.value = undefined;
            this.committer = committer;
        }
        setValue(value) {
            if (value !== noChange && (!isPrimitive(value) || value !== this.value)) {
                this.value = value;
                // If the value is a not a directive, dirty the committer so that it'll
                // call setAttribute. If the value is a directive, it'll dirty the
                // committer if it calls setValue().
                if (!isDirective(value)) {
                    this.committer.dirty = true;
                }
            }
        }
        commit() {
            while (isDirective(this.value)) {
                const directive = this.value;
                this.value = noChange;
                directive(this);
            }
            if (this.value === noChange) {
                return;
            }
            this.committer.commit();
        }
    }
    /**
     * A Part that controls a location within a Node tree. Like a Range, NodePart
     * has start and end locations and can set and update the Nodes between those
     * locations.
     *
     * NodeParts support several value types: primitives, Nodes, TemplateResults,
     * as well as arrays and iterables of those types.
     */
    class NodePart {
        constructor(options) {
            this.value = undefined;
            this.__pendingValue = undefined;
            this.options = options;
        }
        /**
         * Appends this part into a container.
         *
         * This part must be empty, as its contents are not automatically moved.
         */
        appendInto(container) {
            this.startNode = container.appendChild(createMarker());
            this.endNode = container.appendChild(createMarker());
        }
        /**
         * Inserts this part after the `ref` node (between `ref` and `ref`'s next
         * sibling). Both `ref` and its next sibling must be static, unchanging nodes
         * such as those that appear in a literal section of a template.
         *
         * This part must be empty, as its contents are not automatically moved.
         */
        insertAfterNode(ref) {
            this.startNode = ref;
            this.endNode = ref.nextSibling;
        }
        /**
         * Appends this part into a parent part.
         *
         * This part must be empty, as its contents are not automatically moved.
         */
        appendIntoPart(part) {
            part.__insert(this.startNode = createMarker());
            part.__insert(this.endNode = createMarker());
        }
        /**
         * Inserts this part after the `ref` part.
         *
         * This part must be empty, as its contents are not automatically moved.
         */
        insertAfterPart(ref) {
            ref.__insert(this.startNode = createMarker());
            this.endNode = ref.endNode;
            ref.endNode = this.startNode;
        }
        setValue(value) {
            this.__pendingValue = value;
        }
        commit() {
            if (this.startNode.parentNode === null) {
                return;
            }
            while (isDirective(this.__pendingValue)) {
                const directive = this.__pendingValue;
                this.__pendingValue = noChange;
                directive(this);
            }
            const value = this.__pendingValue;
            if (value === noChange) {
                return;
            }
            if (isPrimitive(value)) {
                if (value !== this.value) {
                    this.__commitText(value);
                }
            }
            else if (value instanceof TemplateResult) {
                this.__commitTemplateResult(value);
            }
            else if (value instanceof Node) {
                this.__commitNode(value);
            }
            else if (isIterable(value)) {
                this.__commitIterable(value);
            }
            else if (value === nothing) {
                this.value = nothing;
                this.clear();
            }
            else {
                // Fallback, will render the string representation
                this.__commitText(value);
            }
        }
        __insert(node) {
            this.endNode.parentNode.insertBefore(node, this.endNode);
        }
        __commitNode(value) {
            if (this.value === value) {
                return;
            }
            this.clear();
            this.__insert(value);
            this.value = value;
        }
        __commitText(value) {
            const node = this.startNode.nextSibling;
            value = value == null ? '' : value;
            // If `value` isn't already a string, we explicitly convert it here in case
            // it can't be implicitly converted - i.e. it's a symbol.
            const valueAsString = typeof value === 'string' ? value : String(value);
            if (node === this.endNode.previousSibling &&
                node.nodeType === 3 /* Node.TEXT_NODE */) {
                // If we only have a single text node between the markers, we can just
                // set its value, rather than replacing it.
                // TODO(justinfagnani): Can we just check if this.value is primitive?
                node.data = valueAsString;
            }
            else {
                this.__commitNode(document.createTextNode(valueAsString));
            }
            this.value = value;
        }
        __commitTemplateResult(value) {
            const template = this.options.templateFactory(value);
            if (this.value instanceof TemplateInstance &&
                this.value.template === template) {
                this.value.update(value.values);
            }
            else {
                // Make sure we propagate the template processor from the TemplateResult
                // so that we use its syntax extension, etc. The template factory comes
                // from the render function options so that it can control template
                // caching and preprocessing.
                const instance = new TemplateInstance(template, value.processor, this.options);
                const fragment = instance._clone();
                instance.update(value.values);
                this.__commitNode(fragment);
                this.value = instance;
            }
        }
        __commitIterable(value) {
            // For an Iterable, we create a new InstancePart per item, then set its
            // value to the item. This is a little bit of overhead for every item in
            // an Iterable, but it lets us recurse easily and efficiently update Arrays
            // of TemplateResults that will be commonly returned from expressions like:
            // array.map((i) => html`${i}`), by reusing existing TemplateInstances.
            // If _value is an array, then the previous render was of an
            // iterable and _value will contain the NodeParts from the previous
            // render. If _value is not an array, clear this part and make a new
            // array for NodeParts.
            if (!Array.isArray(this.value)) {
                this.value = [];
                this.clear();
            }
            // Lets us keep track of how many items we stamped so we can clear leftover
            // items from a previous render
            const itemParts = this.value;
            let partIndex = 0;
            let itemPart;
            for (const item of value) {
                // Try to reuse an existing part
                itemPart = itemParts[partIndex];
                // If no existing part, create a new one
                if (itemPart === undefined) {
                    itemPart = new NodePart(this.options);
                    itemParts.push(itemPart);
                    if (partIndex === 0) {
                        itemPart.appendIntoPart(this);
                    }
                    else {
                        itemPart.insertAfterPart(itemParts[partIndex - 1]);
                    }
                }
                itemPart.setValue(item);
                itemPart.commit();
                partIndex++;
            }
            if (partIndex < itemParts.length) {
                // Truncate the parts array so _value reflects the current state
                itemParts.length = partIndex;
                this.clear(itemPart && itemPart.endNode);
            }
        }
        clear(startNode = this.startNode) {
            removeNodes(this.startNode.parentNode, startNode.nextSibling, this.endNode);
        }
    }
    /**
     * Implements a boolean attribute, roughly as defined in the HTML
     * specification.
     *
     * If the value is truthy, then the attribute is present with a value of
     * ''. If the value is falsey, the attribute is removed.
     */
    class BooleanAttributePart {
        constructor(element, name, strings) {
            this.value = undefined;
            this.__pendingValue = undefined;
            if (strings.length !== 2 || strings[0] !== '' || strings[1] !== '') {
                throw new Error('Boolean attributes can only contain a single expression');
            }
            this.element = element;
            this.name = name;
            this.strings = strings;
        }
        setValue(value) {
            this.__pendingValue = value;
        }
        commit() {
            while (isDirective(this.__pendingValue)) {
                const directive = this.__pendingValue;
                this.__pendingValue = noChange;
                directive(this);
            }
            if (this.__pendingValue === noChange) {
                return;
            }
            const value = !!this.__pendingValue;
            if (this.value !== value) {
                if (value) {
                    this.element.setAttribute(this.name, '');
                }
                else {
                    this.element.removeAttribute(this.name);
                }
                this.value = value;
            }
            this.__pendingValue = noChange;
        }
    }
    /**
     * Sets attribute values for PropertyParts, so that the value is only set once
     * even if there are multiple parts for a property.
     *
     * If an expression controls the whole property value, then the value is simply
     * assigned to the property under control. If there are string literals or
     * multiple expressions, then the strings are expressions are interpolated into
     * a string first.
     */
    class PropertyCommitter extends AttributeCommitter {
        constructor(element, name, strings) {
            super(element, name, strings);
            this.single =
                (strings.length === 2 && strings[0] === '' && strings[1] === '');
        }
        _createPart() {
            return new PropertyPart(this);
        }
        _getValue() {
            if (this.single) {
                return this.parts[0].value;
            }
            return super._getValue();
        }
        commit() {
            if (this.dirty) {
                this.dirty = false;
                // eslint-disable-next-line @typescript-eslint/no-explicit-any
                this.element[this.name] = this._getValue();
            }
        }
    }
    class PropertyPart extends AttributePart {
    }
    // Detect event listener options support. If the `capture` property is read
    // from the options object, then options are supported. If not, then the third
    // argument to add/removeEventListener is interpreted as the boolean capture
    // value so we should only pass the `capture` property.
    let eventOptionsSupported = false;
    // Wrap into an IIFE because MS Edge <= v41 does not support having try/catch
    // blocks right into the body of a module
    (() => {
        try {
            const options = {
                get capture() {
                    eventOptionsSupported = true;
                    return false;
                }
            };
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            window.addEventListener('test', options, options);
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            window.removeEventListener('test', options, options);
        }
        catch (_e) {
            // event options not supported
        }
    })();
    class EventPart {
        constructor(element, eventName, eventContext) {
            this.value = undefined;
            this.__pendingValue = undefined;
            this.element = element;
            this.eventName = eventName;
            this.eventContext = eventContext;
            this.__boundHandleEvent = (e) => this.handleEvent(e);
        }
        setValue(value) {
            this.__pendingValue = value;
        }
        commit() {
            while (isDirective(this.__pendingValue)) {
                const directive = this.__pendingValue;
                this.__pendingValue = noChange;
                directive(this);
            }
            if (this.__pendingValue === noChange) {
                return;
            }
            const newListener = this.__pendingValue;
            const oldListener = this.value;
            const shouldRemoveListener = newListener == null ||
                oldListener != null &&
                    (newListener.capture !== oldListener.capture ||
                        newListener.once !== oldListener.once ||
                        newListener.passive !== oldListener.passive);
            const shouldAddListener = newListener != null && (oldListener == null || shouldRemoveListener);
            if (shouldRemoveListener) {
                this.element.removeEventListener(this.eventName, this.__boundHandleEvent, this.__options);
            }
            if (shouldAddListener) {
                this.__options = getOptions(newListener);
                this.element.addEventListener(this.eventName, this.__boundHandleEvent, this.__options);
            }
            this.value = newListener;
            this.__pendingValue = noChange;
        }
        handleEvent(event) {
            if (typeof this.value === 'function') {
                this.value.call(this.eventContext || this.element, event);
            }
            else {
                this.value.handleEvent(event);
            }
        }
    }
    // We copy options because of the inconsistent behavior of browsers when reading
    // the third argument of add/removeEventListener. IE11 doesn't support options
    // at all. Chrome 41 only reads `capture` if the argument is an object.
    const getOptions = (o) => o &&
        (eventOptionsSupported ?
            { capture: o.capture, passive: o.passive, once: o.once } :
            o.capture);

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    /**
     * The default TemplateFactory which caches Templates keyed on
     * result.type and result.strings.
     */
    function templateFactory(result) {
        let templateCache = templateCaches.get(result.type);
        if (templateCache === undefined) {
            templateCache = {
                stringsArray: new WeakMap(),
                keyString: new Map()
            };
            templateCaches.set(result.type, templateCache);
        }
        let template = templateCache.stringsArray.get(result.strings);
        if (template !== undefined) {
            return template;
        }
        // If the TemplateStringsArray is new, generate a key from the strings
        // This key is shared between all templates with identical content
        const key = result.strings.join(marker);
        // Check if we already have a Template for this key
        template = templateCache.keyString.get(key);
        if (template === undefined) {
            // If we have not seen this key before, create a new Template
            template = new Template(result, result.getTemplateElement());
            // Cache the Template for this key
            templateCache.keyString.set(key, template);
        }
        // Cache all future queries for this TemplateStringsArray
        templateCache.stringsArray.set(result.strings, template);
        return template;
    }
    const templateCaches = new Map();

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    const parts = new WeakMap();
    /**
     * Renders a template result or other value to a container.
     *
     * To update a container with new values, reevaluate the template literal and
     * call `render` with the new result.
     *
     * @param result Any value renderable by NodePart - typically a TemplateResult
     *     created by evaluating a template tag like `html` or `svg`.
     * @param container A DOM parent to render to. The entire contents are either
     *     replaced, or efficiently updated if the same result type was previous
     *     rendered there.
     * @param options RenderOptions for the entire render tree rendered to this
     *     container. Render options must *not* change between renders to the same
     *     container, as those changes will not effect previously rendered DOM.
     */
    const render = (result, container, options) => {
        let part = parts.get(container);
        if (part === undefined) {
            removeNodes(container, container.firstChild);
            parts.set(container, part = new NodePart(Object.assign({ templateFactory }, options)));
            part.appendInto(container);
        }
        part.setValue(result);
        part.commit();
    };

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    /**
     * Creates Parts when a template is instantiated.
     */
    class DefaultTemplateProcessor {
        /**
         * Create parts for an attribute-position binding, given the event, attribute
         * name, and string literals.
         *
         * @param element The element containing the binding
         * @param name  The attribute name
         * @param strings The string literals. There are always at least two strings,
         *   event for fully-controlled bindings with a single expression.
         */
        handleAttributeExpressions(element, name, strings, options) {
            const prefix = name[0];
            if (prefix === '.') {
                const committer = new PropertyCommitter(element, name.slice(1), strings);
                return committer.parts;
            }
            if (prefix === '@') {
                return [new EventPart(element, name.slice(1), options.eventContext)];
            }
            if (prefix === '?') {
                return [new BooleanAttributePart(element, name.slice(1), strings)];
            }
            const committer = new AttributeCommitter(element, name, strings);
            return committer.parts;
        }
        /**
         * Create parts for a text-position binding.
         * @param templateFactory
         */
        handleTextExpression(options) {
            return new NodePart(options);
        }
    }
    const defaultTemplateProcessor = new DefaultTemplateProcessor();

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    // IMPORTANT: do not change the property name or the assignment expression.
    // This line will be used in regexes to search for lit-html usage.
    // TODO(justinfagnani): inject version number at build time
    if (typeof window !== 'undefined') {
        (window['litHtmlVersions'] || (window['litHtmlVersions'] = [])).push('1.2.1');
    }
    /**
     * Interprets a template literal as an HTML template that can efficiently
     * render to and update a container.
     */
    const html = (strings, ...values) => new TemplateResult(strings, values, 'html', defaultTemplateProcessor);

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    // Get a key to lookup in `templateCaches`.
    const getTemplateCacheKey = (type, scopeName) => `${type}--${scopeName}`;
    let compatibleShadyCSSVersion = true;
    if (typeof window.ShadyCSS === 'undefined') {
        compatibleShadyCSSVersion = false;
    }
    else if (typeof window.ShadyCSS.prepareTemplateDom === 'undefined') {
        console.warn(`Incompatible ShadyCSS version detected. ` +
            `Please update to at least @webcomponents/webcomponentsjs@2.0.2 and ` +
            `@webcomponents/shadycss@1.3.1.`);
        compatibleShadyCSSVersion = false;
    }
    /**
     * Template factory which scopes template DOM using ShadyCSS.
     * @param scopeName {string}
     */
    const shadyTemplateFactory = (scopeName) => (result) => {
        const cacheKey = getTemplateCacheKey(result.type, scopeName);
        let templateCache = templateCaches.get(cacheKey);
        if (templateCache === undefined) {
            templateCache = {
                stringsArray: new WeakMap(),
                keyString: new Map()
            };
            templateCaches.set(cacheKey, templateCache);
        }
        let template = templateCache.stringsArray.get(result.strings);
        if (template !== undefined) {
            return template;
        }
        const key = result.strings.join(marker);
        template = templateCache.keyString.get(key);
        if (template === undefined) {
            const element = result.getTemplateElement();
            if (compatibleShadyCSSVersion) {
                window.ShadyCSS.prepareTemplateDom(element, scopeName);
            }
            template = new Template(result, element);
            templateCache.keyString.set(key, template);
        }
        templateCache.stringsArray.set(result.strings, template);
        return template;
    };
    const TEMPLATE_TYPES = ['html', 'svg'];
    /**
     * Removes all style elements from Templates for the given scopeName.
     */
    const removeStylesFromLitTemplates = (scopeName) => {
        TEMPLATE_TYPES.forEach((type) => {
            const templates = templateCaches.get(getTemplateCacheKey(type, scopeName));
            if (templates !== undefined) {
                templates.keyString.forEach((template) => {
                    const { element: { content } } = template;
                    // IE 11 doesn't support the iterable param Set constructor
                    const styles = new Set();
                    Array.from(content.querySelectorAll('style')).forEach((s) => {
                        styles.add(s);
                    });
                    removeNodesFromTemplate(template, styles);
                });
            }
        });
    };
    const shadyRenderSet = new Set();
    /**
     * For the given scope name, ensures that ShadyCSS style scoping is performed.
     * This is done just once per scope name so the fragment and template cannot
     * be modified.
     * (1) extracts styles from the rendered fragment and hands them to ShadyCSS
     * to be scoped and appended to the document
     * (2) removes style elements from all lit-html Templates for this scope name.
     *
     * Note, <style> elements can only be placed into templates for the
     * initial rendering of the scope. If <style> elements are included in templates
     * dynamically rendered to the scope (after the first scope render), they will
     * not be scoped and the <style> will be left in the template and rendered
     * output.
     */
    const prepareTemplateStyles = (scopeName, renderedDOM, template) => {
        shadyRenderSet.add(scopeName);
        // If `renderedDOM` is stamped from a Template, then we need to edit that
        // Template's underlying template element. Otherwise, we create one here
        // to give to ShadyCSS, which still requires one while scoping.
        const templateElement = !!template ? template.element : document.createElement('template');
        // Move styles out of rendered DOM and store.
        const styles = renderedDOM.querySelectorAll('style');
        const { length } = styles;
        // If there are no styles, skip unnecessary work
        if (length === 0) {
            // Ensure prepareTemplateStyles is called to support adding
            // styles via `prepareAdoptedCssText` since that requires that
            // `prepareTemplateStyles` is called.
            //
            // ShadyCSS will only update styles containing @apply in the template
            // given to `prepareTemplateStyles`. If no lit Template was given,
            // ShadyCSS will not be able to update uses of @apply in any relevant
            // template. However, this is not a problem because we only create the
            // template for the purpose of supporting `prepareAdoptedCssText`,
            // which doesn't support @apply at all.
            window.ShadyCSS.prepareTemplateStyles(templateElement, scopeName);
            return;
        }
        const condensedStyle = document.createElement('style');
        // Collect styles into a single style. This helps us make sure ShadyCSS
        // manipulations will not prevent us from being able to fix up template
        // part indices.
        // NOTE: collecting styles is inefficient for browsers but ShadyCSS
        // currently does this anyway. When it does not, this should be changed.
        for (let i = 0; i < length; i++) {
            const style = styles[i];
            style.parentNode.removeChild(style);
            condensedStyle.textContent += style.textContent;
        }
        // Remove styles from nested templates in this scope.
        removeStylesFromLitTemplates(scopeName);
        // And then put the condensed style into the "root" template passed in as
        // `template`.
        const content = templateElement.content;
        if (!!template) {
            insertNodeIntoTemplate(template, condensedStyle, content.firstChild);
        }
        else {
            content.insertBefore(condensedStyle, content.firstChild);
        }
        // Note, it's important that ShadyCSS gets the template that `lit-html`
        // will actually render so that it can update the style inside when
        // needed (e.g. @apply native Shadow DOM case).
        window.ShadyCSS.prepareTemplateStyles(templateElement, scopeName);
        const style = content.querySelector('style');
        if (window.ShadyCSS.nativeShadow && style !== null) {
            // When in native Shadow DOM, ensure the style created by ShadyCSS is
            // included in initially rendered output (`renderedDOM`).
            renderedDOM.insertBefore(style.cloneNode(true), renderedDOM.firstChild);
        }
        else if (!!template) {
            // When no style is left in the template, parts will be broken as a
            // result. To fix this, we put back the style node ShadyCSS removed
            // and then tell lit to remove that node from the template.
            // There can be no style in the template in 2 cases (1) when Shady DOM
            // is in use, ShadyCSS removes all styles, (2) when native Shadow DOM
            // is in use ShadyCSS removes the style if it contains no content.
            // NOTE, ShadyCSS creates its own style so we can safely add/remove
            // `condensedStyle` here.
            content.insertBefore(condensedStyle, content.firstChild);
            const removes = new Set();
            removes.add(condensedStyle);
            removeNodesFromTemplate(template, removes);
        }
    };
    /**
     * Extension to the standard `render` method which supports rendering
     * to ShadowRoots when the ShadyDOM (https://github.com/webcomponents/shadydom)
     * and ShadyCSS (https://github.com/webcomponents/shadycss) polyfills are used
     * or when the webcomponentsjs
     * (https://github.com/webcomponents/webcomponentsjs) polyfill is used.
     *
     * Adds a `scopeName` option which is used to scope element DOM and stylesheets
     * when native ShadowDOM is unavailable. The `scopeName` will be added to
     * the class attribute of all rendered DOM. In addition, any style elements will
     * be automatically re-written with this `scopeName` selector and moved out
     * of the rendered DOM and into the document `<head>`.
     *
     * It is common to use this render method in conjunction with a custom element
     * which renders a shadowRoot. When this is done, typically the element's
     * `localName` should be used as the `scopeName`.
     *
     * In addition to DOM scoping, ShadyCSS also supports a basic shim for css
     * custom properties (needed only on older browsers like IE11) and a shim for
     * a deprecated feature called `@apply` that supports applying a set of css
     * custom properties to a given location.
     *
     * Usage considerations:
     *
     * * Part values in `<style>` elements are only applied the first time a given
     * `scopeName` renders. Subsequent changes to parts in style elements will have
     * no effect. Because of this, parts in style elements should only be used for
     * values that will never change, for example parts that set scope-wide theme
     * values or parts which render shared style elements.
     *
     * * Note, due to a limitation of the ShadyDOM polyfill, rendering in a
     * custom element's `constructor` is not supported. Instead rendering should
     * either done asynchronously, for example at microtask timing (for example
     * `Promise.resolve()`), or be deferred until the first time the element's
     * `connectedCallback` runs.
     *
     * Usage considerations when using shimmed custom properties or `@apply`:
     *
     * * Whenever any dynamic changes are made which affect
     * css custom properties, `ShadyCSS.styleElement(element)` must be called
     * to update the element. There are two cases when this is needed:
     * (1) the element is connected to a new parent, (2) a class is added to the
     * element that causes it to match different custom properties.
     * To address the first case when rendering a custom element, `styleElement`
     * should be called in the element's `connectedCallback`.
     *
     * * Shimmed custom properties may only be defined either for an entire
     * shadowRoot (for example, in a `:host` rule) or via a rule that directly
     * matches an element with a shadowRoot. In other words, instead of flowing from
     * parent to child as do native css custom properties, shimmed custom properties
     * flow only from shadowRoots to nested shadowRoots.
     *
     * * When using `@apply` mixing css shorthand property names with
     * non-shorthand names (for example `border` and `border-width`) is not
     * supported.
     */
    const render$1 = (result, container, options) => {
        if (!options || typeof options !== 'object' || !options.scopeName) {
            throw new Error('The `scopeName` option is required.');
        }
        const scopeName = options.scopeName;
        const hasRendered = parts.has(container);
        const needsScoping = compatibleShadyCSSVersion &&
            container.nodeType === 11 /* Node.DOCUMENT_FRAGMENT_NODE */ &&
            !!container.host;
        // Handle first render to a scope specially...
        const firstScopeRender = needsScoping && !shadyRenderSet.has(scopeName);
        // On first scope render, render into a fragment; this cannot be a single
        // fragment that is reused since nested renders can occur synchronously.
        const renderContainer = firstScopeRender ? document.createDocumentFragment() : container;
        render(result, renderContainer, Object.assign({ templateFactory: shadyTemplateFactory(scopeName) }, options));
        // When performing first scope render,
        // (1) We've rendered into a fragment so that there's a chance to
        // `prepareTemplateStyles` before sub-elements hit the DOM
        // (which might cause them to render based on a common pattern of
        // rendering in a custom element's `connectedCallback`);
        // (2) Scope the template with ShadyCSS one time only for this scope.
        // (3) Render the fragment into the container and make sure the
        // container knows its `part` is the one we just rendered. This ensures
        // DOM will be re-used on subsequent renders.
        if (firstScopeRender) {
            const part = parts.get(renderContainer);
            parts.delete(renderContainer);
            // ShadyCSS might have style sheets (e.g. from `prepareAdoptedCssText`)
            // that should apply to `renderContainer` even if the rendered value is
            // not a TemplateInstance. However, it will only insert scoped styles
            // into the document if `prepareTemplateStyles` has already been called
            // for the given scope name.
            const template = part.value instanceof TemplateInstance ?
                part.value.template :
                undefined;
            prepareTemplateStyles(scopeName, renderContainer, template);
            removeNodes(container, container.firstChild);
            container.appendChild(renderContainer);
            parts.set(container, part);
        }
        // After elements have hit the DOM, update styling if this is the
        // initial render to this container.
        // This is needed whenever dynamic changes are made so it would be
        // safest to do every render; however, this would regress performance
        // so we leave it up to the user to call `ShadyCSS.styleElement`
        // for dynamic changes.
        if (!hasRendered && needsScoping) {
            window.ShadyCSS.styleElement(container.host);
        }
    };

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    var _a;
    /**
     * When using Closure Compiler, JSCompiler_renameProperty(property, object) is
     * replaced at compile time by the munged name for object[property]. We cannot
     * alias this function, so we have to use a small shim that has the same
     * behavior when not compiling.
     */
    window.JSCompiler_renameProperty =
        (prop, _obj) => prop;
    const defaultConverter = {
        toAttribute(value, type) {
            switch (type) {
                case Boolean:
                    return value ? '' : null;
                case Object:
                case Array:
                    // if the value is `null` or `undefined` pass this through
                    // to allow removing/no change behavior.
                    return value == null ? value : JSON.stringify(value);
            }
            return value;
        },
        fromAttribute(value, type) {
            switch (type) {
                case Boolean:
                    return value !== null;
                case Number:
                    return value === null ? null : Number(value);
                case Object:
                case Array:
                    return JSON.parse(value);
            }
            return value;
        }
    };
    /**
     * Change function that returns true if `value` is different from `oldValue`.
     * This method is used as the default for a property's `hasChanged` function.
     */
    const notEqual = (value, old) => {
        // This ensures (old==NaN, value==NaN) always returns false
        return old !== value && (old === old || value === value);
    };
    const defaultPropertyDeclaration = {
        attribute: true,
        type: String,
        converter: defaultConverter,
        reflect: false,
        hasChanged: notEqual
    };
    const STATE_HAS_UPDATED = 1;
    const STATE_UPDATE_REQUESTED = 1 << 2;
    const STATE_IS_REFLECTING_TO_ATTRIBUTE = 1 << 3;
    const STATE_IS_REFLECTING_TO_PROPERTY = 1 << 4;
    /**
     * The Closure JS Compiler doesn't currently have good support for static
     * property semantics where "this" is dynamic (e.g.
     * https://github.com/google/closure-compiler/issues/3177 and others) so we use
     * this hack to bypass any rewriting by the compiler.
     */
    const finalized = 'finalized';
    /**
     * Base element class which manages element properties and attributes. When
     * properties change, the `update` method is asynchronously called. This method
     * should be supplied by subclassers to render updates as desired.
     */
    class UpdatingElement extends HTMLElement {
        constructor() {
            super();
            this._updateState = 0;
            this._instanceProperties = undefined;
            // Initialize to an unresolved Promise so we can make sure the element has
            // connected before first update.
            this._updatePromise = new Promise((res) => this._enableUpdatingResolver = res);
            /**
             * Map with keys for any properties that have changed since the last
             * update cycle with previous values.
             */
            this._changedProperties = new Map();
            /**
             * Map with keys of properties that should be reflected when updated.
             */
            this._reflectingProperties = undefined;
            this.initialize();
        }
        /**
         * Returns a list of attributes corresponding to the registered properties.
         * @nocollapse
         */
        static get observedAttributes() {
            // note: piggy backing on this to ensure we're finalized.
            this.finalize();
            const attributes = [];
            // Use forEach so this works even if for/of loops are compiled to for loops
            // expecting arrays
            this._classProperties.forEach((v, p) => {
                const attr = this._attributeNameForProperty(p, v);
                if (attr !== undefined) {
                    this._attributeToPropertyMap.set(attr, p);
                    attributes.push(attr);
                }
            });
            return attributes;
        }
        /**
         * Ensures the private `_classProperties` property metadata is created.
         * In addition to `finalize` this is also called in `createProperty` to
         * ensure the `@property` decorator can add property metadata.
         */
        /** @nocollapse */
        static _ensureClassProperties() {
            // ensure private storage for property declarations.
            if (!this.hasOwnProperty(JSCompiler_renameProperty('_classProperties', this))) {
                this._classProperties = new Map();
                // NOTE: Workaround IE11 not supporting Map constructor argument.
                const superProperties = Object.getPrototypeOf(this)._classProperties;
                if (superProperties !== undefined) {
                    superProperties.forEach((v, k) => this._classProperties.set(k, v));
                }
            }
        }
        /**
         * Creates a property accessor on the element prototype if one does not exist
         * and stores a PropertyDeclaration for the property with the given options.
         * The property setter calls the property's `hasChanged` property option
         * or uses a strict identity check to determine whether or not to request
         * an update.
         *
         * This method may be overridden to customize properties; however,
         * when doing so, it's important to call `super.createProperty` to ensure
         * the property is setup correctly. This method calls
         * `getPropertyDescriptor` internally to get a descriptor to install.
         * To customize what properties do when they are get or set, override
         * `getPropertyDescriptor`. To customize the options for a property,
         * implement `createProperty` like this:
         *
         * static createProperty(name, options) {
         *   options = Object.assign(options, {myOption: true});
         *   super.createProperty(name, options);
         * }
         *
         * @nocollapse
         */
        static createProperty(name, options = defaultPropertyDeclaration) {
            // Note, since this can be called by the `@property` decorator which
            // is called before `finalize`, we ensure storage exists for property
            // metadata.
            this._ensureClassProperties();
            this._classProperties.set(name, options);
            // Do not generate an accessor if the prototype already has one, since
            // it would be lost otherwise and that would never be the user's intention;
            // Instead, we expect users to call `requestUpdate` themselves from
            // user-defined accessors. Note that if the super has an accessor we will
            // still overwrite it
            if (options.noAccessor || this.prototype.hasOwnProperty(name)) {
                return;
            }
            const key = typeof name === 'symbol' ? Symbol() : `__${name}`;
            const descriptor = this.getPropertyDescriptor(name, key, options);
            if (descriptor !== undefined) {
                Object.defineProperty(this.prototype, name, descriptor);
            }
        }
        /**
         * Returns a property descriptor to be defined on the given named property.
         * If no descriptor is returned, the property will not become an accessor.
         * For example,
         *
         *   class MyElement extends LitElement {
         *     static getPropertyDescriptor(name, key, options) {
         *       const defaultDescriptor =
         *           super.getPropertyDescriptor(name, key, options);
         *       const setter = defaultDescriptor.set;
         *       return {
         *         get: defaultDescriptor.get,
         *         set(value) {
         *           setter.call(this, value);
         *           // custom action.
         *         },
         *         configurable: true,
         *         enumerable: true
         *       }
         *     }
         *   }
         *
         * @nocollapse
         */
        static getPropertyDescriptor(name, key, _options) {
            return {
                // tslint:disable-next-line:no-any no symbol in index
                get() {
                    return this[key];
                },
                set(value) {
                    const oldValue = this[name];
                    this[key] = value;
                    this._requestUpdate(name, oldValue);
                },
                configurable: true,
                enumerable: true
            };
        }
        /**
         * Returns the property options associated with the given property.
         * These options are defined with a PropertyDeclaration via the `properties`
         * object or the `@property` decorator and are registered in
         * `createProperty(...)`.
         *
         * Note, this method should be considered "final" and not overridden. To
         * customize the options for a given property, override `createProperty`.
         *
         * @nocollapse
         * @final
         */
        static getPropertyOptions(name) {
            return this._classProperties && this._classProperties.get(name) ||
                defaultPropertyDeclaration;
        }
        /**
         * Creates property accessors for registered properties and ensures
         * any superclasses are also finalized.
         * @nocollapse
         */
        static finalize() {
            // finalize any superclasses
            const superCtor = Object.getPrototypeOf(this);
            if (!superCtor.hasOwnProperty(finalized)) {
                superCtor.finalize();
            }
            this[finalized] = true;
            this._ensureClassProperties();
            // initialize Map populated in observedAttributes
            this._attributeToPropertyMap = new Map();
            // make any properties
            // Note, only process "own" properties since this element will inherit
            // any properties defined on the superClass, and finalization ensures
            // the entire prototype chain is finalized.
            if (this.hasOwnProperty(JSCompiler_renameProperty('properties', this))) {
                const props = this.properties;
                // support symbols in properties (IE11 does not support this)
                const propKeys = [
                    ...Object.getOwnPropertyNames(props),
                    ...(typeof Object.getOwnPropertySymbols === 'function') ?
                        Object.getOwnPropertySymbols(props) :
                        []
                ];
                // This for/of is ok because propKeys is an array
                for (const p of propKeys) {
                    // note, use of `any` is due to TypeSript lack of support for symbol in
                    // index types
                    // tslint:disable-next-line:no-any no symbol in index
                    this.createProperty(p, props[p]);
                }
            }
        }
        /**
         * Returns the property name for the given attribute `name`.
         * @nocollapse
         */
        static _attributeNameForProperty(name, options) {
            const attribute = options.attribute;
            return attribute === false ?
                undefined :
                (typeof attribute === 'string' ?
                    attribute :
                    (typeof name === 'string' ? name.toLowerCase() : undefined));
        }
        /**
         * Returns true if a property should request an update.
         * Called when a property value is set and uses the `hasChanged`
         * option for the property if present or a strict identity check.
         * @nocollapse
         */
        static _valueHasChanged(value, old, hasChanged = notEqual) {
            return hasChanged(value, old);
        }
        /**
         * Returns the property value for the given attribute value.
         * Called via the `attributeChangedCallback` and uses the property's
         * `converter` or `converter.fromAttribute` property option.
         * @nocollapse
         */
        static _propertyValueFromAttribute(value, options) {
            const type = options.type;
            const converter = options.converter || defaultConverter;
            const fromAttribute = (typeof converter === 'function' ? converter : converter.fromAttribute);
            return fromAttribute ? fromAttribute(value, type) : value;
        }
        /**
         * Returns the attribute value for the given property value. If this
         * returns undefined, the property will *not* be reflected to an attribute.
         * If this returns null, the attribute will be removed, otherwise the
         * attribute will be set to the value.
         * This uses the property's `reflect` and `type.toAttribute` property options.
         * @nocollapse
         */
        static _propertyValueToAttribute(value, options) {
            if (options.reflect === undefined) {
                return;
            }
            const type = options.type;
            const converter = options.converter;
            const toAttribute = converter && converter.toAttribute ||
                defaultConverter.toAttribute;
            return toAttribute(value, type);
        }
        /**
         * Performs element initialization. By default captures any pre-set values for
         * registered properties.
         */
        initialize() {
            this._saveInstanceProperties();
            // ensures first update will be caught by an early access of
            // `updateComplete`
            this._requestUpdate();
        }
        /**
         * Fixes any properties set on the instance before upgrade time.
         * Otherwise these would shadow the accessor and break these properties.
         * The properties are stored in a Map which is played back after the
         * constructor runs. Note, on very old versions of Safari (<=9) or Chrome
         * (<=41), properties created for native platform properties like (`id` or
         * `name`) may not have default values set in the element constructor. On
         * these browsers native properties appear on instances and therefore their
         * default value will overwrite any element default (e.g. if the element sets
         * this.id = 'id' in the constructor, the 'id' will become '' since this is
         * the native platform default).
         */
        _saveInstanceProperties() {
            // Use forEach so this works even if for/of loops are compiled to for loops
            // expecting arrays
            this.constructor
                ._classProperties.forEach((_v, p) => {
                if (this.hasOwnProperty(p)) {
                    const value = this[p];
                    delete this[p];
                    if (!this._instanceProperties) {
                        this._instanceProperties = new Map();
                    }
                    this._instanceProperties.set(p, value);
                }
            });
        }
        /**
         * Applies previously saved instance properties.
         */
        _applyInstanceProperties() {
            // Use forEach so this works even if for/of loops are compiled to for loops
            // expecting arrays
            // tslint:disable-next-line:no-any
            this._instanceProperties.forEach((v, p) => this[p] = v);
            this._instanceProperties = undefined;
        }
        connectedCallback() {
            // Ensure first connection completes an update. Updates cannot complete
            // before connection.
            this.enableUpdating();
        }
        enableUpdating() {
            if (this._enableUpdatingResolver !== undefined) {
                this._enableUpdatingResolver();
                this._enableUpdatingResolver = undefined;
            }
        }
        /**
         * Allows for `super.disconnectedCallback()` in extensions while
         * reserving the possibility of making non-breaking feature additions
         * when disconnecting at some point in the future.
         */
        disconnectedCallback() {
        }
        /**
         * Synchronizes property values when attributes change.
         */
        attributeChangedCallback(name, old, value) {
            if (old !== value) {
                this._attributeToProperty(name, value);
            }
        }
        _propertyToAttribute(name, value, options = defaultPropertyDeclaration) {
            const ctor = this.constructor;
            const attr = ctor._attributeNameForProperty(name, options);
            if (attr !== undefined) {
                const attrValue = ctor._propertyValueToAttribute(value, options);
                // an undefined value does not change the attribute.
                if (attrValue === undefined) {
                    return;
                }
                // Track if the property is being reflected to avoid
                // setting the property again via `attributeChangedCallback`. Note:
                // 1. this takes advantage of the fact that the callback is synchronous.
                // 2. will behave incorrectly if multiple attributes are in the reaction
                // stack at time of calling. However, since we process attributes
                // in `update` this should not be possible (or an extreme corner case
                // that we'd like to discover).
                // mark state reflecting
                this._updateState = this._updateState | STATE_IS_REFLECTING_TO_ATTRIBUTE;
                if (attrValue == null) {
                    this.removeAttribute(attr);
                }
                else {
                    this.setAttribute(attr, attrValue);
                }
                // mark state not reflecting
                this._updateState = this._updateState & ~STATE_IS_REFLECTING_TO_ATTRIBUTE;
            }
        }
        _attributeToProperty(name, value) {
            // Use tracking info to avoid deserializing attribute value if it was
            // just set from a property setter.
            if (this._updateState & STATE_IS_REFLECTING_TO_ATTRIBUTE) {
                return;
            }
            const ctor = this.constructor;
            // Note, hint this as an `AttributeMap` so closure clearly understands
            // the type; it has issues with tracking types through statics
            // tslint:disable-next-line:no-unnecessary-type-assertion
            const propName = ctor._attributeToPropertyMap.get(name);
            if (propName !== undefined) {
                const options = ctor.getPropertyOptions(propName);
                // mark state reflecting
                this._updateState = this._updateState | STATE_IS_REFLECTING_TO_PROPERTY;
                this[propName] =
                    // tslint:disable-next-line:no-any
                    ctor._propertyValueFromAttribute(value, options);
                // mark state not reflecting
                this._updateState = this._updateState & ~STATE_IS_REFLECTING_TO_PROPERTY;
            }
        }
        /**
         * This private version of `requestUpdate` does not access or return the
         * `updateComplete` promise. This promise can be overridden and is therefore
         * not free to access.
         */
        _requestUpdate(name, oldValue) {
            let shouldRequestUpdate = true;
            // If we have a property key, perform property update steps.
            if (name !== undefined) {
                const ctor = this.constructor;
                const options = ctor.getPropertyOptions(name);
                if (ctor._valueHasChanged(this[name], oldValue, options.hasChanged)) {
                    if (!this._changedProperties.has(name)) {
                        this._changedProperties.set(name, oldValue);
                    }
                    // Add to reflecting properties set.
                    // Note, it's important that every change has a chance to add the
                    // property to `_reflectingProperties`. This ensures setting
                    // attribute + property reflects correctly.
                    if (options.reflect === true &&
                        !(this._updateState & STATE_IS_REFLECTING_TO_PROPERTY)) {
                        if (this._reflectingProperties === undefined) {
                            this._reflectingProperties = new Map();
                        }
                        this._reflectingProperties.set(name, options);
                    }
                }
                else {
                    // Abort the request if the property should not be considered changed.
                    shouldRequestUpdate = false;
                }
            }
            if (!this._hasRequestedUpdate && shouldRequestUpdate) {
                this._updatePromise = this._enqueueUpdate();
            }
        }
        /**
         * Requests an update which is processed asynchronously. This should
         * be called when an element should update based on some state not triggered
         * by setting a property. In this case, pass no arguments. It should also be
         * called when manually implementing a property setter. In this case, pass the
         * property `name` and `oldValue` to ensure that any configured property
         * options are honored. Returns the `updateComplete` Promise which is resolved
         * when the update completes.
         *
         * @param name {PropertyKey} (optional) name of requesting property
         * @param oldValue {any} (optional) old value of requesting property
         * @returns {Promise} A Promise that is resolved when the update completes.
         */
        requestUpdate(name, oldValue) {
            this._requestUpdate(name, oldValue);
            return this.updateComplete;
        }
        /**
         * Sets up the element to asynchronously update.
         */
        async _enqueueUpdate() {
            this._updateState = this._updateState | STATE_UPDATE_REQUESTED;
            try {
                // Ensure any previous update has resolved before updating.
                // This `await` also ensures that property changes are batched.
                await this._updatePromise;
            }
            catch (e) {
                // Ignore any previous errors. We only care that the previous cycle is
                // done. Any error should have been handled in the previous update.
            }
            const result = this.performUpdate();
            // If `performUpdate` returns a Promise, we await it. This is done to
            // enable coordinating updates with a scheduler. Note, the result is
            // checked to avoid delaying an additional microtask unless we need to.
            if (result != null) {
                await result;
            }
            return !this._hasRequestedUpdate;
        }
        get _hasRequestedUpdate() {
            return (this._updateState & STATE_UPDATE_REQUESTED);
        }
        get hasUpdated() {
            return (this._updateState & STATE_HAS_UPDATED);
        }
        /**
         * Performs an element update. Note, if an exception is thrown during the
         * update, `firstUpdated` and `updated` will not be called.
         *
         * You can override this method to change the timing of updates. If this
         * method is overridden, `super.performUpdate()` must be called.
         *
         * For instance, to schedule updates to occur just before the next frame:
         *
         * ```
         * protected async performUpdate(): Promise<unknown> {
         *   await new Promise((resolve) => requestAnimationFrame(() => resolve()));
         *   super.performUpdate();
         * }
         * ```
         */
        performUpdate() {
            // Mixin instance properties once, if they exist.
            if (this._instanceProperties) {
                this._applyInstanceProperties();
            }
            let shouldUpdate = false;
            const changedProperties = this._changedProperties;
            try {
                shouldUpdate = this.shouldUpdate(changedProperties);
                if (shouldUpdate) {
                    this.update(changedProperties);
                }
                else {
                    this._markUpdated();
                }
            }
            catch (e) {
                // Prevent `firstUpdated` and `updated` from running when there's an
                // update exception.
                shouldUpdate = false;
                // Ensure element can accept additional updates after an exception.
                this._markUpdated();
                throw e;
            }
            if (shouldUpdate) {
                if (!(this._updateState & STATE_HAS_UPDATED)) {
                    this._updateState = this._updateState | STATE_HAS_UPDATED;
                    this.firstUpdated(changedProperties);
                }
                this.updated(changedProperties);
            }
        }
        _markUpdated() {
            this._changedProperties = new Map();
            this._updateState = this._updateState & ~STATE_UPDATE_REQUESTED;
        }
        /**
         * Returns a Promise that resolves when the element has completed updating.
         * The Promise value is a boolean that is `true` if the element completed the
         * update without triggering another update. The Promise result is `false` if
         * a property was set inside `updated()`. If the Promise is rejected, an
         * exception was thrown during the update.
         *
         * To await additional asynchronous work, override the `_getUpdateComplete`
         * method. For example, it is sometimes useful to await a rendered element
         * before fulfilling this Promise. To do this, first await
         * `super._getUpdateComplete()`, then any subsequent state.
         *
         * @returns {Promise} The Promise returns a boolean that indicates if the
         * update resolved without triggering another update.
         */
        get updateComplete() {
            return this._getUpdateComplete();
        }
        /**
         * Override point for the `updateComplete` promise.
         *
         * It is not safe to override the `updateComplete` getter directly due to a
         * limitation in TypeScript which means it is not possible to call a
         * superclass getter (e.g. `super.updateComplete.then(...)`) when the target
         * language is ES5 (https://github.com/microsoft/TypeScript/issues/338).
         * This method should be overridden instead. For example:
         *
         *   class MyElement extends LitElement {
         *     async _getUpdateComplete() {
         *       await super._getUpdateComplete();
         *       await this._myChild.updateComplete;
         *     }
         *   }
         */
        _getUpdateComplete() {
            return this._updatePromise;
        }
        /**
         * Controls whether or not `update` should be called when the element requests
         * an update. By default, this method always returns `true`, but this can be
         * customized to control when to update.
         *
         * @param _changedProperties Map of changed properties with old values
         */
        shouldUpdate(_changedProperties) {
            return true;
        }
        /**
         * Updates the element. This method reflects property values to attributes.
         * It can be overridden to render and keep updated element DOM.
         * Setting properties inside this method will *not* trigger
         * another update.
         *
         * @param _changedProperties Map of changed properties with old values
         */
        update(_changedProperties) {
            if (this._reflectingProperties !== undefined &&
                this._reflectingProperties.size > 0) {
                // Use forEach so this works even if for/of loops are compiled to for
                // loops expecting arrays
                this._reflectingProperties.forEach((v, k) => this._propertyToAttribute(k, this[k], v));
                this._reflectingProperties = undefined;
            }
            this._markUpdated();
        }
        /**
         * Invoked whenever the element is updated. Implement to perform
         * post-updating tasks via DOM APIs, for example, focusing an element.
         *
         * Setting properties inside this method will trigger the element to update
         * again after this update cycle completes.
         *
         * @param _changedProperties Map of changed properties with old values
         */
        updated(_changedProperties) {
        }
        /**
         * Invoked when the element is first updated. Implement to perform one time
         * work on the element after update.
         *
         * Setting properties inside this method will trigger the element to update
         * again after this update cycle completes.
         *
         * @param _changedProperties Map of changed properties with old values
         */
        firstUpdated(_changedProperties) {
        }
    }
    _a = finalized;
    /**
     * Marks class as having finished creating properties.
     */
    UpdatingElement[_a] = true;

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    const legacyCustomElement = (tagName, clazz) => {
        window.customElements.define(tagName, clazz);
        // Cast as any because TS doesn't recognize the return type as being a
        // subtype of the decorated class when clazz is typed as
        // `Constructor<HTMLElement>` for some reason.
        // `Constructor<HTMLElement>` is helpful to make sure the decorator is
        // applied to elements however.
        // tslint:disable-next-line:no-any
        return clazz;
    };
    const standardCustomElement = (tagName, descriptor) => {
        const { kind, elements } = descriptor;
        return {
            kind,
            elements,
            // This callback is called once the class is otherwise fully defined
            finisher(clazz) {
                window.customElements.define(tagName, clazz);
            }
        };
    };
    /**
     * Class decorator factory that defines the decorated class as a custom element.
     *
     * ```
     * @customElement('my-element')
     * class MyElement {
     *   render() {
     *     return html``;
     *   }
     * }
     * ```
     *
     * @param tagName The name of the custom element to define.
     */
    const customElement = (tagName) => (classOrDescriptor) => (typeof classOrDescriptor === 'function') ?
        legacyCustomElement(tagName, classOrDescriptor) :
        standardCustomElement(tagName, classOrDescriptor);
    const standardProperty = (options, element) => {
        // When decorating an accessor, pass it through and add property metadata.
        // Note, the `hasOwnProperty` check in `createProperty` ensures we don't
        // stomp over the user's accessor.
        if (element.kind === 'method' && element.descriptor &&
            !('value' in element.descriptor)) {
            return Object.assign(Object.assign({}, element), { finisher(clazz) {
                    clazz.createProperty(element.key, options);
                } });
        }
        else {
            // createProperty() takes care of defining the property, but we still
            // must return some kind of descriptor, so return a descriptor for an
            // unused prototype field. The finisher calls createProperty().
            return {
                kind: 'field',
                key: Symbol(),
                placement: 'own',
                descriptor: {},
                // When @babel/plugin-proposal-decorators implements initializers,
                // do this instead of the initializer below. See:
                // https://github.com/babel/babel/issues/9260 extras: [
                //   {
                //     kind: 'initializer',
                //     placement: 'own',
                //     initializer: descriptor.initializer,
                //   }
                // ],
                initializer() {
                    if (typeof element.initializer === 'function') {
                        this[element.key] = element.initializer.call(this);
                    }
                },
                finisher(clazz) {
                    clazz.createProperty(element.key, options);
                }
            };
        }
    };
    const legacyProperty = (options, proto, name) => {
        proto.constructor
            .createProperty(name, options);
    };
    /**
     * A property decorator which creates a LitElement property which reflects a
     * corresponding attribute value. A `PropertyDeclaration` may optionally be
     * supplied to configure property features.
     *
     * This decorator should only be used for public fields. Private or protected
     * fields should use the internalProperty decorator.
     *
     * @example
     *
     *     class MyElement {
     *       @property({ type: Boolean })
     *       clicked = false;
     *     }
     *
     * @ExportDecoratedItems
     */
    function property(options) {
        // tslint:disable-next-line:no-any decorator
        return (protoOrDescriptor, name) => (name !== undefined) ?
            legacyProperty(options, protoOrDescriptor, name) :
            standardProperty(options, protoOrDescriptor);
    }
    /**
     * A property decorator that converts a class property into a getter that
     * executes a querySelector on the element's renderRoot.
     *
     * @param selector A DOMString containing one or more selectors to match.
     *
     * See: https://developer.mozilla.org/en-US/docs/Web/API/Document/querySelector
     *
     * @example
     *
     *     class MyElement {
     *       @query('#first')
     *       first;
     *
     *       render() {
     *         return html`
     *           <div id="first"></div>
     *           <div id="second"></div>
     *         `;
     *       }
     *     }
     *
     */
    function query(selector) {
        return (protoOrDescriptor, 
        // tslint:disable-next-line:no-any decorator
        name) => {
            const descriptor = {
                get() {
                    return this.renderRoot.querySelector(selector);
                },
                enumerable: true,
                configurable: true,
            };
            return (name !== undefined) ?
                legacyQuery(descriptor, protoOrDescriptor, name) :
                standardQuery(descriptor, protoOrDescriptor);
        };
    }
    const legacyQuery = (descriptor, proto, name) => {
        Object.defineProperty(proto, name, descriptor);
    };
    const standardQuery = (descriptor, element) => ({
        kind: 'method',
        placement: 'prototype',
        key: element.key,
        descriptor,
    });

    /**
    @license
    Copyright (c) 2019 The Polymer Project Authors. All rights reserved.
    This code may only be used under the BSD style license found at
    http://polymer.github.io/LICENSE.txt The complete set of authors may be found at
    http://polymer.github.io/AUTHORS.txt The complete set of contributors may be
    found at http://polymer.github.io/CONTRIBUTORS.txt Code distributed by Google as
    part of the polymer project is also subject to an additional IP rights grant
    found at http://polymer.github.io/PATENTS.txt
    */
    const supportsAdoptingStyleSheets = ('adoptedStyleSheets' in Document.prototype) &&
        ('replace' in CSSStyleSheet.prototype);
    const constructionToken = Symbol();
    class CSSResult {
        constructor(cssText, safeToken) {
            if (safeToken !== constructionToken) {
                throw new Error('CSSResult is not constructable. Use `unsafeCSS` or `css` instead.');
            }
            this.cssText = cssText;
        }
        // Note, this is a getter so that it's lazy. In practice, this means
        // stylesheets are not created until the first element instance is made.
        get styleSheet() {
            if (this._styleSheet === undefined) {
                // Note, if `adoptedStyleSheets` is supported then we assume CSSStyleSheet
                // is constructable.
                if (supportsAdoptingStyleSheets) {
                    this._styleSheet = new CSSStyleSheet();
                    this._styleSheet.replaceSync(this.cssText);
                }
                else {
                    this._styleSheet = null;
                }
            }
            return this._styleSheet;
        }
        toString() {
            return this.cssText;
        }
    }
    const textFromCSSResult = (value) => {
        if (value instanceof CSSResult) {
            return value.cssText;
        }
        else if (typeof value === 'number') {
            return value;
        }
        else {
            throw new Error(`Value passed to 'css' function must be a 'css' function result: ${value}. Use 'unsafeCSS' to pass non-literal values, but
            take care to ensure page security.`);
        }
    };
    /**
     * Template tag which which can be used with LitElement's `style` property to
     * set element styles. For security reasons, only literal string values may be
     * used. To incorporate non-literal values `unsafeCSS` may be used inside a
     * template string part.
     */
    const css = (strings, ...values) => {
        const cssText = values.reduce((acc, v, idx) => acc + textFromCSSResult(v) + strings[idx + 1], strings[0]);
        return new CSSResult(cssText, constructionToken);
    };

    /**
     * @license
     * Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    // IMPORTANT: do not change the property name or the assignment expression.
    // This line will be used in regexes to search for LitElement usage.
    // TODO(justinfagnani): inject version number at build time
    (window['litElementVersions'] || (window['litElementVersions'] = []))
        .push('2.3.1');
    /**
     * Sentinal value used to avoid calling lit-html's render function when
     * subclasses do not implement `render`
     */
    const renderNotImplemented = {};
    class LitElement extends UpdatingElement {
        /**
         * Return the array of styles to apply to the element.
         * Override this method to integrate into a style management system.
         *
         * @nocollapse
         */
        static getStyles() {
            return this.styles;
        }
        /** @nocollapse */
        static _getUniqueStyles() {
            // Only gather styles once per class
            if (this.hasOwnProperty(JSCompiler_renameProperty('_styles', this))) {
                return;
            }
            // Take care not to call `this.getStyles()` multiple times since this
            // generates new CSSResults each time.
            // TODO(sorvell): Since we do not cache CSSResults by input, any
            // shared styles will generate new stylesheet objects, which is wasteful.
            // This should be addressed when a browser ships constructable
            // stylesheets.
            const userStyles = this.getStyles();
            if (userStyles === undefined) {
                this._styles = [];
            }
            else if (Array.isArray(userStyles)) {
                // De-duplicate styles preserving the _last_ instance in the set.
                // This is a performance optimization to avoid duplicated styles that can
                // occur especially when composing via subclassing.
                // The last item is kept to try to preserve the cascade order with the
                // assumption that it's most important that last added styles override
                // previous styles.
                const addStyles = (styles, set) => styles.reduceRight((set, s) => 
                // Note: On IE set.add() does not return the set
                Array.isArray(s) ? addStyles(s, set) : (set.add(s), set), set);
                // Array.from does not work on Set in IE, otherwise return
                // Array.from(addStyles(userStyles, new Set<CSSResult>())).reverse()
                const set = addStyles(userStyles, new Set());
                const styles = [];
                set.forEach((v) => styles.unshift(v));
                this._styles = styles;
            }
            else {
                this._styles = [userStyles];
            }
        }
        /**
         * Performs element initialization. By default this calls `createRenderRoot`
         * to create the element `renderRoot` node and captures any pre-set values for
         * registered properties.
         */
        initialize() {
            super.initialize();
            this.constructor._getUniqueStyles();
            this.renderRoot =
                this.createRenderRoot();
            // Note, if renderRoot is not a shadowRoot, styles would/could apply to the
            // element's getRootNode(). While this could be done, we're choosing not to
            // support this now since it would require different logic around de-duping.
            if (window.ShadowRoot && this.renderRoot instanceof window.ShadowRoot) {
                this.adoptStyles();
            }
        }
        /**
         * Returns the node into which the element should render and by default
         * creates and returns an open shadowRoot. Implement to customize where the
         * element's DOM is rendered. For example, to render into the element's
         * childNodes, return `this`.
         * @returns {Element|DocumentFragment} Returns a node into which to render.
         */
        createRenderRoot() {
            return this.attachShadow({ mode: 'open' });
        }
        /**
         * Applies styling to the element shadowRoot using the `static get styles`
         * property. Styling will apply using `shadowRoot.adoptedStyleSheets` where
         * available and will fallback otherwise. When Shadow DOM is polyfilled,
         * ShadyCSS scopes styles and adds them to the document. When Shadow DOM
         * is available but `adoptedStyleSheets` is not, styles are appended to the
         * end of the `shadowRoot` to [mimic spec
         * behavior](https://wicg.github.io/construct-stylesheets/#using-constructed-stylesheets).
         */
        adoptStyles() {
            const styles = this.constructor._styles;
            if (styles.length === 0) {
                return;
            }
            // There are three separate cases here based on Shadow DOM support.
            // (1) shadowRoot polyfilled: use ShadyCSS
            // (2) shadowRoot.adoptedStyleSheets available: use it.
            // (3) shadowRoot.adoptedStyleSheets polyfilled: append styles after
            // rendering
            if (window.ShadyCSS !== undefined && !window.ShadyCSS.nativeShadow) {
                window.ShadyCSS.ScopingShim.prepareAdoptedCssText(styles.map((s) => s.cssText), this.localName);
            }
            else if (supportsAdoptingStyleSheets) {
                this.renderRoot.adoptedStyleSheets =
                    styles.map((s) => s.styleSheet);
            }
            else {
                // This must be done after rendering so the actual style insertion is done
                // in `update`.
                this._needsShimAdoptedStyleSheets = true;
            }
        }
        connectedCallback() {
            super.connectedCallback();
            // Note, first update/render handles styleElement so we only call this if
            // connected after first update.
            if (this.hasUpdated && window.ShadyCSS !== undefined) {
                window.ShadyCSS.styleElement(this);
            }
        }
        /**
         * Updates the element. This method reflects property values to attributes
         * and calls `render` to render DOM via lit-html. Setting properties inside
         * this method will *not* trigger another update.
         * @param _changedProperties Map of changed properties with old values
         */
        update(changedProperties) {
            // Setting properties in `render` should not trigger an update. Since
            // updates are allowed after super.update, it's important to call `render`
            // before that.
            const templateResult = this.render();
            super.update(changedProperties);
            // If render is not implemented by the component, don't call lit-html render
            if (templateResult !== renderNotImplemented) {
                this.constructor
                    .render(templateResult, this.renderRoot, { scopeName: this.localName, eventContext: this });
            }
            // When native Shadow DOM is used but adoptedStyles are not supported,
            // insert styling after rendering to ensure adoptedStyles have highest
            // priority.
            if (this._needsShimAdoptedStyleSheets) {
                this._needsShimAdoptedStyleSheets = false;
                this.constructor._styles.forEach((s) => {
                    const style = document.createElement('style');
                    style.textContent = s.cssText;
                    this.renderRoot.appendChild(style);
                });
            }
        }
        /**
         * Invoked on each update to perform rendering tasks. This method may return
         * any value renderable by lit-html's NodePart - typically a TemplateResult.
         * Setting properties inside this method will *not* trigger the element to
         * update.
         */
        render() {
            return renderNotImplemented;
        }
    }
    /**
     * Ensure this class is marked as `finalized` as an optimization ensuring
     * it will not needlessly try to `finalize`.
     *
     * Note this property name is a string to prevent breaking Closure JS Compiler
     * optimizations. See updating-element.ts for more information.
     */
    LitElement['finalized'] = true;
    /**
     * Render method used to render the value to the element's DOM.
     * @param result The value to render.
     * @param container Node into which to render.
     * @param options Element name.
     * @nocollapse
     */
    LitElement.render = render$1;

    function hass() {
      return document.querySelector('home-assistant').hass
    }
    function load_lovelace() {
      if(customElements.get("hui-view")) return true;

      const res = document.createElement("partial-panel-resolver");
      res.hass = hass();
      res.route = {path: "/lovelace/"};
      // res._updateRoutes();
      try {
        document.querySelector("home-assistant").appendChild(res).catch((error) => {});
      } catch (error) {
        document.querySelector("home-assistant").removeChild(res);
      }
      if(customElements.get("hui-view")) return true;
      return false;

    }

    const getConfiguration = async (hass) => {
        const response = await hass.connection.sendMessagePromise({
            type: "hacs/config",
        });
        return response;
    };
    const getRepositories = async (hass) => {
        const response = await hass.connection.sendMessagePromise({
            type: "hacs/repositories",
        });
        return response;
    };
    const getCritical = async (hass) => {
        const response = await hass.connection.sendMessagePromise({
            type: "hacs/get_critical",
        });
        return response;
    };
    const getStatus = async (hass) => {
        const response = await hass.connection.sendMessagePromise({
            type: "hacs/status",
        });
        return response;
    };
    const getRemovedRepositories = async (hass) => {
        const response = await hass.connection.sendMessagePromise({
            type: "hacs/removed",
        });
        return response;
    };
    const repositoryInstall = async (hass, repository) => {
        await hass.connection.sendMessagePromise({
            type: "hacs/repository",
            action: "install",
            repository: repository,
        });
    };
    const repositoryUninstall = async (hass, repository) => {
        await hass.connection.sendMessagePromise({
            type: "hacs/repository",
            action: "uninstall",
            repository: repository,
        });
    };
    const repositoryReleasenotes = async (hass, repository) => {
        const response = await hass.connection.sendMessagePromise({
            type: "hacs/repository",
            action: "release_notes",
            repository: repository,
        });
        return response;
    };
    const repositoryToggleBeta = async (hass, repository) => {
        await hass.connection.sendMessagePromise({
            type: "hacs/repository",
            action: "toggle_beta",
            repository: repository,
        });
    };
    const repositoryInstallVersion = async (hass, repository, version) => {
        await hass.connection.sendMessagePromise({
            type: "hacs/repository/data",
            action: "install",
            repository: repository,
            data: version,
        });
    };
    const repositoryAdd = async (hass, repository, category) => {
        await hass.connection.sendMessagePromise({
            type: "hacs/repository/data",
            action: "add",
            repository: repository,
            data: category,
        });
    };
    const repositorySetNotNew = async (hass, repository) => {
        await hass.connection.sendMessagePromise({
            type: "hacs/repository",
            action: "not_new",
            repository: repository,
        });
    };
    const repositoryUpdate = async (hass, repository) => {
        await hass.connection.sendMessagePromise({
            type: "hacs/repository",
            action: "update",
            repository: repository,
        });
    };
    const repositoryDelete = async (hass, repository) => {
        await hass.connection.sendMessagePromise({
            type: "hacs/repository",
            action: "delete",
            repository: repository,
        });
    };
    const settingsClearAllNewRepositories = async (hass, categories) => {
        await hass.connection.sendMessagePromise({
            type: "hacs/settings",
            action: "clear_new",
            categories,
        });
    };
    const getLovelaceConfiguration = async (hass) => {
        try {
            const response = await hass.connection.sendMessagePromise({
                type: "lovelace/resources",
            });
            return response;
        }
        catch (e) {
            return null;
        }
    };
    const fetchResources = (hass) => hass.connection.sendMessagePromise({
        type: "lovelace/resources",
    });
    const createResource = (hass, values) => hass.callWS(Object.assign({ type: "lovelace/resources/create" }, values));
    const updateResource = (hass, values) => hass.callWS(Object.assign({ type: "lovelace/resources/update" }, values));
    const deleteResource = (hass, id) => hass.callWS({
        type: "lovelace/resources/delete",
        resource_id: id,
    });

    const hacsStyleVariables = css `
  :host {
    --hcv-color-error: var(--hacs-error-color, var(--google-red-500, #f44336));
    --hcv-color-warning: var(--hacs-warning-color, #ff8c00);
    --hcv-color-update: var(--hacs-update-color, #f4b400);
    --hcv-color-new: var(--hacs-new-color, var(--google-blue-500, #2196f3));
    --hcv-color-icon: var(--hacs--default-icon-color, var(--sidebar-icon-color));
    --hcv-color-markdown-background: var(--markdown-code-background-color, #f6f8fa);

    --hcv-text-color-primary: var(--primary-text-color);
    --hcv-text-color-on-background: var(--text-primary-color);
    --hcv-text-color-secondary: var(--secondary-text-color);
    --hcv-text-color-link: var(--link-text-color, var(--accent-color));

    /*hacs-fab*/
    --hcv-color-fab: var(--hacs-fab-color, var(--accent-color));
    --hcv-text-color-fab: var(--hacs-fab-text-color, var(--hcv-text-color-on-background));

    /*hacs-chip*/
    --hcv-color-chip: var(--hacs-chip-color, var(--accent-color));
    --hcv-text-color-chip: var(--hacs-chip-text-color, var(--hcv-text-color-on-background));

    /*hacs-fab*/
    --hcv-text-decoration-link: var(--hacs-link-text-decoration, none);
    /*hacs-fab*/
  }
`;

    function areInputsEqual(newInputs, lastInputs) {
        if (newInputs.length !== lastInputs.length) {
            return false;
        }
        for (var i = 0; i < newInputs.length; i++) {
            if (newInputs[i] !== lastInputs[i]) {
                return false;
            }
        }
        return true;
    }

    function memoizeOne(resultFn, isEqual) {
        if (isEqual === void 0) { isEqual = areInputsEqual; }
        var lastThis;
        var lastArgs = [];
        var lastResult;
        var calledOnce = false;
        function memoized() {
            var newArgs = [];
            for (var _i = 0; _i < arguments.length; _i++) {
                newArgs[_i] = arguments[_i];
            }
            if (calledOnce && lastThis === this && isEqual(newArgs, lastArgs)) {
                return lastResult;
            }
            lastResult = resultFn.apply(this, newArgs);
            calledOnce = true;
            lastThis = this;
            lastArgs = newArgs;
            return lastResult;
        }
        return memoized;
    }

    /*
    Simple logger class to make logging consistent

    Usage:

    Basic: const logger = new Logger()
    Advanced: const logger = new Logger("main")

    logger.info("My message")
    logger.error("My message")
    logger.warning("My message")

    const obj = {test: "string"}
    logger.info(obj)

    */
    class HacsLogger {
        constructor(section) {
            const name = "HACS";
            this.section = section;
            if (section === undefined)
                this.prefix = name;
            else
                this.prefix = `${name}.${section}`;
        }
        debug(content, object) {
            if (content instanceof Object)
                console.debug(`[${this.prefix}] `, content);
            else
                console.debug(`[${this.prefix}] ${content}`, object || "");
        }
        info(content, object) {
            if (content instanceof Object)
                console.info(`[${this.prefix}] `, content);
            else
                console.info(`[${this.prefix}] ${content}`, object || "");
        }
        warning(content, object) {
            if (content instanceof Object)
                console.warn(`[${this.prefix}] `, content);
            else
                console.warn(`[${this.prefix}] ${content}`, object || "");
        }
        error(content, object) {
            if (content instanceof Object)
                console.error(`[${this.prefix}] `, content);
            else
                console.error(`[${this.prefix}] ${content}`, object || "");
        }
    }

    const navigate = (_node, path) => {
        history.pushState(null, "", path);
    };

    const sortRepositoriesByName = (repositories) => {
        return repositories === null || repositories === void 0 ? void 0 : repositories.sort((a, b) => a.name.toLowerCase() > b.name.toLowerCase() ? 1 : -1);
    };

    /**
     * @license
     * Copyright (c) 2018 The Polymer Project Authors. All rights reserved.
     * This code may only be used under the BSD style license found at
     * http://polymer.github.io/LICENSE.txt
     * The complete set of authors may be found at
     * http://polymer.github.io/AUTHORS.txt
     * The complete set of contributors may be found at
     * http://polymer.github.io/CONTRIBUTORS.txt
     * Code distributed by Google as part of the polymer project is also
     * subject to an additional IP rights grant found at
     * http://polymer.github.io/PATENTS.txt
     */
    // IE11 doesn't support classList on SVG elements, so we emulate it with a Set
    class ClassList {
        constructor(element) {
            this.classes = new Set();
            this.changed = false;
            this.element = element;
            const classList = (element.getAttribute('class') || '').split(/\s+/);
            for (const cls of classList) {
                this.classes.add(cls);
            }
        }
        add(cls) {
            this.classes.add(cls);
            this.changed = true;
        }
        remove(cls) {
            this.classes.delete(cls);
            this.changed = true;
        }
        commit() {
            if (this.changed) {
                let classString = '';
                this.classes.forEach((cls) => classString += cls + ' ');
                this.element.setAttribute('class', classString);
            }
        }
    }
    /**
     * Stores the ClassInfo object applied to a given AttributePart.
     * Used to unset existing values when a new ClassInfo object is applied.
     */
    const previousClassesCache = new WeakMap();
    /**
     * A directive that applies CSS classes. This must be used in the `class`
     * attribute and must be the only part used in the attribute. It takes each
     * property in the `classInfo` argument and adds the property name to the
     * element's `class` if the property value is truthy; if the property value is
     * falsey, the property name is removed from the element's `class`. For example
     * `{foo: bar}` applies the class `foo` if the value of `bar` is truthy.
     * @param classInfo {ClassInfo}
     */
    const classMap = directive((classInfo) => (part) => {
        if (!(part instanceof AttributePart) || (part instanceof PropertyPart) ||
            part.committer.name !== 'class' || part.committer.parts.length > 1) {
            throw new Error('The `classMap` directive must be used in the `class` attribute ' +
                'and must be the only part in the attribute.');
        }
        const { committer } = part;
        const { element } = committer;
        let previousClasses = previousClassesCache.get(part);
        if (previousClasses === undefined) {
            // Write static classes once
            // Use setAttribute() because className isn't a string on SVG elements
            element.setAttribute('class', committer.strings.join(' '));
            previousClassesCache.set(part, previousClasses = new Set());
        }
        const classList = (element.classList || new ClassList(element));
        // Remove old classes that no longer apply
        // We use forEach() instead of for-of so that re don't require down-level
        // iteration.
        previousClasses.forEach((name) => {
            if (!(name in classInfo)) {
                classList.remove(name);
                previousClasses.delete(name);
            }
        });
        // Add or remove classes based on their classMap value
        for (const name in classInfo) {
            const value = classInfo[name];
            if (value != previousClasses.has(name)) {
                // We explicitly want a loose truthy check of `value` because it seems
                // more convenient that '' and 0 are skipped.
                if (value) {
                    classList.add(name);
                    previousClasses.add(name);
                }
                else {
                    classList.remove(name);
                    previousClasses.delete(name);
                }
            }
        }
        if (typeof classList.commit === 'function') {
            classList.commit();
        }
    });

    let HacsSinglePageLayout = class HacsSinglePageLayout extends LitElement {
        constructor() {
            super(...arguments);
            this.isWide = false;
        }
        render() {
            this.isWide =
                window.localStorage.getItem("dockedSidebar") === '"always_hidden"';
            return html `
      <div
        class="content ${classMap({
            narrow: !this.isWide,
        })}"
      >
        <div class="header-group">
          ${this.narrow || this.isWide
            ? html `<ha-menu-button
                class="${classMap({
                sidebarhidden: this.isWide && !this.narrow,
            })}"
                .hass=${this.hass}
                .narrow=${this.narrow}
              ></ha-menu-button>`
            : ""}

          <div class="header">${this.header || ""}</div>
        </div>
        <div
          class="together layout ${classMap({
            narrow: this.narrow,
            vertical: !this.isWide || this.narrow,
            horizontal: this.isWide && !this.narrow,
        })}"
        >
          <div class="intro">${this.intro || ""}</div>
          <div class="panel flex-auto"><slot></slot></div>
        </div>
      </div>
    `;
        }
        static get styles() {
            return css `
      :host {
        display: block;
      }
      .header-group {
        display: flex;
        align-items: center;
      }
      .content {
        padding: 28px 20px 0;
        max-width: 1040px;
        margin: 0 auto;
      }
      .layout {
        display: flex;
      }
      .horizontal {
        flex-direction: row;
      }
      .vertical {
        flex-direction: column;
      }
      .flex-auto {
        flex: 1 1 auto;
      }
      .header {
        font-family: var(--paper-font-headline_-_font-family);
        -webkit-font-smoothing: var(
          --paper-font-headline_-_-webkit-font-smoothing
        );
        font-size: var(--paper-font-headline_-_font-size);
        font-weight: var(--paper-font-headline_-_font-weight);
        letter-spacing: var(--paper-font-headline_-_letter-spacing);
        line-height: var(--paper-font-headline_-_line-height);
        opacity: var(--dark-primary-opacity);
      }
      .together {
        margin-top: 32px;
      }
      .intro {
        font-family: var(--paper-font-subhead_-_font-family);
        -webkit-font-smoothing: var(
          --paper-font-subhead_-_-webkit-font-smoothing
        );
        font-weight: var(--paper-font-subhead_-_font-weight);
        line-height: var(--paper-font-subhead_-_line-height);
        width: 100%;
        max-width: 400px;
        margin-right: 40px;
        opacity: var(--dark-primary-opacity);
        font-size: 14px;
        padding-bottom: 20px;
      }
      .panel {
        margin-top: -24px;
      }
      .panel ::slotted(*) {
        margin-top: 24px;
        display: block;
      }
      .narrow.content {
        max-width: 640px;
      }
      .narrow .together {
        margin-top: 0;
      }
      .narrow .intro {
        padding-bottom: 20px;
        margin-right: 0;
        max-width: 500px;
      }
      ha-menu-button {
        --app-header-background-color: var(--primary-background-color);
      }

      ha-menu-button.sidebarhidden {
        position: absolute;
        top: 0;
        left: 0;
      }
    `;
        }
    };
    __decorate([
        property({ attribute: false })
    ], HacsSinglePageLayout.prototype, "hass", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsSinglePageLayout.prototype, "narrow", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsSinglePageLayout.prototype, "route", void 0);
    __decorate([
        property({ type: Boolean })
    ], HacsSinglePageLayout.prototype, "isWide", void 0);
    __decorate([
        property()
    ], HacsSinglePageLayout.prototype, "header", void 0);
    __decorate([
        property()
    ], HacsSinglePageLayout.prototype, "intro", void 0);
    HacsSinglePageLayout = __decorate([
        customElement("hacs-single-page-layout")
    ], HacsSinglePageLayout);

    const hacsLinkStyle = css `
  a {
    text-decoration: var(--hcv-text-decoration-link);
    color: var(--hcv-text-color-link);
  }
`;
    const hacsButtonStyle = css `
  mwc-button[raised] {
    border-radius: 10px;
  }
`;
    const hacsIconStyle = css `
  paper-menu-button,
  ha-icon-button,
  ha-icon {
    color: var(--hcv-color-icon);
  }
`;

    const hacsCommonClasses = css `
  .warning {
    color: var(--hcv-color-warning);
  }
  .pending_update {
    color: var(--hcv-color-update);
  }
  .pending_restart,
  .error,
  .uninstall {
    color: var(--hcv-color-error);
  }
  .header {
    font-size: var(--paper-font-headline_-_font-size);
    opacity: var(--dark-primary-opacity);
    padding: 8px 0 4px 16px;
  }
`;
    const HacsStyles = [
        hacsIconStyle,
        hacsButtonStyle,
        hacsCommonClasses,
        hacsLinkStyle,
        css `
    :root {
      font-family: var(--paper-font-body1_-_font-family);
      -webkit-font-smoothing: var(--paper-font-body1_-_-webkit-font-smoothing);
      font-size: var(--paper-font-body1_-_font-size);
      font-weight: var(--paper-font-body1_-_font-weight);
      line-height: var(--paper-font-body1_-_line-height);
    }
  `,
    ];

    var common = {
    	about: "Om",
    	add: "tilføj",
    	appdaemon_apps: "AppDaemon-apps",
    	appdaemon_plural: "AppDaemon-apps",
    	background_task: "Baggrundsopgave kører. Denne side vil genindlæses automatisk.",
    	check_log_file: "Tjek din logfil for flere detaljer.",
    	"continue": "Fortsæt",
    	disabled: "Deaktiveret",
    	documentation: "Dokumentation",
    	element: "element",
    	hacs_is_disabled: "HACS er deaktiveret",
    	install: "Installer",
    	installed: "installeret",
    	integration: "Integration",
    	integration_plural: "Integrationer",
    	integrations: "Integrationer",
    	lovelace: "Lovelace",
    	lovelace_element: "Lovelace-element",
    	lovelace_elements: "Lovelace-elementer",
    	manage: "administrer",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDaemon-apps",
    	netdaemon_plural: "NetDaemon-apps",
    	plugin: "Lovelace",
    	plugin_plural: "Lovelace-elementer",
    	plugins: "Lovelace-elementer",
    	python_script: "Python-script",
    	python_script_plural: "Python-scripts",
    	python_scripts: "Python-scripts",
    	repositories: "Repositories",
    	repository: "Repository",
    	settings: "indstillinger",
    	theme: "Tema",
    	theme_plural: "Temaer",
    	themes: "Temaer",
    	uninstall: "Afinstaller",
    	update: "Opdater",
    	version: "Version"
    };
    var config = {
    	abort: {
    		single_instance_allowed: "Kun en enkelt konfiguration af HACS er tilladt."
    	},
    	error: {
    		auth: "Personlig adgangstoken er ikke korrekt."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Aktiver opdagelse & sporing af AppDaemon-apps",
    				netdaemon: "Aktiver opdagelse og sporing af NetDaemon-apps",
    				python_script: "Aktivér opdagelse og sporing af python_scripts",
    				sidepanel_icon: "Sidepanelikon",
    				sidepanel_title: "Sidepanelets titel",
    				theme: "Aktivér opdagelse og sporing af temaer",
    				token: "GitHub personlig adgangstoken"
    			},
    			description: "Hvis du har brug for hjælp til konfigurationen, så kig her: https://hacs.xyz/docs/configuration/start"
    		}
    	}
    };
    var confirm = {
    	add_to_lovelace: "Er du sikker på, at du vil tilføje dette til dine Lovelace-ressourcer?",
    	bg_task: "Handlingen er deaktiveret, mens baggrundsopgaver kører.",
    	cancel: "Annuller",
    	"continue": "Er du sikker på, at du vil fortsætte?",
    	"delete": "Er du sikker på, at du vil slette '{Item}'?",
    	delete_installed: "'{item}' er installeret. Du skal afinstallere det, før du kan slette det.",
    	exist: "{item} findes allerede",
    	generic: "Er du sikker?",
    	home_assistant_is_restarting: "Vent venligst - Home Assistant genstarter nu.",
    	home_assistant_version_not_correct: "Du kører Home Assistant version '{haversion}', men dette repository kræver som minimum version '{minversion}'.",
    	no: "Nej",
    	no_upgrades: "Der er ingen opdateringer tilgængelig",
    	ok: "OK",
    	overwrite: "Dette vil overskrive den.",
    	reload_data: "Dette genindlæser data fra alle repositories, som HACS kender til. Dette vil tage nogen tid at fuldføre.",
    	restart_home_assistant: "Er du sikker på, at du vil genstarte Home Assistant?",
    	uninstall: "Er du sikker på, at du vil afinstallere '{Item}'?",
    	upgrade_all: "Dette vil opdatere alle repositories. Sørg for at du har læst udgivelsesnoterne for dem alle, inden du fortsætter.",
    	yes: "Ja"
    };
    var dialog_about = {
    	frontend_version: "Frontend-version",
    	installed_repositories: "Installerede repositories",
    	integration_version: "Integrationsversion",
    	useful_links: "Nyttige links"
    };
    var dialog_add_repo = {
    	limit: "Kun de første 100 repositories vises. Brug søgningen til at filtrere, hvad du har brug for",
    	no_match: "Der blev ikke fundet nogen repositories, der matcher dit filter",
    	sort_by: "Sorter efter",
    	title: "Tilføj repository"
    };
    var dialog_custom_repositories = {
    	category: "Kategori",
    	no_category: "Manglende kategori",
    	no_repository: "Manglende repository",
    	title: "Brugerdefinerede repositories",
    	url_placeholder: "Tilføj brugerdefineret repository-webadresse"
    };
    var dialog_info = {
    	author: "Udvikler",
    	downloads: "Downloads",
    	install: "Installer dette repository i HACS",
    	loading: "Indlæser oplysninger...",
    	no_info: "Udvikleren har ikke givet flere oplysninger om dette repository",
    	open_issues: "Åbn issues",
    	open_repo: "Åbn repository",
    	stars: "Stjerner",
    	version_installed: "Installeret version"
    };
    var dialog_install = {
    	restart: "Husk, at du skal genstarte Home Assistant, før ændringer af integrationer (custom_components) træder i kræft.",
    	select_version: "Vælg version",
    	show_beta: "Vis betaversioner",
    	type: "Type",
    	url: "Webadresse"
    };
    var dialog_update = {
    	available_version: "Tilgængelig version",
    	changelog: "Udgivelsesnoter",
    	installed_version: "Installeret version",
    	releasenotes: "Udgivelsesnoter for {release}",
    	title: "Ventende opdatering"
    };
    var entry = {
    	information: "Oplysninger",
    	intro: "Opdateringer og vigtige meddelelser vises her, hvis der er nogen",
    	messages: {
    		disabled: {
    			content: "Tjek din logfil for flere detaljer",
    			title: "HACS er deaktiveret"
    		},
    		has_pending_tasks: {
    			content: "Nogle repositories vises muligvis ikke, før dette er fuldført",
    			title: "Baggrundsopgaver venter"
    		},
    		resources: {
    			content: "Du har {number} Lovelace-elementer, der ikke er indlæst korrekt i Lovelace.",
    			title: "Ikke indlæst i Lovelace"
    		},
    		restart: {
    			content: "Du har {number} integrationer, der kræver en genstart af Home Assistant. Du kan genstarte fra 'Serveradministration'-sektionen under Indstillinger i Home Assistant-brugerfladen.",
    			title: "Afventer genstart"
    		},
    		startup: {
    			content: "HACS starter op. Der kan i dette tidsrum mangle nogle oplysninger, eller de kan være ukorekte.",
    			title: "HACS starter op"
    		},
    		wrong_frontend_installed: {
    			content: "Du har version {running} af HACS-frontend installeret, men version {expected} var forventet, hvis dette ser du denne besked. Home Assistant kunne ikke installere den nye version. Prøv at genstarte Home Assistant.",
    			title: "Uventet frontend-version"
    		},
    		wrong_frontend_loaded: {
    			content: "Du kører version {running} af HACS-frontend, men version {expected} var forventet. Du bør rydde din browser-cache.",
    			title: "Uventet frontend-version"
    		}
    	},
    	pending_updates: "Ventende opdateringer"
    };
    var menu = {
    	about: "Om HACS",
    	clear: "Ryd alle nye",
    	custom_repositories: "Brugerdefinerede repositories",
    	dismiss: "Afvis alle nye repositories",
    	documentation: "Dokumentation",
    	open_issue: "Opret issue",
    	reload: "Genindlæs vindue"
    };
    var options = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Aktiver opdagelse og sporing af AppDaemon-apps",
    				country: "Filtrer med landekode.",
    				debug: "Aktiver debug.",
    				experimental: "Aktivér eksperimentelle funktioner",
    				netdaemon: "Aktiver opdagelse og sporing af NetDaemon-apps",
    				not_in_use: "Ikke i brug med YAML",
    				release_limit: "Antal udgivelser, der skal vises.",
    				sidepanel_icon: "Sidepanelikon",
    				sidepanel_title: "Sidepanelets titel"
    			}
    		}
    	}
    };
    var repository_banner = {
    	config_flow: "Denne integration understøtter config_flow. Det betyder, at du nu kan gå til integrationssektionen i din brugerflade for at konfigurere den.",
    	config_flow_title: "Brugerfladekonfiguration understøttet",
    	integration_not_loaded: "Denne integration er ikke indlæst i Home Assistant.",
    	no_restart_required: "Ingen genstart påkrævet",
    	not_loaded: "Ikke indlæst",
    	plugin_not_loaded: "Dette element er ikke føjet til dine Lovelace-ressourcer.",
    	restart: "Du skal genstarte Home Assistant.",
    	restart_pending: "Afventer genstart"
    };
    var repository_card = {
    	dismiss: "Afvis",
    	hide: "Skjul",
    	information: "Oplysninger",
    	new_repository: "Nyt repository",
    	not_loaded: "Ikke indlæst",
    	open_issue: "Opret issue",
    	open_source: "Åbn kilde",
    	pending_restart: "Afventer genstart",
    	pending_update: "Ventende opdatering",
    	reinstall: "Geninstaller",
    	report: "Rapporter til fjernelse",
    	update_information: "Opdater oplysninger"
    };
    var repository = {
    	add_to_lovelace: "Tilføj til Lovelace",
    	authors: "Forfattere",
    	available: "Tilgængelig",
    	back_to: "Tilbage til",
    	changelog: "Udgivelsesnoter",
    	downloads: "Downloads",
    	flag_this: "Marker denne",
    	frontend_version: "Frontend-version",
    	github_stars: "GitHub-stjerner",
    	goto_integrations: "Gå til integrationer",
    	hide: "Skjul",
    	hide_beta: "Skjul beta",
    	install: "Installer",
    	installed: "Installeret",
    	lovelace_copy_example: "Kopiér eksemplet til din Udklipsholder",
    	lovelace_instruction: "Tilføj dette til din lovelace-konfiguration",
    	lovelace_no_js_type: "Kunne ikke afgøre typen af dette element, tjek venligst repository'et.",
    	newest: "nyeste",
    	note_appdaemon: "Du skal stadig føje den til filen 'apps.yaml'",
    	note_installed: "Når det er installeret, vil dette være placeret i",
    	note_integration: "du skal stadig føje den til filen 'configuration.yaml'",
    	note_plugin: "du skal stadig tilføje det til din lovelace-konfiguration ('ui-lovelace.yaml' eller Tekstbaseret redigering)",
    	note_plugin_post_107: "du skal stadig tilføje det til din lovelace-konfiguration ('configuration.yaml' eller ressourceeditoren '/config/lovelace/resources')",
    	open_issue: "Opret issue",
    	open_plugin: "Åbn element",
    	reinstall: "Geninstaller",
    	repository: "Repository",
    	restart_home_assistant: "Genstart Home Assistant",
    	show_beta: "Vis beta",
    	uninstall: "Afinstaller",
    	update_information: "Opdater oplysninger",
    	upgrade: "Opdater"
    };
    var search = {
    	placeholder: "Søg efter repository"
    };
    var sections = {
    	about: {
    		description: "Vis information om HACS",
    		title: "Om"
    	},
    	automation: {
    		description: "Det er her, du finder python_scripts, AppDaemon-apps og NetDaemon-apps",
    		title: "Automatisering"
    	},
    	frontend: {
    		description: "Det er her, du finder temaer, brugerdefinerede kort og andre elementer til lovelace",
    		title: "Frontend"
    	},
    	integrations: {
    		description: "Det er her, du finder brugerdefinerede integrationer (custom_components)",
    		title: "Integrationer"
    	},
    	pending_repository_upgrade: "Du kører version {installed}, version {available} er tilgængelig"
    };
    var settings = {
    	add_custom_repository: "TILFØJ ET BRUGERDEFINERET REPOSITORY",
    	adding_new_repo: "Tilføjer nyt repository '{repo}'",
    	adding_new_repo_category: "Med kategorien '{category}'.",
    	bg_task_custom: "Brugerdefinerede repositories er skjult, mens opgaver i baggrunden kører.",
    	category: "Kategori",
    	compact_mode: "Kompakt tilstand",
    	custom_repositories: "BRUGERDEFINEREDE REPOSITORIES",
    	"delete": "Slet",
    	display: "Visning",
    	grid: "Gitter",
    	hacs_repo: "HACS-repo",
    	hidden_repositories: "Skjulte repositories",
    	missing_category: "Du skal vælge en kategori",
    	open_repository: "Åbn repository",
    	reload_data: "Genindlæs data",
    	reload_window: "Genindlæs vindue",
    	repository_configuration: "Konfiguration af repository",
    	save: "Gem",
    	table: "Tabel",
    	table_view: "Tabelvisning",
    	unhide: "Vis",
    	upgrade_all: "Opdater alle"
    };
    var store = {
    	ascending: "stigende",
    	clear_new: "Marker alle som set",
    	descending: "faldende",
    	last_updated: "Sidst opdateret",
    	name: "Navn",
    	new_repositories: "Nye repositories",
    	new_repositories_note: "Du har over 10 nye repositories, der vises her. Hvis du vil rydde dem alle, skal du klikke på de 3 prikker i øverste højre hjørne og afvise dem alle.",
    	no_repositories: "Ingen repositories",
    	no_repositories_desc1: "Det ser ud til, at du ikke har nogen repositories installeret i denne sektion endnu.",
    	no_repositories_desc2: "Klik på + i nederste hjørne for at tilføje dit første!",
    	no_repositories_found_desc1: "Der blev ikke fundet installerede repositories, der matcher \"{searchInput}\" i denne sektion.",
    	no_repositories_found_desc2: "Prøv at søge efter noget andet!",
    	pending_upgrades: "Ventende opdateringer",
    	placeholder_search: "Indtast en søgeterm...",
    	sort: "sorter",
    	stars: "Stjerner",
    	status: "Status"
    };
    var time = {
    	ago: "siden",
    	day: "dag",
    	days: "dage",
    	hour: "time",
    	hours: "timer",
    	minute: "minut",
    	minutes: "minutter",
    	month: "måned",
    	months: "måneder",
    	one: "Et",
    	one_day_ago: "en dag siden",
    	one_hour_ago: "en time siden",
    	one_minute_ago: "et minut siden",
    	one_month_ago: "en måned siden",
    	one_second_ago: "et sekund siden",
    	one_year_ago: "et år siden",
    	second: "sekund",
    	seconds: "sekunder",
    	x_days_ago: "{x} dage siden",
    	x_hours_ago: "{x} timer siden",
    	x_minutes_ago: "{x} minutter siden",
    	x_months_ago: "{x} måneder siden",
    	x_seconds_ago: "{x} sekunder siden",
    	x_years_ago: "{x} år siden",
    	year: "år",
    	years: "år"
    };
    var da = {
    	common: common,
    	config: config,
    	confirm: confirm,
    	dialog_about: dialog_about,
    	dialog_add_repo: dialog_add_repo,
    	dialog_custom_repositories: dialog_custom_repositories,
    	dialog_info: dialog_info,
    	dialog_install: dialog_install,
    	dialog_update: dialog_update,
    	entry: entry,
    	menu: menu,
    	options: options,
    	repository_banner: repository_banner,
    	repository_card: repository_card,
    	repository: repository,
    	search: search,
    	sections: sections,
    	settings: settings,
    	store: store,
    	time: time
    };

    var da$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common,
        config: config,
        confirm: confirm,
        dialog_about: dialog_about,
        dialog_add_repo: dialog_add_repo,
        dialog_custom_repositories: dialog_custom_repositories,
        dialog_info: dialog_info,
        dialog_install: dialog_install,
        dialog_update: dialog_update,
        entry: entry,
        menu: menu,
        options: options,
        repository_banner: repository_banner,
        repository_card: repository_card,
        repository: repository,
        search: search,
        sections: sections,
        settings: settings,
        store: store,
        time: time,
        'default': da
    });

    var common$1 = {
    	about: "Über",
    	add: "hinzufügen",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "AppDaemon Apps",
    	appdaemon_plural: "AppDaemon Apps",
    	background_task: "Hintergrundprozess läuft. Diese Seite lädt neu, sobald dieser fertig ist.",
    	check_log_file: "Überprüfe die Logdatei für weitere Informationen.",
    	"continue": "Fortfahren",
    	disabled: "Deaktiviert",
    	documentation: "Dokumentation",
    	element: "Element",
    	hacs_is_disabled: "HACS ist deaktiviert",
    	install: "Installieren",
    	installed: "Installiert",
    	integration: "Integration",
    	integration_plural: "Integrationen",
    	integrations: "Integrationen",
    	lovelace: "Lovelace",
    	lovelace_element: "Lovelace-Element",
    	lovelace_elements: "Lovelace-Elemente",
    	manage: "verwalten",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDaemon Apps",
    	netdaemon_plural: "NetDaemon Apps",
    	plugin: "Lovelace",
    	plugin_plural: "Lovelace-Elemente",
    	plugins: "Lovelace-Elemente",
    	python_script: "Python Skript",
    	python_script_plural: "Python Skripte",
    	python_scripts: "Python Skripte",
    	repositories: "Repositories",
    	repository: "Repository",
    	settings: "Einstellungen",
    	theme: "Theme",
    	theme_plural: "Themes",
    	themes: "Themes",
    	uninstall: "Deinstallieren",
    	update: "Aktualisieren",
    	version: "Version"
    };
    var config$1 = {
    	abort: {
    		single_instance_allowed: "Es ist nur eine einzelne HACS-Instanz erlaubt."
    	},
    	error: {
    		auth: "Persönlicher Zugriffstoken ist falsch."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "AppDaemon App-Entdeckung & Nachverfolgung aktivieren",
    				netdaemon: "NetDaemon App-Entdeckung & Nachverfolgung aktivieren",
    				python_script: "Python Script-Entdeckung & Nachverfolgung aktivieren",
    				sidepanel_icon: "Sidepanel Symbol",
    				sidepanel_title: "Sidepanel Titel",
    				theme: "Theme-Entdeckung & Nachverfolgung aktivieren",
    				token: "Persönlicher GitHub Zugriffstoken"
    			},
    			description: "Wenn du Hilfe mit den Einstellungen brauchst, kannst du hier nachsehen: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$1 = {
    	add_to_lovelace: "Möchtest du dies wirklich zu deinen Lovelace-Ressourcen hinzufügen?",
    	bg_task: "Die Aktion ist deaktiviert, während Hintergrundprozesse ausgeführt werden.",
    	cancel: "Abbrechen",
    	"continue": "Bist du dir sicher, dass du fortfahren möchtest?",
    	"delete": "Möchtest du '{item}' wirklich löschen?",
    	delete_installed: "'{item}' ist installiert. Du musst es deinstallieren, bevor du es löschen kannst.",
    	exist: "{item} existiert bereits",
    	generic: "Bist du dir sicher?",
    	home_assistant_is_restarting: "Bitte warte, Home Assistant wird jetzt neu gestartet.",
    	home_assistant_version_not_correct: "Du benutzt die Home Assistant-Version '{haversion}', für dieses Repository muss jedoch die Mindestversion '{minversion}' installiert sein.",
    	no: "Nein",
    	no_upgrades: "Keine Upgrades ausstehend",
    	ok: "OK",
    	overwrite: "Dadurch wird die Datei überschrieben.",
    	reload_data: "Hierdurch werden die Daten aller Repositories die HACS kennt neu geladen. Dies wird einige Zeit in Anspruch nehmen.",
    	restart_home_assistant: "Bist du dir sicher, dass du Home Assistant neu starten möchtest?",
    	uninstall: "Möchtest du '{item}' wirklich deinstallieren?",
    	upgrade_all: "Hierdurch werden all diese Repositories aktualisiert. Stelle sicher, dass du die Versionshinweise vorher gelesen hast.",
    	yes: "Ja"
    };
    var dialog_about$1 = {
    	frontend_version: "Frontend Version",
    	installed_repositories: "Installierte Repositories",
    	integration_version: "Integrations Version",
    	useful_links: "Nützliche Links"
    };
    var dialog_add_repo$1 = {
    	limit: "Es werden nur die ersten 100 Repositorys angezeigt. Verwenden Sie die Suche, um zu filtern, was Sie benötigen",
    	no_match: "Es wurden keine Repositorys gefunden, die Ihrem Filter entsprechen",
    	sort_by: "Sortiere nach",
    	title: "Repository hinzufügen"
    };
    var dialog_custom_repositories$1 = {
    	category: "Kategorie",
    	no_category: "Fehlende Kategorie",
    	no_repository: "Fehlendes Repository",
    	title: "Benutzerdefinierte Repositories",
    	url_placeholder: "Fügen Sie eine benutzerdefinierte Repository-URL hinzu"
    };
    var dialog_info$1 = {
    	author: "Autor",
    	downloads: "Downloads",
    	install: "Installieren Sie dieses Repository in HACS",
    	loading: "Informationen laden ...",
    	no_info: "Der Entwickler hat keine weiteren Informationen für dieses Repository bereitgestellt",
    	open_issues: "Probleme melden",
    	open_repo: "Repository öffnen",
    	stars: "Sterne",
    	version_installed: "Version installiert"
    };
    var dialog_install$1 = {
    	restart: "Denken Sie daran, dass Sie Home Assistant neu starten müssen, bevor Änderungen an Integrationen (custom_components) angewendet werden.",
    	select_version: "Version auswählen",
    	show_beta: "Beta-Versionen anzeigen",
    	type: "Typ",
    	url: "URL"
    };
    var dialog_update$1 = {
    	available_version: "Verfügbare Version",
    	changelog: "Änderungsprotokoll",
    	installed_version: "Installierte Version",
    	releasenotes: "Releasenotes für {release}",
    	title: "Update ausstehend"
    };
    var entry$1 = {
    	information: "Information",
    	intro: "Aktualisierungen und wichtige Meldungen werden hier angezeigt, falls vorhanden",
    	messages: {
    		disabled: {
    			content: "Überprüfe die Logdatei für weitere Informationen",
    			title: "HACS ist deaktiviert"
    		},
    		has_pending_tasks: {
    			content: "Einige Repositorys werden möglicherweise erst angezeigt, wenn dies abgeschlossen ist",
    			title: "Hintergrundaufgaben stehen noch aus"
    		},
    		resources: {
    			content: "Sie haben {number} Lovelace-Elemente, die in Lovelace nicht richtig geladen sind.",
    			title: "Nicht in Lovelace geladen"
    		},
    		restart: {
    			content: "Sie haben {number} -Integrationen, die einen Neustart von Home Assistant erfordern. Dies können Sie im Abschnitt 'Server Controls' im Konfigurationsteil der Home Assistant-Benutzeroberfläche tun.",
    			title: "Ausstehender Neustart"
    		},
    		startup: {
    			content: "HACS wird gestartet, während dieser Zeit könnten einige Informationen fehlen oder falsch sein",
    			title: "HACS startet"
    		},
    		wrong_frontend_installed: {
    			content: "Sie haben {running} des HACS-Frontends installiert, aber Version {expected} wurde erwartet. Wenn diese Meldung angezeigt wird, dass Home Assistant die neue Version nicht installieren konnte, starten Sie Home Assistant neu.",
    			title: "Unerwartete Frontend-Version"
    		},
    		wrong_frontend_loaded: {
    			content: "Sie führen die Version {running} des HACS-Frontends aus, aber es wurde die Version {expected} erwartet. Sie sollten Ihren Browser-Cache leeren.",
    			title: "Unerwartete Frontend-Version"
    		}
    	},
    	pending_updates: "Ausstehende Updates"
    };
    var menu$1 = {
    	about: "Über HACS",
    	clear: "Alles neue als gesehen markieren",
    	custom_repositories: "Benutzerdefinierte Repositories",
    	dismiss: "Alle neuen Repositories ausblenden",
    	documentation: "Dokumentation",
    	open_issue: "Problem melden",
    	reload: "Fenster neu laden"
    };
    var options$1 = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "AppDaemon App-Entdeckung & Nachverfolgung aktivieren",
    				country: "Nach Ländercode filtern.",
    				debug: "Debug aktivieren.",
    				experimental: "Experimentelle Funktionen aktivieren",
    				netdaemon: "NetDaemon App-Entdeckung & Nachverfolgung aktivieren",
    				not_in_use: "Nicht in Verwendung mit YAML",
    				release_limit: "Anzahl anzuzeigender Releases.",
    				sidepanel_icon: "Sidepanel Symbol",
    				sidepanel_title: "Sidepanel Titel"
    			}
    		}
    	}
    };
    var repository_banner$1 = {
    	config_flow: "Diese Integration unterstützt config_flow. Das bedeutet, dass du jetzt in den Integrationsbereich deiner Benutzeroberfläche wechseln kannst, um es zu konfigurieren.",
    	config_flow_title: "UI-Konfiguration unterstützt",
    	integration_not_loaded: "Diese Integration ist in Home Assistant nicht geladen.",
    	no_restart_required: "Kein Neustart erforderlich",
    	not_loaded: "Nicht geladen",
    	plugin_not_loaded: "Dieses Element wird nicht zu Ihren Lovelace-Ressourcen hinzugefügt.",
    	restart: "Du musst Home Assistant neu starten.",
    	restart_pending: "Neustart ausstehend"
    };
    var repository_card$1 = {
    	dismiss: "Ausblenden",
    	hide: "Verstecken",
    	information: "Information",
    	new_repository: "Neues Repository",
    	not_loaded: "Nicht geladen",
    	open_issue: "Problem melden",
    	open_source: "Quelldateien öffnen",
    	pending_restart: "Ausstehender Neustart",
    	pending_update: "Ausstehende Aktualisierung",
    	reinstall: "Neu installieren",
    	report: "Melden zur Entfernung des Repositorys",
    	update_information: "Aktualisierungsinformationen"
    };
    var repository$1 = {
    	add_to_lovelace: "Zu Lovelace hinzufügen",
    	authors: "Autoren",
    	available: "Verfügbar",
    	back_to: "Zurück zu",
    	changelog: "Änderungsprotokoll",
    	downloads: "Downloads",
    	flag_this: "Melden",
    	frontend_version: "Frontend Version",
    	github_stars: "GitHub Sterne",
    	goto_integrations: "Gehe zu Integrationen",
    	hide: "Verstecken",
    	hide_beta: "Beta verstecken",
    	install: "Installieren",
    	installed: "Installiert",
    	lovelace_copy_example: "Beispiel in die Zwischenablage kopieren",
    	lovelace_instruction: "Zum Hinzufügen zu deinen Lovelace-Einstellungen, benutze Folgendes",
    	lovelace_no_js_type: "Der Typ dieses Elements konnte nicht ermittelt werden. Überprüfen Sie das Repository.",
    	newest: "neueste",
    	note_appdaemon: "du musst es dann noch in die Datei 'apps.yaml' hinzufügen",
    	note_installed: "Wird installiert nach",
    	note_integration: "du musst es dann noch in die Datei 'configuration.yaml' hinzufügen",
    	note_plugin: "du musst es dann noch in deine Lovelace-Einstellungen ('ui-lovelace.yaml' oder im Raw-Konfigurationseditor) hinzufügen",
    	note_plugin_post_107: "Du musst es noch zu deiner Lovelace-Konfiguration hinzufügen ('configuration.yaml' oder der Ressourceneditor '/config/lovelace/resources')",
    	open_issue: "Problem melden",
    	open_plugin: "Element öffnen",
    	reinstall: "Neu installieren",
    	repository: "Repository",
    	restart_home_assistant: "Home Assistant neu starten",
    	show_beta: "Beta anzeigen",
    	uninstall: "Deinstallieren",
    	update_information: "Aktualisierungsinformationen",
    	upgrade: "Aktualisieren"
    };
    var search$1 = {
    	placeholder: "Suche nach Repository"
    };
    var sections$1 = {
    	about: {
    		description: "Informationen zu HACS anzeigen",
    		title: "Über"
    	},
    	automation: {
    		description: "Hier finden Sie python_scripts, AppDaemon-Apps und NetDaemon-Apps",
    		title: "Automatisierung"
    	},
    	frontend: {
    		description: "Hier finden Sie Themen, individuelle Karten und andere Elemente für Lovelace",
    		title: "Frontend"
    	},
    	integrations: {
    		description: "Hier finden Sie benutzerdefinierte Integrationen (custom_components)",
    		title: "Integrationen"
    	},
    	pending_repository_upgrade: "Sie verwenden Version {installed}, Version {available} ist verfügbar"
    };
    var settings$1 = {
    	add_custom_repository: "BENUTZERDEFINIERTES REPOSITORY HINZUFÜGEN",
    	adding_new_repo: "Hinzufügen eines neuen Repository '{repo}'",
    	adding_new_repo_category: "Mit der Kategorie '{category}'.",
    	bg_task_custom: "Custom Repositorys werden ausgeblendet, während Hintergrundaufgaben ausgeführt werden.",
    	category: "Kategorie",
    	compact_mode: "Kompakter Modus",
    	custom_repositories: "BENUTZERDEFINIERTE REPOSITORIES",
    	"delete": "Löschen",
    	display: "Anzeige",
    	grid: "Gitter",
    	hacs_repo: "HACS repo",
    	hidden_repositories: "versteckte Repositories",
    	missing_category: "Du musst eine Kategorie auswählen.",
    	open_repository: "Repository öffnen",
    	reload_data: "Daten neu laden",
    	reload_window: "Fenster neu laden",
    	repository_configuration: "Repository Konfiguration",
    	save: "Speichern",
    	table: "Tabelle",
    	table_view: "Tabellenansicht",
    	unhide: "einblenden",
    	upgrade_all: "Alles aktualisieren"
    };
    var store$1 = {
    	ascending: "Aufsteigend",
    	clear_new: "Alle neuen Repositories als gesehen markieren",
    	descending: "Absteigend",
    	last_updated: "Zuletzt aktualisiert",
    	name: "Name",
    	new_repositories: "Neue Repositories",
    	new_repositories_note: "Hier werden über 10 neue Repositorys angezeigt. Wenn Sie alle als gelesen markieren möchten, klicken Sie auf die 3 Punkte in der oberen rechten Ecke und blenden sie alle aus.",
    	no_repositories: "Keine Repositories",
    	no_repositories_desc1: "Anscheinend sind in diesem Abschnitt noch keine Repositorys installiert.",
    	no_repositories_desc2: "Klicken Sie auf das + in der unteren Ecke, um Ihre ersten hinzuzufügen!",
    	no_repositories_found_desc1: "In diesem Abschnitt wurden keine installierten Repositorys gefunden, die mit \"{searchInput}\" übereinstimmen.",
    	no_repositories_found_desc2: "Versuchen Sie, nach etwas anderem zu suchen!",
    	pending_upgrades: "Ausstehende Upgrades",
    	placeholder_search: "Suchbegriff eingeben…",
    	sort: "Sortieren",
    	stars: "Sterne",
    	status: "Status"
    };
    var time$1 = {
    	ago: "vor",
    	day: "Tag",
    	days: "Tage",
    	hour: "Stunde",
    	hours: "Stunden",
    	minute: "Minute",
    	minutes: "Minuten",
    	month: "Monat",
    	months: "Monate",
    	one: "Eins",
    	one_day_ago: "vor einem Tag",
    	one_hour_ago: "vor einer Stunde",
    	one_minute_ago: "vor einer Minute",
    	one_month_ago: "vor einem Monat",
    	one_second_ago: "vor einer Sekunde",
    	one_year_ago: "vor einem Jahr",
    	second: "Sekunde",
    	seconds: "Sekunden",
    	x_days_ago: "vor {x} Tagen",
    	x_hours_ago: "vor {x} Stunden",
    	x_minutes_ago: "vor {x} Minuten",
    	x_months_ago: "vor {x} Monaten",
    	x_seconds_ago: "vor {x} Sekunden",
    	x_years_ago: "vor {x} Jahren",
    	year: "Jahr",
    	years: "Jahre"
    };
    var de = {
    	common: common$1,
    	config: config$1,
    	confirm: confirm$1,
    	dialog_about: dialog_about$1,
    	dialog_add_repo: dialog_add_repo$1,
    	dialog_custom_repositories: dialog_custom_repositories$1,
    	dialog_info: dialog_info$1,
    	dialog_install: dialog_install$1,
    	dialog_update: dialog_update$1,
    	entry: entry$1,
    	menu: menu$1,
    	options: options$1,
    	repository_banner: repository_banner$1,
    	repository_card: repository_card$1,
    	repository: repository$1,
    	search: search$1,
    	sections: sections$1,
    	settings: settings$1,
    	store: store$1,
    	time: time$1
    };

    var de$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$1,
        config: config$1,
        confirm: confirm$1,
        dialog_about: dialog_about$1,
        dialog_add_repo: dialog_add_repo$1,
        dialog_custom_repositories: dialog_custom_repositories$1,
        dialog_info: dialog_info$1,
        dialog_install: dialog_install$1,
        dialog_update: dialog_update$1,
        entry: entry$1,
        menu: menu$1,
        options: options$1,
        repository_banner: repository_banner$1,
        repository_card: repository_card$1,
        repository: repository$1,
        search: search$1,
        sections: sections$1,
        settings: settings$1,
        store: store$1,
        time: time$1,
        'default': de
    });

    var common$2 = {
    	about: "Σχετικά με",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "AppDaemon Apps",
    	background_task: "Τρέχει μια διεργασία στο παρασκήνιο, η σελίδα θα ανανεωθεί μόλις αυτό ολοκληρωθεί.",
    	check_log_file: "Ελέγξτε το αρχείο καταγραφής για περισσότερες λεπτομέρειες.",
    	"continue": "Να συνεχίσει",
    	disabled: "Απενεργοποιημένο",
    	documentation: "Τεκμηρίωση",
    	hacs_is_disabled: "Το HACS είναι απενεργοποιημένο",
    	installed: "εγκατεστημένο",
    	integration: "Ενσωμάτωση",
    	integrations: "Ενσωματωμένα",
    	manage: "διαχειρίζονται",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDaemon Apps",
    	plugin: "Lovelace",
    	plugins: "Πρόσθετα",
    	python_script: "Πρόγραμμα Python",
    	python_scripts: "Προγράμματα Python",
    	repositories: "Αποθετήρια",
    	settings: "ρυθμίσεις",
    	theme: "Θέμα",
    	themes: "Θέματα",
    	version: "Έκδοση"
    };
    var config$2 = {
    	abort: {
    		single_instance_allowed: "Μονάχα μία ρύθμιση των παραμέτρων του HACS επιτρέπεται."
    	},
    	error: {
    		auth: "Το διακριτικό πρόσβασης δεν είναι σωστό."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Ενεργοποίηση εύρεσης & παρακολούθησης για το AppDaemon",
    				netdaemon: "Ενεργοποίηση εύρεσης & παρακολούθησης για το NetDaemon",
    				python_script: "Ενεργοποίηση εύρεσης & παρακολούθησης για τα python_scripts",
    				sidepanel_icon: "Εικονίδιο πλαϊνού πάνελ",
    				sidepanel_title: "Τίτλος πλαϊνού πάνελ",
    				theme: "Ενεργοποίηση εύρεσης & παρακολούθησης για τα θεμάτων",
    				token: "Διακριτικό πρόσβασης του GitHub"
    			},
    			description: "Εάν χρειαστείτε βοήθεια με τις ρυθμίσεις ανατρέξτε εδώ: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$2 = {
    	add_to_lovelace: "Είστε βέβαιοι ότι θέλετε να προσθέσετε αυτό στους πόρους του Lovelace;",
    	bg_task: "Η ενέργεια είναι απενεργοποιημένη όσο εκτελούνται εργασίες στο παρασκήνιο",
    	cancel: "Ακύρωση",
    	"continue": "Είστε βέβαιοι ότι θέλετε να συνεχίσετε;",
    	"delete": "Είστε σίγουροι ότι θέλετε να διαγράψετε το '{item}';",
    	delete_installed: "Το '{item}' είναι εγκατεστημένο, πρέπει να το απεγκαταστήσετε πριν να το διαγράψετε.",
    	exist: "{item} υπάρχει ήδη",
    	generic: "Είστε βέβαιοι;",
    	home_assistant_is_restarting: "Περιμένετε, το Home Assistant επανεκκινείται τώρα.",
    	no: "Οχι",
    	no_upgrades: "Δεν υπάρχουν αναβαθμίσεις σε εκκρεμότητα",
    	ok: "Εντάξει",
    	overwrite: "Αυτό θα το αντικαταστήσει.",
    	restart_home_assistant: "Είστε βέβαιοι ότι θέλετε να κάνετε επανεκκίνηση του Home Assistant;",
    	uninstall: "Είστε βέβαιοι ότι θέλετε να απεγκαταστήσετε το '{item}';",
    	upgrade_all: "Αυτό θα αναβαθμίσει όλα αυτά τα αποθετήρια, βεβαιωθείτε ότι έχετε διαβάσει τις σημειώσεις έκδοσης για όλα πριν προχωρήσετε.",
    	yes: "Ναι"
    };
    var options$2 = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Ενεργοποίηση εύρεσης & παρακολούθησης για το AppDaemon",
    				country: "Κριτήριο με βάση τον κωδικό χώρας.",
    				debug: "Ενεργοποίηση εντοπισμού σφαλμάτων.",
    				experimental: "Ενεργοποίση πειραματικών λειτουργιών",
    				netdaemon: "Ενεργοποίηση εύρεσης & παρακολούθησης για το NetDaemon",
    				not_in_use: "Δεν χρησιμοποιείται με το YAML",
    				release_limit: "Αριθμός εκδόσεων που να παραθέτονται.",
    				sidepanel_icon: "Εικονίδιο πλαϊνού πάνελ",
    				sidepanel_title: "Τίτλος πλαϊνού πάνελ"
    			}
    		}
    	}
    };
    var repository_banner$2 = {
    	integration_not_loaded: "Αυτή η ενσωμάτωση δεν φορτώθηκε στο Home Assistant.",
    	no_restart_required: "Δεν απαιτείται επανεκκίνηση",
    	not_loaded: "Δεν έχει φορτωθεί",
    	plugin_not_loaded: "Αυτό το πρόσθετο δεν προστέθηκε στους πόρους του Lovelace.",
    	restart: "Πρέπει να κάνετε επανεκκίνηση του Home Assistant.",
    	restart_pending: "Επανεκκίνηση σε εκκρεμότητα"
    };
    var repository$2 = {
    	add_to_lovelace: "Προσθήκη στο Lovelace",
    	authors: "Συγγραφείς",
    	available: "Διαθέσιμο",
    	back_to: "Πίσω στο",
    	changelog: "Σημειώσεις των αλλαγών",
    	downloads: "Λήψεις",
    	flag_this: "Σημείωσε αυτό",
    	frontend_version: "Έκδοση Frontend",
    	github_stars: "GitHub αστέρια",
    	goto_integrations: "Μετάβαση στις ενσωματώσεις",
    	hide: "Απόκρυψη",
    	hide_beta: "Απόκριση του beta",
    	install: "Εγκατάσταση",
    	installed: "Εγκατεστημένο",
    	lovelace_copy_example: "Αντίγραψε το παράδειγμα στο πρόχειρο",
    	lovelace_instruction: "Όταν το προσθέσετε στις ρυθμίσεις του lovelace χρησιμοποιήστε τούτο",
    	lovelace_no_js_type: "Δεν καταφέραμε να προσδιορίσουμε τον τύπο αυτού του προσθέτου, ελέξτε το αποθετήριο.",
    	newest: "νεότερο",
    	note_appdaemon: "εξακολουθεί να χρειάζεται να το προσθέσετε στο αρχείο 'apps.yaml'",
    	note_installed: "Όταν εγκατασταθεί, θα προστεθεί στο",
    	note_integration: "εξακολουθεί να χρειάζεται να το προσθέσετε στο αρχείο 'configuration.yaml'",
    	note_plugin: "εξακολουθεί να χρειάζετε να το προσθέσετε στις ρυθμίσεις του lovelace ('ui-lovelace.yaml' ή μέσω του γραφικού επεξεργαστή των ρυθμίσεων)",
    	open_issue: "Εκκρεμόν ζήτημα",
    	open_plugin: "Ανοιχτό πρόσθετο",
    	reinstall: "Επανεγκατάσταση",
    	repository: "Αποθετήριο",
    	restart_home_assistant: "Επανεκκίνηση του Home Assistant",
    	show_beta: "Εμφάνιση του beta",
    	uninstall: "Απεγκατάσταση",
    	update_information: "Ενημέρωση πληροφοριών",
    	upgrade: "Ενημέρωση"
    };
    var settings$2 = {
    	add_custom_repository: "ΠΡΟΣΘΕΣΤΕ ΕΝΑ ΕΙΔΙΚΟ ΑΠΟΘΕΤΗΡΙΟ",
    	adding_new_repo: "Προσθήκη νέου αποθετηρίου '{repo}'",
    	adding_new_repo_category: "Με κατηγορία '{category}'.",
    	category: "Κατηγορία",
    	compact_mode: "Συμπαγής λειτουργία",
    	custom_repositories: "ΕΙΔΙΚΑ ΑΠΟΘΕΤΗΡΙΑ",
    	"delete": "Διαγραφή",
    	display: "Εμφάνιση",
    	grid: "Πλέγμα",
    	hacs_repo: "Αποθετήριο του HACS",
    	hidden_repositories: "κρυφά αποθετήρια",
    	missing_category: "Πρέπει να επιλέξετε μια κατηγορία",
    	open_repository: "Ανοίξτε το αποθετήριο",
    	reload_data: "Επαναφόρτωση δεδομένων",
    	reload_window: "Επαναφόρτωση του παραθύρου",
    	repository_configuration: "Διαμόρφωση αποθετηρίου",
    	save: "Αποθήκευση",
    	table: "Πίνακας",
    	table_view: "Προβολή πίνακα",
    	unhide: "αποκρύψω",
    	upgrade_all: "Αναβάθμιση όλων"
    };
    var store$2 = {
    	ascending: "αύξουσα",
    	clear_new: "Απαλοιφή όλων των νέων αποθετηρίων",
    	descending: "φθίνουσα",
    	last_updated: "Τελευταία ενημέρωση",
    	name: "Ονομα",
    	new_repositories: "Νέα αποθετήρια",
    	pending_upgrades: "Εκκρεμείς αναβαθμίσεις",
    	placeholder_search: "Παρακαλώ πληκτρολογήστε έναν όρο προς αναζήτηση...",
    	sort: "είδος",
    	stars: "Αστέρια",
    	status: "Κατάσταση"
    };
    var time$2 = {
    	ago: "πριν",
    	day: "ημέρα",
    	days: "ημέρες",
    	hour: "ώρα",
    	hours: "ώρες",
    	minute: "λεπτό",
    	minutes: "λεπτά",
    	month: "μήνας",
    	months: "μήνες",
    	one: "Ένα",
    	one_day_ago: "πριν από μία ημέρα",
    	one_hour_ago: "πριν από μια ώρα",
    	one_minute_ago: "πριν από ένα λεπτό",
    	one_month_ago: "πριν από ένα μήνα",
    	one_second_ago: "πριν από ένα δευτερόλεπτο",
    	one_year_ago: "πριν από ένα χρόνο",
    	second: "δευτερόλεπτο",
    	seconds: "δευτερόλεπτα",
    	x_days_ago: "{x} ημέρες πριν",
    	x_hours_ago: "{x} ώρες πριν",
    	x_minutes_ago: "{x} λεπτά πριν",
    	x_months_ago: "{x} μήνες πριν",
    	x_seconds_ago: "{x} δευτερόλεπτα πριν",
    	x_years_ago: "{x} χρόνια πριν",
    	year: "έτος",
    	years: "χρόνια"
    };
    var el = {
    	common: common$2,
    	config: config$2,
    	confirm: confirm$2,
    	options: options$2,
    	repository_banner: repository_banner$2,
    	repository: repository$2,
    	settings: settings$2,
    	store: store$2,
    	time: time$2
    };

    var el$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$2,
        config: config$2,
        confirm: confirm$2,
        options: options$2,
        repository_banner: repository_banner$2,
        repository: repository$2,
        settings: settings$2,
        store: store$2,
        time: time$2,
        'default': el
    });

    var common$3 = {
    	about: "About",
    	add: "add",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "AppDaemon Apps",
    	appdaemon_plural: "AppDaemon Apps",
    	background_task: "Background task running, this page will reload when it's done.",
    	check_log_file: "Check your log file for more details.",
    	"continue": "Continue",
    	disabled: "Disabled",
    	documentation: "Documentation",
    	element: "element",
    	hacs_is_disabled: "HACS is disabled",
    	install: "Install",
    	installed: "installed",
    	integration: "Integration",
    	integration_plural: "Integrations",
    	integrations: "Integrations",
    	lovelace: "Lovelace",
    	lovelace_element: "Lovelace element",
    	lovelace_elements: "Lovelace elements",
    	manage: "manage",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDaemon Apps",
    	netdaemon_plural: "NetDaemon Apps",
    	plugin: "Lovelace",
    	plugin_plural: "Lovelace elements",
    	plugins: "Lovelace elements",
    	python_script: "Python Script",
    	python_script_plural: "Python Scripts",
    	python_scripts: "Python Scripts",
    	repositories: "Repositories",
    	repository: "Repository",
    	settings: "settings",
    	theme: "Theme",
    	theme_plural: "Themes",
    	themes: "Themes",
    	uninstall: "Uninstall",
    	update: "Update",
    	version: "Version"
    };
    var config$3 = {
    	abort: {
    		single_instance_allowed: "Only a single configuration of HACS is allowed."
    	},
    	error: {
    		auth: "Personal Access Token is not correct."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Enable AppDaemon apps discovery & tracking",
    				netdaemon: "Enable NetDaemon apps discovery & tracking",
    				python_script: "Enable python_scripts discovery & tracking",
    				sidepanel_icon: "Side panel icon",
    				sidepanel_title: "Side panel title",
    				theme: "Enable Themes discovery & tracking",
    				token: "GitHub Personal Access Token"
    			},
    			description: "If you need help with the configuration have a look here: https://hacs.xyz/docs/configuration/start/",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$3 = {
    	add_to_lovelace: "Are you sure you want to add this to your Lovelace resources?",
    	bg_task: "Action is disabled while background tasks is running.",
    	cancel: "Cancel",
    	"continue": "Are you sure you want to continue?",
    	"delete": "Are you sure you want to delete \"{item}\"?",
    	delete_installed: "'{item}' is installed, you need to uninstall it before you can delete it.",
    	exist: "{item} already exists",
    	generic: "Are you sure?",
    	home_assistant_is_restarting: "Hold on, Home Assistant is now restarting.",
    	home_assistant_version_not_correct: "You are running Home Assistant version '{haversion}', but this repository requires minimum version '{minversion}' to be installed.",
    	no: "No",
    	no_upgrades: "No upgrades pending",
    	ok: "OK",
    	overwrite: "Doing this will overwrite it.",
    	reload_data: "This reloads the data of all repositories HACS knows about, this will take some time to finish.",
    	restart_home_assistant: "Are you sure you want to restart Home Assistant?",
    	uninstall: "Are you sure you want to uninstall \"{item}\"?",
    	upgrade_all: "This will upgrade all of these repositories, make sure that you have read the release notes for all of them before proceeding.",
    	yes: "Yes"
    };
    var dialog_about$2 = {
    	frontend_version: "Frontend version",
    	installed_repositories: "Installed repositories",
    	integration_version: "Integration version",
    	useful_links: "Useful links"
    };
    var dialog_add_repo$2 = {
    	limit: "Only the first 100 repositories are shown, use the search to filter what you need",
    	no_match: "No repositories found matching your filter",
    	sort_by: "Sort by",
    	title: "Add repository"
    };
    var dialog_custom_repositories$2 = {
    	category: "Category",
    	no_category: "Missing category",
    	no_repository: "Missing repository",
    	title: "Custom repositories",
    	url_placeholder: "Add custom repository URL"
    };
    var dialog_info$2 = {
    	author: "Author",
    	downloads: "Downloads",
    	install: "Install this repository in HACS",
    	loading: "Loading information...",
    	no_info: "The developer has not provided any more information for this repository",
    	open_issues: "Open issues",
    	open_repo: "Open repository",
    	stars: "Stars",
    	version_installed: "Version installed"
    };
    var dialog_install$2 = {
    	restart: "Remember that you need to restart Home Assistant before changes to integrations (custom_components) are applied.",
    	select_version: "Select version",
    	show_beta: "Show beta versions",
    	type: "Type",
    	url: "URL"
    };
    var dialog_update$2 = {
    	available_version: "Available version",
    	changelog: "Changelog",
    	installed_version: "Installed version",
    	releasenotes: "Releasenotes for {release}",
    	title: "Update pending"
    };
    var entry$2 = {
    	information: "Information",
    	intro: "Updates and important messages will show here if there are any",
    	messages: {
    		disabled: {
    			content: "Check your log file for more details",
    			title: "HACS is disabled"
    		},
    		has_pending_tasks: {
    			content: "Some repositories might not show untill this is completed",
    			title: "Background tasks pending"
    		},
    		resources: {
    			content: "You have {number} Lovelace elements that are not loaded properly in Lovelace.",
    			title: "Not loaded in Lovelace"
    		},
    		restart: {
    			content: "You have {number} integrations that requires a restart of Home Assistant, you can do that from the 'Server Controls' section under the configuration part of Home Assistant UI.",
    			title: "Pending restart"
    		},
    		startup: {
    			content: "HACS is starting up, during this time some information might be missing or incorrect",
    			title: "HACS is starting up"
    		},
    		wrong_frontend_installed: {
    			content: "You have {running} of the HACS frontend installed, but version {expected} was expected, if this you see this message Home Assistant was not able to install the new version, try restarting Home Assistant.",
    			title: "Unexpected frontend version"
    		},
    		wrong_frontend_loaded: {
    			content: "You are running version {running} of the HACS frontend, but version {expected} was expected, you should clear your browser cache.",
    			title: "Unexpected frontend version"
    		}
    	},
    	pending_updates: "Pending updates"
    };
    var menu$2 = {
    	about: "About HACS",
    	clear: "Clear all new",
    	custom_repositories: "Custom repositories",
    	dismiss: "Dismiss all new repositories",
    	documentation: "Documentation",
    	open_issue: "Open issue",
    	reload: "Reload window"
    };
    var options$3 = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Enable AppDaemon apps discovery & tracking",
    				country: "Filter with country code.",
    				debug: "Enable debug.",
    				experimental: "Enable experimental features",
    				netdaemon: "Enable NetDaemon apps discovery & tracking",
    				not_in_use: "Not in use with YAML",
    				release_limit: "Number of releases to show.",
    				sidepanel_icon: "Side panel icon",
    				sidepanel_title: "Side panel title"
    			}
    		}
    	}
    };
    var repository_banner$3 = {
    	config_flow: "This integration supports config_flow, that means that you now can go to the integration section of your UI to configure it.",
    	config_flow_title: "UI Configuration supported",
    	integration_not_loaded: "This integration is not loaded in Home Assistant.",
    	no_restart_required: "No restart required",
    	not_loaded: "Not loaded",
    	plugin_not_loaded: "This plugin is not added to your Lovelace resources.",
    	restart: "You need to restart Home Assistant.",
    	restart_pending: "Restart pending"
    };
    var repository_card$2 = {
    	dismiss: "dismiss",
    	hide: "Hide",
    	information: "Information",
    	new_repository: "New repository",
    	not_loaded: "Not loaded",
    	open_issue: "Open issue",
    	open_source: "Open source",
    	pending_restart: "Pending restart",
    	pending_update: "Pending update",
    	reinstall: "Reinstall",
    	report: "Report for removal",
    	update_information: "Update information"
    };
    var repository$3 = {
    	add_to_lovelace: "Add to Lovelace",
    	authors: "Authors",
    	available: "Available",
    	back_to: "Back to",
    	changelog: "Change log",
    	downloads: "Downloads",
    	flag_this: "Flag this",
    	frontend_version: "Frontend version",
    	github_stars: "GitHub stars",
    	goto_integrations: "Go to integrations",
    	hide: "Hide",
    	hide_beta: "Hide beta",
    	install: "Install",
    	installed: "Installed",
    	lovelace_copy_example: "Copy the example to your clipboard",
    	lovelace_instruction: "When you add this to your lovelace configuration use this",
    	lovelace_no_js_type: "Could not determine the type of this plugin, check the repository.",
    	newest: "newest",
    	note_appdaemon: "you still need to add it to your 'apps.yaml' file",
    	note_installed: "When installed, this will be located in",
    	note_integration: "you still need to add it to your 'configuration.yaml' file",
    	note_plugin: "you still need to add it to your lovelace configuration ('ui-lovelace.yaml' or the raw UI config editor)",
    	note_plugin_post_107: "you still need to add it to your lovelace configuration ('configuration.yaml' or the resource editor '/config/lovelace/resources')",
    	open_issue: "Open issue",
    	open_plugin: "Open element",
    	reinstall: "Reinstall",
    	repository: "Repository",
    	restart_home_assistant: "Restart Home Assistant",
    	show_beta: "Show beta",
    	uninstall: "Uninstall",
    	update_information: "Update information",
    	upgrade: "Update"
    };
    var search$2 = {
    	placeholder: "Search for repository"
    };
    var sections$2 = {
    	about: {
    		description: "Show information about HACS",
    		title: "About"
    	},
    	automation: {
    		description: "This is where you find python_scripts, AppDaemon apps and NetDaemon apps",
    		title: "Automation"
    	},
    	frontend: {
    		description: "This is where you find themes, custom cards and other elements for lovelace",
    		title: "Frontend"
    	},
    	integrations: {
    		description: "This is where you find custom integrations (custom_components)",
    		title: "Integrations"
    	},
    	pending_repository_upgrade: "You are running version {installed}, version {available} is available"
    };
    var settings$3 = {
    	add_custom_repository: "ADD CUSTOM REPOSITORY",
    	adding_new_repo: "Adding new repository '{repo}'",
    	adding_new_repo_category: "With category '{category}'.",
    	bg_task_custom: "Custom repositories are hidden while background tasks is running.",
    	category: "Category",
    	compact_mode: "Compact mode",
    	custom_repositories: "CUSTOM REPOSITORIES",
    	"delete": "Delete",
    	display: "Display",
    	grid: "Grid",
    	hacs_repo: "HACS repo",
    	hidden_repositories: "hidden repositories",
    	missing_category: "You need to select a category",
    	open_repository: "Open repository",
    	reload_data: "Reload data",
    	reload_window: "Reload window",
    	repository_configuration: "Repository configuration",
    	save: "Save",
    	table: "Table",
    	table_view: "Table view",
    	unhide: "unhide",
    	upgrade_all: "Upgrade all"
    };
    var store$3 = {
    	ascending: "ascending",
    	clear_new: "Clear all new repositories",
    	descending: "descending",
    	last_updated: "Last updated",
    	name: "Name",
    	new_repositories: "New Repositories",
    	new_repositories_note: "You have over 10 new repositories showing here, if you want to clear them all click the 3 dots in the top right corner and dismiss all of them.",
    	no_repositories: "No repositories",
    	no_repositories_desc1: "It seems like you don't have any repositories installed in this section yet.",
    	no_repositories_desc2: "Click on the + in the bottom corner to add your first!",
    	no_repositories_found_desc1: "No installed repositories matching \"{searchInput}\" found in this section.",
    	no_repositories_found_desc2: "Try searching for something else!",
    	pending_upgrades: "Pending upgrades",
    	placeholder_search: "Please enter a search term...",
    	sort: "sort",
    	stars: "Stars",
    	status: "Status"
    };
    var time$3 = {
    	ago: "ago",
    	day: "day",
    	days: "days",
    	hour: "hour",
    	hours: "hours",
    	minute: "minute",
    	minutes: "minutes",
    	month: "month",
    	months: "months",
    	one: "One",
    	one_day_ago: "one day ago",
    	one_hour_ago: "one hour ago",
    	one_minute_ago: "one minute ago",
    	one_month_ago: "one month ago",
    	one_second_ago: "one second ago",
    	one_year_ago: "one year ago",
    	second: "second",
    	seconds: "seconds",
    	x_days_ago: "{x} days ago",
    	x_hours_ago: "{x} hours ago",
    	x_minutes_ago: "{x} minutes ago",
    	x_months_ago: "{x} months ago",
    	x_seconds_ago: "{x} seconds ago",
    	x_years_ago: "{x} years ago",
    	year: "year",
    	years: "years"
    };
    var en = {
    	common: common$3,
    	config: config$3,
    	confirm: confirm$3,
    	dialog_about: dialog_about$2,
    	dialog_add_repo: dialog_add_repo$2,
    	dialog_custom_repositories: dialog_custom_repositories$2,
    	dialog_info: dialog_info$2,
    	dialog_install: dialog_install$2,
    	dialog_update: dialog_update$2,
    	entry: entry$2,
    	menu: menu$2,
    	options: options$3,
    	repository_banner: repository_banner$3,
    	repository_card: repository_card$2,
    	repository: repository$3,
    	search: search$2,
    	sections: sections$2,
    	settings: settings$3,
    	store: store$3,
    	time: time$3
    };

    var en$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$3,
        config: config$3,
        confirm: confirm$3,
        dialog_about: dialog_about$2,
        dialog_add_repo: dialog_add_repo$2,
        dialog_custom_repositories: dialog_custom_repositories$2,
        dialog_info: dialog_info$2,
        dialog_install: dialog_install$2,
        dialog_update: dialog_update$2,
        entry: entry$2,
        menu: menu$2,
        options: options$3,
        repository_banner: repository_banner$3,
        repository_card: repository_card$2,
        repository: repository$3,
        search: search$2,
        sections: sections$2,
        settings: settings$3,
        store: store$3,
        time: time$3,
        'default': en
    });

    var common$4 = {
    	about: "Acerca de",
    	add: "añadir",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "AppDaemon Apps",
    	appdaemon_plural: "AppDaemon Apps",
    	background_task: "Ejecutando tareas en segundo plano. Se refrescará automaticamente esta página al finalizar.",
    	check_log_file: "Compruebe el archivo de registro para obtener más detalles.",
    	"continue": "Continuar",
    	disabled: "Deshabilitado",
    	documentation: "Documentación",
    	element: "elemento",
    	hacs_is_disabled: "HACS está deshabilitado",
    	install: "Instalar",
    	installed: "instalado",
    	integration: "Integración",
    	integration_plural: "Integraciones",
    	integrations: "Integraciones",
    	lovelace: "Lovelace",
    	lovelace_element: "Elemento de Lovelace",
    	lovelace_elements: "Elementos de Lovelace",
    	manage: "Administrar",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDaemon Apps",
    	netdaemon_plural: "Aplicaciones NetDaemon",
    	plugin: "Lovelace",
    	plugin_plural: "Elementos de Lovelace",
    	plugins: "Elementos de Lovelace",
    	python_script: "Python Script",
    	python_script_plural: "Python Scripts",
    	python_scripts: "Python Scripts",
    	repositories: "Repositorios",
    	repository: "Repositorio",
    	settings: "configuraciones",
    	theme: "Tema",
    	theme_plural: "Temas",
    	themes: "Temas",
    	uninstall: "Desinstalar",
    	update: "Actualizar",
    	version: "Versión"
    };
    var config$4 = {
    	abort: {
    		single_instance_allowed: "Sólo se permite una única configuración de HACS."
    	},
    	error: {
    		auth: "El token de acceso personal no es correcto."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Habilitar el descubrimiento y seguimiento de las aplicaciones de AppDaemon",
    				netdaemon: "Habilitar el descubrimiento y seguimiento de las aplicaciones de NetDaemon",
    				python_script: "Habilitar el descubrimiento y seguimiento en python_scripts",
    				sidepanel_icon: "Ícono del panel lateral",
    				sidepanel_title: "Título del panel lateral",
    				theme: "Habilitar el descubrimiento y seguimiento de temas",
    				token: "Token de acceso personal de GitHub"
    			},
    			description: "Si necesitas ayuda con la configuración, visita la siguiente pagina: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$4 = {
    	add_to_lovelace: "¿Está seguro de que desea agregar esto a sus recursos de Lovelace?",
    	bg_task: "La acción está deshabilitada mientras se ejecutan tareas en segundo plano.",
    	cancel: "Cancelar",
    	"continue": "Estás seguro de que quieres continuar?",
    	"delete": "¿Seguro que quieres eliminar '{item}'?",
    	delete_installed: "'{item}' está instalado, debe desinstalarlo antes de poder eliminarlo.",
    	exist: "{item} ya existe",
    	generic: "¿Estás seguro?",
    	home_assistant_is_restarting: "Espera, Home Assistant se está reiniciando.",
    	home_assistant_version_not_correct: "Está ejecutando la versión '{haversion}' de Home Assistant, pero este repositorio requiere la instalación de la versión '{minversion}' mínima.",
    	no: "No",
    	no_upgrades: "No hay actualizaciones pendientes",
    	ok: "OK",
    	overwrite: "Si haces esto, se sobrescribirá.",
    	reload_data: "Esto recarga los datos de todos los repositorios que HACS conoce, esto tardará algún tiempo en finalizar.",
    	restart_home_assistant: "¿Está seguro de que desea reiniciar Home Assistant?",
    	uninstall: "¿Está seguro de que deseas desinstalar '{item}'?",
    	upgrade_all: "Esto actualizará todos estos repositorios, asegúrese de que ha leído las notas de la versión de todos ellos antes de continuar.",
    	yes: "Si"
    };
    var dialog_about$3 = {
    	frontend_version: "Versión del front-end",
    	installed_repositories: "Repositorios instalados",
    	integration_version: "Versión de la integración",
    	useful_links: "Enlaces útiles"
    };
    var dialog_add_repo$3 = {
    	limit: "Sólo se muestran los primeros 100 repositorios, utilice la búsqueda para filtrar lo que necesita",
    	no_match: "No se han encontrado repositorios que coincidan con el filtro",
    	sort_by: "Ordenar por",
    	title: "Añadir repositorio"
    };
    var dialog_custom_repositories$3 = {
    	category: "Categoría",
    	no_category: "Categoría que falta",
    	no_repository: "Falta el repositorio",
    	title: "Repositorios personalizados",
    	url_placeholder: "Agregar URL del repositorio personalizado"
    };
    var dialog_info$3 = {
    	author: "Autor",
    	downloads: "Descargas",
    	install: "Instalar este repositorio en HACS",
    	loading: "Cargando información ...",
    	no_info: "El desarrollador no ha proporcionado más información para este repositorio",
    	open_issues: "Abrir issue",
    	open_repo: "Abrir repositorio",
    	stars: "Estrellas",
    	version_installed: "Versión instalada"
    };
    var dialog_install$3 = {
    	restart: "Recuerde que debe reiniciar Home Assistant antes de aplicar los cambios en las integraciones (custom_components).",
    	select_version: "Seleccione la versión",
    	show_beta: "Mostrar versiones beta",
    	type: "Tipo",
    	url: "URL"
    };
    var dialog_update$3 = {
    	available_version: "Versión disponible",
    	changelog: "Registro de cambios",
    	installed_version: "Versión instalada",
    	releasenotes: "Notas de lanzamiento para {release}",
    	title: "Actualización pendiente"
    };
    var entry$3 = {
    	information: "Información",
    	intro: "Las actualizaciones y los mensajes importantes se mostrarán aquí si hay alguno que mostrar",
    	messages: {
    		disabled: {
    			content: "Compruebe el archivo de registro para obtener más detalles",
    			title: "HACS está deshabilitado"
    		},
    		has_pending_tasks: {
    			content: "Es posible que algunos repositorios no se muestren hasta que esto se complete",
    			title: "Tareas en segundo plano pendientes"
    		},
    		resources: {
    			content: "Tienes {number} elementos de Lovelace que no se cargan correctamente en Lovelace.",
    			title: "No está cargada en Lovelace"
    		},
    		restart: {
    			content: "Tienes {número} integraciones que requieren un reinicio de Home Assistant, puedes hacerlo desde la sección 'Controles del Servidor' en la parte de configuración de la UI de Home Assistant.",
    			title: "Pendiente de reinicio"
    		},
    		startup: {
    			content: "HACS se está iniciando, durante este tiempo alguna información podría faltar o ser incorrecta",
    			title: "HACS se está iniciando"
    		},
    		wrong_frontend_installed: {
    			content: "Tiene {running} la interfaz de HACS instalada, pero se esperaba la versión {expected} , si aparece este mensaje, Home Assistant no pudo instalar la nueva versión, intente reiniciar Home Assistant.",
    			title: "Versión inesperada de la interfaz"
    		},
    		wrong_frontend_loaded: {
    			content: "Estás ejecutando la versión {running} de la interfaz HACS, pero se esperaba la versión {expected} , deberías de limpiar la memoria caché del navegador.",
    			title: "Versión inesperada de la interfaz"
    		}
    	},
    	pending_updates: "Actualizaciones pendientes"
    };
    var menu$3 = {
    	about: "Acerca de HACS",
    	clear: "Borrar todo lo nuevo",
    	custom_repositories: "Repositorios personalizados",
    	dismiss: "Descartar todos los repositorios nuevos",
    	documentation: "Documentación",
    	open_issue: "Abrir issue",
    	reload: "Recargar la ventana"
    };
    var options$4 = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Habilitar el descubrimiento y seguimiento de las aplicaciones de AppDaemon",
    				country: "Filtrar por el código de país.",
    				debug: "Habilitar depuración.",
    				experimental: "Habilitar funciones experimentales",
    				netdaemon: "Habilitar el descubrimiento y seguimiento de las aplicaciones de NetDaemon",
    				not_in_use: "No usarse con YAML",
    				release_limit: "Número de versiones a mostrar.",
    				sidepanel_icon: "Icono del panel lateral",
    				sidepanel_title: "Título del panel lateral"
    			}
    		}
    	}
    };
    var repository_banner$4 = {
    	config_flow: "Esta integración soporta config_flow, lo que significa que ahora puede ir a la sección de integración de su UI para configurarlo.",
    	config_flow_title: "Configuración de UI soportada",
    	integration_not_loaded: "Esta integración no se carga en Home Assistant.",
    	no_restart_required: "No es necesario reiniciar",
    	not_loaded: "No está cargado",
    	plugin_not_loaded: "Este plugin aun no se ha añadido a sus recursos de Lovelace.",
    	restart: "Es necesario reiniciar Home Assistant.",
    	restart_pending: "Reinicio pendiente"
    };
    var repository_card$3 = {
    	dismiss: "descartar",
    	hide: "Ocultar",
    	information: "Información",
    	new_repository: "Nuevo repositorio",
    	not_loaded: "Sin cargar",
    	open_issue: "Abrir issue",
    	open_source: "Código abierto",
    	pending_restart: "Pendiente de reinicio",
    	pending_update: "Actualización pendiente",
    	reinstall: "Reinstalar",
    	report: "Informe para la eliminación",
    	update_information: "Actualizar información"
    };
    var repository$4 = {
    	add_to_lovelace: "Añadir a Lovelace",
    	authors: "Autores",
    	available: "Disponible",
    	back_to: "Volver a",
    	changelog: "Registro de cambios",
    	downloads: "Descargas",
    	flag_this: "Marcar esto",
    	frontend_version: "Versión del frontend",
    	github_stars: "Estrellas de GitHub",
    	goto_integrations: "Ir a integraciones",
    	hide: "Ocultar",
    	hide_beta: "Ocultar beta",
    	install: "Instalar",
    	installed: "Instalado",
    	lovelace_copy_example: "Copiar el ejemplo al clipboard",
    	lovelace_instruction: "Agregue lo siguiente en su configuración de lovelace",
    	lovelace_no_js_type: "No se pudo determinar el tipo de elemento, revise el repositorio.",
    	newest: "más nuevo",
    	note_appdaemon: "deberá agregar esto a su archivo 'apps.yaml'",
    	note_installed: "Cuando esté instalado, se ubicará en",
    	note_integration: "deberá agregar esto a su archivo 'configuration.yaml'",
    	note_plugin: "deberá agregar esto a su configuración de lovelace ('ui-lovelace.yaml' o en el editor UI de lovelace)",
    	note_plugin_post_107: "todavía necesita agregarlo a su configuración de lovelace ('configuration.yaml' o al editor de recursos '/config/lovelace/resources')",
    	open_issue: "Abrir issue",
    	open_plugin: "Abrir elemento",
    	reinstall: "Reinstalar",
    	repository: "Repositorio",
    	restart_home_assistant: "Reiniciar Home Assistant",
    	show_beta: "Mostrar beta",
    	uninstall: "Desinstalar",
    	update_information: "Actualizar información",
    	upgrade: "Actualizar"
    };
    var search$3 = {
    	placeholder: "Buscar repositorio"
    };
    var sections$3 = {
    	about: {
    		description: "Mostrar información sobre HACS",
    		title: "Acerca de"
    	},
    	automation: {
    		description: "Aquí es donde se encuentran python_scripts, aplicaciones AppDaemon y aplicaciones NetDaemon",
    		title: "Automatización"
    	},
    	frontend: {
    		description: "Aquí es donde encontrarás temas, tarjetas personalizadas y otros elementos para lovelace",
    		title: "Interfaz"
    	},
    	integrations: {
    		description: "Aquí es donde se encuentran las integraciones personalizadas (custom_components)",
    		title: "Integraciones"
    	},
    	pending_repository_upgrade: "Está ejecutando la versión {installed}, la versión {available} está disponible"
    };
    var settings$4 = {
    	add_custom_repository: "AGREGAR REPOSITORIO PERSONALIZADO",
    	adding_new_repo: "Añadiendo un nuevo repositorio '{repo}'.",
    	adding_new_repo_category: "Con la categoría '{category}'.",
    	bg_task_custom: "Los repositorios personalizados están ocultos mientras se ejecutan las tareas en segundo plano.",
    	category: "Categoría",
    	compact_mode: "Modo compacto",
    	custom_repositories: "REPOSITORIOS PERSONALIZADOS",
    	"delete": "Eliminar",
    	display: "Mostrar",
    	grid: "Cuadrícula",
    	hacs_repo: "HACS repo",
    	hidden_repositories: "repositorios ocultos",
    	missing_category: "Es necesario seleccionar una categoría",
    	open_repository: "Abrir repositorio",
    	reload_data: "Recargar datos",
    	reload_window: "Recargar ventana",
    	repository_configuration: "Configuración del repositorio",
    	save: "Guardar",
    	table: "Tabla",
    	table_view: "Vista de la tabla",
    	unhide: "mostrar",
    	upgrade_all: "Actualizar todo"
    };
    var store$4 = {
    	ascending: "ascendente",
    	clear_new: "Eliminar la lista los nuevos repositorios",
    	descending: "descendente",
    	last_updated: "Última actualización",
    	name: "Nombre",
    	new_repositories: "Nuevos Repositorios",
    	new_repositories_note: "Tienes más de 10 nuevos repositorios mostrados aquí, si quieres borrarlos todos haz clic en los 3 puntos de la esquina superior derecha y deséchalos todos.",
    	no_repositories: "Sin repositorios",
    	no_repositories_desc1: "Parece que todavía no tiene ningún repositorio instalado en esta sección.",
    	no_repositories_desc2: "Haga clic en el + en la esquina inferior para agregar su primera!",
    	no_repositories_found_desc1: "No se ha encontrado ningún repositorio instalado que coincida con el valor de \"{searchInput}\" en esta sección.",
    	no_repositories_found_desc2: "¡Intenta buscar otra cosa!",
    	pending_upgrades: "Actualizaciones pendientes",
    	placeholder_search: "Por favor escriba una palabra clave de búsqueda...",
    	sort: "ordenar",
    	stars: "Estrellas",
    	status: "Estado"
    };
    var time$4 = {
    	ago: "hace",
    	day: "dia",
    	days: "dias",
    	hour: "hora",
    	hours: "horas",
    	minute: "minuto",
    	minutes: "minutos",
    	month: "mes",
    	months: "meses",
    	one: "Uno",
    	one_day_ago: "hace un día",
    	one_hour_ago: "hace una hora",
    	one_minute_ago: "hace un minuto",
    	one_month_ago: "hace un mes",
    	one_second_ago: "hace un segundo",
    	one_year_ago: "hace un año",
    	second: "segundo",
    	seconds: "segundos",
    	x_days_ago: "hace {x} dias",
    	x_hours_ago: "hace {x} horas",
    	x_minutes_ago: "hace {x} minutos",
    	x_months_ago: "hace {x} meses",
    	x_seconds_ago: "hace {x} segundos",
    	x_years_ago: "hace {x} años",
    	year: "año",
    	years: "años"
    };
    var es = {
    	common: common$4,
    	config: config$4,
    	confirm: confirm$4,
    	dialog_about: dialog_about$3,
    	dialog_add_repo: dialog_add_repo$3,
    	dialog_custom_repositories: dialog_custom_repositories$3,
    	dialog_info: dialog_info$3,
    	dialog_install: dialog_install$3,
    	dialog_update: dialog_update$3,
    	entry: entry$3,
    	menu: menu$3,
    	options: options$4,
    	repository_banner: repository_banner$4,
    	repository_card: repository_card$3,
    	repository: repository$4,
    	search: search$3,
    	sections: sections$3,
    	settings: settings$4,
    	store: store$4,
    	time: time$4
    };

    var es$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$4,
        config: config$4,
        confirm: confirm$4,
        dialog_about: dialog_about$3,
        dialog_add_repo: dialog_add_repo$3,
        dialog_custom_repositories: dialog_custom_repositories$3,
        dialog_info: dialog_info$3,
        dialog_install: dialog_install$3,
        dialog_update: dialog_update$3,
        entry: entry$3,
        menu: menu$3,
        options: options$4,
        repository_banner: repository_banner$4,
        repository_card: repository_card$3,
        repository: repository$4,
        search: search$3,
        sections: sections$3,
        settings: settings$4,
        store: store$4,
        time: time$4,
        'default': es
    });

    var common$5 = {
    	about: "À propos de",
    	add: "ajouter",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "Applications AppDaemon",
    	appdaemon_plural: "AppDaemon Apps",
    	background_task: "Tache de fond en cours, cette page se rechargera une fois terminée",
    	check_log_file: "Consultez votre fichier journal pour plus de détails.",
    	"continue": "Continuer",
    	disabled: "Désactivé",
    	documentation: "Documentation",
    	element: "élément",
    	hacs_is_disabled: "HACS est désactivé",
    	install: "Installer",
    	installed: "installés",
    	integration: "Intégration",
    	integration_plural: "Intégrations",
    	integrations: "Intégrations",
    	lovelace: "Lovelace",
    	lovelace_element: "Élément Lovelace",
    	lovelace_elements: "Éléments Lovelace",
    	manage: "gérer",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "Applications NetDaemon",
    	netdaemon_plural: "NetDaemon Apps",
    	plugin: "Lovelace",
    	plugin_plural: "Éléments Lovelace",
    	plugins: "Éléments Lovelace",
    	python_script: "Script Python",
    	python_script_plural: "Scripts Python",
    	python_scripts: "Scripts Python",
    	repositories: "Dépôts",
    	repository: "Dépôt",
    	settings: "paramètres",
    	theme: "Thème",
    	theme_plural: "Thèmes",
    	themes: "Thèmes",
    	uninstall: "Désinstaller",
    	update: "Mise à jour",
    	version: "Version"
    };
    var config$5 = {
    	abort: {
    		single_instance_allowed: "Une seule configuration de HACS est autorisée."
    	},
    	error: {
    		auth: "Le jeton personnel d'accès est invalide."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Activer la découverte et le suivi des applications AppDaemon",
    				netdaemon: "Activer la découverte et le suivi des applications NetDaemon",
    				python_script: "Activer la découverte et le suivi des scripts python",
    				sidepanel_icon: "Icône de la barre latérale",
    				sidepanel_title: "Titre de la barre latérale",
    				theme: "Activer la découverte et le suivi des thèmes",
    				token: "Jeton d'accès personnel GitHub"
    			},
    			description: "Si vous avez besoin d'aide pour la configuration, jetez un œil ici: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$5 = {
    	add_to_lovelace: "Êtes-vous sûr de vouloir l'ajouter à vos ressources Lovelace ?",
    	bg_task: "L'action est désactivée pendant l'exécution de tâches en arrière-plan.",
    	cancel: "Annuler",
    	"continue": "Es-tu sur de vouloir continuer?",
    	"delete": "Êtes-vous sûr de vouloir supprimer '{item}'?",
    	delete_installed: "'{item}' est installé, vous devez le désinstaller avant de pouvoir le supprimer.",
    	exist: "{item} existe déjà",
    	generic: "Êtes-vous sûr?",
    	home_assistant_is_restarting: "Attendez, Home Assistant redémarre maintenant.",
    	home_assistant_version_not_correct: "Vous utilisez la version '{haversion}' de Home Assistant, mais ce référentiel nécessite l'installation de la version minimale '{minversion}'.",
    	no: "Non",
    	no_upgrades: "Aucune mise à niveau en attente",
    	ok: "OK",
    	overwrite: "En faisant cela, cela l'écrasera.",
    	reload_data: "Cela recharge les données de tous les dépôts dont HACS a connaissance, cela prendra un certain temps à terminer.",
    	restart_home_assistant: "Voulez-vous vraiment redémarrer Home Assistant ?",
    	uninstall: "Êtes-vous sûr de vouloir désinstaller '{item}'?",
    	upgrade_all: "Cela mettra à niveau tous ces dépôts, assurez-vous d'avoir lu les notes de publication pour chacun d'entre eux avant de continuer.",
    	yes: "Oui"
    };
    var dialog_about$4 = {
    	frontend_version: "Version frontend",
    	installed_repositories: "Dépôts installés",
    	integration_version: "Version d'intégration",
    	useful_links: "Liens utiles"
    };
    var dialog_add_repo$4 = {
    	limit: "Seuls les 100 premiers dépôts sont affichés, utilisez la recherche pour filtrer ce dont vous avez besoin",
    	no_match: "Aucun dépôt trouvé correspondant à votre filtre",
    	sort_by: "Trier par",
    	title: "Ajouter un dépôt"
    };
    var dialog_custom_repositories$4 = {
    	category: "Catégorie",
    	no_category: "Catégorie manquante",
    	no_repository: "Dépôt manquant",
    	title: "Dépôts personnalisés",
    	url_placeholder: "Ajouter une URL de dépôt personnalisée"
    };
    var dialog_info$4 = {
    	author: "Auteur",
    	downloads: "Téléchargements",
    	install: "Installer ce dépôt dans HACS",
    	loading: "Chargement des informations ...",
    	no_info: "Le développeur n'a pas fourni plus d'informations pour ce dépôt",
    	open_issues: "Open issues",
    	open_repo: "Dépôt ouvert",
    	stars: "Étoiles",
    	version_installed: "Version installée"
    };
    var dialog_install$4 = {
    	restart: "N'oubliez pas que vous devez redémarrer Home Assistant avant que les modifications aux intégrations (custom_components) soient appliquées.",
    	select_version: "Sélectionnez une version",
    	show_beta: "Afficher les versions bêta",
    	type: "Type",
    	url: "URL"
    };
    var dialog_update$4 = {
    	available_version: "Version disponible",
    	changelog: "Changelog",
    	installed_version: "Version installée",
    	releasenotes: "Notes de version pour {release}",
    	title: "Mise à jour en attente"
    };
    var entry$4 = {
    	information: "Information",
    	intro: "Les mises à jour et les messages importants s'afficheront ici s'il y en a",
    	messages: {
    		disabled: {
    			content: "Vérifiez votre fichier journal pour plus de détails",
    			title: "HACS est désactivé"
    		},
    		has_pending_tasks: {
    			content: "Certains dépôts peuvent ne pas apparaître tant que cette opération n'est pas terminée",
    			title: "Tâches d’arrière-plan en attente"
    		},
    		resources: {
    			content: "Vous avez {number} éléments Lovelace qui ne sont pas chargés correctement dans Lovelace.",
    			title: "Non chargé dans Lovelace"
    		},
    		restart: {
    			content: "Vous disposez de {number} intégrations qui nécessitent un redémarrage de Home Assistant, vous pouvez le faire à partir de la section \"Commandes du serveur\" sous la partie configuration de l'interface utilisateur de Home Assistant.",
    			title: "En attente de redémarrage"
    		},
    		startup: {
    			content: "HACS démarre, pendant ce temps, certaines informations peuvent être manquantes ou incorrectes",
    			title: "HACS est en train de démarrer"
    		},
    		wrong_frontend_installed: {
    			content: "Vous avez {running} du frontend HACS installé, mais la version {expected} était attendue. Si ce message s'affiche, Home Assistant n'a pas pu installer la nouvelle version, essayez de redémarrer Home Assistant.",
    			title: "Version frontend inattendue"
    		},
    		wrong_frontend_loaded: {
    			content: "Vous exécutez la version {running} de l'interface HACS, mais la version {expected} était attendue, vous devez vider le cache de votre navigateur.",
    			title: "Version frontend inattendue"
    		}
    	},
    	pending_updates: "Mises à jour en attente"
    };
    var menu$4 = {
    	about: "À propos de HACS",
    	clear: "Effacer tous les nouveaux",
    	custom_repositories: "Dépôts personnalisés",
    	dismiss: "Rejeter tous les nouveaux dépôts",
    	documentation: "Documentation",
    	open_issue: "Open issue",
    	reload: "Recharger la fenêtre"
    };
    var options$5 = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Activer la découverte et le suivi des applications AppDaemon",
    				country: "Filtrer par code pays.",
    				debug: "Activez le débogage.",
    				experimental: "Activer les fonctionnalités expérimentales",
    				netdaemon: "Activer la découverte et le suivi des applications NetDaemon",
    				not_in_use: "Non utilisé avec YAML",
    				release_limit: "Nombre de recensés à afficher.",
    				sidepanel_icon: "Icône de la barre latérale",
    				sidepanel_title: "Titre de la barre latérale"
    			}
    		}
    	}
    };
    var repository_banner$5 = {
    	config_flow: "Cette intégration prend en charge config_flow, ce qui signifie que vous pouvez maintenant accéder à la section d'intégration de votre interface utilisateur pour le configurer.",
    	config_flow_title: "Configuration de l'interface utilisateur prise en charge",
    	integration_not_loaded: "Cette intégration n'est pas chargée dans Home Assistant.",
    	no_restart_required: "Aucun redémarrage requis",
    	not_loaded: "Non chargé",
    	plugin_not_loaded: "Ce plugin n'est pas ajouté à vos ressources Lovelace.",
    	restart: "Vous devez redémarrer Home Assistant.",
    	restart_pending: "Redémarrage en attente"
    };
    var repository_card$4 = {
    	dismiss: "rejeter",
    	hide: "Cacher",
    	information: "Information",
    	new_repository: "Nouveau dépôt",
    	not_loaded: "Non chargé",
    	open_issue: "Ouvrir issue",
    	open_source: "Open source",
    	pending_restart: "Redémarrage en attente",
    	pending_update: "Mise à jour en attente",
    	reinstall: "Réinstaller",
    	report: "Rapport de suppression",
    	update_information: "Mettre à jour les informations"
    };
    var repository$5 = {
    	add_to_lovelace: "Ajouter à Lovelace",
    	authors: "Auteurs",
    	available: "Disponible",
    	back_to: "Retour",
    	changelog: "Change log",
    	downloads: "Téléchargements",
    	flag_this: "Marquer",
    	frontend_version: "Version de l'interface",
    	github_stars: "Étoiles GitHub",
    	goto_integrations: "Aller aux intégrations",
    	hide: "Masquer",
    	hide_beta: "Masquer les bêta",
    	install: "Installer",
    	installed: "Installé",
    	lovelace_copy_example: "Copier cet exemple dans le presse-papier",
    	lovelace_instruction: "Quand vous l'ajoutez à votre configuration lovelace, utilisez",
    	lovelace_no_js_type: "Impossible de déterminer le type de plugin, veuillez vérifier le dépôt",
    	newest: "nouveau",
    	note_appdaemon: "vous devez toujours l'ajouter à votre fichier 'apps.yaml'",
    	note_installed: "Une fois installé, il se trouvera dans",
    	note_integration: "Vous devez toujours l'ajouter à votre fichier 'configuration.yaml'",
    	note_plugin: "Vous devez toujours l'ajouter à votre configuration lovelace ('ui-lovelace.yaml' ou l'éditeur de configuration de l'interface)",
    	note_plugin_post_107: "Vous devez toujours l'ajouter à votre configuration lovelace ('configuration.yaml' ou l'éditeur de configuration de l'interface '/config/lovelace/resources')",
    	open_issue: "Ouvrir un ticket",
    	open_plugin: "Ouvrir l'élément",
    	reinstall: "Réinstaller",
    	repository: "Dépôt",
    	restart_home_assistant: "Redémarrer Home Assistant",
    	show_beta: "Afficher les bêta",
    	uninstall: "Désinstaller",
    	update_information: "Mettre à jour les informations",
    	upgrade: "Mettre à jour"
    };
    var search$4 = {
    	placeholder: "Rechercher un dépôt"
    };
    var sections$4 = {
    	about: {
    		description: "Afficher des informations sur le système HACS",
    		title: "À propos de"
    	},
    	automation: {
    		description: "C'est ici que vous trouverez les scripts python, les applications AppDaemon et NetDaemon",
    		title: "Automatisation"
    	},
    	frontend: {
    		description: "C'est ici que vous trouverez des thèmes, des cartes personnalisées et d'autres éléments pour lovelace",
    		title: "Frontend"
    	},
    	integrations: {
    		description: "C'est ici que vous trouverez les intégrations personnalisées (custom_components)",
    		title: "Intégrations"
    	},
    	pending_repository_upgrade: "Vous utilisez la version {installed} , la version {available} est disponible"
    };
    var settings$5 = {
    	add_custom_repository: "AJOUTER UN DÉPÔT PERSONNALISÉ",
    	adding_new_repo: "Ajout d'un nouveau dépôt '{repo}'",
    	adding_new_repo_category: "Avec la catégorie '{category}'.",
    	bg_task_custom: "Les dépôts personnalisés sont masqués pendant l'exécution de tâches en arrière-plan.",
    	category: "Catégorie",
    	compact_mode: "Mode compact",
    	custom_repositories: "DÉPÔTS PERSONNALISÉS",
    	"delete": "Supprimer",
    	display: "Affichage",
    	grid: "Grille",
    	hacs_repo: "Dépôt HACS",
    	hidden_repositories: "dépôts cachés",
    	missing_category: "Vous devez sélectionner une catégorie",
    	open_repository: "Ouvrir dépôt",
    	reload_data: "Recharger les données",
    	reload_window: "Recharger la fenêtre",
    	repository_configuration: "Configuration de dépôt",
    	save: "Enregistrer",
    	table: "Tableau",
    	table_view: "Vue table",
    	unhide: "Afficher",
    	upgrade_all: "Tout mettre à jour"
    };
    var store$5 = {
    	ascending: "ascendant",
    	clear_new: "Effacer tous les nouveaux dépôts",
    	descending: "descendant",
    	last_updated: "Dernière mise à jour",
    	name: "Nom",
    	new_repositories: "Nouveaux dépôts",
    	new_repositories_note: "Vous avez plus de 10 nouveaux dépôts qui apparaissent ici. Si vous voulez les effacer tous, cliquez sur les 3 points en haut à droite de l'écran et ignorez-les tous.",
    	no_repositories: "Aucun dépôt",
    	no_repositories_desc1: "Il semble que vous n’avez pas de dépôts installés dans cette section encore.",
    	no_repositories_desc2: "Cliquez sur le + dans le coin inférieur pour ajouter votre premier !",
    	no_repositories_found_desc1: "Aucun référentiel installé correspondant à \" {searchInput} \" n'a été trouvé dans cette section.",
    	no_repositories_found_desc2: "Essayez de chercher autre chose !",
    	pending_upgrades: "Mises à niveau en attente",
    	placeholder_search: "Veuillez entrer un terme de recherche...",
    	sort: "Trier",
    	stars: "Étoiles",
    	status: "Statut"
    };
    var time$5 = {
    	ago: "il y a",
    	day: "jour",
    	days: "jours",
    	hour: "heure",
    	hours: "heures",
    	minute: "minute",
    	minutes: "minutes",
    	month: "mois",
    	months: "mois",
    	one: "Un",
    	one_day_ago: "il y a 1 jour",
    	one_hour_ago: "il y a 1 heure",
    	one_minute_ago: "il y a 1 minute",
    	one_month_ago: "il y a 1 mois",
    	one_second_ago: "il y a 1 seconde",
    	one_year_ago: "il y a 1 an",
    	second: "seconde",
    	seconds: "secondes",
    	x_days_ago: "il y a {x} jours",
    	x_hours_ago: "il y a {x} heures",
    	x_minutes_ago: "il y a {x} minutes",
    	x_months_ago: "il y a {x} mois",
    	x_seconds_ago: "il y a {x} secondes",
    	x_years_ago: "il y a {x} ans",
    	year: "an",
    	years: "ans"
    };
    var fr = {
    	common: common$5,
    	config: config$5,
    	confirm: confirm$5,
    	dialog_about: dialog_about$4,
    	dialog_add_repo: dialog_add_repo$4,
    	dialog_custom_repositories: dialog_custom_repositories$4,
    	dialog_info: dialog_info$4,
    	dialog_install: dialog_install$4,
    	dialog_update: dialog_update$4,
    	entry: entry$4,
    	menu: menu$4,
    	options: options$5,
    	repository_banner: repository_banner$5,
    	repository_card: repository_card$4,
    	repository: repository$5,
    	search: search$4,
    	sections: sections$4,
    	settings: settings$5,
    	store: store$5,
    	time: time$5
    };

    var fr$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$5,
        config: config$5,
        confirm: confirm$5,
        dialog_about: dialog_about$4,
        dialog_add_repo: dialog_add_repo$4,
        dialog_custom_repositories: dialog_custom_repositories$4,
        dialog_info: dialog_info$4,
        dialog_install: dialog_install$4,
        dialog_update: dialog_update$4,
        entry: entry$4,
        menu: menu$4,
        options: options$5,
        repository_banner: repository_banner$5,
        repository_card: repository_card$4,
        repository: repository$5,
        search: search$4,
        sections: sections$4,
        settings: settings$5,
        store: store$5,
        time: time$5,
        'default': fr
    });

    var common$6 = {
    	about: "Névjegy",
    	add: "hozzáadás",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "AppDaemon Appok",
    	appdaemon_plural: "AppDaemon appok",
    	background_task: "Éppen háttérfeladat fut, ez az oldal frissülni fog, ha kész.",
    	check_log_file: "További részletekért ellenőrizd a naplófájlt.",
    	"continue": "Folytatás",
    	disabled: "Tiltva",
    	documentation: "Dokumentáció",
    	element: "bővítmény",
    	hacs_is_disabled: "A HACS le van tiltva",
    	install: "Telepítés",
    	installed: "Telepített",
    	integration: "Integráció",
    	integration_plural: "Integrációk",
    	integrations: "Integrációk",
    	lovelace: "Lovelace",
    	lovelace_element: "Lovelace bővítmény",
    	lovelace_elements: "Lovelace bővítmények",
    	manage: "kezelés",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDaemon Appok",
    	netdaemon_plural: "NetDaemon appok",
    	plugin: "Lovelace",
    	plugin_plural: "Lovelace bővítmények",
    	plugins: "Lovelace bővítmények",
    	python_script: "Python Szkript",
    	python_script_plural: "Python szkriptek",
    	python_scripts: "Python Szkriptek",
    	repositories: "Tárolók",
    	repository: "Tároló",
    	settings: "beállítások",
    	theme: "Téma",
    	theme_plural: "Témák",
    	themes: "Témák",
    	uninstall: "Eltávolítás",
    	update: "Frissítés",
    	version: "Verzió"
    };
    var config$6 = {
    	abort: {
    		single_instance_allowed: "Csak egyetlen HACS-konfiguráció megengedett."
    	},
    	error: {
    		auth: "A Személyes Hozzáférési Token nem megfelelő."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "AppDaemon appok felfedezésének és nyomon követésének engedélyezése",
    				netdaemon: "NetDaemon appok felfedezésének és nyomon követésének engedélyezése",
    				python_script: "Python szkriptek felfedezésének és nyomon követésének engedélyezése",
    				sidepanel_icon: "Oldalsó panel ikon",
    				sidepanel_title: "Oldalsó panel cím",
    				theme: "Témák felfedezésének és nyomon követésének engedélyezése",
    				token: "GitHub Személyes Hozzáférési Token"
    			},
    			description: "Ha segítségre van szükséged a konfigurációval kapcsolatban, akkor tekintsd meg ezt az oldalt: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$6 = {
    	add_to_lovelace: "Biztosan hozzá szeretnéd ezt adni a Lovelace erőforrásokhoz?",
    	bg_task: "A művelet le van tiltva, amíg háttérfeladat fut.",
    	cancel: "Mégse",
    	"continue": "Biztosan folytatni szeretnéd?",
    	"delete": "Biztosan törölni szeretnéd a(z) '{item}'-t?",
    	delete_installed: "A(z) '{item}' telepítve van, törlés előtt el kell távolítanod.",
    	exist: "{item} már létezik",
    	generic: "Biztos vagy benne?",
    	home_assistant_is_restarting: "Várj, a Home Assistant éppen újraindul.",
    	home_assistant_version_not_correct: "A Home Assistant '{haversion}' verzióját használod, de ehhez a tárolóhoz legalább a(z) '{minversion}' verzióra van szükség.",
    	no: "Nem",
    	no_upgrades: "Nincsenek elérhető frissítések",
    	ok: "OK",
    	overwrite: "Ezzel felül fogod írni.",
    	reload_data: "Ez újratölti a HACS által ismert összes tároló adatát, ami némi időbe telhet.",
    	restart_home_assistant: "Biztosan újraindítod a Home Assistant programot?",
    	uninstall: "Biztosan el szeretnéd távolítani a(z) '{item}'-t?",
    	upgrade_all: "Ez frissíteni fogja az összes tárolót. Győzödj meg róla, hogy elolvastad az összes kiadási megjegyzést, mielőtt továbblépnél.",
    	yes: "Igen"
    };
    var dialog_about$5 = {
    	frontend_version: "Frontend verzió",
    	installed_repositories: "Telepített tárolók",
    	integration_version: "Integráció verzió",
    	useful_links: "Hasznos linkek"
    };
    var dialog_add_repo$5 = {
    	limit: "Csak az első 100 tároló jelenik meg, használd a keresőt a találatok szűkítéséhez",
    	no_match: "Nincs a szűrésnek megfelelő tároló",
    	sort_by: "Rendezés",
    	title: "Tároló hozzáadása"
    };
    var dialog_custom_repositories$5 = {
    	category: "Kategória",
    	no_category: "Hiányzó kategória",
    	no_repository: "Hiányzó tároló",
    	title: "Egyéni tárolók",
    	url_placeholder: "Egyéni tároló URL címének hozzáadása"
    };
    var dialog_info$5 = {
    	author: "Szerző",
    	downloads: "Letöltések",
    	install: "Tároló telepítése HACS-ben",
    	loading: "Információ betöltése...",
    	no_info: "A fejlesztő nem adott meg több információt ehhez a tárolóhoz",
    	open_issues: "Jelentett problémák",
    	open_repo: "Tároló megnyitása",
    	stars: "Csillagok",
    	version_installed: "Telepített verzió"
    };
    var dialog_install$5 = {
    	restart: "Ne feledd, hogy az egyéni integrációk (custom_components) módosításainak alkalmazásához újra kell indítanod a Home Assistant alkalmazást.",
    	select_version: "Verzió kiválasztása",
    	show_beta: "Béta verziók megjelenítése",
    	type: "Típus",
    	url: "URL"
    };
    var dialog_update$5 = {
    	available_version: "Elérhető verzió",
    	changelog: "Változási napló",
    	installed_version: "Telepített verzió",
    	title: "Frissítés érhető el"
    };
    var entry$5 = {
    	information: "Információ",
    	intro: "A frissítések és a fontos üzenetek itt jelennek meg, ha vannak",
    	messages: {
    		disabled: {
    			content: "További részletek a naplófájlban",
    			title: "A HACS le van tiltva"
    		},
    		has_pending_tasks: {
    			content: "Előfordulhat, hogy egyes tárolók nem jelennek meg, amíg ez be nem fejeződik",
    			title: "Függőben lévő háttérfeladatok"
    		},
    		startup: {
    			content: "A HACS éppen indul, ezidő alatt egyes információk hiányozhatnak vagy helytelenek lehetnek",
    			title: "A HACS éppen indul"
    		}
    	},
    	pending_updates: "Frissítések érhetők el"
    };
    var menu$5 = {
    	about: "HACS névjegye",
    	clear: "Új jelölések törlése",
    	custom_repositories: "Egyéni tárolók",
    	dismiss: "Minden új tároló elvetése",
    	documentation: "Dokumentáció",
    	open_issue: "Probléma jelentése",
    	reload: "Ablak újratöltése"
    };
    var options$6 = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "AppDaemon appok felfedezésének és nyomon követésének engedélyezése",
    				country: "Szűrés országkóddal.",
    				debug: "Hibakeresés engedélyezése.",
    				experimental: "Kísérleti funkciók engedélyezése",
    				netdaemon: "NetDaemon appok felfedezésének és nyomon követésének engedélyezése",
    				not_in_use: "YAML-lel nem használható",
    				release_limit: "Megjelenítendő kiadások száma.",
    				sidepanel_icon: "Oldalsó panel ikon",
    				sidepanel_title: "Oldalsó panel cím"
    			}
    		}
    	}
    };
    var repository_banner$6 = {
    	config_flow: "Ez az integráció támogatja a config_flow-t, vagyis a felhasználói felületen az integrációk menüben lehet konfigurálni.",
    	config_flow_title: "A felhasználói felület konfigurációja támogatott",
    	integration_not_loaded: "Ez az integráció nincs betöltve a Home Assistantban.",
    	no_restart_required: "Nincs szükség újraindításra",
    	not_loaded: "Nincs betöltve",
    	plugin_not_loaded: "Ez a bővítmény nincs hozzáadva a Lovelace erőforrásaidhoz.",
    	restart: "Indítsd újra a Home Assistant programot.",
    	restart_pending: "Újraindítás függőben"
    };
    var repository_card$5 = {
    	dismiss: "elvetés",
    	hide: "Elrejtés",
    	information: "Információ",
    	new_repository: "Új tároló",
    	open_issue: "Probléma jelentése",
    	open_source: "Forrás megnyitása",
    	pending_update: "Frissítés érhető el",
    	reinstall: "Újratelepítés",
    	report: "Jelentés eltávolításra",
    	update_information: "Frissítési információ"
    };
    var repository$6 = {
    	add_to_lovelace: "Hozzáadás a Lovelace-hez",
    	authors: "Szerzők",
    	available: "Elérhető",
    	back_to: "Vissza -",
    	changelog: "Változási napló",
    	downloads: "Letöltések",
    	flag_this: "Megjelölés",
    	frontend_version: "Frontend verzió",
    	github_stars: "GitHub csillagok",
    	goto_integrations: "Ugrás az integrációkhoz",
    	hide: "Elrejtés",
    	hide_beta: "Béta elrejtése",
    	install: "Telepítés",
    	installed: "Telepített",
    	lovelace_copy_example: "Példa másolása a vágólapra",
    	lovelace_instruction: "Amikor hozzáadod ezt a lovelace konfigurációdhoz, használd ezt",
    	lovelace_no_js_type: "Nem sikerült meghatározni a bővítmény típusát, ellenőrizd a tárolót.",
    	newest: "legújabb",
    	note_appdaemon: "de még hozzá kell adnod az 'apps.yaml' fájlhoz",
    	note_installed: "Telepítéskor a következő helyre kerül:",
    	note_integration: "de még hozzá kell adnod a 'configuration.yaml' fájlhoz",
    	note_plugin: "de még hozzá kell adnod a lovelace konfigurációhoz (az 'ui-lovelace.yaml' fájlban vagy a Lovelace felületen a konfiguráció szerkesztőben)",
    	note_plugin_post_107: "de még hozzá kell adnod a lovelace konfigurációhoz ('configuration.yaml' vagy az erőforrás szerkesztőben '/config/lovelace/resources')",
    	open_issue: "Probléma jelentése",
    	open_plugin: "Bővítmény megnyitása",
    	reinstall: "Újratelepítés",
    	repository: "Tároló",
    	restart_home_assistant: "Home Assistant újraindítása",
    	show_beta: "Béta megjelenítése",
    	uninstall: "Eltávolítás",
    	update_information: "Frissítési információk",
    	upgrade: "Frissítés"
    };
    var search$5 = {
    	placeholder: "Tároló keresése"
    };
    var sections$5 = {
    	about: {
    		description: "Információk megjelenítése a HACS-ről",
    		title: "Névjegy"
    	},
    	automation: {
    		description: "Itt Python szkripteket, AppDaemon és NetDaemon appokat találsz",
    		title: "Automatizálás"
    	},
    	frontend: {
    		description: "Itt témákat, egyéni kártyákat és más bővítményeket találsz a Lovelace-hez",
    		title: "Frontend"
    	},
    	integrations: {
    		description: "Itt találod az egyéni integrációkat (custom_components)",
    		title: "Integrációk"
    	},
    	pending_repository_upgrade: "A(z) {installed} verziót futtatod, a(z) {available} verzió már elérhető"
    };
    var settings$6 = {
    	add_custom_repository: "EGYÉNI TÁROLÓ HOZZÁADÁSA",
    	adding_new_repo: "Új tároló hozzáadása '{repo}'",
    	adding_new_repo_category: "A '{category}' kategóriával.",
    	bg_task_custom: "Az egyéni tárolók rejtve vannak, amíg háttérfeladat fut.",
    	category: "Kategória",
    	compact_mode: "Kompakt mód",
    	custom_repositories: "EGYÉNI TÁROLÓK",
    	"delete": "Törlés",
    	display: "Megjelenítés",
    	grid: "Rács",
    	hacs_repo: "HACS tároló",
    	hidden_repositories: "rejtett tárolók",
    	missing_category: "Ki kell választanod egy kategóriát",
    	open_repository: "Tároló megnyitása",
    	reload_data: "Adatok újratöltése",
    	reload_window: "Ablak újratöltése",
    	repository_configuration: "Tároló konfiguráció",
    	save: "Mentés",
    	table: "Táblázat",
    	table_view: "Táblázat nézet",
    	unhide: "felfedés",
    	upgrade_all: "Minden frissítése"
    };
    var store$6 = {
    	ascending: "növekvő",
    	clear_new: "Új tárolók megjelölése látottként",
    	descending: "csökkenő",
    	last_updated: "Utolsó frissítés",
    	name: "Név",
    	new_repositories: "Új tárolók",
    	new_repositories_note: "Több mint 10 új tároló látható. Ha törölni szeretnéd őket, akkor kattints a jobb felső sarokban lévő 3 pontra, és válaszd ki a 'Minden új tároló elvetése' menüpontot.",
    	no_repositories: "Nincsenek tárolók",
    	no_repositories_desc1: "Úgy tűnik, még nincsenek telepítve tárolók ebben a szekcióban.",
    	no_repositories_desc2: "Kattints az alsó sarokban található + jelre az első hozzáadásához!",
    	pending_upgrades: "Frissítések érhetők el",
    	placeholder_search: "Kérlek adj meg egy keresési kifejezést...",
    	sort: "rendezés",
    	stars: "Csillag",
    	status: "Állapot"
    };
    var time$6 = {
    	ago: "ezelőtt",
    	day: "nap",
    	days: "nap",
    	hour: "óra",
    	hours: "óra",
    	minute: "perc",
    	minutes: "perc",
    	month: "hónap",
    	months: "hónap",
    	one: "Egy",
    	one_day_ago: "egy nappal ezelőtt",
    	one_hour_ago: "egy órával ezelőtt",
    	one_minute_ago: "egy perccel ezelőtt",
    	one_month_ago: "egy hónappal ezelőtt",
    	one_second_ago: "egy másodperccel ezelőtt",
    	one_year_ago: "egy évvel ezelőtt",
    	second: "másodperc",
    	seconds: "másodperc",
    	x_days_ago: "{x} nappal ezelőtt",
    	x_hours_ago: "{x} órával ezelőtt",
    	x_minutes_ago: "{x} perccel ezelőtt",
    	x_months_ago: "{x} hónappal ezelőtt",
    	x_seconds_ago: "{x} másodperccel ezelőtt",
    	x_years_ago: "{x} évvel ezelőtt",
    	year: "év",
    	years: "év"
    };
    var hu = {
    	common: common$6,
    	config: config$6,
    	confirm: confirm$6,
    	dialog_about: dialog_about$5,
    	dialog_add_repo: dialog_add_repo$5,
    	dialog_custom_repositories: dialog_custom_repositories$5,
    	dialog_info: dialog_info$5,
    	dialog_install: dialog_install$5,
    	dialog_update: dialog_update$5,
    	entry: entry$5,
    	menu: menu$5,
    	options: options$6,
    	repository_banner: repository_banner$6,
    	repository_card: repository_card$5,
    	repository: repository$6,
    	search: search$5,
    	sections: sections$5,
    	settings: settings$6,
    	store: store$6,
    	time: time$6
    };

    var hu$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$6,
        config: config$6,
        confirm: confirm$6,
        dialog_about: dialog_about$5,
        dialog_add_repo: dialog_add_repo$5,
        dialog_custom_repositories: dialog_custom_repositories$5,
        dialog_info: dialog_info$5,
        dialog_install: dialog_install$5,
        dialog_update: dialog_update$5,
        entry: entry$5,
        menu: menu$5,
        options: options$6,
        repository_banner: repository_banner$6,
        repository_card: repository_card$5,
        repository: repository$6,
        search: search$5,
        sections: sections$5,
        settings: settings$6,
        store: store$6,
        time: time$6,
        'default': hu
    });

    var common$7 = {
    	about: "Informazioni su",
    	add: "aggiungi",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "Applicazioni AppDaemon",
    	appdaemon_plural: "Applicazioni AppDaemon",
    	background_task: "Attività in esecuzione, questa pagina sarà ricaricata al termine.",
    	check_log_file: "Controlla il tuo file di registro per maggiori dettagli.",
    	"continue": "Continua",
    	disabled: "Disabilitato",
    	documentation: "Documentazione",
    	element: "elemento",
    	hacs_is_disabled: "HACS è disabilitato",
    	install: "Installa",
    	installed: "Installati",
    	integration: "Integrazione",
    	integration_plural: "Integrazioni",
    	integrations: "Integrazioni",
    	lovelace: "Lovelace",
    	lovelace_element: "Elemento Lovelace",
    	lovelace_elements: "Elementi di Lovelace",
    	manage: "gestione",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "Applicazioni NetDaemon",
    	netdaemon_plural: "Applicazioni NetDaemon",
    	plugin: "Lovelace",
    	plugin_plural: "Elementi di Lovelace",
    	plugins: "Elementi di Lovelace",
    	python_script: "Script python",
    	python_script_plural: "Script in Python",
    	python_scripts: "Script python",
    	repositories: "Repository",
    	repository: "Repository",
    	settings: "Impostazioni",
    	theme: "Tema",
    	theme_plural: "Temi",
    	themes: "Temi",
    	uninstall: "Disinstallare",
    	update: "Aggiorna",
    	version: "Versione"
    };
    var config$7 = {
    	abort: {
    		single_instance_allowed: "È consentita una sola configurazione di HACS."
    	},
    	error: {
    		auth: "Il token di accesso personale non è corretto."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Abilita il rilevamento e il monitoraggio delle applicazioni AppDaemon",
    				netdaemon: "Abilita il rilevamento e il monitoraggio delle applicazioni NetDaemon",
    				python_script: "Abilita il rilevamento e il monitoraggio dei python_scripts",
    				sidepanel_icon: "Icona nel pannello laterale",
    				sidepanel_title: "Titolo nel pannello laterale",
    				theme: "Abilita individuazione e tracciamento dei temi",
    				token: "Token di accesso personale GitHub"
    			},
    			description: "Se hai bisogno di aiuto con la configurazione dai un'occhiata qui: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$7 = {
    	add_to_lovelace: "Sei sicuro di voler aggiungerlo alle tue risorse Lovelace?",
    	bg_task: "L'azione è disabilitata mentre sono in esecuzione attività in background.",
    	cancel: "Annulla",
    	"continue": "Sei sicuro di voler continuare?",
    	"delete": "Sei sicuro di voler disinstallare '{item}'?",
    	delete_installed: "'{item}' è installato, è necessario disinstallarlo prima di poterlo eliminare.",
    	exist: "{item} esiste già",
    	generic: "Sei sicuro?",
    	home_assistant_is_restarting: "Aspetta, Home Assistant si sta riavviando.",
    	home_assistant_version_not_correct: "Stai eseguendo la versione Home Assistant '{haversion}', ma questo repository richiede l'installazione della versione minima '{minversion}'.",
    	no: "No",
    	no_upgrades: "Nessun aggiornamento in sospeso",
    	ok: "OK",
    	overwrite: "In questo modo lo sovrascriverà.",
    	reload_data: "Questo ricarica i dati di tutte le repository di cui HACS è a conoscenza, ci vorrà del tempo per finire.",
    	restart_home_assistant: "Sei sicuro di voler riavviare Home Assistant?",
    	uninstall: "Sei sicuro di voler disinstallare '{item}'?",
    	upgrade_all: "Questa azione aggiornerà tutti i repository, assicurati di aver letto le note di rilascio prima di procedere.",
    	yes: "Sì"
    };
    var dialog_about$6 = {
    	frontend_version: "Versione frontend",
    	installed_repositories: "Repository installati",
    	integration_version: "Versione di integrazione",
    	useful_links: "Link utili"
    };
    var dialog_add_repo$6 = {
    	limit: "Vengono visualizzati solo i primi 100 repository, utilizza la ricerca per filtrare ciò di cui hai bisogno",
    	no_match: "Nessun repository trovato corrispondente al filtro",
    	sort_by: "Ordina per",
    	title: "Aggiungi repository"
    };
    var dialog_custom_repositories$6 = {
    	category: "Categoria",
    	no_category: "Categoria mancante",
    	no_repository: "Repository mancante",
    	title: "Repository personalizzati",
    	url_placeholder: "Aggiungi l'URL del repository personalizzato"
    };
    var dialog_info$6 = {
    	author: "Autore",
    	downloads: "Download",
    	install: "Installa questo repository in HACS",
    	loading: "Caricamento informazioni in corso ...",
    	no_info: "Lo sviluppatore non ha fornito ulteriori informazioni per questo repository",
    	open_issues: "Problemi aperti",
    	open_repo: "Apri il repository",
    	stars: "Stelle",
    	version_installed: "Versione installata"
    };
    var dialog_install$6 = {
    	restart: "Tenere presente che è necessario riavviare Home Assistant prima di applicare le modifiche alle integrazioni (custom_components).",
    	select_version: "Seleziona la versione",
    	show_beta: "Mostra le versioni beta",
    	type: "Tipo",
    	url: "URL"
    };
    var dialog_update$6 = {
    	available_version: "Versione disponibile",
    	changelog: "Registro dei cambiamenti",
    	installed_version: "Versione installata",
    	releasenotes: "Note di rilascio per {release}",
    	title: "Aggiornamento in sospeso"
    };
    var entry$6 = {
    	information: "Informazioni",
    	intro: "Gli aggiornamenti e i messaggi importanti verranno visualizzati qui se presenti",
    	messages: {
    		disabled: {
    			content: "Controlla il tuo file di registro per maggiori dettagli",
    			title: "HACS è disabilitato"
    		},
    		has_pending_tasks: {
    			content: "Alcuni repository potrebbero non essere visualizzati fino al completamento",
    			title: "Attività in background in sospeso"
    		},
    		resources: {
    			content: "Hai {number} elementi di Lovelace che non sono stati caricati correttamente in Lovelace.",
    			title: "Non caricato in Lovelace"
    		},
    		restart: {
    			content: "Hai {number} integrazioni che richiedono il riavvio di Home Assistant, puoi farlo dalla sezione \"Controlli del server\" nella sezione Impostazioni dell'interfaccia utente di Home Assistant.",
    			title: "In attesa di riavvio"
    		},
    		startup: {
    			content: "HACS si sta avviando, durante questo periodo alcune informazioni potrebbero essere mancanti o errate",
    			title: "HACS si sta avviando"
    		},
    		wrong_frontend_installed: {
    			content: "Hai installato {running} del frontend HACS, ma era attesa la versione {expected}, se viene visualizzato questo messaggio Home Assistant non è stato in grado di installare la nuova versione, prova a riavviare Home Assistant.",
    			title: "Versione frontend inaspettata"
    		},
    		wrong_frontend_loaded: {
    			content: "Stai eseguendo la versione {running} del frontend HACS, ma la versione {expected} era prevista, è necessario svuotare la cache del browser.",
    			title: "Versione frontend inaspettata"
    		}
    	},
    	pending_updates: "Aggiornamenti in sospeso"
    };
    var menu$6 = {
    	about: "Informazioni su HACS",
    	clear: "Cancella tutte le novità",
    	custom_repositories: "Repository personalizzati",
    	dismiss: "Elimina tutti i nuovi repository",
    	documentation: "Documentazione",
    	open_issue: "Segnala anomalia",
    	reload: "Ricarica la finestra"
    };
    var options$7 = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Abilita il rilevamento e il monitoraggio delle applicazioni AppDaemon",
    				country: "Filtra con prefisso internazionale.",
    				debug: "Abilita debug.",
    				experimental: "Abilita funzionalità sperimentali",
    				netdaemon: "Abilita il rilevamento e il monitoraggio delle applicazioni NetDaemon",
    				not_in_use: "Non in uso con YAML",
    				release_limit: "Numero di versioni da mostrare.",
    				sidepanel_icon: "Icona nel pannello laterale",
    				sidepanel_title: "Titolo nel pannello laterale"
    			}
    		}
    	}
    };
    var repository_banner$7 = {
    	config_flow: "Questa integrazione supporta config_flow, questo significa che è ora possibile passare alla sezione \"IntegrazionI\" dell'interfaccia utente per la configurazione.",
    	config_flow_title: "Configurazione dell'interfaccia utente supportata",
    	integration_not_loaded: "Questa integrazione non è caricata in Home Assistant.",
    	no_restart_required: "Non è necessario riavviare",
    	not_loaded: "Non caricato",
    	plugin_not_loaded: "Questo elemento non è aggiunto alle tue risorse Lovelace.",
    	restart: "È necessario riavviare Home Assistant.",
    	restart_pending: "Riavvio in attesa"
    };
    var repository_card$6 = {
    	dismiss: "elimina",
    	hide: "Nascondi",
    	information: "Informazioni",
    	new_repository: "Nuovo repository",
    	not_loaded: "Non caricato",
    	open_issue: "Segnala anomalia",
    	open_source: "Open source",
    	pending_restart: "In attesa di riavvio",
    	pending_update: "Aggiornamento in sospeso",
    	reinstall: "Reinstallare",
    	report: "Segnala per la rimozione",
    	update_information: "Aggiorna informazioni"
    };
    var repository$7 = {
    	add_to_lovelace: "Aggiungi a Lovelace",
    	authors: "Autori",
    	available: "Disponibile",
    	back_to: "Torna a",
    	changelog: "Change log",
    	downloads: "Downloads",
    	flag_this: "Spunta questo",
    	frontend_version: "Frontend versione",
    	github_stars: "GitHub stelle",
    	goto_integrations: "Vai alle Integrazioni",
    	hide: "Nascondi",
    	hide_beta: "Nascondi beta",
    	install: "Installa",
    	installed: "Installato",
    	lovelace_copy_example: "Copia l'esempio negli appunti",
    	lovelace_instruction: "Quando lo aggiungi nella configurazione di lovelace, usa questo",
    	lovelace_no_js_type: "Impossibile determinare il tipo di plugin, verificare il repository.",
    	newest: "Più recente",
    	note_appdaemon: "dovrai aggiungerlo nel file 'apps.yaml'",
    	note_installed: "Una volta installato, si troverà in",
    	note_integration: "dovrai aggiungerlo nel file 'configuration.yaml'",
    	note_plugin: "devi aggiungere la configurazione nel file 'ui-lovelace.yaml' oppure via Editor RAW della UI.",
    	note_plugin_post_107: "devi aggiungere la configurazione nel file 'ui-lovelace.yaml' oppure via Editor RAW della UI.",
    	open_issue: "Segnala anomalia",
    	open_plugin: "Apri elemento",
    	reinstall: "Reinstalla",
    	repository: "Archivio Software (Repository)",
    	restart_home_assistant: "Riavvia Home Assistant",
    	show_beta: "Visualizza beta",
    	uninstall: "Rimuovi",
    	update_information: "Aggiorna informazioni",
    	upgrade: "Aggiorna"
    };
    var search$6 = {
    	placeholder: "Cerca repository"
    };
    var sections$6 = {
    	about: {
    		description: "Mostra informazioni su HACS",
    		title: "Informazioni su"
    	},
    	automation: {
    		description: "Qui trovi python_scripts, app AppDaemon e app NetDaemon",
    		title: "Automazione"
    	},
    	frontend: {
    		description: "Qui troverai temi, schede personalizzate e altri elementi per Lovelace",
    		title: "Frontend"
    	},
    	integrations: {
    		description: "Qui si trovano le integrazioni personalizzate (custom_components)",
    		title: "Integrazioni"
    	},
    	pending_repository_upgrade: "Stai eseguendo la versione {installed}, la versione {available}è disponibile"
    };
    var settings$7 = {
    	add_custom_repository: "AGGIUNGI REPOSITORY PERSONALIZZATA",
    	adding_new_repo: "Aggiunta di un nuovo repository '{repo}'",
    	adding_new_repo_category: "Con la categoria '{category}'.",
    	bg_task_custom: "I repository personalizzati sono nascosti mentre sono in esecuzione attività in background.",
    	category: "Categoria",
    	compact_mode: "Modalità compatta",
    	custom_repositories: "REPOSITORY PERSONALIZZATE",
    	"delete": "Cancella",
    	display: "Visualizza",
    	grid: "Griglia",
    	hacs_repo: "HACS repo",
    	hidden_repositories: "repository nascosti",
    	missing_category: "Devi selezionare una categoria",
    	open_repository: "Apri il repository",
    	reload_data: "Ricarica i dati",
    	reload_window: "Ricarica la finestra",
    	repository_configuration: "Configurazione del repository",
    	save: "Salva",
    	table: "Tabella",
    	table_view: "Vista tabella",
    	unhide: "Mostra",
    	upgrade_all: "Aggiorna tutto"
    };
    var store$7 = {
    	ascending: "ascendente",
    	clear_new: "Ripulisci i nuovi repository",
    	descending: "discendente",
    	last_updated: "Ultimo aggiornamento",
    	name: "Nome",
    	new_repositories: "Nuovi repository",
    	new_repositories_note: "Hai più di 10 nuovi repository visualizzati qui, se vuoi cancellarli tutti fai clic sui 3 punti nell'angolo in alto a destra e li eliminerai tutti.",
    	no_repositories: "Nessun repository",
    	no_repositories_desc1: "Sembra che non ci siano ancora repository installati in questa sezione.",
    	no_repositories_desc2: "Fai clic sul + nell'angolo in basso per aggiungere il tuo primo!",
    	no_repositories_found_desc1: "Nessun repository installato corrispondente a \"{searchInput}\" trovato in questa sezione.",
    	no_repositories_found_desc2: "Prova a cercare qualcos'altro!",
    	pending_upgrades: "Aggiornamenti in sospeso",
    	placeholder_search: "Inserire un termine di ricerca",
    	sort: "Ordinare",
    	stars: "Stelle",
    	status: "Stato"
    };
    var time$7 = {
    	ago: "fa",
    	day: "giorno",
    	days: "giorni",
    	hour: "ora",
    	hours: "ore",
    	minute: "minuto",
    	minutes: "minuti",
    	month: "mese",
    	months: "mesi",
    	one: "Un",
    	one_day_ago: "un giorno fa",
    	one_hour_ago: "un'ora fa",
    	one_minute_ago: "un minuto fa",
    	one_month_ago: "un mese fa",
    	one_second_ago: "un secondo fa",
    	one_year_ago: "un anno fa",
    	second: "secondo",
    	seconds: "secondi",
    	x_days_ago: "{x} giorni fa",
    	x_hours_ago: "{x} ore fa",
    	x_minutes_ago: "{x} minuti fa",
    	x_months_ago: "{x} mesi fa",
    	x_seconds_ago: "{x} secondi fa",
    	x_years_ago: "{x} anni fa",
    	year: "anno",
    	years: "anni"
    };
    var it = {
    	common: common$7,
    	config: config$7,
    	confirm: confirm$7,
    	dialog_about: dialog_about$6,
    	dialog_add_repo: dialog_add_repo$6,
    	dialog_custom_repositories: dialog_custom_repositories$6,
    	dialog_info: dialog_info$6,
    	dialog_install: dialog_install$6,
    	dialog_update: dialog_update$6,
    	entry: entry$6,
    	menu: menu$6,
    	options: options$7,
    	repository_banner: repository_banner$7,
    	repository_card: repository_card$6,
    	repository: repository$7,
    	search: search$6,
    	sections: sections$6,
    	settings: settings$7,
    	store: store$7,
    	time: time$7
    };

    var it$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$7,
        config: config$7,
        confirm: confirm$7,
        dialog_about: dialog_about$6,
        dialog_add_repo: dialog_add_repo$6,
        dialog_custom_repositories: dialog_custom_repositories$6,
        dialog_info: dialog_info$6,
        dialog_install: dialog_install$6,
        dialog_update: dialog_update$6,
        entry: entry$6,
        menu: menu$6,
        options: options$7,
        repository_banner: repository_banner$7,
        repository_card: repository_card$6,
        repository: repository$7,
        search: search$6,
        sections: sections$6,
        settings: settings$7,
        store: store$7,
        time: time$7,
        'default': it
    });

    var common$8 = {
    	about: "Om",
    	add: "legg til",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "AppDaemon Apper",
    	appdaemon_plural: "AppDaemon-apper",
    	background_task: "Bakgrunnsoppgaven kjører. Denne siden lastes inn på nytt når den er ferdig.",
    	check_log_file: "Sjekk loggfilen din for mer informasjon.",
    	"continue": "Fortsett",
    	disabled: "Deaktivert",
    	documentation: "dokumentasjon",
    	element: "element",
    	hacs_is_disabled: "HACS er deaktivert",
    	install: "Installer",
    	installed: "Installert",
    	integration: "Integrasjon",
    	integration_plural: "Integrasjoner",
    	integrations: "Integrasjoner",
    	lovelace: "Lovelace",
    	lovelace_element: "Lovelace-element",
    	lovelace_elements: "Lovelace-elementer",
    	manage: "manage",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDaemon Apper",
    	netdaemon_plural: "NetDaemon-apper",
    	plugin: "Lovelace",
    	plugin_plural: "Lovelace-elementer",
    	plugins: "Lovelace-elementer",
    	python_script: "Python-skript",
    	python_script_plural: "Python-skript",
    	python_scripts: "Python-skript",
    	repositories: "Repositories",
    	repository: "Repository",
    	settings: "Innstillinger",
    	theme: "Tema",
    	theme_plural: "Temaer",
    	themes: "Temaer",
    	uninstall: "Avinstaller",
    	update: "Oppdater",
    	version: "Versjon"
    };
    var config$8 = {
    	abort: {
    		single_instance_allowed: "Bare en konfigurasjon av HACS er tillatt."
    	},
    	error: {
    		auth: "Personlig tilgangstoken er ikke korrekt."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Aktiver oppdagelse og sporing av AppDaemon-apper",
    				netdaemon: "Aktiver oppdagelse og sporing av NetDaemon-apper",
    				python_script: "Aktiver oppdagelse og sporing av python_scripts",
    				sidepanel_icon: "Sidepanel ikon",
    				sidepanel_title: "Sidepanel tittel",
    				theme: "Aktiver oppdagelse og sporing av temaer",
    				token: "GitHub Personal Access Token"
    			},
    			description: "Hvis du trenger hjelp med konfigurasjonen, ta en titt her: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$8 = {
    	add_to_lovelace: "Er du sikker på at du vil legge dette til i dine Lovelace resources?",
    	bg_task: "Handlingen er deaktivert mens bakgrunnsoppgaver kjører.",
    	cancel: "Avbryt",
    	"continue": "Er du sikker på at du vil fortsette?",
    	"delete": "Er du sikker på at du vil fjerne '{item}'?",
    	delete_installed: "'{item}' er installert, du må avinstallere det før du kan slette det.",
    	exist: "{item} eksisterer allerede",
    	generic: "Er du sikker?",
    	home_assistant_is_restarting: "Vent, Home Assistant starter nå på nytt.",
    	home_assistant_version_not_correct: "Du kjører Home Assistant '{haversion}', men denne krever minimum versjon '{minversion}' for å bli installert.",
    	no: "Nei",
    	no_upgrades: "Ingen oppgraderinger tilgjengelig",
    	ok: "OK",
    	overwrite: "Å gjøre dette vil overskrive det.",
    	reload_data: "Dette laster inn dataene til alle repositories HACS vet om, dette vil ta litt tid å fullføre.",
    	restart_home_assistant: "Er du sikker på at du vil starte Home Assistant på nytt?",
    	uninstall: "Er du sikker på at du vil avinstallere '{item}'?",
    	upgrade_all: "Dette vil oppgradere alle disse repositorene, sørg for at du har lest utgivelses notatene for dem alle før du fortsetter.",
    	yes: "Ja"
    };
    var dialog_about$7 = {
    	frontend_version: "Frontend versjon",
    	installed_repositories: "Installerte repositories",
    	integration_version: "Integrasjonsversjon",
    	useful_links: "Nyttige lenker"
    };
    var dialog_add_repo$7 = {
    	limit: "Bare de første 100 elementene vises, bruk søket til å filtrere det du trenger",
    	no_match: "Ingen elementer funnet som samsvarer med filteret ditt",
    	title: "Legg til repository"
    };
    var dialog_custom_repositories$7 = {
    	category: "Kategori",
    	no_category: "Mangler kategori",
    	no_repository: "Mangler repository",
    	title: "Custom repositories",
    	url_placeholder: "Legg til custom repository"
    };
    var dialog_info$7 = {
    	author: "Utgiver",
    	downloads: "Nedlastinger",
    	install: "Installer dette elementet i HACS",
    	loading: "Laster inn informasjon ...",
    	no_info: "Utvikleren har ikke gitt mer informasjon for dette elementet",
    	open_issues: "Åpne problemer",
    	open_repo: "Åpne repository",
    	stars: "Stjerner",
    	version_installed: "Versjon installert"
    };
    var dialog_install$7 = {
    	select_version: "Velg versjon",
    	show_beta: "Vis betaversjoner",
    	type: "Type",
    	url: "URL"
    };
    var dialog_update$7 = {
    	available_version: "Tilgjengelig versjon",
    	changelog: "Endringslogg",
    	installed_version: "Installert versjon",
    	title: "Oppdatering venter"
    };
    var entry$7 = {
    	information: "Informasjon",
    	intro: "Oppdateringer og viktige meldinger vises her hvis det er noen",
    	messages: {
    		disabled: {
    			content: "Sjekk loggfilen din for mer informasjon",
    			title: "HACS er deaktivert"
    		},
    		has_pending_tasks: {
    			content: "Noen elementer vises kanskje ikke før dette er fullført",
    			title: "Venter på bakgrunnsoppgaver"
    		},
    		startup: {
    			content: "HACS starter opp, i løpet av denne tiden kan det hende at noe informasjon mangler eller er feil",
    			title: "HACS starter opp"
    		}
    	},
    	pending_updates: "Oppdateringer er klare"
    };
    var menu$7 = {
    	about: "Om HACS",
    	clear: "Fjern alt nytt",
    	custom_repositories: "Custom repositories",
    	documentation: "Dokumentasjon",
    	open_issue: "Meld et problem",
    	reload: "Last inn vinduet på nytt"
    };
    var options$8 = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Aktiver oppdagelse og sporing av AppDaemon-apper",
    				country: "Filtrer med landskode.",
    				debug: "Aktiver debug",
    				experimental: "Aktiver eksperimentelle funksjoner",
    				netdaemon: "Aktiver oppdagelse og sporing av NetDaemon-apper",
    				not_in_use: "Ikke i bruk med YAML",
    				release_limit: "Antall utgivelser som skal vises.",
    				sidepanel_icon: "Sidepanel ikon",
    				sidepanel_title: "Sidepanel tittel"
    			}
    		}
    	}
    };
    var repository_banner$8 = {
    	config_flow: "Denne integrasjonen støtter config_flow, det betyr at du nå kan gå til integrasjoner i brukergrensesnittet for å konfigurere den.",
    	config_flow_title: "UI konfigurasjon støttet",
    	integration_not_loaded: "Integrasjonen er ikke lastet inn i Home Assistant.",
    	no_restart_required: "Ingen omstart kreves",
    	not_loaded: "Ikke lastet inn",
    	plugin_not_loaded: "Dette elementet er ikke lagt til i lovelace under \"resource\" delen av konfigurasjonen.",
    	restart: "Du må restart Home Assistant",
    	restart_pending: "Restart er nødvendig"
    };
    var repository_card$7 = {
    	hide: "Skjul",
    	information: "Informasjon",
    	new_repository: "Ny",
    	open_issue: "Meld et problem",
    	open_source: "Åpne kilde",
    	pending_update: "Oppdatering venter",
    	reinstall: "Installer på nytt",
    	report: "Rapporter for fjerning",
    	update_information: "Oppdater informasjon"
    };
    var repository$8 = {
    	add_to_lovelace: "Legg til i Lovelace",
    	authors: "Laget av",
    	available: "Tilgjengelig",
    	back_to: "Tilbake til",
    	changelog: "Endringslogg",
    	downloads: "Nedlastinger",
    	flag_this: "Flag dette",
    	frontend_version: "Frontend versjon",
    	github_stars: "GitHub-stjerner",
    	goto_integrations: "Gå til integrasjoner",
    	hide: "Skjul",
    	hide_beta: "Skjul beta",
    	installed: "Installert",
    	lovelace_copy_example: "Kopier eksemplet til utklippstavlen",
    	lovelace_instruction: "Når du legger til dette i lovelace-konfigurasjonen din, bruk dette",
    	lovelace_no_js_type: "Kunne ikke bestemme typen for dettte elementet, sjekk repository.",
    	newest: "Nyeste",
    	note_appdaemon: "du må fortsatt legge den til i 'apps.yaml' filen",
    	note_installed: "Når det er installert, vil dette ligge i",
    	note_integration: "du må fortsatt legge den til 'configuration.yaml' filen",
    	note_plugin: "du må fortsatt legge den til i lovelace-konfigurasjonen ('ui-lovelace.yaml' eller den rå UI-konfigurasjonsredigereren)",
    	note_plugin_post_107: "du må fortsatt legge den til i lovelace konfigurasjonen ('configuration.yaml' eller via resource behanleren i grensesnittet '/config/lovelace/resources')",
    	open_issue: "Meld et problem",
    	open_plugin: "Åpne kilde",
    	reinstall: "Installer på nytt",
    	repository: "Repository",
    	restart_home_assistant: "Start Home Assistant på nytt",
    	show_beta: "Vis beta",
    	uninstall: "Avinstaller",
    	update_information: "Oppdater informasjon",
    	upgrade: "Oppdater"
    };
    var search$7 = {
    	placeholder: "Søk etter repository"
    };
    var sections$7 = {
    	about: {
    		description: "Vis informasjon om HACS",
    		title: "Om"
    	},
    	automation: {
    		description: "Det er her du finner python_scripts, AppDaemon-apper og NetDaemon-apper",
    		title: "Automasjon"
    	},
    	frontend: {
    		description: "Det er her du finner temaer, tilpassede kort og andre elementer for lovelace",
    		title: "Frontend"
    	},
    	integrations: {
    		description: "Det er her du finner tilpassede integrasjoner (custom_components)",
    		title: "Integrasjoner"
    	},
    	pending_repository_upgrade: "Du kjører versjon {installed} , versjon {available} er tilgjengelig"
    };
    var settings$8 = {
    	add_custom_repository: "LEGG TIL REPOSITORY",
    	adding_new_repo: "Legger til ny repository '{repo}'",
    	adding_new_repo_category: "Med kategori '{category}'.",
    	bg_task_custom: "Custom repositories er skjult mens bakgrunnsoppgaver kjører.",
    	category: "Kategori",
    	compact_mode: "Kompakt modus",
    	custom_repositories: "TILPASSEDE REPOSITORIER",
    	"delete": "Slett",
    	display: "Vise",
    	grid: "Nett",
    	hacs_repo: "HACS repo",
    	hidden_repositories: "Gjemte repositories",
    	missing_category: "Du må velge en kategori",
    	open_repository: "Åpne repository",
    	reload_data: "Last inn data på nytt",
    	reload_window: "Last inn vinduet på nytt",
    	repository_configuration: "Repository konfigurasjon",
    	save: "Lagre",
    	table: "Tabell",
    	table_view: "Tabellvisning",
    	unhide: "Vis igjen",
    	upgrade_all: "Oppgradere alle"
    };
    var store$8 = {
    	ascending: "stigende",
    	clear_new: "Tøm nye repositories",
    	descending: "synkende",
    	last_updated: "Sist oppdatert",
    	name: "Navn",
    	new_repositories: "Nye repositories",
    	no_repositories: "Ingen repositories",
    	no_repositories_desc1: "Det virker som om du ikke har noen elementer installert i denne delen ennå.",
    	no_repositories_desc2: "Klikk på + i nederste hjørne for å legge til din første!",
    	pending_upgrades: "Venter på oppgradering",
    	placeholder_search: "Skriv inn et søkeord ...",
    	sort: "sorter",
    	stars: "Stjerner",
    	status: "Status"
    };
    var time$8 = {
    	ago: "siden",
    	day: "dag",
    	days: "dager",
    	hour: "time",
    	hours: "timer",
    	minute: "minutt",
    	minutes: "minutter",
    	month: "måned",
    	months: "måneder",
    	one: "En",
    	one_day_ago: "for en dag siden",
    	one_hour_ago: "en time siden",
    	one_minute_ago: "ett minutt siden",
    	one_month_ago: "en måned siden",
    	one_second_ago: "ett sekund siden",
    	one_year_ago: "ett år siden",
    	second: "sekund",
    	seconds: "sekunder",
    	x_days_ago: "{x} dager siden",
    	x_hours_ago: "{x} timer siden",
    	x_minutes_ago: "{x} minutter siden",
    	x_months_ago: "{x} måneder siden",
    	x_seconds_ago: "{x} sekunder siden",
    	x_years_ago: "{x} år siden",
    	year: "år",
    	years: "år"
    };
    var nb = {
    	common: common$8,
    	config: config$8,
    	confirm: confirm$8,
    	dialog_about: dialog_about$7,
    	dialog_add_repo: dialog_add_repo$7,
    	dialog_custom_repositories: dialog_custom_repositories$7,
    	dialog_info: dialog_info$7,
    	dialog_install: dialog_install$7,
    	dialog_update: dialog_update$7,
    	entry: entry$7,
    	menu: menu$7,
    	options: options$8,
    	repository_banner: repository_banner$8,
    	repository_card: repository_card$7,
    	repository: repository$8,
    	search: search$7,
    	sections: sections$7,
    	settings: settings$8,
    	store: store$8,
    	time: time$8
    };

    var nb$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$8,
        config: config$8,
        confirm: confirm$8,
        dialog_about: dialog_about$7,
        dialog_add_repo: dialog_add_repo$7,
        dialog_custom_repositories: dialog_custom_repositories$7,
        dialog_info: dialog_info$7,
        dialog_install: dialog_install$7,
        dialog_update: dialog_update$7,
        entry: entry$7,
        menu: menu$7,
        options: options$8,
        repository_banner: repository_banner$8,
        repository_card: repository_card$7,
        repository: repository$8,
        search: search$7,
        sections: sections$7,
        settings: settings$8,
        store: store$8,
        time: time$8,
        'default': nb
    });

    var common$9 = {
    	about: "Over",
    	add: "toevoegen",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "AppDaemon Apps",
    	appdaemon_plural: "AppDaemon Apps",
    	background_task: "Achtergrond taak is draaiende, de pagina herhaalt zichzelf wanneer dit klaar is.",
    	check_log_file: "Controleer het logbestand voor meer details.",
    	"continue": "Doorgaan",
    	disabled: "Uitgeschakeld",
    	documentation: "Documentatie",
    	element: "element",
    	hacs_is_disabled: "HACS is uitgeschakeld",
    	install: "Installeer",
    	installed: "geinstalleerd",
    	integration: "Integratie",
    	integration_plural: "Integraties",
    	integrations: "Integraties",
    	lovelace: "Lovelace",
    	lovelace_element: "Lovelace-element",
    	lovelace_elements: "Lovelace-elementen",
    	manage: "beheer",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDaemon Apps",
    	netdaemon_plural: "NetDaemon Apps",
    	plugin: "Lovelace",
    	plugin_plural: "Lovelace-elementen",
    	plugins: "Plugins",
    	python_script: "Python Script",
    	python_script_plural: "Python Scripts",
    	python_scripts: "Python Scripts",
    	repositories: "Repositories",
    	repository: "Repository",
    	settings: "instellingen",
    	theme: "Thema",
    	theme_plural: "Themas",
    	themes: "Themas",
    	uninstall: "Verwijderen",
    	update: "Update",
    	version: "Versie"
    };
    var config$9 = {
    	abort: {
    		single_instance_allowed: "Je kunt maar een enkele configuratie van HACS tegelijk hebben."
    	},
    	error: {
    		auth: "Persoonlijke Toegang Token is niet correct."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Zet AppDaemon apps ontdekken & traceren aan",
    				netdaemon: "Zet NetDaemon apps ontdekken & traceren aan",
    				python_script: "Zet python_scripts ontdekken & traceren aan",
    				sidepanel_icon: "Zijpaneel icoon",
    				sidepanel_title: "Zijpaneel titel",
    				theme: "Zet Themes ontdekken & traceren aan",
    				token: "GitHub Persoonlijke Toegang Token"
    			},
    			description: "Als je hulp nodig hebt met de configuratie, kun je hier verder kijken: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$9 = {
    	add_to_lovelace: "Weet u zeker dat u dit wilt toevoegen aan uw Lovelace bronnen?",
    	bg_task: "Actie is geblokkeerd terwijl achtergrondtaken actief zijn.",
    	cancel: "Annuleer",
    	"continue": "Weet je zeker dat je wilt doorgaan?",
    	"delete": "Weet u zeker dat u '{item}' wilt verwijderen?",
    	delete_installed: "'{item}' is geïnstalleerd, je dient het eerst te deïnstalleren voordat je het kan verwijderen.",
    	exist: "{item} bestaat al.",
    	generic: "Weet je het zeker?",
    	home_assistant_is_restarting: "Een moment alstublieft, Home Assistant is aan het herstarten.",
    	home_assistant_version_not_correct: "Je gebruikt Home Assistant versie '{haversion}', echter deze repository vereist dat minimaal versie '{minversion}' is geïnstalleerd.",
    	no: "Nee",
    	no_upgrades: "Geen upgrades in afwachting.",
    	ok: "Oké",
    	overwrite: "Door dit te doen, wordt het overschreven.",
    	reload_data: "Dit zal alle bekende data herladen van alle repositories van HACS. Dit kan even duren",
    	restart_home_assistant: "Weet u zeker dat u Home Assistant opnieuw wilt starten?",
    	uninstall: "Weet u zeker dat u '{item}' wilt verwijderen?",
    	upgrade_all: "Hiermee worden al deze repositories geüpgraded. Zorg ervoor dat u de release-opmerkingen van allen heeft gelezen voordat u doorgaat.",
    	yes: "Ja"
    };
    var dialog_about$8 = {
    	frontend_version: "Frontend versie",
    	installed_repositories: "Geïnstalleerde repositories",
    	integration_version: "Integratieversie",
    	useful_links: "Nuttige links"
    };
    var dialog_add_repo$8 = {
    	limit: "Alleen de eerste 100 repositories worden getoond, gebruik de zoekopdracht om te filteren wat je nodig hebt",
    	no_match: "Er zijn geen repositories gevonden die overeenkomen met uw filter",
    	sort_by: "Sorteren op",
    	title: "Repository toevoegen"
    };
    var dialog_custom_repositories$8 = {
    	category: "Categorie",
    	no_category: "Ontbrekende categorie",
    	no_repository: "Ontbrekende repository",
    	title: "Aangepaste repositories",
    	url_placeholder: "Voeg een aangepaste repository-URL toevoegen"
    };
    var dialog_info$8 = {
    	author: "Auteur",
    	downloads: "Downloads",
    	install: "Installeer deze repository in HACS",
    	loading: "Informatie laden ...",
    	no_info: "De ontwikkelaar heeft geen verdere informatie verstrekt voor deze repository",
    	open_issues: "Openstaande problemen",
    	open_repo: "Open repository",
    	stars: "Sterren",
    	version_installed: "Versie geïnstalleerd"
    };
    var dialog_install$8 = {
    	restart: "Onthoud dat u Home Assistant opnieuw moet opstarten voordat wijzigingen aan integraties (custom_components) worden toegepast.",
    	select_version: "Selecteer versie",
    	show_beta: "Bètaversies weergeven",
    	type: "Type",
    	url: "URL"
    };
    var dialog_update$8 = {
    	available_version: "Beschikbare versie",
    	changelog: "Changelog",
    	installed_version: "Geïnstalleerde versie",
    	releasenotes: "Releasenotes voor {release}",
    	title: "Update in behandeling"
    };
    var entry$8 = {
    	information: "Informatie",
    	intro: "Updates en belangrijke berichten worden hier weergegeven als die er zijn",
    	messages: {
    		disabled: {
    			content: "Controleer uw logbestand voor meer details",
    			title: "HACS is uitgeschakeld"
    		},
    		has_pending_tasks: {
    			content: "Sommige repositories worden mogelijk pas weergegeven als dit is voltooid",
    			title: "Achtergrondtaken in behandeling"
    		},
    		resources: {
    			content: "Je hebt {number} Lovelace-elementen die niet correct zijn geladen in Lovelace.",
    			title: "Niet geladen in Lovelace"
    		},
    		restart: {
    			content: "Je hebt {number} integraties waarvoor de Home Assistant opnieuw moet worden opgestart, je kunt dat doen via het gedeelte 'Serverbeheer' onder het configuratiegedeelte van de Home Assistant UI.",
    			title: "In afwachting van herstart"
    		},
    		startup: {
    			content: "HACS is aan het opstarten, gedurende deze tijd kan er informatie ontbreken of onjuist zijn",
    			title: "HACS is aan het opstarten"
    		},
    		wrong_frontend_installed: {
    			content: "Je hebt {running} van de HACS-frontend geïnstalleerd, maar versie {expected} werd verwacht. Als je dit bericht ziet, kon Home Assistant de nieuwe versie niet installeren, probeer dan Home Assistant opnieuw op te starten.",
    			title: "Onverwachte frontend-versie"
    		},
    		wrong_frontend_loaded: {
    			content: "U gebruikt versie {running} van de HACS-frontend, maar versie {expected} werd verwacht, u moet uw browsercache wissen.",
    			title: "Onverwachte frontend-versie"
    		}
    	},
    	pending_updates: "In afwachting van updates"
    };
    var menu$8 = {
    	about: "Over HACS",
    	clear: "Wis alle nieuwe",
    	custom_repositories: "Aangepaste repositories",
    	dismiss: "Sluit alle nieuwe repositories af",
    	documentation: "Documentatie",
    	open_issue: "Meld probleem",
    	reload: "Herlaad venster"
    };
    var options$9 = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Zet AppDaemon apps ontdekken & traceren aan",
    				country: "Filter met land code.",
    				debug: "Schakel debug in.",
    				experimental: "Zet experimentele functies aan",
    				netdaemon: "Zet NetDaemon apps ontdekken & traceren aan",
    				not_in_use: "Niet in gebruik met YAML",
    				release_limit: "Aantal releases om te laten zien.",
    				sidepanel_icon: "Zijpaneel icoon",
    				sidepanel_title: "Zijpaneel titel"
    			}
    		}
    	}
    };
    var repository_banner$9 = {
    	config_flow: "Deze integratie ondersteunt config_flow, wat betekent dat u via uw \"Instellingen\" naar \"Integraties\" kunt gaan om het te configureren.",
    	config_flow_title: "UI-configuratie ondersteund",
    	integration_not_loaded: "Deze integratie wordt niet geladen in Home Assistant.",
    	no_restart_required: "Geen herstart vereist",
    	not_loaded: "Niet geladen",
    	plugin_not_loaded: "Deze plugin wordt niet toegevoegd aan je Lovelace resources.",
    	restart: "U moet Home Assistant opnieuw starten.",
    	restart_pending: "Wachten op herstart"
    };
    var repository_card$8 = {
    	dismiss: "afwijzen",
    	hide: "Verbergen",
    	information: "Informatie",
    	new_repository: "Nieuwe repository",
    	not_loaded: "Niet geladen",
    	open_issue: "Meld probleem",
    	open_source: "Open source",
    	pending_restart: "In afwachting van herstart",
    	pending_update: "In afwachting van update",
    	reinstall: "Herinstalleer",
    	report: "Rapport voor verwijdering",
    	update_information: "Update informatie"
    };
    var repository$9 = {
    	add_to_lovelace: "Toevoegen aan Lovelace",
    	authors: "Auteurs",
    	available: "Beschikbaar",
    	back_to: "Terug naar",
    	changelog: "Changelog",
    	downloads: "Downloads",
    	flag_this: "Vlag dit",
    	frontend_version: "Frontend versie",
    	github_stars: "GitHub-sterren",
    	goto_integrations: "Ga naar integraties",
    	hide: "Verberg",
    	hide_beta: "Verberg beta",
    	install: "Installeer",
    	installed: "Geinstalleerd",
    	lovelace_copy_example: "Kopier het voorbeeld naar je klembord",
    	lovelace_instruction: "Wanneer je dit gaat toevoegen aan je lovelace configuratie gebruik dit",
    	lovelace_no_js_type: "Kon niet achterhalen welk type plugin dit is, controleer de repository van de plugin.",
    	newest: "nieuwste",
    	note_appdaemon: "je moet het nog steeds toevoegen aan je 'apps.yaml' bestand",
    	note_installed: "Wanneer geïnstalleerd, staat het in",
    	note_integration: "je moet het nog steeds toevoegen aan je 'configuration.yaml' bestand",
    	note_plugin: "je moet het nog steeds toevoegen aan je lovelace configuratie ('ui-lovelace.yaml') of raw UI config editor.",
    	note_plugin_post_107: "je moet het nog steeds toevoegen aan je lovelace configuratie ('configuration.yaml' of de resource editor '/config/lovelace/resources')",
    	open_issue: "Meld probleem",
    	open_plugin: "Open plugin",
    	reinstall: "Herinstalleer",
    	repository: "Repository",
    	restart_home_assistant: "Start Home Assistant opnieuw",
    	show_beta: "Laat beta zien",
    	uninstall: "Verwijder",
    	update_information: "Update informatie",
    	upgrade: "Update"
    };
    var search$8 = {
    	placeholder: "Zoek naar repository"
    };
    var sections$8 = {
    	about: {
    		description: "Toon informatie over HACS",
    		title: "Over"
    	},
    	automation: {
    		description: "Hier vind je python_scripts, AppDaemon-apps en NetDaemon-apps",
    		title: "Automatisering"
    	},
    	frontend: {
    		description: "Dit is waar je thema's, aangepaste kaarten en andere elementen voor lovelace",
    		title: "Frontend"
    	},
    	integrations: {
    		description: "Hier vindt u aangepaste integraties (custom_components)",
    		title: "Integraties"
    	},
    	pending_repository_upgrade: "U gebruikt versie {installed} , versie {available} is beschikbaar"
    };
    var settings$9 = {
    	add_custom_repository: "VOEG EIGEN REPOSITORY TOE",
    	adding_new_repo: "Nieuwe repository '{repo}' toevoegen",
    	adding_new_repo_category: "Met categorie '{category}'.",
    	bg_task_custom: "Aangepaste repositories zijn verborgen terwijl de achtergrondtaken actief zijn.",
    	category: "Categorie",
    	compact_mode: "Compacte modus",
    	custom_repositories: "EIGEN REPOSITORIES",
    	"delete": "Verwijder",
    	display: "Weergave",
    	grid: "Rooster",
    	hacs_repo: "HACS repo",
    	hidden_repositories: "verborgen repositories",
    	missing_category: "Je moet een categorie selecteren.",
    	open_repository: "Open repository",
    	reload_data: "Herlaad data",
    	reload_window: "Herlaad venster",
    	repository_configuration: "Repository configuratie",
    	save: "Opslaan",
    	table: "Tabel",
    	table_view: "Tabelweergave",
    	unhide: "zichtbaar maken",
    	upgrade_all: "Upgrade alles"
    };
    var store$9 = {
    	ascending: "oplopend",
    	clear_new: "Wissen van alle nieuwe repositories",
    	descending: "Aflopend",
    	last_updated: "Laatste update",
    	name: "Naam",
    	new_repositories: "Nieuwe Repositories",
    	new_repositories_note: "Je hebt meer dan 10 nieuwe repositories die hier worden weergegeven, als je ze allemaal wilt wissen, klik dan op de 3 puntjes in de rechterbovenhoek en wijs ze af.",
    	no_repositories: "Geen repositories",
    	no_repositories_desc1: "Het lijkt erop dat je nog geen repositories hebt geïnstalleerd in deze sectie.",
    	no_repositories_desc2: "Klik op de + in de benedenhoek om je eerste toe te voegen!",
    	no_repositories_found_desc1: "Er zijn geen geïnstalleerde repositories die overeenkomen met {searchInput}\" in deze sectie.",
    	no_repositories_found_desc2: "Probeer iets anders te zoeken!",
    	pending_upgrades: "Upgrades in afwachting",
    	placeholder_search: "Typ iets om te zoeken...",
    	sort: "sorteer",
    	stars: "Sterren",
    	status: "Status"
    };
    var time$9 = {
    	ago: "geleden",
    	day: "dag",
    	days: "dagen",
    	hour: "uur",
    	hours: "uren",
    	minute: "minuut",
    	minutes: "minuten",
    	month: "maand",
    	months: "maanden",
    	one: "Eén",
    	one_day_ago: "een dag geleden",
    	one_hour_ago: "een uur geleden",
    	one_minute_ago: "een minuut geleden",
    	one_month_ago: "een maand geleden",
    	one_second_ago: "een seconde geleden",
    	one_year_ago: "een jaar geleden",
    	second: "seconde",
    	seconds: "seconden",
    	x_days_ago: "{x} dagen geleden",
    	x_hours_ago: "{x} uur geleden",
    	x_minutes_ago: "{x} minuten geleden",
    	x_months_ago: "{x} maanden geleden",
    	x_seconds_ago: "{x} seconden geleden",
    	x_years_ago: "{x} jaar geleden",
    	year: "jaar",
    	years: "jaren"
    };
    var nl = {
    	common: common$9,
    	config: config$9,
    	confirm: confirm$9,
    	dialog_about: dialog_about$8,
    	dialog_add_repo: dialog_add_repo$8,
    	dialog_custom_repositories: dialog_custom_repositories$8,
    	dialog_info: dialog_info$8,
    	dialog_install: dialog_install$8,
    	dialog_update: dialog_update$8,
    	entry: entry$8,
    	menu: menu$8,
    	options: options$9,
    	repository_banner: repository_banner$9,
    	repository_card: repository_card$8,
    	repository: repository$9,
    	search: search$8,
    	sections: sections$8,
    	settings: settings$9,
    	store: store$9,
    	time: time$9
    };

    var nl$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$9,
        config: config$9,
        confirm: confirm$9,
        dialog_about: dialog_about$8,
        dialog_add_repo: dialog_add_repo$8,
        dialog_custom_repositories: dialog_custom_repositories$8,
        dialog_info: dialog_info$8,
        dialog_install: dialog_install$8,
        dialog_update: dialog_update$8,
        entry: entry$8,
        menu: menu$8,
        options: options$9,
        repository_banner: repository_banner$9,
        repository_card: repository_card$8,
        repository: repository$9,
        search: search$8,
        sections: sections$8,
        settings: settings$9,
        store: store$9,
        time: time$9,
        'default': nl
    });

    var common$a = {
    	about: "Om",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "AppDeamon-appar",
    	appdaemon_plural: "AppDaemon-appar",
    	background_task: "Bakgrunnsoppgåve køyrer. Denne sida kjem til å laste seg omatt når ho er ferdig.",
    	check_log_file: "Sjå i loggfila di for meir detaljar.",
    	"continue": "Hald fram",
    	disabled: "Deaktivert",
    	documentation: "Dokumentasjon",
    	element: "element",
    	hacs_is_disabled: "HACS er deaktivert",
    	installed: "Installert",
    	integration: "Integrasjon",
    	integration_plural: "Integrasjonar",
    	integrations: "Integrasjonar",
    	lovelace: "Lovelace",
    	lovelace_element: "Lovelace-element",
    	lovelace_elements: "Lovelace-element",
    	manage: "Handtere",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDeamon-appar",
    	netdaemon_plural: "NetDaemon-appar",
    	plugin: "Lovelace",
    	plugin_plural: "Lovelace-element",
    	plugins: "Lovelace-element",
    	python_script: "Pythonskript",
    	python_script_plural: "Pythonskript",
    	python_scripts: "Pythonskript",
    	repositories: "Repositories",
    	settings: "innstillingar",
    	theme: "Tema",
    	theme_plural: "Tema",
    	themes: "Tema",
    	version: "Versjon"
    };
    var config$a = {
    	abort: {
    		single_instance_allowed: "Berre éin enkelt konfigurasjon av HACS er tillete."
    	},
    	error: {
    		auth: "Personleg tilgangsnøkkel er ikkje korrekt."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Aktiver AppDeamon-appar-oppdaging og sporing",
    				netdaemon: "Aktiver NetDeamon-appar-oppdaging og sporing",
    				python_script: "Aktiver pythonscript-oppdaging og sporing",
    				sidepanel_icon: "Sidepanelikon",
    				sidepanel_title: "Sidepaneltittel",
    				theme: "Aktiver tema-oppdaging og sporing",
    				token: "Personleg GitHub tilgangsnøkkel"
    			},
    			description: "Dersom du treng hjelp med konfigurasjonen, ta ein kik her: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$a = {
    	add_to_lovelace: "Er du sikker på at du vil legge til dette i Lovelace-ressursane dine?",
    	bg_task: "Handlinga er deaktivert medan bakgrunnsoppgåveene køyrer.",
    	cancel: "Avbryt",
    	"continue": "Er du sikker på at du vil halde fram?",
    	"delete": "Er du sikker på at du vil slette '{item}'?",
    	delete_installed: "'{item}' er installert. Du må avinstallere det før du kan slette det.",
    	exist: "{item} eksisterer allereie",
    	generic: "Er du sikker?",
    	home_assistant_is_restarting: "Vent... Home Assistant starter på nytt no.",
    	home_assistant_version_not_correct: "Du køyrer Home Assistant-versjonen '{haversion}', men dette kodedepoet krev minst versjon '{minversion}' for å bli installert.",
    	no: "Nei",
    	no_upgrades: "Ingen ventande oppgradringer",
    	ok: "OK",
    	overwrite: "Ved å gjere dette kjem du til å overskrive.",
    	reload_data: "Dette laster inn dataa til depota HACS veit om, og dette vil ta litt tid å fullføre.",
    	restart_home_assistant: "Er du sikker på at du vil starte Home Assistant på nytt?",
    	uninstall: "Er du sikker på at du vil avinstallere '{item}'?",
    	upgrade_all: "Dette kjem til å oppgradere alle depota. Ver sikker på at du har lest alle versjonsmerknadene før du held fram.",
    	yes: "Ja"
    };
    var options$a = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Aktiver AppDeamon-appar-oppdaging og sporing",
    				country: "Filterer med landskode",
    				debug: "Aktiver debug.",
    				experimental: "Aktiver ekspreimentelle funksjonar",
    				netdaemon: "Aktiver NetDeamon-appar-oppdaging og sporing",
    				not_in_use: "Kan ikkje brukast saman med YAML",
    				release_limit: "Talet på utgivingar",
    				sidepanel_icon: "Sidepanelikon",
    				sidepanel_title: "Sidepaneltittel"
    			}
    		}
    	}
    };
    var repository_banner$a = {
    	config_flow: "Denne integrasjonen støttar config_flow, som betyr at du no kan gå til integrasjonssida i brukargrensesnittet for å konfigurere den.",
    	config_flow_title: "UI-konfigurasjon støtta",
    	integration_not_loaded: "Integrasjonen er ikkje lasta inn i Home Assistant.",
    	no_restart_required: "Ingen omstart kravd",
    	not_loaded: "Ikkje lasta",
    	plugin_not_loaded: "Tillegget er ikkje lagt til i Lovelace-ressursane dine.",
    	restart: "Du må starte Home Assistant på nytt",
    	restart_pending: "Omstart venter"
    };
    var repository$a = {
    	add_to_lovelace: "Legg til i Lovelace",
    	authors: "Forfatter(e)",
    	available: "Tilgjengeleg",
    	back_to: "Tilbake til",
    	changelog: "Endre logg",
    	downloads: "Nedlastinger",
    	flag_this: "Marker dette",
    	frontend_version: "Frontend-versjon",
    	github_stars: "GitHub-stjerner",
    	goto_integrations: "Gå til integrasjonar",
    	hide: "Gøym",
    	hide_beta: "Gøym beta",
    	install: "Installer",
    	installed: "Installert",
    	lovelace_copy_example: "Kopier eksempelet til utklippsbreittet ditt",
    	lovelace_instruction: "Når du legg til dette i Lovelace-konfigurasjonen din, bruk dette",
    	lovelace_no_js_type: "Kunne ikkje slå fast typen til dette tilegget. Sjå i repositoryet.",
    	newest: "nyaste",
    	note_appdaemon: "du må framleis legge dette til i \"apps.yaml\"-fila di",
    	note_installed: "Når dette er installert, kjem den til å vere plassert i",
    	note_integration: "du må framleis legge dette til i \"configuration.yaml\"-fila di",
    	note_plugin: "du må framleis dette til i Lovelace-konfigurasjonen (\"ui-lovelace.yaml\" eller i rå-brukargrensesnittredigeraren",
    	note_plugin_post_107: "du må framleis legge dette til i lovelace-konfigurasjonen ('configuration.yaml' eller i kjelderedigeraren ''/config/lovelace/resources')",
    	open_issue: "Opne problem",
    	open_plugin: "Opne tillegg",
    	reinstall: "Installer på nytt",
    	repository: "Repository",
    	restart_home_assistant: "Start Home Assistant på nytt",
    	show_beta: "Vis beta",
    	uninstall: "Avinstaller",
    	update_information: "Oppdater informasjonen",
    	upgrade: "Oppdater"
    };
    var sections$9 = {
    	about: {
    		description: "Vis informasjon om HACS",
    		title: "Om"
    	},
    	automation: {
    		description: "Her finn du python_scripts, AppDaemon-appar og NetDaemon-appar",
    		title: "Automasjon"
    	},
    	frontend: {
    		description: "Her finn du tema, eigendefinerte kort og andre element for lovelace",
    		title: "Frontend"
    	},
    	integrations: {
    		description: "Her finn du eigendefinerte ingtegrasjonar (custom_components)",
    		title: "Integrasjonar"
    	},
    	pending_repository_upgrade: "Du køyrer versjon {installed}, og versjon {available} er tilgjengeleg"
    };
    var settings$a = {
    	add_custom_repository: "LEGG TIL EIN ANNAN REPOSITORY",
    	adding_new_repo: "Legger til ny repository '{repo}'",
    	adding_new_repo_category: "Med kategori '{category}'.",
    	bg_task_custom: "Custom repositories er skjult medan bakgrunnsoppgaver køyrer.",
    	category: "Kategori",
    	compact_mode: "Kompaktmodus",
    	custom_repositories: "VANLEG REPOSITORY",
    	"delete": "Slett",
    	display: "Vis",
    	grid: "rutenett",
    	hacs_repo: "HACS repo",
    	hidden_repositories: "gøymde repositories",
    	missing_category: "Du må velje éin kategori",
    	open_repository: "Opne repository",
    	reload_data: "Last om dataa",
    	reload_window: "Last inn vindauget på nytt",
    	repository_configuration: "Repository-konfigurasjon",
    	save: "Lagre",
    	table: "Tavle",
    	table_view: "Tabellvisning",
    	unhide: "vis",
    	upgrade_all: "Oppdater alle"
    };
    var store$a = {
    	ascending: "stigande",
    	clear_new: "Fjern alle nye repositories",
    	descending: "synkande",
    	last_updated: "Sist oppdatert",
    	name: "Namn",
    	new_repositories: "Ny repository",
    	pending_upgrades: "Ventande oppgraderinger",
    	placeholder_search: "Ver vennleg og skriv inn ei søkefrase",
    	sort: "Sorter",
    	stars: "Stjerner",
    	status: "Status"
    };
    var time$a = {
    	ago: "sidan",
    	day: "dag",
    	days: "dagar",
    	hour: "time",
    	hours: "timar",
    	minute: "minutt",
    	minutes: "minutt",
    	month: "månad",
    	months: "månadar",
    	one: "Éin",
    	one_day_ago: "for éin dag sidan",
    	one_hour_ago: "éin time sidan",
    	one_minute_ago: "eitt minutt sidan",
    	one_month_ago: "ein månad sidan",
    	one_second_ago: "eitt sekund sidan",
    	one_year_ago: "eitt år sidan",
    	second: "sekund",
    	seconds: "sekund",
    	x_days_ago: "{x} dagar siden",
    	x_hours_ago: "{x} timer sidan",
    	x_minutes_ago: "{x} minutt sidan",
    	x_months_ago: "{x} månadar sidan",
    	x_seconds_ago: "{x} sekund sidan",
    	x_years_ago: "{x} år sidan",
    	year: "år",
    	years: "år"
    };
    var nn = {
    	common: common$a,
    	config: config$a,
    	confirm: confirm$a,
    	options: options$a,
    	repository_banner: repository_banner$a,
    	repository: repository$a,
    	sections: sections$9,
    	settings: settings$a,
    	store: store$a,
    	time: time$a
    };

    var nn$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$a,
        config: config$a,
        confirm: confirm$a,
        options: options$a,
        repository_banner: repository_banner$a,
        repository: repository$a,
        sections: sections$9,
        settings: settings$a,
        store: store$a,
        time: time$a,
        'default': nn
    });

    var common$b = {
    	about: "O",
    	add: "dodaj",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "Aplikacje AppDaemon",
    	appdaemon_plural: "Aplikacje AppDaemon",
    	background_task: "Wykonywanie zadania w tle, ta strona zostanie odświeżona, gdy zadanie zostanie ukończone.",
    	check_log_file: "Sprawdź plik dziennika, aby uzyskać więcej informacji.",
    	"continue": "Kontynuuj",
    	disabled: "Wyłączony",
    	documentation: "Dokumentacja",
    	element: "element",
    	hacs_is_disabled: "HACS jest wyłączony",
    	install: "Zainstaluj",
    	installed: "zainstalowane",
    	integration: "Integracja",
    	integration_plural: "Integracje",
    	integrations: "Integracje",
    	lovelace: "Lovelace",
    	lovelace_element: "Element Lovelace",
    	lovelace_elements: "Elementy Lovelace",
    	manage: "zarządzaj",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "Aplikacje NetDaemon",
    	netdaemon_plural: "Aplikacje NetDaemon",
    	plugin: "Lovelace",
    	plugin_plural: "Elementy Lovelace",
    	plugins: "Elementy Lovelace",
    	python_script: "Skrypt Python",
    	python_script_plural: "Skrypty języka Python",
    	python_scripts: "Skrypty Python",
    	repositories: "Repozytoria",
    	repository: "Repozytorium",
    	settings: "ustawienia",
    	theme: "Motyw",
    	theme_plural: "Motywy",
    	themes: "Motywy",
    	uninstall: "Odinstaluj",
    	update: "Uaktualnij",
    	version: "Wersja"
    };
    var config$b = {
    	abort: {
    		single_instance_allowed: "Dozwolona jest tylko jedna konfiguracja HACS."
    	},
    	error: {
    		auth: "Osobisty token dostępu jest nieprawidłowy."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Włącz wykrywanie i śledzenie aplikacji AppDaemon",
    				netdaemon: "Włącz wykrywanie i śledzenie aplikacji NetDaemon",
    				python_script: "Włącz wykrywanie i śledzenie skryptów Python",
    				sidepanel_icon: "Ikona w panelu bocznym",
    				sidepanel_title: "Tytuł w panelu bocznym",
    				theme: "Włącz wykrywanie i śledzenie motywów",
    				token: "Osobisty token dostępu GitHub"
    			},
    			description: "Jeśli potrzebujesz pomocy w konfiguracji, przejdź na stronę: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$b = {
    	add_to_lovelace: "Na pewno chcesz dodać to do swoich zasobów Lovelace?",
    	bg_task: "Akcja jest wyłączona podczas wykonywania zadań w tle.",
    	cancel: "Anuluj",
    	"continue": "Na pewno chcesz kontynuować?",
    	"delete": "Na pewno chcesz usunąć '{item}'?",
    	delete_installed: "'{item}' jest zainstalowany, musisz go odinstalować zanim będziesz mógł go usunąć.",
    	exist: "{item} już istnieje",
    	generic: "Jesteś pewny?",
    	home_assistant_is_restarting: "Poczekaj, Home Assistant jest teraz ponownie uruchamiany.",
    	home_assistant_version_not_correct: "Używasz Home Assistant'a w wersji '{haversion}', a to repozytorium wymaga wersji minimum '{minversion}'.",
    	no: "Nie",
    	no_upgrades: "Brak oczekujących aktualizacji",
    	ok: "Ok",
    	overwrite: "Spowoduje to zastąpienie istniejącej kopii.",
    	reload_data: "To przeładowuje dane wszystkich repozytoriów, o których wie HACS, może to trochę potrwać.",
    	restart_home_assistant: "Na pewno chcesz ponownie uruchomić Home Assistant'a?",
    	uninstall: "Na pewno chcesz odinstalować '{item}'?",
    	upgrade_all: "To uaktualni wszystkie te repozytoria, upewnij się, że przeczytałeś uwagi do wydania dla wszystkich z nich przed kontynuacją.",
    	yes: "Tak"
    };
    var dialog_about$9 = {
    	frontend_version: "Wersja interfejsu użytkownika",
    	installed_repositories: "Zainstalowane repozytoria",
    	integration_version: "Wersja integracji",
    	useful_links: "Przydatne linki"
    };
    var dialog_add_repo$9 = {
    	limit: "Wyświetlanych jest tylko pierwszych 100 repozytoriów, użyj wyszukiwania, aby przefiltrować potrzebne informacje",
    	no_match: "Nie znaleziono repozytoriów pasujących do filtra",
    	sort_by: "Sortuj według",
    	title: "Dodawanie repozytorium"
    };
    var dialog_custom_repositories$9 = {
    	category: "Kategoria",
    	no_category: "Brak kategorii",
    	no_repository: "Brak repozytorium",
    	title: "Niestandardowe repozytoria",
    	url_placeholder: "Adres URL niestandardowego repozytorium"
    };
    var dialog_info$9 = {
    	author: "Autor",
    	downloads: "Ilość pobrań",
    	install: "Zainstaluj to repozytorium w HACS",
    	loading: "Pobieranie informacji...",
    	no_info: "Deweloper nie dostarczył więcej informacji na temat tego repozytorium",
    	open_issues: "Problemy",
    	open_repo: "Otwórz repozytorium",
    	stars: "Gwiazdki",
    	version_installed: "Wersja zainstalowana"
    };
    var dialog_install$9 = {
    	restart: "Pamiętaj, że musisz ponownie uruchomić Home Assistanta by zastosować zmiany w integracjach (custom_components).",
    	select_version: "Wybierz wersję",
    	show_beta: "Wyświetl wydania beta",
    	type: "Typ",
    	url: "URL"
    };
    var dialog_update$9 = {
    	available_version: "Dostępna wersja",
    	changelog: "Lista zmian",
    	installed_version: "Zainstalowana wersja",
    	releasenotes: "Informacje o {release}",
    	title: "Dostępna aktualizacja"
    };
    var entry$9 = {
    	information: "Informacje",
    	intro: "Aktualizacje i ważne komunikaty będą wyświetlane w tym miejscu",
    	messages: {
    		disabled: {
    			content: "Sprawdź log, aby uzyskać więcej informacji",
    			title: "HACS jest wyłączony"
    		},
    		has_pending_tasks: {
    			content: "Dopóki nie zostaną zakończone, niektóre repozytoria mogą nie być wyświetlane",
    			title: "Wykonywane są zadania w tle"
    		},
    		resources: {
    			content: "Elementy Lovelace, które nie zostały poprawnie załadowane: {number}",
    			title: "Nie załadowano w Lovelace"
    		},
    		restart: {
    			content: "Integracje, które wymagają ponownego uruchomienia Home Assistanta: {number}\nMożesz to zrobić w sekcji 'Kontrola serwera' konfiguracji Home Assistanta.",
    			title: "Oczekiwanie na restart"
    		},
    		startup: {
    			content: "HACS uruchamia się, w tym czasie może brakować pewnych informacji lub mogą one być nieprawidłowe.",
    			title: "HACS uruchamia się"
    		},
    		wrong_frontend_installed: {
    			content: "Masz zainstalowany interfejs HACS w wersji {running}, a wersja {expected} była oczekiwana. Komunikat ten oznacza, że Home Assistant nie mógł zainstalować nowej wersji interfejsu HACS, spróbuj ponownie uruchomić Home Assistanta.",
    			title: "Nieoczekiwana wersja interfejsu"
    		},
    		wrong_frontend_loaded: {
    			content: "Używasz wersji {running} interfejsu HACS, a wersja {expected} była oczekiwana, powinieneś wyczyścić pamięć podręczną przeglądarki.",
    			title: "Nieoczekiwana wersja interfejsu"
    		}
    	},
    	pending_updates: "Oczekujące aktualizacje"
    };
    var menu$9 = {
    	about: "O HACS",
    	clear: "Wyczyść oznaczenia nowych",
    	custom_repositories: "Niestandardowe repozytoria",
    	dismiss: "Odrzuć wszystkie nowe repozytoria",
    	documentation: "Dokumentacja",
    	open_issue: "Powiadom o problemie",
    	reload: "Załaduj ponownie okno"
    };
    var options$b = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Włącz wykrywanie i śledzenie aplikacji AppDaemon",
    				country: "Filtruj według kodu kraju",
    				debug: "Włącz debugowanie.",
    				experimental: "Włącz funkcje eksperymentalne",
    				netdaemon: "Włącz wykrywanie i śledzenie aplikacji NetDaemon",
    				not_in_use: "Nieużywany z YAML",
    				release_limit: "Liczba wydań do wyświetlenia",
    				sidepanel_icon: "Ikona w panelu bocznym",
    				sidepanel_title: "Tytuł w panelu bocznym"
    			}
    		}
    	}
    };
    var repository_banner$b = {
    	config_flow: "Ta integracja obsługuje config_flow, co oznacza, że możesz teraz przejść do sekcji integracji w interfejsie użytkownika, aby ją skonfigurować.",
    	config_flow_title: "Obsługiwana konfiguracja poprzez interfejs użytkownika",
    	integration_not_loaded: "Ta integracja nie jest załadowana do Home Assistant'a.",
    	no_restart_required: "Ponowne uruchomienie nie jest wymagane",
    	not_loaded: "Nie załadowano",
    	plugin_not_loaded: "Ta wtyczka nie jest dodana do zasobów Lovelace.",
    	restart: "Musisz ponownie uruchomić Home Assistant'a.",
    	restart_pending: "Oczekiwanie na ponowne uruchomienie"
    };
    var repository_card$9 = {
    	dismiss: "odrzuć",
    	hide: "Ukryj",
    	information: "Informacje",
    	new_repository: "Nowe repozytorium",
    	not_loaded: "Nie załadowano",
    	open_issue: "Powiadom o problemie",
    	open_source: "Otwarte źródło",
    	pending_restart: "Oczekiwanie na restart",
    	pending_update: "Oczekująca aktualizacja",
    	reinstall: "Przeinstaluj",
    	report: "Zgłoś do usunięcia",
    	update_information: "Uaktualnij informacje"
    };
    var repository$b = {
    	add_to_lovelace: "Dodaj do Lovelace",
    	authors: "Autorzy",
    	available: "Dostępna",
    	back_to: "Wróć do",
    	changelog: "Lista zmian",
    	downloads: "Ilość pobrań",
    	flag_this: "Oflaguj",
    	frontend_version: "Wersja frontendu",
    	github_stars: "Gwiazdki GitHub",
    	goto_integrations: "Przejdź do integracji",
    	hide: "Ukryj",
    	hide_beta: "Ukryj wydania beta",
    	install: "Zainstaluj",
    	installed: "Zainstalowano",
    	lovelace_copy_example: "Skopiuj przykład do schowka",
    	lovelace_instruction: "Interfejs użytkownika użyje tej wtyczki po dodaniu konfiguracji",
    	lovelace_no_js_type: "Nie można określić typu tej wtyczki, sprawdź repozytorium.",
    	newest: "najnowsza",
    	note_appdaemon: "musisz jeszcze dodać aplikację do pliku 'apps.yaml'",
    	note_installed: "Po zainstalowaniu dodatek będzie znajdować się w",
    	note_integration: "musisz jeszcze dodać integrację do pliku 'configuration.yaml'",
    	note_plugin: "musisz jeszcze dodać wtyczkę do konfiguracji interfejsu użytkownika (plik 'ui-lovelace.yaml' lub edytor interfejsu użytkownika)",
    	note_plugin_post_107: "nadal musisz dodać go do konfiguracji Lovelace ('configuration.yaml' lub edytora zasobów '/config/lovelace/resources')",
    	open_issue: "Powiadom o problemie",
    	open_plugin: "Otwórz element",
    	reinstall: "Przeinstaluj",
    	repository: "Repozytorium",
    	restart_home_assistant: "Uruchom ponownie Home Assistant'a",
    	show_beta: "Wyświetl wydania beta",
    	uninstall: "Odinstaluj",
    	update_information: "Uaktualnij informacje",
    	upgrade: "Uaktualnij"
    };
    var search$9 = {
    	placeholder: "Wyszukaj repozytorium"
    };
    var sections$a = {
    	about: {
    		description: "Informacje o HACS",
    		title: "O HACS"
    	},
    	automation: {
    		description: "Skrypty Pythona, aplikacje AppDaemon i NetDaemon",
    		title: "Automatyzacje"
    	},
    	frontend: {
    		description: "Motywy, niestandardowe karty i inne elementy interfejsu użytkownika",
    		title: "Interfejs użytkownika"
    	},
    	integrations: {
    		description: "Niestandardowe integracje (custom_components)",
    		title: "Integracje"
    	},
    	pending_repository_upgrade: "Używasz wersji {installed}, wersja {available} jest dostępna"
    };
    var settings$b = {
    	add_custom_repository: "DODAJ REPOZYTORIUM NIESTANDARDOWE",
    	adding_new_repo: "Dodawanie nowego repozytorium '{repo}'",
    	adding_new_repo_category: "Z kategorią '{category}'.",
    	bg_task_custom: "Niestandardowe repozytoria są ukryte podczas wykonywania zadań w tle.",
    	category: "Kategoria",
    	compact_mode: "Tryb kompaktowy",
    	custom_repositories: "REPOZYTORIA NIESTANDARDOWE",
    	"delete": "Usuń",
    	display: "Sposób wyświetlania",
    	grid: "Siatka",
    	hacs_repo: "Repozytorium HACS",
    	hidden_repositories: "ukryte repozytoria",
    	missing_category: "Musisz wybrać kategorię",
    	open_repository: "Otwórz repozytorium",
    	reload_data: "Wczytaj ponownie dane",
    	reload_window: "Załaduj ponownie okno",
    	repository_configuration: "Konfiguracja repozytorium",
    	save: "Zapisz",
    	table: "tabela",
    	table_view: "Widok tabeli",
    	unhide: "pokaż",
    	upgrade_all: "Uaktualnij wszystkie"
    };
    var store$b = {
    	ascending: "rosnąco",
    	clear_new: "Wyczyść wszystkie nowe repozytoria",
    	descending: "malejąco",
    	last_updated: "Ostatnia aktualizacja",
    	name: "Nazwa",
    	new_repositories: "Nowe repozytoria",
    	new_repositories_note: "Jest ponad 10 nowych repozytoriów, jeśli chcesz je wyczyścić, kliknij menu z trzema kropkami w prawym górnym rogu i odrzuć je wszystkie.",
    	no_repositories: "Brak repozytoriów",
    	no_repositories_desc1: "Wygląda na to, że nie masz jeszcze zainstalowanych repozytoriów w tej sekcji.",
    	no_repositories_desc2: "Kliknij + w dolnym rogu, aby dodać pierwsze!",
    	no_repositories_found_desc1: "W tej sekcji nie znaleziono zainstalowanych repozytoriów pasujących do \"{searchInput}\".",
    	no_repositories_found_desc2: "Spróbuj wyszukać czegoś innego!",
    	pending_upgrades: "Oczekujące aktualizacje",
    	placeholder_search: "Wprowadź wyszukiwane hasło...",
    	sort: "sortowanie",
    	stars: "Gwiazdki",
    	status: "Status"
    };
    var time$b = {
    	ago: "temu",
    	day: "dzień",
    	days: "dni",
    	hour: "godzina",
    	hours: "godziny",
    	minute: "minuta",
    	minutes: "minuty",
    	month: "miesiąc",
    	months: "miesięcy",
    	one: "Jeden",
    	one_day_ago: "jeden dzień temu",
    	one_hour_ago: "jedna godzina temu",
    	one_minute_ago: "jedna minuta temu",
    	one_month_ago: "jeden miesiąc temu",
    	one_second_ago: "jedna sekunda temu",
    	one_year_ago: "jeden rok temu",
    	second: "sekunda",
    	seconds: "sekundy",
    	x_days_ago: "{x} dni temu",
    	x_hours_ago: "{x} godzin(y) temu",
    	x_minutes_ago: "{x} minut(y) temu",
    	x_months_ago: "{x} miesi(ące/ęcy) temu",
    	x_seconds_ago: "{x} sekund(y) temu",
    	x_years_ago: "{x} lat(a) temu",
    	year: "rok",
    	years: "lata"
    };
    var pl = {
    	common: common$b,
    	config: config$b,
    	confirm: confirm$b,
    	dialog_about: dialog_about$9,
    	dialog_add_repo: dialog_add_repo$9,
    	dialog_custom_repositories: dialog_custom_repositories$9,
    	dialog_info: dialog_info$9,
    	dialog_install: dialog_install$9,
    	dialog_update: dialog_update$9,
    	entry: entry$9,
    	menu: menu$9,
    	options: options$b,
    	repository_banner: repository_banner$b,
    	repository_card: repository_card$9,
    	repository: repository$b,
    	search: search$9,
    	sections: sections$a,
    	settings: settings$b,
    	store: store$b,
    	time: time$b
    };

    var pl$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$b,
        config: config$b,
        confirm: confirm$b,
        dialog_about: dialog_about$9,
        dialog_add_repo: dialog_add_repo$9,
        dialog_custom_repositories: dialog_custom_repositories$9,
        dialog_info: dialog_info$9,
        dialog_install: dialog_install$9,
        dialog_update: dialog_update$9,
        entry: entry$9,
        menu: menu$9,
        options: options$b,
        repository_banner: repository_banner$b,
        repository_card: repository_card$9,
        repository: repository$b,
        search: search$9,
        sections: sections$a,
        settings: settings$b,
        store: store$b,
        time: time$b,
        'default': pl
    });

    var common$c = {
    	about: "Sobre",
    	add: "adicionar",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "AppDaemon Apps",
    	appdaemon_plural: "Aplicativos AppDaemon",
    	background_task: "Tarefa em segundo plano em execução, esta página será recarregada quando terminar.",
    	check_log_file: "Verifique seu arquivo de log para obter mais detalhes.",
    	"continue": "Continuar",
    	disabled: "Desativado",
    	documentation: "Documentação",
    	element: "elemento",
    	hacs_is_disabled: "HACS está desativado",
    	install: "Instalar",
    	installed: "instalado",
    	integration: "Integração",
    	integration_plural: "Integrações",
    	integrations: "Integrações",
    	lovelace: "Lovelace",
    	lovelace_element: "Elemento do Lovelace",
    	lovelace_elements: "Elementos do lovelace",
    	manage: "gerenciar",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDaemon Apps",
    	netdaemon_plural: "Aplicativos NetDaemon",
    	plugin: "Lovelace",
    	plugin_plural: "Elementos do Lovelace",
    	plugins: "Elementos do Lovelace",
    	python_script: "Python Script",
    	python_script_plural: "Scripts python",
    	python_scripts: "Python Scripts",
    	repositories: "Repositórios",
    	repository: "Repositório",
    	settings: "configurações",
    	theme: "Tema",
    	theme_plural: "Temas",
    	themes: "Temas",
    	uninstall: "Desinstalar",
    	update: "Atualizar",
    	version: "Versão"
    };
    var config$c = {
    	abort: {
    		single_instance_allowed: "Apenas uma configuração do HACS é permitida."
    	},
    	error: {
    		auth: "Token de acesso pessoal incorreto."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Habilitar AppDaemon apps descoberta & rastreamento",
    				netdaemon: "Habilitar NetDaemon apps descoberta & rastreamento",
    				python_script: "Habilitar python_scripts descoberta & rastreamento",
    				sidepanel_icon: "Icone do painel lateral",
    				sidepanel_title: "Titulo do painel lateral",
    				theme: "Habilitar temas descoberta & rastreamento",
    				token: "GitHub Token de acesso pessoal"
    			},
    			description: "Se você preciar de ajuda com a configuração olhe aqui: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$c = {
    	add_to_lovelace: "Tem certeza de que deseja adicionar isso aos seus recursos do Lovelace?",
    	bg_task: "A ação é desativada enquanto as tarefas de fundo estão sendo executadas.",
    	cancel: "Cancelar",
    	"continue": "Tem certeza que quer continuar?",
    	"delete": "Tem certeza de que deseja excluir '{item}'?",
    	delete_installed: "'{item}' está instalado, é necessário desinstalá-lo para poder excluí-lo.",
    	exist: "{item} já existe",
    	generic: "Tem certeza?",
    	home_assistant_is_restarting: "Espere, o Home Assistant está agora a reiniciar.",
    	home_assistant_version_not_correct: "Você está executando a versão Home Assistant '{haversion}', mas este repositório requer que a versão mínima '{minversion}' esteja instalada.",
    	no: "Não",
    	no_upgrades: "Não há atualizações pendentes",
    	ok: "OK",
    	overwrite: "Fazer isso irá substituí-lo.",
    	reload_data: "Isso recarrega os dados de todos os repositórios que o HACS conhece e levará algum tempo para concluir.",
    	restart_home_assistant: "Tem certeza de que deseja reiniciar o Home Assistant?",
    	uninstall: "Tem certeza de que deseja desinstalar '{item}'?",
    	upgrade_all: "Isso atualizará todos esses repositórios, verifique se você leu as notas de versão de todos eles antes de continuar.",
    	yes: "Sim"
    };
    var dialog_about$a = {
    	frontend_version: "Versão do frontend",
    	installed_repositories: "Repositórios instalados",
    	integration_version: "Versão da integração",
    	useful_links: "Links úteis"
    };
    var dialog_add_repo$a = {
    	limit: "Apenas os 100 primeiros repositórios são mostrados, use a pesquisa para filtrar o que você precisa",
    	no_match: "Nenhum repositório encontrado correspondente ao seu filtro",
    	sort_by: "Ordenar por",
    	title: "Novo repositório"
    };
    var dialog_custom_repositories$a = {
    	category: "Categoria",
    	no_category: "Categoria ausente",
    	no_repository: "Repositório ausente",
    	title: "Repositórios personalizados",
    	url_placeholder: "Adicionar URL de repositório personalizado"
    };
    var dialog_info$a = {
    	author: "Autor",
    	downloads: "Downloads",
    	install: "Instalar esse repositório no HACS",
    	loading: "Carregando informações...",
    	no_info: "O desenvolvedor não forneceu mais informações para este repositório",
    	open_issues: "Problemas em aberto",
    	open_repo: "Abrir repositório",
    	stars: "Estrelas",
    	version_installed: "Versão instalada"
    };
    var dialog_install$a = {
    	restart: "Lembre-se de que você precisa reiniciar o Home Assistant para que as alterações nas integrações (custom_components) sejam aplicadas.",
    	select_version: "Selecionar versão",
    	show_beta: "Mostrar versões beta",
    	type: "Tipo",
    	url: "URL"
    };
    var dialog_update$a = {
    	available_version: "Versão disponível",
    	changelog: "Registro de mudanças",
    	installed_version: "Versão instalada",
    	releasenotes: "Notas de lançamento para {release}",
    	title: "Atualização pendente"
    };
    var entry$a = {
    	information: "Informações",
    	intro: "Atualizações e mensagens importantes serão mostradas aqui se houver",
    	messages: {
    		disabled: {
    			content: "Verifique seu arquivo de log para mais detalhes.",
    			title: "O HACS está desativado"
    		},
    		has_pending_tasks: {
    			content: "Alguns repositórios podem não aparecer até que isso seja concluído",
    			title: "Tarefas em segundo plano pendentes"
    		},
    		resources: {
    			content: "Tem {número} elementos do Lovelace que não são carregados corretamente no Lovelace.",
    			title: "Não carregado em Lovelace"
    		},
    		restart: {
    			content: "Tem {número} integrações que requerem um reinício do Home Assistant. Você pode fazer isso na seção 'Controles do Servidor' na parte de configuração do Home Assistant UI.",
    			title: "Reinicialização pendente"
    		},
    		startup: {
    			content: "O HACS está sendo iniciado, durante esse período algumas informações podem estar ausentes ou incorretas",
    			title: "O HACS está iniciando"
    		},
    		wrong_frontend_installed: {
    			content: "Você instalou a versão {running} do frontend HACS, mas a versão {expected} era esperada. Se você vir esta mensagem, o Home Assistant não conseguiu instalar a nova versão. Por favor tente reiniciar o Home Assistant.",
    			title: "Versão frontend inesperada"
    		},
    		wrong_frontend_loaded: {
    			content: "Você está executando a versão {running} do frontend HACS, mas a versão {expected} era esperada. Por favor limpe o cache do seu navegador.",
    			title: "Versão frontend inesperada"
    		}
    	},
    	pending_updates: "Atualizações pendentes"
    };
    var menu$a = {
    	about: "Sobre o HACS",
    	clear: "Limpar todos os novos",
    	custom_repositories: "Repositórios personalizados",
    	dismiss: "Limpar todos os novos repositórios",
    	documentation: "Documentação",
    	open_issue: "Relatar problema",
    	reload: "Recarregar janela"
    };
    var options$c = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Habilitar AppDaemon apps descoberta & rastreamento",
    				country: "Filtro pelo código do país.",
    				debug: "Ative a depuração.",
    				experimental: "Ativar recursos experimentais",
    				netdaemon: "Habilitar NetDaemon apps descoberta & rastreamento",
    				not_in_use: "Não está em uso com o YAML",
    				release_limit: "Número de lançamentos a serem exibidos.",
    				sidepanel_icon: "Icone do painel lateral",
    				sidepanel_title: "Titulo do painel lateral"
    			}
    		}
    	}
    };
    var repository_banner$c = {
    	config_flow: "Essa integração oferece suporte ao config_flow, o que significa que agora você pode acessar a seção de integração da sua interface do usuário para configurá-lo.",
    	config_flow_title: "Configuração de interface do usuário suportada",
    	integration_not_loaded: "Esta integração não é carregada no Home Assistant.",
    	no_restart_required: "Não é necessário reiniciar",
    	not_loaded: "Não carregado",
    	plugin_not_loaded: "Este elemento não é adicionado aos seus recursos do Lovelace.",
    	restart: "Você precisa reiniciar o Home Assistant.",
    	restart_pending: "Reiniciar pendente"
    };
    var repository_card$a = {
    	dismiss: "Dispensar",
    	hide: "Esconder",
    	information: "Informações",
    	new_repository: "Novo repositório",
    	not_loaded: "Não carregado",
    	open_issue: "Relatar problema",
    	open_source: "Código aberto",
    	pending_restart: "Reinicialização pendente",
    	pending_update: "Atualização pendente",
    	reinstall: "Reinstalar",
    	report: "Denunciar para remoção",
    	update_information: "Atualizar informações"
    };
    var repository$c = {
    	add_to_lovelace: "Adicionar a Lovelace",
    	authors: "Autores",
    	available: "Disponível",
    	back_to: "Voltar para",
    	changelog: "Changelog",
    	downloads: "Downloads",
    	flag_this: "Sinalizar isso",
    	frontend_version: "Versão Frontend",
    	github_stars: "Estrelas de GitHub",
    	goto_integrations: "Ir para integrações",
    	hide: "Esconder",
    	hide_beta: "Esconder beta",
    	install: "Instalar",
    	installed: "Instalado",
    	lovelace_copy_example: "Copie este exemplo para seu clipboard",
    	lovelace_instruction: "Quando você adicionar isso à sua configuração do lovelace, use este",
    	lovelace_no_js_type: "Não foi possível determinar o tipo desse elemento, verifique o repositório.",
    	newest: "O mais novo",
    	note_appdaemon: "Você ainda precisa adicioná-lo ao seu arquivo 'apps.yaml'",
    	note_installed: "Quando instalado, ele estará localizado em",
    	note_integration: "Você ainda precisa adicioná-lo ao seu arquivo 'configuration.yaml'",
    	note_plugin: "você ainda precisará adicioná-lo à sua configuração do lovelace ('ui-lovelace.yaml' ou o editor de configuração da interface do usuário)",
    	note_plugin_post_107: "você ainda precisará adicioná-lo à sua configuração do lovelace ('configuration.yaml' ou o editor de recursos '/config /lovelace/resources')",
    	open_issue: "Open issue",
    	open_plugin: "Elemento aberto",
    	reinstall: "Reinstalar",
    	repository: "Repositório",
    	restart_home_assistant: "Reiniciar Home Assistant",
    	show_beta: "Mostrar beta",
    	uninstall: "Desinstalar",
    	update_information: "Atualizar informação",
    	upgrade: "Melhorar"
    };
    var search$a = {
    	placeholder: "Procurar repositório"
    };
    var sections$b = {
    	about: {
    		description: "Exibir informações sobre o HACS",
    		title: "Sobre"
    	},
    	automation: {
    		description: "É aqui que você encontra python_scripts, aplicativos AppDaemon e aplicativos NetDaemon",
    		title: "Automação"
    	},
    	frontend: {
    		description: "É aqui que você encontra temas, cartões personalizados e outros elementos para o lovelace",
    		title: "Frontend"
    	},
    	integrations: {
    		description: "É aqui que você encontra integrações personalizadas (custom_components)",
    		title: "Integrações"
    	},
    	pending_repository_upgrade: "Você está executando a versão {installed}, a versão {available} está disponível"
    };
    var settings$c = {
    	add_custom_repository: "ADICIONAR REPOSITÓRIO PERSONALIZADO",
    	adding_new_repo: "Adicionando novo repositório '{repo}'",
    	adding_new_repo_category: "Com a categoria '{category}'.",
    	bg_task_custom: "Os repositórios personalizados ficam ocultos enquanto as tarefas de fundo estão em execução.",
    	category: "Categoria",
    	compact_mode: "Modo compacto",
    	custom_repositories: "REPOSITÓRIOS PERSONALIZADOS",
    	"delete": "Deletar",
    	display: "Display",
    	grid: "Grade",
    	hacs_repo: "HACS repo",
    	hidden_repositories: "repositórios ocultos",
    	missing_category: "Você precisa selecionar uma categoria",
    	open_repository: "Repositório aberto",
    	reload_data: "Recarregar dados",
    	reload_window: "Recarregar janela",
    	repository_configuration: "Configuração do Repositório",
    	save: "Salvar",
    	table: "Tabela",
    	table_view: "Vista de mesa",
    	unhide: "reexibir",
    	upgrade_all: "Atualizar tudo"
    };
    var store$c = {
    	ascending: "ascendente",
    	clear_new: "Limpar todos os novos repositórios",
    	descending: "descendente",
    	last_updated: "Última atualização",
    	name: "Nome",
    	new_repositories: "Novos Repositórios",
    	new_repositories_note: "Você tem mais de 10 novos repositórios sendo mostrados aqui, se quiser limpar todos eles clique nos 3 pontos no canto superior direito e dispense-os.",
    	no_repositories: "Nenhum repositório",
    	no_repositories_desc1: "Parece que você ainda não tem nenhum repositório instalado nesta seção.",
    	no_repositories_desc2: "Clique no + no canto inferior para adicionar o seu primeiro repositório!",
    	no_repositories_found_desc1: "Nenhum repositório instalado foi encontrado que corresponda a \"{searchInput}\" nesta seção.",
    	no_repositories_found_desc2: "Tente procurar por outra coisa!",
    	pending_upgrades: "Atualizações pendentes",
    	placeholder_search: "Por favor insira um termo de pesquisa...",
    	sort: "ordenar",
    	stars: "Estrelas",
    	status: "Status"
    };
    var time$c = {
    	ago: "atrás",
    	day: "dia",
    	days: "dias",
    	hour: "hora",
    	hours: "horas",
    	minute: "minuto",
    	minutes: "minutos",
    	month: "mês",
    	months: "meses",
    	one: "Um",
    	one_day_ago: "um dia atrás",
    	one_hour_ago: "uma hora atrás",
    	one_minute_ago: "um minuto atrás",
    	one_month_ago: "um mês atrás",
    	one_second_ago: "um segundo atrás",
    	one_year_ago: "um ano atrás",
    	second: "segundo",
    	seconds: "segundos",
    	x_days_ago: "{x} dias atrás",
    	x_hours_ago: "{x} horas atrás",
    	x_minutes_ago: "{x} minutos atrás",
    	x_months_ago: "{x} meses atrás",
    	x_seconds_ago: "{x} segundos atrás",
    	x_years_ago: "{x} anos atrás",
    	year: "ano",
    	years: "anos"
    };
    var ptBR = {
    	common: common$c,
    	config: config$c,
    	confirm: confirm$c,
    	dialog_about: dialog_about$a,
    	dialog_add_repo: dialog_add_repo$a,
    	dialog_custom_repositories: dialog_custom_repositories$a,
    	dialog_info: dialog_info$a,
    	dialog_install: dialog_install$a,
    	dialog_update: dialog_update$a,
    	entry: entry$a,
    	menu: menu$a,
    	options: options$c,
    	repository_banner: repository_banner$c,
    	repository_card: repository_card$a,
    	repository: repository$c,
    	search: search$a,
    	sections: sections$b,
    	settings: settings$c,
    	store: store$c,
    	time: time$c
    };

    var pt_BR = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$c,
        config: config$c,
        confirm: confirm$c,
        dialog_about: dialog_about$a,
        dialog_add_repo: dialog_add_repo$a,
        dialog_custom_repositories: dialog_custom_repositories$a,
        dialog_info: dialog_info$a,
        dialog_install: dialog_install$a,
        dialog_update: dialog_update$a,
        entry: entry$a,
        menu: menu$a,
        options: options$c,
        repository_banner: repository_banner$c,
        repository_card: repository_card$a,
        repository: repository$c,
        search: search$a,
        sections: sections$b,
        settings: settings$c,
        store: store$c,
        time: time$c,
        'default': ptBR
    });

    var common$d = {
    	about: "Despre",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "Aplicații AppDaemon",
    	background_task: "Activitatea de fundal se execută, această pagină se va reîncărca atunci când este gata.",
    	check_log_file: "Verificați log-ul pentru mai multe detalii.",
    	"continue": "Continua",
    	disabled: "Dezactivat",
    	documentation: "Documentație",
    	hacs_is_disabled: "HACS este dezactivat",
    	installed: "instalat",
    	integration: "Integrare",
    	integrations: "Integrări",
    	manage: "administra",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "Aplicații NetDaemon",
    	plugin: "Plugin",
    	plugins: "Plugin-uri",
    	python_script: "Script Python",
    	python_scripts: "Scripturi Python",
    	repositories: "Depozite",
    	settings: "setări",
    	theme: "Temă",
    	themes: "Teme",
    	version: "Versiune"
    };
    var config$d = {
    	abort: {
    		single_instance_allowed: "Doar o singură configurație pentru HACS este permisă."
    	},
    	error: {
    		auth: "Token-ul de acces personal nu este corect."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Activați descoperirea și urmărirea aplicațiilor AppDaemon",
    				netdaemon: "Activați descoperirea și urmărirea aplicațiilor NetDaemon",
    				python_script: "Activați descoperirea și urmărirea python_scripts",
    				sidepanel_icon: "Pictogramă Panou lateral",
    				sidepanel_title: "Titlu panou lateral",
    				theme: "Activați descoperirea și urmărirea temelor",
    				token: "Token de acces personal GitHub"
    			},
    			description: "Dacă aveți nevoie de ajutor pentru configurare, uitați-vă aici: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$d = {
    	add_to_lovelace: "Ești sigur că vrei să adaugi asta la resursele tale Lovelace?",
    	bg_task: "Acțiunea este dezactivată în timp ce activitățile de fundal se execută.",
    	cancel: "Anulare",
    	"continue": "Esti sigur ca vrei sa continui?",
    	"delete": "Sigur doriți să ștergeți '{item}'?",
    	exist: "{item} există deja",
    	generic: "Ești sigur?",
    	home_assistant_is_restarting: "Asteptati, Home Assistant repornește.",
    	no: "Nu",
    	no_upgrades: "Nu există actualizări în curs",
    	ok: "OK",
    	overwrite: "Făcând acest lucru, îl va suprascrie.",
    	restart_home_assistant: "Sigur doriți să reporniți Home Assistant?",
    	uninstall: "Sigur doriți să dezinstalați '{item}'?",
    	yes: "Da"
    };
    var options$d = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Activați descoperirea și urmărirea aplicațiilor AppDaemon",
    				country: "Filtrează cu codul țării.",
    				debug: "Activează depanarea.",
    				experimental: "Activați funcțiile experimentale",
    				netdaemon: "Activați descoperirea și urmărirea aplicațiilor NetDaemon",
    				not_in_use: "Nu este utilizat cu YAML",
    				release_limit: "Număr de versiuni afișate.",
    				sidepanel_icon: "Pictogramă Panou lateral",
    				sidepanel_title: "Titlu panou lateral"
    			}
    		}
    	}
    };
    var repository_banner$d = {
    	integration_not_loaded: "Această integrare nu este încărcată în Home Assistant.",
    	no_restart_required: "Nu este necesară repornirea",
    	not_loaded: "Neîncărcat",
    	plugin_not_loaded: "Acest plugin nu este adăugat la resursele Lovelace.",
    	restart: "Trebuie să reporniți Home Assistant.",
    	restart_pending: "Reporniți în așteptare"
    };
    var repository$d = {
    	add_to_lovelace: "Adăugați la Lovelace",
    	authors: "Autori",
    	available: "Disponibil",
    	back_to: "Înapoi la",
    	changelog: "Jurnal modificari",
    	downloads: "Descărcări",
    	flag_this: "Semnalizează",
    	frontend_version: "Versiune frontend",
    	github_stars: "Stele GitHub",
    	goto_integrations: "Mergi la integrări",
    	hide: "Ascunde",
    	hide_beta: "Ascundere beta",
    	install: "Instalează",
    	installed: "Instalat",
    	lovelace_copy_example: "Copiați exemplul în clipboard",
    	lovelace_instruction: "Când adăugați acest lucru la configurația lovelace, folosiți acest",
    	lovelace_no_js_type: "Nu s-a putut determina tipul acestui plugin, verificați \"repository\".",
    	newest: "cele mai noi",
    	note_appdaemon: "mai trebuie să o adăugați in fișierul „apps.yaml”",
    	note_installed: "Când este instalat, acesta va fi localizat în",
    	note_integration: "mai trebuie să o adăugați in fișierul „configuration.yaml”",
    	note_plugin: "mai trebuie să o adăugați la configurația lovelace ('ui-lovelace.yaml' sau in editorul UI)",
    	open_issue: "Deschidere problemă",
    	open_plugin: "Deschide plugin",
    	reinstall: "Reinstalare",
    	repository: "Depozit",
    	restart_home_assistant: "Reporniți Home Assistant",
    	show_beta: "Afișare beta",
    	uninstall: "Dezinstalare",
    	update_information: "Actualizare informație",
    	upgrade: "Actualizare"
    };
    var settings$d = {
    	add_custom_repository: "ADĂUGARE DEPOZIT PERSONALIZAT",
    	adding_new_repo: "Adăugarea unui depozit nou '{repo}'",
    	adding_new_repo_category: "Cu categoria '{categoria}'.",
    	category: "Categorie",
    	compact_mode: "Modul compact",
    	custom_repositories: "DEPOZITE PERSONALIZATE",
    	"delete": "Șterge",
    	display: "Afişeaza",
    	grid: "Grilă",
    	hacs_repo: "HACS repo",
    	hidden_repositories: "depozite ascunse",
    	missing_category: "Trebuie să selectați o categorie",
    	open_repository: "Deschideți depozitul",
    	reload_data: "Reîncărcați datele",
    	reload_window: "Reîncărcați fereastra",
    	repository_configuration: "Configurația depozitului",
    	save: "Salveaza",
    	table: "Tabel",
    	table_view: "Vizualizare tabel",
    	unhide: "Unhide",
    	upgrade_all: "Actualizați toate"
    };
    var store$d = {
    	ascending: "ascendent",
    	clear_new: "Ștergeți toate depozitele noi",
    	descending: "descendent",
    	last_updated: "Ultima actualizare",
    	name: "Nume",
    	new_repositories: "Noi depozite",
    	pending_upgrades: "Actualizări în așteptare",
    	placeholder_search: "Vă rugăm să introduceți un termen de căutare ...",
    	sort: "fel",
    	stars: "Stele",
    	status: "Starea"
    };
    var time$d = {
    	ago: "în urmă",
    	day: "zi",
    	days: "zi",
    	hour: "oră",
    	hours: "ore",
    	minute: "minut",
    	minutes: "minute",
    	month: "lună",
    	months: "luni",
    	one: "Unu",
    	one_day_ago: "acum o zi",
    	one_hour_ago: "acum o oră",
    	one_minute_ago: "acum un minut",
    	one_month_ago: "acum o luna",
    	one_second_ago: "acum o secunda",
    	one_year_ago: "acum un an",
    	second: "al doilea",
    	seconds: "secunde",
    	x_days_ago: "{x} zile în urmă",
    	x_hours_ago: "{x} ore în urmă",
    	x_minutes_ago: "{x} minute în urmă",
    	x_months_ago: "{x} luni în urmă",
    	x_seconds_ago: "{x} secunde în urmă",
    	x_years_ago: "{x} ani în urmă",
    	year: "an",
    	years: "ani"
    };
    var ro = {
    	common: common$d,
    	config: config$d,
    	confirm: confirm$d,
    	options: options$d,
    	repository_banner: repository_banner$d,
    	repository: repository$d,
    	settings: settings$d,
    	store: store$d,
    	time: time$d
    };

    var ro$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$d,
        config: config$d,
        confirm: confirm$d,
        options: options$d,
        repository_banner: repository_banner$d,
        repository: repository$d,
        settings: settings$d,
        store: store$d,
        time: time$d,
        'default': ro
    });

    var common$e = {
    	about: "О проекте",
    	add: "добавить",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "Приложения AppDaemon",
    	appdaemon_plural: "Приложения AppDaemon",
    	background_task: "Выполняется фоновая задача, страница перезагрузится по готовности.",
    	check_log_file: "Проверьте логи для получения более подробной информации.",
    	"continue": "Продолжить",
    	disabled: "Отключено",
    	documentation: "Документация",
    	element: "элемент",
    	hacs_is_disabled: "HACS отключен",
    	install: "Установить",
    	installed: "установлено",
    	integration: "Интеграция",
    	integration_plural: "Интеграции",
    	integrations: "Интеграции",
    	lovelace: "Lovelace",
    	lovelace_element: "Элемент Lovelace",
    	lovelace_elements: "Элементы Lovelace",
    	manage: "управлять",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "Приложения NetDaemon",
    	netdaemon_plural: "Приложения NetDaemon",
    	plugin: "Lovelace",
    	plugin_plural: "Элементы Lovelace",
    	plugins: "Плагины",
    	python_script: "Скрипт Python",
    	python_script_plural: "Скрипты Python",
    	python_scripts: "Скрипты Python",
    	repositories: "Репозитории",
    	repository: "Репозиторий",
    	settings: "настройки",
    	theme: "Тема",
    	theme_plural: "Темы",
    	themes: "Темы",
    	uninstall: "Удалить",
    	update: "Обновить",
    	version: "Версия"
    };
    var config$e = {
    	abort: {
    		single_instance_allowed: "Разрешена только одна настройка HACS."
    	},
    	error: {
    		auth: "Неверный токен персонального доступа."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Включить поиск и отслеживание приложений AppDaemon",
    				netdaemon: "Включить поиск и отслеживание приложений NetDaemon",
    				python_script: "Включить поиск и отслеживание скриптов Python",
    				sidepanel_icon: "Иконка в боковом меню",
    				sidepanel_title: "Название в боковом меню",
    				theme: "Включить поиск и отслеживание тем",
    				token: "Персональный токен доступа GitHub"
    			},
    			description: "Если вам нужна помощь с настройкой, посмотрите: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$e = {
    	add_to_lovelace: "Вы уверены, что хотите добавить это в ресурсы Lovelace?",
    	bg_task: "Действие отключено во время выполнения фоновых задач.",
    	cancel: "Отмена",
    	"continue": "Вы уверены, что хотите продолжить?",
    	"delete": "Вы уверены, что хотите удалить '{item}'?",
    	delete_installed: "'{item}' установлен, вам нужно нажать 'Удалить', чтобы удалить его.",
    	exist: "{item} уже существует.",
    	generic: "Вы уверены?",
    	home_assistant_is_restarting: "Подожди, теперь Home Assistant перезагружается.",
    	no: "Нет",
    	no_upgrades: "Нет обновлений",
    	ok: "ОК",
    	overwrite: "После подтверждения файлы будут перезаписаны.",
    	reload_data: "Выполняется перезагрузка данных всех репозиториев в HACS, это займет некоторое время.",
    	restart_home_assistant: "Вы уверены, что хотите перезапустить Home Assistant?",
    	uninstall: "Вы уверены, что хотите удалить '{item}'?",
    	upgrade_all: "Это произведёт обновление всех этих репозиториев, убедитесь, что вы прочитали примечания к выпуску для каждого из них, прежде чем продолжить.",
    	yes: "Да"
    };
    var dialog_about$b = {
    	frontend_version: "Версия интерфейса",
    	installed_repositories: "Установленные репозитории",
    	integration_version: "Версия интеграции",
    	useful_links: "Полезные ссылки"
    };
    var dialog_add_repo$b = {
    	limit: "Показаны только первые 100 репозиториев, используйте поиск для фильтрации того, что вам нужно",
    	no_match: "Не найдено репозиторий, соответствующих фильтру",
    	sort_by: "Сортировать по",
    	title: "Новый репозиторий"
    };
    var dialog_custom_repositories$b = {
    	category: "Категория",
    	no_category: "Категория не указана",
    	no_repository: "Репозиторий не указан",
    	title: "Пользовательские репозитории",
    	url_placeholder: "Добавить пользовательский URL-адрес репозитория"
    };
    var dialog_info$b = {
    	author: "Автор",
    	downloads: "Загрузки",
    	install: "Установить этот репозиторий в HACS",
    	loading: "Загрузка информации...",
    	no_info: "Разработчик не предоставил никакой дополнительной информации для этого репозитория",
    	open_issues: "Открытые вопросы",
    	open_repo: "Открыть репозиторий",
    	stars: "Звёзды",
    	version_installed: "Установлена версия"
    };
    var dialog_install$b = {
    	restart: "Помните, что вам нужно перезапустить Home Assistant, прежде чем будут применены изменения в интеграциях (custom_components).",
    	select_version: "Выберите версию",
    	show_beta: "Показывать бета-версии",
    	type: "Тип",
    	url: "Ссылка"
    };
    var dialog_update$b = {
    	available_version: "Доступная версия",
    	changelog: "Изменения",
    	installed_version: "Установленная версия",
    	releasenotes: "Примечания к выпуску для {release}",
    	title: "Обновление в ожидании"
    };
    var entry$b = {
    	information: "Информация",
    	intro: "Обновления и важные сообщения будут отображаться здесь, если таковые имеются",
    	messages: {
    		disabled: {
    			content: "Проверьте логи для получения более подробной информации.",
    			title: "HACS отключен"
    		},
    		has_pending_tasks: {
    			content: "Некоторые репозитории могут не отображаться до тех пор, пока это не будет завершено",
    			title: "Выполняются фоновые задачи"
    		},
    		resources: {
    			content: "У вас есть {number} элементов Lovelace, которые не загружаются должным образом.",
    			title: "Не загружено в Lovelace"
    		},
    		restart: {
    			content: "У вас есть {number} интеграций, которые требуют перезагрузки Home Assistant, Вы можете сделать это из раздела 'Сервер' в разделе конфигурации пользовательского интерфейса Home Assistant.",
    			title: "В ожидании перезапуска"
    		},
    		startup: {
    			content: "HACS запускается, в течение этого времени некоторые сведения могут отсутствовать или быть неверными",
    			title: "HACS запускается"
    		},
    		wrong_frontend_installed: {
    			content: "Вы используете версию {running} интерфейса HACS, однако ожидаемая версия — {expected}; если Вы видите это сообщение, то Home Assistant не смог установить новую версию интерфейса; попробуйте перезапустить Home Assistant",
    			title: "Неожиданная версия интерфейса"
    		},
    		wrong_frontend_loaded: {
    			content: "Вы используете версию {running} интерфейса HACS, однако ожидаемая версия — {expected}, попробуйте очистить кеш браузера.",
    			title: "Неожиданная версия интерфейса"
    		}
    	},
    	pending_updates: "Обновления в ожидании"
    };
    var menu$b = {
    	about: "О HACS",
    	clear: "Очистить все новые репозитории",
    	custom_repositories: "Пользовательские репозитории",
    	dismiss: "Убрать все новые репозитории",
    	documentation: "Документация",
    	open_issue: "Сообщить о проблеме",
    	reload: "Перезагрузить окно"
    };
    var options$e = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Включить поиск и отслеживание приложений AppDaemon",
    				country: "Фильтр по стране.",
    				debug: "Включить отладку.",
    				experimental: "Вкл. экспериментальные функции",
    				netdaemon: "Включить поиск и отслеживание приложений NetDaemon",
    				not_in_use: "Не используется с YAML",
    				release_limit: "Число доступных версий.",
    				sidepanel_icon: "Иконка в боковом меню",
    				sidepanel_title: "Название в боковом меню"
    			}
    		}
    	}
    };
    var repository_banner$e = {
    	integration_not_loaded: "Эта интеграция не загружена в Home Assistant.",
    	no_restart_required: "Перезагрузка не требуется",
    	not_loaded: "Не загружено",
    	plugin_not_loaded: "Этот плагин не добавлен к ресурсам Lovelace.",
    	restart: "Вам нужно перезапустить Home Assistant.",
    	restart_pending: "Ожидается перезапуск"
    };
    var repository_card$b = {
    	dismiss: "убрать",
    	hide: "Скрыть",
    	information: "Информация",
    	new_repository: "Новый репозиторий",
    	not_loaded: "Не загружено",
    	open_issue: "Сообщить о проблеме",
    	open_source: "Открыть источник",
    	pending_restart: "В ожидании перезапуска",
    	pending_update: "Ожидается обновление",
    	reinstall: "Переустановить",
    	report: "Сообщить о нарушении",
    	update_information: "Обновить информацию"
    };
    var repository$e = {
    	add_to_lovelace: "Добавить в Lovelace",
    	authors: "Авторы",
    	available: "Доступно",
    	back_to: "Назад к",
    	changelog: "Изменения",
    	downloads: "Загрузки",
    	flag_this: "Пожаловаться",
    	frontend_version: "Версия",
    	github_stars: "Звезды GitHub",
    	goto_integrations: "Перейти к интеграции",
    	hide: "Скрыть",
    	hide_beta: "Скрыть бета",
    	install: "Установить",
    	installed: "Установлено",
    	lovelace_copy_example: "Скопируйте пример в буфер обмена",
    	lovelace_instruction: "Для добавления в конфигурацию Lovelace, используйте",
    	lovelace_no_js_type: "Не удалось определить тип этого плагина, проверьте репозиторий.",
    	newest: "новейшая",
    	note_appdaemon: "вам всё ещё нужно добавить код для настройки приложения в файл 'apps.yaml'",
    	note_installed: "После установки, файлы будут расположены в",
    	note_integration: "вам всё ещё нужно добавить код для настройки интеграции в файл 'configuration.yaml'",
    	note_plugin: "вам всё ещё нужно добавить код для настройки плагина в конфигурацию Lovelace ('ui-lovelace.yaml')",
    	open_issue: "Сообщить о проблеме",
    	open_plugin: "Открыть плагин",
    	reinstall: "Переустановить",
    	repository: "Репозиторий",
    	restart_home_assistant: "Перезагрузка Home Assistant",
    	show_beta: "Показать бета",
    	uninstall: "Удалить",
    	update_information: "Обновить информацию",
    	upgrade: "Обновить"
    };
    var search$b = {
    	placeholder: "Поиск репозитория"
    };
    var sections$c = {
    	about: {
    		description: "Показать информацию о HACS",
    		title: "О проекте"
    	},
    	automation: {
    		description: "Здесь вы найдете python_scripts, приложения AppDaemon и NetDaemon.",
    		title: "Автоматизация"
    	},
    	frontend: {
    		description: "Здесь вы найдете темы, пользовательские карточки и другие элементы для Lovelace",
    		title: "Пользовательский интерфейс"
    	},
    	integrations: {
    		description: "Здесь вы найдете пользовательские интеграции (custom_components)",
    		title: "Интеграции"
    	},
    	pending_repository_upgrade: "Вы используете версию {installed}, доступна версия {available}"
    };
    var settings$e = {
    	add_custom_repository: "ДОБАВИТЬ СВОЙ РЕПОЗИТОРИЙ",
    	adding_new_repo: "Добавление нового репозитория '{repo}'",
    	adding_new_repo_category: "С категорией '{category}'.",
    	bg_task_custom: "Свои репозитории скрыты во время выполнения фоновых задач.",
    	category: "Категория",
    	compact_mode: "Компактный режим",
    	custom_repositories: "СВОИ РЕПОЗИТОРИИ",
    	"delete": "Удалить",
    	display: "Вид",
    	grid: "Сетка",
    	hacs_repo: "Репозиторий HACS",
    	hidden_repositories: "Скрытые репозитории",
    	missing_category: "Вы должны выбрать категорию",
    	open_repository: "Открыть репозиторий",
    	reload_data: "Перезагрузить",
    	reload_window: "Перезагрузить окно",
    	repository_configuration: "Конфигурация репозитория",
    	save: "Сохранить",
    	table: "Таблица",
    	table_view: "Таблица",
    	unhide: "Показать",
    	upgrade_all: "Обновить всё"
    };
    var store$e = {
    	ascending: "по возрастанию",
    	clear_new: "Очистить все новые репозитории",
    	descending: "по убыванию",
    	last_updated: "Последнее обновление",
    	name: "Название",
    	new_repositories: "Новые репозитории",
    	new_repositories_note: "У Вас есть более 10 новых репозиториев, показанных здесь; если Вы хотите очистить их все, нажмите 3 точки в верхнем правом углу и уберите их разом.",
    	no_repositories: "Нет репозиториев",
    	no_repositories_desc1: "Похоже, у вас еще нет репозиториев, установленных в этом разделе.",
    	no_repositories_desc2: "Нажмите на + в нижнем углу, чтобы добавить первый!",
    	no_repositories_found_desc1: "В этом разделе не найдено установленных репозиторий, соответствующих \"{searchinput}\".",
    	no_repositories_found_desc2: "Попробуйте искать что-нибудь другое!",
    	pending_upgrades: "Ожидается обновление",
    	placeholder_search: "Пожалуйста, введите условие для поиска...",
    	sort: "Сортировка",
    	stars: "Звезды",
    	status: "Статус"
    };
    var time$e = {
    	ago: "назад",
    	day: "день",
    	days: "дней",
    	hour: "час",
    	hours: "часов",
    	minute: "минута",
    	minutes: "минут",
    	month: "месяц",
    	months: "месяца",
    	one: "Одна",
    	one_day_ago: "один день назад",
    	one_hour_ago: "час назад",
    	one_minute_ago: "минуту назад",
    	one_month_ago: "месяц назад",
    	one_second_ago: "одну секунду назад",
    	one_year_ago: "один год назад",
    	second: "секунда",
    	seconds: "секунд",
    	x_days_ago: "{x} дней назад",
    	x_hours_ago: "{x} часов назад",
    	x_minutes_ago: "{x} минут назад",
    	x_months_ago: "{x} месяцев назад",
    	x_seconds_ago: "{x} секунд назад",
    	x_years_ago: "{x} лет назад",
    	year: "год",
    	years: "лет"
    };
    var ru = {
    	common: common$e,
    	config: config$e,
    	confirm: confirm$e,
    	dialog_about: dialog_about$b,
    	dialog_add_repo: dialog_add_repo$b,
    	dialog_custom_repositories: dialog_custom_repositories$b,
    	dialog_info: dialog_info$b,
    	dialog_install: dialog_install$b,
    	dialog_update: dialog_update$b,
    	entry: entry$b,
    	menu: menu$b,
    	options: options$e,
    	repository_banner: repository_banner$e,
    	repository_card: repository_card$b,
    	repository: repository$e,
    	search: search$b,
    	sections: sections$c,
    	settings: settings$e,
    	store: store$e,
    	time: time$e
    };

    var ru$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$e,
        config: config$e,
        confirm: confirm$e,
        dialog_about: dialog_about$b,
        dialog_add_repo: dialog_add_repo$b,
        dialog_custom_repositories: dialog_custom_repositories$b,
        dialog_info: dialog_info$b,
        dialog_install: dialog_install$b,
        dialog_update: dialog_update$b,
        entry: entry$b,
        menu: menu$b,
        options: options$e,
        repository_banner: repository_banner$e,
        repository_card: repository_card$b,
        repository: repository$e,
        search: search$b,
        sections: sections$c,
        settings: settings$e,
        store: store$e,
        time: time$e,
        'default': ru
    });

    var common$f = {
    	about: "O tem",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "AppDaemon Aplikacije",
    	background_task: "V ozadju se izvaja naloga, ko se konča, se bo stran osvežila.",
    	check_log_file: "Za več podrobnosti preverite datoteko dnevnika.",
    	"continue": "Nadaljuj",
    	disabled: "Onemogočeno",
    	documentation: "Dokumentacija",
    	hacs_is_disabled: "HACS je onemogočen",
    	installed: "Nameščeno",
    	integration: "Integracija",
    	integrations: "Integracije",
    	manage: "upravljanje",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDaemon Aplikacije",
    	plugin: "Lovelace",
    	plugins: "Vtičniki",
    	python_script: "Python skripta",
    	python_scripts: "Python skripte",
    	repositories: "Repozitoriji",
    	settings: "nastavitve",
    	theme: "Tema",
    	themes: "Teme",
    	version: "Različica"
    };
    var config$f = {
    	abort: {
    		single_instance_allowed: "Dovoljena je samo ena konfiguracija HACS."
    	},
    	error: {
    		auth: "Osebni dostopni žeton ni pravilen."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Omogoči odkrivanje in sledenje aplikacij AppDaemon",
    				netdaemon: "Omogoči odkrivanje in sledenje aplikacij NetDaemon",
    				python_script: "Omogoči odkrivanje in sledenje python_scripts",
    				sidepanel_icon: "Ikona stranske plošče",
    				sidepanel_title: "Naslov stranske plošče",
    				theme: "Omogoči odkrivanje in sledenje tem",
    				token: "GitHub žeton za osebni dostop"
    			},
    			description: "Če potrebujete pomoč pri konfiguraciji, poglejte tukaj: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$f = {
    	add_to_lovelace: "Ali si prepričan, da želiš to dodati med svoje vire Lovelace?",
    	bg_task: "Dejanje je onemogočeno, medtem ko se izvajajo opravila v ozadju.",
    	cancel: "Prekliči",
    	"continue": "Ali ste prepričani, da želite nadaljevati?",
    	"delete": "Ali ste prepričani, da želite izbrisati '{item}'?",
    	delete_installed: "'{item}' je nameščen, pred odstranitvijo ga morate odstraniti.",
    	exist: "{item} že obstaja",
    	generic: "Ali si prepričan?",
    	home_assistant_is_restarting: "Počakaj, Home Assistant se zdaj znova zaganja.",
    	home_assistant_version_not_correct: "Uporabljate Home Assistant verzije '{haversion}', vendar to skladišče zahteva nameščeno najmanj različico '{minversion}'.",
    	no: "Ne",
    	no_upgrades: "Ni nadgradenj",
    	ok: "V redu",
    	overwrite: "Če to storite, bo to prepisano.",
    	reload_data: "Ponovno se naložijo podatki vseh shramb HA, to bo trajalo nekaj časa.",
    	restart_home_assistant: "Ali ste prepričani, da želite znova zagnati Home Assistant?",
    	uninstall: "Ali ste prepričani, da želite odstraniti '{item}'?",
    	upgrade_all: "To bo nadgradil vse te repozitorije, preden nadaljujete poskrbite, da ste prebrali vse opombe ob izdaji.",
    	yes: "Da"
    };
    var options$f = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Omogoči odkrivanje in sledenje aplikacij AppDaemon",
    				country: "Filtrirajte s kodo države.",
    				debug: "Omogoči odpravljanje napak.",
    				experimental: "Omogočite poskusne funkcije",
    				netdaemon: "Omogoči odkrivanje in sledenje aplikacij NetDaemon",
    				not_in_use: "Ni v uporabi z YAML",
    				release_limit: "Število izdaj, ki jih želite prikazati.",
    				sidepanel_icon: "Ikona stranske plošče",
    				sidepanel_title: "Naslov stranske plošče"
    			}
    		}
    	}
    };
    var repository_banner$f = {
    	config_flow: "Ta integracija podpira config_flow, kar pomeni, da lahko zdaj greste na odsek integracije vašega uporabniškega vmesnika, da ga konfigurirate.",
    	config_flow_title: "Konfiguracija uporabniškega vmesnika je podprta",
    	integration_not_loaded: "Ta integracija ni naložena v programu Home Assistant.",
    	no_restart_required: "Ponovni zagon ni potreben",
    	not_loaded: "Ni naloženo",
    	plugin_not_loaded: "Ta vtičnik ni dodan vašim virom Lovelace.",
    	restart: "Znova morate zagnati Home Assistant.",
    	restart_pending: "Vnovični zagon v čakanju"
    };
    var repository$f = {
    	add_to_lovelace: "Dodaj v Lovelace",
    	authors: "Avtorji",
    	available: "Na voljo",
    	back_to: "Nazaj na",
    	changelog: "Dnevnik sprememb",
    	downloads: "Prenosi",
    	flag_this: "Označite to",
    	frontend_version: "Frontend različica",
    	github_stars: "GitHub zvezde",
    	goto_integrations: "Pojdite na integracije",
    	hide: "Skrij",
    	hide_beta: "Skrij različico beta",
    	install: "Namestite",
    	installed: "Nameščeno",
    	lovelace_copy_example: "Kopirajte primer v odložišče",
    	lovelace_instruction: "Ko ga boste dodajali v svojo lovelace konfiguracijo uporabite to:",
    	lovelace_no_js_type: "Ni bilo mogoče določiti vrsto tega vtičnika, preverite repozitorij.",
    	newest: "najnovejše",
    	note_appdaemon: "Še vedno ga morate dodati v svojo 'apps.yaml' datoteko",
    	note_installed: "Ko bo nameščen, se bo nahajal v",
    	note_integration: "Še vedno ga morate dodati v svojo 'configuration.yaml' datoteko",
    	note_plugin: "vendar ga še vedno morate dodati v svojo lovelace konfiguracijo ('ui-lovelace.yaml' ali \"raw\" UI config urejevalnik)",
    	note_plugin_post_107: "še vedno ga morate dodati v svojo konfiguracijo lovelace ('config.yaml' ali urejevalnik virov '/config/lovelace/resources')",
    	open_issue: "Odprite težavo",
    	open_plugin: "Odprite vtičnik",
    	reinstall: "Znova namestite",
    	repository: "Repozitorij",
    	restart_home_assistant: "Znova zaženite Home Assistant",
    	show_beta: "Pokaži različico beta",
    	uninstall: "Odstranite",
    	update_information: "Posodobite podatke",
    	upgrade: "Nadgradnja"
    };
    var settings$f = {
    	add_custom_repository: "DODAJTE SVOJ REPOZITORIJ",
    	adding_new_repo: "Dodajanje novega repozitorija '{repo}'",
    	adding_new_repo_category: "S kategorijo '{category}'.",
    	bg_task_custom: "Med izvajanjem nalog v ozadju so repozitoriji po meri skriti.",
    	category: "Kategorija",
    	compact_mode: "Kompaktni način",
    	custom_repositories: "VAŠI REPOZITORIJI",
    	"delete": "Izbriši",
    	display: "Prikaz",
    	grid: "Mreža",
    	hacs_repo: "HACS repo",
    	hidden_repositories: "skriti repozitoriji",
    	missing_category: "Izbrati moraš kategorijo",
    	open_repository: "Odprite skladišče",
    	reload_data: "Ponovno naloži podatke",
    	reload_window: "Ponovno naloži okno",
    	repository_configuration: "Konfiguracija skladišč",
    	save: "Shranite",
    	table: "Tabela",
    	table_view: "Pogled tabele",
    	unhide: "Razkrij",
    	upgrade_all: "Nadgradite vse"
    };
    var store$f = {
    	ascending: "naraščajoče",
    	clear_new: "Počisti vse nove repozitorije",
    	descending: "padajoče",
    	last_updated: "Nazadnje posodobljeno",
    	name: "Ime",
    	new_repositories: "Novi repozitoriji",
    	pending_upgrades: "Nadgradnje na čakanju",
    	placeholder_search: "Prosim vnesite iskalni izraz ...",
    	sort: "razvrsti",
    	stars: "Zvezd",
    	status: "Stanje"
    };
    var time$f = {
    	ago: "nazaj",
    	day: "dan",
    	days: "Dni",
    	hour: "ura",
    	hours: "Ur",
    	minute: "minuta",
    	minutes: "minut",
    	month: "mesec",
    	months: "mesecev",
    	one: "Eno",
    	one_day_ago: "pred enim dnevom",
    	one_hour_ago: "pred eno uro",
    	one_minute_ago: "pred eno minuto",
    	one_month_ago: "pred enim mesecem",
    	one_second_ago: "pred sekundo",
    	one_year_ago: "pred enim letom",
    	second: "drugič",
    	seconds: "sekund",
    	x_days_ago: "{x} dni nazaj",
    	x_hours_ago: "{x} ur nazaj",
    	x_minutes_ago: "{x} minut nazaj",
    	x_months_ago: "{x} mesecev nazaj",
    	x_seconds_ago: "{x} sekund nazaj",
    	x_years_ago: "{x} let nazaj",
    	year: "leto",
    	years: "Let"
    };
    var sl = {
    	common: common$f,
    	config: config$f,
    	confirm: confirm$f,
    	options: options$f,
    	repository_banner: repository_banner$f,
    	repository: repository$f,
    	settings: settings$f,
    	store: store$f,
    	time: time$f
    };

    var sl$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$f,
        config: config$f,
        confirm: confirm$f,
        options: options$f,
        repository_banner: repository_banner$f,
        repository: repository$f,
        settings: settings$f,
        store: store$f,
        time: time$f,
        'default': sl
    });

    var common$g = {
    	about: "Om",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "Appdaemon Applikationer",
    	background_task: "Bakgrundsjobb körs, denna sida kommer att laddas igen när det är klart.",
    	check_log_file: "Kontrollera din loggfil för mer information.",
    	"continue": "Fortsätta",
    	disabled: "Inaktiverad",
    	documentation: "Dokumentation",
    	hacs_is_disabled: "HACS är inaktiverat",
    	installed: "installerad",
    	integration: "Integration",
    	integrations: "Integrationer",
    	manage: "hantera",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "NetDaemon Applikationer",
    	plugin: "Lovelace",
    	plugins: "Plugins",
    	python_script: "Python skript",
    	python_scripts: "Python skript",
    	repositories: "Repositories",
    	settings: "inställningar",
    	theme: "Tema",
    	themes: "Teman",
    	version: "Version"
    };
    var config$g = {
    	abort: {
    		single_instance_allowed: "Endast en konfiguration kan användas i HACS."
    	},
    	error: {
    		auth: "Personal Access Token är inte korrekt."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Upptäck och följ Appdaemon applikationer",
    				netdaemon: "Upptäck och följ NetDaemon applikationer",
    				python_script: "Upptäck och följ python_scripts",
    				sidepanel_icon: "Ikon för sidpanel",
    				sidepanel_title: "Rubrik för sidpanel",
    				theme: "Upptäck och följ teman",
    				token: "GitHub Personal Access Token"
    			},
    			description: "Om du behöver hjälp med konfigurationen, se här: https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$g = {
    	add_to_lovelace: "Är du säker på att du vill lägga till detta till dina Lovelace-resurser?",
    	bg_task: "Åtgärden är inaktiverad medan aktiviteter i bakgrunden körs",
    	cancel: "Avbryt",
    	"continue": "Är du säker på att du vill fortsätta?",
    	"delete": "Är du säker på att du vill ta bort '{item}'?",
    	delete_installed: "'{item}' är installerat, du måste avinstallera det innan du kan ta bort det.",
    	exist: "{item} existerar redan",
    	generic: "Är du säker?",
    	home_assistant_is_restarting: "Vänta, Home Assistant startar nu om.",
    	home_assistant_version_not_correct: "Du kör Home Assistant-versionen '{haversion}', men detta repository kräver att lägsta versionen '{minversion}' måste installeras.",
    	no: "Nej",
    	no_upgrades: "Inga uppgraderingar väntar",
    	ok: "OK",
    	overwrite: "Detta kommer att skriva över den.",
    	reload_data: "Detta laddar om data för alla repositories HACS vet om, kommer detta att ta lite tid att slutföra.",
    	restart_home_assistant: "Är du säker på att du vill starta om Home Assistant?",
    	uninstall: "Är du säker på att du vill avinstallera '{item}'?",
    	upgrade_all: "Detta kommer uppgradera alla dessa repositories, säkerhetsställ att du läst release anteckningarna för dem alla innan du fortsätter",
    	yes: "Ja"
    };
    var options$g = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Upptäck och följ Appdaemon applikationer",
    				country: "Filtrera på landskod.",
    				debug: "Aktivera felsökning",
    				experimental: "Använd experimentella funktioner",
    				netdaemon: "Upptäck och följ NetDaemon applikationer",
    				not_in_use: "Används inte med YAML",
    				release_limit: "Antalet releaser som visas.",
    				sidepanel_icon: "Ikon för sidpanel",
    				sidepanel_title: "Rubrik för sidpanel"
    			}
    		}
    	}
    };
    var repository_banner$g = {
    	config_flow: "Den här integreringen stöder config_flow, det innebär att du nu kan gå till integrationsdelen i användargränssnittet för att konfigurera det.",
    	config_flow_title: "UI-konfiguration stöds",
    	integration_not_loaded: "Denna integration inte laddats i Hem Assistent.",
    	no_restart_required: "Ingen omstart krävs",
    	not_loaded: "Ej laddad",
    	plugin_not_loaded: "Detta plugin är inte tillagt till din Lovelaceresurs.",
    	restart: "Du måste starta om Home Assistant.",
    	restart_pending: "Omstart väntar"
    };
    var repository$g = {
    	add_to_lovelace: "Lägg till i Lovelace",
    	authors: "Författare",
    	available: "Tillgänglig",
    	back_to: "Tillbaka till",
    	changelog: "Ändringslogg",
    	downloads: "Nedladdningar",
    	flag_this: "Flagga denna",
    	frontend_version: "Frontend-version",
    	github_stars: "GitHub stjärnor",
    	goto_integrations: "Gå till integrationer",
    	hide: "Göm",
    	hide_beta: "Göm betaversioner",
    	install: "Installera",
    	installed: "Installerad",
    	lovelace_copy_example: "Kopiera exemplet till urklipp",
    	lovelace_instruction: "När du lägger till denna till din lovelace konfiguration, använd",
    	lovelace_no_js_type: "Kan inte avgöra villken typ av plugin, kontrollera i GIT \nrepository",
    	newest: "nyaste",
    	note_appdaemon: "du behöver fortfarande lägga till den till filen 'apps.yaml'",
    	note_installed: "När den är installerad kommer den finnas i",
    	note_integration: "du behöver fortfarande lägga den till filen 'configuration.yaml'",
    	note_plugin: "du behöver fortfarande lägga till den till din lovelace konfiguration ('ui-lovelace.yaml' eller raw UI config redigerare)",
    	note_plugin_post_107: "du behöver fortfarande lägga till den i din lovelace-konfiguration ('configuration.yaml' eller resursredigeraren /config/lovelace/resources')",
    	open_issue: "Rapportera problem",
    	open_plugin: "Öppna plugin",
    	reinstall: "Ominstallera",
    	repository: "Repository",
    	restart_home_assistant: "Starta om Home Assistant",
    	show_beta: "Visa betaversioner",
    	uninstall: "Avinstallera",
    	update_information: "Uppdatera information",
    	upgrade: "Uppdatera"
    };
    var settings$g = {
    	add_custom_repository: "LÄGG TILL ETT REPOSITORY",
    	adding_new_repo: "Lägger till nytt repository '{repo}'",
    	adding_new_repo_category: "Med kategori '{kategori}'.",
    	bg_task_custom: "Anpassade repositories är dolda under tiden aktiviteter i bakgrunden körs",
    	category: "Kategori",
    	compact_mode: "Kompakt läge",
    	custom_repositories: "REPOSITORIES",
    	"delete": "Ta bort",
    	display: "Visa",
    	grid: "Rutnät",
    	hacs_repo: "HACS repo",
    	hidden_repositories: "dolda förråd",
    	missing_category: "Du behöver välja en kategori",
    	open_repository: "Öppna Repository",
    	reload_data: "Ladda om data",
    	reload_window: "Ladda om fönstret",
    	repository_configuration: "Konfiguration av repository",
    	save: "Spara",
    	table: "Tabell",
    	table_view: "Tabellvy",
    	unhide: "ta fram",
    	upgrade_all: "Uppgradera alla"
    };
    var store$g = {
    	ascending: "stigande",
    	clear_new: "Rensa alla nya förvar",
    	descending: "fallande",
    	last_updated: "Senast uppdaterad",
    	name: "Namn",
    	new_repositories: "Nya förvar",
    	pending_upgrades: "Väntande uppgraderingar",
    	placeholder_search: "Skriv ett sökord...",
    	sort: "sortera",
    	stars: "Stjärnor",
    	status: "Status"
    };
    var time$g = {
    	ago: "sedan",
    	day: "dag",
    	days: "dagar",
    	hour: "timme",
    	hours: "timmar",
    	minute: "minut",
    	minutes: "minuter",
    	month: "månad",
    	months: "månader",
    	one: "Ett",
    	one_day_ago: "för en dag sedan",
    	one_hour_ago: "för en timme sedan",
    	one_minute_ago: "en minut sedan",
    	one_month_ago: "en månad sedan",
    	one_second_ago: "för en sekund sedan",
    	one_year_ago: "ett år sedan",
    	second: "andra",
    	seconds: "sekunder",
    	x_days_ago: "{x} dagar sedan",
    	x_hours_ago: "{x} timmar sedan",
    	x_minutes_ago: "{x} minuter sedan",
    	x_months_ago: "{x} månader sedan",
    	x_seconds_ago: "{x} sekunder sedan",
    	x_years_ago: "{x} år sedan",
    	year: "år",
    	years: "år"
    };
    var sv = {
    	common: common$g,
    	config: config$g,
    	confirm: confirm$g,
    	options: options$g,
    	repository_banner: repository_banner$g,
    	repository: repository$g,
    	settings: settings$g,
    	store: store$g,
    	time: time$g
    };

    var sv$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$g,
        config: config$g,
        confirm: confirm$g,
        options: options$g,
        repository_banner: repository_banner$g,
        repository: repository$g,
        settings: settings$g,
        store: store$g,
        time: time$g,
        'default': sv
    });

    var common$h = {
    	about: "Thông tin",
    	add: "thêm",
    	appdaemon_apps: "Ứng dụng AppDaemon",
    	appdaemon_plural: "Ứng dụng AppDaemon",
    	background_task: "Tác vụ nền đang chạy, trang này sẽ được tải lại khi hoàn thành.",
    	check_log_file: "Kiểm tra tệp nhật ký để biết thêm chi tiết.",
    	"continue": "Tiếp tục",
    	disabled: "Bị vô hiệu hoá",
    	documentation: "Tài liệu",
    	element: "thành phần",
    	hacs_is_disabled: "HACS đã bị vô hiệu hoá",
    	install: "Cài đặt",
    	installed: "đã cài đặt",
    	integration: "Tích Hợp",
    	integration_plural: "Tích Hợp",
    	integrations: "Tích Hợp",
    	lovelace: "Lovelace",
    	lovelace_element: "Thành phần Lovelace",
    	lovelace_elements: "Các thành phần Lovelace",
    	manage: "quản lý",
    	netdaemon: "NetDaemon",
    	netdaemon_apps: "Ứng dụng NetDaemon",
    	netdaemon_plural: "Ứng dụng NetDaemon",
    	plugin: "Bổ Sung",
    	plugin_plural: "Các thành phần Lovelace",
    	plugins: "Bổ Sung",
    	python_script: "Chương trình Python",
    	python_script_plural: "Chương trình Python",
    	python_scripts: "Chương trình Python",
    	repositories: "Các kho ứng dụng",
    	repository: "Kho lưu trữ",
    	settings: "thiết lập",
    	theme: "Chủ đề",
    	theme_plural: "Chủ đề",
    	themes: "Chủ đề",
    	uninstall: "Gỡ cài đặt",
    	update: "Cập nhật",
    	version: "Phiên bản"
    };
    var config$h = {
    	abort: {
    		single_instance_allowed: "Chỉ cho phép một cấu hình HACS duy nhất."
    	},
    	error: {
    		auth: "Mã Truy Cập Cá Nhân không đúng."
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "Kích hoạt phát hiện & theo dõi các ứng dụng AppDaemon",
    				netdaemon: "Kích hoạt phát hiện & theo dõi các ứng dụng NetDaemon",
    				python_script: "Kích hoạt phát hiện & theo dõi các Chương trình Python",
    				sidepanel_icon: "Biểu tượng bảng điều khiển cạnh bên",
    				sidepanel_title: "Tiêu đề bảng điều khiển cạnh bên",
    				theme: "Kích hoạt phát hiện & theo dõi các Chủ Đề",
    				token: "Mã Truy Cập Cá Nhân từ GitHub"
    			},
    			description: "Nếu bạn cần giúp đỡ về các cấu hình, hãy xem ở đây:\nhttps://hacs.xyz/docs/configuration/start"
    		}
    	}
    };
    var confirm$h = {
    	add_to_lovelace: "Bạn có chắc muốn thêm vào tài nguyên Lovelace của mình không?",
    	bg_task: "Hành động bị vô hiệu hoá trong khi các tác vụ nền đang chạy.",
    	cancel: "Huỷ bỏ",
    	"continue": "Bạn có chắc chắn muốn tiếp tục?",
    	"delete": "Bạn có chắc muốn xoá '{item}'?",
    	delete_installed: "'{item}' đã được cài đặt, bạn cần gỡ cài đặt nó trước khi có thể xoá.",
    	exist: "{item} đang tồn tại",
    	generic: "Bạn có chắc không?",
    	home_assistant_is_restarting: "Khoan đã, Home Assistant đang khởi động lại.",
    	home_assistant_version_not_correct: "Bạn đang chạy phiên bản Home Assistant '{haversion}' nhưng kho ứng dụng này yêu cầu phiên bản thấp nhất '{minversion}' để cài đặt.",
    	no: "Không",
    	no_upgrades: "Không có cập nhật nào đang xử lý",
    	ok: "OK",
    	overwrite: "Tiếp tục sẽ khiến card này bị ghi đè lên.",
    	reload_data: "Điều này tải lại dữ liệu của tất cả các kho ứng dụng trong HACS, sẽ mất một lúc để hoàn tất.",
    	restart_home_assistant: "Bạn có muốn khởi động lại Home Assistant?",
    	uninstall: "Bạn có chắc muốn gỡ cài đặt '{item}'?",
    	upgrade_all: "Điều này sẽ nâng cấp tất cả các kho ứng dụng này, đảm bảo rằng bạn đã đọc tất cả các lưu ý phát hành trước khi tiếp tục.",
    	yes: "Có"
    };
    var dialog_about$c = {
    	frontend_version: "Phiên bản Frontend",
    	installed_repositories: "Kho lưu trữ đã cài đặt",
    	integration_version: "Phiên bản tích hợp",
    	useful_links: "Liên kết hữu ích"
    };
    var dialog_add_repo$c = {
    	limit: "Chỉ 100 kho lưu trữ đầu tiên được hiển thị, sử dụng mục tìm kiếm để lọc những gì bạn cần",
    	no_match: "Không tìm thấy kho lưu trữ phù hợp với bộ lọc của bạn",
    	sort_by: "Sắp xếp theo",
    	title: "Thêm kho lưu trữ"
    };
    var dialog_custom_repositories$c = {
    	category: "Danh mục",
    	no_category: "Thiếu danh mục",
    	no_repository: "Kho lưu trữ bị thiếu",
    	title: "Các kho lưu trữ tuỳ chỉnh",
    	url_placeholder: "Thêm URL của kho lưu trữ tùy chỉnh"
    };
    var dialog_info$c = {
    	author: "Tác giả",
    	downloads: "Tải xuống",
    	install: "Cài đặt kho lưu trữ này trong HACS",
    	loading: "Đang tải thông tin...",
    	no_info: "Nhà phát triển đã không cung cấp thêm thông tin nào cho kho lưu trữ này",
    	open_issues: "Báo cáo vấn đề",
    	open_repo: "Mở Kho ứng dụng",
    	stars: "Số sao",
    	version_installed: "Phiên bản đã cài đặt"
    };
    var dialog_install$c = {
    	restart: "Hãy nhớ rằng bạn cần khởi động lại Home Assistant trước khi các thay đổi với tích hợp (custom_components) được áp dụng.",
    	select_version: "Chọn phiên bản",
    	show_beta: "Hiển thị phiên bản beta",
    	type: "Loại",
    	url: "URL"
    };
    var dialog_update$c = {
    	available_version: "Phiên bản hiện có",
    	changelog: "Thay đổi",
    	installed_version: "Phiên bản đã cài đặt",
    	releasenotes: "Thông tin phiên bản {Release}",
    	title: "Cập nhật đang chờ"
    };
    var entry$c = {
    	information: "Thông tin",
    	intro: "Các cập nhật và thông điệp quan trọng sẽ hiển thị ở đây nếu có",
    	messages: {
    		disabled: {
    			content: "Kiểm tra tệp nhật ký của bạn để biết thêm chi tiết",
    			title: "HACS đã bị vô hiệu hoá"
    		},
    		has_pending_tasks: {
    			content: "Một số kho có thể không thấy cho đến khi điều này hoàn tất",
    			title: "Tác vụ nền đang chờ"
    		},
    		resources: {
    			content: "Bạn có {number} thành phần Lovelace không được tải chính xác.",
    			title: "Không được tải trong Lovelace"
    		},
    		restart: {
    			content: "Bạn có {number} tích hợp yêu cầu khởi động lại Home Assistant, bạn có thể làm điều này từ mục 'Điều khiển máy chủ' bên trong tab Cấu hình trên giao diện Home Assistant.",
    			title: "Đang chờ khởi động lại"
    		},
    		startup: {
    			content: "HACS đang khởi động, suốt quá trình này có thể một số thông tin sẽ bị thiếu hoặc không chính xác",
    			title: "HACS đang khởi động"
    		},
    		wrong_frontend_installed: {
    			content: "Bạn có HACS frontend phiên bản {running} đã được cài đặt, nhưng phiên bản yêu cầu là {version}, khi bạn nhìn thấy thông điệp này có nghĩa là Home Assistant không thể cài đặt phiên bản mới, thử khởi động lại Home Assistant.",
    			title: "Phiên bản frontend không đúng"
    		},
    		wrong_frontend_loaded: {
    			content: "Bạn đang chạy phiên bản {running} của HACS frontend, nhưng phiên bản được yêu cầu là {expected}, bạn nên xoá bộ đệm của trình duyệt web.",
    			title: "Phiên bản frontend không đúng"
    		}
    	},
    	pending_updates: "Đang chờ cập nhật"
    };
    var menu$c = {
    	about: "Giới thiệu HACS",
    	clear: "Ẩn thông báo mục mới",
    	custom_repositories: "Các kho ứng dụng tuỳ chỉnh",
    	dismiss: "Bỏ qua tất cả kho chứa mới",
    	documentation: "Tài liệu",
    	open_issue: "Báo cáo vấn đề",
    	reload: "Tải lại cửa sổ"
    };
    var options$h = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "Kích hoạt phát hiện & theo dõi các ứng dụng AppDaemon",
    				country: "Lọc với mã quốc gia.",
    				debug: "Kích hoạt Trình gỡ lỗi",
    				experimental: "Kích hoạt các tính năng thử nghiệm",
    				netdaemon: "Kích hoạt phát hiện & theo dõi các ứng dụng NetDaemon",
    				not_in_use: "Không sử dụng với YAML",
    				release_limit: "Số lượng phiên bản hiển thị.",
    				sidepanel_icon: "Biểu tượng bảng điều khiển cạnh bên",
    				sidepanel_title: "Tiêu đề bảng điều khiển cạnh bên"
    			}
    		}
    	}
    };
    var repository_banner$h = {
    	config_flow: "Tích hợp này hỗ trợ config_flow, điều này có nghĩa là bây giờ bạn có thể chuyển đến khu vực Tích hợp trên giao diện để cấu hình nó.",
    	config_flow_title: "Hỗ trợ cấu hình qua giao diện người dùng",
    	integration_not_loaded: "Tích hợp này chưa được tải trong Home Assistant.",
    	no_restart_required: "Không cần khởi động lại",
    	not_loaded: "Chưa được tải",
    	plugin_not_loaded: "Bổ sung này chưa được thêm vào tài nguyên Lovelace của bạn.",
    	restart: "Bạn cần khởi động lại Home Assistant.",
    	restart_pending: "Đang chờ khởi động lại"
    };
    var repository_card$c = {
    	dismiss: "bỏ qua",
    	hide: "Ẩn",
    	information: "Thông tin",
    	new_repository: "Kho lưu trữ mới",
    	not_loaded: "Không được tải",
    	open_issue: "Báo cáo vấn đề",
    	open_source: "Mã nguồn mở",
    	pending_restart: "Đang chờ khởi động lại",
    	pending_update: "Cập nhật đang chờ",
    	reinstall: "Cài đặt lại",
    	report: "Báo cáo để loại bỏ",
    	update_information: "Cập nhật thông tin"
    };
    var repository$h = {
    	add_to_lovelace: "Thêm vào Lovelace",
    	authors: "Tác giả",
    	available: "Có sẵn",
    	back_to: "Quay lại",
    	changelog: "Nhật ký thay đổi",
    	downloads: "Tải xuống",
    	flag_this: "Đánh dấu mục này",
    	frontend_version: "Phiên bản Frontend",
    	github_stars: "Số Sao trên GitHub",
    	goto_integrations: "Đi đến Tích hợp",
    	hide: "Ẩn",
    	hide_beta: "Ẩn phiên bản beta",
    	install: "Cài đặt",
    	installed: "Đã cài đặt",
    	lovelace_copy_example: "Sao chép ví dụ vào bộ nhớ tạm",
    	lovelace_instruction: "Khi bạn thêm mục này vào cấu hình lovelace, sử dụng",
    	lovelace_no_js_type: "Không thể xác định loại Bổ sung này, kiểm tra lại kho ứng dụng.",
    	newest: "mới nhất",
    	note_appdaemon: "bạn vẫn cần thêm vào tập tin 'apps.yaml'",
    	note_installed: "Một khi được cài đặt, mục nằm sẽ nằm ở",
    	note_integration: "bạn vẫn cần thêm vào tập tin 'configuration.yaml'",
    	note_plugin: "bạn vẫn cần thêm vào cấu hình lovelace ('ui-lovelace.yaml' hoặc trình soạn thảo giao diện)",
    	note_plugin_post_107: "bạn vẫn cần phải thêm nó vào cấu hình Lovelace ('configuration.yaml' hoặc trình biên tập tài nguyên '/config/Lovelace/Resources ')",
    	open_issue: "Báo cáo vấn đề",
    	open_plugin: "Mở Bổ sung",
    	reinstall: "Cài đặt lại",
    	repository: "Kho ứng dụng",
    	restart_home_assistant: "Khởi động lại Home Assistant",
    	show_beta: "Hiển thị phiên bản beta",
    	uninstall: "Gỡ cài đặt",
    	update_information: "Cập nhật thông tin",
    	upgrade: "Cập nhật"
    };
    var search$c = {
    	placeholder: "Tìm kiếm kho lưu trữ"
    };
    var sections$d = {
    	about: {
    		description: "Hiển thị thông tin về HACS",
    		title: "Thông tin"
    	},
    	automation: {
    		description: "Đây là nơi bạn tìm thấy python_scripts, ứng dụng AppDaemon và ứng dụng NetDaemon",
    		title: "Tự động hóa"
    	},
    	frontend: {
    		description: "Đây là nơi bạn tìm thấy chủ đề, thẻ tùy chỉnh và các thành phần khác cho lovelace",
    		title: "Frontend"
    	},
    	integrations: {
    		description: "Đây là nơi bạn tìm thấy tích hợp tùy chỉnh (custom_components)",
    		title: "Tích Hợp"
    	},
    	pending_repository_upgrade: "Bạn đang chạy phiên bản {installed}, phiên bản {available} có sẵn"
    };
    var settings$h = {
    	add_custom_repository: "THÊM KHO ỨNG DỤNG TUỲ CHỈNH",
    	adding_new_repo: "Thêm kho ứng dụng mới '{repo}'",
    	adding_new_repo_category: "Với danh mục '{category}'",
    	bg_task_custom: "Các kho ứng dụng tuỳ chỉnh bị ẩn khi tác vụ nền đang chạy.",
    	category: "Danh mục",
    	compact_mode: "Chế độ thu gọn",
    	custom_repositories: "KHO ỨNG DỤNG TUỲ CHỈNH",
    	"delete": "Xoá",
    	display: "Hiển thị",
    	grid: "Lưới",
    	hacs_repo: "Kho ứng dụng HACS",
    	hidden_repositories: "các kho ứng dụng bị ẩn",
    	missing_category: "Bạn cần chọn một danh mục",
    	open_repository: "Mở Kho ứng dụng",
    	reload_data: "Tải lại dữ liệu",
    	reload_window: "Tải lại cửa sổ",
    	repository_configuration: "Cấu hình Kho ứng dụng",
    	save: "Lưu",
    	table: "Bảng",
    	table_view: "Xem dạng bảng",
    	unhide: "thôi ẩn",
    	upgrade_all: "Nâng cấp tất cả"
    };
    var store$h = {
    	ascending: "tăng dần",
    	clear_new: "Xoá tất cả kho ứng dụng mới",
    	descending: "giảm dần",
    	last_updated: "Cập nhật lần cuối",
    	name: "Tên",
    	new_repositories: "Kho ứng dụng Mới",
    	new_repositories_note: "Bạn có hơn 10 kho chứa mới được hiển thị, nếu bạn muốn xoá tất cả thông báo này, bấm vào dấu 3 chấm ở góc trên bên phải rồi chọn Bỏ qua tất cả kho chứa mới.",
    	no_repositories: "Không có kho lưu trữ",
    	no_repositories_desc1: "Có vẻ như bạn chưa có bất kỳ kho lưu trữ nào được cài đặt trong phần này.",
    	no_repositories_desc2: "Nhấp vào biểu tượng + ở góc dưới cùng để thêm mục mới đầu tiên của bạn!",
    	no_repositories_found_desc1: "Không tìm thấy kho lưu trữ được cài đặt phù hợp với \"{searchInput}\" trong mục này.",
    	no_repositories_found_desc2: "Thử tìm kiếm với từ khoá khác!",
    	pending_upgrades: "Nâng cấp chờ xử lý",
    	placeholder_search: "Vui lòng nhập cụm từ cần tìm",
    	sort: "sắp xếp",
    	stars: "Sao",
    	status: "Trạng thái"
    };
    var time$h = {
    	ago: "trước",
    	day: "ngày",
    	days: "ngày",
    	hour: "giờ",
    	hours: "giờ",
    	minute: "phút",
    	minutes: "phút",
    	month: "tháng",
    	months: "tháng",
    	one: "Một",
    	one_day_ago: "một ngày trước",
    	one_hour_ago: "một giờ trước",
    	one_minute_ago: "một phút trước",
    	one_month_ago: "một tháng trước",
    	one_second_ago: "một giây trước",
    	one_year_ago: "một năm trước",
    	second: "giây",
    	seconds: "giây",
    	x_days_ago: "{x} ngày trước",
    	x_hours_ago: "{x} giờ trước",
    	x_minutes_ago: "{x} phút trước",
    	x_months_ago: "{x} tháng trước",
    	x_seconds_ago: "{x} giây trước",
    	x_years_ago: "{x} năm trước",
    	year: "năm",
    	years: "năm"
    };
    var vi = {
    	common: common$h,
    	config: config$h,
    	confirm: confirm$h,
    	dialog_about: dialog_about$c,
    	dialog_add_repo: dialog_add_repo$c,
    	dialog_custom_repositories: dialog_custom_repositories$c,
    	dialog_info: dialog_info$c,
    	dialog_install: dialog_install$c,
    	dialog_update: dialog_update$c,
    	entry: entry$c,
    	menu: menu$c,
    	options: options$h,
    	repository_banner: repository_banner$h,
    	repository_card: repository_card$c,
    	repository: repository$h,
    	search: search$c,
    	sections: sections$d,
    	settings: settings$h,
    	store: store$h,
    	time: time$h
    };

    var vi$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$h,
        config: config$h,
        confirm: confirm$h,
        dialog_about: dialog_about$c,
        dialog_add_repo: dialog_add_repo$c,
        dialog_custom_repositories: dialog_custom_repositories$c,
        dialog_info: dialog_info$c,
        dialog_install: dialog_install$c,
        dialog_update: dialog_update$c,
        entry: entry$c,
        menu: menu$c,
        options: options$h,
        repository_banner: repository_banner$h,
        repository_card: repository_card$c,
        repository: repository$h,
        search: search$c,
        sections: sections$d,
        settings: settings$h,
        store: store$h,
        time: time$h,
        'default': vi
    });

    var common$i = {
    	about: "关于",
    	add: "添加",
    	appdaemon: "AppDaemon",
    	appdaemon_apps: "AppDaemon应用",
    	appdaemon_plural: "AppDaemon应用",
    	background_task: "后台任务正在运行，完成后将重新加载此页面。",
    	check_log_file: "请查看日志文件以了解更多信息。",
    	"continue": "继续",
    	disabled: "禁用",
    	documentation: "文档",
    	element: "元素",
    	hacs_is_disabled: "HACS已禁用",
    	install: "安装",
    	installed: "已安装",
    	integration: "集成",
    	integration_plural: "集成",
    	integrations: "集成",
    	lovelace: "Lovelace",
    	lovelace_element: "Lovelace元素",
    	lovelace_elements: "Lovelace元素",
    	manage: "管理",
    	netdaemon: "NetDaemon应用",
    	netdaemon_apps: "NetDaemon应用",
    	netdaemon_plural: "NetDaemon应用",
    	plugin: "Lovelace",
    	plugin_plural: "Lovelace元素",
    	plugins: "Lovelace元素",
    	python_script: "Python脚本",
    	python_script_plural: "Python脚本",
    	python_scripts: "Python脚本",
    	repositories: "储存库数量",
    	repository: "存储库",
    	settings: "设置",
    	theme: "主题",
    	theme_plural: "主题",
    	themes: "主题",
    	uninstall: "卸载",
    	update: "升级",
    	version: "版本"
    };
    var config$i = {
    	abort: {
    		single_instance_allowed: "仅允许单个HACS配置。"
    	},
    	error: {
    		auth: "个人访问令牌不正确。"
    	},
    	step: {
    		user: {
    			data: {
    				appdaemon: "启用AppDaemon应用发现和跟踪",
    				netdaemon: "启用NetDaemon应用发现和跟踪",
    				python_script: "启用python_scripts发现和跟踪",
    				sidepanel_icon: "侧面板图标",
    				sidepanel_title: "侧面板标题",
    				theme: "启用主题发现和跟踪",
    				token: "GitHub个人访问令牌"
    			},
    			description: "如果您需要有关配置的帮助，请在此处查看：https://hacs.xyz/docs/configuration/start",
    			title: "HACS (Home Assistant Community Store)"
    		}
    	},
    	title: "HACS (Home Assistant Community Store)"
    };
    var confirm$i = {
    	add_to_lovelace: "您确定要将此添加到Lovelace资源中吗？",
    	bg_task: "后台任务正在运行时，操作被禁用。",
    	cancel: "取消",
    	"continue": "你确定你要继续吗？",
    	"delete": "是否确实要删除\"{item}\"？",
    	delete_installed: "已安装 '{item}'，需要先将其卸载，然后才能将其删除。",
    	exist: "{item}已经存在",
    	generic: "你确定吗？",
    	home_assistant_is_restarting: "请等待，Home Assistant现在正在重新启动。",
    	home_assistant_version_not_correct: "您正在运行Home Assistant的版本为'{haversion}'，但是这个需要安装最低版本是'{minversion}'。",
    	no: "取消",
    	no_upgrades: "暂无升级",
    	ok: "确定",
    	overwrite: "这样做会覆盖它。",
    	reload_data: "这将重新加载HACS知道的所有存储库的数据，这需要一些时间才能完成。",
    	restart_home_assistant: "您确定要重新启动Home Assistant吗？",
    	uninstall: "您确定要卸载 '{item}' 吗？",
    	upgrade_all: "这将升级所有这些存储库，请确保在继续之前已阅读所有存储库的发行说明。",
    	yes: "确定"
    };
    var dialog_about$d = {
    	frontend_version: "前端版本",
    	installed_repositories: "已安装的存储库",
    	integration_version: "集成版本",
    	useful_links: "有用的链接"
    };
    var dialog_add_repo$d = {
    	limit: "仅显示前100个存储库，使用搜索过滤所需的内容",
    	no_match: "找不到与您的过滤器匹配的存储库",
    	sort_by: "排序方式",
    	title: "添加存储库"
    };
    var dialog_custom_repositories$d = {
    	category: "类别",
    	no_category: "没有选择类别",
    	no_repository: "存储库地址不能为空",
    	title: "自定义存储库",
    	url_placeholder: "添加自定义存储库URL"
    };
    var dialog_info$d = {
    	author: "作者",
    	downloads: "下载",
    	install: "在HACS中安装此存储库",
    	loading: "正在加载详细信息...",
    	no_info: "开发人员尚未为此存储库提供任何更多信息",
    	open_issues: "提交问题",
    	open_repo: "打开存储库",
    	stars: "星级",
    	version_installed: "已安装版本"
    };
    var dialog_install$d = {
    	restart: "请记住，在应用对集成（custom_components）所做的更改之前，您需要重新启动Home Assistant。",
    	select_version: "选择版本",
    	show_beta: "显示测试版",
    	type: "类型",
    	url: "URL"
    };
    var dialog_update$d = {
    	available_version: "可用版本",
    	changelog: "更改日志",
    	installed_version: "已安装版本",
    	releasenotes: "更新日志{release}",
    	title: "待更新"
    };
    var entry$d = {
    	information: "详情",
    	intro: "如果有更新和重要消息，将在此处显示",
    	messages: {
    		disabled: {
    			content: "请查看日志文件以了解更多信息",
    			title: "HACS已禁用"
    		},
    		has_pending_tasks: {
    			content: "在完成此操作之前，某些存储库可能不会显示",
    			title: "待处理的后台任务"
    		},
    		resources: {
    			content: "您有{number} 个Lovelace元素未正确加载到Lovelace中。",
    			title: "未载入Lovelace"
    		},
    		restart: {
    			content: "您有{number}集成，需要重新启动Home Assistant，您可以从Home Assistant 的配置下的“服务器控制”部分执行此操作。",
    			title: "待重启"
    		},
    		startup: {
    			content: "HACS正在启动，在此期间某些信息可能丢失或不正确",
    			title: "HACS正在启动"
    		},
    		wrong_frontend_installed: {
    			content: "您已经安装了{running}的HACS前端，但是{expected}版本是{expected} ，如果看到此消息，则Home Assistant无法安装新版本，请尝试重新启动Home Assistant。",
    			title: "额外的前端版本"
    		},
    		wrong_frontend_loaded: {
    			content: "您正在运行HACS前端的{running}版本，但是{expected}版本，您应该清除浏览器缓存。",
    			title: "额外的前端版本"
    		}
    	},
    	pending_updates: "待升级"
    };
    var menu$d = {
    	about: "关于HACS",
    	clear: "清除所有新标记",
    	custom_repositories: "自定义存储库",
    	dismiss: "忽略所有新的存储库",
    	documentation: "文档",
    	open_issue: "提交问题",
    	reload: "重新加载窗口"
    };
    var options$i = {
    	step: {
    		user: {
    			data: {
    				appdaemon: "启用AppDaemon应用发现和跟踪",
    				country: "用国家代码过滤。",
    				debug: "启用调试。",
    				experimental: "启用实验功能",
    				netdaemon: "启用NetDaemon应用发现和跟踪",
    				not_in_use: "不适用于 YAML",
    				release_limit: "要显示的发行数量。",
    				sidepanel_icon: "侧面板图标",
    				sidepanel_title: "侧面板标题"
    			}
    		}
    	}
    };
    var repository_banner$i = {
    	config_flow: "此组件支持config_flow，这意味着您现在可以转到UI的“集成”部分进行配置。",
    	config_flow_title: "支持UI配置",
    	integration_not_loaded: "此集成未加载到Home Assistant中。",
    	no_restart_required: "无需重启",
    	not_loaded: "未加载",
    	plugin_not_loaded: "该元件未添加到您的Lovelace资源中。",
    	restart: "您需要重新启动Home Assistant。",
    	restart_pending: "重新启动待处理"
    };
    var repository_card$d = {
    	dismiss: "忽略",
    	hide: "隐藏",
    	information: "详情",
    	new_repository: "新存储库",
    	not_loaded: "未加载",
    	open_issue: "提交问题",
    	open_source: "开源",
    	pending_restart: "待重启",
    	pending_update: "待升级",
    	reinstall: "重新安装",
    	report: "报告删除",
    	update_information: "更新信息"
    };
    var repository$i = {
    	add_to_lovelace: "添加到Lovelace",
    	authors: "作者",
    	available: "版本号",
    	back_to: "返回",
    	changelog: "更新日志",
    	downloads: "下载",
    	flag_this: "标记",
    	frontend_version: "前端版本",
    	github_stars: "GitHub星级",
    	goto_integrations: "转到集成",
    	hide: "隐藏",
    	hide_beta: "隐藏测试版",
    	install: "安装",
    	installed: "已安装版本",
    	lovelace_copy_example: "复制样例代码到你的剪贴板",
    	lovelace_instruction: "您仍然需要将下列代码添加到lovelace配置中（“ ui-lovelace.yaml”或原始配置编辑器）:",
    	lovelace_no_js_type: "无法确定此元素的类型，请检查存储库。",
    	newest: "最新",
    	note_appdaemon: "您仍然需要将其添加到“ apps.yaml”文件中",
    	note_installed: "安装后，它将位于",
    	note_integration: "您仍然需要将其添加到“ configuration.yaml”文件中",
    	note_plugin: "您仍然需要将其添加到lovelace配置中（“ ui-lovelace.yaml”或原始配置编辑器）ui-lovelace.yaml",
    	note_plugin_post_107: "您仍然需要将其添加到lovelace配置中（“ configuration.yaml”或资源编辑器“/config/lovelace/resources”）",
    	open_issue: "提交问题",
    	open_plugin: "打开元素",
    	reinstall: "重新安装",
    	repository: "存储库",
    	restart_home_assistant: "重新启动Home Assistant",
    	show_beta: "显示测试版",
    	uninstall: "卸载",
    	update_information: "更新信息",
    	upgrade: "升级"
    };
    var search$d = {
    	placeholder: "搜索存储库"
    };
    var sections$e = {
    	about: {
    		description: "显示有关HACS的信息",
    		title: "关于"
    	},
    	automation: {
    		description: "在这里可以找到python_scripts，AppDaemon应用程序和NetDaemon应用程序",
    		title: "自动化"
    	},
    	frontend: {
    		description: "在这里，您可以找到主题，自定义卡片和其他用于lovelace的元素",
    		title: "前端"
    	},
    	integrations: {
    		description: "在这里您可以找到自定义集成（custom_components）",
    		title: "集成"
    	},
    	pending_repository_upgrade: "您正在运行版本{installed} ，有新版 ({available}) 可用"
    };
    var settings$i = {
    	add_custom_repository: "添加自定义存储库",
    	adding_new_repo: "添加新的储存库 '{repo}'",
    	adding_new_repo_category: "类别为 '{category}'。",
    	bg_task_custom: "自定义存储库在后台任务运行时被隐藏。",
    	category: "类别",
    	compact_mode: "紧凑模式",
    	custom_repositories: "自定义存储库",
    	"delete": "删除",
    	display: "视图方式",
    	grid: "网格",
    	hacs_repo: "HACS存储库",
    	hidden_repositories: "隐藏的存储库",
    	missing_category: "您需要选择一个类别",
    	open_repository: "打开存储库",
    	reload_data: "重载数据",
    	reload_window: "重新加载窗口",
    	repository_configuration: "存储库配置",
    	save: "保存",
    	table: "列表",
    	table_view: "表视图",
    	unhide: "取消隐藏",
    	upgrade_all: "升级全部"
    };
    var store$i = {
    	ascending: "升序",
    	clear_new: "清除所有新存储库标记",
    	descending: "降序",
    	last_updated: "最近更新时间",
    	name: "名称",
    	new_repositories: "新存储库",
    	new_repositories_note: "这里有多于10个新存储库，如果要清除它们，请单击右上角的3点按钮，然后将其全部忽略。",
    	no_repositories: "没有储存库",
    	no_repositories_desc1: "在本节中您似乎尚未安装任何存储库。",
    	no_repositories_desc2: "单击右下角的+，添加第一个！",
    	no_repositories_found_desc1: "在这里找不到与“ {searchInput} ”匹配的已安装存储库。",
    	no_repositories_found_desc2: "尝试寻找其他东西！",
    	pending_upgrades: "待升级",
    	placeholder_search: "搜索项目...",
    	sort: "排序",
    	stars: "星级",
    	status: "状态"
    };
    var time$i = {
    	ago: "过去",
    	day: "天",
    	days: "天",
    	hour: "小时",
    	hours: "小时",
    	minute: "分",
    	minutes: "分",
    	month: "月",
    	months: "月",
    	one: "一",
    	one_day_ago: "一天前",
    	one_hour_ago: "一个小时之前",
    	one_minute_ago: "一分钟前",
    	one_month_ago: "一个月前",
    	one_second_ago: "一秒钟前",
    	one_year_ago: "一年前",
    	second: "秒",
    	seconds: "秒",
    	x_days_ago: "{x}天前",
    	x_hours_ago: "{x}小时前",
    	x_minutes_ago: "{x}分钟前",
    	x_months_ago: "{x}个月前",
    	x_seconds_ago: "{x}秒前",
    	x_years_ago: "{x}年前",
    	year: "年",
    	years: "年"
    };
    var zhHans = {
    	common: common$i,
    	config: config$i,
    	confirm: confirm$i,
    	dialog_about: dialog_about$d,
    	dialog_add_repo: dialog_add_repo$d,
    	dialog_custom_repositories: dialog_custom_repositories$d,
    	dialog_info: dialog_info$d,
    	dialog_install: dialog_install$d,
    	dialog_update: dialog_update$d,
    	entry: entry$d,
    	menu: menu$d,
    	options: options$i,
    	repository_banner: repository_banner$i,
    	repository_card: repository_card$d,
    	repository: repository$i,
    	search: search$d,
    	sections: sections$e,
    	settings: settings$i,
    	store: store$i,
    	time: time$i
    };

    var zh_Hans = /*#__PURE__*/Object.freeze({
        __proto__: null,
        common: common$i,
        config: config$i,
        confirm: confirm$i,
        dialog_about: dialog_about$d,
        dialog_add_repo: dialog_add_repo$d,
        dialog_custom_repositories: dialog_custom_repositories$d,
        dialog_info: dialog_info$d,
        dialog_install: dialog_install$d,
        dialog_update: dialog_update$d,
        entry: entry$d,
        menu: menu$d,
        options: options$i,
        repository_banner: repository_banner$i,
        repository_card: repository_card$d,
        repository: repository$i,
        search: search$d,
        sections: sections$e,
        settings: settings$i,
        store: store$i,
        time: time$i,
        'default': zhHans
    });

    function localize(string, search = undefined, replace = undefined) {
        const languages = {
            da: da$1,
            de: de$1,
            el: el$1,
            en: en$1,
            es: es$1,
            fr: fr$1,
            hu: hu$1,
            it: it$1,
            nb: nb$1,
            nl: nl$1,
            nn: nn$1,
            pl: pl$1,
            pt_BR: pt_BR,
            ro: ro$1,
            ru: ru$1,
            sl: sl$1,
            sv: sv$1,
            vi: vi$1,
            zh_Hans: zh_Hans,
        };
        let translated;
        const split = string.split(".");
        const lang = (localStorage.getItem("selectedLanguage") || "en")
            .replace(/['"]+/g, "")
            .replace("-", "_");
        try {
            translated = languages[lang];
            split.forEach((section) => {
                translated = translated[section];
            });
        }
        catch (e) {
            translated = languages["en"];
            split.forEach((section) => {
                translated = translated[section];
            });
        }
        if (translated === undefined) {
            translated = languages["en"];
            split.forEach((section) => {
                translated = translated[section];
            });
        }
        if (search !== undefined && replace !== undefined) {
            translated = translated.replace(search, replace);
        }
        return translated;
    }

    const version = "202006281222";

    const addedToLovelace = (resources, configuration, repository) => {
        if (configuration.frontend_mode === "yaml") {
            return true;
        }
        if (!repository.installed) {
            return true;
        }
        if (repository.category !== "plugin") {
            return true;
        }
        const url = `/hacsfiles/${repository === null || repository === void 0 ? void 0 : repository.full_name.split("/")[1]}/${repository === null || repository === void 0 ? void 0 : repository.file_name}`;
        return resources.map((resource) => resource.url).includes(url);
    };

    const getMessages = memoizeOne((status, configuration, resources, repositories, removed) => {
        const messages = [];
        const repositoriesNotAddedToLovelace = [];
        const repositoriesRestartPending = [];
        repositories === null || repositories === void 0 ? void 0 : repositories.forEach((repo) => {
            if (repo.status === "pending-restart") {
                repositoriesRestartPending.push(repo);
            }
            if (repo.installed &&
                repo.category === "plugin" &&
                !addedToLovelace(resources, configuration, repo)) {
                repositoriesNotAddedToLovelace.push(repo);
            }
            if (repo.installed && removed.map((r) => r.repository).includes(repo.full_name)) {
                const removedrepo = removed.find((r) => r.repository !== repo.full_name);
            }
        });
        if (configuration.frontend_expected !== configuration.frontend_running) {
            messages.push({
                title: localize("entry.messages.wrong_frontend_installed.title"),
                content: localize("entry.messages.wrong_frontend_installed.content")
                    .replace("{running}", configuration.frontend_running)
                    .replace("{expected}", configuration.frontend_expected),
                severity: "error",
            });
        }
        else if (configuration.frontend_expected !== version) {
            messages.push({
                title: localize("entry.messages.wrong_frontend_loaded.title"),
                content: localize("entry.messages.wrong_frontend_loaded.content")
                    .replace("{running}", version)
                    .replace("{expected}", configuration.frontend_expected),
                severity: "error",
            });
        }
        if (status === null || status === void 0 ? void 0 : status.startup) {
            messages.push({
                title: localize("entry.messages.startup.title"),
                content: localize("entry.messages.startup.content"),
                severity: "information",
            });
        }
        if (status === null || status === void 0 ? void 0 : status.has_pending_tasks) {
            messages.push({
                title: localize("entry.messages.has_pending_tasks.title"),
                content: localize("entry.messages.has_pending_tasks.content"),
                severity: "warning",
            });
        }
        if (status === null || status === void 0 ? void 0 : status.disabled) {
            messages.push({
                title: localize("entry.messages.disabled.title"),
                content: localize("entry.messages.disabled.content"),
                severity: "error",
            });
        }
        if (repositoriesNotAddedToLovelace.length > 0) {
            messages.push({
                title: localize("entry.messages.resources.title"),
                content: localize("entry.messages.resources.content").replace("{number}", repositoriesNotAddedToLovelace.length),
                severity: "error",
                path: "/hacs/frontend",
            });
        }
        if (repositoriesRestartPending.length > 0) {
            messages.push({
                title: localize("entry.messages.restart.title"),
                content: localize("entry.messages.restart.content").replace("{number}", repositoriesRestartPending.length),
                severity: "error",
                path: "/config/server_control",
            });
        }
        return messages;
    });

    const sections$f = {
        updates: [],
        panels: [
            {
                icon: "mdi:puzzle",
                id: "integrations",
                categories: ["integration"],
            },
            {
                icon: "mdi:palette",
                id: "frontend",
                categories: ["plugin", "theme"],
            },
            {
                icon: "mdi:robot",
                id: "automation",
                categories: ["python_script", "appdaemon", "netdaemon"],
            },
        ],
    };
    const activePanel = (route) => {
        return sections$f.panels.find((panel) => panel.id === route.path.replace("/", ""));
    };

    //import "../components/hacs-link";
    let HacsEntryPanel = class HacsEntryPanel extends LitElement {
        constructor() {
            super(...arguments);
            this._panelsEnabled = memoizeOne((sections, config) => {
                return sections.panels.filter((panel) => {
                    const categories = panel.categories;
                    if (categories === undefined)
                        return true;
                    return categories.filter((c) => config === null || config === void 0 ? void 0 : config.categories.includes(c)).length !== 0;
                });
            });
        }
        render() {
            var _a;
            const messages = getMessages(this.status, this.configuration, this.lovelace, this.repositories, this.removed);
            this.isWide = window.localStorage.getItem("dockedSidebar") === '"always_hidden"';
            sections$f.updates = []; // reset so we don't get duplicates
            (_a = this.repositories) === null || _a === void 0 ? void 0 : _a.forEach((repo) => {
                if (repo.pending_upgrade) {
                    sections$f.updates.push(repo);
                }
            });
            const enabledSections = this._panelsEnabled(sections$f, this.configuration);
            return html `
      <hacs-single-page-layout
        .hass=${this.hass}
        .route=${this.route}
        .narrow=${this.narrow}
        .header=${this.narrow ? "HACS" : "Home Assistant Community Store"}
        intro="${this.isWide || (sections$f.updates.length === 0 && messages.length === 0)
            ? localize("entry.intro")
            : ""}"
      >
        ${messages.length !== 0
            ? html ` <ha-card>
              <div class="header">${localize("entry.information")}</div>
              ${messages.map((message) => html `
                    <paper-icon-item
                      ?disabled=${!message.path}
                      information
                      @click=${() => this._navigate(message)}
                    >
                      <ha-icon
                        class="${message.severity}"
                        icon="mdi:alert-circle"
                        slot="item-icon"
                      ></ha-icon>
                      <paper-item-body two-line>
                        ${message.title}
                        <div secondary>
                          ${message.content}
                        </div>
                      </paper-item-body>
                    </paper-icon-item>
                  `)}
            </ha-card>`
            : ""}
        ${sections$f.updates.length !== 0
            ? html ` <ha-card>
              <div class="header">${localize("entry.pending_updates")}</div>
              ${sortRepositoriesByName(sections$f.updates).map((repository) => html `
                    <paper-icon-item @click="${() => this._openUpdateDialog(repository)}">
                      <ha-icon
                        class="pending_update"
                        icon="mdi:arrow-up-circle"
                        slot="item-icon"
                      ></ha-icon>
                      <paper-item-body two-line>
                        ${repository.name}
                        <div secondary>
                          ${localize("sections.pending_repository_upgrade")
                .replace("{installed}", repository.installed_version)
                .replace("{available}", repository.available_version)}
                        </div>
                      </paper-item-body>
                    </paper-icon-item>
                  `)}
            </ha-card>`
            : ""}
        <ha-card>
          ${enabledSections.map((panel) => html `
                <paper-icon-item @click=${() => this._changeLocation(panel.id)}>
                  <ha-icon .icon=${panel.icon} slot="item-icon"></ha-icon>
                  <paper-item-body two-line>
                    ${localize(`sections.${panel.id}.title`)}
                    <div secondary>
                      ${localize(`sections.${panel.id}.description`)}
                    </div>
                  </paper-item-body>
                  <ha-icon-button icon="mdi:chevron-right"></ha-icon-button>
                </paper-icon-item>
              `)}
          <paper-icon-item @click=${this._openAboutDialog}>
            <ha-icon icon="mdi:information" slot="item-icon"></ha-icon>
            <paper-item-body two-line>
              ${localize(`sections.about.title`)}
              <div secondary>
                ${localize(`sections.about.description`)}
              </div>
            </paper-item-body>
          </paper-icon-item>
        </ha-card>
      </hacs-single-page-layout>
    `;
        }
        _openUpdateDialog(repository) {
            this.dispatchEvent(new CustomEvent("hacs-dialog", {
                detail: {
                    type: "update",
                    repository: repository.id,
                },
                bubbles: true,
                composed: true,
            }));
        }
        _openAboutDialog() {
            this.dispatchEvent(new CustomEvent("hacs-dialog", {
                detail: {
                    type: "about",
                    configuration: this.configuration,
                    repositories: this.repositories,
                },
                bubbles: true,
                composed: true,
            }));
        }
        _navigate(message) {
            if (message.path) {
                console.log(message);
                navigate(this, message.path);
                window.location.reload();
            }
        }
        static get styles() {
            return [
                HacsStyles,
                css `
        paper-icon-item {
          cursor: pointer;
        }
        paper-icon-item[information] {
          cursor: normal;
        }

        paper-item-body {
          width: 100%;
          min-height: var(--paper-item-body-two-line-min-height, 72px);
          display: var(--layout-vertical_-_display);
          flex-direction: var(--layout-vertical_-_flex-direction);
          justify-content: var(--layout-center-justified_-_justify-content);
        }
        paper-item-body {
          color: var(--hcv-text-color-primary);
        }
        paper-item-body div {
          font-size: 14px;
          color: var(--hcv-text-color-secondary);
        }
        div[secondary] {
          white-space: normal;
        }
      `,
            ];
        }
        _changeLocation(id) {
            this.route.path = `/${id}`;
            this.dispatchEvent(new CustomEvent("hacs-location-changed", {
                detail: { route: this.route },
                bubbles: true,
                composed: true,
            }));
        }
    };
    __decorate([
        property({ attribute: false })
    ], HacsEntryPanel.prototype, "configuration", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsEntryPanel.prototype, "hass", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsEntryPanel.prototype, "lovelace", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsEntryPanel.prototype, "repositories", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsEntryPanel.prototype, "route", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsEntryPanel.prototype, "status", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsEntryPanel.prototype, "removed", void 0);
    __decorate([
        property({ type: Boolean })
    ], HacsEntryPanel.prototype, "isWide", void 0);
    __decorate([
        property({ type: Boolean })
    ], HacsEntryPanel.prototype, "narrow", void 0);
    HacsEntryPanel = __decorate([
        customElement("hacs-entry-panel")
    ], HacsEntryPanel);

    let HacsTabbedLayout = class HacsTabbedLayout extends LitElement {
        render() {
            return html `
      <div class="main">
        <div class="toolbar">
        <ha-icon-button icon="mdi:chevron-left" @click=${this._goBack}></ha-icon-button>
                ${this.narrow
            ? html `
                        <div class="main-title">
                          ${localize(`sections.${this.selected}.title`)}
                        </div>
                      `
            : ""}
          <div id="tabbar" class=${classMap({ "bottom-bar": this.narrow })}>
            ${this.tabs.map((tab) => html `
                <div
                  class="${classMap({
            "toolbar-button": true,
            selected: this.selected === tab.id,
        })}"
                  @click=${() => this._ChangeTabAction(tab.id)}
                >
                  ${this.narrow
            ? html `<ha-icon-button .icon=${tab.icon}></ha-icon-button
                        ><span class="name"
                          >${localize(`sections.${tab.id}.title`)}</span
                        >`
            : localize(`sections.${tab.id}.title`)}
                </div>
              `)}
          </div>
          <div id="toolbar-icon">
          <slot name="toolbar-icon"></slot>
          </div>
        </div>
        </div>
        <div class="content"><slot></slot></div>
      </div>
    `;
        }
        _ChangeTabAction(tab) {
            window.scrollTo(0, 0);
            this.selected = tab;
            this.route.path = `/${tab}`;
            this._locationChanged();
        }
        _goBack() {
            this.route.path = "";
            this._locationChanged();
        }
        _locationChanged() {
            this.dispatchEvent(new CustomEvent("hacs-location-changed", {
                detail: { route: this.route },
                bubbles: true,
                composed: true,
            }));
        }
        static get styles() {
            return css `
      :host {
        display: block;
        height: 100vh;
        min-height: -webkit-fill-available;
        background-color: var(--primary-background-color);
      }
      ha-menu-button {
        margin-right: 24px;
      }
      .name {
        white-space: nowrap;
      }
      .toolbar {
        display: flex;
        align-items: center;
        font-size: 20px;
        height: 65px;
        background-color: var(--sidebar-background-color);
        font-weight: 400;
        color: var(--sidebar-text-color);
        border-bottom: 1px solid var(--divider-color);
        font-family: var(--paper-font-body1_-_font-family);
        padding: 0 16px;
        box-sizing: border-box;
      }
      .searchbar {
        display: flex;
        align-items: center;
        font-size: 20px;
        top: 65px;
        height: 65px;
        background-color: var(--sidebar-background-color);
        border-bottom: 1px solid var(--divider-color);
        padding: 0 16px;
        box-sizing: border-box;
      }
      #tabbar {
        display: flex;
        font-size: 14px;
      }
      #tabbar.bottom-bar {
        position: absolute;
        bottom: 0;
        left: 0;
        padding: 0 16px;
        box-sizing: border-box;
        background-color: var(--sidebar-background-color);
        border-top: 1px solid var(--divider-color);
        justify-content: space-between;
        z-index: 1;
        font-size: 12px;
        width: 100%;
      }
      #tabbar:not(.bottom-bar) {
        flex: 1;
        justify-content: center;
      }

      :host(:not([narrow])) #toolbar-icon {
        min-width: 40px;
      }
      ha-menu-button,
      ha-icon-button-arrow-prev,
      ::slotted([slot="toolbar-icon"]) {
        flex-shrink: 0;
        pointer-events: auto;
        color: var(--sidebar-icon-color);
      }

      .main-title {
        -webkit-font-smoothing: var(
          --paper-font-headline_-_-webkit-font-smoothing
        );
        flex: 1;
        font-family: var(--paper-font-headline_-_font-family);
        font-size: var(--paper-font-headline_-_font-size);
        font-weight: var(--paper-font-headline_-_font-weight);
        letter-spacing: var(--paper-font-headline_-_letter-spacing);
        line-height: 26px;
        max-height: 26px;
        opacity: var(--dark-primary-opacity);
        overflow: hidden;
        text-overflow: ellipsis;
      }

      .content {
        position: relative;
        width: 100%;
        height: calc(100vh - 65px);
        overflow-y: auto;
        overflow: auto;
        -webkit-overflow-scrolling: touch;
      }

      :host([narrow]) .content {
        height: calc(100vh - 128px);
      }

      .toolbar-button {
        padding: 0 32px;
        display: flex;
        flex-direction: column;
        text-align: center;
        align-items: center;
        justify-content: center;
        height: 64px;
        cursor: pointer;
        position: relative;
        outline: none;
        box-sizing: border-box;
      }

      ha-icon-button {
        cursor: pointer;
      }
      .selected {
        color: var(--primary-color);
        border-bottom: 2px solid var(--primary-color);
      }
      .search-input {
        width: calc(100% - 48px);
        height: 40px;
        border: 0;
        padding: 0 16px;
        font-size: initial;
        color: var(--sidebar-text-color);
        font-family: var(--paper-font-body1_-_font-family);
      }
      input:focus {
        outline-offset: 0;
        outline: 0;
      }
    `;
        }
    };
    __decorate([
        property({ attribute: false })
    ], HacsTabbedLayout.prototype, "hass", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsTabbedLayout.prototype, "route", void 0);
    __decorate([
        property({ type: Boolean })
    ], HacsTabbedLayout.prototype, "narrow", void 0);
    __decorate([
        property()
    ], HacsTabbedLayout.prototype, "selected", void 0);
    __decorate([
        property()
    ], HacsTabbedLayout.prototype, "tabs", void 0);
    HacsTabbedLayout = __decorate([
        customElement("hacs-tabbed-layout")
    ], HacsTabbedLayout);

    let HacsLink = class HacsLink extends LitElement {
        constructor() {
            super(...arguments);
            this.newtab = false;
        }
        render() {
            return html `<div @tap=${this._open} class="link"><slot></slot></div>`;
        }
        _open() {
            const external = this.url.includes("http");
            let features = "";
            let target = "_blank";
            if (external) {
                features = "noreferrer=true";
            }
            if (!external && !this.newtab) {
                target = "_top";
            }
            console.log(this.url, target, features);
            window.open(this.url, target, features);
        }
        static get styles() {
            return css `
      .link {
        cursor: pointer;
        color: var(--hcv-text-color-link);
        text-decoration: var(--hcv-text-decoration-link);
      }
    `;
        }
    };
    __decorate([
        property({ type: Boolean })
    ], HacsLink.prototype, "newtab", void 0);
    __decorate([
        property()
    ], HacsLink.prototype, "url", void 0);
    HacsLink = __decorate([
        customElement("hacs-link")
    ], HacsLink);

    let HacsRepositoryCard = class HacsRepositoryCard extends LitElement {
        get _borderClass() {
            const classes = {};
            if (!this.addedToLovelace || this.repository.status === "pending-restart") {
                classes["status-issue"] = true;
            }
            else if (this.repository.pending_upgrade) {
                classes["status-update"] = true;
            }
            else if (this.repository.new && !this.repository.installed) {
                classes["status-new"] = true;
            }
            if (Object.keys(classes).length !== 0) {
                classes["status-border"] = true;
            }
            return classes;
        }
        get _headerClass() {
            const classes = {};
            if (!this.addedToLovelace || this.repository.status === "pending-restart") {
                classes["issue-header"] = true;
            }
            else if (this.repository.pending_upgrade) {
                classes["update-header"] = true;
            }
            else if (this.repository.new && !this.repository.installed) {
                classes["new-header"] = true;
            }
            else {
                classes["default-header"] = true;
            }
            return classes;
        }
        get _headerTitle() {
            if (!this.addedToLovelace) {
                return localize("repository_card.not_loaded");
            }
            if (this.repository.status === "pending-restart") {
                return localize("repository_card.pending_restart");
            }
            if (this.repository.pending_upgrade) {
                return localize("repository_card.pending_update");
            }
            if (this.repository.new && !this.repository.installed) {
                return localize("repository_card.new_repository");
            }
            return "";
        }
        render() {
            const path = this.repository.local_path.split("/");
            return html `
      <ha-card class=${classMap(this._borderClass)}>
        <div class="card-content">
          <div class="group-header">
            <div class="status-header ${classMap(this._headerClass)}">
              ${this._headerTitle}
            </div>
            <div class="title">
              <h2 class="pointer" @click=${this._showReopsitoryInfo}>
                ${this.repository.name}
              </h2>
              ${this.repository.category !== "integration"
            ? html ` <hacs-chip
                    icon="hacs:hacs"
                    .value=${localize(`common.${this.repository.category}`)}
                  ></hacs-chip>`
            : ""}
            </div>
          </div>
          <paper-item>
            <paper-item-body>${this.repository.description}</paper-item-body></paper-item
          >
        </div>
        <div class="card-actions">
          ${this.repository.new && !this.repository.installed
            ? html `<div>
                  <mwc-button @click=${this._installRepository}
                    >${localize("common.install")}</mwc-button
                  >
                </div>
                <div>
                  <mwc-button @click=${this._showReopsitoryInfo}
                    >${localize("repository_card.information")}</mwc-button
                  >
                </div>
                <div>
                  <hacs-link .url="https://github.com/${this.repository.full_name}"
                    ><mwc-button>${localize("common.repository")}</mwc-button></hacs-link
                  >
                </div>
                <div>
                  <mwc-button @click=${this._setNotNew}
                    >${localize("repository_card.dismiss")}</mwc-button
                  >
                </div>`
            : this.repository.pending_upgrade && this.addedToLovelace
                ? html `<div>
                <mwc-button class="update-header" @click=${this._updateRepository} raised
                  >${localize("common.update")}</mwc-button
                >
              </div>`
                : html `<div>
                <hacs-link .url="https://github.com/${this.repository.full_name}"
                  ><mwc-button>${localize("common.repository")}</mwc-button></hacs-link
                >
              </div>`}
          ${this.repository.installed
            ? html ` <paper-menu-button
                horizontal-align="right"
                vertical-align="top"
                vertical-offset="40"
                close-on-activate
              >
                <ha-icon-button icon="hass:dots-vertical" slot="dropdown-trigger"></ha-icon-button>
                <paper-listbox slot="dropdown-content">
                  <paper-item class="pointer" @tap=${this._showReopsitoryInfo}
                    >${localize("repository_card.information")}</paper-item
                  >

                  <paper-item class="pointer" @tap=${this._updateReopsitoryInfo}
                    >${localize("repository_card.update_information")}</paper-item
                  >

                  <paper-item @tap=${this._installRepository}
                    >${localize("repository_card.reinstall")}</paper-item
                  >

                  ${this.repository.category === "plugin"
                ? html `<hacs-link
                        .url="/hacsfiles/${path.pop()}/${this.repository.file_name}"
                        newtab
                        ><paper-item class="pointer"
                          >${localize("repository_card.open_source")}</paper-item
                        ></hacs-link
                      >`
                : ""}

                  <hacs-link .url="https://github.com/${this.repository.full_name}/issues"
                    ><paper-item class="pointer"
                      >${localize("repository_card.open_issue")}</paper-item
                    ></hacs-link
                  >

                  ${String(this.repository.id) !== "172733314"
                ? html `<hacs-link
                          .url="https://github.com/hacs/integration/issues/new?assignees=ludeeus&labels=flag&template=flag.md&title=${this
                    .repository.full_name}"
                          ><paper-item class="pointer uninstall"
                            >${localize("repository_card.report")}</paper-item
                          ></hacs-link
                        >
                        <paper-item class="pointer uninstall" @tap=${this._uninstallRepository}
                          >${localize("common.uninstall")}</paper-item
                        >`
                : ""}
                </paper-listbox>
              </paper-menu-button>`
            : ""}
        </div>
      </ha-card>
    `;
        }
        async _updateReopsitoryInfo() {
            await repositoryUpdate(this.hass, this.repository.id);
        }
        async _showReopsitoryInfo() {
            this.dispatchEvent(new CustomEvent("hacs-dialog", {
                detail: {
                    type: "repository-info",
                    repository: this.repository.id,
                },
                bubbles: true,
                composed: true,
            }));
        }
        _lovelaceUrl() {
            var _a, _b;
            return `/hacsfiles/${(_a = this.repository) === null || _a === void 0 ? void 0 : _a.full_name.split("/")[1]}/${(_b = this.repository) === null || _b === void 0 ? void 0 : _b.file_name}`;
        }
        async _updateRepository() {
            this.dispatchEvent(new CustomEvent("hacs-dialog", {
                detail: {
                    type: "update",
                    repository: this.repository.id,
                },
                bubbles: true,
                composed: true,
            }));
        }
        async _setNotNew() {
            await repositorySetNotNew(this.hass, this.repository.id);
        }
        _installRepository() {
            this.dispatchEvent(new CustomEvent("hacs-dialog", {
                detail: {
                    type: "install",
                    repository: this.repository.id,
                },
                bubbles: true,
                composed: true,
            }));
        }
        async _uninstallRepository() {
            await repositoryUninstall(this.hass, this.repository.id);
            if (this.repository.category === "plugin" && this.status.lovelace_mode !== "yaml") {
                const resources = await fetchResources(this.hass);
                resources
                    .filter((resource) => resource.url === this._lovelaceUrl())
                    .forEach((resource) => {
                    deleteResource(this.hass, String(resource.id));
                });
            }
        }
        static get styles() {
            return [
                HacsStyles,
                css `
        :host {
          max-width: 500px;
        }
        ha-card {
          display: flex;
          flex-direction: column;
          height: 100%;
        }
        hacs-chip {
          margin: 8px 4px 0 0;
        }
        .title {
          display: flex;
          justify-content: space-between;
        }
        .card-content {
          padding: 0 0 3px 0;
          height: 100%;
        }
        .card-actions {
          border-top: none;
          bottom: 0;
          display: flex;
          justify-content: space-between;
          align-items: center;
          padding: 5px;
        }
        .group-header {
          height: auto;
          align-content: center;
        }
        .group-header h2 {
          margin: 0;
          padding: 8px 16px;
        }
        h2 {
          margin-top: 0;
          min-height: 24px;
        }
        paper-menu-button {
          padding: 0;
          float: right;
        }

        .pointer {
          cursor: pointer;
        }
        paper-item-body {
          opacity: var(--dark-primary-opacity);
        }

        .status-new {
          border-color: var(--hcv-color-new);
        }

        .status-update {
          border-color: var(--hcv-color-update);
        }

        .status-issue {
          border-color: var(--hcv-color-error);
        }

        .new-header {
          background-color: var(--hcv-color-new);
          color: var(--hcv-text-color-on-background);
        }

        .issue-header {
          background-color: var(--hcv-color-error);
          color: var(--hcv-text-color-on-background);
        }

        .update-header {
          background-color: var(--hcv-color-update);
          color: var(--hcv-text-color-on-background);
        }

        .default-header {
          padding: 10px 0 !important;
        }

        mwc-button.update-header {
          --mdc-theme-primary: var(--hcv-color-update);
          --mdc-theme-on-primary: var(--hcv-text-color-on-background);
        }

        .status-border {
          border-style: solid;
          border-width: 1px;
        }

        .status-header {
          top: 0;
          padding: 6px 1px;
          margin: -1px;
          width: 100%;
          font-weight: 300;
          text-align: center;
          left: 0;
          border-top-left-radius: var(--ha-card-border-radius);
          border-top-right-radius: var(--ha-card-border-radius);
        }
      `,
            ];
        }
    };
    __decorate([
        property({ attribute: false })
    ], HacsRepositoryCard.prototype, "hass", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsRepositoryCard.prototype, "repository", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsRepositoryCard.prototype, "status", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsRepositoryCard.prototype, "removed", void 0);
    __decorate([
        property({ type: Boolean })
    ], HacsRepositoryCard.prototype, "narrow", void 0);
    __decorate([
        property({ type: Boolean })
    ], HacsRepositoryCard.prototype, "addedToLovelace", void 0);
    HacsRepositoryCard = __decorate([
        customElement("hacs-repository-card")
    ], HacsRepositoryCard);

    let HacsSearch = class HacsSearch extends LitElement {
        render() {
            return html `
      <div class="searchbar">
        <ha-icon icon="mdi:magnify" role="button"></ha-icon>
        <input
          id="search-input"
          class="search-input"
          placeholder="${localize("search.placeholder")}"
          .value=${this.input || ""}
          @input=${this._inputValueChanged}
        />
        ${this.input
            ? html `
              <ha-icon-button
                icon="mdi:close"
                role="button"
                @click=${this._clearInput}
              ></ha-icon-button>
            `
            : ""}
      </div>
    `;
        }
        _inputValueChanged() {
            var _a;
            this.input = (_a = this._filterInput) === null || _a === void 0 ? void 0 : _a.value;
        }
        _clearInput() {
            this.input = "";
            this.dispatchEvent(new CustomEvent("input", {
                bubbles: true,
                composed: true,
            }));
        }
        static get styles() {
            return [
                HacsStyles,
                css `
        .searchbar {
          display: flex;
          align-items: center;
          font-size: 20px;
          top: 65px;
          height: 65px;
          background-color: var(--sidebar-background-color);
          border-bottom: 1px solid var(--divider-color);
          padding: 0 16px;
          box-sizing: border-box;
        }
        .search-input {
          width: calc(100% - 48px);
          height: 40px;
          border: 0;
          padding: 0 16px;
          font-size: initial;
          color: var(--sidebar-text-color);
          font-family: var(--paper-font-body1_-_font-family);
        }
        input:focus {
          outline-offset: 0;
          outline: 0;
        }
        input {
          background-color: var(--sidebar-background-color);
        }
      `,
            ];
        }
    };
    __decorate([
        property()
    ], HacsSearch.prototype, "input", void 0);
    __decorate([
        query("#search-input")
    ], HacsSearch.prototype, "_filterInput", void 0);
    HacsSearch = __decorate([
        customElement("hacs-search")
    ], HacsSearch);

    let HacsCheckbox = class HacsCheckbox extends LitElement {
        get _checkboxClass() {
            return {
                'checkbox': true,
                'checked': this.checked,
            };
        }
        render() {
            return html `
      <div class="checkbox-container">
        <div class=${classMap(this._checkboxClass)} @click=${this._checkboxClicked}>
          <div class="value">${this.checked ? "✔" : ""}</div>
        </div>
        <div class="label" @click=${this._checkboxClicked}>
          ${this.label}
        </div>
      </div>
    `;
        }
        _checkboxClicked() {
            this.checked = !this.checked;
            this.dispatchEvent(new CustomEvent("checkbox-change", {
                detail: {
                    id: this.id,
                },
                bubbles: true,
                composed: true,
            }));
        }
        static get styles() {
            return [
                HacsStyles,
                css `
        .checkbox-container {
          display: flex;
          color: var(--hcv-text-color-primary);
          -webkit-user-select: none;
          -moz-user-select: none;
          -ms-user-select: none;
          user-select: none;
        }

        .label {
          line-height: 18px;
          opacity: var(--dark-primary-opacity);
          font-family: var(--paper-font-subhead_-_font-family);
          -webkit-font-smoothing: var(--paper-font-subhead_-_-webkit-font-smoothing);
          font-size: var(--paper-font-subhead_-_font-size);
          cursor: pointer;
        }

        .value {
          margin-left: 2px;
          color: var(--hcv-text-color-on-background);
        }

        .checkbox {
          cursor: pointer;
          height: 16px;
          width: 16px;
          font-size: 14px;
          margin-right: 8px;
          background-color: var(--primary-background-color);
          border: 1px solid var(--divider-color);
          border-radius: 4px;
          line-height: 16px;
        }

        .checkbox.checked {
          border-color: var(--accent-color);
          background-color: var(--accent-color);
        }
      `,
            ];
        }
    };
    __decorate([
        property({ attribute: false })
    ], HacsCheckbox.prototype, "checked", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsCheckbox.prototype, "label", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsCheckbox.prototype, "id", void 0);
    HacsCheckbox = __decorate([
        customElement("hacs-checkbox")
    ], HacsCheckbox);

    let HacsFilter = class HacsFilter extends LitElement {
        async firstUpdated() {
            this.addEventListener("checkbox-change", (e) => this._filterClick(e));
        }
        render() {
            var _a;
            return html `
      <div class="filter">
        ${(_a = this.filters) === null || _a === void 0 ? void 0 : _a.map((filter) => html `
            <hacs-checkbox
              class="checkbox"
              .label=${localize(`common.${filter.id}`) || filter.value}
              .id=${filter.id}
              .checked=${filter.checked || false}
            />
            </hacs-checkbox>`)}
      </div>
    `;
        }
        _filterClick(ev) {
            const filter = ev.detail;
            this.dispatchEvent(new CustomEvent("filter-change", {
                detail: {
                    id: filter.id,
                },
                bubbles: true,
                composed: true,
            }));
        }
        static get styles() {
            return [
                HacsStyles,
                css `
        .filter {
          display: flex;
          border-bottom: 1px solid var(--divider-color);
          align-items: center;
          font-size: 16px;
          height: 32px;
          line-height: 4px;
          background-color: var(--sidebar-background-color);
          padding: 0 16px;
          box-sizing: border-box;
        }

        .checkbox:not(:first-child) {
          margin-left: 20px;
        }
      `,
            ];
        }
    };
    __decorate([
        property({ attribute: false })
    ], HacsFilter.prototype, "filters", void 0);
    HacsFilter = __decorate([
        customElement("hacs-filter")
    ], HacsFilter);

    let HacsFab = class HacsFab extends LitElement {
        render() {
            return html `
      <div class="fab" ?narrow=${this.narrow}>
        <div><ha-icon .icon=${this.icon}></ha-icon></div>
      </div>
    `;
        }
        static get styles() {
            return [
                css `
        .fab {
          position: fixed;
          bottom: 16px;
          right: 16px;
          z-index: 1;
          display: inline-flex;
          align-items: center;
          justify-content: center;
          box-sizing: border-box;
          width: 56px;
          height: 56px;
          cursor: pointer;
          user-select: none;
          -webkit-appearance: none;
          background-color: var(--hcv-color-fab);
          -webkit-box-shadow: 2px 2px 8px 1px rgba(0, 0, 0, 0.3);
          -moz-box-shadow: 2px 2px 8px 1px rgba(0, 0, 0, 0.3);
          box-shadow: 2px 2px 8px 1px rgba(0, 0, 0, 0.3);
          border-radius: 50%;
        }
        .fab[narrow] {
          margin-bottom: 65px;
        }
        ha-icon {
          margin: auto;
          color: var(--hcv-text-color-fab);
          height: 100%;
          width: 100%;
          --mdc-icon-size: 32px;
        }
      `,
            ];
        }
    };
    __decorate([
        property()
    ], HacsFab.prototype, "icon", void 0);
    __decorate([
        property({ type: Boolean })
    ], HacsFab.prototype, "narrow", void 0);
    HacsFab = __decorate([
        customElement("hacs-fab")
    ], HacsFab);

    let HacsTabbedMenu = class HacsTabbedMenu extends LitElement {
        render() {
            var _a, _b, _c;
            return html `<paper-menu-button
      slot="toolbar-icon"
      horizontal-align="right"
      vertical-align="top"
      vertical-offset="40"
      close-on-activate
    >
      <ha-icon-button icon="hass:dots-vertical" slot="dropdown-trigger"></ha-icon-button>
      <paper-listbox slot="dropdown-content">
        <hacs-link url="https://hacs.xyz/"
          ><paper-item>${localize("menu.documentation")}</paper-item></hacs-link
        >
        <paper-item @tap=${() => window.location.reload(true)}
          >${localize("menu.reload")}</paper-item
        >
        ${((_a = this.repositories) === null || _a === void 0 ? void 0 : _a.filter((repo) => repo.new).length) !== 0
            ? html ` <paper-item @tap=${this._clearAllNewRepositories}
              >${localize("menu.dismiss")}</paper-item
            >`
            : ""}

        <hacs-link url="https://github.com/hacs"><paper-item>GitHub</paper-item></hacs-link>
        <hacs-link url="https://hacs.xyz/docs/issues"
          ><paper-item>${localize("menu.open_issue")}</paper-item></hacs-link
        >
        ${!((_b = this.status) === null || _b === void 0 ? void 0 : _b.disabled) && !((_c = this.status) === null || _c === void 0 ? void 0 : _c.background_task)
            ? html `<paper-item @tap=${this._showCustomRepositoriesDialog}
              >${localize("menu.custom_repositories")}</paper-item
            >`
            : ""}

        <paper-item @tap=${this._showAboutDialog}>${localize("menu.about")}</paper-item>
      </paper-listbox>
    </paper-menu-button>`;
        }
        async _clearAllNewRepositories() {
            const section = sections$f.panels.find((s) => s.id === this.route.path.replace("/", ""));
            await settingsClearAllNewRepositories(this.hass, section.categories);
        }
        _showAboutDialog() {
            this.dispatchEvent(new CustomEvent("hacs-dialog", {
                detail: {
                    type: "about",
                    configuration: this.configuration,
                    repositories: this.repositories,
                },
                bubbles: true,
                composed: true,
            }));
        }
        _showCustomRepositoriesDialog() {
            this.dispatchEvent(new CustomEvent("hacs-dialog", {
                detail: {
                    type: "custom-repositories",
                    repositories: this.repositories,
                },
                bubbles: true,
                composed: true,
            }));
        }
        static get styles() {
            return css `
      paper-menu-button {
        color: var(--hcv-text-color-secondary);
        padding: 0;
      }
      paper-item {
        cursor: pointer;
      }
      paper-item-body {
        opacity: var(--dark-primary-opacity);
      }
    `;
        }
    };
    __decorate([
        property({ attribute: false })
    ], HacsTabbedMenu.prototype, "configuration", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsTabbedMenu.prototype, "hass", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsTabbedMenu.prototype, "narrow", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsTabbedMenu.prototype, "route", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsTabbedMenu.prototype, "repositories", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsTabbedMenu.prototype, "lovelace", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsTabbedMenu.prototype, "status", void 0);
    HacsTabbedMenu = __decorate([
        customElement("hacs-tabbed-menu")
    ], HacsTabbedMenu);

    function filterRepositoriesByInput(repositories, filter) {
        const _lowcaseFilter = stringify(filter);
        return repositories.filter((_repo) => {
            var _a, _b, _c, _d, _e, _f;
            return ((_a = stringify(_repo.name)) === null || _a === void 0 ? void 0 : _a.includes(_lowcaseFilter)) || ((_b = stringify(_repo.description)) === null || _b === void 0 ? void 0 : _b.includes(_lowcaseFilter)) || ((_c = stringify(_repo.category)) === null || _c === void 0 ? void 0 : _c.includes(_lowcaseFilter)) || ((_d = stringify(_repo.full_name)) === null || _d === void 0 ? void 0 : _d.includes(_lowcaseFilter)) || ((_e = stringify(_repo.authors)) === null || _e === void 0 ? void 0 : _e.includes(_lowcaseFilter)) || ((_f = stringify(_repo.domain)) === null || _f === void 0 ? void 0 : _f.includes(_lowcaseFilter));
        });
    }
    const stringify = (str) => {
        return String(str).toLocaleLowerCase().replace(/-|_| /g, "");
    };

    let HacsStorePanel = class HacsStorePanel extends LitElement {
        constructor() {
            super(...arguments);
            this._searchInput = "";
            this.filters = {};
            this._repositoriesInActiveSection = memoizeOne((repositories, sections, section) => {
                const installedRepositories = repositories === null || repositories === void 0 ? void 0 : repositories.filter((repo) => {
                    var _a;
                    return ((_a = sections.panels
                        .find((panel) => panel.id === section)
                        .categories) === null || _a === void 0 ? void 0 : _a.includes(repo.category)) && repo.installed;
                });
                const newRepositories = repositories === null || repositories === void 0 ? void 0 : repositories.filter((repo) => {
                    var _a;
                    return ((_a = sections.panels
                        .find((panel) => panel.id === section)
                        .categories) === null || _a === void 0 ? void 0 : _a.includes(repo.category)) &&
                        repo.new &&
                        !repo.installed;
                });
                return [installedRepositories || [], newRepositories || []];
            });
            this._panelsEnabled = memoizeOne((sections, config) => {
                return sections.panels.filter((panel) => {
                    const categories = panel.categories;
                    if (categories === undefined)
                        return true;
                    return categories.filter((c) => config === null || config === void 0 ? void 0 : config.categories.includes(c)).length !== 0;
                });
            });
            this._filterRepositories = memoizeOne(filterRepositoriesByInput);
        }
        get allRepositories() {
            const [installedRepositories, newRepositories] = this._repositoriesInActiveSection(this.repositories, sections$f, this.section);
            return newRepositories.concat(installedRepositories);
        }
        get visibleRepositories() {
            const repositories = this.allRepositories.filter((repo) => this.filters[this.section].find((filter) => filter.id === repo.category).checked);
            return this._filterRepositories(repositories, this._searchInput);
        }
        async firstUpdated() {
            this.addEventListener("filter-change", (e) => this._updateFilters(e));
        }
        _updateFilters(e) {
            const current = this.filters[this.section].find((filter) => filter.id === e.detail.id);
            this.filters[this.section].find((filter) => filter.id === current.id).checked = !current.checked;
            this.requestUpdate();
        }
        render() {
            var _a;
            const newRepositories = this._repositoriesInActiveSection(this.repositories, sections$f, this.section)[1];
            if (!this.filters[this.section]) {
                const categories = (_a = activePanel(this.route)) === null || _a === void 0 ? void 0 : _a.categories;
                this.filters[this.section] = [];
                categories === null || categories === void 0 ? void 0 : categories.filter((c) => this.configuration.categories.includes(c)).forEach((category) => {
                    this.filters[this.section].push({
                        id: category,
                        value: category,
                        checked: true,
                    });
                });
            }
            const tabs = this._panelsEnabled(sections$f, this.configuration);
            return html `<hacs-tabbed-layout
      .hass=${this.hass}
      .tabs=${tabs}
      .route=${this.route}
      .narrow=${this.narrow}
      .selected=${this.section}
      ><hacs-tabbed-menu
        slot="toolbar-icon"
        .hass=${this.hass}
        .route=${this.route}
        .narrow=${this.narrow}
        .configuration=${this.configuration}
        .lovelace=${this.lovelace}
        .status=${this.status}
        .repositories=${this.repositories}
      >
      </hacs-tabbed-menu>

      <hacs-search .input=${this._searchInput} @input=${this._inputValueChanged}></hacs-search>
      ${this.filters[this.section].length > 1
            ? html `<hacs-filter class="filter" .filters="${this.filters[this.section]}"></hacs-filter>`
            : ""}
      ${(newRepositories === null || newRepositories === void 0 ? void 0 : newRepositories.length) > 10
            ? html `<div class="new-repositories">
            ${localize("store.new_repositories_note")}
          </div>`
            : ""}
      <div class="content">
        ${this.allRepositories.length === 0
            ? this._renderEmpty()
            : this.visibleRepositories.length === 0
                ? this._renderNoResultsFound()
                : this._renderRepositories()}
      </div>
      <hacs-fab
        .narrow=${this.narrow}
        @click=${this._addRepository}
        icon="mdi:plus"
        title="Add repository"
      >
      </hacs-fab>
    </hacs-tabbed-layout>`;
        }
        _renderRepositories() {
            return this.visibleRepositories.map((repo) => html `<hacs-repository-card
          .hass=${this.hass}
          .repository=${repo}
          .narrow=${this.narrow}
          .status=${this.status}
          .removed=${this.removed}
          .addedToLovelace=${addedToLovelace(this.lovelace, this.configuration, repo)}
        ></hacs-repository-card>`);
        }
        _renderNoResultsFound() {
            return html `<ha-card class="no-repositories">
      <div class="header">${localize("store.no_repositories")} 😕</div>
      <p>
        ${localize("store.no_repositories_found_desc1").replace("{searchInput}", this._searchInput)}
        <br />
        ${localize("store.no_repositories_found_desc2")}
      </p>
    </ha-card>`;
        }
        _renderEmpty() {
            return html `<ha-card class="no-repositories">
      <div class="header">${localize("store.no_repositories")} 😕</div>
      <p>
        ${localize("store.no_repositories_desc1")}<br />${localize("store.no_repositories_desc2")}
      </p>
    </ha-card>`;
        }
        _inputValueChanged(ev) {
            this._searchInput = ev.target.input;
        }
        _addRepository() {
            this.dispatchEvent(new CustomEvent("hacs-dialog", {
                detail: {
                    type: "add-repository",
                    repositories: this.repositories,
                    section: this.section,
                },
                bubbles: true,
                composed: true,
            }));
        }
        static get styles() {
            return [
                HacsStyles,
                css `
        hacs-repository-card {
          max-width: 500px;
          display: flex;
          flex-direction: column;
          justify-content: space-between;
        }
        .filter {
          border-bottom: 1px solid var(--divider-color);
        }
        .content {
          display: grid;
          grid-template-columns: repeat(auto-fit, minmax(360px, 1fr));
          grid-gap: 16px 16px;
          padding: 8px 16px 16px;
          margin-bottom: 64px;
        }
        .no-repositories {
          width: 100%;
          text-align: center;
          margin-top: 12px;
        }
        .new-repositories {
          margin: 4px 16px 0 16px;
        }
        paper-item {
          cursor: pointer;
        }
        ha-icon {
          color: var(--hcv-text-color-on-background);
        }
      `,
            ];
        }
    };
    __decorate([
        property()
    ], HacsStorePanel.prototype, "_searchInput", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsStorePanel.prototype, "configuration", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsStorePanel.prototype, "hass", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsStorePanel.prototype, "narrow", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsStorePanel.prototype, "route", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsStorePanel.prototype, "repositories", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsStorePanel.prototype, "lovelace", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsStorePanel.prototype, "status", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsStorePanel.prototype, "removed", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsStorePanel.prototype, "filters", void 0);
    __decorate([
        property()
    ], HacsStorePanel.prototype, "section", void 0);
    HacsStorePanel = __decorate([
        customElement("hacs-store-panel")
    ], HacsStorePanel);

    class HacsDialogBase extends LitElement {
        constructor() {
            super(...arguments);
            this.active = false;
            this.secondary = false;
            this.loading = true;
        }
        shouldUpdate(changedProperties) {
            changedProperties.forEach((_oldValue, propName) => {
                if (propName === "hass") {
                    this.sidebarDocked = window.localStorage.getItem("dockedSidebar") === '"docked"';
                }
            });
            return (changedProperties.has("sidebarDocked") ||
                changedProperties.has("narrow") ||
                changedProperties.has("active") ||
                changedProperties.has("params") ||
                changedProperties.has("_error") ||
                changedProperties.has("_releaseNotes") ||
                changedProperties.has("_updating"));
        }
        connectedCallback() {
            super.connectedCallback();
            this.sidebarDocked = window.localStorage.getItem("dockedSidebar") === '"docked"';
        }
    }
    __decorate([
        property({ attribute: false })
    ], HacsDialogBase.prototype, "configuration", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsDialogBase.prototype, "critical", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsDialogBase.prototype, "hass", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsDialogBase.prototype, "lovelace", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsDialogBase.prototype, "repositories", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsDialogBase.prototype, "route", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsDialogBase.prototype, "status", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsDialogBase.prototype, "removed", void 0);
    __decorate([
        property({ type: Boolean })
    ], HacsDialogBase.prototype, "active", void 0);
    __decorate([
        property({ type: Boolean })
    ], HacsDialogBase.prototype, "secondary", void 0);
    __decorate([
        property({ type: Boolean })
    ], HacsDialogBase.prototype, "loading", void 0);
    __decorate([
        property({ type: Boolean })
    ], HacsDialogBase.prototype, "narrow", void 0);
    __decorate([
        property({ type: Boolean })
    ], HacsDialogBase.prototype, "sidebarDocked", void 0);

    var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

    function createCommonjsModule(fn, module) {
    	return module = { exports: {} }, fn(module, module.exports), module.exports;
    }

    function getCjsExportFromNamespace (n) {
    	return n && n['default'] || n;
    }

    var defaults = createCommonjsModule(function (module) {
    function getDefaults() {
      return {
        baseUrl: null,
        breaks: false,
        gfm: true,
        headerIds: true,
        headerPrefix: '',
        highlight: null,
        langPrefix: 'language-',
        mangle: true,
        pedantic: false,
        renderer: null,
        sanitize: false,
        sanitizer: null,
        silent: false,
        smartLists: false,
        smartypants: false,
        tokenizer: null,
        walkTokens: null,
        xhtml: false
      };
    }

    function changeDefaults(newDefaults) {
      module.exports.defaults = newDefaults;
    }

    module.exports = {
      defaults: getDefaults(),
      getDefaults,
      changeDefaults
    };
    });
    var defaults_1 = defaults.defaults;
    var defaults_2 = defaults.getDefaults;
    var defaults_3 = defaults.changeDefaults;

    /**
     * Helpers
     */
    const escapeTest = /[&<>"']/;
    const escapeReplace = /[&<>"']/g;
    const escapeTestNoEncode = /[<>"']|&(?!#?\w+;)/;
    const escapeReplaceNoEncode = /[<>"']|&(?!#?\w+;)/g;
    const escapeReplacements = {
      '&': '&amp;',
      '<': '&lt;',
      '>': '&gt;',
      '"': '&quot;',
      "'": '&#39;'
    };
    const getEscapeReplacement = (ch) => escapeReplacements[ch];
    function escape(html, encode) {
      if (encode) {
        if (escapeTest.test(html)) {
          return html.replace(escapeReplace, getEscapeReplacement);
        }
      } else {
        if (escapeTestNoEncode.test(html)) {
          return html.replace(escapeReplaceNoEncode, getEscapeReplacement);
        }
      }

      return html;
    }

    const unescapeTest = /&(#(?:\d+)|(?:#x[0-9A-Fa-f]+)|(?:\w+));?/ig;

    function unescape(html) {
      // explicitly match decimal, hex, and named HTML entities
      return html.replace(unescapeTest, (_, n) => {
        n = n.toLowerCase();
        if (n === 'colon') return ':';
        if (n.charAt(0) === '#') {
          return n.charAt(1) === 'x'
            ? String.fromCharCode(parseInt(n.substring(2), 16))
            : String.fromCharCode(+n.substring(1));
        }
        return '';
      });
    }

    const caret = /(^|[^\[])\^/g;
    function edit(regex, opt) {
      regex = regex.source || regex;
      opt = opt || '';
      const obj = {
        replace: (name, val) => {
          val = val.source || val;
          val = val.replace(caret, '$1');
          regex = regex.replace(name, val);
          return obj;
        },
        getRegex: () => {
          return new RegExp(regex, opt);
        }
      };
      return obj;
    }

    const nonWordAndColonTest = /[^\w:]/g;
    const originIndependentUrl = /^$|^[a-z][a-z0-9+.-]*:|^[?#]/i;
    function cleanUrl(sanitize, base, href) {
      if (sanitize) {
        let prot;
        try {
          prot = decodeURIComponent(unescape(href))
            .replace(nonWordAndColonTest, '')
            .toLowerCase();
        } catch (e) {
          return null;
        }
        if (prot.indexOf('javascript:') === 0 || prot.indexOf('vbscript:') === 0 || prot.indexOf('data:') === 0) {
          return null;
        }
      }
      if (base && !originIndependentUrl.test(href)) {
        href = resolveUrl(base, href);
      }
      try {
        href = encodeURI(href).replace(/%25/g, '%');
      } catch (e) {
        return null;
      }
      return href;
    }

    const baseUrls = {};
    const justDomain = /^[^:]+:\/*[^/]*$/;
    const protocol = /^([^:]+:)[\s\S]*$/;
    const domain = /^([^:]+:\/*[^/]*)[\s\S]*$/;

    function resolveUrl(base, href) {
      if (!baseUrls[' ' + base]) {
        // we can ignore everything in base after the last slash of its path component,
        // but we might need to add _that_
        // https://tools.ietf.org/html/rfc3986#section-3
        if (justDomain.test(base)) {
          baseUrls[' ' + base] = base + '/';
        } else {
          baseUrls[' ' + base] = rtrim(base, '/', true);
        }
      }
      base = baseUrls[' ' + base];
      const relativeBase = base.indexOf(':') === -1;

      if (href.substring(0, 2) === '//') {
        if (relativeBase) {
          return href;
        }
        return base.replace(protocol, '$1') + href;
      } else if (href.charAt(0) === '/') {
        if (relativeBase) {
          return href;
        }
        return base.replace(domain, '$1') + href;
      } else {
        return base + href;
      }
    }

    const noopTest = { exec: function noopTest() {} };

    function merge(obj) {
      let i = 1,
        target,
        key;

      for (; i < arguments.length; i++) {
        target = arguments[i];
        for (key in target) {
          if (Object.prototype.hasOwnProperty.call(target, key)) {
            obj[key] = target[key];
          }
        }
      }

      return obj;
    }

    function splitCells(tableRow, count) {
      // ensure that every cell-delimiting pipe has a space
      // before it to distinguish it from an escaped pipe
      const row = tableRow.replace(/\|/g, (match, offset, str) => {
          let escaped = false,
            curr = offset;
          while (--curr >= 0 && str[curr] === '\\') escaped = !escaped;
          if (escaped) {
            // odd number of slashes means | is escaped
            // so we leave it alone
            return '|';
          } else {
            // add space before unescaped |
            return ' |';
          }
        }),
        cells = row.split(/ \|/);
      let i = 0;

      if (cells.length > count) {
        cells.splice(count);
      } else {
        while (cells.length < count) cells.push('');
      }

      for (; i < cells.length; i++) {
        // leading or trailing whitespace is ignored per the gfm spec
        cells[i] = cells[i].trim().replace(/\\\|/g, '|');
      }
      return cells;
    }

    // Remove trailing 'c's. Equivalent to str.replace(/c*$/, '').
    // /c*$/ is vulnerable to REDOS.
    // invert: Remove suffix of non-c chars instead. Default falsey.
    function rtrim(str, c, invert) {
      const l = str.length;
      if (l === 0) {
        return '';
      }

      // Length of suffix matching the invert condition.
      let suffLen = 0;

      // Step left until we fail to match the invert condition.
      while (suffLen < l) {
        const currChar = str.charAt(l - suffLen - 1);
        if (currChar === c && !invert) {
          suffLen++;
        } else if (currChar !== c && invert) {
          suffLen++;
        } else {
          break;
        }
      }

      return str.substr(0, l - suffLen);
    }

    function findClosingBracket(str, b) {
      if (str.indexOf(b[1]) === -1) {
        return -1;
      }
      const l = str.length;
      let level = 0,
        i = 0;
      for (; i < l; i++) {
        if (str[i] === '\\') {
          i++;
        } else if (str[i] === b[0]) {
          level++;
        } else if (str[i] === b[1]) {
          level--;
          if (level < 0) {
            return i;
          }
        }
      }
      return -1;
    }

    function checkSanitizeDeprecation(opt) {
      if (opt && opt.sanitize && !opt.silent) {
        console.warn('marked(): sanitize and sanitizer parameters are deprecated since version 0.7.0, should not be used and will be removed in the future. Read more here: https://marked.js.org/#/USING_ADVANCED.md#options');
      }
    }

    var helpers = {
      escape,
      unescape,
      edit,
      cleanUrl,
      resolveUrl,
      noopTest,
      merge,
      splitCells,
      rtrim,
      findClosingBracket,
      checkSanitizeDeprecation
    };

    const { defaults: defaults$1 } = defaults;
    const {
      rtrim: rtrim$1,
      splitCells: splitCells$1,
      escape: escape$1,
      findClosingBracket: findClosingBracket$1
    } = helpers;

    function outputLink(cap, link, raw) {
      const href = link.href;
      const title = link.title ? escape$1(link.title) : null;

      if (cap[0].charAt(0) !== '!') {
        return {
          type: 'link',
          raw,
          href,
          title,
          text: cap[1]
        };
      } else {
        return {
          type: 'image',
          raw,
          text: escape$1(cap[1]),
          href,
          title
        };
      }
    }

    function indentCodeCompensation(raw, text) {
      const matchIndentToCode = raw.match(/^(\s+)(?:```)/);

      if (matchIndentToCode === null) {
        return text;
      }

      const indentToCode = matchIndentToCode[1];

      return text
        .split('\n')
        .map(node => {
          const matchIndentInNode = node.match(/^\s+/);
          if (matchIndentInNode === null) {
            return node;
          }

          const [indentInNode] = matchIndentInNode;

          if (indentInNode.length >= indentToCode.length) {
            return node.slice(indentToCode.length);
          }

          return node;
        })
        .join('\n');
    }

    /**
     * Tokenizer
     */
    var Tokenizer_1 = class Tokenizer {
      constructor(options) {
        this.options = options || defaults$1;
      }

      space(src) {
        const cap = this.rules.block.newline.exec(src);
        if (cap) {
          if (cap[0].length > 1) {
            return {
              type: 'space',
              raw: cap[0]
            };
          }
          return { raw: '\n' };
        }
      }

      code(src, tokens) {
        const cap = this.rules.block.code.exec(src);
        if (cap) {
          const lastToken = tokens[tokens.length - 1];
          // An indented code block cannot interrupt a paragraph.
          if (lastToken && lastToken.type === 'paragraph') {
            return {
              raw: cap[0],
              text: cap[0].trimRight()
            };
          }

          const text = cap[0].replace(/^ {4}/gm, '');
          return {
            type: 'code',
            raw: cap[0],
            codeBlockStyle: 'indented',
            text: !this.options.pedantic
              ? rtrim$1(text, '\n')
              : text
          };
        }
      }

      fences(src) {
        const cap = this.rules.block.fences.exec(src);
        if (cap) {
          const raw = cap[0];
          const text = indentCodeCompensation(raw, cap[3] || '');

          return {
            type: 'code',
            raw,
            lang: cap[2] ? cap[2].trim() : cap[2],
            text
          };
        }
      }

      heading(src) {
        const cap = this.rules.block.heading.exec(src);
        if (cap) {
          return {
            type: 'heading',
            raw: cap[0],
            depth: cap[1].length,
            text: cap[2]
          };
        }
      }

      nptable(src) {
        const cap = this.rules.block.nptable.exec(src);
        if (cap) {
          const item = {
            type: 'table',
            header: splitCells$1(cap[1].replace(/^ *| *\| *$/g, '')),
            align: cap[2].replace(/^ *|\| *$/g, '').split(/ *\| */),
            cells: cap[3] ? cap[3].replace(/\n$/, '').split('\n') : [],
            raw: cap[0]
          };

          if (item.header.length === item.align.length) {
            let l = item.align.length;
            let i;
            for (i = 0; i < l; i++) {
              if (/^ *-+: *$/.test(item.align[i])) {
                item.align[i] = 'right';
              } else if (/^ *:-+: *$/.test(item.align[i])) {
                item.align[i] = 'center';
              } else if (/^ *:-+ *$/.test(item.align[i])) {
                item.align[i] = 'left';
              } else {
                item.align[i] = null;
              }
            }

            l = item.cells.length;
            for (i = 0; i < l; i++) {
              item.cells[i] = splitCells$1(item.cells[i], item.header.length);
            }

            return item;
          }
        }
      }

      hr(src) {
        const cap = this.rules.block.hr.exec(src);
        if (cap) {
          return {
            type: 'hr',
            raw: cap[0]
          };
        }
      }

      blockquote(src) {
        const cap = this.rules.block.blockquote.exec(src);
        if (cap) {
          const text = cap[0].replace(/^ *> ?/gm, '');

          return {
            type: 'blockquote',
            raw: cap[0],
            text
          };
        }
      }

      list(src) {
        const cap = this.rules.block.list.exec(src);
        if (cap) {
          let raw = cap[0];
          const bull = cap[2];
          const isordered = bull.length > 1;

          const list = {
            type: 'list',
            raw,
            ordered: isordered,
            start: isordered ? +bull : '',
            loose: false,
            items: []
          };

          // Get each top-level item.
          const itemMatch = cap[0].match(this.rules.block.item);

          let next = false,
            item,
            space,
            b,
            addBack,
            loose,
            istask,
            ischecked;

          const l = itemMatch.length;
          for (let i = 0; i < l; i++) {
            item = itemMatch[i];
            raw = item;

            // Remove the list item's bullet
            // so it is seen as the next token.
            space = item.length;
            item = item.replace(/^ *([*+-]|\d+\.) */, '');

            // Outdent whatever the
            // list item contains. Hacky.
            if (~item.indexOf('\n ')) {
              space -= item.length;
              item = !this.options.pedantic
                ? item.replace(new RegExp('^ {1,' + space + '}', 'gm'), '')
                : item.replace(/^ {1,4}/gm, '');
            }

            // Determine whether the next list item belongs here.
            // Backpedal if it does not belong in this list.
            if (i !== l - 1) {
              b = this.rules.block.bullet.exec(itemMatch[i + 1])[0];
              if (bull.length > 1 ? b.length === 1
                : (b.length > 1 || (this.options.smartLists && b !== bull))) {
                addBack = itemMatch.slice(i + 1).join('\n');
                list.raw = list.raw.substring(0, list.raw.length - addBack.length);
                i = l - 1;
              }
            }

            // Determine whether item is loose or not.
            // Use: /(^|\n)(?! )[^\n]+\n\n(?!\s*$)/
            // for discount behavior.
            loose = next || /\n\n(?!\s*$)/.test(item);
            if (i !== l - 1) {
              next = item.charAt(item.length - 1) === '\n';
              if (!loose) loose = next;
            }

            if (loose) {
              list.loose = true;
            }

            // Check for task list items
            istask = /^\[[ xX]\] /.test(item);
            ischecked = undefined;
            if (istask) {
              ischecked = item[1] !== ' ';
              item = item.replace(/^\[[ xX]\] +/, '');
            }

            list.items.push({
              type: 'list_item',
              raw,
              task: istask,
              checked: ischecked,
              loose: loose,
              text: item
            });
          }

          return list;
        }
      }

      html(src) {
        const cap = this.rules.block.html.exec(src);
        if (cap) {
          return {
            type: this.options.sanitize
              ? 'paragraph'
              : 'html',
            raw: cap[0],
            pre: !this.options.sanitizer
              && (cap[1] === 'pre' || cap[1] === 'script' || cap[1] === 'style'),
            text: this.options.sanitize ? (this.options.sanitizer ? this.options.sanitizer(cap[0]) : escape$1(cap[0])) : cap[0]
          };
        }
      }

      def(src) {
        const cap = this.rules.block.def.exec(src);
        if (cap) {
          if (cap[3]) cap[3] = cap[3].substring(1, cap[3].length - 1);
          const tag = cap[1].toLowerCase().replace(/\s+/g, ' ');
          return {
            tag,
            raw: cap[0],
            href: cap[2],
            title: cap[3]
          };
        }
      }

      table(src) {
        const cap = this.rules.block.table.exec(src);
        if (cap) {
          const item = {
            type: 'table',
            header: splitCells$1(cap[1].replace(/^ *| *\| *$/g, '')),
            align: cap[2].replace(/^ *|\| *$/g, '').split(/ *\| */),
            cells: cap[3] ? cap[3].replace(/\n$/, '').split('\n') : []
          };

          if (item.header.length === item.align.length) {
            item.raw = cap[0];

            let l = item.align.length;
            let i;
            for (i = 0; i < l; i++) {
              if (/^ *-+: *$/.test(item.align[i])) {
                item.align[i] = 'right';
              } else if (/^ *:-+: *$/.test(item.align[i])) {
                item.align[i] = 'center';
              } else if (/^ *:-+ *$/.test(item.align[i])) {
                item.align[i] = 'left';
              } else {
                item.align[i] = null;
              }
            }

            l = item.cells.length;
            for (i = 0; i < l; i++) {
              item.cells[i] = splitCells$1(
                item.cells[i].replace(/^ *\| *| *\| *$/g, ''),
                item.header.length);
            }

            return item;
          }
        }
      }

      lheading(src) {
        const cap = this.rules.block.lheading.exec(src);
        if (cap) {
          return {
            type: 'heading',
            raw: cap[0],
            depth: cap[2].charAt(0) === '=' ? 1 : 2,
            text: cap[1]
          };
        }
      }

      paragraph(src) {
        const cap = this.rules.block.paragraph.exec(src);
        if (cap) {
          return {
            type: 'paragraph',
            raw: cap[0],
            text: cap[1].charAt(cap[1].length - 1) === '\n'
              ? cap[1].slice(0, -1)
              : cap[1]
          };
        }
      }

      text(src, tokens) {
        const cap = this.rules.block.text.exec(src);
        if (cap) {
          const lastToken = tokens[tokens.length - 1];
          if (lastToken && lastToken.type === 'text') {
            return {
              raw: cap[0],
              text: cap[0]
            };
          }

          return {
            type: 'text',
            raw: cap[0],
            text: cap[0]
          };
        }
      }

      escape(src) {
        const cap = this.rules.inline.escape.exec(src);
        if (cap) {
          return {
            type: 'escape',
            raw: cap[0],
            text: escape$1(cap[1])
          };
        }
      }

      tag(src, inLink, inRawBlock) {
        const cap = this.rules.inline.tag.exec(src);
        if (cap) {
          if (!inLink && /^<a /i.test(cap[0])) {
            inLink = true;
          } else if (inLink && /^<\/a>/i.test(cap[0])) {
            inLink = false;
          }
          if (!inRawBlock && /^<(pre|code|kbd|script)(\s|>)/i.test(cap[0])) {
            inRawBlock = true;
          } else if (inRawBlock && /^<\/(pre|code|kbd|script)(\s|>)/i.test(cap[0])) {
            inRawBlock = false;
          }

          return {
            type: this.options.sanitize
              ? 'text'
              : 'html',
            raw: cap[0],
            inLink,
            inRawBlock,
            text: this.options.sanitize
              ? (this.options.sanitizer
                ? this.options.sanitizer(cap[0])
                : escape$1(cap[0]))
              : cap[0]
          };
        }
      }

      link(src) {
        const cap = this.rules.inline.link.exec(src);
        if (cap) {
          const lastParenIndex = findClosingBracket$1(cap[2], '()');
          if (lastParenIndex > -1) {
            const start = cap[0].indexOf('!') === 0 ? 5 : 4;
            const linkLen = start + cap[1].length + lastParenIndex;
            cap[2] = cap[2].substring(0, lastParenIndex);
            cap[0] = cap[0].substring(0, linkLen).trim();
            cap[3] = '';
          }
          let href = cap[2];
          let title = '';
          if (this.options.pedantic) {
            const link = /^([^'"]*[^\s])\s+(['"])(.*)\2/.exec(href);

            if (link) {
              href = link[1];
              title = link[3];
            } else {
              title = '';
            }
          } else {
            title = cap[3] ? cap[3].slice(1, -1) : '';
          }
          href = href.trim().replace(/^<([\s\S]*)>$/, '$1');
          const token = outputLink(cap, {
            href: href ? href.replace(this.rules.inline._escapes, '$1') : href,
            title: title ? title.replace(this.rules.inline._escapes, '$1') : title
          }, cap[0]);
          return token;
        }
      }

      reflink(src, links) {
        let cap;
        if ((cap = this.rules.inline.reflink.exec(src))
            || (cap = this.rules.inline.nolink.exec(src))) {
          let link = (cap[2] || cap[1]).replace(/\s+/g, ' ');
          link = links[link.toLowerCase()];
          if (!link || !link.href) {
            const text = cap[0].charAt(0);
            return {
              type: 'text',
              raw: text,
              text
            };
          }
          const token = outputLink(cap, link, cap[0]);
          return token;
        }
      }

      strong(src) {
        const cap = this.rules.inline.strong.exec(src);
        if (cap) {
          return {
            type: 'strong',
            raw: cap[0],
            text: cap[4] || cap[3] || cap[2] || cap[1]
          };
        }
      }

      em(src) {
        const cap = this.rules.inline.em.exec(src);
        if (cap) {
          return {
            type: 'em',
            raw: cap[0],
            text: cap[6] || cap[5] || cap[4] || cap[3] || cap[2] || cap[1]
          };
        }
      }

      codespan(src) {
        const cap = this.rules.inline.code.exec(src);
        if (cap) {
          let text = cap[2].replace(/\n/g, ' ');
          const hasNonSpaceChars = /[^ ]/.test(text);
          const hasSpaceCharsOnBothEnds = text.startsWith(' ') && text.endsWith(' ');
          if (hasNonSpaceChars && hasSpaceCharsOnBothEnds) {
            text = text.substring(1, text.length - 1);
          }
          text = escape$1(text, true);
          return {
            type: 'codespan',
            raw: cap[0],
            text
          };
        }
      }

      br(src) {
        const cap = this.rules.inline.br.exec(src);
        if (cap) {
          return {
            type: 'br',
            raw: cap[0]
          };
        }
      }

      del(src) {
        const cap = this.rules.inline.del.exec(src);
        if (cap) {
          return {
            type: 'del',
            raw: cap[0],
            text: cap[1]
          };
        }
      }

      autolink(src, mangle) {
        const cap = this.rules.inline.autolink.exec(src);
        if (cap) {
          let text, href;
          if (cap[2] === '@') {
            text = escape$1(this.options.mangle ? mangle(cap[1]) : cap[1]);
            href = 'mailto:' + text;
          } else {
            text = escape$1(cap[1]);
            href = text;
          }

          return {
            type: 'link',
            raw: cap[0],
            text,
            href,
            tokens: [
              {
                type: 'text',
                raw: text,
                text
              }
            ]
          };
        }
      }

      url(src, mangle) {
        let cap;
        if (cap = this.rules.inline.url.exec(src)) {
          let text, href;
          if (cap[2] === '@') {
            text = escape$1(this.options.mangle ? mangle(cap[0]) : cap[0]);
            href = 'mailto:' + text;
          } else {
            // do extended autolink path validation
            let prevCapZero;
            do {
              prevCapZero = cap[0];
              cap[0] = this.rules.inline._backpedal.exec(cap[0])[0];
            } while (prevCapZero !== cap[0]);
            text = escape$1(cap[0]);
            if (cap[1] === 'www.') {
              href = 'http://' + text;
            } else {
              href = text;
            }
          }
          return {
            type: 'link',
            raw: cap[0],
            text,
            href,
            tokens: [
              {
                type: 'text',
                raw: text,
                text
              }
            ]
          };
        }
      }

      inlineText(src, inRawBlock, smartypants) {
        const cap = this.rules.inline.text.exec(src);
        if (cap) {
          let text;
          if (inRawBlock) {
            text = this.options.sanitize ? (this.options.sanitizer ? this.options.sanitizer(cap[0]) : escape$1(cap[0])) : cap[0];
          } else {
            text = escape$1(this.options.smartypants ? smartypants(cap[0]) : cap[0]);
          }
          return {
            type: 'text',
            raw: cap[0],
            text
          };
        }
      }
    };

    const {
      noopTest: noopTest$1,
      edit: edit$1,
      merge: merge$1
    } = helpers;

    /**
     * Block-Level Grammar
     */
    const block = {
      newline: /^\n+/,
      code: /^( {4}[^\n]+\n*)+/,
      fences: /^ {0,3}(`{3,}(?=[^`\n]*\n)|~{3,})([^\n]*)\n(?:|([\s\S]*?)\n)(?: {0,3}\1[~`]* *(?:\n+|$)|$)/,
      hr: /^ {0,3}((?:- *){3,}|(?:_ *){3,}|(?:\* *){3,})(?:\n+|$)/,
      heading: /^ {0,3}(#{1,6}) +([^\n]*?)(?: +#+)? *(?:\n+|$)/,
      blockquote: /^( {0,3}> ?(paragraph|[^\n]*)(?:\n|$))+/,
      list: /^( {0,3})(bull) [\s\S]+?(?:hr|def|\n{2,}(?! )(?!\1bull )\n*|\s*$)/,
      html: '^ {0,3}(?:' // optional indentation
        + '<(script|pre|style)[\\s>][\\s\\S]*?(?:</\\1>[^\\n]*\\n+|$)' // (1)
        + '|comment[^\\n]*(\\n+|$)' // (2)
        + '|<\\?[\\s\\S]*?\\?>\\n*' // (3)
        + '|<![A-Z][\\s\\S]*?>\\n*' // (4)
        + '|<!\\[CDATA\\[[\\s\\S]*?\\]\\]>\\n*' // (5)
        + '|</?(tag)(?: +|\\n|/?>)[\\s\\S]*?(?:\\n{2,}|$)' // (6)
        + '|<(?!script|pre|style)([a-z][\\w-]*)(?:attribute)*? */?>(?=[ \\t]*(?:\\n|$))[\\s\\S]*?(?:\\n{2,}|$)' // (7) open tag
        + '|</(?!script|pre|style)[a-z][\\w-]*\\s*>(?=[ \\t]*(?:\\n|$))[\\s\\S]*?(?:\\n{2,}|$)' // (7) closing tag
        + ')',
      def: /^ {0,3}\[(label)\]: *\n? *<?([^\s>]+)>?(?:(?: +\n? *| *\n *)(title))? *(?:\n+|$)/,
      nptable: noopTest$1,
      table: noopTest$1,
      lheading: /^([^\n]+)\n {0,3}(=+|-+) *(?:\n+|$)/,
      // regex template, placeholders will be replaced according to different paragraph
      // interruption rules of commonmark and the original markdown spec:
      _paragraph: /^([^\n]+(?:\n(?!hr|heading|lheading|blockquote|fences|list|html)[^\n]+)*)/,
      text: /^[^\n]+/
    };

    block._label = /(?!\s*\])(?:\\[\[\]]|[^\[\]])+/;
    block._title = /(?:"(?:\\"?|[^"\\])*"|'[^'\n]*(?:\n[^'\n]+)*\n?'|\([^()]*\))/;
    block.def = edit$1(block.def)
      .replace('label', block._label)
      .replace('title', block._title)
      .getRegex();

    block.bullet = /(?:[*+-]|\d{1,9}\.)/;
    block.item = /^( *)(bull) ?[^\n]*(?:\n(?!\1bull ?)[^\n]*)*/;
    block.item = edit$1(block.item, 'gm')
      .replace(/bull/g, block.bullet)
      .getRegex();

    block.list = edit$1(block.list)
      .replace(/bull/g, block.bullet)
      .replace('hr', '\\n+(?=\\1?(?:(?:- *){3,}|(?:_ *){3,}|(?:\\* *){3,})(?:\\n+|$))')
      .replace('def', '\\n+(?=' + block.def.source + ')')
      .getRegex();

    block._tag = 'address|article|aside|base|basefont|blockquote|body|caption'
      + '|center|col|colgroup|dd|details|dialog|dir|div|dl|dt|fieldset|figcaption'
      + '|figure|footer|form|frame|frameset|h[1-6]|head|header|hr|html|iframe'
      + '|legend|li|link|main|menu|menuitem|meta|nav|noframes|ol|optgroup|option'
      + '|p|param|section|source|summary|table|tbody|td|tfoot|th|thead|title|tr'
      + '|track|ul';
    block._comment = /<!--(?!-?>)[\s\S]*?-->/;
    block.html = edit$1(block.html, 'i')
      .replace('comment', block._comment)
      .replace('tag', block._tag)
      .replace('attribute', / +[a-zA-Z:_][\w.:-]*(?: *= *"[^"\n]*"| *= *'[^'\n]*'| *= *[^\s"'=<>`]+)?/)
      .getRegex();

    block.paragraph = edit$1(block._paragraph)
      .replace('hr', block.hr)
      .replace('heading', ' {0,3}#{1,6} ')
      .replace('|lheading', '') // setex headings don't interrupt commonmark paragraphs
      .replace('blockquote', ' {0,3}>')
      .replace('fences', ' {0,3}(?:`{3,}(?=[^`\\n]*\\n)|~{3,})[^\\n]*\\n')
      .replace('list', ' {0,3}(?:[*+-]|1[.)]) ') // only lists starting from 1 can interrupt
      .replace('html', '</?(?:tag)(?: +|\\n|/?>)|<(?:script|pre|style|!--)')
      .replace('tag', block._tag) // pars can be interrupted by type (6) html blocks
      .getRegex();

    block.blockquote = edit$1(block.blockquote)
      .replace('paragraph', block.paragraph)
      .getRegex();

    /**
     * Normal Block Grammar
     */

    block.normal = merge$1({}, block);

    /**
     * GFM Block Grammar
     */

    block.gfm = merge$1({}, block.normal, {
      nptable: '^ *([^|\\n ].*\\|.*)\\n' // Header
        + ' *([-:]+ *\\|[-| :]*)' // Align
        + '(?:\\n((?:(?!\\n|hr|heading|blockquote|code|fences|list|html).*(?:\\n|$))*)\\n*|$)', // Cells
      table: '^ *\\|(.+)\\n' // Header
        + ' *\\|?( *[-:]+[-| :]*)' // Align
        + '(?:\\n *((?:(?!\\n|hr|heading|blockquote|code|fences|list|html).*(?:\\n|$))*)\\n*|$)' // Cells
    });

    block.gfm.nptable = edit$1(block.gfm.nptable)
      .replace('hr', block.hr)
      .replace('heading', ' {0,3}#{1,6} ')
      .replace('blockquote', ' {0,3}>')
      .replace('code', ' {4}[^\\n]')
      .replace('fences', ' {0,3}(?:`{3,}(?=[^`\\n]*\\n)|~{3,})[^\\n]*\\n')
      .replace('list', ' {0,3}(?:[*+-]|1[.)]) ') // only lists starting from 1 can interrupt
      .replace('html', '</?(?:tag)(?: +|\\n|/?>)|<(?:script|pre|style|!--)')
      .replace('tag', block._tag) // tables can be interrupted by type (6) html blocks
      .getRegex();

    block.gfm.table = edit$1(block.gfm.table)
      .replace('hr', block.hr)
      .replace('heading', ' {0,3}#{1,6} ')
      .replace('blockquote', ' {0,3}>')
      .replace('code', ' {4}[^\\n]')
      .replace('fences', ' {0,3}(?:`{3,}(?=[^`\\n]*\\n)|~{3,})[^\\n]*\\n')
      .replace('list', ' {0,3}(?:[*+-]|1[.)]) ') // only lists starting from 1 can interrupt
      .replace('html', '</?(?:tag)(?: +|\\n|/?>)|<(?:script|pre|style|!--)')
      .replace('tag', block._tag) // tables can be interrupted by type (6) html blocks
      .getRegex();

    /**
     * Pedantic grammar (original John Gruber's loose markdown specification)
     */

    block.pedantic = merge$1({}, block.normal, {
      html: edit$1(
        '^ *(?:comment *(?:\\n|\\s*$)'
        + '|<(tag)[\\s\\S]+?</\\1> *(?:\\n{2,}|\\s*$)' // closed tag
        + '|<tag(?:"[^"]*"|\'[^\']*\'|\\s[^\'"/>\\s]*)*?/?> *(?:\\n{2,}|\\s*$))')
        .replace('comment', block._comment)
        .replace(/tag/g, '(?!(?:'
          + 'a|em|strong|small|s|cite|q|dfn|abbr|data|time|code|var|samp|kbd|sub'
          + '|sup|i|b|u|mark|ruby|rt|rp|bdi|bdo|span|br|wbr|ins|del|img)'
          + '\\b)\\w+(?!:|[^\\w\\s@]*@)\\b')
        .getRegex(),
      def: /^ *\[([^\]]+)\]: *<?([^\s>]+)>?(?: +(["(][^\n]+[")]))? *(?:\n+|$)/,
      heading: /^ *(#{1,6}) *([^\n]+?) *(?:#+ *)?(?:\n+|$)/,
      fences: noopTest$1, // fences not supported
      paragraph: edit$1(block.normal._paragraph)
        .replace('hr', block.hr)
        .replace('heading', ' *#{1,6} *[^\n]')
        .replace('lheading', block.lheading)
        .replace('blockquote', ' {0,3}>')
        .replace('|fences', '')
        .replace('|list', '')
        .replace('|html', '')
        .getRegex()
    });

    /**
     * Inline-Level Grammar
     */
    const inline = {
      escape: /^\\([!"#$%&'()*+,\-./:;<=>?@\[\]\\^_`{|}~])/,
      autolink: /^<(scheme:[^\s\x00-\x1f<>]*|email)>/,
      url: noopTest$1,
      tag: '^comment'
        + '|^</[a-zA-Z][\\w:-]*\\s*>' // self-closing tag
        + '|^<[a-zA-Z][\\w-]*(?:attribute)*?\\s*/?>' // open tag
        + '|^<\\?[\\s\\S]*?\\?>' // processing instruction, e.g. <?php ?>
        + '|^<![a-zA-Z]+\\s[\\s\\S]*?>' // declaration, e.g. <!DOCTYPE html>
        + '|^<!\\[CDATA\\[[\\s\\S]*?\\]\\]>', // CDATA section
      link: /^!?\[(label)\]\(\s*(href)(?:\s+(title))?\s*\)/,
      reflink: /^!?\[(label)\]\[(?!\s*\])((?:\\[\[\]]?|[^\[\]\\])+)\]/,
      nolink: /^!?\[(?!\s*\])((?:\[[^\[\]]*\]|\\[\[\]]|[^\[\]])*)\](?:\[\])?/,
      strong: /^__([^\s_])__(?!_)|^\*\*([^\s*])\*\*(?!\*)|^__([^\s][\s\S]*?[^\s])__(?!_)|^\*\*([^\s][\s\S]*?[^\s])\*\*(?!\*)/,
      em: /^_([^\s_])_(?!_)|^_([^\s_<][\s\S]*?[^\s_])_(?!_|[^\s,punctuation])|^_([^\s_<][\s\S]*?[^\s])_(?!_|[^\s,punctuation])|^\*([^\s*<\[])\*(?!\*)|^\*([^\s<"][\s\S]*?[^\s\[\*])\*(?![\]`punctuation])|^\*([^\s*"<\[][\s\S]*[^\s])\*(?!\*)/,
      code: /^(`+)([^`]|[^`][\s\S]*?[^`])\1(?!`)/,
      br: /^( {2,}|\\)\n(?!\s*$)/,
      del: noopTest$1,
      text: /^(`+|[^`])(?:[\s\S]*?(?:(?=[\\<!\[`*]|\b_|$)|[^ ](?= {2,}\n))|(?= {2,}\n))/
    };

    // list of punctuation marks from common mark spec
    // without ` and ] to workaround Rule 17 (inline code blocks/links)
    // without , to work around example 393
    inline._punctuation = '!"#$%&\'()*+\\-./:;<=>?@\\[^_{|}~';
    inline.em = edit$1(inline.em).replace(/punctuation/g, inline._punctuation).getRegex();

    inline._escapes = /\\([!"#$%&'()*+,\-./:;<=>?@\[\]\\^_`{|}~])/g;

    inline._scheme = /[a-zA-Z][a-zA-Z0-9+.-]{1,31}/;
    inline._email = /[a-zA-Z0-9.!#$%&'*+/=?^_`{|}~-]+(@)[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(?:\.[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)+(?![-_])/;
    inline.autolink = edit$1(inline.autolink)
      .replace('scheme', inline._scheme)
      .replace('email', inline._email)
      .getRegex();

    inline._attribute = /\s+[a-zA-Z:_][\w.:-]*(?:\s*=\s*"[^"]*"|\s*=\s*'[^']*'|\s*=\s*[^\s"'=<>`]+)?/;

    inline.tag = edit$1(inline.tag)
      .replace('comment', block._comment)
      .replace('attribute', inline._attribute)
      .getRegex();

    inline._label = /(?:\[[^\[\]]*\]|\\.|`[^`]*`|[^\[\]\\`])*?/;
    inline._href = /<(?:\\[<>]?|[^\s<>\\])*>|[^\s\x00-\x1f]*/;
    inline._title = /"(?:\\"?|[^"\\])*"|'(?:\\'?|[^'\\])*'|\((?:\\\)?|[^)\\])*\)/;

    inline.link = edit$1(inline.link)
      .replace('label', inline._label)
      .replace('href', inline._href)
      .replace('title', inline._title)
      .getRegex();

    inline.reflink = edit$1(inline.reflink)
      .replace('label', inline._label)
      .getRegex();

    /**
     * Normal Inline Grammar
     */

    inline.normal = merge$1({}, inline);

    /**
     * Pedantic Inline Grammar
     */

    inline.pedantic = merge$1({}, inline.normal, {
      strong: /^__(?=\S)([\s\S]*?\S)__(?!_)|^\*\*(?=\S)([\s\S]*?\S)\*\*(?!\*)/,
      em: /^_(?=\S)([\s\S]*?\S)_(?!_)|^\*(?=\S)([\s\S]*?\S)\*(?!\*)/,
      link: edit$1(/^!?\[(label)\]\((.*?)\)/)
        .replace('label', inline._label)
        .getRegex(),
      reflink: edit$1(/^!?\[(label)\]\s*\[([^\]]*)\]/)
        .replace('label', inline._label)
        .getRegex()
    });

    /**
     * GFM Inline Grammar
     */

    inline.gfm = merge$1({}, inline.normal, {
      escape: edit$1(inline.escape).replace('])', '~|])').getRegex(),
      _extended_email: /[A-Za-z0-9._+-]+(@)[a-zA-Z0-9-_]+(?:\.[a-zA-Z0-9-_]*[a-zA-Z0-9])+(?![-_])/,
      url: /^((?:ftp|https?):\/\/|www\.)(?:[a-zA-Z0-9\-]+\.?)+[^\s<]*|^email/,
      _backpedal: /(?:[^?!.,:;*_~()&]+|\([^)]*\)|&(?![a-zA-Z0-9]+;$)|[?!.,:;*_~)]+(?!$))+/,
      del: /^~+(?=\S)([\s\S]*?\S)~+/,
      text: /^(`+|[^`])(?:[\s\S]*?(?:(?=[\\<!\[`*~]|\b_|https?:\/\/|ftp:\/\/|www\.|$)|[^ ](?= {2,}\n)|[^a-zA-Z0-9.!#$%&'*+\/=?_`{\|}~-](?=[a-zA-Z0-9.!#$%&'*+\/=?_`{\|}~-]+@))|(?= {2,}\n|[a-zA-Z0-9.!#$%&'*+\/=?_`{\|}~-]+@))/
    });

    inline.gfm.url = edit$1(inline.gfm.url, 'i')
      .replace('email', inline.gfm._extended_email)
      .getRegex();
    /**
     * GFM + Line Breaks Inline Grammar
     */

    inline.breaks = merge$1({}, inline.gfm, {
      br: edit$1(inline.br).replace('{2,}', '*').getRegex(),
      text: edit$1(inline.gfm.text)
        .replace('\\b_', '\\b_| {2,}\\n')
        .replace(/\{2,\}/g, '*')
        .getRegex()
    });

    var rules = {
      block,
      inline
    };

    const { defaults: defaults$2 } = defaults;
    const { block: block$1, inline: inline$1 } = rules;

    /**
     * smartypants text replacement
     */
    function smartypants(text) {
      return text
        // em-dashes
        .replace(/---/g, '\u2014')
        // en-dashes
        .replace(/--/g, '\u2013')
        // opening singles
        .replace(/(^|[-\u2014/(\[{"\s])'/g, '$1\u2018')
        // closing singles & apostrophes
        .replace(/'/g, '\u2019')
        // opening doubles
        .replace(/(^|[-\u2014/(\[{\u2018\s])"/g, '$1\u201c')
        // closing doubles
        .replace(/"/g, '\u201d')
        // ellipses
        .replace(/\.{3}/g, '\u2026');
    }

    /**
     * mangle email addresses
     */
    function mangle(text) {
      let out = '',
        i,
        ch;

      const l = text.length;
      for (i = 0; i < l; i++) {
        ch = text.charCodeAt(i);
        if (Math.random() > 0.5) {
          ch = 'x' + ch.toString(16);
        }
        out += '&#' + ch + ';';
      }

      return out;
    }

    /**
     * Block Lexer
     */
    var Lexer_1 = class Lexer {
      constructor(options) {
        this.tokens = [];
        this.tokens.links = Object.create(null);
        this.options = options || defaults$2;
        this.options.tokenizer = this.options.tokenizer || new Tokenizer_1();
        this.tokenizer = this.options.tokenizer;
        this.tokenizer.options = this.options;

        const rules = {
          block: block$1.normal,
          inline: inline$1.normal
        };

        if (this.options.pedantic) {
          rules.block = block$1.pedantic;
          rules.inline = inline$1.pedantic;
        } else if (this.options.gfm) {
          rules.block = block$1.gfm;
          if (this.options.breaks) {
            rules.inline = inline$1.breaks;
          } else {
            rules.inline = inline$1.gfm;
          }
        }
        this.tokenizer.rules = rules;
      }

      /**
       * Expose Rules
       */
      static get rules() {
        return {
          block: block$1,
          inline: inline$1
        };
      }

      /**
       * Static Lex Method
       */
      static lex(src, options) {
        const lexer = new Lexer(options);
        return lexer.lex(src);
      }

      /**
       * Preprocessing
       */
      lex(src) {
        src = src
          .replace(/\r\n|\r/g, '\n')
          .replace(/\t/g, '    ');

        this.blockTokens(src, this.tokens, true);

        this.inline(this.tokens);

        return this.tokens;
      }

      /**
       * Lexing
       */
      blockTokens(src, tokens = [], top = true) {
        src = src.replace(/^ +$/gm, '');
        let token, i, l, lastToken;

        while (src) {
          // newline
          if (token = this.tokenizer.space(src)) {
            src = src.substring(token.raw.length);
            if (token.type) {
              tokens.push(token);
            }
            continue;
          }

          // code
          if (token = this.tokenizer.code(src, tokens)) {
            src = src.substring(token.raw.length);
            if (token.type) {
              tokens.push(token);
            } else {
              lastToken = tokens[tokens.length - 1];
              lastToken.raw += '\n' + token.raw;
              lastToken.text += '\n' + token.text;
            }
            continue;
          }

          // fences
          if (token = this.tokenizer.fences(src)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // heading
          if (token = this.tokenizer.heading(src)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // table no leading pipe (gfm)
          if (token = this.tokenizer.nptable(src)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // hr
          if (token = this.tokenizer.hr(src)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // blockquote
          if (token = this.tokenizer.blockquote(src)) {
            src = src.substring(token.raw.length);
            token.tokens = this.blockTokens(token.text, [], top);
            tokens.push(token);
            continue;
          }

          // list
          if (token = this.tokenizer.list(src)) {
            src = src.substring(token.raw.length);
            l = token.items.length;
            for (i = 0; i < l; i++) {
              token.items[i].tokens = this.blockTokens(token.items[i].text, [], false);
            }
            tokens.push(token);
            continue;
          }

          // html
          if (token = this.tokenizer.html(src)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // def
          if (top && (token = this.tokenizer.def(src))) {
            src = src.substring(token.raw.length);
            if (!this.tokens.links[token.tag]) {
              this.tokens.links[token.tag] = {
                href: token.href,
                title: token.title
              };
            }
            continue;
          }

          // table (gfm)
          if (token = this.tokenizer.table(src)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // lheading
          if (token = this.tokenizer.lheading(src)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // top-level paragraph
          if (top && (token = this.tokenizer.paragraph(src))) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // text
          if (token = this.tokenizer.text(src, tokens)) {
            src = src.substring(token.raw.length);
            if (token.type) {
              tokens.push(token);
            } else {
              lastToken = tokens[tokens.length - 1];
              lastToken.raw += '\n' + token.raw;
              lastToken.text += '\n' + token.text;
            }
            continue;
          }

          if (src) {
            const errMsg = 'Infinite loop on byte: ' + src.charCodeAt(0);
            if (this.options.silent) {
              console.error(errMsg);
              break;
            } else {
              throw new Error(errMsg);
            }
          }
        }

        return tokens;
      }

      inline(tokens) {
        let i,
          j,
          k,
          l2,
          row,
          token;

        const l = tokens.length;
        for (i = 0; i < l; i++) {
          token = tokens[i];
          switch (token.type) {
            case 'paragraph':
            case 'text':
            case 'heading': {
              token.tokens = [];
              this.inlineTokens(token.text, token.tokens);
              break;
            }
            case 'table': {
              token.tokens = {
                header: [],
                cells: []
              };

              // header
              l2 = token.header.length;
              for (j = 0; j < l2; j++) {
                token.tokens.header[j] = [];
                this.inlineTokens(token.header[j], token.tokens.header[j]);
              }

              // cells
              l2 = token.cells.length;
              for (j = 0; j < l2; j++) {
                row = token.cells[j];
                token.tokens.cells[j] = [];
                for (k = 0; k < row.length; k++) {
                  token.tokens.cells[j][k] = [];
                  this.inlineTokens(row[k], token.tokens.cells[j][k]);
                }
              }

              break;
            }
            case 'blockquote': {
              this.inline(token.tokens);
              break;
            }
            case 'list': {
              l2 = token.items.length;
              for (j = 0; j < l2; j++) {
                this.inline(token.items[j].tokens);
              }
              break;
            }
          }
        }

        return tokens;
      }

      /**
       * Lexing/Compiling
       */
      inlineTokens(src, tokens = [], inLink = false, inRawBlock = false) {
        let token;

        while (src) {
          // escape
          if (token = this.tokenizer.escape(src)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // tag
          if (token = this.tokenizer.tag(src, inLink, inRawBlock)) {
            src = src.substring(token.raw.length);
            inLink = token.inLink;
            inRawBlock = token.inRawBlock;
            tokens.push(token);
            continue;
          }

          // link
          if (token = this.tokenizer.link(src)) {
            src = src.substring(token.raw.length);
            if (token.type === 'link') {
              token.tokens = this.inlineTokens(token.text, [], true, inRawBlock);
            }
            tokens.push(token);
            continue;
          }

          // reflink, nolink
          if (token = this.tokenizer.reflink(src, this.tokens.links)) {
            src = src.substring(token.raw.length);
            if (token.type === 'link') {
              token.tokens = this.inlineTokens(token.text, [], true, inRawBlock);
            }
            tokens.push(token);
            continue;
          }

          // strong
          if (token = this.tokenizer.strong(src)) {
            src = src.substring(token.raw.length);
            token.tokens = this.inlineTokens(token.text, [], inLink, inRawBlock);
            tokens.push(token);
            continue;
          }

          // em
          if (token = this.tokenizer.em(src)) {
            src = src.substring(token.raw.length);
            token.tokens = this.inlineTokens(token.text, [], inLink, inRawBlock);
            tokens.push(token);
            continue;
          }

          // code
          if (token = this.tokenizer.codespan(src)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // br
          if (token = this.tokenizer.br(src)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // del (gfm)
          if (token = this.tokenizer.del(src)) {
            src = src.substring(token.raw.length);
            token.tokens = this.inlineTokens(token.text, [], inLink, inRawBlock);
            tokens.push(token);
            continue;
          }

          // autolink
          if (token = this.tokenizer.autolink(src, mangle)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // url (gfm)
          if (!inLink && (token = this.tokenizer.url(src, mangle))) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          // text
          if (token = this.tokenizer.inlineText(src, inRawBlock, smartypants)) {
            src = src.substring(token.raw.length);
            tokens.push(token);
            continue;
          }

          if (src) {
            const errMsg = 'Infinite loop on byte: ' + src.charCodeAt(0);
            if (this.options.silent) {
              console.error(errMsg);
              break;
            } else {
              throw new Error(errMsg);
            }
          }
        }

        return tokens;
      }
    };

    const { defaults: defaults$3 } = defaults;
    const {
      cleanUrl: cleanUrl$1,
      escape: escape$2
    } = helpers;

    /**
     * Renderer
     */
    var Renderer_1 = class Renderer {
      constructor(options) {
        this.options = options || defaults$3;
      }

      code(code, infostring, escaped) {
        const lang = (infostring || '').match(/\S*/)[0];
        if (this.options.highlight) {
          const out = this.options.highlight(code, lang);
          if (out != null && out !== code) {
            escaped = true;
            code = out;
          }
        }

        if (!lang) {
          return '<pre><code>'
            + (escaped ? code : escape$2(code, true))
            + '</code></pre>\n';
        }

        return '<pre><code class="'
          + this.options.langPrefix
          + escape$2(lang, true)
          + '">'
          + (escaped ? code : escape$2(code, true))
          + '</code></pre>\n';
      }

      blockquote(quote) {
        return '<blockquote>\n' + quote + '</blockquote>\n';
      }

      html(html) {
        return html;
      }

      heading(text, level, raw, slugger) {
        if (this.options.headerIds) {
          return '<h'
            + level
            + ' id="'
            + this.options.headerPrefix
            + slugger.slug(raw)
            + '">'
            + text
            + '</h'
            + level
            + '>\n';
        }
        // ignore IDs
        return '<h' + level + '>' + text + '</h' + level + '>\n';
      }

      hr() {
        return this.options.xhtml ? '<hr/>\n' : '<hr>\n';
      }

      list(body, ordered, start) {
        const type = ordered ? 'ol' : 'ul',
          startatt = (ordered && start !== 1) ? (' start="' + start + '"') : '';
        return '<' + type + startatt + '>\n' + body + '</' + type + '>\n';
      }

      listitem(text) {
        return '<li>' + text + '</li>\n';
      }

      checkbox(checked) {
        return '<input '
          + (checked ? 'checked="" ' : '')
          + 'disabled="" type="checkbox"'
          + (this.options.xhtml ? ' /' : '')
          + '> ';
      }

      paragraph(text) {
        return '<p>' + text + '</p>\n';
      }

      table(header, body) {
        if (body) body = '<tbody>' + body + '</tbody>';

        return '<table>\n'
          + '<thead>\n'
          + header
          + '</thead>\n'
          + body
          + '</table>\n';
      }

      tablerow(content) {
        return '<tr>\n' + content + '</tr>\n';
      }

      tablecell(content, flags) {
        const type = flags.header ? 'th' : 'td';
        const tag = flags.align
          ? '<' + type + ' align="' + flags.align + '">'
          : '<' + type + '>';
        return tag + content + '</' + type + '>\n';
      }

      // span level renderer
      strong(text) {
        return '<strong>' + text + '</strong>';
      }

      em(text) {
        return '<em>' + text + '</em>';
      }

      codespan(text) {
        return '<code>' + text + '</code>';
      }

      br() {
        return this.options.xhtml ? '<br/>' : '<br>';
      }

      del(text) {
        return '<del>' + text + '</del>';
      }

      link(href, title, text) {
        href = cleanUrl$1(this.options.sanitize, this.options.baseUrl, href);
        if (href === null) {
          return text;
        }
        let out = '<a href="' + escape$2(href) + '"';
        if (title) {
          out += ' title="' + title + '"';
        }
        out += '>' + text + '</a>';
        return out;
      }

      image(href, title, text) {
        href = cleanUrl$1(this.options.sanitize, this.options.baseUrl, href);
        if (href === null) {
          return text;
        }

        let out = '<img src="' + href + '" alt="' + text + '"';
        if (title) {
          out += ' title="' + title + '"';
        }
        out += this.options.xhtml ? '/>' : '>';
        return out;
      }

      text(text) {
        return text;
      }
    };

    /**
     * TextRenderer
     * returns only the textual part of the token
     */
    var TextRenderer_1 = class TextRenderer {
      // no need for block level renderers
      strong(text) {
        return text;
      }

      em(text) {
        return text;
      }

      codespan(text) {
        return text;
      }

      del(text) {
        return text;
      }

      html(text) {
        return text;
      }

      text(text) {
        return text;
      }

      link(href, title, text) {
        return '' + text;
      }

      image(href, title, text) {
        return '' + text;
      }

      br() {
        return '';
      }
    };

    /**
     * Slugger generates header id
     */
    var Slugger_1 = class Slugger {
      constructor() {
        this.seen = {};
      }

      /**
       * Convert string to unique id
       */
      slug(value) {
        let slug = value
          .toLowerCase()
          .trim()
          // remove html tags
          .replace(/<[!\/a-z].*?>/ig, '')
          // remove unwanted chars
          .replace(/[\u2000-\u206F\u2E00-\u2E7F\\'!"#$%&()*+,./:;<=>?@[\]^`{|}~]/g, '')
          .replace(/\s/g, '-');

        if (this.seen.hasOwnProperty(slug)) {
          const originalSlug = slug;
          do {
            this.seen[originalSlug]++;
            slug = originalSlug + '-' + this.seen[originalSlug];
          } while (this.seen.hasOwnProperty(slug));
        }
        this.seen[slug] = 0;

        return slug;
      }
    };

    const { defaults: defaults$4 } = defaults;
    const {
      unescape: unescape$1
    } = helpers;

    /**
     * Parsing & Compiling
     */
    var Parser_1 = class Parser {
      constructor(options) {
        this.options = options || defaults$4;
        this.options.renderer = this.options.renderer || new Renderer_1();
        this.renderer = this.options.renderer;
        this.renderer.options = this.options;
        this.textRenderer = new TextRenderer_1();
        this.slugger = new Slugger_1();
      }

      /**
       * Static Parse Method
       */
      static parse(tokens, options) {
        const parser = new Parser(options);
        return parser.parse(tokens);
      }

      /**
       * Parse Loop
       */
      parse(tokens, top = true) {
        let out = '',
          i,
          j,
          k,
          l2,
          l3,
          row,
          cell,
          header,
          body,
          token,
          ordered,
          start,
          loose,
          itemBody,
          item,
          checked,
          task,
          checkbox;

        const l = tokens.length;
        for (i = 0; i < l; i++) {
          token = tokens[i];
          switch (token.type) {
            case 'space': {
              continue;
            }
            case 'hr': {
              out += this.renderer.hr();
              continue;
            }
            case 'heading': {
              out += this.renderer.heading(
                this.parseInline(token.tokens),
                token.depth,
                unescape$1(this.parseInline(token.tokens, this.textRenderer)),
                this.slugger);
              continue;
            }
            case 'code': {
              out += this.renderer.code(token.text,
                token.lang,
                token.escaped);
              continue;
            }
            case 'table': {
              header = '';

              // header
              cell = '';
              l2 = token.header.length;
              for (j = 0; j < l2; j++) {
                cell += this.renderer.tablecell(
                  this.parseInline(token.tokens.header[j]),
                  { header: true, align: token.align[j] }
                );
              }
              header += this.renderer.tablerow(cell);

              body = '';
              l2 = token.cells.length;
              for (j = 0; j < l2; j++) {
                row = token.tokens.cells[j];

                cell = '';
                l3 = row.length;
                for (k = 0; k < l3; k++) {
                  cell += this.renderer.tablecell(
                    this.parseInline(row[k]),
                    { header: false, align: token.align[k] }
                  );
                }

                body += this.renderer.tablerow(cell);
              }
              out += this.renderer.table(header, body);
              continue;
            }
            case 'blockquote': {
              body = this.parse(token.tokens);
              out += this.renderer.blockquote(body);
              continue;
            }
            case 'list': {
              ordered = token.ordered;
              start = token.start;
              loose = token.loose;
              l2 = token.items.length;

              body = '';
              for (j = 0; j < l2; j++) {
                item = token.items[j];
                checked = item.checked;
                task = item.task;

                itemBody = '';
                if (item.task) {
                  checkbox = this.renderer.checkbox(checked);
                  if (loose) {
                    if (item.tokens.length > 0 && item.tokens[0].type === 'text') {
                      item.tokens[0].text = checkbox + ' ' + item.tokens[0].text;
                      if (item.tokens[0].tokens && item.tokens[0].tokens.length > 0 && item.tokens[0].tokens[0].type === 'text') {
                        item.tokens[0].tokens[0].text = checkbox + ' ' + item.tokens[0].tokens[0].text;
                      }
                    } else {
                      item.tokens.unshift({
                        type: 'text',
                        text: checkbox
                      });
                    }
                  } else {
                    itemBody += checkbox;
                  }
                }

                itemBody += this.parse(item.tokens, loose);
                body += this.renderer.listitem(itemBody, task, checked);
              }

              out += this.renderer.list(body, ordered, start);
              continue;
            }
            case 'html': {
              // TODO parse inline content if parameter markdown=1
              out += this.renderer.html(token.text);
              continue;
            }
            case 'paragraph': {
              out += this.renderer.paragraph(this.parseInline(token.tokens));
              continue;
            }
            case 'text': {
              body = token.tokens ? this.parseInline(token.tokens) : token.text;
              while (i + 1 < l && tokens[i + 1].type === 'text') {
                token = tokens[++i];
                body += '\n' + (token.tokens ? this.parseInline(token.tokens) : token.text);
              }
              out += top ? this.renderer.paragraph(body) : body;
              continue;
            }
            default: {
              const errMsg = 'Token with "' + token.type + '" type was not found.';
              if (this.options.silent) {
                console.error(errMsg);
                return;
              } else {
                throw new Error(errMsg);
              }
            }
          }
        }

        return out;
      }

      /**
       * Parse Inline Tokens
       */
      parseInline(tokens, renderer) {
        renderer = renderer || this.renderer;
        let out = '',
          i,
          token;

        const l = tokens.length;
        for (i = 0; i < l; i++) {
          token = tokens[i];
          switch (token.type) {
            case 'escape': {
              out += renderer.text(token.text);
              break;
            }
            case 'html': {
              out += renderer.html(token.text);
              break;
            }
            case 'link': {
              out += renderer.link(token.href, token.title, this.parseInline(token.tokens, renderer));
              break;
            }
            case 'image': {
              out += renderer.image(token.href, token.title, token.text);
              break;
            }
            case 'strong': {
              out += renderer.strong(this.parseInline(token.tokens, renderer));
              break;
            }
            case 'em': {
              out += renderer.em(this.parseInline(token.tokens, renderer));
              break;
            }
            case 'codespan': {
              out += renderer.codespan(token.text);
              break;
            }
            case 'br': {
              out += renderer.br();
              break;
            }
            case 'del': {
              out += renderer.del(this.parseInline(token.tokens, renderer));
              break;
            }
            case 'text': {
              out += renderer.text(token.text);
              break;
            }
            default: {
              const errMsg = 'Token with "' + token.type + '" type was not found.';
              if (this.options.silent) {
                console.error(errMsg);
                return;
              } else {
                throw new Error(errMsg);
              }
            }
          }
        }
        return out;
      }
    };

    const {
      merge: merge$2,
      checkSanitizeDeprecation: checkSanitizeDeprecation$1,
      escape: escape$3
    } = helpers;
    const {
      getDefaults,
      changeDefaults,
      defaults: defaults$5
    } = defaults;

    /**
     * Marked
     */
    function marked(src, opt, callback) {
      // throw error in case of non string input
      if (typeof src === 'undefined' || src === null) {
        throw new Error('marked(): input parameter is undefined or null');
      }
      if (typeof src !== 'string') {
        throw new Error('marked(): input parameter is of type '
          + Object.prototype.toString.call(src) + ', string expected');
      }

      if (typeof opt === 'function') {
        callback = opt;
        opt = null;
      }

      opt = merge$2({}, marked.defaults, opt || {});
      checkSanitizeDeprecation$1(opt);

      if (callback) {
        const highlight = opt.highlight;
        let tokens;

        try {
          tokens = Lexer_1.lex(src, opt);
        } catch (e) {
          return callback(e);
        }

        const done = function(err) {
          let out;

          if (!err) {
            try {
              out = Parser_1.parse(tokens, opt);
            } catch (e) {
              err = e;
            }
          }

          opt.highlight = highlight;

          return err
            ? callback(err)
            : callback(null, out);
        };

        if (!highlight || highlight.length < 3) {
          return done();
        }

        delete opt.highlight;

        if (!tokens.length) return done();

        let pending = 0;
        marked.walkTokens(tokens, function(token) {
          if (token.type === 'code') {
            pending++;
            highlight(token.text, token.lang, function(err, code) {
              if (err) {
                return done(err);
              }
              if (code != null && code !== token.text) {
                token.text = code;
                token.escaped = true;
              }

              pending--;
              if (pending === 0) {
                done();
              }
            });
          }
        });

        if (pending === 0) {
          done();
        }

        return;
      }

      try {
        const tokens = Lexer_1.lex(src, opt);
        if (opt.walkTokens) {
          marked.walkTokens(tokens, opt.walkTokens);
        }
        return Parser_1.parse(tokens, opt);
      } catch (e) {
        e.message += '\nPlease report this to https://github.com/markedjs/marked.';
        if (opt.silent) {
          return '<p>An error occurred:</p><pre>'
            + escape$3(e.message + '', true)
            + '</pre>';
        }
        throw e;
      }
    }

    /**
     * Options
     */

    marked.options =
    marked.setOptions = function(opt) {
      merge$2(marked.defaults, opt);
      changeDefaults(marked.defaults);
      return marked;
    };

    marked.getDefaults = getDefaults;

    marked.defaults = defaults$5;

    /**
     * Use Extension
     */

    marked.use = function(extension) {
      const opts = merge$2({}, extension);
      if (extension.renderer) {
        const renderer = marked.defaults.renderer || new Renderer_1();
        for (const prop in extension.renderer) {
          const prevRenderer = renderer[prop];
          renderer[prop] = (...args) => {
            let ret = extension.renderer[prop].apply(renderer, args);
            if (ret === false) {
              ret = prevRenderer.apply(renderer, args);
            }
            return ret;
          };
        }
        opts.renderer = renderer;
      }
      if (extension.tokenizer) {
        const tokenizer = marked.defaults.tokenizer || new Tokenizer_1();
        for (const prop in extension.tokenizer) {
          const prevTokenizer = tokenizer[prop];
          tokenizer[prop] = (...args) => {
            let ret = extension.tokenizer[prop].apply(tokenizer, args);
            if (ret === false) {
              ret = prevTokenizer.apply(tokenizer, args);
            }
            return ret;
          };
        }
        opts.tokenizer = tokenizer;
      }
      if (extension.walkTokens) {
        const walkTokens = marked.defaults.walkTokens;
        opts.walkTokens = (token) => {
          extension.walkTokens(token);
          if (walkTokens) {
            walkTokens(token);
          }
        };
      }
      marked.setOptions(opts);
    };

    /**
     * Run callback for every token
     */

    marked.walkTokens = function(tokens, callback) {
      for (const token of tokens) {
        callback(token);
        switch (token.type) {
          case 'table': {
            for (const cell of token.tokens.header) {
              marked.walkTokens(cell, callback);
            }
            for (const row of token.tokens.cells) {
              for (const cell of row) {
                marked.walkTokens(cell, callback);
              }
            }
            break;
          }
          case 'list': {
            marked.walkTokens(token.items, callback);
            break;
          }
          default: {
            if (token.tokens) {
              marked.walkTokens(token.tokens, callback);
            }
          }
        }
      }
    };

    /**
     * Expose
     */

    marked.Parser = Parser_1;
    marked.parser = Parser_1.parse;

    marked.Renderer = Renderer_1;
    marked.TextRenderer = TextRenderer_1;

    marked.Lexer = Lexer_1;
    marked.lexer = Lexer_1.lex;

    marked.Tokenizer = Tokenizer_1;

    marked.Slugger = Slugger_1;

    marked.parse = marked;

    var marked_1 = marked;

    /*! @license DOMPurify | (c) Cure53 and other contributors | Released under the Apache license 2.0 and Mozilla Public License 2.0 | github.com/cure53/DOMPurify/blob/2.0.8/LICENSE */

    function _toConsumableArray(arr) { if (Array.isArray(arr)) { for (var i = 0, arr2 = Array(arr.length); i < arr.length; i++) { arr2[i] = arr[i]; } return arr2; } else { return Array.from(arr); } }

    var hasOwnProperty = Object.hasOwnProperty,
        setPrototypeOf = Object.setPrototypeOf,
        isFrozen = Object.isFrozen,
        objectKeys = Object.keys;
    var freeze = Object.freeze,
        seal = Object.seal; // eslint-disable-line import/no-mutable-exports

    var _ref = typeof Reflect !== 'undefined' && Reflect,
        apply = _ref.apply,
        construct = _ref.construct;

    if (!apply) {
      apply = function apply(fun, thisValue, args) {
        return fun.apply(thisValue, args);
      };
    }

    if (!freeze) {
      freeze = function freeze(x) {
        return x;
      };
    }

    if (!seal) {
      seal = function seal(x) {
        return x;
      };
    }

    if (!construct) {
      construct = function construct(Func, args) {
        return new (Function.prototype.bind.apply(Func, [null].concat(_toConsumableArray(args))))();
      };
    }

    var arrayForEach = unapply(Array.prototype.forEach);
    var arrayIndexOf = unapply(Array.prototype.indexOf);
    var arrayJoin = unapply(Array.prototype.join);
    var arrayPop = unapply(Array.prototype.pop);
    var arrayPush = unapply(Array.prototype.push);
    var arraySlice = unapply(Array.prototype.slice);

    var stringToLowerCase = unapply(String.prototype.toLowerCase);
    var stringMatch = unapply(String.prototype.match);
    var stringReplace = unapply(String.prototype.replace);
    var stringIndexOf = unapply(String.prototype.indexOf);
    var stringTrim = unapply(String.prototype.trim);

    var regExpTest = unapply(RegExp.prototype.test);
    var regExpCreate = unconstruct(RegExp);

    var typeErrorCreate = unconstruct(TypeError);

    function unapply(func) {
      return function (thisArg) {
        for (var _len = arguments.length, args = Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
          args[_key - 1] = arguments[_key];
        }

        return apply(func, thisArg, args);
      };
    }

    function unconstruct(func) {
      return function () {
        for (var _len2 = arguments.length, args = Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
          args[_key2] = arguments[_key2];
        }

        return construct(func, args);
      };
    }

    /* Add properties to a lookup table */
    function addToSet(set, array) {
      if (setPrototypeOf) {
        // Make 'in' and truthy checks like Boolean(set.constructor)
        // independent of any properties defined on Object.prototype.
        // Prevent prototype setters from intercepting set as a this value.
        setPrototypeOf(set, null);
      }

      var l = array.length;
      while (l--) {
        var element = array[l];
        if (typeof element === 'string') {
          var lcElement = stringToLowerCase(element);
          if (lcElement !== element) {
            // Config presets (e.g. tags.js, attrs.js) are immutable.
            if (!isFrozen(array)) {
              array[l] = lcElement;
            }

            element = lcElement;
          }
        }

        set[element] = true;
      }

      return set;
    }

    /* Shallow clone an object */
    function clone(object) {
      var newObject = {};

      var property = void 0;
      for (property in object) {
        if (apply(hasOwnProperty, object, [property])) {
          newObject[property] = object[property];
        }
      }

      return newObject;
    }

    var html$1 = freeze(['a', 'abbr', 'acronym', 'address', 'area', 'article', 'aside', 'audio', 'b', 'bdi', 'bdo', 'big', 'blink', 'blockquote', 'body', 'br', 'button', 'canvas', 'caption', 'center', 'cite', 'code', 'col', 'colgroup', 'content', 'data', 'datalist', 'dd', 'decorator', 'del', 'details', 'dfn', 'dir', 'div', 'dl', 'dt', 'element', 'em', 'fieldset', 'figcaption', 'figure', 'font', 'footer', 'form', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'head', 'header', 'hgroup', 'hr', 'html', 'i', 'img', 'input', 'ins', 'kbd', 'label', 'legend', 'li', 'main', 'map', 'mark', 'marquee', 'menu', 'menuitem', 'meter', 'nav', 'nobr', 'ol', 'optgroup', 'option', 'output', 'p', 'picture', 'pre', 'progress', 'q', 'rp', 'rt', 'ruby', 's', 'samp', 'section', 'select', 'shadow', 'small', 'source', 'spacer', 'span', 'strike', 'strong', 'style', 'sub', 'summary', 'sup', 'table', 'tbody', 'td', 'template', 'textarea', 'tfoot', 'th', 'thead', 'time', 'tr', 'track', 'tt', 'u', 'ul', 'var', 'video', 'wbr']);

    // SVG
    var svg = freeze(['svg', 'a', 'altglyph', 'altglyphdef', 'altglyphitem', 'animatecolor', 'animatemotion', 'animatetransform', 'audio', 'canvas', 'circle', 'clippath', 'defs', 'desc', 'ellipse', 'filter', 'font', 'g', 'glyph', 'glyphref', 'hkern', 'image', 'line', 'lineargradient', 'marker', 'mask', 'metadata', 'mpath', 'path', 'pattern', 'polygon', 'polyline', 'radialgradient', 'rect', 'stop', 'style', 'switch', 'symbol', 'text', 'textpath', 'title', 'tref', 'tspan', 'video', 'view', 'vkern']);

    var svgFilters = freeze(['feBlend', 'feColorMatrix', 'feComponentTransfer', 'feComposite', 'feConvolveMatrix', 'feDiffuseLighting', 'feDisplacementMap', 'feDistantLight', 'feFlood', 'feFuncA', 'feFuncB', 'feFuncG', 'feFuncR', 'feGaussianBlur', 'feMerge', 'feMergeNode', 'feMorphology', 'feOffset', 'fePointLight', 'feSpecularLighting', 'feSpotLight', 'feTile', 'feTurbulence']);

    var mathMl = freeze(['math', 'menclose', 'merror', 'mfenced', 'mfrac', 'mglyph', 'mi', 'mlabeledtr', 'mmultiscripts', 'mn', 'mo', 'mover', 'mpadded', 'mphantom', 'mroot', 'mrow', 'ms', 'mspace', 'msqrt', 'mstyle', 'msub', 'msup', 'msubsup', 'mtable', 'mtd', 'mtext', 'mtr', 'munder', 'munderover']);

    var text = freeze(['#text']);

    var html$1$1 = freeze(['accept', 'action', 'align', 'alt', 'autocapitalize', 'autocomplete', 'autopictureinpicture', 'autoplay', 'background', 'bgcolor', 'border', 'capture', 'cellpadding', 'cellspacing', 'checked', 'cite', 'class', 'clear', 'color', 'cols', 'colspan', 'controls', 'controlslist', 'coords', 'crossorigin', 'datetime', 'decoding', 'default', 'dir', 'disabled', 'disablepictureinpicture', 'disableremoteplayback', 'download', 'draggable', 'enctype', 'enterkeyhint', 'face', 'for', 'headers', 'height', 'hidden', 'high', 'href', 'hreflang', 'id', 'inputmode', 'integrity', 'ismap', 'kind', 'label', 'lang', 'list', 'loading', 'loop', 'low', 'max', 'maxlength', 'media', 'method', 'min', 'minlength', 'multiple', 'muted', 'name', 'noshade', 'novalidate', 'nowrap', 'open', 'optimum', 'pattern', 'placeholder', 'playsinline', 'poster', 'preload', 'pubdate', 'radiogroup', 'readonly', 'rel', 'required', 'rev', 'reversed', 'role', 'rows', 'rowspan', 'spellcheck', 'scope', 'selected', 'shape', 'size', 'sizes', 'span', 'srclang', 'start', 'src', 'srcset', 'step', 'style', 'summary', 'tabindex', 'title', 'translate', 'type', 'usemap', 'valign', 'value', 'width', 'xmlns']);

    var svg$1 = freeze(['accent-height', 'accumulate', 'additive', 'alignment-baseline', 'ascent', 'attributename', 'attributetype', 'azimuth', 'basefrequency', 'baseline-shift', 'begin', 'bias', 'by', 'class', 'clip', 'clip-path', 'clip-rule', 'color', 'color-interpolation', 'color-interpolation-filters', 'color-profile', 'color-rendering', 'cx', 'cy', 'd', 'dx', 'dy', 'diffuseconstant', 'direction', 'display', 'divisor', 'dur', 'edgemode', 'elevation', 'end', 'fill', 'fill-opacity', 'fill-rule', 'filter', 'filterunits', 'flood-color', 'flood-opacity', 'font-family', 'font-size', 'font-size-adjust', 'font-stretch', 'font-style', 'font-variant', 'font-weight', 'fx', 'fy', 'g1', 'g2', 'glyph-name', 'glyphref', 'gradientunits', 'gradienttransform', 'height', 'href', 'id', 'image-rendering', 'in', 'in2', 'k', 'k1', 'k2', 'k3', 'k4', 'kerning', 'keypoints', 'keysplines', 'keytimes', 'lang', 'lengthadjust', 'letter-spacing', 'kernelmatrix', 'kernelunitlength', 'lighting-color', 'local', 'marker-end', 'marker-mid', 'marker-start', 'markerheight', 'markerunits', 'markerwidth', 'maskcontentunits', 'maskunits', 'max', 'mask', 'media', 'method', 'mode', 'min', 'name', 'numoctaves', 'offset', 'operator', 'opacity', 'order', 'orient', 'orientation', 'origin', 'overflow', 'paint-order', 'path', 'pathlength', 'patterncontentunits', 'patterntransform', 'patternunits', 'points', 'preservealpha', 'preserveaspectratio', 'primitiveunits', 'r', 'rx', 'ry', 'radius', 'refx', 'refy', 'repeatcount', 'repeatdur', 'restart', 'result', 'rotate', 'scale', 'seed', 'shape-rendering', 'specularconstant', 'specularexponent', 'spreadmethod', 'startoffset', 'stddeviation', 'stitchtiles', 'stop-color', 'stop-opacity', 'stroke-dasharray', 'stroke-dashoffset', 'stroke-linecap', 'stroke-linejoin', 'stroke-miterlimit', 'stroke-opacity', 'stroke', 'stroke-width', 'style', 'surfacescale', 'tabindex', 'targetx', 'targety', 'transform', 'text-anchor', 'text-decoration', 'text-rendering', 'textlength', 'type', 'u1', 'u2', 'unicode', 'values', 'viewbox', 'visibility', 'version', 'vert-adv-y', 'vert-origin-x', 'vert-origin-y', 'width', 'word-spacing', 'wrap', 'writing-mode', 'xchannelselector', 'ychannelselector', 'x', 'x1', 'x2', 'xmlns', 'y', 'y1', 'y2', 'z', 'zoomandpan']);

    var mathMl$1 = freeze(['accent', 'accentunder', 'align', 'bevelled', 'close', 'columnsalign', 'columnlines', 'columnspan', 'denomalign', 'depth', 'dir', 'display', 'displaystyle', 'encoding', 'fence', 'frame', 'height', 'href', 'id', 'largeop', 'length', 'linethickness', 'lspace', 'lquote', 'mathbackground', 'mathcolor', 'mathsize', 'mathvariant', 'maxsize', 'minsize', 'movablelimits', 'notation', 'numalign', 'open', 'rowalign', 'rowlines', 'rowspacing', 'rowspan', 'rspace', 'rquote', 'scriptlevel', 'scriptminsize', 'scriptsizemultiplier', 'selection', 'separator', 'separators', 'stretchy', 'subscriptshift', 'supscriptshift', 'symmetric', 'voffset', 'width', 'xmlns']);

    var xml = freeze(['xlink:href', 'xml:id', 'xlink:title', 'xml:space', 'xmlns:xlink']);

    // eslint-disable-next-line unicorn/better-regex
    var MUSTACHE_EXPR = seal(/\{\{[\s\S]*|[\s\S]*\}\}/gm); // Specify template detection regex for SAFE_FOR_TEMPLATES mode
    var ERB_EXPR = seal(/<%[\s\S]*|[\s\S]*%>/gm);
    var DATA_ATTR = seal(/^data-[\-\w.\u00B7-\uFFFF]/); // eslint-disable-line no-useless-escape
    var ARIA_ATTR = seal(/^aria-[\-\w]+$/); // eslint-disable-line no-useless-escape
    var IS_ALLOWED_URI = seal(/^(?:(?:(?:f|ht)tps?|mailto|tel|callto|cid|xmpp):|[^a-z]|[a-z+.\-]+(?:[^a-z+.\-:]|$))/i // eslint-disable-line no-useless-escape
    );
    var IS_SCRIPT_OR_DATA = seal(/^(?:\w+script|data):/i);
    var ATTR_WHITESPACE = seal(/[\u0000-\u0020\u00A0\u1680\u180E\u2000-\u2029\u205f\u3000]/g // eslint-disable-line no-control-regex
    );

    var _typeof = typeof Symbol === "function" && typeof Symbol.iterator === "symbol" ? function (obj) { return typeof obj; } : function (obj) { return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj; };

    function _toConsumableArray$1(arr) { if (Array.isArray(arr)) { for (var i = 0, arr2 = Array(arr.length); i < arr.length; i++) { arr2[i] = arr[i]; } return arr2; } else { return Array.from(arr); } }

    var getGlobal = function getGlobal() {
      return typeof window === 'undefined' ? null : window;
    };

    /**
     * Creates a no-op policy for internal use only.
     * Don't export this function outside this module!
     * @param {?TrustedTypePolicyFactory} trustedTypes The policy factory.
     * @param {Document} document The document object (to determine policy name suffix)
     * @return {?TrustedTypePolicy} The policy created (or null, if Trusted Types
     * are not supported).
     */
    var _createTrustedTypesPolicy = function _createTrustedTypesPolicy(trustedTypes, document) {
      if ((typeof trustedTypes === 'undefined' ? 'undefined' : _typeof(trustedTypes)) !== 'object' || typeof trustedTypes.createPolicy !== 'function') {
        return null;
      }

      // Allow the callers to control the unique policy name
      // by adding a data-tt-policy-suffix to the script element with the DOMPurify.
      // Policy creation with duplicate names throws in Trusted Types.
      var suffix = null;
      var ATTR_NAME = 'data-tt-policy-suffix';
      if (document.currentScript && document.currentScript.hasAttribute(ATTR_NAME)) {
        suffix = document.currentScript.getAttribute(ATTR_NAME);
      }

      var policyName = 'dompurify' + (suffix ? '#' + suffix : '');

      try {
        return trustedTypes.createPolicy(policyName, {
          createHTML: function createHTML(html$$1) {
            return html$$1;
          }
        });
      } catch (_) {
        // Policy creation failed (most likely another DOMPurify script has
        // already run). Skip creating the policy, as this will only cause errors
        // if TT are enforced.
        console.warn('TrustedTypes policy ' + policyName + ' could not be created.');
        return null;
      }
    };

    function createDOMPurify() {
      var window = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : getGlobal();

      var DOMPurify = function DOMPurify(root) {
        return createDOMPurify(root);
      };

      /**
       * Version label, exposed for easier checks
       * if DOMPurify is up to date or not
       */
      DOMPurify.version = '2.0.11';

      /**
       * Array of elements that DOMPurify removed during sanitation.
       * Empty if nothing was removed.
       */
      DOMPurify.removed = [];

      if (!window || !window.document || window.document.nodeType !== 9) {
        // Not running in a browser, provide a factory function
        // so that you can pass your own Window
        DOMPurify.isSupported = false;

        return DOMPurify;
      }

      var originalDocument = window.document;
      var removeTitle = false;

      var document = window.document;
      var DocumentFragment = window.DocumentFragment,
          HTMLTemplateElement = window.HTMLTemplateElement,
          Node = window.Node,
          NodeFilter = window.NodeFilter,
          _window$NamedNodeMap = window.NamedNodeMap,
          NamedNodeMap = _window$NamedNodeMap === undefined ? window.NamedNodeMap || window.MozNamedAttrMap : _window$NamedNodeMap,
          Text = window.Text,
          Comment = window.Comment,
          DOMParser = window.DOMParser,
          trustedTypes = window.trustedTypes;

      // As per issue #47, the web-components registry is inherited by a
      // new document created via createHTMLDocument. As per the spec
      // (http://w3c.github.io/webcomponents/spec/custom/#creating-and-passing-registries)
      // a new empty registry is used when creating a template contents owner
      // document, so we use that as our parent document to ensure nothing
      // is inherited.

      if (typeof HTMLTemplateElement === 'function') {
        var template = document.createElement('template');
        if (template.content && template.content.ownerDocument) {
          document = template.content.ownerDocument;
        }
      }

      var trustedTypesPolicy = _createTrustedTypesPolicy(trustedTypes, originalDocument);
      var emptyHTML = trustedTypesPolicy ? trustedTypesPolicy.createHTML('') : '';

      var _document = document,
          implementation = _document.implementation,
          createNodeIterator = _document.createNodeIterator,
          getElementsByTagName = _document.getElementsByTagName,
          createDocumentFragment = _document.createDocumentFragment;
      var importNode = originalDocument.importNode;


      var hooks = {};

      /**
       * Expose whether this browser supports running the full DOMPurify.
       */
      DOMPurify.isSupported = implementation && typeof implementation.createHTMLDocument !== 'undefined' && document.documentMode !== 9;

      var MUSTACHE_EXPR$$1 = MUSTACHE_EXPR,
          ERB_EXPR$$1 = ERB_EXPR,
          DATA_ATTR$$1 = DATA_ATTR,
          ARIA_ATTR$$1 = ARIA_ATTR,
          IS_SCRIPT_OR_DATA$$1 = IS_SCRIPT_OR_DATA,
          ATTR_WHITESPACE$$1 = ATTR_WHITESPACE;
      var IS_ALLOWED_URI$$1 = IS_ALLOWED_URI;

      /**
       * We consider the elements and attributes below to be safe. Ideally
       * don't add any new ones but feel free to remove unwanted ones.
       */

      /* allowed element names */

      var ALLOWED_TAGS = null;
      var DEFAULT_ALLOWED_TAGS = addToSet({}, [].concat(_toConsumableArray$1(html$1), _toConsumableArray$1(svg), _toConsumableArray$1(svgFilters), _toConsumableArray$1(mathMl), _toConsumableArray$1(text)));

      /* Allowed attribute names */
      var ALLOWED_ATTR = null;
      var DEFAULT_ALLOWED_ATTR = addToSet({}, [].concat(_toConsumableArray$1(html$1$1), _toConsumableArray$1(svg$1), _toConsumableArray$1(mathMl$1), _toConsumableArray$1(xml)));

      /* Explicitly forbidden tags (overrides ALLOWED_TAGS/ADD_TAGS) */
      var FORBID_TAGS = null;

      /* Explicitly forbidden attributes (overrides ALLOWED_ATTR/ADD_ATTR) */
      var FORBID_ATTR = null;

      /* Decide if ARIA attributes are okay */
      var ALLOW_ARIA_ATTR = true;

      /* Decide if custom data attributes are okay */
      var ALLOW_DATA_ATTR = true;

      /* Decide if unknown protocols are okay */
      var ALLOW_UNKNOWN_PROTOCOLS = false;

      /* Output should be safe for jQuery's $() factory? */
      var SAFE_FOR_JQUERY = false;

      /* Output should be safe for common template engines.
       * This means, DOMPurify removes data attributes, mustaches and ERB
       */
      var SAFE_FOR_TEMPLATES = false;

      /* Decide if document with <html>... should be returned */
      var WHOLE_DOCUMENT = false;

      /* Track whether config is already set on this instance of DOMPurify. */
      var SET_CONFIG = false;

      /* Decide if all elements (e.g. style, script) must be children of
       * document.body. By default, browsers might move them to document.head */
      var FORCE_BODY = false;

      /* Decide if a DOM `HTMLBodyElement` should be returned, instead of a html
       * string (or a TrustedHTML object if Trusted Types are supported).
       * If `WHOLE_DOCUMENT` is enabled a `HTMLHtmlElement` will be returned instead
       */
      var RETURN_DOM = false;

      /* Decide if a DOM `DocumentFragment` should be returned, instead of a html
       * string  (or a TrustedHTML object if Trusted Types are supported) */
      var RETURN_DOM_FRAGMENT = false;

      /* If `RETURN_DOM` or `RETURN_DOM_FRAGMENT` is enabled, decide if the returned DOM
       * `Node` is imported into the current `Document`. If this flag is not enabled the
       * `Node` will belong (its ownerDocument) to a fresh `HTMLDocument`, created by
       * DOMPurify. */
      var RETURN_DOM_IMPORT = false;

      /* Try to return a Trusted Type object instead of a string, retrun a string in
       * case Trusted Types are not supported  */
      var RETURN_TRUSTED_TYPE = false;

      /* Output should be free from DOM clobbering attacks? */
      var SANITIZE_DOM = true;

      /* Keep element content when removing element? */
      var KEEP_CONTENT = true;

      /* If a `Node` is passed to sanitize(), then performs sanitization in-place instead
       * of importing it into a new Document and returning a sanitized copy */
      var IN_PLACE = false;

      /* Allow usage of profiles like html, svg and mathMl */
      var USE_PROFILES = {};

      /* Tags to ignore content of when KEEP_CONTENT is true */
      var FORBID_CONTENTS = addToSet({}, ['annotation-xml', 'audio', 'colgroup', 'desc', 'foreignobject', 'head', 'iframe', 'math', 'mi', 'mn', 'mo', 'ms', 'mtext', 'noembed', 'noframes', 'plaintext', 'script', 'style', 'svg', 'template', 'thead', 'title', 'video', 'xmp']);

      /* Tags that are safe for data: URIs */
      var DATA_URI_TAGS = null;
      var DEFAULT_DATA_URI_TAGS = addToSet({}, ['audio', 'video', 'img', 'source', 'image', 'track']);

      /* Attributes safe for values like "javascript:" */
      var URI_SAFE_ATTRIBUTES = null;
      var DEFAULT_URI_SAFE_ATTRIBUTES = addToSet({}, ['alt', 'class', 'for', 'id', 'label', 'name', 'pattern', 'placeholder', 'summary', 'title', 'value', 'style', 'xmlns']);

      /* Keep a reference to config to pass to hooks */
      var CONFIG = null;

      /* Ideally, do not touch anything below this line */
      /* ______________________________________________ */

      var formElement = document.createElement('form');

      /**
       * _parseConfig
       *
       * @param  {Object} cfg optional config literal
       */
      // eslint-disable-next-line complexity
      var _parseConfig = function _parseConfig(cfg) {
        if (CONFIG && CONFIG === cfg) {
          return;
        }

        /* Shield configuration object from tampering */
        if (!cfg || (typeof cfg === 'undefined' ? 'undefined' : _typeof(cfg)) !== 'object') {
          cfg = {};
        }

        /* Set configuration parameters */
        ALLOWED_TAGS = 'ALLOWED_TAGS' in cfg ? addToSet({}, cfg.ALLOWED_TAGS) : DEFAULT_ALLOWED_TAGS;
        ALLOWED_ATTR = 'ALLOWED_ATTR' in cfg ? addToSet({}, cfg.ALLOWED_ATTR) : DEFAULT_ALLOWED_ATTR;
        URI_SAFE_ATTRIBUTES = 'ADD_URI_SAFE_ATTR' in cfg ? addToSet(clone(DEFAULT_URI_SAFE_ATTRIBUTES), cfg.ADD_URI_SAFE_ATTR) : DEFAULT_URI_SAFE_ATTRIBUTES;
        DATA_URI_TAGS = 'ADD_DATA_URI_TAGS' in cfg ? addToSet(clone(DEFAULT_DATA_URI_TAGS), cfg.ADD_DATA_URI_TAGS) : DEFAULT_DATA_URI_TAGS;
        FORBID_TAGS = 'FORBID_TAGS' in cfg ? addToSet({}, cfg.FORBID_TAGS) : {};
        FORBID_ATTR = 'FORBID_ATTR' in cfg ? addToSet({}, cfg.FORBID_ATTR) : {};
        USE_PROFILES = 'USE_PROFILES' in cfg ? cfg.USE_PROFILES : false;
        ALLOW_ARIA_ATTR = cfg.ALLOW_ARIA_ATTR !== false; // Default true
        ALLOW_DATA_ATTR = cfg.ALLOW_DATA_ATTR !== false; // Default true
        ALLOW_UNKNOWN_PROTOCOLS = cfg.ALLOW_UNKNOWN_PROTOCOLS || false; // Default false
        SAFE_FOR_JQUERY = cfg.SAFE_FOR_JQUERY || false; // Default false
        SAFE_FOR_TEMPLATES = cfg.SAFE_FOR_TEMPLATES || false; // Default false
        WHOLE_DOCUMENT = cfg.WHOLE_DOCUMENT || false; // Default false
        RETURN_DOM = cfg.RETURN_DOM || false; // Default false
        RETURN_DOM_FRAGMENT = cfg.RETURN_DOM_FRAGMENT || false; // Default false
        RETURN_DOM_IMPORT = cfg.RETURN_DOM_IMPORT || false; // Default false
        RETURN_TRUSTED_TYPE = cfg.RETURN_TRUSTED_TYPE || false; // Default false
        FORCE_BODY = cfg.FORCE_BODY || false; // Default false
        SANITIZE_DOM = cfg.SANITIZE_DOM !== false; // Default true
        KEEP_CONTENT = cfg.KEEP_CONTENT !== false; // Default true
        IN_PLACE = cfg.IN_PLACE || false; // Default false
        IS_ALLOWED_URI$$1 = cfg.ALLOWED_URI_REGEXP || IS_ALLOWED_URI$$1;
        if (SAFE_FOR_TEMPLATES) {
          ALLOW_DATA_ATTR = false;
        }

        if (RETURN_DOM_FRAGMENT) {
          RETURN_DOM = true;
        }

        /* Parse profile info */
        if (USE_PROFILES) {
          ALLOWED_TAGS = addToSet({}, [].concat(_toConsumableArray$1(text)));
          ALLOWED_ATTR = [];
          if (USE_PROFILES.html === true) {
            addToSet(ALLOWED_TAGS, html$1);
            addToSet(ALLOWED_ATTR, html$1$1);
          }

          if (USE_PROFILES.svg === true) {
            addToSet(ALLOWED_TAGS, svg);
            addToSet(ALLOWED_ATTR, svg$1);
            addToSet(ALLOWED_ATTR, xml);
          }

          if (USE_PROFILES.svgFilters === true) {
            addToSet(ALLOWED_TAGS, svgFilters);
            addToSet(ALLOWED_ATTR, svg$1);
            addToSet(ALLOWED_ATTR, xml);
          }

          if (USE_PROFILES.mathMl === true) {
            addToSet(ALLOWED_TAGS, mathMl);
            addToSet(ALLOWED_ATTR, mathMl$1);
            addToSet(ALLOWED_ATTR, xml);
          }
        }

        /* Merge configuration parameters */
        if (cfg.ADD_TAGS) {
          if (ALLOWED_TAGS === DEFAULT_ALLOWED_TAGS) {
            ALLOWED_TAGS = clone(ALLOWED_TAGS);
          }

          addToSet(ALLOWED_TAGS, cfg.ADD_TAGS);
        }

        if (cfg.ADD_ATTR) {
          if (ALLOWED_ATTR === DEFAULT_ALLOWED_ATTR) {
            ALLOWED_ATTR = clone(ALLOWED_ATTR);
          }

          addToSet(ALLOWED_ATTR, cfg.ADD_ATTR);
        }

        if (cfg.ADD_URI_SAFE_ATTR) {
          addToSet(URI_SAFE_ATTRIBUTES, cfg.ADD_URI_SAFE_ATTR);
        }

        /* Add #text in case KEEP_CONTENT is set to true */
        if (KEEP_CONTENT) {
          ALLOWED_TAGS['#text'] = true;
        }

        /* Add html, head and body to ALLOWED_TAGS in case WHOLE_DOCUMENT is true */
        if (WHOLE_DOCUMENT) {
          addToSet(ALLOWED_TAGS, ['html', 'head', 'body']);
        }

        /* Add tbody to ALLOWED_TAGS in case tables are permitted, see #286, #365 */
        if (ALLOWED_TAGS.table) {
          addToSet(ALLOWED_TAGS, ['tbody']);
          delete FORBID_TAGS.tbody;
        }

        // Prevent further manipulation of configuration.
        // Not available in IE8, Safari 5, etc.
        if (freeze) {
          freeze(cfg);
        }

        CONFIG = cfg;
      };

      /**
       * _forceRemove
       *
       * @param  {Node} node a DOM node
       */
      var _forceRemove = function _forceRemove(node) {
        arrayPush(DOMPurify.removed, { element: node });
        try {
          // eslint-disable-next-line unicorn/prefer-node-remove
          node.parentNode.removeChild(node);
        } catch (_) {
          node.outerHTML = emptyHTML;
        }
      };

      /**
       * _removeAttribute
       *
       * @param  {String} name an Attribute name
       * @param  {Node} node a DOM node
       */
      var _removeAttribute = function _removeAttribute(name, node) {
        try {
          arrayPush(DOMPurify.removed, {
            attribute: node.getAttributeNode(name),
            from: node
          });
        } catch (_) {
          arrayPush(DOMPurify.removed, {
            attribute: null,
            from: node
          });
        }

        node.removeAttribute(name);
      };

      /**
       * _initDocument
       *
       * @param  {String} dirty a string of dirty markup
       * @return {Document} a DOM, filled with the dirty markup
       */
      var _initDocument = function _initDocument(dirty) {
        /* Create a HTML document */
        var doc = void 0;
        var leadingWhitespace = void 0;

        if (FORCE_BODY) {
          dirty = '<remove></remove>' + dirty;
        } else {
          /* If FORCE_BODY isn't used, leading whitespace needs to be preserved manually */
          var matches = stringMatch(dirty, /^[\r\n\t ]+/);
          leadingWhitespace = matches && matches[0];
        }

        var dirtyPayload = trustedTypesPolicy ? trustedTypesPolicy.createHTML(dirty) : dirty;
        /* Use the DOMParser API by default, fallback later if needs be */
        try {
          doc = new DOMParser().parseFromString(dirtyPayload, 'text/html');
        } catch (_) {}

        /* Remove title to fix a mXSS bug in older MS Edge */
        if (removeTitle) {
          addToSet(FORBID_TAGS, ['title']);
        }

        /* Use createHTMLDocument in case DOMParser is not available */
        if (!doc || !doc.documentElement) {
          doc = implementation.createHTMLDocument('');
          var _doc = doc,
              body = _doc.body;

          body.parentNode.removeChild(body.parentNode.firstElementChild);
          body.outerHTML = dirtyPayload;
        }

        if (dirty && leadingWhitespace) {
          doc.body.insertBefore(document.createTextNode(leadingWhitespace), doc.body.childNodes[0] || null);
        }

        /* Work on whole document or just its body */
        return getElementsByTagName.call(doc, WHOLE_DOCUMENT ? 'html' : 'body')[0];
      };

      /* Here we test for a broken feature in Edge that might cause mXSS */
      if (DOMPurify.isSupported) {
        (function () {
          try {
            var doc = _initDocument('<x/><title>&lt;/title&gt;&lt;img&gt;');
            if (regExpTest(/<\/title/, doc.querySelector('title').innerHTML)) {
              removeTitle = true;
            }
          } catch (_) {}
        })();
      }

      /**
       * _createIterator
       *
       * @param  {Document} root document/fragment to create iterator for
       * @return {Iterator} iterator instance
       */
      var _createIterator = function _createIterator(root) {
        return createNodeIterator.call(root.ownerDocument || root, root, NodeFilter.SHOW_ELEMENT | NodeFilter.SHOW_COMMENT | NodeFilter.SHOW_TEXT, function () {
          return NodeFilter.FILTER_ACCEPT;
        }, false);
      };

      /**
       * _isClobbered
       *
       * @param  {Node} elm element to check for clobbering attacks
       * @return {Boolean} true if clobbered, false if safe
       */
      var _isClobbered = function _isClobbered(elm) {
        if (elm instanceof Text || elm instanceof Comment) {
          return false;
        }

        if (typeof elm.nodeName !== 'string' || typeof elm.textContent !== 'string' || typeof elm.removeChild !== 'function' || !(elm.attributes instanceof NamedNodeMap) || typeof elm.removeAttribute !== 'function' || typeof elm.setAttribute !== 'function' || typeof elm.namespaceURI !== 'string') {
          return true;
        }

        return false;
      };

      /**
       * _isNode
       *
       * @param  {Node} obj object to check whether it's a DOM node
       * @return {Boolean} true is object is a DOM node
       */
      var _isNode = function _isNode(object) {
        return (typeof Node === 'undefined' ? 'undefined' : _typeof(Node)) === 'object' ? object instanceof Node : object && (typeof object === 'undefined' ? 'undefined' : _typeof(object)) === 'object' && typeof object.nodeType === 'number' && typeof object.nodeName === 'string';
      };

      /**
       * _executeHook
       * Execute user configurable hooks
       *
       * @param  {String} entryPoint  Name of the hook's entry point
       * @param  {Node} currentNode node to work on with the hook
       * @param  {Object} data additional hook parameters
       */
      var _executeHook = function _executeHook(entryPoint, currentNode, data) {
        if (!hooks[entryPoint]) {
          return;
        }

        arrayForEach(hooks[entryPoint], function (hook) {
          hook.call(DOMPurify, currentNode, data, CONFIG);
        });
      };

      /**
       * _sanitizeElements
       *
       * @protect nodeName
       * @protect textContent
       * @protect removeChild
       *
       * @param   {Node} currentNode to check for permission to exist
       * @return  {Boolean} true if node was killed, false if left alive
       */
      // eslint-disable-next-line complexity
      var _sanitizeElements = function _sanitizeElements(currentNode) {
        var content = void 0;

        /* Execute a hook if present */
        _executeHook('beforeSanitizeElements', currentNode, null);

        /* Check if element is clobbered or can clobber */
        if (_isClobbered(currentNode)) {
          _forceRemove(currentNode);
          return true;
        }

        /* Now let's check the element's type and name */
        var tagName = stringToLowerCase(currentNode.nodeName);

        /* Execute a hook if present */
        _executeHook('uponSanitizeElement', currentNode, {
          tagName: tagName,
          allowedTags: ALLOWED_TAGS
        });

        /* Take care of an mXSS pattern using p, br inside svg, math */
        if ((tagName === 'svg' || tagName === 'math') && currentNode.querySelectorAll('p, br').length !== 0) {
          _forceRemove(currentNode);
          return true;
        }

        /* Remove element if anything forbids its presence */
        if (!ALLOWED_TAGS[tagName] || FORBID_TAGS[tagName]) {
          /* Keep content except for black-listed elements */
          if (KEEP_CONTENT && !FORBID_CONTENTS[tagName] && typeof currentNode.insertAdjacentHTML === 'function') {
            try {
              var htmlToInsert = currentNode.innerHTML;
              currentNode.insertAdjacentHTML('AfterEnd', trustedTypesPolicy ? trustedTypesPolicy.createHTML(htmlToInsert) : htmlToInsert);
            } catch (_) {}
          }

          _forceRemove(currentNode);
          return true;
        }

        /* Remove in case a noscript/noembed XSS is suspected */
        if (tagName === 'noscript' && regExpTest(/<\/noscript/i, currentNode.innerHTML)) {
          _forceRemove(currentNode);
          return true;
        }

        if (tagName === 'noembed' && regExpTest(/<\/noembed/i, currentNode.innerHTML)) {
          _forceRemove(currentNode);
          return true;
        }

        /* Convert markup to cover jQuery behavior */
        if (SAFE_FOR_JQUERY && !currentNode.firstElementChild && (!currentNode.content || !currentNode.content.firstElementChild) && regExpTest(/</g, currentNode.textContent)) {
          arrayPush(DOMPurify.removed, { element: currentNode.cloneNode() });
          if (currentNode.innerHTML) {
            currentNode.innerHTML = stringReplace(currentNode.innerHTML, /</g, '&lt;');
          } else {
            currentNode.innerHTML = stringReplace(currentNode.textContent, /</g, '&lt;');
          }
        }

        /* Sanitize element content to be template-safe */
        if (SAFE_FOR_TEMPLATES && currentNode.nodeType === 3) {
          /* Get the element's text content */
          content = currentNode.textContent;
          content = stringReplace(content, MUSTACHE_EXPR$$1, ' ');
          content = stringReplace(content, ERB_EXPR$$1, ' ');
          if (currentNode.textContent !== content) {
            arrayPush(DOMPurify.removed, { element: currentNode.cloneNode() });
            currentNode.textContent = content;
          }
        }

        /* Execute a hook if present */
        _executeHook('afterSanitizeElements', currentNode, null);

        return false;
      };

      /**
       * _isValidAttribute
       *
       * @param  {string} lcTag Lowercase tag name of containing element.
       * @param  {string} lcName Lowercase attribute name.
       * @param  {string} value Attribute value.
       * @return {Boolean} Returns true if `value` is valid, otherwise false.
       */
      // eslint-disable-next-line complexity
      var _isValidAttribute = function _isValidAttribute(lcTag, lcName, value) {
        /* Make sure attribute cannot clobber */
        if (SANITIZE_DOM && (lcName === 'id' || lcName === 'name') && (value in document || value in formElement)) {
          return false;
        }

        /* Allow valid data-* attributes: At least one character after "-"
            (https://html.spec.whatwg.org/multipage/dom.html#embedding-custom-non-visible-data-with-the-data-*-attributes)
            XML-compatible (https://html.spec.whatwg.org/multipage/infrastructure.html#xml-compatible and http://www.w3.org/TR/xml/#d0e804)
            We don't need to check the value; it's always URI safe. */
        if (ALLOW_DATA_ATTR && regExpTest(DATA_ATTR$$1, lcName)) ; else if (ALLOW_ARIA_ATTR && regExpTest(ARIA_ATTR$$1, lcName)) ; else if (!ALLOWED_ATTR[lcName] || FORBID_ATTR[lcName]) {
          return false;

          /* Check value is safe. First, is attr inert? If so, is safe */
        } else if (URI_SAFE_ATTRIBUTES[lcName]) ; else if (regExpTest(IS_ALLOWED_URI$$1, stringReplace(value, ATTR_WHITESPACE$$1, ''))) ; else if ((lcName === 'src' || lcName === 'xlink:href' || lcName === 'href') && lcTag !== 'script' && stringIndexOf(value, 'data:') === 0 && DATA_URI_TAGS[lcTag]) ; else if (ALLOW_UNKNOWN_PROTOCOLS && !regExpTest(IS_SCRIPT_OR_DATA$$1, stringReplace(value, ATTR_WHITESPACE$$1, ''))) ; else if (!value) ; else {
          return false;
        }

        return true;
      };

      /**
       * _sanitizeAttributes
       *
       * @protect attributes
       * @protect nodeName
       * @protect removeAttribute
       * @protect setAttribute
       *
       * @param  {Node} currentNode to sanitize
       */
      // eslint-disable-next-line complexity
      var _sanitizeAttributes = function _sanitizeAttributes(currentNode) {
        var attr = void 0;
        var value = void 0;
        var lcName = void 0;
        var idAttr = void 0;
        var l = void 0;
        /* Execute a hook if present */
        _executeHook('beforeSanitizeAttributes', currentNode, null);

        var attributes = currentNode.attributes;

        /* Check if we have attributes; if not we might have a text node */

        if (!attributes) {
          return;
        }

        var hookEvent = {
          attrName: '',
          attrValue: '',
          keepAttr: true,
          allowedAttributes: ALLOWED_ATTR
        };
        l = attributes.length;

        /* Go backwards over all attributes; safely remove bad ones */
        while (l--) {
          attr = attributes[l];
          var _attr = attr,
              name = _attr.name,
              namespaceURI = _attr.namespaceURI;

          value = stringTrim(attr.value);
          lcName = stringToLowerCase(name);

          /* Execute a hook if present */
          hookEvent.attrName = lcName;
          hookEvent.attrValue = value;
          hookEvent.keepAttr = true;
          hookEvent.forceKeepAttr = undefined; // Allows developers to see this is a property they can set
          _executeHook('uponSanitizeAttribute', currentNode, hookEvent);
          value = hookEvent.attrValue;
          /* Did the hooks approve of the attribute? */
          if (hookEvent.forceKeepAttr) {
            continue;
          }

          /* Remove attribute */
          // Safari (iOS + Mac), last tested v8.0.5, crashes if you try to
          // remove a "name" attribute from an <img> tag that has an "id"
          // attribute at the time.
          if (lcName === 'name' && currentNode.nodeName === 'IMG' && attributes.id) {
            idAttr = attributes.id;
            attributes = arraySlice(attributes, []);
            _removeAttribute('id', currentNode);
            _removeAttribute(name, currentNode);
            if (arrayIndexOf(attributes, idAttr) > l) {
              currentNode.setAttribute('id', idAttr.value);
            }
          } else if (
          // This works around a bug in Safari, where input[type=file]
          // cannot be dynamically set after type has been removed
          currentNode.nodeName === 'INPUT' && lcName === 'type' && value === 'file' && hookEvent.keepAttr && (ALLOWED_ATTR[lcName] || !FORBID_ATTR[lcName])) {
            continue;
          } else {
            // This avoids a crash in Safari v9.0 with double-ids.
            // The trick is to first set the id to be empty and then to
            // remove the attribute
            if (name === 'id') {
              currentNode.setAttribute(name, '');
            }

            _removeAttribute(name, currentNode);
          }

          /* Did the hooks approve of the attribute? */
          if (!hookEvent.keepAttr) {
            continue;
          }

          /* Work around a security issue in jQuery 3.0 */
          if (SAFE_FOR_JQUERY && regExpTest(/\/>/i, value)) {
            _removeAttribute(name, currentNode);
            continue;
          }

          /* Take care of an mXSS pattern using namespace switches */
          if (regExpTest(/svg|math/i, currentNode.namespaceURI) && regExpTest(regExpCreate('</(' + arrayJoin(objectKeys(FORBID_CONTENTS), '|') + ')', 'i'), value)) {
            _removeAttribute(name, currentNode);
            continue;
          }

          /* Sanitize attribute content to be template-safe */
          if (SAFE_FOR_TEMPLATES) {
            value = stringReplace(value, MUSTACHE_EXPR$$1, ' ');
            value = stringReplace(value, ERB_EXPR$$1, ' ');
          }

          /* Is `value` valid for this attribute? */
          var lcTag = currentNode.nodeName.toLowerCase();
          if (!_isValidAttribute(lcTag, lcName, value)) {
            continue;
          }

          /* Handle invalid data-* attribute set by try-catching it */
          try {
            if (namespaceURI) {
              currentNode.setAttributeNS(namespaceURI, name, value);
            } else {
              /* Fallback to setAttribute() for browser-unrecognized namespaces e.g. "x-schema". */
              currentNode.setAttribute(name, value);
            }

            arrayPop(DOMPurify.removed);
          } catch (_) {}
        }

        /* Execute a hook if present */
        _executeHook('afterSanitizeAttributes', currentNode, null);
      };

      /**
       * _sanitizeShadowDOM
       *
       * @param  {DocumentFragment} fragment to iterate over recursively
       */
      var _sanitizeShadowDOM = function _sanitizeShadowDOM(fragment) {
        var shadowNode = void 0;
        var shadowIterator = _createIterator(fragment);

        /* Execute a hook if present */
        _executeHook('beforeSanitizeShadowDOM', fragment, null);

        while (shadowNode = shadowIterator.nextNode()) {
          /* Execute a hook if present */
          _executeHook('uponSanitizeShadowNode', shadowNode, null);

          /* Sanitize tags and elements */
          if (_sanitizeElements(shadowNode)) {
            continue;
          }

          /* Deep shadow DOM detected */
          if (shadowNode.content instanceof DocumentFragment) {
            _sanitizeShadowDOM(shadowNode.content);
          }

          /* Check attributes, sanitize if necessary */
          _sanitizeAttributes(shadowNode);
        }

        /* Execute a hook if present */
        _executeHook('afterSanitizeShadowDOM', fragment, null);
      };

      /**
       * Sanitize
       * Public method providing core sanitation functionality
       *
       * @param {String|Node} dirty string or DOM node
       * @param {Object} configuration object
       */
      // eslint-disable-next-line complexity
      DOMPurify.sanitize = function (dirty, cfg) {
        var body = void 0;
        var importedNode = void 0;
        var currentNode = void 0;
        var oldNode = void 0;
        var returnNode = void 0;
        /* Make sure we have a string to sanitize.
          DO NOT return early, as this will return the wrong type if
          the user has requested a DOM object rather than a string */
        if (!dirty) {
          dirty = '<!-->';
        }

        /* Stringify, in case dirty is an object */
        if (typeof dirty !== 'string' && !_isNode(dirty)) {
          // eslint-disable-next-line no-negated-condition
          if (typeof dirty.toString !== 'function') {
            throw typeErrorCreate('toString is not a function');
          } else {
            dirty = dirty.toString();
            if (typeof dirty !== 'string') {
              throw typeErrorCreate('dirty is not a string, aborting');
            }
          }
        }

        /* Check we can run. Otherwise fall back or ignore */
        if (!DOMPurify.isSupported) {
          if (_typeof(window.toStaticHTML) === 'object' || typeof window.toStaticHTML === 'function') {
            if (typeof dirty === 'string') {
              return window.toStaticHTML(dirty);
            }

            if (_isNode(dirty)) {
              return window.toStaticHTML(dirty.outerHTML);
            }
          }

          return dirty;
        }

        /* Assign config vars */
        if (!SET_CONFIG) {
          _parseConfig(cfg);
        }

        /* Clean up removed elements */
        DOMPurify.removed = [];

        /* Check if dirty is correctly typed for IN_PLACE */
        if (typeof dirty === 'string') {
          IN_PLACE = false;
        }

        if (IN_PLACE) ; else if (dirty instanceof Node) {
          /* If dirty is a DOM element, append to an empty document to avoid
             elements being stripped by the parser */
          body = _initDocument('<!-->');
          importedNode = body.ownerDocument.importNode(dirty, true);
          if (importedNode.nodeType === 1 && importedNode.nodeName === 'BODY') {
            /* Node is already a body, use as is */
            body = importedNode;
          } else if (importedNode.nodeName === 'HTML') {
            body = importedNode;
          } else {
            // eslint-disable-next-line unicorn/prefer-node-append
            body.appendChild(importedNode);
          }
        } else {
          /* Exit directly if we have nothing to do */
          if (!RETURN_DOM && !SAFE_FOR_TEMPLATES && !WHOLE_DOCUMENT && RETURN_TRUSTED_TYPE &&
          // eslint-disable-next-line unicorn/prefer-includes
          dirty.indexOf('<') === -1) {
            return trustedTypesPolicy ? trustedTypesPolicy.createHTML(dirty) : dirty;
          }

          /* Initialize the document to work on */
          body = _initDocument(dirty);

          /* Check we have a DOM node from the data */
          if (!body) {
            return RETURN_DOM ? null : emptyHTML;
          }
        }

        /* Remove first element node (ours) if FORCE_BODY is set */
        if (body && FORCE_BODY) {
          _forceRemove(body.firstChild);
        }

        /* Get node iterator */
        var nodeIterator = _createIterator(IN_PLACE ? dirty : body);

        /* Now start iterating over the created document */
        while (currentNode = nodeIterator.nextNode()) {
          /* Fix IE's strange behavior with manipulated textNodes #89 */
          if (currentNode.nodeType === 3 && currentNode === oldNode) {
            continue;
          }

          /* Sanitize tags and elements */
          if (_sanitizeElements(currentNode)) {
            continue;
          }

          /* Shadow DOM detected, sanitize it */
          if (currentNode.content instanceof DocumentFragment) {
            _sanitizeShadowDOM(currentNode.content);
          }

          /* Check attributes, sanitize if necessary */
          _sanitizeAttributes(currentNode);

          oldNode = currentNode;
        }

        oldNode = null;

        /* If we sanitized `dirty` in-place, return it. */
        if (IN_PLACE) {
          return dirty;
        }

        /* Return sanitized string or DOM */
        if (RETURN_DOM) {
          if (RETURN_DOM_FRAGMENT) {
            returnNode = createDocumentFragment.call(body.ownerDocument);

            while (body.firstChild) {
              // eslint-disable-next-line unicorn/prefer-node-append
              returnNode.appendChild(body.firstChild);
            }
          } else {
            returnNode = body;
          }

          if (RETURN_DOM_IMPORT) {
            /*
              AdoptNode() is not used because internal state is not reset
              (e.g. the past names map of a HTMLFormElement), this is safe
              in theory but we would rather not risk another attack vector.
              The state that is cloned by importNode() is explicitly defined
              by the specs.
            */
            returnNode = importNode.call(originalDocument, returnNode, true);
          }

          return returnNode;
        }

        var serializedHTML = WHOLE_DOCUMENT ? body.outerHTML : body.innerHTML;

        /* Sanitize final string template-safe */
        if (SAFE_FOR_TEMPLATES) {
          serializedHTML = stringReplace(serializedHTML, MUSTACHE_EXPR$$1, ' ');
          serializedHTML = stringReplace(serializedHTML, ERB_EXPR$$1, ' ');
        }

        return trustedTypesPolicy && RETURN_TRUSTED_TYPE ? trustedTypesPolicy.createHTML(serializedHTML) : serializedHTML;
      };

      /**
       * Public method to set the configuration once
       * setConfig
       *
       * @param {Object} cfg configuration object
       */
      DOMPurify.setConfig = function (cfg) {
        _parseConfig(cfg);
        SET_CONFIG = true;
      };

      /**
       * Public method to remove the configuration
       * clearConfig
       *
       */
      DOMPurify.clearConfig = function () {
        CONFIG = null;
        SET_CONFIG = false;
      };

      /**
       * Public method to check if an attribute value is valid.
       * Uses last set config, if any. Otherwise, uses config defaults.
       * isValidAttribute
       *
       * @param  {string} tag Tag name of containing element.
       * @param  {string} attr Attribute name.
       * @param  {string} value Attribute value.
       * @return {Boolean} Returns true if `value` is valid. Otherwise, returns false.
       */
      DOMPurify.isValidAttribute = function (tag, attr, value) {
        /* Initialize shared config vars if necessary. */
        if (!CONFIG) {
          _parseConfig({});
        }

        var lcTag = stringToLowerCase(tag);
        var lcName = stringToLowerCase(attr);
        return _isValidAttribute(lcTag, lcName, value);
      };

      /**
       * AddHook
       * Public method to add DOMPurify hooks
       *
       * @param {String} entryPoint entry point for the hook to add
       * @param {Function} hookFunction function to execute
       */
      DOMPurify.addHook = function (entryPoint, hookFunction) {
        if (typeof hookFunction !== 'function') {
          return;
        }

        hooks[entryPoint] = hooks[entryPoint] || [];
        arrayPush(hooks[entryPoint], hookFunction);
      };

      /**
       * RemoveHook
       * Public method to remove a DOMPurify hook at a given entryPoint
       * (pops it from the stack of hooks if more are present)
       *
       * @param {String} entryPoint entry point for the hook to remove
       */
      DOMPurify.removeHook = function (entryPoint) {
        if (hooks[entryPoint]) {
          arrayPop(hooks[entryPoint]);
        }
      };

      /**
       * RemoveHooks
       * Public method to remove all DOMPurify hooks at a given entryPoint
       *
       * @param  {String} entryPoint entry point for the hooks to remove
       */
      DOMPurify.removeHooks = function (entryPoint) {
        if (hooks[entryPoint]) {
          hooks[entryPoint] = [];
        }
      };

      /**
       * RemoveAllHooks
       * Public method to remove all DOMPurify hooks
       *
       */
      DOMPurify.removeAllHooks = function () {
        hooks = {};
      };

      return DOMPurify;
    }

    var purify = createDOMPurify();

    /**
     * lodash (Custom Build) <https://lodash.com/>
     * Build: `lodash modularize exports="npm" -o ./`
     * Copyright jQuery Foundation and other contributors <https://jquery.org/>
     * Released under MIT license <https://lodash.com/license>
     * Based on Underscore.js 1.8.3 <http://underscorejs.org/LICENSE>
     * Copyright Jeremy Ashkenas, DocumentCloud and Investigative Reporters & Editors
     */

    /** Used as references for various `Number` constants. */
    var MAX_SAFE_INTEGER = 9007199254740991;

    /** `Object#toString` result references. */
    var argsTag = '[object Arguments]',
        funcTag = '[object Function]',
        genTag = '[object GeneratorFunction]',
        mapTag = '[object Map]',
        objectTag = '[object Object]',
        promiseTag = '[object Promise]',
        setTag = '[object Set]',
        stringTag = '[object String]',
        weakMapTag = '[object WeakMap]';

    var dataViewTag = '[object DataView]';

    /**
     * Used to match `RegExp`
     * [syntax characters](http://ecma-international.org/ecma-262/7.0/#sec-patterns).
     */
    var reRegExpChar = /[\\^$.*+?()[\]{}|]/g;

    /** Used to detect host constructors (Safari). */
    var reIsHostCtor = /^\[object .+?Constructor\]$/;

    /** Used to detect unsigned integer values. */
    var reIsUint = /^(?:0|[1-9]\d*)$/;

    /** Used to compose unicode character classes. */
    var rsAstralRange = '\\ud800-\\udfff',
        rsComboMarksRange = '\\u0300-\\u036f\\ufe20-\\ufe23',
        rsComboSymbolsRange = '\\u20d0-\\u20f0',
        rsVarRange = '\\ufe0e\\ufe0f';

    /** Used to compose unicode capture groups. */
    var rsAstral = '[' + rsAstralRange + ']',
        rsCombo = '[' + rsComboMarksRange + rsComboSymbolsRange + ']',
        rsFitz = '\\ud83c[\\udffb-\\udfff]',
        rsModifier = '(?:' + rsCombo + '|' + rsFitz + ')',
        rsNonAstral = '[^' + rsAstralRange + ']',
        rsRegional = '(?:\\ud83c[\\udde6-\\uddff]){2}',
        rsSurrPair = '[\\ud800-\\udbff][\\udc00-\\udfff]',
        rsZWJ = '\\u200d';

    /** Used to compose unicode regexes. */
    var reOptMod = rsModifier + '?',
        rsOptVar = '[' + rsVarRange + ']?',
        rsOptJoin = '(?:' + rsZWJ + '(?:' + [rsNonAstral, rsRegional, rsSurrPair].join('|') + ')' + rsOptVar + reOptMod + ')*',
        rsSeq = rsOptVar + reOptMod + rsOptJoin,
        rsSymbol = '(?:' + [rsNonAstral + rsCombo + '?', rsCombo, rsRegional, rsSurrPair, rsAstral].join('|') + ')';

    /** Used to match [string symbols](https://mathiasbynens.be/notes/javascript-unicode). */
    var reUnicode = RegExp(rsFitz + '(?=' + rsFitz + ')|' + rsSymbol + rsSeq, 'g');

    /** Used to detect strings with [zero-width joiners or code points from the astral planes](http://eev.ee/blog/2015/09/12/dark-corners-of-unicode/). */
    var reHasUnicode = RegExp('[' + rsZWJ + rsAstralRange  + rsComboMarksRange + rsComboSymbolsRange + rsVarRange + ']');

    /** Detect free variable `global` from Node.js. */
    var freeGlobal = typeof commonjsGlobal == 'object' && commonjsGlobal && commonjsGlobal.Object === Object && commonjsGlobal;

    /** Detect free variable `self`. */
    var freeSelf = typeof self == 'object' && self && self.Object === Object && self;

    /** Used as a reference to the global object. */
    var root = freeGlobal || freeSelf || Function('return this')();

    /**
     * A specialized version of `_.map` for arrays without support for iteratee
     * shorthands.
     *
     * @private
     * @param {Array} [array] The array to iterate over.
     * @param {Function} iteratee The function invoked per iteration.
     * @returns {Array} Returns the new mapped array.
     */
    function arrayMap(array, iteratee) {
      var index = -1,
          length = array ? array.length : 0,
          result = Array(length);

      while (++index < length) {
        result[index] = iteratee(array[index], index, array);
      }
      return result;
    }

    /**
     * Converts an ASCII `string` to an array.
     *
     * @private
     * @param {string} string The string to convert.
     * @returns {Array} Returns the converted array.
     */
    function asciiToArray(string) {
      return string.split('');
    }

    /**
     * The base implementation of `_.times` without support for iteratee shorthands
     * or max array length checks.
     *
     * @private
     * @param {number} n The number of times to invoke `iteratee`.
     * @param {Function} iteratee The function invoked per iteration.
     * @returns {Array} Returns the array of results.
     */
    function baseTimes(n, iteratee) {
      var index = -1,
          result = Array(n);

      while (++index < n) {
        result[index] = iteratee(index);
      }
      return result;
    }

    /**
     * The base implementation of `_.values` and `_.valuesIn` which creates an
     * array of `object` property values corresponding to the property names
     * of `props`.
     *
     * @private
     * @param {Object} object The object to query.
     * @param {Array} props The property names to get values for.
     * @returns {Object} Returns the array of property values.
     */
    function baseValues(object, props) {
      return arrayMap(props, function(key) {
        return object[key];
      });
    }

    /**
     * Gets the value at `key` of `object`.
     *
     * @private
     * @param {Object} [object] The object to query.
     * @param {string} key The key of the property to get.
     * @returns {*} Returns the property value.
     */
    function getValue(object, key) {
      return object == null ? undefined : object[key];
    }

    /**
     * Checks if `string` contains Unicode symbols.
     *
     * @private
     * @param {string} string The string to inspect.
     * @returns {boolean} Returns `true` if a symbol is found, else `false`.
     */
    function hasUnicode(string) {
      return reHasUnicode.test(string);
    }

    /**
     * Checks if `value` is a host object in IE < 9.
     *
     * @private
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is a host object, else `false`.
     */
    function isHostObject(value) {
      // Many host objects are `Object` objects that can coerce to strings
      // despite having improperly defined `toString` methods.
      var result = false;
      if (value != null && typeof value.toString != 'function') {
        try {
          result = !!(value + '');
        } catch (e) {}
      }
      return result;
    }

    /**
     * Converts `iterator` to an array.
     *
     * @private
     * @param {Object} iterator The iterator to convert.
     * @returns {Array} Returns the converted array.
     */
    function iteratorToArray(iterator) {
      var data,
          result = [];

      while (!(data = iterator.next()).done) {
        result.push(data.value);
      }
      return result;
    }

    /**
     * Converts `map` to its key-value pairs.
     *
     * @private
     * @param {Object} map The map to convert.
     * @returns {Array} Returns the key-value pairs.
     */
    function mapToArray(map) {
      var index = -1,
          result = Array(map.size);

      map.forEach(function(value, key) {
        result[++index] = [key, value];
      });
      return result;
    }

    /**
     * Creates a unary function that invokes `func` with its argument transformed.
     *
     * @private
     * @param {Function} func The function to wrap.
     * @param {Function} transform The argument transform.
     * @returns {Function} Returns the new function.
     */
    function overArg(func, transform) {
      return function(arg) {
        return func(transform(arg));
      };
    }

    /**
     * Converts `set` to an array of its values.
     *
     * @private
     * @param {Object} set The set to convert.
     * @returns {Array} Returns the values.
     */
    function setToArray(set) {
      var index = -1,
          result = Array(set.size);

      set.forEach(function(value) {
        result[++index] = value;
      });
      return result;
    }

    /**
     * Converts `string` to an array.
     *
     * @private
     * @param {string} string The string to convert.
     * @returns {Array} Returns the converted array.
     */
    function stringToArray(string) {
      return hasUnicode(string)
        ? unicodeToArray(string)
        : asciiToArray(string);
    }

    /**
     * Converts a Unicode `string` to an array.
     *
     * @private
     * @param {string} string The string to convert.
     * @returns {Array} Returns the converted array.
     */
    function unicodeToArray(string) {
      return string.match(reUnicode) || [];
    }

    /** Used for built-in method references. */
    var funcProto = Function.prototype,
        objectProto = Object.prototype;

    /** Used to detect overreaching core-js shims. */
    var coreJsData = root['__core-js_shared__'];

    /** Used to detect methods masquerading as native. */
    var maskSrcKey = (function() {
      var uid = /[^.]+$/.exec(coreJsData && coreJsData.keys && coreJsData.keys.IE_PROTO || '');
      return uid ? ('Symbol(src)_1.' + uid) : '';
    }());

    /** Used to resolve the decompiled source of functions. */
    var funcToString = funcProto.toString;

    /** Used to check objects for own properties. */
    var hasOwnProperty$1 = objectProto.hasOwnProperty;

    /**
     * Used to resolve the
     * [`toStringTag`](http://ecma-international.org/ecma-262/7.0/#sec-object.prototype.tostring)
     * of values.
     */
    var objectToString = objectProto.toString;

    /** Used to detect if a method is native. */
    var reIsNative = RegExp('^' +
      funcToString.call(hasOwnProperty$1).replace(reRegExpChar, '\\$&')
      .replace(/hasOwnProperty|(function).*?(?=\\\()| for .+?(?=\\\])/g, '$1.*?') + '$'
    );

    /** Built-in value references. */
    var Symbol$1 = root.Symbol,
        iteratorSymbol = Symbol$1 ? Symbol$1.iterator : undefined,
        propertyIsEnumerable = objectProto.propertyIsEnumerable;

    /* Built-in method references for those with the same name as other `lodash` methods. */
    var nativeKeys = overArg(Object.keys, Object);

    /* Built-in method references that are verified to be native. */
    var DataView = getNative(root, 'DataView'),
        Map$1 = getNative(root, 'Map'),
        Promise$1 = getNative(root, 'Promise'),
        Set$1 = getNative(root, 'Set'),
        WeakMap$1 = getNative(root, 'WeakMap');

    /** Used to detect maps, sets, and weakmaps. */
    var dataViewCtorString = toSource(DataView),
        mapCtorString = toSource(Map$1),
        promiseCtorString = toSource(Promise$1),
        setCtorString = toSource(Set$1),
        weakMapCtorString = toSource(WeakMap$1);

    /**
     * Creates an array of the enumerable property names of the array-like `value`.
     *
     * @private
     * @param {*} value The value to query.
     * @param {boolean} inherited Specify returning inherited property names.
     * @returns {Array} Returns the array of property names.
     */
    function arrayLikeKeys(value, inherited) {
      // Safari 8.1 makes `arguments.callee` enumerable in strict mode.
      // Safari 9 makes `arguments.length` enumerable in strict mode.
      var result = (isArray(value) || isArguments(value))
        ? baseTimes(value.length, String)
        : [];

      var length = result.length,
          skipIndexes = !!length;

      for (var key in value) {
        if ((inherited || hasOwnProperty$1.call(value, key)) &&
            !(skipIndexes && (key == 'length' || isIndex(key, length)))) {
          result.push(key);
        }
      }
      return result;
    }

    /**
     * The base implementation of `getTag`.
     *
     * @private
     * @param {*} value The value to query.
     * @returns {string} Returns the `toStringTag`.
     */
    function baseGetTag(value) {
      return objectToString.call(value);
    }

    /**
     * The base implementation of `_.isNative` without bad shim checks.
     *
     * @private
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is a native function,
     *  else `false`.
     */
    function baseIsNative(value) {
      if (!isObject(value) || isMasked(value)) {
        return false;
      }
      var pattern = (isFunction(value) || isHostObject(value)) ? reIsNative : reIsHostCtor;
      return pattern.test(toSource(value));
    }

    /**
     * The base implementation of `_.keys` which doesn't treat sparse arrays as dense.
     *
     * @private
     * @param {Object} object The object to query.
     * @returns {Array} Returns the array of property names.
     */
    function baseKeys(object) {
      if (!isPrototype(object)) {
        return nativeKeys(object);
      }
      var result = [];
      for (var key in Object(object)) {
        if (hasOwnProperty$1.call(object, key) && key != 'constructor') {
          result.push(key);
        }
      }
      return result;
    }

    /**
     * Copies the values of `source` to `array`.
     *
     * @private
     * @param {Array} source The array to copy values from.
     * @param {Array} [array=[]] The array to copy values to.
     * @returns {Array} Returns `array`.
     */
    function copyArray(source, array) {
      var index = -1,
          length = source.length;

      array || (array = Array(length));
      while (++index < length) {
        array[index] = source[index];
      }
      return array;
    }

    /**
     * Gets the native function at `key` of `object`.
     *
     * @private
     * @param {Object} object The object to query.
     * @param {string} key The key of the method to get.
     * @returns {*} Returns the function if it's native, else `undefined`.
     */
    function getNative(object, key) {
      var value = getValue(object, key);
      return baseIsNative(value) ? value : undefined;
    }

    /**
     * Gets the `toStringTag` of `value`.
     *
     * @private
     * @param {*} value The value to query.
     * @returns {string} Returns the `toStringTag`.
     */
    var getTag = baseGetTag;

    // Fallback for data views, maps, sets, and weak maps in IE 11,
    // for data views in Edge < 14, and promises in Node.js.
    if ((DataView && getTag(new DataView(new ArrayBuffer(1))) != dataViewTag) ||
        (Map$1 && getTag(new Map$1) != mapTag) ||
        (Promise$1 && getTag(Promise$1.resolve()) != promiseTag) ||
        (Set$1 && getTag(new Set$1) != setTag) ||
        (WeakMap$1 && getTag(new WeakMap$1) != weakMapTag)) {
      getTag = function(value) {
        var result = objectToString.call(value),
            Ctor = result == objectTag ? value.constructor : undefined,
            ctorString = Ctor ? toSource(Ctor) : undefined;

        if (ctorString) {
          switch (ctorString) {
            case dataViewCtorString: return dataViewTag;
            case mapCtorString: return mapTag;
            case promiseCtorString: return promiseTag;
            case setCtorString: return setTag;
            case weakMapCtorString: return weakMapTag;
          }
        }
        return result;
      };
    }

    /**
     * Checks if `value` is a valid array-like index.
     *
     * @private
     * @param {*} value The value to check.
     * @param {number} [length=MAX_SAFE_INTEGER] The upper bounds of a valid index.
     * @returns {boolean} Returns `true` if `value` is a valid index, else `false`.
     */
    function isIndex(value, length) {
      length = length == null ? MAX_SAFE_INTEGER : length;
      return !!length &&
        (typeof value == 'number' || reIsUint.test(value)) &&
        (value > -1 && value % 1 == 0 && value < length);
    }

    /**
     * Checks if `func` has its source masked.
     *
     * @private
     * @param {Function} func The function to check.
     * @returns {boolean} Returns `true` if `func` is masked, else `false`.
     */
    function isMasked(func) {
      return !!maskSrcKey && (maskSrcKey in func);
    }

    /**
     * Checks if `value` is likely a prototype object.
     *
     * @private
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is a prototype, else `false`.
     */
    function isPrototype(value) {
      var Ctor = value && value.constructor,
          proto = (typeof Ctor == 'function' && Ctor.prototype) || objectProto;

      return value === proto;
    }

    /**
     * Converts `func` to its source code.
     *
     * @private
     * @param {Function} func The function to process.
     * @returns {string} Returns the source code.
     */
    function toSource(func) {
      if (func != null) {
        try {
          return funcToString.call(func);
        } catch (e) {}
        try {
          return (func + '');
        } catch (e) {}
      }
      return '';
    }

    /**
     * Checks if `value` is likely an `arguments` object.
     *
     * @static
     * @memberOf _
     * @since 0.1.0
     * @category Lang
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is an `arguments` object,
     *  else `false`.
     * @example
     *
     * _.isArguments(function() { return arguments; }());
     * // => true
     *
     * _.isArguments([1, 2, 3]);
     * // => false
     */
    function isArguments(value) {
      // Safari 8.1 makes `arguments.callee` enumerable in strict mode.
      return isArrayLikeObject(value) && hasOwnProperty$1.call(value, 'callee') &&
        (!propertyIsEnumerable.call(value, 'callee') || objectToString.call(value) == argsTag);
    }

    /**
     * Checks if `value` is classified as an `Array` object.
     *
     * @static
     * @memberOf _
     * @since 0.1.0
     * @category Lang
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is an array, else `false`.
     * @example
     *
     * _.isArray([1, 2, 3]);
     * // => true
     *
     * _.isArray(document.body.children);
     * // => false
     *
     * _.isArray('abc');
     * // => false
     *
     * _.isArray(_.noop);
     * // => false
     */
    var isArray = Array.isArray;

    /**
     * Checks if `value` is array-like. A value is considered array-like if it's
     * not a function and has a `value.length` that's an integer greater than or
     * equal to `0` and less than or equal to `Number.MAX_SAFE_INTEGER`.
     *
     * @static
     * @memberOf _
     * @since 4.0.0
     * @category Lang
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is array-like, else `false`.
     * @example
     *
     * _.isArrayLike([1, 2, 3]);
     * // => true
     *
     * _.isArrayLike(document.body.children);
     * // => true
     *
     * _.isArrayLike('abc');
     * // => true
     *
     * _.isArrayLike(_.noop);
     * // => false
     */
    function isArrayLike(value) {
      return value != null && isLength(value.length) && !isFunction(value);
    }

    /**
     * This method is like `_.isArrayLike` except that it also checks if `value`
     * is an object.
     *
     * @static
     * @memberOf _
     * @since 4.0.0
     * @category Lang
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is an array-like object,
     *  else `false`.
     * @example
     *
     * _.isArrayLikeObject([1, 2, 3]);
     * // => true
     *
     * _.isArrayLikeObject(document.body.children);
     * // => true
     *
     * _.isArrayLikeObject('abc');
     * // => false
     *
     * _.isArrayLikeObject(_.noop);
     * // => false
     */
    function isArrayLikeObject(value) {
      return isObjectLike(value) && isArrayLike(value);
    }

    /**
     * Checks if `value` is classified as a `Function` object.
     *
     * @static
     * @memberOf _
     * @since 0.1.0
     * @category Lang
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is a function, else `false`.
     * @example
     *
     * _.isFunction(_);
     * // => true
     *
     * _.isFunction(/abc/);
     * // => false
     */
    function isFunction(value) {
      // The use of `Object#toString` avoids issues with the `typeof` operator
      // in Safari 8-9 which returns 'object' for typed array and other constructors.
      var tag = isObject(value) ? objectToString.call(value) : '';
      return tag == funcTag || tag == genTag;
    }

    /**
     * Checks if `value` is a valid array-like length.
     *
     * **Note:** This method is loosely based on
     * [`ToLength`](http://ecma-international.org/ecma-262/7.0/#sec-tolength).
     *
     * @static
     * @memberOf _
     * @since 4.0.0
     * @category Lang
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is a valid length, else `false`.
     * @example
     *
     * _.isLength(3);
     * // => true
     *
     * _.isLength(Number.MIN_VALUE);
     * // => false
     *
     * _.isLength(Infinity);
     * // => false
     *
     * _.isLength('3');
     * // => false
     */
    function isLength(value) {
      return typeof value == 'number' &&
        value > -1 && value % 1 == 0 && value <= MAX_SAFE_INTEGER;
    }

    /**
     * Checks if `value` is the
     * [language type](http://www.ecma-international.org/ecma-262/7.0/#sec-ecmascript-language-types)
     * of `Object`. (e.g. arrays, functions, objects, regexes, `new Number(0)`, and `new String('')`)
     *
     * @static
     * @memberOf _
     * @since 0.1.0
     * @category Lang
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is an object, else `false`.
     * @example
     *
     * _.isObject({});
     * // => true
     *
     * _.isObject([1, 2, 3]);
     * // => true
     *
     * _.isObject(_.noop);
     * // => true
     *
     * _.isObject(null);
     * // => false
     */
    function isObject(value) {
      var type = typeof value;
      return !!value && (type == 'object' || type == 'function');
    }

    /**
     * Checks if `value` is object-like. A value is object-like if it's not `null`
     * and has a `typeof` result of "object".
     *
     * @static
     * @memberOf _
     * @since 4.0.0
     * @category Lang
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is object-like, else `false`.
     * @example
     *
     * _.isObjectLike({});
     * // => true
     *
     * _.isObjectLike([1, 2, 3]);
     * // => true
     *
     * _.isObjectLike(_.noop);
     * // => false
     *
     * _.isObjectLike(null);
     * // => false
     */
    function isObjectLike(value) {
      return !!value && typeof value == 'object';
    }

    /**
     * Checks if `value` is classified as a `String` primitive or object.
     *
     * @static
     * @since 0.1.0
     * @memberOf _
     * @category Lang
     * @param {*} value The value to check.
     * @returns {boolean} Returns `true` if `value` is a string, else `false`.
     * @example
     *
     * _.isString('abc');
     * // => true
     *
     * _.isString(1);
     * // => false
     */
    function isString(value) {
      return typeof value == 'string' ||
        (!isArray(value) && isObjectLike(value) && objectToString.call(value) == stringTag);
    }

    /**
     * Converts `value` to an array.
     *
     * @static
     * @since 0.1.0
     * @memberOf _
     * @category Lang
     * @param {*} value The value to convert.
     * @returns {Array} Returns the converted array.
     * @example
     *
     * _.toArray({ 'a': 1, 'b': 2 });
     * // => [1, 2]
     *
     * _.toArray('abc');
     * // => ['a', 'b', 'c']
     *
     * _.toArray(1);
     * // => []
     *
     * _.toArray(null);
     * // => []
     */
    function toArray(value) {
      if (!value) {
        return [];
      }
      if (isArrayLike(value)) {
        return isString(value) ? stringToArray(value) : copyArray(value);
      }
      if (iteratorSymbol && value[iteratorSymbol]) {
        return iteratorToArray(value[iteratorSymbol]());
      }
      var tag = getTag(value),
          func = tag == mapTag ? mapToArray : (tag == setTag ? setToArray : values);

      return func(value);
    }

    /**
     * Creates an array of the own enumerable property names of `object`.
     *
     * **Note:** Non-object values are coerced to objects. See the
     * [ES spec](http://ecma-international.org/ecma-262/7.0/#sec-object.keys)
     * for more details.
     *
     * @static
     * @since 0.1.0
     * @memberOf _
     * @category Object
     * @param {Object} object The object to query.
     * @returns {Array} Returns the array of property names.
     * @example
     *
     * function Foo() {
     *   this.a = 1;
     *   this.b = 2;
     * }
     *
     * Foo.prototype.c = 3;
     *
     * _.keys(new Foo);
     * // => ['a', 'b'] (iteration order is not guaranteed)
     *
     * _.keys('hi');
     * // => ['0', '1']
     */
    function keys(object) {
      return isArrayLike(object) ? arrayLikeKeys(object) : baseKeys(object);
    }

    /**
     * Creates an array of the own enumerable string keyed property values of `object`.
     *
     * **Note:** Non-object values are coerced to objects.
     *
     * @static
     * @since 0.1.0
     * @memberOf _
     * @category Object
     * @param {Object} object The object to query.
     * @returns {Array} Returns the array of property values.
     * @example
     *
     * function Foo() {
     *   this.a = 1;
     *   this.b = 2;
     * }
     *
     * Foo.prototype.c = 3;
     *
     * _.values(new Foo);
     * // => [1, 2] (iteration order is not guaranteed)
     *
     * _.values('hi');
     * // => ['h', 'i']
     */
    function values(object) {
      return object ? baseValues(object, keys(object)) : [];
    }

    var lodash_toarray = toArray;

    var umbrella_with_rain_drops = "☔";
    var coffee = "☕";
    var aries = "♈";
    var taurus = "♉";
    var sagittarius = "♐";
    var capricorn = "♑";
    var aquarius = "♒";
    var pisces = "♓";
    var anchor = "⚓";
    var white_check_mark = "✅";
    var sparkles = "✨";
    var question = "❓";
    var grey_question = "❔";
    var grey_exclamation = "❕";
    var exclamation = "❗";
    var heavy_exclamation_mark = "❗";
    var heavy_plus_sign = "➕";
    var heavy_minus_sign = "➖";
    var heavy_division_sign = "➗";
    var hash = "#️⃣";
    var keycap_star = "*️⃣";
    var zero = "0️⃣";
    var one = "1️⃣";
    var two = "2️⃣";
    var three = "3️⃣";
    var four = "4️⃣";
    var five = "5️⃣";
    var six = "6️⃣";
    var seven = "7️⃣";
    var eight = "8️⃣";
    var nine = "9️⃣";
    var copyright = "©️";
    var registered = "®️";
    var mahjong = "🀄";
    var black_joker = "🃏";
    var a = "🅰️";
    var b = "🅱️";
    var o2 = "🅾️";
    var parking = "🅿️";
    var ab = "🆎";
    var cl = "🆑";
    var cool = "🆒";
    var free = "🆓";
    var id = "🆔";
    var ng = "🆖";
    var ok = "🆗";
    var sos = "🆘";
    var up = "🆙";
    var vs = "🆚";
    var koko = "🈁";
    var sa = "🈂️";
    var u7121 = "🈚";
    var u6307 = "🈯";
    var u7981 = "🈲";
    var u7a7a = "🈳";
    var u5408 = "🈴";
    var u6e80 = "🈵";
    var u6709 = "🈶";
    var u6708 = "🈷️";
    var u7533 = "🈸";
    var u5272 = "🈹";
    var u55b6 = "🈺";
    var ideograph_advantage = "🉐";
    var accept = "🉑";
    var cyclone = "🌀";
    var foggy = "🌁";
    var closed_umbrella = "🌂";
    var night_with_stars = "🌃";
    var sunrise_over_mountains = "🌄";
    var sunrise = "🌅";
    var city_sunset = "🌆";
    var city_sunrise = "🌇";
    var rainbow = "🌈";
    var bridge_at_night = "🌉";
    var ocean = "🌊";
    var volcano = "🌋";
    var milky_way = "🌌";
    var earth_africa = "🌍";
    var earth_americas = "🌎";
    var earth_asia = "🌏";
    var globe_with_meridians = "🌐";
    var new_moon = "🌑";
    var waxing_crescent_moon = "🌒";
    var first_quarter_moon = "🌓";
    var moon = "🌔";
    var waxing_gibbous_moon = "🌔";
    var full_moon = "🌕";
    var waning_gibbous_moon = "🌖";
    var last_quarter_moon = "🌗";
    var waning_crescent_moon = "🌘";
    var crescent_moon = "🌙";
    var new_moon_with_face = "🌚";
    var first_quarter_moon_with_face = "🌛";
    var last_quarter_moon_with_face = "🌜";
    var full_moon_with_face = "🌝";
    var sun_with_face = "🌞";
    var star2 = "🌟";
    var stars = "🌠";
    var thermometer = "🌡️";
    var mostly_sunny = "🌤️";
    var sun_small_cloud = "🌤️";
    var barely_sunny = "🌥️";
    var sun_behind_cloud = "🌥️";
    var partly_sunny_rain = "🌦️";
    var sun_behind_rain_cloud = "🌦️";
    var rain_cloud = "🌧️";
    var snow_cloud = "🌨️";
    var lightning = "🌩️";
    var lightning_cloud = "🌩️";
    var tornado = "🌪️";
    var tornado_cloud = "🌪️";
    var fog = "🌫️";
    var wind_blowing_face = "🌬️";
    var hotdog = "🌭";
    var taco = "🌮";
    var burrito = "🌯";
    var chestnut = "🌰";
    var seedling = "🌱";
    var evergreen_tree = "🌲";
    var deciduous_tree = "🌳";
    var palm_tree = "🌴";
    var cactus = "🌵";
    var hot_pepper = "🌶️";
    var tulip = "🌷";
    var cherry_blossom = "🌸";
    var rose = "🌹";
    var hibiscus = "🌺";
    var sunflower = "🌻";
    var blossom = "🌼";
    var corn = "🌽";
    var ear_of_rice = "🌾";
    var herb = "🌿";
    var four_leaf_clover = "🍀";
    var maple_leaf = "🍁";
    var fallen_leaf = "🍂";
    var leaves = "🍃";
    var mushroom = "🍄";
    var tomato = "🍅";
    var eggplant = "🍆";
    var grapes = "🍇";
    var melon = "🍈";
    var watermelon = "🍉";
    var tangerine = "🍊";
    var lemon = "🍋";
    var banana = "🍌";
    var pineapple = "🍍";
    var apple = "🍎";
    var green_apple = "🍏";
    var pear = "🍐";
    var peach = "🍑";
    var cherries = "🍒";
    var strawberry = "🍓";
    var hamburger = "🍔";
    var pizza = "🍕";
    var meat_on_bone = "🍖";
    var poultry_leg = "🍗";
    var rice_cracker = "🍘";
    var rice_ball = "🍙";
    var rice = "🍚";
    var curry = "🍛";
    var ramen = "🍜";
    var spaghetti = "🍝";
    var bread = "🍞";
    var fries = "🍟";
    var sweet_potato = "🍠";
    var dango = "🍡";
    var oden = "🍢";
    var sushi = "🍣";
    var fried_shrimp = "🍤";
    var fish_cake = "🍥";
    var icecream = "🍦";
    var shaved_ice = "🍧";
    var ice_cream = "🍨";
    var doughnut = "🍩";
    var cookie = "🍪";
    var chocolate_bar = "🍫";
    var candy = "🍬";
    var lollipop = "🍭";
    var custard = "🍮";
    var honey_pot = "🍯";
    var cake = "🍰";
    var bento = "🍱";
    var stew = "🍲";
    var fried_egg = "🍳";
    var cooking = "🍳";
    var fork_and_knife = "🍴";
    var tea = "🍵";
    var sake = "🍶";
    var wine_glass = "🍷";
    var cocktail = "🍸";
    var tropical_drink = "🍹";
    var beer = "🍺";
    var beers = "🍻";
    var baby_bottle = "🍼";
    var knife_fork_plate = "🍽️";
    var champagne = "🍾";
    var popcorn = "🍿";
    var ribbon = "🎀";
    var gift = "🎁";
    var birthday = "🎂";
    var jack_o_lantern = "🎃";
    var christmas_tree = "🎄";
    var santa = "🎅";
    var fireworks = "🎆";
    var sparkler = "🎇";
    var balloon = "🎈";
    var tada = "🎉";
    var confetti_ball = "🎊";
    var tanabata_tree = "🎋";
    var crossed_flags = "🎌";
    var bamboo = "🎍";
    var dolls = "🎎";
    var flags = "🎏";
    var wind_chime = "🎐";
    var rice_scene = "🎑";
    var school_satchel = "🎒";
    var mortar_board = "🎓";
    var medal = "🎖️";
    var reminder_ribbon = "🎗️";
    var studio_microphone = "🎙️";
    var level_slider = "🎚️";
    var control_knobs = "🎛️";
    var film_frames = "🎞️";
    var admission_tickets = "🎟️";
    var carousel_horse = "🎠";
    var ferris_wheel = "🎡";
    var roller_coaster = "🎢";
    var fishing_pole_and_fish = "🎣";
    var microphone = "🎤";
    var movie_camera = "🎥";
    var cinema = "🎦";
    var headphones = "🎧";
    var art = "🎨";
    var tophat = "🎩";
    var circus_tent = "🎪";
    var ticket = "🎫";
    var clapper = "🎬";
    var performing_arts = "🎭";
    var video_game = "🎮";
    var dart = "🎯";
    var slot_machine = "🎰";
    var game_die = "🎲";
    var bowling = "🎳";
    var flower_playing_cards = "🎴";
    var musical_note = "🎵";
    var notes = "🎶";
    var saxophone = "🎷";
    var guitar = "🎸";
    var musical_keyboard = "🎹";
    var trumpet = "🎺";
    var violin = "🎻";
    var musical_score = "🎼";
    var running_shirt_with_sash = "🎽";
    var tennis = "🎾";
    var ski = "🎿";
    var basketball = "🏀";
    var checkered_flag = "🏁";
    var snowboarder = "🏂";
    var runner = "🏃‍♂️";
    var running = "🏃‍♂️";
    var surfer = "🏄‍♂️";
    var sports_medal = "🏅";
    var trophy = "🏆";
    var horse_racing = "🏇";
    var football = "🏈";
    var rugby_football = "🏉";
    var swimmer = "🏊‍♂️";
    var weight_lifter = "🏋️‍♂️";
    var golfer = "🏌️‍♂️";
    var racing_motorcycle = "🏍️";
    var racing_car = "🏎️";
    var cricket_bat_and_ball = "🏏";
    var volleyball = "🏐";
    var field_hockey_stick_and_ball = "🏑";
    var ice_hockey_stick_and_puck = "🏒";
    var table_tennis_paddle_and_ball = "🏓";
    var snow_capped_mountain = "🏔️";
    var camping = "🏕️";
    var beach_with_umbrella = "🏖️";
    var building_construction = "🏗️";
    var house_buildings = "🏘️";
    var cityscape = "🏙️";
    var derelict_house_building = "🏚️";
    var classical_building = "🏛️";
    var desert = "🏜️";
    var desert_island = "🏝️";
    var national_park = "🏞️";
    var stadium = "🏟️";
    var house = "🏠";
    var house_with_garden = "🏡";
    var office = "🏢";
    var post_office = "🏣";
    var european_post_office = "🏤";
    var hospital = "🏥";
    var bank = "🏦";
    var atm = "🏧";
    var hotel = "🏨";
    var love_hotel = "🏩";
    var convenience_store = "🏪";
    var school = "🏫";
    var department_store = "🏬";
    var factory = "🏭";
    var izakaya_lantern = "🏮";
    var lantern = "🏮";
    var japanese_castle = "🏯";
    var european_castle = "🏰";
    var waving_white_flag = "🏳️";
    var waving_black_flag = "🏴";
    var rosette = "🏵️";
    var label = "🏷️";
    var badminton_racquet_and_shuttlecock = "🏸";
    var bow_and_arrow = "🏹";
    var amphora = "🏺";
    var rat = "🐀";
    var mouse2 = "🐁";
    var ox = "🐂";
    var water_buffalo = "🐃";
    var cow2 = "🐄";
    var tiger2 = "🐅";
    var leopard = "🐆";
    var rabbit2 = "🐇";
    var cat2 = "🐈";
    var dragon = "🐉";
    var crocodile = "🐊";
    var whale2 = "🐋";
    var snail = "🐌";
    var snake = "🐍";
    var racehorse = "🐎";
    var ram = "🐏";
    var goat = "🐐";
    var sheep = "🐑";
    var monkey = "🐒";
    var rooster = "🐓";
    var chicken = "🐔";
    var dog2 = "🐕";
    var pig2 = "🐖";
    var boar = "🐗";
    var elephant = "🐘";
    var octopus = "🐙";
    var shell = "🐚";
    var bug = "🐛";
    var ant = "🐜";
    var bee = "🐝";
    var honeybee = "🐝";
    var beetle = "🐞";
    var fish = "🐟";
    var tropical_fish = "🐠";
    var blowfish = "🐡";
    var turtle = "🐢";
    var hatching_chick = "🐣";
    var baby_chick = "🐤";
    var hatched_chick = "🐥";
    var bird = "🐦";
    var penguin = "🐧";
    var koala = "🐨";
    var poodle = "🐩";
    var dromedary_camel = "🐪";
    var camel = "🐫";
    var dolphin = "🐬";
    var flipper = "🐬";
    var mouse = "🐭";
    var cow = "🐮";
    var tiger = "🐯";
    var rabbit = "🐰";
    var cat = "🐱";
    var dragon_face = "🐲";
    var whale = "🐳";
    var horse = "🐴";
    var monkey_face = "🐵";
    var dog = "🐶";
    var pig = "🐷";
    var frog = "🐸";
    var hamster = "🐹";
    var wolf = "🐺";
    var bear = "🐻";
    var panda_face = "🐼";
    var pig_nose = "🐽";
    var feet = "🐾";
    var paw_prints = "🐾";
    var chipmunk = "🐿️";
    var eyes = "👀";
    var eye = "👁️";
    var ear = "👂";
    var nose = "👃";
    var lips = "👄";
    var tongue = "👅";
    var point_up_2 = "👆";
    var point_down = "👇";
    var point_left = "👈";
    var point_right = "👉";
    var facepunch = "👊";
    var punch = "👊";
    var wave = "👋";
    var ok_hand = "👌";
    var thumbsup = "👍";
    var thumbsdown = "👎";
    var clap = "👏";
    var open_hands = "👐";
    var crown = "👑";
    var womans_hat = "👒";
    var eyeglasses = "👓";
    var necktie = "👔";
    var shirt = "👕";
    var tshirt = "👕";
    var jeans = "👖";
    var dress = "👗";
    var kimono = "👘";
    var bikini = "👙";
    var womans_clothes = "👚";
    var purse = "👛";
    var handbag = "👜";
    var pouch = "👝";
    var mans_shoe = "👞";
    var shoe = "👞";
    var athletic_shoe = "👟";
    var high_heel = "👠";
    var sandal = "👡";
    var boot = "👢";
    var footprints = "👣";
    var bust_in_silhouette = "👤";
    var busts_in_silhouette = "👥";
    var boy = "👦";
    var girl = "👧";
    var family = "👨‍👩‍👦";
    var man = "👨";
    var couple_with_heart = "👩‍❤️‍👨";
    var couplekiss = "👩‍❤️‍💋‍👨";
    var woman = "👩";
    var couple = "👫";
    var man_and_woman_holding_hands = "👫";
    var two_men_holding_hands = "👬";
    var two_women_holding_hands = "👭";
    var cop = "👮‍♂️";
    var dancers = "👯‍♀️";
    var bride_with_veil = "👰";
    var person_with_blond_hair = "👱‍♂️";
    var man_with_gua_pi_mao = "👲";
    var man_with_turban = "👳‍♂️";
    var older_man = "👴";
    var older_woman = "👵";
    var baby = "👶";
    var construction_worker = "👷‍♂️";
    var princess = "👸";
    var japanese_ogre = "👹";
    var japanese_goblin = "👺";
    var ghost = "👻";
    var angel = "👼";
    var alien = "👽";
    var space_invader = "👾";
    var imp = "👿";
    var skull = "💀";
    var information_desk_person = "💁‍♀️";
    var guardsman = "💂‍♂️";
    var dancer = "💃";
    var lipstick = "💄";
    var nail_care = "💅";
    var massage = "💆‍♀️";
    var haircut = "💇‍♀️";
    var barber = "💈";
    var syringe = "💉";
    var pill = "💊";
    var kiss = "💋";
    var love_letter = "💌";
    var ring = "💍";
    var gem = "💎";
    var bouquet = "💐";
    var wedding = "💒";
    var heartbeat = "💓";
    var broken_heart = "💔";
    var two_hearts = "💕";
    var sparkling_heart = "💖";
    var heartpulse = "💗";
    var cupid = "💘";
    var blue_heart = "💙";
    var green_heart = "💚";
    var yellow_heart = "💛";
    var purple_heart = "💜";
    var gift_heart = "💝";
    var revolving_hearts = "💞";
    var heart_decoration = "💟";
    var diamond_shape_with_a_dot_inside = "💠";
    var bulb = "💡";
    var anger = "💢";
    var bomb = "💣";
    var zzz = "💤";
    var boom = "💥";
    var collision = "💥";
    var sweat_drops = "💦";
    var droplet = "💧";
    var dash = "💨";
    var poop = "💩";
    var muscle = "💪";
    var dizzy = "💫";
    var speech_balloon = "💬";
    var thought_balloon = "💭";
    var white_flower = "💮";
    var moneybag = "💰";
    var currency_exchange = "💱";
    var heavy_dollar_sign = "💲";
    var credit_card = "💳";
    var yen = "💴";
    var dollar = "💵";
    var euro = "💶";
    var pound = "💷";
    var money_with_wings = "💸";
    var chart = "💹";
    var seat = "💺";
    var computer = "💻";
    var briefcase = "💼";
    var minidisc = "💽";
    var floppy_disk = "💾";
    var cd = "💿";
    var dvd = "📀";
    var file_folder = "📁";
    var open_file_folder = "📂";
    var page_with_curl = "📃";
    var page_facing_up = "📄";
    var date = "📅";
    var calendar = "📆";
    var card_index = "📇";
    var chart_with_upwards_trend = "📈";
    var chart_with_downwards_trend = "📉";
    var bar_chart = "📊";
    var clipboard = "📋";
    var pushpin = "📌";
    var round_pushpin = "📍";
    var paperclip = "📎";
    var straight_ruler = "📏";
    var triangular_ruler = "📐";
    var bookmark_tabs = "📑";
    var ledger = "📒";
    var notebook = "📓";
    var notebook_with_decorative_cover = "📔";
    var closed_book = "📕";
    var book = "📖";
    var open_book = "📖";
    var green_book = "📗";
    var blue_book = "📘";
    var orange_book = "📙";
    var books = "📚";
    var name_badge = "📛";
    var scroll = "📜";
    var memo = "📝";
    var pencil = "📝";
    var telephone_receiver = "📞";
    var pager = "📟";
    var fax = "📠";
    var satellite_antenna = "📡";
    var loudspeaker = "📢";
    var mega = "📣";
    var outbox_tray = "📤";
    var inbox_tray = "📥";
    var incoming_envelope = "📨";
    var envelope_with_arrow = "📩";
    var mailbox_closed = "📪";
    var mailbox = "📫";
    var mailbox_with_mail = "📬";
    var mailbox_with_no_mail = "📭";
    var postbox = "📮";
    var postal_horn = "📯";
    var newspaper = "📰";
    var iphone = "📱";
    var calling = "📲";
    var vibration_mode = "📳";
    var mobile_phone_off = "📴";
    var no_mobile_phones = "📵";
    var signal_strength = "📶";
    var camera = "📷";
    var camera_with_flash = "📸";
    var video_camera = "📹";
    var tv = "📺";
    var radio = "📻";
    var vhs = "📼";
    var film_projector = "📽️";
    var prayer_beads = "📿";
    var twisted_rightwards_arrows = "🔀";
    var repeat = "🔁";
    var repeat_one = "🔂";
    var arrows_clockwise = "🔃";
    var arrows_counterclockwise = "🔄";
    var low_brightness = "🔅";
    var high_brightness = "🔆";
    var mute = "🔇";
    var speaker = "🔈";
    var sound = "🔉";
    var loud_sound = "🔊";
    var battery = "🔋";
    var electric_plug = "🔌";
    var mag = "🔍";
    var mag_right = "🔎";
    var lock_with_ink_pen = "🔏";
    var closed_lock_with_key = "🔐";
    var key = "🔑";
    var lock = "🔒";
    var unlock = "🔓";
    var bell = "🔔";
    var no_bell = "🔕";
    var bookmark = "🔖";
    var link = "🔗";
    var radio_button = "🔘";
    var back = "🔙";
    var end = "🔚";
    var on = "🔛";
    var soon = "🔜";
    var underage = "🔞";
    var keycap_ten = "🔟";
    var capital_abcd = "🔠";
    var abcd = "🔡";
    var symbols = "🔣";
    var abc = "🔤";
    var fire = "🔥";
    var flashlight = "🔦";
    var wrench = "🔧";
    var hammer = "🔨";
    var nut_and_bolt = "🔩";
    var hocho = "🔪";
    var knife = "🔪";
    var gun = "🔫";
    var microscope = "🔬";
    var telescope = "🔭";
    var crystal_ball = "🔮";
    var six_pointed_star = "🔯";
    var beginner = "🔰";
    var trident = "🔱";
    var black_square_button = "🔲";
    var white_square_button = "🔳";
    var red_circle = "🔴";
    var large_blue_circle = "🔵";
    var large_orange_diamond = "🔶";
    var large_blue_diamond = "🔷";
    var small_orange_diamond = "🔸";
    var small_blue_diamond = "🔹";
    var small_red_triangle = "🔺";
    var small_red_triangle_down = "🔻";
    var arrow_up_small = "🔼";
    var arrow_down_small = "🔽";
    var om_symbol = "🕉️";
    var dove_of_peace = "🕊️";
    var kaaba = "🕋";
    var mosque = "🕌";
    var synagogue = "🕍";
    var menorah_with_nine_branches = "🕎";
    var candle = "🕯️";
    var mantelpiece_clock = "🕰️";
    var hole = "🕳️";
    var man_in_business_suit_levitating = "🕴️";
    var sleuth_or_spy = "🕵️‍♂️";
    var dark_sunglasses = "🕶️";
    var spider = "🕷️";
    var spider_web = "🕸️";
    var joystick = "🕹️";
    var man_dancing = "🕺";
    var linked_paperclips = "🖇️";
    var lower_left_ballpoint_pen = "🖊️";
    var lower_left_fountain_pen = "🖋️";
    var lower_left_paintbrush = "🖌️";
    var lower_left_crayon = "🖍️";
    var raised_hand_with_fingers_splayed = "🖐️";
    var middle_finger = "🖕";
    var reversed_hand_with_middle_finger_extended = "🖕";
    var black_heart = "🖤";
    var desktop_computer = "🖥️";
    var printer = "🖨️";
    var three_button_mouse = "🖱️";
    var trackball = "🖲️";
    var frame_with_picture = "🖼️";
    var card_index_dividers = "🗂️";
    var card_file_box = "🗃️";
    var file_cabinet = "🗄️";
    var wastebasket = "🗑️";
    var spiral_note_pad = "🗒️";
    var spiral_calendar_pad = "🗓️";
    var compression = "🗜️";
    var old_key = "🗝️";
    var rolled_up_newspaper = "🗞️";
    var dagger_knife = "🗡️";
    var speaking_head_in_silhouette = "🗣️";
    var left_speech_bubble = "🗨️";
    var right_anger_bubble = "🗯️";
    var ballot_box_with_ballot = "🗳️";
    var world_map = "🗺️";
    var mount_fuji = "🗻";
    var tokyo_tower = "🗼";
    var statue_of_liberty = "🗽";
    var japan = "🗾";
    var moyai = "🗿";
    var grinning = "😀";
    var grin = "😁";
    var joy = "😂";
    var smiley = "😃";
    var smile = "😄";
    var sweat_smile = "😅";
    var laughing = "😆";
    var satisfied = "😆";
    var innocent = "😇";
    var smiling_imp = "😈";
    var wink = "😉";
    var blush = "😊";
    var yum = "😋";
    var relieved = "😌";
    var heart_eyes = "😍";
    var sunglasses = "😎";
    var smirk = "😏";
    var neutral_face = "😐";
    var expressionless = "😑";
    var unamused = "😒";
    var sweat = "😓";
    var pensive = "😔";
    var confused = "😕";
    var confounded = "😖";
    var kissing = "😗";
    var kissing_heart = "😘";
    var kissing_smiling_eyes = "😙";
    var kissing_closed_eyes = "😚";
    var stuck_out_tongue = "😛";
    var stuck_out_tongue_winking_eye = "😜";
    var stuck_out_tongue_closed_eyes = "😝";
    var disappointed = "😞";
    var worried = "😟";
    var angry = "😠";
    var rage = "😡";
    var cry = "😢";
    var persevere = "😣";
    var triumph = "😤";
    var disappointed_relieved = "😥";
    var frowning = "😦";
    var anguished = "😧";
    var fearful = "😨";
    var weary = "😩";
    var sleepy = "😪";
    var tired_face = "😫";
    var grimacing = "😬";
    var sob = "😭";
    var open_mouth = "😮";
    var hushed = "😯";
    var cold_sweat = "😰";
    var scream = "😱";
    var astonished = "😲";
    var flushed = "😳";
    var sleeping = "😴";
    var dizzy_face = "😵";
    var no_mouth = "😶";
    var mask = "😷";
    var smile_cat = "😸";
    var joy_cat = "😹";
    var smiley_cat = "😺";
    var heart_eyes_cat = "😻";
    var smirk_cat = "😼";
    var kissing_cat = "😽";
    var pouting_cat = "😾";
    var crying_cat_face = "😿";
    var scream_cat = "🙀";
    var slightly_frowning_face = "🙁";
    var slightly_smiling_face = "🙂";
    var upside_down_face = "🙃";
    var face_with_rolling_eyes = "🙄";
    var no_good = "🙅‍♀️";
    var ok_woman = "🙆‍♀️";
    var bow = "🙇‍♂️";
    var see_no_evil = "🙈";
    var hear_no_evil = "🙉";
    var speak_no_evil = "🙊";
    var raising_hand = "🙋‍♀️";
    var raised_hands = "🙌";
    var person_frowning = "🙍‍♀️";
    var person_with_pouting_face = "🙎‍♀️";
    var pray = "🙏";
    var rocket = "🚀";
    var helicopter = "🚁";
    var steam_locomotive = "🚂";
    var railway_car = "🚃";
    var bullettrain_side = "🚄";
    var bullettrain_front = "🚅";
    var train2 = "🚆";
    var metro = "🚇";
    var light_rail = "🚈";
    var station = "🚉";
    var tram = "🚊";
    var train = "🚋";
    var bus = "🚌";
    var oncoming_bus = "🚍";
    var trolleybus = "🚎";
    var busstop = "🚏";
    var minibus = "🚐";
    var ambulance = "🚑";
    var fire_engine = "🚒";
    var police_car = "🚓";
    var oncoming_police_car = "🚔";
    var taxi = "🚕";
    var oncoming_taxi = "🚖";
    var car = "🚗";
    var red_car = "🚗";
    var oncoming_automobile = "🚘";
    var blue_car = "🚙";
    var truck = "🚚";
    var articulated_lorry = "🚛";
    var tractor = "🚜";
    var monorail = "🚝";
    var mountain_railway = "🚞";
    var suspension_railway = "🚟";
    var mountain_cableway = "🚠";
    var aerial_tramway = "🚡";
    var ship = "🚢";
    var rowboat = "🚣‍♂️";
    var speedboat = "🚤";
    var traffic_light = "🚥";
    var vertical_traffic_light = "🚦";
    var construction = "🚧";
    var rotating_light = "🚨";
    var triangular_flag_on_post = "🚩";
    var door = "🚪";
    var no_entry_sign = "🚫";
    var smoking = "🚬";
    var no_smoking = "🚭";
    var put_litter_in_its_place = "🚮";
    var do_not_litter = "🚯";
    var potable_water = "🚰";
    var bike = "🚲";
    var no_bicycles = "🚳";
    var bicyclist = "🚴‍♂️";
    var mountain_bicyclist = "🚵‍♂️";
    var walking = "🚶‍♂️";
    var no_pedestrians = "🚷";
    var children_crossing = "🚸";
    var mens = "🚹";
    var womens = "🚺";
    var restroom = "🚻";
    var baby_symbol = "🚼";
    var toilet = "🚽";
    var wc = "🚾";
    var shower = "🚿";
    var bath = "🛀";
    var bathtub = "🛁";
    var passport_control = "🛂";
    var customs = "🛃";
    var baggage_claim = "🛄";
    var left_luggage = "🛅";
    var couch_and_lamp = "🛋️";
    var sleeping_accommodation = "🛌";
    var shopping_bags = "🛍️";
    var bellhop_bell = "🛎️";
    var bed = "🛏️";
    var place_of_worship = "🛐";
    var octagonal_sign = "🛑";
    var shopping_trolley = "🛒";
    var hammer_and_wrench = "🛠️";
    var shield = "🛡️";
    var oil_drum = "🛢️";
    var motorway = "🛣️";
    var railway_track = "🛤️";
    var motor_boat = "🛥️";
    var small_airplane = "🛩️";
    var airplane_departure = "🛫";
    var airplane_arriving = "🛬";
    var satellite = "🛰️";
    var passenger_ship = "🛳️";
    var scooter = "🛴";
    var motor_scooter = "🛵";
    var canoe = "🛶";
    var sled = "🛷";
    var flying_saucer = "🛸";
    var zipper_mouth_face = "🤐";
    var money_mouth_face = "🤑";
    var face_with_thermometer = "🤒";
    var nerd_face = "🤓";
    var thinking_face = "🤔";
    var face_with_head_bandage = "🤕";
    var robot_face = "🤖";
    var hugging_face = "🤗";
    var the_horns = "🤘";
    var sign_of_the_horns = "🤘";
    var call_me_hand = "🤙";
    var raised_back_of_hand = "🤚";
    var handshake = "🤝";
    var crossed_fingers = "🤞";
    var hand_with_index_and_middle_fingers_crossed = "🤞";
    var i_love_you_hand_sign = "🤟";
    var face_with_cowboy_hat = "🤠";
    var clown_face = "🤡";
    var nauseated_face = "🤢";
    var rolling_on_the_floor_laughing = "🤣";
    var drooling_face = "🤤";
    var lying_face = "🤥";
    var face_palm = "🤦";
    var sneezing_face = "🤧";
    var face_with_raised_eyebrow = "🤨";
    var face_with_one_eyebrow_raised = "🤨";
    var grinning_face_with_star_eyes = "🤩";
    var zany_face = "🤪";
    var grinning_face_with_one_large_and_one_small_eye = "🤪";
    var shushing_face = "🤫";
    var face_with_finger_covering_closed_lips = "🤫";
    var face_with_symbols_on_mouth = "🤬";
    var serious_face_with_symbols_covering_mouth = "🤬";
    var face_with_hand_over_mouth = "🤭";
    var smiling_face_with_smiling_eyes_and_hand_covering_mouth = "🤭";
    var face_vomiting = "🤮";
    var face_with_open_mouth_vomiting = "🤮";
    var exploding_head = "🤯";
    var shocked_face_with_exploding_head = "🤯";
    var pregnant_woman = "🤰";
    var palms_up_together = "🤲";
    var selfie = "🤳";
    var prince = "🤴";
    var man_in_tuxedo = "🤵";
    var mrs_claus = "🤶";
    var mother_christmas = "🤶";
    var shrug = "🤷";
    var person_doing_cartwheel = "🤸";
    var juggling = "🤹";
    var fencer = "🤺";
    var wrestlers = "🤼";
    var water_polo = "🤽";
    var handball = "🤾";
    var wilted_flower = "🥀";
    var drum_with_drumsticks = "🥁";
    var clinking_glasses = "🥂";
    var tumbler_glass = "🥃";
    var spoon = "🥄";
    var goal_net = "🥅";
    var first_place_medal = "🥇";
    var second_place_medal = "🥈";
    var third_place_medal = "🥉";
    var boxing_glove = "🥊";
    var martial_arts_uniform = "🥋";
    var curling_stone = "🥌";
    var croissant = "🥐";
    var avocado = "🥑";
    var cucumber = "🥒";
    var bacon = "🥓";
    var potato = "🥔";
    var carrot = "🥕";
    var baguette_bread = "🥖";
    var green_salad = "🥗";
    var shallow_pan_of_food = "🥘";
    var stuffed_flatbread = "🥙";
    var egg = "🥚";
    var glass_of_milk = "🥛";
    var peanuts = "🥜";
    var kiwifruit = "🥝";
    var pancakes = "🥞";
    var dumpling = "🥟";
    var fortune_cookie = "🥠";
    var takeout_box = "🥡";
    var chopsticks = "🥢";
    var bowl_with_spoon = "🥣";
    var cup_with_straw = "🥤";
    var coconut = "🥥";
    var broccoli = "🥦";
    var pie = "🥧";
    var pretzel = "🥨";
    var cut_of_meat = "🥩";
    var sandwich = "🥪";
    var canned_food = "🥫";
    var crab = "🦀";
    var lion_face = "🦁";
    var scorpion = "🦂";
    var turkey = "🦃";
    var unicorn_face = "🦄";
    var eagle = "🦅";
    var duck = "🦆";
    var bat = "🦇";
    var shark = "🦈";
    var owl = "🦉";
    var fox_face = "🦊";
    var butterfly = "🦋";
    var deer = "🦌";
    var gorilla = "🦍";
    var lizard = "🦎";
    var rhinoceros = "🦏";
    var shrimp = "🦐";
    var squid = "🦑";
    var giraffe_face = "🦒";
    var zebra_face = "🦓";
    var hedgehog = "🦔";
    var sauropod = "🦕";
    var cricket = "🦗";
    var cheese_wedge = "🧀";
    var face_with_monocle = "🧐";
    var adult = "🧑";
    var child = "🧒";
    var older_adult = "🧓";
    var bearded_person = "🧔";
    var person_with_headscarf = "🧕";
    var woman_in_steamy_room = "🧖‍♀️";
    var man_in_steamy_room = "🧖‍♂️";
    var person_in_steamy_room = "🧖‍♂️";
    var woman_climbing = "🧗‍♀️";
    var person_climbing = "🧗‍♀️";
    var man_climbing = "🧗‍♂️";
    var woman_in_lotus_position = "🧘‍♀️";
    var person_in_lotus_position = "🧘‍♀️";
    var man_in_lotus_position = "🧘‍♂️";
    var female_mage = "🧙‍♀️";
    var mage = "🧙‍♀️";
    var male_mage = "🧙‍♂️";
    var female_fairy = "🧚‍♀️";
    var fairy = "🧚‍♀️";
    var male_fairy = "🧚‍♂️";
    var female_vampire = "🧛‍♀️";
    var vampire = "🧛‍♀️";
    var male_vampire = "🧛‍♂️";
    var mermaid = "🧜‍♀️";
    var merman = "🧜‍♂️";
    var merperson = "🧜‍♂️";
    var female_elf = "🧝‍♀️";
    var male_elf = "🧝‍♂️";
    var elf = "🧝‍♂️";
    var female_genie = "🧞‍♀️";
    var male_genie = "🧞‍♂️";
    var genie = "🧞‍♂️";
    var female_zombie = "🧟‍♀️";
    var male_zombie = "🧟‍♂️";
    var zombie = "🧟‍♂️";
    var brain = "🧠";
    var orange_heart = "🧡";
    var billed_cap = "🧢";
    var scarf = "🧣";
    var gloves = "🧤";
    var coat = "🧥";
    var socks = "🧦";
    var bangbang = "‼️";
    var interrobang = "⁉️";
    var tm = "™️";
    var information_source = "ℹ️";
    var left_right_arrow = "↔️";
    var arrow_up_down = "↕️";
    var arrow_upper_left = "↖️";
    var arrow_upper_right = "↗️";
    var arrow_lower_right = "↘️";
    var arrow_lower_left = "↙️";
    var leftwards_arrow_with_hook = "↩️";
    var arrow_right_hook = "↪️";
    var watch = "⌚";
    var hourglass = "⌛";
    var keyboard = "⌨️";
    var eject = "⏏️";
    var fast_forward = "⏩";
    var rewind = "⏪";
    var arrow_double_up = "⏫";
    var arrow_double_down = "⏬";
    var black_right_pointing_double_triangle_with_vertical_bar = "⏭️";
    var black_left_pointing_double_triangle_with_vertical_bar = "⏮️";
    var black_right_pointing_triangle_with_double_vertical_bar = "⏯️";
    var alarm_clock = "⏰";
    var stopwatch = "⏱️";
    var timer_clock = "⏲️";
    var hourglass_flowing_sand = "⏳";
    var double_vertical_bar = "⏸️";
    var black_square_for_stop = "⏹️";
    var black_circle_for_record = "⏺️";
    var m = "Ⓜ️";
    var black_small_square = "▪️";
    var white_small_square = "▫️";
    var arrow_forward = "▶️";
    var arrow_backward = "◀️";
    var white_medium_square = "◻️";
    var black_medium_square = "◼️";
    var white_medium_small_square = "◽";
    var black_medium_small_square = "◾";
    var sunny = "☀️";
    var cloud = "☁️";
    var umbrella = "☂️";
    var snowman = "☃️";
    var comet = "☄️";
    var phone = "☎️";
    var telephone = "☎️";
    var ballot_box_with_check = "☑️";
    var shamrock = "☘️";
    var point_up = "☝️";
    var skull_and_crossbones = "☠️";
    var radioactive_sign = "☢️";
    var biohazard_sign = "☣️";
    var orthodox_cross = "☦️";
    var star_and_crescent = "☪️";
    var peace_symbol = "☮️";
    var yin_yang = "☯️";
    var wheel_of_dharma = "☸️";
    var white_frowning_face = "☹️";
    var relaxed = "☺️";
    var female_sign = "♀️";
    var male_sign = "♂️";
    var gemini = "♊";
    var cancer = "♋";
    var leo = "♌";
    var virgo = "♍";
    var libra = "♎";
    var scorpius = "♏";
    var spades = "♠️";
    var clubs = "♣️";
    var hearts = "♥️";
    var diamonds = "♦️";
    var hotsprings = "♨️";
    var recycle = "♻️";
    var wheelchair = "♿";
    var hammer_and_pick = "⚒️";
    var crossed_swords = "⚔️";
    var medical_symbol = "⚕️";
    var staff_of_aesculapius = "⚕️";
    var scales = "⚖️";
    var alembic = "⚗️";
    var gear = "⚙️";
    var atom_symbol = "⚛️";
    var fleur_de_lis = "⚜️";
    var warning = "⚠️";
    var zap = "⚡";
    var white_circle = "⚪";
    var black_circle = "⚫";
    var coffin = "⚰️";
    var funeral_urn = "⚱️";
    var soccer = "⚽";
    var baseball = "⚾";
    var snowman_without_snow = "⛄";
    var partly_sunny = "⛅";
    var thunder_cloud_and_rain = "⛈️";
    var ophiuchus = "⛎";
    var pick = "⛏️";
    var helmet_with_white_cross = "⛑️";
    var chains = "⛓️";
    var no_entry = "⛔";
    var shinto_shrine = "⛩️";
    var church = "⛪";
    var mountain = "⛰️";
    var umbrella_on_ground = "⛱️";
    var fountain = "⛲";
    var golf = "⛳";
    var ferry = "⛴️";
    var boat = "⛵";
    var sailboat = "⛵";
    var skier = "⛷️";
    var ice_skate = "⛸️";
    var person_with_ball = "⛹️‍♂️";
    var tent = "⛺";
    var fuelpump = "⛽";
    var scissors = "✂️";
    var airplane = "✈️";
    var email = "✉️";
    var envelope = "✉️";
    var fist = "✊";
    var hand = "✋";
    var raised_hand = "✋";
    var v = "✌️";
    var writing_hand = "✍️";
    var pencil2 = "✏️";
    var black_nib = "✒️";
    var heavy_check_mark = "✔️";
    var heavy_multiplication_x = "✖️";
    var latin_cross = "✝️";
    var star_of_david = "✡️";
    var eight_spoked_asterisk = "✳️";
    var eight_pointed_black_star = "✴️";
    var snowflake = "❄️";
    var sparkle = "❇️";
    var x = "❌";
    var negative_squared_cross_mark = "❎";
    var heavy_heart_exclamation_mark_ornament = "❣️";
    var heart = "❤️";
    var arrow_right = "➡️";
    var curly_loop = "➰";
    var loop = "➿";
    var arrow_heading_up = "⤴️";
    var arrow_heading_down = "⤵️";
    var arrow_left = "⬅️";
    var arrow_up = "⬆️";
    var arrow_down = "⬇️";
    var black_large_square = "⬛";
    var white_large_square = "⬜";
    var star = "⭐";
    var o = "⭕";
    var wavy_dash = "〰️";
    var part_alternation_mark = "〽️";
    var congratulations = "㊗️";
    var secret = "㊙️";
    var emoji = {
    	"100": "💯",
    	"1234": "🔢",
    	umbrella_with_rain_drops: umbrella_with_rain_drops,
    	coffee: coffee,
    	aries: aries,
    	taurus: taurus,
    	sagittarius: sagittarius,
    	capricorn: capricorn,
    	aquarius: aquarius,
    	pisces: pisces,
    	anchor: anchor,
    	white_check_mark: white_check_mark,
    	sparkles: sparkles,
    	question: question,
    	grey_question: grey_question,
    	grey_exclamation: grey_exclamation,
    	exclamation: exclamation,
    	heavy_exclamation_mark: heavy_exclamation_mark,
    	heavy_plus_sign: heavy_plus_sign,
    	heavy_minus_sign: heavy_minus_sign,
    	heavy_division_sign: heavy_division_sign,
    	hash: hash,
    	keycap_star: keycap_star,
    	zero: zero,
    	one: one,
    	two: two,
    	three: three,
    	four: four,
    	five: five,
    	six: six,
    	seven: seven,
    	eight: eight,
    	nine: nine,
    	copyright: copyright,
    	registered: registered,
    	mahjong: mahjong,
    	black_joker: black_joker,
    	a: a,
    	b: b,
    	o2: o2,
    	parking: parking,
    	ab: ab,
    	cl: cl,
    	cool: cool,
    	free: free,
    	id: id,
    	"new": "🆕",
    	ng: ng,
    	ok: ok,
    	sos: sos,
    	up: up,
    	vs: vs,
    	koko: koko,
    	sa: sa,
    	u7121: u7121,
    	u6307: u6307,
    	u7981: u7981,
    	u7a7a: u7a7a,
    	u5408: u5408,
    	u6e80: u6e80,
    	u6709: u6709,
    	u6708: u6708,
    	u7533: u7533,
    	u5272: u5272,
    	u55b6: u55b6,
    	ideograph_advantage: ideograph_advantage,
    	accept: accept,
    	cyclone: cyclone,
    	foggy: foggy,
    	closed_umbrella: closed_umbrella,
    	night_with_stars: night_with_stars,
    	sunrise_over_mountains: sunrise_over_mountains,
    	sunrise: sunrise,
    	city_sunset: city_sunset,
    	city_sunrise: city_sunrise,
    	rainbow: rainbow,
    	bridge_at_night: bridge_at_night,
    	ocean: ocean,
    	volcano: volcano,
    	milky_way: milky_way,
    	earth_africa: earth_africa,
    	earth_americas: earth_americas,
    	earth_asia: earth_asia,
    	globe_with_meridians: globe_with_meridians,
    	new_moon: new_moon,
    	waxing_crescent_moon: waxing_crescent_moon,
    	first_quarter_moon: first_quarter_moon,
    	moon: moon,
    	waxing_gibbous_moon: waxing_gibbous_moon,
    	full_moon: full_moon,
    	waning_gibbous_moon: waning_gibbous_moon,
    	last_quarter_moon: last_quarter_moon,
    	waning_crescent_moon: waning_crescent_moon,
    	crescent_moon: crescent_moon,
    	new_moon_with_face: new_moon_with_face,
    	first_quarter_moon_with_face: first_quarter_moon_with_face,
    	last_quarter_moon_with_face: last_quarter_moon_with_face,
    	full_moon_with_face: full_moon_with_face,
    	sun_with_face: sun_with_face,
    	star2: star2,
    	stars: stars,
    	thermometer: thermometer,
    	mostly_sunny: mostly_sunny,
    	sun_small_cloud: sun_small_cloud,
    	barely_sunny: barely_sunny,
    	sun_behind_cloud: sun_behind_cloud,
    	partly_sunny_rain: partly_sunny_rain,
    	sun_behind_rain_cloud: sun_behind_rain_cloud,
    	rain_cloud: rain_cloud,
    	snow_cloud: snow_cloud,
    	lightning: lightning,
    	lightning_cloud: lightning_cloud,
    	tornado: tornado,
    	tornado_cloud: tornado_cloud,
    	fog: fog,
    	wind_blowing_face: wind_blowing_face,
    	hotdog: hotdog,
    	taco: taco,
    	burrito: burrito,
    	chestnut: chestnut,
    	seedling: seedling,
    	evergreen_tree: evergreen_tree,
    	deciduous_tree: deciduous_tree,
    	palm_tree: palm_tree,
    	cactus: cactus,
    	hot_pepper: hot_pepper,
    	tulip: tulip,
    	cherry_blossom: cherry_blossom,
    	rose: rose,
    	hibiscus: hibiscus,
    	sunflower: sunflower,
    	blossom: blossom,
    	corn: corn,
    	ear_of_rice: ear_of_rice,
    	herb: herb,
    	four_leaf_clover: four_leaf_clover,
    	maple_leaf: maple_leaf,
    	fallen_leaf: fallen_leaf,
    	leaves: leaves,
    	mushroom: mushroom,
    	tomato: tomato,
    	eggplant: eggplant,
    	grapes: grapes,
    	melon: melon,
    	watermelon: watermelon,
    	tangerine: tangerine,
    	lemon: lemon,
    	banana: banana,
    	pineapple: pineapple,
    	apple: apple,
    	green_apple: green_apple,
    	pear: pear,
    	peach: peach,
    	cherries: cherries,
    	strawberry: strawberry,
    	hamburger: hamburger,
    	pizza: pizza,
    	meat_on_bone: meat_on_bone,
    	poultry_leg: poultry_leg,
    	rice_cracker: rice_cracker,
    	rice_ball: rice_ball,
    	rice: rice,
    	curry: curry,
    	ramen: ramen,
    	spaghetti: spaghetti,
    	bread: bread,
    	fries: fries,
    	sweet_potato: sweet_potato,
    	dango: dango,
    	oden: oden,
    	sushi: sushi,
    	fried_shrimp: fried_shrimp,
    	fish_cake: fish_cake,
    	icecream: icecream,
    	shaved_ice: shaved_ice,
    	ice_cream: ice_cream,
    	doughnut: doughnut,
    	cookie: cookie,
    	chocolate_bar: chocolate_bar,
    	candy: candy,
    	lollipop: lollipop,
    	custard: custard,
    	honey_pot: honey_pot,
    	cake: cake,
    	bento: bento,
    	stew: stew,
    	fried_egg: fried_egg,
    	cooking: cooking,
    	fork_and_knife: fork_and_knife,
    	tea: tea,
    	sake: sake,
    	wine_glass: wine_glass,
    	cocktail: cocktail,
    	tropical_drink: tropical_drink,
    	beer: beer,
    	beers: beers,
    	baby_bottle: baby_bottle,
    	knife_fork_plate: knife_fork_plate,
    	champagne: champagne,
    	popcorn: popcorn,
    	ribbon: ribbon,
    	gift: gift,
    	birthday: birthday,
    	jack_o_lantern: jack_o_lantern,
    	christmas_tree: christmas_tree,
    	santa: santa,
    	fireworks: fireworks,
    	sparkler: sparkler,
    	balloon: balloon,
    	tada: tada,
    	confetti_ball: confetti_ball,
    	tanabata_tree: tanabata_tree,
    	crossed_flags: crossed_flags,
    	bamboo: bamboo,
    	dolls: dolls,
    	flags: flags,
    	wind_chime: wind_chime,
    	rice_scene: rice_scene,
    	school_satchel: school_satchel,
    	mortar_board: mortar_board,
    	medal: medal,
    	reminder_ribbon: reminder_ribbon,
    	studio_microphone: studio_microphone,
    	level_slider: level_slider,
    	control_knobs: control_knobs,
    	film_frames: film_frames,
    	admission_tickets: admission_tickets,
    	carousel_horse: carousel_horse,
    	ferris_wheel: ferris_wheel,
    	roller_coaster: roller_coaster,
    	fishing_pole_and_fish: fishing_pole_and_fish,
    	microphone: microphone,
    	movie_camera: movie_camera,
    	cinema: cinema,
    	headphones: headphones,
    	art: art,
    	tophat: tophat,
    	circus_tent: circus_tent,
    	ticket: ticket,
    	clapper: clapper,
    	performing_arts: performing_arts,
    	video_game: video_game,
    	dart: dart,
    	slot_machine: slot_machine,
    	"8ball": "🎱",
    	game_die: game_die,
    	bowling: bowling,
    	flower_playing_cards: flower_playing_cards,
    	musical_note: musical_note,
    	notes: notes,
    	saxophone: saxophone,
    	guitar: guitar,
    	musical_keyboard: musical_keyboard,
    	trumpet: trumpet,
    	violin: violin,
    	musical_score: musical_score,
    	running_shirt_with_sash: running_shirt_with_sash,
    	tennis: tennis,
    	ski: ski,
    	basketball: basketball,
    	checkered_flag: checkered_flag,
    	snowboarder: snowboarder,
    	"woman-running": "🏃‍♀️",
    	"man-running": "🏃‍♂️",
    	runner: runner,
    	running: running,
    	"woman-surfing": "🏄‍♀️",
    	"man-surfing": "🏄‍♂️",
    	surfer: surfer,
    	sports_medal: sports_medal,
    	trophy: trophy,
    	horse_racing: horse_racing,
    	football: football,
    	rugby_football: rugby_football,
    	"woman-swimming": "🏊‍♀️",
    	"man-swimming": "🏊‍♂️",
    	swimmer: swimmer,
    	"woman-lifting-weights": "🏋️‍♀️",
    	"man-lifting-weights": "🏋️‍♂️",
    	weight_lifter: weight_lifter,
    	"woman-golfing": "🏌️‍♀️",
    	"man-golfing": "🏌️‍♂️",
    	golfer: golfer,
    	racing_motorcycle: racing_motorcycle,
    	racing_car: racing_car,
    	cricket_bat_and_ball: cricket_bat_and_ball,
    	volleyball: volleyball,
    	field_hockey_stick_and_ball: field_hockey_stick_and_ball,
    	ice_hockey_stick_and_puck: ice_hockey_stick_and_puck,
    	table_tennis_paddle_and_ball: table_tennis_paddle_and_ball,
    	snow_capped_mountain: snow_capped_mountain,
    	camping: camping,
    	beach_with_umbrella: beach_with_umbrella,
    	building_construction: building_construction,
    	house_buildings: house_buildings,
    	cityscape: cityscape,
    	derelict_house_building: derelict_house_building,
    	classical_building: classical_building,
    	desert: desert,
    	desert_island: desert_island,
    	national_park: national_park,
    	stadium: stadium,
    	house: house,
    	house_with_garden: house_with_garden,
    	office: office,
    	post_office: post_office,
    	european_post_office: european_post_office,
    	hospital: hospital,
    	bank: bank,
    	atm: atm,
    	hotel: hotel,
    	love_hotel: love_hotel,
    	convenience_store: convenience_store,
    	school: school,
    	department_store: department_store,
    	factory: factory,
    	izakaya_lantern: izakaya_lantern,
    	lantern: lantern,
    	japanese_castle: japanese_castle,
    	european_castle: european_castle,
    	"rainbow-flag": "🏳️‍🌈",
    	waving_white_flag: waving_white_flag,
    	"flag-england": "🏴󠁧󠁢󠁥󠁮󠁧󠁿",
    	"flag-scotland": "🏴󠁧󠁢󠁳󠁣󠁴󠁿",
    	"flag-wales": "🏴󠁧󠁢󠁷󠁬󠁳󠁿",
    	waving_black_flag: waving_black_flag,
    	rosette: rosette,
    	label: label,
    	badminton_racquet_and_shuttlecock: badminton_racquet_and_shuttlecock,
    	bow_and_arrow: bow_and_arrow,
    	amphora: amphora,
    	rat: rat,
    	mouse2: mouse2,
    	ox: ox,
    	water_buffalo: water_buffalo,
    	cow2: cow2,
    	tiger2: tiger2,
    	leopard: leopard,
    	rabbit2: rabbit2,
    	cat2: cat2,
    	dragon: dragon,
    	crocodile: crocodile,
    	whale2: whale2,
    	snail: snail,
    	snake: snake,
    	racehorse: racehorse,
    	ram: ram,
    	goat: goat,
    	sheep: sheep,
    	monkey: monkey,
    	rooster: rooster,
    	chicken: chicken,
    	dog2: dog2,
    	pig2: pig2,
    	boar: boar,
    	elephant: elephant,
    	octopus: octopus,
    	shell: shell,
    	bug: bug,
    	ant: ant,
    	bee: bee,
    	honeybee: honeybee,
    	beetle: beetle,
    	fish: fish,
    	tropical_fish: tropical_fish,
    	blowfish: blowfish,
    	turtle: turtle,
    	hatching_chick: hatching_chick,
    	baby_chick: baby_chick,
    	hatched_chick: hatched_chick,
    	bird: bird,
    	penguin: penguin,
    	koala: koala,
    	poodle: poodle,
    	dromedary_camel: dromedary_camel,
    	camel: camel,
    	dolphin: dolphin,
    	flipper: flipper,
    	mouse: mouse,
    	cow: cow,
    	tiger: tiger,
    	rabbit: rabbit,
    	cat: cat,
    	dragon_face: dragon_face,
    	whale: whale,
    	horse: horse,
    	monkey_face: monkey_face,
    	dog: dog,
    	pig: pig,
    	frog: frog,
    	hamster: hamster,
    	wolf: wolf,
    	bear: bear,
    	panda_face: panda_face,
    	pig_nose: pig_nose,
    	feet: feet,
    	paw_prints: paw_prints,
    	chipmunk: chipmunk,
    	eyes: eyes,
    	"eye-in-speech-bubble": "👁️‍🗨️",
    	eye: eye,
    	ear: ear,
    	nose: nose,
    	lips: lips,
    	tongue: tongue,
    	point_up_2: point_up_2,
    	point_down: point_down,
    	point_left: point_left,
    	point_right: point_right,
    	facepunch: facepunch,
    	punch: punch,
    	wave: wave,
    	ok_hand: ok_hand,
    	"+1": "👍",
    	thumbsup: thumbsup,
    	"-1": "👎",
    	thumbsdown: thumbsdown,
    	clap: clap,
    	open_hands: open_hands,
    	crown: crown,
    	womans_hat: womans_hat,
    	eyeglasses: eyeglasses,
    	necktie: necktie,
    	shirt: shirt,
    	tshirt: tshirt,
    	jeans: jeans,
    	dress: dress,
    	kimono: kimono,
    	bikini: bikini,
    	womans_clothes: womans_clothes,
    	purse: purse,
    	handbag: handbag,
    	pouch: pouch,
    	mans_shoe: mans_shoe,
    	shoe: shoe,
    	athletic_shoe: athletic_shoe,
    	high_heel: high_heel,
    	sandal: sandal,
    	boot: boot,
    	footprints: footprints,
    	bust_in_silhouette: bust_in_silhouette,
    	busts_in_silhouette: busts_in_silhouette,
    	boy: boy,
    	girl: girl,
    	"male-farmer": "👨‍🌾",
    	"male-cook": "👨‍🍳",
    	"male-student": "👨‍🎓",
    	"male-singer": "👨‍🎤",
    	"male-artist": "👨‍🎨",
    	"male-teacher": "👨‍🏫",
    	"male-factory-worker": "👨‍🏭",
    	"man-boy-boy": "👨‍👦‍👦",
    	"man-boy": "👨‍👦",
    	"man-girl-boy": "👨‍👧‍👦",
    	"man-girl-girl": "👨‍👧‍👧",
    	"man-girl": "👨‍👧",
    	"man-man-boy": "👨‍👨‍👦",
    	"man-man-boy-boy": "👨‍👨‍👦‍👦",
    	"man-man-girl": "👨‍👨‍👧",
    	"man-man-girl-boy": "👨‍👨‍👧‍👦",
    	"man-man-girl-girl": "👨‍👨‍👧‍👧",
    	"man-woman-boy": "👨‍👩‍👦",
    	family: family,
    	"man-woman-boy-boy": "👨‍👩‍👦‍👦",
    	"man-woman-girl": "👨‍👩‍👧",
    	"man-woman-girl-boy": "👨‍👩‍👧‍👦",
    	"man-woman-girl-girl": "👨‍👩‍👧‍👧",
    	"male-technologist": "👨‍💻",
    	"male-office-worker": "👨‍💼",
    	"male-mechanic": "👨‍🔧",
    	"male-scientist": "👨‍🔬",
    	"male-astronaut": "👨‍🚀",
    	"male-firefighter": "👨‍🚒",
    	"male-doctor": "👨‍⚕️",
    	"male-judge": "👨‍⚖️",
    	"male-pilot": "👨‍✈️",
    	"man-heart-man": "👨‍❤️‍👨",
    	"man-kiss-man": "👨‍❤️‍💋‍👨",
    	man: man,
    	"female-farmer": "👩‍🌾",
    	"female-cook": "👩‍🍳",
    	"female-student": "👩‍🎓",
    	"female-singer": "👩‍🎤",
    	"female-artist": "👩‍🎨",
    	"female-teacher": "👩‍🏫",
    	"female-factory-worker": "👩‍🏭",
    	"woman-boy-boy": "👩‍👦‍👦",
    	"woman-boy": "👩‍👦",
    	"woman-girl-boy": "👩‍👧‍👦",
    	"woman-girl-girl": "👩‍👧‍👧",
    	"woman-girl": "👩‍👧",
    	"woman-woman-boy": "👩‍👩‍👦",
    	"woman-woman-boy-boy": "👩‍👩‍👦‍👦",
    	"woman-woman-girl": "👩‍👩‍👧",
    	"woman-woman-girl-boy": "👩‍👩‍👧‍👦",
    	"woman-woman-girl-girl": "👩‍👩‍👧‍👧",
    	"female-technologist": "👩‍💻",
    	"female-office-worker": "👩‍💼",
    	"female-mechanic": "👩‍🔧",
    	"female-scientist": "👩‍🔬",
    	"female-astronaut": "👩‍🚀",
    	"female-firefighter": "👩‍🚒",
    	"female-doctor": "👩‍⚕️",
    	"female-judge": "👩‍⚖️",
    	"female-pilot": "👩‍✈️",
    	"woman-heart-man": "👩‍❤️‍👨",
    	couple_with_heart: couple_with_heart,
    	"woman-heart-woman": "👩‍❤️‍👩",
    	"woman-kiss-man": "👩‍❤️‍💋‍👨",
    	couplekiss: couplekiss,
    	"woman-kiss-woman": "👩‍❤️‍💋‍👩",
    	woman: woman,
    	couple: couple,
    	man_and_woman_holding_hands: man_and_woman_holding_hands,
    	two_men_holding_hands: two_men_holding_hands,
    	two_women_holding_hands: two_women_holding_hands,
    	"female-police-officer": "👮‍♀️",
    	"male-police-officer": "👮‍♂️",
    	cop: cop,
    	"woman-with-bunny-ears-partying": "👯‍♀️",
    	dancers: dancers,
    	"man-with-bunny-ears-partying": "👯‍♂️",
    	bride_with_veil: bride_with_veil,
    	"blond-haired-woman": "👱‍♀️",
    	"blond-haired-man": "👱‍♂️",
    	person_with_blond_hair: person_with_blond_hair,
    	man_with_gua_pi_mao: man_with_gua_pi_mao,
    	"woman-wearing-turban": "👳‍♀️",
    	"man-wearing-turban": "👳‍♂️",
    	man_with_turban: man_with_turban,
    	older_man: older_man,
    	older_woman: older_woman,
    	baby: baby,
    	"female-construction-worker": "👷‍♀️",
    	"male-construction-worker": "👷‍♂️",
    	construction_worker: construction_worker,
    	princess: princess,
    	japanese_ogre: japanese_ogre,
    	japanese_goblin: japanese_goblin,
    	ghost: ghost,
    	angel: angel,
    	alien: alien,
    	space_invader: space_invader,
    	imp: imp,
    	skull: skull,
    	"woman-tipping-hand": "💁‍♀️",
    	information_desk_person: information_desk_person,
    	"man-tipping-hand": "💁‍♂️",
    	"female-guard": "💂‍♀️",
    	"male-guard": "💂‍♂️",
    	guardsman: guardsman,
    	dancer: dancer,
    	lipstick: lipstick,
    	nail_care: nail_care,
    	"woman-getting-massage": "💆‍♀️",
    	massage: massage,
    	"man-getting-massage": "💆‍♂️",
    	"woman-getting-haircut": "💇‍♀️",
    	haircut: haircut,
    	"man-getting-haircut": "💇‍♂️",
    	barber: barber,
    	syringe: syringe,
    	pill: pill,
    	kiss: kiss,
    	love_letter: love_letter,
    	ring: ring,
    	gem: gem,
    	bouquet: bouquet,
    	wedding: wedding,
    	heartbeat: heartbeat,
    	broken_heart: broken_heart,
    	two_hearts: two_hearts,
    	sparkling_heart: sparkling_heart,
    	heartpulse: heartpulse,
    	cupid: cupid,
    	blue_heart: blue_heart,
    	green_heart: green_heart,
    	yellow_heart: yellow_heart,
    	purple_heart: purple_heart,
    	gift_heart: gift_heart,
    	revolving_hearts: revolving_hearts,
    	heart_decoration: heart_decoration,
    	diamond_shape_with_a_dot_inside: diamond_shape_with_a_dot_inside,
    	bulb: bulb,
    	anger: anger,
    	bomb: bomb,
    	zzz: zzz,
    	boom: boom,
    	collision: collision,
    	sweat_drops: sweat_drops,
    	droplet: droplet,
    	dash: dash,
    	poop: poop,
    	muscle: muscle,
    	dizzy: dizzy,
    	speech_balloon: speech_balloon,
    	thought_balloon: thought_balloon,
    	white_flower: white_flower,
    	moneybag: moneybag,
    	currency_exchange: currency_exchange,
    	heavy_dollar_sign: heavy_dollar_sign,
    	credit_card: credit_card,
    	yen: yen,
    	dollar: dollar,
    	euro: euro,
    	pound: pound,
    	money_with_wings: money_with_wings,
    	chart: chart,
    	seat: seat,
    	computer: computer,
    	briefcase: briefcase,
    	minidisc: minidisc,
    	floppy_disk: floppy_disk,
    	cd: cd,
    	dvd: dvd,
    	file_folder: file_folder,
    	open_file_folder: open_file_folder,
    	page_with_curl: page_with_curl,
    	page_facing_up: page_facing_up,
    	date: date,
    	calendar: calendar,
    	card_index: card_index,
    	chart_with_upwards_trend: chart_with_upwards_trend,
    	chart_with_downwards_trend: chart_with_downwards_trend,
    	bar_chart: bar_chart,
    	clipboard: clipboard,
    	pushpin: pushpin,
    	round_pushpin: round_pushpin,
    	paperclip: paperclip,
    	straight_ruler: straight_ruler,
    	triangular_ruler: triangular_ruler,
    	bookmark_tabs: bookmark_tabs,
    	ledger: ledger,
    	notebook: notebook,
    	notebook_with_decorative_cover: notebook_with_decorative_cover,
    	closed_book: closed_book,
    	book: book,
    	open_book: open_book,
    	green_book: green_book,
    	blue_book: blue_book,
    	orange_book: orange_book,
    	books: books,
    	name_badge: name_badge,
    	scroll: scroll,
    	memo: memo,
    	pencil: pencil,
    	telephone_receiver: telephone_receiver,
    	pager: pager,
    	fax: fax,
    	satellite_antenna: satellite_antenna,
    	loudspeaker: loudspeaker,
    	mega: mega,
    	outbox_tray: outbox_tray,
    	inbox_tray: inbox_tray,
    	"package": "📦",
    	"e-mail": "📧",
    	incoming_envelope: incoming_envelope,
    	envelope_with_arrow: envelope_with_arrow,
    	mailbox_closed: mailbox_closed,
    	mailbox: mailbox,
    	mailbox_with_mail: mailbox_with_mail,
    	mailbox_with_no_mail: mailbox_with_no_mail,
    	postbox: postbox,
    	postal_horn: postal_horn,
    	newspaper: newspaper,
    	iphone: iphone,
    	calling: calling,
    	vibration_mode: vibration_mode,
    	mobile_phone_off: mobile_phone_off,
    	no_mobile_phones: no_mobile_phones,
    	signal_strength: signal_strength,
    	camera: camera,
    	camera_with_flash: camera_with_flash,
    	video_camera: video_camera,
    	tv: tv,
    	radio: radio,
    	vhs: vhs,
    	film_projector: film_projector,
    	prayer_beads: prayer_beads,
    	twisted_rightwards_arrows: twisted_rightwards_arrows,
    	repeat: repeat,
    	repeat_one: repeat_one,
    	arrows_clockwise: arrows_clockwise,
    	arrows_counterclockwise: arrows_counterclockwise,
    	low_brightness: low_brightness,
    	high_brightness: high_brightness,
    	mute: mute,
    	speaker: speaker,
    	sound: sound,
    	loud_sound: loud_sound,
    	battery: battery,
    	electric_plug: electric_plug,
    	mag: mag,
    	mag_right: mag_right,
    	lock_with_ink_pen: lock_with_ink_pen,
    	closed_lock_with_key: closed_lock_with_key,
    	key: key,
    	lock: lock,
    	unlock: unlock,
    	bell: bell,
    	no_bell: no_bell,
    	bookmark: bookmark,
    	link: link,
    	radio_button: radio_button,
    	back: back,
    	end: end,
    	on: on,
    	soon: soon,
    	underage: underage,
    	keycap_ten: keycap_ten,
    	capital_abcd: capital_abcd,
    	abcd: abcd,
    	symbols: symbols,
    	abc: abc,
    	fire: fire,
    	flashlight: flashlight,
    	wrench: wrench,
    	hammer: hammer,
    	nut_and_bolt: nut_and_bolt,
    	hocho: hocho,
    	knife: knife,
    	gun: gun,
    	microscope: microscope,
    	telescope: telescope,
    	crystal_ball: crystal_ball,
    	six_pointed_star: six_pointed_star,
    	beginner: beginner,
    	trident: trident,
    	black_square_button: black_square_button,
    	white_square_button: white_square_button,
    	red_circle: red_circle,
    	large_blue_circle: large_blue_circle,
    	large_orange_diamond: large_orange_diamond,
    	large_blue_diamond: large_blue_diamond,
    	small_orange_diamond: small_orange_diamond,
    	small_blue_diamond: small_blue_diamond,
    	small_red_triangle: small_red_triangle,
    	small_red_triangle_down: small_red_triangle_down,
    	arrow_up_small: arrow_up_small,
    	arrow_down_small: arrow_down_small,
    	om_symbol: om_symbol,
    	dove_of_peace: dove_of_peace,
    	kaaba: kaaba,
    	mosque: mosque,
    	synagogue: synagogue,
    	menorah_with_nine_branches: menorah_with_nine_branches,
    	candle: candle,
    	mantelpiece_clock: mantelpiece_clock,
    	hole: hole,
    	man_in_business_suit_levitating: man_in_business_suit_levitating,
    	"female-detective": "🕵️‍♀️",
    	"male-detective": "🕵️‍♂️",
    	sleuth_or_spy: sleuth_or_spy,
    	dark_sunglasses: dark_sunglasses,
    	spider: spider,
    	spider_web: spider_web,
    	joystick: joystick,
    	man_dancing: man_dancing,
    	linked_paperclips: linked_paperclips,
    	lower_left_ballpoint_pen: lower_left_ballpoint_pen,
    	lower_left_fountain_pen: lower_left_fountain_pen,
    	lower_left_paintbrush: lower_left_paintbrush,
    	lower_left_crayon: lower_left_crayon,
    	raised_hand_with_fingers_splayed: raised_hand_with_fingers_splayed,
    	middle_finger: middle_finger,
    	reversed_hand_with_middle_finger_extended: reversed_hand_with_middle_finger_extended,
    	"spock-hand": "🖖",
    	black_heart: black_heart,
    	desktop_computer: desktop_computer,
    	printer: printer,
    	three_button_mouse: three_button_mouse,
    	trackball: trackball,
    	frame_with_picture: frame_with_picture,
    	card_index_dividers: card_index_dividers,
    	card_file_box: card_file_box,
    	file_cabinet: file_cabinet,
    	wastebasket: wastebasket,
    	spiral_note_pad: spiral_note_pad,
    	spiral_calendar_pad: spiral_calendar_pad,
    	compression: compression,
    	old_key: old_key,
    	rolled_up_newspaper: rolled_up_newspaper,
    	dagger_knife: dagger_knife,
    	speaking_head_in_silhouette: speaking_head_in_silhouette,
    	left_speech_bubble: left_speech_bubble,
    	right_anger_bubble: right_anger_bubble,
    	ballot_box_with_ballot: ballot_box_with_ballot,
    	world_map: world_map,
    	mount_fuji: mount_fuji,
    	tokyo_tower: tokyo_tower,
    	statue_of_liberty: statue_of_liberty,
    	japan: japan,
    	moyai: moyai,
    	grinning: grinning,
    	grin: grin,
    	joy: joy,
    	smiley: smiley,
    	smile: smile,
    	sweat_smile: sweat_smile,
    	laughing: laughing,
    	satisfied: satisfied,
    	innocent: innocent,
    	smiling_imp: smiling_imp,
    	wink: wink,
    	blush: blush,
    	yum: yum,
    	relieved: relieved,
    	heart_eyes: heart_eyes,
    	sunglasses: sunglasses,
    	smirk: smirk,
    	neutral_face: neutral_face,
    	expressionless: expressionless,
    	unamused: unamused,
    	sweat: sweat,
    	pensive: pensive,
    	confused: confused,
    	confounded: confounded,
    	kissing: kissing,
    	kissing_heart: kissing_heart,
    	kissing_smiling_eyes: kissing_smiling_eyes,
    	kissing_closed_eyes: kissing_closed_eyes,
    	stuck_out_tongue: stuck_out_tongue,
    	stuck_out_tongue_winking_eye: stuck_out_tongue_winking_eye,
    	stuck_out_tongue_closed_eyes: stuck_out_tongue_closed_eyes,
    	disappointed: disappointed,
    	worried: worried,
    	angry: angry,
    	rage: rage,
    	cry: cry,
    	persevere: persevere,
    	triumph: triumph,
    	disappointed_relieved: disappointed_relieved,
    	frowning: frowning,
    	anguished: anguished,
    	fearful: fearful,
    	weary: weary,
    	sleepy: sleepy,
    	tired_face: tired_face,
    	grimacing: grimacing,
    	sob: sob,
    	open_mouth: open_mouth,
    	hushed: hushed,
    	cold_sweat: cold_sweat,
    	scream: scream,
    	astonished: astonished,
    	flushed: flushed,
    	sleeping: sleeping,
    	dizzy_face: dizzy_face,
    	no_mouth: no_mouth,
    	mask: mask,
    	smile_cat: smile_cat,
    	joy_cat: joy_cat,
    	smiley_cat: smiley_cat,
    	heart_eyes_cat: heart_eyes_cat,
    	smirk_cat: smirk_cat,
    	kissing_cat: kissing_cat,
    	pouting_cat: pouting_cat,
    	crying_cat_face: crying_cat_face,
    	scream_cat: scream_cat,
    	slightly_frowning_face: slightly_frowning_face,
    	slightly_smiling_face: slightly_smiling_face,
    	upside_down_face: upside_down_face,
    	face_with_rolling_eyes: face_with_rolling_eyes,
    	"woman-gesturing-no": "🙅‍♀️",
    	no_good: no_good,
    	"man-gesturing-no": "🙅‍♂️",
    	"woman-gesturing-ok": "🙆‍♀️",
    	ok_woman: ok_woman,
    	"man-gesturing-ok": "🙆‍♂️",
    	"woman-bowing": "🙇‍♀️",
    	"man-bowing": "🙇‍♂️",
    	bow: bow,
    	see_no_evil: see_no_evil,
    	hear_no_evil: hear_no_evil,
    	speak_no_evil: speak_no_evil,
    	"woman-raising-hand": "🙋‍♀️",
    	raising_hand: raising_hand,
    	"man-raising-hand": "🙋‍♂️",
    	raised_hands: raised_hands,
    	"woman-frowning": "🙍‍♀️",
    	person_frowning: person_frowning,
    	"man-frowning": "🙍‍♂️",
    	"woman-pouting": "🙎‍♀️",
    	person_with_pouting_face: person_with_pouting_face,
    	"man-pouting": "🙎‍♂️",
    	pray: pray,
    	rocket: rocket,
    	helicopter: helicopter,
    	steam_locomotive: steam_locomotive,
    	railway_car: railway_car,
    	bullettrain_side: bullettrain_side,
    	bullettrain_front: bullettrain_front,
    	train2: train2,
    	metro: metro,
    	light_rail: light_rail,
    	station: station,
    	tram: tram,
    	train: train,
    	bus: bus,
    	oncoming_bus: oncoming_bus,
    	trolleybus: trolleybus,
    	busstop: busstop,
    	minibus: minibus,
    	ambulance: ambulance,
    	fire_engine: fire_engine,
    	police_car: police_car,
    	oncoming_police_car: oncoming_police_car,
    	taxi: taxi,
    	oncoming_taxi: oncoming_taxi,
    	car: car,
    	red_car: red_car,
    	oncoming_automobile: oncoming_automobile,
    	blue_car: blue_car,
    	truck: truck,
    	articulated_lorry: articulated_lorry,
    	tractor: tractor,
    	monorail: monorail,
    	mountain_railway: mountain_railway,
    	suspension_railway: suspension_railway,
    	mountain_cableway: mountain_cableway,
    	aerial_tramway: aerial_tramway,
    	ship: ship,
    	"woman-rowing-boat": "🚣‍♀️",
    	"man-rowing-boat": "🚣‍♂️",
    	rowboat: rowboat,
    	speedboat: speedboat,
    	traffic_light: traffic_light,
    	vertical_traffic_light: vertical_traffic_light,
    	construction: construction,
    	rotating_light: rotating_light,
    	triangular_flag_on_post: triangular_flag_on_post,
    	door: door,
    	no_entry_sign: no_entry_sign,
    	smoking: smoking,
    	no_smoking: no_smoking,
    	put_litter_in_its_place: put_litter_in_its_place,
    	do_not_litter: do_not_litter,
    	potable_water: potable_water,
    	"non-potable_water": "🚱",
    	bike: bike,
    	no_bicycles: no_bicycles,
    	"woman-biking": "🚴‍♀️",
    	"man-biking": "🚴‍♂️",
    	bicyclist: bicyclist,
    	"woman-mountain-biking": "🚵‍♀️",
    	"man-mountain-biking": "🚵‍♂️",
    	mountain_bicyclist: mountain_bicyclist,
    	"woman-walking": "🚶‍♀️",
    	"man-walking": "🚶‍♂️",
    	walking: walking,
    	no_pedestrians: no_pedestrians,
    	children_crossing: children_crossing,
    	mens: mens,
    	womens: womens,
    	restroom: restroom,
    	baby_symbol: baby_symbol,
    	toilet: toilet,
    	wc: wc,
    	shower: shower,
    	bath: bath,
    	bathtub: bathtub,
    	passport_control: passport_control,
    	customs: customs,
    	baggage_claim: baggage_claim,
    	left_luggage: left_luggage,
    	couch_and_lamp: couch_and_lamp,
    	sleeping_accommodation: sleeping_accommodation,
    	shopping_bags: shopping_bags,
    	bellhop_bell: bellhop_bell,
    	bed: bed,
    	place_of_worship: place_of_worship,
    	octagonal_sign: octagonal_sign,
    	shopping_trolley: shopping_trolley,
    	hammer_and_wrench: hammer_and_wrench,
    	shield: shield,
    	oil_drum: oil_drum,
    	motorway: motorway,
    	railway_track: railway_track,
    	motor_boat: motor_boat,
    	small_airplane: small_airplane,
    	airplane_departure: airplane_departure,
    	airplane_arriving: airplane_arriving,
    	satellite: satellite,
    	passenger_ship: passenger_ship,
    	scooter: scooter,
    	motor_scooter: motor_scooter,
    	canoe: canoe,
    	sled: sled,
    	flying_saucer: flying_saucer,
    	zipper_mouth_face: zipper_mouth_face,
    	money_mouth_face: money_mouth_face,
    	face_with_thermometer: face_with_thermometer,
    	nerd_face: nerd_face,
    	thinking_face: thinking_face,
    	face_with_head_bandage: face_with_head_bandage,
    	robot_face: robot_face,
    	hugging_face: hugging_face,
    	the_horns: the_horns,
    	sign_of_the_horns: sign_of_the_horns,
    	call_me_hand: call_me_hand,
    	raised_back_of_hand: raised_back_of_hand,
    	"left-facing_fist": "🤛",
    	"right-facing_fist": "🤜",
    	handshake: handshake,
    	crossed_fingers: crossed_fingers,
    	hand_with_index_and_middle_fingers_crossed: hand_with_index_and_middle_fingers_crossed,
    	i_love_you_hand_sign: i_love_you_hand_sign,
    	face_with_cowboy_hat: face_with_cowboy_hat,
    	clown_face: clown_face,
    	nauseated_face: nauseated_face,
    	rolling_on_the_floor_laughing: rolling_on_the_floor_laughing,
    	drooling_face: drooling_face,
    	lying_face: lying_face,
    	"woman-facepalming": "🤦‍♀️",
    	"man-facepalming": "🤦‍♂️",
    	face_palm: face_palm,
    	sneezing_face: sneezing_face,
    	face_with_raised_eyebrow: face_with_raised_eyebrow,
    	face_with_one_eyebrow_raised: face_with_one_eyebrow_raised,
    	"star-struck": "🤩",
    	grinning_face_with_star_eyes: grinning_face_with_star_eyes,
    	zany_face: zany_face,
    	grinning_face_with_one_large_and_one_small_eye: grinning_face_with_one_large_and_one_small_eye,
    	shushing_face: shushing_face,
    	face_with_finger_covering_closed_lips: face_with_finger_covering_closed_lips,
    	face_with_symbols_on_mouth: face_with_symbols_on_mouth,
    	serious_face_with_symbols_covering_mouth: serious_face_with_symbols_covering_mouth,
    	face_with_hand_over_mouth: face_with_hand_over_mouth,
    	smiling_face_with_smiling_eyes_and_hand_covering_mouth: smiling_face_with_smiling_eyes_and_hand_covering_mouth,
    	face_vomiting: face_vomiting,
    	face_with_open_mouth_vomiting: face_with_open_mouth_vomiting,
    	exploding_head: exploding_head,
    	shocked_face_with_exploding_head: shocked_face_with_exploding_head,
    	pregnant_woman: pregnant_woman,
    	"breast-feeding": "🤱",
    	palms_up_together: palms_up_together,
    	selfie: selfie,
    	prince: prince,
    	man_in_tuxedo: man_in_tuxedo,
    	mrs_claus: mrs_claus,
    	mother_christmas: mother_christmas,
    	"woman-shrugging": "🤷‍♀️",
    	"man-shrugging": "🤷‍♂️",
    	shrug: shrug,
    	"woman-cartwheeling": "🤸‍♀️",
    	"man-cartwheeling": "🤸‍♂️",
    	person_doing_cartwheel: person_doing_cartwheel,
    	"woman-juggling": "🤹‍♀️",
    	"man-juggling": "🤹‍♂️",
    	juggling: juggling,
    	fencer: fencer,
    	"woman-wrestling": "🤼‍♀️",
    	"man-wrestling": "🤼‍♂️",
    	wrestlers: wrestlers,
    	"woman-playing-water-polo": "🤽‍♀️",
    	"man-playing-water-polo": "🤽‍♂️",
    	water_polo: water_polo,
    	"woman-playing-handball": "🤾‍♀️",
    	"man-playing-handball": "🤾‍♂️",
    	handball: handball,
    	wilted_flower: wilted_flower,
    	drum_with_drumsticks: drum_with_drumsticks,
    	clinking_glasses: clinking_glasses,
    	tumbler_glass: tumbler_glass,
    	spoon: spoon,
    	goal_net: goal_net,
    	first_place_medal: first_place_medal,
    	second_place_medal: second_place_medal,
    	third_place_medal: third_place_medal,
    	boxing_glove: boxing_glove,
    	martial_arts_uniform: martial_arts_uniform,
    	curling_stone: curling_stone,
    	croissant: croissant,
    	avocado: avocado,
    	cucumber: cucumber,
    	bacon: bacon,
    	potato: potato,
    	carrot: carrot,
    	baguette_bread: baguette_bread,
    	green_salad: green_salad,
    	shallow_pan_of_food: shallow_pan_of_food,
    	stuffed_flatbread: stuffed_flatbread,
    	egg: egg,
    	glass_of_milk: glass_of_milk,
    	peanuts: peanuts,
    	kiwifruit: kiwifruit,
    	pancakes: pancakes,
    	dumpling: dumpling,
    	fortune_cookie: fortune_cookie,
    	takeout_box: takeout_box,
    	chopsticks: chopsticks,
    	bowl_with_spoon: bowl_with_spoon,
    	cup_with_straw: cup_with_straw,
    	coconut: coconut,
    	broccoli: broccoli,
    	pie: pie,
    	pretzel: pretzel,
    	cut_of_meat: cut_of_meat,
    	sandwich: sandwich,
    	canned_food: canned_food,
    	crab: crab,
    	lion_face: lion_face,
    	scorpion: scorpion,
    	turkey: turkey,
    	unicorn_face: unicorn_face,
    	eagle: eagle,
    	duck: duck,
    	bat: bat,
    	shark: shark,
    	owl: owl,
    	fox_face: fox_face,
    	butterfly: butterfly,
    	deer: deer,
    	gorilla: gorilla,
    	lizard: lizard,
    	rhinoceros: rhinoceros,
    	shrimp: shrimp,
    	squid: squid,
    	giraffe_face: giraffe_face,
    	zebra_face: zebra_face,
    	hedgehog: hedgehog,
    	sauropod: sauropod,
    	"t-rex": "🦖",
    	cricket: cricket,
    	cheese_wedge: cheese_wedge,
    	face_with_monocle: face_with_monocle,
    	adult: adult,
    	child: child,
    	older_adult: older_adult,
    	bearded_person: bearded_person,
    	person_with_headscarf: person_with_headscarf,
    	woman_in_steamy_room: woman_in_steamy_room,
    	man_in_steamy_room: man_in_steamy_room,
    	person_in_steamy_room: person_in_steamy_room,
    	woman_climbing: woman_climbing,
    	person_climbing: person_climbing,
    	man_climbing: man_climbing,
    	woman_in_lotus_position: woman_in_lotus_position,
    	person_in_lotus_position: person_in_lotus_position,
    	man_in_lotus_position: man_in_lotus_position,
    	female_mage: female_mage,
    	mage: mage,
    	male_mage: male_mage,
    	female_fairy: female_fairy,
    	fairy: fairy,
    	male_fairy: male_fairy,
    	female_vampire: female_vampire,
    	vampire: vampire,
    	male_vampire: male_vampire,
    	mermaid: mermaid,
    	merman: merman,
    	merperson: merperson,
    	female_elf: female_elf,
    	male_elf: male_elf,
    	elf: elf,
    	female_genie: female_genie,
    	male_genie: male_genie,
    	genie: genie,
    	female_zombie: female_zombie,
    	male_zombie: male_zombie,
    	zombie: zombie,
    	brain: brain,
    	orange_heart: orange_heart,
    	billed_cap: billed_cap,
    	scarf: scarf,
    	gloves: gloves,
    	coat: coat,
    	socks: socks,
    	bangbang: bangbang,
    	interrobang: interrobang,
    	tm: tm,
    	information_source: information_source,
    	left_right_arrow: left_right_arrow,
    	arrow_up_down: arrow_up_down,
    	arrow_upper_left: arrow_upper_left,
    	arrow_upper_right: arrow_upper_right,
    	arrow_lower_right: arrow_lower_right,
    	arrow_lower_left: arrow_lower_left,
    	leftwards_arrow_with_hook: leftwards_arrow_with_hook,
    	arrow_right_hook: arrow_right_hook,
    	watch: watch,
    	hourglass: hourglass,
    	keyboard: keyboard,
    	eject: eject,
    	fast_forward: fast_forward,
    	rewind: rewind,
    	arrow_double_up: arrow_double_up,
    	arrow_double_down: arrow_double_down,
    	black_right_pointing_double_triangle_with_vertical_bar: black_right_pointing_double_triangle_with_vertical_bar,
    	black_left_pointing_double_triangle_with_vertical_bar: black_left_pointing_double_triangle_with_vertical_bar,
    	black_right_pointing_triangle_with_double_vertical_bar: black_right_pointing_triangle_with_double_vertical_bar,
    	alarm_clock: alarm_clock,
    	stopwatch: stopwatch,
    	timer_clock: timer_clock,
    	hourglass_flowing_sand: hourglass_flowing_sand,
    	double_vertical_bar: double_vertical_bar,
    	black_square_for_stop: black_square_for_stop,
    	black_circle_for_record: black_circle_for_record,
    	m: m,
    	black_small_square: black_small_square,
    	white_small_square: white_small_square,
    	arrow_forward: arrow_forward,
    	arrow_backward: arrow_backward,
    	white_medium_square: white_medium_square,
    	black_medium_square: black_medium_square,
    	white_medium_small_square: white_medium_small_square,
    	black_medium_small_square: black_medium_small_square,
    	sunny: sunny,
    	cloud: cloud,
    	umbrella: umbrella,
    	snowman: snowman,
    	comet: comet,
    	phone: phone,
    	telephone: telephone,
    	ballot_box_with_check: ballot_box_with_check,
    	shamrock: shamrock,
    	point_up: point_up,
    	skull_and_crossbones: skull_and_crossbones,
    	radioactive_sign: radioactive_sign,
    	biohazard_sign: biohazard_sign,
    	orthodox_cross: orthodox_cross,
    	star_and_crescent: star_and_crescent,
    	peace_symbol: peace_symbol,
    	yin_yang: yin_yang,
    	wheel_of_dharma: wheel_of_dharma,
    	white_frowning_face: white_frowning_face,
    	relaxed: relaxed,
    	female_sign: female_sign,
    	male_sign: male_sign,
    	gemini: gemini,
    	cancer: cancer,
    	leo: leo,
    	virgo: virgo,
    	libra: libra,
    	scorpius: scorpius,
    	spades: spades,
    	clubs: clubs,
    	hearts: hearts,
    	diamonds: diamonds,
    	hotsprings: hotsprings,
    	recycle: recycle,
    	wheelchair: wheelchair,
    	hammer_and_pick: hammer_and_pick,
    	crossed_swords: crossed_swords,
    	medical_symbol: medical_symbol,
    	staff_of_aesculapius: staff_of_aesculapius,
    	scales: scales,
    	alembic: alembic,
    	gear: gear,
    	atom_symbol: atom_symbol,
    	fleur_de_lis: fleur_de_lis,
    	warning: warning,
    	zap: zap,
    	white_circle: white_circle,
    	black_circle: black_circle,
    	coffin: coffin,
    	funeral_urn: funeral_urn,
    	soccer: soccer,
    	baseball: baseball,
    	snowman_without_snow: snowman_without_snow,
    	partly_sunny: partly_sunny,
    	thunder_cloud_and_rain: thunder_cloud_and_rain,
    	ophiuchus: ophiuchus,
    	pick: pick,
    	helmet_with_white_cross: helmet_with_white_cross,
    	chains: chains,
    	no_entry: no_entry,
    	shinto_shrine: shinto_shrine,
    	church: church,
    	mountain: mountain,
    	umbrella_on_ground: umbrella_on_ground,
    	fountain: fountain,
    	golf: golf,
    	ferry: ferry,
    	boat: boat,
    	sailboat: sailboat,
    	skier: skier,
    	ice_skate: ice_skate,
    	"woman-bouncing-ball": "⛹️‍♀️",
    	"man-bouncing-ball": "⛹️‍♂️",
    	person_with_ball: person_with_ball,
    	tent: tent,
    	fuelpump: fuelpump,
    	scissors: scissors,
    	airplane: airplane,
    	email: email,
    	envelope: envelope,
    	fist: fist,
    	hand: hand,
    	raised_hand: raised_hand,
    	v: v,
    	writing_hand: writing_hand,
    	pencil2: pencil2,
    	black_nib: black_nib,
    	heavy_check_mark: heavy_check_mark,
    	heavy_multiplication_x: heavy_multiplication_x,
    	latin_cross: latin_cross,
    	star_of_david: star_of_david,
    	eight_spoked_asterisk: eight_spoked_asterisk,
    	eight_pointed_black_star: eight_pointed_black_star,
    	snowflake: snowflake,
    	sparkle: sparkle,
    	x: x,
    	negative_squared_cross_mark: negative_squared_cross_mark,
    	heavy_heart_exclamation_mark_ornament: heavy_heart_exclamation_mark_ornament,
    	heart: heart,
    	arrow_right: arrow_right,
    	curly_loop: curly_loop,
    	loop: loop,
    	arrow_heading_up: arrow_heading_up,
    	arrow_heading_down: arrow_heading_down,
    	arrow_left: arrow_left,
    	arrow_up: arrow_up,
    	arrow_down: arrow_down,
    	black_large_square: black_large_square,
    	white_large_square: white_large_square,
    	star: star,
    	o: o,
    	wavy_dash: wavy_dash,
    	part_alternation_mark: part_alternation_mark,
    	congratulations: congratulations,
    	secret: secret
    };

    var emoji$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        umbrella_with_rain_drops: umbrella_with_rain_drops,
        coffee: coffee,
        aries: aries,
        taurus: taurus,
        sagittarius: sagittarius,
        capricorn: capricorn,
        aquarius: aquarius,
        pisces: pisces,
        anchor: anchor,
        white_check_mark: white_check_mark,
        sparkles: sparkles,
        question: question,
        grey_question: grey_question,
        grey_exclamation: grey_exclamation,
        exclamation: exclamation,
        heavy_exclamation_mark: heavy_exclamation_mark,
        heavy_plus_sign: heavy_plus_sign,
        heavy_minus_sign: heavy_minus_sign,
        heavy_division_sign: heavy_division_sign,
        hash: hash,
        keycap_star: keycap_star,
        zero: zero,
        one: one,
        two: two,
        three: three,
        four: four,
        five: five,
        six: six,
        seven: seven,
        eight: eight,
        nine: nine,
        copyright: copyright,
        registered: registered,
        mahjong: mahjong,
        black_joker: black_joker,
        a: a,
        b: b,
        o2: o2,
        parking: parking,
        ab: ab,
        cl: cl,
        cool: cool,
        free: free,
        id: id,
        ng: ng,
        ok: ok,
        sos: sos,
        up: up,
        vs: vs,
        koko: koko,
        sa: sa,
        u7121: u7121,
        u6307: u6307,
        u7981: u7981,
        u7a7a: u7a7a,
        u5408: u5408,
        u6e80: u6e80,
        u6709: u6709,
        u6708: u6708,
        u7533: u7533,
        u5272: u5272,
        u55b6: u55b6,
        ideograph_advantage: ideograph_advantage,
        accept: accept,
        cyclone: cyclone,
        foggy: foggy,
        closed_umbrella: closed_umbrella,
        night_with_stars: night_with_stars,
        sunrise_over_mountains: sunrise_over_mountains,
        sunrise: sunrise,
        city_sunset: city_sunset,
        city_sunrise: city_sunrise,
        rainbow: rainbow,
        bridge_at_night: bridge_at_night,
        ocean: ocean,
        volcano: volcano,
        milky_way: milky_way,
        earth_africa: earth_africa,
        earth_americas: earth_americas,
        earth_asia: earth_asia,
        globe_with_meridians: globe_with_meridians,
        new_moon: new_moon,
        waxing_crescent_moon: waxing_crescent_moon,
        first_quarter_moon: first_quarter_moon,
        moon: moon,
        waxing_gibbous_moon: waxing_gibbous_moon,
        full_moon: full_moon,
        waning_gibbous_moon: waning_gibbous_moon,
        last_quarter_moon: last_quarter_moon,
        waning_crescent_moon: waning_crescent_moon,
        crescent_moon: crescent_moon,
        new_moon_with_face: new_moon_with_face,
        first_quarter_moon_with_face: first_quarter_moon_with_face,
        last_quarter_moon_with_face: last_quarter_moon_with_face,
        full_moon_with_face: full_moon_with_face,
        sun_with_face: sun_with_face,
        star2: star2,
        stars: stars,
        thermometer: thermometer,
        mostly_sunny: mostly_sunny,
        sun_small_cloud: sun_small_cloud,
        barely_sunny: barely_sunny,
        sun_behind_cloud: sun_behind_cloud,
        partly_sunny_rain: partly_sunny_rain,
        sun_behind_rain_cloud: sun_behind_rain_cloud,
        rain_cloud: rain_cloud,
        snow_cloud: snow_cloud,
        lightning: lightning,
        lightning_cloud: lightning_cloud,
        tornado: tornado,
        tornado_cloud: tornado_cloud,
        fog: fog,
        wind_blowing_face: wind_blowing_face,
        hotdog: hotdog,
        taco: taco,
        burrito: burrito,
        chestnut: chestnut,
        seedling: seedling,
        evergreen_tree: evergreen_tree,
        deciduous_tree: deciduous_tree,
        palm_tree: palm_tree,
        cactus: cactus,
        hot_pepper: hot_pepper,
        tulip: tulip,
        cherry_blossom: cherry_blossom,
        rose: rose,
        hibiscus: hibiscus,
        sunflower: sunflower,
        blossom: blossom,
        corn: corn,
        ear_of_rice: ear_of_rice,
        herb: herb,
        four_leaf_clover: four_leaf_clover,
        maple_leaf: maple_leaf,
        fallen_leaf: fallen_leaf,
        leaves: leaves,
        mushroom: mushroom,
        tomato: tomato,
        eggplant: eggplant,
        grapes: grapes,
        melon: melon,
        watermelon: watermelon,
        tangerine: tangerine,
        lemon: lemon,
        banana: banana,
        pineapple: pineapple,
        apple: apple,
        green_apple: green_apple,
        pear: pear,
        peach: peach,
        cherries: cherries,
        strawberry: strawberry,
        hamburger: hamburger,
        pizza: pizza,
        meat_on_bone: meat_on_bone,
        poultry_leg: poultry_leg,
        rice_cracker: rice_cracker,
        rice_ball: rice_ball,
        rice: rice,
        curry: curry,
        ramen: ramen,
        spaghetti: spaghetti,
        bread: bread,
        fries: fries,
        sweet_potato: sweet_potato,
        dango: dango,
        oden: oden,
        sushi: sushi,
        fried_shrimp: fried_shrimp,
        fish_cake: fish_cake,
        icecream: icecream,
        shaved_ice: shaved_ice,
        ice_cream: ice_cream,
        doughnut: doughnut,
        cookie: cookie,
        chocolate_bar: chocolate_bar,
        candy: candy,
        lollipop: lollipop,
        custard: custard,
        honey_pot: honey_pot,
        cake: cake,
        bento: bento,
        stew: stew,
        fried_egg: fried_egg,
        cooking: cooking,
        fork_and_knife: fork_and_knife,
        tea: tea,
        sake: sake,
        wine_glass: wine_glass,
        cocktail: cocktail,
        tropical_drink: tropical_drink,
        beer: beer,
        beers: beers,
        baby_bottle: baby_bottle,
        knife_fork_plate: knife_fork_plate,
        champagne: champagne,
        popcorn: popcorn,
        ribbon: ribbon,
        gift: gift,
        birthday: birthday,
        jack_o_lantern: jack_o_lantern,
        christmas_tree: christmas_tree,
        santa: santa,
        fireworks: fireworks,
        sparkler: sparkler,
        balloon: balloon,
        tada: tada,
        confetti_ball: confetti_ball,
        tanabata_tree: tanabata_tree,
        crossed_flags: crossed_flags,
        bamboo: bamboo,
        dolls: dolls,
        flags: flags,
        wind_chime: wind_chime,
        rice_scene: rice_scene,
        school_satchel: school_satchel,
        mortar_board: mortar_board,
        medal: medal,
        reminder_ribbon: reminder_ribbon,
        studio_microphone: studio_microphone,
        level_slider: level_slider,
        control_knobs: control_knobs,
        film_frames: film_frames,
        admission_tickets: admission_tickets,
        carousel_horse: carousel_horse,
        ferris_wheel: ferris_wheel,
        roller_coaster: roller_coaster,
        fishing_pole_and_fish: fishing_pole_and_fish,
        microphone: microphone,
        movie_camera: movie_camera,
        cinema: cinema,
        headphones: headphones,
        art: art,
        tophat: tophat,
        circus_tent: circus_tent,
        ticket: ticket,
        clapper: clapper,
        performing_arts: performing_arts,
        video_game: video_game,
        dart: dart,
        slot_machine: slot_machine,
        game_die: game_die,
        bowling: bowling,
        flower_playing_cards: flower_playing_cards,
        musical_note: musical_note,
        notes: notes,
        saxophone: saxophone,
        guitar: guitar,
        musical_keyboard: musical_keyboard,
        trumpet: trumpet,
        violin: violin,
        musical_score: musical_score,
        running_shirt_with_sash: running_shirt_with_sash,
        tennis: tennis,
        ski: ski,
        basketball: basketball,
        checkered_flag: checkered_flag,
        snowboarder: snowboarder,
        runner: runner,
        running: running,
        surfer: surfer,
        sports_medal: sports_medal,
        trophy: trophy,
        horse_racing: horse_racing,
        football: football,
        rugby_football: rugby_football,
        swimmer: swimmer,
        weight_lifter: weight_lifter,
        golfer: golfer,
        racing_motorcycle: racing_motorcycle,
        racing_car: racing_car,
        cricket_bat_and_ball: cricket_bat_and_ball,
        volleyball: volleyball,
        field_hockey_stick_and_ball: field_hockey_stick_and_ball,
        ice_hockey_stick_and_puck: ice_hockey_stick_and_puck,
        table_tennis_paddle_and_ball: table_tennis_paddle_and_ball,
        snow_capped_mountain: snow_capped_mountain,
        camping: camping,
        beach_with_umbrella: beach_with_umbrella,
        building_construction: building_construction,
        house_buildings: house_buildings,
        cityscape: cityscape,
        derelict_house_building: derelict_house_building,
        classical_building: classical_building,
        desert: desert,
        desert_island: desert_island,
        national_park: national_park,
        stadium: stadium,
        house: house,
        house_with_garden: house_with_garden,
        office: office,
        post_office: post_office,
        european_post_office: european_post_office,
        hospital: hospital,
        bank: bank,
        atm: atm,
        hotel: hotel,
        love_hotel: love_hotel,
        convenience_store: convenience_store,
        school: school,
        department_store: department_store,
        factory: factory,
        izakaya_lantern: izakaya_lantern,
        lantern: lantern,
        japanese_castle: japanese_castle,
        european_castle: european_castle,
        waving_white_flag: waving_white_flag,
        waving_black_flag: waving_black_flag,
        rosette: rosette,
        label: label,
        badminton_racquet_and_shuttlecock: badminton_racquet_and_shuttlecock,
        bow_and_arrow: bow_and_arrow,
        amphora: amphora,
        rat: rat,
        mouse2: mouse2,
        ox: ox,
        water_buffalo: water_buffalo,
        cow2: cow2,
        tiger2: tiger2,
        leopard: leopard,
        rabbit2: rabbit2,
        cat2: cat2,
        dragon: dragon,
        crocodile: crocodile,
        whale2: whale2,
        snail: snail,
        snake: snake,
        racehorse: racehorse,
        ram: ram,
        goat: goat,
        sheep: sheep,
        monkey: monkey,
        rooster: rooster,
        chicken: chicken,
        dog2: dog2,
        pig2: pig2,
        boar: boar,
        elephant: elephant,
        octopus: octopus,
        shell: shell,
        bug: bug,
        ant: ant,
        bee: bee,
        honeybee: honeybee,
        beetle: beetle,
        fish: fish,
        tropical_fish: tropical_fish,
        blowfish: blowfish,
        turtle: turtle,
        hatching_chick: hatching_chick,
        baby_chick: baby_chick,
        hatched_chick: hatched_chick,
        bird: bird,
        penguin: penguin,
        koala: koala,
        poodle: poodle,
        dromedary_camel: dromedary_camel,
        camel: camel,
        dolphin: dolphin,
        flipper: flipper,
        mouse: mouse,
        cow: cow,
        tiger: tiger,
        rabbit: rabbit,
        cat: cat,
        dragon_face: dragon_face,
        whale: whale,
        horse: horse,
        monkey_face: monkey_face,
        dog: dog,
        pig: pig,
        frog: frog,
        hamster: hamster,
        wolf: wolf,
        bear: bear,
        panda_face: panda_face,
        pig_nose: pig_nose,
        feet: feet,
        paw_prints: paw_prints,
        chipmunk: chipmunk,
        eyes: eyes,
        eye: eye,
        ear: ear,
        nose: nose,
        lips: lips,
        tongue: tongue,
        point_up_2: point_up_2,
        point_down: point_down,
        point_left: point_left,
        point_right: point_right,
        facepunch: facepunch,
        punch: punch,
        wave: wave,
        ok_hand: ok_hand,
        thumbsup: thumbsup,
        thumbsdown: thumbsdown,
        clap: clap,
        open_hands: open_hands,
        crown: crown,
        womans_hat: womans_hat,
        eyeglasses: eyeglasses,
        necktie: necktie,
        shirt: shirt,
        tshirt: tshirt,
        jeans: jeans,
        dress: dress,
        kimono: kimono,
        bikini: bikini,
        womans_clothes: womans_clothes,
        purse: purse,
        handbag: handbag,
        pouch: pouch,
        mans_shoe: mans_shoe,
        shoe: shoe,
        athletic_shoe: athletic_shoe,
        high_heel: high_heel,
        sandal: sandal,
        boot: boot,
        footprints: footprints,
        bust_in_silhouette: bust_in_silhouette,
        busts_in_silhouette: busts_in_silhouette,
        boy: boy,
        girl: girl,
        family: family,
        man: man,
        couple_with_heart: couple_with_heart,
        couplekiss: couplekiss,
        woman: woman,
        couple: couple,
        man_and_woman_holding_hands: man_and_woman_holding_hands,
        two_men_holding_hands: two_men_holding_hands,
        two_women_holding_hands: two_women_holding_hands,
        cop: cop,
        dancers: dancers,
        bride_with_veil: bride_with_veil,
        person_with_blond_hair: person_with_blond_hair,
        man_with_gua_pi_mao: man_with_gua_pi_mao,
        man_with_turban: man_with_turban,
        older_man: older_man,
        older_woman: older_woman,
        baby: baby,
        construction_worker: construction_worker,
        princess: princess,
        japanese_ogre: japanese_ogre,
        japanese_goblin: japanese_goblin,
        ghost: ghost,
        angel: angel,
        alien: alien,
        space_invader: space_invader,
        imp: imp,
        skull: skull,
        information_desk_person: information_desk_person,
        guardsman: guardsman,
        dancer: dancer,
        lipstick: lipstick,
        nail_care: nail_care,
        massage: massage,
        haircut: haircut,
        barber: barber,
        syringe: syringe,
        pill: pill,
        kiss: kiss,
        love_letter: love_letter,
        ring: ring,
        gem: gem,
        bouquet: bouquet,
        wedding: wedding,
        heartbeat: heartbeat,
        broken_heart: broken_heart,
        two_hearts: two_hearts,
        sparkling_heart: sparkling_heart,
        heartpulse: heartpulse,
        cupid: cupid,
        blue_heart: blue_heart,
        green_heart: green_heart,
        yellow_heart: yellow_heart,
        purple_heart: purple_heart,
        gift_heart: gift_heart,
        revolving_hearts: revolving_hearts,
        heart_decoration: heart_decoration,
        diamond_shape_with_a_dot_inside: diamond_shape_with_a_dot_inside,
        bulb: bulb,
        anger: anger,
        bomb: bomb,
        zzz: zzz,
        boom: boom,
        collision: collision,
        sweat_drops: sweat_drops,
        droplet: droplet,
        dash: dash,
        poop: poop,
        muscle: muscle,
        dizzy: dizzy,
        speech_balloon: speech_balloon,
        thought_balloon: thought_balloon,
        white_flower: white_flower,
        moneybag: moneybag,
        currency_exchange: currency_exchange,
        heavy_dollar_sign: heavy_dollar_sign,
        credit_card: credit_card,
        yen: yen,
        dollar: dollar,
        euro: euro,
        pound: pound,
        money_with_wings: money_with_wings,
        chart: chart,
        seat: seat,
        computer: computer,
        briefcase: briefcase,
        minidisc: minidisc,
        floppy_disk: floppy_disk,
        cd: cd,
        dvd: dvd,
        file_folder: file_folder,
        open_file_folder: open_file_folder,
        page_with_curl: page_with_curl,
        page_facing_up: page_facing_up,
        date: date,
        calendar: calendar,
        card_index: card_index,
        chart_with_upwards_trend: chart_with_upwards_trend,
        chart_with_downwards_trend: chart_with_downwards_trend,
        bar_chart: bar_chart,
        clipboard: clipboard,
        pushpin: pushpin,
        round_pushpin: round_pushpin,
        paperclip: paperclip,
        straight_ruler: straight_ruler,
        triangular_ruler: triangular_ruler,
        bookmark_tabs: bookmark_tabs,
        ledger: ledger,
        notebook: notebook,
        notebook_with_decorative_cover: notebook_with_decorative_cover,
        closed_book: closed_book,
        book: book,
        open_book: open_book,
        green_book: green_book,
        blue_book: blue_book,
        orange_book: orange_book,
        books: books,
        name_badge: name_badge,
        scroll: scroll,
        memo: memo,
        pencil: pencil,
        telephone_receiver: telephone_receiver,
        pager: pager,
        fax: fax,
        satellite_antenna: satellite_antenna,
        loudspeaker: loudspeaker,
        mega: mega,
        outbox_tray: outbox_tray,
        inbox_tray: inbox_tray,
        incoming_envelope: incoming_envelope,
        envelope_with_arrow: envelope_with_arrow,
        mailbox_closed: mailbox_closed,
        mailbox: mailbox,
        mailbox_with_mail: mailbox_with_mail,
        mailbox_with_no_mail: mailbox_with_no_mail,
        postbox: postbox,
        postal_horn: postal_horn,
        newspaper: newspaper,
        iphone: iphone,
        calling: calling,
        vibration_mode: vibration_mode,
        mobile_phone_off: mobile_phone_off,
        no_mobile_phones: no_mobile_phones,
        signal_strength: signal_strength,
        camera: camera,
        camera_with_flash: camera_with_flash,
        video_camera: video_camera,
        tv: tv,
        radio: radio,
        vhs: vhs,
        film_projector: film_projector,
        prayer_beads: prayer_beads,
        twisted_rightwards_arrows: twisted_rightwards_arrows,
        repeat: repeat,
        repeat_one: repeat_one,
        arrows_clockwise: arrows_clockwise,
        arrows_counterclockwise: arrows_counterclockwise,
        low_brightness: low_brightness,
        high_brightness: high_brightness,
        mute: mute,
        speaker: speaker,
        sound: sound,
        loud_sound: loud_sound,
        battery: battery,
        electric_plug: electric_plug,
        mag: mag,
        mag_right: mag_right,
        lock_with_ink_pen: lock_with_ink_pen,
        closed_lock_with_key: closed_lock_with_key,
        key: key,
        lock: lock,
        unlock: unlock,
        bell: bell,
        no_bell: no_bell,
        bookmark: bookmark,
        link: link,
        radio_button: radio_button,
        back: back,
        end: end,
        on: on,
        soon: soon,
        underage: underage,
        keycap_ten: keycap_ten,
        capital_abcd: capital_abcd,
        abcd: abcd,
        symbols: symbols,
        abc: abc,
        fire: fire,
        flashlight: flashlight,
        wrench: wrench,
        hammer: hammer,
        nut_and_bolt: nut_and_bolt,
        hocho: hocho,
        knife: knife,
        gun: gun,
        microscope: microscope,
        telescope: telescope,
        crystal_ball: crystal_ball,
        six_pointed_star: six_pointed_star,
        beginner: beginner,
        trident: trident,
        black_square_button: black_square_button,
        white_square_button: white_square_button,
        red_circle: red_circle,
        large_blue_circle: large_blue_circle,
        large_orange_diamond: large_orange_diamond,
        large_blue_diamond: large_blue_diamond,
        small_orange_diamond: small_orange_diamond,
        small_blue_diamond: small_blue_diamond,
        small_red_triangle: small_red_triangle,
        small_red_triangle_down: small_red_triangle_down,
        arrow_up_small: arrow_up_small,
        arrow_down_small: arrow_down_small,
        om_symbol: om_symbol,
        dove_of_peace: dove_of_peace,
        kaaba: kaaba,
        mosque: mosque,
        synagogue: synagogue,
        menorah_with_nine_branches: menorah_with_nine_branches,
        candle: candle,
        mantelpiece_clock: mantelpiece_clock,
        hole: hole,
        man_in_business_suit_levitating: man_in_business_suit_levitating,
        sleuth_or_spy: sleuth_or_spy,
        dark_sunglasses: dark_sunglasses,
        spider: spider,
        spider_web: spider_web,
        joystick: joystick,
        man_dancing: man_dancing,
        linked_paperclips: linked_paperclips,
        lower_left_ballpoint_pen: lower_left_ballpoint_pen,
        lower_left_fountain_pen: lower_left_fountain_pen,
        lower_left_paintbrush: lower_left_paintbrush,
        lower_left_crayon: lower_left_crayon,
        raised_hand_with_fingers_splayed: raised_hand_with_fingers_splayed,
        middle_finger: middle_finger,
        reversed_hand_with_middle_finger_extended: reversed_hand_with_middle_finger_extended,
        black_heart: black_heart,
        desktop_computer: desktop_computer,
        printer: printer,
        three_button_mouse: three_button_mouse,
        trackball: trackball,
        frame_with_picture: frame_with_picture,
        card_index_dividers: card_index_dividers,
        card_file_box: card_file_box,
        file_cabinet: file_cabinet,
        wastebasket: wastebasket,
        spiral_note_pad: spiral_note_pad,
        spiral_calendar_pad: spiral_calendar_pad,
        compression: compression,
        old_key: old_key,
        rolled_up_newspaper: rolled_up_newspaper,
        dagger_knife: dagger_knife,
        speaking_head_in_silhouette: speaking_head_in_silhouette,
        left_speech_bubble: left_speech_bubble,
        right_anger_bubble: right_anger_bubble,
        ballot_box_with_ballot: ballot_box_with_ballot,
        world_map: world_map,
        mount_fuji: mount_fuji,
        tokyo_tower: tokyo_tower,
        statue_of_liberty: statue_of_liberty,
        japan: japan,
        moyai: moyai,
        grinning: grinning,
        grin: grin,
        joy: joy,
        smiley: smiley,
        smile: smile,
        sweat_smile: sweat_smile,
        laughing: laughing,
        satisfied: satisfied,
        innocent: innocent,
        smiling_imp: smiling_imp,
        wink: wink,
        blush: blush,
        yum: yum,
        relieved: relieved,
        heart_eyes: heart_eyes,
        sunglasses: sunglasses,
        smirk: smirk,
        neutral_face: neutral_face,
        expressionless: expressionless,
        unamused: unamused,
        sweat: sweat,
        pensive: pensive,
        confused: confused,
        confounded: confounded,
        kissing: kissing,
        kissing_heart: kissing_heart,
        kissing_smiling_eyes: kissing_smiling_eyes,
        kissing_closed_eyes: kissing_closed_eyes,
        stuck_out_tongue: stuck_out_tongue,
        stuck_out_tongue_winking_eye: stuck_out_tongue_winking_eye,
        stuck_out_tongue_closed_eyes: stuck_out_tongue_closed_eyes,
        disappointed: disappointed,
        worried: worried,
        angry: angry,
        rage: rage,
        cry: cry,
        persevere: persevere,
        triumph: triumph,
        disappointed_relieved: disappointed_relieved,
        frowning: frowning,
        anguished: anguished,
        fearful: fearful,
        weary: weary,
        sleepy: sleepy,
        tired_face: tired_face,
        grimacing: grimacing,
        sob: sob,
        open_mouth: open_mouth,
        hushed: hushed,
        cold_sweat: cold_sweat,
        scream: scream,
        astonished: astonished,
        flushed: flushed,
        sleeping: sleeping,
        dizzy_face: dizzy_face,
        no_mouth: no_mouth,
        mask: mask,
        smile_cat: smile_cat,
        joy_cat: joy_cat,
        smiley_cat: smiley_cat,
        heart_eyes_cat: heart_eyes_cat,
        smirk_cat: smirk_cat,
        kissing_cat: kissing_cat,
        pouting_cat: pouting_cat,
        crying_cat_face: crying_cat_face,
        scream_cat: scream_cat,
        slightly_frowning_face: slightly_frowning_face,
        slightly_smiling_face: slightly_smiling_face,
        upside_down_face: upside_down_face,
        face_with_rolling_eyes: face_with_rolling_eyes,
        no_good: no_good,
        ok_woman: ok_woman,
        bow: bow,
        see_no_evil: see_no_evil,
        hear_no_evil: hear_no_evil,
        speak_no_evil: speak_no_evil,
        raising_hand: raising_hand,
        raised_hands: raised_hands,
        person_frowning: person_frowning,
        person_with_pouting_face: person_with_pouting_face,
        pray: pray,
        rocket: rocket,
        helicopter: helicopter,
        steam_locomotive: steam_locomotive,
        railway_car: railway_car,
        bullettrain_side: bullettrain_side,
        bullettrain_front: bullettrain_front,
        train2: train2,
        metro: metro,
        light_rail: light_rail,
        station: station,
        tram: tram,
        train: train,
        bus: bus,
        oncoming_bus: oncoming_bus,
        trolleybus: trolleybus,
        busstop: busstop,
        minibus: minibus,
        ambulance: ambulance,
        fire_engine: fire_engine,
        police_car: police_car,
        oncoming_police_car: oncoming_police_car,
        taxi: taxi,
        oncoming_taxi: oncoming_taxi,
        car: car,
        red_car: red_car,
        oncoming_automobile: oncoming_automobile,
        blue_car: blue_car,
        truck: truck,
        articulated_lorry: articulated_lorry,
        tractor: tractor,
        monorail: monorail,
        mountain_railway: mountain_railway,
        suspension_railway: suspension_railway,
        mountain_cableway: mountain_cableway,
        aerial_tramway: aerial_tramway,
        ship: ship,
        rowboat: rowboat,
        speedboat: speedboat,
        traffic_light: traffic_light,
        vertical_traffic_light: vertical_traffic_light,
        construction: construction,
        rotating_light: rotating_light,
        triangular_flag_on_post: triangular_flag_on_post,
        door: door,
        no_entry_sign: no_entry_sign,
        smoking: smoking,
        no_smoking: no_smoking,
        put_litter_in_its_place: put_litter_in_its_place,
        do_not_litter: do_not_litter,
        potable_water: potable_water,
        bike: bike,
        no_bicycles: no_bicycles,
        bicyclist: bicyclist,
        mountain_bicyclist: mountain_bicyclist,
        walking: walking,
        no_pedestrians: no_pedestrians,
        children_crossing: children_crossing,
        mens: mens,
        womens: womens,
        restroom: restroom,
        baby_symbol: baby_symbol,
        toilet: toilet,
        wc: wc,
        shower: shower,
        bath: bath,
        bathtub: bathtub,
        passport_control: passport_control,
        customs: customs,
        baggage_claim: baggage_claim,
        left_luggage: left_luggage,
        couch_and_lamp: couch_and_lamp,
        sleeping_accommodation: sleeping_accommodation,
        shopping_bags: shopping_bags,
        bellhop_bell: bellhop_bell,
        bed: bed,
        place_of_worship: place_of_worship,
        octagonal_sign: octagonal_sign,
        shopping_trolley: shopping_trolley,
        hammer_and_wrench: hammer_and_wrench,
        shield: shield,
        oil_drum: oil_drum,
        motorway: motorway,
        railway_track: railway_track,
        motor_boat: motor_boat,
        small_airplane: small_airplane,
        airplane_departure: airplane_departure,
        airplane_arriving: airplane_arriving,
        satellite: satellite,
        passenger_ship: passenger_ship,
        scooter: scooter,
        motor_scooter: motor_scooter,
        canoe: canoe,
        sled: sled,
        flying_saucer: flying_saucer,
        zipper_mouth_face: zipper_mouth_face,
        money_mouth_face: money_mouth_face,
        face_with_thermometer: face_with_thermometer,
        nerd_face: nerd_face,
        thinking_face: thinking_face,
        face_with_head_bandage: face_with_head_bandage,
        robot_face: robot_face,
        hugging_face: hugging_face,
        the_horns: the_horns,
        sign_of_the_horns: sign_of_the_horns,
        call_me_hand: call_me_hand,
        raised_back_of_hand: raised_back_of_hand,
        handshake: handshake,
        crossed_fingers: crossed_fingers,
        hand_with_index_and_middle_fingers_crossed: hand_with_index_and_middle_fingers_crossed,
        i_love_you_hand_sign: i_love_you_hand_sign,
        face_with_cowboy_hat: face_with_cowboy_hat,
        clown_face: clown_face,
        nauseated_face: nauseated_face,
        rolling_on_the_floor_laughing: rolling_on_the_floor_laughing,
        drooling_face: drooling_face,
        lying_face: lying_face,
        face_palm: face_palm,
        sneezing_face: sneezing_face,
        face_with_raised_eyebrow: face_with_raised_eyebrow,
        face_with_one_eyebrow_raised: face_with_one_eyebrow_raised,
        grinning_face_with_star_eyes: grinning_face_with_star_eyes,
        zany_face: zany_face,
        grinning_face_with_one_large_and_one_small_eye: grinning_face_with_one_large_and_one_small_eye,
        shushing_face: shushing_face,
        face_with_finger_covering_closed_lips: face_with_finger_covering_closed_lips,
        face_with_symbols_on_mouth: face_with_symbols_on_mouth,
        serious_face_with_symbols_covering_mouth: serious_face_with_symbols_covering_mouth,
        face_with_hand_over_mouth: face_with_hand_over_mouth,
        smiling_face_with_smiling_eyes_and_hand_covering_mouth: smiling_face_with_smiling_eyes_and_hand_covering_mouth,
        face_vomiting: face_vomiting,
        face_with_open_mouth_vomiting: face_with_open_mouth_vomiting,
        exploding_head: exploding_head,
        shocked_face_with_exploding_head: shocked_face_with_exploding_head,
        pregnant_woman: pregnant_woman,
        palms_up_together: palms_up_together,
        selfie: selfie,
        prince: prince,
        man_in_tuxedo: man_in_tuxedo,
        mrs_claus: mrs_claus,
        mother_christmas: mother_christmas,
        shrug: shrug,
        person_doing_cartwheel: person_doing_cartwheel,
        juggling: juggling,
        fencer: fencer,
        wrestlers: wrestlers,
        water_polo: water_polo,
        handball: handball,
        wilted_flower: wilted_flower,
        drum_with_drumsticks: drum_with_drumsticks,
        clinking_glasses: clinking_glasses,
        tumbler_glass: tumbler_glass,
        spoon: spoon,
        goal_net: goal_net,
        first_place_medal: first_place_medal,
        second_place_medal: second_place_medal,
        third_place_medal: third_place_medal,
        boxing_glove: boxing_glove,
        martial_arts_uniform: martial_arts_uniform,
        curling_stone: curling_stone,
        croissant: croissant,
        avocado: avocado,
        cucumber: cucumber,
        bacon: bacon,
        potato: potato,
        carrot: carrot,
        baguette_bread: baguette_bread,
        green_salad: green_salad,
        shallow_pan_of_food: shallow_pan_of_food,
        stuffed_flatbread: stuffed_flatbread,
        egg: egg,
        glass_of_milk: glass_of_milk,
        peanuts: peanuts,
        kiwifruit: kiwifruit,
        pancakes: pancakes,
        dumpling: dumpling,
        fortune_cookie: fortune_cookie,
        takeout_box: takeout_box,
        chopsticks: chopsticks,
        bowl_with_spoon: bowl_with_spoon,
        cup_with_straw: cup_with_straw,
        coconut: coconut,
        broccoli: broccoli,
        pie: pie,
        pretzel: pretzel,
        cut_of_meat: cut_of_meat,
        sandwich: sandwich,
        canned_food: canned_food,
        crab: crab,
        lion_face: lion_face,
        scorpion: scorpion,
        turkey: turkey,
        unicorn_face: unicorn_face,
        eagle: eagle,
        duck: duck,
        bat: bat,
        shark: shark,
        owl: owl,
        fox_face: fox_face,
        butterfly: butterfly,
        deer: deer,
        gorilla: gorilla,
        lizard: lizard,
        rhinoceros: rhinoceros,
        shrimp: shrimp,
        squid: squid,
        giraffe_face: giraffe_face,
        zebra_face: zebra_face,
        hedgehog: hedgehog,
        sauropod: sauropod,
        cricket: cricket,
        cheese_wedge: cheese_wedge,
        face_with_monocle: face_with_monocle,
        adult: adult,
        child: child,
        older_adult: older_adult,
        bearded_person: bearded_person,
        person_with_headscarf: person_with_headscarf,
        woman_in_steamy_room: woman_in_steamy_room,
        man_in_steamy_room: man_in_steamy_room,
        person_in_steamy_room: person_in_steamy_room,
        woman_climbing: woman_climbing,
        person_climbing: person_climbing,
        man_climbing: man_climbing,
        woman_in_lotus_position: woman_in_lotus_position,
        person_in_lotus_position: person_in_lotus_position,
        man_in_lotus_position: man_in_lotus_position,
        female_mage: female_mage,
        mage: mage,
        male_mage: male_mage,
        female_fairy: female_fairy,
        fairy: fairy,
        male_fairy: male_fairy,
        female_vampire: female_vampire,
        vampire: vampire,
        male_vampire: male_vampire,
        mermaid: mermaid,
        merman: merman,
        merperson: merperson,
        female_elf: female_elf,
        male_elf: male_elf,
        elf: elf,
        female_genie: female_genie,
        male_genie: male_genie,
        genie: genie,
        female_zombie: female_zombie,
        male_zombie: male_zombie,
        zombie: zombie,
        brain: brain,
        orange_heart: orange_heart,
        billed_cap: billed_cap,
        scarf: scarf,
        gloves: gloves,
        coat: coat,
        socks: socks,
        bangbang: bangbang,
        interrobang: interrobang,
        tm: tm,
        information_source: information_source,
        left_right_arrow: left_right_arrow,
        arrow_up_down: arrow_up_down,
        arrow_upper_left: arrow_upper_left,
        arrow_upper_right: arrow_upper_right,
        arrow_lower_right: arrow_lower_right,
        arrow_lower_left: arrow_lower_left,
        leftwards_arrow_with_hook: leftwards_arrow_with_hook,
        arrow_right_hook: arrow_right_hook,
        watch: watch,
        hourglass: hourglass,
        keyboard: keyboard,
        eject: eject,
        fast_forward: fast_forward,
        rewind: rewind,
        arrow_double_up: arrow_double_up,
        arrow_double_down: arrow_double_down,
        black_right_pointing_double_triangle_with_vertical_bar: black_right_pointing_double_triangle_with_vertical_bar,
        black_left_pointing_double_triangle_with_vertical_bar: black_left_pointing_double_triangle_with_vertical_bar,
        black_right_pointing_triangle_with_double_vertical_bar: black_right_pointing_triangle_with_double_vertical_bar,
        alarm_clock: alarm_clock,
        stopwatch: stopwatch,
        timer_clock: timer_clock,
        hourglass_flowing_sand: hourglass_flowing_sand,
        double_vertical_bar: double_vertical_bar,
        black_square_for_stop: black_square_for_stop,
        black_circle_for_record: black_circle_for_record,
        m: m,
        black_small_square: black_small_square,
        white_small_square: white_small_square,
        arrow_forward: arrow_forward,
        arrow_backward: arrow_backward,
        white_medium_square: white_medium_square,
        black_medium_square: black_medium_square,
        white_medium_small_square: white_medium_small_square,
        black_medium_small_square: black_medium_small_square,
        sunny: sunny,
        cloud: cloud,
        umbrella: umbrella,
        snowman: snowman,
        comet: comet,
        phone: phone,
        telephone: telephone,
        ballot_box_with_check: ballot_box_with_check,
        shamrock: shamrock,
        point_up: point_up,
        skull_and_crossbones: skull_and_crossbones,
        radioactive_sign: radioactive_sign,
        biohazard_sign: biohazard_sign,
        orthodox_cross: orthodox_cross,
        star_and_crescent: star_and_crescent,
        peace_symbol: peace_symbol,
        yin_yang: yin_yang,
        wheel_of_dharma: wheel_of_dharma,
        white_frowning_face: white_frowning_face,
        relaxed: relaxed,
        female_sign: female_sign,
        male_sign: male_sign,
        gemini: gemini,
        cancer: cancer,
        leo: leo,
        virgo: virgo,
        libra: libra,
        scorpius: scorpius,
        spades: spades,
        clubs: clubs,
        hearts: hearts,
        diamonds: diamonds,
        hotsprings: hotsprings,
        recycle: recycle,
        wheelchair: wheelchair,
        hammer_and_pick: hammer_and_pick,
        crossed_swords: crossed_swords,
        medical_symbol: medical_symbol,
        staff_of_aesculapius: staff_of_aesculapius,
        scales: scales,
        alembic: alembic,
        gear: gear,
        atom_symbol: atom_symbol,
        fleur_de_lis: fleur_de_lis,
        warning: warning,
        zap: zap,
        white_circle: white_circle,
        black_circle: black_circle,
        coffin: coffin,
        funeral_urn: funeral_urn,
        soccer: soccer,
        baseball: baseball,
        snowman_without_snow: snowman_without_snow,
        partly_sunny: partly_sunny,
        thunder_cloud_and_rain: thunder_cloud_and_rain,
        ophiuchus: ophiuchus,
        pick: pick,
        helmet_with_white_cross: helmet_with_white_cross,
        chains: chains,
        no_entry: no_entry,
        shinto_shrine: shinto_shrine,
        church: church,
        mountain: mountain,
        umbrella_on_ground: umbrella_on_ground,
        fountain: fountain,
        golf: golf,
        ferry: ferry,
        boat: boat,
        sailboat: sailboat,
        skier: skier,
        ice_skate: ice_skate,
        person_with_ball: person_with_ball,
        tent: tent,
        fuelpump: fuelpump,
        scissors: scissors,
        airplane: airplane,
        email: email,
        envelope: envelope,
        fist: fist,
        hand: hand,
        raised_hand: raised_hand,
        v: v,
        writing_hand: writing_hand,
        pencil2: pencil2,
        black_nib: black_nib,
        heavy_check_mark: heavy_check_mark,
        heavy_multiplication_x: heavy_multiplication_x,
        latin_cross: latin_cross,
        star_of_david: star_of_david,
        eight_spoked_asterisk: eight_spoked_asterisk,
        eight_pointed_black_star: eight_pointed_black_star,
        snowflake: snowflake,
        sparkle: sparkle,
        x: x,
        negative_squared_cross_mark: negative_squared_cross_mark,
        heavy_heart_exclamation_mark_ornament: heavy_heart_exclamation_mark_ornament,
        heart: heart,
        arrow_right: arrow_right,
        curly_loop: curly_loop,
        loop: loop,
        arrow_heading_up: arrow_heading_up,
        arrow_heading_down: arrow_heading_down,
        arrow_left: arrow_left,
        arrow_up: arrow_up,
        arrow_down: arrow_down,
        black_large_square: black_large_square,
        white_large_square: white_large_square,
        star: star,
        o: o,
        wavy_dash: wavy_dash,
        part_alternation_mark: part_alternation_mark,
        congratulations: congratulations,
        secret: secret,
        'default': emoji
    });

    var emojiByName = getCjsExportFromNamespace(emoji$1);

    /**
     * regex to parse emoji in a string - finds emoji, e.g. :coffee:
     */
    var emojiNameRegex = /:([a-zA-Z0-9_\-\+]+):/g;

    /**
     * regex to trim whitespace
     * use instead of String.prototype.trim() for IE8 support
     */
    var trimSpaceRegex = /^[\s\uFEFF\xA0]+|[\s\uFEFF\xA0]+$/g;

    /**
     * Removes colons on either side
     * of the string if present
     * @param  {string} str
     * @return {string}
     */
    function stripColons(str) {
      var colonIndex = str.indexOf(':');
      if (colonIndex > -1) {
        // :emoji: (http://www.emoji-cheat-sheet.com/)
        if (colonIndex === str.length - 1) {
          str = str.substring(0, colonIndex);
          return stripColons(str);
        } else {
          str = str.substr(colonIndex + 1);
          return stripColons(str);
        }
      }

      return str;
    }

    /**
     * Adds colons to either side
     * of the string
     * @param {string} str
     * @return {string}
     */
    function wrapColons(str) {
      return (typeof str === 'string' && str.length > 0) ? ':' + str + ':' : str;
    }

    /**
     * Ensure that the word is wrapped in colons
     * by only adding them, if they are not there.
     * @param {string} str
     * @return {string}
     */
    function ensureColons(str) {
      return (typeof str === 'string' && str[0] !== ':') ? wrapColons(str) : str;
    }

    // Remove the non-spacing-mark from the code, never send a stripped version
    // to the client, as it kills graphical emoticons.
    function stripNSB(code) {
      return code;
    }
    // Reversed hash table, where as emojiByName contains a { heart: '❤' }
    // dictionary emojiByCode contains { ❤: 'heart' }. The codes are normalized
    // to the text version.
    var emojiByCode = Object.keys(emojiByName).reduce(function (h, k) {
      h[stripNSB(emojiByName[k])] = k;
      return h;
    }, {});

    /**
     * Emoji namespace
     */
    var Emoji = {
      emoji: emojiByName,
    };

    /**
     * get emoji code from name
     * @param  {string} emoji
     * @return {string}
     */
    Emoji._get = function _get(emoji) {
      if (emojiByName.hasOwnProperty(emoji)) {
        return emojiByName[emoji];
      }

      return ensureColons(emoji);
    };

    /**
     * get emoji code from :emoji: string or name
     * @param  {string} emoji
     * @return {string}
     */
    Emoji.get = function get(emoji) {
      emoji = stripColons(emoji);

      return Emoji._get(emoji);
    };

    /**
     * find the emoji by either code or name
     * @param {string} nameOrCode The emoji to find, either `coffee`, `:coffee:` or `☕`;
     * @return {object}
     */
    Emoji.find = function find(nameOrCode) {
      return Emoji.findByName(nameOrCode) || Emoji.findByCode(nameOrCode);
    };

    /**
     * find the emoji by name
     * @param {string} name The emoji to find either `coffee` or `:coffee:`;
     * @return {object}
     */
    Emoji.findByName = function findByName(name) {
      var stripped = stripColons(name);
      var emoji = emojiByName[stripped];

      return emoji ? ({ emoji: emoji, key: stripped }) : undefined;
    };

    /**
     * find the emoji by code (emoji)
     * @param {string} code The emoji to find; for example `☕` or `☔`
     * @return {object}
     */
    Emoji.findByCode = function findByCode(code) {
      var stripped = stripNSB(code);
      var name = emojiByCode[stripped];

      // lookup emoji to ensure the Variant Form is returned
      return name ? ({ emoji: emojiByName[name], key: name }) : undefined;
    };


    /**
     * Check if an emoji is known by this library
     * @param {string} nameOrCode The emoji to validate, either `coffee`, `:coffee:` or `☕`;
     * @return {object}
     */
    Emoji.hasEmoji = function hasEmoji(nameOrCode) {
      return Emoji.hasEmojiByName(nameOrCode) || Emoji.hasEmojiByCode(nameOrCode);
    };

    /**
     * Check if an emoji with given name is known by this library
     * @param {string} name The emoji to validate either `coffee` or `:coffee:`;
     * @return {object}
     */
    Emoji.hasEmojiByName = function hasEmojiByName(name) {
      var result = Emoji.findByName(name);
      return !!result && result.key === stripColons(name);
    };

    /**
     * Check if a given emoji is known by this library
     * @param {string} code The emoji to validate; for example `☕` or `☔`
     * @return {object}
     */
    Emoji.hasEmojiByCode = function hasEmojiByCode(code) {
      var result = Emoji.findByCode(code);
      return !!result && stripNSB(result.emoji) === stripNSB(code);
    };

    /**
     * get emoji name from code
     * @param  {string} emoji
     * @param  {boolean} includeColons should the result include the ::
     * @return {string}
     */
    Emoji.which = function which(emoji_code, includeColons) {
      var code = stripNSB(emoji_code);
      var word = emojiByCode[code];

      return includeColons ? wrapColons(word) : word;
    };

    /**
     * emojify a string (replace :emoji: with an emoji)
     * @param  {string} str
     * @param  {function} on_missing (gets emoji name without :: and returns a proper emoji if no emoji was found)
     * @param  {function} format (wrap the returned emoji in a custom element)
     * @return {string}
     */
    Emoji.emojify = function emojify(str, on_missing, format) {
      if (!str) return '';

      return str.split(emojiNameRegex) // parse emoji via regex
        .map(function parseEmoji(s, i) {
          // every second element is an emoji, e.g. "test :fast_forward:" -> [ "test ", "fast_forward" ]
          if (i % 2 === 0) return s;
          var emoji = Emoji._get(s);
          var isMissing = emoji.indexOf(':') > -1;

          if (isMissing && typeof on_missing === 'function') {
            return on_missing(s);
          }

          if (!isMissing && typeof format === 'function') {
            return format(emoji, s);
          }

          return emoji;
        })
        .join('') // convert back to string
        ;
    };

    /**
     * return a random emoji
     * @return {string}
     */
    Emoji.random = function random() {
      var emojiKeys = Object.keys(emojiByName);
      var randomIndex = Math.floor(Math.random() * emojiKeys.length);
      var key = emojiKeys[randomIndex];
      var emoji = Emoji._get(key);
      return { key: key, emoji: emoji };
    };

    /**
     *  return an collection of potential emoji matches
     *  @param {string} str
     *  @return {Array.<Object>}
     */
    Emoji.search = function search(str) {
      var emojiKeys = Object.keys(emojiByName);
      var matcher = stripColons(str);
      var matchingKeys = emojiKeys.filter(function (key) {
        return key.toString().indexOf(matcher) === 0;
      });
      return matchingKeys.map(function (key) {
        return {
          key: key,
          emoji: Emoji._get(key),
        };
      });
    };

    /**
     * unemojify a string (replace emoji with :emoji:)
     * @param  {string} str
     * @return {string}
     */
    Emoji.unemojify = function unemojify(str) {
      if (!str) return '';
      var words = lodash_toarray(str);

      return words.map(function (word) {
        return Emoji.which(word, true) || word;
      }).join('');
    };

    /**
     * replace emojis with replacement value
     * @param {string} str
     * @param {function|string} the string or callback function to replace the emoji with
     * @param {boolean} should trailing whitespaces be cleaned? Defaults false
     * @return {string}
     */
    Emoji.replace = function replace(str, replacement, cleanSpaces) {
      if (!str) return '';

      var replace = typeof replacement === 'function' ? replacement : function () { return replacement; };
      var words = lodash_toarray(str);

      var replaced = words.map(function (word, idx) {
        var emoji = Emoji.findByCode(word);
        
        if (emoji && cleanSpaces && words[idx + 1] === ' ') {
          words[idx + 1] = '';
        }

        return emoji ? replace(emoji) : word;
      }).join('');

      return cleanSpaces ? replaced.replace(trimSpaceRegex, '') : replaced;
    };


    /**
     * remove all emojis from a string
     * @param {string} str
     * @return {string}
     */
    Emoji.strip = function strip(str) {
      return Emoji.replace(str, '', true);
    };

    var emoji$2 = Emoji;

    var nodeEmoji = emoji$2;

    // https://github.com/substack/deep-freeze/blob/master/index.js
    /** @param {any} obj */
    function deepFreeze(obj) {
      Object.freeze(obj);

      var objIsFunction = typeof obj === 'function';

      Object.getOwnPropertyNames(obj).forEach(function(prop) {
        if (Object.hasOwnProperty.call(obj, prop)
        && obj[prop] !== null
        && (typeof obj[prop] === "object" || typeof obj[prop] === "function")
        // IE11 fix: https://github.com/highlightjs/highlight.js/issues/2318
        // TODO: remove in the future
        && (objIsFunction ? prop !== 'caller' && prop !== 'callee' && prop !== 'arguments' : true)
        && !Object.isFrozen(obj[prop])) {
          deepFreeze(obj[prop]);
        }
      });

      return obj;
    }

    class Response {
      /**
       * @param {CompiledMode} mode
       */
      constructor(mode) {
        // eslint-disable-next-line no-undefined
        if (mode.data === undefined) mode.data = {};

        this.data = mode.data;
      }

      ignoreMatch() {
        this.ignore = true;
      }
    }

    /**
     * @param {string} value
     * @returns {string}
     */
    function escapeHTML(value) {
      return value
        .replace(/&/g, '&amp;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;')
        .replace(/"/g, '&quot;')
        .replace(/'/g, '&#x27;');
    }

    /**
     * performs a shallow merge of multiple objects into one
     *
     * @template T
     * @param {T} original
     * @param {Record<string,any>[]} objects
     * @returns {T} a single new object
     */
    function inherit(original, ...objects) {
      /** @type Record<string,any> */
      var result = {};

      for (const key in original) {
        result[key] = original[key];
      }
      objects.forEach(function(obj) {
        for (const key in obj) {
          result[key] = obj[key];
        }
      });
      return /** @type {T} */ (result);
    }

    /* Stream merging */

    /**
     * @typedef Event
     * @property {'start'|'stop'} event
     * @property {number} offset
     * @property {Node} node
     */

    /**
     * @param {Node} node
     */
    function tag(node) {
      return node.nodeName.toLowerCase();
    }

    /**
     * @param {Node} node
     */
    function nodeStream(node) {
      /** @type Event[] */
      var result = [];
      (function _nodeStream(node, offset) {
        for (var child = node.firstChild; child; child = child.nextSibling) {
          if (child.nodeType === 3) {
            offset += child.nodeValue.length;
          } else if (child.nodeType === 1) {
            result.push({
              event: 'start',
              offset: offset,
              node: child
            });
            offset = _nodeStream(child, offset);
            // Prevent void elements from having an end tag that would actually
            // double them in the output. There are more void elements in HTML
            // but we list only those realistically expected in code display.
            if (!tag(child).match(/br|hr|img|input/)) {
              result.push({
                event: 'stop',
                offset: offset,
                node: child
              });
            }
          }
        }
        return offset;
      })(node, 0);
      return result;
    }

    /**
     * @param {any} original - the original stream
     * @param {any} highlighted - stream of the highlighted source
     * @param {string} value - the original source itself
     */
    function mergeStreams(original, highlighted, value) {
      var processed = 0;
      var result = '';
      var nodeStack = [];

      function selectStream() {
        if (!original.length || !highlighted.length) {
          return original.length ? original : highlighted;
        }
        if (original[0].offset !== highlighted[0].offset) {
          return (original[0].offset < highlighted[0].offset) ? original : highlighted;
        }

        /*
        To avoid starting the stream just before it should stop the order is
        ensured that original always starts first and closes last:

        if (event1 == 'start' && event2 == 'start')
          return original;
        if (event1 == 'start' && event2 == 'stop')
          return highlighted;
        if (event1 == 'stop' && event2 == 'start')
          return original;
        if (event1 == 'stop' && event2 == 'stop')
          return highlighted;

        ... which is collapsed to:
        */
        return highlighted[0].event === 'start' ? original : highlighted;
      }

      /**
       * @param {Node} node
       */
      function open(node) {
        /** @param {Attr} attr */
        function attr_str(attr) {
          return ' ' + attr.nodeName + '="' + escapeHTML(attr.value) + '"';
        }
        // @ts-ignore
        result += '<' + tag(node) + [].map.call(node.attributes, attr_str).join('') + '>';
      }

      /**
       * @param {Node} node
       */
      function close(node) {
        result += '</' + tag(node) + '>';
      }

      /**
       * @param {Event} event
       */
      function render(event) {
        (event.event === 'start' ? open : close)(event.node);
      }

      while (original.length || highlighted.length) {
        var stream = selectStream();
        result += escapeHTML(value.substring(processed, stream[0].offset));
        processed = stream[0].offset;
        if (stream === original) {
          /*
          On any opening or closing tag of the original markup we first close
          the entire highlighted node stack, then render the original tag along
          with all the following original tags at the same offset and then
          reopen all the tags on the highlighted stack.
          */
          nodeStack.reverse().forEach(close);
          do {
            render(stream.splice(0, 1)[0]);
            stream = selectStream();
          } while (stream === original && stream.length && stream[0].offset === processed);
          nodeStack.reverse().forEach(open);
        } else {
          if (stream[0].event === 'start') {
            nodeStack.push(stream[0].node);
          } else {
            nodeStack.pop();
          }
          render(stream.splice(0, 1)[0]);
        }
      }
      return result + escapeHTML(value.substr(processed));
    }

    var utils = /*#__PURE__*/Object.freeze({
      __proto__: null,
      escapeHTML: escapeHTML,
      inherit: inherit,
      nodeStream: nodeStream,
      mergeStreams: mergeStreams
    });

    /**
     * @typedef {object} Renderer
     * @property {(text: string) => void} addText
     * @property {(node: Node) => void} openNode
     * @property {(node: Node) => void} closeNode
     * @property {() => string} value
     */

    /** @typedef {{kind?: string, sublanguage?: boolean}} Node */
    /** @typedef {{walk: (r: Renderer) => void}} Tree */
    /** */

    const SPAN_CLOSE = '</span>';

    /**
     * Determines if a node needs to be wrapped in <span>
     *
     * @param {Node} node */
    const emitsWrappingTags = (node) => {
      return !!node.kind;
    };

    /** @type {Renderer} */
    class HTMLRenderer {
      /**
       * Creates a new HTMLRenderer
       *
       * @param {Tree} parseTree - the parse tree (must support `walk` API)
       * @param {{classPrefix: string}} options
       */
      constructor(parseTree, options) {
        this.buffer = "";
        this.classPrefix = options.classPrefix;
        parseTree.walk(this);
      }

      /**
       * Adds texts to the output stream
       *
       * @param {string} text */
      addText(text) {
        this.buffer += escapeHTML(text);
      }

      /**
       * Adds a node open to the output stream (if needed)
       *
       * @param {Node} node */
      openNode(node) {
        if (!emitsWrappingTags(node)) return;

        let className = node.kind;
        if (!node.sublanguage) {
          className = `${this.classPrefix}${className}`;
        }
        this.span(className);
      }

      /**
       * Adds a node close to the output stream (if needed)
       *
       * @param {Node} node */
      closeNode(node) {
        if (!emitsWrappingTags(node)) return;

        this.buffer += SPAN_CLOSE;
      }

      /**
       * returns the accumulated buffer
      */
      value() {
        return this.buffer;
      }

      // helpers

      /**
       * Builds a span element
       *
       * @param {string} className */
      span(className) {
        this.buffer += `<span class="${className}">`;
      }
    }

    /** @typedef {{kind?: string, sublanguage?: boolean, children: Node[]} | string} Node */
    /** @typedef {{kind?: string, sublanguage?: boolean, children: Node[]} } DataNode */
    /**  */

    class TokenTree {
      constructor() {
        /** @type DataNode */
        this.rootNode = { children: [] };
        this.stack = [this.rootNode];
      }

      get top() {
        return this.stack[this.stack.length - 1];
      }

      get root() { return this.rootNode; }

      /** @param {Node} node */
      add(node) {
        this.top.children.push(node);
      }

      /** @param {string} kind */
      openNode(kind) {
        /** @type Node */
        const node = { kind, children: [] };
        this.add(node);
        this.stack.push(node);
      }

      closeNode() {
        if (this.stack.length > 1) {
          return this.stack.pop();
        }
        // eslint-disable-next-line no-undefined
        return undefined;
      }

      closeAllNodes() {
        while (this.closeNode());
      }

      toJSON() {
        return JSON.stringify(this.rootNode, null, 4);
      }

      /**
       * @typedef { import("./html_renderer").Renderer } Renderer
       * @param {Renderer} builder
       */
      walk(builder) {
        // this does not
        return this.constructor._walk(builder, this.rootNode);
        // this works
        // return TokenTree._walk(builder, this.rootNode);
      }

      /**
       * @param {Renderer} builder
       * @param {Node} node
       */
      static _walk(builder, node) {
        if (typeof node === "string") {
          builder.addText(node);
        } else if (node.children) {
          builder.openNode(node);
          node.children.forEach((child) => this._walk(builder, child));
          builder.closeNode(node);
        }
        return builder;
      }

      /**
       * @param {Node} node
       */
      static _collapse(node) {
        if (typeof node === "string") return;
        if (!node.children) return;

        if (node.children.every(el => typeof el === "string")) {
          // node.text = node.children.join("");
          // delete node.children;
          node.children = [node.children.join("")];
        } else {
          node.children.forEach((child) => {
            TokenTree._collapse(child);
          });
        }
      }
    }

    /**
      Currently this is all private API, but this is the minimal API necessary
      that an Emitter must implement to fully support the parser.

      Minimal interface:

      - addKeyword(text, kind)
      - addText(text)
      - addSublanguage(emitter, subLanguageName)
      - finalize()
      - openNode(kind)
      - closeNode()
      - closeAllNodes()
      - toHTML()

    */

    /**
     * @implements {Emitter}
     */
    class TokenTreeEmitter extends TokenTree {
      /**
       * @param {*} options
       */
      constructor(options) {
        super();
        this.options = options;
      }

      /**
       * @param {string} text
       * @param {string} kind
       */
      addKeyword(text, kind) {
        if (text === "") { return; }

        this.openNode(kind);
        this.addText(text);
        this.closeNode();
      }

      /**
       * @param {string} text
       */
      addText(text) {
        if (text === "") { return; }

        this.add(text);
      }

      /**
       * @param {Emitter & {root: DataNode}} emitter
       * @param {string} name
       */
      addSublanguage(emitter, name) {
        /** @type DataNode */
        const node = emitter.root;
        node.kind = name;
        node.sublanguage = true;
        this.add(node);
      }

      toHTML() {
        const renderer = new HTMLRenderer(this, this.options);
        return renderer.value();
      }

      finalize() {
        return true;
      }
    }

    /**
     * @param {string} value
     * @returns {RegExp}
     * */
    function escape$4(value) {
      return new RegExp(value.replace(/[-/\\^$*+?.()|[\]{}]/g, '\\$&'), 'm');
    }

    /**
     * @param {RegExp | string } re
     * @returns {string}
     */
    function source(re) {
      if (!re) return null;
      if (typeof re === "string") return re;

      return re.source;
    }

    /**
     * @param {...(RegExp | string) } args
     * @returns {string}
     */
    function concat(...args) {
      const joined = args.map((x) => source(x)).join("");
      return joined;
    }

    /**
     * @param {RegExp} re
     * @returns {number}
     */
    function countMatchGroups(re) {
      return (new RegExp(re.toString() + '|')).exec('').length - 1;
    }

    /**
     * Does lexeme start with a regular expression match at the beginning
     * @param {RegExp} re
     * @param {string} lexeme
     */
    function startsWith(re, lexeme) {
      var match = re && re.exec(lexeme);
      return match && match.index === 0;
    }

    // join logically computes regexps.join(separator), but fixes the
    // backreferences so they continue to match.
    // it also places each individual regular expression into it's own
    // match group, keeping track of the sequencing of those match groups
    // is currently an exercise for the caller. :-)
    /**
     * @param {(string | RegExp)[]} regexps
     * @param {string} separator
     * @returns {string}
     */
    function join(regexps, separator = "|") {
      // backreferenceRe matches an open parenthesis or backreference. To avoid
      // an incorrect parse, it additionally matches the following:
      // - [...] elements, where the meaning of parentheses and escapes change
      // - other escape sequences, so we do not misparse escape sequences as
      //   interesting elements
      // - non-matching or lookahead parentheses, which do not capture. These
      //   follow the '(' with a '?'.
      var backreferenceRe = /\[(?:[^\\\]]|\\.)*\]|\(\??|\\([1-9][0-9]*)|\\./;
      var numCaptures = 0;
      var ret = '';
      for (var i = 0; i < regexps.length; i++) {
        numCaptures += 1;
        var offset = numCaptures;
        var re = source(regexps[i]);
        if (i > 0) {
          ret += separator;
        }
        ret += "(";
        while (re.length > 0) {
          var match = backreferenceRe.exec(re);
          if (match == null) {
            ret += re;
            break;
          }
          ret += re.substring(0, match.index);
          re = re.substring(match.index + match[0].length);
          if (match[0][0] === '\\' && match[1]) {
            // Adjust the backreference.
            ret += '\\' + String(Number(match[1]) + offset);
          } else {
            ret += match[0];
            if (match[0] === '(') {
              numCaptures++;
            }
          }
        }
        ret += ")";
      }
      return ret;
    }

    // Common regexps
    const IDENT_RE = '[a-zA-Z]\\w*';
    const UNDERSCORE_IDENT_RE = '[a-zA-Z_]\\w*';
    const NUMBER_RE = '\\b\\d+(\\.\\d+)?';
    const C_NUMBER_RE = '(-?)(\\b0[xX][a-fA-F0-9]+|(\\b\\d+(\\.\\d*)?|\\.\\d+)([eE][-+]?\\d+)?)'; // 0x..., 0..., decimal, float
    const BINARY_NUMBER_RE = '\\b(0b[01]+)'; // 0b...
    const RE_STARTERS_RE = '!|!=|!==|%|%=|&|&&|&=|\\*|\\*=|\\+|\\+=|,|-|-=|/=|/|:|;|<<|<<=|<=|<|===|==|=|>>>=|>>=|>=|>>>|>>|>|\\?|\\[|\\{|\\(|\\^|\\^=|\\||\\|=|\\|\\||~';

    /**
    * @param { Partial<Mode> & {binary?: string | RegExp} } opts
    */
    const SHEBANG = (opts = {}) => {
      const beginShebang = /^#![ ]*\//;
      if (opts.binary) {
        opts.begin = concat(
          beginShebang,
          /.*\b/,
          opts.binary,
          /\b.*/);
      }
      return inherit({
        className: 'meta',
        begin: beginShebang,
        end: /$/,
        relevance: 0,
        /** @type {ModeCallback} */
        "on:begin": (m, resp) => {
          if (m.index !== 0) resp.ignoreMatch();
        }
      }, opts);
    };

    // Common modes
    const BACKSLASH_ESCAPE = {
      begin: '\\\\[\\s\\S]', relevance: 0
    };
    const APOS_STRING_MODE = {
      className: 'string',
      begin: '\'',
      end: '\'',
      illegal: '\\n',
      contains: [BACKSLASH_ESCAPE]
    };
    const QUOTE_STRING_MODE = {
      className: 'string',
      begin: '"',
      end: '"',
      illegal: '\\n',
      contains: [BACKSLASH_ESCAPE]
    };
    const PHRASAL_WORDS_MODE = {
      begin: /\b(a|an|the|are|I'm|isn't|don't|doesn't|won't|but|just|should|pretty|simply|enough|gonna|going|wtf|so|such|will|you|your|they|like|more)\b/
    };
    /**
     * Creates a comment mode
     *
     * @param {string | RegExp} begin
     * @param {string | RegExp} end
     * @param {Mode | {}} [modeOptions]
     * @returns {Partial<Mode>}
     */
    const COMMENT = function(begin, end, modeOptions = {}) {
      var mode = inherit(
        {
          className: 'comment',
          begin,
          end,
          contains: []
        },
        modeOptions
      );
      mode.contains.push(PHRASAL_WORDS_MODE);
      mode.contains.push({
        className: 'doctag',
        begin: '(?:TODO|FIXME|NOTE|BUG|OPTIMIZE|HACK|XXX):',
        relevance: 0
      });
      return mode;
    };
    const C_LINE_COMMENT_MODE = COMMENT('//', '$');
    const C_BLOCK_COMMENT_MODE = COMMENT('/\\*', '\\*/');
    const HASH_COMMENT_MODE = COMMENT('#', '$');
    const NUMBER_MODE = {
      className: 'number',
      begin: NUMBER_RE,
      relevance: 0
    };
    const C_NUMBER_MODE = {
      className: 'number',
      begin: C_NUMBER_RE,
      relevance: 0
    };
    const BINARY_NUMBER_MODE = {
      className: 'number',
      begin: BINARY_NUMBER_RE,
      relevance: 0
    };
    const CSS_NUMBER_MODE = {
      className: 'number',
      begin: NUMBER_RE + '(' +
        '%|em|ex|ch|rem' +
        '|vw|vh|vmin|vmax' +
        '|cm|mm|in|pt|pc|px' +
        '|deg|grad|rad|turn' +
        '|s|ms' +
        '|Hz|kHz' +
        '|dpi|dpcm|dppx' +
        ')?',
      relevance: 0
    };
    const REGEXP_MODE = {
      // this outer rule makes sure we actually have a WHOLE regex and not simply
      // an expression such as:
      //
      //     3 / something
      //
      // (which will then blow up when regex's `illegal` sees the newline)
      begin: /(?=\/[^/\n]*\/)/,
      contains: [{
        className: 'regexp',
        begin: /\//,
        end: /\/[gimuy]*/,
        illegal: /\n/,
        contains: [
          BACKSLASH_ESCAPE,
          {
            begin: /\[/,
            end: /\]/,
            relevance: 0,
            contains: [BACKSLASH_ESCAPE]
          }
        ]
      }]
    };
    const TITLE_MODE = {
      className: 'title',
      begin: IDENT_RE,
      relevance: 0
    };
    const UNDERSCORE_TITLE_MODE = {
      className: 'title',
      begin: UNDERSCORE_IDENT_RE,
      relevance: 0
    };
    const METHOD_GUARD = {
      // excludes method names from keyword processing
      begin: '\\.\\s*' + UNDERSCORE_IDENT_RE,
      relevance: 0
    };

    /**
     * Adds end same as begin mechanics to a mode
     *
     * Your mode must include at least a single () match group as that first match
     * group is what is used for comparison
     * @param {Partial<Mode>} mode
     */
    const END_SAME_AS_BEGIN = function(mode) {
      return Object.assign(mode,
        {
          /** @type {ModeCallback} */
          'on:begin': (m, resp) => { resp.data._beginMatch = m[1]; },
          /** @type {ModeCallback} */
          'on:end': (m, resp) => { if (resp.data._beginMatch !== m[1]) resp.ignoreMatch(); }
        });
    };

    var MODES = /*#__PURE__*/Object.freeze({
      __proto__: null,
      IDENT_RE: IDENT_RE,
      UNDERSCORE_IDENT_RE: UNDERSCORE_IDENT_RE,
      NUMBER_RE: NUMBER_RE,
      C_NUMBER_RE: C_NUMBER_RE,
      BINARY_NUMBER_RE: BINARY_NUMBER_RE,
      RE_STARTERS_RE: RE_STARTERS_RE,
      SHEBANG: SHEBANG,
      BACKSLASH_ESCAPE: BACKSLASH_ESCAPE,
      APOS_STRING_MODE: APOS_STRING_MODE,
      QUOTE_STRING_MODE: QUOTE_STRING_MODE,
      PHRASAL_WORDS_MODE: PHRASAL_WORDS_MODE,
      COMMENT: COMMENT,
      C_LINE_COMMENT_MODE: C_LINE_COMMENT_MODE,
      C_BLOCK_COMMENT_MODE: C_BLOCK_COMMENT_MODE,
      HASH_COMMENT_MODE: HASH_COMMENT_MODE,
      NUMBER_MODE: NUMBER_MODE,
      C_NUMBER_MODE: C_NUMBER_MODE,
      BINARY_NUMBER_MODE: BINARY_NUMBER_MODE,
      CSS_NUMBER_MODE: CSS_NUMBER_MODE,
      REGEXP_MODE: REGEXP_MODE,
      TITLE_MODE: TITLE_MODE,
      UNDERSCORE_TITLE_MODE: UNDERSCORE_TITLE_MODE,
      METHOD_GUARD: METHOD_GUARD,
      END_SAME_AS_BEGIN: END_SAME_AS_BEGIN
    });

    // keywords that should have no default relevance value
    var COMMON_KEYWORDS = 'of and for in not or if then'.split(' ');

    // compilation

    /**
     * Compiles a language definition result
     *
     * Given the raw result of a language definition (Language), compiles this so
     * that it is ready for highlighting code.
     * @param {Language} language
     * @returns {CompiledLanguage}
     */
    function compileLanguage(language) {
      /**
       * Builds a regex with the case sensativility of the current language
       *
       * @param {RegExp | string} value
       * @param {boolean} [global]
       */
      function langRe(value, global) {
        return new RegExp(
          source(value),
          'm' + (language.case_insensitive ? 'i' : '') + (global ? 'g' : '')
        );
      }

      /**
        Stores multiple regular expressions and allows you to quickly search for
        them all in a string simultaneously - returning the first match.  It does
        this by creating a huge (a|b|c) regex - each individual item wrapped with ()
        and joined by `|` - using match groups to track position.  When a match is
        found checking which position in the array has content allows us to figure
        out which of the original regexes / match groups triggered the match.

        The match object itself (the result of `Regex.exec`) is returned but also
        enhanced by merging in any meta-data that was registered with the regex.
        This is how we keep track of which mode matched, and what type of rule
        (`illegal`, `begin`, end, etc).
      */
      class MultiRegex {
        constructor() {
          this.matchIndexes = {};
          // @ts-ignore
          this.regexes = [];
          this.matchAt = 1;
          this.position = 0;
        }

        // @ts-ignore
        addRule(re, opts) {
          opts.position = this.position++;
          // @ts-ignore
          this.matchIndexes[this.matchAt] = opts;
          this.regexes.push([opts, re]);
          this.matchAt += countMatchGroups(re) + 1;
        }

        compile() {
          if (this.regexes.length === 0) {
            // avoids the need to check length every time exec is called
            // @ts-ignore
            this.exec = () => null;
          }
          const terminators = this.regexes.map(el => el[1]);
          this.matcherRe = langRe(join(terminators), true);
          this.lastIndex = 0;
        }

        /** @param {string} s */
        exec(s) {
          this.matcherRe.lastIndex = this.lastIndex;
          const match = this.matcherRe.exec(s);
          if (!match) { return null; }

          // eslint-disable-next-line no-undefined
          const i = match.findIndex((el, i) => i > 0 && el !== undefined);
          // @ts-ignore
          const matchData = this.matchIndexes[i];
          // trim off any earlier non-relevant match groups (ie, the other regex
          // match groups that make up the multi-matcher)
          match.splice(0, i);

          return Object.assign(match, matchData);
        }
      }

      /*
        Created to solve the key deficiently with MultiRegex - there is no way to
        test for multiple matches at a single location.  Why would we need to do
        that?  In the future a more dynamic engine will allow certain matches to be
        ignored.  An example: if we matched say the 3rd regex in a large group but
        decided to ignore it - we'd need to started testing again at the 4th
        regex... but MultiRegex itself gives us no real way to do that.

        So what this class creates MultiRegexs on the fly for whatever search
        position they are needed.

        NOTE: These additional MultiRegex objects are created dynamically.  For most
        grammars most of the time we will never actually need anything more than the
        first MultiRegex - so this shouldn't have too much overhead.

        Say this is our search group, and we match regex3, but wish to ignore it.

          regex1 | regex2 | regex3 | regex4 | regex5    ' ie, startAt = 0

        What we need is a new MultiRegex that only includes the remaining
        possibilities:

          regex4 | regex5                               ' ie, startAt = 3

        This class wraps all that complexity up in a simple API... `startAt` decides
        where in the array of expressions to start doing the matching. It
        auto-increments, so if a match is found at position 2, then startAt will be
        set to 3.  If the end is reached startAt will return to 0.

        MOST of the time the parser will be setting startAt manually to 0.
      */
      class ResumableMultiRegex {
        constructor() {
          // @ts-ignore
          this.rules = [];
          // @ts-ignore
          this.multiRegexes = [];
          this.count = 0;

          this.lastIndex = 0;
          this.regexIndex = 0;
        }

        // @ts-ignore
        getMatcher(index) {
          if (this.multiRegexes[index]) return this.multiRegexes[index];

          const matcher = new MultiRegex();
          this.rules.slice(index).forEach(([re, opts]) => matcher.addRule(re, opts));
          matcher.compile();
          this.multiRegexes[index] = matcher;
          return matcher;
        }

        considerAll() {
          this.regexIndex = 0;
        }

        // @ts-ignore
        addRule(re, opts) {
          this.rules.push([re, opts]);
          if (opts.type === "begin") this.count++;
        }

        /** @param {string} s */
        exec(s) {
          const m = this.getMatcher(this.regexIndex);
          m.lastIndex = this.lastIndex;
          const result = m.exec(s);
          if (result) {
            this.regexIndex += result.position + 1;
            if (this.regexIndex === this.count) { // wrap-around
              this.regexIndex = 0;
            }
          }

          // this.regexIndex = 0;
          return result;
        }
      }

      /**
       * Given a mode, builds a huge ResumableMultiRegex that can be used to walk
       * the content and find matches.
       *
       * @param {CompiledMode} mode
       * @returns {ResumableMultiRegex}
       */
      function buildModeRegex(mode) {
        const mm = new ResumableMultiRegex();

        mode.contains.forEach(term => mm.addRule(term.begin, { rule: term, type: "begin" }));

        if (mode.terminator_end) {
          mm.addRule(mode.terminator_end, { type: "end" });
        }
        if (mode.illegal) {
          mm.addRule(mode.illegal, { type: "illegal" });
        }

        return mm;
      }

      // TODO: We need negative look-behind support to do this properly
      /**
       * Skip a match if it has a preceding or trailing dot
       *
       * This is used for `beginKeywords` to prevent matching expressions such as
       * `bob.keyword.do()`. The mode compiler automatically wires this up as a
       * special _internal_ 'on:begin' callback for modes with `beginKeywords`
       * @param {RegExpMatchArray} match
       * @param {CallbackResponse} response
       */
      function skipIfhasPrecedingOrTrailingDot(match, response) {
        const before = match.input[match.index - 1];
        const after = match.input[match.index + match[0].length];
        if (before === "." || after === ".") {
          response.ignoreMatch();
        }
      }

      /** skip vs abort vs ignore
       *
       * @skip   - The mode is still entered and exited normally (and contains rules apply),
       *           but all content is held and added to the parent buffer rather than being
       *           output when the mode ends.  Mostly used with `sublanguage` to build up
       *           a single large buffer than can be parsed by sublanguage.
       *
       *             - The mode begin ands ends normally.
       *             - Content matched is added to the parent mode buffer.
       *             - The parser cursor is moved forward normally.
       *
       * @abort  - A hack placeholder until we have ignore.  Aborts the mode (as if it
       *           never matched) but DOES NOT continue to match subsequent `contains`
       *           modes.  Abort is bad/suboptimal because it can result in modes
       *           farther down not getting applied because an earlier rule eats the
       *           content but then aborts.
       *
       *             - The mode does not begin.
       *             - Content matched by `begin` is added to the mode buffer.
       *             - The parser cursor is moved forward accordingly.
       *
       * @ignore - Ignores the mode (as if it never matched) and continues to match any
       *           subsequent `contains` modes.  Ignore isn't technically possible with
       *           the current parser implementation.
       *
       *             - The mode does not begin.
       *             - Content matched by `begin` is ignored.
       *             - The parser cursor is not moved forward.
       */

      /**
       * Compiles an individual mode
       *
       * This can raise an error if the mode contains certain detectable known logic
       * issues.
       * @param {Mode} mode
       * @param {CompiledMode | null} [parent]
       * @returns {CompiledMode | never}
       */
      function compileMode(mode, parent) {
        const cmode = /** @type CompiledMode */ (mode);
        if (mode.compiled) return cmode;
        mode.compiled = true;

        // __beforeBegin is considered private API, internal use only
        mode.__beforeBegin = null;

        mode.keywords = mode.keywords || mode.beginKeywords;

        let kw_pattern = null;
        if (typeof mode.keywords === "object") {
          kw_pattern = mode.keywords.$pattern;
          delete mode.keywords.$pattern;
        }

        if (mode.keywords) {
          mode.keywords = compileKeywords(mode.keywords, language.case_insensitive);
        }

        // both are not allowed
        if (mode.lexemes && kw_pattern) {
          throw new Error("ERR: Prefer `keywords.$pattern` to `mode.lexemes`, BOTH are not allowed. (see mode reference) ");
        }

        // `mode.lexemes` was the old standard before we added and now recommend
        // using `keywords.$pattern` to pass the keyword pattern
        cmode.keywordPatternRe = langRe(mode.lexemes || kw_pattern || /\w+/, true);

        if (parent) {
          if (mode.beginKeywords) {
            // for languages with keywords that include non-word characters checking for
            // a word boundary is not sufficient, so instead we check for a word boundary
            // or whitespace - this does no harm in any case since our keyword engine
            // doesn't allow spaces in keywords anyways and we still check for the boundary
            // first
            mode.begin = '\\b(' + mode.beginKeywords.split(' ').join('|') + ')(?=\\b|\\s)';
            mode.__beforeBegin = skipIfhasPrecedingOrTrailingDot;
          }
          if (!mode.begin) mode.begin = /\B|\b/;
          cmode.beginRe = langRe(mode.begin);
          if (mode.endSameAsBegin) mode.end = mode.begin;
          if (!mode.end && !mode.endsWithParent) mode.end = /\B|\b/;
          if (mode.end) cmode.endRe = langRe(mode.end);
          cmode.terminator_end = source(mode.end) || '';
          if (mode.endsWithParent && parent.terminator_end) {
            cmode.terminator_end += (mode.end ? '|' : '') + parent.terminator_end;
          }
        }
        if (mode.illegal) cmode.illegalRe = langRe(mode.illegal);
        // eslint-disable-next-line no-undefined
        if (mode.relevance === undefined) mode.relevance = 1;
        if (!mode.contains) mode.contains = [];

        mode.contains = [].concat(...mode.contains.map(function(c) {
          return expand_or_clone_mode(c === 'self' ? mode : c);
        }));
        mode.contains.forEach(function(c) { compileMode(/** @type Mode */ (c), cmode); });

        if (mode.starts) {
          compileMode(mode.starts, parent);
        }

        cmode.matcher = buildModeRegex(cmode);
        return cmode;
      }

      // self is not valid at the top-level
      if (language.contains && language.contains.includes('self')) {
        throw new Error("ERR: contains `self` is not supported at the top-level of a language.  See documentation.");
      }
      return compileMode(/** @type Mode */ (language));
    }

    /**
     * Determines if a mode has a dependency on it's parent or not
     *
     * If a mode does have a parent dependency then often we need to clone it if
     * it's used in multiple places so that each copy points to the correct parent,
     * where-as modes without a parent can often safely be re-used at the bottom of
     * a mode chain.
     *
     * @param {Mode | null} mode
     * @returns {boolean} - is there a dependency on the parent?
     * */
    function dependencyOnParent(mode) {
      if (!mode) return false;

      return mode.endsWithParent || dependencyOnParent(mode.starts);
    }

    /**
     * Expands a mode or clones it if necessary
     *
     * This is necessary for modes with parental dependenceis (see notes on
     * `dependencyOnParent`) and for nodes that have `variants` - which must then be
     * exploded into their own individual modes at compile time.
     *
     * @param {Mode} mode
     * @returns {Mode | Mode[]}
     * */
    function expand_or_clone_mode(mode) {
      if (mode.variants && !mode.cached_variants) {
        mode.cached_variants = mode.variants.map(function(variant) {
          return inherit(mode, { variants: null }, variant);
        });
      }

      // EXPAND
      // if we have variants then essentially "replace" the mode with the variants
      // this happens in compileMode, where this function is called from
      if (mode.cached_variants) {
        return mode.cached_variants;
      }

      // CLONE
      // if we have dependencies on parents then we need a unique
      // instance of ourselves, so we can be reused with many
      // different parents without issue
      if (dependencyOnParent(mode)) {
        return inherit(mode, { starts: mode.starts ? inherit(mode.starts) : null });
      }

      if (Object.isFrozen(mode)) {
        return inherit(mode);
      }

      // no special dependency issues, just return ourselves
      return mode;
    }

    /***********************************************
      Keywords
    ***********************************************/

    /**
     * Given raw keywords from a language definition, compile them.
     *
     * @param {string | Record<string,string>} rawKeywords
     * @param {boolean} case_insensitive
     */
    function compileKeywords(rawKeywords, case_insensitive) {
      /** @type KeywordDict */
      var compiled_keywords = {};

      if (typeof rawKeywords === 'string') { // string
        splitAndCompile('keyword', rawKeywords);
      } else {
        Object.keys(rawKeywords).forEach(function(className) {
          splitAndCompile(className, rawKeywords[className]);
        });
      }
      return compiled_keywords;

      // ---

      /**
       * Compiles an individual list of keywords
       *
       * Ex: "for if when while|5"
       *
       * @param {string} className
       * @param {string} keywordList
       */
      function splitAndCompile(className, keywordList) {
        if (case_insensitive) {
          keywordList = keywordList.toLowerCase();
        }
        keywordList.split(' ').forEach(function(keyword) {
          var pair = keyword.split('|');
          compiled_keywords[pair[0]] = [className, scoreForKeyword(pair[0], pair[1])];
        });
      }
    }

    /**
     * Returns the proper score for a given keyword
     *
     * Also takes into account comment keywords, which will be scored 0 UNLESS
     * another score has been manually assigned.
     * @param {string} keyword
     * @param {string} [providedScore]
     */
    function scoreForKeyword(keyword, providedScore) {
      // manual scores always win over common keywords
      // so you can force a score of 1 if you really insist
      if (providedScore) {
        return Number(providedScore);
      }

      return commonKeyword(keyword) ? 0 : 1;
    }

    /**
     * Determines if a given keyword is common or not
     *
     * @param {string} keyword */
    function commonKeyword(keyword) {
      return COMMON_KEYWORDS.includes(keyword.toLowerCase());
    }

    var version$1 = "10.1.1";

    /*
    Syntax highlighting with language autodetection.
    https://highlightjs.org/
    */

    const escape$1$1 = escapeHTML;
    const inherit$1 = inherit;

    const { nodeStream: nodeStream$1, mergeStreams: mergeStreams$1 } = utils;
    const NO_MATCH = Symbol("nomatch");

    /**
     * @param {any} hljs - object that is extended (legacy)
     * @returns {HLJSApi}
     */
    const HLJS = function(hljs) {
      // Convenience variables for build-in objects
      /** @type {unknown[]} */
      var ArrayProto = [];

      // Global internal variables used within the highlight.js library.
      /** @type {Record<string, Language>} */
      var languages = {};
      /** @type {Record<string, string>} */
      var aliases = {};
      /** @type {HLJSPlugin[]} */
      var plugins = [];

      // safe/production mode - swallows more errors, tries to keep running
      // even if a single syntax or parse hits a fatal error
      var SAFE_MODE = true;
      var fixMarkupRe = /(^(<[^>]+>|\t|)+|\n)/gm;
      var LANGUAGE_NOT_FOUND = "Could not find the language '{}', did you forget to load/include a language module?";
      /** @type {Language} */
      const PLAINTEXT_LANGUAGE = { disableAutodetect: true, name: 'Plain text', contains: [] };

      // Global options used when within external APIs. This is modified when
      // calling the `hljs.configure` function.
      /** @type HLJSOptions */
      var options = {
        noHighlightRe: /^(no-?highlight)$/i,
        languageDetectRe: /\blang(?:uage)?-([\w-]+)\b/i,
        classPrefix: 'hljs-',
        tabReplace: null,
        useBR: false,
        languages: null,
        // beta configuration options, subject to change, welcome to discuss
        // https://github.com/highlightjs/highlight.js/issues/1086
        __emitter: TokenTreeEmitter
      };

      /* Utility functions */

      /**
       * Tests a language name to see if highlighting should be skipped
       * @param {string} languageName
       */
      function shouldNotHighlight(languageName) {
        return options.noHighlightRe.test(languageName);
      }

      /**
       * @param {HighlightedHTMLElement} block - the HTML element to determine language for
       */
      function blockLanguage(block) {
        var classes = block.className + ' ';

        classes += block.parentNode ? block.parentNode.className : '';

        // language-* takes precedence over non-prefixed class names.
        const match = options.languageDetectRe.exec(classes);
        if (match) {
          var language = getLanguage(match[1]);
          if (!language) {
            console.warn(LANGUAGE_NOT_FOUND.replace("{}", match[1]));
            console.warn("Falling back to no-highlight mode for this block.", block);
          }
          return language ? match[1] : 'no-highlight';
        }

        return classes
          .split(/\s+/)
          .find((_class) => shouldNotHighlight(_class) || getLanguage(_class));
      }

      /**
       * Core highlighting function.
       *
       * @param {string} languageName - the language to use for highlighting
       * @param {string} code - the code to highlight
       * @param {boolean} [ignoreIllegals] - whether to ignore illegal matches, default is to bail
       * @param {Mode} [continuation] - current continuation mode, if any
       *
       * @returns {HighlightResult} Result - an object that represents the result
       * @property {string} language - the language name
       * @property {number} relevance - the relevance score
       * @property {string} value - the highlighted HTML code
       * @property {string} code - the original raw code
       * @property {Mode} top - top of the current mode stack
       * @property {boolean} illegal - indicates whether any illegal matches were found
      */
      function highlight(languageName, code, ignoreIllegals, continuation) {
        /** @type {{ code: string, language: string, result?: any }} */
        var context = {
          code,
          language: languageName
        };
        // the plugin can change the desired language or the code to be highlighted
        // just be changing the object it was passed
        fire("before:highlight", context);

        // a before plugin can usurp the result completely by providing it's own
        // in which case we don't even need to call highlight
        var result = context.result ?
          context.result :
          _highlight(context.language, context.code, ignoreIllegals, continuation);

        result.code = context.code;
        // the plugin can change anything in result to suite it
        fire("after:highlight", result);

        return result;
      }

      /**
       * private highlight that's used internally and does not fire callbacks
       *
       * @param {string} languageName - the language to use for highlighting
       * @param {string} code - the code to highlight
       * @param {boolean} [ignoreIllegals] - whether to ignore illegal matches, default is to bail
       * @param {Mode} [continuation] - current continuation mode, if any
      */
      function _highlight(languageName, code, ignoreIllegals, continuation) {
        var codeToHighlight = code;

        /**
         * Return keyword data if a match is a keyword
         * @param {CompiledMode} mode - current mode
         * @param {RegExpMatchArray} match - regexp match data
         * @returns {KeywordData | false}
         */
        function keywordData(mode, match) {
          var matchText = language.case_insensitive ? match[0].toLowerCase() : match[0];
          return Object.prototype.hasOwnProperty.call(mode.keywords, matchText) && mode.keywords[matchText];
        }

        function processKeywords() {
          if (!top.keywords) {
            emitter.addText(mode_buffer);
            return;
          }

          let last_index = 0;
          top.keywordPatternRe.lastIndex = 0;
          let match = top.keywordPatternRe.exec(mode_buffer);
          let buf = "";

          while (match) {
            buf += mode_buffer.substring(last_index, match.index);
            const data = keywordData(top, match);
            if (data) {
              const [kind, keywordRelevance] = data;
              emitter.addText(buf);
              buf = "";

              relevance += keywordRelevance;
              emitter.addKeyword(match[0], kind);
            } else {
              buf += match[0];
            }
            last_index = top.keywordPatternRe.lastIndex;
            match = top.keywordPatternRe.exec(mode_buffer);
          }
          buf += mode_buffer.substr(last_index);
          emitter.addText(buf);
        }

        function processSubLanguage() {
          if (mode_buffer === "") return;
          /** @type HighlightResult */
          var result = null;

          if (typeof top.subLanguage === 'string') {
            if (!languages[top.subLanguage]) {
              emitter.addText(mode_buffer);
              return;
            }
            result = _highlight(top.subLanguage, mode_buffer, true, continuations[top.subLanguage]);
            continuations[top.subLanguage] = result.top;
          } else {
            result = highlightAuto(mode_buffer, top.subLanguage.length ? top.subLanguage : null);
          }

          // Counting embedded language score towards the host language may be disabled
          // with zeroing the containing mode relevance. Use case in point is Markdown that
          // allows XML everywhere and makes every XML snippet to have a much larger Markdown
          // score.
          if (top.relevance > 0) {
            relevance += result.relevance;
          }
          emitter.addSublanguage(result.emitter, result.language);
        }

        function processBuffer() {
          if (top.subLanguage != null) {
            processSubLanguage();
          } else {
            processKeywords();
          }
          mode_buffer = '';
        }

        /**
         * @param {Mode} mode - new mode to start
         */
        function startNewMode(mode) {
          if (mode.className) {
            emitter.openNode(mode.className);
          }
          top = Object.create(mode, { parent: { value: top } });
          return top;
        }

        /**
         * @param {CompiledMode } mode - the mode to potentially end
         * @param {RegExpMatchArray} match - the latest match
         * @param {string} matchPlusRemainder - match plus remainder of content
         * @returns {CompiledMode | void} - the next mode, or if void continue on in current mode
         */
        function endOfMode(mode, match, matchPlusRemainder) {
          let matched = startsWith(mode.endRe, matchPlusRemainder);

          if (matched) {
            if (mode["on:end"]) {
              const resp = new Response(mode);
              mode["on:end"](match, resp);
              if (resp.ignore) matched = false;
            }

            if (matched) {
              while (mode.endsParent && mode.parent) {
                mode = mode.parent;
              }
              return mode;
            }
          }
          // even if on:end fires an `ignore` it's still possible
          // that we might trigger the end node because of a parent mode
          if (mode.endsWithParent) {
            return endOfMode(mode.parent, match, matchPlusRemainder);
          }
        }

        /**
         * Handle matching but then ignoring a sequence of text
         *
         * @param {string} lexeme - string containing full match text
         */
        function doIgnore(lexeme) {
          if (top.matcher.regexIndex === 0) {
            // no more regexs to potentially match here, so we move the cursor forward one
            // space
            mode_buffer += lexeme[0];
            return 1;
          } else {
            // no need to move the cursor, we still have additional regexes to try and
            // match at this very spot
            continueScanAtSamePosition = true;
            return 0;
          }
        }

        /**
         * Handle the start of a new potential mode match
         *
         * @param {EnhancedMatch} match - the current match
         * @returns {number} how far to advance the parse cursor
         */
        function doBeginMatch(match) {
          var lexeme = match[0];
          var new_mode = match.rule;

          const resp = new Response(new_mode);
          // first internal before callbacks, then the public ones
          const beforeCallbacks = [new_mode.__beforeBegin, new_mode["on:begin"]];
          for (const cb of beforeCallbacks) {
            if (!cb) continue;
            cb(match, resp);
            if (resp.ignore) return doIgnore(lexeme);
          }

          if (new_mode && new_mode.endSameAsBegin) {
            new_mode.endRe = escape$4(lexeme);
          }

          if (new_mode.skip) {
            mode_buffer += lexeme;
          } else {
            if (new_mode.excludeBegin) {
              mode_buffer += lexeme;
            }
            processBuffer();
            if (!new_mode.returnBegin && !new_mode.excludeBegin) {
              mode_buffer = lexeme;
            }
          }
          startNewMode(new_mode);
          // if (mode["after:begin"]) {
          //   let resp = new Response(mode);
          //   mode["after:begin"](match, resp);
          // }
          return new_mode.returnBegin ? 0 : lexeme.length;
        }

        /**
         * Handle the potential end of mode
         *
         * @param {RegExpMatchArray} match - the current match
         */
        function doEndMatch(match) {
          var lexeme = match[0];
          var matchPlusRemainder = codeToHighlight.substr(match.index);

          var end_mode = endOfMode(top, match, matchPlusRemainder);
          if (!end_mode) { return NO_MATCH; }

          var origin = top;
          if (origin.skip) {
            mode_buffer += lexeme;
          } else {
            if (!(origin.returnEnd || origin.excludeEnd)) {
              mode_buffer += lexeme;
            }
            processBuffer();
            if (origin.excludeEnd) {
              mode_buffer = lexeme;
            }
          }
          do {
            if (top.className) {
              emitter.closeNode();
            }
            if (!top.skip && !top.subLanguage) {
              relevance += top.relevance;
            }
            top = top.parent;
          } while (top !== end_mode.parent);
          if (end_mode.starts) {
            if (end_mode.endSameAsBegin) {
              end_mode.starts.endRe = end_mode.endRe;
            }
            startNewMode(end_mode.starts);
          }
          return origin.returnEnd ? 0 : lexeme.length;
        }

        function processContinuations() {
          var list = [];
          for (var current = top; current !== language; current = current.parent) {
            if (current.className) {
              list.unshift(current.className);
            }
          }
          list.forEach(item => emitter.openNode(item));
        }

        /** @type {{type?: MatchType, index?: number, rule?: Mode}}} */
        var lastMatch = {};

        /**
         *  Process an individual match
         *
         * @param {string} textBeforeMatch - text preceeding the match (since the last match)
         * @param {EnhancedMatch} [match] - the match itself
         */
        function processLexeme(textBeforeMatch, match) {
          var lexeme = match && match[0];

          // add non-matched text to the current mode buffer
          mode_buffer += textBeforeMatch;

          if (lexeme == null) {
            processBuffer();
            return 0;
          }

          // we've found a 0 width match and we're stuck, so we need to advance
          // this happens when we have badly behaved rules that have optional matchers to the degree that
          // sometimes they can end up matching nothing at all
          // Ref: https://github.com/highlightjs/highlight.js/issues/2140
          if (lastMatch.type === "begin" && match.type === "end" && lastMatch.index === match.index && lexeme === "") {
            // spit the "skipped" character that our regex choked on back into the output sequence
            mode_buffer += codeToHighlight.slice(match.index, match.index + 1);
            if (!SAFE_MODE) {
              /** @type {AnnotatedError} */
              const err = new Error('0 width match regex');
              err.languageName = languageName;
              err.badRule = lastMatch.rule;
              throw err;
            }
            return 1;
          }
          lastMatch = match;

          if (match.type === "begin") {
            return doBeginMatch(match);
          } else if (match.type === "illegal" && !ignoreIllegals) {
            // illegal match, we do not continue processing
            /** @type {AnnotatedError} */
            const err = new Error('Illegal lexeme "' + lexeme + '" for mode "' + (top.className || '<unnamed>') + '"');
            err.mode = top;
            throw err;
          } else if (match.type === "end") {
            var processed = doEndMatch(match);
            if (processed !== NO_MATCH) {
              return processed;
            }
          }

          // edge case for when illegal matches $ (end of line) which is technically
          // a 0 width match but not a begin/end match so it's not caught by the
          // first handler (when ignoreIllegals is true)
          if (match.type === "illegal" && lexeme === "") {
            // advance so we aren't stuck in an infinite loop
            return 1;
          }

          // infinite loops are BAD, this is a last ditch catch all. if we have a
          // decent number of iterations yet our index (cursor position in our
          // parsing) still 3x behind our index then something is very wrong
          // so we bail
          if (iterations > 100000 && iterations > match.index * 3) {
            const err = new Error('potential infinite loop, way more iterations than matches');
            throw err;
          }

          /*
          Why might be find ourselves here?  Only one occasion now.  An end match that was
          triggered but could not be completed.  When might this happen?  When an `endSameasBegin`
          rule sets the end rule to a specific match.  Since the overall mode termination rule that's
          being used to scan the text isn't recompiled that means that any match that LOOKS like
          the end (but is not, because it is not an exact match to the beginning) will
          end up here.  A definite end match, but when `doEndMatch` tries to "reapply"
          the end rule and fails to match, we wind up here, and just silently ignore the end.

          This causes no real harm other than stopping a few times too many.
          */

          mode_buffer += lexeme;
          return lexeme.length;
        }

        var language = getLanguage(languageName);
        if (!language) {
          console.error(LANGUAGE_NOT_FOUND.replace("{}", languageName));
          throw new Error('Unknown language: "' + languageName + '"');
        }

        var md = compileLanguage(language);
        var result = '';
        /** @type {CompiledMode} */
        var top = continuation || md;
        /** @type Record<string,Mode> */
        var continuations = {}; // keep continuations for sub-languages
        var emitter = new options.__emitter(options);
        processContinuations();
        var mode_buffer = '';
        var relevance = 0;
        var index = 0;
        var iterations = 0;
        var continueScanAtSamePosition = false;

        try {
          top.matcher.considerAll();

          for (;;) {
            iterations++;
            if (continueScanAtSamePosition) {
              // only regexes not matched previously will now be
              // considered for a potential match
              continueScanAtSamePosition = false;
            } else {
              top.matcher.lastIndex = index;
              top.matcher.considerAll();
            }
            const match = top.matcher.exec(codeToHighlight);
            // console.log("match", match[0], match.rule && match.rule.begin)
            if (!match) break;

            const beforeMatch = codeToHighlight.substring(index, match.index);
            const processedCount = processLexeme(beforeMatch, match);
            index = match.index + processedCount;
          }
          processLexeme(codeToHighlight.substr(index));
          emitter.closeAllNodes();
          emitter.finalize();
          result = emitter.toHTML();

          return {
            relevance: relevance,
            value: result,
            language: languageName,
            illegal: false,
            emitter: emitter,
            top: top
          };
        } catch (err) {
          if (err.message && err.message.includes('Illegal')) {
            return {
              illegal: true,
              illegalBy: {
                msg: err.message,
                context: codeToHighlight.slice(index - 100, index + 100),
                mode: err.mode
              },
              sofar: result,
              relevance: 0,
              value: escape$1$1(codeToHighlight),
              emitter: emitter
            };
          } else if (SAFE_MODE) {
            return {
              illegal: false,
              relevance: 0,
              value: escape$1$1(codeToHighlight),
              emitter: emitter,
              language: languageName,
              top: top,
              errorRaised: err
            };
          } else {
            throw err;
          }
        }
      }

      /**
       * returns a valid highlight result, without actually doing any actual work,
       * auto highlight starts with this and it's possible for small snippets that
       * auto-detection may not find a better match
       * @param {string} code
       * @returns {HighlightResult}
       */
      function justTextHighlightResult(code) {
        const result = {
          relevance: 0,
          emitter: new options.__emitter(options),
          value: escape$1$1(code),
          illegal: false,
          top: PLAINTEXT_LANGUAGE
        };
        result.emitter.addText(code);
        return result;
      }

      /**
      Highlighting with language detection. Accepts a string with the code to
      highlight. Returns an object with the following properties:

      - language (detected language)
      - relevance (int)
      - value (an HTML string with highlighting markup)
      - second_best (object with the same structure for second-best heuristically
        detected language, may be absent)

        @param {string} code
        @param {Array<string>} [languageSubset]
        @returns {AutoHighlightResult}
      */
      function highlightAuto(code, languageSubset) {
        languageSubset = languageSubset || options.languages || Object.keys(languages);
        var result = justTextHighlightResult(code);
        var secondBest = result;
        languageSubset.filter(getLanguage).filter(autoDetection).forEach(function(name) {
          var current = _highlight(name, code, false);
          current.language = name;
          if (current.relevance > secondBest.relevance) {
            secondBest = current;
          }
          if (current.relevance > result.relevance) {
            secondBest = result;
            result = current;
          }
        });
        if (secondBest.language) {
          // second_best (with underscore) is the expected API
          result.second_best = secondBest;
        }
        return result;
      }

      /**
      Post-processing of the highlighted markup:

      - replace TABs with something more useful
      - replace real line-breaks with '<br>' for non-pre containers

        @param {string} html
        @returns {string}
      */
      function fixMarkup(html) {
        if (!(options.tabReplace || options.useBR)) {
          return html;
        }

        return html.replace(fixMarkupRe, match => {
          if (match === '\n') {
            return options.useBR ? '<br>' : match;
          } else if (options.tabReplace) {
            return match.replace(/\t/g, options.tabReplace);
          }
          return match;
        });
      }

      /**
       * Builds new class name for block given the language name
       *
       * @param {string} prevClassName
       * @param {string} [currentLang]
       * @param {string} [resultLang]
       */
      function buildClassName(prevClassName, currentLang, resultLang) {
        var language = currentLang ? aliases[currentLang] : resultLang;
        var result = [prevClassName.trim()];

        if (!prevClassName.match(/\bhljs\b/)) {
          result.push('hljs');
        }

        if (!prevClassName.includes(language)) {
          result.push(language);
        }

        return result.join(' ').trim();
      }

      /**
       * Applies highlighting to a DOM node containing code. Accepts a DOM node and
       * two optional parameters for fixMarkup.
       *
       * @param {HighlightedHTMLElement} element - the HTML element to highlight
      */
      function highlightBlock(element) {
        /** @type HTMLElement */
        let node = null;
        const language = blockLanguage(element);

        if (shouldNotHighlight(language)) return;

        fire("before:highlightBlock",
          { block: element, language: language });

        if (options.useBR) {
          node = document.createElement('div');
          node.innerHTML = element.innerHTML.replace(/\n/g, '').replace(/<br[ /]*>/g, '\n');
        } else {
          node = element;
        }
        const text = node.textContent;
        const result = language ? highlight(language, text, true) : highlightAuto(text);

        const originalStream = nodeStream$1(node);
        if (originalStream.length) {
          const resultNode = document.createElement('div');
          resultNode.innerHTML = result.value;
          result.value = mergeStreams$1(originalStream, nodeStream$1(resultNode), text);
        }
        result.value = fixMarkup(result.value);

        fire("after:highlightBlock", { block: element, result: result });

        element.innerHTML = result.value;
        element.className = buildClassName(element.className, language, result.language);
        element.result = {
          language: result.language,
          // TODO: remove with version 11.0
          re: result.relevance,
          relavance: result.relevance
        };
        if (result.second_best) {
          element.second_best = {
            language: result.second_best.language,
            // TODO: remove with version 11.0
            re: result.second_best.relevance,
            relavance: result.second_best.relevance
          };
        }
      }

      /**
       * Updates highlight.js global options with the passed options
       *
       * @param {{}} userOptions
       */
      function configure(userOptions) {
        options = inherit$1(options, userOptions);
      }

      /**
       * Highlights to all <pre><code> blocks on a page
       *
       * @type {Function & {called?: boolean}}
       */
      const initHighlighting = () => {
        if (initHighlighting.called) return;
        initHighlighting.called = true;

        var blocks = document.querySelectorAll('pre code');
        ArrayProto.forEach.call(blocks, highlightBlock);
      };

      // Higlights all when DOMContentLoaded fires
      function initHighlightingOnLoad() {
        // @ts-ignore
        window.addEventListener('DOMContentLoaded', initHighlighting, false);
      }

      /**
       * Register a language grammar module
       *
       * @param {string} languageName
       * @param {LanguageFn} languageDefinition
       */
      function registerLanguage(languageName, languageDefinition) {
        var lang = null;
        try {
          lang = languageDefinition(hljs);
        } catch (error) {
          console.error("Language definition for '{}' could not be registered.".replace("{}", languageName));
          // hard or soft error
          if (!SAFE_MODE) { throw error; } else { console.error(error); }
          // languages that have serious errors are replaced with essentially a
          // "plaintext" stand-in so that the code blocks will still get normal
          // css classes applied to them - and one bad language won't break the
          // entire highlighter
          lang = PLAINTEXT_LANGUAGE;
        }
        // give it a temporary name if it doesn't have one in the meta-data
        if (!lang.name) lang.name = languageName;
        languages[languageName] = lang;
        lang.rawDefinition = languageDefinition.bind(null, hljs);

        if (lang.aliases) {
          registerAliases(lang.aliases, { languageName });
        }
      }

      /**
       * @returns {string[]} List of language internal names
       */
      function listLanguages() {
        return Object.keys(languages);
      }

      /**
        intended usage: When one language truly requires another

        Unlike `getLanguage`, this will throw when the requested language
        is not available.

        @param {string} name - name of the language to fetch/require
        @returns {Language | never}
      */
      function requireLanguage(name) {
        var lang = getLanguage(name);
        if (lang) { return lang; }

        var err = new Error('The \'{}\' language is required, but not loaded.'.replace('{}', name));
        throw err;
      }

      /**
       * @param {string} name - name of the language to retrieve
       * @returns {Language | undefined}
       */
      function getLanguage(name) {
        name = (name || '').toLowerCase();
        return languages[name] || languages[aliases[name]];
      }

      /**
       *
       * @param {string|string[]} aliasList - single alias or list of aliases
       * @param {{languageName: string}} opts
       */
      function registerAliases(aliasList, { languageName }) {
        if (typeof aliasList === 'string') {
          aliasList = [aliasList];
        }
        aliasList.forEach(alias => { aliases[alias] = languageName; });
      }

      /**
       * Determines if a given language has auto-detection enabled
       * @param {string} name - name of the language
       */
      function autoDetection(name) {
        var lang = getLanguage(name);
        return lang && !lang.disableAutodetect;
      }

      /**
       * @param {HLJSPlugin} plugin
       */
      function addPlugin(plugin) {
        plugins.push(plugin);
      }

      /**
       *
       * @param {PluginEvent} event
       * @param {any} args
       */
      function fire(event, args) {
        var cb = event;
        plugins.forEach(function(plugin) {
          if (plugin[cb]) {
            plugin[cb](args);
          }
        });
      }

      /* Interface definition */

      Object.assign(hljs, {
        highlight,
        highlightAuto,
        fixMarkup,
        highlightBlock,
        configure,
        initHighlighting,
        initHighlightingOnLoad,
        registerLanguage,
        listLanguages,
        getLanguage,
        registerAliases,
        requireLanguage,
        autoDetection,
        inherit: inherit$1,
        addPlugin
      });

      hljs.debugMode = function() { SAFE_MODE = false; };
      hljs.safeMode = function() { SAFE_MODE = true; };
      hljs.versionString = version$1;

      for (const key in MODES) {
        // @ts-ignore
        if (typeof MODES[key] === "object") {
          // @ts-ignore
          deepFreeze(MODES[key]);
        }
      }

      // merge all the modes/regexs into our main object
      Object.assign(hljs, MODES);

      return hljs;
    };

    // export an "instance" of the highlighter
    var highlight = HLJS({});

    var core = highlight;

    /*
    Language: YAML
    Description: Yet Another Markdown Language
    Author: Stefan Wienert <stwienert@gmail.com>
    Contributors: Carl Baxter <carl@cbax.tech>
    Requires: ruby.js
    Website: https://yaml.org
    Category: common, config
    */
    function yaml(hljs) {
      var LITERALS = 'true false yes no null';

      // YAML spec allows non-reserved URI characters in tags.
      var URI_CHARACTERS = '[\\w#;/?:@&=+$,.~*\\\'()[\\]]+';

      // Define keys as starting with a word character
      // ...containing word chars, spaces, colons, forward-slashes, hyphens and periods
      // ...and ending with a colon followed immediately by a space, tab or newline.
      // The YAML spec allows for much more than this, but this covers most use-cases.
      var KEY = {
        className: 'attr',
        variants: [
          { begin: '\\w[\\w :\\/.-]*:(?=[ \t]|$)' },
          { begin: '"\\w[\\w :\\/.-]*":(?=[ \t]|$)' }, // double quoted keys
          { begin: '\'\\w[\\w :\\/.-]*\':(?=[ \t]|$)' } // single quoted keys
        ]
      };

      var TEMPLATE_VARIABLES = {
        className: 'template-variable',
        variants: [
          { begin: '{{', end: '}}' }, // jinja templates Ansible
          { begin: '%{', end: '}' } // Ruby i18n
        ]
      };
      var STRING = {
        className: 'string',
        relevance: 0,
        variants: [
          { begin: /'/, end: /'/ },
          { begin: /"/, end: /"/ },
          { begin: /\S+/ }
        ],
        contains: [
          hljs.BACKSLASH_ESCAPE,
          TEMPLATE_VARIABLES
        ]
      };

      // Strings inside of value containers (objects) can't contain braces,
      // brackets, or commas
      var CONTAINER_STRING = hljs.inherit(STRING, {
        variants: [
          { begin: /'/, end: /'/ },
          { begin: /"/, end: /"/ },
          { begin: /[^\s,{}[\]]+/ }
        ]
      });

      var DATE_RE = '[0-9]{4}(-[0-9][0-9]){0,2}';
      var TIME_RE = '([Tt \\t][0-9][0-9]?(:[0-9][0-9]){2})?';
      var FRACTION_RE = '(\\.[0-9]*)?';
      var ZONE_RE = '([ \\t])*(Z|[-+][0-9][0-9]?(:[0-9][0-9])?)?';
      var TIMESTAMP = {
        className: 'number',
        begin: '\\b' + DATE_RE + TIME_RE + FRACTION_RE + ZONE_RE + '\\b'
      };

      var VALUE_CONTAINER = {
        end: ',',
        endsWithParent: true,
        excludeEnd: true,
        contains: [],
        keywords: LITERALS,
        relevance: 0
      };
      var OBJECT = {
        begin: '{',
        end: '}',
        contains: [VALUE_CONTAINER],
        illegal: '\\n',
        relevance: 0
      };
      var ARRAY = {
        begin: '\\[',
        end: '\\]',
        contains: [VALUE_CONTAINER],
        illegal: '\\n',
        relevance: 0
      };

      var MODES = [
        KEY,
        {
          className: 'meta',
          begin: '^---\s*$',
          relevance: 10
        },
        { // multi line string
          // Blocks start with a | or > followed by a newline
          //
          // Indentation of subsequent lines must be the same to
          // be considered part of the block
          className: 'string',
          begin: '[\\|>]([0-9]?[+-])?[ ]*\\n( *)[\\S ]+\\n(\\2[\\S ]+\\n?)*'
        },
        { // Ruby/Rails erb
          begin: '<%[%=-]?',
          end: '[%-]?%>',
          subLanguage: 'ruby',
          excludeBegin: true,
          excludeEnd: true,
          relevance: 0
        },
        { // named tags
          className: 'type',
          begin: '!\\w+!' + URI_CHARACTERS
        },
        // https://yaml.org/spec/1.2/spec.html#id2784064
        { // verbatim tags
          className: 'type',
          begin: '!<' + URI_CHARACTERS + ">"
        },
        { // primary tags
          className: 'type',
          begin: '!' + URI_CHARACTERS
        },
        { // secondary tags
          className: 'type',
          begin: '!!' + URI_CHARACTERS
        },
        { // fragment id &ref
          className: 'meta',
          begin: '&' + hljs.UNDERSCORE_IDENT_RE + '$'
        },
        { // fragment reference *ref
          className: 'meta',
          begin: '\\*' + hljs.UNDERSCORE_IDENT_RE + '$'
        },
        { // array listing
          className: 'bullet',
          // TODO: remove |$ hack when we have proper look-ahead support
          begin: '\\-(?=[ ]|$)',
          relevance: 0
        },
        hljs.HASH_COMMENT_MODE,
        {
          beginKeywords: LITERALS,
          keywords: { literal: LITERALS }
        },
        TIMESTAMP,
        // numbers are any valid C-style number that
        // sit isolated from other words
        {
          className: 'number',
          begin: hljs.C_NUMBER_RE + '\\b'
        },
        OBJECT,
        ARRAY,
        STRING
      ];

      var VALUE_MODES = [...MODES];
      VALUE_MODES.pop();
      VALUE_MODES.push(CONTAINER_STRING);
      VALUE_CONTAINER.contains = VALUE_MODES;

      return {
        name: 'YAML',
        case_insensitive: true,
        aliases: ['yml', 'YAML'],
        contains: MODES
      };
    }

    var yaml_1 = yaml;

    const IDENT_RE$1 = '[A-Za-z$_][0-9A-Za-z$_]*';
    const KEYWORDS = [
      "as", // for exports
      "in",
      "of",
      "if",
      "for",
      "while",
      "finally",
      "var",
      "new",
      "function",
      "do",
      "return",
      "void",
      "else",
      "break",
      "catch",
      "instanceof",
      "with",
      "throw",
      "case",
      "default",
      "try",
      "switch",
      "continue",
      "typeof",
      "delete",
      "let",
      "yield",
      "const",
      "class",
      // JS handles these with a special rule
      // "get",
      // "set",
      "debugger",
      "async",
      "await",
      "static",
      "import",
      "from",
      "export",
      "extends"
    ];
    const LITERALS = [
      "true",
      "false",
      "null",
      "undefined",
      "NaN",
      "Infinity"
    ];

    const TYPES = [
      "Intl",
      "DataView",
      "Number",
      "Math",
      "Date",
      "String",
      "RegExp",
      "Object",
      "Function",
      "Boolean",
      "Error",
      "Symbol",
      "Set",
      "Map",
      "WeakSet",
      "WeakMap",
      "Proxy",
      "Reflect",
      "JSON",
      "Promise",
      "Float64Array",
      "Int16Array",
      "Int32Array",
      "Int8Array",
      "Uint16Array",
      "Uint32Array",
      "Float32Array",
      "Array",
      "Uint8Array",
      "Uint8ClampedArray",
      "ArrayBuffer"
    ];

    const ERROR_TYPES = [
      "EvalError",
      "InternalError",
      "RangeError",
      "ReferenceError",
      "SyntaxError",
      "TypeError",
      "URIError"
    ];

    const BUILT_IN_GLOBALS = [
      "setInterval",
      "setTimeout",
      "clearInterval",
      "clearTimeout",

      "require",
      "exports",

      "eval",
      "isFinite",
      "isNaN",
      "parseFloat",
      "parseInt",
      "decodeURI",
      "decodeURIComponent",
      "encodeURI",
      "encodeURIComponent",
      "escape",
      "unescape"
    ];

    const BUILT_IN_VARIABLES = [
      "arguments",
      "this",
      "super",
      "console",
      "window",
      "document",
      "localStorage",
      "module",
      "global" // Node.js
    ];

    const BUILT_INS = [].concat(
      BUILT_IN_GLOBALS,
      BUILT_IN_VARIABLES,
      TYPES,
      ERROR_TYPES
    );

    /**
     * @param {string} value
     * @returns {RegExp}
     * */

    /**
     * @param {RegExp | string } re
     * @returns {string}
     */
    function source$1(re) {
      if (!re) return null;
      if (typeof re === "string") return re;

      return re.source;
    }

    /**
     * @param {RegExp | string } re
     * @returns {string}
     */
    function lookahead(re) {
      return concat$1('(?=', re, ')');
    }

    /**
     * @param {...(RegExp | string) } args
     * @returns {string}
     */
    function concat$1(...args) {
      const joined = args.map((x) => source$1(x)).join("");
      return joined;
    }

    /*
    Language: JavaScript
    Description: JavaScript (JS) is a lightweight, interpreted, or just-in-time compiled programming language with first-class functions.
    Category: common, scripting
    Website: https://developer.mozilla.org/en-US/docs/Web/JavaScript
    */

    function javascript(hljs) {
      var IDENT_RE$1$1 = IDENT_RE$1;
      var FRAGMENT = {
        begin: '<>',
        end: '</>'
      };
      var XML_TAG = {
        begin: /<[A-Za-z0-9\\._:-]+/,
        end: /\/[A-Za-z0-9\\._:-]+>|\/>/
      };
      var KEYWORDS$1 = {
        $pattern: IDENT_RE$1,
        keyword: KEYWORDS.join(" "),
        literal: LITERALS.join(" "),
        built_in: BUILT_INS.join(" ")
      };
      var NUMBER = {
        className: 'number',
        variants: [
          { begin: '\\b(0[bB][01]+)n?' },
          { begin: '\\b(0[oO][0-7]+)n?' },
          { begin: hljs.C_NUMBER_RE + 'n?' }
        ],
        relevance: 0
      };
      var SUBST = {
        className: 'subst',
        begin: '\\$\\{', end: '\\}',
        keywords: KEYWORDS$1,
        contains: []  // defined later
      };
      var HTML_TEMPLATE = {
        begin: 'html`', end: '',
        starts: {
          end: '`', returnEnd: false,
          contains: [
            hljs.BACKSLASH_ESCAPE,
            SUBST
          ],
          subLanguage: 'xml',
        }
      };
      var CSS_TEMPLATE = {
        begin: 'css`', end: '',
        starts: {
          end: '`', returnEnd: false,
          contains: [
            hljs.BACKSLASH_ESCAPE,
            SUBST
          ],
          subLanguage: 'css',
        }
      };
      var TEMPLATE_STRING = {
        className: 'string',
        begin: '`', end: '`',
        contains: [
          hljs.BACKSLASH_ESCAPE,
          SUBST
        ]
      };
      SUBST.contains = [
        hljs.APOS_STRING_MODE,
        hljs.QUOTE_STRING_MODE,
        HTML_TEMPLATE,
        CSS_TEMPLATE,
        TEMPLATE_STRING,
        NUMBER,
        hljs.REGEXP_MODE
      ];
      var PARAMS_CONTAINS = SUBST.contains.concat([
        // eat recursive parens in sub expressions
        { begin: /\(/, end: /\)/,
          contains: ["self"].concat(SUBST.contains, [hljs.C_BLOCK_COMMENT_MODE, hljs.C_LINE_COMMENT_MODE])
        },
        hljs.C_BLOCK_COMMENT_MODE,
        hljs.C_LINE_COMMENT_MODE
      ]);
      var PARAMS = {
        className: 'params',
        begin: /\(/, end: /\)/,
        excludeBegin: true,
        excludeEnd: true,
        contains: PARAMS_CONTAINS
      };

      return {
        name: 'JavaScript',
        aliases: ['js', 'jsx', 'mjs', 'cjs'],
        keywords: KEYWORDS$1,
        contains: [
          hljs.SHEBANG({
            binary: "node",
            relevance: 5
          }),
          {
            className: 'meta',
            relevance: 10,
            begin: /^\s*['"]use (strict|asm)['"]/
          },
          hljs.APOS_STRING_MODE,
          hljs.QUOTE_STRING_MODE,
          HTML_TEMPLATE,
          CSS_TEMPLATE,
          TEMPLATE_STRING,
          hljs.C_LINE_COMMENT_MODE,
          hljs.COMMENT(
            '/\\*\\*',
            '\\*/',
            {
              relevance : 0,
              contains : [
                {
                  className : 'doctag',
                  begin : '@[A-Za-z]+',
                  contains : [
                    {
                      className: 'type',
                      begin: '\\{',
                      end: '\\}',
                      relevance: 0
                    },
                    {
                      className: 'variable',
                      begin: IDENT_RE$1$1 + '(?=\\s*(-)|$)',
                      endsParent: true,
                      relevance: 0
                    },
                    // eat spaces (not newlines) so we can find
                    // types or variables
                    {
                      begin: /(?=[^\n])\s/,
                      relevance: 0
                    },
                  ]
                }
              ]
            }
          ),
          hljs.C_BLOCK_COMMENT_MODE,
          NUMBER,
          { // object attr container
            begin: concat$1(/[{,\n]\s*/,
              // we need to look ahead to make sure that we actually have an
              // attribute coming up so we don't steal a comma from a potential
              // "value" container
              //
              // NOTE: this might not work how you think.  We don't actually always
              // enter this mode and stay.  Instead it might merely match `,
              // <comments up next>` and then immediately end after the , because it
              // fails to find any actual attrs. But this still does the job because
              // it prevents the value contain rule from grabbing this instead and
              // prevening this rule from firing when we actually DO have keys.
              lookahead(concat$1(
                // we also need to allow for multiple possible comments inbetween
                // the first key:value pairing
                /(((\/\/.*)|(\/\*(.|\n)*\*\/))\s*)*/,
                IDENT_RE$1$1 + '\\s*:'))),
            relevance: 0,
            contains: [
              {
                className: 'attr',
                begin: IDENT_RE$1$1 + lookahead('\\s*:'),
                relevance: 0,
              },
            ]
          },
          { // "value" container
            begin: '(' + hljs.RE_STARTERS_RE + '|\\b(case|return|throw)\\b)\\s*',
            keywords: 'return throw case',
            contains: [
              hljs.C_LINE_COMMENT_MODE,
              hljs.C_BLOCK_COMMENT_MODE,
              hljs.REGEXP_MODE,
              {
                className: 'function',
                // we have to count the parens to make sure we actually have the
                // correct bounding ( ) before the =>.  There could be any number of
                // sub-expressions inside also surrounded by parens.
                begin: '(\\([^(]*' +
                  '(\\([^(]*' +
                    '(\\([^(]*' +
                    '\\))?' +
                  '\\))?' +
                '\\)|' + hljs.UNDERSCORE_IDENT_RE + ')\\s*=>', returnBegin: true,
                end: '\\s*=>',
                contains: [
                  {
                    className: 'params',
                    variants: [
                      {
                        begin: hljs.UNDERSCORE_IDENT_RE
                      },
                      {
                        className: null,
                        begin: /\(\s*\)/,
                        skip: true
                      },
                      {
                        begin: /\(/, end: /\)/,
                        excludeBegin: true, excludeEnd: true,
                        keywords: KEYWORDS$1,
                        contains: PARAMS_CONTAINS
                      }
                    ]
                  }
                ]
              },
              { // could be a comma delimited list of params to a function call
                begin: /,/, relevance: 0,
              },
              {
                className: '',
                begin: /\s/,
                end: /\s*/,
                skip: true,
              },
              { // JSX
                variants: [
                  { begin: FRAGMENT.begin, end: FRAGMENT.end },
                  { begin: XML_TAG.begin, end: XML_TAG.end }
                ],
                subLanguage: 'xml',
                contains: [
                  {
                    begin: XML_TAG.begin, end: XML_TAG.end, skip: true,
                    contains: ['self']
                  }
                ]
              },
            ],
            relevance: 0
          },
          {
            className: 'function',
            beginKeywords: 'function', end: /\{/, excludeEnd: true,
            contains: [
              hljs.inherit(hljs.TITLE_MODE, {begin: IDENT_RE$1$1}),
              PARAMS
            ],
            illegal: /\[|%/
          },
          {
            begin: /\$[(.]/ // relevance booster for a pattern common to JS libs: `$(something)` and `$.something`
          },

          hljs.METHOD_GUARD,
          { // ES6 class
            className: 'class',
            beginKeywords: 'class', end: /[{;=]/, excludeEnd: true,
            illegal: /[:"\[\]]/,
            contains: [
              {beginKeywords: 'extends'},
              hljs.UNDERSCORE_TITLE_MODE
            ]
          },
          {
            beginKeywords: 'constructor', end: /\{/, excludeEnd: true
          },
          {
            begin: '(get|set)\\s+(?=' + IDENT_RE$1$1 + '\\()',
            end: /{/,
            keywords: "get set",
            contains: [
              hljs.inherit(hljs.TITLE_MODE, {begin: IDENT_RE$1$1}),
              { begin: /\(\)/ }, // eat to avoid empty params
              PARAMS
            ]

          }
        ],
        illegal: /#(?!!)/
      };
    }

    var javascript_1 = javascript;

    /*
    Language: JSON
    Description: JSON (JavaScript Object Notation) is a lightweight data-interchange format.
    Author: Ivan Sagalaev <maniac@softwaremaniacs.org>
    Website: http://www.json.org
    Category: common, protocols
    */

    function json(hljs) {
      var LITERALS = {literal: 'true false null'};
      var ALLOWED_COMMENTS = [
        hljs.C_LINE_COMMENT_MODE,
        hljs.C_BLOCK_COMMENT_MODE
      ];
      var TYPES = [
        hljs.QUOTE_STRING_MODE,
        hljs.C_NUMBER_MODE
      ];
      var VALUE_CONTAINER = {
        end: ',', endsWithParent: true, excludeEnd: true,
        contains: TYPES,
        keywords: LITERALS
      };
      var OBJECT = {
        begin: '{', end: '}',
        contains: [
          {
            className: 'attr',
            begin: /"/, end: /"/,
            contains: [hljs.BACKSLASH_ESCAPE],
            illegal: '\\n',
          },
          hljs.inherit(VALUE_CONTAINER, {begin: /:/})
        ].concat(ALLOWED_COMMENTS),
        illegal: '\\S'
      };
      var ARRAY = {
        begin: '\\[', end: '\\]',
        contains: [hljs.inherit(VALUE_CONTAINER)], // inherit is a workaround for a bug that makes shared modes with endsWithParent compile only the ending of one of the parents
        illegal: '\\S'
      };
      TYPES.push(OBJECT, ARRAY);
      ALLOWED_COMMENTS.forEach(function(rule) {
        TYPES.push(rule);
      });
      return {
        name: 'JSON',
        contains: TYPES,
        keywords: LITERALS,
        illegal: '\\S'
      };
    }

    var json_1 = json;

    // from https://raw.githubusercontent.com/highlightjs/highlight.js/master/src/styles/github.css
    const HLJS$1 = `
  /*

github.com style (c) Vasily Polovnyov <vast@whiteants.net>

*/

  .hljs {
    display: block;
    overflow-x: auto;
    padding: 0.5em;
    color: #333;
    background: #f8f8f8;
  }

  .hljs-comment,
  .hljs-quote {
    color: #998;
    font-style: italic;
  }

  .hljs-keyword,
  .hljs-selector-tag,
  .hljs-subst {
    color: #333;
    font-weight: bold;
  }

  .hljs-number,
  .hljs-literal,
  .hljs-variable,
  .hljs-template-variable,
  .hljs-tag .hljs-attr {
    color: #008080;
  }

  .hljs-string,
  .hljs-doctag {
    color: #d14;
  }

  .hljs-title,
  .hljs-section,
  .hljs-selector-id {
    color: #900;
    font-weight: bold;
  }

  .hljs-subst {
    font-weight: normal;
  }

  .hljs-type,
  .hljs-class .hljs-title {
    color: #458;
    font-weight: bold;
  }

  .hljs-tag,
  .hljs-name,
  .hljs-attribute {
    color: #000080;
    font-weight: normal;
  }

  .hljs-regexp,
  .hljs-link {
    color: #009926;
  }

  .hljs-symbol,
  .hljs-bullet {
    color: #990073;
  }

  .hljs-built_in,
  .hljs-builtin-name {
    color: #0086b3;
  }

  .hljs-meta {
    color: #999;
    font-weight: bold;
  }

  .hljs-deletion {
    background: #fdd;
  }

  .hljs-addition {
    background: #dfd;
  }

  .hljs-emphasis {
    font-style: italic;
  }

  .hljs-strong {
    font-weight: bold;
  }
`;
    // From https://github.com/sindresorhus/github-markdown-css/blob/gh-pages/github-markdown.css
    const GFM1 = `
  @font-face {
    font-family: octicons-link;
    src: url(data:font/woff;charset=utf-8;base64,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)
      format("woff");
  }
`;
    const GFM2 = `
  .markdown-body .octicon {
    display: inline-block;
    fill: currentColor;
    vertical-align: text-bottom;
  }

  .markdown-body .anchor {
    float: left;
    line-height: 1;
    margin-left: -20px;
    padding-right: 4px;
  }

  .markdown-body .anchor:focus {
    outline: none;
  }

  .markdown-body h1 .octicon-link,
  .markdown-body h2 .octicon-link,
  .markdown-body h3 .octicon-link,
  .markdown-body h4 .octicon-link,
  .markdown-body h5 .octicon-link,
  .markdown-body h6 .octicon-link {
    color: #1b1f23;
    vertical-align: middle;
    visibility: hidden;
  }

  .markdown-body h1:hover .anchor,
  .markdown-body h2:hover .anchor,
  .markdown-body h3:hover .anchor,
  .markdown-body h4:hover .anchor,
  .markdown-body h5:hover .anchor,
  .markdown-body h6:hover .anchor {
    text-decoration: none;
  }

  .markdown-body h1:hover .anchor .octicon-link,
  .markdown-body h2:hover .anchor .octicon-link,
  .markdown-body h3:hover .anchor .octicon-link,
  .markdown-body h4:hover .anchor .octicon-link,
  .markdown-body h5:hover .anchor .octicon-link,
  .markdown-body h6:hover .anchor .octicon-link {
    visibility: visible;
  }
  .markdown-body {
    -ms-text-size-adjust: 100%;
    -webkit-text-size-adjust: 100%;
    line-height: 1.5;
    font-family: -apple-system, BlinkMacSystemFont, Segoe UI, Helvetica, Arial,
      sans-serif, Apple Color Emoji, Segoe UI Emoji;
    font-size: 16px;
    line-height: 1.5;
    word-wrap: break-word;
  }

  .markdown-body .pl-c {
    color: #6a737d;
  }

  .markdown-body .pl-c1,
  .markdown-body .pl-s .pl-v {
    color: #005cc5;
  }

  .markdown-body .pl-e,
  .markdown-body .pl-en {
    color: #6f42c1;
  }

  .markdown-body .pl-s .pl-s1,
  .markdown-body .pl-smi {
    color: #24292e;
  }

  .markdown-body .pl-ent {
    color: #22863a;
  }

  .markdown-body .pl-k {
    color: #d73a49;
  }

  .markdown-body .pl-pds,
  .markdown-body .pl-s,
  .markdown-body .pl-s .pl-pse .pl-s1,
  .markdown-body .pl-sr,
  .markdown-body .pl-sr .pl-cce,
  .markdown-body .pl-sr .pl-sra,
  .markdown-body .pl-sr .pl-sre {
    color: #032f62;
  }

  .markdown-body .pl-smw,
  .markdown-body .pl-v {
    color: #e36209;
  }

  .markdown-body .pl-bu {
    color: #b31d28;
  }

  .markdown-body .pl-ii {
    color: #fafbfc;
    background-color: #b31d28;
  }

  .markdown-body .pl-c2 {
    color: #fafbfc;
    background-color: #d73a49;
  }

  .markdown-body .pl-c2:before {
    content: "^M";
  }

  .markdown-body .pl-sr .pl-cce {
    font-weight: 700;
    color: #22863a;
  }

  .markdown-body .pl-ml {
    color: #735c0f;
  }

  .markdown-body .pl-mh,
  .markdown-body .pl-mh .pl-en,
  .markdown-body .pl-ms {
    font-weight: 700;
    color: #005cc5;
  }

  .markdown-body .pl-mi {
    font-style: italic;
    color: #24292e;
  }

  .markdown-body .pl-mb {
    font-weight: 700;
    color: #24292e;
  }

  .markdown-body .pl-md {
    color: #b31d28;
    background-color: #ffeef0;
  }

  .markdown-body .pl-mi1 {
    color: #22863a;
    background-color: #f0fff4;
  }

  .markdown-body .pl-mc {
    color: #e36209;
    background-color: #ffebda;
  }

  .markdown-body .pl-mi2 {
    color: #f6f8fa;
    background-color: #005cc5;
  }

  .markdown-body .pl-mdr {
    font-weight: 700;
    color: #6f42c1;
  }

  .markdown-body .pl-ba {
    color: #586069;
  }

  .markdown-body .pl-sg {
    color: #959da5;
  }

  .markdown-body .pl-corl {
    text-decoration: underline;
    color: #032f62;
  }

  .markdown-body details {
    display: block;
  }

  .markdown-body summary {
    display: list-item;
  }

  .markdown-body a {
    background-color: initial;
  }

  .markdown-body a:active,
  .markdown-body a:hover {
    outline-width: 0;
  }

  .markdown-body strong {
    font-weight: inherit;
    font-weight: bolder;
  }

  .markdown-body h1 {
    font-size: 2em;
    margin: 0.67em 0;
  }

  .markdown-body img {
    border-style: none;
  }

  .markdown-body code,
  .markdown-body kbd,
  .markdown-body pre {
    font-family: monospace, monospace;
    font-size: 1em;
    background-color: var(--hcv-color-markdown-background);
  }

  .markdown-body hr {
    box-sizing: initial;
    height: 0;
    overflow: visible;
  }

  .markdown-body input {
    font: inherit;
    margin: 0;
  }

  .markdown-body input {
    overflow: visible;
  }

  .markdown-body [type="checkbox"] {
    box-sizing: border-box;
    padding: 0;
  }

  .markdown-body * {
    box-sizing: border-box;
  }

  .markdown-body input {
    font-family: inherit;
    font-size: inherit;
    line-height: inherit;
  }

  .markdown-body a {
    color: var(--link-text-color, var(--accent-color));
    font-weight: 600;
    text-decoration: none;
  }

  .markdown-body a:hover {
    text-decoration: underline;
  }

  .markdown-body strong {
    font-weight: 600;
  }

  .markdown-body hr {
    height: 0;
    margin: 15px 0;
    overflow: hidden;
    background: transparent;
    border: 0;
    border-bottom: 1px solid #dfe2e5;
  }

  .markdown-body hr:after,
  .markdown-body hr:before {
    display: table;
    content: "";
  }

  .markdown-body hr:after {
    clear: both;
  }

  .markdown-body table {
    border-spacing: 0;
    border-collapse: collapse;
  }

  .markdown-body td,
  .markdown-body th {
    padding: 0;
  }

  .markdown-body details summary {
    cursor: pointer;
  }

  .markdown-body kbd {
    display: inline-block;
    padding: 3px 5px;
    font: 11px SFMono-Regular, Consolas, Liberation Mono, Menlo, monospace;
    line-height: 10px;
    color: #444d56;
    vertical-align: middle;
    background-color: var(--hcv-color-markdown-background);
    border: 1px solid #d1d5da;
    border-radius: 3px;
    box-shadow: inset 0 -1px 0 #d1d5da;
  }

  .markdown-body h1,
  .markdown-body h2,
  .markdown-body h3,
  .markdown-body h4,
  .markdown-body h5,
  .markdown-body h6 {
    margin-top: 0;
    margin-bottom: 0;
  }

  .markdown-body h1 {
    font-size: 32px;
  }

  .markdown-body h1,
  .markdown-body h2 {
    font-weight: 600;
  }

  .markdown-body h2 {
    font-size: 24px;
  }

  .markdown-body h3 {
    font-size: 20px;
  }

  .markdown-body h3,
  .markdown-body h4 {
    font-weight: 600;
  }

  .markdown-body h4 {
    font-size: 16px;
  }

  .markdown-body h5 {
    font-size: 14px;
  }

  .markdown-body h5,
  .markdown-body h6 {
    font-weight: 600;
  }

  .markdown-body h6 {
    font-size: 12px;
  }

  .markdown-body p {
    margin-top: 0;
    margin-bottom: 10px;
  }

  .markdown-body blockquote {
    margin: 0;
  }

  .markdown-body ol,
  .markdown-body ul {
    padding-left: 0;
    margin-top: 0;
    margin-bottom: 0;
  }

  .markdown-body ol ol,
  .markdown-body ul ol {
    list-style-type: lower-roman;
  }

  .markdown-body ol ol ol,
  .markdown-body ol ul ol,
  .markdown-body ul ol ol,
  .markdown-body ul ul ol {
    list-style-type: lower-alpha;
  }

  .markdown-body dd {
    margin-left: 0;
  }

  .markdown-body code,
  .markdown-body pre {
    font-family: SFMono-Regular, Consolas, Liberation Mono, Menlo, monospace;
    font-size: 12px;
  }

  .markdown-body pre {
    margin-top: 0;
    margin-bottom: 0;
  }

  .markdown-body input::-webkit-inner-spin-button,
  .markdown-body input::-webkit-outer-spin-button {
    margin: 0;
    -webkit-appearance: none;
    appearance: none;
  }

  .markdown-body .border {
    border: 1px solid #e1e4e8 !important;
  }

  .markdown-body .border-0 {
    border: 0 !important;
  }

  .markdown-body .border-bottom {
    border-bottom: 1px solid #e1e4e8 !important;
  }

  .markdown-body .rounded-1 {
    border-radius: 3px !important;
  }

  .markdown-body .bg-white {
    background-color: #fff !important;
  }

  .markdown-body .bg-gray-light {
    background-color: #fafbfc !important;
  }

  .markdown-body .text-gray-light {
    color: #6a737d !important;
  }

  .markdown-body .mb-0 {
    margin-bottom: 0 !important;
  }

  .markdown-body .my-2 {
    margin-top: 8px !important;
    margin-bottom: 8px !important;
  }

  .markdown-body .pl-0 {
    padding-left: 0 !important;
  }

  .markdown-body .py-0 {
    padding-top: 0 !important;
    padding-bottom: 0 !important;
  }

  .markdown-body .pl-1 {
    padding-left: 4px !important;
  }

  .markdown-body .pl-2 {
    padding-left: 8px !important;
  }

  .markdown-body .py-2 {
    padding-top: 8px !important;
    padding-bottom: 8px !important;
  }

  .markdown-body .pl-3,
  .markdown-body .px-3 {
    padding-left: 16px !important;
  }

  .markdown-body .px-3 {
    padding-right: 16px !important;
  }

  .markdown-body .pl-4 {
    padding-left: 24px !important;
  }

  .markdown-body .pl-5 {
    padding-left: 32px !important;
  }

  .markdown-body .pl-6 {
    padding-left: 40px !important;
  }

  .markdown-body .f6 {
    font-size: 12px !important;
  }

  .markdown-body .lh-condensed {
    line-height: 1.25 !important;
  }

  .markdown-body .text-bold {
    font-weight: 600 !important;
  }

  .markdown-body .pl-7 {
    padding-left: 48px !important;
  }

  .markdown-body .pl-8 {
    padding-left: 64px !important;
  }

  .markdown-body .pl-9 {
    padding-left: 80px !important;
  }

  .markdown-body .pl-10 {
    padding-left: 96px !important;
  }

  .markdown-body .pl-11 {
    padding-left: 112px !important;
  }

  .markdown-body .pl-12 {
    padding-left: 128px !important;
  }

  .markdown-body hr {
    border-bottom-color: #eee;
  }

  .markdown-body kbd {
    display: inline-block;
    padding: 3px 5px;
    font: 11px SFMono-Regular, Consolas, Liberation Mono, Menlo, monospace;
    line-height: 10px;
    color: #444d56;
    vertical-align: middle;
    background-color: var(--hcv-color-markdown-background);
    border: 1px solid #d1d5da;
    border-radius: 3px;
    box-shadow: inset 0 -1px 0 #d1d5da;
  }

  .markdown-body:after,
  .markdown-body:before {
    display: table;
    content: "";
  }

  .markdown-body:after {
    clear: both;
  }

  .markdown-body > :first-child {
    margin-top: 0 !important;
  }

  .markdown-body > :last-child {
    margin-bottom: 0 !important;
  }

  .markdown-body a:not([href]) {
    color: inherit;
    text-decoration: none;
  }
`;
    const GFM3 = `
  .markdown-body blockquote,
  .markdown-body details,
  .markdown-body dl,
  .markdown-body ol,
  .markdown-body p,
  .markdown-body pre,
  .markdown-body table,
  .markdown-body ul {
    margin-top: 0;
    margin-bottom: 16px;
  }

  .markdown-body hr {
    height: 0.25em;
    padding: 0;
    margin: 24px 0;
    background-color: #e1e4e8;
    border: 0;
  }

  .markdown-body blockquote {
    padding: 0 1em;
    color: #6a737d;
    border-left: 0.25em solid #dfe2e5;
  }

  .markdown-body blockquote > :first-child {
    margin-top: 0;
  }

  .markdown-body blockquote > :last-child {
    margin-bottom: 0;
  }

  .markdown-body h1,
  .markdown-body h2,
  .markdown-body h3,
  .markdown-body h4,
  .markdown-body h5,
  .markdown-body h6 {
    margin-top: 24px;
    margin-bottom: 16px;
    font-weight: 600;
    line-height: 1.25;
  }

  .markdown-body h1 {
    font-size: 2em;
  }

  .markdown-body h1,
  .markdown-body h2 {
    padding-bottom: 0.3em;
    border-bottom: 1px solid #eaecef;
  }

  .markdown-body h2 {
    font-size: 1.5em;
  }

  .markdown-body h3 {
    font-size: 1.25em;
  }

  .markdown-body h4 {
    font-size: 1em;
  }

  .markdown-body h5 {
    font-size: 0.875em;
  }

  .markdown-body h6 {
    font-size: 0.85em;
    color: #6a737d;
  }

  .markdown-body ol,
  .markdown-body ul {
    padding-left: 2em;
  }

  .markdown-body ol ol,
  .markdown-body ol ul,
  .markdown-body ul ol,
  .markdown-body ul ul {
    margin-top: 0;
    margin-bottom: 0;
  }

  .markdown-body li {
    word-wrap: break-all;
  }

  .markdown-body li > p {
    margin-top: 16px;
  }

  .markdown-body li + li {
    margin-top: 0.25em;
  }

  .markdown-body dl {
    padding: 0;
  }

  .markdown-body dl dt {
    padding: 0;
    margin-top: 16px;
    font-size: 1em;
    font-style: italic;
    font-weight: 600;
  }

  .markdown-body dl dd {
    padding: 0 16px;
    margin-bottom: 16px;
  }

  .markdown-body table {
    display: block;
    width: 100%;
    overflow: auto;
  }

  .markdown-body table th {
    font-weight: 600;
  }

  .markdown-body table td,
  .markdown-body table th {
    padding: 6px 13px;
    border: 1px solid #dfe2e5;
  }

  .markdown-body table th {
    background-color: var(--primary-background-color, #fff);
  }

  .markdown-body table tr {
    background-color: hsla(var(--primary-background-color, #fff), 0.1);
    border-top: 1px solid #c6cbd1;
  }

  .markdown-body table tr:nth-child(2n) {
    background-color: var(--primary-background-color, #fff);
  }

  .markdown-body img {
    max-width: 100%;
    box-sizing: initial;
    background-color: #fff;
    border-radius: var(--ha-card-border-radius);
  }

  .markdown-body img[align="right"] {
    padding-left: 20px;
  }

  .markdown-body img[align="left"] {
    padding-right: 20px;
  }

  .markdown-body code {
    padding: 0.2em 0.4em;
    margin: 0;
    font-size: 85%;
    background-color: var(--hcv-color-markdown-background);
    border-radius: 3px;
  }

  .markdown-body pre {
    word-wrap: normal;
  }

  .markdown-body pre > code {
    padding: 0;
    margin: 0;
    font-size: 100%;
    word-break: normal;
    white-space: pre;
    background: transparent;
    border: 0;
  }

  .markdown-body .highlight {
    margin-bottom: 16px;
  }

  .markdown-body .highlight pre {
    margin-bottom: 0;
    word-break: normal;
  }

  .markdown-body .highlight pre,
  .markdown-body pre {
    padding: 16px;
    overflow: auto;
    font-size: 85%;
    line-height: 1.45;
    background-color: var(--hcv-color-markdown-background);
    border-radius: var(--ha-card-border-radius, 3px);
  }

  .markdown-body pre code {
    display: inline;
    max-width: auto;
    padding: 0;
    margin: 0;
    overflow: visible;
    line-height: inherit;
    word-wrap: normal;
    background-color: initial;
    border: 0;
  }

  .markdown-body .commit-tease-sha {
    display: inline-block;
    font-family: SFMono-Regular, Consolas, Liberation Mono, Menlo, monospace;
    font-size: 90%;
    color: #444d56;
  }

  .markdown-body .full-commit .btn-outline:not(:disabled):hover {
    color: #005cc5;
    border-color: #005cc5;
  }

  .markdown-body .blob-wrapper {
    overflow-x: auto;
    overflow-y: hidden;
  }

  .markdown-body .blob-wrapper-embedded {
    max-height: 240px;
    overflow-y: auto;
  }

  .markdown-body .blob-num {
    width: 1%;
    min-width: 50px;
    padding-right: 10px;
    padding-left: 10px;
    font-family: SFMono-Regular, Consolas, Liberation Mono, Menlo, monospace;
    font-size: 12px;
    line-height: 20px;
    color: rgba(27, 31, 35, 0.3);
    text-align: right;
    white-space: nowrap;
    vertical-align: top;
    cursor: pointer;
    -webkit-user-select: none;
    -moz-user-select: none;
    -ms-user-select: none;
    user-select: none;
  }

  .markdown-body .blob-num:hover {
    color: rgba(27, 31, 35, 0.6);
  }

  .markdown-body .blob-num:before {
    content: attr(data-line-number);
  }

  .markdown-body .blob-code {
    position: relative;
    padding-right: 10px;
    padding-left: 10px;
    line-height: 20px;
    vertical-align: top;
  }

  .markdown-body .blob-code-inner {
    overflow: visible;
    font-family: SFMono-Regular, Consolas, Liberation Mono, Menlo, monospace;
    font-size: 12px;
    color: #24292e;
    word-wrap: normal;
    white-space: pre;
  }

  .markdown-body .pl-token.active,
  .markdown-body .pl-token:hover {
    cursor: pointer;
    background: #ffea7f;
  }

  .markdown-body :checked + .radio-label {
    position: relative;
    z-index: 1;
    border-color: var(--link-text-color, var(--accent-color));
  }

  .markdown-body
    .select-menu-item
    input[type="radio"]:not(:checked)
    + .octicon-check,
  .markdown-body
    .select-menu-item
    input[type="radio"]:not(:checked)
    + .octicon-circle-slash {
    visibility: hidden;
  }

  .markdown-body .pl-7 {
    padding-left: 48px !important;
  }

  .markdown-body .pl-8 {
    padding-left: 64px !important;
  }

  .markdown-body .pl-9 {
    padding-left: 80px !important;
  }

  .markdown-body .pl-10 {
    padding-left: 96px !important;
  }

  .markdown-body .pl-11 {
    padding-left: 112px !important;
  }

  .markdown-body .pl-12 {
    padding-left: 128px !important;
  }

  .markdown-body .tab-size[data-tab-size="1"] {
    -moz-tab-size: 1;
    tab-size: 1;
  }

  .markdown-body .tab-size[data-tab-size="2"] {
    -moz-tab-size: 2;
    tab-size: 2;
  }

  .markdown-body .tab-size[data-tab-size="3"] {
    -moz-tab-size: 3;
    tab-size: 3;
  }

  .markdown-body .tab-size[data-tab-size="4"] {
    -moz-tab-size: 4;
    tab-size: 4;
  }

  .markdown-body .tab-size[data-tab-size="5"] {
    -moz-tab-size: 5;
    tab-size: 5;
  }

  .markdown-body .tab-size[data-tab-size="6"] {
    -moz-tab-size: 6;
    tab-size: 6;
  }

  .markdown-body .tab-size[data-tab-size="7"] {
    -moz-tab-size: 7;
    tab-size: 7;
  }

  .markdown-body .tab-size[data-tab-size="8"] {
    -moz-tab-size: 8;
    tab-size: 8;
  }

  .markdown-body .tab-size[data-tab-size="9"] {
    -moz-tab-size: 9;
    tab-size: 9;
  }

  .markdown-body .tab-size[data-tab-size="10"] {
    -moz-tab-size: 10;
    tab-size: 10;
  }

  .markdown-body .tab-size[data-tab-size="11"] {
    -moz-tab-size: 11;
    tab-size: 11;
  }

  .markdown-body .tab-size[data-tab-size="12"] {
    -moz-tab-size: 12;
    tab-size: 12;
  }

  .markdown-body .task-list-item {
    list-style-type: none;
  }

  .markdown-body .task-list-item + .task-list-item {
    margin-top: 3px;
  }

  .markdown-body .task-list-item input {
    margin: 0 0.2em 0.25em -1.6em;
    vertical-align: middle;
  }
`;
    const GFM = `${GFM1}${GFM2}${GFM3}`;

    core.registerLanguage("yaml", yaml_1);
    core.registerLanguage("javascript", javascript_1);
    core.registerLanguage("json", json_1);
    const marked$1 = marked_1;
    marked$1.setOptions({
        highlight: function (code, lang) {
            if (lang && core.getLanguage(lang)) {
                return core.highlight(lang, code, true).value;
            }
            else {
                return core.highlightAuto(code).value;
            }
        },
        breaks: true,
        gfm: true,
        tables: true,
        langPrefix: "",
    });
    class markdown {
        static convert(input) {
            return marked$1(input);
        }
        static html(input, repo) {
            // Convert emoji short codes to real emojis
            input = nodeEmoji.emojify(input);
            // Handle convertion to raw GitHub URL
            input = input.replace(/(https:\/\/github\.com\/.*.\/blob*.[^\s]+)/g, function (x) {
                let url = x
                    .replace("https://github.com/", "https://raw.githubusercontent.com/")
                    .replace("/blob/", "/");
                return url;
            });
            // Handle relative links
            input = input.replace(/\!\[*.*\]\(\w*\.\w*\)/g, function (x) {
                let url = x
                    .replace("(", `(https://raw.githubusercontent.com/${repo === null || repo === void 0 ? void 0 : repo.full_name}/master/`)
                    .replace("/blob/", "/");
                return url;
            });
            const content = document.createElement("div");
            content.className = "markdown-body";
            content.innerHTML = purify.sanitize(marked$1(input), {
                css: false,
            }).replace(/\<a href="http\w:\/\/.*.\">.*.\<\/a>\W/g, function (x) {
                return x.replace(/<a href=/gm, "<hacs-link url=").replace(/<\/a>/gm, "</hacs-link>");
            });
            const style = document.createElement("style");
            style.innerText = `${HLJS$1}${GFM}`;
            return html `${style}${content} `;
        }
    }

    let HacsGenericDialog = class HacsGenericDialog extends HacsDialogBase {
        constructor() {
            super(...arguments);
            this.markdown = false;
            this._getRepository = memoizeOne((repositories, repository) => repositories === null || repositories === void 0 ? void 0 : repositories.find((repo) => repo.id === repository));
        }
        render() {
            if (!this.active)
                return html ``;
            const repository = this._getRepository(this.repositories, this.repository);
            return html `
      <hacs-dialog .active=${this.active} .narrow=${this.narrow} .hass=${this.hass}>
        <div slot="header">${this.header || ""}</div>
        ${this.markdown
            ? this.repository
                ? markdown.html(this.content || "", repository)
                : markdown.html(this.content || "")
            : this.content || ""}
      </hacs-dialog>
    `;
        }
    };
    __decorate([
        property({ type: Boolean })
    ], HacsGenericDialog.prototype, "markdown", void 0);
    __decorate([
        property()
    ], HacsGenericDialog.prototype, "repository", void 0);
    __decorate([
        property()
    ], HacsGenericDialog.prototype, "header", void 0);
    __decorate([
        property()
    ], HacsGenericDialog.prototype, "content", void 0);
    HacsGenericDialog = __decorate([
        customElement("hacs-generic-dialog")
    ], HacsGenericDialog);

    let HacsDialog = class HacsDialog extends HacsDialogBase {
        constructor() {
            super(...arguments);
            this.noActions = false;
            this.hasContent = true;
            this.dynamicHeight = false;
            this.hasSearch = false;
            this.hasFilter = false;
        }
        render() {
            if (!this.active) {
                return html ``;
            }
            const sidebar = window.localStorage.getItem("dockedSidebar");
            const sidebarDocked = sidebar === '"docked"';
            return html `
      <div
        class=${classMap({
            backdrop: true,
            docked: sidebarDocked,
            "full-width": sidebar === '"always_hidden"',
            narrow: this.narrow,
        })}
      >
        <ha-card
          class=${classMap({
            dialog: true,
            "has-content": this.hasContent,
            "dynamic-height": this.dynamicHeight,
            narrow: this.narrow,
        })}
        >
          <div class="header-group">
            <ha-icon-button class="close" icon="mdi:close" @click=${this._close}></ha-icon-button>

            <div
              class=${classMap({
            header: true,
            "narrow-header": this.narrow,
        })}
            >
              <slot name="header"></slot>
            </div>
          </div>
          <slot name="search"></slot>
          <slot name="filter"></slot>
          <div
            @scroll=${this._scrollEvent}
            class=${classMap({
            "card-content": true,
            noactions: this.noActions && !this.hasSearch,
            "search-and-filter": this.hasSearch && this.hasFilter,
            "dynamic-height": !this.narrow && this.dynamicHeight,
            "narrow-content": this.narrow,
        })}
          >
            <div class="content">
              <slot></slot>
            </div>
          </div>
          ${!this.noActions
            ? html ` <div
                class=${classMap({
                "card-actions": true,
                "narrow-actions": this.narrow,
            })}
              >
                <slot name="actions"></slot>
              </div>`
            : ""}
        </ha-card>
      </div>
    `;
        }
        _scrollEvent(ev) {
            this.dispatchEvent(new CustomEvent("scroll", {
                detail: {
                    target: ev.target,
                },
                bubbles: true,
                composed: true,
            }));
        }
        _close() {
            this.dispatchEvent(new Event(this.secondary ? "hacs-secondary-dialog-closed" : "hacs-dialog-closed", {
                bubbles: true,
                composed: true,
            }));
        }
        static get styles() {
            return [
                HacsStyles,
                css `
        ha-card {
          background-color: var(--paper-dialog-background-color, var(--card-background-color));
          transition: none;
        }
        .header-group {
          position: absolute;
          width: 100%;
          display: flex;
          align-items: center;
          font-size: 20px;
          height: 65px;
          background-color: var(--primary-background-color);
          border-bottom: 1px solid var(--divider-color);
          font-weight: 400;
          font-family: var(--paper-font-body1_-_font-family);
          padding: 12px 16px;
          box-sizing: border-box;
        }
        .header {
          -webkit-font-smoothing: var(--paper-font-headline_-_-webkit-font-smoothing);
          font-family: var(--paper-font-headline_-_font-family);
          font-size: var(--paper-font-headline_-_font-size);
          font-weight: var(--paper-font-headline_-_font-weight);
          letter-spacing: var(--paper-font-headline_-_letter-spacing);
          line-height: 28px;
          max-height: 26px;
          opacity: var(--dark-primary-opacity);
          overflow: hidden;
          padding: 4px;
          text-overflow: ellipsis;
        }
        .close {
          pointer-events: auto;
          color: var(--sidebar-icon-color);
          --mdc-icon-size: 32px;
        }

        .card-content {
          position: relative;
          padding: 0;
          width: 100%;
          padding: 0;
          height: calc(100% - 136px);
          overflow-y: auto;
          overflow-x: hidden;
          -webkit-overflow-scrolling: touch;
          margin-top: 65px;
        }
        ha-card.dynamic-height .noactions {
          margin-bottom: -65px;
        }
        .noactions {
          height: calc(100% - 68px);
        }
        .content {
          padding: 8px;
          height: fit-content;
        }

        ::slotted([slot="actions"]) {
          position: absolute;
          width: calc(100% - 32px);
          display: flex;
          justify-content: space-between;
          align-items: center;
          font-size: 20px;
          height: 65px;
          background-color: var(--sidebar-background-color);
          font-weight: 400;
          color: var(--sidebar-text-color);
          font-family: var(--paper-font-body1_-_font-family);
          padding: 12px 16px;
          box-sizing: border-box;
        }

        .backdrop {
          background-color: rgba(0, 0, 0, 0.75);
          height: 100%;
          position: fixed;
          z-index: 1;
          top: 0;
          left: var(--app-drawer-width);
          width: calc(100% - var(--app-drawer-width));
        }

        .narrow {
          top: 0 !important;
          left: 0 !important;
          bottom: 0 !important;
          right: 0 !important;
          height: 100% !important;
          width: 100% !important;
          max-height: 100% !important;
          max-width: 100% !important;
        }

        .dialog {
          height: 300px;
          margin: auto;
          max-height: calc(100% - 130px);
          max-width: 90%;
          min-width: 350px;
          overflow-x: hidden;
          padding-bottom: 65px;
          padding: unset;
          top: 64px;
          width: fit-content;
          z-index: 2;
        }
        .has-content {
          height: var(--hacs-dialog-height, -webkit-fill-available);
          height: var(--hacs-dialog-height, 100%);
        }

        .dynamic-height {
          height: auto;
        }
        ha-card.dynamic-height {
          padding-bottom: 65px;
        }

        .search-and-filter {
          height: calc(100% - 166px) !important;
        }

        .full-width {
          --app-drawer-width: 0%;
        }
      `,
            ];
        }
    };
    __decorate([
        property({ type: Boolean })
    ], HacsDialog.prototype, "noActions", void 0);
    __decorate([
        property({ type: Boolean })
    ], HacsDialog.prototype, "hasContent", void 0);
    __decorate([
        property({ type: Boolean })
    ], HacsDialog.prototype, "dynamicHeight", void 0);
    __decorate([
        property({ type: Boolean })
    ], HacsDialog.prototype, "hasSearch", void 0);
    __decorate([
        property({ type: Boolean })
    ], HacsDialog.prototype, "hasFilter", void 0);
    HacsDialog = __decorate([
        customElement("hacs-dialog")
    ], HacsDialog);

    let HacsAboutDialog = class HacsAboutDialog extends HacsDialogBase {
        render() {
            if (!this.active)
                return html ``;
            return html `
      <hacs-dialog
        .active=${this.active}
        .narrow=${this.narrow}
        .hass=${this.hass}
        noActions
        dynamicHeight
      >
        <div slot="header">
          ${this.narrow ? "HACS" : "Home Assistant Community Store"}
        </div>
        <div class="content">
          ${markdown.html(`
**${localize("dialog_about.integration_version")}:** | ${this.configuration.version}
--|--
**${localize("dialog_about.frontend_version")}:** | ${version}
**${localize("common.repositories")}:** | ${this.repositories.length}
**${localize("dialog_about.installed_repositories")}:** | ${this.repositories.filter((repo) => repo.installed).length}

**${localize("dialog_about.useful_links")}:**

- [General documentation](https://hacs.xyz/)
- [Configuration](https://hacs.xyz/docs/configuration/start)
- [FAQ](https://hacs.xyz/docs/faq/what)
- [GitHub](https://github.com/hacs)
- [Discord](https://discord.gg/apgchf8)
- [Become a GitHub sponsor? ❤️](https://github.com/sponsors/ludeeus)
- [BuyMe~~Coffee~~Beer? 🍺🙈](https://buymeacoffee.com/ludeeus)

***

_Everything you find in HACS is **not** tested by Home Assistant, that includes HACS itself._
_The HACS and Home Assistant teams do not support **anything** you find here._
        `)}
        </div>
      </hacs-dialog>
    `;
        }
        static get styles() {
            return [
                css `
        .content {
          font-size: 16px;
          padding-top: 16px;
        }
      `,
            ];
        }
    };
    HacsAboutDialog = __decorate([
        customElement("hacs-about-dialog")
    ], HacsAboutDialog);

    async function updateLovelaceResources(hass, repository) {
        const resources = await fetchResources(hass);
        const namespace = `/hacsfiles/${repository.full_name.split("/")[1]}`;
        const url = `${namespace}/${repository.file_name}`;
        const exsisting = resources.find((resource) => resource.url.includes(namespace));
        if (exsisting && exsisting.url !== url) {
            console.debug(`Updating exsusting resource for ${namespace}`);
            updateResource(hass, {
                url,
                resource_id: exsisting.id,
                res_type: exsisting.type,
            });
        }
        else if (!resources.map((resource) => resource.url).includes(url)) {
            console.debug(`Adding ${url} to Lovelace resources`);
            createResource(hass, {
                url,
                res_type: "module",
            });
        }
    }

    let HacsUpdateDialog = class HacsUpdateDialog extends HacsDialogBase {
        constructor() {
            super(...arguments);
            this._updating = false;
            this._releaseNotes = [];
            this._getRepository = memoizeOne((repositories, repository) => repositories === null || repositories === void 0 ? void 0 : repositories.find((repo) => repo.id === repository));
        }
        async firstUpdated() {
            const repository = this._getRepository(this.repositories, this.repository);
            if (repository.version_or_commit !== "commit") {
                this._releaseNotes = await repositoryReleasenotes(this.hass, repository.id);
                this._releaseNotes = this._releaseNotes.filter((release) => release.tag > repository.installed_version);
            }
            this.hass.connection.subscribeEvents((msg) => (this._error = msg.data), "hacs/error");
        }
        render() {
            if (!this.active)
                return html ``;
            const repository = this._getRepository(this.repositories, this.repository);
            return html `
      <hacs-dialog
        .active=${this.active}
        .narrow=${this.narrow}
        .hass=${this.hass}
        ?dynamicHeight=${this._releaseNotes.length === 0}
        ?hasContent=${this._releaseNotes.length > 0}
      >
        <div slot="header">${localize("dialog_update.title")}</div>
        <div class="content">
          ${repository.name}
          <p>
            <b>${localize("dialog_update.installed_version")}:</b>
            ${repository.installed_version}
          </p>
          <p>
            <b>${localize("dialog_update.available_version")}:</b>
            ${repository.available_version}
          </p>
          ${this._releaseNotes.length > 0
            ? this._releaseNotes.map((release) => html `<details>
                  <summary
                    >${localize("dialog_update.releasenotes").replace("{release}", release.tag)}</summary
                  >
                  ${markdown.html(release.body)}
                </details>`)
            : ""}
          ${!repository.can_install
            ? html `<p class="error">
                ${localize("confirm.home_assistant_version_not_correct")
                .replace("{haversion}", this.hass.config.version)
                .replace("{minversion}", repository.homeassistant)}
              </p>`
            : ""}
          ${repository.category === "integration"
            ? html `<p>${localize("dialog_install.restart")}</p>`
            : ""}
          ${this._error ? html `<div class="error">${this._error.message}</div>` : ""}
        </div>
        <div slot="actions">
          <mwc-button ?disabled=${!repository.can_install} @click=${this._updateRepository}
            >${this._updating
            ? html `<paper-spinner active></paper-spinner>`
            : localize("common.update")}</mwc-button
          >
          <hacs-link .url=${this._getChanglogURL()}
            ><mwc-button>${localize("dialog_update.changelog")}</mwc-button></hacs-link
          >
          <hacs-link .url="https://github.com/${repository.full_name}"
            ><mwc-button>${localize("common.repository")}</mwc-button></hacs-link
          >
        </div>
      </hacs-dialog>
    `;
        }
        async _updateRepository() {
            this._updating = true;
            const repository = this._getRepository(this.repositories, this.repository);
            if (repository.version_or_commit !== "commit") {
                await repositoryInstallVersion(this.hass, repository.id, repository.available_version);
            }
            else {
                await repositoryInstall(this.hass, repository.id);
            }
            if (repository.category === "plugin") {
                if (this.status.lovelace_mode !== "yaml") {
                    await updateLovelaceResources(this.hass, repository);
                }
                window.location.reload(true);
            }
            this._updating = false;
            this.dispatchEvent(new Event("hacs-dialog-closed", { bubbles: true, composed: true }));
        }
        _getChanglogURL() {
            const repository = this._getRepository(this.repositories, this.repository);
            if (repository.version_or_commit === "commit") {
                return `https://github.com/${repository.full_name}/compare/${repository.installed_version}...${repository.available_version}`;
            }
            return `https://github.com/${repository.full_name}/releases`;
        }
        static get styles() {
            return [
                css `
        .content {
          padding: 32px 8px;
        }
        .error {
          color: var(--hacs-error-color, var(--google-red-500));
        }
        details {
          padding: 12px 0;
        }
        summary {
          padding: 4px;
          cursor: pointer;
        }
      `,
            ];
        }
    };
    __decorate([
        property()
    ], HacsUpdateDialog.prototype, "repository", void 0);
    __decorate([
        property()
    ], HacsUpdateDialog.prototype, "_updating", void 0);
    __decorate([
        property()
    ], HacsUpdateDialog.prototype, "_error", void 0);
    __decorate([
        property()
    ], HacsUpdateDialog.prototype, "_releaseNotes", void 0);
    HacsUpdateDialog = __decorate([
        customElement("hacs-update-dialog")
    ], HacsUpdateDialog);

    let HacsChip = class HacsChip extends LitElement {
        render() {
            return html `
      <div class="chip">
        <div class="icon"><ha-icon .icon=${this.icon}></ha-icon></div>
        <div class="value">${String(this.value) || ""}</div>
      </div>
    `;
        }
        static get styles() {
            return [
                css `
        .chip {
          background-color: var(--hcv-color-chip);
          height: 24px;
          color: var(--hcv-text-color-chip);
          max-width: fit-content;
          display: flex;
          border-radius: 50px;
          padding: 0 4px;
          box-shadow: 2px 2px 8px 1px rgba(0, 0, 0, 0.3);
        }
        .icon {
          margin: auto;
          color: var(--hcv-color-chip);
          height: 20px;
          width: 20px;
          line-height: 20px;
          text-align: center;

          margin-left: -2px;
          background-color: var(--hcv-text-color-chip);
          border-radius: 50px;
        }
        .value {
          width: max-content;
          margin: auto;
          margin-left: 5px;
          margin-right: 5px;
        }
        ha-icon {
          --mdc-icon-size: 16px;
        }
      `,
            ];
        }
    };
    __decorate([
        property()
    ], HacsChip.prototype, "icon", void 0);
    __decorate([
        property()
    ], HacsChip.prototype, "value", void 0);
    HacsChip = __decorate([
        customElement("hacs-chip")
    ], HacsChip);

    let HacsRepositoryDialog = class HacsRepositoryDialog extends HacsDialogBase {
        constructor() {
            super(...arguments);
            this._getRepository = memoizeOne((repositories, repository) => repositories === null || repositories === void 0 ? void 0 : repositories.find((repo) => repo.id === repository));
            this._getAuthors = memoizeOne((repository) => {
                const authors = [];
                if (!repository.authors)
                    return authors;
                repository.authors.forEach((author) => authors.push(author.replace("@", "")));
                if (authors.length === 0) {
                    const author = repository.full_name.split("/")[0];
                    if (["custom-cards", "custom-components", "home-assistant-community-themes"].includes(author)) {
                        return authors;
                    }
                    authors.push(author);
                }
                return authors;
            });
        }
        shouldUpdate(changedProperties) {
            changedProperties.forEach((_oldValue, propName) => {
                if (propName === "hass") {
                    this.sidebarDocked = window.localStorage.getItem("dockedSidebar") === '"docked"';
                }
                if (propName === "repositories") {
                    this._repository = this._getRepository(this.repositories, this.repository);
                }
            });
            return (changedProperties.has("sidebarDocked") ||
                changedProperties.has("narrow") ||
                changedProperties.has("active") ||
                changedProperties.has("_repository"));
        }
        async firstUpdated() {
            this._repository = this._getRepository(this.repositories, this.repository);
            if (!this._repository.updated_info) {
                await repositoryUpdate(this.hass, this._repository.id);
                this.repositories = await getRepositories(this.hass);
            }
        }
        render() {
            var _a, _b, _c, _d, _e, _f;
            if (!this.active)
                return html ``;
            const authors = this._getAuthors(this._repository);
            return html `
      <hacs-dialog
        .active=${this.active}
        .narrow=${this.narrow}
        .hass=${this.hass}
        .secondary=${this.secondary}
        ?hasContent=${((_b = (_a = this._repository) === null || _a === void 0 ? void 0 : _a.additional_info) === null || _b === void 0 ? void 0 : _b.length) > 100}
        ?noActions=${(_c = this._repository) === null || _c === void 0 ? void 0 : _c.installed}
        ?dynamicHeight=${((_d = this._repository) === null || _d === void 0 ? void 0 : _d.installed) &&
            ((_f = (_e = this._repository) === null || _e === void 0 ? void 0 : _e.additional_info) === null || _f === void 0 ? void 0 : _f.length) < 100}
      >
        <div slot="header">${this._repository.name || ""}</div>
        <div class="chips">
          ${this._repository.installed
            ? html `<hacs-chip
                title="${localize("dialog_info.version_installed")}"
                icon="mdi:cube"
                .value=${this._repository.installed_version}
              ></hacs-chip>`
            : ""}
          ${authors
            ? authors.map((author) => html `<hacs-link .url="https://github.com/${author}"
                  ><hacs-chip
                    title="${localize("dialog_info.author")}"
                    icon="mdi:account"
                    .value="@${author}"
                  ></hacs-chip
                ></hacs-link>`)
            : ""}
          ${this._repository.downloads
            ? html ` <hacs-chip
                title="${localize("dialog_info.downloads")}"
                icon="mdi:arrow-down-bold"
                .value=${this._repository.downloads}
              ></hacs-chip>`
            : ""}
          <hacs-chip
            title="${localize("dialog_info.stars")}"
            icon="mdi:star"
            .value=${this._repository.stars}
          ></hacs-chip>
          <hacs-link .url="https://github.com/${this._repository.full_name}/issues">
            <hacs-chip
              title="${localize("dialog_info.open_issues")}"
              icon="mdi:exclamation-thick"
              .value=${this._repository.issues}
            ></hacs-chip
          ></hacs-link>
        </div>
        ${this._repository.updated_info
            ? markdown.html(this._repository.additional_info || localize("dialog_info.no_info"), this._repository)
            : localize("dialog_info.loading")}
        ${!this._repository.installed && this._repository.updated_info
            ? html ` <div slot="actions">
              <mwc-button @click=${this._installRepository} raised
                >${localize("dialog_info.install")}</mwc-button
              ><hacs-link .url="https://github.com/${this._repository.full_name}"
                ><mwc-button>${localize("dialog_info.open_repo")}</mwc-button></hacs-link
              >
            </div>`
            : ""}
      </hacs-dialog>
    `;
        }
        static get styles() {
            return css `
      img {
        max-width: 100%;
      }
      .chips {
        margin: 0 -4px;
        display: flex;
        padding-bottom: 8px;
      }
      hacs-chip {
        margin: 0 4px;
      }
      div.chips hacs-link {
        margin: -17px 4px;
      }
    `;
        }
        async _installRepository() {
            this.dispatchEvent(new CustomEvent("hacs-dialog-secondary", {
                detail: {
                    type: "install",
                    repository: this._repository.id,
                },
                bubbles: true,
                composed: true,
            }));
        }
    };
    __decorate([
        property()
    ], HacsRepositoryDialog.prototype, "repository", void 0);
    __decorate([
        property()
    ], HacsRepositoryDialog.prototype, "_repository", void 0);
    HacsRepositoryDialog = __decorate([
        customElement("hacs-repository-info-dialog")
    ], HacsRepositoryDialog);

    let HacsCustomRepositoriesDialog = class HacsCustomRepositoriesDialog extends HacsDialogBase {
        shouldUpdate(changedProperties) {
            changedProperties.forEach((_oldValue, propName) => {
                if (propName === "hass") {
                    this.sidebarDocked =
                        window.localStorage.getItem("dockedSidebar") === '"docked"';
                }
            });
            return (changedProperties.has("sidebarDocked") ||
                changedProperties.has("narrow") ||
                changedProperties.has("active") ||
                changedProperties.has("_error") ||
                changedProperties.has("repositories"));
        }
        render() {
            var _a;
            if (!this.active)
                return html ``;
            const repositories = (_a = this.repositories) === null || _a === void 0 ? void 0 : _a.filter((repo) => repo.custom);
            return html `
      <hacs-dialog
        .active=${this.active}
        .narrow=${this.narrow}
        .hass=${this.hass}
        noActions
        dynamicHeight
      >
        <div slot="header">${localize("dialog_custom_repositories.title")}</div>
        <div class="content">
          ${this._error
            ? html `<div class="error">${this._error.message}</div>`
            : ""}
          <div class="list">
            ${repositories === null || repositories === void 0 ? void 0 : repositories.map((repo) => html `<paper-icon-item>
                ${repo.category === "integration"
            ? html `
                      <img
                        src="https://brands.home-assistant.io/_/${repo.domain}/icon.png"
                        referrerpolicy="no-referrer"
                        @error=${this._onImageError}
                        @load=${this._onImageLoad}
                      />
                    `
            : html `<ha-icon
                      icon="mdi:github-circle"
                      slot="item-icon"
                    ></ha-icon>`}
                <paper-item-body
                  @click=${() => this._showReopsitoryInfo(String(repo.id))}
                  three-line
                  >${repo.name}
                  <div secondary>${repo.description}</div>
                  <div secondary>
                    Category: ${repo.category}
                  </div></paper-item-body
                ><ha-icon-button
                  class="delete"
                  icon="mdi:delete"
                  @click=${() => this._removeRepository(repo.id)}
                ></ha-icon-button>
              </paper-icon-item>`)}
          </div>
          <div class="add">
            <input
              id="add-input"
              class="add-input"
              placeholder="${localize("dialog_custom_repositories.url_placeholder")}"
              .value=${this._inputRepository || ""}
              @input=${this._inputValueChanged}
            />
          </div>
          <div class="add-actions">
            <paper-dropdown-menu
              class="category"
              label="${localize("dialog_custom_repositories.category")}"
            >
              <paper-listbox
                id="category"
                slot="dropdown-content"
                selected="-1"
              >
                ${this.configuration.categories.map((category) => html `
                    <paper-item class="categoryitem" .category=${category}>
                      ${localize(`common.${category}`)}
                    </paper-item>
                  `)}
              </paper-listbox>
            </paper-dropdown-menu>
            <mwc-button raised @click=${this._addRepository}
              >${localize("common.add")}</mwc-button
            >
          </div>
        </div>
      </hacs-dialog>
    `;
        }
        firstUpdated() {
            this.hass.connection.subscribeEvents((msg) => (this._error = msg.data), "hacs/error");
        }
        _inputValueChanged() {
            var _a;
            this._inputRepository = (_a = this._addInput) === null || _a === void 0 ? void 0 : _a.value;
        }
        async _addRepository() {
            var _a, _b;
            this._error = undefined;
            const repository = this._inputRepository;
            const category = (_b = (_a = this._addCategory) === null || _a === void 0 ? void 0 : _a.selectedItem) === null || _b === void 0 ? void 0 : _b.category;
            if (!category) {
                this._error = {
                    message: localize("dialog_custom_repositories.no_category"),
                };
                return;
            }
            if (!repository) {
                this._error = {
                    message: localize("dialog_custom_repositories.no_repository"),
                };
                return;
            }
            await repositoryAdd(this.hass, repository, category);
            this.repositories = await getRepositories(this.hass);
        }
        async _removeRepository(repository) {
            this._error = undefined;
            await repositoryDelete(this.hass, repository);
            this.repositories = await getRepositories(this.hass);
        }
        _onImageLoad(ev) {
            ev.target.style.visibility = "initial";
        }
        _onImageError(ev) {
            ev.target.outerHTML = `<ha-icon
      icon="mdi:github-circle"
      slot="item-icon"
    ></ha-icon>`;
        }
        async _showReopsitoryInfo(repository) {
            this.dispatchEvent(new CustomEvent("hacs-dialog-secondary", {
                detail: {
                    type: "repository-info",
                    repository,
                },
                bubbles: true,
                composed: true,
            }));
        }
        static get styles() {
            return css `
      .content {
        width: 1024px;
        display: contents;
      }
      .list {
        margin-top: 16px;
      }
      ha-icon-button,
      ha-icon {
        color: var(--secondary-text-color);
      }
      ha-icon {
        --mdc-icon-size: 36px;
      }
      img {
        align-items: center;
        display: block;
        justify-content: center;
        margin-bottom: 16px;
        max-height: 36px;
        max-width: 36px;
        position: absolute;
      }
      .delete,
      paper-item-body {
        cursor: pointer;
      }
      .error {
        line-height: 0px;
        margin: 12px;
        color: var(--hacs-error-color, var(--google-red-500));
      }

      paper-item-body {
        width: 100%;
        min-height: var(--paper-item-body-two-line-min-height, 72px);
        display: var(--layout-vertical_-_display);
        flex-direction: var(--layout-vertical_-_flex-direction);
        justify-content: var(--layout-center-justified_-_justify-content);
      }
      paper-item-body div {
        font-size: 14px;
        color: var(--secondary-text-color);
      }
      .add {
        border-top: 1px solid var(--divider-color);
        margin-top: 32px;
      }
      .add-actions {
        justify-content: space-between;
      }
      .add,
      .add-actions {
        display: flex;
        align-items: center;
        font-size: 20px;
        height: 65px;
        background-color: var(--sidebar-background-color);
        border-bottom: 1px solid var(--divider-color);
        padding: 0 16px;
        box-sizing: border-box;
      }
      .add-input {
        width: calc(100% - 80px);
        height: 40px;
        border: 0;
        padding: 0 16px;
        font-size: initial;
        color: var(--sidebar-text-color);
        font-family: var(--paper-font-body1_-_font-family);
      }
      input:focus {
        outline-offset: 0;
        outline: 0;
      }
      input {
        background-color: var(--sidebar-background-color);
      }
      paper-dropdown-menu {
        width: 75%;
      }
    `;
        }
    };
    __decorate([
        property()
    ], HacsCustomRepositoriesDialog.prototype, "_inputRepository", void 0);
    __decorate([
        property()
    ], HacsCustomRepositoriesDialog.prototype, "_error", void 0);
    __decorate([
        query("#add-input")
    ], HacsCustomRepositoriesDialog.prototype, "_addInput", void 0);
    __decorate([
        query("#category")
    ], HacsCustomRepositoriesDialog.prototype, "_addCategory", void 0);
    HacsCustomRepositoriesDialog = __decorate([
        customElement("hacs-custom-repositories-dialog")
    ], HacsCustomRepositoriesDialog);

    let HacsAddRepositoryDialog = class HacsAddRepositoryDialog extends HacsDialogBase {
        constructor() {
            super(...arguments);
            this.filters = [];
            this._load = 30;
            this._top = 0;
            this._searchInput = "";
            this._sortBy = "stars";
            this._repositoriesInActiveCategory = (repositories, categories) => repositories === null || repositories === void 0 ? void 0 : repositories.filter((repo) => {
                var _a, _b;
                return !repo.installed && ((_a = sections$f === null || sections$f === void 0 ? void 0 : sections$f.panels.find((panel) => panel.id === this.section).categories) === null || _a === void 0 ? void 0 : _a.includes(repo.category)) &&
                    !repo.installed && (categories === null || categories === void 0 ? void 0 : categories.includes(repo.category)) && ((_b = this.filters.find((filter) => filter.id === repo.category)) === null || _b === void 0 ? void 0 : _b.checked);
            });
            this._filterRepositories = memoizeOne(filterRepositoriesByInput);
        }
        shouldUpdate(changedProperties) {
            changedProperties.forEach((_oldValue, propName) => {
                if (propName === "hass") {
                    this.sidebarDocked = window.localStorage.getItem("dockedSidebar") === '"docked"';
                }
            });
            return (changedProperties.has("sidebarDocked") ||
                changedProperties.has("narrow") ||
                changedProperties.has("filters") ||
                changedProperties.has("active") ||
                changedProperties.has("_searchInput") ||
                changedProperties.has("_load") ||
                changedProperties.has("_sortBy"));
        }
        async firstUpdated() {
            this.addEventListener("filter-change", (e) => this._updateFilters(e));
        }
        _updateFilters(e) {
            const current = this.filters.find((filter) => filter.id === e.detail.id);
            this.filters.find((filter) => filter.id === current.id).checked = !current.checked;
            this.requestUpdate("filters");
        }
        render() {
            var _a, _b;
            this._searchInput = window.localStorage.getItem("hacs-search") || "";
            if (!this.active)
                return html ``;
            if (this.filters.length === 0) {
                const categories = (_a = activePanel(this.route)) === null || _a === void 0 ? void 0 : _a.categories;
                categories === null || categories === void 0 ? void 0 : categories.filter((c) => this.configuration.categories.includes(c)).forEach((category) => {
                    this.filters.push({
                        id: category,
                        value: category,
                        checked: true,
                    });
                });
            }
            const repositories = this._filterRepositories(this._repositoriesInActiveCategory(this.repositories, (_b = this.configuration) === null || _b === void 0 ? void 0 : _b.categories), this._searchInput);
            return html `
      <hacs-dialog
        .active=${this.active}
        .narrow=${this.narrow}
        .hass=${this.hass}
        @scroll=${this._loadMore}
        noActions
        ?hasFilter=${this.filters.length > 1}
        hasSearch
      >
        <div slot="header">
          ${localize("dialog_add_repo.title")}
        </div>
        <div slot="search" class="filter">
          <hacs-search .input=${this._searchInput} @input=${this._inputValueChanged}></hacs-search>
          <paper-dropdown-menu label="${localize("dialog_add_repo.sort_by")}">
            <paper-listbox slot="dropdown-content" selected="0">
              <paper-item @tap=${() => (this._sortBy = "stars")}
                >${localize("store.stars")}</paper-item
              >
              <paper-item @tap=${() => (this._sortBy = "name")}
                >${localize("store.name")}</paper-item
              >
              <paper-item @tap=${() => (this._sortBy = "last_updated")}
                >${localize("store.last_updated")}</paper-item
              >
            </paper-listbox>
          </paper-dropdown-menu>
        </div>
        ${this.filters.length > 1
            ? html `<div slot="filter" class="filter">
              <hacs-filter .filters="${this.filters}"></hacs-filter>
            </div>`
            : ""}
        <div class=${classMap({ content: true, narrow: this.narrow })}>
          <div class=${classMap({ list: true, narrow: this.narrow })}>
            ${repositories
            .sort((a, b) => {
            if (this._sortBy === "name") {
                return a.name.toLocaleLowerCase() < b.name.toLocaleLowerCase() ? -1 : 1;
            }
            return a[this._sortBy] > b[this._sortBy] ? -1 : 1;
        })
            .slice(0, this._load)
            .map((repo) => html `<paper-icon-item
                  class=${classMap({ narrow: this.narrow })}
                  @click=${() => this._openInformation(repo)}
                >
                  ${repo.category === "integration"
            ? html `
                        <img
                          src="https://brands.home-assistant.io/_/${repo.domain}/icon.png"
                          referrerpolicy="no-referrer"
                          @error=${this._onImageError}
                          @load=${this._onImageLoad}
                        />
                      `
            : html `<ha-icon icon="mdi:github-circle" slot="item-icon"></ha-icon>`}
                  <paper-item-body two-line
                    >${repo.name}
                    <div class="category-chip">
                      <hacs-chip
                        icon="hacs:hacs"
                        .value=${localize(`common.${repo.category}`)}
                      ></hacs-chip>
                    </div>
                    <div secondary>${repo.description}</div>
                  </paper-item-body>
                </paper-icon-item>`)}
            ${repositories.length === 0 ? html `<p>${localize("dialog_add_repo.no_match")}</p>` : ""}
          </div>
        </div>
      </hacs-dialog>
    `;
        }
        _loadMore(ev) {
            const top = ev.detail.target.scrollTop;
            if (top >= this._top) {
                this._load += 1;
            }
            else {
                this._load -= 1;
            }
            this._top = top;
        }
        _inputValueChanged(ev) {
            this._searchInput = ev.target.input;
            window.localStorage.setItem("hacs-search", this._searchInput);
        }
        _openInformation(repo) {
            this.dispatchEvent(new CustomEvent("hacs-dialog-secondary", {
                detail: {
                    type: "repository-info",
                    repository: repo.id,
                },
                bubbles: true,
                composed: true,
            }));
        }
        _onImageLoad(ev) {
            ev.target.style.visibility = "initial";
        }
        _onImageError(ev) {
            if (ev.target) {
                ev.target.outerHTML = `<ha-icon
    icon="mdi:github-circle"
    slot="item-icon"
  ></ha-icon>`;
            }
        }
        static get styles() {
            return css `
      .content {
        width: 100%;
        margin-bottom: -65px;
      }
      .filter {
        margin-bottom: -65px;
        margin-top: 65px;
        display: flex;
      }
      .narrow {
        min-width: unset !important;
        width: 100% !important;
      }
      .list {
        margin-top: 16px;
        width: 1024px;
        max-width: 100%;
      }
      .category-chip {
        position: absolute;
        top: 8px;
        right: 8px;
      }
      ha-icon {
        --mdc-icon-size: 36px;
      }
      img {
        align-items: center;
        display: block;
        justify-content: center;
        margin-bottom: 16px;
        max-height: 36px;
        max-width: 36px;
        position: absolute;
      }

      paper-icon-item:focus {
        background-color: var(--divider-color);
      }

      paper-icon-item {
        cursor: pointer;
        padding: 2px 0;
      }

      paper-dropdown-menu {
        max-width: 30%;
        margin: 11px 4px -5px;
      }

      paper-item-body {
        width: 100%;
        min-height: var(--paper-item-body-two-line-min-height, 72px);
        display: var(--layout-vertical_-_display);
        flex-direction: var(--layout-vertical_-_flex-direction);
        justify-content: var(--layout-center-justified_-_justify-content);
      }
      paper-icon-item.narrow {
        border-bottom: 1px solid var(--divider-color);
        padding: 8px 0;
      }
      paper-item-body div {
        font-size: 14px;
        color: var(--secondary-text-color);
      }
      .add {
        border-top: 1px solid var(--divider-color);
        margin-top: 32px;
      }
      .add-actions {
        justify-content: space-between;
      }
      .add,
      .add-actions {
        display: flex;
        align-items: center;
        font-size: 20px;
        height: 65px;
        background-color: var(--sidebar-background-color);
        border-bottom: 1px solid var(--divider-color);
        padding: 0 16px;
        box-sizing: border-box;
      }
      .add-input {
        width: calc(100% - 80px);
        height: 40px;
        border: 0;
        padding: 0 16px;
        font-size: initial;
        color: var(--sidebar-text-color);
        font-family: var(--paper-font-body1_-_font-family);
      }
      input:focus {
        outline-offset: 0;
        outline: 0;
      }
      input {
        background-color: var(--sidebar-background-color);
      }
      paper-dropdown-menu {
        width: 75%;
      }
      hacs-search,
      hacs-filter {
        width: 100%;
      }
    `;
        }
    };
    __decorate([
        property({ attribute: false })
    ], HacsAddRepositoryDialog.prototype, "filters", void 0);
    __decorate([
        property()
    ], HacsAddRepositoryDialog.prototype, "_load", void 0);
    __decorate([
        property()
    ], HacsAddRepositoryDialog.prototype, "_top", void 0);
    __decorate([
        property()
    ], HacsAddRepositoryDialog.prototype, "_searchInput", void 0);
    __decorate([
        property()
    ], HacsAddRepositoryDialog.prototype, "_sortBy", void 0);
    __decorate([
        property()
    ], HacsAddRepositoryDialog.prototype, "section", void 0);
    HacsAddRepositoryDialog = __decorate([
        customElement("hacs-add-repository-dialog")
    ], HacsAddRepositoryDialog);

    let HacsInstallDialog = class HacsInstallDialog extends HacsDialogBase {
        constructor() {
            super(...arguments);
            this._toggle = true;
            this._installing = false;
            this._getRepository = memoizeOne((repositories, repository) => repositories === null || repositories === void 0 ? void 0 : repositories.find((repo) => repo.id === repository));
            this._getInstallPath = memoizeOne((repository) => {
                let path = repository.local_path;
                if (repository.category === "theme") {
                    path = `${path}/${repository.file_name}`;
                }
                return path;
            });
        }
        shouldUpdate(changedProperties) {
            changedProperties.forEach((_oldValue, propName) => {
                if (propName === "hass") {
                    this.sidebarDocked = window.localStorage.getItem("dockedSidebar") === '"docked"';
                }
                if (propName === "repositories") {
                    this._repository = this._getRepository(this.repositories, this.repository);
                }
            });
            return (changedProperties.has("sidebarDocked") ||
                changedProperties.has("narrow") ||
                changedProperties.has("active") ||
                changedProperties.has("_toggle") ||
                changedProperties.has("_error") ||
                changedProperties.has("_repository") ||
                changedProperties.has("_installing"));
        }
        async firstUpdated() {
            this._repository = this._getRepository(this.repositories, this.repository);
            if (!this._repository.updated_info) {
                await repositoryUpdate(this.hass, this._repository.id);
                this.repositories = await getRepositories(this.hass);
            }
            this._toggle = false;
            this.hass.connection.subscribeEvents((msg) => (this._error = msg.data), "hacs/error");
        }
        render() {
            if (!this.active)
                return html ``;
            const installPath = this._getInstallPath(this._repository);
            return html `
      <hacs-dialog
        .active=${this.active}
        .narrow=${this.narrow}
        .hass=${this.hass}
        .secondary=${this.secondary}
        dynamicHeight
      >
        <div slot="header">${this._repository.name}</div>
        <div class="content">
          ${this._repository.version_or_commit === "version"
            ? html `<div class="beta-container">
                  <ha-switch
                    ?disabled=${this._toggle}
                    .checked=${this._repository.beta}
                    @change=${this._toggleBeta}
                    >${localize("dialog_install.show_beta")}</ha-switch
                  >
                </div>
                <div class="version-select-container">
                  <paper-dropdown-menu
                    ?disabled=${this._toggle}
                    class="version-select-dropdown"
                    label="${localize("dialog_install.select_version")}"
                  >
                    <paper-listbox
                      id="version"
                      class="version-select-list"
                      slot="dropdown-content"
                      selected="0"
                    >
                      ${this._repository.releases.map((release) => {
                return html `<paper-item .version=${release} class="version-select-item"
                          >${release}</paper-item
                        >`;
            })}
                      ${this._repository.full_name === "hacs/integration" ||
                this._repository.hide_default_branch
                ? ""
                : html `
                            <paper-item
                              .version=${this._repository.default_branch}
                              class="version-select-item"
                              >${this._repository.default_branch}</paper-item
                            >
                          `}
                    </paper-listbox>
                  </paper-dropdown-menu>
                </div>`
            : ""}
          ${!this._repository.can_install
            ? html `<p class="error">
                ${localize("confirm.home_assistant_version_not_correct")
                .replace("{haversion}", this.hass.config.version)
                .replace("{minversion}", this._repository.homeassistant)}
              </p>`
            : ""}
          <div class="note">
            ${localize(`repository.note_installed`)}
            <code>'${installPath}'</code>
            ${this._repository.category === "plugin" && this.status.lovelace_mode === "yaml"
            ? html ` <table class="lovelace">
                  <tr>
                    <td>${localize("dialog_install.url")}:</td>
                    <td>
                      <code>${this._lovelaceUrl()}</code>
                    </td>
                  </tr>
                  <tr>
                    <td>${localize("dialog_install.type")}:</td>
                    <td>
                      module
                    </td>
                  </tr>
                </table>`
            : ""}
            ${this._repository.category === "integration"
            ? html `<p>${localize("dialog_install.restart")}</p>`
            : ""}
          </div>
          ${this._error ? html `<div class="error">${this._error.message}</div>` : ""}
        </div>
        <div slot="actions">
          <mwc-button
            ?disabled=${!this._repository.can_install || this._toggle}
            @click=${this._installRepository}
            >${this._installing
            ? html `<paper-spinner active></paper-spinner>`
            : localize("common.install")}</mwc-button
          >

          <hacs-link .url="https://github.com/${this._repository.full_name}"
            ><mwc-button>${localize("common.repository")}</mwc-button></hacs-link
          >
        </div>
      </hacs-dialog>
    `;
        }
        _lovelaceUrl() {
            var _a, _b;
            return `/hacsfiles/${(_a = this._repository) === null || _a === void 0 ? void 0 : _a.full_name.split("/")[1]}/${(_b = this._repository) === null || _b === void 0 ? void 0 : _b.file_name}`;
        }
        async _toggleBeta() {
            this._toggle = true;
            await repositoryToggleBeta(this.hass, this.repository);
            this.repositories = await getRepositories(this.hass);
            this._toggle = false;
        }
        async _installRepository() {
            var _a, _b;
            this._installing = true;
            if (this._repository.version_or_commit !== "commit") {
                const selectedVersion = ((_b = (_a = this._version) === null || _a === void 0 ? void 0 : _a.selectedItem) === null || _b === void 0 ? void 0 : _b.version) ||
                    this._repository.available_version ||
                    this._repository.default_branch;
                await repositoryInstallVersion(this.hass, this._repository.id, selectedVersion);
            }
            else {
                await repositoryInstall(this.hass, this._repository.id);
            }
            if (this._repository.category === "plugin" && this.status.lovelace_mode !== "yaml") {
                await updateLovelaceResources(this.hass, this._repository);
            }
            this._installing = false;
            if (this._repository.category === "plugin") {
                window.location.reload(true);
            }
            this.dispatchEvent(new Event("hacs-secondary-dialog-closed", {
                bubbles: true,
                composed: true,
            }));
            this.dispatchEvent(new Event("hacs-dialog-closed", {
                bubbles: true,
                composed: true,
            }));
        }
        static get styles() {
            return [
                css `
        .version-select-dropdown {
          width: 100%;
        }
        .content {
          padding: 32px 8px;
        }
        .note {
          margin-bottom: -32px;
          margin-top: 12px;
        }
        .lovelace {
          margin-top: 8px;
        }
        .error {
          color: var(--hacs-error-color, var(--google-red-500));
        }
        paper-menu-button {
          color: var(--secondary-text-color);
          padding: 0;
        }
        paper-item {
          cursor: pointer;
        }
        paper-item-body {
          opacity: var(--dark-primary-opacity);
        }
      `,
            ];
        }
    };
    __decorate([
        property()
    ], HacsInstallDialog.prototype, "repository", void 0);
    __decorate([
        property()
    ], HacsInstallDialog.prototype, "_repository", void 0);
    __decorate([
        property()
    ], HacsInstallDialog.prototype, "_toggle", void 0);
    __decorate([
        property()
    ], HacsInstallDialog.prototype, "_installing", void 0);
    __decorate([
        property()
    ], HacsInstallDialog.prototype, "_error", void 0);
    __decorate([
        query("#version")
    ], HacsInstallDialog.prototype, "_version", void 0);
    HacsInstallDialog = __decorate([
        customElement("hacs-install-dialog")
    ], HacsInstallDialog);

    let HacsEventDialog = class HacsEventDialog extends HacsDialogBase {
        render() {
            if (!this.active)
                return html ``;
            const el = document.createElement(`hacs-${this.params.type || "generic"}-dialog`);
            el.active = true;
            el.hass = this.hass;
            el.narrow = this.narrow;
            el.configuration = this.configuration;
            el.lovelace = this.lovelace;
            el.secondary = this.secondary;
            el.repositories = this.repositories;
            el.route = this.route;
            el.status = this.status;
            if (this.params) {
                for (let [key, value] of Object.entries(this.params)) {
                    el[key] = value;
                }
            }
            return html `${el}`;
        }
    };
    __decorate([
        property({ attribute: false })
    ], HacsEventDialog.prototype, "params", void 0);
    HacsEventDialog = __decorate([
        customElement("hacs-event-dialog")
    ], HacsEventDialog);

    let HacsResolver = class HacsResolver extends LitElement {
        constructor() {
            super(...arguments);
            this.logger = new HacsLogger();
            this._sortRepositoriesByName = memoizeOne((repositories) => repositories.sort((a, b) => a.name.toLowerCase() > b.name.toLowerCase() ? 1 : -1));
        }
        connectedCallback() {
            super.connectedCallback();
            this.addEventListener("hacs-location-changed", (e) => this._setRoute(e));
            this.addEventListener("hacs-dialog", (e) => this._showDialog(e));
            this.addEventListener("hacs-dialog-secondary", (e) => this._showDialogSecondary(e));
        }
        async firstUpdated() {
            window.onpopstate = function () {
                if (window.location.pathname.includes("hacs")) {
                    window.location.reload();
                }
            };
            /* Backend event subscription */
            this.hass.connection.subscribeEvents(async () => await this._updateProperties(), "hacs/config");
            this.hass.connection.subscribeEvents(async () => await this._updateProperties(), "hacs/status");
            this.hass.connection.subscribeEvents(async () => await this._updateProperties(), "hacs/repository");
            this.hass.connection.subscribeEvents(async () => await this._updateProperties(), "lovelace_updated");
            await this._updateProperties();
        }
        async _updateProperties() {
            const [repositories, configuration, status, critical, lovelace, removed] = await Promise.all([
                getRepositories(this.hass),
                getConfiguration(this.hass),
                getStatus(this.hass),
                getCritical(this.hass),
                getLovelaceConfiguration(this.hass),
                getRemovedRepositories(this.hass),
            ]);
            this.configuration = configuration;
            this.status = status;
            this.removed = removed;
            this.critical = critical;
            this.lovelace = lovelace;
            this.configuration = configuration;
            this.repositories = this._sortRepositoriesByName(repositories);
        }
        render() {
            if (this.route.path === "" || this.route.path === "/") {
                this.route.path = "/dashboard";
            }
            return html `${["/integrations", "/frontend", "/automation"].includes(this.route.path)
            ? html `<hacs-store-panel
            .hass=${this.hass}
            .route=${this.route}
            .narrow=${this.narrow}
            .configuration=${this.configuration}
            .lovelace=${this.lovelace}
            .repositories=${this.repositories}
            .status=${this.status}
            .removed=${this.removed}
            .section=${this.route.path.split("/")[1]}
          ></hacs-store-panel>`
            : html `<hacs-entry-panel
            .hass=${this.hass}
            .route=${this.route}
            .narrow=${this.narrow}
            .configuration=${this.configuration}
            .lovelace=${this.lovelace}
            .status=${this.status}
            .removed=${this.removed}
            .repositories=${this.repositories}
          ></hacs-entry-panel>`}
      <hacs-event-dialog
        .hass=${this.hass}
        .route=${this.route}
        .narrow=${this.narrow}
        .configuration=${this.configuration}
        .lovelace=${this.lovelace}
        .status=${this.status}
        .removed=${this.removed}
        .repositories=${this.repositories}
        id="hacs-dialog"
      ></hacs-event-dialog>
      <hacs-event-dialog
        .hass=${this.hass}
        .route=${this.route}
        .narrow=${this.narrow}
        .configuration=${this.configuration}
        .lovelace=${this.lovelace}
        .status=${this.status}
        .removed=${this.removed}
        .repositories=${this.repositories}
        id="hacs-dialog-secondary"
      ></hacs-event-dialog>`;
        }
        _showDialog(ev) {
            const dialogParams = ev.detail;
            this._hacsDialog.active = true;
            this._hacsDialog.params = dialogParams;
            this.addEventListener("hacs-dialog-closed", () => (this._hacsDialog.active = false));
        }
        _showDialogSecondary(ev) {
            const dialogParams = ev.detail;
            this._hacsDialogSecondary.active = true;
            this._hacsDialogSecondary.secondary = true;
            this._hacsDialogSecondary.params = dialogParams;
            this.addEventListener("hacs-secondary-dialog-closed", () => (this._hacsDialogSecondary.active = false));
        }
        _setRoute(ev) {
            this.route = ev.detail.route;
            navigate(this, this.route.prefix + this.route.path);
            this.requestUpdate();
        }
    };
    __decorate([
        property({ attribute: false })
    ], HacsResolver.prototype, "configuration", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsResolver.prototype, "critical", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsResolver.prototype, "hass", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsResolver.prototype, "lovelace", void 0);
    __decorate([
        property({ type: Boolean })
    ], HacsResolver.prototype, "narrow", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsResolver.prototype, "repositories", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsResolver.prototype, "route", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsResolver.prototype, "status", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsResolver.prototype, "removed", void 0);
    __decorate([
        query("#hacs-dialog")
    ], HacsResolver.prototype, "_hacsDialog", void 0);
    __decorate([
        query("#hacs-dialog-secondary")
    ], HacsResolver.prototype, "_hacsDialogSecondary", void 0);
    HacsResolver = __decorate([
        customElement("hacs-resolver")
    ], HacsResolver);

    let HacsFrontend = class HacsFrontend extends LitElement {
        async connectedCallback() {
            super.connectedCallback();
            load_lovelace();
            this.configuration = await getConfiguration(this.hass);
        }
        render() {
            if (!this.hass || !this.configuration)
                return html ``;
            return html `
      <hacs-resolver
        .hass=${this.hass}
        .route=${this.route}
        .narrow=${this.narrow}
        .configuration=${this.configuration}
      ></hacs-resolver>
    `;
        }
        static get styles() {
            return hacsStyleVariables;
        }
    };
    __decorate([
        property({ attribute: false })
    ], HacsFrontend.prototype, "hass", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsFrontend.prototype, "narrow", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsFrontend.prototype, "route", void 0);
    __decorate([
        property({ attribute: false })
    ], HacsFrontend.prototype, "configuration", void 0);
    HacsFrontend = __decorate([
        customElement("hacs-frontend")
    ], HacsFrontend);

}());
