"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Alarm = exports.TreatMissingData = exports.ComparisonOperator = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const alarm_base_1 = require("./alarm-base");
const cloudwatch_generated_1 = require("./cloudwatch.generated");
const metric_util_1 = require("./private/metric-util");
const object_1 = require("./private/object");
const rendering_1 = require("./private/rendering");
const statistic_1 = require("./private/statistic");
/**
 * Comparison operator for evaluating alarms.
 *
 * @stability stable
 */
var ComparisonOperator;
(function (ComparisonOperator) {
    ComparisonOperator["GREATER_THAN_OR_EQUAL_TO_THRESHOLD"] = "GreaterThanOrEqualToThreshold";
    ComparisonOperator["GREATER_THAN_THRESHOLD"] = "GreaterThanThreshold";
    ComparisonOperator["LESS_THAN_THRESHOLD"] = "LessThanThreshold";
    ComparisonOperator["LESS_THAN_OR_EQUAL_TO_THRESHOLD"] = "LessThanOrEqualToThreshold";
    ComparisonOperator["LESS_THAN_LOWER_OR_GREATER_THAN_UPPER_THRESHOLD"] = "LessThanLowerOrGreaterThanUpperThreshold";
    ComparisonOperator["GREATER_THAN_UPPER_THRESHOLD"] = "GreaterThanUpperThreshold";
    ComparisonOperator["LESS_THAN_LOWER_THRESHOLD"] = "LessThanLowerThreshold";
})(ComparisonOperator = exports.ComparisonOperator || (exports.ComparisonOperator = {}));
const OPERATOR_SYMBOLS = {
    GreaterThanOrEqualToThreshold: '>=',
    GreaterThanThreshold: '>',
    LessThanThreshold: '<',
    LessThanOrEqualToThreshold: '<=',
};
/**
 * Specify how missing data points are treated during alarm evaluation.
 *
 * @stability stable
 */
var TreatMissingData;
(function (TreatMissingData) {
    TreatMissingData["BREACHING"] = "breaching";
    TreatMissingData["NOT_BREACHING"] = "notBreaching";
    TreatMissingData["IGNORE"] = "ignore";
    TreatMissingData["MISSING"] = "missing";
})(TreatMissingData = exports.TreatMissingData || (exports.TreatMissingData = {}));
/**
 * An alarm on a CloudWatch metric.
 *
 * @stability stable
 */
class Alarm extends alarm_base_1.AlarmBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.alarmName,
        });
        jsiiDeprecationWarnings._aws_cdk_aws_cloudwatch_AlarmProps(props);
        const comparisonOperator = props.comparisonOperator || ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD;
        // Render metric, process potential overrides from the alarm
        // (It would be preferable if the statistic etc. was worked into the metric,
        // but hey we're allowing overrides...)
        const metricProps = this.renderMetric(props.metric);
        if (props.period) {
            metricProps.period = props.period.toSeconds();
        }
        if (props.statistic) {
            // Will overwrite both fields if present
            Object.assign(metricProps, {
                statistic: renderIfSimpleStatistic(props.statistic),
                extendedStatistic: renderIfExtendedStatistic(props.statistic),
            });
        }
        const alarm = new cloudwatch_generated_1.CfnAlarm(this, 'Resource', {
            // Meta
            alarmDescription: props.alarmDescription,
            alarmName: this.physicalName,
            // Evaluation
            comparisonOperator,
            threshold: props.threshold,
            datapointsToAlarm: props.datapointsToAlarm,
            evaluateLowSampleCountPercentile: props.evaluateLowSampleCountPercentile,
            evaluationPeriods: props.evaluationPeriods,
            treatMissingData: props.treatMissingData,
            // Actions
            actionsEnabled: props.actionsEnabled,
            alarmActions: core_1.Lazy.list({ produce: () => this.alarmActionArns }),
            insufficientDataActions: core_1.Lazy.list({ produce: (() => this.insufficientDataActionArns) }),
            okActions: core_1.Lazy.list({ produce: () => this.okActionArns }),
            // Metric
            ...metricProps,
        });
        this.alarmArn = this.getResourceArnAttribute(alarm.attrArn, {
            service: 'cloudwatch',
            resource: 'alarm',
            resourceName: this.physicalName,
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
        });
        this.alarmName = this.getResourceNameAttribute(alarm.ref);
        this.metric = props.metric;
        const datapoints = props.datapointsToAlarm || props.evaluationPeriods;
        this.annotation = {
            // eslint-disable-next-line max-len
            label: `${this.metric} ${OPERATOR_SYMBOLS[comparisonOperator]} ${props.threshold} for ${datapoints} datapoints within ${describePeriod(props.evaluationPeriods * metric_util_1.metricPeriod(props.metric).toSeconds())}`,
            value: props.threshold,
        };
    }
    /**
     * Import an existing CloudWatch alarm provided an ARN.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param alarmArn Alarm ARN (i.e. arn:aws:cloudwatch:<region>:<account-id>:alarm:Foo).
     * @stability stable
     */
    static fromAlarmArn(scope, id, alarmArn) {
        class Import extends alarm_base_1.AlarmBase {
            constructor() {
                super(...arguments);
                this.alarmArn = alarmArn;
                this.alarmName = core_1.Stack.of(scope).splitArn(alarmArn, core_1.ArnFormat.COLON_RESOURCE_NAME).resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Turn this alarm into a horizontal annotation.
     *
     * This is useful if you want to represent an Alarm in a non-AlarmWidget.
     * An `AlarmWidget` can directly show an alarm, but it can only show a
     * single alarm and no other metrics. Instead, you can convert the alarm to
     * a HorizontalAnnotation and add it as an annotation to another graph.
     *
     * This might be useful if:
     *
     * - You want to show multiple alarms inside a single graph, for example if
     *    you have both a "small margin/long period" alarm as well as a
     *    "large margin/short period" alarm.
     *
     * - You want to show an Alarm line in a graph with multiple metrics in it.
     *
     * @stability stable
     */
    toAnnotation() {
        return this.annotation;
    }
    /**
     * Trigger this action if the alarm fires.
     *
     * Typically the ARN of an SNS topic or ARN of an AutoScaling policy.
     *
     * @stability stable
     */
    addAlarmAction(...actions) {
        jsiiDeprecationWarnings._aws_cdk_aws_cloudwatch_IAlarmAction(actions);
        if (this.alarmActionArns === undefined) {
            this.alarmActionArns = [];
        }
        this.alarmActionArns.push(...actions.map(a => this.validateActionArn(a.bind(this, this).alarmActionArn)));
    }
    validateActionArn(actionArn) {
        var _b, _c, _d;
        const ec2ActionsRegexp = /arn:aws:automate:[a-z|\d|-]+:ec2:[a-z]+/;
        if (ec2ActionsRegexp.test(actionArn)) {
            // Check per-instance metric
            const metricConfig = this.metric.toMetricConfig();
            if (((_c = (_b = metricConfig.metricStat) === null || _b === void 0 ? void 0 : _b.dimensions) === null || _c === void 0 ? void 0 : _c.length) != 1 || ((_d = metricConfig.metricStat) === null || _d === void 0 ? void 0 : _d.dimensions[0].name) != 'InstanceId') {
                throw new Error(`EC2 alarm actions requires an EC2 Per-Instance Metric. (${JSON.stringify(metricConfig)} does not have an 'InstanceId' dimension)`);
            }
        }
        return actionArn;
    }
    renderMetric(metric) {
        const self = this;
        return metric_util_1.dispatchMetric(metric, {
            withStat(stat, conf) {
                var _b, _c, _d;
                self.validateMetricStat(stat, metric);
                const canRenderAsLegacyMetric = ((_b = conf.renderingProperties) === null || _b === void 0 ? void 0 : _b.label) == undefined && !self.requiresAccountId(stat);
                // Do this to disturb existing templates as little as possible
                if (canRenderAsLegacyMetric) {
                    return object_1.dropUndefined({
                        dimensions: stat.dimensions,
                        namespace: stat.namespace,
                        metricName: stat.metricName,
                        period: (_c = stat.period) === null || _c === void 0 ? void 0 : _c.toSeconds(),
                        statistic: renderIfSimpleStatistic(stat.statistic),
                        extendedStatistic: renderIfExtendedStatistic(stat.statistic),
                        unit: stat.unitFilter,
                    });
                }
                return {
                    metrics: [
                        {
                            metricStat: {
                                metric: {
                                    metricName: stat.metricName,
                                    namespace: stat.namespace,
                                    dimensions: stat.dimensions,
                                },
                                period: stat.period.toSeconds(),
                                stat: stat.statistic,
                                unit: stat.unitFilter,
                            },
                            id: 'm1',
                            accountId: self.requiresAccountId(stat) ? stat.account : undefined,
                            label: (_d = conf.renderingProperties) === null || _d === void 0 ? void 0 : _d.label,
                            returnData: true,
                        },
                    ],
                };
            },
            withExpression() {
                // Expand the math expression metric into a set
                const mset = new rendering_1.MetricSet();
                mset.addTopLevel(true, metric);
                let eid = 0;
                function uniqueMetricId() {
                    return `expr_${++eid}`;
                }
                return {
                    metrics: mset.entries.map(entry => metric_util_1.dispatchMetric(entry.metric, {
                        withStat(stat, conf) {
                            var _b;
                            self.validateMetricStat(stat, entry.metric);
                            return {
                                metricStat: {
                                    metric: {
                                        metricName: stat.metricName,
                                        namespace: stat.namespace,
                                        dimensions: stat.dimensions,
                                    },
                                    period: stat.period.toSeconds(),
                                    stat: stat.statistic,
                                    unit: stat.unitFilter,
                                },
                                id: entry.id || uniqueMetricId(),
                                accountId: self.requiresAccountId(stat) ? stat.account : undefined,
                                label: (_b = conf.renderingProperties) === null || _b === void 0 ? void 0 : _b.label,
                                returnData: entry.tag ? undefined : false,
                            };
                        },
                        withExpression(expr, conf) {
                            var _b;
                            const hasSubmetrics = mathExprHasSubmetrics(expr);
                            if (hasSubmetrics) {
                                assertSubmetricsCount(expr);
                            }
                            self.validateMetricExpression(expr);
                            return {
                                expression: expr.expression,
                                id: entry.id || uniqueMetricId(),
                                label: (_b = conf.renderingProperties) === null || _b === void 0 ? void 0 : _b.label,
                                period: hasSubmetrics ? undefined : expr.period,
                                returnData: entry.tag ? undefined : false,
                            };
                        },
                    })),
                };
            },
        });
    }
    /**
     * Validate that if a region is in the given stat config, they match the Alarm
     */
    validateMetricStat(stat, metric) {
        const stack = core_1.Stack.of(this);
        if (definitelyDifferent(stat.region, stack.region)) {
            throw new Error(`Cannot create an Alarm in region '${stack.region}' based on metric '${metric}' in '${stat.region}'`);
        }
    }
    /**
     * Validates that the expression config does not specify searchAccount or searchRegion props
     * as search expressions are not supported by Alarms.
     */
    validateMetricExpression(expr) {
        if (expr.searchAccount !== undefined || expr.searchRegion !== undefined) {
            throw new Error('Cannot create an Alarm based on a MathExpression which specifies a searchAccount or searchRegion');
        }
    }
    /**
     * Determine if the accountId property should be included in the metric.
     */
    requiresAccountId(stat) {
        const stackAccount = core_1.Stack.of(this).account;
        // if stat.account is undefined, it's by definition in the same account
        if (stat.account === undefined) {
            return false;
        }
        // Return true if they're different. The ACCOUNT_ID token is interned
        // so will always have the same string value (and even if we guess wrong
        // it will still work).
        return stackAccount !== stat.account;
    }
}
exports.Alarm = Alarm;
_a = JSII_RTTI_SYMBOL_1;
Alarm[_a] = { fqn: "@aws-cdk/aws-cloudwatch.Alarm", version: "1.138.2" };
function definitelyDifferent(x, y) {
    return x && !core_1.Token.isUnresolved(y) && x !== y;
}
/**
 * Return a human readable string for this period
 *
 * We know the seconds are always one of a handful of allowed values.
 */
function describePeriod(seconds) {
    if (seconds === 60) {
        return '1 minute';
    }
    if (seconds === 1) {
        return '1 second';
    }
    if (seconds > 60) {
        return (seconds / 60) + ' minutes';
    }
    return seconds + ' seconds';
}
function renderIfSimpleStatistic(statistic) {
    if (statistic === undefined) {
        return undefined;
    }
    const parsed = statistic_1.parseStatistic(statistic);
    if (parsed.type === 'simple') {
        return parsed.statistic;
    }
    return undefined;
}
function renderIfExtendedStatistic(statistic) {
    if (statistic === undefined) {
        return undefined;
    }
    const parsed = statistic_1.parseStatistic(statistic);
    if (parsed.type === 'percentile') {
        // Already percentile. Avoid parsing because we might get into
        // floating point rounding issues, return as-is but lowercase the p.
        return statistic.toLowerCase();
    }
    else if (parsed.type === 'generic') {
        return statistic;
    }
    return undefined;
}
function mathExprHasSubmetrics(expr) {
    return Object.keys(expr.usingMetrics).length > 0;
}
function assertSubmetricsCount(expr) {
    if (Object.keys(expr.usingMetrics).length > 10) {
        // https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/AlarmThatSendsEmail.html#alarms-on-metric-math-expressions
        throw new Error('Alarms on math expressions cannot contain more than 10 individual metrics');
    }
    ;
}
//# sourceMappingURL=data:application/json;base64,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