"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const consts_1 = require("./cluster-resource-handler/consts");
const cluster_resource_provider_1 = require("./cluster-resource-provider");
/**
 * A low-level CFN resource Amazon EKS cluster implemented through a custom
 * resource.
 *
 * Implements EKS create/update/delete through a CloudFormation custom resource
 * in order to allow us to control the IAM role which creates the cluster. This
 * is required in order to be able to allow CloudFormation to interact with the
 * cluster via `kubectl` to enable Kubernetes management capabilities like apply
 * manifest and IAM role/user RBAC mapping.
 */
class ClusterResource extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.trustedPrincipals = [];
        const stack = core_1.Stack.of(this);
        const provider = cluster_resource_provider_1.ClusterResourceProvider.getOrCreate(this);
        if (!props.roleArn) {
            throw new Error('"roleArn" is required');
        }
        // the role used to create the cluster. this becomes the administrator role
        // of the cluster.
        this.creationRole = new iam.Role(this, 'CreationRole', {
            assumedBy: new iam.CompositePrincipal(...provider.roles.map(x => new iam.ArnPrincipal(x.roleArn))),
        });
        // the CreateCluster API will allow the cluster to assume this role, so we
        // need to allow the lambda execution role to pass it.
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [props.roleArn],
        }));
        // if we know the cluster name, restrict the policy to only allow
        // interacting with this specific cluster otherwise, we will have to grant
        // this role to manage all clusters in the account. this must be lazy since
        // `props.name` may contain a lazy value that conditionally resolves to a
        // physical name.
        const resourceArns = core_1.Lazy.listValue({
            produce: () => {
                const arn = stack.formatArn(clusterArnComponents(stack.resolve(props.name)));
                return stack.resolve(props.name)
                    ? [arn, `${arn}/*`] // see https://github.com/aws/aws-cdk/issues/6060
                    : ['*'];
            },
        });
        const fargateProfileResourceArn = core_1.Lazy.stringValue({
            produce: () => stack.resolve(props.name)
                ? stack.formatArn({ service: 'eks', resource: 'fargateprofile', resourceName: stack.resolve(props.name) + '/*' })
                : '*',
        });
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'ec2:DescribeSubnets',
                'ec2:DescribeRouteTables',
            ],
            resources: ['*'],
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'eks:CreateCluster',
                'eks:DescribeCluster',
                'eks:DescribeUpdate',
                'eks:DeleteCluster',
                'eks:UpdateClusterVersion',
                'eks:UpdateClusterConfig',
                'eks:CreateFargateProfile',
                'eks:TagResource',
                'eks:UntagResource',
            ],
            resources: resourceArns,
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['eks:DescribeFargateProfile', 'eks:DeleteFargateProfile'],
            resources: [fargateProfileResourceArn],
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:GetRole'],
            resources: ['*'],
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:CreateServiceLinkedRole'],
            resources: ['*'],
        }));
        const resource = new core_1.CustomResource(this, 'Resource', {
            resourceType: consts_1.CLUSTER_RESOURCE_TYPE,
            serviceToken: provider.serviceToken,
            properties: {
                Config: props,
                AssumeRoleArn: this.creationRole.roleArn,
            },
        });
        resource.node.addDependency(this.creationRole);
        this.ref = resource.ref;
        this.attrEndpoint = core_1.Token.asString(resource.getAtt('Endpoint'));
        this.attrArn = core_1.Token.asString(resource.getAtt('Arn'));
        this.attrCertificateAuthorityData = core_1.Token.asString(resource.getAtt('CertificateAuthorityData'));
        this.attrOpenIdConnectIssuerUrl = core_1.Token.asString(resource.getAtt('OpenIdConnectIssuerUrl'));
    }
    /**
     * Returns the ARN of the cluster creation role and grants `trustedRole`
     * permissions to assume this role.
     */
    getCreationRoleArn(trustedRole) {
        if (!trustedRole) {
            return this.creationRole.roleArn;
        }
        if (!this.trustedPrincipals.includes(trustedRole.roleArn)) {
            if (!this.creationRole.assumeRolePolicy) {
                throw new Error('unexpected: cluster creation role must have trust policy');
            }
            this.creationRole.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                actions: ['sts:AssumeRole'],
                principals: [new iam.ArnPrincipal(trustedRole.roleArn)],
            }));
            this.trustedPrincipals.push(trustedRole.roleArn);
        }
        return this.creationRole.roleArn;
    }
}
exports.ClusterResource = ClusterResource;
function clusterArnComponents(clusterName) {
    return {
        service: 'eks',
        resource: 'cluster',
        resourceName: clusterName,
    };
}
exports.clusterArnComponents = clusterArnComponents;
//# sourceMappingURL=data:application/json;base64,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