"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobDefinition = void 0;
const core_1 = require("@aws-cdk/core");
const batch_generated_1 = require("./batch.generated");
const job_definition_image_config_1 = require("./job-definition-image-config");
/**
 * Batch Job Definition.
 *
 * Defines a batch job definition to execute a specific batch job.
 */
class JobDefinition extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.jobDefinitionName,
        });
        this.imageConfig = new job_definition_image_config_1.JobDefinitionImageConfig(this, props.container);
        const jobDef = new batch_generated_1.CfnJobDefinition(this, 'Resource', {
            jobDefinitionName: props.jobDefinitionName,
            containerProperties: this.buildJobContainer(props.container),
            type: 'container',
            nodeProperties: props.nodeProps
                ? { mainNode: props.nodeProps.mainNode,
                    nodeRangeProperties: this.buildNodeRangeProps(props.nodeProps),
                    numNodes: props.nodeProps.count }
                : undefined,
            parameters: props.parameters,
            retryStrategy: {
                attempts: props.retryAttempts || 1,
            },
            timeout: {
                attemptDurationSeconds: props.timeout ? props.timeout.toSeconds() : undefined,
            },
        });
        this.jobDefinitionArn = this.getResourceArnAttribute(jobDef.ref, {
            service: 'batch',
            resource: 'job-definition',
            resourceName: this.physicalName,
        });
        this.jobDefinitionName = this.getResourceNameAttribute(jobDef.ref);
    }
    /**
     * Imports an existing batch job definition by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param jobDefinitionArn
     */
    static fromJobDefinitionArn(scope, id, jobDefinitionArn) {
        const stack = core_1.Stack.of(scope);
        const jobDefName = stack.parseArn(jobDefinitionArn).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.jobDefinitionArn = jobDefinitionArn;
                this.jobDefinitionName = jobDefName;
            }
        }
        return new Import(scope, id);
    }
    deserializeEnvVariables(env) {
        const vars = new Array();
        if (env === undefined) {
            return undefined;
        }
        Object.keys(env).map((name) => {
            vars.push({ name, value: env[name] });
        });
        return vars;
    }
    buildJobContainer(container) {
        if (container === undefined) {
            return undefined;
        }
        return {
            command: container.command,
            environment: this.deserializeEnvVariables(container.environment),
            image: this.imageConfig.imageName,
            instanceType: container.instanceType && container.instanceType.toString(),
            jobRoleArn: container.jobRole && container.jobRole.roleArn,
            linuxParameters: container.linuxParams
                ? { devices: container.linuxParams.renderLinuxParameters().devices }
                : undefined,
            memory: container.memoryLimitMiB || 4,
            mountPoints: container.mountPoints,
            privileged: container.privileged || false,
            resourceRequirements: container.gpuCount
                ? [{ type: 'GPU', value: String(container.gpuCount) }]
                : undefined,
            readonlyRootFilesystem: container.readOnly || false,
            ulimits: container.ulimits,
            user: container.user,
            vcpus: container.vcpus || 1,
            volumes: container.volumes,
        };
    }
    buildNodeRangeProps(multiNodeProps) {
        const rangeProps = new Array();
        for (const prop of multiNodeProps.rangeProps) {
            rangeProps.push({
                container: this.buildJobContainer(prop.container),
                targetNodes: `${prop.fromNodeIndex || 0}:${prop.toNodeIndex || multiNodeProps.count}`,
            });
        }
        return rangeProps;
    }
}
exports.JobDefinition = JobDefinition;
//# sourceMappingURL=data:application/json;base64,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